// Copyright 2021 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.
package main

import (
	"os"
	"path/filepath"
	"sort"
	"strings"
	"text/template"

	"honnef.co/go/tools/analysis/lint"
	"honnef.co/go/tools/simple"
	"honnef.co/go/tools/staticcheck"
	"honnef.co/go/tools/stylecheck"
)

const (
	readmeContent = `All of the code in this directory is generated by generate-staticcheck for use in Bazel.
`
	rootBuildBazelContent = `exports_files(["def.bzl"])
`
	analysisFileTemplate = `// Copyright 2021 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.
//
// Code generated by generate-staticcheck; DO NOT EDIT.
//
//go:build bazel
// +build bazel

package {{ .Package }}

import (
	util "github.com/cockroachdb/cockroach/pkg/testutils/lint/passes/staticcheck"
	"golang.org/x/tools/go/analysis"
	"honnef.co/go/tools/{{ .CheckType }}"
)

var Analyzer *analysis.Analyzer

func init() {
	for _, analyzer := range {{ .CheckType }}.Analyzers {
		if analyzer.Analyzer.Name == "{{ .Check }}" {
			Analyzer = analyzer.Analyzer
			break
		}
	}
	util.MungeAnalyzer(Analyzer)
}
`
	buildBzlFileTemplate = `# Code generated by generate-staticcheck; DO NOT EDIT.

load("@io_bazel_rules_go//go:def.bzl", "go_library")

go_library(
    name = "{{ .Package }}",
    srcs = ["analyzer.go"],
    importpath = "github.com/cockroachdb/cockroach/build/bazelutil/staticcheckanalyzers/{{ .Package }}",
    visibility = ["//visibility:public"],
    deps = [
        "//pkg/testutils/lint/passes/staticcheck",
        "@co_honnef_go_tools//{{ .CheckType }}",
        "@org_golang_x_tools//go/analysis",
    ],
)
`
	defBzlFileTemplate = `# Code generated by generate-staticcheck; DO NOT EDIT.

STATICCHECK_CHECKS = [
{{range $i,$a := .AllAnalyzers}}    "{{.}}",
{{end}}]
`
)

func main() {
	fileTpl := template.Must(template.New("source").Parse(analysisFileTemplate))
	buildTpl := template.Must(template.New("buildbazel").Parse(buildBzlFileTemplate))
	rootDir := "build/bazelutil/staticcheckanalyzers"
	err := os.RemoveAll(rootDir)
	if err != nil {
		panic(err)
	}
	err = os.MkdirAll(rootDir, 0755)
	if err != nil {
		panic(err)
	}
	err = os.WriteFile(filepath.Join(rootDir, "README.md"), []byte(readmeContent), 0644)
	if err != nil {
		panic(err)
	}
	err = os.WriteFile(filepath.Join(rootDir, "BUILD.bazel"), []byte(rootBuildBazelContent), 0644)
	if err != nil {
		panic(err)
	}
	// All of these analyzers will be written to def.bzl.
	var allAnalyzers []string
	for _, check := range []struct {
		Analyzers []*lint.Analyzer
		CheckType string
	}{
		// TODO: Consider adding quickfix checks.
		{Analyzers: staticcheck.Analyzers, CheckType: "staticcheck"},
		{Analyzers: stylecheck.Analyzers, CheckType: "stylecheck"},
		{Analyzers: simple.Analyzers, CheckType: "simple"},
	} {
		for _, v := range check.Analyzers {
			analyzer := v.Analyzer
			pkgname := strings.ToLower(analyzer.Name)
			dirname := filepath.Join(rootDir, pkgname)
			err := os.MkdirAll(dirname, 0755)
			if err != nil {
				panic(err)
			}
			outFile, err := os.Create(filepath.Join(dirname, "analyzer.go"))
			if err != nil {
				panic(err)
			}
			vars := struct {
				Package   string
				Check     string
				CheckType string
			}{Package: pkgname, Check: analyzer.Name, CheckType: check.CheckType}
			err = fileTpl.Execute(outFile, vars)
			if err != nil {
				panic(err)
			}
			err = outFile.Close()
			if err != nil {
				panic(err)
			}
			outBuild, err := os.Create(filepath.Join(dirname, "BUILD.bazel"))
			if err != nil {
				panic(err)
			}
			err = buildTpl.Execute(outBuild, vars)
			if err != nil {
				panic(err)
			}
			err = outBuild.Close()
			if err != nil {
				panic(err)
			}
			allAnalyzers = append(allAnalyzers, "//"+dirname)
		}
	}
	sort.Strings(allAnalyzers)
	fileTpl = template.Must(template.New("defbzl").Parse(defBzlFileTemplate))
	defBzlFile, err := os.Create(filepath.Join(rootDir, "def.bzl"))
	if err != nil {
		panic(err)
	}
	err = fileTpl.Execute(defBzlFile, struct{ AllAnalyzers []string }{AllAnalyzers: allAnalyzers})
	if err != nil {
		panic(err)
	}
}
