// Copyright 2024 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package main

import (
	"fmt"
	"io"
	"strings"
	"time"

	"github.com/cockroachdb/cockroach/pkg/util/syncutil"
	"github.com/cockroachdb/cockroach/pkg/util/timeutil"
)

// EventType denotes the type of an event.
type EventType int

const (
	// EventOutput is the default event type. Used for miscellaneous
	// info or log lines with no other semantic meaning.
	EventOutput EventType = iota
	// EventStart denotes a start event, i.e. something started running.
	EventStart
	// EventStart denotes a finish event, i.e. something finished running.
	EventFinish
	// EventError denotes an error event.
	EventError
)

// EventSource denotes the type of source that generated this event.
type EventSource uint8

const (
	// SourceOperation denotes an operation as the event source.
	SourceOperation EventSource = iota
	// SourceWorkload denotes a workload as the event source.
	SourceWorkload
)

// Number of events retained in memory for every workload.
const perWorkloadEventRetention = 1000

// Number of events retained in memory for all operations combined.
const operationEventRetention = 1000

// Event represents a drt-run Event, generated by either a workload
// or an operation. Reflects a log entry.
type Event struct {
	Type       EventType
	Source     EventSource
	SourceName string
	Details    string
	Timestamp  time.Time
}

// String() implements the Stringer interface.
func (e *Event) String() string {
	var builder strings.Builder
	fmt.Fprintf(&builder, "[%s] ", e.Timestamp.Format(time.RFC3339))
	switch e.Source {
	case SourceWorkload:
		fmt.Fprintf(&builder, "workload %s", e.SourceName)
	case SourceOperation:
		fmt.Fprintf(&builder, "operation %s", e.SourceName)
	}
	switch e.Type {
	case EventOutput:
		fmt.Fprintf(&builder, ": %s", e.Details)
	case EventFinish:
		fmt.Fprintf(&builder, " finished with %s", e.Details)
	case EventStart:
		builder.WriteString(" started")
		if e.Details != "" {
			fmt.Fprintf(&builder, " with %s", e.Details)
		}
	case EventError:
		fmt.Fprintf(&builder, " returned an error %s", e.Details)
	}
	return builder.String()
}

// workloadEvents holds events for all workload workers.
type workloadEvents struct {
	workloadName string
	mu           struct {
		syncutil.Mutex
		startIdx  int
		numEvents int
	}
	events [perWorkloadEventRetention]Event
}

// eventLogger is a logger for events generated by workloads and operations
// in drt-run.
type eventLogger struct {
	outputFile      io.Writer
	workloadEvents  []workloadEvents
	operationEvents [operationEventRetention]Event

	mu struct {
		syncutil.Mutex
		operationStartIdx int
		operationEvents   int
	}
}

// logWorkloadEvent logs an event in the workload worker's log. Up to
// perWorkloadEventRetention events are retained in memory for each workload worker.
func (l *eventLogger) logWorkloadEvent(ev Event, workloadIdx int) {
	ev.Timestamp = timeutil.Now()
	ev.Source = SourceWorkload

	we := &l.workloadEvents[workloadIdx]
	var idx int
	func() {
		we.mu.Lock()
		defer we.mu.Unlock()
		idx = we.mu.startIdx
		we.mu.startIdx = (we.mu.startIdx + 1) % perWorkloadEventRetention
		if we.mu.numEvents != len(we.events) {
			we.mu.numEvents++
		}
	}()
	we.events[idx] = ev
	_, _ = io.WriteString(l.outputFile, ev.String()+"\n")
}

// logOperationEvent logs an event in the operation log. Up to operationEventRetention
// events are retained in-memory.
func (l *eventLogger) logOperationEvent(ev Event) {
	ev.Timestamp = timeutil.Now()
	ev.Source = SourceOperation

	var idx int
	func() {
		l.mu.Lock()
		defer l.mu.Unlock()
		idx = l.mu.operationStartIdx
		l.mu.operationStartIdx = (l.mu.operationStartIdx + 1) % operationEventRetention
		if l.mu.operationEvents != len(l.operationEvents) {
			l.mu.operationEvents++
		}
	}()
	l.operationEvents[idx] = ev
	_, _ = io.WriteString(l.outputFile, ev.String()+"\n")
}

// getWorkloadEvents returns workload events for a given workload worker. Up to
// perWorkloadEventRetention events are returned.
func (l *eventLogger) getWorkloadEvents(workloadIdx int) []Event {
	we := &l.workloadEvents[workloadIdx]
	we.mu.Lock()
	defer we.mu.Unlock()
	events := make([]Event, we.mu.numEvents)
	j := 0
	for i := (perWorkloadEventRetention - we.mu.numEvents); i < perWorkloadEventRetention; i++ {
		events[j] = we.events[(we.mu.startIdx+i)%perWorkloadEventRetention]
		j++
	}
	return events
}

// getOperationEvents returns all operation events. Up to operationEventRetention
// events are returned.
func (l *eventLogger) getOperationEvents() []Event {
	l.mu.Lock()
	defer l.mu.Unlock()
	events := make([]Event, l.mu.operationEvents)
	j := 0
	for i := operationEventRetention - l.mu.operationEvents; i < operationEventRetention; i++ {
		events[j] = l.operationEvents[(l.mu.operationStartIdx+i)%operationEventRetention]
		j++
	}
	return events
}

// makeEventLogger creates a new eventLogger. All log entries are also
// tee'd to the supplied io.Writer.
func makeEventLogger(out io.Writer, c config) *eventLogger {
	el := &eventLogger{
		outputFile: out,
	}
	el.workloadEvents = make([]workloadEvents, len(c.Workloads))
	for i := range c.Workloads {
		el.workloadEvents[i].workloadName = c.Workloads[i].Name
	}
	return el
}
