// Copyright 2015 The Cockroach Authors.
//
// Use of this software is governed by the CockroachDB Software License
// included in the /LICENSE file.

package cli

import (
	"fmt"
	"net/url"
	"os"
	"strings"
	"time"

	"github.com/cockroachdb/cockroach/pkg/cli/clierrorplus"
	"github.com/cockroachdb/cockroach/pkg/cli/clisqlexec"
	"github.com/cockroachdb/cockroach/pkg/security"
	"github.com/cockroachdb/cockroach/pkg/security/username"
	"github.com/cockroachdb/errors"
	"github.com/spf13/cobra"
)

const defaultKeySize = 2048

// We use 366 days on certificate lifetimes to at least match X years,
// otherwise leap years risk putting us just under.
const defaultCALifetime = 10 * 366 * 24 * time.Hour  // ten years
const defaultCertLifetime = 5 * 366 * 24 * time.Hour // five years

// A createCACert command generates a CA certificate and stores it
// in the cert directory.
var createCACertCmd = &cobra.Command{
	Use:   "create-ca --certs-dir=<path to cockroach certs dir> --ca-key=<path-to-ca-key>",
	Short: "create CA certificate and key",
	Long: `
Generate a CA certificate "<certs-dir>/ca.crt" and CA key "<ca-key>".
The certs directory is created if it does not exist.

If the CA key exists and --allow-ca-key-reuse is true, the key is used.
If the CA certificate exists and --overwrite is true, the new CA certificate is prepended to it.
`,
	Args: cobra.NoArgs,
	RunE: clierrorplus.MaybeDecorateError(runCreateCACert),
}

// runCreateCACert generates a key and CA certificate and writes them
// to their corresponding files.
func runCreateCACert(cmd *cobra.Command, args []string) error {
	return errors.Wrap(
		security.CreateCAPair(
			certCtx.certsDir,
			certCtx.caKey,
			certCtx.keySize,
			certCtx.caCertificateLifetime,
			certCtx.allowCAKeyReuse,
			certCtx.overwriteFiles),
		"failed to generate CA cert and key")
}

// A createClientCACert command generates a client CA certificate and stores it
// in the cert directory.
var createClientCACertCmd = &cobra.Command{
	Use:   "create-client-ca --certs-dir=<path to cockroach certs dir> --ca-key=<path-to-client-ca-key>",
	Short: "create client CA certificate and key",
	Long: `
Generate a client CA certificate "<certs-dir>/ca-client.crt" and CA key "<client-ca-key>".
The certs directory is created if it does not exist.

If the CA key exists and --allow-ca-key-reuse is true, the key is used.
If the CA certificate exists and --overwrite is true, the new CA certificate is prepended to it.

The client CA is optional and should only be used when separate CAs are desired for server certificates
and client certificates.

If the client CA exists, a client.node.crt client certificate must be created using:
  cockroach cert create-client node

Once the client.node.crt exists, all client certificates will be verified using the client CA.
`,
	Args: cobra.NoArgs,
	RunE: clierrorplus.MaybeDecorateError(runCreateClientCACert),
}

// runCreateClientCACert generates a key and CA certificate and writes them
// to their corresponding files.
func runCreateClientCACert(cmd *cobra.Command, args []string) error {
	return errors.Wrap(
		security.CreateClientCAPair(
			certCtx.certsDir,
			certCtx.caKey,
			certCtx.keySize,
			certCtx.caCertificateLifetime,
			certCtx.allowCAKeyReuse,
			certCtx.overwriteFiles),
		"failed to generate client CA cert and key")
}

// A createNodeCert command generates a node certificate and stores it
// in the cert directory.
var createNodeCertCmd = &cobra.Command{
	Use:   "create-node --certs-dir=<path to cockroach certs dir> --ca-key=<path-to-ca-key> <host 1> <host 2> ... <host N>",
	Short: "create node certificate and key",
	Long: `
Generate a node certificate "<certs-dir>/node.crt" and key "<certs-dir>/node.key".

If --overwrite is true, any existing files are overwritten.

At least one host should be passed in (either IP address or dns name).

Requires a CA cert in "<certs-dir>/ca.crt" and matching key in "--ca-key".
If "ca.crt" contains more than one certificate, the first is used.
Creation fails if the CA expiration time is before the desired certificate expiration.
`,
	Args: func(cmd *cobra.Command, args []string) error {
		if len(args) == 0 {
			return errors.Errorf("create-node requires at least one host name or address, none was specified")
		}
		return nil
	},
	RunE: clierrorplus.MaybeDecorateError(runCreateNodeCert),
}

// runCreateNodeCert generates key pair and CA certificate and writes them
// to their corresponding files.
// TODO(marc): there is currently no way to specify which CA cert to use if more
// than one is present. We should try to load each certificate along with the key
// and pick the one that works. That way, the key specifies the certificate.
func runCreateNodeCert(cmd *cobra.Command, args []string) error {
	return errors.Wrap(
		security.CreateNodePair(
			certCtx.certsDir,
			certCtx.caKey,
			certCtx.keySize,
			certCtx.certificateLifetime,
			certCtx.overwriteFiles,
			args),
		"failed to generate node certificate and key")
}

// A createClientCert command generates a client certificate and stores it
// in the cert directory under <username>.crt and key under <username>.key.
var createClientCertCmd = &cobra.Command{
	Use:   "create-client --certs-dir=<path to cockroach certs dir> --ca-key=<path-to-ca-key> <username>",
	Short: "create client certificate and key",
	Long: `
Generate a client certificate "<certs-dir>/client.<username>.crt" and key
"<certs-dir>/client.<username>.key".

If --overwrite is true, any existing files are overwritten.

Requires a CA cert in "<certs-dir>/ca.crt" and matching key in "--ca-key".
If "ca.crt" contains more than one certificate, the first is used.
Creation fails if the CA expiration time is before the desired certificate expiration.
`,
	Args: cobra.ExactArgs(1),
	RunE: clierrorplus.MaybeDecorateError(runCreateClientCert),
}

// runCreateClientCert generates key pair and CA certificate and writes them
// to their corresponding files.
// TODO(marc): there is currently no way to specify which CA cert to use if more
// than one if present.
func runCreateClientCert(cmd *cobra.Command, args []string) error {
	user, err := username.MakeSQLUsernameFromUserInput(args[0], username.PurposeCreation)
	if err != nil {
		const genError = "failed to generate client certificate and key"
		if _, err := username.MakeSQLUsernameFromUserInput(args[0], username.PurposeValidation); err != nil {
			return errors.Wrap(err, genError)
		}
		if certCtx.disableUsernameValidation {
			// The username is not valid SQL structure, but we're still OK
			// minting a TLS certificate for it. Simply inform the user they
			// will need extra work to use it with SQL.
			fmt.Fprintf(stderr, "warning: the specified identity %q is not a valid SQL username.\n"+
				"Before it can be used to log in, an identity map rule will need to be set on the server.",
				args[0])
		} else {
			return errors.Wrap(err, genError)
		}
	}

	return errors.Wrap(
		security.CreateClientPair(
			certCtx.certsDir,
			certCtx.caKey,
			certCtx.keySize,
			certCtx.certificateLifetime,
			certCtx.overwriteFiles,
			user,
			certCtx.tenantScope,
			certCtx.tenantNameScope,
			certCtx.generatePKCS8Key),
		"failed to generate client certificate and key")
}

// A listCerts command generates a client certificate and stores it
// in the cert directory under <username>.crt and key under <username>.key.
var listCertsCmd = &cobra.Command{
	Use:   "list",
	Short: "list certs in --certs-dir",
	Long: `
List certificates and keys found in the certificate directory.
`,
	Args: cobra.NoArgs,
	RunE: clierrorplus.MaybeDecorateError(runListCerts),
}

// runListCerts loads and lists all certs.
func runListCerts(cmd *cobra.Command, args []string) error {
	cm, err := security.NewCertificateManager(certCtx.certsDir, security.CommandTLSSettings{})
	if err != nil {
		return errors.Wrap(err, "cannot load certificates")
	}

	fmt.Fprintf(os.Stdout, "Certificate directory: %s\n", certCtx.certsDir)

	certTableHeaders := []string{"Usage", "Certificate File", "Key File", "Expires", "Notes", "Error"}
	alignment := "llllll"
	var rows [][]string

	addRow := func(ci *security.CertInfo, notes string) {
		var errString string
		if ci.Error != nil {
			errString = ci.Error.Error()
		}
		rows = append(rows, []string{
			ci.FileUsage.String(),
			ci.Filename,
			ci.KeyFilename,
			ci.ExpirationTime.Format("2006/01/02"),
			notes,
			errString,
		})
	}

	if cert := cm.CACert(); cert != nil {
		var notes string
		if cert.Error == nil && len(cert.ParsedCertificates) > 0 {
			notes = fmt.Sprintf("num certs: %d", len(cert.ParsedCertificates))
		}
		addRow(cert, notes)
	}

	if cert := cm.ClientCACert(); cert != nil {
		var notes string
		if cert.Error == nil && len(cert.ParsedCertificates) > 0 {
			notes = fmt.Sprintf("num certs: %d", len(cert.ParsedCertificates))
		}
		addRow(cert, notes)
	}

	if cert := cm.UICACert(); cert != nil {
		var notes string
		if cert.Error == nil && len(cert.ParsedCertificates) > 0 {
			notes = fmt.Sprintf("num certs: %d", len(cert.ParsedCertificates))
		}
		addRow(cert, notes)
	}

	if cert := cm.NodeCert(); cert != nil {
		var addresses []string
		if cert.Error == nil && len(cert.ParsedCertificates) > 0 {
			addresses = cert.ParsedCertificates[0].DNSNames
			for _, ip := range cert.ParsedCertificates[0].IPAddresses {
				addresses = append(addresses, ip.String())
			}
		} else {
			addresses = append(addresses, "<unknown>")
		}

		addRow(cert, fmt.Sprintf("addresses: %s", strings.Join(addresses, ",")))
	}

	if cert := cm.UICert(); cert != nil {
		var addresses []string
		if cert.Error == nil && len(cert.ParsedCertificates) > 0 {
			addresses = cert.ParsedCertificates[0].DNSNames
			for _, ip := range cert.ParsedCertificates[0].IPAddresses {
				addresses = append(addresses, ip.String())
			}
		} else {
			addresses = append(addresses, "<unknown>")
		}

		addRow(cert, fmt.Sprintf("addresses: %s", strings.Join(addresses, ",")))
	}

	for _, cert := range cm.ClientCerts() {
		var user string
		if cert.Error == nil && len(cert.ParsedCertificates) > 0 {
			user = cert.ParsedCertificates[0].Subject.CommonName
		} else {
			user = "<unknown>"
		}

		addRow(cert, fmt.Sprintf("user: %s", user))
	}

	return sqlExecCtx.PrintQueryOutput(os.Stdout, stderr, certTableHeaders, clisqlexec.NewRowSliceIter(rows, alignment))
}

// encodeURICmd creates a PG URI for the given parameters.
var encodeURICmd = func() *cobra.Command {
	cmd := &cobra.Command{
		Use:   "encode-uri [postgres://][USERNAME[:PASSWORD]@]HOST",
		Short: "encode a CRDB connection URL",
		Args:  cobra.ExactArgs(1),
		RunE:  clierrorplus.MaybeDecorateError(encodeURI),
	}
	f := cmd.PersistentFlags()
	f.BoolVar(&encodeURIOpts.sslInline, "inline", false, "whether to inline certificates (supported by CRDB's Physical Replication feature)")
	f.StringVar(&encodeURIOpts.user, "user", "", "username (overrides any username in the passed URL)")
	f.StringVar(&encodeURIOpts.cluster, "cluster", "system", "virtual cluster to connect to")
	f.StringVar(&encodeURIOpts.certsDir, "certs-dir", "", "certs directory in which to find certs automatically")
	f.StringVar(&encodeURIOpts.caCertPath, "ca-cert", "", "path to CA certificate")
	f.StringVar(&encodeURIOpts.certPath, "cert", "", "path to certificate for client-cert authentication")
	f.StringVar(&encodeURIOpts.keyPath, "key", "", "path to key for client-cert authentication")
	f.StringVar(&encodeURIOpts.database, "database", "defaultdb", "database to connect to")
	return cmd
}()

var encodeURIOpts = struct {
	sslInline  bool
	user       string
	cluster    string
	certsDir   string
	caCertPath string
	certPath   string
	keyPath    string
	database   string
}{}

func encodeURI(cmd *cobra.Command, args []string) error {
	pgURL, err := url.Parse(args[0])
	if err != nil {
		return err
	}

	if pgURL.Scheme == "" {
		pgURL.Scheme = "postgresql://"
	}

	if encodeURIOpts.database != "" {
		pgURL.Path = encodeURIOpts.database
	}

	userName := encodeURIOpts.user
	if userName == "" && pgURL.User != nil {
		userName = pgURL.User.Username()
	}

	user := username.RootUserName()
	if userName != "" {
		u, err := username.MakeSQLUsernameFromPreNormalizedStringChecked(userName)
		if err != nil {
			return err
		}
		user = u
	}

	// Now that we've established the username, update it in the URL.
	if pgURL.User == nil {
		pgURL.User = url.User(user.Normalized())
	} else {
		if pass, hasPass := pgURL.User.Password(); hasPass {
			pgURL.User = url.UserPassword(user.Normalized(), pass)
		} else {
			pgURL.User = url.User(user.Normalized())
		}
	}

	if encodeURIOpts.certsDir != "" {
		cm, err := security.NewCertificateManager(encodeURIOpts.certsDir, security.CommandTLSSettings{})
		if err != nil {
			return errors.Wrap(err, "cannot load certificates")
		}
		if encodeURIOpts.caCertPath == "" {
			encodeURIOpts.caCertPath = cm.CACertPath()
		}

		if encodeURIOpts.certPath == "" {
			encodeURIOpts.certPath = cm.ClientCertPath(user)
		}
		if encodeURIOpts.keyPath == "" {
			encodeURIOpts.keyPath = cm.ClientKeyPath(user)
		}
	}

	options := pgURL.Query()
	if encodeURIOpts.cluster != "" {
		options.Set("options", fmt.Sprintf("-ccluster=%s", encodeURIOpts.cluster))
	}

	if encodeURIOpts.sslInline {
		options.Set("sslinline", "true")
	}

	getCert := func(path string) (string, error) {
		if !encodeURIOpts.sslInline {
			return path, nil
		}
		content, err := os.ReadFile(path)
		if err != nil {
			return "", err
		}
		return string(content), err
	}

	if encodeURIOpts.caCertPath != "" {
		sslRootCert, err := getCert(encodeURIOpts.caCertPath)
		if err != nil {
			return err
		}
		options.Set("sslrootcert", sslRootCert)
		options.Set("sslmode", "verify-full")
	}

	if encodeURIOpts.certPath != "" {
		sslClientCert, err := getCert(encodeURIOpts.certPath)
		if err != nil {
			return err
		}
		options.Set("sslcert", sslClientCert)
	}

	if encodeURIOpts.keyPath != "" {
		sslClientKey, err := getCert(encodeURIOpts.keyPath)
		if err != nil {
			return err
		}
		options.Set("sslkey", sslClientKey)
	}
	pgURL.RawQuery = options.Encode()
	fmt.Printf("%s\n", pgURL.String())
	return nil
}

var certCmds = []*cobra.Command{
	createCACertCmd,
	createClientCACertCmd,
	mtCreateTenantCACertCmd,
	createNodeCertCmd,
	createClientCertCmd,
	mtCreateTenantCertCmd,
	mtCreateTenantSigningCertCmd,
	listCertsCmd,
}

var certCmd = func() *cobra.Command {
	cmd := &cobra.Command{
		Use:   "cert",
		Short: "create ca, node, and client certs",
		RunE:  UsageAndErr,
	}
	cmd.AddCommand(certCmds...)
	return cmd
}()
