/* Copyright 2019 Chris Smeele
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#pragma once

#include "common.hh"
#include "frame.hh"

/**
 * \namespace Interrupt::Handler
 *
 * Interrupt Service Routines.
 *
 * In this file we define all interrupt and exception handlers.
 *
 * Interrupt numbers 0x00-0x1f are exceptions, they are fired on e.g. memory
 * access violations, illegal instructions, and divisions by zero. Interrupts
 * 0x20 - 0x2f are hardware interrupt requests (IRQs), fired e.g. by a keyboard
 * or a timer.
 *
 * Interrupt space above 0x2f we can freely fill in. For example, we allow
 * userspace programs to trigger software interrupt 0xca to perform a syscall.
 * In comparison, Linux uses int 0x80 for this.
 *
 * Side note: In modern OSes, a driver may ask a device to use a different IRQ
 * number in order to avoid conflicts. For simplicity's sake, we do no such
 * thing: All supported hardware uses fixed IRQ numbers, and we do not allow
 * devices to share IRQ numbers.
 *
 * We need to define as many unique functions as there are interrupt numbers
 * that we want to handle, because the handler itself is not told the interrupt
 * number by the CPU.
 *
 * Our approach then, is to create 256 tiny handler functions, that each push
 * their own interrupt number and then call a common handler routine.
 * Interrupts that we do not want to handle we simply detect and ignore.
 * This lets us avoid dealing with interrupt masks and patching the IDT.
 *
 * see handlers.cc for the implementation.
 *
 * ---
 *
 * All drivers that need to handle IRQs register themselves on initialisation.
 *
 * Our fixed IRQ numbers are as follows:
 *
 * - IRQ  0 (int 20h): Programmable Interrupt Timer (PIT)
 * - IRQ  1 (int 21h): PS/2 Keyboard
 * - IRQ  4 (int 24h): Serial port 1
 * - IRQ 12 (int 2ch): PS/2 Mouse
 * - IRQ 14 (int 2eh): Primary ATA bus
 * - IRQ 15 (int 2fh): Secondary ATA bus
 */
namespace Interrupt::Handler {

    /// An IRQ handler is a function that receives a frame and returns nothing.
    using irq_handler_t = function_ptr<void(const interrupt_frame_t &frame)>;

    /// This allows drivers to register their own interrupt handling routines.
    /// (irq must be a value in the range of 0-15, inclusive)
    void register_irq_handler(u8 irq, irq_handler_t handler);

    // Declare all interrupt service routines.
    // We use a macro to avoid boilerplate.

#define I(n) [[gnu::naked]] void isr_##n();

    I(0x00) I(0x01) I(0x02) I(0x03) I(0x04) I(0x05) I(0x06) I(0x07)
    I(0x08) I(0x09) I(0x0a) I(0x0b) I(0x0c) I(0x0d) I(0x0e) I(0x0f)
    I(0x10) I(0x11) I(0x12) I(0x13) I(0x14) I(0x15) I(0x16) I(0x17)
    I(0x18) I(0x19) I(0x1a) I(0x1b) I(0x1c) I(0x1d) I(0x1e) I(0x1f)
    I(0x20) I(0x21) I(0x22) I(0x23) I(0x24) I(0x25) I(0x26) I(0x27)
    I(0x28) I(0x29) I(0x2a) I(0x2b) I(0x2c) I(0x2d) I(0x2e) I(0x2f)
    I(0x30) I(0x31) I(0x32) I(0x33) I(0x34) I(0x35) I(0x36) I(0x37)
    I(0x38) I(0x39) I(0x3a) I(0x3b) I(0x3c) I(0x3d) I(0x3e) I(0x3f)
    I(0x40) I(0x41) I(0x42) I(0x43) I(0x44) I(0x45) I(0x46) I(0x47)
    I(0x48) I(0x49) I(0x4a) I(0x4b) I(0x4c) I(0x4d) I(0x4e) I(0x4f)
    I(0x50) I(0x51) I(0x52) I(0x53) I(0x54) I(0x55) I(0x56) I(0x57)
    I(0x58) I(0x59) I(0x5a) I(0x5b) I(0x5c) I(0x5d) I(0x5e) I(0x5f)
    I(0x60) I(0x61) I(0x62) I(0x63) I(0x64) I(0x65) I(0x66) I(0x67)
    I(0x68) I(0x69) I(0x6a) I(0x6b) I(0x6c) I(0x6d) I(0x6e) I(0x6f)
    I(0x70) I(0x71) I(0x72) I(0x73) I(0x74) I(0x75) I(0x76) I(0x77)
    I(0x78) I(0x79) I(0x7a) I(0x7b) I(0x7c) I(0x7d) I(0x7e) I(0x7f)
    I(0x80) I(0x81) I(0x82) I(0x83) I(0x84) I(0x85) I(0x86) I(0x87)
    I(0x88) I(0x89) I(0x8a) I(0x8b) I(0x8c) I(0x8d) I(0x8e) I(0x8f)
    I(0x90) I(0x91) I(0x92) I(0x93) I(0x94) I(0x95) I(0x96) I(0x97)
    I(0x98) I(0x99) I(0x9a) I(0x9b) I(0x9c) I(0x9d) I(0x9e) I(0x9f)
    I(0xa0) I(0xa1) I(0xa2) I(0xa3) I(0xa4) I(0xa5) I(0xa6) I(0xa7)
    I(0xa8) I(0xa9) I(0xaa) I(0xab) I(0xac) I(0xad) I(0xae) I(0xaf)
    I(0xb0) I(0xb1) I(0xb2) I(0xb3) I(0xb4) I(0xb5) I(0xb6) I(0xb7)
    I(0xb8) I(0xb9) I(0xba) I(0xbb) I(0xbc) I(0xbd) I(0xbe) I(0xbf)
    I(0xc0) I(0xc1) I(0xc2) I(0xc3) I(0xc4) I(0xc5) I(0xc6) I(0xc7)
    I(0xc8) I(0xc9) I(0xca) I(0xcb) I(0xcc) I(0xcd) I(0xce) I(0xcf)
    I(0xd0) I(0xd1) I(0xd2) I(0xd3) I(0xd4) I(0xd5) I(0xd6) I(0xd7)
    I(0xd8) I(0xd9) I(0xda) I(0xdb) I(0xdc) I(0xdd) I(0xde) I(0xdf)
    I(0xe0) I(0xe1) I(0xe2) I(0xe3) I(0xe4) I(0xe5) I(0xe6) I(0xe7)
    I(0xe8) I(0xe9) I(0xea) I(0xeb) I(0xec) I(0xed) I(0xee) I(0xef)
    I(0xf0) I(0xf1) I(0xf2) I(0xf3) I(0xf4) I(0xf5) I(0xf6) I(0xf7)
    I(0xf8) I(0xf9) I(0xfa) I(0xfb) I(0xfc) I(0xfd) I(0xfe) I(0xff)

#undef I
}
