:- module(etermsvar, [], [assertions,regtypes,basicmodes,datafacts]).

:- doc(title,"etermsvar: eterms with variables (abstract domain)").
:- doc(author, "Claudio Vaucheret").
:- doc(author, "Francisco Bueno").
:- doc(author, "Alejandro Serrano (etermsvar)").
:- doc(author, "Ciao Development Team").

% TODO: document widening and abstract domain (Name)
% (See bugs inherited from eterms.pl)

:- doc(module,"This module implements an (experimental) variant of the
   @lib{eterms} domain with @tt{vr} elements in the lattice.").
% ===========================================
% NOTES ABOUT ETERMSVAR ABSTRACT DOMAIN (ASM)
% ===========================================
% 
% The etermsvar domain enhances the eterms domain of regular types by adding
% a new native type: "vr", which corresponds to variables. This is different
% to the combination of the eterms domain (which gives information only about
% ground terms) and shfr (which gives information about freeness of variables).
% For example, a "list of variables" can be represented in etermsvar as
% list(vr), but cannot be represented using eterms+shfr. Conversely, the shfr
% is able to represent "X and Y are alias to the same variable", whereas
% etermsvar cannot.
% 
% The main user of etermsvar right now is the "res_plai" abstract domain,
% which analyzes sized types and resources. The combined types+freeness
% information that etermsvar gives is used for determining which parts of the
% head variables should be considered as input and which as output. And this
% etermsvar domain is the only one that allows to tell that the head of a list
% is a variable, while the tail is ground.
% 
% Basically, the etermsvar domain adds a new type to the domain, "vr", which
% is not comparable with any other type but with "term" (which is the top of
% the lattice) and "$bottom". Most of the operations are then the same as in
% etermsvar, where the new element is added to the lattice.
% 
% This means that in general the LUB of a "list of numbers" and "vr" will be
% be a new type which is the disjunction of those two. In most cases this 
% behavior inherited from the eterms domain and the typeslib is correct.
% However, in a couple of places changes are needed and this is the objective
% of the etermsvar domain.
% 
% For example, let's consider the predicates:
% 
%   f(X,Y) :- g(Y).
%   g([]).
% 
% and analyze f with the call substitution num(X), vr(Y). At some point we will
% analyze g, for which we would get an assertion vr(Y) => list(Y). If in the
% extend step we perform the same as in etermsvar, we find a puzzling result.
% So the correct solution is to create, for those cases, a special way of inter-
% secting types which takes into account that intersect(vr,X) = X for every X.
% ===========================================================================

% ===========================================================================
% TODO: WARNING!!! This is work in progress. Some known problems is
%   that this domain may produce incorrect results if the case of
%   variable sharing:
%
%   :- entry p(X) : var(X).
%   p(Z) :-
%     Y = Z,
%     X = f(Y,Z),
%     X = f(1,_).
%
%   Note that this may not be a problem for programs without (logical)
%   variable sharing, such as those typically generated by translating
%   imperative programs to Horn clauses.
%
%   NOTE: This domain is required for resource analysis.
%
%   Related domains: svterms, optim_comp's domains
% ===========================================================================

:- include(ciaopp(plai/plai_domain)).
:- dom_def(etermsvar).

:- use_module(typeslib(typeslib), [
    dz_type_included/2,
    insert_rule/2,
    insert_type_name/3,
    new_type_name/1,
    new_type_symbol/1,
    pure_type_term/1,
    lab_intersect/2,
    set_float_type/1,
    set_numeric_type/1,
    set_top_type/1,
    set_var_type/1,
    set_ground_type/1,
    top_type/1,
    var_type/1,
    type_escape_term_list/2,
    type_intersection_2/3,
    type_intersection_2_VR/3,
    resetunion_VR/0,
    type_union_VR/3,
    lnewiden_el_VR/4,
    get_canonical_name/2,
    concrete/4]).

% CiaoPP library
:- use_module(ciaopp(preprocess_flags), [current_pp_flag/2]).

%:- use_module(engine(hiord_rt), [call/1]).
%:- use_module(engine(hiord_rt), ['$meta_call'/1]).

:- use_module(ciaopp(plai/apply_assertions_old), [apply_trusted0/7]).

:- use_module(library(messages)).
:- use_module(library(aggregates), [setof/3]).
:- use_module(library(terms_vars), [varset/2]).
:- use_module(library(terms_check), [variant/2]).
:- use_module(library(lists), [member/2, append/3]).
:- use_module(library(sets), [merge/3, insert/3]).
:- use_module(library(sort), [sort/2]).

:- regtype absu(A) # "@var{A} is an abstract substitution".

absu('$bottom').
absu([]).
absu([Elem|Absu]):- 
    absu_elem(Elem),
    absu(Absu).

:- regtype absu_elem(E) # "@var{E} is a single substitution".

absu_elem(Var:Type):-
    var(Var),
    pure_type_term(Type).

:- regtype extrainfo(_).
extrainfo(_). % TODO: define

%------------------------------------------------------------------%

:- use_module(ciaopp(preprocess_flags), [push_pp_flag/2]).

:- dom_impl(etermsvar, init_abstract_domain/1).
:- export(etermsvar_init_abstract_domain/1).
etermsvar_init_abstract_domain([type_eval,widen]) :-
    push_pp_flag(type_eval,on),
    push_pp_flag(widen,on).

%------------------------------------------------------------------%
:- dom_impl(etermsvar, compute_lub/2).
:- export(etermsvar_compute_lub/2).
:- pred etermsvar_compute_lub(+ListASub,-Lub) : list(absu) * absu.

etermsvar_compute_lub([ASub1,ASub2|Rest],Lub):-
    etermsvar_compute_lub_el(ASub1,ASub2,ASub3),
    !,
    etermsvar_compute_lub([ASub3|Rest],Lub).
etermsvar_compute_lub([ASub],ASub).

%------------------------------------------------------------------%

:- export(etermsvar_compute_lub_el/3).
etermsvar_compute_lub_el('$bottom',ASub,ASub):- !.
etermsvar_compute_lub_el(ASub,'$bottom',ASub):- !.
etermsvar_compute_lub_el(ASub1,ASub2,ASub3):-
    ASub1 == ASub2, !,
    ASub3 = ASub2.
etermsvar_compute_lub_el(ASub1,ASub2,ASub3):-
    etermsvar_lub0(ASub1,ASub2,ASub3).

etermsvar_lub0([X:(N1_e,T1)|ASub1],[Y:(N2_e,T2)|ASub2],[X:(N2,T3)|ASub3]):-
    X==Y,
    get_canonical_name(N1_e,N1),
    get_canonical_name(N2_e,N2),
    ( 
        ( top_type(T2) ; top_type(T1) ) -> set_top_type(T3) 
    ;
        resetunion_VR,
        type_union_VR(T1,T2,T3)
    ),
%       lab_intersect(N1,N2),
    lab_intersect(N2,N1),
    etermsvar_lub0(ASub1,ASub2,ASub3).
etermsvar_lub0([],[],[]).

%---------------------------------------------------------------------%  
% Widening

:- dom_impl(etermsvar, widencall/3).
:- export(etermsvar_widencall/3).
etermsvar_widencall(Prime0,Prime1,Result):-
    % display(user,'widencall'),
    etermsvar_widen(Prime0,Prime1,Result).  


:- dom_impl(etermsvar, needs/1).
:- export(etermsvar_needs/1).
etermsvar_needs(X) :-
    eterms_needs(X).

:- dom_impl(etermsvar, widen/3).
:- export(etermsvar_widen/3).
:- pred etermsvar_widen(+Prime0,+Prime1,-NewPrime) : absu * absu * absu.

etermsvar_widen(Prime0,Prime1,NewPrime):-
%       display(user,'widen'),nl(user),
    etermsvar_compute_lub_el(Prime0,Prime1,Prime),
    ewiden(Prime0,Prime,NewPrime).

ewiden('$bottom','$bottom','$bottom').
ewiden('$bottom',Prime,Prime).
ewiden([],[],[]).
ewiden([X:T1|Prime0],[X:T2|Prime],[X:T|NewPrime]):-
    ( current_pp_flag(type_eval,on) -> TEval = yes ; TEval = no ),
    lnewiden_el_VR(T1,T2,TEval,T),
    ewiden(Prime0,Prime,NewPrime).

%------------------------------------------------------------------%
:- dom_impl(etermsvar, call_to_entry/9).
:- export(etermsvar_call_to_entry/9).
:- pred etermsvar_call_to_entry(+Sv,+Sg,+Hv,+Head,+K,+Fv,+Proj,-Entry,-ExtraInfo)
   : term * cgoal * list * cgoal * term * list * absu * absu * extrainfo.

etermsvar_call_to_entry(_Sv,Sg,Hv,Head,_K,Fv,Proj,Entry,(yes,Proj)):- 
    variant(Sg,Head), !,
    copy_term((Sg,Proj),(NewTerm,NewProj_u)),
    Head = NewTerm,
    eterms_abs_sort(NewProj_u,NewProj),
    eterms_project(Sg,Hv,not_provided_HvFv_u,NewProj,NewProj1),
    variables_are_variable_type(Fv,Free),
    merge(Free,NewProj1,Entry).
etermsvar_call_to_entry(_Sv,Sg,Hv,Head,_K,Fv,Proj,Entry,(no,Proj)):-
    unify_term_and_type_term(Head,Hv,Sg,Proj,TmpEntry), !,
    variables_are_variable_type(Fv,Tmp),
    merge(Tmp,TmpEntry,Entry).
etermsvar_call_to_entry(_Sv,_Sg,_Hv,_Head,_K,_Fv,_Proj,'$bottom',no).

% (new version by ASM)
:- pred variables_are_variable_type(+Fv,-ASub)
   :: list * absu
   # "It assigns the value var_type to the variables in @var{Fv} and
   return the abstract substitution @var{ASub}".

variables_are_variable_type([V|Fv],[V:(Name,Type)|ASub]):-
    set_var_type(Type),
    new_type_name(Name),
    insert_type_name(Name,[],0),
    variables_are_variable_type(Fv,ASub).
variables_are_variable_type([],[]).

%------------------------------------------------------------------%
:- dom_impl(etermsvar, exit_to_prime/7).
:- export(etermsvar_exit_to_prime/7).
:- pred etermsvar_exit_to_prime(+Sg,+Hv,+Head,+Sv,+Exit,-ExtraInfo,-Prime)
   : list * list * cgoal * cgoal * absu * extrainfo * absu.

% TODO: almost duplicated
etermsvar_exit_to_prime(_Sg,_Hv,_Head,_Sv,'$bottom',_ExtraInfo,Prime) :- !,
    Prime = '$bottom'.
etermsvar_exit_to_prime(Sg,Hv,Head,_Sv,Exit,(yes,Proj),Prime):- !,
    eterms_project(Sg,Hv,not_provided_HvFv_u,Exit,BPrime),
    copy_term((Head,BPrime),(NewTerm,NewPrime)),
    Sg = NewTerm,
    eterms_abs_sort(NewPrime,Prime1),
    replace_names(Proj,Prime1,Prime).       
etermsvar_exit_to_prime(Sg,Hv,Head,Sv,Exit,(no,ExtraInfo),Prime):- 
    eterms_project(Sg,Hv,not_provided_HvFv_u,Exit,BPrime),
    unify_term_and_type_term_exit(Sg,Sv,Head,BPrime,ExtraInfo,Prime). %,!, %change

% probar agregar sinonimos de ExtraInfo a Prime
%%      replace_names(ExtraInfo,Prime1,Prime).
etermsvar_exit_to_prime(_Sg,_Hv,_Head,_Sv,_Exit,_ExtraInfo,'$bottom').

:- pred unify_term_and_type_term(+Term1,+Tv,+Term2,+ASub,-NewASub)
   : cgoal * list * cgoal * absu * absu.

unify_term_and_type_term_exit(Term1,Tv,Term2,ASub,Proj,NewASub):-
    copy_term((Term2,ASub),(TypeTerm,ASub0)),
    Term2 =.. [_|HeadArg], 
    TypeTerm =.. [_|Types],
    Term1 =.. [_|Args],
    type_escape_term_list(Types,EscTypes),
    apply(ASub0),
    generate_a_type_assignment_VR(EscTypes,Args,TypeAss),
    ( 
        member(_:bot,TypeAss) -> fail
    ;
        sort(Proj,Proj_s),
        eterms_abs_sort(TypeAss,ASub1),
        obtains_names(HeadArg,Args,ASub,TypeNameAss),

        % generate_subs_exit(ASub1,Proj,Subs),
        % update_names(TypeNameAss,Subs),

        sort(TypeNameAss,TypeNameAss_s),
        generate_subs_exit(ASub1,Proj_s,TypeNameAss_s,Subs),

        eterms_project(not_provided_Sg,Tv,not_provided_HvFv_u,Subs,NASub),
        normal_asub(NASub,NewASub)
    ).

unify_term_and_type_term(Term1,Tv,Term2,ASub,NewASub):-
    copy_term((Term2,ASub),(TypeTerm,ASub0)),
    Term2 =.. [_|HeadArg], 
    TypeTerm =.. [_|Types],
    Term1 =.. [_|Args],
    type_escape_term_list(Types,EscTypes),
    apply(ASub0),
    generate_a_type_assignment_VR(EscTypes,Args,TypeAss),
    ( 
        member(_:bot,TypeAss) -> fail
    ;
        sort(ASub,ASub_s),
        eterms_abs_sort(TypeAss,ASub1),
        obtains_names(HeadArg,Args,ASub_s,TypeNameAss),
        sort(TypeNameAss,ASub2),
        generate_subs(ASub1,ASub2,Subs),
        obtains_names(Args,HeadArg,Subs,TypeNameAss2),
        sort(TypeNameAss2,TypeNameAss2_s),
        update_names(TypeNameAss2_s,ASub_s),
        eterms_project(not_provided_Sg,Tv,not_provided_HvFv_u,Subs,NASub),
        normal_asub(NASub,NewASub)
    ).

% TODO: duplicated
:- pred apply(+ASub) : absu.

apply([X:(_N,Term)|ASub]):-
    X=Term,
    apply(ASub).
apply([]).

generate_a_type_assignment_VR(Type_List, Term_List, TypeAss):- 
    varset(Term_List, Term_Vars),
    get_var_types_by_unification(Type_List, Term_List, Types),
    intersec_types_2_VR(Term_Vars, Types, [], TypeAss).

% TODO:[new-resources] preliminary support for var_type (for etermsvar)
% TODO: merge with intersec_types_2/4

% ASM, 6 Sep 2012
% Special intersection for type assignment
% Variables which do not have type are assigned the top type.
% If a variable may be assigned the type var, a special case is used:
% - if only var is present, the resulting type is var
% - if not, the resulting type is the intersection without any var

intersec_types_2_VR([], _Var_Types, OTypeAss, OTypeAss):- !.
intersec_types_2_VR([Var|List], Var_Types, ITypeAss, OTypeAss):-
    find_list_entry(Var_Types, Var, Entry),
    ( var(Entry) ->
        Types = _
    ; Entry = vt(_, Types)
    ),
    set_top_type(Top),
    ( contains_var_type(Types) ->
        delete_var_type(Types,DTypes),
        ( var(DTypes) -> % Is list empty?
            set_var_type(LType)
        ; intersec_type_list_2_VR(DTypes, Top, LType)
        )
    ; intersec_type_list_2_VR(Types, Top, LType)
    ),
    % \+ bot_type(LType),
    intersec_types_2_VR(List, Var_Types, [Var:LType|ITypeAss], OTypeAss).

contains_var_type([Type|_]) :-
    nonvar(Type),
    var_type(Type),
    !.
contains_var_type([_|List]) :-
    nonvar(List),
    contains_var_type(List).

delete_var_type(Type_List,Type_List) :-
    var(Type_List),
    !.
delete_var_type([Type|List],DList) :-
    var_type(Type),
    !,
    delete_var_type(List,DList).
delete_var_type([Type|List],[Type|DList]) :-
    delete_var_type(List,DList).

intersec_type_list_2_VR(Type_List, Type, Type):-
    var(Type_List), 
    !.
intersec_type_list_2_VR(Type_List, InType, OutType):-
    nonvar(Type_List),
    Type_List = [Type|List],
    ( var_type(Type) ->
        intersec_type_list_2_VR(List, InType, Intersec),
        ( ( top_type(Intersec) ; var_type(Intersec) ) ->
            set_var_type(OutType)
        ; OutType = Intersec
        )
    ; type_intersection_2_VR(InType, Type, Intersec),
      intersec_type_list_2_VR(List, Intersec, OutType)
    ).

%------------------------------------------------------------------%
:- dom_impl(etermsvar, project/5).
:- export(etermsvar_project/5).
:- pred etermsvar_project(+Sg,+Vars,+HvFv_u,+Asub,-Proj)
   : term * list * list * absu * absu.

etermsvar_project(Sg,Vars,HvFv_u,ASub,Proj) :- 
    eterms_project(Sg,Vars,HvFv_u,ASub,Proj).

%------------------------------------------------------------------%
:- dom_impl(etermsvar, abs_sort/2).
:- export(etermsvar_abs_sort/2).
:- pred etermsvar_abs_sort(+Asub,-Asub_s) : absu * absu.

etermsvar_abs_sort(ASub,ASub_s) :-
    eterms_abs_sort(ASub,ASub_s).

%------------------------------------------------------------------%
:- dom_impl(etermsvar, extend/5).
:- export(etermsvar_extend/5).
:- pred etermsvar_extend(+Sg,+Prime,+Sv,+Call,-Succ)
   : term * absu * list * absu * absu.

etermsvar_extend(Sg,Prime,Sv,Call,Succ) :-
    eterms_extend(Sg,Prime,Sv,Call,Succ).

%------------------------------------------------------------------%
:- dom_impl(etermsvar, less_or_equal/2).
:- export(etermsvar_less_or_equal/2).
:- pred etermsvar_less_or_equal(+ASub0,+ASub1) : absu * absu.

etermsvar_less_or_equal('$bottom',_ASub):- !.
etermsvar_less_or_equal(ASub1,ASub2):-
    ASub1 == ASub2, !.
etermsvar_less_or_equal(ASub1,ASub2):-
    etermsvar_less_or_equal0(ASub1,ASub2).

etermsvar_less_or_equal0([X:(_N1,T1)|ASub1],[Y:(_N2,T2)|ASub2]):-
    X==Y,
    dz_type_included(T1,T2),
    etermsvar_less_or_equal0(ASub1,ASub2).
etermsvar_less_or_equal0([],[]).

%------------------------------------------------------------------%
:- dom_impl(etermsvar, glb/3).
:- export(etermsvar_glb/3).
:- pred etermsvar_glb(+ASub0,+ASub1,-Glb) : absu * absu * absu.

etermsvar_glb('$bottom',_ASub,ASub3) :- !, ASub3='$bottom'.
etermsvar_glb(_ASub,'$bottom',ASub3) :- !, ASub3='$bottom'.
etermsvar_glb(ASub1,ASub2,ASub3):-
    ASub1 == ASub2, !,
    ASub3 = ASub2.
etermsvar_glb(ASub1,ASub2,ASub3):-
    etermsvar_glb0(ASub1,ASub2,ASub33), !,
    eterms_glbnames(ASub1,ASub33,ASub3).
etermsvar_glb(_ASub1,_ASub2,'$bottom').

etermsvar_glb0([X:(_N1,T1)|ASub1],[Y:(N2,T2)|ASub2],[X:(N2,T3)|ASub3]):-
    X==Y,
    type_intersection_2_VR(T1,T2,T3),
    ( T3==bot ->
        fail 
    ; etermsvar_glb0(ASub1,ASub2,ASub3)
    ).
etermsvar_glb0([],[],[]).

% TODO: why?
etermsvar_glb_without_var('$bottom',_ASub,'$bottom'):- !.
etermsvar_glb_without_var(_ASub,'$bottom','$bottom'):- !.
etermsvar_glb_without_var(ASub1,ASub2,ASub3):-
    ASub1 == ASub2, !,
    ASub3 = ASub2.
etermsvar_glb_without_var(ASub1,ASub2,ASub3):-
    etermsvar_glb0_without_var(ASub1,ASub2,ASub33), !,
    eterms_glbnames(ASub1,ASub33,ASub3).
etermsvar_glb_without_var(_ASub1,_ASub2,'$bottom').

etermsvar_glb0_without_var([X:(_N1,T1)|ASub1],[Y:(N2,T2)|ASub2],[X:(N2,T3)|ASub3]):-
    X==Y,
    ( var_type(T1) -> T3 = T2
    ; var_type(T2) -> T3 = T1
    ; type_intersection_2(T1,T2,T3) % TODO: why not _VR?
    ),
    ( T3==bot ->
        fail 
    ; etermsvar_glb0_without_var(ASub1,ASub2,ASub3)
    ).
etermsvar_glb0_without_var([],[],[]).

%------------------------------------------------------------------%

% :- dom_impl(etermsvar, concrete/3).
:- export(etermsvar_concrete/3).
etermsvar_concrete(Var,ASub,List):-
    eterms_concrete(Var,ASub,List).

%------------------------------------------------------------------%
:- dom_impl(etermsvar, unknown_entry/3).
:- export(etermsvar_unknown_entry/3).
:- pred etermsvar_unknown_entry(+Sg,+Qv,-Call) : cgoal * list * absu.

etermsvar_unknown_entry(_Sg,Vars,ASub):-
    variables_are_top_type(Vars,ASub).

:- dom_impl(etermsvar, empty_entry/3).
:- export(etermsvar_empty_entry/3).
:- pred etermsvar_empty_entry(+Sg,+Vars,-Entry) : cgoal * list * absu.

etermsvar_empty_entry(_Sg,Vars,ASub):-
    variables_are_variable_type(Vars,ASub).

%------------------------------------------------------------------%
:- dom_impl(etermsvar, unknown_call/4).
:- export(etermsvar_unknown_call/4).
:- pred etermsvar_unknown_call(+Sg,+Vars,+Call,-Succ)
   : cgoal * list * absu * absu.

etermsvar_unknown_call(Sg,Vars,Call,Succ):-
    eterms_unknown_call(Sg,Vars,Call,Succ).

%------------------------------------------------------------------%
:- dom_impl(etermsvar, call_to_success_fact/9).
:- export(etermsvar_call_to_success_fact/9).
:- pred etermsvar_call_to_success_fact(+Sg,+Hv,+Head,+K,+Sv,+Call,+Proj,-Prime,-Succ)
   : cgoal * list * cgoal * term * list * absu * absu * absu * absu.

etermsvar_call_to_success_fact(Sg,Hv,Head,K,Sv,Call,Proj,Prime,Succ):-
    etermsvar_call_to_entry(Sv,Sg,Hv,Head,K,[],Proj,Entry,ExtraInfo),
    etermsvar_exit_to_prime(Sg,Hv,Head,Sv,Entry,ExtraInfo,Prime),
    eterms_extend(Sg,Prime,Sv,Call,Succ).

%------------------------------------------------------------------------%
% Builtins

:- dom_impl(etermsvar, special_builtin/5).
:- export(etermsvar_special_builtin/5).
:- pred etermsvar_special_builtin(+SgKey,+Sg,+Subgoal,-Type,-Condvars).

% (reuses eterms_special_builtin/5, with some special cases)
etermsvar_special_builtin('var/1',Sg,_,type(T),Condvars):- !, % TODO: var/1 vs free/1?
    set_var_type(T),
    varset(Sg,Condvars).
etermsvar_special_builtin('free/1',Sg,_,type(T),Condvars):- !,
    set_var_type(T),
    varset(Sg,Condvars).
etermsvar_special_builtin('nonvar/1',Sg,_,type(T),Condvars):- !, % TODO: nonvar/1 vs not_free/1?
    set_ground_type(T), % TODO: wrong?
    varset(Sg,Condvars).
etermsvar_special_builtin('not_free/1',Sg,_,type(T),Condvars):- !,
    set_ground_type(T), % TODO: wrong?
    varset(Sg,Condvars).
etermsvar_special_builtin('ground/1',Sg,_,type(T),Condvars):- !,
    set_ground_type(T),
    varset(Sg,Condvars).
etermsvar_special_builtin('float/1',Sg,_,type(T),Condvars):- !,
    set_float_type(T),
    varset(Sg,Condvars).
etermsvar_special_builtin('number/1',Sg,_,type(T),Condvars):- !,
    set_numeric_type(T),
    varset(Sg,Condvars).
%
etermsvar_special_builtin(Key,_Sg,_Subgoal,special(Key),[]):- arith_builtin(Key), !. 
etermsvar_special_builtin(SgKey,Sg,Subgoal,Type,Condvars) :-
    eterms_special_builtin(SgKey,Sg,Subgoal,Type,Condvars).

% TODO: eterms.pl only enables them when current_pp_flag(type_eval,on)
arith_builtin('is/2').
arith_builtin('>/2').
arith_builtin('>=/2').
arith_builtin('=</2').
arith_builtin('</2').

%------------------------------------------------------------------------%
:- dom_impl(etermsvar, success_builtin/6).
:- export(etermsvar_success_builtin/6).
:- pred etermsvar_success_builtin(+Type,+Sv_uns,+Condvars,+HvFv_u,+Call,-Succ).

etermsvar_success_builtin(Type,Sv_uns,Condvars,HvFv_u,Call,Succ):-
    eterms_success_builtin(Type,Sv_uns,Condvars,HvFv_u,Call,Succ).

%------------------------------------------------------------------------%
:- export(etermsvar_arg_call_to_success/9).
etermsvar_arg_call_to_success(Sg,Hv,arg(X,Y,Z),Sv,Call,Proj,Succ,TypeX,TypeY):-
    etermsvar_call_to_entry(Sv,Sg,Hv,arg(X,Y,Z),not_provided,[],Proj,Entry,ExtraInfo), % TODO: add some ClauseKey? (JF)
    get_type(X,Entry,TypeX),
    get_type(Y,Entry,TypeY),
    new_type_name(NX),
    new_type_name(NY),
    new_type_name(NZ),
    insert_type_name(NX,[],0),
    insert_type_name(NY,[],0),
    insert_type_name(NZ,[],0),
    sort([X:(NX,int),Y:(NY,term),Z:(NZ,term)],Prime1), % postcondition builtin
    ( concrete(TypeX,ValuesX,[],[]) -> 
        ( getargtypes(TypeY,ValuesX,ValuesY,[],_,_) ->
            resetunion_VR,
            set_union_VR(ValuesY,TZ),
            replacetype(Z,Entry,TZ,Prime0),
            etermsvar_glb(Prime0,Prime1,Prime2)
        ;
            Prime2 = Prime1
        )
    ; 
        Prime2 = Prime1
    ),
    etermsvar_glb(Prime2,Entry,Prime3),
    etermsvar_exit_to_prime(Sg,Hv,arg(X,Y,Z),Sv,Prime3,ExtraInfo,Prime),
    eterms_extend(Sg,Prime,Sv,Call,Succ).

set_union_VR([T],T).
set_union_VR([T1,T2|L],T):-
    type_union_VR(T1,T2,T3),
    set_union_VR([T3|L],T).

:- dom_impl(etermsvar, call_to_success_builtin/6).
:- export(etermsvar_call_to_success_builtin/6).
:- pred etermsvar_call_to_success_builtin(+SgKey,+Sg,+Sv,+Call,+Proj,-Succ).

etermsvar_call_to_success_builtin('arg/3',Sg,Sv,Call,Proj,Succ):-
    sort([X,Y,Z],Hv),
    etermsvar_arg_call_to_success(Sg,Hv,arg(X,Y,Z),Sv,Call,Proj,Succ,_,_).
%
etermsvar_call_to_success_builtin('functor/3',Sg,Sv,Call,Proj,Succ):-
    % TODO: share with eterms.pl version; call domain operations instead?
    sort([X,Y,Z],Hv),
    Head = functor(X,Y,Z),
    etermsvar_call_to_entry(Sv,Sg,Hv,Head,not_provided,[],Proj,Entry,ExtraInfo), % TODO: add some ClauseKey?
    get_type(X,Entry,TypeX),
    get_type(Y,Entry,TypeY),
    get_type(Z,Entry,TypeZ),
    ( getfunctors(TypeX,ValuesX) -> true ; true),
    ( concrete(TypeY,ValuesY,[],[]) -> true ; true),
    ( concrete(TypeZ,ValuesZ,[],[]) -> true ; true),
    new_type_name(NX),
    new_type_name(NY),
    new_type_name(NZ),
    insert_type_name(NX,[],0),
    insert_type_name(NY,[],0),
    insert_type_name(NZ,[],0),
    sort([X:(NX,term),Y:(NY,atm),Z:(NZ,int)],Prime1),
    ( setof(f(X,Y,Z),(getvalue(X,ValuesX),getvalue(Y,ValuesY),getvalue(Z,ValuesZ),functor(X,Y,Z)),ListF) -> 
        split_f(ListF,ListX,ListY,ListZ),
        new_type_symbol(TX),
        new_type_symbol(TY),
        new_type_symbol(TZ),
        sort(ListX,DefX1),
        sort(ListY,DefY1),
        sort(ListZ,DefZ1),
        varset(DefX1,VarsX),
        type_escape_term_list(DefX1,DefX),
        type_escape_term_list(DefY1,DefY),
        type_escape_term_list(DefZ1,DefZ),
        unifytoterm(VarsX),
        insert_rule(TX,DefX),
        insert_rule(TY,DefY),
        insert_rule(TZ,DefZ),
        sort([X:(NX,TX),Y:(NY,TY),Z:(NZ,TZ)],Prime0),
        etermsvar_glb(Prime0,Prime1,Prime2)
    ;
        Prime2 = Prime1
    ),
    etermsvar_glb(Prime2,Entry,Prime3),
    etermsvar_exit_to_prime(Sg,Hv,Head,Sv,Prime3,ExtraInfo,Prime),
    eterms_extend(Sg,Prime,Sv,Call,Succ).
%
etermsvar_call_to_success_builtin('=/2',X=Y,Sv,Call,Proj,Succ):-
    etermsvar_call_to_success_fact(p(X,Y),[W],p(W,W),not_provided,Sv,Call,Proj,_Prime,Succ). % TODO: add some ClauseKey?
%
etermsvar_call_to_success_builtin(Key,Sg,Sv,Call,Proj,Succ):-
    member(Key,['>/2','>=/2','=</2','</2']),
    TY = 'arithmetic:arithexpression',
    TX = 'arithmetic:arithexpression',
    new_type_name(NX),
    new_type_name(NY),
    insert_type_name(NX,[],0),
    insert_type_name(NY,[],0),
    Exit_u = [X:(NX,TX),Y:(NY,TY)],
    Bv_u = [X,Y],
    sort(Exit_u,Exit),
    sort(Bv_u,Bv),
    functor(Sg,F,2),
    functor(G,F,2),
    arg(1,G,X),
    arg(2,G,Y),
    etermsvar_exit_to_prime(Sg,Bv,G,Sv,Exit,(no,Proj),Prime1),
    etermsvar_glb(Proj,Prime1,Prime2),
    ( Prime2 \== '$bottom' ->
        abs_eval_arithcomp(Sg,Sv,Prime2,Prime)
    ;
        Prime = '$bottom'
    ),
    eterms_extend(Sg,Prime,Sv,Call,Succ).
%
etermsvar_call_to_success_builtin('is/2',(X is Y),Sv,Call,Proj,Succ):-
    ( (var(X);number(X)) -> % (it was precondition_builtin)
        TY = 'arithmetic:arithexpression',
        new_type_name(NY),
        insert_type_name(NY,[],0),
        varset(Y,Svy),
        eterms_project(not_provided_Sg,Svy,not_provided_HvFv_u,Proj,Projy0),
        etermsvar_exit_to_prime(p(Y),[Y1],p(Y1),Svy,[Y1:(NY,TY)],(no,Projy0),Primey),
        normalize_names(Projy0, Projy),
        etermsvar_glb(Projy,Primey,Primey2),
        ( Primey2 \== '$bottom' ->
            abs_eval_arith(Y,Primey2,Type),         
            TX = Type,
            new_type_name(NX),
            get_list_names_is(Projy,NameSelec),
            insert_type_name(NX,NameSelec,0),
            varset(X,Svx),
            eterms_project(not_provided_Sg,Svx,not_provided_HvFv_u,Proj,Projx),
            etermsvar_exit_to_prime(p(X),[X1],p(X1),Svx,[X1:(NX,TX)],(no,Projx),Primex),
            % ASM - take care of variables in first term
            etermsvar_glb_without_var(Projx,Primex,Primex2),
            (
                Primex2 \== '$bottom' ->
                append(Primex2,Primey2,Prime_u),
                sort(Prime_u,Prime),
                eterms_extend(not_provided_Sg,Prime,Sv,Call,Succ)
            ;
                Succ = '$bottom'
            )
        ;
            Succ = '$bottom'
        )
    ;
        Succ = '$bottom'
    ).
%        
etermsvar_call_to_success_builtin(key(Key,SubGoal),Sg,Sv,Call,Proj,Succ):-
    ( getvalues_comp_cond(Sv,Proj,Vals) ->
        ( generateSucc0_cond(Vals,SubGoal,Vals,Proj,Prime) ->
            true
        ; Prime = '$bottom'
        )
    ; apply_trusted0(Proj,Key,SubGoal,Sv,eterms,_,Prime)
    ),
    eterms_extend(Sg,Prime,Sv,Call,Succ).

normalize_names([], []).
normalize_names([(X, Type)|T], [(X, Type_)|T_] ):-
    get_canonical_name(Type, Type_),
    normalize_names(T, T_).
normalize_names([(X:(Type, Y))|T], [(X:(Type_, Y))|T_] ):-
    get_canonical_name(Type, Type_),
    normalize_names(T, T_).

%------------------------------------------------------------------------%

:- use_module(ciaopp(plai/domains), [asub_to_info/5]).

:- dom_impl(etermsvar, obtain_info/4).
:- export(etermsvar_obtain_info/4).
etermsvar_obtain_info(_Prop,Vars,ASub,Info) :- asub_to_info(etermsvar,ASub,Vars,Info,_CompProps).

%------------------------------------------------------------------------%
% User interface

:- dom_impl(etermsvar, input_user_interface/5).
:- export(etermsvar_input_user_interface/5).
:- pred etermsvar_input_user_interface(+InputUser,+Qv,-ASub,+Sg,+MaybeCallASub).

etermsvar_input_user_interface(InputUser,Qv,ASub,Sg,MaybeCallASub) :-
    eterms_input_user_interface(InputUser,Qv,ASub,Sg,MaybeCallASub).

:- dom_impl(etermsvar, input_interface/4).
:- export(etermsvar_input_interface/4).
etermsvar_input_interface(X,Mode,Acc,R):-
    etermsvar_input_interface_convert_free_and_ground(X,XConv),
    etermsvar_input_interface_(XConv,Mode,Acc,R).

etermsvar_input_interface_(vr(X),perfect,Acc,[vr(X)|Acc]) :- !.
etermsvar_input_interface_(gnd(X),perfect,Acc,[gnd(X)|Acc]) :- !.
etermsvar_input_interface_(XConv,Mode,Acc,R) :-
    eterms_input_interface(XConv,Mode,Acc,R).

% TODO: strange coding
% note: this must be reversible (see etermsvar_asub_to_native/5)
etermsvar_input_interface_convert_free_and_ground(X,X) :- var(X), !.
etermsvar_input_interface_convert_free_and_ground(free(X),vr(X)) :- !.
etermsvar_input_interface_convert_free_and_ground(ground(X),gnd(X)) :- !.
etermsvar_input_interface_convert_free_and_ground('term_typing:var',vr) :- !.
etermsvar_input_interface_convert_free_and_ground('term_typing:ground',gnd) :- !.
etermsvar_input_interface_convert_free_and_ground(FunctorA,FunctorB) :-
    nonvar(FunctorA), FunctorA =.. [F|ArgsA], ArgsA \== [], !,
    etermsvar_input_interface_convert_free_and_ground_args(ArgsA,ArgsB),
    FunctorB =.. [F|ArgsB].
etermsvar_input_interface_convert_free_and_ground(FunctorA,FunctorB) :-
    nonvar(FunctorB), FunctorB =.. [F|ArgsB], ArgsB \== [], !,
    etermsvar_input_interface_convert_free_and_ground_args(ArgsA,ArgsB),
    FunctorA =.. [F|ArgsA].
etermsvar_input_interface_convert_free_and_ground(X,X).

etermsvar_input_interface_convert_free_and_ground_args([],[]).
etermsvar_input_interface_convert_free_and_ground_args([A|R],[AConv|RConv]) :-
    etermsvar_input_interface_convert_free_and_ground(A,AConv),
    etermsvar_input_interface_convert_free_and_ground_args(R,RConv).

%------------------------------------------------------------------------%
:- dom_impl(etermsvar, asub_to_native/5).
:- export(etermsvar_asub_to_native/5).
:- pred eterms_asub_to_native(+ASub,+Qv,+OutFlag,-OutputUser,-Comps).

etermsvar_asub_to_native(ASub,Qv,OutFlag,OutputUser,[]):-
    eterms_asub_to_native(ASub,Qv,OutFlag,OutputUserPre,[]),
    etermsvar_input_interface_convert_free_and_ground_args(OutputUser,OutputUserPre).

%------------------------------------------------------------------------%
:- export(etermsvar_output_interface/2).
:- pred etermsvar_output_interface(+ASub,-Output).

etermsvar_output_interface(ASub,ASub).

%------------------------------------------------------------------------%

:- dom_impl(etermsvar, collect_auxinfo_asub/3).
:- export(etermsvar_collect_auxinfo_asub/3).
etermsvar_collect_auxinfo_asub(Abs,Types0,Types) :-
    eterms_collect_auxinfo_asub(Abs,Types0,Types).

:- dom_impl(etermsvar, rename_auxinfo_asub/3).
:- export(etermsvar_rename_auxinfo_asub/3).
etermsvar_rename_auxinfo_asub(Call,Dict,RenCall) :-
    eterms_rename_auxinfo_asub(Call,Dict,RenCall).

%------------------------------------------------------------------%

:- dom_impl(etermsvar, identical_abstract/2).
:- export(etermsvar_identical_abstract/2).
etermsvar_identical_abstract(ASub1,ASub2):-
    eterms_identical_abstract(ASub1,ASub2).

%-------------------- types operations

:- dom_impl(etermsvar, multi_part_conc/3).
:- export(etermsvar_multi_part_conc/3).
etermsvar_multi_part_conc(A,ASub,List) :- 
    eterms_multi_part_conc(A,ASub,List).

%-----------------------------------------------------------

:- dom_impl(etermsvar, part_conc/4).
:- export(etermsvar_part_conc/4).
etermsvar_part_conc(A,ASub,NA,NASub):- 
    eterms_part_conc(A,ASub,NA,NASub).

% ---------------------------------------------------------------------------

% (internal?)
:- use_module(domain(eterms), [
    get_type/3,
    replace_names/3,
    obtains_names/4,
    normal_asub/2,
    update_names/2,
    generate_subs_exit/4,
    generate_subs/3,
    variables_are_top_type/2,
    eterms_glbnames/3,
    getargtypes/6,
    %
    replacetype/4,
    getfunctors/2,
    getvalue/2,
    split_f/4,
    unifytoterm/1,
    abs_eval_arithcomp/4,
    abs_eval_arith/3,
    get_list_names_is/2,
    getvalues_comp_cond/3,
    generateSucc0_cond/5,
    %
    eterms_needs/1,
    eterms_project/5,
    eterms_abs_sort/2,
    eterms_extend/5,
    eterms_concrete/3,
    eterms_unknown_call/4,
    eterms_input_user_interface/5,
    eterms_input_interface/4,
    eterms_asub_to_native/5,
    eterms_success_builtin/6,
    %
    eterms_special_builtin/5,
    %
    eterms_collect_auxinfo_asub/3,
    eterms_rename_auxinfo_asub/3,
    eterms_identical_abstract/2,
    eterms_multi_part_conc/3,
    eterms_part_conc/4
]).
:- use_module(domain(termsd), [
    find_list_entry/3,
    get_var_types_by_unification/3]).