import datetime
import decimal
import uuid
import proto
from proto import proto
from proto import test
from unittest import TestCase


class TestSerializationJson(TestCase):

    def test_serialization_json_domain(self):
        # Define a source JSON string
        json = r'{"id":1,"name":"Test","state":6,"wallet":{"currency":"USD","amount":1000.0},"asset":{"currency":"EUR","amount":100.0},"orders":[{"id":1,"symbol":"EURUSD","side":0,"type":0,"price":1.23456,"volume":1000.0},{"id":2,"symbol":"EURUSD","side":1,"type":1,"price":1.0,"volume":100.0},{"id":3,"symbol":"EURUSD","side":0,"type":2,"price":1.5,"volume":10.0}]}'

        # Create a new account from the source JSON string
        account1 = proto.Account.from_json(json)

        # Serialize the account to the JSON string
        json = account1.to_json()

        # Check the serialized JSON size
        self.assertGreater(len(json), 0)

        # Deserialize the account from the JSON string
        account2 = proto.Account.from_json(json)

        self.assertEqual(account2.id, 1)
        self.assertEqual(account2.name, "Test")
        self.assertTrue(account2.state.has_flags(proto.State.good))
        self.assertEqual(account2.wallet.currency, "USD")
        self.assertEqual(account2.wallet.amount, 1000.0)
        self.assertNotEqual(account2.asset, None)
        self.assertEqual(account2.asset.currency, "EUR")
        self.assertEqual(account2.asset.amount, 100.0)
        self.assertEqual(len(account2.orders), 3)
        self.assertEqual(account2.orders[0].id, 1)
        self.assertEqual(account2.orders[0].symbol, "EURUSD")
        self.assertEqual(account2.orders[0].side, proto.OrderSide.buy)
        self.assertEqual(account2.orders[0].type, proto.OrderType.market)
        self.assertEqual(account2.orders[0].price, 1.23456)
        self.assertEqual(account2.orders[0].volume, 1000.0)
        self.assertEqual(account2.orders[1].id, 2)
        self.assertEqual(account2.orders[1].symbol, "EURUSD")
        self.assertEqual(account2.orders[1].side, proto.OrderSide.sell)
        self.assertEqual(account2.orders[1].type, proto.OrderType.limit)
        self.assertEqual(account2.orders[1].price, 1.0)
        self.assertEqual(account2.orders[1].volume, 100.0)
        self.assertEqual(account2.orders[2].id, 3)
        self.assertEqual(account2.orders[2].symbol, "EURUSD")
        self.assertEqual(account2.orders[2].side, proto.OrderSide.buy)
        self.assertEqual(account2.orders[2].type, proto.OrderType.stop)
        self.assertEqual(account2.orders[2].price, 1.5)
        self.assertEqual(account2.orders[2].volume, 10.0)

    def test_serialization_json_struct_simple(self):
        # Define a source JSON string
        json = r'{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145597933463000,"f36":"00000000-0000-0000-0000-000000000000","f37":"e7854072-f0a5-11e8-8f69-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}'

        # Create a new struct from the source JSON string
        struct1 = test.StructSimple.from_json(json)

        # Serialize the struct to the JSON string
        json = struct1.to_json()

        # Check the serialized JSON size
        self.assertGreater(len(json), 0)

        # Deserialize the struct from the JSON string
        struct2 = test.StructSimple.from_json(json)

        self.assertEqual(struct2.f1, False)
        self.assertEqual(struct2.f2, True)
        self.assertEqual(struct2.f3, 0)
        self.assertEqual(struct2.f4, 0xFF)
        self.assertEqual(struct2.f5, '\0')
        self.assertEqual(struct2.f6, '!')
        self.assertEqual(struct2.f7, chr(0))
        self.assertEqual(struct2.f8, chr(0x0444))
        self.assertEqual(struct2.f9, 0)
        self.assertEqual(struct2.f10, 127)
        self.assertEqual(struct2.f11, 0)
        self.assertEqual(struct2.f12, 0xFF)
        self.assertEqual(struct2.f13, 0)
        self.assertEqual(struct2.f14, 32767)
        self.assertEqual(struct2.f15, 0)
        self.assertEqual(struct2.f16, 0xFFFF)
        self.assertEqual(struct2.f17, 0)
        self.assertEqual(struct2.f18, 2147483647)
        self.assertEqual(struct2.f19, 0)
        self.assertEqual(struct2.f20, 0xFFFFFFFF)
        self.assertEqual(struct2.f21, 0)
        self.assertEqual(struct2.f22, 9223372036854775807)
        self.assertEqual(struct2.f23, 0)
        self.assertEqual(struct2.f24, 0xFFFFFFFFFFFFFFFF)
        self.assertEqual(struct2.f25, 0.0)
        self.assertLess(abs(struct2.f26 - 123.456), 0.0001)
        self.assertEqual(struct2.f27, 0.0)
        self.assertLess(abs(struct2.f28 - -123.567e+123), 1e+123)
        self.assertEqual(struct2.f29, decimal.Decimal("0"))
        self.assertEqual(struct2.f30, decimal.Decimal("123456.123456"))
        self.assertEqual(struct2.f31, "")
        self.assertEqual(struct2.f32, "Initial string!")
        self.assertEqual(datetime.datetime.utcfromtimestamp(struct2.f33 / 1000000000), datetime.datetime(1970, 1, 1))
        self.assertEqual(datetime.datetime.utcfromtimestamp(struct2.f34 / 1000000000), datetime.datetime(1970, 1, 1))
        self.assertGreater(datetime.datetime.utcfromtimestamp(struct2.f35 / 1000000000), datetime.datetime(2018, 1, 1))
        self.assertEqual(struct2.f36, uuid.UUID(int=0))
        self.assertNotEqual(struct2.f37, uuid.UUID(int=0))
        self.assertEqual(struct2.f38, uuid.UUID("123e4567-e89b-12d3-a456-426655440000"))

        self.assertEqual(struct2.f1, struct1.f1)
        self.assertEqual(struct2.f2, struct1.f2)
        self.assertEqual(struct2.f3, struct1.f3)
        self.assertEqual(struct2.f4, struct1.f4)
        self.assertEqual(struct2.f5, struct1.f5)
        self.assertEqual(struct2.f6, struct1.f6)
        self.assertEqual(struct2.f7, struct1.f7)
        self.assertEqual(struct2.f8, struct1.f8)
        self.assertEqual(struct2.f9, struct1.f9)
        self.assertEqual(struct2.f10, struct1.f10)
        self.assertEqual(struct2.f11, struct1.f11)
        self.assertEqual(struct2.f12, struct1.f12)
        self.assertEqual(struct2.f13, struct1.f13)
        self.assertEqual(struct2.f14, struct1.f14)
        self.assertEqual(struct2.f15, struct1.f15)
        self.assertEqual(struct2.f16, struct1.f16)
        self.assertEqual(struct2.f17, struct1.f17)
        self.assertEqual(struct2.f18, struct1.f18)
        self.assertEqual(struct2.f19, struct1.f19)
        self.assertEqual(struct2.f20, struct1.f20)
        self.assertEqual(struct2.f21, struct1.f21)
        self.assertEqual(struct2.f22, struct1.f22)
        self.assertEqual(struct2.f23, struct1.f23)
        self.assertEqual(struct2.f24, struct1.f24)
        self.assertEqual(struct2.f25, struct1.f25)
        self.assertLess(abs(struct2.f26 - struct1.f26), 0.0001)
        self.assertEqual(struct2.f27, struct1.f27)
        self.assertLess(abs(struct2.f28 - struct1.f28), 1e+123)
        self.assertEqual(struct2.f29, struct1.f29)
        self.assertEqual(struct2.f30, struct1.f30)
        self.assertEqual(struct2.f31, struct1.f31)
        self.assertEqual(struct2.f32, struct1.f32)
        self.assertEqual(struct2.f33, struct1.f33)
        self.assertEqual(struct2.f34, struct1.f34)
        self.assertEqual(struct2.f35, struct1.f35)
        self.assertEqual(struct2.f36, struct1.f36)
        self.assertEqual(struct2.f37, struct1.f37)
        self.assertEqual(struct2.f38, struct1.f38)
        self.assertEqual(struct2.f39, struct1.f39)
        self.assertEqual(struct2.f40, struct1.f40)

    def test_serialization_json_struct_optional(self):
        # Define a source JSON string
        json = r'{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145860677797000,"f36":"00000000-0000-0000-0000-000000000000","f37":"8420d1c6-f0a6-11e8-80fc-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]},"f100":null,"f101":true,"f102":null,"f103":null,"f104":255,"f105":null,"f106":null,"f107":33,"f108":null,"f109":null,"f110":1092,"f111":null,"f112":null,"f113":127,"f114":null,"f115":null,"f116":255,"f117":null,"f118":null,"f119":32767,"f120":null,"f121":null,"f122":65535,"f123":null,"f124":null,"f125":2147483647,"f126":null,"f127":null,"f128":4294967295,"f129":null,"f130":null,"f131":9223372036854775807,"f132":null,"f133":null,"f134":18446744073709551615,"f135":null,"f136":null,"f137":123.456,"f138":null,"f139":null,"f140":-1.23456e+125,"f141":null,"f142":null,"f143":"123456.123456","f144":null,"f145":null,"f146":"Initial string!","f147":null,"f148":null,"f149":1543145860678429000,"f150":null,"f151":null,"f152":"123e4567-e89b-12d3-a456-426655440000","f153":null,"f154":null,"f155":null,"f156":null,"f157":null,"f158":null,"f159":null,"f160":null,"f161":null,"f162":null,"f163":null,"f164":null,"f165":null}'

        # Create a new struct from the source JSON string
        struct1 = test.StructOptional.from_json(json)

        # Serialize the struct to the JSON string
        json = struct1.to_json()

        # Check the serialized JSON size
        self.assertGreater(len(json), 0)

        # Deserialize the struct from the JSON string
        struct2 = test.StructOptional.from_json(json)

        self.assertEqual(struct2.f1, False)
        self.assertEqual(struct2.f2, True)
        self.assertEqual(struct2.f3, 0)
        self.assertEqual(struct2.f4, 0xFF)
        self.assertEqual(struct2.f5, '\0')
        self.assertEqual(struct2.f6, '!')
        self.assertEqual(struct2.f7, chr(0))
        self.assertEqual(struct2.f8, chr(0x0444))
        self.assertEqual(struct2.f9, 0)
        self.assertEqual(struct2.f10, 127)
        self.assertEqual(struct2.f11, 0)
        self.assertEqual(struct2.f12, 0xFF)
        self.assertEqual(struct2.f13, 0)
        self.assertEqual(struct2.f14, 32767)
        self.assertEqual(struct2.f15, 0)
        self.assertEqual(struct2.f16, 0xFFFF)
        self.assertEqual(struct2.f17, 0)
        self.assertEqual(struct2.f18, 2147483647)
        self.assertEqual(struct2.f19, 0)
        self.assertEqual(struct2.f20, 0xFFFFFFFF)
        self.assertEqual(struct2.f21, 0)
        self.assertEqual(struct2.f22, 9223372036854775807)
        self.assertEqual(struct2.f23, 0)
        self.assertEqual(struct2.f24, 0xFFFFFFFFFFFFFFFF)
        self.assertEqual(struct2.f25, 0.0)
        self.assertLess(abs(struct2.f26 - 123.456), 0.0001)
        self.assertEqual(struct2.f27, 0.0)
        self.assertLess(abs(struct2.f28 - -123.567e+123), 1e+123)
        self.assertEqual(struct2.f29, decimal.Decimal("0"))
        self.assertEqual(struct2.f30, decimal.Decimal("123456.123456"))
        self.assertEqual(struct2.f31, "")
        self.assertEqual(struct2.f32, "Initial string!")
        self.assertEqual(datetime.datetime.utcfromtimestamp(struct2.f33 / 1000000000), datetime.datetime(1970, 1, 1))
        self.assertEqual(datetime.datetime.utcfromtimestamp(struct2.f34 / 1000000000), datetime.datetime(1970, 1, 1))
        self.assertGreater(datetime.datetime.utcfromtimestamp(struct2.f35 / 1000000000), datetime.datetime(2018, 1, 1))
        self.assertEqual(struct2.f36, uuid.UUID(int=0))
        self.assertNotEqual(struct2.f37, uuid.UUID(int=0))
        self.assertEqual(struct2.f38, uuid.UUID("123e4567-e89b-12d3-a456-426655440000"))

        self.assertEqual(struct2.f100, None)
        self.assertNotEqual(struct2.f101, None)
        self.assertEqual(struct2.f101, True)
        self.assertEqual(struct2.f102, None)
        self.assertEqual(struct2.f103, None)
        self.assertNotEqual(struct2.f104, None)
        self.assertEqual(struct2.f104, 0xFF)
        self.assertEqual(struct2.f105, None)
        self.assertEqual(struct2.f106, None)
        self.assertNotEqual(struct2.f107, None)
        self.assertEqual(struct2.f107, '!')
        self.assertEqual(struct2.f108, None)
        self.assertEqual(struct2.f109, None)
        self.assertNotEqual(struct2.f110, None)
        self.assertEqual(struct2.f110, chr(0x0444))
        self.assertEqual(struct2.f111, None)
        self.assertEqual(struct2.f112, None)
        self.assertNotEqual(struct2.f113, None)
        self.assertEqual(struct2.f113, 127)
        self.assertEqual(struct2.f114, None)
        self.assertEqual(struct2.f115, None)
        self.assertNotEqual(struct2.f116, None)
        self.assertEqual(struct2.f116, 0xFF)
        self.assertEqual(struct2.f117, None)
        self.assertEqual(struct2.f118, None)
        self.assertNotEqual(struct2.f119, None)
        self.assertEqual(struct2.f119, 32767)
        self.assertEqual(struct2.f120, None)
        self.assertEqual(struct2.f121, None)
        self.assertNotEqual(struct2.f122, None)
        self.assertEqual(struct2.f122, 0xFFFF)
        self.assertEqual(struct2.f123, None)
        self.assertEqual(struct2.f124, None)
        self.assertNotEqual(struct2.f125, None)
        self.assertEqual(struct2.f125, 2147483647)
        self.assertEqual(struct2.f126, None)
        self.assertEqual(struct2.f127, None)
        self.assertNotEqual(struct2.f128, None)
        self.assertEqual(struct2.f128, 0xFFFFFFFF)
        self.assertEqual(struct2.f129, None)
        self.assertEqual(struct2.f130, None)
        self.assertNotEqual(struct2.f131, None)
        self.assertEqual(struct2.f131, 9223372036854775807)
        self.assertEqual(struct2.f132, None)
        self.assertEqual(struct2.f133, None)
        self.assertNotEqual(struct2.f131, None)
        self.assertEqual(struct2.f134, 0xFFFFFFFFFFFFFFFF)
        self.assertEqual(struct2.f135, None)
        self.assertEqual(struct2.f136, None)
        self.assertNotEqual(struct2.f137, None)
        self.assertLess(abs(struct2.f137 - 123.456), 0.0001)
        self.assertEqual(struct2.f138, None)
        self.assertEqual(struct2.f139, None)
        self.assertNotEqual(struct2.f140, None)
        self.assertLess(abs(struct2.f140 - -123.567e+123), 1e+123)
        self.assertEqual(struct2.f141, None)
        self.assertEqual(struct2.f142, None)
        self.assertNotEqual(struct2.f143, None)
        self.assertEqual(struct2.f143, decimal.Decimal("123456.123456"))
        self.assertEqual(struct2.f144, None)
        self.assertEqual(struct2.f145, None)
        self.assertNotEqual(struct2.f146, None)
        self.assertEqual(struct2.f146, "Initial string!")
        self.assertEqual(struct2.f147, None)
        self.assertEqual(struct2.f148, None)
        self.assertNotEqual(struct2.f149, None)
        self.assertGreater(datetime.datetime.utcfromtimestamp(struct2.f149 / 1000000000), datetime.datetime(2018, 1, 1))
        self.assertEqual(struct2.f150, None)
        self.assertEqual(struct2.f151, None)
        self.assertNotEqual(struct2.f152, None)
        self.assertEqual(struct2.f152, uuid.UUID("123e4567-e89b-12d3-a456-426655440000"))
        self.assertEqual(struct2.f153, None)
        self.assertEqual(struct2.f154, None)
        self.assertEqual(struct2.f155, None)
        self.assertEqual(struct2.f156, None)
        self.assertEqual(struct2.f157, None)
        self.assertEqual(struct2.f158, None)
        self.assertEqual(struct2.f159, None)
        self.assertEqual(struct2.f160, None)
        self.assertEqual(struct2.f161, None)
        self.assertEqual(struct2.f162, None)
        self.assertEqual(struct2.f163, None)
        self.assertEqual(struct2.f164, None)
        self.assertEqual(struct2.f165, None)

        self.assertEqual(struct2.f1, struct1.f1)
        self.assertEqual(struct2.f2, struct1.f2)
        self.assertEqual(struct2.f3, struct1.f3)
        self.assertEqual(struct2.f4, struct1.f4)
        self.assertEqual(struct2.f5, struct1.f5)
        self.assertEqual(struct2.f6, struct1.f6)
        self.assertEqual(struct2.f7, struct1.f7)
        self.assertEqual(struct2.f8, struct1.f8)
        self.assertEqual(struct2.f9, struct1.f9)
        self.assertEqual(struct2.f10, struct1.f10)
        self.assertEqual(struct2.f11, struct1.f11)
        self.assertEqual(struct2.f12, struct1.f12)
        self.assertEqual(struct2.f13, struct1.f13)
        self.assertEqual(struct2.f14, struct1.f14)
        self.assertEqual(struct2.f15, struct1.f15)
        self.assertEqual(struct2.f16, struct1.f16)
        self.assertEqual(struct2.f17, struct1.f17)
        self.assertEqual(struct2.f18, struct1.f18)
        self.assertEqual(struct2.f19, struct1.f19)
        self.assertEqual(struct2.f20, struct1.f20)
        self.assertEqual(struct2.f21, struct1.f21)
        self.assertEqual(struct2.f22, struct1.f22)
        self.assertEqual(struct2.f23, struct1.f23)
        self.assertEqual(struct2.f24, struct1.f24)
        self.assertEqual(struct2.f25, struct1.f25)
        self.assertLess(abs(struct2.f26 - struct1.f26), 0.0001)
        self.assertEqual(struct2.f27, struct1.f27)
        self.assertLess(abs(struct2.f28 - struct1.f28), 1e+123)
        self.assertEqual(struct2.f29, struct1.f29)
        self.assertEqual(struct2.f30, struct1.f30)
        self.assertEqual(struct2.f31, struct1.f31)
        self.assertEqual(struct2.f32, struct1.f32)
        self.assertEqual(struct2.f33, struct1.f33)
        self.assertEqual(struct2.f34, struct1.f34)
        self.assertEqual(struct2.f35, struct1.f35)
        self.assertEqual(struct2.f36, struct1.f36)
        self.assertEqual(struct2.f37, struct1.f37)

        self.assertEqual(struct2.f100, struct1.f100)
        self.assertEqual(struct2.f101, struct1.f101)
        self.assertEqual(struct2.f102, struct1.f102)
        self.assertEqual(struct2.f103, struct1.f103)
        self.assertEqual(struct2.f104, struct1.f104)
        self.assertEqual(struct2.f105, struct1.f105)
        self.assertEqual(struct2.f106, struct1.f106)
        self.assertEqual(struct2.f107, struct1.f107)
        self.assertEqual(struct2.f108, struct1.f108)
        self.assertEqual(struct2.f109, struct1.f109)
        self.assertEqual(struct2.f110, struct1.f110)
        self.assertEqual(struct2.f111, struct1.f111)
        self.assertEqual(struct2.f112, struct1.f112)
        self.assertEqual(struct2.f113, struct1.f113)
        self.assertEqual(struct2.f114, struct1.f114)
        self.assertEqual(struct2.f115, struct1.f115)
        self.assertEqual(struct2.f116, struct1.f116)
        self.assertEqual(struct2.f117, struct1.f117)
        self.assertEqual(struct2.f118, struct1.f118)
        self.assertEqual(struct2.f119, struct1.f119)
        self.assertEqual(struct2.f120, struct1.f120)
        self.assertEqual(struct2.f121, struct1.f121)
        self.assertEqual(struct2.f122, struct1.f122)
        self.assertEqual(struct2.f123, struct1.f123)
        self.assertEqual(struct2.f124, struct1.f124)
        self.assertEqual(struct2.f125, struct1.f125)
        self.assertEqual(struct2.f126, struct1.f126)
        self.assertEqual(struct2.f127, struct1.f127)
        self.assertEqual(struct2.f128, struct1.f128)
        self.assertEqual(struct2.f129, struct1.f129)
        self.assertEqual(struct2.f130, struct1.f130)
        self.assertEqual(struct2.f131, struct1.f131)
        self.assertEqual(struct2.f132, struct1.f132)
        self.assertEqual(struct2.f133, struct1.f133)
        self.assertEqual(struct2.f134, struct1.f134)
        self.assertEqual(struct2.f135, struct1.f135)
        self.assertEqual(struct2.f136, struct1.f136)
        self.assertLess(abs(struct2.f137 - struct1.f137), 0.0001)
        self.assertEqual(struct2.f138, struct1.f138)
        self.assertEqual(struct2.f139, struct1.f139)
        self.assertLess(abs(struct2.f140 - struct1.f140), 1e+123)
        self.assertEqual(struct2.f141, struct1.f141)
        self.assertEqual(struct2.f142, struct1.f142)
        self.assertEqual(struct2.f143, struct1.f143)
        self.assertEqual(struct2.f144, struct1.f144)
        self.assertEqual(struct2.f145, struct1.f145)
        self.assertEqual(struct2.f146, struct1.f146)
        self.assertEqual(struct2.f147, struct1.f147)
        self.assertEqual(struct2.f148, struct1.f148)
        self.assertEqual(struct2.f149, struct1.f149)
        self.assertEqual(struct2.f150, struct1.f150)
        self.assertEqual(struct2.f151, struct1.f151)
        self.assertEqual(struct2.f152, struct1.f152)
        self.assertEqual(struct2.f153, struct1.f153)
        self.assertEqual(struct2.f154, struct1.f154)
        self.assertEqual(struct2.f155, struct1.f155)
        self.assertEqual(struct2.f156, struct1.f156)
        self.assertEqual(struct2.f157, struct1.f157)

    def test_serialization_json_struct_nested(self):
        # Define a source JSON string
        json = r'{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145901646321000,"f36":"00000000-0000-0000-0000-000000000000","f37":"9c8c268e-f0a6-11e8-a777-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]},"f100":null,"f101":true,"f102":null,"f103":null,"f104":255,"f105":null,"f106":null,"f107":33,"f108":null,"f109":null,"f110":1092,"f111":null,"f112":null,"f113":127,"f114":null,"f115":null,"f116":255,"f117":null,"f118":null,"f119":32767,"f120":null,"f121":null,"f122":65535,"f123":null,"f124":null,"f125":2147483647,"f126":null,"f127":null,"f128":4294967295,"f129":null,"f130":null,"f131":9223372036854775807,"f132":null,"f133":null,"f134":18446744073709551615,"f135":null,"f136":null,"f137":123.456,"f138":null,"f139":null,"f140":-1.23456e+125,"f141":null,"f142":null,"f143":"123456.123456","f144":null,"f145":null,"f146":"Initial string!","f147":null,"f148":null,"f149":1543145901647155000,"f150":null,"f151":null,"f152":"123e4567-e89b-12d3-a456-426655440000","f153":null,"f154":null,"f155":null,"f156":null,"f157":null,"f158":null,"f159":null,"f160":null,"f161":null,"f162":null,"f163":null,"f164":null,"f165":null,"f1000":0,"f1001":null,"f1002":50,"f1003":null,"f1004":0,"f1005":null,"f1006":42,"f1007":null,"f1008":{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145901647367000,"f36":"00000000-0000-0000-0000-000000000000","f37":"9c8c54c4-f0a6-11e8-a777-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"f1009":null,"f1010":{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145901648310000,"f36":"00000000-0000-0000-0000-000000000000","f37":"9c8c6b76-f0a6-11e8-a777-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]},"f100":null,"f101":true,"f102":null,"f103":null,"f104":255,"f105":null,"f106":null,"f107":33,"f108":null,"f109":null,"f110":1092,"f111":null,"f112":null,"f113":127,"f114":null,"f115":null,"f116":255,"f117":null,"f118":null,"f119":32767,"f120":null,"f121":null,"f122":65535,"f123":null,"f124":null,"f125":2147483647,"f126":null,"f127":null,"f128":4294967295,"f129":null,"f130":null,"f131":9223372036854775807,"f132":null,"f133":null,"f134":18446744073709551615,"f135":null,"f136":null,"f137":123.456,"f138":null,"f139":null,"f140":-1.23456e+125,"f141":null,"f142":null,"f143":"123456.123456","f144":null,"f145":null,"f146":"Initial string!","f147":null,"f148":null,"f149":1543145901648871000,"f150":null,"f151":null,"f152":"123e4567-e89b-12d3-a456-426655440000","f153":null,"f154":null,"f155":null,"f156":null,"f157":null,"f158":null,"f159":null,"f160":null,"f161":null,"f162":null,"f163":null,"f164":null,"f165":null},"f1011":null}'

        # Create a new struct from the source JSON string
        struct1 = test.StructNested.from_json(json)

        # Serialize the struct to the JSON string
        json = struct1.to_json()

        # Check the serialized JSON size
        self.assertGreater(len(json), 0)

        # Deserialize the struct from the JSON string
        struct2 = test.StructNested.from_json(json)

        self.assertEqual(struct2.f1, False)
        self.assertEqual(struct2.f2, True)
        self.assertEqual(struct2.f3, 0)
        self.assertEqual(struct2.f4, 0xFF)
        self.assertEqual(struct2.f5, '\0')
        self.assertEqual(struct2.f6, '!')
        self.assertEqual(struct2.f7, chr(0))
        self.assertEqual(struct2.f8, chr(0x0444))
        self.assertEqual(struct2.f9, 0)
        self.assertEqual(struct2.f10, 127)
        self.assertEqual(struct2.f11, 0)
        self.assertEqual(struct2.f12, 0xFF)
        self.assertEqual(struct2.f13, 0)
        self.assertEqual(struct2.f14, 32767)
        self.assertEqual(struct2.f15, 0)
        self.assertEqual(struct2.f16, 0xFFFF)
        self.assertEqual(struct2.f17, 0)
        self.assertEqual(struct2.f18, 2147483647)
        self.assertEqual(struct2.f19, 0)
        self.assertEqual(struct2.f20, 0xFFFFFFFF)
        self.assertEqual(struct2.f21, 0)
        self.assertEqual(struct2.f22, 9223372036854775807)
        self.assertEqual(struct2.f23, 0)
        self.assertEqual(struct2.f24, 0xFFFFFFFFFFFFFFFF)
        self.assertEqual(struct2.f25, 0.0)
        self.assertLess(abs(struct2.f26 - 123.456), 0.0001)
        self.assertEqual(struct2.f27, 0.0)
        self.assertLess(abs(struct2.f28 - -123.567e+123), 1e+123)
        self.assertEqual(struct2.f29, decimal.Decimal("0"))
        self.assertEqual(struct2.f30, decimal.Decimal("123456.123456"))
        self.assertEqual(struct2.f31, "")
        self.assertEqual(struct2.f32, "Initial string!")
        self.assertEqual(datetime.datetime.utcfromtimestamp(struct2.f33 / 1000000000), datetime.datetime(1970, 1, 1))
        self.assertEqual(datetime.datetime.utcfromtimestamp(struct2.f34 / 1000000000), datetime.datetime(1970, 1, 1))
        self.assertGreater(datetime.datetime.utcfromtimestamp(struct2.f35 / 1000000000), datetime.datetime(2018, 1, 1))
        self.assertEqual(struct2.f36, uuid.UUID(int=0))
        self.assertNotEqual(struct2.f37, uuid.UUID(int=0))
        self.assertEqual(struct2.f38, uuid.UUID("123e4567-e89b-12d3-a456-426655440000"))

        self.assertEqual(struct2.f100, None)
        self.assertNotEqual(struct2.f101, None)
        self.assertEqual(struct2.f101, True)
        self.assertEqual(struct2.f102, None)
        self.assertEqual(struct2.f103, None)
        self.assertNotEqual(struct2.f104, None)
        self.assertEqual(struct2.f104, 0xFF)
        self.assertEqual(struct2.f105, None)
        self.assertEqual(struct2.f106, None)
        self.assertNotEqual(struct2.f107, None)
        self.assertEqual(struct2.f107, '!')
        self.assertEqual(struct2.f108, None)
        self.assertEqual(struct2.f109, None)
        self.assertNotEqual(struct2.f110, None)
        self.assertEqual(struct2.f110, chr(0x0444))
        self.assertEqual(struct2.f111, None)
        self.assertEqual(struct2.f112, None)
        self.assertNotEqual(struct2.f113, None)
        self.assertEqual(struct2.f113, 127)
        self.assertEqual(struct2.f114, None)
        self.assertEqual(struct2.f115, None)
        self.assertNotEqual(struct2.f116, None)
        self.assertEqual(struct2.f116, 0xFF)
        self.assertEqual(struct2.f117, None)
        self.assertEqual(struct2.f118, None)
        self.assertNotEqual(struct2.f119, None)
        self.assertEqual(struct2.f119, 32767)
        self.assertEqual(struct2.f120, None)
        self.assertEqual(struct2.f121, None)
        self.assertNotEqual(struct2.f122, None)
        self.assertEqual(struct2.f122, 0xFFFF)
        self.assertEqual(struct2.f123, None)
        self.assertEqual(struct2.f124, None)
        self.assertNotEqual(struct2.f125, None)
        self.assertEqual(struct2.f125, 2147483647)
        self.assertEqual(struct2.f126, None)
        self.assertEqual(struct2.f127, None)
        self.assertNotEqual(struct2.f128, None)
        self.assertEqual(struct2.f128, 0xFFFFFFFF)
        self.assertEqual(struct2.f129, None)
        self.assertEqual(struct2.f130, None)
        self.assertNotEqual(struct2.f131, None)
        self.assertEqual(struct2.f131, 9223372036854775807)
        self.assertEqual(struct2.f132, None)
        self.assertEqual(struct2.f133, None)
        self.assertNotEqual(struct2.f131, None)
        self.assertEqual(struct2.f134, 0xFFFFFFFFFFFFFFFF)
        self.assertEqual(struct2.f135, None)
        self.assertEqual(struct2.f136, None)
        self.assertNotEqual(struct2.f137, None)
        self.assertLess(abs(struct2.f137 - 123.456), 0.0001)
        self.assertEqual(struct2.f138, None)
        self.assertEqual(struct2.f139, None)
        self.assertNotEqual(struct2.f140, None)
        self.assertLess(abs(struct2.f140 - -123.567e+123), 1e+123)
        self.assertEqual(struct2.f141, None)
        self.assertEqual(struct2.f142, None)
        self.assertNotEqual(struct2.f143, None)
        self.assertEqual(struct2.f143, decimal.Decimal("123456.123456"))
        self.assertEqual(struct2.f144, None)
        self.assertEqual(struct2.f145, None)
        self.assertNotEqual(struct2.f146, None)
        self.assertEqual(struct2.f146, "Initial string!")
        self.assertEqual(struct2.f147, None)
        self.assertEqual(struct2.f148, None)
        self.assertNotEqual(struct2.f149, None)
        self.assertGreater(datetime.datetime.utcfromtimestamp(struct2.f149 / 1000000000), datetime.datetime(2018, 1, 1))
        self.assertEqual(struct2.f150, None)
        self.assertEqual(struct2.f151, None)
        self.assertNotEqual(struct2.f152, None)
        self.assertEqual(struct2.f152, uuid.UUID("123e4567-e89b-12d3-a456-426655440000"))
        self.assertEqual(struct2.f153, None)
        self.assertEqual(struct2.f154, None)
        self.assertEqual(struct2.f155, None)
        self.assertEqual(struct2.f156, None)
        self.assertEqual(struct2.f157, None)
        self.assertEqual(struct2.f158, None)
        self.assertEqual(struct2.f159, None)
        self.assertEqual(struct2.f160, None)
        self.assertEqual(struct2.f161, None)
        self.assertEqual(struct2.f162, None)
        self.assertEqual(struct2.f163, None)
        self.assertEqual(struct2.f164, None)
        self.assertEqual(struct2.f165, None)

        self.assertEqual(struct2.f1000, test.EnumSimple.ENUM_VALUE_0)
        self.assertEqual(struct2.f1001, None)
        self.assertEqual(struct2.f1002, test.EnumTyped.ENUM_VALUE_2)
        self.assertEqual(struct2.f1003, None)
        self.assertEqual(struct2.f1004, test.FlagsSimple.FLAG_VALUE_0)
        self.assertEqual(struct2.f1005, None)
        self.assertEqual(struct2.f1006, test.FlagsTyped.FLAG_VALUE_2 | test.FlagsTyped.FLAG_VALUE_4 | test.FlagsTyped.FLAG_VALUE_6)
        self.assertEqual(struct2.f1007, None)
        self.assertEqual(struct2.f1009, None)
        self.assertEqual(struct2.f1011, None)

        self.assertEqual(struct2.f1, struct1.f1)
        self.assertEqual(struct2.f2, struct1.f2)
        self.assertEqual(struct2.f3, struct1.f3)
        self.assertEqual(struct2.f4, struct1.f4)
        self.assertEqual(struct2.f5, struct1.f5)
        self.assertEqual(struct2.f6, struct1.f6)
        self.assertEqual(struct2.f7, struct1.f7)
        self.assertEqual(struct2.f8, struct1.f8)
        self.assertEqual(struct2.f9, struct1.f9)
        self.assertEqual(struct2.f10, struct1.f10)
        self.assertEqual(struct2.f11, struct1.f11)
        self.assertEqual(struct2.f12, struct1.f12)
        self.assertEqual(struct2.f13, struct1.f13)
        self.assertEqual(struct2.f14, struct1.f14)
        self.assertEqual(struct2.f15, struct1.f15)
        self.assertEqual(struct2.f16, struct1.f16)
        self.assertEqual(struct2.f17, struct1.f17)
        self.assertEqual(struct2.f18, struct1.f18)
        self.assertEqual(struct2.f19, struct1.f19)
        self.assertEqual(struct2.f20, struct1.f20)
        self.assertEqual(struct2.f21, struct1.f21)
        self.assertEqual(struct2.f22, struct1.f22)
        self.assertEqual(struct2.f23, struct1.f23)
        self.assertEqual(struct2.f24, struct1.f24)
        self.assertEqual(struct2.f25, struct1.f25)
        self.assertLess(abs(struct2.f26 - struct1.f26), 0.0001)
        self.assertEqual(struct2.f27, struct1.f27)
        self.assertLess(abs(struct2.f28 - struct2.f28), 1e+123)
        self.assertEqual(struct2.f29, struct1.f29)
        self.assertEqual(struct2.f30, struct1.f30)
        self.assertEqual(struct2.f31, struct1.f31)
        self.assertEqual(struct2.f32, struct1.f32)
        self.assertEqual(struct2.f33, struct1.f33)
        self.assertEqual(struct2.f34, struct1.f34)
        self.assertEqual(struct2.f35, struct1.f35)
        self.assertEqual(struct2.f36, struct1.f36)
        self.assertEqual(struct2.f37, struct1.f37)
        self.assertEqual(struct2.f38, struct1.f38)
        self.assertEqual(struct2.f39, struct1.f39)
        self.assertEqual(struct2.f40, struct1.f40)

        self.assertEqual(struct2.f100, struct1.f100)
        self.assertEqual(struct2.f101, struct1.f101)
        self.assertEqual(struct2.f102, struct1.f102)
        self.assertEqual(struct2.f103, struct1.f103)
        self.assertEqual(struct2.f104, struct1.f104)
        self.assertEqual(struct2.f105, struct1.f105)
        self.assertEqual(struct2.f106, struct1.f106)
        self.assertEqual(struct2.f107, struct1.f107)
        self.assertEqual(struct2.f108, struct1.f108)
        self.assertEqual(struct2.f109, struct1.f109)
        self.assertEqual(struct2.f110, struct1.f110)
        self.assertEqual(struct2.f111, struct1.f111)
        self.assertEqual(struct2.f112, struct1.f112)
        self.assertEqual(struct2.f113, struct1.f113)
        self.assertEqual(struct2.f114, struct1.f114)
        self.assertEqual(struct2.f115, struct1.f115)
        self.assertEqual(struct2.f116, struct1.f116)
        self.assertEqual(struct2.f117, struct1.f117)
        self.assertEqual(struct2.f118, struct1.f118)
        self.assertEqual(struct2.f119, struct1.f119)
        self.assertEqual(struct2.f120, struct1.f120)
        self.assertEqual(struct2.f121, struct1.f121)
        self.assertEqual(struct2.f122, struct1.f122)
        self.assertEqual(struct2.f123, struct1.f123)
        self.assertEqual(struct2.f124, struct1.f124)
        self.assertEqual(struct2.f125, struct1.f125)
        self.assertEqual(struct2.f126, struct1.f126)
        self.assertEqual(struct2.f127, struct1.f127)
        self.assertEqual(struct2.f128, struct1.f128)
        self.assertEqual(struct2.f129, struct1.f129)
        self.assertEqual(struct2.f130, struct1.f130)
        self.assertEqual(struct2.f131, struct1.f131)
        self.assertEqual(struct2.f132, struct1.f132)
        self.assertEqual(struct2.f133, struct1.f133)
        self.assertEqual(struct2.f134, struct1.f134)
        self.assertEqual(struct2.f135, struct1.f135)
        self.assertEqual(struct2.f136, struct1.f136)
        self.assertLess(abs(struct2.f137 - struct1.f137), 0.0001)
        self.assertEqual(struct2.f138, struct1.f138)
        self.assertEqual(struct2.f139, struct1.f139)
        self.assertLess(abs(struct2.f140 - struct1.f140), 1e+123)
        self.assertEqual(struct2.f141, struct1.f141)
        self.assertEqual(struct2.f142, struct1.f142)
        self.assertEqual(struct2.f143, struct1.f143)
        self.assertEqual(struct2.f144, struct1.f144)
        self.assertEqual(struct2.f145, struct1.f145)
        self.assertEqual(struct2.f146, struct1.f146)
        self.assertEqual(struct2.f147, struct1.f147)
        self.assertEqual(struct2.f148, struct1.f148)
        self.assertEqual(struct2.f149, struct1.f149)
        self.assertEqual(struct2.f150, struct1.f150)
        self.assertEqual(struct2.f151, struct1.f151)
        self.assertEqual(struct2.f152, struct1.f152)
        self.assertEqual(struct2.f153, struct1.f153)
        self.assertEqual(struct2.f154, struct1.f154)
        self.assertEqual(struct2.f155, struct1.f155)
        self.assertEqual(struct2.f156, struct1.f156)
        self.assertEqual(struct2.f157, struct1.f157)

        self.assertEqual(struct2.f1000, struct1.f1000)
        self.assertEqual(struct2.f1001, struct1.f1001)
        self.assertEqual(struct2.f1002, struct1.f1002)
        self.assertEqual(struct2.f1003, struct1.f1003)
        self.assertEqual(struct2.f1004, struct1.f1004)
        self.assertEqual(struct2.f1005, struct1.f1005)
        self.assertEqual(struct2.f1006, struct1.f1006)
        self.assertEqual(struct2.f1007, struct1.f1007)

    def test_serialization_json_struct_bytes(self):
        # Define a source JSON string
        json = r'{"f1":"QUJD","f2":"dGVzdA==","f3":null}'

        # Create a new struct from the source JSON string
        struct1 = test.StructBytes.from_json(json)

        # Serialize the struct to the JSON string
        json = struct1.to_json()

        # Check the serialized JSON size
        self.assertGreater(len(json), 0)

        # Deserialize the struct from the JSON string
        struct2 = test.StructBytes.from_json(json)

        self.assertEqual(len(struct2.f1), 3)
        self.assertEqual(chr(struct2.f1[0]), 'A')
        self.assertEqual(chr(struct2.f1[1]), 'B')
        self.assertEqual(chr(struct2.f1[2]), 'C')
        self.assertNotEqual(struct2.f2, None)
        self.assertEqual(len(struct2.f2), 4)
        self.assertEqual(chr(struct2.f2[0]), 't')
        self.assertEqual(chr(struct2.f2[1]), 'e')
        self.assertEqual(chr(struct2.f2[2]), 's')
        self.assertEqual(chr(struct2.f2[3]), 't')
        self.assertEqual(struct2.f3, None)

    def test_serialization_json_struct_array(self):
        # Define a source JSON string
        json = r'{"f1":[48,65],"f2":[97,null],"f3":["MDAw","QUFB"],"f4":["YWFh",null],"f5":[1,2],"f6":[1,null],"f7":[3,7],"f8":[3,null],"f9":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145986060361000,"f36":"00000000-0000-0000-0000-000000000000","f37":"cedcad98-f0a6-11e8-9f47-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145986060910000,"f36":"00000000-0000-0000-0000-000000000000","f37":"cedcc274-f0a6-11e8-9f47-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}],"f10":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145986061436000,"f36":"00000000-0000-0000-0000-000000000000","f37":"cedcd714-f0a6-11e8-9f47-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},null]}'

        # Create a new struct from the source JSON string
        struct1 = test.StructArray.from_json(json)

        # Serialize the struct to the JSON string
        json = struct1.to_json()

        # Check the serialized JSON size
        self.assertGreater(len(json), 0)

        # Deserialize the struct from the JSON string
        struct2 = test.StructArray.from_json(json)

        self.assertEqual(len(struct2.f1), 2)
        self.assertEqual(struct2.f1[0], 48)
        self.assertEqual(struct2.f1[1], 65)
        self.assertEqual(len(struct2.f2), 2)
        self.assertEqual(struct2.f2[0], 97)
        self.assertEqual(struct2.f2[1], None)
        self.assertEqual(len(struct2.f3), 2)
        self.assertEqual(len(struct2.f3[0]), 3)
        self.assertEqual(struct2.f3[0][0], 48)
        self.assertEqual(struct2.f3[0][1], 48)
        self.assertEqual(struct2.f3[0][2], 48)
        self.assertEqual(len(struct2.f3[1]), 3)
        self.assertEqual(struct2.f3[1][0], 65)
        self.assertEqual(struct2.f3[1][1], 65)
        self.assertEqual(struct2.f3[1][2], 65)
        self.assertEqual(len(struct2.f4), 2)
        self.assertNotEqual(struct2.f4[0], None)
        self.assertEqual(len(struct2.f4[0]), 3)
        self.assertEqual(struct2.f4[0][0], 97)
        self.assertEqual(struct2.f4[0][1], 97)
        self.assertEqual(struct2.f4[0][2], 97)
        self.assertEqual(struct2.f4[1], None)
        self.assertEqual(len(struct2.f5), 2)
        self.assertEqual(struct2.f5[0], test.EnumSimple.ENUM_VALUE_1)
        self.assertEqual(struct2.f5[1], test.EnumSimple.ENUM_VALUE_2)
        self.assertEqual(len(struct2.f6), 2)
        self.assertEqual(struct2.f6[0], test.EnumSimple.ENUM_VALUE_1)
        self.assertEqual(struct2.f6[1], None)
        self.assertEqual(len(struct2.f7), 2)
        self.assertEqual(struct2.f7[0], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2)
        self.assertEqual(struct2.f7[1], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2 | test.FlagsSimple.FLAG_VALUE_3)
        self.assertEqual(len(struct2.f8), 2)
        self.assertEqual(struct2.f8[0], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2)
        self.assertEqual(struct2.f8[1], None)
        self.assertEqual(len(struct2.f9), 2)
        self.assertEqual(struct2.f9[0].f2, True)
        self.assertEqual(struct2.f9[0].f12, 0xFF)
        self.assertEqual(struct2.f9[0].f32, "Initial string!")
        self.assertEqual(struct2.f9[1].f2, True)
        self.assertEqual(struct2.f9[1].f12, 0xFF)
        self.assertEqual(struct2.f9[1].f32, "Initial string!")
        self.assertEqual(len(struct2.f10), 2)
        self.assertNotEqual(struct2.f10[0], None)
        self.assertEqual(struct2.f10[0].f2, True)
        self.assertEqual(struct2.f10[0].f12, 0xFF)
        self.assertEqual(struct2.f10[0].f32, "Initial string!")
        self.assertEqual(struct2.f10[1], None)

    def test_serialization_json_struct_vector(self):
        # Define a source JSON string
        json = r'{"f1":[48,65],"f2":[97,null],"f3":["MDAw","QUFB"],"f4":["YWFh",null],"f5":[1,2],"f6":[1,null],"f7":[3,7],"f8":[3,null],"f9":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146157127964000,"f36":"00000000-0000-0000-0000-000000000000","f37":"34d38702-f0a7-11e8-b30e-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146157128572000,"f36":"00000000-0000-0000-0000-000000000000","f37":"34d39c88-f0a7-11e8-b30e-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}],"f10":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146157129063000,"f36":"00000000-0000-0000-0000-000000000000","f37":"34d3b038-f0a7-11e8-b30e-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},null]}'

        # Create a new struct from the source JSON string
        struct1 = test.StructVector.from_json(json)

        # Serialize the struct to the JSON string
        json = struct1.to_json()

        # Check the serialized JSON size
        self.assertGreater(len(json), 0)

        # Deserialize the struct from the JSON string
        struct2 = test.StructVector.from_json(json)

        self.assertEqual(len(struct2.f1), 2)
        self.assertEqual(struct2.f1[0], 48)
        self.assertEqual(struct2.f1[1], 65)
        self.assertEqual(len(struct2.f2), 2)
        self.assertEqual(struct2.f2[0], 97)
        self.assertEqual(struct2.f2[1], None)
        self.assertEqual(len(struct2.f3), 2)
        self.assertEqual(len(struct2.f3[0]), 3)
        self.assertEqual(struct2.f3[0][0], 48)
        self.assertEqual(struct2.f3[0][1], 48)
        self.assertEqual(struct2.f3[0][2], 48)
        self.assertEqual(len(struct2.f3[1]), 3)
        self.assertEqual(struct2.f3[1][0], 65)
        self.assertEqual(struct2.f3[1][1], 65)
        self.assertEqual(struct2.f3[1][2], 65)
        self.assertEqual(len(struct2.f4), 2)
        self.assertNotEqual(struct2.f4[0], None)
        self.assertEqual(len(struct2.f4[0]), 3)
        self.assertEqual(struct2.f4[0][0], 97)
        self.assertEqual(struct2.f4[0][1], 97)
        self.assertEqual(struct2.f4[0][2], 97)
        self.assertEqual(struct2.f4[1], None)
        self.assertEqual(len(struct2.f5), 2)
        self.assertEqual(struct2.f5[0], test.EnumSimple.ENUM_VALUE_1)
        self.assertEqual(struct2.f5[1], test.EnumSimple.ENUM_VALUE_2)
        self.assertEqual(len(struct2.f6), 2)
        self.assertEqual(struct2.f6[0], test.EnumSimple.ENUM_VALUE_1)
        self.assertEqual(struct2.f6[1], None)
        self.assertEqual(len(struct2.f7), 2)
        self.assertEqual(struct2.f7[0], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2)
        self.assertEqual(struct2.f7[1], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2 | test.FlagsSimple.FLAG_VALUE_3)
        self.assertEqual(len(struct2.f8), 2)
        self.assertEqual(struct2.f8[0], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2)
        self.assertEqual(struct2.f8[1], None)
        self.assertEqual(len(struct2.f9), 2)
        self.assertEqual(struct2.f9[0].f2, True)
        self.assertEqual(struct2.f9[0].f12, 0xFF)
        self.assertEqual(struct2.f9[0].f32, "Initial string!")
        self.assertEqual(struct2.f9[1].f2, True)
        self.assertEqual(struct2.f9[1].f12, 0xFF)
        self.assertEqual(struct2.f9[1].f32, "Initial string!")
        self.assertEqual(len(struct2.f10), 2)
        self.assertNotEqual(struct2.f10[0], None)
        self.assertEqual(struct2.f10[0].f2, True)
        self.assertEqual(struct2.f10[0].f12, 0xFF)
        self.assertEqual(struct2.f10[0].f32, "Initial string!")
        self.assertEqual(struct2.f10[1], None)

    def test_serialization_json_struct_list(self):
        # Define a source JSON string
        json = r'{"f1":[48,65],"f2":[97,null],"f3":["MDAw","QUFB"],"f4":["YWFh",null],"f5":[1,2],"f6":[1,null],"f7":[3,7],"f8":[3,null],"f9":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146220253760000,"f36":"00000000-0000-0000-0000-000000000000","f37":"5a73e7fe-f0a7-11e8-89e6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146220255725000,"f36":"00000000-0000-0000-0000-000000000000","f37":"5a741990-f0a7-11e8-89e6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}],"f10":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146220256802000,"f36":"00000000-0000-0000-0000-000000000000","f37":"5a74e4b0-f0a7-11e8-89e6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},null]}'

        # Create a new struct from the source JSON string
        struct1 = test.StructList.from_json(json)

        # Serialize the struct to the JSON string
        json = struct1.to_json()

        # Check the serialized JSON size
        self.assertGreater(len(json), 0)

        # Deserialize the struct from the JSON string
        struct2 = test.StructList.from_json(json)

        self.assertEqual(len(struct2.f1), 2)
        self.assertEqual(struct2.f1[0], 48)
        self.assertEqual(struct2.f1[1], 65)
        self.assertEqual(len(struct2.f2), 2)
        self.assertEqual(struct2.f2[0], 97)
        self.assertEqual(struct2.f2[1], None)
        self.assertEqual(len(struct2.f3), 2)
        self.assertEqual(len(struct2.f3[0]), 3)
        self.assertEqual(struct2.f3[0][0], 48)
        self.assertEqual(struct2.f3[0][1], 48)
        self.assertEqual(struct2.f3[0][2], 48)
        self.assertEqual(len(struct2.f3[1]), 3)
        self.assertEqual(struct2.f3[1][0], 65)
        self.assertEqual(struct2.f3[1][1], 65)
        self.assertEqual(struct2.f3[1][2], 65)
        self.assertEqual(len(struct2.f4), 2)
        self.assertNotEqual(struct2.f4[0], None)
        self.assertEqual(len(struct2.f4[0]), 3)
        self.assertEqual(struct2.f4[0][0], 97)
        self.assertEqual(struct2.f4[0][1], 97)
        self.assertEqual(struct2.f4[0][2], 97)
        self.assertEqual(struct2.f4[1], None)
        self.assertEqual(len(struct2.f5), 2)
        self.assertEqual(struct2.f5[0], test.EnumSimple.ENUM_VALUE_1)
        self.assertEqual(struct2.f5[1], test.EnumSimple.ENUM_VALUE_2)
        self.assertEqual(len(struct2.f6), 2)
        self.assertEqual(struct2.f6[0], test.EnumSimple.ENUM_VALUE_1)
        self.assertEqual(struct2.f6[1], None)
        self.assertEqual(len(struct2.f7), 2)
        self.assertEqual(struct2.f7[0], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2)
        self.assertEqual(struct2.f7[1], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2 | test.FlagsSimple.FLAG_VALUE_3)
        self.assertEqual(len(struct2.f8), 2)
        self.assertEqual(struct2.f8[0], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2)
        self.assertEqual(struct2.f8[1], None)
        self.assertEqual(len(struct2.f9), 2)
        self.assertEqual(struct2.f9[0].f2, True)
        self.assertEqual(struct2.f9[0].f12, 0xFF)
        self.assertEqual(struct2.f9[0].f32, "Initial string!")
        self.assertEqual(struct2.f9[1].f2, True)
        self.assertEqual(struct2.f9[1].f12, 0xFF)
        self.assertEqual(struct2.f9[1].f32, "Initial string!")
        self.assertEqual(len(struct2.f10), 2)
        self.assertNotEqual(struct2.f10[0], None)
        self.assertEqual(struct2.f10[0].f2, True)
        self.assertEqual(struct2.f10[0].f12, 0xFF)
        self.assertEqual(struct2.f10[0].f32, "Initial string!")
        self.assertEqual(struct2.f10[1], None)

    def test_serialization_json_struct_set(self):
        # Define a source JSON string
        json = r'{"f1":[48,65,97],"f2":[1,2],"f3":[3,7],"f4":[{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146299848353000,"f36":"00000000-0000-0000-0000-000000000000","f37":"89e4edd0-f0a7-11e8-9dde-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":65,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146299848966000,"f36":"00000000-0000-0000-0000-000000000000","f37":"89e503f6-f0a7-11e8-9dde-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}]}'

        # Create a new struct from the source JSON string
        struct1 = test.StructSet.from_json(json)

        # Serialize the struct to the JSON string
        json = struct1.to_json()

        # Check the serialized JSON size
        self.assertGreater(len(json), 0)

        # Deserialize the struct from the JSON string
        struct2 = test.StructSet.from_json(json)

        self.assertEqual(len(struct2.f1), 3)
        self.assertTrue(48 in struct2.f1)
        self.assertTrue(65 in struct2.f1)
        self.assertTrue(97 in struct2.f1)
        self.assertEqual(len(struct2.f2), 2)
        self.assertTrue(test.EnumSimple.ENUM_VALUE_1 in struct2.f2)
        self.assertTrue(test.EnumSimple.ENUM_VALUE_2 in struct2.f2)
        self.assertEqual(len(struct2.f3), 2)
        self.assertTrue((test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2) in struct2.f3)
        self.assertTrue((test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2 | test.FlagsSimple.FLAG_VALUE_3) in struct2.f3)
        self.assertEqual(len(struct2.f4), 2)
        s1 = test.StructSimple()
        s1.id = 48
        self.assertTrue(s1 in struct2.f4)
        s2 = test.StructSimple()
        s2.id = 65
        self.assertTrue(s2 in struct2.f4)

    def test_serialization_json_struct_map(self):
        # Define a source JSON string
        json = r'{"f1":{"10":48,"20":65},"f2":{"10":97,"20":null},"f3":{"10":"MDAw","20":"QUFB"},"f4":{"10":"YWFh","20":null},"f5":{"10":1,"20":2},"f6":{"10":1,"20":null},"f7":{"10":3,"20":7},"f8":{"10":3,"20":null},"f9":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146345803483000,"f36":"00000000-0000-0000-0000-000000000000","f37":"a549215e-f0a7-11e8-90f6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":{"id":65,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146345804184000,"f36":"00000000-0000-0000-0000-000000000000","f37":"a54942ce-f0a7-11e8-90f6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}},"f10":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146345803483000,"f36":"00000000-0000-0000-0000-000000000000","f37":"a549215e-f0a7-11e8-90f6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":null}}'

        # Create a new struct from the source JSON string
        struct1 = test.StructMap.from_json(json)

        # Serialize the struct to the JSON string
        json = struct1.to_json()

        # Check the serialized JSON size
        self.assertGreater(len(json), 0)

        # Deserialize the struct from the JSON string
        struct2 = test.StructMap.from_json(json)

        self.assertEqual(len(struct2.f1), 2)
        self.assertEqual(struct2.f1["10"], 48)
        self.assertEqual(struct2.f1["20"], 65)
        self.assertEqual(len(struct2.f2), 2)
        self.assertEqual(struct2.f2["10"], 97)
        self.assertEqual(struct2.f2["20"], None)
        self.assertEqual(len(struct2.f3), 2)
        self.assertEqual(len(struct2.f3["10"]), 3)
        self.assertEqual(len(struct2.f3["20"]), 3)
        self.assertEqual(len(struct2.f4), 2)
        self.assertEqual(len(struct2.f4["10"]), 3)
        self.assertEqual(struct2.f4["20"], None)
        self.assertEqual(len(struct2.f5), 2)
        self.assertEqual(struct2.f5["10"], test.EnumSimple.ENUM_VALUE_1)
        self.assertEqual(struct2.f5["20"], test.EnumSimple.ENUM_VALUE_2)
        self.assertEqual(len(struct2.f6), 2)
        self.assertEqual(struct2.f6["10"], test.EnumSimple.ENUM_VALUE_1)
        self.assertEqual(struct2.f6["20"], None)
        self.assertEqual(len(struct2.f7), 2)
        self.assertEqual(struct2.f7["10"], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2)
        self.assertEqual(struct2.f7["20"], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2 | test.FlagsSimple.FLAG_VALUE_3)
        self.assertEqual(len(struct2.f8), 2)
        self.assertEqual(struct2.f8["10"], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2)
        self.assertEqual(struct2.f8["20"], None)
        self.assertEqual(len(struct2.f9), 2)
        self.assertEqual(struct2.f9["10"].id, 48)
        self.assertEqual(struct2.f9["20"].id, 65)
        self.assertEqual(len(struct2.f10), 2)
        self.assertEqual(struct2.f10["10"].id, 48)
        self.assertEqual(struct2.f10["20"], None)

    def test_serialization_json_struct_hash(self):
        # Define a source JSON string
        json = r'{"f1":{"10":48,"20":65},"f2":{"10":97,"20":null},"f3":{"10":"MDAw","20":"QUFB"},"f4":{"10":"YWFh","20":null},"f5":{"10":1,"20":2},"f6":{"10":1,"20":null},"f7":{"10":3,"20":7},"f8":{"10":3,"20":null},"f9":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146381450913000,"f36":"00000000-0000-0000-0000-000000000000","f37":"ba8885d2-f0a7-11e8-81fa-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":{"id":65,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146381452825000,"f36":"00000000-0000-0000-0000-000000000000","f37":"ba88ced4-f0a7-11e8-81fa-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}},"f10":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146381450913000,"f36":"00000000-0000-0000-0000-000000000000","f37":"ba8885d2-f0a7-11e8-81fa-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":null}}'

        # Create a new struct from the source JSON string
        struct1 = test.StructHash.from_json(json)

        # Serialize the struct to the JSON string
        json = struct1.to_json()

        # Check the serialized JSON size
        self.assertGreater(len(json), 0)

        # Deserialize the struct from the JSON string
        struct2 = test.StructHash.from_json(json)

        self.assertEqual(len(struct2.f1), 2)
        self.assertEqual(struct2.f1["10"], 48)
        self.assertEqual(struct2.f1["20"], 65)
        self.assertEqual(len(struct2.f2), 2)
        self.assertEqual(struct2.f2["10"], 97)
        self.assertEqual(struct2.f2["20"], None)
        self.assertEqual(len(struct2.f3), 2)
        self.assertEqual(len(struct2.f3["10"]), 3)
        self.assertEqual(len(struct2.f3["20"]), 3)
        self.assertEqual(len(struct2.f4), 2)
        self.assertEqual(len(struct2.f4["10"]), 3)
        self.assertEqual(struct2.f4["20"], None)
        self.assertEqual(len(struct2.f5), 2)
        self.assertEqual(struct2.f5["10"], test.EnumSimple.ENUM_VALUE_1)
        self.assertEqual(struct2.f5["20"], test.EnumSimple.ENUM_VALUE_2)
        self.assertEqual(len(struct2.f6), 2)
        self.assertEqual(struct2.f6["10"], test.EnumSimple.ENUM_VALUE_1)
        self.assertEqual(struct2.f6["20"], None)
        self.assertEqual(len(struct2.f7), 2)
        self.assertEqual(struct2.f7["10"], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2)
        self.assertEqual(struct2.f7["20"], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2 | test.FlagsSimple.FLAG_VALUE_3)
        self.assertEqual(len(struct2.f8), 2)
        self.assertEqual(struct2.f8["10"], test.FlagsSimple.FLAG_VALUE_1 | test.FlagsSimple.FLAG_VALUE_2)
        self.assertEqual(struct2.f8["20"], None)
        self.assertEqual(len(struct2.f9), 2)
        self.assertEqual(struct2.f9["10"].id, 48)
        self.assertEqual(struct2.f9["20"].id, 65)
        self.assertEqual(len(struct2.f10), 2)
        self.assertEqual(struct2.f10["10"].id, 48)
        self.assertEqual(struct2.f10["20"], None)
