/* eslint-disable prefer-const,no-loss-of-precision */
'use strict'

const test = require('tape')

const big = require('../proto/big')
const int64 = require('../proto/int64')
const proto = require('../proto/proto')
const prototest = require('../proto/test')
const uuid = require('../proto/uuid')

const Big = big.Big
const Int64 = int64.Int64
const UInt64 = int64.UInt64
const UUID = uuid.UUID

test('Serialization (JSON): domain', function (t) {
  // Define a source JSON string
  let json = String.raw`{"id":1,"name":"Test","state":6,"wallet":{"currency":"USD","amount":1000.0},"asset":{"currency":"EUR","amount":100.0},"orders":[{"id":1,"symbol":"EURUSD","side":0,"type":0,"price":1.23456,"volume":1000.0},{"id":2,"symbol":"EURUSD","side":1,"type":1,"price":1.0,"volume":100.0},{"id":3,"symbol":"EURUSD","side":0,"type":2,"price":1.5,"volume":10.0}]}`

  // Create a new account from the source JSON string
  let account1 = proto.Account.fromJSON(json)

  // Serialize the account to the JSON stream
  json = JSON.stringify(account1)

  // Check the serialized JSON size
  t.true(json.length > 0)

  // Deserialize the account from the JSON stream
  let account2 = proto.Account.fromJSON(json)

  t.equal(account2.id, 1)
  t.equal(account2.name, 'Test')
  t.true(account2.state.hasFlags(proto.State.good))
  t.equal(account2.wallet.currency, 'USD')
  t.equal(account2.wallet.amount, 1000.0)
  t.notEqual(account2.asset, undefined)
  t.equal(account2.asset.currency, 'EUR')
  t.equal(account2.asset.amount, 100.0)
  t.equal(account2.orders.length, 3)
  t.equal(account2.orders[0].id, 1)
  t.equal(account2.orders[0].symbol, 'EURUSD')
  t.true(account2.orders[0].side.eq(proto.OrderSide.buy))
  t.true(account2.orders[0].type.eq(proto.OrderType.market))
  t.equal(account2.orders[0].price, 1.23456)
  t.equal(account2.orders[0].volume, 1000.0)
  t.equal(account2.orders[1].id, 2)
  t.equal(account2.orders[1].symbol, 'EURUSD')
  t.true(account2.orders[1].side.eq(proto.OrderSide.sell))
  t.true(account2.orders[1].type.eq(proto.OrderType.limit))
  t.equal(account2.orders[1].price, 1.0)
  t.equal(account2.orders[1].volume, 100.0)
  t.equal(account2.orders[2].id, 3)
  t.equal(account2.orders[2].symbol, 'EURUSD')
  t.true(account2.orders[2].side.eq(proto.OrderSide.buy))
  t.true(account2.orders[2].type.eq(proto.OrderType.stop))
  t.equal(account2.orders[2].price, 1.5)
  t.equal(account2.orders[2].volume, 10.0)
  t.end()
})

test('Serialization (JSON): struct simple', function (t) {
  // Define a source JSON string
  let json = String.raw`{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145597933463000,"f36":"00000000-0000-0000-0000-000000000000","f37":"e7854072-f0a5-11e8-8f69-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}`

  // Create a new struct from the source JSON string
  let struct1 = prototest.StructSimple.fromJSON(json)

  // Serialize the struct to the JSON string
  json = JSON.stringify(struct1)

  // Check the serialized JSON size
  t.true(json.length > 0)

  // Deserialize the struct from the JSON string
  let struct2 = prototest.StructSimple.fromJSON(json)

  t.equal(struct2.f1, false)
  t.equal(struct2.f2, true)
  t.equal(struct2.f3, 0)
  t.equal(struct2.f4, 0xFF)
  t.equal(struct2.f5, '\0')
  t.equal(struct2.f6, '!')
  t.equal(struct2.f7, String.fromCharCode(0))
  t.equal(struct2.f8, String.fromCharCode(0x0444))
  t.equal(struct2.f9, 0)
  t.equal(struct2.f10, 127)
  t.equal(struct2.f11, 0)
  t.equal(struct2.f12, 0xFF)
  t.equal(struct2.f13, 0)
  t.equal(struct2.f14, 32767)
  t.equal(struct2.f15, 0)
  t.equal(struct2.f16, 0xFFFF)
  t.equal(struct2.f17, 0)
  t.equal(struct2.f18, 2147483647)
  t.equal(struct2.f19, 0)
  t.equal(struct2.f20, 0xFFFFFFFF)
  t.true(struct2.f21.eq(Int64.fromNumber(0)))
  t.true(struct2.f22.eq(Int64.fromNumber(9223372036854775807)))
  t.true(struct2.f23.eq(UInt64.fromNumber(0)))
  t.true(struct2.f24.eq(UInt64.fromNumber(0xFFFFFFFFFFFFFFFF)))
  t.equal(struct2.f25, 0.0)
  t.true(Math.abs(struct2.f26 - 123.456) < 0.0001)
  t.equal(struct2.f27, 0.0)
  t.true(Math.abs(struct2.f28 - -123.567e+123) < 1e+123)
  t.true(struct2.f29.eq(new Big('0')))
  t.true(struct2.f30.eq(new Big('123456.123456')))
  t.equal(struct2.f31, '')
  t.equal(struct2.f32, 'Initial string!')
  t.equal(struct2.f33.getTime(), new Date(Date.UTC(1970, 0, 1)).getTime())
  t.equal(struct2.f34.getTime(), new Date(Date.UTC(1970, 0, 1)).getTime())
  t.true(struct2.f35.getTime() > new Date(Date.UTC(2018, 1, 1)).getTime())
  t.true(struct2.f36.eq(new UUID()))
  t.false(struct2.f37.eq(new UUID()))
  t.true(struct2.f38.eq(new UUID('123e4567-e89b-12d3-a456-426655440000')))

  t.equal(struct2.f1, struct1.f1)
  t.equal(struct2.f2, struct1.f2)
  t.equal(struct2.f3, struct1.f3)
  t.equal(struct2.f4, struct1.f4)
  t.equal(struct2.f5, struct1.f5)
  t.equal(struct2.f6, struct1.f6)
  t.equal(struct2.f7, struct1.f7)
  t.equal(struct2.f8, struct1.f8)
  t.equal(struct2.f9, struct1.f9)
  t.equal(struct2.f10, struct1.f10)
  t.equal(struct2.f11, struct1.f11)
  t.equal(struct2.f12, struct1.f12)
  t.equal(struct2.f13, struct1.f13)
  t.equal(struct2.f14, struct1.f14)
  t.equal(struct2.f15, struct1.f15)
  t.equal(struct2.f16, struct1.f16)
  t.equal(struct2.f17, struct1.f17)
  t.equal(struct2.f18, struct1.f18)
  t.equal(struct2.f19, struct1.f19)
  t.equal(struct2.f20, struct1.f20)
  t.true(struct2.f21.eq(struct1.f21))
  t.true(struct2.f22.eq(struct1.f22))
  t.true(struct2.f23.eq(struct1.f23))
  t.true(struct2.f24.eq(struct1.f24))
  t.equal(struct2.f25, struct1.f25)
  t.true(Math.abs(struct2.f26 - struct1.f26) < 0.0001)
  t.equal(struct2.f27, struct1.f27)
  t.true(Math.abs(struct2.f28 - struct1.f28) < 1e+123)
  t.true(struct2.f29.eq(struct1.f29))
  t.true(struct2.f29.eq(struct1.f29))
  t.equal(struct2.f31, struct1.f31)
  t.equal(struct2.f32, struct1.f32)
  t.equal(struct2.f33.getTime(), struct1.f33.getTime())
  t.equal(struct2.f34.getTime(), struct1.f34.getTime())
  t.equal(struct2.f35.getTime(), struct1.f35.getTime())
  t.true(struct2.f36.eq(struct1.f36))
  t.true(struct2.f37.eq(struct1.f37))
  t.true(struct2.f38.eq(struct1.f38))
  t.true(struct2.f39.eq(struct1.f39))
  t.true(struct2.f40.eq(struct1.f40))
  t.end()
})

test('Serialization (JSON): struct optional', function (t) {
  // Define a source JSON string
  let json = String.raw`{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145860677797000,"f36":"00000000-0000-0000-0000-000000000000","f37":"8420d1c6-f0a6-11e8-80fc-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]},"f100":null,"f101":true,"f102":null,"f103":null,"f104":255,"f105":null,"f106":null,"f107":33,"f108":null,"f109":null,"f110":1092,"f111":null,"f112":null,"f113":127,"f114":null,"f115":null,"f116":255,"f117":null,"f118":null,"f119":32767,"f120":null,"f121":null,"f122":65535,"f123":null,"f124":null,"f125":2147483647,"f126":null,"f127":null,"f128":4294967295,"f129":null,"f130":null,"f131":9223372036854775807,"f132":null,"f133":null,"f134":18446744073709551615,"f135":null,"f136":null,"f137":123.456,"f138":null,"f139":null,"f140":-1.23456e+125,"f141":null,"f142":null,"f143":"123456.123456","f144":null,"f145":null,"f146":"Initial string!","f147":null,"f148":null,"f149":1543145860678429000,"f150":null,"f151":null,"f152":"123e4567-e89b-12d3-a456-426655440000","f153":null,"f154":null,"f155":null,"f156":null,"f157":null,"f158":null,"f159":null,"f160":null,"f161":null,"f162":null,"f163":null,"f164":null,"f165":null}`

  // Create a new struct from the source JSON string
  let struct1 = prototest.StructOptional.fromJSON(json)

  // Serialize the struct to the JSON string
  json = JSON.stringify(struct1)

  // Check the serialized JSON size
  t.true(json.length > 0)

  // Deserialize the struct from the JSON string
  let struct2 = prototest.StructOptional.fromJSON(json)

  t.equal(struct2.f1, false)
  t.equal(struct2.f2, true)
  t.equal(struct2.f3, 0)
  t.equal(struct2.f4, 0xFF)
  t.equal(struct2.f5, '\0')
  t.equal(struct2.f6, '!')
  t.equal(struct2.f7, String.fromCharCode(0))
  t.equal(struct2.f8, String.fromCharCode(0x0444))
  t.equal(struct2.f9, 0)
  t.equal(struct2.f10, 127)
  t.equal(struct2.f11, 0)
  t.equal(struct2.f12, 0xFF)
  t.equal(struct2.f13, 0)
  t.equal(struct2.f14, 32767)
  t.equal(struct2.f15, 0)
  t.equal(struct2.f16, 0xFFFF)
  t.equal(struct2.f17, 0)
  t.equal(struct2.f18, 2147483647)
  t.equal(struct2.f19, 0)
  t.equal(struct2.f20, 0xFFFFFFFF)
  t.true(struct2.f21.eq(Int64.fromNumber(0)))
  t.true(struct2.f22.eq(Int64.fromNumber(9223372036854775807)))
  t.true(struct2.f23.eq(UInt64.fromNumber(0)))
  t.true(struct2.f24.eq(UInt64.fromNumber(0xFFFFFFFFFFFFFFFF)))
  t.equal(struct2.f25, 0.0)
  t.true(Math.abs(struct2.f26 - 123.456) < 0.0001)
  t.equal(struct2.f27, 0.0)
  t.true(Math.abs(struct2.f28 - -123.567e+123) < 1e+123)
  t.true(struct2.f29.eq(new Big('0')))
  t.true(struct2.f30.eq(new Big('123456.123456')))
  t.equal(struct2.f31, '')
  t.equal(struct2.f32, 'Initial string!')
  t.equal(struct2.f33.getTime(), new Date(Date.UTC(1970, 0, 1)).getTime())
  t.equal(struct2.f34.getTime(), new Date(Date.UTC(1970, 0, 1)).getTime())
  t.true(struct2.f35.getTime() > new Date(Date.UTC(2018, 1, 1)).getTime())
  t.true(struct2.f36.eq(new UUID()))
  t.false(struct2.f37.eq(new UUID()))
  t.true(struct2.f38.eq(new UUID('123e4567-e89b-12d3-a456-426655440000')))

  t.equal(struct2.f100, undefined)
  t.notEqual(struct2.f101, null)
  t.equal(struct2.f101, true)
  t.equal(struct2.f102, undefined)
  t.equal(struct2.f103, undefined)
  t.notEqual(struct2.f104, null)
  t.equal(struct2.f104, 0xFF)
  t.equal(struct2.f105, undefined)
  t.equal(struct2.f106, undefined)
  t.notEqual(struct2.f107, null)
  t.equal(struct2.f107, '!')
  t.equal(struct2.f108, undefined)
  t.equal(struct2.f109, undefined)
  t.notEqual(struct2.f110, null)
  t.equal(struct2.f110, String.fromCharCode(0x0444))
  t.equal(struct2.f111, undefined)
  t.equal(struct2.f112, undefined)
  t.notEqual(struct2.f113, null)
  t.equal(struct2.f113, 127)
  t.equal(struct2.f114, undefined)
  t.equal(struct2.f115, undefined)
  t.notEqual(struct2.f116, null)
  t.equal(struct2.f116, 0xFF)
  t.equal(struct2.f117, undefined)
  t.equal(struct2.f118, undefined)
  t.notEqual(struct2.f119, null)
  t.equal(struct2.f119, 32767)
  t.equal(struct2.f120, undefined)
  t.equal(struct2.f121, undefined)
  t.notEqual(struct2.f122, null)
  t.equal(struct2.f122, 0xFFFF)
  t.equal(struct2.f123, undefined)
  t.equal(struct2.f124, undefined)
  t.notEqual(struct2.f125, null)
  t.equal(struct2.f125, 2147483647)
  t.equal(struct2.f126, undefined)
  t.equal(struct2.f127, undefined)
  t.notEqual(struct2.f128, null)
  t.equal(struct2.f128, 0xFFFFFFFF)
  t.equal(struct2.f129, undefined)
  t.equal(struct2.f130, undefined)
  t.notEqual(struct2.f131, null)
  t.true(struct2.f131.eq(Int64.fromNumber(9223372036854775807)))
  t.equal(struct2.f132, undefined)
  t.equal(struct2.f133, undefined)
  t.notEqual(struct2.f134, null)
  t.true(struct2.f134.eq(UInt64.fromNumber(0xFFFFFFFFFFFFFFFF)))
  t.equal(struct2.f135, undefined)
  t.equal(struct2.f136, undefined)
  t.notEqual(struct2.f137, null)
  t.true(Math.abs(struct2.f137 - 123.456) < 0.0001)
  t.equal(struct2.f138, undefined)
  t.equal(struct2.f139, undefined)
  t.notEqual(struct2.f140, null)
  t.true(Math.abs(struct2.f140 - -123.567e+123) < 1e+123)
  t.equal(struct2.f141, undefined)
  t.equal(struct2.f142, undefined)
  t.notEqual(struct2.f143, null)
  t.true(struct2.f143.eq(new Big('123456.123456')))
  t.equal(struct2.f144, undefined)
  t.equal(struct2.f145, undefined)
  t.notEqual(struct2.f146, null)
  t.equal(struct2.f146, 'Initial string!')
  t.equal(struct2.f147, undefined)
  t.equal(struct2.f148, undefined)
  t.notEqual(struct2.f149, null)
  t.true(struct2.f149.getTime() > new Date(Date.UTC(2018, 1, 1)).getTime())
  t.equal(struct2.f150, undefined)
  t.equal(struct2.f151, undefined)
  t.notEqual(struct2.f152, null)
  t.true(struct2.f152.eq(new UUID('123e4567-e89b-12d3-a456-426655440000')))
  t.equal(struct2.f153, undefined)
  t.equal(struct2.f154, undefined)
  t.equal(struct2.f155, undefined)
  t.equal(struct2.f156, undefined)
  t.equal(struct2.f157, undefined)
  t.equal(struct2.f158, undefined)
  t.equal(struct2.f159, undefined)
  t.equal(struct2.f160, undefined)
  t.equal(struct2.f161, undefined)
  t.equal(struct2.f162, undefined)
  t.equal(struct2.f163, undefined)
  t.equal(struct2.f164, undefined)
  t.equal(struct2.f165, undefined)

  t.equal(struct2.f1, struct1.f1)
  t.equal(struct2.f2, struct1.f2)
  t.equal(struct2.f3, struct1.f3)
  t.equal(struct2.f4, struct1.f4)
  t.equal(struct2.f5, struct1.f5)
  t.equal(struct2.f6, struct1.f6)
  t.equal(struct2.f7, struct1.f7)
  t.equal(struct2.f8, struct1.f8)
  t.equal(struct2.f9, struct1.f9)
  t.equal(struct2.f10, struct1.f10)
  t.equal(struct2.f11, struct1.f11)
  t.equal(struct2.f12, struct1.f12)
  t.equal(struct2.f13, struct1.f13)
  t.equal(struct2.f14, struct1.f14)
  t.equal(struct2.f15, struct1.f15)
  t.equal(struct2.f16, struct1.f16)
  t.equal(struct2.f17, struct1.f17)
  t.equal(struct2.f18, struct1.f18)
  t.equal(struct2.f19, struct1.f19)
  t.equal(struct2.f20, struct1.f20)
  t.true(struct2.f21.eq(struct1.f21))
  t.true(struct2.f22.eq(struct1.f22))
  t.true(struct2.f23.eq(struct1.f23))
  t.true(struct2.f24.eq(struct1.f24))
  t.equal(struct2.f25, struct1.f25)
  t.true(Math.abs(struct2.f26 - struct1.f26) < 0.0001)
  t.equal(struct2.f27, struct1.f27)
  t.true(Math.abs(struct2.f28 - struct1.f28) < 1e+123)
  t.true(struct2.f29.eq(struct1.f29))
  t.true(struct2.f30.eq(struct1.f30))
  t.equal(struct2.f31, struct1.f31)
  t.equal(struct2.f32, struct1.f32)
  t.equal(struct2.f33.getTime(), struct1.f33.getTime())
  t.equal(struct2.f34.getTime(), struct1.f34.getTime())
  t.equal(struct2.f35.getTime(), struct1.f35.getTime())
  t.true(struct2.f36.eq(struct1.f36))
  t.true(struct2.f37.eq(struct1.f37))
  t.true(struct2.f38.eq(struct1.f38))
  t.true(struct2.f39.eq(struct1.f39))
  t.true(struct2.f40.eq(struct1.f40))

  t.equal(struct2.f100, struct1.f100)
  t.equal(struct2.f101, struct1.f101)
  t.equal(struct2.f102, struct1.f102)
  t.equal(struct2.f103, struct1.f103)
  t.equal(struct2.f104, struct1.f104)
  t.equal(struct2.f105, struct1.f105)
  t.equal(struct2.f106, struct1.f106)
  t.equal(struct2.f107, struct1.f107)
  t.equal(struct2.f108, struct1.f108)
  t.equal(struct2.f109, struct1.f109)
  t.equal(struct2.f110, struct1.f110)
  t.equal(struct2.f111, struct1.f111)
  t.equal(struct2.f112, struct1.f112)
  t.equal(struct2.f113, struct1.f113)
  t.equal(struct2.f114, struct1.f114)
  t.equal(struct2.f115, struct1.f115)
  t.equal(struct2.f116, struct1.f116)
  t.equal(struct2.f117, struct1.f117)
  t.equal(struct2.f118, struct1.f118)
  t.equal(struct2.f119, struct1.f119)
  t.equal(struct2.f120, struct1.f120)
  t.equal(struct2.f121, struct1.f121)
  t.equal(struct2.f122, struct1.f122)
  t.equal(struct2.f123, struct1.f123)
  t.equal(struct2.f124, struct1.f124)
  t.equal(struct2.f125, struct1.f125)
  t.equal(struct2.f126, struct1.f126)
  t.equal(struct2.f127, struct1.f127)
  t.equal(struct2.f128, struct1.f128)
  t.equal(struct2.f129, struct1.f129)
  t.equal(struct2.f130, struct1.f130)
  t.true(struct2.f131.eq(struct1.f131))
  t.equal(struct2.f132, struct1.f132)
  t.equal(struct2.f133, struct1.f133)
  t.true(struct2.f134.eq(struct1.f134))
  t.equal(struct2.f135, struct1.f135)
  t.equal(struct2.f136, struct1.f136)
  t.true(Math.abs(struct2.f137 - struct1.f137) < 0.0001)
  t.equal(struct2.f138, struct1.f138)
  t.equal(struct2.f139, struct1.f139)
  t.true(Math.abs(struct2.f140 - struct1.f140) < 1e+123)
  t.equal(struct2.f141, struct1.f141)
  t.equal(struct2.f142, struct1.f142)
  t.true(struct2.f143.eq(struct1.f143))
  t.equal(struct2.f144, struct1.f144)
  t.equal(struct2.f145, struct1.f145)
  t.equal(struct2.f146, struct1.f146)
  t.equal(struct2.f147, struct1.f147)
  t.equal(struct2.f148, struct1.f148)
  t.equal(struct2.f149.getTime(), struct1.f149.getTime())
  t.equal(struct2.f150, struct1.f150)
  t.equal(struct2.f151, struct1.f151)
  t.true(struct2.f152.eq(struct1.f152))
  t.equal(struct2.f153, struct1.f153)
  t.equal(struct2.f154, struct1.f154)
  t.equal(struct2.f155, struct1.f155)
  t.equal(struct2.f156, struct1.f156)
  t.equal(struct2.f157, struct1.f157)
  t.end()
})

test('Serialization (JSON): struct nested', function (t) {
  // Define a source JSON string
  let json = String.raw`{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145901646321000,"f36":"00000000-0000-0000-0000-000000000000","f37":"9c8c268e-f0a6-11e8-a777-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]},"f100":null,"f101":true,"f102":null,"f103":null,"f104":255,"f105":null,"f106":null,"f107":33,"f108":null,"f109":null,"f110":1092,"f111":null,"f112":null,"f113":127,"f114":null,"f115":null,"f116":255,"f117":null,"f118":null,"f119":32767,"f120":null,"f121":null,"f122":65535,"f123":null,"f124":null,"f125":2147483647,"f126":null,"f127":null,"f128":4294967295,"f129":null,"f130":null,"f131":9223372036854775807,"f132":null,"f133":null,"f134":18446744073709551615,"f135":null,"f136":null,"f137":123.456,"f138":null,"f139":null,"f140":-1.23456e+125,"f141":null,"f142":null,"f143":"123456.123456","f144":null,"f145":null,"f146":"Initial string!","f147":null,"f148":null,"f149":1543145901647155000,"f150":null,"f151":null,"f152":"123e4567-e89b-12d3-a456-426655440000","f153":null,"f154":null,"f155":null,"f156":null,"f157":null,"f158":null,"f159":null,"f160":null,"f161":null,"f162":null,"f163":null,"f164":null,"f165":null,"f1000":0,"f1001":null,"f1002":50,"f1003":null,"f1004":0,"f1005":null,"f1006":42,"f1007":null,"f1008":{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145901647367000,"f36":"00000000-0000-0000-0000-000000000000","f37":"9c8c54c4-f0a6-11e8-a777-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"f1009":null,"f1010":{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145901648310000,"f36":"00000000-0000-0000-0000-000000000000","f37":"9c8c6b76-f0a6-11e8-a777-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]},"f100":null,"f101":true,"f102":null,"f103":null,"f104":255,"f105":null,"f106":null,"f107":33,"f108":null,"f109":null,"f110":1092,"f111":null,"f112":null,"f113":127,"f114":null,"f115":null,"f116":255,"f117":null,"f118":null,"f119":32767,"f120":null,"f121":null,"f122":65535,"f123":null,"f124":null,"f125":2147483647,"f126":null,"f127":null,"f128":4294967295,"f129":null,"f130":null,"f131":9223372036854775807,"f132":null,"f133":null,"f134":18446744073709551615,"f135":null,"f136":null,"f137":123.456,"f138":null,"f139":null,"f140":-1.23456e+125,"f141":null,"f142":null,"f143":"123456.123456","f144":null,"f145":null,"f146":"Initial string!","f147":null,"f148":null,"f149":1543145901648871000,"f150":null,"f151":null,"f152":"123e4567-e89b-12d3-a456-426655440000","f153":null,"f154":null,"f155":null,"f156":null,"f157":null,"f158":null,"f159":null,"f160":null,"f161":null,"f162":null,"f163":null,"f164":null,"f165":null},"f1011":null}`

  // Create a new struct from the source JSON string
  let struct1 = prototest.StructNested.fromJSON(json)

  // Serialize the struct to the JSON string
  json = JSON.stringify(struct1)

  // Check the serialized JSON size
  t.true(json.length > 0)

  // Deserialize the struct from the JSON string
  let struct2 = prototest.StructNested.fromJSON(json)

  t.equal(struct2.f1, false)
  t.equal(struct2.f2, true)
  t.equal(struct2.f3, 0)
  t.equal(struct2.f4, 0xFF)
  t.equal(struct2.f5, '\0')
  t.equal(struct2.f6, '!')
  t.equal(struct2.f7, String.fromCharCode(0))
  t.equal(struct2.f8, String.fromCharCode(0x0444))
  t.equal(struct2.f9, 0)
  t.equal(struct2.f10, 127)
  t.equal(struct2.f11, 0)
  t.equal(struct2.f12, 0xFF)
  t.equal(struct2.f13, 0)
  t.equal(struct2.f14, 32767)
  t.equal(struct2.f15, 0)
  t.equal(struct2.f16, 0xFFFF)
  t.equal(struct2.f17, 0)
  t.equal(struct2.f18, 2147483647)
  t.equal(struct2.f19, 0)
  t.equal(struct2.f20, 0xFFFFFFFF)
  t.true(struct2.f21.eq(Int64.fromNumber(0)))
  t.true(struct2.f22.eq(Int64.fromNumber(9223372036854775807)))
  t.true(struct2.f23.eq(UInt64.fromNumber(0)))
  t.true(struct2.f24.eq(UInt64.fromNumber(0xFFFFFFFFFFFFFFFF)))
  t.equal(struct2.f25, 0.0)
  t.true(Math.abs(struct2.f26 - 123.456) < 0.0001)
  t.equal(struct2.f27, 0.0)
  t.true(Math.abs(struct2.f28 - -123.567e+123) < 1e+123)
  t.true(struct2.f29.eq(new Big('0')))
  t.true(struct2.f30.eq(new Big('123456.123456')))
  t.equal(struct2.f31, '')
  t.equal(struct2.f32, 'Initial string!')
  t.equal(struct2.f33.getTime(), new Date(Date.UTC(1970, 0, 1)).getTime())
  t.equal(struct2.f34.getTime(), new Date(Date.UTC(1970, 0, 1)).getTime())
  t.true(struct2.f35.getTime() > new Date(Date.UTC(2018, 1, 1)).getTime())
  t.true(struct2.f36.eq(new UUID()))
  t.false(struct2.f37.eq(new UUID()))
  t.true(struct2.f38.eq(new UUID('123e4567-e89b-12d3-a456-426655440000')))

  t.equal(struct2.f100, undefined)
  t.notEqual(struct2.f101, null)
  t.equal(struct2.f101, true)
  t.equal(struct2.f102, undefined)
  t.equal(struct2.f103, undefined)
  t.notEqual(struct2.f104, null)
  t.equal(struct2.f104, 0xFF)
  t.equal(struct2.f105, undefined)
  t.equal(struct2.f106, undefined)
  t.notEqual(struct2.f107, null)
  t.equal(struct2.f107, '!')
  t.equal(struct2.f108, undefined)
  t.equal(struct2.f109, undefined)
  t.notEqual(struct2.f110, null)
  t.equal(struct2.f110, String.fromCharCode(0x0444))
  t.equal(struct2.f111, undefined)
  t.equal(struct2.f112, undefined)
  t.notEqual(struct2.f113, null)
  t.equal(struct2.f113, 127)
  t.equal(struct2.f114, undefined)
  t.equal(struct2.f115, undefined)
  t.notEqual(struct2.f116, null)
  t.equal(struct2.f116, 0xFF)
  t.equal(struct2.f117, undefined)
  t.equal(struct2.f118, undefined)
  t.notEqual(struct2.f119, null)
  t.equal(struct2.f119, 32767)
  t.equal(struct2.f120, undefined)
  t.equal(struct2.f121, undefined)
  t.notEqual(struct2.f122, null)
  t.equal(struct2.f122, 0xFFFF)
  t.equal(struct2.f123, undefined)
  t.equal(struct2.f124, undefined)
  t.notEqual(struct2.f125, null)
  t.equal(struct2.f125, 2147483647)
  t.equal(struct2.f126, undefined)
  t.equal(struct2.f127, undefined)
  t.notEqual(struct2.f128, null)
  t.equal(struct2.f128, 0xFFFFFFFF)
  t.equal(struct2.f129, undefined)
  t.equal(struct2.f130, undefined)
  t.notEqual(struct2.f131, null)
  t.true(struct2.f131.eq(Int64.fromNumber(9223372036854775807)))
  t.equal(struct2.f132, undefined)
  t.equal(struct2.f133, undefined)
  t.notEqual(struct2.f134, null)
  t.true(struct2.f134.eq(UInt64.fromNumber(0xFFFFFFFFFFFFFFFF)))
  t.equal(struct2.f135, undefined)
  t.equal(struct2.f136, undefined)
  t.notEqual(struct2.f137, null)
  t.true(Math.abs(struct2.f137 - 123.456) < 0.0001)
  t.equal(struct2.f138, undefined)
  t.equal(struct2.f139, undefined)
  t.notEqual(struct2.f140, null)
  t.true(Math.abs(struct2.f140 - -123.567e+123) < 1e+123)
  t.equal(struct2.f141, undefined)
  t.equal(struct2.f142, undefined)
  t.notEqual(struct2.f143, null)
  t.true(struct2.f143.eq(new Big('123456.123456')))
  t.equal(struct2.f144, undefined)
  t.equal(struct2.f145, undefined)
  t.notEqual(struct2.f146, null)
  t.equal(struct2.f146, 'Initial string!')
  t.equal(struct2.f147, undefined)
  t.equal(struct2.f148, undefined)
  t.notEqual(struct2.f149, null)
  t.true(struct2.f149.getTime() > new Date(Date.UTC(2018, 1, 1)).getTime())
  t.equal(struct2.f150, undefined)
  t.equal(struct2.f151, undefined)
  t.notEqual(struct2.f152, null)
  t.true(struct2.f152.eq(new UUID('123e4567-e89b-12d3-a456-426655440000')))
  t.equal(struct2.f153, undefined)
  t.equal(struct2.f154, undefined)
  t.equal(struct2.f155, undefined)
  t.equal(struct2.f156, undefined)
  t.equal(struct2.f157, undefined)
  t.equal(struct2.f158, undefined)
  t.equal(struct2.f159, undefined)
  t.equal(struct2.f160, undefined)
  t.equal(struct2.f161, undefined)
  t.equal(struct2.f162, undefined)
  t.equal(struct2.f163, undefined)
  t.equal(struct2.f164, undefined)
  t.equal(struct2.f165, undefined)

  t.true(struct2.f1000.eq(prototest.EnumSimple.ENUM_VALUE_0))
  t.equal(struct2.f1001, undefined)
  t.true(struct2.f1002.eq(prototest.EnumTyped.ENUM_VALUE_2))
  t.equal(struct2.f1003, undefined)
  t.true(struct2.f1004.eq(prototest.FlagsSimple.FLAG_VALUE_0))
  t.equal(struct2.f1005, undefined)
  t.true(struct2.f1006.hasFlags(prototest.FlagsTyped.FLAG_VALUE_2 | prototest.FlagsTyped.FLAG_VALUE_4 | prototest.FlagsTyped.FLAG_VALUE_6))
  t.equal(struct2.f1007, undefined)
  t.equal(struct2.f1009, undefined)
  t.equal(struct2.f1011, undefined)

  t.equal(struct2.f1, struct1.f1)
  t.equal(struct2.f2, struct1.f2)
  t.equal(struct2.f3, struct1.f3)
  t.equal(struct2.f4, struct1.f4)
  t.equal(struct2.f5, struct1.f5)
  t.equal(struct2.f6, struct1.f6)
  t.equal(struct2.f7, struct1.f7)
  t.equal(struct2.f8, struct1.f8)
  t.equal(struct2.f9, struct1.f9)
  t.equal(struct2.f10, struct1.f10)
  t.equal(struct2.f11, struct1.f11)
  t.equal(struct2.f12, struct1.f12)
  t.equal(struct2.f13, struct1.f13)
  t.equal(struct2.f14, struct1.f14)
  t.equal(struct2.f15, struct1.f15)
  t.equal(struct2.f16, struct1.f16)
  t.equal(struct2.f17, struct1.f17)
  t.equal(struct2.f18, struct1.f18)
  t.equal(struct2.f19, struct1.f19)
  t.equal(struct2.f20, struct1.f20)
  t.true(struct2.f21.eq(struct1.f21))
  t.true(struct2.f22.eq(struct1.f22))
  t.true(struct2.f23.eq(struct1.f23))
  t.true(struct2.f24.eq(struct1.f24))
  t.equal(struct2.f25, struct1.f25)
  t.true(Math.abs(struct2.f26 - struct1.f26) < 0.0001)
  t.equal(struct2.f27, struct1.f27)
  t.true(Math.abs(struct2.f28 - struct1.f28) < 1e+123)
  t.true(struct2.f29.eq(struct1.f29))
  t.true(struct2.f29.eq(struct1.f29))
  t.equal(struct2.f31, struct1.f31)
  t.equal(struct2.f32, struct1.f32)
  t.equal(struct2.f33.getTime(), struct1.f33.getTime())
  t.equal(struct2.f34.getTime(), struct1.f34.getTime())
  t.equal(struct2.f35.getTime(), struct1.f35.getTime())
  t.true(struct2.f36.eq(struct1.f36))
  t.true(struct2.f37.eq(struct1.f37))
  t.true(struct2.f38.eq(struct1.f38))
  t.true(struct2.f39.eq(struct1.f39))
  t.true(struct2.f40.eq(struct1.f40))

  t.equal(struct2.f100, struct1.f100)
  t.equal(struct2.f101, struct1.f101)
  t.equal(struct2.f102, struct1.f102)
  t.equal(struct2.f103, struct1.f103)
  t.equal(struct2.f104, struct1.f104)
  t.equal(struct2.f105, struct1.f105)
  t.equal(struct2.f106, struct1.f106)
  t.equal(struct2.f107, struct1.f107)
  t.equal(struct2.f108, struct1.f108)
  t.equal(struct2.f109, struct1.f109)
  t.equal(struct2.f110, struct1.f110)
  t.equal(struct2.f111, struct1.f111)
  t.equal(struct2.f112, struct1.f112)
  t.equal(struct2.f113, struct1.f113)
  t.equal(struct2.f114, struct1.f114)
  t.equal(struct2.f115, struct1.f115)
  t.equal(struct2.f116, struct1.f116)
  t.equal(struct2.f117, struct1.f117)
  t.equal(struct2.f118, struct1.f118)
  t.equal(struct2.f119, struct1.f119)
  t.equal(struct2.f120, struct1.f120)
  t.equal(struct2.f121, struct1.f121)
  t.equal(struct2.f122, struct1.f122)
  t.equal(struct2.f123, struct1.f123)
  t.equal(struct2.f124, struct1.f124)
  t.equal(struct2.f125, struct1.f125)
  t.equal(struct2.f126, struct1.f126)
  t.equal(struct2.f127, struct1.f127)
  t.equal(struct2.f128, struct1.f128)
  t.equal(struct2.f129, struct1.f129)
  t.equal(struct2.f130, struct1.f130)
  t.true(struct2.f131.eq(struct1.f131))
  t.equal(struct2.f132, struct1.f132)
  t.equal(struct2.f133, struct1.f133)
  t.true(struct2.f134.eq(struct1.f134))
  t.equal(struct2.f135, struct1.f135)
  t.equal(struct2.f136, struct1.f136)
  t.true(Math.abs(struct2.f137 - struct1.f137) < 0.0001)
  t.equal(struct2.f138, struct1.f138)
  t.equal(struct2.f139, struct1.f139)
  t.true(Math.abs(struct2.f140 - struct1.f140) < 1e+123)
  t.equal(struct2.f141, struct1.f141)
  t.equal(struct2.f142, struct1.f142)
  t.true(struct2.f143.eq(struct1.f143))
  t.equal(struct2.f144, struct1.f144)
  t.equal(struct2.f145, struct1.f145)
  t.equal(struct2.f146, struct1.f146)
  t.equal(struct2.f147, struct1.f147)
  t.equal(struct2.f148, struct1.f148)
  t.equal(struct2.f149.getTime(), struct1.f149.getTime())
  t.equal(struct2.f150, struct1.f150)
  t.equal(struct2.f151, struct1.f151)
  t.true(struct2.f152.eq(struct1.f152))
  t.equal(struct2.f153, struct1.f153)
  t.equal(struct2.f154, struct1.f154)
  t.equal(struct2.f155, struct1.f155)
  t.equal(struct2.f156, struct1.f156)
  t.equal(struct2.f157, struct1.f157)

  t.true(struct2.f1000.eq(struct1.f1000))
  t.equal(struct2.f1001, struct1.f1001)
  t.true(struct2.f1002.eq(struct1.f1002))
  t.equal(struct2.f1003, struct1.f1003)
  t.true(struct2.f1004.eq(struct1.f1004))
  t.equal(struct2.f1005, struct1.f1005)
  t.true(struct2.f1006.eq(struct1.f1006))
  t.equal(struct2.f1007, struct1.f1007)
  t.end()
})

test('Serialization (JSON): struct bytes', function (t) {
  // Define a source JSON string
  let json = String.raw`{"f1":"QUJD\n","f2":"dGVzdA==\n","f3":null}`

  // Create a new struct from the source JSON string
  let struct1 = prototest.StructBytes.fromJSON(json)

  // Serialize the struct to the JSON string
  json = JSON.stringify(struct1)

  // Check the serialized JSON size
  t.true(json.length > 0)

  // Deserialize the struct from the JSON string
  let struct2 = prototest.StructBytes.fromJSON(json)

  t.equal(struct2.f1.length, 3)
  t.equal(struct2.f1[0], 'A'.charCodeAt(0))
  t.equal(struct2.f1[1], 'B'.charCodeAt(0))
  t.equal(struct2.f1[2], 'C'.charCodeAt(0))
  t.notEqual(struct2.f2, undefined)
  t.equal(struct2.f2.length, 4)
  t.equal(struct2.f2[0], 't'.charCodeAt(0))
  t.equal(struct2.f2[1], 'e'.charCodeAt(0))
  t.equal(struct2.f2[2], 's'.charCodeAt(0))
  t.equal(struct2.f2[3], 't'.charCodeAt(0))
  t.equal(struct2.f3, undefined)
  t.end()
})

test('Serialization (JSON): struct array', function (t) {
  // Define a source JSON string
  let json = String.raw`{"f1":[48,65],"f2":[97,null],"f3":["MDAw\\n","QUFB\\n"],"f4":["YWFh\\n",null],"f5":[1,2],"f6":[1,null],"f7":[3,7],"f8":[3,null],"f9":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145986060361000,"f36":"00000000-0000-0000-0000-000000000000","f37":"cedcad98-f0a6-11e8-9f47-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145986060910000,"f36":"00000000-0000-0000-0000-000000000000","f37":"cedcc274-f0a6-11e8-9f47-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}],"f10":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145986061436000,"f36":"00000000-0000-0000-0000-000000000000","f37":"cedcd714-f0a6-11e8-9f47-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},null]}`

  // Create a new struct from the source JSON string
  let struct1 = prototest.StructArray.fromJSON(json)

  // Serialize the struct to the JSON string
  json = JSON.stringify(struct1)

  // Check the serialized JSON size
  t.true(json.length > 0)

  // Deserialize the struct from the JSON string
  let struct2 = prototest.StructArray.fromJSON(json)

  t.equal(struct2.f1.length, 2)
  t.equal(struct2.f1[0], 48)
  t.equal(struct2.f1[1], 65)
  t.equal(struct2.f2.length, 2)
  t.equal(struct2.f2[0], 97)
  t.equal(struct2.f2[1], undefined)
  t.equal(struct2.f3.length, 2)
  t.equal(struct2.f3[0].length, 3)
  t.equal(struct2.f3[0][0], 48)
  t.equal(struct2.f3[0][1], 48)
  t.equal(struct2.f3[0][2], 48)
  t.equal(struct2.f3[1].length, 3)
  t.equal(struct2.f3[1][0], 65)
  t.equal(struct2.f3[1][1], 65)
  t.equal(struct2.f3[1][2], 65)
  t.equal(struct2.f4.length, 2)
  t.notEqual(struct2.f4[0], undefined)
  t.equal(struct2.f4[0].length, 3)
  t.equal(struct2.f4[0][0], 97)
  t.equal(struct2.f4[0][1], 97)
  t.equal(struct2.f4[0][2], 97)
  t.equal(struct2.f4[1], undefined)
  t.equal(struct2.f5.length, 2)
  t.true(struct2.f5[0].eq(prototest.EnumSimple.ENUM_VALUE_1))
  t.true(struct2.f5[1].eq(prototest.EnumSimple.ENUM_VALUE_2))
  t.equal(struct2.f6.length, 2)
  t.true(struct2.f6[0].eq(prototest.EnumSimple.ENUM_VALUE_1))
  t.equal(struct2.f6[1], undefined)
  t.equal(struct2.f7.length, 2)
  t.true(struct2.f7[0].hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2))
  t.true(struct2.f7[1].hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2 | prototest.FlagsSimple.FLAG_VALUE_3))
  t.equal(struct2.f8.length, 2)
  t.true(struct2.f8[0].hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2))
  t.equal(struct2.f8[1], undefined)
  t.equal(struct2.f9.length, 2)
  t.equal(struct2.f9[0].f2, true)
  t.equal(struct2.f9[0].f12, 0xFF)
  t.equal(struct2.f9[0].f32, 'Initial string!')
  t.equal(struct2.f9[1].f2, true)
  t.equal(struct2.f9[1].f12, 0xFF)
  t.equal(struct2.f9[1].f32, 'Initial string!')
  t.equal(struct2.f10.length, 2)
  t.notEqual(struct2.f10[0], undefined)
  t.equal(struct2.f10[0].f2, true)
  t.equal(struct2.f10[0].f12, 0xFF)
  t.equal(struct2.f10[0].f32, 'Initial string!')
  t.equal(struct2.f10[1], undefined)
  t.end()
})

test('Serialization (JSON): struct vector', function (t) {
  // Define a source JSON string
  let json = String.raw`{"f1":[48,65],"f2":[97,null],"f3":["MDAw\\n","QUFB\\n"],"f4":["YWFh\\n",null],"f5":[1,2],"f6":[1,null],"f7":[3,7],"f8":[3,null],"f9":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146157127964000,"f36":"00000000-0000-0000-0000-000000000000","f37":"34d38702-f0a7-11e8-b30e-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146157128572000,"f36":"00000000-0000-0000-0000-000000000000","f37":"34d39c88-f0a7-11e8-b30e-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}],"f10":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146157129063000,"f36":"00000000-0000-0000-0000-000000000000","f37":"34d3b038-f0a7-11e8-b30e-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},null]}`

  // Create a new struct from the source JSON string
  let struct1 = prototest.StructVector.fromJSON(json)

  // Serialize the struct to the JSON string
  json = JSON.stringify(struct1)

  // Check the serialized JSON size
  t.true(json.length > 0)

  // Deserialize the struct from the JSON string
  let struct2 = prototest.StructVector.fromJSON(json)

  t.equal(struct2.f1.length, 2)
  t.equal(struct2.f1[0], 48)
  t.equal(struct2.f1[1], 65)
  t.equal(struct2.f2.length, 2)
  t.equal(struct2.f2[0], 97)
  t.equal(struct2.f2[1], undefined)
  t.equal(struct2.f3.length, 2)
  t.equal(struct2.f3[0].length, 3)
  t.equal(struct2.f3[0][0], 48)
  t.equal(struct2.f3[0][1], 48)
  t.equal(struct2.f3[0][2], 48)
  t.equal(struct2.f3[1].length, 3)
  t.equal(struct2.f3[1][0], 65)
  t.equal(struct2.f3[1][1], 65)
  t.equal(struct2.f3[1][2], 65)
  t.equal(struct2.f4.length, 2)
  t.notEqual(struct2.f4[0], undefined)
  t.equal(struct2.f4[0].length, 3)
  t.equal(struct2.f4[0][0], 97)
  t.equal(struct2.f4[0][1], 97)
  t.equal(struct2.f4[0][2], 97)
  t.equal(struct2.f4[1], undefined)
  t.equal(struct2.f5.length, 2)
  t.true(struct2.f5[0].eq(prototest.EnumSimple.ENUM_VALUE_1))
  t.true(struct2.f5[1].eq(prototest.EnumSimple.ENUM_VALUE_2))
  t.equal(struct2.f6.length, 2)
  t.true(struct2.f6[0].eq(prototest.EnumSimple.ENUM_VALUE_1))
  t.equal(struct2.f6[1], undefined)
  t.equal(struct2.f7.length, 2)
  t.true(struct2.f7[0].hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2))
  t.true(struct2.f7[1].hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2 | prototest.FlagsSimple.FLAG_VALUE_3))
  t.equal(struct2.f8.length, 2)
  t.true(struct2.f8[0].hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2))
  t.equal(struct2.f8[1], undefined)
  t.equal(struct2.f9.length, 2)
  t.equal(struct2.f9[0].f2, true)
  t.equal(struct2.f9[0].f12, 0xFF)
  t.equal(struct2.f9[0].f32, 'Initial string!')
  t.equal(struct2.f9[1].f2, true)
  t.equal(struct2.f9[1].f12, 0xFF)
  t.equal(struct2.f9[1].f32, 'Initial string!')
  t.equal(struct2.f10.length, 2)
  t.notEqual(struct2.f10[0], undefined)
  t.equal(struct2.f10[0].f2, true)
  t.equal(struct2.f10[0].f12, 0xFF)
  t.equal(struct2.f10[0].f32, 'Initial string!')
  t.equal(struct2.f10[1], undefined)
  t.end()
})

test('Serialization (JSON): struct list', function (t) {
  // Define a source JSON string
  let json = String.raw`{"f1":[48,65],"f2":[97,null],"f3":["MDAw\\n","QUFB\\n"],"f4":["YWFh\\n",null],"f5":[1,2],"f6":[1,null],"f7":[3,7],"f8":[3,null],"f9":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146220253760000,"f36":"00000000-0000-0000-0000-000000000000","f37":"5a73e7fe-f0a7-11e8-89e6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146220255725000,"f36":"00000000-0000-0000-0000-000000000000","f37":"5a741990-f0a7-11e8-89e6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}],"f10":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146220256802000,"f36":"00000000-0000-0000-0000-000000000000","f37":"5a74e4b0-f0a7-11e8-89e6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},null]}`

  // Create a new struct from the source JSON string
  let struct1 = prototest.StructList.fromJSON(json)

  // Serialize the struct to the JSON string
  json = JSON.stringify(struct1)

  // Check the serialized JSON size
  t.true(json.length > 0)

  // Deserialize the struct from the JSON string
  let struct2 = prototest.StructList.fromJSON(json)

  t.equal(struct2.f1.length, 2)
  t.equal(struct2.f1[0], 48)
  t.equal(struct2.f1[1], 65)
  t.equal(struct2.f2.length, 2)
  t.equal(struct2.f2[0], 97)
  t.equal(struct2.f2[1], undefined)
  t.equal(struct2.f3.length, 2)
  t.equal(struct2.f3[0].length, 3)
  t.equal(struct2.f3[0][0], 48)
  t.equal(struct2.f3[0][1], 48)
  t.equal(struct2.f3[0][2], 48)
  t.equal(struct2.f3[1].length, 3)
  t.equal(struct2.f3[1][0], 65)
  t.equal(struct2.f3[1][1], 65)
  t.equal(struct2.f3[1][2], 65)
  t.equal(struct2.f4.length, 2)
  t.notEqual(struct2.f4[0], undefined)
  t.equal(struct2.f4[0].length, 3)
  t.equal(struct2.f4[0][0], 97)
  t.equal(struct2.f4[0][1], 97)
  t.equal(struct2.f4[0][2], 97)
  t.equal(struct2.f4[1], undefined)
  t.equal(struct2.f5.length, 2)
  t.true(struct2.f5[0].eq(prototest.EnumSimple.ENUM_VALUE_1))
  t.true(struct2.f5[1].eq(prototest.EnumSimple.ENUM_VALUE_2))
  t.equal(struct2.f6.length, 2)
  t.true(struct2.f6[0].eq(prototest.EnumSimple.ENUM_VALUE_1))
  t.equal(struct2.f6[1], undefined)
  t.equal(struct2.f7.length, 2)
  t.true(struct2.f7[0].hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2))
  t.true(struct2.f7[1].hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2 | prototest.FlagsSimple.FLAG_VALUE_3))
  t.equal(struct2.f8.length, 2)
  t.true(struct2.f8[0].hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2))
  t.equal(struct2.f8[1], undefined)
  t.equal(struct2.f9.length, 2)
  t.equal(struct2.f9[0].f2, true)
  t.equal(struct2.f9[0].f12, 0xFF)
  t.equal(struct2.f9[0].f32, 'Initial string!')
  t.equal(struct2.f9[1].f2, true)
  t.equal(struct2.f9[1].f12, 0xFF)
  t.equal(struct2.f9[1].f32, 'Initial string!')
  t.equal(struct2.f10.length, 2)
  t.notEqual(struct2.f10[0], undefined)
  t.equal(struct2.f10[0].f2, true)
  t.equal(struct2.f10[0].f12, 0xFF)
  t.equal(struct2.f10[0].f32, 'Initial string!')
  t.equal(struct2.f10[1], undefined)
  t.end()
})

test('Serialization (JSON): struct set', function (t) {
  // Define a source JSON string
  let json = String.raw`{"f1":[48,65,97],"f2":[1,2],"f3":[3,7],"f4":[{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146299848353000,"f36":"00000000-0000-0000-0000-000000000000","f37":"89e4edd0-f0a7-11e8-9dde-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":65,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146299848966000,"f36":"00000000-0000-0000-0000-000000000000","f37":"89e503f6-f0a7-11e8-9dde-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}]}`

  // Create a new struct from the source JSON string
  let struct1 = prototest.StructSet.fromJSON(json)

  // Serialize the struct to the JSON string
  json = JSON.stringify(struct1)

  // Check the serialized JSON size
  t.true(json.length > 0)

  // Deserialize the struct from the JSON string
  let struct2 = prototest.StructSet.fromJSON(json)

  t.equal(struct2.f1.size, 3)
  t.true(struct2.f1.has(48))
  t.true(struct2.f1.has(65))
  t.true(struct2.f1.has(97))
  t.equal(struct2.f2.size, 2)
  // t.true(struct2.f2.has(prototest.EnumSimple.ENUM_VALUE_1))
  // t.true(struct2.f2.has(prototest.EnumSimple.ENUM_VALUE_2))
  t.equal(struct2.f3.size, 2)
  // t.true(struct2.f3.has(prototest.FlagsSimple.fromFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2)))
  // t.true(struct2.f3.has(prototest.FlagsSimple.fromFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2 | prototest.FlagsSimple.FLAG_VALUE_3)))
  t.equal(struct2.f4.size, 2)
  // let s1 = new prototest.StructSimple()
  // s1.id = 48
  // t.true(struct2.f4.has(s1))
  // let s2 = new prototest.StructSimple()
  // s2.id = 65
  // t.true(struct2.f4.has(s2))
  t.end()
})

test('Serialization (JSON): struct map', function (t) {
  // Define a source JSON string
  let json = String.raw`{"f1":{"10":48,"20":65},"f2":{"10":97,"20":null},"f3":{"10":"MDAw\\n","20":"QUFB\\n"},"f4":{"10":"YWFh\\n","20":null},"f5":{"10":1,"20":2},"f6":{"10":1,"20":null},"f7":{"10":3,"20":7},"f8":{"10":3,"20":null},"f9":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146345803483000,"f36":"00000000-0000-0000-0000-000000000000","f37":"a549215e-f0a7-11e8-90f6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":{"id":65,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146345804184000,"f36":"00000000-0000-0000-0000-000000000000","f37":"a54942ce-f0a7-11e8-90f6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}},"f10":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146345803483000,"f36":"00000000-0000-0000-0000-000000000000","f37":"a549215e-f0a7-11e8-90f6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":null}}`

  // Create a new struct from the source JSON string
  let struct1 = prototest.StructMap.fromJSON(json)

  // Serialize the struct to the JSON string
  json = JSON.stringify(struct1)

  // Check the serialized JSON size
  t.true(json.length > 0)

  // Deserialize the struct from the JSON string
  let struct2 = prototest.StructMap.fromJSON(json)

  t.equal(struct2.f1.size, 2)
  t.equal(struct2.f1.get('10'), 48)
  t.equal(struct2.f1.get('20'), 65)
  t.equal(struct2.f2.size, 2)
  t.equal(struct2.f2.get('10'), 97)
  t.equal(struct2.f2.get('20'), undefined)
  t.equal(struct2.f3.size, 2)
  t.equal(struct2.f3.get('10').length, 3)
  t.equal(struct2.f3.get('20').length, 3)
  t.equal(struct2.f4.size, 2)
  t.equal(struct2.f4.get('10').length, 3)
  t.equal(struct2.f4.get('20'), undefined)
  t.equal(struct2.f5.size, 2)
  t.true(struct2.f5.get('10').eq(prototest.EnumSimple.ENUM_VALUE_1))
  t.true(struct2.f5.get('20').eq(prototest.EnumSimple.ENUM_VALUE_2))
  t.equal(struct2.f6.size, 2)
  t.true(struct2.f6.get('10').eq(prototest.EnumSimple.ENUM_VALUE_1))
  t.equal(struct2.f6.get('20'), undefined)
  t.equal(struct2.f7.size, 2)
  t.true(struct2.f7.get('10').hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2))
  t.true(struct2.f7.get('20').hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2 | prototest.FlagsSimple.FLAG_VALUE_3))
  t.equal(struct2.f8.size, 2)
  t.true(struct2.f8.get('10').hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2))
  t.equal(struct2.f8.get('20'), undefined)
  t.equal(struct2.f9.size, 2)
  t.equal(struct2.f9.get('10').id, 48)
  t.equal(struct2.f9.get('20').id, 65)
  t.equal(struct2.f10.size, 2)
  t.equal(struct2.f10.get('10').id, 48)
  t.equal(struct2.f10.get('20'), undefined)
  t.end()
})

test('Serialization (JSON): struct hash', function (t) {
  // Define a source JSON string
  let json = String.raw`{"f1":{"10":48,"20":65},"f2":{"10":97,"20":null},"f3":{"10":"MDAw\\n","20":"QUFB\\n"},"f4":{"10":"YWFh\\n","20":null},"f5":{"10":1,"20":2},"f6":{"10":1,"20":null},"f7":{"10":3,"20":7},"f8":{"10":3,"20":null},"f9":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146381450913000,"f36":"00000000-0000-0000-0000-000000000000","f37":"ba8885d2-f0a7-11e8-81fa-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":{"id":65,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146381452825000,"f36":"00000000-0000-0000-0000-000000000000","f37":"ba88ced4-f0a7-11e8-81fa-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}},"f10":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146381450913000,"f36":"00000000-0000-0000-0000-000000000000","f37":"ba8885d2-f0a7-11e8-81fa-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":null}}`

  // Create a new struct from the source JSON string
  let struct1 = prototest.StructHash.fromJSON(json)

  // Serialize the struct to the JSON string
  json = JSON.stringify(struct1)

  // Check the serialized JSON size
  t.true(json.length > 0)

  // Deserialize the struct from the JSON string
  let struct2 = prototest.StructHash.fromJSON(json)

  t.equal(struct2.f1.size, 2)
  t.equal(struct2.f1.get('10'), 48)
  t.equal(struct2.f1.get('20'), 65)
  t.equal(struct2.f2.size, 2)
  t.equal(struct2.f2.get('10'), 97)
  t.equal(struct2.f2.get('20'), undefined)
  t.equal(struct2.f3.size, 2)
  t.equal(struct2.f3.get('10').length, 3)
  t.equal(struct2.f3.get('20').length, 3)
  t.equal(struct2.f4.size, 2)
  t.equal(struct2.f4.get('10').length, 3)
  t.equal(struct2.f4.get('20'), undefined)
  t.equal(struct2.f5.size, 2)
  t.true(struct2.f5.get('10').eq(prototest.EnumSimple.ENUM_VALUE_1))
  t.true(struct2.f5.get('20').eq(prototest.EnumSimple.ENUM_VALUE_2))
  t.equal(struct2.f6.size, 2)
  t.true(struct2.f6.get('10').eq(prototest.EnumSimple.ENUM_VALUE_1))
  t.equal(struct2.f6.get('20'), undefined)
  t.equal(struct2.f7.size, 2)
  t.true(struct2.f7.get('10').hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2))
  t.true(struct2.f7.get('20').hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2 | prototest.FlagsSimple.FLAG_VALUE_3))
  t.equal(struct2.f8.size, 2)
  t.true(struct2.f8.get('10').hasFlags(prototest.FlagsSimple.FLAG_VALUE_1 | prototest.FlagsSimple.FLAG_VALUE_2))
  t.equal(struct2.f8.get('20'), undefined)
  t.equal(struct2.f9.size, 2)
  t.equal(struct2.f9.get('10').id, 48)
  t.equal(struct2.f9.get('20').id, 65)
  t.equal(struct2.f10.size, 2)
  t.equal(struct2.f10.get('10').id, 48)
  t.equal(struct2.f10.get('20'), undefined)
  t.end()
})
