package tests

import "testing"
import "github.com/stretchr/testify/assert"
import "../proto/fbe"
import "../proto/proto"
import "../proto/test"

func TestSerializationJsonDomain(t *testing.T) {
	// Define a source JSON string
	json := []byte(`{"id":1,"name":"Test","state":6,"wallet":{"currency":"USD","amount":1000.0},"asset":{"currency":"EUR","amount":100.0},"orders":[{"id":1,"symbol":"EURUSD","side":0,"type":0,"price":1.23456,"volume":1000.0},{"id":2,"symbol":"EURUSD","side":1,"type":1,"price":1.0,"volume":100.0},{"id":3,"symbol":"EURUSD","side":0,"type":2,"price":1.5,"volume":10.0}]}`)

	// Create a new account from the source JSON string
	account1, _ := proto.NewAccountFromJSON(json)

	// Serialize the account to the JSON string
	json, _ = account1.JSON()

	// Check the serialized JSON size
	assert.NotEmpty(t, json)

	// Deserialize the account from the JSON string
	account2, _ := proto.NewAccountFromJSON(json)

	assert.EqualValues(t, account2.Id, 1)
	assert.EqualValues(t, account2.Name, "Test")
	assert.True(t, account2.State.HasFlags(proto.State_good))
	assert.EqualValues(t, account2.Wallet.Currency, "USD")
	assert.EqualValues(t, account2.Wallet.Amount, 1000.0)
	assert.NotNil(t, account2.Asset)
	assert.EqualValues(t, account2.Asset.Currency, "EUR")
	assert.EqualValues(t, account2.Asset.Amount, 100.0)
	assert.EqualValues(t, len(account2.Orders), 3)
	assert.EqualValues(t, account2.Orders[0].Id, 1)
	assert.EqualValues(t, account2.Orders[0].Symbol, "EURUSD")
	assert.EqualValues(t, account2.Orders[0].Side, proto.OrderSide_buy)
	assert.EqualValues(t, account2.Orders[0].Type, proto.OrderType_market)
	assert.EqualValues(t, account2.Orders[0].Price, 1.23456)
	assert.EqualValues(t, account2.Orders[0].Volume, 1000.0)
	assert.EqualValues(t, account2.Orders[1].Id, 2)
	assert.EqualValues(t, account2.Orders[1].Symbol, "EURUSD")
	assert.EqualValues(t, account2.Orders[1].Side, proto.OrderSide_sell)
	assert.EqualValues(t, account2.Orders[1].Type, proto.OrderType_limit)
	assert.EqualValues(t, account2.Orders[1].Price, 1.0)
	assert.EqualValues(t, account2.Orders[1].Volume, 100.0)
	assert.EqualValues(t, account2.Orders[2].Id, 3)
	assert.EqualValues(t, account2.Orders[2].Symbol, "EURUSD")
	assert.EqualValues(t, account2.Orders[2].Side, proto.OrderSide_buy)
	assert.EqualValues(t, account2.Orders[2].Type, proto.OrderType_stop)
	assert.EqualValues(t, account2.Orders[2].Price, 1.5)
	assert.EqualValues(t, account2.Orders[2].Volume, 10.0)
}

func TestSerializationJsonStructSimple(t *testing.T) {
	// Define a source JSON string
	json := []byte(`{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145597933463000,"f36":"00000000-0000-0000-0000-000000000000","f37":"e7854072-f0a5-11e8-8f69-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}`)

	// Create a new struct from the source JSON string
	struct1, _ := test.NewStructSimpleFromJSON(json)

	// Serialize the struct to the JSON string
	json, _ = struct1.JSON()

	// Check the serialized JSON size
	assert.NotEmpty(t, json)

	// Deserialize the struct from the JSON string
	struct2, _ := test.NewStructSimpleFromJSON(json)

	assert.EqualValues(t, struct2.F1, false)
	assert.EqualValues(t, struct2.F2, true)
	assert.EqualValues(t, struct2.F3, 0)
	assert.EqualValues(t, struct2.F4, 0xFF)
	assert.EqualValues(t, struct2.F5, '\000')
	assert.EqualValues(t, struct2.F6, '!')
	assert.EqualValues(t, struct2.F7, rune(0))
	assert.EqualValues(t, struct2.F8, rune(0x0444))
	assert.EqualValues(t, struct2.F9, 0)
	assert.EqualValues(t, struct2.F10, 127)
	assert.EqualValues(t, struct2.F11, 0)
	assert.EqualValues(t, struct2.F12, 0xFF)
	assert.EqualValues(t, struct2.F13, 0)
	assert.EqualValues(t, struct2.F14, 32767)
	assert.EqualValues(t, struct2.F15, 0)
	assert.EqualValues(t, struct2.F16, 0xFFFF)
	assert.EqualValues(t, struct2.F17, 0)
	assert.EqualValues(t, struct2.F18, 2147483647)
	assert.EqualValues(t, struct2.F19, 0)
	assert.EqualValues(t, struct2.F20, 0xFFFFFFFF)
	assert.EqualValues(t, struct2.F21, 0)
	assert.EqualValues(t, struct2.F22, int64(9223372036854775807))
	assert.EqualValues(t, struct2.F23, 0)
	assert.EqualValues(t, struct2.F24, uint64(0xFFFFFFFFFFFFFFFF))
	assert.EqualValues(t, struct2.F25, 0.0)
	assert.InEpsilon(t, struct2.F26, 123.456, 0.0001)
	assert.EqualValues(t, struct2.F27, 0.0)
	assert.InEpsilon(t, struct2.F28, -123.567e+123, 1e+123)
	assert.EqualValues(t, struct2.F29, fbe.DecimalFromString("0"))
	assert.EqualValues(t, struct2.F30, fbe.DecimalFromString("123456.123456"))
	assert.EqualValues(t, struct2.F31, "")
	assert.EqualValues(t, struct2.F32, "Initial string!")
	assert.EqualValues(t, struct2.F33, fbe.TimestampEpoch())
	assert.EqualValues(t, struct2.F34, fbe.TimestampEpoch())
	assert.True(t, struct2.F35.UnixNano() > fbe.TimestampFromDate(2018, 1, 1).UnixNano())
	assert.EqualValues(t, struct2.F36, fbe.UUIDNil())
	assert.NotEqual(t, struct2.F37, fbe.UUIDNil())
	assert.EqualValues(t, struct2.F38, fbe.UUIDFromString("123e4567-e89b-12d3-a456-426655440000"))

	assert.EqualValues(t, struct2.F1, struct1.F1)
	assert.EqualValues(t, struct2.F2, struct1.F2)
	assert.EqualValues(t, struct2.F3, struct1.F3)
	assert.EqualValues(t, struct2.F4, struct1.F4)
	assert.EqualValues(t, struct2.F5, struct1.F5)
	assert.EqualValues(t, struct2.F6, struct1.F6)
	assert.EqualValues(t, struct2.F7, struct1.F7)
	assert.EqualValues(t, struct2.F8, struct1.F8)
	assert.EqualValues(t, struct2.F9, struct1.F9)
	assert.EqualValues(t, struct2.F10, struct1.F10)
	assert.EqualValues(t, struct2.F11, struct1.F11)
	assert.EqualValues(t, struct2.F12, struct1.F12)
	assert.EqualValues(t, struct2.F13, struct1.F13)
	assert.EqualValues(t, struct2.F14, struct1.F14)
	assert.EqualValues(t, struct2.F15, struct1.F15)
	assert.EqualValues(t, struct2.F16, struct1.F16)
	assert.EqualValues(t, struct2.F17, struct1.F17)
	assert.EqualValues(t, struct2.F18, struct1.F18)
	assert.EqualValues(t, struct2.F19, struct1.F19)
	assert.EqualValues(t, struct2.F20, struct1.F20)
	assert.EqualValues(t, struct2.F21, struct1.F21)
	assert.EqualValues(t, struct2.F22, struct1.F22)
	assert.EqualValues(t, struct2.F23, struct1.F23)
	assert.EqualValues(t, struct2.F24, struct1.F24)
	assert.EqualValues(t, struct2.F25, struct1.F25)
	assert.InEpsilon(t, struct2.F26, struct1.F26, 0.0001)
	assert.EqualValues(t, struct2.F27, struct1.F27)
	assert.InEpsilon(t, struct2.F28, struct1.F28, 1e+123)
	assert.EqualValues(t, struct2.F29, struct1.F29)
	assert.EqualValues(t, struct2.F30, struct1.F30)
	assert.EqualValues(t, struct2.F31, struct1.F31)
	assert.EqualValues(t, struct2.F32, struct1.F32)
	assert.EqualValues(t, struct2.F33, struct1.F33)
	assert.EqualValues(t, struct2.F34, struct1.F34)
	assert.EqualValues(t, struct2.F35, struct1.F35)
	assert.EqualValues(t, struct2.F36, struct1.F36)
	assert.EqualValues(t, struct2.F37, struct1.F37)
	assert.EqualValues(t, struct2.F38, struct1.F38)
	assert.EqualValues(t, struct2.F39, struct1.F39)
	assert.EqualValues(t, struct2.F40, struct1.F40)
}

func TestSerializationJsonStructOptional(t *testing.T) {
	// Define a source JSON string
	json := []byte(`{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145860677797000,"f36":"00000000-0000-0000-0000-000000000000","f37":"8420d1c6-f0a6-11e8-80fc-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]},"f100":null,"f101":true,"f102":null,"f103":null,"f104":255,"f105":null,"f106":null,"f107":33,"f108":null,"f109":null,"f110":1092,"f111":null,"f112":null,"f113":127,"f114":null,"f115":null,"f116":255,"f117":null,"f118":null,"f119":32767,"f120":null,"f121":null,"f122":65535,"f123":null,"f124":null,"f125":2147483647,"f126":null,"f127":null,"f128":4294967295,"f129":null,"f130":null,"f131":9223372036854775807,"f132":null,"f133":null,"f134":18446744073709551615,"f135":null,"f136":null,"f137":123.456,"f138":null,"f139":null,"f140":-1.23456e+125,"f141":null,"f142":null,"f143":"123456.123456","f144":null,"f145":null,"f146":"Initial string!","f147":null,"f148":null,"f149":1543145860678429000,"f150":null,"f151":null,"f152":"123e4567-e89b-12d3-a456-426655440000","f153":null,"f154":null,"f155":null,"f156":null,"f157":null,"f158":null,"f159":null,"f160":null,"f161":null,"f162":null,"f163":null,"f164":null,"f165":null}`)

	// Create a new struct from the source JSON string
	struct1, _ := test.NewStructOptionalFromJSON(json)

	// Serialize the struct to the JSON string
	json, _ = struct1.JSON()

	// Check the serialized JSON size
	assert.NotEmpty(t, json)

	// Deserialize the struct from the JSON string
	struct2, _ := test.NewStructOptionalFromJSON(json)

	assert.EqualValues(t, struct2.F1, false)
	assert.EqualValues(t, struct2.F2, true)
	assert.EqualValues(t, struct2.F3, 0)
	assert.EqualValues(t, struct2.F4, 0xFF)
	assert.EqualValues(t, struct2.F5, '\000')
	assert.EqualValues(t, struct2.F6, '!')
	assert.EqualValues(t, struct2.F7, rune(0))
	assert.EqualValues(t, struct2.F8, rune(0x0444))
	assert.EqualValues(t, struct2.F9, 0)
	assert.EqualValues(t, struct2.F10, 127)
	assert.EqualValues(t, struct2.F11, 0)
	assert.EqualValues(t, struct2.F12, 0xFF)
	assert.EqualValues(t, struct2.F13, 0)
	assert.EqualValues(t, struct2.F14, 32767)
	assert.EqualValues(t, struct2.F15, 0)
	assert.EqualValues(t, struct2.F16, 0xFFFF)
	assert.EqualValues(t, struct2.F17, 0)
	assert.EqualValues(t, struct2.F18, 2147483647)
	assert.EqualValues(t, struct2.F19, 0)
	assert.EqualValues(t, struct2.F20, 0xFFFFFFFF)
	assert.EqualValues(t, struct2.F21, 0)
	assert.EqualValues(t, struct2.F22, int64(9223372036854775807))
	assert.EqualValues(t, struct2.F23, 0)
	assert.EqualValues(t, struct2.F24, uint64(0xFFFFFFFFFFFFFFFF))
	assert.EqualValues(t, struct2.F25, 0.0)
	assert.InEpsilon(t, struct2.F26, 123.456, 0.0001)
	assert.EqualValues(t, struct2.F27, 0.0)
	assert.InEpsilon(t, struct2.F28, -123.567e+123, 1e+123)
	assert.EqualValues(t, struct2.F29, fbe.DecimalFromString("0"))
	assert.EqualValues(t, struct2.F30, fbe.DecimalFromString("123456.123456"))
	assert.EqualValues(t, struct2.F31, "")
	assert.EqualValues(t, struct2.F32, "Initial string!")
	assert.EqualValues(t, struct2.F33, fbe.TimestampEpoch())
	assert.EqualValues(t, struct2.F34, fbe.TimestampEpoch())
	assert.True(t, struct2.F35.UnixNano() > fbe.TimestampFromDate(2018, 1, 1).UnixNano())
	assert.EqualValues(t, struct2.F36, fbe.UUIDNil())
	assert.NotEqual(t, struct2.F37, fbe.UUIDNil())
	assert.EqualValues(t, struct2.F38, fbe.UUIDFromString("123e4567-e89b-12d3-a456-426655440000"))

	assert.Nil(t, struct2.F100)
	assert.NotNil(t, struct2.F101)
	assert.EqualValues(t, *struct2.F101, true)
	assert.Nil(t, struct2.F102)
	assert.Nil(t, struct2.F103)
	assert.NotNil(t, struct2.F104)
	assert.EqualValues(t, *struct2.F104, 0xFF)
	assert.Nil(t, struct2.F105)
	assert.Nil(t, struct2.F106)
	assert.NotNil(t, struct2.F107)
	assert.EqualValues(t, *struct2.F107, '!')
	assert.Nil(t, struct2.F108)
	assert.Nil(t, struct2.F109)
	assert.NotNil(t, struct2.F110)
	assert.EqualValues(t, *struct2.F110, rune(0x0444))
	assert.Nil(t, struct2.F111)
	assert.Nil(t, struct2.F112)
	assert.NotNil(t, struct2.F113)
	assert.EqualValues(t, *struct2.F113, 127)
	assert.Nil(t, struct2.F114)
	assert.Nil(t, struct2.F115)
	assert.NotNil(t, struct2.F116)
	assert.EqualValues(t, *struct2.F116, 0xFF)
	assert.Nil(t, struct2.F117)
	assert.Nil(t, struct2.F118)
	assert.NotNil(t, struct2.F119)
	assert.EqualValues(t, *struct2.F119, 32767)
	assert.Nil(t, struct2.F120)
	assert.Nil(t, struct2.F121)
	assert.NotNil(t, struct2.F122)
	assert.EqualValues(t, *struct2.F122, 0xFFFF)
	assert.Nil(t, struct2.F123)
	assert.Nil(t, struct2.F124)
	assert.NotNil(t, struct2.F125)
	assert.EqualValues(t, *struct2.F125, 2147483647)
	assert.Nil(t, struct2.F126)
	assert.Nil(t, struct2.F127)
	assert.NotNil(t, struct2.F128)
	assert.EqualValues(t, *struct2.F128, 0xFFFFFFFF)
	assert.Nil(t, struct2.F129)
	assert.Nil(t, struct2.F130)
	assert.NotNil(t, struct2.F131)
	assert.EqualValues(t, *struct2.F131, 9223372036854775807)
	assert.Nil(t, struct2.F132)
	assert.Nil(t, struct2.F133)
	assert.NotNil(t, struct2.F131)
	assert.EqualValues(t, *struct2.F134, uint64(0xFFFFFFFFFFFFFFFF))
	assert.Nil(t, struct2.F135)
	assert.Nil(t, struct2.F136)
	assert.NotNil(t, struct2.F137)
	assert.InEpsilon(t, *struct2.F137, 123.456, 0.0001)
	assert.Nil(t, struct2.F138)
	assert.Nil(t, struct2.F139)
	assert.NotNil(t, struct2.F140)
	assert.InEpsilon(t, *struct2.F140, -123.567e+123, 1e+123)
	assert.Nil(t, struct2.F141)
	assert.Nil(t, struct2.F142)
	assert.NotNil(t, struct2.F143)
	assert.EqualValues(t, *struct2.F143, fbe.DecimalFromString("123456.123456"))
	assert.Nil(t, struct2.F144)
	assert.Nil(t, struct2.F145)
	assert.NotNil(t, struct2.F146)
	assert.EqualValues(t, *struct2.F146, "Initial string!")
	assert.Nil(t, struct2.F147)
	assert.Nil(t, struct2.F148)
	assert.NotNil(t, struct2.F149)
	assert.True(t, struct2.F149.UnixNano() > fbe.TimestampFromDate(2018, 1, 1).UnixNano())
	assert.Nil(t, struct2.F150)
	assert.Nil(t, struct2.F151)
	assert.NotNil(t, struct2.F152)
	assert.EqualValues(t, *struct2.F152, fbe.UUIDFromString("123e4567-e89b-12d3-a456-426655440000"))
	assert.Nil(t, struct2.F153)
	assert.Nil(t, struct2.F154)
	assert.Nil(t, struct2.F155)
	assert.Nil(t, struct2.F156)
	assert.Nil(t, struct2.F157)
	assert.Nil(t, struct2.F158)
	assert.Nil(t, struct2.F159)
	assert.Nil(t, struct2.F160)
	assert.Nil(t, struct2.F161)
	assert.Nil(t, struct2.F162)
	assert.Nil(t, struct2.F163)
	assert.Nil(t, struct2.F164)
	assert.Nil(t, struct2.F165)

	assert.EqualValues(t, struct2.F1, struct1.F1)
	assert.EqualValues(t, struct2.F2, struct1.F2)
	assert.EqualValues(t, struct2.F3, struct1.F3)
	assert.EqualValues(t, struct2.F4, struct1.F4)
	assert.EqualValues(t, struct2.F5, struct1.F5)
	assert.EqualValues(t, struct2.F6, struct1.F6)
	assert.EqualValues(t, struct2.F7, struct1.F7)
	assert.EqualValues(t, struct2.F8, struct1.F8)
	assert.EqualValues(t, struct2.F9, struct1.F9)
	assert.EqualValues(t, struct2.F10, struct1.F10)
	assert.EqualValues(t, struct2.F11, struct1.F11)
	assert.EqualValues(t, struct2.F12, struct1.F12)
	assert.EqualValues(t, struct2.F13, struct1.F13)
	assert.EqualValues(t, struct2.F14, struct1.F14)
	assert.EqualValues(t, struct2.F15, struct1.F15)
	assert.EqualValues(t, struct2.F16, struct1.F16)
	assert.EqualValues(t, struct2.F17, struct1.F17)
	assert.EqualValues(t, struct2.F18, struct1.F18)
	assert.EqualValues(t, struct2.F19, struct1.F19)
	assert.EqualValues(t, struct2.F20, struct1.F20)
	assert.EqualValues(t, struct2.F21, struct1.F21)
	assert.EqualValues(t, struct2.F22, struct1.F22)
	assert.EqualValues(t, struct2.F23, struct1.F23)
	assert.EqualValues(t, struct2.F24, struct1.F24)
	assert.EqualValues(t, struct2.F25, struct1.F25)
	assert.InEpsilon(t, struct2.F26, struct1.F26, 0.0001)
	assert.EqualValues(t, struct2.F27, struct1.F27)
	assert.InEpsilon(t, struct2.F28, struct1.F28, 1e+123)
	assert.EqualValues(t, struct2.F29, struct1.F29)
	assert.EqualValues(t, struct2.F30, struct1.F30)
	assert.EqualValues(t, struct2.F31, struct1.F31)
	assert.EqualValues(t, struct2.F32, struct1.F32)
	assert.EqualValues(t, struct2.F33, struct1.F33)
	assert.EqualValues(t, struct2.F34, struct1.F34)
	assert.EqualValues(t, struct2.F35, struct1.F35)
	assert.EqualValues(t, struct2.F36, struct1.F36)
	assert.EqualValues(t, struct2.F37, struct1.F37)
	assert.EqualValues(t, struct2.F38, struct1.F38)
	assert.EqualValues(t, struct2.F39, struct1.F39)
	assert.EqualValues(t, struct2.F40, struct1.F40)

	assert.EqualValues(t, struct2.F100, struct1.F100)
	assert.EqualValues(t, struct2.F101, struct1.F101)
	assert.EqualValues(t, struct2.F102, struct1.F102)
	assert.EqualValues(t, struct2.F103, struct1.F103)
	assert.EqualValues(t, struct2.F104, struct1.F104)
	assert.EqualValues(t, struct2.F105, struct1.F105)
	assert.EqualValues(t, struct2.F106, struct1.F106)
	assert.EqualValues(t, struct2.F107, struct1.F107)
	assert.EqualValues(t, struct2.F108, struct1.F108)
	assert.EqualValues(t, struct2.F109, struct1.F109)
	assert.EqualValues(t, struct2.F110, struct1.F110)
	assert.EqualValues(t, struct2.F111, struct1.F111)
	assert.EqualValues(t, struct2.F112, struct1.F112)
	assert.EqualValues(t, struct2.F113, struct1.F113)
	assert.EqualValues(t, struct2.F114, struct1.F114)
	assert.EqualValues(t, struct2.F115, struct1.F115)
	assert.EqualValues(t, struct2.F116, struct1.F116)
	assert.EqualValues(t, struct2.F117, struct1.F117)
	assert.EqualValues(t, struct2.F118, struct1.F118)
	assert.EqualValues(t, struct2.F119, struct1.F119)
	assert.EqualValues(t, struct2.F120, struct1.F120)
	assert.EqualValues(t, struct2.F121, struct1.F121)
	assert.EqualValues(t, struct2.F122, struct1.F122)
	assert.EqualValues(t, struct2.F123, struct1.F123)
	assert.EqualValues(t, struct2.F124, struct1.F124)
	assert.EqualValues(t, struct2.F125, struct1.F125)
	assert.EqualValues(t, struct2.F126, struct1.F126)
	assert.EqualValues(t, struct2.F127, struct1.F127)
	assert.EqualValues(t, struct2.F128, struct1.F128)
	assert.EqualValues(t, struct2.F129, struct1.F129)
	assert.EqualValues(t, struct2.F130, struct1.F130)
	assert.EqualValues(t, struct2.F131, struct1.F131)
	assert.EqualValues(t, struct2.F132, struct1.F132)
	assert.EqualValues(t, struct2.F133, struct1.F133)
	assert.EqualValues(t, struct2.F134, struct1.F134)
	assert.EqualValues(t, struct2.F135, struct1.F135)
	assert.EqualValues(t, struct2.F136, struct1.F136)
	assert.InEpsilon(t, *struct2.F137, *struct1.F137, 0.0001)
	assert.EqualValues(t, struct2.F138, struct1.F138)
	assert.EqualValues(t, struct2.F139, struct1.F139)
	assert.InEpsilon(t, *struct2.F140, *struct1.F140, 1e+123)
	assert.EqualValues(t, struct2.F141, struct1.F141)
	assert.EqualValues(t, struct2.F142, struct1.F142)
	assert.EqualValues(t, struct2.F143, struct1.F143)
	assert.EqualValues(t, struct2.F144, struct1.F144)
	assert.EqualValues(t, struct2.F145, struct1.F145)
	assert.EqualValues(t, struct2.F146, struct1.F146)
	assert.EqualValues(t, struct2.F147, struct1.F147)
	assert.EqualValues(t, struct2.F148, struct1.F148)
	assert.EqualValues(t, struct2.F149, struct1.F149)
	assert.EqualValues(t, struct2.F150, struct1.F150)
	assert.EqualValues(t, struct2.F151, struct1.F151)
	assert.EqualValues(t, struct2.F152, struct1.F152)
	assert.EqualValues(t, struct2.F153, struct1.F153)
	assert.EqualValues(t, struct2.F154, struct1.F154)
	assert.EqualValues(t, struct2.F155, struct1.F155)
	assert.EqualValues(t, struct2.F156, struct1.F156)
	assert.EqualValues(t, struct2.F157, struct1.F157)
}

func TestSerializationJsonStructNested(t *testing.T) {
	// Define a source JSON string
	json := []byte(`{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145901646321000,"f36":"00000000-0000-0000-0000-000000000000","f37":"9c8c268e-f0a6-11e8-a777-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]},"f100":null,"f101":true,"f102":null,"f103":null,"f104":255,"f105":null,"f106":null,"f107":33,"f108":null,"f109":null,"f110":1092,"f111":null,"f112":null,"f113":127,"f114":null,"f115":null,"f116":255,"f117":null,"f118":null,"f119":32767,"f120":null,"f121":null,"f122":65535,"f123":null,"f124":null,"f125":2147483647,"f126":null,"f127":null,"f128":4294967295,"f129":null,"f130":null,"f131":9223372036854775807,"f132":null,"f133":null,"f134":18446744073709551615,"f135":null,"f136":null,"f137":123.456,"f138":null,"f139":null,"f140":-1.23456e+125,"f141":null,"f142":null,"f143":"123456.123456","f144":null,"f145":null,"f146":"Initial string!","f147":null,"f148":null,"f149":1543145901647155000,"f150":null,"f151":null,"f152":"123e4567-e89b-12d3-a456-426655440000","f153":null,"f154":null,"f155":null,"f156":null,"f157":null,"f158":null,"f159":null,"f160":null,"f161":null,"f162":null,"f163":null,"f164":null,"f165":null,"f1000":0,"f1001":null,"f1002":50,"f1003":null,"f1004":0,"f1005":null,"f1006":42,"f1007":null,"f1008":{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145901647367000,"f36":"00000000-0000-0000-0000-000000000000","f37":"9c8c54c4-f0a6-11e8-a777-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"f1009":null,"f1010":{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145901648310000,"f36":"00000000-0000-0000-0000-000000000000","f37":"9c8c6b76-f0a6-11e8-a777-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]},"f100":null,"f101":true,"f102":null,"f103":null,"f104":255,"f105":null,"f106":null,"f107":33,"f108":null,"f109":null,"f110":1092,"f111":null,"f112":null,"f113":127,"f114":null,"f115":null,"f116":255,"f117":null,"f118":null,"f119":32767,"f120":null,"f121":null,"f122":65535,"f123":null,"f124":null,"f125":2147483647,"f126":null,"f127":null,"f128":4294967295,"f129":null,"f130":null,"f131":9223372036854775807,"f132":null,"f133":null,"f134":18446744073709551615,"f135":null,"f136":null,"f137":123.456,"f138":null,"f139":null,"f140":-1.23456e+125,"f141":null,"f142":null,"f143":"123456.123456","f144":null,"f145":null,"f146":"Initial string!","f147":null,"f148":null,"f149":1543145901648871000,"f150":null,"f151":null,"f152":"123e4567-e89b-12d3-a456-426655440000","f153":null,"f154":null,"f155":null,"f156":null,"f157":null,"f158":null,"f159":null,"f160":null,"f161":null,"f162":null,"f163":null,"f164":null,"f165":null},"f1011":null}`)

	// Create a new struct from the source JSON string
	struct1, _ := test.NewStructNestedFromJSON(json)

	// Serialize the struct to the JSON string
	json, _ = struct1.JSON()

	// Check the serialized JSON size
	assert.NotEmpty(t, json)

	// Deserialize the struct from the JSON string
	struct2, _ := test.NewStructNestedFromJSON(json)

	assert.EqualValues(t, struct2.F1, false)
	assert.EqualValues(t, struct2.F2, true)
	assert.EqualValues(t, struct2.F3, 0)
	assert.EqualValues(t, struct2.F4, 0xFF)
	assert.EqualValues(t, struct2.F5, '\000')
	assert.EqualValues(t, struct2.F6, '!')
	assert.EqualValues(t, struct2.F7, rune(0))
	assert.EqualValues(t, struct2.F8, rune(0x0444))
	assert.EqualValues(t, struct2.F9, 0)
	assert.EqualValues(t, struct2.F10, 127)
	assert.EqualValues(t, struct2.F11, 0)
	assert.EqualValues(t, struct2.F12, 0xFF)
	assert.EqualValues(t, struct2.F13, 0)
	assert.EqualValues(t, struct2.F14, 32767)
	assert.EqualValues(t, struct2.F15, 0)
	assert.EqualValues(t, struct2.F16, 0xFFFF)
	assert.EqualValues(t, struct2.F17, 0)
	assert.EqualValues(t, struct2.F18, 2147483647)
	assert.EqualValues(t, struct2.F19, 0)
	assert.EqualValues(t, struct2.F20, 0xFFFFFFFF)
	assert.EqualValues(t, struct2.F21, 0)
	assert.EqualValues(t, struct2.F22, int64(9223372036854775807))
	assert.EqualValues(t, struct2.F23, 0)
	assert.EqualValues(t, struct2.F24, uint64(0xFFFFFFFFFFFFFFFF))
	assert.EqualValues(t, struct2.F25, 0.0)
	assert.InEpsilon(t, struct2.F26, 123.456, 0.0001)
	assert.EqualValues(t, struct2.F27, 0.0)
	assert.InEpsilon(t, struct2.F28, -123.567e+123, 1e+123)
	assert.EqualValues(t, struct2.F29, fbe.DecimalFromString("0"))
	assert.EqualValues(t, struct2.F30, fbe.DecimalFromString("123456.123456"))
	assert.EqualValues(t, struct2.F31, "")
	assert.EqualValues(t, struct2.F32, "Initial string!")
	assert.EqualValues(t, struct2.F33, fbe.TimestampEpoch())
	assert.EqualValues(t, struct2.F34, fbe.TimestampEpoch())
	assert.True(t, struct2.F35.UnixNano() > fbe.TimestampFromDate(2018, 1, 1).UnixNano())
	assert.EqualValues(t, struct2.F36, fbe.UUIDNil())
	assert.NotEqual(t, struct2.F37, fbe.UUIDNil())
	assert.EqualValues(t, struct2.F38, fbe.UUIDFromString("123e4567-e89b-12d3-a456-426655440000"))

	assert.Nil(t, struct2.F100)
	assert.NotNil(t, struct2.F101)
	assert.EqualValues(t, *struct2.F101, true)
	assert.Nil(t, struct2.F102)
	assert.Nil(t, struct2.F103)
	assert.NotNil(t, struct2.F104)
	assert.EqualValues(t, *struct2.F104, 0xFF)
	assert.Nil(t, struct2.F105)
	assert.Nil(t, struct2.F106)
	assert.NotNil(t, struct2.F107)
	assert.EqualValues(t, *struct2.F107, '!')
	assert.Nil(t, struct2.F108)
	assert.Nil(t, struct2.F109)
	assert.NotNil(t, struct2.F110)
	assert.EqualValues(t, *struct2.F110, rune(0x0444))
	assert.Nil(t, struct2.F111)
	assert.Nil(t, struct2.F112)
	assert.NotNil(t, struct2.F113)
	assert.EqualValues(t, *struct2.F113, 127)
	assert.Nil(t, struct2.F114)
	assert.Nil(t, struct2.F115)
	assert.NotNil(t, struct2.F116)
	assert.EqualValues(t, *struct2.F116, 0xFF)
	assert.Nil(t, struct2.F117)
	assert.Nil(t, struct2.F118)
	assert.NotNil(t, struct2.F119)
	assert.EqualValues(t, *struct2.F119, 32767)
	assert.Nil(t, struct2.F120)
	assert.Nil(t, struct2.F121)
	assert.NotNil(t, struct2.F122)
	assert.EqualValues(t, *struct2.F122, 0xFFFF)
	assert.Nil(t, struct2.F123)
	assert.Nil(t, struct2.F124)
	assert.NotNil(t, struct2.F125)
	assert.EqualValues(t, *struct2.F125, 2147483647)
	assert.Nil(t, struct2.F126)
	assert.Nil(t, struct2.F127)
	assert.NotNil(t, struct2.F128)
	assert.EqualValues(t, *struct2.F128, 0xFFFFFFFF)
	assert.Nil(t, struct2.F129)
	assert.Nil(t, struct2.F130)
	assert.NotNil(t, struct2.F131)
	assert.EqualValues(t, *struct2.F131, 9223372036854775807)
	assert.Nil(t, struct2.F132)
	assert.Nil(t, struct2.F133)
	assert.NotNil(t, struct2.F131)
	assert.EqualValues(t, *struct2.F134, uint64(0xFFFFFFFFFFFFFFFF))
	assert.Nil(t, struct2.F135)
	assert.Nil(t, struct2.F136)
	assert.NotNil(t, struct2.F137)
	assert.InEpsilon(t, *struct2.F137, 123.456, 0.0001)
	assert.Nil(t, struct2.F138)
	assert.Nil(t, struct2.F139)
	assert.NotNil(t, struct2.F140)
	assert.InEpsilon(t, *struct2.F140, -123.567e+123, 1e+123)
	assert.Nil(t, struct2.F141)
	assert.Nil(t, struct2.F142)
	assert.NotNil(t, struct2.F143)
	assert.EqualValues(t, *struct2.F143, fbe.DecimalFromString("123456.123456"))
	assert.Nil(t, struct2.F144)
	assert.Nil(t, struct2.F145)
	assert.NotNil(t, struct2.F146)
	assert.EqualValues(t, *struct2.F146, "Initial string!")
	assert.Nil(t, struct2.F147)
	assert.Nil(t, struct2.F148)
	assert.NotNil(t, struct2.F149)
	assert.True(t, struct2.F149.UnixNano() > fbe.TimestampFromDate(2018, 1, 1).UnixNano())
	assert.Nil(t, struct2.F150)
	assert.Nil(t, struct2.F151)
	assert.NotNil(t, struct2.F152)
	assert.EqualValues(t, *struct2.F152, fbe.UUIDFromString("123e4567-e89b-12d3-a456-426655440000"))
	assert.Nil(t, struct2.F153)
	assert.Nil(t, struct2.F154)
	assert.Nil(t, struct2.F155)
	assert.Nil(t, struct2.F156)
	assert.Nil(t, struct2.F157)
	assert.Nil(t, struct2.F158)
	assert.Nil(t, struct2.F159)
	assert.Nil(t, struct2.F160)
	assert.Nil(t, struct2.F161)
	assert.Nil(t, struct2.F162)
	assert.Nil(t, struct2.F163)
	assert.Nil(t, struct2.F164)
	assert.Nil(t, struct2.F165)

	assert.EqualValues(t, struct2.F1000, test.EnumSimple_ENUM_VALUE_0)
	assert.Nil(t, struct2.F1001)
	assert.EqualValues(t, struct2.F1002, test.EnumTyped_ENUM_VALUE_2)
	assert.Nil(t, struct2.F1003)
	assert.EqualValues(t, struct2.F1004, test.FlagsSimple_FLAG_VALUE_0)
	assert.Nil(t, struct2.F1005)
	assert.EqualValues(t, struct2.F1006, test.FlagsTyped_FLAG_VALUE_2 | test.FlagsTyped_FLAG_VALUE_4 | test.FlagsTyped_FLAG_VALUE_6)
	assert.Nil(t, struct2.F1007)
	assert.Nil(t, struct2.F1009)
	assert.Nil(t, struct2.F1011)

	assert.EqualValues(t, struct2.F1, struct1.F1)
	assert.EqualValues(t, struct2.F2, struct1.F2)
	assert.EqualValues(t, struct2.F3, struct1.F3)
	assert.EqualValues(t, struct2.F4, struct1.F4)
	assert.EqualValues(t, struct2.F5, struct1.F5)
	assert.EqualValues(t, struct2.F6, struct1.F6)
	assert.EqualValues(t, struct2.F7, struct1.F7)
	assert.EqualValues(t, struct2.F8, struct1.F8)
	assert.EqualValues(t, struct2.F9, struct1.F9)
	assert.EqualValues(t, struct2.F10, struct1.F10)
	assert.EqualValues(t, struct2.F11, struct1.F11)
	assert.EqualValues(t, struct2.F12, struct1.F12)
	assert.EqualValues(t, struct2.F13, struct1.F13)
	assert.EqualValues(t, struct2.F14, struct1.F14)
	assert.EqualValues(t, struct2.F15, struct1.F15)
	assert.EqualValues(t, struct2.F16, struct1.F16)
	assert.EqualValues(t, struct2.F17, struct1.F17)
	assert.EqualValues(t, struct2.F18, struct1.F18)
	assert.EqualValues(t, struct2.F19, struct1.F19)
	assert.EqualValues(t, struct2.F20, struct1.F20)
	assert.EqualValues(t, struct2.F21, struct1.F21)
	assert.EqualValues(t, struct2.F22, struct1.F22)
	assert.EqualValues(t, struct2.F23, struct1.F23)
	assert.EqualValues(t, struct2.F24, struct1.F24)
	assert.EqualValues(t, struct2.F25, struct1.F25)
	assert.InEpsilon(t, struct2.F26, struct1.F26, 0.0001)
	assert.EqualValues(t, struct2.F27, struct1.F27)
	assert.InEpsilon(t, struct2.F28, struct1.F28, 1e+123)
	assert.EqualValues(t, struct2.F29, struct1.F29)
	assert.EqualValues(t, struct2.F30, struct1.F30)
	assert.EqualValues(t, struct2.F31, struct1.F31)
	assert.EqualValues(t, struct2.F32, struct1.F32)
	assert.EqualValues(t, struct2.F33, struct1.F33)
	assert.EqualValues(t, struct2.F34, struct1.F34)
	assert.EqualValues(t, struct2.F35, struct1.F35)
	assert.EqualValues(t, struct2.F36, struct1.F36)
	assert.EqualValues(t, struct2.F37, struct1.F37)
	assert.EqualValues(t, struct2.F38, struct1.F38)
	assert.EqualValues(t, struct2.F39, struct1.F39)
	assert.EqualValues(t, struct2.F40, struct1.F40)

	assert.EqualValues(t, struct2.F100, struct1.F100)
	assert.EqualValues(t, struct2.F101, struct1.F101)
	assert.EqualValues(t, struct2.F102, struct1.F102)
	assert.EqualValues(t, struct2.F103, struct1.F103)
	assert.EqualValues(t, struct2.F104, struct1.F104)
	assert.EqualValues(t, struct2.F105, struct1.F105)
	assert.EqualValues(t, struct2.F106, struct1.F106)
	assert.EqualValues(t, struct2.F107, struct1.F107)
	assert.EqualValues(t, struct2.F108, struct1.F108)
	assert.EqualValues(t, struct2.F109, struct1.F109)
	assert.EqualValues(t, struct2.F110, struct1.F110)
	assert.EqualValues(t, struct2.F111, struct1.F111)
	assert.EqualValues(t, struct2.F112, struct1.F112)
	assert.EqualValues(t, struct2.F113, struct1.F113)
	assert.EqualValues(t, struct2.F114, struct1.F114)
	assert.EqualValues(t, struct2.F115, struct1.F115)
	assert.EqualValues(t, struct2.F116, struct1.F116)
	assert.EqualValues(t, struct2.F117, struct1.F117)
	assert.EqualValues(t, struct2.F118, struct1.F118)
	assert.EqualValues(t, struct2.F119, struct1.F119)
	assert.EqualValues(t, struct2.F120, struct1.F120)
	assert.EqualValues(t, struct2.F121, struct1.F121)
	assert.EqualValues(t, struct2.F122, struct1.F122)
	assert.EqualValues(t, struct2.F123, struct1.F123)
	assert.EqualValues(t, struct2.F124, struct1.F124)
	assert.EqualValues(t, struct2.F125, struct1.F125)
	assert.EqualValues(t, struct2.F126, struct1.F126)
	assert.EqualValues(t, struct2.F127, struct1.F127)
	assert.EqualValues(t, struct2.F128, struct1.F128)
	assert.EqualValues(t, struct2.F129, struct1.F129)
	assert.EqualValues(t, struct2.F130, struct1.F130)
	assert.EqualValues(t, struct2.F131, struct1.F131)
	assert.EqualValues(t, struct2.F132, struct1.F132)
	assert.EqualValues(t, struct2.F133, struct1.F133)
	assert.EqualValues(t, struct2.F134, struct1.F134)
	assert.EqualValues(t, struct2.F135, struct1.F135)
	assert.EqualValues(t, struct2.F136, struct1.F136)
	assert.InEpsilon(t, *struct2.F137, *struct1.F137, 0.0001)
	assert.EqualValues(t, struct2.F138, struct1.F138)
	assert.EqualValues(t, struct2.F139, struct1.F139)
	assert.InEpsilon(t, *struct2.F140, *struct1.F140, 1e+123)
	assert.EqualValues(t, struct2.F141, struct1.F141)
	assert.EqualValues(t, struct2.F142, struct1.F142)
	assert.EqualValues(t, struct2.F143, struct1.F143)
	assert.EqualValues(t, struct2.F144, struct1.F144)
	assert.EqualValues(t, struct2.F145, struct1.F145)
	assert.EqualValues(t, struct2.F146, struct1.F146)
	assert.EqualValues(t, struct2.F147, struct1.F147)
	assert.EqualValues(t, struct2.F148, struct1.F148)
	assert.EqualValues(t, struct2.F149, struct1.F149)
	assert.EqualValues(t, struct2.F150, struct1.F150)
	assert.EqualValues(t, struct2.F151, struct1.F151)
	assert.EqualValues(t, struct2.F152, struct1.F152)
	assert.EqualValues(t, struct2.F153, struct1.F153)
	assert.EqualValues(t, struct2.F154, struct1.F154)
	assert.EqualValues(t, struct2.F155, struct1.F155)
	assert.EqualValues(t, struct2.F156, struct1.F156)
	assert.EqualValues(t, struct2.F157, struct1.F157)

	assert.EqualValues(t, struct2.F1000, struct1.F1000)
	assert.EqualValues(t, struct2.F1001, struct1.F1001)
	assert.EqualValues(t, struct2.F1002, struct1.F1002)
	assert.EqualValues(t, struct2.F1003, struct1.F1003)
	assert.EqualValues(t, struct2.F1004, struct1.F1004)
	assert.EqualValues(t, struct2.F1005, struct1.F1005)
	assert.EqualValues(t, struct2.F1006, struct1.F1006)
	assert.EqualValues(t, struct2.F1007, struct1.F1007)
}

func TestSerializationJsonStructBytes(t *testing.T) {
	// Define a source JSON string
	json := []byte(`{"f1":"QUJD","f2":"dGVzdA==","f3":null}`)

	// Create a new struct from the source JSON string
	struct1, _ := test.NewStructBytesFromJSON(json)

	// Serialize the struct to the JSON string
	json, _ = struct1.JSON()

	// Check the serialized JSON size
	assert.NotEmpty(t, json)

	// Deserialize the struct from the JSON string
	struct2, _ := test.NewStructBytesFromJSON(json)

	assert.EqualValues(t, len(struct2.F1), 3)
	assert.EqualValues(t, struct2.F1[0], 65)
	assert.EqualValues(t, struct2.F1[1], 66)
	assert.EqualValues(t, struct2.F1[2], 67)
	assert.NotNil(t, struct2.F2)
	assert.EqualValues(t, len(*struct2.F2), 4)
	assert.EqualValues(t, (*struct2.F2)[0], 116)
	assert.EqualValues(t, (*struct2.F2)[1], 101)
	assert.EqualValues(t, (*struct2.F2)[2], 115)
	assert.EqualValues(t, (*struct2.F2)[3], 116)
	assert.Nil(t, struct2.F3)
}

func TestSerializationJsonStructArray(t *testing.T) {
	// Define a source JSON string
	json := []byte(`{"f1":[48,65],"f2":[97,null],"f3":["MDAw","QUFB"],"f4":["YWFh",null],"f5":[1,2],"f6":[1,null],"f7":[3,7],"f8":[3,null],"f9":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145986060361000,"f36":"00000000-0000-0000-0000-000000000000","f37":"cedcad98-f0a6-11e8-9f47-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145986060910000,"f36":"00000000-0000-0000-0000-000000000000","f37":"cedcc274-f0a6-11e8-9f47-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}],"f10":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543145986061436000,"f36":"00000000-0000-0000-0000-000000000000","f37":"cedcd714-f0a6-11e8-9f47-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},null]}`)

	// Create a new struct from the source JSON string
	struct1, _ := test.NewStructArrayFromJSON(json)

	// Serialize the struct to the JSON string
	json, _ = struct1.JSON()

	// Check the serialized JSON size
	assert.NotEmpty(t, json)

	// Deserialize the struct from the JSON string
	struct2, _ := test.NewStructArrayFromJSON(json)

	assert.EqualValues(t, len(struct2.F1), 2)
	assert.EqualValues(t, struct2.F1[0], 48)
	assert.EqualValues(t, struct2.F1[1], 65)
	assert.EqualValues(t, len(struct2.F2), 2)
	assert.EqualValues(t, *struct2.F2[0], 97)
	assert.Nil(t, struct2.F2[1])
	assert.EqualValues(t, len(struct2.F3), 2)
	assert.EqualValues(t, len(struct2.F3[0]), 3)
	assert.EqualValues(t, struct2.F3[0][0], 48)
	assert.EqualValues(t, struct2.F3[0][1], 48)
	assert.EqualValues(t, struct2.F3[0][2], 48)
	assert.EqualValues(t, len(struct2.F3[1]), 3)
	assert.EqualValues(t, struct2.F3[1][0], 65)
	assert.EqualValues(t, struct2.F3[1][1], 65)
	assert.EqualValues(t, struct2.F3[1][2], 65)
	assert.EqualValues(t, len(struct2.F4), 2)
	assert.NotNil(t, struct2.F4[0])
	assert.EqualValues(t, len(*struct2.F4[0]), 3)
	assert.EqualValues(t, (*struct2.F4[0])[0], 97)
	assert.EqualValues(t, (*struct2.F4[0])[1], 97)
	assert.EqualValues(t, (*struct2.F4[0])[2], 97)
	assert.Nil(t, struct2.F4[1])
	assert.EqualValues(t, len(struct2.F5), 2)
	assert.EqualValues(t, struct2.F5[0], test.EnumSimple_ENUM_VALUE_1)
	assert.EqualValues(t, struct2.F5[1], test.EnumSimple_ENUM_VALUE_2)
	assert.EqualValues(t, len(struct2.F6), 2)
	assert.EqualValues(t, *struct2.F6[0], test.EnumSimple_ENUM_VALUE_1)
	assert.Nil(t, struct2.F6[1])
	assert.EqualValues(t, len(struct2.F7), 2)
	assert.EqualValues(t, struct2.F7[0], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2)
	assert.EqualValues(t, struct2.F7[1], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2 | test.FlagsSimple_FLAG_VALUE_3)
	assert.EqualValues(t, len(struct2.F8), 2)
	assert.EqualValues(t, *struct2.F8[0], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2)
	assert.Nil(t, struct2.F8[1])
	assert.EqualValues(t, len(struct2.F9), 2)
	assert.EqualValues(t, struct2.F9[0].F2, true)
	assert.EqualValues(t, struct2.F9[0].F12, 0xFF)
	assert.EqualValues(t, struct2.F9[0].F32, "Initial string!")
	assert.EqualValues(t, struct2.F9[1].F2, true)
	assert.EqualValues(t, struct2.F9[1].F12, 0xFF)
	assert.EqualValues(t, struct2.F9[1].F32, "Initial string!")
	assert.EqualValues(t, len(struct2.F10), 2)
	assert.NotNil(t, struct2.F10[0])
	assert.EqualValues(t, struct2.F10[0].F2, true)
	assert.EqualValues(t, struct2.F10[0].F12, 0xFF)
	assert.EqualValues(t, struct2.F10[0].F32, "Initial string!")
	assert.Nil(t, struct2.F10[1])
}

func TestSerializationJsonStructVector(t *testing.T) {
	// Define a source JSON string
	json := []byte(`{"f1":[48,65],"f2":[97,null],"f3":["MDAw","QUFB"],"f4":["YWFh",null],"f5":[1,2],"f6":[1,null],"f7":[3,7],"f8":[3,null],"f9":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146157127964000,"f36":"00000000-0000-0000-0000-000000000000","f37":"34d38702-f0a7-11e8-b30e-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146157128572000,"f36":"00000000-0000-0000-0000-000000000000","f37":"34d39c88-f0a7-11e8-b30e-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}],"f10":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146157129063000,"f36":"00000000-0000-0000-0000-000000000000","f37":"34d3b038-f0a7-11e8-b30e-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},null]}`)

	// Create a new struct from the source JSON string
	struct1, _ := test.NewStructVectorFromJSON(json)

	// Serialize the struct to the JSON string
	json, _ = struct1.JSON()

	// Check the serialized JSON size
	assert.NotEmpty(t, json)

	// Deserialize the struct from the JSON string
	struct2, _ := test.NewStructVectorFromJSON(json)

	assert.EqualValues(t, len(struct2.F1), 2)
	assert.EqualValues(t, struct2.F1[0], 48)
	assert.EqualValues(t, struct2.F1[1], 65)
	assert.EqualValues(t, len(struct2.F2), 2)
	assert.EqualValues(t, *struct2.F2[0], 97)
	assert.Nil(t, struct2.F2[1])
	assert.EqualValues(t, len(struct2.F3), 2)
	assert.EqualValues(t, len(struct2.F3[0]), 3)
	assert.EqualValues(t, struct2.F3[0][0], 48)
	assert.EqualValues(t, struct2.F3[0][1], 48)
	assert.EqualValues(t, struct2.F3[0][2], 48)
	assert.EqualValues(t, len(struct2.F3[1]), 3)
	assert.EqualValues(t, struct2.F3[1][0], 65)
	assert.EqualValues(t, struct2.F3[1][1], 65)
	assert.EqualValues(t, struct2.F3[1][2], 65)
	assert.EqualValues(t, len(struct2.F4), 2)
	assert.NotNil(t, struct2.F4[0])
	assert.EqualValues(t, len(*struct2.F4[0]), 3)
	assert.EqualValues(t, (*struct2.F4[0])[0], 97)
	assert.EqualValues(t, (*struct2.F4[0])[1], 97)
	assert.EqualValues(t, (*struct2.F4[0])[2], 97)
	assert.Nil(t, struct2.F4[1])
	assert.EqualValues(t, len(struct2.F5), 2)
	assert.EqualValues(t, struct2.F5[0], test.EnumSimple_ENUM_VALUE_1)
	assert.EqualValues(t, struct2.F5[1], test.EnumSimple_ENUM_VALUE_2)
	assert.EqualValues(t, len(struct2.F6), 2)
	assert.EqualValues(t, *struct2.F6[0], test.EnumSimple_ENUM_VALUE_1)
	assert.Nil(t, struct2.F6[1])
	assert.EqualValues(t, len(struct2.F7), 2)
	assert.EqualValues(t, struct2.F7[0], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2)
	assert.EqualValues(t, struct2.F7[1], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2 | test.FlagsSimple_FLAG_VALUE_3)
	assert.EqualValues(t, len(struct2.F8), 2)
	assert.EqualValues(t, *struct2.F8[0], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2)
	assert.Nil(t, struct2.F8[1])
	assert.EqualValues(t, len(struct2.F9), 2)
	assert.EqualValues(t, struct2.F9[0].F2, true)
	assert.EqualValues(t, struct2.F9[0].F12, 0xFF)
	assert.EqualValues(t, struct2.F9[0].F32, "Initial string!")
	assert.EqualValues(t, struct2.F9[1].F2, true)
	assert.EqualValues(t, struct2.F9[1].F12, 0xFF)
	assert.EqualValues(t, struct2.F9[1].F32, "Initial string!")
	assert.EqualValues(t, len(struct2.F10), 2)
	assert.NotNil(t, struct2.F10[0])
	assert.EqualValues(t, struct2.F10[0].F2, true)
	assert.EqualValues(t, struct2.F10[0].F12, 0xFF)
	assert.EqualValues(t, struct2.F10[0].F32, "Initial string!")
	assert.Nil(t, struct2.F10[1])
}

func TestSerializationJsonStructList(t *testing.T) {
	// Define a source JSON string
	json := []byte(`{"f1":[48,65],"f2":[97,null],"f3":["MDAw","QUFB"],"f4":["YWFh",null],"f5":[1,2],"f6":[1,null],"f7":[3,7],"f8":[3,null],"f9":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146220253760000,"f36":"00000000-0000-0000-0000-000000000000","f37":"5a73e7fe-f0a7-11e8-89e6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146220255725000,"f36":"00000000-0000-0000-0000-000000000000","f37":"5a741990-f0a7-11e8-89e6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}],"f10":[{"id":0,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146220256802000,"f36":"00000000-0000-0000-0000-000000000000","f37":"5a74e4b0-f0a7-11e8-89e6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},null]}`)

	// Create a new struct from the source JSON string
	struct1, _ := test.NewStructListFromJSON(json)

	// Serialize the struct to the JSON string
	json, _ = struct1.JSON()

	// Check the serialized JSON size
	assert.NotEmpty(t, json)

	// Deserialize the struct from the JSON string
	struct2, _ := test.NewStructListFromJSON(json)

	assert.EqualValues(t, len(struct2.F1), 2)
	assert.EqualValues(t, struct2.F1[0], 48)
	assert.EqualValues(t, struct2.F1[1], 65)
	assert.EqualValues(t, len(struct2.F2), 2)
	assert.EqualValues(t, *struct2.F2[0], 97)
	assert.Nil(t, struct2.F2[1])
	assert.EqualValues(t, len(struct2.F3), 2)
	assert.EqualValues(t, len(struct2.F3[0]), 3)
	assert.EqualValues(t, struct2.F3[0][0], 48)
	assert.EqualValues(t, struct2.F3[0][1], 48)
	assert.EqualValues(t, struct2.F3[0][2], 48)
	assert.EqualValues(t, len(struct2.F3[1]), 3)
	assert.EqualValues(t, struct2.F3[1][0], 65)
	assert.EqualValues(t, struct2.F3[1][1], 65)
	assert.EqualValues(t, struct2.F3[1][2], 65)
	assert.EqualValues(t, len(struct2.F4), 2)
	assert.NotNil(t, struct2.F4[0])
	assert.EqualValues(t, len(*struct2.F4[0]), 3)
	assert.EqualValues(t, (*struct2.F4[0])[0], 97)
	assert.EqualValues(t, (*struct2.F4[0])[1], 97)
	assert.EqualValues(t, (*struct2.F4[0])[2], 97)
	assert.Nil(t, struct2.F4[1])
	assert.EqualValues(t, len(struct2.F5), 2)
	assert.EqualValues(t, struct2.F5[0], test.EnumSimple_ENUM_VALUE_1)
	assert.EqualValues(t, struct2.F5[1], test.EnumSimple_ENUM_VALUE_2)
	assert.EqualValues(t, len(struct2.F6), 2)
	assert.EqualValues(t, *struct2.F6[0], test.EnumSimple_ENUM_VALUE_1)
	assert.Nil(t, struct2.F6[1])
	assert.EqualValues(t, len(struct2.F7), 2)
	assert.EqualValues(t, struct2.F7[0], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2)
	assert.EqualValues(t, struct2.F7[1], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2 | test.FlagsSimple_FLAG_VALUE_3)
	assert.EqualValues(t, len(struct2.F8), 2)
	assert.EqualValues(t, *struct2.F8[0], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2)
	assert.Nil(t, struct2.F8[1])
	assert.EqualValues(t, len(struct2.F9), 2)
	assert.EqualValues(t, struct2.F9[0].F2, true)
	assert.EqualValues(t, struct2.F9[0].F12, 0xFF)
	assert.EqualValues(t, struct2.F9[0].F32, "Initial string!")
	assert.EqualValues(t, struct2.F9[1].F2, true)
	assert.EqualValues(t, struct2.F9[1].F12, 0xFF)
	assert.EqualValues(t, struct2.F9[1].F32, "Initial string!")
	assert.EqualValues(t, len(struct2.F10), 2)
	assert.NotNil(t, struct2.F10[0])
	assert.EqualValues(t, struct2.F10[0].F2, true)
	assert.EqualValues(t, struct2.F10[0].F12, 0xFF)
	assert.EqualValues(t, struct2.F10[0].F32, "Initial string!")
	assert.Nil(t, struct2.F10[1])
}

func TestSerializationJsonStructSet(t *testing.T) {
	// Define a source JSON string
	json := []byte(`{"f1":[48,65,97],"f2":[1,2],"f3":[3,7],"f4":[{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146299848353000,"f36":"00000000-0000-0000-0000-000000000000","f37":"89e4edd0-f0a7-11e8-9dde-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},{"id":65,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146299848966000,"f36":"00000000-0000-0000-0000-000000000000","f37":"89e503f6-f0a7-11e8-9dde-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}]}`)

	// Create a new struct from the source JSON string
	struct1, _ := test.NewStructSetFromJSON(json)

	// Serialize the struct to the JSON string
	json, _ = struct1.JSON()

	// Check the serialized JSON size
	assert.NotEmpty(t, json)

	// Deserialize the struct from the JSON string
	struct2, _ := test.NewStructSetFromJSON(json)

	assert.EqualValues(t, len(struct2.F1), 3)
	assert.True(t, struct2.F1.Contains(48))
	assert.True(t, struct2.F1.Contains(65))
	assert.True(t, struct2.F1.Contains(97))
	assert.EqualValues(t, len(struct2.F2), 2)
	assert.True(t, struct2.F2.Contains(test.EnumSimple_ENUM_VALUE_1))
	assert.True(t, struct2.F2.Contains(test.EnumSimple_ENUM_VALUE_2))
	assert.EqualValues(t, len(struct2.F3), 2)
	assert.True(t, struct2.F3.Contains(test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2))
	assert.True(t, struct2.F3.Contains(test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2 | test.FlagsSimple_FLAG_VALUE_3))
	assert.EqualValues(t, len(struct2.F4), 2)
	s1 := *test.NewStructSimple()
	s1.Id = 48
	assert.True(t, struct2.F4.Contains(s1))
	s2 := *test.NewStructSimple()
	s2.Id = 65
	assert.True(t, struct2.F4.Contains(s2))
}

func TestSerializationJsonStructMap(t *testing.T) {
	// Define a source JSON string
	json := []byte(`{"f1":{"10":48,"20":65},"f2":{"10":97,"20":null},"f3":{"10":"MDAw","20":"QUFB"},"f4":{"10":"YWFh","20":null},"f5":{"10":1,"20":2},"f6":{"10":1,"20":null},"f7":{"10":3,"20":7},"f8":{"10":3,"20":null},"f9":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146345803483000,"f36":"00000000-0000-0000-0000-000000000000","f37":"a549215e-f0a7-11e8-90f6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":{"id":65,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146345804184000,"f36":"00000000-0000-0000-0000-000000000000","f37":"a54942ce-f0a7-11e8-90f6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}},"f10":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146345803483000,"f36":"00000000-0000-0000-0000-000000000000","f37":"a549215e-f0a7-11e8-90f6-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":null}}`)

	// Create a new struct from the source JSON string
	struct1, _ := test.NewStructMapFromJSON(json)

	// Serialize the struct to the JSON string
	json, _ = struct1.JSON()

	// Check the serialized JSON size
	assert.NotEmpty(t, json)

	// Deserialize the struct from the JSON string
	struct2, _ := test.NewStructMapFromJSON(json)

	assert.EqualValues(t, len(struct2.F1), 2)
	assert.EqualValues(t, struct2.F1[10], 48)
	assert.EqualValues(t, struct2.F1[20], 65)
	assert.EqualValues(t, len(struct2.F2), 2)
	assert.EqualValues(t, *struct2.F2[10], 97)
	assert.Nil(t, struct2.F2[20])
	assert.EqualValues(t, len(struct2.F3), 2)
	assert.EqualValues(t, len(struct2.F3[10]), 3)
	assert.EqualValues(t, len(struct2.F3[20]), 3)
	assert.EqualValues(t, len(struct2.F4), 2)
	assert.EqualValues(t, len(*struct2.F4[10]), 3)
	assert.Nil(t, struct2.F4[20])
	assert.EqualValues(t, len(struct2.F5), 2)
	assert.EqualValues(t, struct2.F5[10], test.EnumSimple_ENUM_VALUE_1)
	assert.EqualValues(t, struct2.F5[20], test.EnumSimple_ENUM_VALUE_2)
	assert.EqualValues(t, len(struct2.F6), 2)
	assert.EqualValues(t, *struct2.F6[10], test.EnumSimple_ENUM_VALUE_1)
	assert.Nil(t, struct2.F6[20])
	assert.EqualValues(t, len(struct2.F7), 2)
	assert.EqualValues(t, struct2.F7[10], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2)
	assert.EqualValues(t, struct2.F7[20], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2 | test.FlagsSimple_FLAG_VALUE_3)
	assert.EqualValues(t, len(struct2.F8), 2)
	assert.EqualValues(t, *struct2.F8[10], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2)
	assert.Nil(t, struct2.F8[20])
	assert.EqualValues(t, len(struct2.F9), 2)
	assert.EqualValues(t, struct2.F9[10].Id, 48)
	assert.EqualValues(t, struct2.F9[20].Id, 65)
	assert.EqualValues(t, len(struct2.F10), 2)
	assert.EqualValues(t, struct2.F10[10].Id, 48)
	assert.Nil(t, struct2.F10[20])
}

func TestSerializationJsonStructHash(t *testing.T) {
	// Define a source JSON string
	json := []byte(`{"f1":{"10":48,"20":65},"f2":{"10":97,"20":null},"f3":{"10":"MDAw","20":"QUFB"},"f4":{"10":"YWFh","20":null},"f5":{"10":1,"20":2},"f6":{"10":1,"20":null},"f7":{"10":3,"20":7},"f8":{"10":3,"20":null},"f9":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146381450913000,"f36":"00000000-0000-0000-0000-000000000000","f37":"ba8885d2-f0a7-11e8-81fa-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":{"id":65,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146381452825000,"f36":"00000000-0000-0000-0000-000000000000","f37":"ba88ced4-f0a7-11e8-81fa-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}}},"f10":{"10":{"id":48,"f1":false,"f2":true,"f3":0,"f4":255,"f5":0,"f6":33,"f7":0,"f8":1092,"f9":0,"f10":127,"f11":0,"f12":255,"f13":0,"f14":32767,"f15":0,"f16":65535,"f17":0,"f18":2147483647,"f19":0,"f20":4294967295,"f21":0,"f22":9223372036854775807,"f23":0,"f24":18446744073709551615,"f25":0.0,"f26":123.456,"f27":0.0,"f28":-1.23456e+125,"f29":"0.0","f30":"123456.123456","f31":"","f32":"Initial string!","f33":0,"f34":0,"f35":1543146381450913000,"f36":"00000000-0000-0000-0000-000000000000","f37":"ba8885d2-f0a7-11e8-81fa-ac220bcdd8e0","f38":"123e4567-e89b-12d3-a456-426655440000","f39":0,"f40":0,"f41":{"id":0,"symbol":"","side":0,"type":0,"price":0.0,"volume":0.0},"f42":{"currency":"","amount":0.0},"f43":0,"f44":{"id":0,"name":"","state":11,"wallet":{"currency":"","amount":0.0},"asset":null,"orders":[]}},"20":null}}`)

	// Create a new struct from the source JSON string
	struct1, _ := test.NewStructHashFromJSON(json)

	// Serialize the struct to the JSON string
	json, _ = struct1.JSON()

	// Check the serialized JSON size
	assert.NotEmpty(t, json)

	// Deserialize the struct from the JSON string
	struct2, _ := test.NewStructHashFromJSON(json)

	assert.EqualValues(t, len(struct2.F1), 2)
	assert.EqualValues(t, struct2.F1["10"], 48)
	assert.EqualValues(t, struct2.F1["20"], 65)
	assert.EqualValues(t, len(struct2.F2), 2)
	assert.EqualValues(t, *struct2.F2["10"], 97)
	assert.Nil(t, struct2.F2["20"])
	assert.EqualValues(t, len(struct2.F3), 2)
	assert.EqualValues(t, len(struct2.F3["10"]), 3)
	assert.EqualValues(t, len(struct2.F3["20"]), 3)
	assert.EqualValues(t, len(struct2.F4), 2)
	assert.EqualValues(t, len(*struct2.F4["10"]), 3)
	assert.Nil(t, struct2.F4["20"])
	assert.EqualValues(t, len(struct2.F5), 2)
	assert.EqualValues(t, struct2.F5["10"], test.EnumSimple_ENUM_VALUE_1)
	assert.EqualValues(t, struct2.F5["20"], test.EnumSimple_ENUM_VALUE_2)
	assert.EqualValues(t, len(struct2.F6), 2)
	assert.EqualValues(t, *struct2.F6["10"], test.EnumSimple_ENUM_VALUE_1)
	assert.Nil(t, struct2.F6["20"])
	assert.EqualValues(t, len(struct2.F7), 2)
	assert.EqualValues(t, struct2.F7["10"], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2)
	assert.EqualValues(t, struct2.F7["20"], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2 | test.FlagsSimple_FLAG_VALUE_3)
	assert.EqualValues(t, len(struct2.F8), 2)
	assert.EqualValues(t, *struct2.F8["10"], test.FlagsSimple_FLAG_VALUE_1 | test.FlagsSimple_FLAG_VALUE_2)
	assert.Nil(t, struct2.F8["20"])
	assert.EqualValues(t, len(struct2.F9), 2)
	assert.EqualValues(t, struct2.F9["10"].Id, 48)
	assert.EqualValues(t, struct2.F9["20"].Id, 65)
	assert.EqualValues(t, len(struct2.F10), 2)
	assert.EqualValues(t, struct2.F10["10"].Id, 48)
	assert.Nil(t, struct2.F10["20"])
}
