package com.thomasjensen.checkstyle.addons.build.tasks;
/*
 * Checkstyle-Addons - Additional Checkstyle checks
 * Copyright (c) 2015-2024, the Checkstyle Addons contributors
 *
 * This program is free software: you can redistribute it and/or modify it under the
 * terms of the GNU General Public License, version 3, as published by the Free
 * Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with this
 * program.  If not, see <http://www.gnu.org/licenses/>.
 */

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import javax.annotation.Nonnull;

import org.gradle.api.DefaultTask;
import org.gradle.api.GradleException;
import org.gradle.api.Project;
import org.gradle.api.plugins.BasePlugin;
import org.gradle.api.provider.Property;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.Optional;
import org.gradle.api.tasks.OutputFile;
import org.gradle.api.tasks.TaskAction;
import org.gradle.api.tasks.TaskInputs;

import com.thomasjensen.checkstyle.addons.build.BuildUtil;


/**
 * Gradle task to generate a 'pom.properties' file.
 */
public class GeneratePomPropsTask
    extends DefaultTask
{
    private final File pluginPomProps = new File(getTemporaryDir(), "pom.properties");

    private final Property<String> appendix;



    /**
     * Constructor.
     */
    public GeneratePomPropsTask()
    {
        super();
        setGroup(BasePlugin.BUILD_GROUP);

        final Project project = getProject();
        appendix = project.getObjects().property(String.class);
        appendix.set((String) null);
        updateDescription();

        // Task Inputs: the property values identifying the artifact
        final TaskInputs inputs = getInputs();
        inputs.property(BuildUtil.GROUP_ID, project.getGroup());
        inputs.property(BuildUtil.ARTIFACT_ID, project.getName());
        inputs.property(BuildUtil.VERSION, project.getVersion().toString());

        // Task Outputs: the 'pom.properties' file
        getOutputs().file(pluginPomProps);
    }



    @TaskAction
    public void generateFile()
    {
        final TaskInputs inputs = getInputs();
        String effectiveArtifactId = (String) inputs.getProperties().get(BuildUtil.ARTIFACT_ID);
        if (appendix.isPresent()) {
            effectiveArtifactId += '-' + appendix.get();
        }

        List<String> entries = new ArrayList<>();
        entries.add("#Generated by Maven");
        entries.add("#@buildTimestamp@");
        entries.add(BuildUtil.GROUP_ID + "=" + inputs.getProperties().get(BuildUtil.GROUP_ID));
        entries.add(BuildUtil.ARTIFACT_ID + "=" + effectiveArtifactId);
        entries.add(BuildUtil.VERSION + "=" + inputs.getProperties().get(BuildUtil.VERSION));

        //noinspection ResultOfMethodCallIgnored
        pluginPomProps.getParentFile().mkdirs();
        try {
            Files.write(pluginPomProps.toPath(), entries, StandardCharsets.UTF_8);
        }
        catch (IOException e) {
            throw new GradleException("Failed to create file: " + pluginPomProps.getAbsolutePath(), e);
        }
    }



    @Input
    @Nonnull
    @Optional
    public Property<String> getAppendix()
    {
        return appendix;
    }



    /**
     * Update the task description with the value of the {@code appendix} property.
     */
    public void updateDescription()
    {
        final String appendixValue = appendix.getOrNull();
        String text = "Create file '" + pluginPomProps.getName() + "' for use in JAR ";
        if (appendixValue != null && appendixValue.length() > 0) {
            text += "(appendix: " + appendixValue + ")";
        }
        else {
            text += "(no appendix)";
        }
        setDescription(text);
    }



    @OutputFile
    public File getPluginPomProps()
    {
        return pluginPomProps;
    }
}
