#ifndef EXTERNAL_H
#define EXTERNAL_H

// --- internally added ---
typedef unsigned char bool;
typedef signed char int8_t;
typedef unsigned char uint8_t;
typedef signed short int16_t;
typedef unsigned short uint16_t;
typedef signed int int32_t;
typedef unsigned int uint32_t;
typedef signed long int64_t;
typedef unsigned long uint64_t;
typedef long ssize_t;
typedef unsigned long size_t;
#define true 1
#define false 0
#define NULL ((void*)0)
#define ARRAY_SIZE(x) (sizeof(x)/sizeof(x[0]))
#define offsetof(TYPE, MEMBER) ((unsigned long) &((TYPE *)0)->MEMBER)
#define to_container(type, member, ptr) ((type *)((char *)(ptr)-(unsigned long)(&((type *)0)->member)))

int dprintf(int fd, const char *format, ...);
void abort(void);

static void c2_assert(bool condition, const char* location, const char* condstr) {
  if (condition) return;
  static const char me[] = "c2c";
  dprintf(2, "%s: %s: Assertion '%s' failed\n", me, location, condstr);
  abort();
}

static bool c2_strequal(const char* s1, const char* s2) {
  while (*s1 == *s2) {
    if (*s1 == 0) return true;
    s1++;
    s2++;
  }
  return false;
}

#endif

// --- module c2 ---

typedef char c2_c_char;

typedef uint8_t c2_c_uchar;

typedef int16_t c2_c_short;

typedef uint16_t c2_c_ushort;

typedef int32_t c2_c_int;

typedef uint32_t c2_c_uint;

typedef int64_t c2_c_longlong;

typedef uint64_t c2_c_ulonglong;

typedef float c2_c_float;

typedef double c2_c_double;

typedef int64_t c2_c_long;

typedef uint64_t c2_c_ulong;

typedef uint64_t c2_c_size;

typedef int64_t c2_c_ssize;

#define c2_min_i8 -128l
#define c2_max_i8 127
#define c2_min_u8 0
#define c2_max_u8 255
#define c2_min_i16 -32768l
#define c2_max_i16 32767
#define c2_min_u16 0
#define c2_max_u16 65535
#define c2_min_i32 -2147483648l
#define c2_max_i32 2147483647
#define c2_min_u32 0
#define c2_max_u32 4294967295
#define c2_min_i64 -9223372036854775807l
#define c2_max_i64 9223372036854775807l
#define c2_min_u64 0
#define c2_max_u64 18446744073709551615lu
#define c2_min_isize -9223372036854775807l
#define c2_max_isize 9223372036854775807l
#define c2_min_usize 0
#define c2_max_usize 18446744073709551615lu

// --- module c_errno ---

#define ENOENT 2
#define EEXIST 17
int32_t* __errno_location(void);

// --- module csetjmp ---
typedef struct __jmp_buf_tag_ __jmp_buf_tag;

struct __jmp_buf_tag_ {
   char data[200];
};

typedef __jmp_buf_tag* jmp_buf;

int32_t setjmp(jmp_buf __env);
void longjmp(jmp_buf __env, int32_t __val);

// --- module ctype ---

int32_t isalnum(int32_t c);
int32_t isalpha(int32_t c);
int32_t isdigit(int32_t c);
int32_t islower(int32_t c);
int32_t isprint(int32_t c);
int32_t isspace(int32_t c);
int32_t isupper(int32_t c);
int32_t isxdigit(int32_t c);
int32_t toupper(int32_t c);

// --- module libc_dirent ---
typedef struct DIR_ DIR;
typedef struct dirent_ dirent;

struct DIR_ {
};

typedef int32_t (*FilterFn)(const dirent* _arg0);

typedef int32_t (*DirentCompareFn)(const dirent** _arg0, const dirent** _arg1);

struct dirent_ {
   uint64_t d_ino;
   int64_t d_off;
   uint16_t d_reclen;
   uint8_t d_type;
   char d_name[256];
};

#define DT_DIR 4
#define DT_REG 8
DIR* opendir(const char* name);
int32_t closedir(DIR* dirp);
dirent* readdir(DIR* dirp);

// --- module libc_fcntl ---

#define O_RDONLY 0
#define O_WRONLY 01
#define O_CREAT 0100
#define O_NOCTTY 0400
#define O_TRUNC 01000
#define O_NONBLOCK 04000
#define O_DIRECTORY 0200000
#define O_NOFOLLOW 0400000
#define F_SETFD 2
#define AT_FDCWD -100
#define FD_CLOEXEC 1
int32_t open(const char* __file, int32_t __oflag, ...);
int32_t openat(int32_t dirfd, const char* pathname, int32_t flags, ...);
int32_t fcntl(int32_t __fd, int32_t __cmd, ...);

// --- module stdio ---
typedef struct _IO_marker_ _IO_marker;
typedef struct FILE_ FILE;

struct _IO_marker_ {
   _IO_marker* next;
   FILE* sbuf;
   int32_t _pos;
};

struct FILE_ {
};

typedef uint64_t off_t;

extern FILE* stdout;

extern FILE* stderr;

int32_t fflush(FILE* __stream);
__attribute__((__format__(printf, 2, 3))) 
int32_t fprintf(FILE* __stream, const char* __format, ...);
__attribute__((__format__(printf, 1, 2))) 
int32_t printf(const char* __format, ...);
__attribute__((__format__(printf, 2, 3))) 
int32_t sprintf(char* __s, const char* __format, ...);
int32_t fputs(const char* __s, FILE* __stream);
int32_t puts(const char* __s);
void perror(const char* __s);

// --- module stdlib ---
typedef struct div_t_ div_t;
typedef struct Ldiv_t_ Ldiv_t;
typedef struct random_data_ random_data;
typedef struct drand48_data_ drand48_data;

struct div_t_ {
   int32_t quot;
   int32_t rem;
};

struct Ldiv_t_ {
   int64_t quot;
   int64_t rem;
};

struct random_data_ {
};

struct drand48_data_ {
};

typedef void (*AtExitFn)(void);

typedef void (*OnExitFn)(int32_t _arg0, void* _arg1);

typedef int32_t (*__compar_fn_t)(const void* _arg0, const void* _arg1);

#define EXIT_FAILURE 1
#define EXIT_SUCCESS 0
void* calloc(uint64_t count, uint64_t size);
void* malloc(uint64_t size);
void free(void* ptr);
double strtod(const char* nptr, char** endptr);
uint64_t strtoull(const char* nptr, char** endptr, int32_t base);
void exit(int32_t __status);
void _exit(int32_t __status);
char* getenv(const char* __name);

// --- module string ---

void* memcpy(void* dest, const void* src, uint64_t n);
void* memset(void* b, int32_t c, uint64_t len);
char* strcpy(char* dest, const char* src);
char* strncpy(char* dest, const char* src, uint64_t n);
char* strcat(char* dest, const char* src);
int32_t strcmp(const char* s1, const char* s2);
int32_t strncmp(const char* s1, const char* s2, uint64_t n);
int32_t strcasecmp(const char* s1, const char* s2);
char* strdup(const char* s);
char* strchr(const char* s, int32_t c);
char* strstr(const char* haystack, const char* needle);
char* strtok(char* s, const char* delim);
uint64_t strlen(const char* s);
char* strerror(int32_t errnum);

// --- module sys_stat ---

struct stat {
   uint64_t st_dev;
   uint64_t st_ino;
   uint64_t st_nlink;
   uint32_t st_mode;
   uint32_t st_uid;
   uint32_t st_gid;
   uint64_t st_rdev;
   int64_t st_size;
   int64_t st_blksize;
   int64_t st_blocks;
   int64_t st_atime;
   uint64_t st_atime_nsec;
   uint64_t st_mtime;
   uint64_t st_mtime_nsec;
   uint64_t st_ctime;
   uint64_t st_ctime_nsec;
   uint32_t __unused4;
   uint32_t __unused5;
   int64_t reserved[2];
};

typedef uint32_t Mode;

#define S_IFMT 0170000
#define S_IFREG 0100000
int32_t fstat(int32_t fd, struct stat* buf);
int32_t stat(const char* pathname, struct stat* buf);
int32_t mkdir(const char* __file, uint32_t mode);

// --- module sys_time ---
typedef struct timeval_ timeval;
typedef struct timezone_ timezone;

typedef int64_t time_t;

typedef int64_t suseconds_t;

struct timeval_ {
   time_t tv_sec;
   suseconds_t tv_usec;
};

struct timezone_ {
   int32_t tz_minuteswest;
   int32_t tz_dsttime;
};

int32_t gettimeofday(timeval* tv, timezone* tz);

// --- module sys_utsname ---
typedef struct utsname_ utsname;

#define NAME_LEN 65
struct utsname_ {
   char sysname[65];
   char nodename[65];
   char release[65];
   char version[65];
   char machine[65];
   char domainname[65];
};

int32_t uname(utsname* buf);

// --- module unistd ---

typedef int32_t pid_t;

#define STDOUT_FILENO 1
#define STDERR_FILENO 2
char* getcwd(char* buf, uint64_t size);
int32_t chdir(const char* path);
int32_t fchdir(int32_t fd);
int32_t close(int32_t fd);
int64_t read(int32_t fd, void* buf, uint64_t count);
int32_t isatty(int32_t fd);
int64_t write(int32_t fd, const void* buf, uint64_t count);
int32_t pipe(int32_t* pipefd);
int32_t fsync(int32_t fd);
pid_t fork(void);
pid_t waitpid(pid_t pid, int32_t* wstatus, int32_t options);
int32_t dup(int32_t oldfd);
int32_t execv(const char* pathname, char** argv);

// --- module stdarg ---
// Note: this module is a special case and is custom generated

#define va_list __builtin_va_list
#define va_start __builtin_va_start
#define va_end __builtin_va_end

int32_t vdprintf(int32_t __fd, const char* __fmt, va_list __arg);
int32_t vsprintf(char* str, const char* format, va_list __ap);
int32_t vsnprintf(char* str, uint64_t size, const char* format, va_list __ap);


// --- module dlfcn ---

#define RTLD_NOW 0x2
#define RTLD_LOCAL 0
void* dlopen(const char* file, int32_t mode);
int32_t dlclose(void* handle);
void* dlsym(void* handle, const char* name);
char* dlerror(void);
// WARNING: this file is auto-generated by the C2 compiler.
// Any changes you make might be lost!

#include "external.h"


// --- module git_version ---

#define git_version_Describe "f2ede6a"

// --- module file_utils ---
typedef struct file_utils_Reader_ file_utils_Reader;
typedef struct file_utils_Writer_ file_utils_Writer;

struct file_utils_Reader_ {
   void* region;
   uint32_t size;
   int32_t errno;
};

static uint8_t file_utils_empty = 0;

#define file_utils_Err_not_a_file 2001
static bool file_utils_Reader_open(file_utils_Reader* file, const char* filename);
static void file_utils_Reader_close(file_utils_Reader* file);
static bool file_utils_Reader_isOpen(const file_utils_Reader* file);
static const uint8_t* file_utils_Reader_data(file_utils_Reader* file);
static const char* file_utils_Reader_char_data(file_utils_Reader* file);
static bool file_utils_Reader_isEmpty(const file_utils_Reader* file);
static const char* file_utils_find_char(const char* full, char delim);
static int32_t file_utils_create_dir(const char* path, bool follow);
static int32_t file_utils_create_directory(const char* path);
static bool file_utils_exists(const char* filename);
struct file_utils_Writer_ {
   char msg[512];
};

static bool file_utils_Writer_write(file_utils_Writer* writer, const char* filename, const uint8_t* data, uint32_t len);
static const char* file_utils_Writer_getError(const file_utils_Writer* writer);
static bool file_utils_Reader_open(file_utils_Reader* file, const char* filename)
{
   file->region = NULL;
   file->size = 0;
   int32_t fd = open(filename, O_RDONLY);
   if ((fd == -1)) {
      file->errno = *__errno_location();
      return false;
   }
   struct stat statbuf;
   int32_t err = fstat(fd, &statbuf);
   if (err) {
      file->errno = *__errno_location();
      return false;
   }
   if (((statbuf.st_mode & S_IFMT) != S_IFREG)) {
      close(fd);
      file->errno = file_utils_Err_not_a_file;
      return false;
   }
   file->size = ((uint32_t)(statbuf.st_size));
   if ((file->size == 0)) {
      file->region = &file_utils_empty;
   } else {
      file->region = malloc((file->size + 1));
      ssize_t numread = read(fd, file->region, file->size);
      if ((numread != file->size)) return false;

      uint8_t* ptr = file->region;
      ptr[file->size] = 0;
   }
   close(fd);
   return true;
}

static void file_utils_Reader_close(file_utils_Reader* file)
{
   if (file->size) {
      free(file->region);
      file->region = NULL;
   }
}

static bool file_utils_Reader_isOpen(const file_utils_Reader* file)
{
   return (file->region != NULL);
}

static const uint8_t* file_utils_Reader_data(file_utils_Reader* file)
{
   return ((uint8_t*)(file->region));
}

static const char* file_utils_Reader_char_data(file_utils_Reader* file)
{
   return ((char*)(file->region));
}

static bool file_utils_Reader_isEmpty(const file_utils_Reader* file)
{
   return (file->size == 0);
}

static const char* file_utils_find_char(const char* full, char delim)
{
   while (1) {
      char c = *full;
      if ((c == delim)) return full;

      if ((c == 0)) break;

      full++;
   }
   return NULL;
}

static int32_t file_utils_create_dir(const char* path, bool follow)
{
   int32_t err = mkdir(path, 0777);
   if ((err && (*__errno_location() != EEXIST))) return -1;

   if (!follow) return 0;

   int32_t fd = openat(AT_FDCWD, path, ((((O_RDONLY | O_NOCTTY) | O_NONBLOCK) | O_NOFOLLOW) | O_DIRECTORY));
   if ((fd == -1)) return *__errno_location();

   err = fchdir(fd);
   if ((err == -1)) return -1;

   close(fd);
   return 0;
}

static int32_t file_utils_create_directory(const char* path)
{
   int32_t fd = openat(AT_FDCWD, ".", O_RDONLY);
   if ((fd == -1)) return *__errno_location();

   char tmp[128];
   const char* cp = path;
   int32_t err = 0;
   while (*cp) {
      const char* slash = file_utils_find_char(cp, '/');
      if (slash) {
         size_t len = ((size_t)((slash - cp)));
         memcpy(tmp, cp, len);
         tmp[len] = 0;
         cp = (slash + 1);
         err = file_utils_create_dir(tmp, true);
         if ((err != 0)) break;

      } else {
         err = file_utils_create_dir(cp, false);
         break;
      }
   }
   int32_t errno_ = 0;
   if (err) errno_ = *__errno_location();
   fchdir(fd);
   close(fd);
   return errno_;
}

static bool file_utils_exists(const char* filename)
{
   struct stat statbuf;
   return ((stat(filename, &statbuf) == 0));
}

static bool file_utils_Writer_write(file_utils_Writer* writer, const char* filename, const uint8_t* data, uint32_t len)
{
   writer->msg[0] = 0;
   int32_t fd = open(filename, ((O_CREAT | O_WRONLY) | O_TRUNC), 0660);
   if ((fd == -1)) {
      sprintf(writer->msg, "error opening %s: %s", filename, strerror(*__errno_location()));
      return false;
   }
   int64_t written = write(fd, data, len);
   if ((written != len)) {
      sprintf(writer->msg, "error writing %s: %s", filename, strerror(*__errno_location()));
      close(fd);
      return false;
   }
   close(fd);
   return true;
}

static const char* file_utils_Writer_getError(const file_utils_Writer* writer)
{
   return writer->msg;
}


// --- module yaml ---
typedef struct yaml_Node_ yaml_Node;
typedef struct yaml_StackLevel_ yaml_StackLevel;
typedef struct yaml_Data_ yaml_Data;
typedef struct yaml_Iter_ yaml_Iter;
typedef struct yaml_Location_ yaml_Location;
typedef struct yaml_Token_ yaml_Token;
typedef struct yaml_Tokenizer_ yaml_Tokenizer;
typedef struct yaml_Parser_ yaml_Parser;

typedef enum {
   yaml_NodeKind_Unknown,
   yaml_NodeKind_Scalar,
   yaml_NodeKind_Map,
   yaml_NodeKind_Sequence,
   _yaml_NodeKind_max = 255
} __attribute__((packed)) yaml_NodeKind;

struct yaml_Node_ {
   yaml_NodeKind kind;
   uint32_t next_idx;
   uint32_t name_idx;
   union {
      uint32_t text_idx;
      uint32_t child_idx;
   };
};

struct yaml_StackLevel_ {
   int32_t indent;
   yaml_Node* node;
   yaml_Node* last_child;
};

struct yaml_Data_ {
   char* text;
   uint32_t text_size;
   char* text_cur;
   yaml_Node* nodes;
   uint32_t nodes_count;
   yaml_Node* nodes_cur;
   yaml_StackLevel* stack;
};

#define yaml_MaxDepth 8
static const char* yaml_node_names[4] = { "UNK", "SCA", "MAP", "SEQ" };

static void yaml_Data_init(yaml_Data* d, uint32_t text_size, uint32_t nodes_count, yaml_StackLevel* stack);
static void yaml_Data_destroy(yaml_Data* d);
static void yaml_Data_resize_nodes(yaml_Data* d);
static void yaml_Data_resize_text(yaml_Data* d);
static yaml_Node* yaml_Data_add_node(yaml_Data* d, yaml_NodeKind kind, uint32_t name_idx);
__inline__ 
static uint32_t yaml_Data_node2idx(const yaml_Data* d, const yaml_Node* n);
static uint32_t yaml_Data_add_text(yaml_Data* d, const char* text, uint32_t len);
static void yaml_Parser_dump(const yaml_Parser* p, bool verbose);
static void yaml_Data_dump(const yaml_Data* d, bool verbose);
static yaml_Node* yaml_Data_idx2node(const yaml_Data* d, uint32_t idx);
static void yaml_Data_dump_node(const yaml_Data* d, const yaml_Node* n, int32_t indent);
struct yaml_Iter_ {
   const void* data;
   const yaml_Node* node;
};

static bool yaml_Node_isMap(const yaml_Node* n);
static bool yaml_Node_isSequence(const yaml_Node* n);
static bool yaml_Node_isScalar(const yaml_Node* n);
static const yaml_Node* yaml_Parser_getRoot(const yaml_Parser* p);
static const char* yaml_Parser_getScalarValue(const yaml_Parser* p, const char* path);
static const yaml_Node* yaml_Parser_findNode(const yaml_Parser* p, const char* path);
static const yaml_Node* yaml_Data_findNode(const yaml_Data* d, const char* path);
static const yaml_Node* yaml_Data_findChildNode(const yaml_Data* d, const char* path, uint32_t next);
static yaml_Iter yaml_Parser_getNodeChildIter(const yaml_Parser* p, const yaml_Node* n);
static void yaml_Iter_next(yaml_Iter* iter);
static bool yaml_Iter_done(const yaml_Iter* iter);
static const char* yaml_Iter_getName(const yaml_Iter* iter);
static const char* yaml_Iter_getValue(const yaml_Iter* iter);
static yaml_Iter yaml_Iter_getChildIter(yaml_Iter* parent);
static const char* yaml_Iter_getChildScalarValue(yaml_Iter* iter, const char* path);
static const char* yaml_starts_with(const char* full, const char* start);
struct yaml_Location_ {
   uint32_t line;
   uint32_t column;
};

typedef enum {
   yaml_TokenKind_None,
   yaml_TokenKind_Plain_Scalar,
   yaml_TokenKind_Single_Quoted_Scalar,
   yaml_TokenKind_Double_Quoted_Scalar,
   yaml_TokenKind_Colon,
   yaml_TokenKind_Dash,
   yaml_TokenKind_Indent,
   yaml_TokenKind_Dedent,
   yaml_TokenKind_Doc_Start,
   yaml_TokenKind_Doc_End,
   yaml_TokenKind_Directive,
   yaml_TokenKind_Eof,
   yaml_TokenKind_Error,
   _yaml_TokenKind_max = 255
} __attribute__((packed)) yaml_TokenKind;

struct yaml_Token_ {
   yaml_Location loc;
   yaml_TokenKind kind;
   bool same_line;
   union {
      const char* error_msg;
      uint32_t text_idx;
      int32_t indent;
   };
};

struct yaml_Tokenizer_ {
   const char* cur;
   yaml_Location loc;
   const char* input_start;
   char* error_msg;
   int32_t cur_indent;
   bool same_line;
   yaml_Data* data;
   yaml_Token next;
};

static const char* yaml_token_names[13] = {
   "none",
   "scalar",
   "'scalar'",
   "\"scalar\"",
   ":",
   "-",
   "indent",
   "dedent",
   "---",
   "...",
   "%",
   "eof",
   "error"
};

static const char* yaml_Location_str(const yaml_Location* loc);
static const char* yaml_Token_str(const yaml_Token* tok);
static void yaml_Tokenizer_init(yaml_Tokenizer* t, const char* input, yaml_Data* d, char* error_msg);
static void yaml_Tokenizer_lex(yaml_Tokenizer* t, yaml_Token* result);
static yaml_Token* yaml_Tokenizer_lex_next(yaml_Tokenizer* t);
static bool yaml_Tokenizer_lex_indent(yaml_Tokenizer* t, yaml_Token* result);
static void yaml_Tokenizer_lex_comment(yaml_Tokenizer* t);
static void yaml_Tokenizer_lex_directive(yaml_Tokenizer* t, yaml_Token* result);
static void yaml_Tokenizer_lex_quoted_string(yaml_Tokenizer* t, yaml_Token* result, char delim);
static bool yaml_is_string(char c);
static void yaml_Tokenizer_lex_string(yaml_Tokenizer* t, yaml_Token* result);
static void yaml_Tokenizer_error(yaml_Tokenizer* t, yaml_Token* result);
#define yaml_MaxDiag 256
struct yaml_Parser_ {
   yaml_Token token;
   yaml_Tokenizer tokenizer;
   int32_t cur_indent;
   bool doc_started;
   bool in_document;
   yaml_StackLevel stack[8];
   uint32_t stack_size;
   yaml_Data data;
   __jmp_buf_tag jmp_err;
   char message[256];
};

static yaml_Parser* yaml_Parser_create(void);
static void yaml_Parser_destroy(yaml_Parser* p);
static bool yaml_Parser_parse(yaml_Parser* p, const char* input);
static const char* yaml_Parser_getMessage(const yaml_Parser* p);
__attribute__((__format__(printf, 2, 3))) 
static void yaml_Parser_error(yaml_Parser* p, const char* format, ...);
static void yaml_Parser_consumeToken(yaml_Parser* p);
static void yaml_Parser_expectAndConsume(yaml_Parser* p, yaml_TokenKind kind);
static void yaml_Parser_parse_doc(yaml_Parser* p);
static void yaml_Parser_parse_node(yaml_Parser* p);
static void yaml_Parser_parse_value(yaml_Parser* p);
static void yaml_Parser_parse_node_or_value(yaml_Parser* p);
static void yaml_Parser_doc_start(yaml_Parser* p);
static void yaml_Parser_doc_end(yaml_Parser* p);
static void yaml_Parser_add_scalar_value(yaml_Parser* p, uint32_t value_idx);
static void yaml_Parser_pop(yaml_Parser* p);
static void yaml_Parser_push_root(yaml_Parser* p);
static void yaml_Parser_push_node(yaml_Parser* p, yaml_Node* n, yaml_NodeKind parent_kind, int32_t indent);
static void yaml_Data_init(yaml_Data* d, uint32_t text_size, uint32_t nodes_count, yaml_StackLevel* stack)
{
   d->text = malloc(text_size);
   d->text_size = text_size;
   d->text_cur = (d->text + 1);
   d->text[0] = 0;
   d->nodes = malloc((nodes_count * 16));
   d->nodes_count = nodes_count;
   d->nodes_cur = &d->nodes[1];
   memset(&d->nodes[0], 0, 16);
   d->stack = stack;
}

static void yaml_Data_destroy(yaml_Data* d)
{
   free(d->text);
   free(d->nodes);
}

static void yaml_Data_resize_nodes(yaml_Data* d)
{
   uint32_t idx = ((uint32_t)((d->nodes_cur - d->nodes)));
   d->nodes_count *= 2;
   yaml_Node* nodes2 = malloc((d->nodes_count * 16));
   memcpy(nodes2, d->nodes, (idx * 16));
   for (uint32_t i = 0; (i < yaml_MaxDepth); i++) {
      yaml_StackLevel* sl = &d->stack[i];
      if (sl->node) {
         uint32_t node_idx = ((uint32_t)((sl->node - d->nodes)));
         sl->node = &nodes2[node_idx];
      }
      if (sl->last_child) {
         uint32_t last_child_idx = ((uint32_t)((sl->last_child - d->nodes)));
         sl->last_child = &nodes2[last_child_idx];
      }
   }
   free(d->nodes);
   d->nodes = nodes2;
   d->nodes_cur = &d->nodes[idx];
}

static void yaml_Data_resize_text(yaml_Data* d)
{
   uint32_t idx = ((uint32_t)((d->text_cur - d->text)));
   d->text_size *= 2;
   char* text2 = malloc(d->text_size);
   memcpy(text2, d->text, (idx + 1));
   free(d->text);
   d->text = text2;
   d->text_cur = &d->text[idx];
}

static yaml_Node* yaml_Data_add_node(yaml_Data* d, yaml_NodeKind kind, uint32_t name_idx)
{
   uint32_t idx = ((uint32_t)((d->nodes_cur - d->nodes)));
   if ((idx >= (d->nodes_count - 1))) yaml_Data_resize_nodes(d);
   yaml_Node* result = d->nodes_cur;
   d->nodes_cur++;
   result->kind = kind;
   result->next_idx = 0;
   result->name_idx = name_idx;
   result->child_idx = 0;
   return result;
}

__inline__ 
static uint32_t yaml_Data_node2idx(const yaml_Data* d, const yaml_Node* n)
{
   return ((uint32_t)((n - d->nodes)));
}

static uint32_t yaml_Data_add_text(yaml_Data* d, const char* text, uint32_t len)
{
   uint32_t idx = ((uint32_t)((d->text_cur - d->text)));
   while ((((idx + len) + 1) >= d->text_size)) yaml_Data_resize_text(d);
   memcpy(d->text_cur, text, len);
   d->text_cur[len] = 0;
   d->text_cur += (len + 1);
   return idx;
}

static void yaml_Parser_dump(const yaml_Parser* p, bool verbose)
{
   yaml_Data_dump(&p->data, verbose);
}

static void yaml_Data_dump(const yaml_Data* d, bool verbose)
{
   uint32_t node_count = ((uint32_t)((d->nodes_cur - d->nodes)));
   if (verbose) {
      printf("Text %u/%u\n", ((uint32_t)((d->text_cur - d->text))), d->text_size);
      const char* cp = (d->text + 1);
      while ((cp < d->text_cur)) {
         uint32_t len = ((uint32_t)(strlen(cp)));
         uint32_t offset = ((uint32_t)((cp - d->text)));
         printf("  [%3u] %s\n", offset, cp);
         cp += (len + 1);
      }
      printf("Nodes %u/%u\n", node_count, d->nodes_count);
      for (uint32_t i = 1; (i < node_count); i++) {
         const yaml_Node* n = &d->nodes[i];
         printf("  [%2u] %s  next %3u  name %3u  value/child %3u\n", i, yaml_node_names[n->kind], n->next_idx, n->name_idx, n->text_idx);
      }
   }
   if ((node_count > 1)) yaml_Data_dump_node(d, &d->nodes[1], 0);
}

static yaml_Node* yaml_Data_idx2node(const yaml_Data* d, uint32_t idx)
{
   return &d->nodes[idx];
}

static void yaml_Data_dump_node(const yaml_Data* d, const yaml_Node* n, int32_t indent)
{
   for (int32_t i = 0; (i < indent); i++) printf("   ");
   printf("[%2u] %s", yaml_Data_node2idx(d, n), yaml_node_names[n->kind]);
   printf("  name: ");
   if (n->name_idx) printf("%s", &d->text[n->name_idx]);
   else printf("-");
   printf("  value: ");
   switch (n->kind) {
   case yaml_NodeKind_Unknown:
      printf("-\n");
      break;
   case yaml_NodeKind_Scalar:
      if (n->text_idx) printf("%s", &d->text[n->text_idx]);
      printf("\n");
      break;
   case yaml_NodeKind_Map:
      __attribute__((fallthrough));
   case yaml_NodeKind_Sequence:
      printf("-\n");
      if (n->child_idx) yaml_Data_dump_node(d, yaml_Data_idx2node(d, n->child_idx), (indent + 1));
      break;
   }
   if (n->next_idx) {
      yaml_Data_dump_node(d, yaml_Data_idx2node(d, n->next_idx), indent);
   }
}

static bool yaml_Node_isMap(const yaml_Node* n)
{
   return (n->kind == yaml_NodeKind_Map);
}

static bool yaml_Node_isSequence(const yaml_Node* n)
{
   return (n->kind == yaml_NodeKind_Sequence);
}

static bool yaml_Node_isScalar(const yaml_Node* n)
{
   return (n->kind == yaml_NodeKind_Scalar);
}

static const yaml_Node* yaml_Parser_getRoot(const yaml_Parser* p)
{
   uint32_t node_count = (((uint32_t)((p->data.nodes_cur - p->data.nodes))) - 1);
   if ((node_count == 0)) return NULL;

   return &p->data.nodes[1];
}

static const char* yaml_Parser_getScalarValue(const yaml_Parser* p, const char* path)
{
   const yaml_Node* n = yaml_Parser_findNode(p, path);
   if ((n && yaml_Node_isScalar(n))) return &p->data.text[n->text_idx];

   return NULL;
}

static const yaml_Node* yaml_Parser_findNode(const yaml_Parser* p, const char* path)
{
   return yaml_Data_findNode(&p->data, path);
}

static const yaml_Node* yaml_Data_findNode(const yaml_Data* d, const char* path)
{
   uint32_t node_count = (((uint32_t)((d->nodes_cur - d->nodes))) - 1);
   if ((node_count == 0)) return NULL;

   const yaml_Node* root = &d->nodes[1];
   if ((root->kind == yaml_NodeKind_Sequence)) return NULL;

   return yaml_Data_findChildNode(d, path, root->child_idx);
}

static const yaml_Node* yaml_Data_findChildNode(const yaml_Data* d, const char* path, uint32_t next)
{
   while (next) {
      const yaml_Node* node = yaml_Data_idx2node(d, next);
      if (node->name_idx) {
         const char* name = &d->text[node->name_idx];
         const char* rest = yaml_starts_with(path, name);
         if (rest) {
            path = rest;
            if ((path[0] == 0)) return node;

            if ((node->kind == yaml_NodeKind_Sequence)) return NULL;

            next = node->child_idx;
            continue;
         }
      }
      next = node->next_idx;
   }
   return NULL;
}

static yaml_Iter yaml_Parser_getNodeChildIter(const yaml_Parser* p, const yaml_Node* n)
{
   yaml_Iter iter = { .data = &p->data, .node = NULL };
   if (((n && (n->kind != yaml_NodeKind_Scalar)) && n->child_idx)) {
      iter.node = yaml_Data_idx2node(&p->data, n->child_idx);
   }
   return iter;
}

static void yaml_Iter_next(yaml_Iter* iter)
{
   const yaml_Data* d = ((yaml_Data*)(iter->data));
   if (iter->node) {
      if (iter->node->next_idx) iter->node = yaml_Data_idx2node(d, iter->node->next_idx);
      else iter->node = NULL;
   }
}

static bool yaml_Iter_done(const yaml_Iter* iter)
{
   return (iter->node == NULL);
}

static const char* yaml_Iter_getName(const yaml_Iter* iter)
{
   const yaml_Data* d = ((yaml_Data*)(iter->data));
   if (iter->node) return &d->text[iter->node->name_idx];

   return NULL;
}

static const char* yaml_Iter_getValue(const yaml_Iter* iter)
{
   const yaml_Data* d = ((yaml_Data*)(iter->data));
   if ((iter->node && (iter->node->kind == yaml_NodeKind_Scalar))) return &d->text[iter->node->text_idx];

   return NULL;
}

static yaml_Iter yaml_Iter_getChildIter(yaml_Iter* parent)
{
   yaml_Iter iter = { .data = parent->data, .node = NULL };
   if ((parent->node == NULL)) return iter;

   const yaml_Node* n = parent->node;
   if (((n->kind != yaml_NodeKind_Scalar) && n->child_idx)) {
      const yaml_Data* d = ((yaml_Data*)(iter.data));
      iter.node = yaml_Data_idx2node(d, n->child_idx);
   }
   return iter;
}

static const char* yaml_Iter_getChildScalarValue(yaml_Iter* iter, const char* path)
{
   if (!iter->node) return NULL;

   if ((iter->node->kind == yaml_NodeKind_Sequence)) return NULL;

   const yaml_Data* d = ((yaml_Data*)(iter->data));
   const yaml_Node* n = yaml_Data_findChildNode(d, path, iter->node->child_idx);
   if ((n && yaml_Node_isScalar(n))) return &d->text[n->text_idx];

   return NULL;
}

static const char* yaml_starts_with(const char* full, const char* start)
{
   uint32_t len = ((uint32_t)(strlen(start)));
   if ((strncmp(full, start, len) == 0)) {
      full += len;
      if ((full[0] == '.')) return (full + 1);

      if ((full[0] == 0)) return full;

   }
   return NULL;
}

static const char* yaml_Location_str(const yaml_Location* loc)
{
   static char msg[32];
   sprintf(msg, "at line %u:%u", loc->line, loc->column);
   return msg;
}

static const char* yaml_Token_str(const yaml_Token* tok)
{
   return yaml_token_names[tok->kind];
}

static void yaml_Tokenizer_init(yaml_Tokenizer* t, const char* input, yaml_Data* d, char* error_msg)
{
   memset(t, 0, 72);
   t->cur = input;
   t->input_start = input;
   t->loc.line = 1;
   t->loc.column = 1;
   t->error_msg = error_msg;
   t->data = d;
   t->next.kind = yaml_TokenKind_None;
}

static void yaml_Tokenizer_lex(yaml_Tokenizer* t, yaml_Token* result)
{
   if ((t->next.kind != yaml_TokenKind_None)) {
      memcpy(result, &t->next, 24);
      t->next.kind = yaml_TokenKind_None;
      return;
   }
   result->same_line = t->same_line;
   t->same_line = true;
   result->text_idx = 0;
   while (1) {
      if ((((((t->loc.column == 1) && t->cur_indent) && (*t->cur != ' ')) && (*t->cur != '\r')) && (*t->cur != '\n'))) {
         result->loc = t->loc;
         result->kind = yaml_TokenKind_Dedent;
         result->indent = 0;
         t->cur_indent = 0;
         t->same_line = false;
         return;
      }
      switch (*t->cur) {
      case 0:
         result->loc = t->loc;
         result->kind = yaml_TokenKind_Eof;
         return;
      case '\t':
         sprintf(t->error_msg, "file contains TAB characters %s", yaml_Location_str(&t->loc));
         yaml_Tokenizer_error(t, result);
         return;
      case '\r':
         t->cur++;
         if ((*t->cur != '\n')) {
            sprintf(t->error_msg, "unexpected char 0x%02X %s", *t->cur, yaml_Location_str(&t->loc));
            yaml_Tokenizer_error(t, result);
            return;
         }
         __attribute__((fallthrough));
      case '\n':
         t->cur++;
         t->loc.line++;
         t->loc.column = 1;
         t->same_line = true;
         result->same_line = false;
         break;
      case ' ':
         if ((t->loc.column == 1)) {
            if (yaml_Tokenizer_lex_indent(t, result)) return;

            break;
         }
         t->cur++;
         t->loc.column++;
         break;
      case '"':
         yaml_Tokenizer_lex_quoted_string(t, result, '"');
         return;
      case '#':
         yaml_Tokenizer_lex_comment(t);
         break;
      case '%':
         yaml_Tokenizer_lex_directive(t, result);
         return;
      case '\'':
         yaml_Tokenizer_lex_quoted_string(t, result, '\'');
         return;
      case '-':
         if ((((t->cur[1] == ' ') || (t->cur[1] == '\r')) || (t->cur[1] == '\n'))) {
            t->cur++;
            result->loc = t->loc;
            result->kind = yaml_TokenKind_Dash;
            t->loc.column++;
            return;
         }
         if ((((t->loc.column == 1) && (t->cur[1] == '-')) && (t->cur[2] == '-'))) {
            t->cur += 3;
            result->loc = t->loc;
            result->kind = yaml_TokenKind_Doc_Start;
            t->loc.column += 3;
            return;
         }
         yaml_Tokenizer_lex_string(t, result);
         return;
      case '.':
         if ((((t->loc.column == 1) && (t->cur[1] == '.')) && (t->cur[2] == '.'))) {
            result->loc = t->loc;
            result->kind = yaml_TokenKind_Doc_End;
            t->cur += 3;
            t->loc.column += 3;
            return;
         }
         yaml_Tokenizer_lex_string(t, result);
         return;
      case ':':
         t->cur++;
         result->loc = t->loc;
         result->kind = yaml_TokenKind_Colon;
         t->loc.column++;
         return;
      default:
         if (yaml_is_string(*t->cur)) {
            yaml_Tokenizer_lex_string(t, result);
            return;
         }
         sprintf(t->error_msg, "unhandled char 0x%02x (%c) %s", *t->cur, isprint(*t->cur) ? *t->cur : ' ', yaml_Location_str(&t->loc));
         yaml_Tokenizer_error(t, result);
         return;
      }
   }
}

static yaml_Token* yaml_Tokenizer_lex_next(yaml_Tokenizer* t)
{
   if ((t->next.kind == yaml_TokenKind_None)) yaml_Tokenizer_lex(t, &t->next);
   return &t->next;
}

static bool yaml_Tokenizer_lex_indent(yaml_Tokenizer* t, yaml_Token* result)
{
   const char* start = t->cur;
   while ((*t->cur == ' ')) t->cur++;
   int32_t indent = ((int32_t)((t->cur - start)));
   result->loc = t->loc;
   t->loc.column += indent;
   if ((t->cur_indent == indent)) return false;

   if ((t->cur_indent > indent)) result->kind = yaml_TokenKind_Dedent;
   else result->kind = yaml_TokenKind_Indent;
   result->indent = indent;
   t->cur_indent = indent;
   return true;
}

static void yaml_Tokenizer_lex_comment(yaml_Tokenizer* t)
{
   const char* start = t->cur;
   t->cur++;
   while (1) {
      switch (*t->cur) {
      case 0:
         __attribute__((fallthrough));
      case '\r':
         __attribute__((fallthrough));
      case '\n':
         t->loc.column += ((t->cur - start));
         return;
      default:
         t->cur++;
         break;
      }
   }
}

static void yaml_Tokenizer_lex_directive(yaml_Tokenizer* t, yaml_Token* result)
{
   t->cur++;
   const char* start = t->cur;
   uint32_t count;
   while (1) {
      switch (*t->cur) {
      case 0:
         __attribute__((fallthrough));
      case '\r':
         __attribute__((fallthrough));
      case '\n':
         goto out;
      default:
         t->cur++;
         break;
      }
   }
   out:
   count = ((uint32_t)((t->cur - start)));
   t->error_msg[count] = 0;
   result->loc = t->loc;
   result->kind = yaml_TokenKind_Directive;
   result->text_idx = yaml_Data_add_text(t->data, start, count);
   t->loc.column += (count + 1);
}

static void yaml_Tokenizer_lex_quoted_string(yaml_Tokenizer* t, yaml_Token* result, char delim)
{
   t->cur++;
   const char* start = t->cur;
   uint32_t count;
   while (1) {
      switch (*t->cur) {
      case 0:
         __attribute__((fallthrough));
      case '\r':
         __attribute__((fallthrough));
      case '\n':
         t->loc.column += ((t->cur - start));
         sprintf(t->error_msg, "unterminated string %s", yaml_Location_str(&t->loc));
         yaml_Tokenizer_error(t, result);
         return;
      default:
         if ((*t->cur == delim)) goto out;

         t->cur++;
         break;
      }
   }
   out:
   count = ((uint32_t)((t->cur - start)));
   t->cur++;
   result->loc = t->loc;
   result->kind = ((delim == '"')) ? yaml_TokenKind_Double_Quoted_Scalar : yaml_TokenKind_Single_Quoted_Scalar;
   result->text_idx = yaml_Data_add_text(t->data, start, count);
   t->loc.column += (count + 2);
}

static bool yaml_is_string(char c)
{
   if (((((((isalpha(c) || isdigit(c)) || (c == '_')) || (c == '-')) || (c == '.')) || (c == '/')) || (c == '~'))) {
      return true;
   }
   return false;
}

static void yaml_Tokenizer_lex_string(yaml_Tokenizer* t, yaml_Token* result)
{
   const char* start = t->cur;
   t->cur++;
   while (1) {
      char c = *t->cur;
      if (yaml_is_string(c)) {
         t->cur++;
         continue;
      }
      if (((c == ' ') && yaml_is_string(t->cur[1]))) {
         t->cur += 2;
         continue;
      }
      break;
   }
   uint32_t count = ((uint32_t)((t->cur - start)));
   result->loc = t->loc;
   result->kind = yaml_TokenKind_Plain_Scalar;
   result->text_idx = yaml_Data_add_text(t->data, start, count);
   t->loc.column += count;
}

static void yaml_Tokenizer_error(yaml_Tokenizer* t, yaml_Token* result)
{
   result->loc = t->loc;
   result->kind = yaml_TokenKind_Error;
   result->error_msg = t->error_msg;
}

static yaml_Parser* yaml_Parser_create(void)
{
   yaml_Parser* p = calloc(1, 816);
   yaml_Data_init(&p->data, 1024, 32, p->stack);
   return p;
}

static void yaml_Parser_destroy(yaml_Parser* p)
{
   yaml_Data_destroy(&p->data);
   free(p);
}

static bool yaml_Parser_parse(yaml_Parser* p, const char* input)
{
   yaml_Tokenizer_init(&p->tokenizer, input, &p->data, p->message);
   p->token.kind = yaml_TokenKind_None;
   int32_t res = setjmp(&p->jmp_err);
   if ((res == 0)) {
      yaml_Parser_consumeToken(p);
      while ((p->token.kind != yaml_TokenKind_Eof)) yaml_Parser_parse_doc(p);
   } else {
      return false;
   }
   return true;
}

static const char* yaml_Parser_getMessage(const yaml_Parser* p)
{
   return p->message;
}

__attribute__((__format__(printf, 2, 3))) 
static void yaml_Parser_error(yaml_Parser* p, const char* format, ...)
{
   va_list args;
   va_start(args, format);
   char* cp = p->message;
   cp += vsnprintf(cp, (yaml_MaxDiag - 1), format, args);
   va_end(args);
   sprintf(cp, " %s", yaml_Location_str(&p->token.loc));
   longjmp(&p->jmp_err, 1);
}

static void yaml_Parser_consumeToken(yaml_Parser* p)
{
   yaml_Tokenizer_lex(&p->tokenizer, &p->token);
   if ((p->token.kind == yaml_TokenKind_Error)) longjmp(&p->jmp_err, 1);
}

static void yaml_Parser_expectAndConsume(yaml_Parser* p, yaml_TokenKind kind)
{
   if ((p->token.kind != kind)) {
      yaml_Parser_error(p, "expected '%s', got '%s'", yaml_token_names[kind], yaml_Token_str(&p->token));
   }
   yaml_Parser_consumeToken(p);
}

static void yaml_Parser_parse_doc(yaml_Parser* p)
{
   while (1) {
      switch (p->token.kind) {
      case yaml_TokenKind_Doc_Start:
         yaml_Parser_consumeToken(p);
         if (p->doc_started) yaml_Parser_doc_end(p);
         yaml_Parser_doc_start(p);
         break;
      case yaml_TokenKind_Doc_End:
         if ((!p->doc_started || !p->in_document)) {
            yaml_Parser_error(p, "END document without start");
         }
         yaml_Parser_consumeToken(p);
         yaml_Parser_doc_end(p);
         return;
      case yaml_TokenKind_Directive:
         yaml_Parser_consumeToken(p);
         break;
      case yaml_TokenKind_Eof:
         return;
      default:
         if (!p->doc_started) yaml_Parser_doc_start(p);
         yaml_Parser_parse_node(p);
         break;
      }
   }
}

static void yaml_Parser_parse_node(yaml_Parser* p)
{
   switch (p->token.kind) {
   case yaml_TokenKind_Plain_Scalar:
      __attribute__((fallthrough));
   case yaml_TokenKind_Single_Quoted_Scalar:
      __attribute__((fallthrough));
   case yaml_TokenKind_Double_Quoted_Scalar: {
      yaml_Node* n = yaml_Data_add_node(&p->data, yaml_NodeKind_Unknown, p->token.text_idx);
      yaml_Parser_push_node(p, n, yaml_NodeKind_Unknown, p->cur_indent);
      yaml_Parser_consumeToken(p);
      yaml_Parser_expectAndConsume(p, yaml_TokenKind_Colon);
      yaml_Parser_parse_value(p);
      break;
   }
   case yaml_TokenKind_Dash: {
      yaml_Parser_consumeToken(p);
      yaml_Node* n = yaml_Data_add_node(&p->data, yaml_NodeKind_Unknown, 0);
      yaml_Parser_push_node(p, n, yaml_NodeKind_Sequence, (p->cur_indent + 1));
      yaml_Parser_parse_node_or_value(p);
      break;
   }
   case yaml_TokenKind_Indent:
      p->cur_indent = p->token.indent;
      yaml_Parser_consumeToken(p);
      break;
   case yaml_TokenKind_Dedent:
      p->cur_indent = p->token.indent;
      yaml_Parser_consumeToken(p);
      yaml_Parser_pop(p);
      break;
   case yaml_TokenKind_Doc_Start:
      __attribute__((fallthrough));
   case yaml_TokenKind_Doc_End:
      break;
   default:
      yaml_Parser_error(p, "%s() unhandled token '%s'", "parse_node", yaml_Token_str(&p->token));
      break;
   }
}

static void yaml_Parser_parse_value(yaml_Parser* p)
{
   switch (p->token.kind) {
   case yaml_TokenKind_Plain_Scalar:
      __attribute__((fallthrough));
   case yaml_TokenKind_Single_Quoted_Scalar:
      __attribute__((fallthrough));
   case yaml_TokenKind_Double_Quoted_Scalar:
      if (p->token.same_line) {
         yaml_Parser_add_scalar_value(p, p->token.text_idx);
         yaml_Parser_consumeToken(p);
      } else {
         c2_assert((0) != 0, "common/yaml/yaml_parser.c2:173: yaml.Parser.parse_value", "0");
      }
      return;
   case yaml_TokenKind_Dash: {
      yaml_Parser_consumeToken(p);
      yaml_Node* n = yaml_Data_add_node(&p->data, yaml_NodeKind_Unknown, 0);
      yaml_Parser_push_node(p, n, yaml_NodeKind_Sequence, (p->cur_indent + 1));
      yaml_Parser_parse_node_or_value(p);
      return;
   }
   case yaml_TokenKind_Indent:
      p->cur_indent = p->token.indent;
      yaml_Parser_consumeToken(p);
      yaml_Parser_parse_node(p);
      return;
   case yaml_TokenKind_Dedent:
      p->cur_indent = p->token.indent;
      yaml_Parser_consumeToken(p);
      yaml_Parser_pop(p);
      return;
   case yaml_TokenKind_Doc_Start:
      __attribute__((fallthrough));
   case yaml_TokenKind_Doc_End:
      return;
   case yaml_TokenKind_Eof:
      yaml_Parser_add_scalar_value(p, 0);
      return;
   default:
      yaml_Parser_error(p, "%s() unhandled token '%s'", "parse_value", yaml_Token_str(&p->token));
      break;
   }
}

static void yaml_Parser_parse_node_or_value(yaml_Parser* p)
{
   switch (p->token.kind) {
   case yaml_TokenKind_Plain_Scalar:
      __attribute__((fallthrough));
   case yaml_TokenKind_Single_Quoted_Scalar:
      __attribute__((fallthrough));
   case yaml_TokenKind_Double_Quoted_Scalar: {
      yaml_Token* next = yaml_Tokenizer_lex_next(&p->tokenizer);
      if ((next->kind == yaml_TokenKind_Colon)) {
         p->cur_indent += 2;
         p->tokenizer.cur_indent += 2;
         yaml_Parser_parse_node(p);
         return;
      }
      break;
   }
   default:
      break;
   }
   yaml_Parser_parse_value(p);
}

static void yaml_Parser_doc_start(yaml_Parser* p)
{
   yaml_Parser_push_root(p);
   p->doc_started = true;
   p->in_document = true;
}

static void yaml_Parser_doc_end(yaml_Parser* p)
{
   p->cur_indent = -1;
   if (((p->stack_size == 1) && (p->stack[0].node->kind == yaml_NodeKind_Unknown))) {
      p->stack[0].node->kind = yaml_NodeKind_Map;
   }
   yaml_Parser_pop(p);
   p->cur_indent = 0;
   p->in_document = false;
}

static void yaml_Parser_add_scalar_value(yaml_Parser* p, uint32_t value_idx)
{
   yaml_StackLevel* top = &p->stack[(p->stack_size - 1)];
   yaml_Node* n = top->node;
   if ((n->kind != yaml_NodeKind_Unknown)) {
      yaml_Parser_error(p, "%s() cannot add scalar to node", "add_scalar_value");
   }
   n->kind = yaml_NodeKind_Scalar;
   n->text_idx = value_idx;
}

static void yaml_Parser_pop(yaml_Parser* p)
{
   int32_t indent = p->cur_indent;
   while (1) {
      yaml_StackLevel* top = &p->stack[(p->stack_size - 1)];
      if ((top->indent <= indent)) break;

      if ((p->stack_size >= 1)) {
         yaml_StackLevel* prev = &p->stack[(p->stack_size - 2)];
         prev->last_child = top->node;
      }
      if ((top->node->kind == yaml_NodeKind_Unknown)) top->node->kind = yaml_NodeKind_Scalar;
      top->indent = 0;
      top->node = NULL;
      top->last_child = NULL;
      p->stack_size--;
   }
}

static void yaml_Parser_push_root(yaml_Parser* p)
{
   yaml_Node* root = yaml_Data_add_node(&p->data, yaml_NodeKind_Unknown, 0);
   yaml_StackLevel* top = &p->stack[0];
   if (p->stack_size) {
      top->node->next_idx = yaml_Data_node2idx(&p->data, root);
   }
   top->node = root;
   top->indent = -1;
   top->last_child = NULL;
   p->stack_size = 1;
}

static void yaml_Parser_push_node(yaml_Parser* p, yaml_Node* n, yaml_NodeKind parent_kind, int32_t indent)
{
   c2_assert((p->stack_size) != 0, "common/yaml/yaml_parser.c2:285: yaml.Parser.push_node", "p.stack_size");
   uint32_t n_idx = yaml_Data_node2idx(&p->data, n);
   yaml_StackLevel* top = &p->stack[(p->stack_size - 1)];
   if ((indent < top->indent)) {
      c2_assert((((indent + 1) == top->indent)) != 0, "common/yaml/yaml_parser.c2:290: yaml.Parser.push_node", "indent+1==top.indent");
      yaml_Parser_pop(p);
      top = &p->stack[(p->stack_size - 1)];
   }
   if ((top->indent == indent)) {
      if (top->node) {
         if ((top->node->kind == yaml_NodeKind_Unknown)) top->node->kind = yaml_NodeKind_Scalar;
         top->node->next_idx = n_idx;
      }
      top->last_child = NULL;
   } else {
      c2_assert((((p->stack_size + 1) < yaml_MaxDepth)) != 0, "common/yaml/yaml_parser.c2:303: yaml.Parser.push_node", "p.stack_size+1<MaxDepth");
      c2_assert(((indent > top->indent)) != 0, "common/yaml/yaml_parser.c2:304: yaml.Parser.push_node", "indent>top.indent");
      yaml_Node* parent = top->node;
      if ((parent->kind == yaml_NodeKind_Unknown)) {
         if ((parent_kind == yaml_NodeKind_Unknown)) parent_kind = yaml_NodeKind_Map;
         parent->kind = parent_kind;
      }
      if (top->last_child) {
         top->last_child->next_idx = n_idx;
      } else {
         c2_assert(((parent->child_idx == 0)) != 0, "common/yaml/yaml_parser.c2:315: yaml.Parser.push_node", "parent.child_idx==0");
         parent->child_idx = n_idx;
      }
      top->last_child = n;
      p->stack_size++;
      top = &p->stack[(p->stack_size - 1)];
   }
   top->indent = indent;
   top->node = n;
   yaml_StackLevel* prev = &p->stack[(p->stack_size - 2)];
   yaml_Node* parent = prev->node;
   if (((parent->kind != parent_kind) && !(((parent->kind == yaml_NodeKind_Map) && (parent_kind == yaml_NodeKind_Unknown))))) {
      if ((parent->kind == yaml_NodeKind_Sequence)) {
         yaml_Parser_error(p, "invalid scalar after sequence");
      } else {
         yaml_Parser_error(p, "invalid scalar after %s", yaml_node_names[parent->kind]);
      }
   }
}


// --- module constants ---

#define constants_MaxIdentifierLen 31
#define constants_MaxFeatureName 31
#define constants_MaxFeatureDepth 6
#define constants_MaxErrorMsgLen 31
#define constants_Max_path 512
#define constants_Max_open_files 200
static const char* constants_output_dir = "output";

static const char* constants_recipe_name = "recipe.txt";

static const char* constants_buildfile_name = "build.yaml";

static const char* constants_manifest_name = "manifest.yaml";

static const char* constants_refs_filename = "refs";


// --- module dsm_sorter ---
typedef struct dsm_sorter_Sorter_ dsm_sorter_Sorter;

struct dsm_sorter_Sorter_ {
   uint8_t* array;
   uint32_t count;
};

static void dsm_sorter_Sorter_init(dsm_sorter_Sorter* s, uint32_t count);
static void dsm_sorter_Sorter_free(dsm_sorter_Sorter* s);
static void dsm_sorter_Sorter_add_dep(dsm_sorter_Sorter* s, uint32_t src, uint32_t dst);
static const uint8_t* dsm_sorter_Sorter_get_array(dsm_sorter_Sorter* s);
static const uint8_t* dsm_sorter_Sorter_sort(dsm_sorter_Sorter* s);
static void dsm_sorter_Sorter_init(dsm_sorter_Sorter* s, uint32_t count)
{
   s->array = calloc(1, (count * ((count + 2))));
   s->count = count;
   c2_assert(((count <= 256)) != 0, "common/dsm_sorter.c2:28: dsm_sorter.Sorter.init", "count<=256");
}

static void dsm_sorter_Sorter_free(dsm_sorter_Sorter* s)
{
   free(s->array);
}

static void dsm_sorter_Sorter_add_dep(dsm_sorter_Sorter* s, uint32_t src, uint32_t dst)
{
   uint32_t offset = ((src * s->count) + dst);
   s->array[offset] = 1;
}

static const uint8_t* dsm_sorter_Sorter_get_array(dsm_sorter_Sorter* s)
{
   return s->array;
}

static const uint8_t* dsm_sorter_Sorter_sort(dsm_sorter_Sorter* s)
{
   const uint32_t count = s->count;
   uint8_t* sorted = &s->array[(count * ((count + 1)))];
   uint8_t* ringbuf = &s->array[(count * count)];
   uint32_t head = 0;
   uint32_t size = count;
   for (uint8_t i = 0; (i < count); i++) ringbuf[i] = i;
   uint32_t iterations = 0;
   while (size) {
      if ((iterations > size)) {
         return NULL;
      }
      uint8_t idx = ringbuf[head];
      head = (((head + 1)) % count);
      uint32_t offset = (idx * count);
      bool has_deps = false;
      for (uint32_t j = 0; (j < count); j++) {
         if ((s->array[(offset + j)] != 0)) {
            has_deps = true;
            break;
         }
      }
      if (has_deps) {
         ringbuf[((((head + size) - 1)) % count)] = idx;
         iterations++;
      } else {
         sorted[(count - size)] = ((uint8_t)(idx));
         iterations = 0;
         size--;
         for (uint32_t x = 0; (x < count); x++) s->array[((x * count) + idx)] = 0;
      }
   }
   return sorted;
}


// --- module library_list ---
typedef struct library_list_Lib_ library_list_Lib;
typedef struct library_list_List_ library_list_List;

struct library_list_Lib_ {
   uint32_t name;
   bool is_static;
};

struct library_list_List_ {
   library_list_Lib* libs;
   uint32_t count;
   uint32_t capacity;
};

typedef void (*library_list_Visitor)(void* arg, uint32_t name, bool is_static);

static void library_list_List_init(library_list_List* l);
static void library_list_List_free(library_list_List* l);
static void library_list_List_visit(const library_list_List* l, library_list_Visitor visitor, void* arg);
static void library_list_List_add(library_list_List* l, uint32_t name, bool is_static);
static bool library_list_List_contains(const library_list_List* l, uint32_t name);
static uint32_t library_list_List_size(const library_list_List* l);
static void library_list_List_resize(library_list_List* l, uint32_t cap);
static void library_list_List_init(library_list_List* l)
{
   memset(l, 0, 16);
   library_list_List_resize(l, 2);
}

static void library_list_List_free(library_list_List* l)
{
   free(l->libs);
}

static void library_list_List_visit(const library_list_List* l, library_list_Visitor visitor, void* arg)
{
   for (uint32_t i = 0; (i < l->count); i++) {
      const library_list_Lib* lib = &l->libs[i];
      visitor(arg, lib->name, lib->is_static);
   }
}

static void library_list_List_add(library_list_List* l, uint32_t name, bool is_static)
{
   if ((l->count == l->capacity)) library_list_List_resize(l, (l->capacity * 2));
   l->libs[l->count].name = name;
   l->libs[l->count].is_static = is_static;
   l->count++;
}

static bool library_list_List_contains(const library_list_List* l, uint32_t name)
{
   for (uint32_t i = 0; (i < l->count); i++) {
      if ((l->libs[i].name == name)) return true;

   }
   return false;
}

static uint32_t library_list_List_size(const library_list_List* l)
{
   return l->count;
}

static void library_list_List_resize(library_list_List* l, uint32_t cap)
{
   l->capacity = cap;
   library_list_Lib* libs2 = malloc((l->capacity * 8));
   if (l->libs) {
      memcpy(libs2, l->libs, (l->count * 8));
      free(l->libs);
   }
   l->libs = libs2;
}


// --- module linked_list ---
typedef struct linked_list_Element_ linked_list_Element;

struct linked_list_Element_ {
   linked_list_Element* prev;
   linked_list_Element* next;
};

static void linked_list_Element_init(linked_list_Element* src);
static void linked_list_Element_addTail(linked_list_Element* src, linked_list_Element* item);
static void linked_list_Element_addFront(linked_list_Element* src, linked_list_Element* item);
static linked_list_Element* linked_list_Element_popFront(linked_list_Element* item);
static void linked_list_Element_remove(linked_list_Element* item);
static uint64_t linked_list_Element_size(const linked_list_Element* src);
static bool linked_list_Element_isEmpty(const linked_list_Element* src);
static void linked_list_Element_move(linked_list_Element* src, linked_list_Element* dest);
static void linked_list_Element_init(linked_list_Element* src)
{
   src->prev = src;
   src->next = src;
}

static void linked_list_Element_addTail(linked_list_Element* src, linked_list_Element* item)
{
   linked_list_Element* old_tail = src->prev;
   src->prev = item;
   item->next = src;
   item->prev = old_tail;
   old_tail->next = item;
}

static void linked_list_Element_addFront(linked_list_Element* src, linked_list_Element* item)
{
   linked_list_Element* old_head = src->next;
   old_head->prev = item;
   item->next = old_head;
   item->prev = src;
   src->next = item;
}

static linked_list_Element* linked_list_Element_popFront(linked_list_Element* item)
{
   linked_list_Element* node = item->next;
   linked_list_Element_remove(node);
   return node;
}

static void linked_list_Element_remove(linked_list_Element* item)
{
   linked_list_Element* prev = item->prev;
   linked_list_Element* next = item->next;
   prev->next = next;
   next->prev = prev;
}

static uint64_t linked_list_Element_size(const linked_list_Element* src)
{
   uint64_t count = 0;
   linked_list_Element* node = src->next;
   while ((node != src)) {
      count++;
      node = node->next;
   }
   return count;
}

static bool linked_list_Element_isEmpty(const linked_list_Element* src)
{
   return (src->next == src);
}

static void linked_list_Element_move(linked_list_Element* src, linked_list_Element* dest)
{
   linked_list_Element* node = src->next;
   while ((node != src)) {
      linked_list_Element* tmp = node;
      node = node->next;
      linked_list_Element_remove(tmp);
      linked_list_Element_addTail(dest, tmp);
   }
}


// --- module process_utils ---

#define process_utils_MAX_ARG_LEN 512
#define process_utils_MAX_ARGS 16
static bool process_utils_doWIFSIGNALED(int32_t state);
static bool process_utils_doWIFEXITED(int32_t state);
static char process_utils_getWEXITSTATUS(int32_t state);
static void process_utils_child_error(int32_t fd, const char* msg);
static int32_t process_utils_run(const char* path, const char* cmd, const char* logfile);
static void process_utils_parseArgs(const char* cmd, const char* args, char** argv, uint32_t _arg3);
static const char* process_utils_find_bin(const char* name);
static int32_t process_utils_run_args(const char* path, const char* cmd, const char* logfile, const char* args);
static int32_t process_utils_run2(const char* path, const char* cmd, const char* args, char* output);
static void process_utils_stripNewLine(char* buf);
static void process_utils_split(const char* input, char* cmd, char* args);
static bool process_utils_doWIFSIGNALED(int32_t state)
{
   return (((((state & 0x7f)) > 0) && (((state & 0x7f)) < 0x7f)));
}

static bool process_utils_doWIFEXITED(int32_t state)
{
   return ((((state & 0xff)) == 0));
}

static char process_utils_getWEXITSTATUS(int32_t state)
{
   return ((char)((((state >> 8)) & 0xff)));
}

static void process_utils_child_error(int32_t fd, const char* msg)
{
   size_t len = strlen(msg);
   ssize_t written = write(fd, msg, len);
   if ((written != ((ssize_t)(len)))) perror("write");
   fsync(fd);
   fprintf(stderr, "[exec] %s\n", msg);
   fflush(stderr);
   _exit(-1);
}

static int32_t process_utils_run(const char* path, const char* cmd, const char* logfile)
{
   int32_t error_pipe[2];
   if (pipe(error_pipe)) {
      fprintf(stderr, "pipe() failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   if ((fcntl(error_pipe[0], F_SETFD, FD_CLOEXEC) != 0)) {
      fprintf(stderr, "fcncl(FD_CLOEXEC() failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   if ((fcntl(error_pipe[1], F_SETFD, FD_CLOEXEC) != 0)) {
      fprintf(stderr, "fcncl(FD_CLOEXEC) failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   pid_t child_pid = fork();
   if ((child_pid == 0)) {
      char errmsg[256];
      if (close(error_pipe[0])) {
         perror("close(errpipe)");
      }
      char output[512];
      sprintf(output, "%s%s", path, logfile);
      fflush(stdout);
      close(STDOUT_FILENO);
      int32_t fdout = open(output, ((O_TRUNC | O_CREAT) | O_WRONLY), 0644);
      if ((fdout == -1)) {
         sprintf(errmsg, "cannot open output '%s': %s", output, strerror(*__errno_location()));
         process_utils_child_error(error_pipe[1], errmsg);
      }
      close(STDERR_FILENO);
      if ((dup(STDOUT_FILENO) == -1)) {
         sprintf(errmsg, "dup(): %s", strerror(*__errno_location()));
         process_utils_child_error(error_pipe[1], errmsg);
      }
      printf("current dir: %s\n", getcwd(NULL, 0));
      if ((chdir(path) != 0)) {
         sprintf(errmsg, "cannot change to dir '%s': %s", path, strerror(*__errno_location()));
         process_utils_child_error(error_pipe[1], errmsg);
      }
      printf("changing to dir: %s\n", path);
      c2_assert(((strlen(cmd) < process_utils_MAX_ARG_LEN)) != 0, "common/process_utils.c2:105: process_utils.run", "CALL TODO<MAX_ARG_LEN");
      char self[513];
      strcpy(self, cmd);
      char* argv[2] = { self, NULL };
      printf("running command: %s\n", cmd);
      execv(cmd, argv);
      int32_t lasterr = *__errno_location();
      fprintf(stderr, "failed to start %s: %s\n", cmd, strerror(lasterr));
      sprintf(errmsg, "error starting %s: %s", cmd, strerror(lasterr));
      process_utils_child_error(error_pipe[1], errmsg);
      _exit(-1);
   } else {
      if (close(error_pipe[1])) {
      }
      char error[256];
      memset(error, 0, 256);
      ssize_t numread = read(error_pipe[0], error, (256 - 1));
      if ((numread < 0)) {
         fprintf(stderr, "Error reading pipe\n");
         return -1;
      } else error[numread] = 0;
      close(error_pipe[0]);
      if ((numread != 0)) {
         return -1;
      }
      int32_t state = 0;
      pid_t pid = waitpid(child_pid, &state, 0);
      if ((pid == -1)) {
         fprintf(stderr, "Error waiting for pid: %s\n", strerror(*__errno_location()));
         return -1;
      }
      if (process_utils_doWIFSIGNALED(state)) {
         return -1;
      }
      if (process_utils_doWIFEXITED(state)) {
         char exitcode = process_utils_getWEXITSTATUS(state);
         if ((exitcode != 0)) return -1;

      } else {
         return -1;
      }
   }
   return 0;
}

static void process_utils_parseArgs(const char* cmd, const char* args, char** argv, uint32_t _arg3)
{
   static char tmp[512];
   uint32_t argc = 0;
   argv[argc++] = ((char*)(cmd));
   size_t len = (strlen(args) + 1);
   c2_assert(((len < process_utils_MAX_ARG_LEN)) != 0, "common/process_utils.c2:164: process_utils.parseArgs", "len<MAX_ARG_LEN");
   memcpy(tmp, args, len);
   char* token = strtok(tmp, " ");
   while (token) {
      argv[argc] = token;
      argc++;
      token = strtok(NULL, " ");
   }
   argv[argc] = NULL;
}

static const char* process_utils_find_bin(const char* name)
{
   static char result[512];
   struct stat statbuf;
   char* path = strdup(getenv("PATH"));
   char* s = path;
   char* p = NULL;
   do {
      p = strchr(s, ':');
      if ((p != NULL)) p[0] = 0;
      sprintf(result, "%s/%s", s, name);
      if ((stat(result, &statbuf) == 0)) {
         free(path);
         return result;
      }
      s = (p + 1);
   } while ((p != NULL));
   free(path);
   return NULL;
}

static int32_t process_utils_run_args(const char* path, const char* cmd, const char* logfile, const char* args)
{
   int32_t error_pipe[2];
   if (pipe(error_pipe)) {
      fprintf(stderr, "pipe() failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   if ((fcntl(error_pipe[0], F_SETFD, FD_CLOEXEC) != 0)) {
      fprintf(stderr, "fcncl(FD_CLOEXEC() failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   if ((fcntl(error_pipe[1], F_SETFD, FD_CLOEXEC) != 0)) {
      fprintf(stderr, "fcncl(FD_CLOEXEC) failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   pid_t child_pid = fork();
   if ((child_pid == 0)) {
      char errmsg[256];
      if (close(error_pipe[0])) {
         perror("close(errpipe)");
      }
      char output[512];
      sprintf(output, "%s%s", path, logfile);
      fflush(stdout);
      close(STDOUT_FILENO);
      int32_t fdout = open(output, ((O_TRUNC | O_CREAT) | O_WRONLY), 0644);
      if ((fdout == -1)) {
         sprintf(errmsg, "cannot open output '%s': %s", output, strerror(*__errno_location()));
         process_utils_child_error(error_pipe[1], errmsg);
      }
      close(STDERR_FILENO);
      if ((dup(STDOUT_FILENO) == -1)) {
         sprintf(errmsg, "dup(): %s", strerror(*__errno_location()));
         process_utils_child_error(error_pipe[1], errmsg);
      }
      printf("current dir: %s\n", getcwd(NULL, 0));
      if ((chdir(path) != 0)) {
         sprintf(errmsg, "cannot change to dir '%s': %s", path, strerror(*__errno_location()));
         process_utils_child_error(error_pipe[1], errmsg);
      }
      printf("changing to dir: %s\n", path);
      printf("running command: %s %s\n", cmd, args);
      const char* self = process_utils_find_bin(cmd);
      if (!self) {
         printf("command not found\n");
         _exit(EXIT_FAILURE);
      }
      char* argv[16];
      process_utils_parseArgs(self, args, argv, process_utils_MAX_ARGS);
      execv(self, argv);
      int32_t lasterr = *__errno_location();
      fprintf(stderr, "failed to start %s: %s\n", cmd, strerror(lasterr));
      sprintf(errmsg, "error starting %s: %s", cmd, strerror(lasterr));
      process_utils_child_error(error_pipe[1], errmsg);
      _exit(EXIT_FAILURE);
   } else {
      if (close(error_pipe[1])) {
      }
      char error[256];
      memset(error, 0, 256);
      ssize_t numread = read(error_pipe[0], error, (256 - 1));
      if ((numread < 0)) {
         fprintf(stderr, "Error reading pipe\n");
         return -1;
      } else error[numread] = 0;
      close(error_pipe[0]);
      if ((numread != 0)) {
         return -1;
      }
      int32_t state = 0;
      pid_t pid = waitpid(child_pid, &state, 0);
      if ((pid == -1)) {
         fprintf(stderr, "Error waiting for pid: %s\n", strerror(*__errno_location()));
         return -1;
      }
      if (process_utils_doWIFSIGNALED(state)) {
         return -1;
      }
      if (process_utils_doWIFEXITED(state)) {
         char exitcode = process_utils_getWEXITSTATUS(state);
         if ((exitcode != 0)) return -1;

      } else {
         return -1;
      }
   }
   return 0;
}

static int32_t process_utils_run2(const char* path, const char* cmd, const char* args, char* output)
{
   int32_t error_pipe[2];
   if (pipe(error_pipe)) {
      fprintf(stderr, "pipe() failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   if ((fcntl(error_pipe[0], F_SETFD, FD_CLOEXEC) != 0)) {
      fprintf(stderr, "fcncl(FD_CLOEXEC() failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   if ((fcntl(error_pipe[1], F_SETFD, FD_CLOEXEC) != 0)) {
      fprintf(stderr, "fcncl(FD_CLOEXEC) failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   int32_t output_pipe[2];
   if (pipe(output_pipe)) {
      fprintf(stderr, "pipe() failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   if ((fcntl(output_pipe[0], F_SETFD, FD_CLOEXEC) != 0)) {
      fprintf(stderr, "fcncl(FD_CLOEXEC() failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   if ((fcntl(output_pipe[1], F_SETFD, FD_CLOEXEC) != 0)) {
      fprintf(stderr, "fcncl(FD_CLOEXEC) failed: %s\n", strerror(*__errno_location()));
      return -1;
   }
   pid_t child_pid = fork();
   if ((child_pid == 0)) {
      char errmsg[256];
      if (close(output_pipe[0])) {
         perror("close(outputpipe)");
      }
      if (close(error_pipe[0])) {
         perror("close(error_pipe)");
      }
      fflush(stdout);
      close(STDOUT_FILENO);
      if ((dup(output_pipe[1]) == -1)) {
         sprintf(errmsg, "dup(): %s", strerror(*__errno_location()));
         process_utils_child_error(error_pipe[1], errmsg);
      }
      close(STDERR_FILENO);
      if ((dup(STDOUT_FILENO) == -1)) {
         sprintf(errmsg, "dup(): %s", strerror(*__errno_location()));
         process_utils_child_error(error_pipe[1], errmsg);
      }
      if ((chdir(path) != 0)) {
         sprintf(errmsg, "cannot change to dir '%s': %s", path, strerror(*__errno_location()));
         process_utils_child_error(error_pipe[1], errmsg);
      }
      const char* self = process_utils_find_bin(cmd);
      if (!self) {
         sprintf(errmsg, "command %s not found", self);
         process_utils_child_error(error_pipe[1], errmsg);
      }
      char* argv[16];
      process_utils_parseArgs(self, args, argv, process_utils_MAX_ARGS);
      execv(self, argv);
      int32_t lasterr = *__errno_location();
      fprintf(stderr, "failed to start %s: %s\n", self, strerror(lasterr));
      sprintf(errmsg, "error starting %s: %s", self, strerror(lasterr));
      process_utils_child_error(error_pipe[1], errmsg);
      _exit(-1);
   } else {
      if (close(error_pipe[1])) {
      }
      if (close(output_pipe[1])) {
      }
      char error[256];
      memset(error, 0, 256);
      ssize_t numread = read(error_pipe[0], error, (256 - 1));
      if ((numread < 0)) {
         fprintf(stderr, "Error reading pipe\n");
         return -1;
      } else error[numread] = 0;
      close(error_pipe[0]);
      if ((numread != 0)) {
         printf("ERROR %s\n", error);
         return -1;
      }
      int32_t state = 0;
      pid_t pid = waitpid(child_pid, &state, 0);
      if ((pid == -1)) {
         fprintf(stderr, "Error waiting for pid: %s\n", strerror(*__errno_location()));
         return -1;
      }
      if (process_utils_doWIFSIGNALED(state)) {
         return -1;
      }
      if (process_utils_doWIFEXITED(state)) {
         char exitcode = process_utils_getWEXITSTATUS(state);
         if ((exitcode != 0)) return -1;

         numread = read(output_pipe[0], error, (256 - 1));
         error[numread] = 0;
         char* cp = error;
         while (((*cp == ' ') || (*cp == '\t'))) cp++;
         process_utils_stripNewLine(cp);
         strcpy(output, cp);
      } else {
         fprintf(stderr, "child exited ABNORMALLY\n");
         return -1;
      }
   }
   return 0;
}

static void process_utils_stripNewLine(char* buf)
{
   size_t len = strlen(buf);
   if ((buf[(len - 1)] == '\n')) {
      buf[(len - 1)] = 0;
   }
}

static void process_utils_split(const char* input, char* cmd, char* args)
{
   const char* delim = strstr(input, " ");
   if (delim) {
      uint32_t len = ((uint32_t)((delim - input)));
      memcpy(cmd, input, len);
      cmd[len] = 0;
      strcpy(args, (delim + 1));
   } else {
      strcpy(cmd, input);
      args[0] = 0;
   }
}


// --- module quicksort ---

typedef bool (*quicksort_CompareFn)(void* arg, const void* left, const void* right);

static void quicksort_swap(uint8_t* item, uint8_t* other, size_t size);
static void quicksort_sort(void* items, size_t count, size_t item_size, quicksort_CompareFn is_less, void* arg);
static void quicksort_swap(uint8_t* item, uint8_t* other, size_t size)
{
   const uint8_t* end = (item + size);
   while ((item < end)) {
      uint8_t tmp = *other;
      *other = *item;
      *item = tmp;
      other++;
      item++;
   }
}

static void quicksort_sort(void* items, size_t count, size_t item_size, quicksort_CompareFn is_less, void* arg)
{
   if ((count <= 1)) return;

   uint8_t* begin = items;
   uint8_t* end = (begin + (count * item_size));
   uint8_t* left = begin;
   uint8_t* pivot = (begin + (((count / 2)) * item_size));
   uint8_t* right = (end - item_size);
   do {
      while (is_less(arg, left, pivot)) left += item_size;
      while (is_less(arg, pivot, right)) right -= item_size;
      if ((left < right)) {
         quicksort_swap(left, right, item_size);
         if ((left == pivot)) {
            pivot = right;
         } else if ((right == pivot)) {
            pivot = left;
         }

      }
      if ((left <= right)) {
         left += item_size;
         right -= item_size;
      }
   } while ((left <= right));
   if ((right > begin)) {
      size_t part_items = ((((right - begin) + item_size)) / item_size);
      quicksort_sort(begin, part_items, item_size, is_less, arg);
   }
   if ((left < end)) {
      size_t part_items = (((end - left)) / item_size);
      quicksort_sort(left, part_items, item_size, is_less, arg);
   }
}


// --- module string_utils ---

static uint32_t string_utils_count_tabs(const char* text, uint32_t len);
static void string_utils_toUpper(const char* input, char* output);
static uint32_t string_utils_count_tabs(const char* text, uint32_t len)
{
   uint32_t count = 0;
   const char* cp = text;
   for (uint32_t i = 0; (i < len); i++) {
      if ((*cp == 0)) break;

      if ((*cp == '\t')) count++;
      if (((*cp == '\n') || (*cp == '\r'))) break;

      cp++;
      if ((((cp - text)) >= 256)) break;

   }
   return count;
}

static void string_utils_toUpper(const char* input, char* output)
{
   while (*input) {
      *output = ((char)(toupper(*input)));
      input++;
      output++;
   }
   *output = 0;
}


// --- module utils ---
typedef struct utils_PathInfo_ utils_PathInfo;

struct utils_PathInfo_ {
   char orig2root[512];
   char root2orig[512];
};

static uint64_t utils_now(void);
static bool utils_findProjectDir(utils_PathInfo* info);
static const char* utils_findBuildFile(void);
static uint64_t utils_now(void)
{
   timeval tv;
   gettimeofday(&tv, NULL);
   uint64_t now64 = ((uint64_t)(tv.tv_sec));
   now64 *= 1000000;
   now64 += tv.tv_usec;
   return now64;
}

static bool utils_findProjectDir(utils_PathInfo* info)
{
   if (info) {
      info->root2orig[0] = 0;
      info->orig2root[0] = 0;
   }
   char base_path[512];
   char rel_path[512];
   base_path[0] = 0;
   rel_path[0] = 0;
   char buffer[512];
   while (1) {
      char* path = getcwd(buffer, constants_Max_path);
      if ((path == NULL)) {
         perror("getcwd");
         return false;
      }
      if ((base_path[0] == 0)) strcpy(base_path, buffer);
      struct stat buf;
      int32_t error = stat(constants_recipe_name, &buf);
      if (error) {
         if (((buffer[0] == '/') && (buffer[1] == 0))) return false;

         if ((*__errno_location() != ENOENT)) {
            perror("stat");
            return false;
         }
      } else {
         if (((buf.st_mode & S_IFMT) == S_IFREG)) {
            char* path_prefix = (base_path + strlen(buffer));
            if ((*path_prefix == '/')) path_prefix++;
            if (info) {
               strcpy(info->orig2root, path_prefix);
               strcpy(info->root2orig, rel_path);
            }
            return true;
         }
      }
      error = chdir("..");
      if (error) {
         perror("chdir");
         return false;
      }
      strcat(rel_path, "../");
   }
   return false;
}

static const char* utils_findBuildFile(void)
{
   struct stat buf;
   int32_t error = stat(constants_buildfile_name, &buf);
   if (error) return NULL;

   return constants_buildfile_name;
}


// --- module warning_flags ---
typedef struct warning_flags_Flags_ warning_flags_Flags;

struct warning_flags_Flags_ {
   bool no_unused;
   bool no_unused_variable;
   bool no_unused_function;
   bool no_unused_parameter;
   bool no_unused_type;
   bool no_unused_module;
   bool no_unused_import;
   bool no_unused_public;
   bool no_unused_label;
   bool no_unused_enum_constant;
   bool are_errors;
};


// --- module name_vector ---
typedef struct name_vector_NameVector_ name_vector_NameVector;

struct name_vector_NameVector_ {
   uint32_t* data;
   uint32_t count;
   uint32_t capacity;
};

static void name_vector_NameVector_init(name_vector_NameVector* v, uint32_t capacity);
static void name_vector_NameVector_free(name_vector_NameVector* v);
static void name_vector_NameVector_clear(name_vector_NameVector* v);
static void name_vector_NameVector_resize(name_vector_NameVector* v);
static uint32_t name_vector_NameVector_add(name_vector_NameVector* v, uint32_t name_idx);
static uint32_t name_vector_NameVector_get(const name_vector_NameVector* v, uint32_t idx);
static bool name_vector_NameVector_find(name_vector_NameVector* v, uint32_t name_idx, uint32_t* index);
static void name_vector_NameVector_init(name_vector_NameVector* v, uint32_t capacity)
{
   v->data = NULL;
   v->count = 0;
   v->capacity = (capacity / 2);
   name_vector_NameVector_resize(v);
}

static void name_vector_NameVector_free(name_vector_NameVector* v)
{
   free(v->data);
   v->count = 0;
   v->capacity = 0;
   v->data = NULL;
}

static void name_vector_NameVector_clear(name_vector_NameVector* v)
{
   v->count = 0;
}

static void name_vector_NameVector_resize(name_vector_NameVector* v)
{
   v->capacity = (v->capacity == 0) ? 4 : (v->capacity * 2);
   void* data2 = malloc((v->capacity * 4));
   if (v->data) {
      memcpy(data2, v->data, (v->count * 4));
      free(v->data);
   }
   v->data = data2;
}

static uint32_t name_vector_NameVector_add(name_vector_NameVector* v, uint32_t name_idx)
{
   if ((v->count == v->capacity)) name_vector_NameVector_resize(v);
   uint32_t index = v->count;
   v->data[index] = name_idx;
   v->count++;
   return index;
}

static uint32_t name_vector_NameVector_get(const name_vector_NameVector* v, uint32_t idx)
{
   return v->data[idx];
}

static bool name_vector_NameVector_find(name_vector_NameVector* v, uint32_t name_idx, uint32_t* index)
{
   for (uint32_t i = 0; (i < v->count); i++) {
      if ((v->data[i] == name_idx)) {
         *index = i;
         return true;
      }
   }
   return false;
}


// --- module color ---

#define color_Black "\033[0;30m"
#define color_Red "\033[0;31m"
#define color_Green "\033[0;32m"
#define color_Yellow "\033[0;33m"
#define color_Blue "\033[0;34m"
#define color_Magenta "\033[0;35m"
#define color_Cyan "\033[0;36m"
#define color_Grey "\033[0;37m"
#define color_Darkgrey "\033[01;30m"
#define color_Bred "\033[01;31m"
#define color_Bgreen "\033[01;32m"
#define color_Byellow "\033[01;33m"
#define color_Bblue "\033[01;34m"
#define color_Bmagenta "\033[01;35m"
#define color_Bcyan "\033[01;36m"
#define color_White "\033[01;37m"
#define color_Normal "\033[0m"
static bool color_useColor(void);
static bool color_useColor(void)
{
   return isatty(1);
}


// --- module ast_context ---
typedef struct ast_context_Block_ ast_context_Block;
typedef struct ast_context_Context_ ast_context_Context;

struct ast_context_Block_ {
   uint8_t* data;
   uint32_t size;
   ast_context_Block* next;
};

struct ast_context_Context_ {
   ast_context_Block* blk_head;
   ast_context_Block* blk_tail;
   uint32_t blk_size;
   uint32_t num_allocs;
};

static ast_context_Block* ast_context_Block_create(uint32_t blk_size);
static ast_context_Block* ast_context_Block_free(ast_context_Block* b);
static uint32_t ast_context_Block_count(const ast_context_Block* b);
static uint32_t ast_context_Block_get_total(const ast_context_Block* b);
static ast_context_Context* ast_context_create(uint32_t blk_size);
static void ast_context_Context_freeBlocks(ast_context_Context* c);
static void ast_context_Context_free(ast_context_Context* c);
static void ast_context_Context_init(ast_context_Context* c);
static void* ast_context_Context_alloc(ast_context_Context* c, uint32_t len);
static void ast_context_Context_reserve(ast_context_Context* c, uint32_t len);
static void ast_context_Context_report(const ast_context_Context* c);
static ast_context_Block* ast_context_Block_create(uint32_t blk_size)
{
   ast_context_Block* b = malloc(24);
   b->data = malloc(blk_size);
   b->size = 0;
   b->next = NULL;
   return b;
}

static ast_context_Block* ast_context_Block_free(ast_context_Block* b)
{
   ast_context_Block* next = b->next;
   free(b->data);
   free(b);
   return next;
}

static uint32_t ast_context_Block_count(const ast_context_Block* b)
{
   uint32_t total = 0;
   const ast_context_Block* blk = b;
   while (blk) {
      total++;
      blk = blk->next;
   }
   return total;
}

static uint32_t ast_context_Block_get_total(const ast_context_Block* b)
{
   uint32_t total = 0;
   const ast_context_Block* blk = b;
   while (blk) {
      total += blk->size;
      blk = blk->next;
   }
   return total;
}

static ast_context_Context* ast_context_create(uint32_t blk_size)
{
   ast_context_Context* c = calloc(1, 24);
   c->blk_size = blk_size;
   ast_context_Context_init(c);
   return c;
}

static void ast_context_Context_freeBlocks(ast_context_Context* c)
{
   ast_context_Block* blk = c->blk_head;
   while (blk) blk = ast_context_Block_free(blk);
}

static void ast_context_Context_free(ast_context_Context* c)
{
   ast_context_Context_freeBlocks(c);
   free(c);
}

static void ast_context_Context_init(ast_context_Context* c)
{
   c->blk_head = ast_context_Block_create(c->blk_size);
   c->blk_tail = c->blk_head;
}

static void* ast_context_Context_alloc(ast_context_Context* c, uint32_t len)
{
   len = (((len + 7)) & ~0x7);
   c->num_allocs++;
   ast_context_Block* last = c->blk_tail;
   if (((last->size + len) >= c->blk_size)) {
      c->blk_tail = ast_context_Block_create(c->blk_size);
      last->next = c->blk_tail;
      last = last->next;
   }
   void* cur = (last->data + last->size);
   last->size += len;
   return cur;
}

static void ast_context_Context_reserve(ast_context_Context* c, uint32_t len)
{
   len = (((len + 7)) & ~0x7);
   ast_context_Block* last = c->blk_tail;
   if (((last->size + len) >= c->blk_size)) {
      c->blk_tail = ast_context_Block_create(c->blk_size);
      last->next = c->blk_tail;
      last = last->next;
   }
}

static void ast_context_Context_report(const ast_context_Context* c)
{
   uint32_t num = ast_context_Block_count(c->blk_head);
   uint32_t total = ast_context_Block_get_total(c->blk_head);
   uint32_t avg = 0;
   if (c->num_allocs) avg = (total / c->num_allocs);
   printf("context: %u allocs (avg %u bytes), %u blocks (%u), total %u Kb (%u)\n", c->num_allocs, avg, num, c->blk_size, (((total / 1024)) + 1), total);
}


// --- module src_loc ---
typedef struct src_loc_SrcRange_ src_loc_SrcRange;

typedef uint32_t src_loc_SrcLoc;

struct src_loc_SrcRange_ {
   src_loc_SrcLoc start;
   src_loc_SrcLoc end;
};


// --- module string_buffer ---
typedef struct string_buffer_Buf_ string_buffer_Buf;

struct string_buffer_Buf_ {
   uint32_t capacity;
   uint32_t size_;
   uint32_t indent_step;
   char* data_;
   bool colors;
   bool own;
};

static string_buffer_Buf* string_buffer_create(uint32_t capacity, bool colors, uint32_t indent_step);
static string_buffer_Buf* string_buffer_create_static(uint32_t capacity, bool colors, char* data);
static void string_buffer_Buf_free(string_buffer_Buf* buf);
static uint32_t string_buffer_Buf_size(const string_buffer_Buf* buf);
static const char* string_buffer_Buf_data(const string_buffer_Buf* buf);
static const uint8_t* string_buffer_Buf_udata(const string_buffer_Buf* buf);
static void string_buffer_Buf_clear(string_buffer_Buf* buf);
static void string_buffer_Buf_color(string_buffer_Buf* buf, const char* color);
static void string_buffer_Buf_add1(string_buffer_Buf* buf, char c);
static void string_buffer_Buf_add(string_buffer_Buf* buf, const char* text);
static void string_buffer_Buf_add2(string_buffer_Buf* buf, const char* text, uint32_t len);
static void string_buffer_Buf_add_line(string_buffer_Buf* buf, const char* text);
static void string_buffer_Buf_newline(string_buffer_Buf* buf);
static void string_buffer_Buf_space(string_buffer_Buf* buf);
static void string_buffer_Buf_lparen(string_buffer_Buf* buf);
static void string_buffer_Buf_rparen(string_buffer_Buf* buf);
__attribute__((__format__(printf, 2, 3))) 
static void string_buffer_Buf_print(string_buffer_Buf* buf, const char* format, ...);
static void string_buffer_Buf_indent(string_buffer_Buf* buf, uint32_t indent);
static bool string_buffer_Buf_endsWith(const string_buffer_Buf* buf, char c);
static void string_buffer_Buf_resize(string_buffer_Buf* buf, uint32_t capacity);
static void string_buffer_Buf_stripNewline(string_buffer_Buf* buf);
static string_buffer_Buf* string_buffer_create(uint32_t capacity, bool colors, uint32_t indent_step)
{
   string_buffer_Buf* buf = malloc(32);
   buf->capacity = capacity;
   buf->size_ = 0;
   buf->indent_step = indent_step;
   buf->data_ = malloc(capacity);
   buf->colors = colors;
   buf->own = true;
   buf->data_[0] = 0;
   return buf;
}

static string_buffer_Buf* string_buffer_create_static(uint32_t capacity, bool colors, char* data)
{
   string_buffer_Buf* buf = malloc(32);
   buf->capacity = capacity;
   buf->size_ = 0;
   buf->data_ = data;
   buf->colors = colors;
   buf->own = false;
   buf->data_[0] = 0;
   return buf;
}

static void string_buffer_Buf_free(string_buffer_Buf* buf)
{
   if (buf->own) free(buf->data_);
   free(buf);
}

static uint32_t string_buffer_Buf_size(const string_buffer_Buf* buf)
{
   return buf->size_;
}

static const char* string_buffer_Buf_data(const string_buffer_Buf* buf)
{
   return buf->data_;
}

static const uint8_t* string_buffer_Buf_udata(const string_buffer_Buf* buf)
{
   return ((uint8_t*)(buf->data_));
}

static void string_buffer_Buf_clear(string_buffer_Buf* buf)
{
   buf->size_ = 0;
}

static void string_buffer_Buf_color(string_buffer_Buf* buf, const char* color)
{
   if (!buf->colors) return;

   uint32_t len = ((uint32_t)(strlen(color)));
   string_buffer_Buf_add2(buf, color, len);
}

static void string_buffer_Buf_add1(string_buffer_Buf* buf, char c)
{
   if (((buf->size_ + 2) > buf->capacity)) {
      uint32_t new_cap = (buf->capacity * 2);
      while (((buf->size_ + 2) > new_cap)) new_cap *= 2;
      string_buffer_Buf_resize(buf, new_cap);
   }
   buf->data_[buf->size_] = c;
   buf->size_ += 1;
   buf->data_[buf->size_] = 0;
}

static void string_buffer_Buf_add(string_buffer_Buf* buf, const char* text)
{
   uint32_t len = ((uint32_t)(strlen(text)));
   string_buffer_Buf_add2(buf, text, len);
}

static void string_buffer_Buf_add2(string_buffer_Buf* buf, const char* text, uint32_t len)
{
   if ((((buf->size_ + len) + 1) > buf->capacity)) {
      uint32_t new_cap = (buf->capacity * 2);
      while ((((buf->size_ + len) + 1) > new_cap)) new_cap *= 2;
      string_buffer_Buf_resize(buf, new_cap);
   }
   memcpy(&buf->data_[buf->size_], text, len);
   buf->size_ += len;
   buf->data_[buf->size_] = 0;
}

static void string_buffer_Buf_add_line(string_buffer_Buf* buf, const char* text)
{
   c2_assert((text) != NULL, "ast_utils/string_buffer.c2:114: string_buffer.Buf.add_line", "text");
   const char* end = text;
   while (*end) {
      if (((*end == '\n') || (*end == '\r'))) break;

      end++;
      if ((((end - text)) >= 256)) break;

   }
   uint32_t len = ((uint32_t)((end - text)));
   string_buffer_Buf_add2(buf, text, len);
}

static void string_buffer_Buf_newline(string_buffer_Buf* buf)
{
   if (((buf->size_ + 2) > buf->capacity)) {
      uint32_t new_cap = (buf->capacity * 2);
      while (((buf->size_ + 2) > new_cap)) new_cap *= 2;
      string_buffer_Buf_resize(buf, new_cap);
   }
   buf->data_[buf->size_] = '\n';
   buf->size_ += 1;
   buf->data_[buf->size_] = 0;
}

static void string_buffer_Buf_space(string_buffer_Buf* buf)
{
   string_buffer_Buf_add1(buf, ' ');
}

static void string_buffer_Buf_lparen(string_buffer_Buf* buf)
{
   string_buffer_Buf_add1(buf, '(');
}

static void string_buffer_Buf_rparen(string_buffer_Buf* buf)
{
   string_buffer_Buf_add1(buf, ')');
}

__attribute__((__format__(printf, 2, 3))) 
static void string_buffer_Buf_print(string_buffer_Buf* buf, const char* format, ...)
{
   char tmp[4096];
   va_list args;
   va_start(args, format);
   int32_t len = vsnprintf(tmp, 4096, format, args);
   c2_assert(((len < 4096)) != 0, "ast_utils/string_buffer.c2:148: string_buffer.Buf.print", "len<sizeof()");
   string_buffer_Buf_add2(buf, tmp, ((uint32_t)(len)));
   va_end(args);
}

static void string_buffer_Buf_indent(string_buffer_Buf* buf, uint32_t indent)
{
   if ((indent == 0)) return;

   indent *= buf->indent_step;
   if ((((buf->size_ + indent) + 1) > buf->capacity)) {
      string_buffer_Buf_resize(buf, (buf->capacity * 2));
   }
   char* cur = (buf->data_ + buf->size_);
   memset(cur, ' ', indent);
   cur[indent] = 0;
   buf->size_ += indent;
}

static bool string_buffer_Buf_endsWith(const string_buffer_Buf* buf, char c)
{
   if ((buf->size_ && (buf->data_[(buf->size_ - 1)] == c))) return true;

   return false;
}

static void string_buffer_Buf_resize(string_buffer_Buf* buf, uint32_t capacity)
{
   c2_assert((buf->own) != 0, "ast_utils/string_buffer.c2:173: string_buffer.Buf.resize", "buf.own");
   buf->capacity = capacity;
   char* data2 = malloc(buf->capacity);
   memcpy(data2, buf->data_, buf->size_);
   free(buf->data_);
   buf->data_ = data2;
}

static void string_buffer_Buf_stripNewline(string_buffer_Buf* buf)
{
   if ((buf->size_ == 0)) return;

   if ((buf->data_[(buf->size_ - 1)] == '\n')) {
      buf->data_[(buf->size_ - 1)] = 0;
      buf->size_--;
   }
}


// --- module string_pool ---
typedef struct string_pool_Node_ string_pool_Node;
typedef struct string_pool_Pool_ string_pool_Pool;

typedef enum {
   string_pool_Color_Black,
   string_pool_Color_Red,
   _string_pool_Color_max = 255
} __attribute__((packed)) string_pool_Color;

typedef enum {
   string_pool_Dir_Left,
   string_pool_Dir_Right,
   _string_pool_Dir_max = 255
} __attribute__((packed)) string_pool_Dir;

struct string_pool_Node_ {
   uint16_t child[2];
   uint32_t word_idx;
   uint16_t parent;
   string_pool_Color color;
};

struct string_pool_Pool_ {
   uint32_t data_size;
   uint32_t data_capacity;
   char* data;
   string_pool_Node* nodes;
   uint16_t node_capacity;
   uint16_t node_count;
   string_pool_Node* root;
   uint32_t num_adds;
   uint32_t total_size;
};

static string_pool_Pool* string_pool_create(uint32_t data_capacity, uint16_t node_capacity);
static void string_pool_Pool_free(string_pool_Pool* p);
static const char* string_pool_Pool_getStart(const string_pool_Pool* pool);
static const char* string_pool_Pool_idx2str(const string_pool_Pool* pool, uint32_t idx);
static string_pool_Node* string_pool_Pool_getChild(string_pool_Pool* pool, string_pool_Node* x, string_pool_Dir dir);
static uint16_t string_pool_Pool_toIndex(const string_pool_Pool* pool, const string_pool_Node* x);
static void string_pool_Pool_rotate(string_pool_Pool* pool, string_pool_Node* p, string_pool_Dir dir);
static void string_pool_Pool_balance(string_pool_Pool* pool, string_pool_Node* n, string_pool_Node* p);
static uint32_t string_pool_compare(const char* left, const char* right, size_t rlen);
static uint32_t string_pool_Pool_add(string_pool_Pool* pool, const char* text, size_t len, bool filter);
static uint32_t string_pool_Pool_addStr(string_pool_Pool* pool, const char* text, bool filter);
static void string_pool_Pool_resize_data(string_pool_Pool* p, uint32_t capacity);
static void string_pool_Pool_resize_nodes(string_pool_Pool* p, uint16_t capacity);
static void string_pool_Pool_report(const string_pool_Pool* pool);
static string_pool_Pool* string_pool_create(uint32_t data_capacity, uint16_t node_capacity)
{
   string_pool_Pool* p = calloc(1, 48);
   string_pool_Pool_resize_data(p, data_capacity);
   p->data[0] = 0;
   p->data_size = 1;
   string_pool_Pool_resize_nodes(p, node_capacity);
   p->node_count = 1;
   return p;
}

static void string_pool_Pool_free(string_pool_Pool* p)
{
   free(p->data);
   free(p->nodes);
   free(p);
}

static const char* string_pool_Pool_getStart(const string_pool_Pool* pool)
{
   return pool->data;
}

static const char* string_pool_Pool_idx2str(const string_pool_Pool* pool, uint32_t idx)
{
   return (pool->data + idx);
}

static string_pool_Node* string_pool_Pool_getChild(string_pool_Pool* pool, string_pool_Node* x, string_pool_Dir dir)
{
   c2_assert((x) != NULL, "ast_utils/string_pool.c2:71: string_pool.Pool.getChild", "x");
   uint16_t idx = x->child[dir];
   if (idx) return (pool->nodes + idx);

   return NULL;
}

static uint16_t string_pool_Pool_toIndex(const string_pool_Pool* pool, const string_pool_Node* x)
{
   return ((uint16_t)((x - pool->nodes)));
}

static void string_pool_Pool_rotate(string_pool_Pool* pool, string_pool_Node* p, string_pool_Dir dir)
{
   string_pool_Dir rdir = ((string_pool_Dir)((1 - dir)));
   uint16_t g = p->parent;
   uint16_t s = p->child[rdir];
   uint16_t c = pool->nodes[s].child[dir];
   p->child[rdir] = c;
   uint16_t p_idx = string_pool_Pool_toIndex(pool, p);
   if (c) pool->nodes[c].parent = p_idx;
   pool->nodes[s].child[dir] = p_idx;
   p->parent = s;
   pool->nodes[s].parent = g;
   if (g) {
      pool->nodes[g].child[(p_idx == pool->nodes[g].child[string_pool_Dir_Right]) ? string_pool_Dir_Right : string_pool_Dir_Left] = s;
   } else {
      pool->root = &pool->nodes[s];
   }
}

static void string_pool_Pool_balance(string_pool_Pool* pool, string_pool_Node* n, string_pool_Node* p)
{
   n->color = string_pool_Color_Red;
   if ((p == NULL)) {
      pool->root = n;
      return;
   }
   string_pool_Node* g;
   string_pool_Dir dir;
   string_pool_Dir rdir;
   do {
      if ((p->color == string_pool_Color_Black)) return;

      if ((p->parent == 0)) goto Case_I4;

      g = (pool->nodes + p->parent);
      dir = ((p == ((pool->nodes + g->child[string_pool_Dir_Right])))) ? string_pool_Dir_Right : string_pool_Dir_Left;
      rdir = ((string_pool_Dir)((1 - dir)));
      uint16_t u_idx = g->child[rdir];
      if ((u_idx == 0)) goto Case_I56;

      string_pool_Node* u = (pool->nodes + u_idx);
      if ((u->color == string_pool_Color_Black)) goto Case_I56;

      p->color = string_pool_Color_Black;
      u->color = string_pool_Color_Black;
      g->color = string_pool_Color_Red;
      n = g;
      if ((n->parent == 0)) break;

      p = (pool->nodes + n->parent);
   } while (1);
   return;
   Case_I4:
   p->color = string_pool_Color_Black;
   return;
   Case_I56:
   if ((n == string_pool_Pool_getChild(pool, p, rdir))) {
      string_pool_Pool_rotate(pool, p, dir);
      p = (pool->nodes + g->child[dir]);
   }
   string_pool_Pool_rotate(pool, g, rdir);
   p->color = string_pool_Color_Black;
   g->color = string_pool_Color_Red;
   return;
}

static uint32_t string_pool_compare(const char* left, const char* right, size_t rlen)
{
   uint32_t i = 0;
   while ((i < rlen)) {
      char l = left[i];
      char r = right[i];
      char c = (l - r);
      if ((c < 0)) return 1;

      if ((c > 0)) return 0;

      i++;
   }
   if ((left[rlen] == 0)) return 2;

   return 0;
}

static uint32_t string_pool_Pool_add(string_pool_Pool* pool, const char* text, size_t len, bool filter)
{
   pool->num_adds++;
   pool->total_size += len;
   if (filter) {
      string_pool_Node* parent;
      string_pool_Node* n = pool->root;
      while (n) {
         const char* word = (pool->data + n->word_idx);
         switch (string_pool_compare(word, text, len)) {
         case 0:
            if (n->child[string_pool_Dir_Left]) {
               n = (pool->nodes + n->child[string_pool_Dir_Left]);
               continue;
            } else {
               n->child[string_pool_Dir_Left] = pool->node_count;
               goto after_loop;
            }
            break;
         case 1:
            if (n->child[string_pool_Dir_Right]) {
               n = (pool->nodes + n->child[string_pool_Dir_Right]);
               continue;
            } else {
               n->child[string_pool_Dir_Right] = pool->node_count;
               goto after_loop;
            }
            break;
         case 2:
            return n->word_idx;
         }
      }
      after_loop:
      parent = n;
      uint16_t parent_idx = ((uint16_t)(n ? ((uint16_t)((n - pool->nodes))) : 0));
      if ((pool->node_count == pool->node_capacity)) {
         string_pool_Pool_resize_nodes(pool, (pool->node_capacity * 2));
         parent = (pool->nodes + parent_idx);
      }
      n = (pool->nodes + pool->node_count);
      n->parent = parent_idx;
      pool->node_count++;
      n->word_idx = pool->data_size;
      n->child[string_pool_Dir_Left] = 0;
      n->child[string_pool_Dir_Right] = 0;
      string_pool_Pool_balance(pool, n, parent);
   }
   while ((((pool->data_size + len) + 1) > pool->data_capacity)) {
      string_pool_Pool_resize_data(pool, (pool->data_capacity * 2));
   }
   uint32_t idx = pool->data_size;
   char* dest = (pool->data + pool->data_size);
   memcpy(dest, text, len);
   dest[len] = 0;
   pool->data_size += (len + 1);
   c2_assert(((pool->data_size <= pool->data_capacity)) != 0, "ast_utils/string_pool.c2:236: string_pool.Pool.add", "pool.data_size<=pool.data_capacity");
   return idx;
}

static uint32_t string_pool_Pool_addStr(string_pool_Pool* pool, const char* text, bool filter)
{
   return string_pool_Pool_add(pool, text, strlen(text), filter);
}

static void string_pool_Pool_resize_data(string_pool_Pool* p, uint32_t capacity)
{
   p->data_capacity = capacity;
   char* data2 = malloc(capacity);
   if (p->data_size) {
      memcpy(data2, p->data, p->data_size);
      free(p->data);
   }
   p->data = data2;
}

static void string_pool_Pool_resize_nodes(string_pool_Pool* p, uint16_t capacity)
{
   p->node_capacity = capacity;
   string_pool_Node* nodes = malloc((p->node_capacity * 12));
   if (p->nodes) {
      c2_assert((p->root) != NULL, "ast_utils/string_pool.c2:258: string_pool.Pool.resize_nodes", "p.root");
      uint32_t root_idx = ((uint32_t)((p->root - p->nodes)));
      memcpy(nodes, p->nodes, (p->node_count * 12));
      free(p->nodes);
      p->root = &nodes[root_idx];
   }
   p->nodes = nodes;
}

static void string_pool_Pool_report(const string_pool_Pool* pool)
{
   uint32_t index_used = (pool->node_count * 12);
   uint32_t index_size = (pool->node_capacity * 12);
   index_used = (((index_used + 1023)) / 1024);
   index_size = (((index_size + 1023)) / 1024);
   printf("pool: %d(%u) entries, data %u(%u)/%u bytes, index %u/%u Kb\n", (pool->node_count - 1), pool->num_adds, pool->data_size, pool->total_size, pool->data_capacity, index_used, index_size);
}


// --- module console ---

static bool console_use_color = false;

static bool console_show_debug = false;

static bool console_show_timing = false;

static void console_init(void);
static void console_setDebug(bool enable);
static void console_setTiming(bool enable);
__attribute__((__format__(printf, 1, 2))) 
static void console_debug(const char* format, ...);
__attribute__((__format__(printf, 1, 2))) 
static void console_log(const char* format, ...);
__attribute__((__format__(printf, 1, 2))) 
static void console_warn(const char* format, ...);
__attribute__((__format__(printf, 1, 2))) 
static void console_error(const char* format, ...);
static void console_log_time(const char* item, uint64_t duration);
static void console_init(void)
{
   console_use_color = color_useColor();
}

static void console_setDebug(bool enable)
{
   console_show_debug = enable;
}

static void console_setTiming(bool enable)
{
   console_show_timing = enable;
}

__attribute__((__format__(printf, 1, 2))) 
static void console_debug(const char* format, ...)
{
   if (!console_show_debug) return;

   char buf[256];
   va_list args;
   va_start(args, format);
   vsnprintf(buf, 256, format, args);
   va_end(args);
   if (console_use_color) {
      printf("%s%s%s\n", color_Blue, buf, color_Normal);
   } else {
      printf("%s\n", buf);
   }
}

__attribute__((__format__(printf, 1, 2))) 
static void console_log(const char* format, ...)
{
   char buf[256];
   va_list args;
   va_start(args, format);
   vsnprintf(buf, 256, format, args);
   va_end(args);
   printf("%s\n", buf);
}

__attribute__((__format__(printf, 1, 2))) 
static void console_warn(const char* format, ...)
{
   char buf[256];
   va_list args;
   va_start(args, format);
   vsnprintf(buf, 256, format, args);
   va_end(args);
   if (console_use_color) {
      fprintf(stderr, "%swarning: %s%s\n", color_Yellow, buf, color_Normal);
   } else {
      fprintf(stderr, "warning: %s\n", buf);
   }
}

__attribute__((__format__(printf, 1, 2))) 
static void console_error(const char* format, ...)
{
   char buf[256];
   va_list args;
   va_start(args, format);
   vsprintf(buf, format, args);
   va_end(args);
   if (console_use_color) {
      fprintf(stderr, "%serror: %s%s\n", color_Red, buf, color_Normal);
   } else {
      fprintf(stderr, "error: %s\n", buf);
   }
}

static void console_log_time(const char* item, uint64_t duration)
{
   if (!console_show_timing) return;

   if (console_use_color) {
      printf("%s%s took %lu usec%s\n", color_Blue, item, duration, color_Normal);
   } else {
      printf("%s took %lu usec\n", item, duration);
   }
}


// --- module source_mgr ---
typedef struct source_mgr_CheckPoint_ source_mgr_CheckPoint;
typedef struct source_mgr_Location_ source_mgr_Location;
typedef struct source_mgr_File_ source_mgr_File;
typedef struct source_mgr_SourceMgr_ source_mgr_SourceMgr;

struct source_mgr_CheckPoint_ {
   uint32_t offset;
   uint32_t line;
};

struct source_mgr_Location_ {
   uint32_t line;
   uint32_t column;
   const char* filename;
   const char* line_start;
};

struct source_mgr_File_ {
   uint32_t filename;
   uint32_t offset;
   union {
      file_utils_Reader file;
      string_buffer_Buf* contents;
   };
   bool needed;
   bool is_generated;
   uint32_t last_offset;
   source_mgr_Location last_loc;
   uint32_t checkpoint_count;
   uint32_t checkpoint_capacity;
   uint32_t next_checkpoint;
   source_mgr_CheckPoint* checkpoints;
};

struct source_mgr_SourceMgr_ {
   const string_pool_Pool* pool;
   source_mgr_File* files;
   uint32_t num_files;
   uint32_t max_files;
   uint32_t num_open;
   uint32_t max_open;
   uint32_t last_file;
   uint32_t last_file_offset;
   uint32_t other_count;
   uint32_t other_size;
   uint32_t sources_count;
   uint32_t sources_size;
};

#define source_mgr_InitialMaxFiles 32
#define source_mgr_CheckPointSize 128
static void source_mgr_File_close(source_mgr_File* f);
static bool source_mgr_File_isOpen(const source_mgr_File* f);
static void source_mgr_File_clear(source_mgr_File* f);
static uint32_t source_mgr_File_size(const source_mgr_File* f);
static const char* source_mgr_File_data(source_mgr_File* f);
static void source_mgr_File_addCheckPoint(source_mgr_File* f, uint32_t offset, uint32_t line);
static const source_mgr_CheckPoint* source_mgr_File_findCheckPoint(source_mgr_File* f, uint32_t offset);
static source_mgr_SourceMgr* source_mgr_create(const string_pool_Pool* pool, uint32_t max_open);
static void source_mgr_SourceMgr_free(source_mgr_SourceMgr* sm);
static void source_mgr_SourceMgr_clear(source_mgr_SourceMgr* sm, int32_t handle);
static file_utils_Reader source_mgr_SourceMgr_openInternal(source_mgr_SourceMgr* sm, const char* filename, src_loc_SrcLoc loc);
static bool source_mgr_SourceMgr_close_oldest(source_mgr_SourceMgr* sm);
static void source_mgr_SourceMgr_resize(source_mgr_SourceMgr* sm);
static int32_t source_mgr_SourceMgr_addGenerated(source_mgr_SourceMgr* sm, string_buffer_Buf* contents, uint32_t name);
static int32_t source_mgr_SourceMgr_open(source_mgr_SourceMgr* sm, uint32_t filename, src_loc_SrcLoc loc, bool is_source);
static void source_mgr_SourceMgr_close(source_mgr_SourceMgr* sm, int32_t file_id);
static void source_mgr_SourceMgr_checkOpen(source_mgr_SourceMgr* sm, int32_t handle);
static const char* source_mgr_SourceMgr_get_content(source_mgr_SourceMgr* sm, int32_t handle);
static const char* source_mgr_SourceMgr_get_token_source(source_mgr_SourceMgr* sm, src_loc_SrcLoc loc);
static uint32_t source_mgr_SourceMgr_get_offset(source_mgr_SourceMgr* sm, int32_t handle);
static uint32_t source_mgr_SourceMgr_getFileNameIdx(source_mgr_SourceMgr* sm, int32_t handle);
static const char* source_mgr_SourceMgr_getFileName(source_mgr_SourceMgr* sm, int32_t handle);
static void source_mgr_SourceMgr_dump(const source_mgr_SourceMgr* sm);
static source_mgr_File* source_mgr_SourceMgr_find_file(source_mgr_SourceMgr* sm, src_loc_SrcLoc loc);
static source_mgr_Location source_mgr_SourceMgr_locate(source_mgr_SourceMgr* sm, src_loc_SrcLoc loc);
static source_mgr_Location source_mgr_SourceMgr_getLocation(source_mgr_SourceMgr* sm, src_loc_SrcLoc sloc);
static const char* source_mgr_SourceMgr_loc2str(source_mgr_SourceMgr* sm, src_loc_SrcLoc sloc);
static void source_mgr_SourceMgr_report(const source_mgr_SourceMgr* sm);
static void source_mgr_File_close(source_mgr_File* f)
{
   if (!f->is_generated) file_utils_Reader_close(&f->file);
}

static bool source_mgr_File_isOpen(const source_mgr_File* f)
{
   if (f->is_generated) return true;

   return file_utils_Reader_isOpen(&f->file);
}

static void source_mgr_File_clear(source_mgr_File* f)
{
   if (f->is_generated) {
      free(f->contents);
   } else {
      file_utils_Reader_close(&f->file);
   }
   free(f->checkpoints);
   f->checkpoints = NULL;
}

static uint32_t source_mgr_File_size(const source_mgr_File* f)
{
   if (f->is_generated) return string_buffer_Buf_size(f->contents);

   return f->file.size;
}

static const char* source_mgr_File_data(source_mgr_File* f)
{
   if (f->is_generated) return string_buffer_Buf_data(f->contents);

   return file_utils_Reader_char_data(&f->file);
}

static void source_mgr_File_addCheckPoint(source_mgr_File* f, uint32_t offset, uint32_t line)
{
   if ((f->checkpoint_count == f->checkpoint_capacity)) {
      if (f->checkpoint_capacity) f->checkpoint_capacity *= 2;
      else f->checkpoint_capacity = 16;
      source_mgr_CheckPoint* checkpoints = malloc((f->checkpoint_capacity * 8));
      if (f->checkpoints) {
         memcpy(checkpoints, f->checkpoints, (f->checkpoint_count * 8));
         free(f->checkpoints);
      }
      f->checkpoints = checkpoints;
   }
   source_mgr_CheckPoint* c = &f->checkpoints[f->checkpoint_count];
   f->checkpoint_count++;
   c->offset = offset;
   c->line = line;
   f->next_checkpoint = (offset + source_mgr_CheckPointSize);
}

static const source_mgr_CheckPoint* source_mgr_File_findCheckPoint(source_mgr_File* f, uint32_t offset)
{
   if ((f->checkpoint_count && (offset > f->checkpoints[(f->checkpoint_count - 1)].offset))) return &f->checkpoints[(f->checkpoint_count - 1)];

   uint32_t left = 0;
   uint32_t right = f->checkpoint_count;
   while ((left != right)) {
      uint32_t middle = (((left + right)) / 2);
      const source_mgr_CheckPoint* cur = &f->checkpoints[middle];
      if ((offset < cur->offset)) {
         if ((right == middle)) return cur;

         right = middle;
         continue;
      }
      if ((offset > f->checkpoints[(middle + 1)].offset)) {
         if ((left == middle)) return cur;

         left = middle;
         continue;
      }
      return cur;
   }
   return NULL;
}

static source_mgr_SourceMgr* source_mgr_create(const string_pool_Pool* pool, uint32_t max_open)
{
   source_mgr_SourceMgr* sm = calloc(1, 56);
   sm->pool = pool;
   sm->max_files = source_mgr_InitialMaxFiles;
   sm->files = malloc((80 * sm->max_files));
   sm->max_open = max_open;
   return sm;
}

static void source_mgr_SourceMgr_free(source_mgr_SourceMgr* sm)
{
   for (uint32_t i = 0; (i < sm->num_files); i++) {
      source_mgr_File_close(&sm->files[i]);
   }
   free(sm->files);
   free(sm);
}

static void source_mgr_SourceMgr_clear(source_mgr_SourceMgr* sm, int32_t handle)
{
   handle++;
   for (int32_t i = handle; (i < ((int32_t)(sm->num_files))); i++) {
      source_mgr_File* f = &sm->files[i];
      if ((f->needed && !f->is_generated)) {
         printf("WARN %s still not closed\n", string_pool_Pool_idx2str(sm->pool, f->filename));
      }
      source_mgr_File_clear(f);
   }
   sm->num_files = ((uint32_t)(handle));
   sm->num_open = 0;
   sm->other_count = ((uint32_t)(handle));
   sm->other_size = 0;
   sm->sources_count = 0;
   sm->sources_size = 0;
   for (uint32_t i = 0; (i < sm->num_files); i++) {
      const source_mgr_File* f = &sm->files[i];
      if ((!f->is_generated && source_mgr_File_isOpen(f))) sm->num_open++;
      sm->other_size += source_mgr_File_size(f);
   }
}

static file_utils_Reader source_mgr_SourceMgr_openInternal(source_mgr_SourceMgr* sm, const char* filename, src_loc_SrcLoc loc)
{
   file_utils_Reader file;
   if (file_utils_Reader_open(&file, filename)) {
      sm->num_open++;
   } else {
      char error_msg[256];
      if ((file.errno == file_utils_Err_not_a_file)) {
         sprintf(error_msg, "cannot open %s: %s\n", filename, "not a regular file");
      } else {
         sprintf(error_msg, "cannot open %s: %s\n", filename, strerror(file.errno));
      }
      if (loc) {
         fprintf(stderr, "%s: %serror:%s %s\n", source_mgr_SourceMgr_loc2str(sm, loc), color_Red, color_Normal, error_msg);
      } else {
         fprintf(stderr, "%serror%s: %s\n", color_Red, color_Normal, error_msg);
      }
   }
   return file;
}

static bool source_mgr_SourceMgr_close_oldest(source_mgr_SourceMgr* sm)
{
   for (uint32_t i = 0; (i < sm->num_files); i++) {
      source_mgr_File* f = &sm->files[i];
      if ((source_mgr_File_isOpen(f) && !f->needed)) {
         source_mgr_File_close(f);
         sm->num_open--;
         return true;
      }
   }
   return false;
}

static void source_mgr_SourceMgr_resize(source_mgr_SourceMgr* sm)
{
   sm->max_files *= 2;
   source_mgr_File* files2 = malloc((80 * sm->max_files));
   memcpy(files2, sm->files, (sm->num_files * 80));
   free(sm->files);
   sm->files = files2;
}

static int32_t source_mgr_SourceMgr_addGenerated(source_mgr_SourceMgr* sm, string_buffer_Buf* contents, uint32_t name)
{
   if ((sm->num_files == sm->max_files)) source_mgr_SourceMgr_resize(sm);
   source_mgr_File* f = &sm->files[sm->num_files];
   memset(f, 0, 80);
   uint32_t offset = 1;
   if (sm->num_files) {
      source_mgr_File* last = &sm->files[(sm->num_files - 1)];
      offset = (last->offset + source_mgr_File_size(last));
   }
   f->filename = name;
   f->offset = offset;
   f->contents = contents;
   f->next_checkpoint = source_mgr_CheckPointSize;
   f->needed = true;
   f->is_generated = true;
   int32_t file_id = ((int32_t)(sm->num_files));
   sm->num_files++;
   return file_id;
}

static int32_t source_mgr_SourceMgr_open(source_mgr_SourceMgr* sm, uint32_t filename, src_loc_SrcLoc loc, bool is_source)
{
   if ((sm->num_open == sm->max_open)) {
      if (!source_mgr_SourceMgr_close_oldest(sm)) {
         fprintf(stderr, "%serror%s: too many files open\n", color_Red, color_Normal);
         return -1;
      }
   }
   file_utils_Reader file = source_mgr_SourceMgr_openInternal(sm, string_pool_Pool_idx2str(sm->pool, filename), loc);
   if (!file_utils_Reader_isOpen(&file)) return -1;

   if ((sm->num_files == sm->max_files)) source_mgr_SourceMgr_resize(sm);
   int32_t file_id = ((int32_t)(sm->num_files));
   source_mgr_File* f = &sm->files[sm->num_files];
   memset(f, 0, 80);
   uint32_t offset = 1;
   if (sm->num_files) {
      source_mgr_File* last = &sm->files[(sm->num_files - 1)];
      offset = (last->offset + source_mgr_File_size(last));
   }
   f->filename = filename;
   f->offset = offset;
   f->file = file;
   f->next_checkpoint = source_mgr_CheckPointSize;
   f->needed = true;
   if (is_source) {
      sm->sources_count++;
      sm->sources_size += file.size;
   } else {
      sm->other_count++;
      sm->other_size += file.size;
   }
   sm->num_files++;
   return file_id;
}

static void source_mgr_SourceMgr_close(source_mgr_SourceMgr* sm, int32_t file_id)
{
   sm->files[file_id].needed = false;
}

static void source_mgr_SourceMgr_checkOpen(source_mgr_SourceMgr* sm, int32_t handle)
{
   source_mgr_File* f = &sm->files[handle];
   if (source_mgr_File_isOpen(f)) return;

   if ((sm->num_open == sm->max_open)) {
      if (!source_mgr_SourceMgr_close_oldest(sm)) {
         fprintf(stderr, "%serror%s: too many files open\n", color_Red, color_Normal);
         exit(-1);
      }
   }
   c2_assert((!f->is_generated) != 0, "common/source_mgr.c2:329: source_mgr.SourceMgr.checkOpen", "!f.is_generated");
   f->file = source_mgr_SourceMgr_openInternal(sm, string_pool_Pool_idx2str(sm->pool, f->filename), 0);
   if (!source_mgr_File_isOpen(f)) exit(-1);
}

static const char* source_mgr_SourceMgr_get_content(source_mgr_SourceMgr* sm, int32_t handle)
{
   source_mgr_SourceMgr_checkOpen(sm, handle);
   return source_mgr_File_data(&sm->files[handle]);
}

static const char* source_mgr_SourceMgr_get_token_source(source_mgr_SourceMgr* sm, src_loc_SrcLoc loc)
{
   source_mgr_File* f = source_mgr_SourceMgr_find_file(sm, loc);
   if (!f) return "";

   uint32_t offset = (loc - f->offset);
   const char* data = source_mgr_File_data(f);
   data += offset;
   return data;
}

static uint32_t source_mgr_SourceMgr_get_offset(source_mgr_SourceMgr* sm, int32_t handle)
{
   return sm->files[handle].offset;
}

static uint32_t source_mgr_SourceMgr_getFileNameIdx(source_mgr_SourceMgr* sm, int32_t handle)
{
   return sm->files[handle].filename;
}

static const char* source_mgr_SourceMgr_getFileName(source_mgr_SourceMgr* sm, int32_t handle)
{
   uint32_t idx = sm->files[handle].filename;
   return string_pool_Pool_idx2str(sm->pool, idx);
}

static void source_mgr_SourceMgr_dump(const source_mgr_SourceMgr* sm)
{
   printf("SourceMgr  files %u  (open %u/%u)\n", sm->num_files, sm->num_open, sm->max_open);
   uint32_t total_size = 0;
   for (uint32_t i = 0; (i < sm->num_files); i++) {
      source_mgr_File* f = &sm->files[i];
      total_size += f->file.size;
      printf("  [%2u]  %7u  %7u %u %s\n", i, f->offset, f->file.size, f->is_generated, string_pool_Pool_idx2str(sm->pool, f->filename));
   }
   printf("Total size %u\n", total_size);
}

static source_mgr_File* source_mgr_SourceMgr_find_file(source_mgr_SourceMgr* sm, src_loc_SrcLoc loc)
{
   uint32_t left = 0;
   if ((loc >= sm->last_file_offset)) {
      left = sm->last_file;
      if ((loc <= (sm->last_file_offset + sm->files[sm->last_file].file.size))) {
         return &sm->files[sm->last_file];
      }
   }
   uint32_t right = sm->num_files;
   while ((left != right)) {
      uint32_t middle = (((left + right)) / 2);
      source_mgr_File* f = &sm->files[middle];
      if ((loc < f->offset)) {
         right = middle;
         continue;
      }
      if ((loc > (f->offset + source_mgr_File_size(f)))) {
         left = middle;
         continue;
      }
      source_mgr_SourceMgr_checkOpen(sm, ((int32_t)(middle)));
      sm->last_file = middle;
      sm->last_file_offset = f->offset;
      return f;
   }
   return NULL;
}

static source_mgr_Location source_mgr_SourceMgr_locate(source_mgr_SourceMgr* sm, src_loc_SrcLoc loc)
{
   source_mgr_Location l = { 1, 1, "-", NULL };
   if ((loc == 0)) return l;

   source_mgr_File* f = source_mgr_SourceMgr_find_file(sm, loc);
   if (f) {
      l.filename = string_pool_Pool_idx2str(sm->pool, f->filename);
      uint32_t offset = (loc - f->offset);
      uint32_t last_offset = 0;
      const char* data = source_mgr_File_data(f);
      const char* line = data;
      uint32_t line_nr = l.line;
      const source_mgr_CheckPoint* cp = source_mgr_File_findCheckPoint(f, offset);
      if (cp) {
         line_nr = cp->line;
         last_offset = cp->offset;
         line = (data + cp->offset);
      } else {
      }
      for (uint32_t i = last_offset; (i < offset); i++) {
         if ((data[i] == '\n')) {
            line_nr++;
            line = ((data + i) + 1);
            if ((i >= f->next_checkpoint)) {
               source_mgr_File_addCheckPoint(f, (i + 1), line_nr);
            }
         }
      }
      l.line = line_nr;
      l.column = (((uint32_t)((&data[offset] - line))) + 1);
      l.line_start = line;
      f->last_offset = offset;
      f->last_loc = l;
   }
   return l;
}

static source_mgr_Location source_mgr_SourceMgr_getLocation(source_mgr_SourceMgr* sm, src_loc_SrcLoc sloc)
{
   return source_mgr_SourceMgr_locate(sm, sloc);
}

static const char* source_mgr_SourceMgr_loc2str(source_mgr_SourceMgr* sm, src_loc_SrcLoc sloc)
{
   static char tmp[256];
   if ((sloc == 0)) {
      strcpy(tmp, "-");
   } else {
      source_mgr_Location loc = source_mgr_SourceMgr_locate(sm, sloc);
      sprintf(tmp, "%s:%u:%u", loc.filename, loc.line, loc.column);
   }
   return tmp;
}

static void source_mgr_SourceMgr_report(const source_mgr_SourceMgr* sm)
{
   printf("source-mgr: %u files, %u sources (%u bytes), %u other (%u bytes)\n", sm->num_files, sm->sources_count, sm->sources_size, sm->other_count, sm->other_size);
   uint32_t total_size = 0;
   for (uint32_t i = 0; (i < sm->num_files); i++) {
      source_mgr_File* f = &sm->files[i];
      total_size += ((f->checkpoint_capacity * 8));
      printf("  %6u  %u  %4u  %s\n", source_mgr_File_size(f), f->is_generated, (f->checkpoint_count * 8), string_pool_Pool_idx2str(sm->pool, f->filename));
   }
   printf("  Total size %u\n", total_size);
}


// --- module string_list ---
typedef struct string_list_List_ string_list_List;

struct string_list_List_ {
   const string_pool_Pool* pool;
   uint32_t* indexes;
   uint32_t count;
   uint32_t capacity;
};

static void string_list_List_init(string_list_List* l, const string_pool_Pool* pool);
static void string_list_List_free(string_list_List* l);
static void string_list_List_clear(string_list_List* l);
static void string_list_List_resize(string_list_List* l, uint32_t capacity);
static void string_list_List_add(string_list_List* l, uint32_t name_idx);
static bool string_list_List_contains(const string_list_List* l, const char* name);
static bool string_list_List_contains_idx(const string_list_List* l, uint32_t idx);
static uint32_t string_list_List_length(const string_list_List* l);
static const char* string_list_List_get(const string_list_List* l, uint32_t idx);
static uint32_t string_list_List_get_idx(const string_list_List* l, uint32_t idx);
static uint32_t* string_list_List_getData(string_list_List* l);
static void string_list_List_init(string_list_List* l, const string_pool_Pool* pool)
{
   memset(l, 0, 24);
   l->pool = pool;
   string_list_List_resize(l, 16);
}

static void string_list_List_free(string_list_List* l)
{
   free(l->indexes);
}

static void string_list_List_clear(string_list_List* l)
{
   l->count = 0;
}

static void string_list_List_resize(string_list_List* l, uint32_t capacity)
{
   l->capacity = capacity;
   uint32_t* indexes2 = malloc((capacity * 4));
   if (l->count) {
      memcpy(indexes2, l->indexes, (l->count * 4));
      free(l->indexes);
   }
   l->indexes = indexes2;
}

static void string_list_List_add(string_list_List* l, uint32_t name_idx)
{
   if ((l->count == l->capacity)) string_list_List_resize(l, (l->capacity * 2));
   l->indexes[l->count] = name_idx;
   l->count++;
}

static bool string_list_List_contains(const string_list_List* l, const char* name)
{
   for (uint32_t i = 0; (i < l->count); i++) {
      if ((strcmp(string_pool_Pool_idx2str(l->pool, l->indexes[i]), name) == 0)) return true;

   }
   return false;
}

static bool string_list_List_contains_idx(const string_list_List* l, uint32_t idx)
{
   for (uint32_t i = 0; (i < l->count); i++) {
      if ((l->indexes[i] == idx)) return true;

   }
   return false;
}

static uint32_t string_list_List_length(const string_list_List* l)
{
   return l->count;
}

static const char* string_list_List_get(const string_list_List* l, uint32_t idx)
{
   return string_pool_Pool_idx2str(l->pool, l->indexes[idx]);
}

static uint32_t string_list_List_get_idx(const string_list_List* l, uint32_t idx)
{
   return l->indexes[idx];
}

static uint32_t* string_list_List_getData(string_list_List* l)
{
   return l->indexes;
}


// --- module target_info ---
typedef struct target_info_Info_ target_info_Info;

typedef enum {
   target_info_Arch_Unknown,
   target_info_Arch_I686,
   target_info_Arch_Arm,
   target_info_Arch_X86_64,
   target_info_Arch_Arm64,
   target_info_Arch_Riscv_32,
   _target_info_Arch_max = 255
} __attribute__((packed)) target_info_Arch;

typedef enum {
   target_info_System_Unknown,
   target_info_System_Linux,
   target_info_System_Darwin,
   target_info_System_Cygwin,
   _target_info_System_max = 255
} __attribute__((packed)) target_info_System;

typedef enum {
   target_info_Vendor_Unknown,
   target_info_Vendor_Apple,
   _target_info_Vendor_max = 255
} __attribute__((packed)) target_info_Vendor;

typedef enum {
   target_info_Abi_Unknown,
   target_info_Abi_GNU,
   target_info_Abi_GNU_EABI,
   target_info_Abi_MACHO,
   target_info_Abi_WIN32,
   target_info_Abi_Rv32G,
   _target_info_Abi_max = 255
} __attribute__((packed)) target_info_Abi;

struct target_info_Info_ {
   target_info_Arch arch;
   target_info_System sys;
   target_info_Vendor vendor;
   target_info_Abi abi;
   uint32_t intWidth;
   char triple[80];
};

static const char* target_info_system_names[4] = { "unknown", "linux", "darwin", "cygwin" };

static const char* target_info_arch_names[6] = { "unknown", "i686", "arm", "x86_64", "arm_64", "riscv32" };

static const char* target_info_vendor_names[2] = { "unknown", "apple" };

static const char* target_info_abi_names[6] = { "unknown", "gnu", "gnueabi", "macho", "win32", "rv32" };

static target_info_System target_info_str2sys(const char* name);
static target_info_Arch target_info_str2arch(const char* name);
static target_info_Vendor target_info_str2vendor(const char* name);
static target_info_Abi target_info_str2abi(const char* name);
static void target_info_Info_getNative(target_info_Info* info);
static void target_info_Info_init(target_info_Info* info);
static bool target_info_Info_fromString(target_info_Info* info, const char* triple);
static const char* target_info_Info_str(const target_info_Info* info);
static target_info_System target_info_str2sys(const char* name)
{
   for (uint32_t i = 0; (i < 4); i++) {
      if ((strcasecmp(target_info_system_names[i], name) == 0)) return ((target_info_System)(i));

   }
   return target_info_System_Unknown;
}

static target_info_Arch target_info_str2arch(const char* name)
{
   for (uint32_t i = 0; (i < 6); i++) {
      if ((strcasecmp(target_info_arch_names[i], name) == 0)) return ((target_info_Arch)(i));

   }
   return target_info_Arch_Unknown;
}

static target_info_Vendor target_info_str2vendor(const char* name)
{
   for (uint32_t i = 0; (i < 2); i++) {
      if ((strcasecmp(target_info_vendor_names[i], name) == 0)) return ((target_info_Vendor)(i));

   }
   return target_info_Vendor_Unknown;
}

static target_info_Abi target_info_str2abi(const char* name)
{
   for (uint32_t i = 0; (i < 6); i++) {
      if ((strcasecmp(target_info_abi_names[i], name) == 0)) return ((target_info_Abi)(i));

   }
   return target_info_Abi_Unknown;
}

static void target_info_Info_getNative(target_info_Info* info)
{
   utsname un;
   if ((uname(&un) != 0)) {
      console_error("error getting system info: %s", strerror(*__errno_location()));
      exit(EXIT_FAILURE);
   }
   info->sys = target_info_str2sys(un.sysname);
   switch (info->sys) {
   case target_info_System_Unknown:
      console_error("unsupported system: '%s'", un.sysname);
      exit(EXIT_FAILURE);
      break;
   case target_info_System_Linux:
      info->vendor = target_info_Vendor_Unknown;
      info->abi = target_info_Abi_GNU;
      break;
   case target_info_System_Darwin:
      info->vendor = target_info_Vendor_Apple;
      info->abi = target_info_Abi_MACHO;
      break;
   case target_info_System_Cygwin:
      info->vendor = target_info_Vendor_Unknown;
      info->abi = target_info_Abi_WIN32;
      break;
   }
   info->arch = target_info_str2arch(un.machine);
   if ((info->arch == target_info_Arch_Unknown)) {
      console_error("unsupported arch: '%s'", un.machine);
      exit(EXIT_FAILURE);
   }
   target_info_Info_init(info);
}

static void target_info_Info_init(target_info_Info* info)
{
   switch (info->arch) {
   case target_info_Arch_Unknown:
      info->intWidth = 64;
      break;
   case target_info_Arch_I686:
      __attribute__((fallthrough));
   case target_info_Arch_Arm:
      info->intWidth = 32;
      break;
   case target_info_Arch_X86_64:
      __attribute__((fallthrough));
   case target_info_Arch_Arm64:
      info->intWidth = 64;
      break;
   case target_info_Arch_Riscv_32:
      info->intWidth = 32;
      break;
   }
   sprintf(info->triple, "%s-%s-%s-%s", target_info_arch_names[info->arch], target_info_vendor_names[info->vendor], target_info_system_names[info->sys], target_info_abi_names[info->abi]);
}

static bool target_info_Info_fromString(target_info_Info* info, const char* triple)
{
   char arch_str[32];
   char vendor_str[32];
   char sys_str[32];
   char abi_str[32];
   char* matches[4] = { arch_str, vendor_str, sys_str, abi_str };
   int32_t match = 0;
   const char* start = triple;
   const char* cp = start;
   while (*cp) {
      if ((*cp == '-')) {
         uint32_t len = ((uint32_t)((cp - start)));
         strncpy(matches[match], start, len);
         matches[match][len] = 0;
         start = (cp + 1);
         match++;
         if ((match == 5)) return false;

      }
      cp++;
   }
   if ((cp != start)) {
      uint32_t len = ((uint32_t)((cp - start)));
      strncpy(matches[match], start, len);
      matches[match][len] = 0;
      match++;
   }
   if ((match != 4)) return false;

   info->arch = target_info_str2arch(arch_str);
   if ((info->arch == target_info_Arch_Unknown)) {
      console_error("unsupported arch: %s", arch_str);
      return false;
   }
   info->vendor = target_info_str2vendor(vendor_str);
   info->sys = target_info_str2sys(sys_str);
   if ((info->sys == target_info_System_Unknown)) {
   }
   info->abi = target_info_str2abi(abi_str);
   if ((info->abi == target_info_Abi_Unknown)) {
      console_error("unsupported ABI: %s", abi_str);
      return false;
   }
   target_info_Info_init(info);
   return true;
}

static const char* target_info_Info_str(const target_info_Info* info)
{
   return info->triple;
}


// --- module token ---
typedef struct token_Token_ token_Token;

typedef enum {
   token_Kind_None,
   token_Kind_Identifier,
   token_Kind_IntegerLiteral,
   token_Kind_FloatLiteral,
   token_Kind_CharLiteral,
   token_Kind_StringLiteral,
   token_Kind_LParen,
   token_Kind_RParen,
   token_Kind_LSquare,
   token_Kind_RSquare,
   token_Kind_LBrace,
   token_Kind_RBrace,
   token_Kind_Exclaim,
   token_Kind_ExclaimEqual,
   token_Kind_Star,
   token_Kind_StarEqual,
   token_Kind_Amp,
   token_Kind_AmpAmp,
   token_Kind_AmpEqual,
   token_Kind_Pipe,
   token_Kind_PipePipe,
   token_Kind_PipeEqual,
   token_Kind_Equal,
   token_Kind_EqualEqual,
   token_Kind_Semicolon,
   token_Kind_Colon,
   token_Kind_At,
   token_Kind_Caret,
   token_Kind_CaretEqual,
   token_Kind_Question,
   token_Kind_Dot,
   token_Kind_Ellipsis,
   token_Kind_Comma,
   token_Kind_Plus,
   token_Kind_PlusPlus,
   token_Kind_PlusEqual,
   token_Kind_Minus,
   token_Kind_MinusMinus,
   token_Kind_MinusEqual,
   token_Kind_Tilde,
   token_Kind_Slash,
   token_Kind_SlashEqual,
   token_Kind_Percent,
   token_Kind_PercentEqual,
   token_Kind_Less,
   token_Kind_LessLess,
   token_Kind_LessEqual,
   token_Kind_LessLessEqual,
   token_Kind_Greater,
   token_Kind_GreaterGreater,
   token_Kind_GreaterEqual,
   token_Kind_GreaterGreaterEqual,
   token_Kind_KW_bool,
   token_Kind_KW_char,
   token_Kind_KW_i8,
   token_Kind_KW_i16,
   token_Kind_KW_i32,
   token_Kind_KW_i64,
   token_Kind_KW_u8,
   token_Kind_KW_u16,
   token_Kind_KW_u32,
   token_Kind_KW_u64,
   token_Kind_KW_reg8,
   token_Kind_KW_reg16,
   token_Kind_KW_reg32,
   token_Kind_KW_reg64,
   token_Kind_KW_isize,
   token_Kind_KW_usize,
   token_Kind_KW_f32,
   token_Kind_KW_f64,
   token_Kind_KW_void,
   token_Kind_KW_as,
   token_Kind_KW_asm,
   token_Kind_KW_assert,
   token_Kind_KW_break,
   token_Kind_KW_case,
   token_Kind_KW_cast,
   token_Kind_KW_const,
   token_Kind_KW_continue,
   token_Kind_KW_default,
   token_Kind_KW_do,
   token_Kind_KW_elemsof,
   token_Kind_KW_else,
   token_Kind_KW_enum_max,
   token_Kind_KW_enum_min,
   token_Kind_KW_enum,
   token_Kind_KW_fallthrough,
   token_Kind_KW_false,
   token_Kind_KW_fn,
   token_Kind_KW_for,
   token_Kind_KW_goto,
   token_Kind_KW_if,
   token_Kind_KW_import,
   token_Kind_KW_local,
   token_Kind_KW_module,
   token_Kind_KW_nil,
   token_Kind_KW_offsetof,
   token_Kind_KW_public,
   token_Kind_KW_return,
   token_Kind_KW_sizeof,
   token_Kind_KW_sswitch,
   token_Kind_KW_static_assert,
   token_Kind_KW_struct,
   token_Kind_KW_switch,
   token_Kind_KW_template,
   token_Kind_KW_to_container,
   token_Kind_KW_true,
   token_Kind_KW_type,
   token_Kind_KW_union,
   token_Kind_KW_volatile,
   token_Kind_KW_while,
   token_Kind_Feat_if,
   token_Kind_Feat_else,
   token_Kind_Feat_endif,
   token_Kind_LineComment,
   token_Kind_BlockComment,
   token_Kind_Eof,
   token_Kind_Warning,
   token_Kind_Error,
   _token_Kind_max = 255
} __attribute__((packed)) token_Kind;

struct token_Token_ {
   src_loc_SrcLoc loc;
   token_Kind kind;
   bool more;
   uint8_t radix;
   union {
      const char* error_msg;
      struct {
         uint32_t text_idx;
         uint32_t text_len;
      };
      uint64_t int_value;
      double float_value;
      uint8_t char_value;
   };
};

static const char* token_token_names[119] = {
   "none",
   "identifier",
   "integer",
   "float",
   "character",
   "string",
   "(",
   ")",
   "[",
   "]",
   "{",
   "}",
   "!",
   "!=",
   "*",
   "*=",
   "&",
   "&&",
   "&=",
   "|",
   "||",
   "|=",
   "=",
   "==",
   ";",
   ":",
   "@",
   "^",
   "^=",
   "?",
   ".",
   "...",
   ",",
   "+",
   "++",
   "+=",
   "-",
   "--",
   "-=",
   "~",
   "/",
   "/=",
   "%",
   "%=",
   "<",
   "<<",
   "<=",
   "<<=",
   ">",
   ">>",
   ">=",
   ">>=",
   "bool",
   "char",
   "i8",
   "i16",
   "i32",
   "i64",
   "u8",
   "u16",
   "u32",
   "u64",
   "reg8",
   "reg16",
   "reg32",
   "reg64",
   "isize",
   "usize",
   "f32",
   "f64",
   "void",
   "as",
   "asm",
   "assert",
   "break",
   "case",
   "cast",
   "const",
   "continue",
   "default",
   "do",
   "elemsof",
   "else",
   "enum_max",
   "enum_min",
   "enum",
   "fallthrough",
   "false",
   "fn",
   "for",
   "goto",
   "if",
   "import",
   "local",
   "module",
   "nil",
   "offsetof",
   "public",
   "return",
   "sizeof",
   "sswitch",
   "static_assert",
   "struct",
   "switch",
   "template",
   "to_container",
   "true",
   "type",
   "union",
   "volatile",
   "while",
   "#if",
   "#else",
   "#endif",
   "l-comment",
   "b-comment",
   "eof",
   "warning",
   "error"
};

static const char* token_kind2str(token_Kind kind);
static void token_Token_init(token_Token* tok);
static const char* token_kind2str(token_Kind kind)
{
   return token_token_names[kind];
}

static void token_Token_init(token_Token* tok)
{
   memset(tok, 0, 16);
   tok->more = true;
}


// --- module init_checker ---
typedef struct init_checker_InitEntry_ init_checker_InitEntry;
typedef struct init_checker_Checker_ init_checker_Checker;

struct init_checker_InitEntry_ {
   uint32_t index;
   src_loc_SrcLoc loc;
};

struct init_checker_Checker_ {
   init_checker_InitEntry* entries;
   uint32_t count;
   uint32_t capacity;
   uint32_t max;
};

static init_checker_Checker init_checker_Checker_create(uint32_t capacity);
static void init_checker_Checker_free(init_checker_Checker* c);
static uint32_t init_checker_Checker_getCount(const init_checker_Checker* c);
static void init_checker_Checker_add(init_checker_Checker* c, uint32_t index, src_loc_SrcLoc loc);
static src_loc_SrcLoc init_checker_Checker_find(init_checker_Checker* c, uint32_t index);
static init_checker_Checker init_checker_Checker_create(uint32_t capacity)
{
   c2_assert(((capacity != 0)) != 0, "analyser/init_checker.c2:37: init_checker.Checker.create", "capacity!=0");
   init_checker_Checker c;
   c.count = 0;
   c.capacity = capacity;
   c.max = 0;
   c.entries = malloc((capacity * 8));
   return c;
}

static void init_checker_Checker_free(init_checker_Checker* c)
{
   free(c->entries);
}

static uint32_t init_checker_Checker_getCount(const init_checker_Checker* c)
{
   return c->count;
}

static void init_checker_Checker_add(init_checker_Checker* c, uint32_t index, src_loc_SrcLoc loc)
{
   if ((c->count >= c->capacity)) {
      c->capacity *= 2;
      init_checker_InitEntry* entries = malloc((c->capacity * 8));
      memcpy(entries, c->entries, (c->count * 8));
      free(c->entries);
      c->entries = entries;
   }
   init_checker_InitEntry* entry = &c->entries[c->count];
   entry->index = index;
   entry->loc = loc;
   if ((index > c->max)) c->max = index;
   c->count++;
}

static src_loc_SrcLoc init_checker_Checker_find(init_checker_Checker* c, uint32_t index)
{
   if ((index > c->max)) return 0;

   for (uint32_t i = 0; (i < c->count); i++) {
      if ((c->entries[i].index == index)) return c->entries[i].loc;

   }
   return 0;
}


// --- module attr ---
typedef struct attr_Value_ attr_Value;
typedef struct attr_Attr_ attr_Attr;

typedef enum {
   attr_AttrKind_Unknown,
   attr_AttrKind_Export,
   attr_AttrKind_Packed,
   attr_AttrKind_Unused,
   attr_AttrKind_UnusedParams,
   attr_AttrKind_Section,
   attr_AttrKind_NoReturn,
   attr_AttrKind_Inline,
   attr_AttrKind_PrintfFormat,
   attr_AttrKind_Aligned,
   attr_AttrKind_Weak,
   attr_AttrKind_Opaque,
   attr_AttrKind_CName,
   attr_AttrKind_NoTypeDef,
   attr_AttrKind_Constructor,
   attr_AttrKind_Destructor,
   attr_AttrKind_Pure,
   attr_AttrKind_AutoFile,
   attr_AttrKind_AutoLine,
   _attr_AttrKind_max = 255
} __attribute__((packed)) attr_AttrKind;

typedef enum {
   attr_ValueKind_None,
   attr_ValueKind_Number,
   attr_ValueKind_String,
   _attr_ValueKind_max = 255
} __attribute__((packed)) attr_ValueKind;

struct attr_Value_ {
   src_loc_SrcLoc loc;
   union {
      uint32_t text;
      uint32_t number;
   };
};

struct attr_Attr_ {
   uint32_t name;
   attr_AttrKind kind;
   attr_ValueKind value_kind;
   src_loc_SrcLoc loc;
   attr_Value value;
};

typedef enum {
   attr_AttrReq_NoArg = 0,
   attr_AttrReq_Arg,
   attr_AttrReq_Number,
   attr_AttrReq_String,
   attr_AttrReq_Power2,
   attr_AttrReq_Ok,
   _attr_AttrReq_max = 255
} __attribute__((packed)) attr_AttrReq;

static const char* attr_attrKind_names[19] = {
   "?",
   "export",
   "packed",
   "unused",
   "unused_params",
   "section",
   "noreturn",
   "inline",
   "printf_format",
   "aligned",
   "weak",
   "opaque",
   "cname",
   "no_typedef",
   "constructor",
   "destructor",
   "pure",
   "auto_file",
   "auto_line"
};

static const uint32_t* attr_name_indexes = NULL;

static const attr_AttrReq attr_Required_arg[19] = {
   [attr_AttrKind_Unknown] = attr_AttrReq_NoArg,
   [attr_AttrKind_Export] = attr_AttrReq_NoArg,
   [attr_AttrKind_Packed] = attr_AttrReq_NoArg,
   [attr_AttrKind_Unused] = attr_AttrReq_NoArg,
   [attr_AttrKind_UnusedParams] = attr_AttrReq_NoArg,
   [attr_AttrKind_Section] = attr_AttrReq_String,
   [attr_AttrKind_NoReturn] = attr_AttrReq_NoArg,
   [attr_AttrKind_Inline] = attr_AttrReq_NoArg,
   [attr_AttrKind_PrintfFormat] = attr_AttrReq_NoArg,
   [attr_AttrKind_Aligned] = attr_AttrReq_Number,
   [attr_AttrKind_Weak] = attr_AttrReq_NoArg,
   [attr_AttrKind_Opaque] = attr_AttrReq_NoArg,
   [attr_AttrKind_CName] = attr_AttrReq_String,
   [attr_AttrKind_NoTypeDef] = attr_AttrReq_NoArg,
   [attr_AttrKind_Constructor] = attr_AttrReq_NoArg,
   [attr_AttrKind_Destructor] = attr_AttrReq_NoArg,
   [attr_AttrKind_Pure] = attr_AttrReq_NoArg,
   [attr_AttrKind_AutoFile] = attr_AttrReq_NoArg,
   [attr_AttrKind_AutoLine] = attr_AttrReq_NoArg
};

static const char* attr_kind2name(attr_AttrKind k);
static void attr_register(string_pool_Pool* pool, uint32_t* indexes);
static void attr_init(const uint32_t* indexes);
static attr_AttrKind attr_find(uint32_t name_idx);
static bool attr_isPowerOf2(uint32_t val);
static attr_AttrReq attr_check(const attr_Attr* a);
static const char* attr_kind2name(attr_AttrKind k)
{
   return attr_attrKind_names[k];
}

static void attr_register(string_pool_Pool* pool, uint32_t* indexes)
{
   indexes[0] = 0;
   for (uint32_t i = 1; (i < 19); i++) {
      indexes[i] = string_pool_Pool_addStr(pool, attr_attrKind_names[i], true);
   }
}

static void attr_init(const uint32_t* indexes)
{
   attr_name_indexes = indexes;
}

static attr_AttrKind attr_find(uint32_t name_idx)
{
   for (uint32_t i = 1; (i < 19); i++) {
      if ((name_idx == attr_name_indexes[i])) return ((attr_AttrKind)(i));

   }
   return attr_AttrKind_Unknown;
}

static bool attr_isPowerOf2(uint32_t val)
{
   return (val && !((val & ((val - 1)))));
}

static attr_AttrReq attr_check(const attr_Attr* a)
{
   switch (attr_Required_arg[a->kind]) {
   case attr_AttrReq_NoArg:
      if ((a->value_kind != attr_ValueKind_None)) return attr_AttrReq_NoArg;

      break;
   case attr_AttrReq_Number:
      switch (a->value_kind) {
      case attr_ValueKind_None:
         return attr_AttrReq_Arg;
      case attr_ValueKind_Number:
         if ((a->kind == attr_AttrKind_Aligned)) {
            if (!attr_isPowerOf2(a->value.number)) return attr_AttrReq_Power2;

         }
         break;
      case attr_ValueKind_String:
         return attr_AttrReq_Number;
      }
      break;
   case attr_AttrReq_String:
      switch (a->value_kind) {
      case attr_ValueKind_None:
         return attr_AttrReq_Arg;
      case attr_ValueKind_Number:
         return attr_AttrReq_String;
      case attr_ValueKind_String:
         break;
      }
      break;
   default:
      break;
   }
   return attr_AttrReq_Ok;
}


// --- module attr_table ---
typedef struct attr_table_Attr_ attr_table_Attr;
typedef struct attr_table_Table_ attr_table_Table;

struct attr_table_Attr_ {
   void* decl;
   attr_Value value;
   attr_AttrKind kind;
};

struct attr_table_Table_ {
   uint32_t count;
   uint32_t capacity;
   attr_table_Attr* attrs;
};

static attr_table_Table* attr_table_create(void);
static void attr_table_Table_free(attr_table_Table* t);
static void attr_table_Table_resize(attr_table_Table* t, uint32_t capacity);
static void attr_table_Table_add(attr_table_Table* t, void* decl, attr_AttrKind kind, const attr_Value* value);
static const attr_Value* attr_table_Table_find(const attr_table_Table* t, const void* decl, attr_AttrKind kind);
static attr_table_Table* attr_table_create(void)
{
   attr_table_Table* t = calloc(1, 16);
   attr_table_Table_resize(t, 2);
   return t;
}

static void attr_table_Table_free(attr_table_Table* t)
{
   free(t->attrs);
   free(t);
}

static void attr_table_Table_resize(attr_table_Table* t, uint32_t capacity)
{
   t->capacity = capacity;
   attr_table_Attr* attrs2 = malloc((capacity * 24));
   if (t->count) {
      memcpy(attrs2, t->attrs, (t->count * 24));
      free(t->attrs);
   }
   t->attrs = attrs2;
}

static void attr_table_Table_add(attr_table_Table* t, void* decl, attr_AttrKind kind, const attr_Value* value)
{
   if ((t->count == t->capacity)) attr_table_Table_resize(t, (t->capacity * 2));
   attr_table_Attr* a = &t->attrs[t->count];
   a->decl = decl;
   a->value = *value;
   a->kind = kind;
   t->count++;
}

static const attr_Value* attr_table_Table_find(const attr_table_Table* t, const void* decl, attr_AttrKind kind)
{
   for (uint32_t i = 0; (i < t->count); i++) {
      const attr_table_Attr* a = &t->attrs[i];
      if (((a->decl == decl) && (a->kind == kind))) return &a->value;

   }
   return NULL;
}


// --- module build_file ---
typedef struct build_file_Plugin_ build_file_Plugin;
typedef struct build_file_Info_ build_file_Info;

struct build_file_Plugin_ {
   uint32_t name;
   uint32_t options;
};

struct build_file_Info_ {
   string_pool_Pool* pool;
   const char* filename;
   uint32_t target;
   uint32_t output_dir;
   uint32_t cc;
   uint32_t cflags;
   uint32_t ldflags;
   uint32_t ldflags2;
   uint32_t asmflags;
   uint32_t linkerscript;
   string_list_List lib_dirs;
   string_list_List plugin_dirs;
   build_file_Plugin* plugins;
   uint32_t plugin_count;
   uint32_t plugin_max;
};

static void build_file_Info_addPlugin(build_file_Info* info, const char* name, const char* options);
static const char* build_file_Info_getTarget(const build_file_Info* info);
static const char* build_file_Info_getOutputDir(const build_file_Info* info);
static const char* build_file_Info_getCC(const build_file_Info* info);
static const char* build_file_Info_getCFlags(const build_file_Info* info);
static const char* build_file_Info_getLdFlags(const build_file_Info* info);
static const char* build_file_Info_getLdFlags2(const build_file_Info* info);
static const char* build_file_Info_getAsmFlags(const build_file_Info* info);
static const char* build_file_Info_getLinkerScript(const build_file_Info* info);
static const string_list_List* build_file_Info_getLibDirs(const build_file_Info* info);
static const string_list_List* build_file_Info_getPluginDirs(const build_file_Info* info);
static const build_file_Plugin* build_file_Info_getPlugin(const build_file_Info* info, uint32_t idx);
static uint32_t build_file_Info_getNumPlugins(const build_file_Info* info);
static const yaml_Node* build_file_get_checked(yaml_Parser* parser, const char* path);
static uint32_t build_file_Info_expand(build_file_Info* info, const char* raw);
static bool build_file_getYamlInfo(yaml_Parser* parser, build_file_Info* info);
static bool build_file_Info_parse(build_file_Info* info, const char* data);
static build_file_Info* build_file_parse(source_mgr_SourceMgr* sm, string_pool_Pool* pool, const char* filename);
static void build_file_Info_free(build_file_Info* info);
static void build_file_Info_addPlugin(build_file_Info* info, const char* name, const char* options)
{
   if ((info->plugin_count == info->plugin_max)) {
      info->plugin_max += 2;
      build_file_Plugin* plugins2 = malloc((info->plugin_max * 8));
      if (info->plugins) {
         memcpy(plugins2, info->plugins, (info->plugin_count * 8));
         free(info->plugins);
      }
      info->plugins = plugins2;
   }
   build_file_Plugin* p = &info->plugins[info->plugin_count];
   info->plugin_count++;
   p->name = string_pool_Pool_addStr(info->pool, name, false);
   p->options = string_pool_Pool_addStr(info->pool, options, false);
}

static const char* build_file_Info_getTarget(const build_file_Info* info)
{
   if (info->target) return string_pool_Pool_idx2str(info->pool, info->target);

   return NULL;
}

static const char* build_file_Info_getOutputDir(const build_file_Info* info)
{
   if (info->output_dir) return string_pool_Pool_idx2str(info->pool, info->output_dir);

   return NULL;
}

static const char* build_file_Info_getCC(const build_file_Info* info)
{
   if (info->cc) return string_pool_Pool_idx2str(info->pool, info->cc);

   return NULL;
}

static const char* build_file_Info_getCFlags(const build_file_Info* info)
{
   if (info->cflags) return string_pool_Pool_idx2str(info->pool, info->cflags);

   return NULL;
}

static const char* build_file_Info_getLdFlags(const build_file_Info* info)
{
   if (info->ldflags) return string_pool_Pool_idx2str(info->pool, info->ldflags);

   return NULL;
}

static const char* build_file_Info_getLdFlags2(const build_file_Info* info)
{
   if (info->ldflags2) return string_pool_Pool_idx2str(info->pool, info->ldflags2);

   return NULL;
}

static const char* build_file_Info_getAsmFlags(const build_file_Info* info)
{
   if (info->asmflags) return string_pool_Pool_idx2str(info->pool, info->asmflags);

   return NULL;
}

static const char* build_file_Info_getLinkerScript(const build_file_Info* info)
{
   if (info->linkerscript) return string_pool_Pool_idx2str(info->pool, info->linkerscript);

   return NULL;
}

static const string_list_List* build_file_Info_getLibDirs(const build_file_Info* info)
{
   return &info->lib_dirs;
}

static const string_list_List* build_file_Info_getPluginDirs(const build_file_Info* info)
{
   return &info->plugin_dirs;
}

static const build_file_Plugin* build_file_Info_getPlugin(const build_file_Info* info, uint32_t idx)
{
   return &info->plugins[idx];
}

static uint32_t build_file_Info_getNumPlugins(const build_file_Info* info)
{
   return info->plugin_count;
}

static const yaml_Node* build_file_get_checked(yaml_Parser* parser, const char* path)
{
   const yaml_Node* node = yaml_Parser_findNode(parser, path);
   if (!node) {
      fprintf(stderr, "missing node %s\n", path);
      exit(-1);
   }
   return node;
}

static uint32_t build_file_Info_expand(build_file_Info* info, const char* raw)
{
   if (!raw) return 0;

   if ((raw[0] == '$')) {
      const char* expand = getenv((raw + 1));
      if (!expand) {
         fprintf(stderr, "[build-file] warning: environment variable '%s' not set!\n", (raw + 1));
         return 0;
      }
      raw = expand;
   }
   return string_pool_Pool_addStr(info->pool, raw, false);
}

static bool build_file_getYamlInfo(yaml_Parser* parser, build_file_Info* info)
{
   const char* target = yaml_Parser_getScalarValue(parser, "target");
   info->target = build_file_Info_expand(info, target);
   const char* outputDir = yaml_Parser_getScalarValue(parser, "output_dir");
   info->output_dir = build_file_Info_expand(info, outputDir);
   const char* cc = yaml_Parser_getScalarValue(parser, "toolchain.cc");
   info->cc = build_file_Info_expand(info, cc);
   const char* cflags = yaml_Parser_getScalarValue(parser, "toolchain.cflags");
   info->cflags = build_file_Info_expand(info, cflags);
   const char* ldflags = yaml_Parser_getScalarValue(parser, "toolchain.ldflags");
   info->ldflags = build_file_Info_expand(info, ldflags);
   const char* ldflags2 = yaml_Parser_getScalarValue(parser, "toolchain.ldflags2");
   info->ldflags2 = build_file_Info_expand(info, ldflags2);
   const char* asmflags = yaml_Parser_getScalarValue(parser, "toolchain.asmflags");
   info->asmflags = build_file_Info_expand(info, asmflags);
   const char* linkerscript = yaml_Parser_getScalarValue(parser, "toolchain.linkerscript");
   info->linkerscript = build_file_Info_expand(info, linkerscript);
   const yaml_Node* dirs = yaml_Parser_findNode(parser, "libdir");
   yaml_Iter iter = yaml_Parser_getNodeChildIter(parser, dirs);
   while (!yaml_Iter_done(&iter)) {
      const char* dir = yaml_Iter_getValue(&iter);
      string_list_List_add(&info->lib_dirs, build_file_Info_expand(info, dir));
      yaml_Iter_next(&iter);
   }
   dirs = yaml_Parser_findNode(parser, "plugindir");
   iter = yaml_Parser_getNodeChildIter(parser, dirs);
   while (!yaml_Iter_done(&iter)) {
      const char* dir = yaml_Iter_getValue(&iter);
      string_list_List_add(&info->plugin_dirs, build_file_Info_expand(info, dir));
      yaml_Iter_next(&iter);
   }
   const yaml_Node* root = yaml_Parser_getRoot(parser);
   iter = yaml_Parser_getNodeChildIter(parser, root);
   while (!yaml_Iter_done(&iter)) {
      const char* name = yaml_Iter_getName(&iter);
      if ((strncmp(name, "plugin.,", 7) == 0)) {
         const char* options = yaml_Iter_getChildScalarValue(&iter, "options");
         if (!options) {
            fprintf(stderr, "[build-file] missing options for %s\n", name);
            exit(-1);
         }
         build_file_Info_addPlugin(info, (name + 7), options);
      }
      yaml_Iter_next(&iter);
   }
   return true;
}

static bool build_file_Info_parse(build_file_Info* info, const char* data)
{
   yaml_Parser* parser = yaml_Parser_create();
   bool ok = yaml_Parser_parse(parser, ((char*)(data)));
   if (ok) {
      ok = build_file_getYamlInfo(parser, info);
   } else {
      fprintf(stderr, "Error: %s\n", yaml_Parser_getMessage(parser));
   }
   yaml_Parser_destroy(parser);
   return ok;
}

static build_file_Info* build_file_parse(source_mgr_SourceMgr* sm, string_pool_Pool* pool, const char* filename)
{
   build_file_Info info = { };
   info.pool = pool;
   info.filename = filename;
   string_list_List_init(&info.lib_dirs, pool);
   string_list_List_init(&info.plugin_dirs, pool);
   uint32_t filename_idx = string_pool_Pool_addStr(pool, filename, false);
   int32_t file_id = source_mgr_SourceMgr_open(sm, filename_idx, 0, false);
   if ((file_id == -1)) return NULL;

   bool ok = build_file_Info_parse(&info, source_mgr_SourceMgr_get_content(sm, file_id));
   source_mgr_SourceMgr_close(sm, file_id);
   if (!ok) return NULL;

   build_file_Info* result = malloc(112);
   memcpy(result, &info, 112);
   return result;
}

static void build_file_Info_free(build_file_Info* info)
{
   string_list_List_free(&info->lib_dirs);
   string_list_List_free(&info->plugin_dirs);
   free(info);
}


// --- module build_target ---
typedef struct build_target_Plugin_ build_target_Plugin;
typedef struct build_target_PluginList_ build_target_PluginList;
typedef struct build_target_File_ build_target_File;
typedef struct build_target_Target_ build_target_Target;

typedef enum {
   build_target_Kind_Image,
   build_target_Kind_Executable,
   build_target_Kind_StaticLibrary,
   build_target_Kind_DynamicLibrary,
   _build_target_Kind_max = 255
} __attribute__((packed)) build_target_Kind;

struct build_target_Plugin_ {
   uint32_t name;
   uint32_t options;
   src_loc_SrcLoc loc;
};

struct build_target_PluginList_ {
   build_target_Plugin* plugins;
   uint32_t count;
   uint32_t capacity;
};

struct build_target_File_ {
   uint32_t name;
   src_loc_SrcLoc loc;
};

struct build_target_Target_ {
   uint32_t name_idx;
   src_loc_SrcLoc loc;
   warning_flags_Flags warnings;
   build_target_Kind kind;
   bool disable_asserts;
   bool no_libc;
   bool backend_c;
   bool cgen_no_build;
   bool cgen_fast;
   bool backend_llvm;
   bool backend_qbe;
   string_list_List features;
   library_list_List libs;
   string_list_List exports;
   build_target_PluginList plugins;
   build_target_File* files;
   uint32_t num_files;
   uint32_t max_files;
   build_target_File* asm_files;
   uint32_t asm_file_count;
   uint32_t asm_file_max;
};

static void build_target_PluginList_init(build_target_PluginList* l);
static uint32_t build_target_PluginList_size(const build_target_PluginList* l);
static const build_target_Plugin* build_target_PluginList_get(const build_target_PluginList* l, uint32_t idx);
static void build_target_PluginList_add(build_target_PluginList* l, uint32_t name, uint32_t options, src_loc_SrcLoc loc);
static build_target_Target* build_target_create(uint32_t name_idx, src_loc_SrcLoc loc, build_target_Kind kind, string_pool_Pool* pool);
static void build_target_Target_free(build_target_Target* t);
static bool build_target_Target_hasBackend(const build_target_Target* t);
static uint32_t build_target_Target_getNameIdx(const build_target_Target* t);
static uint32_t build_target_Target_numFiles(const build_target_Target* t);
static uint32_t build_target_Target_numAsmFiles(const build_target_Target* t);
static const string_list_List* build_target_Target_getFeatures(const build_target_Target* t);
static void build_target_Target_addFeature(build_target_Target* t, uint32_t feature);
static void build_target_Target_addPlugin(build_target_Target* t, uint32_t name, uint32_t options, src_loc_SrcLoc loc);
static const build_target_PluginList* build_target_Target_getPlugins(const build_target_Target* t);
static void build_target_Target_disableAsserts(build_target_Target* t);
static bool build_target_Target_hasAsserts(const build_target_Target* t);
static void build_target_Target_visitLibs(const build_target_Target* t, library_list_Visitor visitor, void* arg);
static bool build_target_Target_hasLib(const build_target_Target* t, uint32_t lib);
static void build_target_Target_addLib(build_target_Target* t, uint32_t lib, bool is_static);
static const warning_flags_Flags* build_target_Target_getWarnings(const build_target_Target* t);
static warning_flags_Flags* build_target_Target_getWarnings2(build_target_Target* t);
static void build_target_Target_addExport(build_target_Target* t, uint32_t export);
static const string_list_List* build_target_Target_getExports(const build_target_Target* t);
static build_target_Kind build_target_Target_getKind(const build_target_Target* t);
static bool build_target_Target_needsMain(const build_target_Target* t);
static bool build_target_Target_addFile(build_target_Target* t, uint32_t filename, src_loc_SrcLoc loc);
static const build_target_File* build_target_Target_getFile(const build_target_Target* t, uint32_t idx);
static bool build_target_Target_addAsmFile(build_target_Target* t, uint32_t filename, src_loc_SrcLoc loc);
static const build_target_File* build_target_Target_getAsmFile(const build_target_Target* t, uint32_t idx);
static void build_target_Target_setNoLibC(build_target_Target* t);
static bool build_target_Target_getNoLibC(const build_target_Target* t);
static void build_target_Target_setBackendC(build_target_Target* t);
static bool build_target_Target_getBackendC(const build_target_Target* t);
static void build_target_Target_setBackendLLVM(build_target_Target* t);
static bool build_target_Target_getBackendLLVM(const build_target_Target* t);
static void build_target_Target_setBackendQBE(build_target_Target* t);
static bool build_target_Target_getBackendQBE(const build_target_Target* t);
static void build_target_Target_setCGenNoBuild(build_target_Target* t);
static bool build_target_Target_getCGenNoBuild(const build_target_Target* t);
static void build_target_Target_setCGenFastBuild(build_target_Target* t);
static bool build_target_Target_getCGenFastBuild(const build_target_Target* t);
static void build_target_PluginList_init(build_target_PluginList* l)
{
   memset(l, 0, 16);
}

static uint32_t build_target_PluginList_size(const build_target_PluginList* l)
{
   return l->count;
}

static const build_target_Plugin* build_target_PluginList_get(const build_target_PluginList* l, uint32_t idx)
{
   return &l->plugins[idx];
}

static void build_target_PluginList_add(build_target_PluginList* l, uint32_t name, uint32_t options, src_loc_SrcLoc loc)
{
   if ((l->count == l->capacity)) {
      l->capacity = l->capacity ? (l->capacity * 2) : 4;
      build_target_Plugin* plugins2 = malloc((l->capacity * 12));
      if (l->count) {
         memcpy(plugins2, l->plugins, (l->count * 12));
         free(l->plugins);
      }
      l->plugins = plugins2;
   }
   build_target_Plugin* p = &l->plugins[l->count];
   p->name = name;
   p->options = options;
   p->loc = loc;
   l->count++;
}

static build_target_Target* build_target_create(uint32_t name_idx, src_loc_SrcLoc loc, build_target_Kind kind, string_pool_Pool* pool)
{
   build_target_Target* t = calloc(1, 144);
   t->name_idx = name_idx;
   t->loc = loc;
   t->kind = kind;
   t->max_files = 8;
   string_list_List_init(&t->features, pool);
   library_list_List_init(&t->libs);
   string_list_List_init(&t->exports, pool);
   t->files = malloc((t->max_files * 8));
   t->asm_file_max = 4;
   t->asm_files = malloc((t->asm_file_max * 8));
   return t;
}

static void build_target_Target_free(build_target_Target* t)
{
   string_list_List_free(&t->exports);
   library_list_List_free(&t->libs);
   string_list_List_free(&t->features);
   free(t->files);
   free(t->asm_files);
   free(t);
}

static bool build_target_Target_hasBackend(const build_target_Target* t)
{
   return ((t->backend_c || t->backend_llvm) || t->backend_qbe);
}

static uint32_t build_target_Target_getNameIdx(const build_target_Target* t)
{
   return t->name_idx;
}

static uint32_t build_target_Target_numFiles(const build_target_Target* t)
{
   return t->num_files;
}

static uint32_t build_target_Target_numAsmFiles(const build_target_Target* t)
{
   return t->asm_file_count;
}

static const string_list_List* build_target_Target_getFeatures(const build_target_Target* t)
{
   return &t->features;
}

static void build_target_Target_addFeature(build_target_Target* t, uint32_t feature)
{
   string_list_List_add(&t->features, feature);
}

static void build_target_Target_addPlugin(build_target_Target* t, uint32_t name, uint32_t options, src_loc_SrcLoc loc)
{
   build_target_PluginList_add(&t->plugins, name, options, loc);
}

static const build_target_PluginList* build_target_Target_getPlugins(const build_target_Target* t)
{
   return &t->plugins;
}

static void build_target_Target_disableAsserts(build_target_Target* t)
{
   t->disable_asserts = true;
}

static bool build_target_Target_hasAsserts(const build_target_Target* t)
{
   return !t->disable_asserts;
}

static void build_target_Target_visitLibs(const build_target_Target* t, library_list_Visitor visitor, void* arg)
{
   library_list_List_visit(&t->libs, visitor, arg);
}

static bool build_target_Target_hasLib(const build_target_Target* t, uint32_t lib)
{
   return library_list_List_contains(&t->libs, lib);
}

static void build_target_Target_addLib(build_target_Target* t, uint32_t lib, bool is_static)
{
   library_list_List_add(&t->libs, lib, is_static);
}

static const warning_flags_Flags* build_target_Target_getWarnings(const build_target_Target* t)
{
   return &t->warnings;
}

static warning_flags_Flags* build_target_Target_getWarnings2(build_target_Target* t)
{
   return &t->warnings;
}

static void build_target_Target_addExport(build_target_Target* t, uint32_t export)
{
   string_list_List_add(&t->exports, export);
}

static const string_list_List* build_target_Target_getExports(const build_target_Target* t)
{
   return &t->exports;
}

static build_target_Kind build_target_Target_getKind(const build_target_Target* t)
{
   return t->kind;
}

static bool build_target_Target_needsMain(const build_target_Target* t)
{
   switch (t->kind) {
   case build_target_Kind_Image:
      __attribute__((fallthrough));
   case build_target_Kind_Executable:
      return true;
   case build_target_Kind_StaticLibrary:
      __attribute__((fallthrough));
   case build_target_Kind_DynamicLibrary:
      break;
   }
   return false;
}

static bool build_target_Target_addFile(build_target_Target* t, uint32_t filename, src_loc_SrcLoc loc)
{
   for (uint32_t i = 0; (i < t->num_files); i++) {
      if ((t->files[i].name == filename)) return false;

   }
   if ((t->num_files == t->max_files)) {
      t->max_files *= 2;
      build_target_File* files2 = malloc((t->max_files * 8));
      memcpy(files2, t->files, (t->num_files * 8));
      free(t->files);
      t->files = files2;
   }
   t->files[t->num_files].name = filename;
   t->files[t->num_files].loc = loc;
   t->num_files++;
   return true;
}

static const build_target_File* build_target_Target_getFile(const build_target_Target* t, uint32_t idx)
{
   return &t->files[idx];
}

static bool build_target_Target_addAsmFile(build_target_Target* t, uint32_t filename, src_loc_SrcLoc loc)
{
   for (uint32_t i = 0; (i < t->asm_file_count); i++) {
      if ((t->asm_files[i].name == filename)) return false;

   }
   if ((t->asm_file_count == t->asm_file_max)) {
      t->asm_file_max *= 2;
      build_target_File* files2 = malloc((t->asm_file_max * 8));
      memcpy(files2, t->files, (t->asm_file_count * 8));
      free(t->asm_files);
      t->asm_files = files2;
   }
   t->asm_files[t->asm_file_count].name = filename;
   t->asm_files[t->asm_file_count].loc = loc;
   t->asm_file_count++;
   return true;
}

static const build_target_File* build_target_Target_getAsmFile(const build_target_Target* t, uint32_t idx)
{
   return &t->asm_files[idx];
}

static void build_target_Target_setNoLibC(build_target_Target* t)
{
   t->no_libc = true;
}

static bool build_target_Target_getNoLibC(const build_target_Target* t)
{
   return t->no_libc;
}

static void build_target_Target_setBackendC(build_target_Target* t)
{
   t->backend_c = true;
}

static bool build_target_Target_getBackendC(const build_target_Target* t)
{
   return t->backend_c;
}

static void build_target_Target_setBackendLLVM(build_target_Target* t)
{
   t->backend_llvm = true;
}

static bool build_target_Target_getBackendLLVM(const build_target_Target* t)
{
   return t->backend_llvm;
}

static void build_target_Target_setBackendQBE(build_target_Target* t)
{
   t->backend_qbe = true;
}

static bool build_target_Target_getBackendQBE(const build_target_Target* t)
{
   return t->backend_qbe;
}

static void build_target_Target_setCGenNoBuild(build_target_Target* t)
{
   t->cgen_no_build = true;
}

static bool build_target_Target_getCGenNoBuild(const build_target_Target* t)
{
   return t->cgen_no_build;
}

static void build_target_Target_setCGenFastBuild(build_target_Target* t)
{
   t->cgen_fast = true;
}

static bool build_target_Target_getCGenFastBuild(const build_target_Target* t)
{
   return t->cgen_fast;
}


// --- module diagnostics ---
typedef struct diagnostics_Diags_ diagnostics_Diags;

typedef src_loc_SrcLoc (*diagnostics_GetTokenEndFn)(const char* input, src_loc_SrcLoc start);

struct diagnostics_Diags_ {
   source_mgr_SourceMgr* sm;
   string_buffer_Buf* out;
   uint32_t num_errors;
   uint32_t num_warnings;
   bool promote_warnings;
   diagnostics_GetTokenEndFn get_token_end;
};

typedef enum {
   diagnostics_Category_Note,
   diagnostics_Category_Warning,
   diagnostics_Category_Error,
   _diagnostics_Category_max = 255
} __attribute__((packed)) diagnostics_Category;

static const char* diagnostics_category_names[3] = { "note", "warning", "error" };

static const char* diagnostics_category_colors[3] = { color_Grey, color_Magenta, color_Bred };

static diagnostics_Diags* diagnostics_create(source_mgr_SourceMgr* sm, bool use_color, diagnostics_GetTokenEndFn get_token_end);
static void diagnostics_Diags_free(diagnostics_Diags* diags);
static void diagnostics_Diags_clear(diagnostics_Diags* diags);
static void diagnostics_Diags_setWarningAsError(diagnostics_Diags* diags, bool are_errors);
__attribute__((__format__(printf, 3, 4))) 
static void diagnostics_Diags_error(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, ...);
static void diagnostics_Diags_error2(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, va_list args);
__attribute__((__format__(printf, 3, 4))) 
static void diagnostics_Diags_note(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, ...);
static void diagnostics_Diags_note2(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, va_list args);
__attribute__((__format__(printf, 3, 4))) 
static void diagnostics_Diags_warn(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, ...);
static void diagnostics_Diags_warn2(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, va_list args);
__attribute__((__format__(printf, 4, 5))) 
static void diagnostics_Diags_errorRange(diagnostics_Diags* diags, src_loc_SrcLoc loc, src_loc_SrcRange range, const char* format, ...);
static void diagnostics_Diags_errorRange2(diagnostics_Diags* diags, src_loc_SrcLoc loc, src_loc_SrcRange range, const char* format, va_list args);
static void diagnostics_Diags_internal(diagnostics_Diags* diags, diagnostics_Category category, src_loc_SrcLoc sloc, src_loc_SrcRange range, const char* format, va_list args);
static bool diagnostics_Diags_isOk(const diagnostics_Diags* diags);
static bool diagnostics_Diags_hasErrors(const diagnostics_Diags* diags);
static uint32_t diagnostics_Diags_getNumErrors(const diagnostics_Diags* diags);
static uint32_t diagnostics_Diags_getNumWarnings(const diagnostics_Diags* diags);
static void diagnostics_Diags_printStatus(const diagnostics_Diags* diags);
static diagnostics_Diags* diagnostics_create(source_mgr_SourceMgr* sm, bool use_color, diagnostics_GetTokenEndFn get_token_end)
{
   diagnostics_Diags* diags = calloc(1, 40);
   diags->sm = sm;
   diags->out = string_buffer_create(512, use_color, 1);
   diags->get_token_end = get_token_end;
   return diags;
}

static void diagnostics_Diags_free(diagnostics_Diags* diags)
{
   string_buffer_Buf_free(diags->out);
   free(diags);
}

static void diagnostics_Diags_clear(diagnostics_Diags* diags)
{
   diags->num_errors = 0;
   diags->num_warnings = 0;
}

static void diagnostics_Diags_setWarningAsError(diagnostics_Diags* diags, bool are_errors)
{
   diags->promote_warnings = are_errors;
}

__attribute__((__format__(printf, 3, 4))) 
static void diagnostics_Diags_error(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, ...)
{
   diagnostics_Category category = diagnostics_Category_Error;
   va_list args;
   va_start(args, format);
   src_loc_SrcRange range = { 0, 0 };
   diagnostics_Diags_internal(diags, category, loc, range, format, args);
   va_end(args);
}

static void diagnostics_Diags_error2(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, va_list args)
{
   src_loc_SrcRange range = { 0, 0 };
   diagnostics_Diags_internal(diags, diagnostics_Category_Error, loc, range, format, args);
}

__attribute__((__format__(printf, 3, 4))) 
static void diagnostics_Diags_note(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, ...)
{
   va_list args;
   va_start(args, format);
   src_loc_SrcRange range = { 0, 0 };
   diagnostics_Diags_internal(diags, diagnostics_Category_Note, loc, range, format, args);
   va_end(args);
}

static void diagnostics_Diags_note2(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, va_list args)
{
   src_loc_SrcRange range = { 0, 0 };
   diagnostics_Diags_internal(diags, diagnostics_Category_Note, loc, range, format, args);
}

__attribute__((__format__(printf, 3, 4))) 
static void diagnostics_Diags_warn(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, ...)
{
   va_list args;
   va_start(args, format);
   src_loc_SrcRange range = { 0, 0 };
   diagnostics_Category category = diagnostics_Category_Warning;
   if (diags->promote_warnings) category = diagnostics_Category_Error;
   diagnostics_Diags_internal(diags, category, loc, range, format, args);
   va_end(args);
}

static void diagnostics_Diags_warn2(diagnostics_Diags* diags, src_loc_SrcLoc loc, const char* format, va_list args)
{
   src_loc_SrcRange range = { 0, 0 };
   diagnostics_Category category = diagnostics_Category_Warning;
   if (diags->promote_warnings) category = diagnostics_Category_Error;
   diagnostics_Diags_internal(diags, category, loc, range, format, args);
}

__attribute__((__format__(printf, 4, 5))) 
static void diagnostics_Diags_errorRange(diagnostics_Diags* diags, src_loc_SrcLoc loc, src_loc_SrcRange range, const char* format, ...)
{
   diagnostics_Category category = diagnostics_Category_Error;
   va_list args;
   va_start(args, format);
   diagnostics_Diags_internal(diags, category, loc, range, format, args);
   va_end(args);
}

static void diagnostics_Diags_errorRange2(diagnostics_Diags* diags, src_loc_SrcLoc loc, src_loc_SrcRange range, const char* format, va_list args)
{
   diagnostics_Diags_internal(diags, diagnostics_Category_Error, loc, range, format, args);
}

static void diagnostics_Diags_internal(diagnostics_Diags* diags, diagnostics_Category category, src_loc_SrcLoc sloc, src_loc_SrcRange range, const char* format, va_list args)
{
   if ((category == diagnostics_Category_Error)) {
      diags->num_errors++;
   } else {
      diags->num_warnings++;
   }
   string_buffer_Buf* out = diags->out;
   string_buffer_Buf_clear(out);
   source_mgr_Location startLoc = source_mgr_SourceMgr_getLocation(diags->sm, range.start);
   source_mgr_Location loc = source_mgr_SourceMgr_getLocation(diags->sm, sloc);
   if (range.end) {
      const char* src = source_mgr_SourceMgr_get_token_source(diags->sm, range.end);
      range.end = diags->get_token_end(src, range.end);
   }
   source_mgr_Location endLoc = source_mgr_SourceMgr_getLocation(diags->sm, range.end);
   if (sloc) {
      string_buffer_Buf_print(out, "%s:%u:%u: ", loc.filename, loc.line, loc.column);
   }
   string_buffer_Buf_color(out, diagnostics_category_colors[category]);
   string_buffer_Buf_add(out, diagnostics_category_names[category]);
   string_buffer_Buf_add(out, ": ");
   string_buffer_Buf_color(out, color_Normal);
   char tmp[256];
   vsprintf(tmp, format, args);
   string_buffer_Buf_add(out, tmp);
   string_buffer_Buf_add(out, "\n");
   if (sloc) {
      c2_assert((loc.line_start) != NULL, "common/diagnostics.c2:172: diagnostics.Diags.internal", "loc.line_start");
      string_buffer_Buf_add_line(out, loc.line_start);
      string_buffer_Buf_add(out, "\n");
      if ((range.start && range.end)) {
         c2_assert(((endLoc.column >= startLoc.column)) != 0, "common/diagnostics.c2:178: diagnostics.Diags.internal", "endLoc.column>=startLoc.column");
         uint32_t offset = (startLoc.column - 1);
         uint32_t tabs = string_utils_count_tabs(startLoc.line_start, offset);
         c2_assert(((offset >= tabs)) != 0, "common/diagnostics.c2:182: diagnostics.Diags.internal", "offset>=tabs");
         offset -= tabs;
         for (uint32_t i = 0; (i < tabs); i++) string_buffer_Buf_add1(out, '\t');
         string_buffer_Buf_indent(out, offset);
         string_buffer_Buf_color(out, color_Bgreen);
         for (uint32_t i = startLoc.column; (i <= endLoc.column); i++) {
            if ((i == loc.column)) string_buffer_Buf_add(out, "^");
            else string_buffer_Buf_add(out, "~");
         }
         string_buffer_Buf_color(out, color_Normal);
      } else {
         uint32_t offset = (loc.column - 1);
         uint32_t tabs = string_utils_count_tabs(loc.line_start, offset);
         c2_assert(((offset >= tabs)) != 0, "common/diagnostics.c2:196: diagnostics.Diags.internal", "offset>=tabs");
         offset -= tabs;
         for (uint32_t i = 0; (i < tabs); i++) string_buffer_Buf_add1(out, '\t');
         string_buffer_Buf_indent(out, offset);
         string_buffer_Buf_color(out, color_Bgreen);
         string_buffer_Buf_add(out, "^");
         string_buffer_Buf_color(out, color_Normal);
      }
   }
   fprintf(stderr, "%s\n", string_buffer_Buf_data(out));
}

static bool diagnostics_Diags_isOk(const diagnostics_Diags* diags)
{
   return (diags->num_errors == 0);
}

static bool diagnostics_Diags_hasErrors(const diagnostics_Diags* diags)
{
   return (diags->num_errors != 0);
}

static uint32_t diagnostics_Diags_getNumErrors(const diagnostics_Diags* diags)
{
   return diags->num_errors;
}

static uint32_t diagnostics_Diags_getNumWarnings(const diagnostics_Diags* diags)
{
   return diags->num_warnings;
}

static void diagnostics_Diags_printStatus(const diagnostics_Diags* diags)
{
   string_buffer_Buf* out = diags->out;
   string_buffer_Buf_clear(out);
   if (diags->num_warnings) {
      string_buffer_Buf_print(out, "%u warning%s", diags->num_warnings, (diags->num_warnings > 1) ? "s" : "");
   }
   if (diags->num_errors) {
      if (diags->num_warnings) string_buffer_Buf_add(out, " and ");
      string_buffer_Buf_print(out, "%u error%s", diags->num_errors, (diags->num_errors > 1) ? "s" : "");
   }
   if (string_buffer_Buf_size(out)) {
      string_buffer_Buf_add(out, " generated.\n");
      fputs(string_buffer_Buf_data(out), stderr);
   }
}


// --- module c2_tokenizer ---
typedef struct c2_tokenizer_Keyword_ c2_tokenizer_Keyword;
typedef struct c2_tokenizer_Feature_ c2_tokenizer_Feature;
typedef struct c2_tokenizer_Tokenizer_ c2_tokenizer_Tokenizer;

struct c2_tokenizer_Keyword_ {
   const char* name;
   token_Kind kind;
   uint8_t len;
};

typedef enum {
   c2_tokenizer_Action_TABSPACE,
   c2_tokenizer_Action_IDENT_OR_KEYWORD,
   c2_tokenizer_Action_IDENT,
   c2_tokenizer_Action_DIGIT,
   c2_tokenizer_Action_LPAREN,
   c2_tokenizer_Action_RPAREN,
   c2_tokenizer_Action_LSQUARE,
   c2_tokenizer_Action_RSQUARE,
   c2_tokenizer_Action_NEWLINE,
   c2_tokenizer_Action_EXCLAIM,
   c2_tokenizer_Action_DQUOTE,
   c2_tokenizer_Action_SQUOTE,
   c2_tokenizer_Action_POUND,
   c2_tokenizer_Action_STAR,
   c2_tokenizer_Action_PLUS,
   c2_tokenizer_Action_MINUS,
   c2_tokenizer_Action_COMMA,
   c2_tokenizer_Action_DOT,
   c2_tokenizer_Action_PERCENT,
   c2_tokenizer_Action_SLASH,
   c2_tokenizer_Action_COLON,
   c2_tokenizer_Action_SEMI_COLON,
   c2_tokenizer_Action_LESS,
   c2_tokenizer_Action_EQUAL,
   c2_tokenizer_Action_GREATER,
   c2_tokenizer_Action_QUESTION,
   c2_tokenizer_Action_AT,
   c2_tokenizer_Action_AMP,
   c2_tokenizer_Action_CARET,
   c2_tokenizer_Action_LBRACE,
   c2_tokenizer_Action_RBRACE,
   c2_tokenizer_Action_PIPE,
   c2_tokenizer_Action_TILDE,
   c2_tokenizer_Action_CR,
   c2_tokenizer_Action_EOF,
   c2_tokenizer_Action_INVALID,
   _c2_tokenizer_Action_max = 255
} __attribute__((packed)) c2_tokenizer_Action;

struct c2_tokenizer_Feature_ {
   bool is_if;
   bool enabled;
};

#define c2_tokenizer_MaxLookahead 16
struct c2_tokenizer_Tokenizer_ {
   const char* cur;
   src_loc_SrcLoc loc_start;
   const char* input_start;
   token_Token next[16];
   uint32_t next_count;
   uint32_t next_head;
   const char* line_start;
   string_pool_Pool* pool;
   string_buffer_Buf* buf;
   c2_tokenizer_Feature feature_stack[6];
   uint32_t feature_count;
   const string_list_List* features;
   bool raw_mode;
   char error_msg[256];
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_a[4] = {
   { "as", token_Kind_KW_as, 2 },
   { "asm", token_Kind_KW_asm, 3 },
   { "assert", token_Kind_KW_assert, 6 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_b[3] = {
   { "bool", token_Kind_KW_bool, 4 },
   { "break", token_Kind_KW_break, 5 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_c[6] = {
   { "case", token_Kind_KW_case, 4 },
   { "cast", token_Kind_KW_cast, 4 },
   { "char", token_Kind_KW_char, 4 },
   { "const", token_Kind_KW_const, 5 },
   { "continue", token_Kind_KW_continue, 8 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_d[3] = {
   { "default", token_Kind_KW_default, 7 },
   { "do", token_Kind_KW_do, 2 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_e[6] = {
   { "elemsof", token_Kind_KW_elemsof, 7 },
   { "else", token_Kind_KW_else, 4 },
   { "enum", token_Kind_KW_enum, 4 },
   { "enum_max", token_Kind_KW_enum_max, 8 },
   { "enum_min", token_Kind_KW_enum_min, 8 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_f[7] = {
   { "f32", token_Kind_KW_f32, 3 },
   { "f64", token_Kind_KW_f64, 3 },
   { "fallthrough", token_Kind_KW_fallthrough, 11 },
   { "false", token_Kind_KW_false, 5 },
   { "fn", token_Kind_KW_fn, 2 },
   { "for", token_Kind_KW_for, 3 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_g[2] = {
   { "goto", token_Kind_KW_goto, 4 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_i[8] = {
   { "i16", token_Kind_KW_i16, 3 },
   { "i32", token_Kind_KW_i32, 3 },
   { "i64", token_Kind_KW_i64, 3 },
   { "i8", token_Kind_KW_i8, 2 },
   { "if", token_Kind_KW_if, 2 },
   { "import", token_Kind_KW_import, 6 },
   { "isize", token_Kind_KW_isize, 5 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_l[2] = {
   { "local", token_Kind_KW_local, 5 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_m[2] = {
   { "module", token_Kind_KW_module, 6 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_n[2] = {
   { "nil", token_Kind_KW_nil, 3 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_o[2] = {
   { "offsetof", token_Kind_KW_offsetof, 8 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_p[2] = {
   { "public", token_Kind_KW_public, 6 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_r[6] = {
   { "reg16", token_Kind_KW_reg16, 5 },
   { "reg32", token_Kind_KW_reg32, 5 },
   { "reg64", token_Kind_KW_reg64, 5 },
   { "reg8", token_Kind_KW_reg8, 4 },
   { "return", token_Kind_KW_return, 6 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_s[6] = {
   { "sizeof", token_Kind_KW_sizeof, 6 },
   { "sswitch", token_Kind_KW_sswitch, 7 },
   { "static_assert", token_Kind_KW_static_assert, 13 },
   { "struct", token_Kind_KW_struct, 6 },
   { "switch", token_Kind_KW_switch, 6 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_t[5] = {
   { "template", token_Kind_KW_template, 8 },
   { "to_container", token_Kind_KW_to_container, 12 },
   { "true", token_Kind_KW_true, 4 },
   { "type", token_Kind_KW_type, 4 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_u[7] = {
   { "u16", token_Kind_KW_u16, 3 },
   { "u32", token_Kind_KW_u32, 3 },
   { "u64", token_Kind_KW_u64, 3 },
   { "u8", token_Kind_KW_u8, 2 },
   { "union", token_Kind_KW_union, 5 },
   { "usize", token_Kind_KW_usize, 5 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_v[3] = {
   { "void", token_Kind_KW_void, 4 },
   { "volatile", token_Kind_KW_volatile, 8 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword c2_tokenizer_Keywords_w[2] = {
   { "while", token_Kind_KW_while, 5 },
   { NULL, token_Kind_None, 0 }
};

static const c2_tokenizer_Keyword* c2_tokenizer_keywords[26] = {
   c2_tokenizer_Keywords_a,
   c2_tokenizer_Keywords_b,
   c2_tokenizer_Keywords_c,
   c2_tokenizer_Keywords_d,
   c2_tokenizer_Keywords_e,
   c2_tokenizer_Keywords_f,
   c2_tokenizer_Keywords_g,
   NULL,
   c2_tokenizer_Keywords_i,
   NULL,
   NULL,
   c2_tokenizer_Keywords_l,
   c2_tokenizer_Keywords_m,
   c2_tokenizer_Keywords_n,
   c2_tokenizer_Keywords_o,
   c2_tokenizer_Keywords_p,
   NULL,
   c2_tokenizer_Keywords_r,
   c2_tokenizer_Keywords_s,
   c2_tokenizer_Keywords_t,
   c2_tokenizer_Keywords_u,
   c2_tokenizer_Keywords_v,
   c2_tokenizer_Keywords_w,
   NULL,
   NULL,
   NULL
};

static const c2_tokenizer_Action c2_tokenizer_Char_lookup[128] = {
   [0] = c2_tokenizer_Action_EOF,
   ['\t'] = c2_tokenizer_Action_TABSPACE,
   ['\n'] = c2_tokenizer_Action_NEWLINE,
   ['\r'] = c2_tokenizer_Action_CR,
   [' '] = c2_tokenizer_Action_TABSPACE,
   ['!'] = c2_tokenizer_Action_EXCLAIM,
   ['"'] = c2_tokenizer_Action_DQUOTE,
   ['#'] = c2_tokenizer_Action_POUND,
   ['%'] = c2_tokenizer_Action_PERCENT,
   ['&'] = c2_tokenizer_Action_AMP,
   ['\''] = c2_tokenizer_Action_SQUOTE,
   ['('] = c2_tokenizer_Action_LPAREN,
   [')'] = c2_tokenizer_Action_RPAREN,
   ['*'] = c2_tokenizer_Action_STAR,
   ['+'] = c2_tokenizer_Action_PLUS,
   [','] = c2_tokenizer_Action_COMMA,
   ['-'] = c2_tokenizer_Action_MINUS,
   ['.'] = c2_tokenizer_Action_DOT,
   ['/'] = c2_tokenizer_Action_SLASH,
   ['0'] = c2_tokenizer_Action_DIGIT,
   ['1'] = c2_tokenizer_Action_DIGIT,
   ['2'] = c2_tokenizer_Action_DIGIT,
   ['3'] = c2_tokenizer_Action_DIGIT,
   ['4'] = c2_tokenizer_Action_DIGIT,
   ['5'] = c2_tokenizer_Action_DIGIT,
   ['6'] = c2_tokenizer_Action_DIGIT,
   ['7'] = c2_tokenizer_Action_DIGIT,
   ['8'] = c2_tokenizer_Action_DIGIT,
   ['9'] = c2_tokenizer_Action_DIGIT,
   [':'] = c2_tokenizer_Action_COLON,
   [';'] = c2_tokenizer_Action_SEMI_COLON,
   ['<'] = c2_tokenizer_Action_LESS,
   ['='] = c2_tokenizer_Action_EQUAL,
   ['>'] = c2_tokenizer_Action_GREATER,
   ['?'] = c2_tokenizer_Action_QUESTION,
   ['@'] = c2_tokenizer_Action_AT,
   ['A'] = c2_tokenizer_Action_IDENT,
   ['B'] = c2_tokenizer_Action_IDENT,
   ['C'] = c2_tokenizer_Action_IDENT,
   ['D'] = c2_tokenizer_Action_IDENT,
   ['E'] = c2_tokenizer_Action_IDENT,
   ['F'] = c2_tokenizer_Action_IDENT,
   ['G'] = c2_tokenizer_Action_IDENT,
   ['H'] = c2_tokenizer_Action_IDENT,
   ['I'] = c2_tokenizer_Action_IDENT,
   ['J'] = c2_tokenizer_Action_IDENT,
   ['K'] = c2_tokenizer_Action_IDENT,
   ['L'] = c2_tokenizer_Action_IDENT,
   ['M'] = c2_tokenizer_Action_IDENT,
   ['N'] = c2_tokenizer_Action_IDENT,
   ['O'] = c2_tokenizer_Action_IDENT,
   ['P'] = c2_tokenizer_Action_IDENT,
   ['Q'] = c2_tokenizer_Action_IDENT,
   ['R'] = c2_tokenizer_Action_IDENT,
   ['S'] = c2_tokenizer_Action_IDENT,
   ['T'] = c2_tokenizer_Action_IDENT,
   ['U'] = c2_tokenizer_Action_IDENT,
   ['V'] = c2_tokenizer_Action_IDENT,
   ['W'] = c2_tokenizer_Action_IDENT,
   ['X'] = c2_tokenizer_Action_IDENT,
   ['Y'] = c2_tokenizer_Action_IDENT,
   ['Z'] = c2_tokenizer_Action_IDENT,
   ['['] = c2_tokenizer_Action_LSQUARE,
   ['\\'] = c2_tokenizer_Action_TABSPACE,
   [']'] = c2_tokenizer_Action_RSQUARE,
   ['^'] = c2_tokenizer_Action_CARET,
   ['_'] = c2_tokenizer_Action_IDENT,
   ['a'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['b'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['c'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['d'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['e'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['f'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['g'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['h'] = c2_tokenizer_Action_IDENT,
   ['i'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['j'] = c2_tokenizer_Action_IDENT,
   ['k'] = c2_tokenizer_Action_IDENT,
   ['l'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['m'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['n'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['o'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['p'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['q'] = c2_tokenizer_Action_IDENT,
   ['r'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['s'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['t'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['u'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['v'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['w'] = c2_tokenizer_Action_IDENT_OR_KEYWORD,
   ['x'] = c2_tokenizer_Action_IDENT,
   ['y'] = c2_tokenizer_Action_IDENT,
   ['z'] = c2_tokenizer_Action_IDENT,
   ['{'] = c2_tokenizer_Action_LBRACE,
   ['|'] = c2_tokenizer_Action_PIPE,
   ['}'] = c2_tokenizer_Action_RBRACE,
   ['~'] = c2_tokenizer_Action_TILDE
};

static const uint8_t c2_tokenizer_Identifier_char[128] = {
   ['0'] = 1,
   ['1'] = 1,
   ['2'] = 1,
   ['3'] = 1,
   ['4'] = 1,
   ['5'] = 1,
   ['6'] = 1,
   ['7'] = 1,
   ['8'] = 1,
   ['9'] = 1,
   ['A'] = 1,
   ['B'] = 1,
   ['C'] = 1,
   ['D'] = 1,
   ['E'] = 1,
   ['F'] = 1,
   ['G'] = 1,
   ['H'] = 1,
   ['I'] = 1,
   ['J'] = 1,
   ['K'] = 1,
   ['L'] = 1,
   ['M'] = 1,
   ['N'] = 1,
   ['O'] = 1,
   ['P'] = 1,
   ['Q'] = 1,
   ['R'] = 1,
   ['S'] = 1,
   ['T'] = 1,
   ['U'] = 1,
   ['V'] = 1,
   ['W'] = 1,
   ['X'] = 1,
   ['Y'] = 1,
   ['Z'] = 1,
   ['_'] = 1,
   ['a'] = 1,
   ['b'] = 1,
   ['c'] = 1,
   ['d'] = 1,
   ['e'] = 1,
   ['f'] = 1,
   ['g'] = 1,
   ['h'] = 1,
   ['i'] = 1,
   ['j'] = 1,
   ['k'] = 1,
   ['l'] = 1,
   ['m'] = 1,
   ['n'] = 1,
   ['o'] = 1,
   ['p'] = 1,
   ['q'] = 1,
   ['r'] = 1,
   ['s'] = 1,
   ['t'] = 1,
   ['u'] = 1,
   ['v'] = 1,
   ['w'] = 1,
   ['x'] = 1,
   ['y'] = 1,
   ['z'] = 1
};

static const c2_tokenizer_Keyword* c2_tokenizer_check_keyword(const char* cp);
static void c2_tokenizer_Tokenizer_init(c2_tokenizer_Tokenizer* t, string_pool_Pool* pool, string_buffer_Buf* buf, const char* input, src_loc_SrcLoc loc_start, const string_list_List* features, bool raw_mode);
static void c2_tokenizer_Tokenizer_lex(c2_tokenizer_Tokenizer* t, token_Token* result);
static void c2_tokenizer_Tokenizer_lex_internal(c2_tokenizer_Tokenizer* t, token_Token* result);
static token_Token c2_tokenizer_Tokenizer_lookahead(c2_tokenizer_Tokenizer* t, uint32_t n);
__attribute__((__format__(printf, 3, 4))) 
static void c2_tokenizer_Tokenizer_error(c2_tokenizer_Tokenizer* t, token_Token* result, const char* format, ...);
static void c2_tokenizer_Tokenizer_lex_identifier(c2_tokenizer_Tokenizer* t, token_Token* result);
static uint8_t c2_tokenizer_hex2val(char c);
static bool c2_tokenizer_is_octal(char c);
static bool c2_tokenizer_is_binary(char c);
static void c2_tokenizer_Tokenizer_lex_number(c2_tokenizer_Tokenizer* t, token_Token* result);
static void c2_tokenizer_Tokenizer_lex_floating_point(c2_tokenizer_Tokenizer* t, token_Token* result, const char* start);
static uint32_t c2_tokenizer_Tokenizer_lex_escaped_char(c2_tokenizer_Tokenizer* t, token_Token* result);
static void c2_tokenizer_Tokenizer_lex_char_literal(c2_tokenizer_Tokenizer* t, token_Token* result);
static void c2_tokenizer_Tokenizer_lex_string_literal(c2_tokenizer_Tokenizer* t, token_Token* result);
static bool c2_tokenizer_Tokenizer_lex_string_literal_multi(c2_tokenizer_Tokenizer* t, token_Token* result, uint32_t* num_escapes);
static bool c2_tokenizer_Tokenizer_lex_line_comment(c2_tokenizer_Tokenizer* t, token_Token* result);
static bool c2_tokenizer_Tokenizer_lex_block_comment(c2_tokenizer_Tokenizer* t, token_Token* result);
static bool c2_tokenizer_compare_word(const char* cur, const char* expect);
static bool c2_tokenizer_Tokenizer_lex_feature_cmd(c2_tokenizer_Tokenizer* t, token_Token* result);
static bool c2_tokenizer_Tokenizer_parse_error_warn(c2_tokenizer_Tokenizer* t, token_Token* result, bool is_error);
static bool c2_tokenizer_Tokenizer_is_enabled(const c2_tokenizer_Tokenizer* t);
static bool c2_tokenizer_Tokenizer_handle_if(c2_tokenizer_Tokenizer* t, token_Token* result);
static bool c2_tokenizer_Tokenizer_parse_feature(c2_tokenizer_Tokenizer* t, token_Token* result, bool* enabled);
static bool c2_tokenizer_Tokenizer_handle_else(c2_tokenizer_Tokenizer* t, token_Token* result);
static bool c2_tokenizer_Tokenizer_handle_endif(c2_tokenizer_Tokenizer* t, token_Token* result);
static bool c2_tokenizer_Tokenizer_skip_feature(c2_tokenizer_Tokenizer* t, token_Token* result);
static void c2_tokenizer_Tokenizer_skip_string_literal(c2_tokenizer_Tokenizer* t);
static void c2_tokenizer_Tokenizer_skip_char_literal(c2_tokenizer_Tokenizer* t);
static bool c2_tokenizer_Tokenizer_is_multi_string(c2_tokenizer_Tokenizer* t);
static bool c2_tokenizer_Tokenizer_skip_to_next_string(c2_tokenizer_Tokenizer* t, token_Token* result);
static const c2_tokenizer_Keyword* c2_tokenizer_check_keyword(const char* cp)
{
   const c2_tokenizer_Keyword* table = c2_tokenizer_keywords[(*cp - 'a')];
   uint32_t i = 0;
   while (table[i].name) {
      const char* word = cp;
      const char* kw = table[i].name;
      uint32_t idx = 0;
      while (1) {
         char a = kw[idx];
         char b = word[idx];
         if ((a == 0)) {
            if (!c2_tokenizer_Identifier_char[b]) return &table[i];

            break;
         }
         if ((a != b)) {
            if ((b < a)) return NULL;

            break;
         }
         idx++;
      }
      i++;
   }
   return NULL;
}

static void c2_tokenizer_Tokenizer_init(c2_tokenizer_Tokenizer* t, string_pool_Pool* pool, string_buffer_Buf* buf, const char* input, src_loc_SrcLoc loc_start, const string_list_List* features, bool raw_mode)
{
   memset(t, 0, 600);
   t->cur = input;
   t->input_start = input;
   t->loc_start = loc_start;
   t->line_start = input;
   t->pool = pool;
   t->buf = buf;
   t->features = features;
   t->raw_mode = raw_mode;
   for (uint32_t i = 0; (i < c2_tokenizer_MaxLookahead); i++) {
      token_Token_init(&t->next[i]);
   }
}

static void c2_tokenizer_Tokenizer_lex(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   if (t->next_count) {
      memcpy(result, &t->next[t->next_head], 16);
      t->next_head = (((t->next_head + 1)) % c2_tokenizer_MaxLookahead);
      t->next_count--;
      return;
   }
   c2_tokenizer_Tokenizer_lex_internal(t, result);
}

static void c2_tokenizer_Tokenizer_lex_internal(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   while (1) {
      c2_tokenizer_Action act = c2_tokenizer_Char_lookup[*t->cur];
      switch (act) {
      case c2_tokenizer_Action_TABSPACE:
         t->cur++;
         continue;
      case c2_tokenizer_Action_IDENT_OR_KEYWORD: {
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         const c2_tokenizer_Keyword* kw = c2_tokenizer_check_keyword(t->cur);
         if (kw) {
            result->kind = kw->kind;
            t->cur += kw->len;
         } else {
            c2_tokenizer_Tokenizer_lex_identifier(t, result);
         }
         return;
      }
      case c2_tokenizer_Action_IDENT:
         c2_tokenizer_Tokenizer_lex_identifier(t, result);
         return;
      case c2_tokenizer_Action_DIGIT:
         c2_tokenizer_Tokenizer_lex_number(t, result);
         return;
      case c2_tokenizer_Action_LPAREN:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_LParen;
         t->cur++;
         return;
      case c2_tokenizer_Action_RPAREN:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_RParen;
         t->cur++;
         return;
      case c2_tokenizer_Action_LSQUARE:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_LSquare;
         t->cur++;
         return;
      case c2_tokenizer_Action_RSQUARE:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_RSquare;
         t->cur++;
         return;
      case c2_tokenizer_Action_NEWLINE:
         t->cur++;
         t->line_start = t->cur;
         continue;
      case c2_tokenizer_Action_EXCLAIM:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '=')) {
            result->kind = token_Kind_ExclaimEqual;
            t->cur++;
         } else {
            result->kind = token_Kind_Exclaim;
         }
         return;
      case c2_tokenizer_Action_DQUOTE:
         c2_tokenizer_Tokenizer_lex_string_literal(t, result);
         return;
      case c2_tokenizer_Action_SQUOTE:
         c2_tokenizer_Tokenizer_lex_char_literal(t, result);
         return;
      case c2_tokenizer_Action_POUND:
         if (c2_tokenizer_Tokenizer_lex_feature_cmd(t, result)) return;

         if (!c2_tokenizer_Tokenizer_is_enabled(t)) {
            if (c2_tokenizer_Tokenizer_skip_feature(t, result)) return;

         }
         continue;
      case c2_tokenizer_Action_STAR:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '=')) {
            result->kind = token_Kind_StarEqual;
            t->cur++;
         } else {
            result->kind = token_Kind_Star;
         }
         return;
      case c2_tokenizer_Action_PLUS:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '+')) {
            t->cur++;
            result->kind = token_Kind_PlusPlus;
            return;
         }
         if ((*t->cur == '=')) {
            t->cur++;
            result->kind = token_Kind_PlusEqual;
            return;
         }
         result->kind = token_Kind_Plus;
         return;
      case c2_tokenizer_Action_MINUS:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '-')) {
            t->cur++;
            result->kind = token_Kind_MinusMinus;
            return;
         }
         if ((*t->cur == '=')) {
            t->cur++;
            result->kind = token_Kind_MinusEqual;
            return;
         }
         if ((*t->cur == '>')) {
            t->cur--;
            c2_tokenizer_Tokenizer_error(t, result, "use the dot operators instead of '->'");
            return;
         }
         result->kind = token_Kind_Minus;
         return;
      case c2_tokenizer_Action_COMMA:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_Comma;
         t->cur++;
         return;
      case c2_tokenizer_Action_DOT:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if (((t->cur[0] == '.') && (t->cur[1] == '.'))) {
            t->cur += 2;
            result->kind = token_Kind_Ellipsis;
         } else {
            result->kind = token_Kind_Dot;
         }
         return;
      case c2_tokenizer_Action_PERCENT:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '=')) {
            result->kind = token_Kind_PercentEqual;
            t->cur++;
         } else {
            result->kind = token_Kind_Percent;
         }
         return;
      case c2_tokenizer_Action_SLASH:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '=')) {
            result->kind = token_Kind_SlashEqual;
            t->cur++;
            return;
         }
         if ((*t->cur == '/')) {
            if (c2_tokenizer_Tokenizer_lex_line_comment(t, result)) return;

            continue;
         }
         if ((*t->cur == '*')) {
            if (c2_tokenizer_Tokenizer_lex_block_comment(t, result)) return;

            continue;
         }
         result->kind = token_Kind_Slash;
         return;
      case c2_tokenizer_Action_COLON:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_Colon;
         t->cur++;
         return;
      case c2_tokenizer_Action_SEMI_COLON:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_Semicolon;
         t->cur++;
         return;
      case c2_tokenizer_Action_LESS:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '=')) {
            t->cur++;
            result->kind = token_Kind_LessEqual;
            return;
         }
         if ((*t->cur == '<')) {
            t->cur++;
            if ((*t->cur == '=')) {
               t->cur++;
               result->kind = token_Kind_LessLessEqual;
            } else {
               result->kind = token_Kind_LessLess;
            }
            return;
         }
         result->kind = token_Kind_Less;
         return;
      case c2_tokenizer_Action_EQUAL:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '=')) {
            result->kind = token_Kind_EqualEqual;
            t->cur++;
         } else {
            result->kind = token_Kind_Equal;
         }
         return;
      case c2_tokenizer_Action_GREATER:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '=')) {
            t->cur++;
            result->kind = token_Kind_GreaterEqual;
            return;
         }
         if ((*t->cur == '>')) {
            t->cur++;
            if ((*t->cur == '=')) {
               t->cur++;
               result->kind = token_Kind_GreaterGreaterEqual;
            } else {
               result->kind = token_Kind_GreaterGreater;
            }
            return;
         }
         result->kind = token_Kind_Greater;
         return;
      case c2_tokenizer_Action_QUESTION:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_Question;
         t->cur++;
         return;
      case c2_tokenizer_Action_AT:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_At;
         t->cur++;
         return;
      case c2_tokenizer_Action_AMP:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '&')) {
            result->kind = token_Kind_AmpAmp;
            t->cur++;
            return;
         }
         if ((*t->cur == '=')) {
            result->kind = token_Kind_AmpEqual;
            t->cur++;
            return;
         }
         result->kind = token_Kind_Amp;
         return;
      case c2_tokenizer_Action_CARET:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '=')) {
            t->cur++;
            result->kind = token_Kind_CaretEqual;
            return;
         }
         result->kind = token_Kind_Caret;
         return;
      case c2_tokenizer_Action_LBRACE:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_LBrace;
         t->cur++;
         return;
      case c2_tokenizer_Action_RBRACE:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_RBrace;
         t->cur++;
         return;
      case c2_tokenizer_Action_PIPE:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         t->cur++;
         if ((*t->cur == '|')) {
            result->kind = token_Kind_PipePipe;
            t->cur++;
            return;
         }
         if ((*t->cur == '=')) {
            result->kind = token_Kind_PipeEqual;
            t->cur++;
            return;
         }
         result->kind = token_Kind_Pipe;
         return;
      case c2_tokenizer_Action_TILDE:
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
         result->kind = token_Kind_Tilde;
         t->cur++;
         return;
      case c2_tokenizer_Action_CR:
         t->cur++;
         if ((*t->cur != '\n')) {
            c2_tokenizer_Tokenizer_error(t, result, "unexpected char 0x%02X", *t->cur);
            return;
         }
         t->cur++;
         return;
      case c2_tokenizer_Action_EOF:
         result->loc = ((t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start)))) - 1);
         result->kind = token_Kind_Eof;
         result->more = false;
         return;
      case c2_tokenizer_Action_INVALID:
         c2_tokenizer_Tokenizer_error(t, result, "invalid char '%c'", *t->cur);
         return;
      }
   }
}

static token_Token c2_tokenizer_Tokenizer_lookahead(c2_tokenizer_Tokenizer* t, uint32_t n)
{
   c2_assert(((n > 0)) != 0, "parser/c2_tokenizer.c2:790: c2_tokenizer.Tokenizer.lookahead", "n>0");
   c2_assert(((n <= c2_tokenizer_MaxLookahead)) != 0, "parser/c2_tokenizer.c2:791: c2_tokenizer.Tokenizer.lookahead", "n<=MaxLookahead");
   while ((t->next_count < n)) {
      const uint32_t slot = (((t->next_head + t->next_count)) % c2_tokenizer_MaxLookahead);
      c2_tokenizer_Tokenizer_lex_internal(t, &t->next[slot]);
      t->next_count++;
   }
   uint32_t slot = ((((t->next_head + n) - 1)) % c2_tokenizer_MaxLookahead);
   return t->next[slot];
}

__attribute__((__format__(printf, 3, 4))) 
static void c2_tokenizer_Tokenizer_error(c2_tokenizer_Tokenizer* t, token_Token* result, const char* format, ...)
{
   va_list args;
   va_start(args, format);
   vsnprintf(t->error_msg, (256 - 1), format, args);
   va_end(args);
   result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
   result->kind = token_Kind_Error;
   result->error_msg = t->error_msg;
   result->more = false;
}

static void c2_tokenizer_Tokenizer_lex_identifier(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   result->kind = token_Kind_Identifier;
   result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
   const char* start = t->cur;
   const char* end = (t->cur + 1);
   while (c2_tokenizer_Identifier_char[*end]) end++;
   size_t len = ((size_t)((end - start)));
   if ((len > constants_MaxIdentifierLen)) {
      c2_tokenizer_Tokenizer_error(t, result, "identifier too long (max %u chars)", constants_MaxIdentifierLen);
      return;
   }
   t->cur += len;
   result->text_idx = string_pool_Pool_add(t->pool, start, len, true);
}

static uint8_t c2_tokenizer_hex2val(char c)
{
   if (((c >= '0') && (c <= '9'))) return ((uint8_t)((c - '0')));

   if (((c >= 'a') && (c <= 'f'))) return ((uint8_t)((c - 'a')));

   return ((uint8_t)((c - 'A')));
}

static bool c2_tokenizer_is_octal(char c)
{
   return (((c >= '0') && (c <= '7')));
}

static bool c2_tokenizer_is_binary(char c)
{
   return (((c >= '0') && (c <= '1')));
}

static void c2_tokenizer_Tokenizer_lex_number(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   result->kind = token_Kind_IntegerLiteral;
   result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
   const char* start;
   if ((t->cur[0] == '0')) {
      if ((t->cur[1] == 'x')) {
         result->radix = 16;
         t->cur += 2;
         start = t->cur;
         while (isxdigit(*t->cur)) t->cur++;
         if (isalpha(*t->cur)) {
            c2_tokenizer_Tokenizer_error(t, result, "invalid character '%c' in hexadecimal constant", *t->cur);
            return;
         }
         result->int_value = strtoull(start, NULL, 16);
         return;
      }
      if (c2_tokenizer_is_octal(t->cur[1])) {
         result->radix = 8;
         t->cur++;
         start = t->cur;
         while (c2_tokenizer_is_octal(*t->cur)) t->cur++;
         if (isxdigit(*t->cur)) {
            c2_tokenizer_Tokenizer_error(t, result, "invalid digit '%c' in octal constant", *t->cur);
            return;
         }
         result->int_value = strtoull(start, NULL, 8);
         return;
      }
      if ((t->cur[1] == 'b')) {
         result->radix = 2;
         t->cur += 2;
         start = t->cur;
         while (c2_tokenizer_is_binary(*t->cur)) t->cur++;
         if (isdigit(*t->cur)) {
            c2_tokenizer_Tokenizer_error(t, result, "invalid digit '%c' in binary constant", *t->cur);
            return;
         }
         result->int_value = strtoull(start, NULL, 2);
         return;
      }
      if ((t->cur[1] == '.')) {
         t->cur++;
         c2_tokenizer_Tokenizer_lex_floating_point(t, result, t->cur);
         return;
      }
      if (isdigit(t->cur[1])) {
         c2_tokenizer_Tokenizer_error(t, result, "decimal numbers may not start with a 0");
         return;
      }
      t->cur++;
      result->radix = 10;
      result->int_value = 0;
      return;
   }
   result->radix = 10;
   start = t->cur;
   while (isdigit(*t->cur)) t->cur++;
   if ((t->cur[0] == '.')) {
      c2_tokenizer_Tokenizer_lex_floating_point(t, result, start);
      return;
   }
   uint32_t len = ((uint32_t)((t->cur - start)));
   if ((len >= 20)) {
      if (((len > 20) || (strncmp(start, "18446744073709551615", 20) > 0))) {
         t->cur -= len;
         c2_tokenizer_Tokenizer_error(t, result, "integer literal is too large to be represented in any integer type");
         return;
      }
   }
   result->int_value = strtoull(start, NULL, 10);
}

static void c2_tokenizer_Tokenizer_lex_floating_point(c2_tokenizer_Tokenizer* t, token_Token* result, const char* start)
{
   t->cur++;
   result->kind = token_Kind_FloatLiteral;
   result->loc = (t->loc_start + ((src_loc_SrcLoc)((start - t->input_start))));
   result->float_value = strtod(start, NULL);
   while (isdigit(*t->cur)) t->cur++;
}

static uint32_t c2_tokenizer_Tokenizer_lex_escaped_char(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   const char* input = (t->cur + 1);
   switch (input[0]) {
   case '"':
      result->char_value = '"';
      break;
   case '\'':
      result->char_value = '\'';
      break;
   case '?':
      result->char_value = '?';
      break;
   case '\\':
      result->char_value = '\\';
      break;
   case 'a':
      result->char_value = '\a';
      break;
   case 'b':
      result->char_value = '\b';
      break;
   case 'f':
      result->char_value = '\f';
      break;
   case 'n':
      result->char_value = '\n';
      break;
   case 'r':
      result->char_value = '\r';
      break;
   case 't':
      result->char_value = '\t';
      break;
   case 'u':
      c2_tokenizer_Tokenizer_error(t, result, "unicode escape sequences not supported yet");
      return 0;
   case 'v':
      result->char_value = '\v';
      break;
   case 'x':
      if ((!isxdigit(input[1]) || !isxdigit(input[2]))) {
         t->cur++;
         c2_tokenizer_Tokenizer_error(t, result, "expect hexadecimal number after '\\x'");
         return 0;
      }
      result->char_value = ((c2_tokenizer_hex2val(input[1]) * 16) + c2_tokenizer_hex2val(input[2]));
      result->radix = 16;
      return 3;
   default:
      if (c2_tokenizer_is_octal(input[0])) {
         uint32_t offset = 0;
         uint32_t value = 0;
         while ((c2_tokenizer_is_octal(input[offset]) && (offset <= 3))) {
            value *= 8;
            value += ((uint32_t)((input[offset] - '0')));
            offset++;
         }
         if ((value > 127)) {
            t->cur++;
            c2_tokenizer_Tokenizer_error(t, result, "octal escape sequence out of range");
            return 0;
         }
         result->char_value = ((uint8_t)(value));
         result->radix = 8;
         return offset;
      }
      t->cur++;
      c2_tokenizer_Tokenizer_error(t, result, "unknown escape sequence '\\%c'", input[0]);
      return 0;
   }
   return 1;
}

static void c2_tokenizer_Tokenizer_lex_char_literal(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   result->kind = token_Kind_CharLiteral;
   result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
   result->radix = 10;
   if ((t->cur[1] == '\\')) {
      t->cur++;
      uint32_t len = c2_tokenizer_Tokenizer_lex_escaped_char(t, result);
      if ((len == 0)) return;

      t->cur += ((len - 1));
   } else {
      result->char_value = ((uint8_t)(t->cur[1]));
   }
   if ((t->cur[2] != '\'')) {
      if (((t->cur[2] != 0) && (t->cur[3] == '\''))) {
         c2_tokenizer_Tokenizer_error(t, result, "multi-character character constant");
      } else {
         c2_tokenizer_Tokenizer_error(t, result, "missing terminating ' character (GOT %c)", t->cur[2]);
      }
      return;
   }
   t->cur += 3;
}

static void c2_tokenizer_Tokenizer_lex_string_literal(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   result->kind = token_Kind_StringLiteral;
   result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start))));
   t->cur++;
   const char* start = t->cur;
   uint32_t len;
   uint32_t num_escapes = 0;
   while (1) {
      switch (*t->cur) {
      case 0:
         __attribute__((fallthrough));
      case '\r':
         __attribute__((fallthrough));
      case '\n':
         t->cur--;
         c2_tokenizer_Tokenizer_error(t, result, "unterminated string");
         return;
      case '\\': {
         uint32_t esc_len = c2_tokenizer_Tokenizer_lex_escaped_char(t, result);
         if ((esc_len == 0)) return;

         num_escapes += esc_len;
         t->cur += ((esc_len + 1));
         break;
      }
      case '"':
         goto out;
      default:
         t->cur++;
         break;
      }
   }
   out:
   len = ((uint32_t)((t->cur - start)));
   t->cur++;
   if ((!t->raw_mode && c2_tokenizer_Tokenizer_is_multi_string(t))) {
      string_buffer_Buf_clear(t->buf);
      string_buffer_Buf_add2(t->buf, start, len);
      while (1) {
         if (!c2_tokenizer_Tokenizer_skip_to_next_string(t, result)) return;

         if (!c2_tokenizer_Tokenizer_lex_string_literal_multi(t, result, &num_escapes)) return;

         if (!c2_tokenizer_Tokenizer_is_multi_string(t)) break;

      }
      result->text_len = ((string_buffer_Buf_size(t->buf) + 1) - num_escapes);
      result->text_idx = string_pool_Pool_add(t->pool, string_buffer_Buf_data(t->buf), string_buffer_Buf_size(t->buf), false);
   } else {
      result->text_len = ((len + 1) - num_escapes);
      result->text_idx = string_pool_Pool_add(t->pool, start, len, false);
   }
}

static bool c2_tokenizer_Tokenizer_lex_string_literal_multi(c2_tokenizer_Tokenizer* t, token_Token* result, uint32_t* num_escapes)
{
   uint32_t len;
   t->cur++;
   const char* start = t->cur;
   while (1) {
      switch (*t->cur) {
      case 0:
         __attribute__((fallthrough));
      case '\r':
         __attribute__((fallthrough));
      case '\n':
         t->cur--;
         c2_tokenizer_Tokenizer_error(t, result, "unterminated string");
         return false;
      case '\\': {
         uint32_t esc_len = c2_tokenizer_Tokenizer_lex_escaped_char(t, result);
         if ((esc_len == 0)) return false;

         *num_escapes += esc_len;
         t->cur += ((esc_len + 1));
         break;
      }
      case '"':
         goto out;
      default:
         t->cur++;
         break;
      }
   }
   out:
   len = ((uint32_t)((t->cur - start)));
   string_buffer_Buf_add2(t->buf, start, len);
   t->cur++;
   return true;
}

static bool c2_tokenizer_Tokenizer_lex_line_comment(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   t->cur += 1;
   const char* start = t->cur;
   const char* end = start;
   while (*end) {
      if (((*end == '\r') || (*end == '\n'))) break;

      end++;
   }
   uint32_t len = ((uint32_t)((end - start)));
   t->cur += len;
   if (t->raw_mode) {
      result->kind = token_Kind_LineComment;
      result->loc = (t->loc_start + ((src_loc_SrcLoc)(((start - t->input_start) - 2))));
      result->text_idx = string_pool_Pool_add(t->pool, start, len, false);
      return true;
   }
   return false;
}

static bool c2_tokenizer_Tokenizer_lex_block_comment(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   t->cur += 1;
   const char* start = t->cur;
   while (1) {
      switch (*t->cur) {
      case 0:
         t->cur--;
         c2_tokenizer_Tokenizer_error(t, result, "un-terminated block comment");
         return true;
      case '/':
         if ((t->cur[1] == '*')) {
            c2_tokenizer_Tokenizer_error(t, result, "'/*' within block comment");
            return true;
         }
         break;
      case '*':
         if ((t->cur[1] == '/')) {
            t->cur += 2;
            if (t->raw_mode) {
               size_t len = ((size_t)(((t->cur - start) - 2)));
               result->kind = token_Kind_BlockComment;
               result->loc = (t->loc_start + ((src_loc_SrcLoc)(((start - t->input_start) - 2))));
               result->text_idx = string_pool_Pool_add(t->pool, start, len, false);
               return true;
            }
            return false;
         }
         break;
      default:
         break;
      }
      t->cur++;
   }
   return false;
}

static bool c2_tokenizer_compare_word(const char* cur, const char* expect)
{
   while (*expect) {
      if ((*cur != *expect)) return false;

      cur++;
      expect++;
   }
   return !c2_tokenizer_Identifier_char[*cur];
}

static bool c2_tokenizer_Tokenizer_lex_feature_cmd(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   if ((t->cur != t->line_start)) {
      c2_tokenizer_Tokenizer_error(t, result, "#if/#else/#endif must be at start of line");
      return true;
   }
   t->cur++;
   if (c2_tokenizer_compare_word(t->cur, "if")) {
      t->cur += 2;
      if (c2_tokenizer_Tokenizer_handle_if(t, result)) return true;

   } else if (c2_tokenizer_compare_word(t->cur, "else")) {
      t->cur += 4;
      if (c2_tokenizer_Tokenizer_handle_else(t, result)) return true;

   } else if (c2_tokenizer_compare_word(t->cur, "endif")) {
      t->cur += 5;
      if (c2_tokenizer_Tokenizer_handle_endif(t, result)) return true;

   } else if (c2_tokenizer_compare_word(t->cur, "error")) {
      t->cur += 5;
      if (t->raw_mode) return false;

      return c2_tokenizer_Tokenizer_parse_error_warn(t, result, true);
   } else if (c2_tokenizer_compare_word(t->cur, "warn")) {
      t->cur += 4;
      if (t->raw_mode) return false;

      return c2_tokenizer_Tokenizer_parse_error_warn(t, result, false);
   } else {
      c2_tokenizer_Tokenizer_error(t, result, "unknown feature-selection command");
      return true;
   }




   return false;
}

static bool c2_tokenizer_Tokenizer_parse_error_warn(c2_tokenizer_Tokenizer* t, token_Token* result, bool is_error)
{
   t->cur++;
   if ((*t->cur != '"')) {
      c2_tokenizer_Tokenizer_error(t, result, "expect '\"'");
      return true;
   }
   t->cur++;
   const char* start = t->cur;
   while ((*t->cur != '"')) {
      switch (*t->cur) {
      case 0:
         __attribute__((fallthrough));
      case '\r':
         __attribute__((fallthrough));
      case '\n':
         c2_tokenizer_Tokenizer_error(t, result, "unterminated string");
         return true;
      case '"':
         break;
      default:
         t->cur++;
         break;
      }
   }
   size_t len = ((size_t)((t->cur - start)));
   t->cur++;
   if ((len > constants_MaxIdentifierLen)) {
      c2_tokenizer_Tokenizer_error(t, result, "error msg too long (max %u bytes)", constants_MaxErrorMsgLen);
      return true;
   }
   char msg[32];
   memcpy(msg, start, len);
   msg[len] = 0;
   if (c2_tokenizer_Tokenizer_is_enabled(t)) {
      if (is_error) {
         t->cur = t->line_start;
         c2_tokenizer_Tokenizer_error(t, result, "%s", msg);
      } else {
         strcpy(t->error_msg, msg);
         result->loc = (t->loc_start + ((src_loc_SrcLoc)((t->line_start - t->input_start))));
         result->kind = token_Kind_Warning;
         result->error_msg = t->error_msg;
      }
      return true;
   }
   return false;
}

static bool c2_tokenizer_Tokenizer_is_enabled(const c2_tokenizer_Tokenizer* t)
{
   for (uint32_t i = 0; (i < t->feature_count); i++) {
      if (!t->feature_stack[i].enabled) return false;

   }
   return true;
}

static bool c2_tokenizer_Tokenizer_handle_if(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   if (t->raw_mode) {
      result->kind = token_Kind_Feat_if;
      result->loc = ((t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start)))) - 3);
      return true;
   }
   if ((t->feature_count >= constants_MaxFeatureDepth)) {
      c2_tokenizer_Tokenizer_error(t, result, "feature nesting too much");
      return true;
   }
   t->cur++;
   bool enabled = false;
   c2_tokenizer_Action act = c2_tokenizer_Char_lookup[*t->cur];
   switch (act) {
   case c2_tokenizer_Action_INVALID:
      c2_tokenizer_Tokenizer_error(t, result, "invalid char '%c'", *t->cur);
      return true;
   case c2_tokenizer_Action_IDENT_OR_KEYWORD:
      __attribute__((fallthrough));
   case c2_tokenizer_Action_IDENT:
      if (c2_tokenizer_Tokenizer_parse_feature(t, result, &enabled)) return true;

      break;
   case c2_tokenizer_Action_DIGIT:
      if ((*t->cur != '0')) enabled = true;
      t->cur++;
      break;
   case c2_tokenizer_Action_EOF:
      t->cur--;
      c2_tokenizer_Tokenizer_error(t, result, "expected feature");
      return true;
   default:
      c2_tokenizer_Tokenizer_error(t, result, "invalid feature value");
      return true;
   }
   c2_tokenizer_Feature* next = &t->feature_stack[t->feature_count];
   next->is_if = true;
   next->enabled = enabled;
   t->feature_count++;
   return false;
}

static bool c2_tokenizer_Tokenizer_parse_feature(c2_tokenizer_Tokenizer* t, token_Token* result, bool* enabled)
{
   const char* start = t->cur;
   while (c2_tokenizer_Identifier_char[*t->cur]) t->cur++;
   size_t len = ((size_t)((t->cur - start)));
   if ((len > constants_MaxFeatureName)) {
      c2_tokenizer_Tokenizer_error(t, result, "feature name too long (max %u chars)", constants_MaxFeatureName);
      return true;
   }
   char name[32];
   memcpy(name, start, len);
   name[len] = 0;
   if (string_list_List_contains(t->features, name)) *enabled = true;
   return false;
}

static bool c2_tokenizer_Tokenizer_handle_else(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   if (t->raw_mode) {
      result->kind = token_Kind_Feat_else;
      result->loc = ((t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start)))) - 5);
      return true;
   }
   if ((t->feature_count == 0)) {
      c2_tokenizer_Tokenizer_error(t, result, "#else without #if");
      return true;
   }
   c2_tokenizer_Feature* top = &t->feature_stack[(t->feature_count - 1)];
   if (!top->is_if) {
      c2_tokenizer_Tokenizer_error(t, result, "#else in #else");
      return true;
   }
   top->is_if = false;
   top->enabled = !top->enabled;
   return false;
}

static bool c2_tokenizer_Tokenizer_handle_endif(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   if (t->raw_mode) {
      result->kind = token_Kind_Feat_endif;
      result->loc = ((t->loc_start + ((src_loc_SrcLoc)((t->cur - t->input_start)))) - 6);
      return true;
   }
   if ((t->feature_count == 0)) {
      c2_tokenizer_Tokenizer_error(t, result, "#endif without #if/#else");
      return true;
   }
   t->feature_count--;
   return false;
}

static bool c2_tokenizer_Tokenizer_skip_feature(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   while (1) {
      c2_tokenizer_Action act = c2_tokenizer_Char_lookup[*t->cur];
      switch (act) {
      case c2_tokenizer_Action_INVALID:
         t->cur++;
         break;
      case c2_tokenizer_Action_NEWLINE:
         t->cur++;
         t->line_start = t->cur;
         break;
      case c2_tokenizer_Action_DQUOTE:
         c2_tokenizer_Tokenizer_skip_string_literal(t);
         break;
      case c2_tokenizer_Action_POUND:
         if (c2_tokenizer_Tokenizer_lex_feature_cmd(t, result)) return true;

         if (c2_tokenizer_Tokenizer_is_enabled(t)) return false;

         break;
      case c2_tokenizer_Action_EOF: {
         t->cur--;
         c2_tokenizer_Feature* top = &t->feature_stack[(t->feature_count - 1)];
         c2_tokenizer_Tokenizer_error(t, result, "un-terminated #%s", top->is_if ? "if" : "else");
         return true;
      }
      default:
         t->cur++;
         break;
      }
   }
   return false;
}

static void c2_tokenizer_Tokenizer_skip_string_literal(c2_tokenizer_Tokenizer* t)
{
   t->cur++;
   while (1) {
      switch (*t->cur) {
      case 0:
         return;
      case '\r':
         t->cur++;
         return;
      case '\n':
         return;
      case '"':
         t->cur++;
         return;
      default:
         t->cur++;
         break;
      }
   }
}

static void c2_tokenizer_Tokenizer_skip_char_literal(c2_tokenizer_Tokenizer* t)
{
   t->cur++;
   while (1) {
      switch (*t->cur) {
      case 0:
         return;
      case '\'':
         t->cur++;
         return;
      default:
         t->cur++;
         break;
      }
   }
}

static bool c2_tokenizer_Tokenizer_is_multi_string(c2_tokenizer_Tokenizer* t)
{
   const char* c = t->cur;
   while (1) {
      switch (*c) {
      case '\t':
         __attribute__((fallthrough));
      case '\n':
         __attribute__((fallthrough));
      case '\r':
         __attribute__((fallthrough));
      case ' ':
         c++;
         break;
      case '"':
         return true;
      default:
         return false;
      }
   }
   return false;
}

static bool c2_tokenizer_Tokenizer_skip_to_next_string(c2_tokenizer_Tokenizer* t, token_Token* result)
{
   while (1) {
      switch (*t->cur) {
      case '\t':
         t->cur++;
         break;
      case '\n':
         t->cur++;
         t->line_start = t->cur;
         break;
      case '\r':
         t->cur++;
         if ((*t->cur != '\n')) {
            c2_tokenizer_Tokenizer_error(t, result, "unexpected char 0x%02X", *t->cur);
            return false;
         }
         t->cur++;
         break;
      case ' ':
         t->cur++;
         break;
      case '"':
         return true;
      }
   }
   return true;
}


// --- module parser_utils ---

static src_loc_SrcLoc parser_utils_getTokenEnd(const char* input, src_loc_SrcLoc start);
static src_loc_SrcLoc parser_utils_getTokenEnd(const char* input, src_loc_SrcLoc start)
{
   c2_tokenizer_Tokenizer tokenizer;
   string_pool_Pool* pool = string_pool_create(128, 20);
   string_buffer_Buf* buf = string_buffer_create(1024, 0, false);
   string_list_List features;
   string_list_List_init(&features, pool);
   c2_tokenizer_Tokenizer_init(&tokenizer, pool, buf, input, start, &features, false);
   token_Token result;
   token_Token_init(&result);
   c2_tokenizer_Tokenizer_lex(&tokenizer, &result);
   string_list_List_free(&features);
   string_pool_Pool_free(pool);
   string_buffer_Buf_free(buf);
   return ((start + ((src_loc_SrcLoc)((tokenizer.cur - tokenizer.input_start)))) - 1);
}


// --- module ast ---
typedef struct ast_DeclBits_ ast_DeclBits;
typedef struct ast_Decl_ ast_Decl;
typedef struct ast_AliasTypeDecl_ ast_AliasTypeDecl;
typedef struct ast_ArrayValue_ ast_ArrayValue;
typedef struct ast_DeclStmt_ ast_DeclStmt;
typedef struct ast_EnumConstantDeclBits_ ast_EnumConstantDeclBits;
typedef struct ast_EnumConstantDecl_ ast_EnumConstantDecl;
typedef struct ast_EnumTypeDeclBits_ ast_EnumTypeDeclBits;
typedef struct ast_EnumTypeDecl_ ast_EnumTypeDecl;
typedef struct ast_FunctionDeclBits_ ast_FunctionDeclBits;
typedef struct ast_FunctionDecl_ ast_FunctionDecl;
typedef struct ast_FunctionTypeDecl_ ast_FunctionTypeDecl;
typedef struct ast_ImportDeclBits_ ast_ImportDeclBits;
typedef struct ast_ImportDecl_ ast_ImportDecl;
typedef struct ast_StaticAssert_ ast_StaticAssert;
typedef struct ast_StructTypeDeclBits_ ast_StructTypeDeclBits;
typedef struct ast_StructLayout_ ast_StructLayout;
typedef struct ast_StructTypeDecl_ ast_StructTypeDecl;
typedef struct ast_Value_ ast_Value;
typedef struct ast_VarDeclBits_ ast_VarDeclBits;
typedef struct ast_VarDecl_ ast_VarDecl;
typedef struct ast_StmtBits_ ast_StmtBits;
typedef struct ast_Stmt_ ast_Stmt;
typedef struct ast_AsmStmtBits_ ast_AsmStmtBits;
typedef struct ast_AsmStmt_ ast_AsmStmt;
typedef struct ast_AssertStmtBits_ ast_AssertStmtBits;
typedef struct ast_AssertStmt_ ast_AssertStmt;
typedef struct ast_BreakStmt_ ast_BreakStmt;
typedef struct ast_CompoundStmtBits_ ast_CompoundStmtBits;
typedef struct ast_CompoundStmt_ ast_CompoundStmt;
typedef struct ast_ContinueStmt_ ast_ContinueStmt;
typedef struct ast_DoStmt_ ast_DoStmt;
typedef struct ast_FallthroughStmt_ ast_FallthroughStmt;
typedef struct ast_ForStmt_ ast_ForStmt;
typedef struct ast_GotoStmt_ ast_GotoStmt;
typedef struct ast_IfStmtBits_ ast_IfStmtBits;
typedef struct ast_IfStmt_ ast_IfStmt;
typedef struct ast_LabelStmt_ ast_LabelStmt;
typedef struct ast_ReturnStmtBits_ ast_ReturnStmtBits;
typedef struct ast_ReturnStmt_ ast_ReturnStmt;
typedef struct ast_SwitchCaseBits_ ast_SwitchCaseBits;
typedef struct ast_SwitchCase_ ast_SwitchCase;
typedef struct ast_SwitchStmtBits_ ast_SwitchStmtBits;
typedef struct ast_SwitchStmt_ ast_SwitchStmt;
typedef struct ast_WhileStmt_ ast_WhileStmt;
typedef struct ast_ExprBits_ ast_ExprBits;
typedef struct ast_Expr_ ast_Expr;
typedef struct ast_ArrayDesignatedInitExpr_ ast_ArrayDesignatedInitExpr;
typedef struct ast_ArraySubscriptExpr_ ast_ArraySubscriptExpr;
typedef struct ast_BinaryOperatorBits_ ast_BinaryOperatorBits;
typedef struct ast_BinaryOperator_ ast_BinaryOperator;
typedef struct ast_BitOffsetExprBits_ ast_BitOffsetExprBits;
typedef struct ast_BitOffsetExpr_ ast_BitOffsetExpr;
typedef struct ast_BooleanLiteralBits_ ast_BooleanLiteralBits;
typedef struct ast_BooleanLiteral_ ast_BooleanLiteral;
typedef struct ast_BuiltinExprBits_ ast_BuiltinExprBits;
typedef struct ast_ToContainerData_ ast_ToContainerData;
typedef struct ast_OffsetOfData_ ast_OffsetOfData;
typedef struct ast_BuiltinExpr_ ast_BuiltinExpr;
typedef struct ast_CallExprBits_ ast_CallExprBits;
typedef struct ast_CallExpr_ ast_CallExpr;
typedef struct ast_CharLiteralBits_ ast_CharLiteralBits;
typedef struct ast_CharLiteral_ ast_CharLiteral;
typedef struct ast_ConditionalOperator_ ast_ConditionalOperator;
typedef struct ast_ExplicitCastExpr_ ast_ExplicitCastExpr;
typedef struct ast_FieldDesignatedInitExpr_ ast_FieldDesignatedInitExpr;
typedef struct ast_FloatLiteral_ ast_FloatLiteral;
typedef struct ast_IdentifierExprBits_ ast_IdentifierExprBits;
typedef struct ast_IdentifierExpr_ ast_IdentifierExpr;
typedef struct ast_ImplicitCastBits_ ast_ImplicitCastBits;
typedef struct ast_ImplicitCastExpr_ ast_ImplicitCastExpr;
typedef struct ast_InitListExprBits_ ast_InitListExprBits;
typedef struct ast_InitListExpr_ ast_InitListExpr;
typedef struct ast_IntegerLiteralBits_ ast_IntegerLiteralBits;
typedef struct ast_IntegerLiteral_ ast_IntegerLiteral;
typedef struct ast_MemberExprBits_ ast_MemberExprBits;
typedef union ast_MemberRef_ ast_MemberRef;
typedef struct ast_MemberExpr_ ast_MemberExpr;
typedef struct ast_NilExpr_ ast_NilExpr;
typedef struct ast_ParenExpr_ ast_ParenExpr;
typedef struct ast_StringLiteral_ ast_StringLiteral;
typedef struct ast_TypeExpr_ ast_TypeExpr;
typedef struct ast_UnaryOperatorBits_ ast_UnaryOperatorBits;
typedef struct ast_UnaryOperator_ ast_UnaryOperator;
typedef struct ast_TypeBits_ ast_TypeBits;
typedef struct ast_Type_ ast_Type;
typedef struct ast_AliasType_ ast_AliasType;
typedef struct ast_ArrayTypeBits_ ast_ArrayTypeBits;
typedef struct ast_ArrayType_ ast_ArrayType;
typedef struct ast_BuiltinTypeBits_ ast_BuiltinTypeBits;
typedef struct ast_BuiltinType_ ast_BuiltinType;
typedef struct ast_EnumType_ ast_EnumType;
typedef struct ast_FunctionType_ ast_FunctionType;
typedef struct ast_ModuleType_ ast_ModuleType;
typedef struct ast_PointerType_ ast_PointerType;
typedef struct ast_QualType_ ast_QualType;
typedef struct ast_StructType_ ast_StructType;
typedef struct ast_TypeRefBits_ ast_TypeRefBits;
typedef struct ast_Ref_ ast_Ref;
typedef struct ast_TypeRef_ ast_TypeRef;
typedef struct ast_TypeRefHolder_ ast_TypeRefHolder;
typedef struct ast_ArrayValueList_ ast_ArrayValueList;
typedef struct ast_AST_ ast_AST;
typedef struct ast_DeclList_ ast_DeclList;
typedef struct ast_ExprList_ ast_ExprList;
typedef struct ast_FunctionDeclList_ ast_FunctionDeclList;
typedef struct ast_ImportDeclList_ ast_ImportDeclList;
typedef struct ast_TemplateInstance_ ast_TemplateInstance;
typedef struct ast_TemplateFunction_ ast_TemplateFunction;
typedef struct ast_InstanceTable_ ast_InstanceTable;
typedef struct ast_Instantiator_ ast_Instantiator;
typedef struct ast_Module_ ast_Module;
typedef struct ast_PointerPoolSlot_ ast_PointerPoolSlot;
typedef struct ast_PointerPool_ ast_PointerPool;
typedef struct ast_StaticAssertList_ ast_StaticAssertList;
typedef struct ast_Stat_ ast_Stat;
typedef struct ast_Stats_ ast_Stats;
typedef struct ast_StringTypeSlot_ ast_StringTypeSlot;
typedef struct ast_StringTypePool_ ast_StringTypePool;
typedef struct ast_SymbolTable_ ast_SymbolTable;
typedef struct ast_Globals_ ast_Globals;

typedef enum {
   ast_DeclKind_Function,
   ast_DeclKind_Import,
   ast_DeclKind_StructType,
   ast_DeclKind_EnumType,
   ast_DeclKind_EnumConstant,
   ast_DeclKind_FunctionType,
   ast_DeclKind_AliasType,
   ast_DeclKind_Variable,
   _ast_DeclKind_max = 255
} __attribute__((packed)) ast_DeclKind;

typedef enum {
   ast_DeclCheckState_Unchecked,
   ast_DeclCheckState_InProgress,
   ast_DeclCheckState_Checked,
   _ast_DeclCheckState_max = 255
} __attribute__((packed)) ast_DeclCheckState;

struct ast_DeclBits_ {
   uint32_t kind : 8;
   uint32_t check_state : 2;
   uint32_t is_public : 1;
   uint32_t is_used : 1;
   uint32_t is_used_public : 1;
   uint32_t has_attr : 1;
   uint32_t attr_export : 1;
   uint32_t attr_unused : 1;
   uint32_t is_external : 1;
   uint32_t is_generated : 1;
};

#define ast_NumDeclBits 18
struct ast_ImportDeclBits_ {
   uint32_t  : 18;
   uint32_t is_local : 1;
};

struct ast_FunctionDeclBits_ {
   uint32_t  : 18;
   uint32_t is_variadic : 1;
   uint32_t has_prefix : 1;
   uint32_t call_kind : 2;
   uint32_t has_return : 1;
   uint32_t attr_unused_params : 1;
   uint32_t attr_noreturn : 1;
   uint32_t attr_inline : 1;
   uint32_t attr_weak : 1;
   uint32_t attr_constructor : 1;
   uint32_t attr_destructor : 1;
   uint32_t attr_pure : 1;
   uint32_t is_template : 1;
   uint32_t is_type : 1;
};

struct ast_StructTypeDeclBits_ {
   uint32_t  : 18;
   uint32_t is_struct : 1;
   uint32_t is_global : 1;
   uint32_t attr_packed : 1;
   uint32_t attr_opaque : 1;
   uint32_t attr_notypedef : 1;
   uint32_t size_analysed : 1;
};

struct ast_EnumTypeDeclBits_ {
   uint32_t  : 18;
   uint32_t is_incremental : 1;
   uint32_t num_constants : 12;
};

struct ast_EnumConstantDeclBits_ {
   uint32_t  : 18;
   uint32_t has_init : 1;
   uint32_t enum_index : 13;
};

struct ast_VarDeclBits_ {
   uint32_t  : 18;
   uint32_t kind : 3;
   uint32_t has_init_or_bitfield : 1;
   uint32_t has_local : 1;
   uint32_t attr_weak : 1;
   uint32_t addr_used : 1;
   uint32_t auto_file : 1;
   uint32_t auto_line : 1;
   uint32_t printf_format : 1;
};

struct ast_QualType_ {
   size_t ptr;
};

struct ast_Decl_ {
   union {
      ast_DeclBits declBits;
      ast_ImportDeclBits importDeclBits;
      ast_FunctionDeclBits functionDeclBits;
      ast_StructTypeDeclBits structTypeDeclBits;
      ast_EnumTypeDeclBits enumTypeDeclBits;
      ast_EnumConstantDeclBits enumConstantDeclBits;
      ast_VarDeclBits varDeclBits;
      uint32_t bits;
   };
   src_loc_SrcLoc loc;
   uint32_t name_idx;
   uint32_t ast_idx;
   ast_QualType qt;
};

static const char* ast_declCheckState_names[3] = { "unchecked", "in-progress", "checked" };

static const char* ast_declKind_names[8] = {
   "FunctionDecl",
   "ImportDecl",
   "StructTypeDecl",
   "EnumTypeDecl",
   "EnumConstantDecl",
   "FunctionType",
   "AliasTypeDecl",
   "VarDecl"
};

static void ast_Decl_init(ast_Decl* d, ast_DeclKind k, uint32_t name_idx, src_loc_SrcLoc loc, bool is_public, ast_QualType qt, uint32_t ast_idx);
static ast_DeclKind ast_Decl_getKind(const ast_Decl* d);
static ast_DeclCheckState ast_Decl_getCheckState(const ast_Decl* d);
static void ast_Decl_setCheckState(ast_Decl* d, ast_DeclCheckState s);
static bool ast_Decl_isChecked(const ast_Decl* d);
static void ast_Decl_setChecked(ast_Decl* d);
static void ast_Decl_setHasAttr(ast_Decl* d);
static bool ast_Decl_hasAttr(const ast_Decl* d);
static void ast_Decl_setAttrExport(ast_Decl* d);
static void ast_Decl_setExportedIfPublic(ast_Decl* d);
static bool ast_Decl_isExported(const ast_Decl* d);
static void ast_Decl_setAttrUnused(ast_Decl* d);
static bool ast_Decl_hasAttrUnused(const ast_Decl* d);
static bool ast_Decl_isStructType(const ast_Decl* d);
static bool ast_Decl_isImport(const ast_Decl* d);
static bool ast_Decl_isEnum(const ast_Decl* d);
static bool ast_Decl_isEnumConstant(const ast_Decl* d);
static bool ast_Decl_isFunction(const ast_Decl* d);
static bool ast_Decl_isFunctionType(const ast_Decl* d);
static bool ast_Decl_isVariable(const ast_Decl* d);
static const char* ast_Decl_getName(const ast_Decl* d);
static uint32_t ast_Decl_getNameIdx(const ast_Decl* d);
static const char* ast_Decl_getModuleName(const ast_Decl* d);
static src_loc_SrcLoc ast_Decl_getLoc(const ast_Decl* d);
static ast_QualType ast_Decl_getType(const ast_Decl* d);
static void ast_Decl_setType(ast_Decl* d, ast_QualType qt);
static ast_AST* ast_Decl_getAST(const ast_Decl* d);
static uint32_t ast_Decl_getASTIdx(const ast_Decl* d);
static ast_Module* ast_Decl_getModule(const ast_Decl* d);
static bool ast_Decl_isPublic(const ast_Decl* d);
static bool ast_Decl_isUsed(const ast_Decl* d);
static bool ast_Decl_isUsedPublic(const ast_Decl* d);
static void ast_Decl_setUsed(ast_Decl* d);
static void ast_Decl_setUsedPublic(ast_Decl* d);
static bool ast_Decl_isExternal(const ast_Decl* d);
static void ast_Decl_setExternal(ast_Decl* d);
static bool ast_Decl_isGenerated(const ast_Decl* d);
static void ast_Decl_setGenerated(ast_Decl* d);
static void ast_Decl_clearGenerated(ast_Decl* d);
static void ast_Decl_dump(const ast_Decl* d);
static bool ast_Decl_isTypeDecl(const ast_Decl* d);
static bool ast_Decl_isVarDecl(const ast_Decl* d);
static const char* ast_Decl_getKindName(const ast_Decl* d);
static const char* ast_Decl_getCName(const ast_Decl* d);
static const char* ast_Decl_getSection(const ast_Decl* d);
static const char* ast_Decl_getFullName(const ast_Decl* d);
static void ast_Decl_print(const ast_Decl* d, string_buffer_Buf* out, uint32_t indent);
static void ast_Decl_printKind(const ast_Decl* d, string_buffer_Buf* out, uint32_t indent, bool print_type);
static void ast_Decl_printName(const ast_Decl* d, string_buffer_Buf* out);
static void ast_Decl_printBits(const ast_Decl* d, string_buffer_Buf* out);
static void ast_Decl_printAttrs(const ast_Decl* d, string_buffer_Buf* out);
static void ast_Decl_printUsed(const ast_Decl* d, string_buffer_Buf* out);
struct ast_TypeRefBits_ {
   uint32_t is_const : 1;
   uint32_t is_volatile : 1;
   uint32_t num_ptrs : 2;
   uint32_t num_arrays : 2;
   uint32_t incr_array : 1;
   uint32_t is_user : 1;
   uint32_t has_prefix : 1;
   uint32_t builtin_kind : 4;
};

struct ast_Ref_ {
   src_loc_SrcLoc loc;
   uint32_t name_idx;
   ast_Decl* decl;
};

struct ast_TypeRef_ {
   union {
      ast_TypeRefBits flags;
      uint32_t flagBits;
   };
   union {
      uint32_t dest;
      src_loc_SrcLoc loc;
   };
   ast_Ref refs[0];
};

struct ast_AliasTypeDecl_ {
   ast_Decl parent;
   ast_TypeRef typeRef;
};

static ast_AliasTypeDecl* ast_AliasTypeDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, const ast_TypeRefHolder* ref);
static ast_Decl* ast_AliasTypeDecl_asDecl(ast_AliasTypeDecl* d);
static ast_TypeRef* ast_AliasTypeDecl_getTypeRef(ast_AliasTypeDecl* d);
static void ast_AliasTypeDecl_print(const ast_AliasTypeDecl* d, string_buffer_Buf* out, uint32_t indent);
struct ast_ArrayValue_ {
   uint32_t name_idx;
   src_loc_SrcLoc loc;
   ast_Expr* value;
};

static ast_ArrayValue* ast_ArrayValue_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, ast_Expr* value);
static uint32_t ast_ArrayValue_getNameIdx(const ast_ArrayValue* d);
static src_loc_SrcLoc ast_ArrayValue_getLoc(const ast_ArrayValue* d);
static ast_Expr* ast_ArrayValue_getValue(const ast_ArrayValue* d);
static void ast_ArrayValue_print(const ast_ArrayValue* d, string_buffer_Buf* out);
#define ast_NumStmtBits 5
struct ast_StmtBits_ {
   uint32_t kind : 5;
};

struct ast_AssertStmtBits_ {
   uint32_t  : 5;
   uint32_t is_pointer : 1;
};

struct ast_AsmStmtBits_ {
   uint32_t  : 5;
   uint32_t is_basic : 1;
   uint32_t is_volatile : 1;
};

struct ast_ReturnStmtBits_ {
   uint32_t  : 5;
   uint32_t has_value : 1;
};

struct ast_SwitchStmtBits_ {
   uint32_t  : 5;
   uint32_t is_sswitch : 1;
   uint32_t num_cases : 26;
};

struct ast_CompoundStmtBits_ {
   uint32_t  : 5;
   uint32_t count : 27;
};

struct ast_ExprBits_ {
   uint32_t  : 5;
   uint32_t kind : 8;
   uint32_t is_ctv : 1;
   uint32_t is_ctc : 1;
   uint32_t valtype : 2;
   uint32_t has_effect : 1;
};

struct ast_IfStmtBits_ {
   uint32_t  : 5;
   uint32_t has_else : 1;
};

#define ast_NumExprBits (ast_NumStmtBits + 13)
struct ast_BuiltinExprBits_ {
   uint32_t  : 18;
   uint32_t kind : 3;
};

struct ast_BooleanLiteralBits_ {
   uint32_t  : 18;
   uint32_t value : 1;
};

struct ast_CharLiteralBits_ {
   uint32_t  : 18;
   uint32_t value : 8;
   uint32_t radix : 5;
};

struct ast_IdentifierExprBits_ {
   uint32_t  : 18;
   uint32_t has_decl : 1;
   uint32_t kind : 4;
   uint32_t is_case_range : 1;
};

struct ast_MemberExprBits_ {
   uint32_t  : 18;
   uint32_t kind : 4;
   uint32_t num_refs : 3;
   uint32_t num_decls : 3;
   uint32_t has_expr : 1;
   uint32_t is_struct_func : 1;
   uint32_t is_static_sf : 1;
   uint32_t is_const_base : 1;
};

struct ast_IntegerLiteralBits_ {
   uint32_t  : 18;
   uint32_t radix : 5;
   uint32_t is_signed : 1;
};

struct ast_UnaryOperatorBits_ {
   uint32_t  : 18;
   uint32_t kind : 4;
};

struct ast_BinaryOperatorBits_ {
   uint32_t  : 18;
   uint32_t kind : 5;
};

struct ast_BitOffsetExprBits_ {
   uint32_t  : 18;
   uint32_t width : 8;
};

struct ast_CallExprBits_ {
   uint32_t  : 18;
   uint32_t calls_struct_func : 1;
   uint32_t calls_static_sf : 1;
   uint32_t is_template_call : 1;
   uint32_t printf_format : 4;
   uint32_t change_format : 1;
   uint32_t has_auto_args : 1;
};

struct ast_InitListExprBits_ {
   uint32_t  : 18;
   uint32_t num_values : 14;
};

struct ast_ImplicitCastBits_ {
   uint32_t  : 18;
   uint32_t kind : 3;
};

struct ast_Stmt_ {
   union {
      ast_StmtBits stmtBits;
      ast_AssertStmtBits assertStmtBits;
      ast_AsmStmtBits asmStmtBits;
      ast_ReturnStmtBits returnStmtBits;
      ast_SwitchStmtBits switchStmtBits;
      ast_CompoundStmtBits compoundStmtBits;
      ast_ExprBits exprBits;
      ast_IfStmtBits ifStmtBits;
      ast_BuiltinExprBits builtinExprBits;
      ast_BooleanLiteralBits booleanLiteralBits;
      ast_CharLiteralBits charLiteralBits;
      ast_IdentifierExprBits identifierExprBits;
      ast_MemberExprBits memberExprBits;
      ast_IntegerLiteralBits integerLiteralBits;
      ast_UnaryOperatorBits unaryOperatorBits;
      ast_BinaryOperatorBits binaryOperatorBits;
      ast_BitOffsetExprBits bitOffsetBits;
      ast_CallExprBits callExprBits;
      ast_InitListExprBits initListExprBits;
      ast_ImplicitCastBits implicitCastBits;
      uint32_t bits;
   };
};

struct ast_DeclStmt_ {
   ast_Stmt parent;
   ast_VarDecl* decl;
};

static ast_DeclStmt* ast_DeclStmt_create(ast_context_Context* c, ast_VarDecl* decl);
static ast_Stmt* ast_DeclStmt_instantiate(ast_DeclStmt* s, ast_Instantiator* inst);
static ast_VarDecl* ast_DeclStmt_getDecl(const ast_DeclStmt* d);
static void ast_DeclStmt_print(const ast_DeclStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_Value_ {
   bool is_signed;
   union {
      uint64_t uvalue;
      int64_t svalue;
   };
};

struct ast_EnumConstantDecl_ {
   ast_Decl parent;
   ast_Value value;
   ast_Expr* init[0];
};

static ast_EnumConstantDecl* ast_EnumConstantDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, ast_Expr* initValue);
static ast_Decl* ast_EnumConstantDecl_asDecl(ast_EnumConstantDecl* d);
static ast_Value ast_EnumConstantDecl_getValue(const ast_EnumConstantDecl* d);
static void ast_EnumConstantDecl_setValue(ast_EnumConstantDecl* d, ast_Value value);
static void ast_EnumConstantDecl_setIndex(ast_EnumConstantDecl* d, uint32_t index);
static uint32_t ast_EnumConstantDecl_getIndex(const ast_EnumConstantDecl* d);
static ast_Expr* ast_EnumConstantDecl_getInit(const ast_EnumConstantDecl* d);
static ast_Expr** ast_EnumConstantDecl_getInit2(ast_EnumConstantDecl* d);
static ast_EnumTypeDecl* ast_EnumConstantDecl_getEnum(const ast_EnumConstantDecl* d);
static void ast_EnumConstantDecl_print(const ast_EnumConstantDecl* d, string_buffer_Buf* out, uint32_t indent);
struct ast_EnumTypeDecl_ {
   ast_Decl parent;
   ast_QualType implType;
   ast_EnumConstantDecl* constants[0];
   ast_EnumConstantDecl** incr_constants[0];
};

static ast_EnumTypeDecl* ast_EnumTypeDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, ast_QualType implType, bool is_incremental, ast_EnumConstantDecl** constants, uint32_t num_constants);
static void ast_EnumTypeDecl_setIncrMembers(ast_EnumTypeDecl* d, ast_Decl** constants, uint32_t num_constants);
static ast_QualType ast_EnumTypeDecl_getImplType(const ast_EnumTypeDecl* d);
static ast_Decl* ast_EnumTypeDecl_asDecl(ast_EnumTypeDecl* d);
static bool ast_EnumTypeDecl_isIncremental(const ast_EnumTypeDecl* d);
static uint32_t ast_EnumTypeDecl_getNumConstants(const ast_EnumTypeDecl* d);
static ast_EnumConstantDecl** ast_EnumTypeDecl_getConstants(ast_EnumTypeDecl* d);
static void ast_EnumTypeDecl_setIncrConstants(ast_EnumTypeDecl* d, ast_context_Context* c, ast_IdentifierExpr** constants, uint32_t count);
static ast_EnumConstantDecl* ast_EnumTypeDecl_findConstant(ast_EnumTypeDecl* d, uint32_t name_idx);
static ast_EnumConstantDecl* ast_EnumTypeDecl_findConstantIdx(ast_EnumTypeDecl* d, uint32_t name_idx, uint32_t* idx);
static ast_EnumConstantDecl* ast_EnumTypeDecl_getConstant(const ast_EnumTypeDecl* d, uint32_t idx);
static void ast_EnumTypeDecl_print(ast_EnumTypeDecl* d, string_buffer_Buf* out, uint32_t indent);
typedef enum {
   ast_CallKind_Invalid,
   ast_CallKind_Normal,
   ast_CallKind_StructFunc,
   ast_CallKind_StaticStructFunc,
   _ast_CallKind_max = 255
} __attribute__((packed)) ast_CallKind;

struct ast_FunctionDecl_ {
   ast_Decl parent;
   ast_CompoundStmt* body;
   ast_QualType rt;
   uint8_t num_params;
   uint8_t attr_printf_arg;
   uint16_t instance_idx;
   uint32_t template_name;
   src_loc_SrcLoc template_loc;
   uint32_t num_auto_args : 4;
   ast_TypeRef rtype;
};

static const char* ast_callKind_names[4] = { "Invalid", "Normal", "SF", "SSF" };

static ast_FunctionDecl* ast_FunctionDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, const ast_TypeRefHolder* rtype, const ast_Ref* prefix, ast_VarDecl** params, uint32_t num_params, bool is_variadic, bool is_type);
static ast_FunctionDecl* ast_FunctionDecl_createTemplate(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, const ast_TypeRefHolder* rtype, uint32_t template_name, src_loc_SrcLoc template_loc, ast_VarDecl** params, uint32_t num_params, bool is_variadic);
static ast_FunctionDecl* ast_FunctionDecl_instantiate(const ast_FunctionDecl* fd, ast_Instantiator* inst);
static void ast_FunctionDecl_setBody(ast_FunctionDecl* d, ast_CompoundStmt* body);
static ast_CompoundStmt* ast_FunctionDecl_getBody(const ast_FunctionDecl* d);
static bool ast_FunctionDecl_isType(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setRType(ast_FunctionDecl* d, ast_QualType rt);
static ast_QualType ast_FunctionDecl_getRType(const ast_FunctionDecl* d);
static bool ast_FunctionDecl_hasReturn(const ast_FunctionDecl* d);
static ast_Decl* ast_FunctionDecl_asDecl(ast_FunctionDecl* d);
static ast_TypeRef* ast_FunctionDecl_getReturnTypeRef(ast_FunctionDecl* d);
static bool ast_FunctionDecl_hasPrefix(const ast_FunctionDecl* d);
static bool ast_FunctionDecl_isTemplate(const ast_FunctionDecl* d);
static uint32_t ast_FunctionDecl_getTemplateNameIdx(const ast_FunctionDecl* d);
static src_loc_SrcLoc ast_FunctionDecl_getTemplateLoc(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setTemplateInstanceIdx(ast_FunctionDecl* d, uint16_t idx);
static uint16_t ast_FunctionDecl_getTemplateInstanceIdx(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setInstanceName(ast_FunctionDecl* d, uint32_t name_idx);
static ast_Ref* ast_FunctionDecl_getPrefix(const ast_FunctionDecl* d);
static const char* ast_FunctionDecl_getPrefixName(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setCallKind(ast_FunctionDecl* d, ast_CallKind kind);
static ast_CallKind ast_FunctionDecl_getCallKind(const ast_FunctionDecl* d);
static bool ast_FunctionDecl_isVariadic(const ast_FunctionDecl* d);
static uint32_t ast_FunctionDecl_getNumParams(const ast_FunctionDecl* d);
static ast_VarDecl** ast_FunctionDecl_getParams(const ast_FunctionDecl* d);
static uint32_t ast_FunctionDecl_getNumAutoArgs(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setNumAutoArgs(ast_FunctionDecl* d, uint32_t num);
static void ast_FunctionDecl_setAttrUnusedParams(ast_FunctionDecl* d);
static bool ast_FunctionDecl_hasAttrUnusedParams(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setAttrNoReturn(ast_FunctionDecl* d);
static bool ast_FunctionDecl_hasAttrNoReturn(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setAttrInline(ast_FunctionDecl* d);
static bool ast_FunctionDecl_hasAttrInline(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setAttrWeak(ast_FunctionDecl* d);
static bool ast_FunctionDecl_hasAttrWeak(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setAttrConstructor(ast_FunctionDecl* d);
static bool ast_FunctionDecl_hasAttrConstructor(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setAttrDestructor(ast_FunctionDecl* d);
static bool ast_FunctionDecl_hasAttrDestructor(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setAttrPure(ast_FunctionDecl* d);
static bool ast_FunctionDecl_hasAttrPure(const ast_FunctionDecl* d);
static void ast_FunctionDecl_setAttrPrintf(ast_FunctionDecl* d, uint8_t arg);
static bool ast_FunctionDecl_hasAttrPrintf(const ast_FunctionDecl* d);
static uint8_t ast_FunctionDecl_getAttrPrintf(const ast_FunctionDecl* d);
static const char* ast_FunctionDecl_getDiagKind(const ast_FunctionDecl* d);
static void ast_FunctionDecl_print(const ast_FunctionDecl* d, string_buffer_Buf* out, uint32_t indent);
static void ast_FunctionDecl_printType(const ast_FunctionDecl* d, string_buffer_Buf* out);
struct ast_FunctionTypeDecl_ {
   ast_Decl parent;
   ast_FunctionDecl* func;
};

static ast_FunctionTypeDecl* ast_FunctionTypeDecl_create(ast_context_Context* c, ast_FunctionDecl* func);
static ast_Decl* ast_FunctionTypeDecl_asDecl(ast_FunctionTypeDecl* t);
static ast_FunctionDecl* ast_FunctionTypeDecl_getDecl(const ast_FunctionTypeDecl* d);
static void ast_FunctionTypeDecl_print(const ast_FunctionTypeDecl* d, string_buffer_Buf* out, uint32_t indent);
struct ast_ImportDecl_ {
   ast_Decl parent;
   uint32_t alias_idx;
   src_loc_SrcLoc alias_loc;
   ast_Module* dest;
};

static ast_ImportDecl* ast_ImportDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, uint32_t alias_name, src_loc_SrcLoc alias_loc, uint32_t ast_idx, bool is_local);
static ast_Decl* ast_ImportDecl_asDecl(ast_ImportDecl* d);
static const char* ast_ImportDecl_getAliasName(const ast_ImportDecl* d);
static uint32_t ast_ImportDecl_getAliasNameIdx(const ast_ImportDecl* d);
static uint32_t ast_ImportDecl_getImportNameIdx(const ast_ImportDecl* d);
static src_loc_SrcLoc ast_ImportDecl_getLoc(const ast_ImportDecl* d);
static void ast_ImportDecl_setDest(ast_ImportDecl* d, ast_Module* mod);
static ast_Module* ast_ImportDecl_getDest(const ast_ImportDecl* d);
static bool ast_ImportDecl_isLocal(const ast_ImportDecl* d);
static void ast_ImportDecl_print(const ast_ImportDecl* d, string_buffer_Buf* out, uint32_t indent);
struct ast_StaticAssert_ {
   uint32_t ast_idx;
   ast_Expr* lhs;
   ast_Expr* rhs;
};

static ast_StaticAssert* ast_StaticAssert_create(ast_context_Context* c, uint32_t ast_idx, src_loc_SrcLoc loc, ast_Expr* lhs, ast_Expr* rhs);
static ast_AST* ast_StaticAssert_getAST(const ast_StaticAssert* d);
static ast_Expr* ast_StaticAssert_getLhs(const ast_StaticAssert* d);
static ast_Expr* ast_StaticAssert_getRhs(const ast_StaticAssert* d);
static void ast_StaticAssert_print(const ast_StaticAssert* d, string_buffer_Buf* out, uint32_t indent);
struct ast_StructLayout_ {
   uint32_t size;
   uint32_t alignment;
   uint32_t attr_alignment;
   uint32_t member_offsets[0];
};

struct ast_StructTypeDecl_ {
   ast_Decl parent;
   uint32_t num_members;
   uint32_t num_struct_functions;
   ast_FunctionDecl** struct_functions;
   ast_Decl* members[0];
};

static ast_StructTypeDecl* ast_StructTypeDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, bool is_struct, bool is_global, ast_Decl** members, uint32_t num_members);
static ast_Decl* ast_StructTypeDecl_asDecl(ast_StructTypeDecl* d);
static uint32_t ast_StructTypeDecl_getNumMembers(const ast_StructTypeDecl* d);
static ast_Decl** ast_StructTypeDecl_getMembers(ast_StructTypeDecl* d);
static bool ast_StructTypeDecl_isStruct(const ast_StructTypeDecl* d);
static bool ast_StructTypeDecl_isUnion(const ast_StructTypeDecl* d);
static const ast_FunctionDecl** ast_StructTypeDecl_getStructFunctions(const ast_StructTypeDecl* d);
static uint32_t ast_StructTypeDecl_getNumStructFunctions(const ast_StructTypeDecl* d);
static ast_StructLayout* ast_StructTypeDecl_getLayoutPtr(const ast_StructTypeDecl* d);
static void ast_StructTypeDecl_setMemberOffset(ast_StructTypeDecl* d, uint32_t member_idx, uint32_t offset);
static uint32_t ast_StructTypeDecl_getMemberOffset(const ast_StructTypeDecl* d, uint32_t member_idx);
static uint32_t ast_StructTypeDecl_getSize(const ast_StructTypeDecl* d);
static void ast_StructTypeDecl_setSizeAlignment(ast_StructTypeDecl* d, uint32_t size, uint32_t alignment);
static uint32_t ast_StructTypeDecl_getAlignment(const ast_StructTypeDecl* d);
static uint32_t ast_StructTypeDecl_getAttrAlignment(const ast_StructTypeDecl* d);
static void ast_StructTypeDecl_setAttrAlignment(ast_StructTypeDecl* d, uint32_t alignment);
static void ast_StructTypeDecl_setPacked(ast_StructTypeDecl* d);
static bool ast_StructTypeDecl_isPacked(const ast_StructTypeDecl* d);
static void ast_StructTypeDecl_setOpaque(ast_StructTypeDecl* d);
static bool ast_StructTypeDecl_isOpaque(const ast_StructTypeDecl* d);
static bool ast_StructTypeDecl_isGlobal(const ast_StructTypeDecl* d);
static void ast_StructTypeDecl_setAttrNoTypeDef(ast_StructTypeDecl* d);
static bool ast_StructTypeDecl_hasAttrNoTypeDef(const ast_StructTypeDecl* d);
static void ast_StructTypeDecl_setStructFunctions(ast_StructTypeDecl* d, ast_context_Context* c, ast_FunctionDecl** funcs, uint32_t count);
static ast_Decl* ast_StructTypeDecl_findAny(const ast_StructTypeDecl* s, uint32_t name_idx);
static ast_Decl* ast_StructTypeDecl_findMember(const ast_StructTypeDecl* s, uint32_t name_idx, uint32_t* offset);
static void ast_StructTypeDecl_print(const ast_StructTypeDecl* d, string_buffer_Buf* out, uint32_t indent);
static bool ast_Value_isNegative(const ast_Value* v);
static bool ast_Value_equals(const ast_Value* v1, const ast_Value* v2);
static bool ast_Value_less_than(const ast_Value* v1, const ast_Value* v2);
static ast_Value ast_Value_minus(const ast_Value* v1, const ast_Value* v2);
static void ast_Value_mask(ast_Value* v, uint32_t width);
static bool ast_Value_ugt(const ast_Value* v1, uint64_t max);
static void ast_Value_incr(ast_Value* v);
static const char* ast_Value_str(const ast_Value* v);
typedef enum {
   ast_VarDeclKind_GlobalVar,
   ast_VarDeclKind_LocalVar,
   ast_VarDeclKind_FunctionParam,
   ast_VarDeclKind_StructMember,
   _ast_VarDeclKind_max = 255
} __attribute__((packed)) ast_VarDeclKind;

struct ast_VarDecl_ {
   ast_Decl parent;
   ast_TypeRef typeRef;
};

static const char* ast_varDeclNames[4] = { " global", " local", " parameter", " member" };

static ast_VarDecl* ast_VarDecl_create(ast_context_Context* c, ast_VarDeclKind kind, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* ref, uint32_t ast_idx, src_loc_SrcLoc assignLoc, ast_Expr* initValue);
static ast_VarDecl* ast_VarDecl_createStructMember(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* ref, uint32_t ast_idx, ast_Expr* bitfield);
static ast_VarDecl* ast_VarDecl_instantiate(const ast_VarDecl* vd, ast_Instantiator* inst);
static ast_Decl* ast_VarDecl_asDecl(ast_VarDecl* d);
static ast_VarDeclKind ast_VarDecl_getKind(const ast_VarDecl* d);
static bool ast_VarDecl_isGlobal(const ast_VarDecl* d);
static bool ast_VarDecl_isLocal(const ast_VarDecl* d);
static bool ast_VarDecl_isParameter(const ast_VarDecl* d);
static bool ast_VarDecl_isStructMember(const ast_VarDecl* d);
static bool ast_VarDecl_isAddrUsed(const ast_VarDecl* d);
static void ast_VarDecl_setAddrUsed(ast_VarDecl* d);
static ast_TypeRef* ast_VarDecl_getTypeRef(ast_VarDecl* d);
static src_loc_SrcLoc ast_VarDecl_getAssignLoc(const ast_VarDecl* d);
static bool ast_VarDecl_hasInit(const ast_VarDecl* d);
static ast_Expr* ast_VarDecl_getInit(const ast_VarDecl* d);
static ast_Expr** ast_VarDecl_getInit2(ast_VarDecl* d);
static void ast_VarDecl_setInit(ast_VarDecl* d, ast_Expr* initValue);
static ast_Expr* ast_VarDecl_getBitfield(const ast_VarDecl* d);
static bool ast_VarDecl_hasLocalQualifier(const ast_VarDecl* d);
static void ast_VarDecl_setLocal(ast_VarDecl* d, bool has_local);
static void ast_VarDecl_setAttrWeak(ast_VarDecl* d);
static bool ast_VarDecl_hasAttrWeak(const ast_VarDecl* d);
static void ast_VarDecl_setAttrAutoFile(ast_VarDecl* d);
static bool ast_VarDecl_hasAttrAutoFile(const ast_VarDecl* d);
static void ast_VarDecl_setAttrAutoLine(ast_VarDecl* d);
static bool ast_VarDecl_hasAttrAutoLine(const ast_VarDecl* d);
static bool ast_VarDecl_hasAutoAttr(const ast_VarDecl* d);
static void ast_VarDecl_setPrintfFormat(ast_VarDecl* d);
static bool ast_VarDecl_hasPrintfFormat(const ast_VarDecl* d);
static void ast_VarDecl_print(const ast_VarDecl* d, string_buffer_Buf* out, uint32_t indent);
static void ast_VarDecl_printType(const ast_VarDecl* d, string_buffer_Buf* out);
typedef enum {
   ast_StmtKind_Return,
   ast_StmtKind_Expr,
   ast_StmtKind_If,
   ast_StmtKind_While,
   ast_StmtKind_Do,
   ast_StmtKind_For,
   ast_StmtKind_Switch,
   ast_StmtKind_Break,
   ast_StmtKind_Continue,
   ast_StmtKind_Fallthrough,
   ast_StmtKind_Label,
   ast_StmtKind_Goto,
   ast_StmtKind_Compound,
   ast_StmtKind_Decl,
   ast_StmtKind_Asm,
   ast_StmtKind_Assert,
   _ast_StmtKind_max = 255
} __attribute__((packed)) ast_StmtKind;

static const char* ast_stmtKind_names[16] = {
   "ReturnStmt",
   "ExprStmt",
   "IfStmt",
   "WhileStmt",
   "DoStmt",
   "ForStmt",
   "SwitchStmt",
   "BreakStmt",
   "ContinueStmt",
   "FallthroughStmt",
   "LabelStmt",
   "GotoStmt",
   "CompoundStmt",
   "DeclStmt",
   "Asm",
   "AssertStmt"
};

static void ast_Stmt_init(ast_Stmt* s, ast_StmtKind k);
static ast_Stmt* ast_Stmt_instantiate(ast_Stmt* s, ast_Instantiator* inst);
static ast_StmtKind ast_Stmt_getKind(const ast_Stmt* s);
static bool ast_Stmt_isReturn(const ast_Stmt* s);
static bool ast_Stmt_isExpr(const ast_Stmt* s);
static bool ast_Stmt_isCompound(const ast_Stmt* s);
static bool ast_Stmt_isFallthrough(const ast_Stmt* s);
static bool ast_Stmt_isDecl(const ast_Stmt* s);
static src_loc_SrcLoc ast_Stmt_getLoc(const ast_Stmt* s);
static void ast_Stmt_dump(const ast_Stmt* s);
static void ast_Stmt_print(const ast_Stmt* s, string_buffer_Buf* out, uint32_t indent);
static void ast_Stmt_printKind(const ast_Stmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_AsmStmt_ {
   ast_Stmt parent;
   src_loc_SrcLoc loc;
   ast_StringLiteral* asm_string;
   uint8_t num_constraints;
   uint8_t num_exprs;
   uint8_t num_clobbers;
   uint8_t num_outputs;
   uint8_t num_inputs;
   uint8_t pad[2];
   ast_Expr* constraints[0];
   ast_Expr* exprs[0];
   ast_Expr* clobbers[0];
   uint32_t names[0];
};

static ast_AsmStmt* ast_AsmStmt_create(ast_context_Context* c, src_loc_SrcLoc loc, bool is_basic, bool is_volatile, uint32_t num_outputs, uint32_t num_inputs, const uint32_t* names, ast_ExprList* constraints, ast_ExprList* exprs, ast_ExprList* clobbers, ast_Expr* str);
static ast_Stmt* ast_AsmStmt_instantiate(ast_AsmStmt* s, ast_Instantiator* inst);
static src_loc_SrcLoc ast_AsmStmt_getLoc(const ast_AsmStmt* s);
static bool ast_AsmStmt_isVolatile(const ast_AsmStmt* s);
static uint32_t ast_AsmStmt_getNumConstraints(const ast_AsmStmt* s);
static uint32_t ast_AsmStmt_getNumClobbers(const ast_AsmStmt* s);
static uint32_t ast_AsmStmt_getNumExprs(const ast_AsmStmt* s);
static uint32_t ast_AsmStmt_getNumOutputs(const ast_AsmStmt* s);
static uint32_t ast_AsmStmt_getNumInputs(const ast_AsmStmt* s);
static ast_StringLiteral* ast_AsmStmt_getString(const ast_AsmStmt* s);
static const ast_Expr** ast_AsmStmt_getConstraints(const ast_AsmStmt* s);
static ast_Expr** ast_AsmStmt_getExprs(const ast_AsmStmt* s);
static ast_Expr** ast_AsmStmt_getClobbers(const ast_AsmStmt* s);
static uint32_t* ast_AsmStmt_getNames(const ast_AsmStmt* s);
static void ast_AsmStmt_print(const ast_AsmStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_AssertStmt_ {
   ast_Stmt parent;
   src_loc_SrcLoc loc;
   ast_Expr* inner;
};

static ast_AssertStmt* ast_AssertStmt_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* inner);
static ast_Stmt* ast_AssertStmt_instantiate(ast_AssertStmt* s, ast_Instantiator* inst);
static src_loc_SrcLoc ast_AssertStmt_getLoc(const ast_AssertStmt* s);
static ast_Expr* ast_AssertStmt_getInner(const ast_AssertStmt* s);
static ast_Expr** ast_AssertStmt_getInner2(ast_AssertStmt* s);
static void ast_AssertStmt_setPointer(ast_AssertStmt* s);
static bool ast_AssertStmt_isPointer(const ast_AssertStmt* s);
static void ast_AssertStmt_print(const ast_AssertStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_BreakStmt_ {
   ast_Stmt parent;
   src_loc_SrcLoc loc;
};

static ast_BreakStmt* ast_BreakStmt_create(ast_context_Context* c, src_loc_SrcLoc loc);
static src_loc_SrcLoc ast_BreakStmt_getLoc(const ast_BreakStmt* s);
static void ast_BreakStmt_print(const ast_BreakStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_CompoundStmt_ {
   ast_Stmt parent;
   src_loc_SrcLoc endLoc;
   ast_Stmt* stmts[0];
};

static ast_CompoundStmt* ast_CompoundStmt_create(ast_context_Context* c, src_loc_SrcLoc end, ast_Stmt** stmts, uint32_t count);
static ast_CompoundStmt* ast_CompoundStmt_instantiate(ast_CompoundStmt* s, ast_Instantiator* inst);
static src_loc_SrcLoc ast_CompoundStmt_getEndLoc(const ast_CompoundStmt* s);
static uint32_t ast_CompoundStmt_getCount(const ast_CompoundStmt* s);
static ast_Stmt** ast_CompoundStmt_getStmts(ast_CompoundStmt* s);
static ast_Stmt* ast_CompoundStmt_getLastStmt(const ast_CompoundStmt* s);
static void ast_CompoundStmt_print(const ast_CompoundStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_ContinueStmt_ {
   ast_Stmt parent;
   src_loc_SrcLoc loc;
};

static ast_ContinueStmt* ast_ContinueStmt_create(ast_context_Context* c, src_loc_SrcLoc loc);
static src_loc_SrcLoc ast_ContinueStmt_getLoc(const ast_ContinueStmt* s);
static void ast_ContinueStmt_print(const ast_ContinueStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_DoStmt_ {
   ast_Stmt parent;
   ast_Stmt* cond;
   ast_Stmt* body;
};

static ast_DoStmt* ast_DoStmt_create(ast_context_Context* c, ast_Stmt* cond, ast_Stmt* body);
static ast_Stmt* ast_DoStmt_instantiate(ast_DoStmt* s, ast_Instantiator* inst);
static void ast_DoStmt_print(const ast_DoStmt* s, string_buffer_Buf* out, uint32_t indent);
static ast_Stmt* ast_DoStmt_getCond(const ast_DoStmt* s);
static ast_Stmt* ast_DoStmt_getBody(const ast_DoStmt* s);
struct ast_FallthroughStmt_ {
   ast_Stmt parent;
   src_loc_SrcLoc loc;
};

static ast_FallthroughStmt* ast_FallthroughStmt_create(ast_context_Context* c, src_loc_SrcLoc loc);
static src_loc_SrcLoc ast_FallthroughStmt_getLoc(const ast_FallthroughStmt* s);
static void ast_FallthroughStmt_print(const ast_FallthroughStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_ForStmt_ {
   ast_Stmt parent;
   src_loc_SrcLoc loc;
   ast_Stmt* init;
   ast_Expr* cond;
   ast_Expr* incr;
   ast_Stmt* body;
};

static ast_ForStmt* ast_ForStmt_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Stmt* init_, ast_Expr* cond, ast_Expr* incr, ast_Stmt* body);
static ast_Stmt* ast_ForStmt_instantiate(ast_ForStmt* s, ast_Instantiator* inst);
static src_loc_SrcLoc ast_ForStmt_getLoc(const ast_ForStmt* s);
static ast_Stmt* ast_ForStmt_getInit(const ast_ForStmt* s);
static ast_Expr* ast_ForStmt_getCond(const ast_ForStmt* s);
static ast_Expr* ast_ForStmt_getIncr(const ast_ForStmt* s);
static ast_Stmt* ast_ForStmt_getBody(const ast_ForStmt* s);
static ast_Stmt** ast_ForStmt_getInit2(ast_ForStmt* s);
static ast_Expr** ast_ForStmt_getCond2(ast_ForStmt* s);
static ast_Expr** ast_ForStmt_getIncr2(ast_ForStmt* s);
static ast_Stmt** ast_ForStmt_getBody2(ast_ForStmt* s);
static void ast_ForStmt_print(const ast_ForStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_GotoStmt_ {
   ast_Stmt parent;
   src_loc_SrcLoc loc;
   uint32_t name;
};

static ast_GotoStmt* ast_GotoStmt_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc);
static const char* ast_GotoStmt_getName(const ast_GotoStmt* g);
static uint32_t ast_GotoStmt_getNameIdx(const ast_GotoStmt* g);
static src_loc_SrcLoc ast_GotoStmt_getLoc(const ast_GotoStmt* g);
static void ast_GotoStmt_print(const ast_GotoStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_IfStmt_ {
   ast_Stmt parent;
   ast_Stmt* cond;
   ast_Stmt* then;
   ast_Stmt* else_stmt[0];
};

static ast_IfStmt* ast_IfStmt_create(ast_context_Context* c, ast_Stmt* cond, ast_Stmt* then, ast_Stmt* else_stmt);
static ast_Stmt* ast_IfStmt_instantiate(ast_IfStmt* s, ast_Instantiator* inst);
static ast_Stmt* ast_IfStmt_getCond(const ast_IfStmt* s);
static ast_Stmt** ast_IfStmt_getCond2(ast_IfStmt* s);
static ast_Stmt* ast_IfStmt_getThen(const ast_IfStmt* s);
static ast_Stmt* ast_IfStmt_getElse(const ast_IfStmt* s);
static void ast_IfStmt_print(const ast_IfStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_LabelStmt_ {
   ast_Stmt parent;
   src_loc_SrcLoc loc;
   uint32_t name;
};

static ast_LabelStmt* ast_LabelStmt_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc);
static const char* ast_LabelStmt_getName(const ast_LabelStmt* s);
static uint32_t ast_LabelStmt_getNameIdx(const ast_LabelStmt* s);
static src_loc_SrcLoc ast_LabelStmt_getLoc(const ast_LabelStmt* s);
static void ast_LabelStmt_print(const ast_LabelStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_ReturnStmt_ {
   ast_Stmt parent;
   src_loc_SrcLoc loc;
   ast_Expr* value[0];
};

static ast_ReturnStmt* ast_ReturnStmt_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* value);
static ast_Stmt* ast_ReturnStmt_instantiate(ast_ReturnStmt* s, ast_Instantiator* inst);
static src_loc_SrcLoc ast_ReturnStmt_getLoc(const ast_ReturnStmt* s);
static ast_Expr* ast_ReturnStmt_getValue(const ast_ReturnStmt* s);
static ast_Expr** ast_ReturnStmt_getValue2(ast_ReturnStmt* s);
static void ast_ReturnStmt_print(const ast_ReturnStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_SwitchCaseBits_ {
   uint32_t num_subcases : 8;
   uint32_t num_stmts : 10;
   uint32_t is_default : 1;
   uint32_t has_decls : 1;
};

struct ast_SwitchCase_ {
   union {
      ast_SwitchCaseBits bits;
      uint32_t allbits;
   };
   src_loc_SrcLoc loc;
   union {
      ast_Expr* cond;
      ast_IdentifierExpr* multi_cond[1];
   };
};

static ast_SwitchCase* ast_SwitchCase_create(ast_context_Context* c, src_loc_SrcLoc loc, bool is_default, ast_Expr* cond, ast_IdentifierExpr** multi, uint32_t num_multi, ast_Stmt** stmts, uint32_t numStmts);
static ast_SwitchCase* ast_SwitchCase_instantiate(ast_SwitchCase* s, ast_Instantiator* inst);
static uint32_t ast_SwitchCase_getNumStmts(const ast_SwitchCase* s);
static ast_Stmt** ast_SwitchCase_getStmts(const ast_SwitchCase* s);
static bool ast_SwitchCase_isDefault(const ast_SwitchCase* s);
static uint32_t ast_SwitchCase_numMulti(const ast_SwitchCase* s);
static bool ast_SwitchCase_hasDecls(const ast_SwitchCase* s);
static void ast_SwitchCase_setHasDecls(ast_SwitchCase* s);
static src_loc_SrcLoc ast_SwitchCase_getLoc(const ast_SwitchCase* s);
static ast_Expr* ast_SwitchCase_getCond(const ast_SwitchCase* s);
static ast_Expr** ast_SwitchCase_getCond2(ast_SwitchCase* s);
static ast_IdentifierExpr** ast_SwitchCase_getMultiCond(ast_SwitchCase* s);
static void ast_SwitchCase_print(const ast_SwitchCase* s, string_buffer_Buf* out, uint32_t indent);
struct ast_SwitchStmt_ {
   ast_Stmt parent;
   src_loc_SrcLoc loc;
   ast_Expr* cond;
   ast_SwitchCase* cases[0];
};

static ast_SwitchStmt* ast_SwitchStmt_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* cond, ast_SwitchCase** cases, uint32_t numCases, bool is_sswitch);
static ast_Stmt* ast_SwitchStmt_instantiate(ast_SwitchStmt* s, ast_Instantiator* inst);
static src_loc_SrcLoc ast_SwitchStmt_getLoc(const ast_SwitchStmt* s);
static ast_Expr* ast_SwitchStmt_getCond(const ast_SwitchStmt* s);
static ast_Expr** ast_SwitchStmt_getCond2(ast_SwitchStmt* s);
static bool ast_SwitchStmt_isSSwitch(const ast_SwitchStmt* s);
static uint32_t ast_SwitchStmt_getNumCases(const ast_SwitchStmt* s);
static ast_SwitchCase** ast_SwitchStmt_getCases(ast_SwitchStmt* s);
static void ast_SwitchStmt_print(const ast_SwitchStmt* s, string_buffer_Buf* out, uint32_t indent);
struct ast_WhileStmt_ {
   ast_Stmt parent;
   ast_Stmt* cond;
   ast_Stmt* body;
};

static ast_WhileStmt* ast_WhileStmt_create(ast_context_Context* c, ast_Stmt* cond, ast_Stmt* body);
static ast_Stmt* ast_WhileStmt_instantiate(ast_WhileStmt* s, ast_Instantiator* inst);
static void ast_WhileStmt_print(const ast_WhileStmt* s, string_buffer_Buf* out, uint32_t indent);
static ast_Stmt* ast_WhileStmt_getCond(const ast_WhileStmt* s);
static ast_Stmt** ast_WhileStmt_getCond2(ast_WhileStmt* s);
static ast_Stmt* ast_WhileStmt_getBody(const ast_WhileStmt* s);
typedef enum {
   ast_ExprKind_IntegerLiteral,
   ast_ExprKind_FloatLiteral,
   ast_ExprKind_BooleanLiteral,
   ast_ExprKind_CharLiteral,
   ast_ExprKind_StringLiteral,
   ast_ExprKind_Nil,
   ast_ExprKind_Identifier,
   ast_ExprKind_Type,
   ast_ExprKind_Call,
   ast_ExprKind_InitList,
   ast_ExprKind_FieldDesignatedInit,
   ast_ExprKind_ArrayDesignatedInit,
   ast_ExprKind_BinaryOperator,
   ast_ExprKind_UnaryOperator,
   ast_ExprKind_ConditionalOperator,
   ast_ExprKind_Builtin,
   ast_ExprKind_ArraySubscript,
   ast_ExprKind_Member,
   ast_ExprKind_Paren,
   ast_ExprKind_BitOffset,
   ast_ExprKind_ExplicitCast,
   ast_ExprKind_ImplicitCast,
   _ast_ExprKind_max = 255
} __attribute__((packed)) ast_ExprKind;

typedef enum {
   ast_ValType_NValue,
   ast_ValType_RValue,
   ast_ValType_LValue,
   _ast_ValType_max = 255
} __attribute__((packed)) ast_ValType;

struct ast_Expr_ {
   ast_Stmt parent;
   src_loc_SrcLoc loc;
   ast_QualType qt;
};

static const char* ast_exprKind_names[22] = {
   "IntegerLiteral",
   "FloatLiteral",
   "BooleanLiteral",
   "CharLiteral",
   "StringLiteral",
   "Nil",
   "Identifier",
   "TypeExpr",
   "Call",
   "InitList",
   "FieldDesignatedInit",
   "ArrayDesignatedInit",
   "BinaryOperator",
   "UnaryOperator",
   "ConditionalOp",
   "Builtin",
   "ArraySubscript",
   "Member",
   "Paren",
   "BitOffset",
   "ExplicitCast",
   "ImplicitCast"
};

static const char* ast_valType_names[3] = { "nvalue", "rvalue", "lvalue" };

static void ast_Expr_init(ast_Expr* e, ast_ExprKind k, src_loc_SrcLoc loc, bool ctv, bool ctc, bool has_effect, ast_ValType valtype);
static ast_Expr* ast_Expr_instantiate(ast_Expr* e, ast_Instantiator* inst);
static ast_Stmt* ast_Expr_asStmt(ast_Expr* e);
static ast_ExprKind ast_Expr_getKind(const ast_Expr* e);
static bool ast_Expr_isIntegerLiteral(const ast_Expr* e);
static bool ast_Expr_isStringLiteral(const ast_Expr* e);
static bool ast_Expr_isNil(const ast_Expr* e);
static bool ast_Expr_isIdentifier(const ast_Expr* e);
static bool ast_Expr_isCall(const ast_Expr* e);
static bool ast_Expr_isImplicitCast(const ast_Expr* e);
static bool ast_Expr_isType(const ast_Expr* e);
static bool ast_Expr_isInitList(const ast_Expr* e);
static bool ast_Expr_isBinaryOperator(const ast_Expr* e);
static bool ast_Expr_isMember(const ast_Expr* e);
static bool ast_Expr_isFieldDesignatedInit(const ast_Expr* e);
static bool ast_Expr_isArrayDesignatedInit(const ast_Expr* e);
static bool ast_Expr_isBitOffset(const ast_Expr* e);
static bool ast_Expr_isCtv(const ast_Expr* e);
static bool ast_Expr_isCtc(const ast_Expr* e);
static void ast_Expr_setCtv(ast_Expr* e);
static void ast_Expr_setCtc(ast_Expr* e);
static void ast_Expr_copyCtcFlags(ast_Expr* e, const ast_Expr* other);
static void ast_Expr_copyConstantFlags(ast_Expr* e, const ast_Expr* other);
static void ast_Expr_combineConstantFlags(ast_Expr* e, const ast_Expr* lhs, const ast_Expr* rhs);
static bool ast_Expr_hasEffect(const ast_Expr* e);
static ast_ValType ast_Expr_getValType(const ast_Expr* e);
static bool ast_Expr_isNValue(const ast_Expr* e);
static bool ast_Expr_isRValue(const ast_Expr* e);
static bool ast_Expr_isLValue(const ast_Expr* e);
static void ast_Expr_setLValue(ast_Expr* e);
static void ast_Expr_setRValue(ast_Expr* e);
static void ast_Expr_setValType(ast_Expr* e, ast_ValType valtype);
static void ast_Expr_copyValType(ast_Expr* e, const ast_Expr* other);
static src_loc_SrcLoc ast_Expr_getLoc(const ast_Expr* e);
static src_loc_SrcLoc ast_Expr_getStartLoc(const ast_Expr* e);
static src_loc_SrcLoc ast_Expr_getEndLoc(const ast_Expr* e);
static src_loc_SrcRange ast_Expr_getRange(const ast_Expr* e);
static void ast_Expr_setType(ast_Expr* e, ast_QualType qt_);
static ast_QualType ast_Expr_getType(const ast_Expr* e);
static void ast_Expr_dump(const ast_Expr* e);
static void ast_Expr_print(const ast_Expr* e, string_buffer_Buf* out, uint32_t indent);
static void ast_Expr_printLiteral(const ast_Expr* e, string_buffer_Buf* out);
static void ast_Expr_printKind(const ast_Expr* e, string_buffer_Buf* out, uint32_t indent);
static void ast_Expr_printTypeBits(const ast_Expr* e, string_buffer_Buf* out);
struct ast_ArrayDesignatedInitExpr_ {
   ast_Expr parent;
   ast_Expr* designator;
   ast_Expr* initValue;
};

static ast_ArrayDesignatedInitExpr* ast_ArrayDesignatedInitExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* designator, ast_Expr* initValue);
static ast_Expr* ast_ArrayDesignatedInitExpr_instantiate(ast_ArrayDesignatedInitExpr* e, ast_Instantiator* inst);
static ast_Expr* ast_ArrayDesignatedInitExpr_getDesignator(const ast_ArrayDesignatedInitExpr* e);
static ast_Expr** ast_ArrayDesignatedInitExpr_getDesignator2(ast_ArrayDesignatedInitExpr* e);
static ast_Expr* ast_ArrayDesignatedInitExpr_getInit(const ast_ArrayDesignatedInitExpr* e);
static ast_Expr** ast_ArrayDesignatedInitExpr_getInit2(ast_ArrayDesignatedInitExpr* e);
static void ast_ArrayDesignatedInitExpr_print(const ast_ArrayDesignatedInitExpr* e, string_buffer_Buf* out, uint32_t indent);
struct ast_ArraySubscriptExpr_ {
   ast_Expr parent;
   ast_Expr* base;
   ast_Expr* idx;
};

static ast_ArraySubscriptExpr* ast_ArraySubscriptExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* base, ast_Expr* idx);
static ast_Expr* ast_ArraySubscriptExpr_instantiate(ast_ArraySubscriptExpr* e, ast_Instantiator* inst);
static ast_Expr* ast_ArraySubscriptExpr_getBase(const ast_ArraySubscriptExpr* e);
static ast_Expr** ast_ArraySubscriptExpr_getBase2(ast_ArraySubscriptExpr* e);
static ast_Expr* ast_ArraySubscriptExpr_getIndex(const ast_ArraySubscriptExpr* e);
static ast_Expr** ast_ArraySubscriptExpr_getIndex2(ast_ArraySubscriptExpr* e);
static void ast_ArraySubscriptExpr_printLiteral(const ast_ArraySubscriptExpr* e, string_buffer_Buf* out);
static void ast_ArraySubscriptExpr_print(const ast_ArraySubscriptExpr* e, string_buffer_Buf* out, uint32_t indent);
typedef enum {
   ast_BinaryOpcode_Multiply,
   ast_BinaryOpcode_Divide,
   ast_BinaryOpcode_Reminder,
   ast_BinaryOpcode_Add,
   ast_BinaryOpcode_Subtract,
   ast_BinaryOpcode_ShiftLeft,
   ast_BinaryOpcode_ShiftRight,
   ast_BinaryOpcode_LessThan,
   ast_BinaryOpcode_GreaterThan,
   ast_BinaryOpcode_LessEqual,
   ast_BinaryOpcode_GreaterEqual,
   ast_BinaryOpcode_Equal,
   ast_BinaryOpcode_NotEqual,
   ast_BinaryOpcode_And,
   ast_BinaryOpcode_Xor,
   ast_BinaryOpcode_Or,
   ast_BinaryOpcode_LAnd,
   ast_BinaryOpcode_LOr,
   ast_BinaryOpcode_Assign,
   ast_BinaryOpcode_MulAssign,
   ast_BinaryOpcode_DivAssign,
   ast_BinaryOpcode_RemAssign,
   ast_BinaryOpcode_AddAssign,
   ast_BinaryOpcode_SubAssign,
   ast_BinaryOpcode_ShlAssign,
   ast_BinaryOpcode_ShrAssign,
   ast_BinaryOpcode_AndAssign,
   ast_BinaryOpcode_XorAssign,
   ast_BinaryOpcode_OrAssign,
   _ast_BinaryOpcode_max = 255
} __attribute__((packed)) ast_BinaryOpcode;

struct ast_BinaryOperator_ {
   ast_Expr parent;
   ast_Expr* lhs;
   ast_Expr* rhs;
};

static const char* ast_binaryOpcode_names[29] = {
   "*",
   "/",
   "%",
   "+",
   "-",
   "<<",
   ">>",
   "<",
   ">",
   "<=",
   ">=",
   "==",
   "!=",
   "&",
   "^",
   "|",
   "&&",
   "||",
   "=",
   "*=",
   "/=",
   "%=",
   "+=",
   "-=",
   "<<=",
   ">>=",
   "&=",
   "^=",
   "|="
};

static ast_BinaryOperator* ast_BinaryOperator_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_BinaryOpcode kind, ast_Expr* lhs, ast_Expr* rhs);
static ast_Expr* ast_BinaryOperator_instantiate(ast_BinaryOperator* e, ast_Instantiator* inst);
static ast_BinaryOpcode ast_BinaryOperator_getOpcode(const ast_BinaryOperator* e);
static ast_Expr* ast_BinaryOperator_getLHS(const ast_BinaryOperator* e);
static ast_Expr** ast_BinaryOperator_getLHS2(ast_BinaryOperator* e);
static ast_Expr* ast_BinaryOperator_getRHS(const ast_BinaryOperator* e);
static ast_Expr** ast_BinaryOperator_getRHS2(ast_BinaryOperator* e);
static const char* ast_BinaryOperator_getOpcodeStr(const ast_BinaryOperator* e);
static void ast_BinaryOperator_print(const ast_BinaryOperator* e, string_buffer_Buf* out, uint32_t indent);
static void ast_BinaryOperator_printLiteral(const ast_BinaryOperator* e, string_buffer_Buf* out);
struct ast_BitOffsetExpr_ {
   ast_Expr parent;
   ast_Expr* lhs;
   ast_Expr* rhs;
};

static ast_BitOffsetExpr* ast_BitOffsetExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* lhs, ast_Expr* rhs);
static ast_Expr* ast_BitOffsetExpr_instantiate(ast_BitOffsetExpr* e, ast_Instantiator* inst);
static ast_Expr* ast_BitOffsetExpr_getLHS(ast_BitOffsetExpr* e);
static ast_Expr** ast_BitOffsetExpr_getLHS2(ast_BitOffsetExpr* e);
static ast_Expr* ast_BitOffsetExpr_getRHS(ast_BitOffsetExpr* e);
static ast_Expr** ast_BitOffsetExpr_getRHS2(ast_BitOffsetExpr* e);
static void ast_BitOffsetExpr_setWidth(ast_BitOffsetExpr* e, uint8_t width);
static uint32_t ast_BitOffsetExpr_getWidth(const ast_BitOffsetExpr* e);
static void ast_BitOffsetExpr_printLiteral(const ast_BitOffsetExpr* e, string_buffer_Buf* out);
static void ast_BitOffsetExpr_print(const ast_BitOffsetExpr* e, string_buffer_Buf* out, uint32_t indent);
struct ast_BooleanLiteral_ {
   ast_Expr parent;
};

static ast_BooleanLiteral* ast_BooleanLiteral_create(ast_context_Context* c, src_loc_SrcLoc loc, bool val);
static bool ast_BooleanLiteral_getValue(const ast_BooleanLiteral* e);
static void ast_BooleanLiteral_print(const ast_BooleanLiteral* e, string_buffer_Buf* out, uint32_t indent);
static void ast_BooleanLiteral_printLiteral(const ast_BooleanLiteral* e, string_buffer_Buf* out);
typedef enum {
   ast_BuiltinExprKind_Sizeof,
   ast_BuiltinExprKind_Elemsof,
   ast_BuiltinExprKind_EnumMin,
   ast_BuiltinExprKind_EnumMax,
   ast_BuiltinExprKind_OffsetOf,
   ast_BuiltinExprKind_ToContainer,
   _ast_BuiltinExprKind_max = 255
} __attribute__((packed)) ast_BuiltinExprKind;

struct ast_ToContainerData_ {
   ast_Expr* member;
   ast_Expr* pointer;
};

struct ast_OffsetOfData_ {
   ast_Expr* member;
};

struct ast_BuiltinExpr_ {
   ast_Expr parent;
   ast_Expr* inner;
   ast_Value value;
   ast_OffsetOfData offset[0];
   ast_ToContainerData container[0];
};

static const char* ast_builtin_names[6] = { "sizeof", "elemsof", "enum_min", "enum_max", "offsetof", "to_container" };

static ast_BuiltinExpr* ast_BuiltinExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* inner, ast_BuiltinExprKind kind);
static ast_BuiltinExpr* ast_BuiltinExpr_createOffsetOf(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* typeExpr, ast_Expr* member);
static ast_BuiltinExpr* ast_BuiltinExpr_createToContainer(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* typeExpr, ast_Expr* member, ast_Expr* pointer);
static ast_Expr* ast_BuiltinExpr_instantiate(ast_BuiltinExpr* e, ast_Instantiator* inst);
static ast_BuiltinExprKind ast_BuiltinExpr_getKind(const ast_BuiltinExpr* e);
static ast_Value ast_BuiltinExpr_getValue(const ast_BuiltinExpr* e);
static void ast_BuiltinExpr_setValue(ast_BuiltinExpr* e, ast_Value value);
static void ast_BuiltinExpr_setUValue(ast_BuiltinExpr* e, uint64_t val);
static ast_Expr* ast_BuiltinExpr_getInner(const ast_BuiltinExpr* e);
static src_loc_SrcLoc ast_BuiltinExpr_getEndLoc(const ast_BuiltinExpr* e);
static ast_Expr* ast_BuiltinExpr_getOffsetOfMember(const ast_BuiltinExpr* b);
static ast_Expr* ast_BuiltinExpr_getToContainerMember(const ast_BuiltinExpr* b);
static ast_Expr* ast_BuiltinExpr_getToContainerPointer(const ast_BuiltinExpr* b);
static ast_Expr** ast_BuiltinExpr_getToContainerPointer2(ast_BuiltinExpr* b);
static void ast_BuiltinExpr_print(const ast_BuiltinExpr* e, string_buffer_Buf* out, uint32_t indent);
static void ast_BuiltinExpr_printLiteral(const ast_BuiltinExpr* e, string_buffer_Buf* out);
struct ast_CallExpr_ {
   ast_Expr parent;
   src_loc_SrcLoc endLoc;
   uint16_t template_idx;
   uint8_t num_args;
   ast_Expr* func;
   ast_Expr* args[0];
};

static ast_CallExpr* ast_CallExpr_create(ast_context_Context* c, src_loc_SrcLoc endLoc, ast_Expr* func, ast_Expr** args, uint32_t num_args);
static ast_CallExpr* ast_CallExpr_createTemplate(ast_context_Context* c, src_loc_SrcLoc endLoc, ast_Expr* func, ast_Expr** args, uint32_t num_args, const ast_TypeRefHolder* ref);
static ast_Expr* ast_CallExpr_instantiate(ast_CallExpr* e, ast_Instantiator* inst);
static void ast_CallExpr_setCallsStructFunc(ast_CallExpr* e);
static bool ast_CallExpr_isStructFunc(const ast_CallExpr* e);
static void ast_CallExpr_setCallsStaticStructFunc(ast_CallExpr* e);
static bool ast_CallExpr_isStaticStructFunc(const ast_CallExpr* e);
static bool ast_CallExpr_isTemplateCall(const ast_CallExpr* e);
static ast_TypeRef* ast_CallExpr_getTemplateArg(const ast_CallExpr* e);
static void ast_CallExpr_setTemplateIdx(ast_CallExpr* e, uint32_t idx);
static uint32_t ast_CallExpr_getTemplateIdx(const ast_CallExpr* e);
static void ast_CallExpr_setPrintfFormat(ast_CallExpr* e, uint32_t format_idx, bool change_format);
static bool ast_CallExpr_isPrintfCall(const ast_CallExpr* e);
static uint32_t ast_CallExpr_getPrintfFormat(const ast_CallExpr* e);
static bool ast_CallExpr_needFormatChange(const ast_CallExpr* e);
static void ast_CallExpr_setHasAutoArgs(ast_CallExpr* e);
static bool ast_CallExpr_hasAutoArgs(const ast_CallExpr* e);
static src_loc_SrcLoc ast_CallExpr_getEndLoc(const ast_CallExpr* e);
static ast_Expr* ast_CallExpr_getFunc(const ast_CallExpr* e);
static ast_Expr** ast_CallExpr_getFunc2(ast_CallExpr* e);
static uint32_t ast_CallExpr_getNumArgs(const ast_CallExpr* e);
static ast_Expr** ast_CallExpr_getArgs(ast_CallExpr* e);
static void ast_CallExpr_printLiteral(const ast_CallExpr* e, string_buffer_Buf* out);
static void ast_CallExpr_print(const ast_CallExpr* e, string_buffer_Buf* out, uint32_t indent);
struct ast_CharLiteral_ {
   ast_Expr parent;
};

static ast_CharLiteral* ast_CharLiteral_create(ast_context_Context* c, src_loc_SrcLoc loc, uint8_t val, uint8_t radix);
static uint8_t ast_CharLiteral_getValue(const ast_CharLiteral* e);
static void ast_CharLiteral_print(const ast_CharLiteral* e, string_buffer_Buf* out, uint32_t indent);
static void ast_CharLiteral_printLiteral(const ast_CharLiteral* e, string_buffer_Buf* out);
struct ast_ConditionalOperator_ {
   ast_Expr parent;
   src_loc_SrcLoc colonLoc;
   ast_Expr* cond;
   ast_Expr* lhs;
   ast_Expr* rhs;
};

static ast_ConditionalOperator* ast_ConditionalOperator_create(ast_context_Context* c, src_loc_SrcLoc questionLoc, src_loc_SrcLoc colonLoc, ast_Expr* cond, ast_Expr* lhs, ast_Expr* rhs);
static ast_Expr* ast_ConditionalOperator_instantiate(ast_ConditionalOperator* e, ast_Instantiator* inst);
static ast_Expr* ast_ConditionalOperator_getCond(const ast_ConditionalOperator* e);
static ast_Expr** ast_ConditionalOperator_getCond2(ast_ConditionalOperator* e);
static ast_Expr* ast_ConditionalOperator_getLHS(const ast_ConditionalOperator* e);
static ast_Expr** ast_ConditionalOperator_getLHS2(ast_ConditionalOperator* e);
static ast_Expr* ast_ConditionalOperator_getRHS(const ast_ConditionalOperator* e);
static ast_Expr** ast_ConditionalOperator_getRHS2(ast_ConditionalOperator* e);
static void ast_ConditionalOperator_printLiteral(const ast_ConditionalOperator* e, string_buffer_Buf* out);
static void ast_ConditionalOperator_print(const ast_ConditionalOperator* e, string_buffer_Buf* out, uint32_t indent);
struct ast_ExplicitCastExpr_ {
   ast_Expr parent;
   ast_Expr* inner;
   ast_TypeRef dest;
};

static ast_ExplicitCastExpr* ast_ExplicitCastExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, const ast_TypeRefHolder* ref, ast_Expr* inner);
static ast_Expr* ast_ExplicitCastExpr_instantiate(ast_ExplicitCastExpr* e, ast_Instantiator* inst);
static ast_Expr* ast_ExplicitCastExpr_getInner(const ast_ExplicitCastExpr* e);
static ast_Expr** ast_ExplicitCastExpr_getInner2(ast_ExplicitCastExpr* e);
static ast_TypeRef* ast_ExplicitCastExpr_getTypeRef(ast_ExplicitCastExpr* e);
static void ast_ExplicitCastExpr_printLiteral(const ast_ExplicitCastExpr* e, string_buffer_Buf* out);
static void ast_ExplicitCastExpr_print(const ast_ExplicitCastExpr* e, string_buffer_Buf* out, uint32_t indent);
struct ast_FieldDesignatedInitExpr_ {
   ast_Expr parent;
   uint32_t field;
   ast_Expr* initValue;
   ast_Decl* decl;
};

static ast_FieldDesignatedInitExpr* ast_FieldDesignatedInitExpr_create(ast_context_Context* c, uint32_t field, src_loc_SrcLoc loc, ast_Expr* initValue);
static ast_Expr* ast_FieldDesignatedInitExpr_instantiate(ast_FieldDesignatedInitExpr* e, ast_Instantiator* inst);
static uint32_t ast_FieldDesignatedInitExpr_getField(const ast_FieldDesignatedInitExpr* e);
static const char* ast_FieldDesignatedInitExpr_getFieldName(const ast_FieldDesignatedInitExpr* e);
static ast_Expr* ast_FieldDesignatedInitExpr_getInit(const ast_FieldDesignatedInitExpr* e);
static ast_Expr** ast_FieldDesignatedInitExpr_getInit2(ast_FieldDesignatedInitExpr* e);
static void ast_FieldDesignatedInitExpr_setDecl(ast_FieldDesignatedInitExpr* e, ast_Decl* d);
static ast_Decl* ast_FieldDesignatedInitExpr_getDecl(const ast_FieldDesignatedInitExpr* e);
static void ast_FieldDesignatedInitExpr_print(const ast_FieldDesignatedInitExpr* e, string_buffer_Buf* out, uint32_t indent);
struct ast_FloatLiteral_ {
   ast_Expr parent;
   double val;
};

static ast_FloatLiteral* ast_FloatLiteral_create(ast_context_Context* c, src_loc_SrcLoc loc, double val);
static double ast_FloatLiteral_getValue(const ast_FloatLiteral* e);
static void ast_FloatLiteral_print(const ast_FloatLiteral* e, string_buffer_Buf* out, uint32_t indent);
static void ast_FloatLiteral_printLiteral(const ast_FloatLiteral* e, string_buffer_Buf* out);
typedef enum {
   ast_IdentifierKind_Unresolved,
   ast_IdentifierKind_Module,
   ast_IdentifierKind_Function,
   ast_IdentifierKind_Type,
   ast_IdentifierKind_Var,
   ast_IdentifierKind_EnumConstant,
   ast_IdentifierKind_StructMember,
   ast_IdentifierKind_Label,
   _ast_IdentifierKind_max = 255
} __attribute__((packed)) ast_IdentifierKind;

struct ast_IdentifierExpr_ {
   ast_Expr parent;
   union {
      uint32_t name_idx;
      ast_Decl* decl;
   };
};

static const char* ast_identifierKind_names[8] = {
   "Unresolved",
   "Module",
   "Function",
   "Type",
   "Var",
   "EnumConstant",
   "StructMember",
   "Label"
};

static ast_IdentifierExpr* ast_IdentifierExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, uint32_t name);
static ast_Expr* ast_IdentifierExpr_instantiate(ast_IdentifierExpr* e, ast_Instantiator* inst);
static uint32_t ast_IdentifierExpr_getSize(void);
static ast_Expr* ast_IdentifierExpr_asExpr(ast_IdentifierExpr* e);
static void ast_IdentifierExpr_setDecl(ast_IdentifierExpr* e, ast_Decl* decl);
static ast_Decl* ast_IdentifierExpr_getDecl(const ast_IdentifierExpr* e);
static ast_Ref ast_IdentifierExpr_getRef(const ast_IdentifierExpr* e);
static void ast_IdentifierExpr_setKind(ast_IdentifierExpr* e, ast_IdentifierKind kind);
static ast_IdentifierKind ast_IdentifierExpr_getKind(const ast_IdentifierExpr* e);
static const char* ast_IdentifierExpr_getName(const ast_IdentifierExpr* e);
static void ast_IdentifierExpr_setCaseRange(ast_IdentifierExpr* e);
static bool ast_IdentifierExpr_isCaseRange(const ast_IdentifierExpr* e);
static uint32_t ast_IdentifierExpr_getNameIdx(const ast_IdentifierExpr* e);
static void ast_IdentifierExpr_print(const ast_IdentifierExpr* e, string_buffer_Buf* out, uint32_t indent);
static void ast_IdentifierExpr_printLiteral(const ast_IdentifierExpr* e, string_buffer_Buf* out);
typedef enum {
   ast_ImplicitCastKind_ArrayToPointerDecay,
   ast_ImplicitCastKind_FunctionToPointerDecay,
   ast_ImplicitCastKind_LValueToRValue,
   ast_ImplicitCastKind_PointerToBoolean,
   ast_ImplicitCastKind_PointerToInteger,
   ast_ImplicitCastKind_IntegralCast,
   ast_ImplicitCastKind_BitCast,
   _ast_ImplicitCastKind_max = 255
} __attribute__((packed)) ast_ImplicitCastKind;

struct ast_ImplicitCastExpr_ {
   ast_Expr parent;
   ast_Expr* inner;
};

static const char* ast_implicitCastKind_names[7] = {
   "ArrayToPointerDecay",
   "FunctionToPointerDecay",
   "LValueToRValue",
   "PointerToBoolean",
   "PointerToInteger",
   "IntegralCast",
   "BitCast"
};

static ast_ImplicitCastExpr* ast_ImplicitCastExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_ImplicitCastKind kind, ast_Expr* inner);
static ast_ImplicitCastKind ast_ImplicitCastExpr_getKind(const ast_ImplicitCastExpr* e);
static bool ast_ImplicitCastExpr_isArrayToPointerDecay(const ast_ImplicitCastExpr* e);
static ast_Expr* ast_ImplicitCastExpr_getInner(const ast_ImplicitCastExpr* e);
static void ast_ImplicitCastExpr_printLiteral(const ast_ImplicitCastExpr* e, string_buffer_Buf* out);
static void ast_ImplicitCastExpr_print(const ast_ImplicitCastExpr* e, string_buffer_Buf* out, uint32_t indent);
struct ast_InitListExpr_ {
   ast_Expr parent;
   src_loc_SrcLoc right;
   ast_Expr* values[0];
};

static ast_InitListExpr* ast_InitListExpr_create(ast_context_Context* c, src_loc_SrcLoc left, src_loc_SrcLoc right, ast_Expr** values, uint32_t num_values);
static ast_Expr* ast_InitListExpr_instantiate(ast_InitListExpr* e, ast_Instantiator* inst);
static uint32_t ast_InitListExpr_getNumValues(const ast_InitListExpr* e);
static ast_Expr** ast_InitListExpr_getValues(ast_InitListExpr* e);
static void ast_InitListExpr_print(const ast_InitListExpr* e, string_buffer_Buf* out, uint32_t indent);
struct ast_IntegerLiteral_ {
   ast_Expr parent;
   uint64_t val;
};

static ast_IntegerLiteral* ast_IntegerLiteral_create(ast_context_Context* c, src_loc_SrcLoc loc, uint8_t radix, uint64_t val);
static ast_IntegerLiteral* ast_IntegerLiteral_createUnsignedConstant(ast_context_Context* c, src_loc_SrcLoc loc, uint64_t val, ast_QualType qt);
static ast_IntegerLiteral* ast_IntegerLiteral_createSignedConstant(ast_context_Context* c, src_loc_SrcLoc loc, int64_t val, ast_QualType qt);
static uint64_t ast_IntegerLiteral_getValue(const ast_IntegerLiteral* e);
static bool ast_IntegerLiteral_isDecimal(const ast_IntegerLiteral* e);
static bool ast_IntegerLiteral_isSigned(const ast_IntegerLiteral* e);
static void ast_printBinary(string_buffer_Buf* out, uint64_t value);
static void ast_printOctal(string_buffer_Buf* out, uint64_t value);
static void ast_IntegerLiteral_print(const ast_IntegerLiteral* e, string_buffer_Buf* out, uint32_t indent);
static void ast_IntegerLiteral_printLiteral(const ast_IntegerLiteral* e, string_buffer_Buf* out);
static void ast_IntegerLiteral_printDecimal(const ast_IntegerLiteral* e, string_buffer_Buf* out);
union ast_MemberRef_ {
   uint32_t name_idx;
   ast_Decl* decl;
   ast_Expr* expr;
};

struct ast_MemberExpr_ {
   ast_Expr parent;
   ast_MemberRef refs[0];
};

#define ast_MemberExprMaxDepth 7
static ast_MemberExpr* ast_MemberExpr_create(ast_context_Context* c, ast_Expr* base, const ast_Ref* refs, uint32_t refcount);
static ast_Expr* ast_MemberExpr_instantiate(ast_MemberExpr* e, ast_Instantiator* inst);
static bool ast_MemberExpr_hasExpr(const ast_MemberExpr* e);
static ast_Expr* ast_MemberExpr_getExprBase(const ast_MemberExpr* e);
static const char* ast_MemberExpr_getName(const ast_MemberExpr* e, uint32_t ref_idx);
static uint32_t ast_MemberExpr_getNumRefs(const ast_MemberExpr* e);
static uint32_t ast_MemberExpr_getNameIdx(const ast_MemberExpr* e, uint32_t ref_idx);
static src_loc_SrcLoc ast_MemberExpr_getLoc(const ast_MemberExpr* e, uint32_t ref_idx);
static ast_Ref ast_MemberExpr_getRef(const ast_MemberExpr* e, uint32_t ref_idx);
static ast_IdentifierKind ast_MemberExpr_getKind(const ast_MemberExpr* e);
static void ast_MemberExpr_setKind(ast_MemberExpr* e, ast_IdentifierKind kind);
static bool ast_MemberExpr_isVarKind(const ast_MemberExpr* e);
static void ast_MemberExpr_setIsStructFunc(ast_MemberExpr* e);
static bool ast_MemberExpr_isStructFunc(const ast_MemberExpr* e);
static void ast_MemberExpr_setIsStaticStructFunc(ast_MemberExpr* e);
static bool ast_MemberExpr_isStaticStructFunc(const ast_MemberExpr* e);
static void ast_MemberExpr_setConstBase(ast_MemberExpr* e, bool b);
static bool ast_MemberExpr_isConstBase(const ast_MemberExpr* e);
static ast_Decl* ast_MemberExpr_getPrevLastDecl(const ast_MemberExpr* e);
static ast_Decl* ast_MemberExpr_getFullDecl(const ast_MemberExpr* e);
static ast_Decl* ast_MemberExpr_getDecl(const ast_MemberExpr* e, uint32_t ref_idx);
static void ast_MemberExpr_setDecl(ast_MemberExpr* e, ast_Decl* d, uint32_t ref_idx);
static src_loc_SrcLoc ast_MemberExpr_getStartLoc(const ast_MemberExpr* e);
static src_loc_SrcLoc ast_MemberExpr_getEndLoc(const ast_MemberExpr* e);
static ast_QualType ast_MemberExpr_getBaseType(const ast_MemberExpr* m);
static const char* ast_MemberExpr_getLastMemberName(const ast_MemberExpr* e);
static void ast_MemberExpr_print(const ast_MemberExpr* e, string_buffer_Buf* out, uint32_t indent);
static void ast_MemberExpr_printLiteral(const ast_MemberExpr* e, string_buffer_Buf* out);
static void ast_MemberExpr_dump(const ast_MemberExpr* m);
struct ast_NilExpr_ {
   ast_Expr parent;
};

static ast_NilExpr* ast_NilExpr_create(ast_context_Context* c, src_loc_SrcLoc loc);
static void ast_NilExpr_print(const ast_NilExpr* e, string_buffer_Buf* out, uint32_t indent);
static void ast_NilExpr_printLiteral(const ast_NilExpr* _arg0, string_buffer_Buf* out);
struct ast_ParenExpr_ {
   ast_Expr parent;
   ast_Expr* inner;
};

static ast_ParenExpr* ast_ParenExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* inner);
static ast_Expr* ast_ParenExpr_instantiate(ast_ParenExpr* e, ast_Instantiator* inst);
static ast_Expr* ast_ParenExpr_getInner(const ast_ParenExpr* e);
static ast_Expr** ast_ParenExpr_getInner2(ast_ParenExpr* e);
static void ast_ParenExpr_print(const ast_ParenExpr* e, string_buffer_Buf* out, uint32_t indent);
static void ast_ParenExpr_printLiteral(const ast_ParenExpr* e, string_buffer_Buf* out);
struct ast_StringLiteral_ {
   ast_Expr parent;
   uint32_t value;
};

static ast_StringLiteral* ast_StringLiteral_create(ast_context_Context* c, src_loc_SrcLoc loc, uint32_t value, uint32_t len);
static const char* ast_StringLiteral_getText(const ast_StringLiteral* e);
static void ast_StringLiteral_printLiteral(const ast_StringLiteral* e, string_buffer_Buf* out);
static void ast_StringLiteral_print(const ast_StringLiteral* e, string_buffer_Buf* out, uint32_t indent);
struct ast_TypeExpr_ {
   ast_Expr parent;
   ast_TypeRef typeRef;
};

static ast_TypeExpr* ast_TypeExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, const ast_TypeRefHolder* ref);
static ast_Expr* ast_TypeExpr_instantiate(ast_TypeExpr* e, ast_Instantiator* inst);
static ast_TypeRef* ast_TypeExpr_getTypeRef(ast_TypeExpr* e);
static void ast_TypeExpr_print(const ast_TypeExpr* e, string_buffer_Buf* out, uint32_t indent);
typedef enum {
   ast_UnaryOpcode_PostInc,
   ast_UnaryOpcode_PostDec,
   ast_UnaryOpcode_PreInc,
   ast_UnaryOpcode_PreDec,
   ast_UnaryOpcode_AddrOf,
   ast_UnaryOpcode_Deref,
   ast_UnaryOpcode_Minus,
   ast_UnaryOpcode_Not,
   ast_UnaryOpcode_LNot,
   _ast_UnaryOpcode_max = 255
} __attribute__((packed)) ast_UnaryOpcode;

struct ast_UnaryOperator_ {
   ast_Expr parent;
   ast_Expr* inner;
};

static const char* ast_unaryOpcode_names[9] = {
   "++",
   "--",
   "++",
   "--",
   "&",
   "*",
   "-",
   "~",
   "!"
};

static ast_UnaryOperator* ast_UnaryOperator_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_UnaryOpcode kind, ast_Expr* inner);
static ast_Expr* ast_UnaryOperator_instantiate(ast_UnaryOperator* e, ast_Instantiator* inst);
static ast_UnaryOpcode ast_UnaryOperator_getOpcode(const ast_UnaryOperator* e);
static ast_Expr* ast_UnaryOperator_getInner(const ast_UnaryOperator* e);
static ast_Expr** ast_UnaryOperator_getInner2(ast_UnaryOperator* e);
static ast_Expr* ast_UnaryOperator_asExpr(ast_UnaryOperator* e);
static bool ast_UnaryOperator_isBefore(const ast_UnaryOperator* e);
static src_loc_SrcLoc ast_UnaryOperator_getStartLoc(const ast_UnaryOperator* e);
static src_loc_SrcLoc ast_UnaryOperator_getEndLoc(const ast_UnaryOperator* e);
static const char* ast_UnaryOperator_getOpcodeStr(const ast_UnaryOperator* e);
static void ast_UnaryOperator_print(const ast_UnaryOperator* e, string_buffer_Buf* out, uint32_t indent);
static void ast_UnaryOperator_printLiteral(const ast_UnaryOperator* e, string_buffer_Buf* out);
typedef enum {
   ast_TypeKind_Builtin,
   ast_TypeKind_Pointer,
   ast_TypeKind_Array,
   ast_TypeKind_Struct,
   ast_TypeKind_Enum,
   ast_TypeKind_Function,
   ast_TypeKind_Alias,
   ast_TypeKind_Module,
   _ast_TypeKind_max = 255
} __attribute__((packed)) ast_TypeKind;

struct ast_TypeBits_ {
   uint32_t kind : 8;
};

#define ast_NumTypeBits 8
struct ast_BuiltinTypeBits_ {
   uint32_t  : 8;
   uint32_t kind : 4;
};

struct ast_ArrayTypeBits_ {
   uint32_t  : 8;
   uint32_t has_size : 1;
   uint32_t is_incremental : 1;
};

struct ast_Type_ {
   union {
      ast_TypeBits typeBits;
      ast_BuiltinTypeBits builtinTypeBits;
      ast_ArrayTypeBits arrayTypeBits;
      uint32_t bits;
   };
   uint32_t ptr_pool_idx;
   ast_QualType canonicalType;
};

static const char* ast_typeKind_names[8] = {
   "Builtin",
   "Pointer",
   "Array",
   "Struct",
   "Enum",
   "Function",
   "Alias",
   "Module"
};

static void ast_Type_init(ast_Type* t, ast_TypeKind k);
static ast_TypeKind ast_Type_getKind(const ast_Type* t);
static bool ast_Type_hasCanonicalType(const ast_Type* t);
static ast_QualType ast_Type_getCanonicalType(const ast_Type* t);
static void ast_Type_setCanonicalType(ast_Type* t, ast_QualType canon);
static uint32_t ast_Type_getIndex(const ast_Type* t);
static ast_ArrayType* ast_Type_asArray(ast_Type* t);
static bool ast_Type_isBuiltinType(const ast_Type* t);
static bool ast_Type_isArrayType(const ast_Type* t);
static bool ast_Type_isStructType(const ast_Type* t);
static bool ast_Type_isPointerType(const ast_Type* t);
static bool ast_Type_isFunctionType(const ast_Type* t);
static bool ast_Type_isEnumType(const ast_Type* t);
static bool ast_Type_isVoidType(const ast_Type* t);
static void ast_Type_dump(const ast_Type* t);
static uint32_t ast_Type_getAlignment(const ast_Type* t);
static uint32_t ast_Type_getSize(const ast_Type* t);
static void ast_Type_print(const ast_Type* t, string_buffer_Buf* out);
static void ast_Type_fullPrint(const ast_Type* t, string_buffer_Buf* out, uint32_t indent);
struct ast_AliasType_ {
   ast_Type parent;
   ast_AliasTypeDecl* decl;
};

static ast_AliasType* ast_AliasType_create(ast_context_Context* c, ast_AliasTypeDecl* decl);
static ast_AliasTypeDecl* ast_AliasType_getDecl(const ast_AliasType* t);
static void ast_AliasType_print(const ast_AliasType* t, string_buffer_Buf* out);
static void ast_AliasType_fullPrint(const ast_AliasType* t, string_buffer_Buf* out, uint32_t indent);
struct ast_ArrayType_ {
   ast_Type parent;
   ast_QualType elem;
   uint32_t size;
};

static ast_ArrayType* ast_ArrayType_create(ast_context_Context* c, ast_QualType elem, bool has_size, uint32_t size);
static ast_ArrayType* ast_ArrayType_createIncremental(ast_context_Context* c, ast_QualType elem);
static ast_Type* ast_ArrayType_asType(ast_ArrayType* t);
static ast_QualType ast_ArrayType_getElemType(const ast_ArrayType* t);
static bool ast_ArrayType_isIncremental(const ast_ArrayType* t);
static uint32_t ast_ArrayType_hasSize(const ast_ArrayType* t);
static uint32_t ast_ArrayType_getSize(const ast_ArrayType* t);
static void ast_ArrayType_setSize(ast_ArrayType* t, uint32_t size);
static void ast_ArrayType_printPreName(const ast_ArrayType* t, string_buffer_Buf* out);
static void ast_ArrayType_printPostName(const ast_ArrayType* t, string_buffer_Buf* out);
static void ast_ArrayType_print(const ast_ArrayType* t, string_buffer_Buf* out);
static void ast_ArrayType_fullPrint(const ast_ArrayType* t, string_buffer_Buf* out, uint32_t indent);
typedef enum {
   ast_BuiltinKind_Char,
   ast_BuiltinKind_Int8,
   ast_BuiltinKind_Int16,
   ast_BuiltinKind_Int32,
   ast_BuiltinKind_Int64,
   ast_BuiltinKind_UInt8,
   ast_BuiltinKind_UInt16,
   ast_BuiltinKind_UInt32,
   ast_BuiltinKind_UInt64,
   ast_BuiltinKind_Float32,
   ast_BuiltinKind_Float64,
   ast_BuiltinKind_ISize,
   ast_BuiltinKind_USize,
   ast_BuiltinKind_Bool,
   ast_BuiltinKind_Void,
   _ast_BuiltinKind_max = 255
} __attribute__((packed)) ast_BuiltinKind;

struct ast_BuiltinType_ {
   ast_Type parent;
};

static const char* ast_builtinType_names[15] = {
   "char",
   "i8",
   "i16",
   "i32",
   "i64",
   "u8",
   "u16",
   "u32",
   "u64",
   "f32",
   "f64",
   "isize",
   "usize",
   "bool",
   "void"
};

static const bool ast_BuiltinType_promotable[15] = {
   true,
   true,
   true,
   false,
   false,
   true,
   true,
   false,
   false,
   false,
   false,
   false,
   false,
   true,
   false
};

static const bool ast_BuiltinType_signed[15] = {
   true,
   true,
   true,
   true,
   true,
   false,
   false,
   false,
   false,
   true,
   true,
   true,
   false,
   false,
   false
};

static const bool ast_BuiltinType_unsigned[15] = {
   false,
   false,
   false,
   false,
   false,
   true,
   true,
   true,
   true,
   false,
   false,
   false,
   true,
   false,
   false
};

static const bool ast_BuiltinType_integer[15] = {
   true,
   true,
   true,
   true,
   true,
   true,
   true,
   true,
   true,
   false,
   false,
   true,
   true,
   false,
   false
};

static const uint32_t ast_BuiltinType_default_sizes[15] = {
   1,
   1,
   2,
   4,
   8,
   1,
   2,
   4,
   8,
   4,
   8,
   8,
   8,
   1,
   0
};

static const uint32_t ast_BuiltinType_default_widths[15] = {
   7,
   7,
   15,
   31,
   63,
   8,
   16,
   32,
   64,
   0,
   0,
   63,
   64,
   1,
   0
};

static bool ast_builtinKind2Signed(ast_BuiltinKind kind);
static ast_BuiltinType* ast_BuiltinType_create(ast_context_Context* c, ast_BuiltinKind kind);
static ast_BuiltinKind ast_BuiltinType_getKind(const ast_BuiltinType* b);
static ast_BuiltinKind ast_BuiltinType_getBaseKind(const ast_BuiltinType* b);
static bool ast_BuiltinType_isChar(const ast_BuiltinType* b);
static bool ast_BuiltinType_isInt8(const ast_BuiltinType* b);
static bool ast_BuiltinType_isUInt8(const ast_BuiltinType* b);
static bool ast_BuiltinType_isInt32(const ast_BuiltinType* b);
static bool ast_BuiltinType_isBool(const ast_BuiltinType* b);
static bool ast_BuiltinType_isVoid(const ast_BuiltinType* b);
static bool ast_BuiltinType_isArithmeticType(const ast_BuiltinType* b);
static const char* ast_BuiltinType_kind2str(const ast_BuiltinType* b);
static bool ast_BuiltinType_isPromotableIntegerType(const ast_BuiltinType* b);
static bool ast_BuiltinType_isInteger(const ast_BuiltinType* b);
static bool ast_BuiltinType_isIntegerOrBool(const ast_BuiltinType* b);
static bool ast_BuiltinType_isFloatingPoint(const ast_BuiltinType* b);
static bool ast_BuiltinType_isSigned(const ast_BuiltinType* b);
static bool ast_BuiltinType_isUnsigned(const ast_BuiltinType* b);
static uint32_t ast_BuiltinType_getAlignment(const ast_BuiltinType* b);
static uint32_t ast_BuiltinType_getWidth(const ast_BuiltinType* b);
static void ast_BuiltinType_print(const ast_BuiltinType* b, string_buffer_Buf* out);
static void ast_BuiltinType_fullPrint(const ast_BuiltinType* t, string_buffer_Buf* out, uint32_t indent);
struct ast_EnumType_ {
   ast_Type parent;
   ast_EnumTypeDecl* decl;
};

static ast_EnumType* ast_EnumType_create(ast_context_Context* c, ast_EnumTypeDecl* decl);
static ast_EnumTypeDecl* ast_EnumType_getDecl(const ast_EnumType* t);
static ast_Type* ast_EnumType_asType(ast_EnumType* t);
static ast_QualType ast_EnumType_getImplType(const ast_EnumType* t);
static const char* ast_EnumType_getName(const ast_EnumType* t);
static void ast_EnumType_print(const ast_EnumType* t, string_buffer_Buf* out);
static void ast_EnumType_fullPrint(const ast_EnumType* t, string_buffer_Buf* out, uint32_t indent);
struct ast_FunctionType_ {
   ast_Type parent;
   ast_FunctionDecl* decl;
};

static ast_FunctionType* ast_FunctionType_create(ast_context_Context* c, ast_FunctionDecl* decl);
static ast_FunctionDecl* ast_FunctionType_getDecl(const ast_FunctionType* t);
static ast_Type* ast_FunctionType_asType(ast_FunctionType* t);
static void ast_FunctionType_print(const ast_FunctionType* t, string_buffer_Buf* out);
static void ast_FunctionType_fullPrint(const ast_FunctionType* t, string_buffer_Buf* out, uint32_t indent);
struct ast_ModuleType_ {
   ast_Type parent;
   ast_Module* mod;
};

static ast_ModuleType* ast_ModuleType_create(ast_context_Context* c, ast_Module* mod);
static ast_Type* ast_ModuleType_asType(ast_ModuleType* t);
static ast_Module* ast_ModuleType_getModule(const ast_ModuleType* t);
static void ast_ModuleType_print(const ast_ModuleType* t, string_buffer_Buf* out);
static void ast_ModuleType_fullPrint(const ast_ModuleType* t, string_buffer_Buf* out, uint32_t indent);
struct ast_PointerType_ {
   ast_Type parent;
   ast_QualType inner;
};

static ast_PointerType* ast_PointerType_create(ast_context_Context* c, ast_QualType inner);
static ast_Type* ast_PointerType_asType(ast_PointerType* t);
static ast_QualType ast_PointerType_getInner(const ast_PointerType* t);
static void ast_PointerType_print(const ast_PointerType* t, string_buffer_Buf* out);
static void ast_PointerType_fullPrint(const ast_PointerType* t, string_buffer_Buf* out, uint32_t indent);
#define ast_QualType_Const 0x1
#define ast_QualType_Volatile 0x2
#define ast_QualType_Mask 0x3
static ast_QualType ast_QualType_init(ast_Type* t);
static void ast_QualType_set(ast_QualType* qt, ast_Type* t);
static void ast_QualType_setConst(ast_QualType* qt);
static void ast_QualType_unsetConst(ast_QualType* qt);
static bool ast_QualType_isConst(ast_QualType* qt);
static bool ast_QualType_isVolatile(ast_QualType* qt);
static void ast_QualType_setVolatile(ast_QualType* qt);
static uint32_t ast_QualType_getQuals(const ast_QualType* qt);
static void ast_QualType_copyQuals(ast_QualType* qt, ast_QualType other);
static void ast_QualType_clearQuals(ast_QualType* qt);
static bool ast_QualType_isConstant(const ast_QualType* qt);
static bool ast_QualType_isValid(const ast_QualType* qt);
static bool ast_QualType_isInvalid(const ast_QualType* qt);
static bool ast_QualType_isScalar(const ast_QualType* qt);
static ast_Type* ast_QualType_getType(const ast_QualType* qt);
static ast_Type* ast_QualType_getTypeOrNil(const ast_QualType* qt);
static bool ast_QualType_hasCanonicalType(const ast_QualType* qt);
static ast_QualType ast_QualType_getCanonicalType(const ast_QualType* qt);
static void ast_QualType_setCanonicalType(ast_QualType* qt, ast_QualType canon);
static bool ast_QualType_isConstPtr(ast_QualType* qt);
static ast_TypeKind ast_QualType_getKind(ast_QualType* qt);
static uint32_t ast_QualType_getIndex(ast_QualType* qt);
static uint32_t ast_QualType_getAlignment(ast_QualType* qt);
static uint32_t ast_QualType_getSize(ast_QualType* qt);
static bool ast_QualType_isBool(const ast_QualType* qt);
static bool ast_QualType_isBuiltin(const ast_QualType* qt);
static bool ast_QualType_isArray(const ast_QualType* qt);
static bool ast_QualType_isStruct(const ast_QualType* qt);
static bool ast_QualType_isInteger(const ast_QualType* qt);
static bool ast_QualType_isCharPointer(const ast_QualType* qt);
static bool ast_QualType_isPointer(const ast_QualType* qt);
static bool ast_QualType_isFunction(const ast_QualType* qt);
static bool ast_QualType_isEnum(const ast_QualType* qt);
static bool ast_QualType_isVoid(const ast_QualType* qt);
static ast_BuiltinType* ast_QualType_getBuiltin(const ast_QualType* qt);
static ast_BuiltinType* ast_QualType_getBuiltinTypeOrNil(const ast_QualType* qt);
static ast_StructType* ast_QualType_getStructType(const ast_QualType* qt);
static ast_PointerType* ast_QualType_getPointerType(const ast_QualType* qt);
static ast_FunctionType* ast_QualType_getFunctionType(const ast_QualType* qt);
static ast_ArrayType* ast_QualType_getArrayType(const ast_QualType* qt);
static ast_EnumType* ast_QualType_getEnum(const ast_QualType* qt);
static ast_FunctionType* ast_QualType_getFunctionTypeOrNil(const ast_QualType* qt);
static ast_StructType* ast_QualType_getStructTypeOrNil(const ast_QualType* qt);
static ast_PointerType* ast_QualType_getPointerTypeOrNil(const ast_QualType* qt);
static ast_ArrayType* ast_QualType_getArrayTypeOrNil(const ast_QualType* qt);
static ast_EnumType* ast_QualType_getEnumType(const ast_QualType* qt);
static ast_EnumType* ast_QualType_getEnumTypeOrNil(const ast_QualType* qt);
static bool ast_QualType_isChar(const ast_QualType* qt);
static bool ast_QualType_isInt8(const ast_QualType* qt);
static bool ast_QualType_isUInt8(const ast_QualType* qt);
static bool ast_QualType_needsCtvInit(const ast_QualType* qt);
static const char* ast_QualType_diagName(const ast_QualType* qt);
static const char* ast_QualType_diagNameBare(const ast_QualType* qt);
static void ast_QualType_dump(const ast_QualType* qt);
static void ast_QualType_dump_full(const ast_QualType* qt);
static void ast_QualType_printQuoted(const ast_QualType* qt, string_buffer_Buf* out);
static void ast_QualType_print(const ast_QualType* qt, string_buffer_Buf* out);
static void ast_QualType_printInner(const ast_QualType* qt, string_buffer_Buf* out, bool printCanon, bool printModifiers, bool print_error);
static void ast_QualType_fullPrint(const ast_QualType* qt, string_buffer_Buf* out, uint32_t indent);
struct ast_StructType_ {
   ast_Type parent;
   ast_StructTypeDecl* decl;
};

static ast_StructType* ast_StructType_create(ast_context_Context* c, ast_StructTypeDecl* decl);
static ast_StructTypeDecl* ast_StructType_getDecl(const ast_StructType* t);
static ast_Type* ast_StructType_asType(ast_StructType* t);
static void ast_StructType_print(const ast_StructType* t, string_buffer_Buf* out);
static void ast_StructType_fullPrint(const ast_StructType* t, string_buffer_Buf* out, uint32_t indent);
struct ast_TypeRefHolder_ {
   uint64_t ref;
   ast_Ref user;
   ast_Ref prefix;
   ast_Expr* arrays[3];
};

static const char* ast_Ref_getName(const ast_Ref* r);
static void ast_TypeRefHolder_init(ast_TypeRefHolder* h);
static uint32_t ast_TypeRefHolder_getExtraSize(const ast_TypeRefHolder* h);
static void ast_TypeRefHolder_setQualifiers(ast_TypeRefHolder* h, uint32_t qualifiers);
static void ast_TypeRefHolder_setConst(ast_TypeRefHolder* h);
static void ast_TypeRefHolder_setVolatile(ast_TypeRefHolder* h);
static void ast_TypeRefHolder_addPointer(ast_TypeRefHolder* h);
static bool ast_TypeRefHolder_isIncrArray(const ast_TypeRefHolder* h);
static void ast_TypeRefHolder_setIncrArray(ast_TypeRefHolder* h);
static uint32_t ast_TypeRefHolder_getNumArrays(const ast_TypeRefHolder* h);
static void ast_TypeRefHolder_addArray(ast_TypeRefHolder* h, ast_Expr* array);
static void ast_TypeRefHolder_setBuiltin(ast_TypeRefHolder* h, ast_BuiltinKind kind, src_loc_SrcLoc loc);
static void ast_TypeRefHolder_setUser(ast_TypeRefHolder* h, src_loc_SrcLoc loc, uint32_t name_idx);
static void ast_TypeRefHolder_setPrefix(ast_TypeRefHolder* h, src_loc_SrcLoc loc, uint32_t name_idx);
static void ast_TypeRefHolder_fill(const ast_TypeRefHolder* h, ast_TypeRef* dest);
static void ast_TypeRefHolder_dump(const ast_TypeRefHolder* h);
static bool ast_TypeRef_matchesTemplate(const ast_TypeRef* r, uint32_t template_arg);
static void ast_TypeRef_initRef0(ast_TypeRef* r, const ast_TypeRef* r2);
static void ast_TypeRef_instantiate(ast_TypeRef* r, const ast_TypeRef* r1, ast_Instantiator* inst);
static void ast_TypeRef_setDest(ast_TypeRef* r, uint32_t dest);
static uint32_t ast_TypeRef_getDest2(const ast_TypeRef* r);
static uint32_t ast_TypeRef_getExtraSize(const ast_TypeRef* r);
static uint32_t ast_TypeRef_getMaxSizeNoArray(void);
static void* ast_TypeRef_getPointerAfter(const ast_TypeRef* r);
static bool ast_TypeRef_isConst(const ast_TypeRef* r);
static bool ast_TypeRef_isVolatile(const ast_TypeRef* r);
static bool ast_TypeRef_isUser(const ast_TypeRef* r);
static bool ast_TypeRef_isVoid(const ast_TypeRef* r);
static bool ast_TypeRef_isConstCharPtr(const ast_TypeRef* r);
static bool ast_TypeRef_isU32(const ast_TypeRef* r);
static bool ast_TypeRef_hasPrefix(const ast_TypeRef* r);
static bool ast_TypeRef_isIncrArray(const ast_TypeRef* r);
static bool ast_TypeRef_isPointerTo(const ast_TypeRef* r, uint32_t ptr_idx);
static ast_BuiltinKind ast_TypeRef_getBuiltinKind(const ast_TypeRef* r);
static src_loc_SrcLoc ast_TypeRef_getLoc(const ast_TypeRef* r);
static uint32_t ast_TypeRef_getNumPointers(const ast_TypeRef* r);
static const ast_Ref* ast_TypeRef_getUser(const ast_TypeRef* r);
static const ast_Decl* ast_TypeRef_getUserDecl(const ast_TypeRef* r);
static const ast_Ref* ast_TypeRef_getPrefix(const ast_TypeRef* r);
static void ast_TypeRef_setPrefix(ast_TypeRef* r, ast_Decl* d);
static void ast_TypeRef_setUser(ast_TypeRef* r, ast_Decl* d);
static uint32_t ast_TypeRef_getNumArrays(const ast_TypeRef* r);
static ast_Expr* ast_TypeRef_getArray(const ast_TypeRef* r, uint32_t idx);
static ast_Expr** ast_TypeRef_getArray2(ast_TypeRef* r, uint32_t idx);
static void ast_TypeRef_printLiteral(const ast_TypeRef* r, string_buffer_Buf* out, bool print_prefix);
static void ast_TypeRef_print(const ast_TypeRef* r, string_buffer_Buf* out, bool filled);
static void ast_TypeRef_dump(const ast_TypeRef* r);
static const char* ast_TypeRef_diagName(const ast_TypeRef* r);
struct ast_ArrayValueList_ {
   uint32_t count;
   uint32_t capacity;
   ast_ArrayValue** values;
};

static void ast_ArrayValueList_init(ast_ArrayValueList* l, uint32_t initial_size);
static void ast_ArrayValueList_free(ast_ArrayValueList* l);
static void ast_ArrayValueList_add(ast_ArrayValueList* l, ast_ArrayValue* v);
static uint32_t ast_ArrayValueList_size(const ast_ArrayValueList* l);
static ast_ArrayValue** ast_ArrayValueList_get(ast_ArrayValueList* l);
struct ast_ImportDeclList_ {
   uint32_t count;
   uint32_t capacity;
   ast_ImportDecl** decls;
};

struct ast_DeclList_ {
   uint32_t count;
   uint32_t capacity;
   ast_Decl** decls;
};

struct ast_FunctionDeclList_ {
   uint32_t count;
   uint32_t capacity;
   ast_FunctionDecl** decls;
};

struct ast_StaticAssertList_ {
   uint32_t count;
   uint32_t capacity;
   ast_StaticAssert** asserts;
};

struct ast_AST_ {
   ast_Module* mod;
   string_pool_Pool* auxPool;
   void* ptr;
   uint32_t name;
   uint32_t idx;
   bool is_generated;
   ast_ImportDeclList imports;
   ast_DeclList types;
   ast_DeclList variables;
   ast_FunctionDeclList functions;
   ast_StaticAssertList static_asserts;
   ast_ArrayValueList array_values;
   attr_table_Table* attrs;
};

typedef void (*ast_ImportVisitor)(void* arg, ast_ImportDecl* d);

typedef void (*ast_ArrayValueVisitor)(void* arg, ast_ArrayValue* avd);

typedef void (*ast_FunctionVisitor)(void* arg, ast_FunctionDecl* d);

typedef void (*ast_TypeDeclVisitor)(void* arg, ast_Decl* d);

typedef void (*ast_VarDeclVisitor)(void* arg, ast_VarDecl* d);

typedef void (*ast_StaticAssertVisitor)(void* arg, ast_StaticAssert* d);

typedef void (*ast_DeclVisitor)(void* arg, ast_Decl* d);

static ast_AST* ast_AST_create(string_pool_Pool* auxPool, uint32_t name, ast_Module* mod, bool is_generated);
static void ast_AST_free(ast_AST* a);
static const char* ast_AST_getFilename(const ast_AST* a);
static uint32_t ast_AST_getIdx(const ast_AST* a);
static const char* ast_AST_getName(const ast_AST* a);
static uint32_t ast_AST_getNameIdx(const ast_AST* a);
static src_loc_SrcLoc ast_AST_getLoc(const ast_AST* a);
static void ast_AST_setPtr(ast_AST* a, void* ptr);
static void* ast_AST_getPtr(const ast_AST* a);
static ast_Module* ast_AST_getMod(const ast_AST* a);
static void ast_AST_addImport(ast_AST* a, ast_ImportDecl* d);
static ast_ImportDecl* ast_AST_findImport(const ast_AST* a, uint32_t name);
static bool ast_AST_isGenerated(const ast_AST* a);
static void ast_AST_addFunc(ast_AST* a, ast_FunctionDecl* d);
static void ast_AST_addTypeDecl(ast_AST* a, ast_Decl* d);
static void ast_AST_addVarDecl(ast_AST* a, ast_Decl* d);
static void ast_AST_addStaticAssert(ast_AST* a, ast_StaticAssert* s);
static void ast_AST_addArrayValue(ast_AST* a, ast_ArrayValue* v);
static void ast_AST_visitImports(const ast_AST* a, ast_ImportVisitor visitor, void* arg);
static const ast_ImportDeclList* ast_AST_getImports(const ast_AST* a);
static void ast_AST_visitArrayValues(ast_AST* a, ast_ArrayValueVisitor visitor, void* arg);
static void ast_AST_visitStructFunctions(const ast_AST* a, ast_FunctionVisitor visitor, void* arg);
static void ast_AST_visitFunctions(const ast_AST* a, ast_FunctionVisitor visitor, void* arg);
static void ast_AST_visitTypeDecls(const ast_AST* a, ast_TypeDeclVisitor visitor, void* arg);
static void ast_AST_visitVarDecls(const ast_AST* a, ast_VarDeclVisitor visitor, void* arg);
static void ast_AST_visitStaticAsserts(ast_AST* a, ast_StaticAssertVisitor visitor, void* arg);
static void ast_AST_visitDecls(const ast_AST* a, ast_DeclVisitor visitor, void* arg);
static void ast_AST_visitDeclsWithoutImports(const ast_AST* a, ast_DeclVisitor visitor, void* arg);
static ast_Decl* ast_AST_findType(const ast_AST* a, uint32_t name_idx);
static void ast_AST_storeAttr(ast_AST* a, ast_Decl* d, attr_AttrKind kind, const attr_Value* value);
static const attr_Value* ast_AST_getAttr(const ast_AST* a, const ast_Decl* d, attr_AttrKind kind);
static void ast_AST_info(const ast_AST* a, string_buffer_Buf* out);
static void ast_AST_print(const ast_AST* a, string_buffer_Buf* out, bool show_funcs);
static void ast_AST_setExported(ast_AST* a);
static void ast_DeclList_init(ast_DeclList* l, uint32_t initial_size);
static void ast_DeclList_free(ast_DeclList* l);
static void ast_DeclList_add(ast_DeclList* l, ast_Decl* d);
static void ast_DeclList_clear(ast_DeclList* l);
static uint32_t ast_DeclList_size(const ast_DeclList* l);
static ast_Decl* ast_DeclList_get(const ast_DeclList* l, uint32_t idx);
static ast_Decl** ast_DeclList_getDecls(const ast_DeclList* l);
static void ast_DeclList_setSize(ast_DeclList* l, uint32_t size);
static uint32_t ast_DeclList_findIdx(const ast_DeclList* l, const ast_Decl* d);
struct ast_ExprList_ {
   uint32_t count;
   uint32_t capacity;
   ast_Expr** exprs;
};

static void ast_ExprList_init(ast_ExprList* l, uint32_t initial_size);
static void ast_ExprList_free(ast_ExprList* l);
static void ast_ExprList_add(ast_ExprList* l, ast_Expr* d);
static uint32_t ast_ExprList_size(const ast_ExprList* l);
static ast_Expr** ast_ExprList_getExprs(const ast_ExprList* l);
static void ast_FunctionDeclList_init(ast_FunctionDeclList* l);
static void ast_FunctionDeclList_free(ast_FunctionDeclList* l);
static void ast_FunctionDeclList_clear(ast_FunctionDeclList* l);
static void ast_FunctionDeclList_add(ast_FunctionDeclList* l, ast_FunctionDecl* d);
static uint32_t ast_FunctionDeclList_size(const ast_FunctionDeclList* l);
static ast_FunctionDecl** ast_FunctionDeclList_getDecls(const ast_FunctionDeclList* l);
static ast_FunctionDecl* ast_FunctionDeclList_find(const ast_FunctionDeclList* l, uint32_t name_idx);
static void ast_ImportDeclList_init(ast_ImportDeclList* l);
static void ast_ImportDeclList_free(ast_ImportDeclList* l);
static void ast_ImportDeclList_add(ast_ImportDeclList* l, ast_ImportDecl* d);
static uint32_t ast_ImportDeclList_size(const ast_ImportDeclList* l);
static ast_ImportDecl** ast_ImportDeclList_getDecls(const ast_ImportDeclList* l);
static ast_ImportDecl* ast_ImportDeclList_find(const ast_ImportDeclList* l, uint32_t name_idx);
static ast_ImportDecl* ast_ImportDeclList_findAny(const ast_ImportDeclList* l, uint32_t name_idx);
struct ast_TemplateInstance_ {
   ast_QualType qt;
   ast_FunctionDecl* instance;
};

struct ast_TemplateFunction_ {
   const ast_FunctionDecl* fd;
   uint16_t count;
   uint16_t capacity;
   ast_TemplateInstance* instances;
};

struct ast_InstanceTable_ {
   uint32_t count;
   uint32_t capacity;
   ast_TemplateFunction* funcs;
};

static void ast_TemplateFunction_init(ast_TemplateFunction* f, const ast_FunctionDecl* fd);
static void ast_TemplateFunction_resize(ast_TemplateFunction* f, uint16_t capacity);
static uint16_t ast_TemplateFunction_add(ast_TemplateFunction* f, ast_QualType qt, ast_FunctionDecl* instance);
static ast_FunctionDecl* ast_TemplateFunction_find(const ast_TemplateFunction* f, ast_QualType qt);
static ast_FunctionDecl* ast_TemplateFunction_get(const ast_TemplateFunction* f, uint32_t idx);
static void ast_InstanceTable_init(ast_InstanceTable* t);
static void ast_InstanceTable_free(ast_InstanceTable* t);
static void ast_InstanceTable_resize(ast_InstanceTable* t, uint32_t capacity);
static ast_TemplateFunction* ast_InstanceTable_findFunc(const ast_InstanceTable* t, const ast_FunctionDecl* fd);
static ast_FunctionDecl* ast_InstanceTable_find(const ast_InstanceTable* t, const ast_FunctionDecl* fd, ast_QualType qt);
static ast_FunctionDecl* ast_InstanceTable_get(const ast_InstanceTable* t, const ast_FunctionDecl* fd, uint32_t idx);
static uint16_t ast_InstanceTable_add(ast_InstanceTable* t, const ast_FunctionDecl* fd, ast_QualType qt, ast_FunctionDecl* instance);
typedef void (*ast_TemplateVisitor)(void* arg, ast_FunctionDecl* fd, uint32_t idx);

static void ast_InstanceTable_visit(const ast_InstanceTable* t, const ast_FunctionDecl* fd, ast_TemplateVisitor visitor, void* arg);
typedef void (*ast_OpaqueErrorfn)(void* arg, src_loc_SrcLoc loc, ast_Decl* decl);

struct ast_Instantiator_ {
   ast_context_Context* c;
   const ast_TypeRef* ref;
   uint32_t template_name;
   bool used_opaque;
   void* arg;
   ast_OpaqueErrorfn on_error;
};

static void ast_Instantiator_on_opaque(ast_Instantiator* inst, src_loc_SrcLoc loc, ast_Decl* decl);
struct ast_SymbolTable_ {
   uint32_t num_symbols;
   uint32_t max_symbols;
   uint32_t* symbols;
   ast_DeclList decls;
};

struct ast_Module_ {
   uint32_t name_idx;
   bool is_used;
   bool is_external;
   bool is_internal;
   bool is_direct;
   bool is_exported;
   ast_ModuleType* mt;
   ast_AST** files;
   uint32_t num_files;
   uint32_t max_files;
   ast_SymbolTable symbols;
   ast_InstanceTable instances;
};

typedef void (*ast_ASTVisitor)(void* arg, ast_AST* d);

static ast_Module* ast_Module_create(ast_context_Context* c, uint32_t name_idx, bool is_external, bool is_direct);
static void ast_Module_free(ast_Module* m);
static void ast_Module_setUsed(ast_Module* m);
static bool ast_Module_isUsed(const ast_Module* m);
static void ast_Module_setInternal(ast_Module* m);
static bool ast_Module_isInternal(const ast_Module* m);
static bool ast_Module_isExternal(const ast_Module* m);
static void ast_Module_setExported(ast_Module* m);
static bool ast_Module_isExported(const ast_Module* m);
static bool ast_Module_isDirect(const ast_Module* m);
static const ast_SymbolTable* ast_Module_getSymbols(const ast_Module* m);
static ast_ModuleType* ast_Module_getType(const ast_Module* m);
static const char* ast_Module_getFirstFilename(const ast_Module* m);
static void ast_Module_visitASTs(const ast_Module* m, ast_ASTVisitor visitor, void* arg);
static void ast_Module_visitImports(const ast_Module* m, ast_ImportVisitor visitor, void* arg);
static void ast_Module_visitArrayValues(const ast_Module* m, ast_ArrayValueVisitor visitor, void* arg);
static void ast_Module_visitStructFunctions(const ast_Module* m, ast_FunctionVisitor visitor, void* arg);
static void ast_Module_visitFunctions(const ast_Module* m, ast_FunctionVisitor visitor, void* arg);
static void ast_Module_visitTypeDecls(const ast_Module* m, ast_TypeDeclVisitor visitor, void* arg);
static void ast_Module_visitVarDecls(const ast_Module* m, ast_VarDeclVisitor visitor, void* arg);
static void ast_Module_visitStaticAsserts(const ast_Module* m, ast_StaticAssertVisitor visitor, void* arg);
static void ast_Module_visitDecls(const ast_Module* m, ast_DeclVisitor visitor, void* arg);
static ast_Decl* ast_Module_findType(const ast_Module* m, uint32_t name_idx);
static const char* ast_Module_getName(const ast_Module* m);
static uint32_t ast_Module_getNameIdx(const ast_Module* m);
static void ast_Module_resizeFiles(ast_Module* m, uint32_t cap);
static ast_AST* ast_Module_add(ast_Module* m, string_pool_Pool* auxPool, uint32_t filename, bool is_generated);
static void ast_Module_addSymbol(ast_Module* m, uint32_t name_idx, ast_Decl* d);
static ast_Decl* ast_Module_findSymbol(const ast_Module* m, uint32_t name_idx);
static ast_FunctionDecl* ast_Module_findInstance(const ast_Module* m, ast_FunctionDecl* fd, ast_QualType qt);
static uint16_t ast_Module_addInstance(ast_Module* m, ast_FunctionDecl* fd, ast_QualType qt, ast_FunctionDecl* instance);
static ast_FunctionDecl* ast_Module_getInstance(const ast_Module* m, ast_FunctionDecl* fd, uint32_t idx);
static void ast_Module_visitInstances(const ast_Module* m, ast_FunctionDecl* fd, ast_TemplateVisitor visitor, void* arg);
static void ast_Module_info(const ast_Module* m, string_buffer_Buf* out);
static void ast_Module_print(const ast_Module* m, string_buffer_Buf* out, bool show_funcs);
struct ast_PointerPoolSlot_ {
   ast_Type* ptrs[4];
};

struct ast_PointerPool_ {
   ast_context_Context* context;
   uint32_t count;
   uint32_t capacity;
   ast_PointerPoolSlot* slots;
};

static void ast_PointerPool_init(ast_PointerPool* p, ast_context_Context* c);
static void ast_PointerPool_clear(ast_PointerPool* p);
static void ast_PointerPool_resize(ast_PointerPool* p, uint32_t cap);
static ast_Type* ast_PointerPool_getPointer(ast_PointerPool* p, ast_QualType qt);
static void ast_StaticAssertList_init(ast_StaticAssertList* l, uint32_t initial_size);
static void ast_StaticAssertList_free(ast_StaticAssertList* l);
static void ast_StaticAssertList_add(ast_StaticAssertList* l, ast_StaticAssert* v);
static uint32_t ast_StaticAssertList_size(const ast_StaticAssertList* l);
static ast_StaticAssert* ast_StaticAssertList_getAt(const ast_StaticAssertList* l, uint32_t idx);
static ast_StaticAssert** ast_StaticAssertList_get(ast_StaticAssertList* l);
struct ast_Stat_ {
   uint32_t count;
   uint32_t size;
};

struct ast_Stats_ {
   ast_Stat types[8];
   ast_Stat exprs[22];
   ast_Stat stmts[16];
   ast_Stat decls[8];
   ast_Stat others[3];
   ast_Stat arrayValues;
   ast_Stat staticAsserts;
};

static const char* ast_other_names[3] = { "ArrayValue", "StaticAssert", "SwitchCase" };

static void ast_Stats_reset(ast_Stats* s);
static void ast_Stats_addType(ast_TypeKind kind, uint32_t size);
static void ast_Stats_addExpr(ast_ExprKind kind, uint32_t size);
static void ast_Stats_addStmt(ast_StmtKind kind, uint32_t size);
static void ast_Stats_addDecl(ast_DeclKind kind, uint32_t size);
static void ast_Stats_addArrayValue(uint32_t size);
static void ast_Stats_addStaticAssert(uint32_t size);
static void ast_Stats_addSwitchCase(uint32_t size);
static void ast_Stats_dump(const ast_Stats* s);
struct ast_StringTypeSlot_ {
   uint32_t len;
   ast_Type* type_;
};

struct ast_StringTypePool_ {
   uint32_t count;
   uint32_t capacity;
   ast_StringTypeSlot* slots;
   ast_context_Context* context;
};

static void ast_StringTypePool_init(ast_StringTypePool* p, ast_context_Context* c);
static void ast_StringTypePool_clear(ast_StringTypePool* p);
static void ast_StringTypePool_resize(ast_StringTypePool* p, uint32_t cap);
static ast_QualType ast_StringTypePool_get(ast_StringTypePool* p, uint32_t len);
static void ast_SymbolTable_init(ast_SymbolTable* t, uint32_t initial);
static void ast_SymbolTable_free(ast_SymbolTable* t);
static uint32_t ast_SymbolTable_size(const ast_SymbolTable* t);
static ast_Decl** ast_SymbolTable_getDecls(const ast_SymbolTable* l);
static void ast_SymbolTable_crop(ast_SymbolTable* t, uint32_t size);
static void ast_SymbolTable_resize(ast_SymbolTable* t, uint32_t capacity);
static void ast_SymbolTable_add(ast_SymbolTable* t, uint32_t name_idx, ast_Decl* d);
static ast_Decl* ast_SymbolTable_find(const ast_SymbolTable* t, uint32_t name_idx);
static void ast_SymbolTable_print(const ast_SymbolTable* t, string_buffer_Buf* out);
static void ast_SymbolTable_dump(const ast_SymbolTable* t);
struct ast_Globals_ {
   const char* names_start;
   ast_PointerPool pointers;
   ast_StringTypePool string_types;
   uint32_t wordsize;
   bool use_color;
   uint32_t ast_count;
   uint32_t ast_capacity;
   ast_AST** ast_list;
   uint32_t builtinType_sizes[15];
   uint32_t builtinType_width[15];
   ast_BuiltinKind builtinType_baseTypes[15];
   uint32_t attr_name_indexes[19];
   ast_Stats stats;
};

typedef bool (*ast_AttrHandlerFn)(void* arg, ast_Decl* d, const attr_Attr* a);

static const ast_QualType ast_QualType_Invalid = { };

static ast_Globals* ast_globals = NULL;

static ast_QualType* ast_builtins = NULL;

static const char* ast_col_Stmt = color_Bmagenta;

static const char* ast_col_Decl = color_Bgreen;

static const char* ast_col_Expr = color_Bmagenta;

static const char* ast_col_Attr = color_Blue;

static const char* ast_col_Template = color_Green;

static const char* ast_col_Type = color_Green;

static const char* ast_col_Value = color_Bcyan;

static const char* ast_col_Error = color_Red;

static const char* ast_col_Calc = color_Yellow;

static const char* ast_col_Normal = color_Normal;

static ast_Globals* ast_getGlobals(void);
static void ast_setGlobals(ast_Globals* g);
static void ast_init(ast_context_Context* c, string_pool_Pool* astPool, uint32_t wordsize, bool use_color);
static void ast_deinit(bool print_stats);
static uint32_t ast_getWordSize(void);
static bool ast_useColor(void);
static ast_QualType ast_getStringType(uint32_t len);
static const char* ast_idx2name(uint32_t idx);
static ast_QualType ast_getVoidPtr(void);
static ast_Type* ast_getPointerType(ast_QualType inner);
static uint32_t ast_addAST(ast_AST* ast_);
static uint32_t ast_ast2idx(const ast_AST* ast_);
static ast_AST* ast_idx2ast(uint32_t idx);
static void ast_setTypePublicUsed(ast_QualType qt);
static const char* ast_getPrefixedName(const ast_Decl* d);
static bool ast_isGlobal(const ast_Decl* d);
static ast_QualType ast_getNativeType(void);
static void ast_Decl_init(ast_Decl* d, ast_DeclKind k, uint32_t name_idx, src_loc_SrcLoc loc, bool is_public, ast_QualType qt, uint32_t ast_idx)
{
   d->bits = 0;
   d->declBits.kind = k;
   d->declBits.is_public = is_public;
   d->loc = loc;
   d->name_idx = name_idx;
   d->ast_idx = ast_idx;
   d->qt = qt;
}

static ast_DeclKind ast_Decl_getKind(const ast_Decl* d)
{
   return ((ast_DeclKind)(d->declBits.kind));
}

static ast_DeclCheckState ast_Decl_getCheckState(const ast_Decl* d)
{
   return ((ast_DeclCheckState)(d->declBits.check_state));
}

static void ast_Decl_setCheckState(ast_Decl* d, ast_DeclCheckState s)
{
   d->declBits.check_state = s;
}

static bool ast_Decl_isChecked(const ast_Decl* d)
{
   return (d->declBits.check_state == ast_DeclCheckState_Checked);
}

static void ast_Decl_setChecked(ast_Decl* d)
{
   d->declBits.check_state = ast_DeclCheckState_Checked;
}

static void ast_Decl_setHasAttr(ast_Decl* d)
{
   d->declBits.has_attr = 1;
}

static bool ast_Decl_hasAttr(const ast_Decl* d)
{
   return d->declBits.has_attr;
}

static void ast_Decl_setAttrExport(ast_Decl* d)
{
   d->declBits.attr_export = 1;
}

static void ast_Decl_setExportedIfPublic(ast_Decl* d)
{
   if (d->declBits.is_public) d->declBits.attr_export = 1;
}

static bool ast_Decl_isExported(const ast_Decl* d)
{
   return d->declBits.attr_export;
}

static void ast_Decl_setAttrUnused(ast_Decl* d)
{
   d->declBits.attr_unused = 1;
}

static bool ast_Decl_hasAttrUnused(const ast_Decl* d)
{
   return d->declBits.attr_unused;
}

static bool ast_Decl_isStructType(const ast_Decl* d)
{
   return (ast_Decl_getKind(d) == ast_DeclKind_StructType);
}

static bool ast_Decl_isImport(const ast_Decl* d)
{
   return (ast_Decl_getKind(d) == ast_DeclKind_Import);
}

static bool ast_Decl_isEnum(const ast_Decl* d)
{
   return (ast_Decl_getKind(d) == ast_DeclKind_EnumType);
}

static bool ast_Decl_isEnumConstant(const ast_Decl* d)
{
   return (ast_Decl_getKind(d) == ast_DeclKind_EnumConstant);
}

static bool ast_Decl_isFunction(const ast_Decl* d)
{
   return (ast_Decl_getKind(d) == ast_DeclKind_Function);
}

static bool ast_Decl_isFunctionType(const ast_Decl* d)
{
   return (ast_Decl_getKind(d) == ast_DeclKind_FunctionType);
}

static bool ast_Decl_isVariable(const ast_Decl* d)
{
   return (ast_Decl_getKind(d) == ast_DeclKind_Variable);
}

static const char* ast_Decl_getName(const ast_Decl* d)
{
   return ast_idx2name(d->name_idx);
}

static uint32_t ast_Decl_getNameIdx(const ast_Decl* d)
{
   return d->name_idx;
}

static const char* ast_Decl_getModuleName(const ast_Decl* d)
{
   if ((d->ast_idx == 0)) return NULL;

   const ast_AST* a = ast_Decl_getAST(d);
   const ast_Module* mod = ast_AST_getMod(a);
   return ast_Module_getName(mod);
}

static src_loc_SrcLoc ast_Decl_getLoc(const ast_Decl* d)
{
   return d->loc;
}

static ast_QualType ast_Decl_getType(const ast_Decl* d)
{
   return d->qt;
}

static void ast_Decl_setType(ast_Decl* d, ast_QualType qt)
{
   d->qt = qt;
}

static ast_AST* ast_Decl_getAST(const ast_Decl* d)
{
   return ast_idx2ast(d->ast_idx);
}

static uint32_t ast_Decl_getASTIdx(const ast_Decl* d)
{
   return d->ast_idx;
}

static ast_Module* ast_Decl_getModule(const ast_Decl* d)
{
   return ast_idx2ast(d->ast_idx)->mod;
}

static bool ast_Decl_isPublic(const ast_Decl* d)
{
   return d->declBits.is_public;
}

static bool ast_Decl_isUsed(const ast_Decl* d)
{
   return d->declBits.is_used;
}

static bool ast_Decl_isUsedPublic(const ast_Decl* d)
{
   return d->declBits.is_used_public;
}

static void ast_Decl_setUsed(ast_Decl* d)
{
   d->declBits.is_used = true;
}

static void ast_Decl_setUsedPublic(ast_Decl* d)
{
   d->declBits.is_used_public = true;
}

static bool ast_Decl_isExternal(const ast_Decl* d)
{
   return d->declBits.is_external;
}

static void ast_Decl_setExternal(ast_Decl* d)
{
   d->declBits.is_external = 1;
}

static bool ast_Decl_isGenerated(const ast_Decl* d)
{
   return d->declBits.is_generated;
}

static void ast_Decl_setGenerated(ast_Decl* d)
{
   d->declBits.is_generated = 1;
}

static void ast_Decl_clearGenerated(ast_Decl* d)
{
   d->declBits.is_generated = 0;
}

static void ast_Decl_dump(const ast_Decl* d)
{
   string_buffer_Buf* out = string_buffer_create((10 * 4096), ast_useColor(), 2);
   ast_Decl_print(d, out, 0);
   string_buffer_Buf_color(out, ast_col_Normal);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static bool ast_Decl_isTypeDecl(const ast_Decl* d)
{
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_Function:
      break;
   case ast_DeclKind_Import:
      break;
   case ast_DeclKind_StructType:
      return true;
   case ast_DeclKind_EnumType:
      return true;
   case ast_DeclKind_EnumConstant:
      break;
   case ast_DeclKind_FunctionType:
      return true;
   case ast_DeclKind_AliasType:
      return true;
   case ast_DeclKind_Variable:
      break;
   }
   return false;
}

static bool ast_Decl_isVarDecl(const ast_Decl* d)
{
   return (ast_Decl_getKind(d) == ast_DeclKind_Variable);
}

static const char* ast_Decl_getKindName(const ast_Decl* d)
{
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_Function:
      return "function";
   case ast_DeclKind_Import:
      return "import";
   case ast_DeclKind_StructType:
      return "type";
   case ast_DeclKind_EnumType:
      return "type";
   case ast_DeclKind_EnumConstant:
      return "enum constant";
   case ast_DeclKind_FunctionType:
      return "type";
   case ast_DeclKind_AliasType:
      return "type";
   case ast_DeclKind_Variable:
      return "variable";
   }
   return "";
}

static const char* ast_Decl_getCName(const ast_Decl* d)
{
   if (!ast_Decl_hasAttr(d)) return NULL;

   const ast_AST* a = ast_Decl_getAST(d);
   const attr_Value* cname = ast_AST_getAttr(a, d, attr_AttrKind_CName);
   if (cname) return ast_idx2name(cname->text);

   return NULL;
}

static const char* ast_Decl_getSection(const ast_Decl* d)
{
   if (!ast_Decl_hasAttr(d)) return NULL;

   const ast_AST* a = ast_Decl_getAST(d);
   const attr_Value* section = ast_AST_getAttr(a, d, attr_AttrKind_Section);
   if (section) return ast_idx2name(section->text);

   return NULL;
}

static const char* ast_Decl_getFullName(const ast_Decl* d)
{
   static char tmp[128];
   const char* modname = ast_Decl_getModuleName(d);
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_Function: {
      const ast_FunctionDecl* fd = ((ast_FunctionDecl*)(d));
      if (ast_FunctionDecl_hasPrefix(fd)) {
         sprintf(tmp, "%s.%s.%s", modname, ast_FunctionDecl_getPrefixName(fd), ast_Decl_getName(d));
      } else {
         sprintf(tmp, "%s.%s", modname, ast_Decl_getName(d));
      }
      break;
   }
   case ast_DeclKind_Import:
      sprintf(tmp, "%s", ast_Decl_getName(d));
      break;
   case ast_DeclKind_EnumConstant: {
      ast_QualType qt = ast_Decl_getType(d);
      ast_EnumType* et = ast_QualType_getEnumType(&qt);
      ast_EnumTypeDecl* etd = ast_EnumType_getDecl(et);
      ast_Decl* ed = ((ast_Decl*)(etd));
      sprintf(tmp, "%s.%s.%s", modname, ast_Decl_getName(ed), ast_Decl_getName(d));
      break;
   }
   default:
      sprintf(tmp, "%s.%s", modname, ast_Decl_getName(d));
      break;
   }
   return tmp;
}

static void ast_Decl_print(const ast_Decl* d, string_buffer_Buf* out, uint32_t indent)
{
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_Function:
      ast_FunctionDecl_print(((ast_FunctionDecl*)(d)), out, indent);
      break;
   case ast_DeclKind_Import:
      ast_ImportDecl_print(((ast_ImportDecl*)(d)), out, indent);
      break;
   case ast_DeclKind_StructType:
      ast_StructTypeDecl_print(((ast_StructTypeDecl*)(d)), out, indent);
      break;
   case ast_DeclKind_EnumType:
      ast_EnumTypeDecl_print(((ast_EnumTypeDecl*)(d)), out, indent);
      break;
   case ast_DeclKind_EnumConstant:
      ast_EnumConstantDecl_print(((ast_EnumConstantDecl*)(d)), out, indent);
      break;
   case ast_DeclKind_FunctionType:
      ast_FunctionTypeDecl_print(((ast_FunctionTypeDecl*)(d)), out, indent);
      break;
   case ast_DeclKind_AliasType:
      ast_AliasTypeDecl_print(((ast_AliasTypeDecl*)(d)), out, indent);
      break;
   case ast_DeclKind_Variable:
      ast_VarDecl_print(((ast_VarDecl*)(d)), out, indent);
      break;
   }
}

static void ast_Decl_printKind(const ast_Decl* d, string_buffer_Buf* out, uint32_t indent, bool print_type)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_color(out, ast_col_Decl);
   string_buffer_Buf_add(out, ast_declKind_names[ast_Decl_getKind(d)]);
   if (print_type) {
      string_buffer_Buf_space(out);
      ast_QualType_printQuoted(&d->qt, out);
   }
}

static void ast_Decl_printName(const ast_Decl* d, string_buffer_Buf* out)
{
   string_buffer_Buf_space(out);
   string_buffer_Buf_color(out, ast_col_Value);
   if (d->name_idx) {
      string_buffer_Buf_add(out, ast_Decl_getName(d));
   } else {
      string_buffer_Buf_add(out, "(nil)");
   }
}

static void ast_Decl_printBits(const ast_Decl* d, string_buffer_Buf* out)
{
   string_buffer_Buf_color(out, ast_col_Attr);
   if (ast_Decl_isPublic(d)) string_buffer_Buf_add(out, " public");
   ast_DeclCheckState cs = ast_Decl_getCheckState(d);
   if ((cs != ast_DeclCheckState_Checked)) {
      string_buffer_Buf_space(out);
      string_buffer_Buf_add(out, ast_declCheckState_names[cs]);
   }
   if (d->declBits.attr_unused) {
      string_buffer_Buf_add(out, " unused");
   }
   if (d->declBits.has_attr) string_buffer_Buf_add(out, " attr");
   if (d->declBits.attr_export) string_buffer_Buf_add(out, " export");
   if (!ast_Decl_isUsed(d)) {
      string_buffer_Buf_color(out, ast_col_Expr);
      string_buffer_Buf_add(out, " unused");
   }
}

static void ast_Decl_printAttrs(const ast_Decl* d, string_buffer_Buf* out)
{
   if (!ast_Decl_hasAttr(d)) return;

   const ast_AST* a = ast_Decl_getAST(d);
   const attr_Value* cname = ast_AST_getAttr(a, d, attr_AttrKind_CName);
   if (cname) {
      string_buffer_Buf_print(out, " cname='%s'", ast_idx2name(cname->text));
   }
   const attr_Value* section = ast_AST_getAttr(a, d, attr_AttrKind_Section);
   if (section) {
      string_buffer_Buf_print(out, " section='%s'", ast_idx2name(section->text));
   }
}

static void ast_Decl_printUsed(const ast_Decl* d, string_buffer_Buf* out)
{
   string_buffer_Buf_color(out, ast_col_Attr);
   string_buffer_Buf_print(out, " used=%u/%u", ast_Decl_isUsed(d), ast_Decl_isUsedPublic(d));
}

static ast_AliasTypeDecl* ast_AliasTypeDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, const ast_TypeRefHolder* ref)
{
   uint32_t size = (32 + ast_TypeRefHolder_getExtraSize(ref));
   ast_AliasTypeDecl* d = ast_context_Context_alloc(c, size);
   ast_AliasType* at = ast_AliasType_create(c, d);
   ast_Decl_init(&d->parent, ast_DeclKind_AliasType, name, loc, is_public, ast_QualType_init(((ast_Type*)(at))), ast_idx);
   ast_TypeRefHolder_fill(ref, &d->typeRef);
   ast_Stats_addDecl(ast_DeclKind_AliasType, size);
   return d;
}

static ast_Decl* ast_AliasTypeDecl_asDecl(ast_AliasTypeDecl* d)
{
   return &d->parent;
}

static ast_TypeRef* ast_AliasTypeDecl_getTypeRef(ast_AliasTypeDecl* d)
{
   return &d->typeRef;
}

static void ast_AliasTypeDecl_print(const ast_AliasTypeDecl* d, string_buffer_Buf* out, uint32_t indent)
{
   ast_Decl_printKind(&d->parent, out, indent, true);
   ast_Decl_printBits(&d->parent, out);
   ast_Decl_printAttrs(&d->parent, out);
   ast_Decl_printName(&d->parent, out);
   if (ast_QualType_isInvalid(&d->parent.qt)) {
      string_buffer_Buf_space(out);
      ast_TypeRef_print(&d->typeRef, out, true);
      string_buffer_Buf_newline(out);
   }
}

static ast_ArrayValue* ast_ArrayValue_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, ast_Expr* value)
{
   ast_ArrayValue* d = ast_context_Context_alloc(c, 16);
   d->name_idx = name;
   d->loc = loc;
   d->value = value;
   ast_Stats_addArrayValue(16);
   return d;
}

static uint32_t ast_ArrayValue_getNameIdx(const ast_ArrayValue* d)
{
   return d->name_idx;
}

static src_loc_SrcLoc ast_ArrayValue_getLoc(const ast_ArrayValue* d)
{
   return d->loc;
}

static ast_Expr* ast_ArrayValue_getValue(const ast_ArrayValue* d)
{
   return d->value;
}

static void ast_ArrayValue_print(const ast_ArrayValue* d, string_buffer_Buf* out)
{
}

static ast_DeclStmt* ast_DeclStmt_create(ast_context_Context* c, ast_VarDecl* decl)
{
   ast_DeclStmt* s = ast_context_Context_alloc(c, 16);
   ast_Stmt_init(&s->parent, ast_StmtKind_Decl);
   s->decl = decl;
   ast_Stats_addStmt(ast_StmtKind_Decl, 16);
   return s;
}

static ast_Stmt* ast_DeclStmt_instantiate(ast_DeclStmt* s, ast_Instantiator* inst)
{
   ast_VarDecl* decl2 = ast_VarDecl_instantiate(s->decl, inst);
   return ((ast_Stmt*)(ast_DeclStmt_create(inst->c, decl2)));
}

static ast_VarDecl* ast_DeclStmt_getDecl(const ast_DeclStmt* d)
{
   return d->decl;
}

static void ast_DeclStmt_print(const ast_DeclStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
   ast_VarDecl_print(s->decl, out, (indent + 1));
}

static ast_EnumConstantDecl* ast_EnumConstantDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, ast_Expr* initValue)
{
   uint32_t size = 40;
   if (initValue) size += 8;
   ast_EnumConstantDecl* d = ast_context_Context_alloc(c, size);
   ast_Decl_init(&d->parent, ast_DeclKind_EnumConstant, name, loc, is_public, ast_QualType_Invalid, ast_idx);
   d->value.is_signed = false;
   d->value.uvalue = 0;
   if (initValue) {
      d->parent.enumConstantDeclBits.has_init = 1;
      d->init[0] = initValue;
   }
   ast_Stats_addDecl(ast_DeclKind_EnumConstant, size);
   return d;
}

static ast_Decl* ast_EnumConstantDecl_asDecl(ast_EnumConstantDecl* d)
{
   return &d->parent;
}

static ast_Value ast_EnumConstantDecl_getValue(const ast_EnumConstantDecl* d)
{
   return d->value;
}

static void ast_EnumConstantDecl_setValue(ast_EnumConstantDecl* d, ast_Value value)
{
   d->value = value;
}

static void ast_EnumConstantDecl_setIndex(ast_EnumConstantDecl* d, uint32_t index)
{
   d->parent.enumConstantDeclBits.enum_index = index;
}

static uint32_t ast_EnumConstantDecl_getIndex(const ast_EnumConstantDecl* d)
{
   return d->parent.enumConstantDeclBits.enum_index;
}

static ast_Expr* ast_EnumConstantDecl_getInit(const ast_EnumConstantDecl* d)
{
   if (d->parent.enumConstantDeclBits.has_init) return d->init[0];

   return NULL;
}

static ast_Expr** ast_EnumConstantDecl_getInit2(ast_EnumConstantDecl* d)
{
   if (d->parent.enumConstantDeclBits.has_init) return &d->init[0];

   return NULL;
}

static ast_EnumTypeDecl* ast_EnumConstantDecl_getEnum(const ast_EnumConstantDecl* d)
{
   ast_QualType qt = ast_Decl_getType(&d->parent);
   ast_EnumType* et = ((ast_EnumType*)(ast_QualType_getType(&qt)));
   return ast_EnumType_getDecl(et);
}

static void ast_EnumConstantDecl_print(const ast_EnumConstantDecl* d, string_buffer_Buf* out, uint32_t indent)
{
   ast_Decl_printKind(&d->parent, out, indent, true);
   ast_Decl_printBits(&d->parent, out);
   ast_Decl_printName(&d->parent, out);
   string_buffer_Buf_color(out, ast_col_Calc);
   string_buffer_Buf_print(out, " [%u] %s", ast_EnumConstantDecl_getIndex(d), ast_Value_str(&d->value));
   string_buffer_Buf_newline(out);
   if (d->parent.enumConstantDeclBits.has_init) ast_Expr_print(d->init[0], out, (indent + 1));
}

static ast_EnumTypeDecl* ast_EnumTypeDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, ast_QualType implType, bool is_incremental, ast_EnumConstantDecl** constants, uint32_t num_constants)
{
   uint32_t size = (32 + (num_constants * 8));
   if (is_incremental) size += 8;
   ast_EnumTypeDecl* d = ast_context_Context_alloc(c, size);
   ast_EnumType* etype = ast_EnumType_create(c, d);
   ast_QualType qt = ast_QualType_init(((ast_Type*)(etype)));
   ast_Decl_init(&d->parent, ast_DeclKind_EnumType, name, loc, is_public, qt, ast_idx);
   d->parent.enumTypeDeclBits.is_incremental = is_incremental;
   d->parent.enumTypeDeclBits.num_constants = num_constants;
   d->implType = implType;
   if (is_incremental) {
      c2_assert(((num_constants == 0)) != 0, "ast/enum_type_decl.c2:57: ast.EnumTypeDecl.create", "num_constants==0");
      d->incr_constants[0] = NULL;
   } else {
      memcpy(((void*)(d->constants)), ((void*)(constants)), (num_constants * 8));
      for (uint32_t i = 0; (i < num_constants); i++) {
         ast_Decl_setType(ast_EnumConstantDecl_asDecl(constants[i]), qt);
      }
   }
   ast_Stats_addDecl(ast_DeclKind_EnumType, size);
   return d;
}

static void ast_EnumTypeDecl_setIncrMembers(ast_EnumTypeDecl* d, ast_Decl** constants, uint32_t num_constants)
{
   d->parent.enumTypeDeclBits.num_constants = num_constants;
   memcpy(((void*)(d->constants)), ((void*)(constants)), (num_constants * 8));
}

static ast_QualType ast_EnumTypeDecl_getImplType(const ast_EnumTypeDecl* d)
{
   return d->implType;
}

static ast_Decl* ast_EnumTypeDecl_asDecl(ast_EnumTypeDecl* d)
{
   return &d->parent;
}

static bool ast_EnumTypeDecl_isIncremental(const ast_EnumTypeDecl* d)
{
   return d->parent.enumTypeDeclBits.is_incremental;
}

static uint32_t ast_EnumTypeDecl_getNumConstants(const ast_EnumTypeDecl* d)
{
   return d->parent.enumTypeDeclBits.num_constants;
}

static ast_EnumConstantDecl** ast_EnumTypeDecl_getConstants(ast_EnumTypeDecl* d)
{
   if (ast_EnumTypeDecl_isIncremental(d)) {
      return d->incr_constants[0];
   }
   return d->constants;
}

static void ast_EnumTypeDecl_setIncrConstants(ast_EnumTypeDecl* d, ast_context_Context* c, ast_IdentifierExpr** constants, uint32_t count)
{
   const uint32_t size = (count * 8);
   ast_EnumConstantDecl** decls = ast_context_Context_alloc(c, size);
   ast_QualType qt = ast_Decl_getType(&d->parent);
   for (uint32_t i = 0; (i < count); i++) {
      ast_IdentifierExpr* id = constants[i];
      decls[i] = ast_EnumConstantDecl_create(c, ast_IdentifierExpr_getNameIdx(id), ast_Expr_getLoc(ast_IdentifierExpr_asExpr(id)), ast_Decl_isPublic(&d->parent), d->parent.ast_idx, NULL);
      ast_Decl_setType(ast_EnumConstantDecl_asDecl(decls[i]), qt);
   }
   d->incr_constants[0] = decls;
   d->parent.enumTypeDeclBits.num_constants = count;
}

static ast_EnumConstantDecl* ast_EnumTypeDecl_findConstant(ast_EnumTypeDecl* d, uint32_t name_idx)
{
   ast_EnumConstantDecl** constants = d->constants;
   if (ast_EnumTypeDecl_isIncremental(d)) constants = d->incr_constants[0];
   for (uint32_t i = 0; (i < ast_EnumTypeDecl_getNumConstants(d)); i++) {
      ast_EnumConstantDecl* ecd = constants[i];
      ast_Decl* ed = ((ast_Decl*)(ecd));
      if ((ast_Decl_getNameIdx(ed) == name_idx)) return ecd;

   }
   return NULL;
}

static ast_EnumConstantDecl* ast_EnumTypeDecl_findConstantIdx(ast_EnumTypeDecl* d, uint32_t name_idx, uint32_t* idx)
{
   ast_EnumConstantDecl** constants = d->constants;
   if (ast_EnumTypeDecl_isIncremental(d)) constants = d->incr_constants[0];
   for (uint32_t i = 0; (i < ast_EnumTypeDecl_getNumConstants(d)); i++) {
      ast_EnumConstantDecl* ecd = constants[i];
      ast_Decl* ed = ((ast_Decl*)(ecd));
      if ((ast_Decl_getNameIdx(ed) == name_idx)) {
         *idx = i;
         return ecd;
      }
   }
   return NULL;
}

static ast_EnumConstantDecl* ast_EnumTypeDecl_getConstant(const ast_EnumTypeDecl* d, uint32_t idx)
{
   if (ast_EnumTypeDecl_isIncremental(d)) return d->incr_constants[0][idx];

   return d->constants[idx];
}

static void ast_EnumTypeDecl_print(ast_EnumTypeDecl* d, string_buffer_Buf* out, uint32_t indent)
{
   ast_Decl_printKind(&d->parent, out, indent, true);
   ast_Decl_printBits(&d->parent, out);
   if (ast_EnumTypeDecl_isIncremental(d)) string_buffer_Buf_add(out, " incremental");
   ast_Decl_printAttrs(&d->parent, out);
   ast_Decl_printName(&d->parent, out);
   string_buffer_Buf_space(out);
   ast_QualType_print(&d->implType, out);
   string_buffer_Buf_newline(out);
   ast_EnumConstantDecl** constants = d->constants;
   if (ast_EnumTypeDecl_isIncremental(d)) constants = d->incr_constants[0];
   for (uint32_t i = 0; (i < d->parent.enumTypeDeclBits.num_constants); i++) {
      ast_EnumConstantDecl_print(constants[i], out, (indent + 1));
   }
}

static ast_FunctionDecl* ast_FunctionDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, const ast_TypeRefHolder* rtype, const ast_Ref* prefix, ast_VarDecl** params, uint32_t num_params, bool is_variadic, bool is_type)
{
   uint32_t size = ((64 + (num_params * 8)) + ast_TypeRefHolder_getExtraSize(rtype));
   if (prefix) size += 16;
   ast_FunctionDecl* d = ast_context_Context_alloc(c, size);
   ast_FunctionType* ftype = ast_FunctionType_create(c, d);
   ast_QualType qt = ast_QualType_init(ast_FunctionType_asType(ftype));
   ast_Decl_init(&d->parent, ast_DeclKind_Function, name, loc, is_public, qt, ast_idx);
   d->parent.functionDeclBits.is_variadic = is_variadic;
   d->parent.functionDeclBits.call_kind = prefix ? ast_CallKind_StaticStructFunc : ast_CallKind_Normal;
   d->parent.functionDeclBits.is_type = is_type;
   d->num_params = ((uint8_t)(num_params));
   d->attr_printf_arg = 0;
   d->instance_idx = 0;
   d->template_name = 0;
   d->template_loc = 0;
   d->num_auto_args = 0;
   d->rt = ast_QualType_Invalid;
   ast_TypeRefHolder_fill(rtype, &d->rtype);
   d->body = NULL;
   uint8_t* tail = ast_TypeRef_getPointerAfter(&d->rtype);
   if (prefix) {
      d->parent.functionDeclBits.has_prefix = 1;
      memcpy(tail, prefix, 16);
      tail += 16;
   }
   if (num_params) {
      memcpy(tail, ((void*)(params)), (num_params * 8));
   }
   ast_Stats_addDecl(ast_DeclKind_Function, size);
   return d;
}

static ast_FunctionDecl* ast_FunctionDecl_createTemplate(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, const ast_TypeRefHolder* rtype, uint32_t template_name, src_loc_SrcLoc template_loc, ast_VarDecl** params, uint32_t num_params, bool is_variadic)
{
   uint32_t size = ((64 + (num_params * 8)) + ast_TypeRefHolder_getExtraSize(rtype));
   ast_FunctionDecl* d = ast_context_Context_alloc(c, size);
   ast_FunctionType* ftype = ast_FunctionType_create(c, d);
   ast_QualType qt = ast_QualType_init(ast_FunctionType_asType(ftype));
   ast_Decl_init(&d->parent, ast_DeclKind_Function, name, loc, is_public, qt, ast_idx);
   d->parent.functionDeclBits.is_variadic = is_variadic;
   d->parent.functionDeclBits.call_kind = ast_CallKind_Normal;
   d->parent.functionDeclBits.is_template = 1;
   d->parent.functionDeclBits.is_type = false;
   d->num_params = ((uint8_t)(num_params));
   d->attr_printf_arg = 0;
   d->instance_idx = 0;
   d->template_name = template_name;
   d->template_loc = template_loc;
   d->num_auto_args = 0;
   d->rt = ast_QualType_Invalid;
   ast_TypeRefHolder_fill(rtype, &d->rtype);
   d->body = NULL;
   uint8_t* tail = ast_TypeRef_getPointerAfter(&d->rtype);
   if (num_params) {
      memcpy(tail, ((void*)(params)), (num_params * 8));
   }
   ast_Stats_addDecl(ast_DeclKind_Function, size);
   return d;
}

static ast_FunctionDecl* ast_FunctionDecl_instantiate(const ast_FunctionDecl* fd, ast_Instantiator* inst)
{
   bool rtype_matches = ast_TypeRef_matchesTemplate(&fd->rtype, fd->template_name);
   uint32_t extra = rtype_matches ? ast_TypeRef_getExtraSize(inst->ref) : ast_TypeRef_getExtraSize(&fd->rtype);
   uint32_t size = ((64 + (fd->num_params * 8)) + extra);
   ast_FunctionDecl* fd2 = ast_context_Context_alloc(inst->c, size);
   memcpy(&fd2->parent, &fd->parent, 24);
   fd2->parent.functionDeclBits.is_template = 0;
   ast_FunctionType* ftype = ast_FunctionType_create(inst->c, fd2);
   fd2->parent.qt = ast_QualType_init(ast_FunctionType_asType(ftype));
   fd2->body = ast_CompoundStmt_instantiate(fd->body, inst);
   fd2->rt = ast_QualType_Invalid;
   fd2->num_params = fd->num_params;
   fd2->attr_printf_arg = fd->attr_printf_arg;
   fd2->instance_idx = 0;
   fd2->template_name = 0;
   ast_TypeRef_instantiate(&fd2->rtype, &fd->rtype, inst);
   ast_VarDecl** src = ((ast_VarDecl**)(ast_TypeRef_getPointerAfter(&fd->rtype)));
   ast_VarDecl** dst = ((ast_VarDecl**)(ast_TypeRef_getPointerAfter(&fd2->rtype)));
   for (uint32_t i = 0; (i < fd2->num_params); i++) {
      dst[i] = ast_VarDecl_instantiate(src[i], inst);
   }
   ast_Stats_addDecl(ast_DeclKind_Function, size);
   return fd2;
}

static void ast_FunctionDecl_setBody(ast_FunctionDecl* d, ast_CompoundStmt* body)
{
   d->body = body;
}

static ast_CompoundStmt* ast_FunctionDecl_getBody(const ast_FunctionDecl* d)
{
   return d->body;
}

static bool ast_FunctionDecl_isType(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.is_type;
}

static void ast_FunctionDecl_setRType(ast_FunctionDecl* d, ast_QualType rt)
{
   if (!ast_QualType_isVoid(&rt)) d->parent.functionDeclBits.has_return = 1;
   d->rt = rt;
}

static ast_QualType ast_FunctionDecl_getRType(const ast_FunctionDecl* d)
{
   return d->rt;
}

static bool ast_FunctionDecl_hasReturn(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.has_return;
}

static ast_Decl* ast_FunctionDecl_asDecl(ast_FunctionDecl* d)
{
   return &d->parent;
}

static ast_TypeRef* ast_FunctionDecl_getReturnTypeRef(ast_FunctionDecl* d)
{
   return &d->rtype;
}

static bool ast_FunctionDecl_hasPrefix(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.has_prefix;
}

static bool ast_FunctionDecl_isTemplate(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.is_template;
}

static uint32_t ast_FunctionDecl_getTemplateNameIdx(const ast_FunctionDecl* d)
{
   return d->template_name;
}

static src_loc_SrcLoc ast_FunctionDecl_getTemplateLoc(const ast_FunctionDecl* d)
{
   return d->template_loc;
}

static void ast_FunctionDecl_setTemplateInstanceIdx(ast_FunctionDecl* d, uint16_t idx)
{
   d->instance_idx = idx;
}

static uint16_t ast_FunctionDecl_getTemplateInstanceIdx(const ast_FunctionDecl* d)
{
   return d->instance_idx;
}

static void ast_FunctionDecl_setInstanceName(ast_FunctionDecl* d, uint32_t name_idx)
{
   d->parent.name_idx = name_idx;
}

static ast_Ref* ast_FunctionDecl_getPrefix(const ast_FunctionDecl* d)
{
   if (ast_FunctionDecl_hasPrefix(d)) return ast_TypeRef_getPointerAfter(&d->rtype);

   return NULL;
}

static const char* ast_FunctionDecl_getPrefixName(const ast_FunctionDecl* d)
{
   if (!ast_FunctionDecl_hasPrefix(d)) return NULL;

   const ast_Ref* ref = ast_TypeRef_getPointerAfter(&d->rtype);
   return ast_Ref_getName(ref);
}

static void ast_FunctionDecl_setCallKind(ast_FunctionDecl* d, ast_CallKind kind)
{
   d->parent.functionDeclBits.call_kind = kind;
}

static ast_CallKind ast_FunctionDecl_getCallKind(const ast_FunctionDecl* d)
{
   return ((ast_CallKind)(d->parent.functionDeclBits.call_kind));
}

static bool ast_FunctionDecl_isVariadic(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.is_variadic;
}

static uint32_t ast_FunctionDecl_getNumParams(const ast_FunctionDecl* d)
{
   return d->num_params;
}

static ast_VarDecl** ast_FunctionDecl_getParams(const ast_FunctionDecl* d)
{
   uint8_t* tail = ast_TypeRef_getPointerAfter(&d->rtype);
   if (ast_FunctionDecl_hasPrefix(d)) tail += 16;
   ast_VarDecl** params = ((ast_VarDecl**)(tail));
   return params;
}

static uint32_t ast_FunctionDecl_getNumAutoArgs(const ast_FunctionDecl* d)
{
   return d->num_auto_args;
}

static void ast_FunctionDecl_setNumAutoArgs(ast_FunctionDecl* d, uint32_t num)
{
   d->num_auto_args = num;
}

static void ast_FunctionDecl_setAttrUnusedParams(ast_FunctionDecl* d)
{
   d->parent.functionDeclBits.attr_unused_params = 1;
}

static bool ast_FunctionDecl_hasAttrUnusedParams(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.attr_unused_params;
}

static void ast_FunctionDecl_setAttrNoReturn(ast_FunctionDecl* d)
{
   d->parent.functionDeclBits.attr_noreturn = 1;
}

static bool ast_FunctionDecl_hasAttrNoReturn(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.attr_noreturn;
}

static void ast_FunctionDecl_setAttrInline(ast_FunctionDecl* d)
{
   d->parent.functionDeclBits.attr_inline = 1;
}

static bool ast_FunctionDecl_hasAttrInline(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.attr_inline;
}

static void ast_FunctionDecl_setAttrWeak(ast_FunctionDecl* d)
{
   d->parent.functionDeclBits.attr_weak = 1;
}

static bool ast_FunctionDecl_hasAttrWeak(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.attr_weak;
}

static void ast_FunctionDecl_setAttrConstructor(ast_FunctionDecl* d)
{
   d->parent.functionDeclBits.attr_constructor = 1;
}

static bool ast_FunctionDecl_hasAttrConstructor(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.attr_constructor;
}

static void ast_FunctionDecl_setAttrDestructor(ast_FunctionDecl* d)
{
   d->parent.functionDeclBits.attr_destructor = 1;
}

static bool ast_FunctionDecl_hasAttrDestructor(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.attr_destructor;
}

static void ast_FunctionDecl_setAttrPure(ast_FunctionDecl* d)
{
   d->parent.functionDeclBits.attr_pure = 1;
}

static bool ast_FunctionDecl_hasAttrPure(const ast_FunctionDecl* d)
{
   return d->parent.functionDeclBits.attr_pure;
}

static void ast_FunctionDecl_setAttrPrintf(ast_FunctionDecl* d, uint8_t arg)
{
   d->attr_printf_arg = arg;
}

static bool ast_FunctionDecl_hasAttrPrintf(const ast_FunctionDecl* d)
{
   return (d->attr_printf_arg != 0);
}

static uint8_t ast_FunctionDecl_getAttrPrintf(const ast_FunctionDecl* d)
{
   return d->attr_printf_arg;
}

static const char* ast_FunctionDecl_getDiagKind(const ast_FunctionDecl* d)
{
   ast_CallKind ck = ast_FunctionDecl_getCallKind(d);
   if ((ck == ast_CallKind_StructFunc)) return "struct-";

   return "";
}

static void ast_FunctionDecl_print(const ast_FunctionDecl* d, string_buffer_Buf* out, uint32_t indent)
{
   bool valid_type = ast_QualType_isValid(&d->parent.qt);
   ast_Decl_printKind(&d->parent, out, indent, valid_type);
   if (!valid_type) {
      string_buffer_Buf_add(out, " ");
      ast_TypeRef_print(&d->rtype, out, true);
   }
   ast_Decl_printBits(&d->parent, out);
   string_buffer_Buf_space(out);
   string_buffer_Buf_color(out, ast_col_Attr);
   string_buffer_Buf_add(out, ast_callKind_names[ast_FunctionDecl_getCallKind(d)]);
   ast_Decl_printAttrs(&d->parent, out);
   if (d->parent.functionDeclBits.is_type) string_buffer_Buf_add(out, " Type");
   string_buffer_Buf_color(out, ast_col_Expr);
   if (ast_FunctionDecl_hasAttrUnusedParams(d)) string_buffer_Buf_add(out, " unused-params");
   if (ast_FunctionDecl_hasAttrNoReturn(d)) string_buffer_Buf_add(out, " noreturn");
   if (ast_FunctionDecl_hasAttrInline(d)) string_buffer_Buf_add(out, " inline");
   if (ast_FunctionDecl_hasAttrWeak(d)) string_buffer_Buf_add(out, " weak");
   if ((d->attr_printf_arg != 0)) string_buffer_Buf_print(out, " printf_format=%u", d->attr_printf_arg);
   if (ast_FunctionDecl_hasAttrConstructor(d)) string_buffer_Buf_add(out, " constructor");
   if (ast_FunctionDecl_hasAttrDestructor(d)) string_buffer_Buf_add(out, " destructor");
   if (ast_FunctionDecl_hasAttrPure(d)) string_buffer_Buf_add(out, " pure");
   string_buffer_Buf_space(out);
   string_buffer_Buf_color(out, ast_col_Value);
   const uint8_t* tail = ast_TypeRef_getPointerAfter(&d->rtype);
   if (ast_FunctionDecl_hasPrefix(d)) {
      const ast_Ref* prefix = ((ast_Ref*)(tail));
      string_buffer_Buf_add(out, ast_Ref_getName(prefix));
      string_buffer_Buf_add1(out, '.');
      tail += 16;
   }
   string_buffer_Buf_add(out, ast_Decl_getName(&d->parent));
   string_buffer_Buf_newline(out);
   if (d->parent.functionDeclBits.is_template) {
      string_buffer_Buf_indent(out, (indent + 1));
      string_buffer_Buf_color(out, ast_col_Template);
      string_buffer_Buf_print(out, "template %s\n", ast_idx2name(d->template_name));
   }
   ast_VarDecl** params = ((ast_VarDecl**)(tail));
   for (uint32_t i = 0; (i < d->num_params); i++) {
      ast_VarDecl_print(params[i], out, (indent + 1));
   }
   if (d->body) {
      ast_CompoundStmt_print(d->body, out, (indent + 1));
   }
}

static void ast_FunctionDecl_printType(const ast_FunctionDecl* d, string_buffer_Buf* out)
{
   if (ast_FunctionDecl_isType(d)) {
      string_buffer_Buf_add(out, ast_Decl_getName(&d->parent));
      return;
   }
   if (ast_QualType_isValid(&d->rt)) {
      ast_QualType_print(&d->rt, out);
   } else {
      ast_TypeRef_print(&d->rtype, out, true);
   }
   string_buffer_Buf_add(out, " (");
   const uint8_t* tail = ast_TypeRef_getPointerAfter(&d->rtype);
   if (ast_FunctionDecl_hasPrefix(d)) tail += 16;
   ast_VarDecl** params = ((ast_VarDecl**)(tail));
   for (uint32_t i = 0; (i < d->num_params); i++) {
      if ((i != 0)) string_buffer_Buf_add(out, ", ");
      ast_VarDecl_printType(params[i], out);
   }
   if (d->parent.functionDeclBits.is_variadic) string_buffer_Buf_add(out, ", ...");
   string_buffer_Buf_rparen(out);
}

static ast_FunctionTypeDecl* ast_FunctionTypeDecl_create(ast_context_Context* c, ast_FunctionDecl* func)
{
   ast_FunctionTypeDecl* ftd = ast_context_Context_alloc(c, 32);
   ast_Decl* d = ast_FunctionDecl_asDecl(func);
   ast_Decl_init(&ftd->parent, ast_DeclKind_FunctionType, ast_Decl_getNameIdx(d), ast_Decl_getLoc(d), ast_Decl_isPublic(d), ast_Decl_getType(d), ast_Decl_getASTIdx(d));
   ftd->func = func;
   ast_Stats_addDecl(ast_DeclKind_FunctionType, 32);
   return ftd;
}

static ast_Decl* ast_FunctionTypeDecl_asDecl(ast_FunctionTypeDecl* t)
{
   return &t->parent;
}

static ast_FunctionDecl* ast_FunctionTypeDecl_getDecl(const ast_FunctionTypeDecl* d)
{
   return d->func;
}

static void ast_FunctionTypeDecl_print(const ast_FunctionTypeDecl* d, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_color(out, ast_col_Decl);
   string_buffer_Buf_add(out, "FunctionTypeDecl");
   ast_Decl_printAttrs(&d->parent, out);
   string_buffer_Buf_newline(out);
   ast_FunctionDecl_print(d->func, out, (indent + 1));
}

static ast_ImportDecl* ast_ImportDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, uint32_t alias_name, src_loc_SrcLoc alias_loc, uint32_t ast_idx, bool is_local)
{
   ast_ImportDecl* d = ast_context_Context_alloc(c, 40);
   ast_Decl_init(&d->parent, ast_DeclKind_Import, name, loc, false, ast_QualType_Invalid, ast_idx);
   d->parent.importDeclBits.is_local = is_local;
   d->alias_idx = alias_name;
   d->alias_loc = alias_loc;
   d->dest = NULL;
   ast_Stats_addDecl(ast_DeclKind_Import, 40);
   return d;
}

static ast_Decl* ast_ImportDecl_asDecl(ast_ImportDecl* d)
{
   return &d->parent;
}

static const char* ast_ImportDecl_getAliasName(const ast_ImportDecl* d)
{
   return ast_idx2name(d->alias_idx);
}

static uint32_t ast_ImportDecl_getAliasNameIdx(const ast_ImportDecl* d)
{
   return d->alias_idx;
}

static uint32_t ast_ImportDecl_getImportNameIdx(const ast_ImportDecl* d)
{
   if (d->alias_idx) return d->alias_idx;

   return d->parent.name_idx;
}

static src_loc_SrcLoc ast_ImportDecl_getLoc(const ast_ImportDecl* d)
{
   if (d->alias_idx) return d->alias_loc;

   return ast_Decl_getLoc(&d->parent);
}

static void ast_ImportDecl_setDest(ast_ImportDecl* d, ast_Module* mod)
{
   d->dest = mod;
}

static ast_Module* ast_ImportDecl_getDest(const ast_ImportDecl* d)
{
   return d->dest;
}

static bool ast_ImportDecl_isLocal(const ast_ImportDecl* d)
{
   return d->parent.importDeclBits.is_local;
}

static void ast_ImportDecl_print(const ast_ImportDecl* d, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_color(out, ast_col_Decl);
   string_buffer_Buf_add(out, "ImportDecl");
   ast_Decl_printUsed(&d->parent, out);
   string_buffer_Buf_add(out, " module=");
   if (d->dest) {
      string_buffer_Buf_add(out, ast_Module_getName(d->dest));
   } else {
      string_buffer_Buf_add(out, "<nil>");
   }
   if (d->parent.importDeclBits.is_local) {
      string_buffer_Buf_color(out, ast_col_Attr);
      string_buffer_Buf_add(out, " local");
   }
   ast_Decl_printName(&d->parent, out);
   if (d->alias_idx) {
      string_buffer_Buf_color(out, ast_col_Attr);
      string_buffer_Buf_add(out, " as ");
      string_buffer_Buf_color(out, ast_col_Value);
      string_buffer_Buf_print(out, "%s", ast_idx2name(d->alias_idx));
   }
   string_buffer_Buf_newline(out);
}

static ast_StaticAssert* ast_StaticAssert_create(ast_context_Context* c, uint32_t ast_idx, src_loc_SrcLoc loc, ast_Expr* lhs, ast_Expr* rhs)
{
   ast_StaticAssert* d = ast_context_Context_alloc(c, 24);
   d->ast_idx = ast_idx;
   d->lhs = lhs;
   d->rhs = rhs;
   ast_Stats_addStaticAssert(24);
   return d;
}

static ast_AST* ast_StaticAssert_getAST(const ast_StaticAssert* d)
{
   return ast_idx2ast(d->ast_idx);
}

static ast_Expr* ast_StaticAssert_getLhs(const ast_StaticAssert* d)
{
   return d->lhs;
}

static ast_Expr* ast_StaticAssert_getRhs(const ast_StaticAssert* d)
{
   return d->rhs;
}

static void ast_StaticAssert_print(const ast_StaticAssert* d, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_color(out, ast_col_Decl);
   string_buffer_Buf_print(out, "StaticAssert\n");
   ast_Expr_print(d->lhs, out, (indent + 1));
   ast_Expr_print(d->rhs, out, (indent + 1));
}

static ast_StructTypeDecl* ast_StructTypeDecl_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, uint32_t ast_idx, bool is_struct, bool is_global, ast_Decl** members, uint32_t num_members)
{
   uint32_t size = (40 + (num_members * 8));
   size += 12;
   size += (num_members * 4);
   size = (((size + 7)) & ~0x7);
   ast_StructTypeDecl* d = ast_context_Context_alloc(c, size);
   ast_StructType* stype = ast_StructType_create(c, d);
   ast_QualType qt = ast_QualType_init(ast_StructType_asType(stype));
   ast_Type_setCanonicalType(ast_StructType_asType(stype), qt);
   ast_Decl_init(&d->parent, ast_DeclKind_StructType, name, loc, is_public, qt, ast_idx);
   d->parent.structTypeDeclBits.is_struct = is_struct;
   d->parent.structTypeDeclBits.is_global = is_global;
   if (!is_global) ast_Decl_setUsed(&d->parent);
   d->num_members = num_members;
   d->num_struct_functions = 0;
   d->struct_functions = NULL;
   ast_StructLayout* layout = ast_StructTypeDecl_getLayoutPtr(d);
   layout->size = 0;
   layout->alignment = 0;
   layout->attr_alignment = 1;
   uint32_t* member_offsets = ((uint32_t*)(&d->members[d->num_members]));
   if (num_members) {
      memcpy(((void*)(d->members)), ((void*)(members)), (num_members * 8));
      memset(layout->member_offsets, 0, (num_members * 4));
   }
   ast_Stats_addDecl(ast_DeclKind_StructType, size);
   return d;
}

static ast_Decl* ast_StructTypeDecl_asDecl(ast_StructTypeDecl* d)
{
   return &d->parent;
}

static uint32_t ast_StructTypeDecl_getNumMembers(const ast_StructTypeDecl* d)
{
   return d->num_members;
}

static ast_Decl** ast_StructTypeDecl_getMembers(ast_StructTypeDecl* d)
{
   return d->members;
}

static bool ast_StructTypeDecl_isStruct(const ast_StructTypeDecl* d)
{
   return d->parent.structTypeDeclBits.is_struct;
}

static bool ast_StructTypeDecl_isUnion(const ast_StructTypeDecl* d)
{
   return !d->parent.structTypeDeclBits.is_struct;
}

static const ast_FunctionDecl** ast_StructTypeDecl_getStructFunctions(const ast_StructTypeDecl* d)
{
   return ((const ast_FunctionDecl**)(d->struct_functions));
}

static uint32_t ast_StructTypeDecl_getNumStructFunctions(const ast_StructTypeDecl* d)
{
   return d->num_struct_functions;
}

static ast_StructLayout* ast_StructTypeDecl_getLayoutPtr(const ast_StructTypeDecl* d)
{
   return ((ast_StructLayout*)(&d->members[d->num_members]));
}

static void ast_StructTypeDecl_setMemberOffset(ast_StructTypeDecl* d, uint32_t member_idx, uint32_t offset)
{
   ast_StructLayout* layout = ast_StructTypeDecl_getLayoutPtr(d);
   layout->member_offsets[member_idx] = offset;
}

static uint32_t ast_StructTypeDecl_getMemberOffset(const ast_StructTypeDecl* d, uint32_t member_idx)
{
   ast_StructLayout* layout = ast_StructTypeDecl_getLayoutPtr(d);
   return layout->member_offsets[member_idx];
}

static uint32_t ast_StructTypeDecl_getSize(const ast_StructTypeDecl* d)
{
   ast_StructLayout* layout = ast_StructTypeDecl_getLayoutPtr(d);
   return layout->size;
}

static void ast_StructTypeDecl_setSizeAlignment(ast_StructTypeDecl* d, uint32_t size, uint32_t alignment)
{
   ast_StructLayout* layout = ast_StructTypeDecl_getLayoutPtr(d);
   d->parent.structTypeDeclBits.size_analysed = true;
   layout->size = size;
   layout->alignment = alignment;
}

static uint32_t ast_StructTypeDecl_getAlignment(const ast_StructTypeDecl* d)
{
   ast_StructLayout* layout = ast_StructTypeDecl_getLayoutPtr(d);
   return layout->alignment;
}

static uint32_t ast_StructTypeDecl_getAttrAlignment(const ast_StructTypeDecl* d)
{
   ast_StructLayout* layout = ast_StructTypeDecl_getLayoutPtr(d);
   return layout->attr_alignment;
}

static void ast_StructTypeDecl_setAttrAlignment(ast_StructTypeDecl* d, uint32_t alignment)
{
   ast_StructLayout* layout = ast_StructTypeDecl_getLayoutPtr(d);
   layout->attr_alignment = alignment;
}

static void ast_StructTypeDecl_setPacked(ast_StructTypeDecl* d)
{
   d->parent.structTypeDeclBits.attr_packed = 1;
}

static bool ast_StructTypeDecl_isPacked(const ast_StructTypeDecl* d)
{
   return d->parent.structTypeDeclBits.attr_packed;
}

static void ast_StructTypeDecl_setOpaque(ast_StructTypeDecl* d)
{
   d->parent.structTypeDeclBits.attr_opaque = 1;
}

static bool ast_StructTypeDecl_isOpaque(const ast_StructTypeDecl* d)
{
   return d->parent.structTypeDeclBits.attr_opaque;
}

static bool ast_StructTypeDecl_isGlobal(const ast_StructTypeDecl* d)
{
   return d->parent.structTypeDeclBits.is_global;
}

static void ast_StructTypeDecl_setAttrNoTypeDef(ast_StructTypeDecl* d)
{
   d->parent.structTypeDeclBits.attr_notypedef = 1;
}

static bool ast_StructTypeDecl_hasAttrNoTypeDef(const ast_StructTypeDecl* d)
{
   return d->parent.structTypeDeclBits.attr_notypedef;
}

static void ast_StructTypeDecl_setStructFunctions(ast_StructTypeDecl* d, ast_context_Context* c, ast_FunctionDecl** funcs, uint32_t count)
{
   const uint32_t size = (count * 8);
   void* dest = ast_context_Context_alloc(c, size);
   memcpy(dest, ((void*)(funcs)), size);
   d->struct_functions = dest;
   d->num_struct_functions = count;
}

static ast_Decl* ast_StructTypeDecl_findAny(const ast_StructTypeDecl* s, uint32_t name_idx)
{
   for (uint32_t i = 0; (i < ast_StructTypeDecl_getNumMembers(s)); i++) {
      ast_Decl* d = s->members[i];
      uint32_t member_name = ast_Decl_getNameIdx(d);
      if ((member_name == name_idx)) return d;

      if (((member_name == 0) && ast_Decl_isStructType(d))) {
         ast_StructTypeDecl* sub = ((ast_StructTypeDecl*)(d));
         d = ast_StructTypeDecl_findAny(sub, name_idx);
         if (d) return d;

      }
   }
   if (s->parent.structTypeDeclBits.is_global) {
      for (uint32_t i = 0; (i < s->num_struct_functions); i++) {
         ast_Decl* sf = ((ast_Decl*)(s->struct_functions[i]));
         if ((ast_Decl_getNameIdx(sf) == name_idx)) return sf;

      }
   }
   return NULL;
}

static ast_Decl* ast_StructTypeDecl_findMember(const ast_StructTypeDecl* s, uint32_t name_idx, uint32_t* offset)
{
   for (uint32_t i = 0; (i < ast_StructTypeDecl_getNumMembers(s)); i++) {
      ast_Decl* d = s->members[i];
      uint32_t member_name = ast_Decl_getNameIdx(d);
      if ((member_name == name_idx)) {
         if (offset) *offset += ast_StructTypeDecl_getMemberOffset(s, i);
         return d;
      }
      if (((member_name == 0) && ast_Decl_isStructType(d))) {
         ast_StructTypeDecl* sub = ((ast_StructTypeDecl*)(d));
         d = ast_StructTypeDecl_findMember(sub, name_idx, offset);
         if (d) {
            if (offset) *offset += ast_StructTypeDecl_getMemberOffset(s, i);
            return d;
         }
      }
   }
   return NULL;
}

static void ast_StructTypeDecl_print(const ast_StructTypeDecl* d, string_buffer_Buf* out, uint32_t indent)
{
   ast_Decl_printKind(&d->parent, out, indent, true);
   ast_Decl_printBits(&d->parent, out);
   bool is_global = d->parent.structTypeDeclBits.is_global;
   if (is_global) string_buffer_Buf_add(out, " global");
   if (d->parent.structTypeDeclBits.is_struct) string_buffer_Buf_add(out, " struct");
   else string_buffer_Buf_add(out, " union");
   if (ast_StructTypeDecl_isPacked(d)) string_buffer_Buf_add(out, " packed");
   if (ast_StructTypeDecl_isOpaque(d)) string_buffer_Buf_add(out, " opaque");
   if (ast_StructTypeDecl_hasAttrNoTypeDef(d)) string_buffer_Buf_add(out, " notypedef");
   if (is_global) ast_Decl_printAttrs(&d->parent, out);
   if (d->parent.structTypeDeclBits.size_analysed) {
      string_buffer_Buf_color(out, ast_col_Calc);
      ast_StructLayout* layout = ast_StructTypeDecl_getLayoutPtr(d);
      string_buffer_Buf_print(out, " size=%u align=%u", layout->size, layout->alignment);
   }
   string_buffer_Buf_space(out);
   string_buffer_Buf_color(out, ast_col_Value);
   if (ast_Decl_getName(&d->parent)) string_buffer_Buf_add(out, ast_Decl_getName(&d->parent));
   else string_buffer_Buf_add(out, "<anonymous>");
   string_buffer_Buf_newline(out);
   for (uint32_t i = 0; (i < d->num_members); i++) {
      string_buffer_Buf_indent(out, (indent + 1));
      string_buffer_Buf_color(out, ast_col_Calc);
      string_buffer_Buf_print(out, "offset=%u\n", ast_StructTypeDecl_getMemberOffset(d, i));
      ast_Decl_print(d->members[i], out, (indent + 1));
   }
}

static bool ast_Value_isNegative(const ast_Value* v)
{
   if ((v->is_signed && (v->svalue < 0))) return true;

   return false;
}

static bool ast_Value_equals(const ast_Value* v1, const ast_Value* v2)
{
   if ((v1->is_signed == v2->is_signed)) {
      return (v1->uvalue == v2->uvalue);
   }
   if (v1->is_signed) {
      if ((v1->svalue >= 0)) return (v1->uvalue == v2->uvalue);

   } else {
      if ((v2->svalue >= 0)) return (v1->uvalue == v2->uvalue);

   }
   return false;
}

static bool ast_Value_less_than(const ast_Value* v1, const ast_Value* v2)
{
   if (v1->is_signed) {
      if (v2->is_signed) {
         return (v1->svalue < v2->svalue);
      }
      if ((v1->svalue < 0)) return true;

      uint64_t lval = ((uint64_t)(v1->svalue));
      return (lval < v2->uvalue);
   } else {
      if ((v2->is_signed && (v2->svalue <= 0))) return false;

      return (v1->uvalue < ((uint64_t)(v2->svalue)));
   }
   return true;
}

static ast_Value ast_Value_minus(const ast_Value* v1, const ast_Value* v2)
{
   ast_Value result;
   result.is_signed = false;
   if (v1->is_signed) {
      result.is_signed = true;
      if (v2->is_signed) {
         result.svalue = (v1->svalue - v2->svalue);
      } else {
         result.svalue = (v1->svalue - ((int64_t)(v2->uvalue)));
      }
   } else {
      if (v2->is_signed) {
         result.is_signed = true;
         result.svalue = ((int64_t)((v1->uvalue - v2->svalue)));
      } else {
         result.uvalue = (v1->uvalue - v2->uvalue);
      }
   }
   return result;
}

static void ast_Value_mask(ast_Value* v, uint32_t width)
{
   uint64_t mask = 0;
   for (uint32_t i = 0; (i < width); i++) mask |= ((1 << i));
   v->uvalue &= mask;
}

static bool ast_Value_ugt(const ast_Value* v1, uint64_t max)
{
   if (v1->is_signed) {
      if ((v1->svalue < 0)) return false;

      uint64_t lval = ((uint64_t)(v1->svalue));
      return (lval > max);
   }
   return (v1->uvalue > max);
}

static void ast_Value_incr(ast_Value* v)
{
   if (v->is_signed) v->svalue++;
   else v->uvalue++;
}

static const char* ast_Value_str(const ast_Value* v)
{
   static char text[4][32];
   static uint8_t index = 0;
   char* out = text[index];
   index = (((index + 1)) % 4);
   if (v->is_signed) {
      sprintf(out, "%ld", v->svalue);
   } else {
      sprintf(out, "%lu", v->uvalue);
   }
   return out;
}

static ast_VarDecl* ast_VarDecl_create(ast_context_Context* c, ast_VarDeclKind kind, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* ref, uint32_t ast_idx, src_loc_SrcLoc assignLoc, ast_Expr* initValue)
{
   uint32_t size = (32 + ast_TypeRefHolder_getExtraSize(ref));
   if ((initValue || ast_TypeRefHolder_isIncrArray(ref))) size += (8 + 8);
   c2_assert(((kind != ast_VarDeclKind_StructMember)) != 0, "ast/var_decl.c2:71: ast.VarDecl.create", "kind!=VarDeclKind.StructMember");
   ast_VarDecl* d = ast_context_Context_alloc(c, size);
   ast_Decl_init(&d->parent, ast_DeclKind_Variable, name, loc, is_public, ast_QualType_Invalid, ast_idx);
   d->parent.varDeclBits.kind = kind;
   ast_TypeRefHolder_fill(ref, &d->typeRef);
   if (initValue) {
      d->parent.varDeclBits.has_init_or_bitfield = 1;
      src_loc_SrcLoc* locpos = ast_TypeRef_getPointerAfter(&d->typeRef);
      *locpos = assignLoc;
      ast_Expr** i = ast_VarDecl_getInit2(d);
      *i = initValue;
   }
   ast_Stats_addDecl(ast_DeclKind_Variable, size);
   return d;
}

static ast_VarDecl* ast_VarDecl_createStructMember(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* ref, uint32_t ast_idx, ast_Expr* bitfield)
{
   uint32_t size = (32 + ast_TypeRefHolder_getExtraSize(ref));
   if (bitfield) size += (8 + 8);
   size = (((size + 7)) & ~0x7);
   ast_VarDecl* d = ast_context_Context_alloc(c, size);
   ast_Decl_init(&d->parent, ast_DeclKind_Variable, name, loc, is_public, ast_QualType_Invalid, ast_idx);
   d->parent.varDeclBits.kind = ast_VarDeclKind_StructMember;
   if ((name == 0)) ast_Decl_setUsed(&d->parent);
   ast_TypeRefHolder_fill(ref, &d->typeRef);
   if (bitfield) {
      d->parent.varDeclBits.has_init_or_bitfield = 1;
      ast_Expr** i = ast_VarDecl_getInit2(d);
      *i = bitfield;
   }
   ast_Stats_addDecl(ast_DeclKind_Variable, size);
   return d;
}

static ast_VarDecl* ast_VarDecl_instantiate(const ast_VarDecl* vd, ast_Instantiator* inst)
{
   bool matches = ast_TypeRef_matchesTemplate(&vd->typeRef, inst->template_name);
   uint32_t extra = matches ? ast_TypeRef_getExtraSize(inst->ref) : ast_TypeRef_getExtraSize(&vd->typeRef);
   uint32_t size = (32 + extra);
   ast_VarDecl* vd2 = ast_context_Context_alloc(inst->c, size);
   vd2->parent = vd->parent;
   ast_TypeRef_instantiate(&vd2->typeRef, &vd->typeRef, inst);
   ast_Expr* ie = ast_VarDecl_getInit(vd);
   if (ie) {
      ast_Expr** ie2 = ast_VarDecl_getInit2(vd2);
      *ie2 = ast_Expr_instantiate(ie, inst);
   }
   ast_Stats_addDecl(ast_DeclKind_Variable, size);
   return vd2;
}

static ast_Decl* ast_VarDecl_asDecl(ast_VarDecl* d)
{
   return &d->parent;
}

static ast_VarDeclKind ast_VarDecl_getKind(const ast_VarDecl* d)
{
   return ((ast_VarDeclKind)(d->parent.varDeclBits.kind));
}

static bool ast_VarDecl_isGlobal(const ast_VarDecl* d)
{
   return (ast_VarDecl_getKind(d) == ast_VarDeclKind_GlobalVar);
}

static bool ast_VarDecl_isLocal(const ast_VarDecl* d)
{
   return (ast_VarDecl_getKind(d) == ast_VarDeclKind_LocalVar);
}

static bool ast_VarDecl_isParameter(const ast_VarDecl* d)
{
   return (ast_VarDecl_getKind(d) == ast_VarDeclKind_FunctionParam);
}

static bool ast_VarDecl_isStructMember(const ast_VarDecl* d)
{
   return (ast_VarDecl_getKind(d) == ast_VarDeclKind_StructMember);
}

static bool ast_VarDecl_isAddrUsed(const ast_VarDecl* d)
{
   return d->parent.varDeclBits.addr_used;
}

static void ast_VarDecl_setAddrUsed(ast_VarDecl* d)
{
   d->parent.varDeclBits.addr_used = 1;
}

static ast_TypeRef* ast_VarDecl_getTypeRef(ast_VarDecl* d)
{
   return &d->typeRef;
}

static src_loc_SrcLoc ast_VarDecl_getAssignLoc(const ast_VarDecl* d)
{
   if (d->parent.varDeclBits.has_init_or_bitfield) {
      src_loc_SrcLoc* locpos = ast_TypeRef_getPointerAfter(&d->typeRef);
      return *locpos;
   }
   return 0;
}

static bool ast_VarDecl_hasInit(const ast_VarDecl* d)
{
   return d->parent.varDeclBits.has_init_or_bitfield;
}

static ast_Expr* ast_VarDecl_getInit(const ast_VarDecl* d)
{
   if (d->parent.varDeclBits.has_init_or_bitfield) {
      uint8_t* tail = ast_TypeRef_getPointerAfter(&d->typeRef);
      tail += 8;
      ast_Expr** e = ((ast_Expr**)(tail));
      return *e;
   }
   return NULL;
}

static ast_Expr** ast_VarDecl_getInit2(ast_VarDecl* d)
{
   if (d->parent.varDeclBits.has_init_or_bitfield) {
      uint8_t* tail = ast_TypeRef_getPointerAfter(&d->typeRef);
      tail += 8;
      return ((ast_Expr**)(tail));
   }
   return NULL;
}

static void ast_VarDecl_setInit(ast_VarDecl* d, ast_Expr* initValue)
{
   d->parent.varDeclBits.has_init_or_bitfield = 1;
   ast_Expr** i = ast_VarDecl_getInit2(d);
   *i = initValue;
}

static ast_Expr* ast_VarDecl_getBitfield(const ast_VarDecl* d)
{
   if (ast_VarDecl_isStructMember(d)) return ast_VarDecl_getInit(d);

   return NULL;
}

static bool ast_VarDecl_hasLocalQualifier(const ast_VarDecl* d)
{
   return d->parent.varDeclBits.has_local;
}

static void ast_VarDecl_setLocal(ast_VarDecl* d, bool has_local)
{
   d->parent.varDeclBits.has_local = has_local;
}

static void ast_VarDecl_setAttrWeak(ast_VarDecl* d)
{
   d->parent.varDeclBits.attr_weak = 1;
}

static bool ast_VarDecl_hasAttrWeak(const ast_VarDecl* d)
{
   return d->parent.varDeclBits.attr_weak;
}

static void ast_VarDecl_setAttrAutoFile(ast_VarDecl* d)
{
   d->parent.varDeclBits.auto_file = 1;
}

static bool ast_VarDecl_hasAttrAutoFile(const ast_VarDecl* d)
{
   return d->parent.varDeclBits.auto_file;
}

static void ast_VarDecl_setAttrAutoLine(ast_VarDecl* d)
{
   d->parent.varDeclBits.auto_line = 1;
}

static bool ast_VarDecl_hasAttrAutoLine(const ast_VarDecl* d)
{
   return d->parent.varDeclBits.auto_line;
}

static bool ast_VarDecl_hasAutoAttr(const ast_VarDecl* d)
{
   return (d->parent.varDeclBits.auto_file || d->parent.varDeclBits.auto_line);
}

static void ast_VarDecl_setPrintfFormat(ast_VarDecl* d)
{
   d->parent.varDeclBits.printf_format = 1;
}

static bool ast_VarDecl_hasPrintfFormat(const ast_VarDecl* d)
{
   return d->parent.varDeclBits.printf_format;
}

static void ast_VarDecl_print(const ast_VarDecl* d, string_buffer_Buf* out, uint32_t indent)
{
   bool valid_type = ast_QualType_isValid(&d->parent.qt);
   ast_Decl_printKind(&d->parent, out, indent, valid_type);
   if (!valid_type) {
      string_buffer_Buf_space(out);
      ast_TypeRef_print(&d->typeRef, out, true);
   }
   string_buffer_Buf_color(out, ast_col_Attr);
   ast_VarDeclKind k = ast_VarDecl_getKind(d);
   string_buffer_Buf_add(out, ast_varDeclNames[k]);
   bool has_init_or_bitfield = d->parent.varDeclBits.has_init_or_bitfield;
   if (((k == ast_VarDeclKind_StructMember) && has_init_or_bitfield)) string_buffer_Buf_add(out, " bitfield");
   if (d->parent.varDeclBits.attr_weak) string_buffer_Buf_add(out, " weak");
   if (d->parent.varDeclBits.addr_used) string_buffer_Buf_add(out, " addr_used");
   if (d->parent.varDeclBits.auto_file) string_buffer_Buf_add(out, " auto_file");
   if (d->parent.varDeclBits.auto_line) string_buffer_Buf_add(out, " auto_line");
   if (d->parent.varDeclBits.printf_format) string_buffer_Buf_add(out, " printf_format");
   ast_Decl_printBits(&d->parent, out);
   ast_Decl_printAttrs(&d->parent, out);
   string_buffer_Buf_color(out, ast_col_Value);
   ast_Decl_printName(&d->parent, out);
   string_buffer_Buf_newline(out);
   if (has_init_or_bitfield) {
      ast_Expr* i = ast_VarDecl_getInit(d);
      ast_Expr_print(i, out, (indent + 1));
   }
}

static void ast_VarDecl_printType(const ast_VarDecl* d, string_buffer_Buf* out)
{
   if (ast_QualType_isValid(&d->parent.qt)) {
      ast_QualType_printQuoted(&d->parent.qt, out);
   } else {
      ast_TypeRef_print(&d->typeRef, out, true);
   }
}

static void ast_Stmt_init(ast_Stmt* s, ast_StmtKind k)
{
   s->bits = 0;
   s->stmtBits.kind = k;
}

static ast_Stmt* ast_Stmt_instantiate(ast_Stmt* s, ast_Instantiator* inst)
{
   switch (ast_Stmt_getKind(s)) {
   case ast_StmtKind_Return:
      return ast_ReturnStmt_instantiate(((ast_ReturnStmt*)(s)), inst);
   case ast_StmtKind_Expr:
      return ((ast_Stmt*)(ast_Expr_instantiate(((ast_Expr*)(s)), inst)));
   case ast_StmtKind_If:
      return ast_IfStmt_instantiate(((ast_IfStmt*)(s)), inst);
   case ast_StmtKind_While:
      return ast_WhileStmt_instantiate(((ast_WhileStmt*)(s)), inst);
   case ast_StmtKind_Do:
      return ast_DoStmt_instantiate(((ast_DoStmt*)(s)), inst);
   case ast_StmtKind_For:
      return ast_ForStmt_instantiate(((ast_ForStmt*)(s)), inst);
   case ast_StmtKind_Switch:
      return ast_SwitchStmt_instantiate(((ast_SwitchStmt*)(s)), inst);
   case ast_StmtKind_Break:
      return s;
   case ast_StmtKind_Continue:
      return s;
   case ast_StmtKind_Fallthrough:
      return s;
   case ast_StmtKind_Label:
      return s;
   case ast_StmtKind_Goto:
      return s;
   case ast_StmtKind_Compound:
      return ((ast_Stmt*)(ast_CompoundStmt_instantiate(((ast_CompoundStmt*)(s)), inst)));
   case ast_StmtKind_Decl:
      return ast_DeclStmt_instantiate(((ast_DeclStmt*)(s)), inst);
   case ast_StmtKind_Asm:
      break;
   case ast_StmtKind_Assert:
      return ast_AssertStmt_instantiate(((ast_AssertStmt*)(s)), inst);
   }
   ast_Stmt_dump(s);
   c2_assert((0) != 0, "ast/stmt.c2:136: ast.Stmt.instantiate", "0");
   return NULL;
}

static ast_StmtKind ast_Stmt_getKind(const ast_Stmt* s)
{
   return ((ast_StmtKind)(s->stmtBits.kind));
}

static bool ast_Stmt_isReturn(const ast_Stmt* s)
{
   return (ast_Stmt_getKind(s) == ast_StmtKind_Return);
}

static bool ast_Stmt_isExpr(const ast_Stmt* s)
{
   return (ast_Stmt_getKind(s) == ast_StmtKind_Expr);
}

static bool ast_Stmt_isCompound(const ast_Stmt* s)
{
   return (ast_Stmt_getKind(s) == ast_StmtKind_Compound);
}

static bool ast_Stmt_isFallthrough(const ast_Stmt* s)
{
   return (ast_Stmt_getKind(s) == ast_StmtKind_Fallthrough);
}

static bool ast_Stmt_isDecl(const ast_Stmt* s)
{
   return (ast_Stmt_getKind(s) == ast_StmtKind_Decl);
}

static src_loc_SrcLoc ast_Stmt_getLoc(const ast_Stmt* s)
{
   switch (ast_Stmt_getKind(s)) {
   case ast_StmtKind_Return: {
      const ast_ReturnStmt* r = ((ast_ReturnStmt*)(s));
      return ast_ReturnStmt_getLoc(r);
   }
   case ast_StmtKind_Expr: {
      const ast_Expr* e = ((ast_Expr*)(s));
      return ast_Expr_getLoc(e);
   }
   case ast_StmtKind_If: {
      const ast_IfStmt* i = ((ast_IfStmt*)(s));
      break;
   }
   case ast_StmtKind_While: {
      const ast_WhileStmt* w = ((ast_WhileStmt*)(s));
      break;
   }
   case ast_StmtKind_Do: {
      const ast_DoStmt* d = ((ast_DoStmt*)(s));
      break;
   }
   case ast_StmtKind_For: {
      const ast_ForStmt* f = ((ast_ForStmt*)(s));
      return ast_ForStmt_getLoc(f);
   }
   case ast_StmtKind_Switch: {
      const ast_SwitchStmt* sw = ((ast_SwitchStmt*)(s));
      return ast_SwitchStmt_getLoc(sw);
   }
   case ast_StmtKind_Break: {
      const ast_BreakStmt* b = ((ast_BreakStmt*)(s));
      return ast_BreakStmt_getLoc(b);
   }
   case ast_StmtKind_Continue: {
      const ast_ContinueStmt* c = ((ast_ContinueStmt*)(s));
      return ast_ContinueStmt_getLoc(c);
   }
   case ast_StmtKind_Fallthrough: {
      const ast_FallthroughStmt* f = ((ast_FallthroughStmt*)(s));
      return ast_FallthroughStmt_getLoc(f);
   }
   case ast_StmtKind_Label: {
      const ast_LabelStmt* l = ((ast_LabelStmt*)(s));
      return ast_LabelStmt_getLoc(l);
   }
   case ast_StmtKind_Goto: {
      const ast_GotoStmt* g = ((ast_GotoStmt*)(s));
      return ast_GotoStmt_getLoc(g);
   }
   case ast_StmtKind_Compound: {
      const ast_CompoundStmt* c = ((ast_CompoundStmt*)(s));
      return ast_CompoundStmt_getEndLoc(c);
   }
   case ast_StmtKind_Decl: {
      const ast_DeclStmt* d = ((ast_DeclStmt*)(s));
      ast_Decl* vd = ((ast_Decl*)(ast_DeclStmt_getDecl(d)));
      return ast_Decl_getLoc(vd);
   }
   case ast_StmtKind_Asm: {
      const ast_AsmStmt* a = ((ast_AsmStmt*)(s));
      return ast_AsmStmt_getLoc(a);
   }
   case ast_StmtKind_Assert: {
      const ast_AssertStmt* a = ((ast_AssertStmt*)(s));
      return ast_AssertStmt_getLoc(a);
   }
   }
   return 0;
}

static void ast_Stmt_dump(const ast_Stmt* s)
{
   string_buffer_Buf* out = string_buffer_create((10 * 4096), ast_useColor(), 2);
   ast_Stmt_print(s, out, 0);
   string_buffer_Buf_color(out, ast_col_Normal);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static void ast_Stmt_print(const ast_Stmt* s, string_buffer_Buf* out, uint32_t indent)
{
   switch (ast_Stmt_getKind(s)) {
   case ast_StmtKind_Return:
      ast_ReturnStmt_print(((ast_ReturnStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Expr:
      ast_Expr_print(((ast_Expr*)(s)), out, indent);
      break;
   case ast_StmtKind_If:
      ast_IfStmt_print(((ast_IfStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_While:
      ast_WhileStmt_print(((ast_WhileStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Do:
      ast_DoStmt_print(((ast_DoStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_For:
      ast_ForStmt_print(((ast_ForStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Switch:
      ast_SwitchStmt_print(((ast_SwitchStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Break:
      ast_BreakStmt_print(((ast_BreakStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Continue:
      ast_ContinueStmt_print(((ast_ContinueStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Fallthrough:
      ast_FallthroughStmt_print(((ast_FallthroughStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Label:
      ast_LabelStmt_print(((ast_LabelStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Goto:
      ast_GotoStmt_print(((ast_GotoStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Compound:
      ast_CompoundStmt_print(((ast_CompoundStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Decl:
      ast_DeclStmt_print(((ast_DeclStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Asm:
      ast_AsmStmt_print(((ast_AsmStmt*)(s)), out, indent);
      break;
   case ast_StmtKind_Assert:
      ast_AssertStmt_print(((ast_AssertStmt*)(s)), out, indent);
      break;
   }
}

static void ast_Stmt_printKind(const ast_Stmt* s, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_color(out, ast_col_Stmt);
   string_buffer_Buf_add(out, ast_stmtKind_names[ast_Stmt_getKind(s)]);
}

static ast_AsmStmt* ast_AsmStmt_create(ast_context_Context* c, src_loc_SrcLoc loc, bool is_basic, bool is_volatile, uint32_t num_outputs, uint32_t num_inputs, const uint32_t* names, ast_ExprList* constraints, ast_ExprList* exprs, ast_ExprList* clobbers, ast_Expr* str)
{
   uint32_t size = 24;
   size += ((ast_ExprList_size(constraints) * 8));
   size += ((ast_ExprList_size(exprs) * 8));
   size += ((ast_ExprList_size(clobbers) * 8));
   size += (((num_inputs + num_outputs)) * 4);
   ast_AsmStmt* s = ast_context_Context_alloc(c, size);
   ast_Stmt_init(&s->parent, ast_StmtKind_Asm);
   s->parent.asmStmtBits.is_basic = is_basic;
   s->parent.asmStmtBits.is_volatile = is_volatile;
   s->num_outputs = ((uint8_t)(num_outputs));
   s->num_inputs = ((uint8_t)(num_inputs));
   s->num_constraints = ((uint8_t)(ast_ExprList_size(constraints)));
   s->num_exprs = ((uint8_t)(ast_ExprList_size(exprs)));
   s->num_clobbers = ((uint8_t)(ast_ExprList_size(clobbers)));
   s->loc = loc;
   s->asm_string = ((ast_StringLiteral*)(str));
   uint8_t* tail = ((uint8_t*)(s->constraints));
   if (ast_ExprList_size(constraints)) {
      uint32_t sz = (ast_ExprList_size(constraints) * 8);
      memcpy(tail, ast_ExprList_getExprs(constraints), sz);
      tail += sz;
   }
   if (ast_ExprList_size(exprs)) {
      uint32_t sz = (ast_ExprList_size(exprs) * 8);
      memcpy(tail, ast_ExprList_getExprs(exprs), sz);
      tail += sz;
   }
   if (ast_ExprList_size(clobbers)) {
      uint32_t sz = (ast_ExprList_size(clobbers) * 8);
      memcpy(tail, ast_ExprList_getExprs(clobbers), sz);
      tail += sz;
   }
   uint32_t num_names = (num_outputs + num_inputs);
   if (num_names) {
      memcpy(tail, names, (num_names * 4));
   }
   ast_Stats_addStmt(ast_StmtKind_Asm, size);
   return s;
}

static ast_Stmt* ast_AsmStmt_instantiate(ast_AsmStmt* s, ast_Instantiator* inst)
{
   return ((ast_Stmt*)(s));
}

static src_loc_SrcLoc ast_AsmStmt_getLoc(const ast_AsmStmt* s)
{
   return s->loc;
}

static bool ast_AsmStmt_isVolatile(const ast_AsmStmt* s)
{
   return s->parent.asmStmtBits.is_volatile;
}

static uint32_t ast_AsmStmt_getNumConstraints(const ast_AsmStmt* s)
{
   return s->num_constraints;
}

static uint32_t ast_AsmStmt_getNumClobbers(const ast_AsmStmt* s)
{
   return s->num_clobbers;
}

static uint32_t ast_AsmStmt_getNumExprs(const ast_AsmStmt* s)
{
   return s->num_exprs;
}

static uint32_t ast_AsmStmt_getNumOutputs(const ast_AsmStmt* s)
{
   return s->num_outputs;
}

static uint32_t ast_AsmStmt_getNumInputs(const ast_AsmStmt* s)
{
   return s->num_inputs;
}

static ast_StringLiteral* ast_AsmStmt_getString(const ast_AsmStmt* s)
{
   return s->asm_string;
}

static const ast_Expr** ast_AsmStmt_getConstraints(const ast_AsmStmt* s)
{
   return ((const ast_Expr**)(s->constraints));
}

static ast_Expr** ast_AsmStmt_getExprs(const ast_AsmStmt* s)
{
   uint8_t* tail = ((uint8_t*)(s->constraints));
   tail += (s->num_constraints * 8);
   return ((ast_Expr**)(tail));
}

static ast_Expr** ast_AsmStmt_getClobbers(const ast_AsmStmt* s)
{
   uint8_t* tail = ((uint8_t*)(s->constraints));
   tail += (s->num_constraints * 8);
   tail += (s->num_exprs * 8);
   return ((ast_Expr**)(tail));
}

static uint32_t* ast_AsmStmt_getNames(const ast_AsmStmt* s)
{
   uint8_t* tail = ((uint8_t*)(s->constraints));
   tail += (s->num_constraints * 8);
   tail += (s->num_exprs * 8);
   tail += (s->num_clobbers * 8);
   return ((uint32_t*)(tail));
}

static void ast_AsmStmt_print(const ast_AsmStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
   string_buffer_Buf_indent(out, indent);
   ast_StringLiteral_print(s->asm_string, out, 0);
   if (s->num_outputs) {
      string_buffer_Buf_indent(out, indent);
      string_buffer_Buf_color(out, ast_col_Attr);
      string_buffer_Buf_add(out, "outputs\n");
      const uint32_t* names = ast_AsmStmt_getNames(s);
      const ast_Expr** constraints = ast_AsmStmt_getConstraints(s);
      ast_Expr** exprs = ast_AsmStmt_getExprs(s);
      for (uint32_t i = 0; (i < s->num_outputs); i++) {
         string_buffer_Buf_indent(out, (indent + 1));
         if (names[i]) {
            string_buffer_Buf_color(out, ast_col_Value);
            string_buffer_Buf_add(out, ast_idx2name(names[i]));
            string_buffer_Buf_space(out);
         }
         string_buffer_Buf_color(out, ast_col_Value);
         const ast_StringLiteral* sl = ((ast_StringLiteral*)(constraints[i]));
         ast_StringLiteral_printLiteral(sl, out);
         string_buffer_Buf_newline(out);
         const ast_Expr* e = exprs[i];
         ast_Expr_print(e, out, (indent + 1));
      }
   }
   if (s->num_inputs) {
      string_buffer_Buf_indent(out, indent);
      string_buffer_Buf_color(out, ast_col_Attr);
      string_buffer_Buf_add(out, "inputs\n");
      const uint32_t* names = ast_AsmStmt_getNames(s);
      const ast_Expr** constraints = ast_AsmStmt_getConstraints(s);
      ast_Expr** exprs = ast_AsmStmt_getExprs(s);
      for (uint32_t i = 0; (i < s->num_inputs); i++) {
         string_buffer_Buf_indent(out, (indent + 1));
         if (names[(i + s->num_outputs)]) {
            string_buffer_Buf_color(out, ast_col_Value);
            string_buffer_Buf_add(out, ast_idx2name(names[(i + s->num_outputs)]));
            string_buffer_Buf_space(out);
         }
         string_buffer_Buf_color(out, ast_col_Value);
         const ast_StringLiteral* sl = ((ast_StringLiteral*)(constraints[(i + s->num_outputs)]));
         ast_StringLiteral_printLiteral(sl, out);
         string_buffer_Buf_newline(out);
         const ast_Expr* e = exprs[(i + s->num_outputs)];
         ast_Expr_print(e, out, (indent + 1));
      }
   }
   if (s->num_clobbers) {
      string_buffer_Buf_indent(out, indent);
      string_buffer_Buf_color(out, ast_col_Attr);
      string_buffer_Buf_add(out, "clobbers: ");
      ast_Expr** clobbers = ast_AsmStmt_getClobbers(s);
      for (uint32_t i = 0; (i < s->num_clobbers); i++) {
         if ((i != 0)) string_buffer_Buf_space(out);
      }
   }
}

static ast_AssertStmt* ast_AssertStmt_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* inner)
{
   ast_AssertStmt* s = ast_context_Context_alloc(c, 16);
   ast_Stmt_init(&s->parent, ast_StmtKind_Assert);
   s->loc = loc;
   s->inner = inner;
   ast_Stats_addStmt(ast_StmtKind_Assert, 16);
   return s;
}

static ast_Stmt* ast_AssertStmt_instantiate(ast_AssertStmt* s, ast_Instantiator* inst)
{
   ast_AssertStmt* s2 = ast_AssertStmt_create(inst->c, s->loc, ast_Expr_instantiate(s->inner, inst));
   return ((ast_Stmt*)(s2));
}

static src_loc_SrcLoc ast_AssertStmt_getLoc(const ast_AssertStmt* s)
{
   return s->loc;
}

static ast_Expr* ast_AssertStmt_getInner(const ast_AssertStmt* s)
{
   return s->inner;
}

static ast_Expr** ast_AssertStmt_getInner2(ast_AssertStmt* s)
{
   return &s->inner;
}

static void ast_AssertStmt_setPointer(ast_AssertStmt* s)
{
   s->parent.assertStmtBits.is_pointer = 1;
}

static bool ast_AssertStmt_isPointer(const ast_AssertStmt* s)
{
   return s->parent.assertStmtBits.is_pointer;
}

static void ast_AssertStmt_print(const ast_AssertStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
   ast_Expr_print(s->inner, out, (indent + 1));
}

static ast_BreakStmt* ast_BreakStmt_create(ast_context_Context* c, src_loc_SrcLoc loc)
{
   ast_BreakStmt* s = ast_context_Context_alloc(c, 8);
   ast_Stmt_init(&s->parent, ast_StmtKind_Break);
   s->loc = loc;
   ast_Stats_addStmt(ast_StmtKind_Break, 8);
   return s;
}

static src_loc_SrcLoc ast_BreakStmt_getLoc(const ast_BreakStmt* s)
{
   return s->loc;
}

static void ast_BreakStmt_print(const ast_BreakStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
}

static ast_CompoundStmt* ast_CompoundStmt_create(ast_context_Context* c, src_loc_SrcLoc end, ast_Stmt** stmts, uint32_t count)
{
   c2_assert(((count < 65556)) != 0, "ast/compound_stmt.c2:36: ast.CompoundStmt.create", "count<65556");
   uint32_t size = (8 + (count * 8));
   ast_CompoundStmt* s = ast_context_Context_alloc(c, size);
   ast_Stmt_init(&s->parent, ast_StmtKind_Compound);
   s->parent.compoundStmtBits.count = count;
   s->endLoc = end;
   if (count) {
      memcpy(((void*)(s->stmts)), ((void*)(stmts)), (count * 8));
   }
   ast_Stats_addStmt(ast_StmtKind_Compound, size);
   return s;
}

static ast_CompoundStmt* ast_CompoundStmt_instantiate(ast_CompoundStmt* s, ast_Instantiator* inst)
{
   const uint32_t count = s->parent.compoundStmtBits.count;
   uint32_t size = (8 + (count * 8));
   ast_CompoundStmt* s2 = ast_context_Context_alloc(inst->c, size);
   s2->parent = s->parent;
   s2->endLoc = s->endLoc;
   for (uint32_t i = 0; (i < count); i++) {
      s2->stmts[i] = ast_Stmt_instantiate(s->stmts[i], inst);
   }
   return s2;
}

static src_loc_SrcLoc ast_CompoundStmt_getEndLoc(const ast_CompoundStmt* s)
{
   return s->endLoc;
}

static uint32_t ast_CompoundStmt_getCount(const ast_CompoundStmt* s)
{
   return s->parent.compoundStmtBits.count;
}

static ast_Stmt** ast_CompoundStmt_getStmts(ast_CompoundStmt* s)
{
   if (ast_CompoundStmt_getCount(s)) return s->stmts;

   return NULL;
}

static ast_Stmt* ast_CompoundStmt_getLastStmt(const ast_CompoundStmt* s)
{
   uint32_t count = ast_CompoundStmt_getCount(s);
   if (count) return s->stmts[(count - 1)];

   return NULL;
}

static void ast_CompoundStmt_print(const ast_CompoundStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
   const uint32_t count = s->parent.compoundStmtBits.count;
   for (uint32_t i = 0; (i < count); i++) {
      ast_Stmt_print(s->stmts[i], out, (indent + 1));
   }
}

static ast_ContinueStmt* ast_ContinueStmt_create(ast_context_Context* c, src_loc_SrcLoc loc)
{
   ast_ContinueStmt* s = ast_context_Context_alloc(c, 8);
   ast_Stmt_init(&s->parent, ast_StmtKind_Continue);
   s->loc = loc;
   ast_Stats_addStmt(ast_StmtKind_Continue, 8);
   return s;
}

static src_loc_SrcLoc ast_ContinueStmt_getLoc(const ast_ContinueStmt* s)
{
   return s->loc;
}

static void ast_ContinueStmt_print(const ast_ContinueStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
}

static ast_DoStmt* ast_DoStmt_create(ast_context_Context* c, ast_Stmt* cond, ast_Stmt* body)
{
   ast_DoStmt* s = ast_context_Context_alloc(c, 24);
   ast_Stmt_init(&s->parent, ast_StmtKind_Do);
   s->cond = cond;
   s->body = body;
   ast_Stats_addStmt(ast_StmtKind_Do, 24);
   return s;
}

static ast_Stmt* ast_DoStmt_instantiate(ast_DoStmt* s, ast_Instantiator* inst)
{
   ast_Stmt* cond2 = ast_Stmt_instantiate(s->cond, inst);
   ast_Stmt* body2 = ast_Stmt_instantiate(s->body, inst);
   return ((ast_Stmt*)(ast_DoStmt_create(inst->c, cond2, body2)));
}

static void ast_DoStmt_print(const ast_DoStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
   ast_Stmt_print(s->cond, out, (indent + 1));
   ast_Stmt_print(s->body, out, (indent + 1));
}

static ast_Stmt* ast_DoStmt_getCond(const ast_DoStmt* s)
{
   return s->cond;
}

static ast_Stmt* ast_DoStmt_getBody(const ast_DoStmt* s)
{
   return s->body;
}

static ast_FallthroughStmt* ast_FallthroughStmt_create(ast_context_Context* c, src_loc_SrcLoc loc)
{
   ast_FallthroughStmt* s = ast_context_Context_alloc(c, 8);
   ast_Stmt_init(&s->parent, ast_StmtKind_Fallthrough);
   s->loc = loc;
   ast_Stats_addStmt(ast_StmtKind_Fallthrough, 8);
   return s;
}

static src_loc_SrcLoc ast_FallthroughStmt_getLoc(const ast_FallthroughStmt* s)
{
   return s->loc;
}

static void ast_FallthroughStmt_print(const ast_FallthroughStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
}

static ast_ForStmt* ast_ForStmt_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Stmt* init_, ast_Expr* cond, ast_Expr* incr, ast_Stmt* body)
{
   ast_ForStmt* s = ast_context_Context_alloc(c, 40);
   ast_Stmt_init(&s->parent, ast_StmtKind_For);
   s->loc = loc;
   s->init = init_;
   s->cond = cond;
   s->incr = incr;
   s->body = body;
   ast_Stats_addStmt(ast_StmtKind_For, 40);
   return s;
}

static ast_Stmt* ast_ForStmt_instantiate(ast_ForStmt* s, ast_Instantiator* inst)
{
   ast_Stmt* init2 = s->init ? ast_Stmt_instantiate(s->init, inst) : NULL;
   ast_Expr* cond2 = s->cond ? ast_Expr_instantiate(s->cond, inst) : NULL;
   ast_Expr* incr2 = s->incr ? ast_Expr_instantiate(s->incr, inst) : NULL;
   ast_Stmt* body2 = s->body ? ast_Stmt_instantiate(s->body, inst) : NULL;
   return ((ast_Stmt*)(ast_ForStmt_create(inst->c, s->loc, init2, cond2, incr2, body2)));
}

static src_loc_SrcLoc ast_ForStmt_getLoc(const ast_ForStmt* s)
{
   return s->loc;
}

static ast_Stmt* ast_ForStmt_getInit(const ast_ForStmt* s)
{
   return s->init;
}

static ast_Expr* ast_ForStmt_getCond(const ast_ForStmt* s)
{
   return s->cond;
}

static ast_Expr* ast_ForStmt_getIncr(const ast_ForStmt* s)
{
   return s->incr;
}

static ast_Stmt* ast_ForStmt_getBody(const ast_ForStmt* s)
{
   return s->body;
}

static ast_Stmt** ast_ForStmt_getInit2(ast_ForStmt* s)
{
   return s->init ? &s->init : NULL;
}

static ast_Expr** ast_ForStmt_getCond2(ast_ForStmt* s)
{
   return s->cond ? &s->cond : NULL;
}

static ast_Expr** ast_ForStmt_getIncr2(ast_ForStmt* s)
{
   return s->incr ? &s->incr : NULL;
}

static ast_Stmt** ast_ForStmt_getBody2(ast_ForStmt* s)
{
   return s->body ? &s->body : NULL;
}

static void ast_ForStmt_print(const ast_ForStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
   if (s->init) ast_Stmt_print(s->init, out, (indent + 1));
   if (s->cond) ast_Expr_print(s->cond, out, (indent + 1));
   if (s->incr) ast_Expr_print(s->incr, out, (indent + 1));
   if (s->body) ast_Stmt_print(s->body, out, (indent + 1));
}

static ast_GotoStmt* ast_GotoStmt_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc)
{
   ast_GotoStmt* s = ast_context_Context_alloc(c, 12);
   ast_Stmt_init(&s->parent, ast_StmtKind_Goto);
   s->loc = loc;
   s->name = name;
   ast_Stats_addStmt(ast_StmtKind_Goto, 12);
   return s;
}

static const char* ast_GotoStmt_getName(const ast_GotoStmt* g)
{
   return ast_idx2name(g->name);
}

static uint32_t ast_GotoStmt_getNameIdx(const ast_GotoStmt* g)
{
   return g->name;
}

static src_loc_SrcLoc ast_GotoStmt_getLoc(const ast_GotoStmt* g)
{
   return g->loc;
}

static void ast_GotoStmt_print(const ast_GotoStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_color(out, ast_col_Value);
   string_buffer_Buf_print(out, " %s\n", ast_idx2name(s->name));
}

static ast_IfStmt* ast_IfStmt_create(ast_context_Context* c, ast_Stmt* cond, ast_Stmt* then, ast_Stmt* else_stmt)
{
   uint32_t size = 24;
   if (else_stmt) size += 8;
   ast_IfStmt* s = ast_context_Context_alloc(c, size);
   ast_Stmt_init(&s->parent, ast_StmtKind_If);
   s->cond = cond;
   s->then = then;
   if (else_stmt) {
      s->parent.ifStmtBits.has_else = 1;
      s->else_stmt[0] = else_stmt;
   }
   ast_Stats_addStmt(ast_StmtKind_If, size);
   return s;
}

static ast_Stmt* ast_IfStmt_instantiate(ast_IfStmt* s, ast_Instantiator* inst)
{
   ast_Stmt* cond2 = ast_Stmt_instantiate(s->cond, inst);
   ast_Stmt* then2 = ast_Stmt_instantiate(s->then, inst);
   ast_Stmt* else2 = NULL;
   if (s->parent.ifStmtBits.has_else) else2 = ast_Stmt_instantiate(s->else_stmt[0], inst);
   return ((ast_Stmt*)(ast_IfStmt_create(inst->c, cond2, then2, else2)));
}

static ast_Stmt* ast_IfStmt_getCond(const ast_IfStmt* s)
{
   return s->cond;
}

static ast_Stmt** ast_IfStmt_getCond2(ast_IfStmt* s)
{
   return &s->cond;
}

static ast_Stmt* ast_IfStmt_getThen(const ast_IfStmt* s)
{
   return s->then;
}

static ast_Stmt* ast_IfStmt_getElse(const ast_IfStmt* s)
{
   if (s->parent.ifStmtBits.has_else) return s->else_stmt[0];

   return NULL;
}

static void ast_IfStmt_print(const ast_IfStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
   ast_Stmt_print(s->cond, out, (indent + 1));
   if (s->then) ast_Stmt_print(s->then, out, (indent + 1));
   if (s->parent.ifStmtBits.has_else) ast_Stmt_print(s->else_stmt[0], out, (indent + 1));
}

static ast_LabelStmt* ast_LabelStmt_create(ast_context_Context* c, uint32_t name, src_loc_SrcLoc loc)
{
   ast_LabelStmt* s = ast_context_Context_alloc(c, 12);
   ast_Stmt_init(&s->parent, ast_StmtKind_Label);
   s->loc = loc;
   s->name = name;
   ast_Stats_addStmt(ast_StmtKind_Label, 12);
   return s;
}

static const char* ast_LabelStmt_getName(const ast_LabelStmt* s)
{
   return ast_idx2name(s->name);
}

static uint32_t ast_LabelStmt_getNameIdx(const ast_LabelStmt* s)
{
   return s->name;
}

static src_loc_SrcLoc ast_LabelStmt_getLoc(const ast_LabelStmt* s)
{
   return s->loc;
}

static void ast_LabelStmt_print(const ast_LabelStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_color(out, ast_col_Value);
   string_buffer_Buf_print(out, " %s\n", ast_idx2name(s->name));
}

static ast_ReturnStmt* ast_ReturnStmt_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* value)
{
   uint32_t size = 8;
   if (value) size += 8;
   ast_ReturnStmt* s = ast_context_Context_alloc(c, size);
   ast_Stmt_init(&s->parent, ast_StmtKind_Return);
   s->loc = loc;
   if (value) {
      s->parent.returnStmtBits.has_value = 1;
      s->value[0] = value;
   }
   ast_Stats_addStmt(ast_StmtKind_Return, size);
   return s;
}

static ast_Stmt* ast_ReturnStmt_instantiate(ast_ReturnStmt* s, ast_Instantiator* inst)
{
   if (!s->parent.returnStmtBits.has_value) return ((ast_Stmt*)(s));

   return ((ast_Stmt*)(ast_ReturnStmt_create(inst->c, s->loc, ast_Expr_instantiate(s->value[0], inst))));
}

static src_loc_SrcLoc ast_ReturnStmt_getLoc(const ast_ReturnStmt* s)
{
   return s->loc;
}

static ast_Expr* ast_ReturnStmt_getValue(const ast_ReturnStmt* s)
{
   if (s->parent.returnStmtBits.has_value) return s->value[0];

   return NULL;
}

static ast_Expr** ast_ReturnStmt_getValue2(ast_ReturnStmt* s)
{
   if (s->parent.returnStmtBits.has_value) return &s->value[0];

   return NULL;
}

static void ast_ReturnStmt_print(const ast_ReturnStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
   if (s->parent.returnStmtBits.has_value) {
      ast_Expr_print(s->value[0], out, (indent + 1));
   }
}

static ast_SwitchCase* ast_SwitchCase_create(ast_context_Context* c, src_loc_SrcLoc loc, bool is_default, ast_Expr* cond, ast_IdentifierExpr** multi, uint32_t num_multi, ast_Stmt** stmts, uint32_t numStmts)
{
   c2_assert(((numStmts < 1024)) != 0, "ast/switch_case.c2:54: ast.SwitchCase.create", "numStmts<1024");
   c2_assert(((num_multi < 256)) != 0, "ast/switch_case.c2:55: ast.SwitchCase.create", "num_multi<256");
   uint32_t size = (16 + (numStmts * 8));
   if (num_multi) {
      size += (((num_multi - 1)) * 8);
   }
   ast_SwitchCase* s = ast_context_Context_alloc(c, size);
   s->allbits = 0;
   s->bits.is_default = is_default;
   s->bits.num_stmts = numStmts;
   s->bits.num_subcases = num_multi;
   s->loc = loc;
   if (num_multi) {
      memcpy(s->multi_cond, multi, (num_multi * 8));
   } else {
      s->cond = cond;
   }
   ast_Stmt** dst_stmts = ast_SwitchCase_getStmts(s);
   memcpy(((void*)(dst_stmts)), ((void*)(stmts)), (numStmts * 8));
   ast_Stats_addSwitchCase(size);
   return s;
}

static ast_SwitchCase* ast_SwitchCase_instantiate(ast_SwitchCase* s, ast_Instantiator* inst)
{
   uint32_t numStmts = ast_SwitchCase_getNumStmts(s);
   uint32_t size = (16 + (numStmts * 8));
   if (s->bits.num_subcases) {
      size += (((s->bits.num_subcases - 1)) * 8);
   }
   ast_SwitchCase* s2 = ast_context_Context_alloc(inst->c, size);
   s2->allbits = s->allbits;
   s2->loc = s->loc;
   if (s->bits.num_subcases) {
      memcpy(s2->multi_cond, s->multi_cond, (s->bits.num_subcases * 8));
   } else {
      s2->cond = s->cond;
   }
   ast_Stmt** src_stmts = ast_SwitchCase_getStmts(s);
   ast_Stmt** dst_stmts = ast_SwitchCase_getStmts(s);
   for (uint32_t i = 0; (i < numStmts); i++) {
      dst_stmts[i] = ast_Stmt_instantiate(src_stmts[i], inst);
   }
   ast_Stats_addSwitchCase(size);
   return s;
}

static uint32_t ast_SwitchCase_getNumStmts(const ast_SwitchCase* s)
{
   return s->bits.num_stmts;
}

static ast_Stmt** ast_SwitchCase_getStmts(const ast_SwitchCase* s)
{
   uint8_t* stmts = ((uint8_t*)(&s->cond));
   if (s->bits.num_subcases) {
      stmts += (s->bits.num_subcases * 8);
   } else {
      stmts += 8;
   }
   return ((ast_Stmt**)(stmts));
}

static bool ast_SwitchCase_isDefault(const ast_SwitchCase* s)
{
   return s->bits.is_default;
}

static uint32_t ast_SwitchCase_numMulti(const ast_SwitchCase* s)
{
   return s->bits.num_subcases;
}

static bool ast_SwitchCase_hasDecls(const ast_SwitchCase* s)
{
   return s->bits.has_decls;
}

static void ast_SwitchCase_setHasDecls(ast_SwitchCase* s)
{
   s->bits.has_decls = 1;
}

static src_loc_SrcLoc ast_SwitchCase_getLoc(const ast_SwitchCase* s)
{
   return s->loc;
}

static ast_Expr* ast_SwitchCase_getCond(const ast_SwitchCase* s)
{
   return s->cond;
}

static ast_Expr** ast_SwitchCase_getCond2(ast_SwitchCase* s)
{
   return &s->cond;
}

static ast_IdentifierExpr** ast_SwitchCase_getMultiCond(ast_SwitchCase* s)
{
   return s->multi_cond;
}

static void ast_SwitchCase_print(const ast_SwitchCase* s, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_color(out, ast_col_Stmt);
   string_buffer_Buf_add(out, "SwitchCase");
   if (s->bits.is_default) {
      string_buffer_Buf_color(out, ast_col_Attr);
      string_buffer_Buf_add(out, " default");
   }
   if (s->bits.has_decls) {
      string_buffer_Buf_color(out, ast_col_Attr);
      string_buffer_Buf_add(out, " decls");
   }
   if (s->bits.num_subcases) {
      string_buffer_Buf_color(out, ast_col_Attr);
      string_buffer_Buf_add(out, " multi");
   }
   string_buffer_Buf_newline(out);
   if (s->cond) {
      if (s->bits.num_subcases) {
         for (uint32_t i = 0; (i < s->bits.num_subcases); i++) {
            ast_IdentifierExpr_print(s->multi_cond[i], out, (indent + 1));
         }
      } else {
         ast_Expr_print(s->cond, out, (indent + 1));
      }
   }
   ast_Stmt** stmts = ast_SwitchCase_getStmts(s);
   for (uint32_t i = 0; (i < s->bits.num_stmts); i++) {
      ast_Stmt_print(stmts[i], out, (indent + 1));
   }
}

static ast_SwitchStmt* ast_SwitchStmt_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* cond, ast_SwitchCase** cases, uint32_t numCases, bool is_sswitch)
{
   uint32_t size = (16 + (numCases * 8));
   ast_SwitchStmt* s = ast_context_Context_alloc(c, size);
   ast_Stmt_init(&s->parent, ast_StmtKind_Switch);
   s->parent.switchStmtBits.is_sswitch = is_sswitch;
   s->parent.switchStmtBits.num_cases = numCases;
   s->loc = loc;
   s->cond = cond;
   memcpy(((void*)(s->cases)), ((void*)(cases)), (numCases * 8));
   ast_Stats_addStmt(ast_StmtKind_Switch, size);
   return s;
}

static ast_Stmt* ast_SwitchStmt_instantiate(ast_SwitchStmt* s, ast_Instantiator* inst)
{
   uint32_t numCases = ast_SwitchStmt_getNumCases(s);
   uint32_t size = (16 + (numCases * 8));
   ast_SwitchStmt* s2 = ast_context_Context_alloc(inst->c, size);
   s2->parent = s->parent;
   s2->cond = ast_Expr_instantiate(s->cond, inst);
   for (uint32_t i = 0; (i < numCases); i++) {
      s2->cases[i] = ast_SwitchCase_instantiate(s->cases[i], inst);
   }
   ast_Stats_addStmt(ast_StmtKind_Switch, size);
   return ((ast_Stmt*)(s2));
}

static src_loc_SrcLoc ast_SwitchStmt_getLoc(const ast_SwitchStmt* s)
{
   return s->loc;
}

static ast_Expr* ast_SwitchStmt_getCond(const ast_SwitchStmt* s)
{
   return s->cond;
}

static ast_Expr** ast_SwitchStmt_getCond2(ast_SwitchStmt* s)
{
   return s->cond ? &s->cond : NULL;
}

static bool ast_SwitchStmt_isSSwitch(const ast_SwitchStmt* s)
{
   return s->parent.switchStmtBits.is_sswitch;
}

static uint32_t ast_SwitchStmt_getNumCases(const ast_SwitchStmt* s)
{
   return s->parent.switchStmtBits.num_cases;
}

static ast_SwitchCase** ast_SwitchStmt_getCases(ast_SwitchStmt* s)
{
   return s->cases;
}

static void ast_SwitchStmt_print(const ast_SwitchStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   if (ast_SwitchStmt_isSSwitch(s)) {
      string_buffer_Buf_color(out, ast_col_Attr);
      string_buffer_Buf_add(out, " string");
   }
   string_buffer_Buf_newline(out);
   ast_Expr_print(s->cond, out, (indent + 1));
   for (uint32_t i = 0; (i < s->parent.switchStmtBits.num_cases); i++) {
      ast_SwitchCase_print(s->cases[i], out, (indent + 1));
   }
}

static ast_WhileStmt* ast_WhileStmt_create(ast_context_Context* c, ast_Stmt* cond, ast_Stmt* body)
{
   ast_WhileStmt* s = ast_context_Context_alloc(c, 24);
   ast_Stmt_init(&s->parent, ast_StmtKind_While);
   s->cond = cond;
   s->body = body;
   ast_Stats_addStmt(ast_StmtKind_While, 24);
   return s;
}

static ast_Stmt* ast_WhileStmt_instantiate(ast_WhileStmt* s, ast_Instantiator* inst)
{
   ast_Stmt* cond2 = ast_Stmt_instantiate(s->cond, inst);
   ast_Stmt* body2 = ast_Stmt_instantiate(s->body, inst);
   return ((ast_Stmt*)(ast_WhileStmt_create(inst->c, cond2, body2)));
}

static void ast_WhileStmt_print(const ast_WhileStmt* s, string_buffer_Buf* out, uint32_t indent)
{
   ast_Stmt_printKind(&s->parent, out, indent);
   string_buffer_Buf_newline(out);
   ast_Stmt_print(s->cond, out, (indent + 1));
   ast_Stmt_print(s->body, out, (indent + 1));
}

static ast_Stmt* ast_WhileStmt_getCond(const ast_WhileStmt* s)
{
   return s->cond;
}

static ast_Stmt** ast_WhileStmt_getCond2(ast_WhileStmt* s)
{
   return &s->cond;
}

static ast_Stmt* ast_WhileStmt_getBody(const ast_WhileStmt* s)
{
   return s->body;
}

static void ast_Expr_init(ast_Expr* e, ast_ExprKind k, src_loc_SrcLoc loc, bool ctv, bool ctc, bool has_effect, ast_ValType valtype)
{
   ast_Stmt_init(&e->parent, ast_StmtKind_Expr);
   e->parent.exprBits.kind = k;
   e->parent.exprBits.is_ctv = ctv;
   e->parent.exprBits.is_ctc = ctc;
   e->parent.exprBits.has_effect = has_effect;
   e->parent.exprBits.valtype = valtype;
   e->loc = loc;
   e->qt.ptr = 0;
}

static ast_Expr* ast_Expr_instantiate(ast_Expr* e, ast_Instantiator* inst)
{
   switch (ast_Expr_getKind(e)) {
   case ast_ExprKind_IntegerLiteral:
      return e;
   case ast_ExprKind_FloatLiteral:
      return e;
   case ast_ExprKind_BooleanLiteral:
      return e;
   case ast_ExprKind_CharLiteral:
      return e;
   case ast_ExprKind_StringLiteral:
      return e;
   case ast_ExprKind_Nil:
      return e;
   case ast_ExprKind_Identifier:
      return ast_IdentifierExpr_instantiate(((ast_IdentifierExpr*)(e)), inst);
   case ast_ExprKind_Type:
      return ast_TypeExpr_instantiate(((ast_TypeExpr*)(e)), inst);
   case ast_ExprKind_Call:
      return ast_CallExpr_instantiate(((ast_CallExpr*)(e)), inst);
   case ast_ExprKind_InitList:
      return ast_InitListExpr_instantiate(((ast_InitListExpr*)(e)), inst);
   case ast_ExprKind_FieldDesignatedInit:
      return ast_FieldDesignatedInitExpr_instantiate(((ast_FieldDesignatedInitExpr*)(e)), inst);
   case ast_ExprKind_ArrayDesignatedInit:
      return ast_ArrayDesignatedInitExpr_instantiate(((ast_ArrayDesignatedInitExpr*)(e)), inst);
   case ast_ExprKind_BinaryOperator:
      return ast_BinaryOperator_instantiate(((ast_BinaryOperator*)(e)), inst);
   case ast_ExprKind_UnaryOperator:
      return ast_UnaryOperator_instantiate(((ast_UnaryOperator*)(e)), inst);
   case ast_ExprKind_ConditionalOperator:
      return ast_ConditionalOperator_instantiate(((ast_ConditionalOperator*)(e)), inst);
   case ast_ExprKind_Builtin:
      return ast_BuiltinExpr_instantiate(((ast_BuiltinExpr*)(e)), inst);
   case ast_ExprKind_ArraySubscript:
      return ast_ArraySubscriptExpr_instantiate(((ast_ArraySubscriptExpr*)(e)), inst);
   case ast_ExprKind_Member:
      return ast_MemberExpr_instantiate(((ast_MemberExpr*)(e)), inst);
   case ast_ExprKind_Paren:
      return ast_ParenExpr_instantiate(((ast_ParenExpr*)(e)), inst);
   case ast_ExprKind_BitOffset:
      return ast_BitOffsetExpr_instantiate(((ast_BitOffsetExpr*)(e)), inst);
   case ast_ExprKind_ExplicitCast:
      return ast_ExplicitCastExpr_instantiate(((ast_ExplicitCastExpr*)(e)), inst);
   case ast_ExprKind_ImplicitCast:
      break;
   }
   ast_Expr_dump(e);
   c2_assert((0) != 0, "ast/expr.c2:190: ast.Expr.instantiate", "0");
   return NULL;
}

static ast_Stmt* ast_Expr_asStmt(ast_Expr* e)
{
   return &e->parent;
}

static ast_ExprKind ast_Expr_getKind(const ast_Expr* e)
{
   return ((ast_ExprKind)(e->parent.exprBits.kind));
}

static bool ast_Expr_isIntegerLiteral(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_IntegerLiteral);
}

static bool ast_Expr_isStringLiteral(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_StringLiteral);
}

static bool ast_Expr_isNil(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_Nil);
}

static bool ast_Expr_isIdentifier(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_Identifier);
}

static bool ast_Expr_isCall(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_Call);
}

static bool ast_Expr_isImplicitCast(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_ImplicitCast);
}

static bool ast_Expr_isType(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_Type);
}

static bool ast_Expr_isInitList(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_InitList);
}

static bool ast_Expr_isBinaryOperator(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_BinaryOperator);
}

static bool ast_Expr_isMember(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_Member);
}

static bool ast_Expr_isFieldDesignatedInit(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_FieldDesignatedInit);
}

static bool ast_Expr_isArrayDesignatedInit(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_ArrayDesignatedInit);
}

static bool ast_Expr_isBitOffset(const ast_Expr* e)
{
   return (ast_Expr_getKind(e) == ast_ExprKind_BitOffset);
}

static bool ast_Expr_isCtv(const ast_Expr* e)
{
   return e->parent.exprBits.is_ctv;
}

static bool ast_Expr_isCtc(const ast_Expr* e)
{
   return e->parent.exprBits.is_ctc;
}

static void ast_Expr_setCtv(ast_Expr* e)
{
   e->parent.exprBits.is_ctv = true;
}

static void ast_Expr_setCtc(ast_Expr* e)
{
   e->parent.exprBits.is_ctc = true;
}

static void ast_Expr_copyCtcFlags(ast_Expr* e, const ast_Expr* other)
{
   e->parent.exprBits.is_ctc = other->parent.exprBits.is_ctc;
}

static void ast_Expr_copyConstantFlags(ast_Expr* e, const ast_Expr* other)
{
   e->parent.exprBits.is_ctc = other->parent.exprBits.is_ctc;
   e->parent.exprBits.is_ctv = other->parent.exprBits.is_ctv;
}

static void ast_Expr_combineConstantFlags(ast_Expr* e, const ast_Expr* lhs, const ast_Expr* rhs)
{
   e->parent.exprBits.is_ctc = (lhs->parent.exprBits.is_ctc && rhs->parent.exprBits.is_ctc);
   e->parent.exprBits.is_ctv = (lhs->parent.exprBits.is_ctv && rhs->parent.exprBits.is_ctv);
}

static bool ast_Expr_hasEffect(const ast_Expr* e)
{
   return e->parent.exprBits.has_effect;
}

static ast_ValType ast_Expr_getValType(const ast_Expr* e)
{
   return ((ast_ValType)(e->parent.exprBits.valtype));
}

static bool ast_Expr_isNValue(const ast_Expr* e)
{
   return (ast_Expr_getValType(e) == ast_ValType_NValue);
}

static bool ast_Expr_isRValue(const ast_Expr* e)
{
   return (ast_Expr_getValType(e) == ast_ValType_RValue);
}

static bool ast_Expr_isLValue(const ast_Expr* e)
{
   return (ast_Expr_getValType(e) == ast_ValType_LValue);
}

static void ast_Expr_setLValue(ast_Expr* e)
{
   e->parent.exprBits.valtype = ast_ValType_LValue;
}

static void ast_Expr_setRValue(ast_Expr* e)
{
   e->parent.exprBits.valtype = ast_ValType_RValue;
}

static void ast_Expr_setValType(ast_Expr* e, ast_ValType valtype)
{
   e->parent.exprBits.valtype = valtype;
}

static void ast_Expr_copyValType(ast_Expr* e, const ast_Expr* other)
{
   e->parent.exprBits.valtype = other->parent.exprBits.valtype;
}

static src_loc_SrcLoc ast_Expr_getLoc(const ast_Expr* e)
{
   return e->loc;
}

static src_loc_SrcLoc ast_Expr_getStartLoc(const ast_Expr* e)
{
   switch (ast_Expr_getKind(e)) {
   case ast_ExprKind_IntegerLiteral:
      __attribute__((fallthrough));
   case ast_ExprKind_FloatLiteral:
      __attribute__((fallthrough));
   case ast_ExprKind_BooleanLiteral:
      __attribute__((fallthrough));
   case ast_ExprKind_CharLiteral:
      __attribute__((fallthrough));
   case ast_ExprKind_StringLiteral:
      __attribute__((fallthrough));
   case ast_ExprKind_Nil:
      __attribute__((fallthrough));
   case ast_ExprKind_Identifier:
      break;
   case ast_ExprKind_Type:
      break;
   case ast_ExprKind_Call:
      break;
   case ast_ExprKind_InitList:
      break;
   case ast_ExprKind_FieldDesignatedInit:
      break;
   case ast_ExprKind_ArrayDesignatedInit:
      break;
   case ast_ExprKind_BinaryOperator: {
      const ast_BinaryOperator* b = ((ast_BinaryOperator*)(e));
      return ast_Expr_getStartLoc(ast_BinaryOperator_getLHS(b));
   }
   case ast_ExprKind_UnaryOperator: {
      const ast_UnaryOperator* u = ((ast_UnaryOperator*)(e));
      return ast_UnaryOperator_getStartLoc(u);
   }
   case ast_ExprKind_ConditionalOperator: {
      const ast_ConditionalOperator* c = ((ast_ConditionalOperator*)(e));
      return ast_Expr_getStartLoc(ast_ConditionalOperator_getCond(c));
   }
   case ast_ExprKind_Builtin:
      break;
   case ast_ExprKind_ArraySubscript: {
      const ast_ArraySubscriptExpr* a = ((ast_ArraySubscriptExpr*)(e));
      return ast_Expr_getStartLoc(ast_ArraySubscriptExpr_getBase(a));
   }
   case ast_ExprKind_Member: {
      const ast_MemberExpr* m = ((ast_MemberExpr*)(e));
      return ast_MemberExpr_getStartLoc(m);
   }
   case ast_ExprKind_Paren:
      break;
   case ast_ExprKind_BitOffset:
      break;
   case ast_ExprKind_ExplicitCast:
      break;
   case ast_ExprKind_ImplicitCast: {
      const ast_ImplicitCastExpr* c = ((ast_ImplicitCastExpr*)(e));
      return ast_Expr_getStartLoc(ast_ImplicitCastExpr_getInner(c));
   }
   }
   return e->loc;
}

static src_loc_SrcLoc ast_Expr_getEndLoc(const ast_Expr* e)
{
   switch (ast_Expr_getKind(e)) {
   case ast_ExprKind_IntegerLiteral:
      __attribute__((fallthrough));
   case ast_ExprKind_FloatLiteral:
      __attribute__((fallthrough));
   case ast_ExprKind_BooleanLiteral:
      __attribute__((fallthrough));
   case ast_ExprKind_CharLiteral:
      __attribute__((fallthrough));
   case ast_ExprKind_StringLiteral:
      __attribute__((fallthrough));
   case ast_ExprKind_Nil:
      __attribute__((fallthrough));
   case ast_ExprKind_Identifier:
      break;
   case ast_ExprKind_Type:
      break;
   case ast_ExprKind_Call:
      break;
   case ast_ExprKind_InitList:
      break;
   case ast_ExprKind_FieldDesignatedInit:
      break;
   case ast_ExprKind_ArrayDesignatedInit:
      break;
   case ast_ExprKind_BinaryOperator: {
      const ast_BinaryOperator* b = ((ast_BinaryOperator*)(e));
      return ast_Expr_getEndLoc(ast_BinaryOperator_getRHS(b));
   }
   case ast_ExprKind_UnaryOperator: {
      const ast_UnaryOperator* u = ((ast_UnaryOperator*)(e));
      return ast_UnaryOperator_getEndLoc(u);
   }
   case ast_ExprKind_ConditionalOperator: {
      const ast_ConditionalOperator* c = ((ast_ConditionalOperator*)(e));
      return ast_Expr_getEndLoc(ast_ConditionalOperator_getRHS(c));
   }
   case ast_ExprKind_Builtin: {
      const ast_BuiltinExpr* bi = ((ast_BuiltinExpr*)(e));
      return ast_BuiltinExpr_getEndLoc(bi);
   }
   case ast_ExprKind_ArraySubscript:
      break;
   case ast_ExprKind_Member: {
      const ast_MemberExpr* m = ((ast_MemberExpr*)(e));
      return ast_MemberExpr_getEndLoc(m);
   }
   case ast_ExprKind_Paren: {
      const ast_ParenExpr* p = ((ast_ParenExpr*)(e));
      return (ast_Expr_getEndLoc(ast_ParenExpr_getInner(p)) + 1);
   }
   case ast_ExprKind_BitOffset:
      break;
   case ast_ExprKind_ExplicitCast: {
      const ast_ExplicitCastExpr* c = ((ast_ExplicitCastExpr*)(e));
      return (ast_Expr_getEndLoc(ast_ExplicitCastExpr_getInner(c)) + 1);
   }
   case ast_ExprKind_ImplicitCast: {
      const ast_ImplicitCastExpr* c = ((ast_ImplicitCastExpr*)(e));
      return ast_Expr_getEndLoc(ast_ImplicitCastExpr_getInner(c));
   }
   }
   return e->loc;
}

static src_loc_SrcRange ast_Expr_getRange(const ast_Expr* e)
{
   src_loc_SrcRange range = { ast_Expr_getStartLoc(e), ast_Expr_getEndLoc(e) };
   return range;
}

static void ast_Expr_setType(ast_Expr* e, ast_QualType qt_)
{
   e->qt = qt_;
}

static ast_QualType ast_Expr_getType(const ast_Expr* e)
{
   return e->qt;
}

static void ast_Expr_dump(const ast_Expr* e)
{
   string_buffer_Buf* out = string_buffer_create((10 * 4096), ast_useColor(), 2);
   ast_Expr_print(e, out, 0);
   string_buffer_Buf_color(out, ast_col_Normal);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static void ast_Expr_print(const ast_Expr* e, string_buffer_Buf* out, uint32_t indent)
{
   switch (ast_Expr_getKind(e)) {
   case ast_ExprKind_IntegerLiteral:
      ast_IntegerLiteral_print(((ast_IntegerLiteral*)(e)), out, indent);
      break;
   case ast_ExprKind_FloatLiteral:
      ast_FloatLiteral_print(((ast_FloatLiteral*)(e)), out, indent);
      break;
   case ast_ExprKind_BooleanLiteral:
      ast_BooleanLiteral_print(((ast_BooleanLiteral*)(e)), out, indent);
      break;
   case ast_ExprKind_CharLiteral:
      ast_CharLiteral_print(((ast_CharLiteral*)(e)), out, indent);
      break;
   case ast_ExprKind_StringLiteral:
      ast_StringLiteral_print(((ast_StringLiteral*)(e)), out, indent);
      break;
   case ast_ExprKind_Nil:
      ast_NilExpr_print(((ast_NilExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_Identifier:
      ast_IdentifierExpr_print(((ast_IdentifierExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_Type:
      ast_TypeExpr_print(((ast_TypeExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_Call:
      ast_CallExpr_print(((ast_CallExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_InitList:
      ast_InitListExpr_print(((ast_InitListExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_FieldDesignatedInit:
      ast_FieldDesignatedInitExpr_print(((ast_FieldDesignatedInitExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_ArrayDesignatedInit:
      ast_ArrayDesignatedInitExpr_print(((ast_ArrayDesignatedInitExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_BinaryOperator:
      ast_BinaryOperator_print(((ast_BinaryOperator*)(e)), out, indent);
      break;
   case ast_ExprKind_UnaryOperator:
      ast_UnaryOperator_print(((ast_UnaryOperator*)(e)), out, indent);
      break;
   case ast_ExprKind_ConditionalOperator:
      ast_ConditionalOperator_print(((ast_ConditionalOperator*)(e)), out, indent);
      break;
   case ast_ExprKind_Builtin:
      ast_BuiltinExpr_print(((ast_BuiltinExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_ArraySubscript:
      ast_ArraySubscriptExpr_print(((ast_ArraySubscriptExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_Member:
      ast_MemberExpr_print(((ast_MemberExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_Paren:
      ast_ParenExpr_print(((ast_ParenExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_BitOffset:
      ast_BitOffsetExpr_print(((ast_BitOffsetExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_ExplicitCast:
      ast_ExplicitCastExpr_print(((ast_ExplicitCastExpr*)(e)), out, indent);
      break;
   case ast_ExprKind_ImplicitCast:
      ast_ImplicitCastExpr_print(((ast_ImplicitCastExpr*)(e)), out, indent);
      break;
   }
}

static void ast_Expr_printLiteral(const ast_Expr* e, string_buffer_Buf* out)
{
   switch (ast_Expr_getKind(e)) {
   case ast_ExprKind_IntegerLiteral:
      ast_IntegerLiteral_printLiteral(((ast_IntegerLiteral*)(e)), out);
      return;
   case ast_ExprKind_FloatLiteral:
      ast_FloatLiteral_printLiteral(((ast_FloatLiteral*)(e)), out);
      return;
   case ast_ExprKind_BooleanLiteral:
      ast_BooleanLiteral_printLiteral(((ast_BooleanLiteral*)(e)), out);
      return;
   case ast_ExprKind_CharLiteral:
      ast_CharLiteral_printLiteral(((ast_CharLiteral*)(e)), out);
      return;
   case ast_ExprKind_StringLiteral:
      ast_StringLiteral_printLiteral(((ast_StringLiteral*)(e)), out);
      return;
   case ast_ExprKind_Nil:
      ast_NilExpr_printLiteral(((ast_NilExpr*)(e)), out);
      return;
   case ast_ExprKind_Identifier:
      ast_IdentifierExpr_printLiteral(((ast_IdentifierExpr*)(e)), out);
      return;
   case ast_ExprKind_Type:
      break;
   case ast_ExprKind_Call:
      ast_CallExpr_printLiteral(((ast_CallExpr*)(e)), out);
      return;
   case ast_ExprKind_InitList:
      break;
   case ast_ExprKind_FieldDesignatedInit:
      break;
   case ast_ExprKind_ArrayDesignatedInit:
      break;
   case ast_ExprKind_BinaryOperator:
      ast_BinaryOperator_printLiteral(((ast_BinaryOperator*)(e)), out);
      return;
   case ast_ExprKind_UnaryOperator:
      ast_UnaryOperator_printLiteral(((ast_UnaryOperator*)(e)), out);
      return;
   case ast_ExprKind_ConditionalOperator:
      ast_ConditionalOperator_printLiteral(((ast_ConditionalOperator*)(e)), out);
      return;
   case ast_ExprKind_Builtin:
      ast_BuiltinExpr_printLiteral(((ast_BuiltinExpr*)(e)), out);
      return;
   case ast_ExprKind_ArraySubscript:
      ast_ArraySubscriptExpr_printLiteral(((ast_ArraySubscriptExpr*)(e)), out);
      return;
   case ast_ExprKind_Member:
      ast_MemberExpr_printLiteral(((ast_MemberExpr*)(e)), out);
      return;
   case ast_ExprKind_Paren:
      ast_ParenExpr_printLiteral(((ast_ParenExpr*)(e)), out);
      return;
   case ast_ExprKind_BitOffset:
      ast_BitOffsetExpr_printLiteral(((ast_BitOffsetExpr*)(e)), out);
      return;
   case ast_ExprKind_ExplicitCast:
      ast_ExplicitCastExpr_printLiteral(((ast_ExplicitCastExpr*)(e)), out);
      return;
   case ast_ExprKind_ImplicitCast:
      ast_ImplicitCastExpr_printLiteral(((ast_ImplicitCastExpr*)(e)), out);
      return;
   }
   string_buffer_Buf_print(out, "?? kind=%u", ast_Expr_getKind(e));
}

static void ast_Expr_printKind(const ast_Expr* e, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_color(out, ast_col_Expr);
   string_buffer_Buf_add(out, ast_exprKind_names[ast_Expr_getKind(e)]);
}

static void ast_Expr_printTypeBits(const ast_Expr* e, string_buffer_Buf* out)
{
   string_buffer_Buf_space(out);
   ast_QualType_printQuoted(&e->qt, out);
   string_buffer_Buf_color(out, ast_col_Attr);
   if (e->parent.exprBits.is_ctc) string_buffer_Buf_add(out, " CTC");
   if (e->parent.exprBits.is_ctv) string_buffer_Buf_add(out, " CTV");
   string_buffer_Buf_space(out);
   string_buffer_Buf_add(out, ast_valType_names[ast_Expr_getValType(e)]);
}

static ast_ArrayDesignatedInitExpr* ast_ArrayDesignatedInitExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* designator, ast_Expr* initValue)
{
   ast_ArrayDesignatedInitExpr* e = ast_context_Context_alloc(c, 32);
   ast_Expr_init(&e->parent, ast_ExprKind_ArrayDesignatedInit, loc, 0, 0, 0, ast_ValType_RValue);
   e->designator = designator;
   e->initValue = initValue;
   ast_Stats_addExpr(ast_ExprKind_ArrayDesignatedInit, 32);
   return e;
}

static ast_Expr* ast_ArrayDesignatedInitExpr_instantiate(ast_ArrayDesignatedInitExpr* e, ast_Instantiator* inst)
{
   ast_ArrayDesignatedInitExpr* f = ast_ArrayDesignatedInitExpr_create(inst->c, e->parent.loc, ast_Expr_instantiate(e->designator, inst), ast_Expr_instantiate(e->initValue, inst));
   return ((ast_Expr*)(f));
}

static ast_Expr* ast_ArrayDesignatedInitExpr_getDesignator(const ast_ArrayDesignatedInitExpr* e)
{
   return e->designator;
}

static ast_Expr** ast_ArrayDesignatedInitExpr_getDesignator2(ast_ArrayDesignatedInitExpr* e)
{
   return &e->designator;
}

static ast_Expr* ast_ArrayDesignatedInitExpr_getInit(const ast_ArrayDesignatedInitExpr* e)
{
   return e->initValue;
}

static ast_Expr** ast_ArrayDesignatedInitExpr_getInit2(ast_ArrayDesignatedInitExpr* e)
{
   return &e->initValue;
}

static void ast_ArrayDesignatedInitExpr_print(const ast_ArrayDesignatedInitExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_newline(out);
   ast_Expr_print(e->designator, out, (indent + 1));
   ast_Expr_print(e->initValue, out, (indent + 1));
}

static ast_ArraySubscriptExpr* ast_ArraySubscriptExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* base, ast_Expr* idx)
{
   ast_ArraySubscriptExpr* e = ast_context_Context_alloc(c, 32);
   ast_Expr_init(&e->parent, ast_ExprKind_ArraySubscript, loc, 0, 0, 0, ast_ValType_LValue);
   e->base = base;
   e->idx = idx;
   ast_Stats_addExpr(ast_ExprKind_ArraySubscript, 32);
   return e;
}

static ast_Expr* ast_ArraySubscriptExpr_instantiate(ast_ArraySubscriptExpr* e, ast_Instantiator* inst)
{
   ast_ArraySubscriptExpr* a = ast_ArraySubscriptExpr_create(inst->c, e->parent.loc, ast_Expr_instantiate(e->base, inst), ast_Expr_instantiate(e->idx, inst));
   return ((ast_Expr*)(a));
}

static ast_Expr* ast_ArraySubscriptExpr_getBase(const ast_ArraySubscriptExpr* e)
{
   return e->base;
}

static ast_Expr** ast_ArraySubscriptExpr_getBase2(ast_ArraySubscriptExpr* e)
{
   return &e->base;
}

static ast_Expr* ast_ArraySubscriptExpr_getIndex(const ast_ArraySubscriptExpr* e)
{
   return e->idx;
}

static ast_Expr** ast_ArraySubscriptExpr_getIndex2(ast_ArraySubscriptExpr* e)
{
   return &e->idx;
}

static void ast_ArraySubscriptExpr_printLiteral(const ast_ArraySubscriptExpr* e, string_buffer_Buf* out)
{
   ast_Expr_printLiteral(e->base, out);
   string_buffer_Buf_add1(out, '[');
   ast_Expr_printLiteral(e->idx, out);
   string_buffer_Buf_add1(out, ']');
}

static void ast_ArraySubscriptExpr_print(const ast_ArraySubscriptExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_newline(out);
   ast_Expr_print(e->base, out, (indent + 1));
   ast_Expr_print(e->idx, out, (indent + 1));
}

static ast_BinaryOperator* ast_BinaryOperator_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_BinaryOpcode kind, ast_Expr* lhs, ast_Expr* rhs)
{
   ast_BinaryOperator* e = ast_context_Context_alloc(c, 32);
   ast_Expr_init(&e->parent, ast_ExprKind_BinaryOperator, loc, 0, 0, (kind >= ast_BinaryOpcode_Assign), ast_ValType_RValue);
   e->parent.parent.binaryOperatorBits.kind = kind;
   e->lhs = lhs;
   e->rhs = rhs;
   ast_Stats_addExpr(ast_ExprKind_BinaryOperator, 32);
   return e;
}

static ast_Expr* ast_BinaryOperator_instantiate(ast_BinaryOperator* e, ast_Instantiator* inst)
{
   return ((ast_Expr*)(ast_BinaryOperator_create(inst->c, e->parent.loc, ast_BinaryOperator_getOpcode(e), ast_Expr_instantiate(e->lhs, inst), ast_Expr_instantiate(e->rhs, inst))));
}

static ast_BinaryOpcode ast_BinaryOperator_getOpcode(const ast_BinaryOperator* e)
{
   return ((ast_BinaryOpcode)(e->parent.parent.binaryOperatorBits.kind));
}

static ast_Expr* ast_BinaryOperator_getLHS(const ast_BinaryOperator* e)
{
   return e->lhs;
}

static ast_Expr** ast_BinaryOperator_getLHS2(ast_BinaryOperator* e)
{
   return &e->lhs;
}

static ast_Expr* ast_BinaryOperator_getRHS(const ast_BinaryOperator* e)
{
   return e->rhs;
}

static ast_Expr** ast_BinaryOperator_getRHS2(ast_BinaryOperator* e)
{
   return &e->rhs;
}

static const char* ast_BinaryOperator_getOpcodeStr(const ast_BinaryOperator* e)
{
   return ast_binaryOpcode_names[ast_BinaryOperator_getOpcode(e)];
}

static void ast_BinaryOperator_print(const ast_BinaryOperator* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_color(out, ast_col_Value);
   string_buffer_Buf_space(out);
   string_buffer_Buf_add(out, ast_binaryOpcode_names[ast_BinaryOperator_getOpcode(e)]);
   string_buffer_Buf_newline(out);
   string_buffer_Buf_indent(out, (indent + 1));
   string_buffer_Buf_color(out, ast_col_Attr);
   string_buffer_Buf_add(out, "LHS=\n");
   ast_Expr_print(e->lhs, out, (indent + 1));
   string_buffer_Buf_indent(out, (indent + 1));
   string_buffer_Buf_color(out, ast_col_Attr);
   string_buffer_Buf_add(out, "RHS=\n");
   ast_Expr_print(e->rhs, out, (indent + 1));
}

static void ast_BinaryOperator_printLiteral(const ast_BinaryOperator* e, string_buffer_Buf* out)
{
   ast_Expr_printLiteral(e->lhs, out);
   string_buffer_Buf_add(out, ast_binaryOpcode_names[ast_BinaryOperator_getOpcode(e)]);
   ast_Expr_printLiteral(e->rhs, out);
}

static ast_BitOffsetExpr* ast_BitOffsetExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* lhs, ast_Expr* rhs)
{
   ast_BitOffsetExpr* e = ast_context_Context_alloc(c, 32);
   ast_Expr_init(&e->parent, ast_ExprKind_BitOffset, loc, 0, 0, 0, ast_ValType_RValue);
   e->lhs = lhs;
   e->rhs = rhs;
   ast_Stats_addExpr(ast_ExprKind_BitOffset, 32);
   return e;
}

static ast_Expr* ast_BitOffsetExpr_instantiate(ast_BitOffsetExpr* e, ast_Instantiator* inst)
{
   ast_BitOffsetExpr* b = ast_BitOffsetExpr_create(inst->c, e->parent.loc, ast_Expr_instantiate(e->lhs, inst), ast_Expr_instantiate(e->rhs, inst));
   return ((ast_Expr*)(b));
}

static ast_Expr* ast_BitOffsetExpr_getLHS(ast_BitOffsetExpr* e)
{
   return e->lhs;
}

static ast_Expr** ast_BitOffsetExpr_getLHS2(ast_BitOffsetExpr* e)
{
   return &e->lhs;
}

static ast_Expr* ast_BitOffsetExpr_getRHS(ast_BitOffsetExpr* e)
{
   return e->rhs;
}

static ast_Expr** ast_BitOffsetExpr_getRHS2(ast_BitOffsetExpr* e)
{
   return &e->rhs;
}

static void ast_BitOffsetExpr_setWidth(ast_BitOffsetExpr* e, uint8_t width)
{
   e->parent.parent.bitOffsetBits.width = width;
}

static uint32_t ast_BitOffsetExpr_getWidth(const ast_BitOffsetExpr* e)
{
   return e->parent.parent.bitOffsetBits.width;
}

static void ast_BitOffsetExpr_printLiteral(const ast_BitOffsetExpr* e, string_buffer_Buf* out)
{
   ast_Expr_printLiteral(e->lhs, out);
   string_buffer_Buf_add1(out, ':');
   ast_Expr_printLiteral(e->rhs, out);
}

static void ast_BitOffsetExpr_print(const ast_BitOffsetExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_space(out);
   string_buffer_Buf_color(out, ast_col_Calc);
   string_buffer_Buf_print(out, "%u", e->parent.parent.bitOffsetBits.width);
   string_buffer_Buf_newline(out);
   ast_Expr_print(e->lhs, out, (indent + 1));
   ast_Expr_print(e->rhs, out, (indent + 1));
}

static ast_BooleanLiteral* ast_BooleanLiteral_create(ast_context_Context* c, src_loc_SrcLoc loc, bool val)
{
   ast_BooleanLiteral* e = ast_context_Context_alloc(c, 16);
   ast_Expr_init(&e->parent, ast_ExprKind_BooleanLiteral, loc, 1, 1, 0, ast_ValType_RValue);
   e->parent.parent.booleanLiteralBits.value = val;
   ast_Expr_setType(&e->parent, ast_builtins[ast_BuiltinKind_Bool]);
   ast_Stats_addExpr(ast_ExprKind_BooleanLiteral, 16);
   return e;
}

static bool ast_BooleanLiteral_getValue(const ast_BooleanLiteral* e)
{
   return e->parent.parent.booleanLiteralBits.value;
}

static void ast_BooleanLiteral_print(const ast_BooleanLiteral* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_space(out);
   string_buffer_Buf_color(out, ast_col_Value);
   string_buffer_Buf_add(out, e->parent.parent.booleanLiteralBits.value ? "true" : "false");
   string_buffer_Buf_newline(out);
}

static void ast_BooleanLiteral_printLiteral(const ast_BooleanLiteral* e, string_buffer_Buf* out)
{
   string_buffer_Buf_add(out, e->parent.parent.booleanLiteralBits.value ? "true" : "false");
}

static ast_BuiltinExpr* ast_BuiltinExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* inner, ast_BuiltinExprKind kind)
{
   const uint32_t size = 40;
   ast_BuiltinExpr* e = ast_context_Context_alloc(c, size);
   ast_Expr_init(&e->parent, ast_ExprKind_Builtin, loc, true, true, false, ast_ValType_RValue);
   e->parent.parent.builtinExprBits.kind = kind;
   e->inner = inner;
   e->value.is_signed = false;
   e->value.uvalue = 0;
   ast_Stats_addExpr(ast_ExprKind_Builtin, size);
   return e;
}

static ast_BuiltinExpr* ast_BuiltinExpr_createOffsetOf(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* typeExpr, ast_Expr* member)
{
   const uint32_t size = (40 + 8);
   ast_BuiltinExpr* e = ast_context_Context_alloc(c, size);
   ast_Expr_init(&e->parent, ast_ExprKind_Builtin, loc, true, true, false, ast_ValType_RValue);
   e->parent.parent.builtinExprBits.kind = ast_BuiltinExprKind_OffsetOf;
   e->inner = typeExpr;
   e->offset[0].member = member;
   ast_Stats_addExpr(ast_ExprKind_Builtin, size);
   return e;
}

static ast_BuiltinExpr* ast_BuiltinExpr_createToContainer(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* typeExpr, ast_Expr* member, ast_Expr* pointer)
{
   const uint32_t size = (40 + 16);
   ast_BuiltinExpr* e = ast_context_Context_alloc(c, size);
   ast_Expr_init(&e->parent, ast_ExprKind_Builtin, loc, false, false, false, ast_ValType_RValue);
   e->parent.parent.builtinExprBits.kind = ast_BuiltinExprKind_ToContainer;
   e->inner = typeExpr;
   e->container[0].member = member;
   e->container[0].pointer = pointer;
   ast_Stats_addExpr(ast_ExprKind_Builtin, size);
   return e;
}

static ast_Expr* ast_BuiltinExpr_instantiate(ast_BuiltinExpr* e, ast_Instantiator* inst)
{
   ast_BuiltinExpr* bi = NULL;
   switch (ast_BuiltinExpr_getKind(e)) {
   case ast_BuiltinExprKind_Sizeof:
      __attribute__((fallthrough));
   case ast_BuiltinExprKind_Elemsof:
      __attribute__((fallthrough));
   case ast_BuiltinExprKind_EnumMin:
      __attribute__((fallthrough));
   case ast_BuiltinExprKind_EnumMax:
      bi = ast_BuiltinExpr_create(inst->c, e->parent.loc, ast_Expr_instantiate(e->inner, inst), ast_BuiltinExpr_getKind(e));
      break;
   case ast_BuiltinExprKind_OffsetOf:
      bi = ast_BuiltinExpr_createOffsetOf(inst->c, e->parent.loc, ast_Expr_instantiate(e->inner, inst), ast_Expr_instantiate(e->offset[0].member, inst));
      break;
   case ast_BuiltinExprKind_ToContainer:
      bi = ast_BuiltinExpr_createToContainer(inst->c, e->parent.loc, ast_Expr_instantiate(e->inner, inst), ast_Expr_instantiate(e->container[0].member, inst), ast_Expr_instantiate(e->container[0].pointer, inst));
      break;
   }
   return ((ast_Expr*)(bi));
}

static ast_BuiltinExprKind ast_BuiltinExpr_getKind(const ast_BuiltinExpr* e)
{
   return ((ast_BuiltinExprKind)(e->parent.parent.builtinExprBits.kind));
}

static ast_Value ast_BuiltinExpr_getValue(const ast_BuiltinExpr* e)
{
   return e->value;
}

static void ast_BuiltinExpr_setValue(ast_BuiltinExpr* e, ast_Value value)
{
   e->value = value;
}

static void ast_BuiltinExpr_setUValue(ast_BuiltinExpr* e, uint64_t val)
{
   e->value.uvalue = val;
}

static ast_Expr* ast_BuiltinExpr_getInner(const ast_BuiltinExpr* e)
{
   return e->inner;
}

static src_loc_SrcLoc ast_BuiltinExpr_getEndLoc(const ast_BuiltinExpr* e)
{
   switch (ast_BuiltinExpr_getKind(e)) {
   case ast_BuiltinExprKind_Sizeof:
      __attribute__((fallthrough));
   case ast_BuiltinExprKind_Elemsof:
      __attribute__((fallthrough));
   case ast_BuiltinExprKind_EnumMin:
      __attribute__((fallthrough));
   case ast_BuiltinExprKind_EnumMax:
      break;
   case ast_BuiltinExprKind_OffsetOf:
      return (ast_Expr_getEndLoc(e->offset[0].member) + 1);
   case ast_BuiltinExprKind_ToContainer:
      return (ast_Expr_getEndLoc(e->container[0].pointer) + 1);
   }
   return ast_Expr_getEndLoc(e->inner);
}

static ast_Expr* ast_BuiltinExpr_getOffsetOfMember(const ast_BuiltinExpr* b)
{
   c2_assert(((ast_BuiltinExpr_getKind(b) == ast_BuiltinExprKind_OffsetOf)) != 0, "ast/builtin_expr.c2:162: ast.BuiltinExpr.getOffsetOfMember", "CALL TODO==BuiltinExprKind.OffsetOf");
   return b->offset[0].member;
}

static ast_Expr* ast_BuiltinExpr_getToContainerMember(const ast_BuiltinExpr* b)
{
   c2_assert(((ast_BuiltinExpr_getKind(b) == ast_BuiltinExprKind_ToContainer)) != 0, "ast/builtin_expr.c2:167: ast.BuiltinExpr.getToContainerMember", "CALL TODO==BuiltinExprKind.ToContainer");
   return b->container[0].member;
}

static ast_Expr* ast_BuiltinExpr_getToContainerPointer(const ast_BuiltinExpr* b)
{
   c2_assert(((ast_BuiltinExpr_getKind(b) == ast_BuiltinExprKind_ToContainer)) != 0, "ast/builtin_expr.c2:172: ast.BuiltinExpr.getToContainerPointer", "CALL TODO==BuiltinExprKind.ToContainer");
   return b->container[0].pointer;
}

static ast_Expr** ast_BuiltinExpr_getToContainerPointer2(ast_BuiltinExpr* b)
{
   c2_assert(((ast_BuiltinExpr_getKind(b) == ast_BuiltinExprKind_ToContainer)) != 0, "ast/builtin_expr.c2:177: ast.BuiltinExpr.getToContainerPointer2", "CALL TODO==BuiltinExprKind.ToContainer");
   return &b->container[0].pointer;
}

static void ast_BuiltinExpr_print(const ast_BuiltinExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_color(out, ast_col_Value);
   string_buffer_Buf_print(out, " %s", ast_builtin_names[ast_BuiltinExpr_getKind(e)]);
   string_buffer_Buf_color(out, ast_col_Calc);
   string_buffer_Buf_print(out, " %s", ast_Value_str(&e->value));
   string_buffer_Buf_newline(out);
   ast_Expr_print(e->inner, out, (indent + 1));
   switch (ast_BuiltinExpr_getKind(e)) {
   case ast_BuiltinExprKind_Sizeof:
      break;
   case ast_BuiltinExprKind_Elemsof:
      break;
   case ast_BuiltinExprKind_EnumMin:
      break;
   case ast_BuiltinExprKind_EnumMax:
      break;
   case ast_BuiltinExprKind_OffsetOf:
      ast_Expr_print(e->offset[0].member, out, (indent + 1));
      break;
   case ast_BuiltinExprKind_ToContainer:
      ast_Expr_print(e->container[0].member, out, (indent + 1));
      ast_Expr_print(e->container[0].pointer, out, (indent + 1));
      break;
   }
}

static void ast_BuiltinExpr_printLiteral(const ast_BuiltinExpr* e, string_buffer_Buf* out)
{
   string_buffer_Buf_add(out, ast_builtin_names[ast_BuiltinExpr_getKind(e)]);
   string_buffer_Buf_lparen(out);
   string_buffer_Buf_rparen(out);
   switch (ast_BuiltinExpr_getKind(e)) {
   case ast_BuiltinExprKind_Sizeof:
      break;
   case ast_BuiltinExprKind_Elemsof:
      break;
   case ast_BuiltinExprKind_EnumMin:
      break;
   case ast_BuiltinExprKind_EnumMax:
      break;
   case ast_BuiltinExprKind_OffsetOf:
      break;
   case ast_BuiltinExprKind_ToContainer:
      break;
   }
}

static ast_CallExpr* ast_CallExpr_create(ast_context_Context* c, src_loc_SrcLoc endLoc, ast_Expr* func, ast_Expr** args, uint32_t num_args)
{
   uint32_t size = (32 + (num_args * 8));
   ast_CallExpr* e = ast_context_Context_alloc(c, size);
   ast_Expr_init(&e->parent, ast_ExprKind_Call, ast_Expr_getLoc(func), 0, 0, 1, ast_ValType_RValue);
   e->endLoc = endLoc;
   e->template_idx = 0;
   e->num_args = ((uint8_t)(num_args));
   e->func = func;
   memcpy(((void*)(e->args)), ((void*)(args)), (num_args * 8));
   ast_Stats_addExpr(ast_ExprKind_Call, size);
   return e;
}

static ast_CallExpr* ast_CallExpr_createTemplate(ast_context_Context* c, src_loc_SrcLoc endLoc, ast_Expr* func, ast_Expr** args, uint32_t num_args, const ast_TypeRefHolder* ref)
{
   uint32_t size = (32 + (num_args * 8));
   size += (8 + ast_TypeRefHolder_getExtraSize(ref));
   ast_CallExpr* e = ast_context_Context_alloc(c, size);
   ast_Expr_init(&e->parent, ast_ExprKind_Call, ast_Expr_getLoc(func), 0, 0, 1, ast_ValType_RValue);
   e->parent.parent.callExprBits.is_template_call = 1;
   e->endLoc = endLoc;
   e->template_idx = 0;
   e->num_args = ((uint8_t)(num_args));
   e->func = func;
   memcpy(((void*)(e->args)), ((void*)(args)), (num_args * 8));
   ast_TypeRef* destRef = ((ast_TypeRef*)(&e->args[num_args]));
   ast_TypeRefHolder_fill(ref, destRef);
   ast_Stats_addExpr(ast_ExprKind_Call, size);
   return e;
}

static ast_Expr* ast_CallExpr_instantiate(ast_CallExpr* e, ast_Instantiator* inst)
{
   c2_assert((!ast_CallExpr_isTemplateCall(e)) != 0, "ast/call_expr.c2:89: ast.CallExpr.instantiate", "!CALL TODO");
   uint32_t size = (32 + (e->num_args * 8));
   ast_CallExpr* e2 = ast_context_Context_alloc(inst->c, size);
   e2->parent = e->parent;
   e2->endLoc = e->endLoc;
   e2->num_args = e->num_args;
   e2->func = ast_Expr_instantiate(e->func, inst);
   for (uint32_t i = 0; (i < e->num_args); i++) {
      e2->args[i] = ast_Expr_instantiate(e->args[i], inst);
   }
   ast_Stats_addExpr(ast_ExprKind_Call, size);
   return ((ast_Expr*)(e2));
}

static void ast_CallExpr_setCallsStructFunc(ast_CallExpr* e)
{
   e->parent.parent.callExprBits.calls_struct_func = 1;
}

static bool ast_CallExpr_isStructFunc(const ast_CallExpr* e)
{
   return e->parent.parent.callExprBits.calls_struct_func;
}

static void ast_CallExpr_setCallsStaticStructFunc(ast_CallExpr* e)
{
   e->parent.parent.callExprBits.calls_static_sf = 1;
}

static bool ast_CallExpr_isStaticStructFunc(const ast_CallExpr* e)
{
   return e->parent.parent.callExprBits.calls_static_sf;
}

static bool ast_CallExpr_isTemplateCall(const ast_CallExpr* e)
{
   return e->parent.parent.callExprBits.is_template_call;
}

static ast_TypeRef* ast_CallExpr_getTemplateArg(const ast_CallExpr* e)
{
   if (ast_CallExpr_isTemplateCall(e)) return ((ast_TypeRef*)(&e->args[e->num_args]));

   return NULL;
}

static void ast_CallExpr_setTemplateIdx(ast_CallExpr* e, uint32_t idx)
{
   e->template_idx = ((uint16_t)(idx));
}

static uint32_t ast_CallExpr_getTemplateIdx(const ast_CallExpr* e)
{
   return e->template_idx;
}

static void ast_CallExpr_setPrintfFormat(ast_CallExpr* e, uint32_t format_idx, bool change_format)
{
   e->parent.parent.callExprBits.printf_format = (format_idx + 1);
   e->parent.parent.callExprBits.change_format = change_format;
}

static bool ast_CallExpr_isPrintfCall(const ast_CallExpr* e)
{
   return (e->parent.parent.callExprBits.printf_format != 0);
}

static uint32_t ast_CallExpr_getPrintfFormat(const ast_CallExpr* e)
{
   return (e->parent.parent.callExprBits.printf_format - 1);
}

static bool ast_CallExpr_needFormatChange(const ast_CallExpr* e)
{
   return e->parent.parent.callExprBits.change_format;
}

static void ast_CallExpr_setHasAutoArgs(ast_CallExpr* e)
{
   e->parent.parent.callExprBits.has_auto_args = 1;
}

static bool ast_CallExpr_hasAutoArgs(const ast_CallExpr* e)
{
   return e->parent.parent.callExprBits.has_auto_args;
}

static src_loc_SrcLoc ast_CallExpr_getEndLoc(const ast_CallExpr* e)
{
   return e->endLoc;
}

static ast_Expr* ast_CallExpr_getFunc(const ast_CallExpr* e)
{
   return e->func;
}

static ast_Expr** ast_CallExpr_getFunc2(ast_CallExpr* e)
{
   return &e->func;
}

static uint32_t ast_CallExpr_getNumArgs(const ast_CallExpr* e)
{
   return e->num_args;
}

static ast_Expr** ast_CallExpr_getArgs(ast_CallExpr* e)
{
   return e->args;
}

static void ast_CallExpr_printLiteral(const ast_CallExpr* e, string_buffer_Buf* out)
{
   string_buffer_Buf_add(out, "CALL TODO");
}

static void ast_CallExpr_print(const ast_CallExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   if (e->parent.parent.callExprBits.calls_struct_func) string_buffer_Buf_add(out, " SF");
   if (e->parent.parent.callExprBits.calls_static_sf) string_buffer_Buf_add(out, " SSF");
   if (ast_CallExpr_isPrintfCall(e)) {
      string_buffer_Buf_print(out, " printf=%u|%u", ast_CallExpr_getPrintfFormat(e), ast_CallExpr_needFormatChange(e));
   }
   if (ast_CallExpr_hasAutoArgs(e)) string_buffer_Buf_add(out, " auto-args");
   string_buffer_Buf_newline(out);
   if (e->parent.parent.callExprBits.is_template_call) {
      string_buffer_Buf_indent(out, (indent + 1));
      string_buffer_Buf_color(out, ast_col_Template);
      string_buffer_Buf_add(out, "template ");
      ast_TypeRef* ref = ((ast_TypeRef*)(&e->args[e->num_args]));
      ast_TypeRef_print(ref, out, true);
      string_buffer_Buf_newline(out);
   }
   ast_Expr_print(e->func, out, (indent + 1));
   for (uint32_t i = 0; (i < ast_CallExpr_getNumArgs(e)); i++) {
      ast_Expr_print(e->args[i], out, (indent + 1));
   }
}

static ast_CharLiteral* ast_CharLiteral_create(ast_context_Context* c, src_loc_SrcLoc loc, uint8_t val, uint8_t radix)
{
   ast_CharLiteral* e = ast_context_Context_alloc(c, 16);
   ast_Expr_init(&e->parent, ast_ExprKind_CharLiteral, loc, 1, 1, 0, ast_ValType_RValue);
   e->parent.parent.charLiteralBits.value = val;
   e->parent.parent.charLiteralBits.radix = radix;
   ast_Stats_addExpr(ast_ExprKind_CharLiteral, 16);
   ast_Expr_setType(&e->parent, ast_builtins[ast_BuiltinKind_Char]);
   return e;
}

static uint8_t ast_CharLiteral_getValue(const ast_CharLiteral* e)
{
   return ((uint8_t)(e->parent.parent.charLiteralBits.value));
}

static void ast_CharLiteral_print(const ast_CharLiteral* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_space(out);
   string_buffer_Buf_color(out, ast_col_Value);
   ast_CharLiteral_printLiteral(e, out);
   string_buffer_Buf_newline(out);
}

static void ast_CharLiteral_printLiteral(const ast_CharLiteral* e, string_buffer_Buf* out)
{
   char c = ((char)(e->parent.parent.charLiteralBits.value));
   switch (e->parent.parent.charLiteralBits.radix) {
   case 8:
      string_buffer_Buf_print(out, "'\\%o'", c);
      return;
   case 16:
      string_buffer_Buf_print(out, "'\\x%x'", c);
      return;
   default:
      break;
   }
   switch (c) {
   case '\a':
      string_buffer_Buf_add(out, "'\\a'");
      break;
   case '\b':
      string_buffer_Buf_add(out, "'\\b'");
      break;
   case '\f':
      string_buffer_Buf_add(out, "'\\f'");
      break;
   case '\n':
      string_buffer_Buf_add(out, "'\\n'");
      break;
   case '\r':
      string_buffer_Buf_add(out, "'\\r'");
      break;
   case '\t':
      string_buffer_Buf_add(out, "'\\t'");
      break;
   case '\v':
      string_buffer_Buf_add(out, "'\\v'");
      break;
   case '\'':
      string_buffer_Buf_add(out, "'\\''");
      break;
   case '\\':
      string_buffer_Buf_add(out, "'\\\\'");
      break;
   default:
      string_buffer_Buf_print(out, "'%c'", c);
      break;
   }
}

static ast_ConditionalOperator* ast_ConditionalOperator_create(ast_context_Context* c, src_loc_SrcLoc questionLoc, src_loc_SrcLoc colonLoc, ast_Expr* cond, ast_Expr* lhs, ast_Expr* rhs)
{
   ast_ConditionalOperator* e = ast_context_Context_alloc(c, 48);
   ast_Expr_init(&e->parent, ast_ExprKind_ConditionalOperator, questionLoc, 0, 1, 1, ast_ValType_RValue);
   e->colonLoc = colonLoc;
   e->cond = cond;
   e->lhs = lhs;
   e->rhs = rhs;
   ast_Stats_addExpr(ast_ExprKind_ConditionalOperator, 48);
   return e;
}

static ast_Expr* ast_ConditionalOperator_instantiate(ast_ConditionalOperator* e, ast_Instantiator* inst)
{
   ast_ConditionalOperator* o = ast_ConditionalOperator_create(inst->c, e->parent.loc, e->colonLoc, ast_Expr_instantiate(e->cond, inst), ast_Expr_instantiate(e->lhs, inst), ast_Expr_instantiate(e->rhs, inst));
   return ((ast_Expr*)(o));
}

static ast_Expr* ast_ConditionalOperator_getCond(const ast_ConditionalOperator* e)
{
   return e->cond;
}

static ast_Expr** ast_ConditionalOperator_getCond2(ast_ConditionalOperator* e)
{
   return &e->cond;
}

static ast_Expr* ast_ConditionalOperator_getLHS(const ast_ConditionalOperator* e)
{
   return e->lhs;
}

static ast_Expr** ast_ConditionalOperator_getLHS2(ast_ConditionalOperator* e)
{
   return &e->lhs;
}

static ast_Expr* ast_ConditionalOperator_getRHS(const ast_ConditionalOperator* e)
{
   return e->rhs;
}

static ast_Expr** ast_ConditionalOperator_getRHS2(ast_ConditionalOperator* e)
{
   return &e->rhs;
}

static void ast_ConditionalOperator_printLiteral(const ast_ConditionalOperator* e, string_buffer_Buf* out)
{
   ast_Expr_printLiteral(e->cond, out);
   string_buffer_Buf_add(out, " ? ");
   ast_Expr_printLiteral(e->lhs, out);
   string_buffer_Buf_add(out, " : ");
   ast_Expr_printLiteral(e->rhs, out);
}

static void ast_ConditionalOperator_print(const ast_ConditionalOperator* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_newline(out);
   ast_Expr_print(e->cond, out, (indent + 1));
   ast_Expr_print(e->lhs, out, (indent + 1));
   ast_Expr_print(e->rhs, out, (indent + 1));
}

static ast_ExplicitCastExpr* ast_ExplicitCastExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, const ast_TypeRefHolder* ref, ast_Expr* inner)
{
   uint32_t size = (32 + ast_TypeRefHolder_getExtraSize(ref));
   ast_ExplicitCastExpr* e = ast_context_Context_alloc(c, size);
   ast_Expr_init(&e->parent, ast_ExprKind_ExplicitCast, loc, 0, 0, 0, ast_ValType_NValue);
   e->inner = inner;
   ast_TypeRefHolder_fill(ref, &e->dest);
   ast_Stats_addExpr(ast_ExprKind_ExplicitCast, size);
   return e;
}

static ast_Expr* ast_ExplicitCastExpr_instantiate(ast_ExplicitCastExpr* e, ast_Instantiator* inst)
{
   bool matches = ast_TypeRef_matchesTemplate(&e->dest, inst->template_name);
   uint32_t extra = matches ? ast_TypeRef_getExtraSize(inst->ref) : ast_TypeRef_getExtraSize(&e->dest);
   uint32_t size = (32 + extra);
   ast_ExplicitCastExpr* e2 = ast_context_Context_alloc(inst->c, size);
   e2->parent = e->parent;
   e2->inner = ast_Expr_instantiate(e->inner, inst);
   ast_TypeRef_instantiate(&e2->dest, &e->dest, inst);
   return ((ast_Expr*)(e2));
}

static ast_Expr* ast_ExplicitCastExpr_getInner(const ast_ExplicitCastExpr* e)
{
   return e->inner;
}

static ast_Expr** ast_ExplicitCastExpr_getInner2(ast_ExplicitCastExpr* e)
{
   return &e->inner;
}

static ast_TypeRef* ast_ExplicitCastExpr_getTypeRef(ast_ExplicitCastExpr* e)
{
   return &e->dest;
}

static void ast_ExplicitCastExpr_printLiteral(const ast_ExplicitCastExpr* e, string_buffer_Buf* out)
{
   string_buffer_Buf_add(out, "cast<");
   ast_TypeRef_print(&e->dest, out, true);
   string_buffer_Buf_add(out, ">(");
   ast_Expr_printLiteral(e->inner, out);
   string_buffer_Buf_add1(out, ')');
}

static void ast_ExplicitCastExpr_print(const ast_ExplicitCastExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_add(out, " -> ");
   ast_TypeRef_print(&e->dest, out, true);
   string_buffer_Buf_newline(out);
   ast_Expr_print(e->inner, out, (indent + 1));
}

static ast_FieldDesignatedInitExpr* ast_FieldDesignatedInitExpr_create(ast_context_Context* c, uint32_t field, src_loc_SrcLoc loc, ast_Expr* initValue)
{
   ast_FieldDesignatedInitExpr* e = ast_context_Context_alloc(c, 40);
   ast_Expr_init(&e->parent, ast_ExprKind_FieldDesignatedInit, loc, 0, 0, 0, ast_ValType_RValue);
   e->field = field;
   e->initValue = initValue;
   e->decl = NULL;
   ast_Stats_addExpr(ast_ExprKind_FieldDesignatedInit, 40);
   return e;
}

static ast_Expr* ast_FieldDesignatedInitExpr_instantiate(ast_FieldDesignatedInitExpr* e, ast_Instantiator* inst)
{
   return ((ast_Expr*)(ast_FieldDesignatedInitExpr_create(inst->c, e->field, e->parent.loc, ast_Expr_instantiate(e->initValue, inst))));
}

static uint32_t ast_FieldDesignatedInitExpr_getField(const ast_FieldDesignatedInitExpr* e)
{
   return e->field;
}

static const char* ast_FieldDesignatedInitExpr_getFieldName(const ast_FieldDesignatedInitExpr* e)
{
   return ast_idx2name(e->field);
}

static ast_Expr* ast_FieldDesignatedInitExpr_getInit(const ast_FieldDesignatedInitExpr* e)
{
   return e->initValue;
}

static ast_Expr** ast_FieldDesignatedInitExpr_getInit2(ast_FieldDesignatedInitExpr* e)
{
   return &e->initValue;
}

static void ast_FieldDesignatedInitExpr_setDecl(ast_FieldDesignatedInitExpr* e, ast_Decl* d)
{
   e->decl = d;
}

static ast_Decl* ast_FieldDesignatedInitExpr_getDecl(const ast_FieldDesignatedInitExpr* e)
{
   return e->decl;
}

static void ast_FieldDesignatedInitExpr_print(const ast_FieldDesignatedInitExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_color(out, ast_col_Value);
   string_buffer_Buf_print(out, " %s", ast_idx2name(e->field));
   if (!e->decl) {
      string_buffer_Buf_color(out, ast_col_Error);
      string_buffer_Buf_add(out, " ??");
   }
   string_buffer_Buf_newline(out);
   ast_Expr_print(e->initValue, out, (indent + 1));
}

static ast_FloatLiteral* ast_FloatLiteral_create(ast_context_Context* c, src_loc_SrcLoc loc, double val)
{
   ast_FloatLiteral* i = ast_context_Context_alloc(c, 24);
   ast_Expr_init(&i->parent, ast_ExprKind_FloatLiteral, loc, 1, 1, 0, ast_ValType_RValue);
   i->val = val;
   ast_Stats_addExpr(ast_ExprKind_FloatLiteral, 24);
   ast_Expr_setType(&i->parent, ast_builtins[ast_BuiltinKind_Float32]);
   return i;
}

static double ast_FloatLiteral_getValue(const ast_FloatLiteral* e)
{
   return e->val;
}

static void ast_FloatLiteral_print(const ast_FloatLiteral* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_color(out, ast_col_Value);
   string_buffer_Buf_space(out);
   ast_FloatLiteral_printLiteral(e, out);
   string_buffer_Buf_newline(out);
}

static void ast_FloatLiteral_printLiteral(const ast_FloatLiteral* e, string_buffer_Buf* out)
{
   string_buffer_Buf_print(out, "%lf", e->val);
}

static ast_IdentifierExpr* ast_IdentifierExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, uint32_t name)
{
   ast_IdentifierExpr* e = ast_context_Context_alloc(c, 24);
   ast_Expr_init(&e->parent, ast_ExprKind_Identifier, loc, 0, 0, 0, ast_ValType_NValue);
   e->name_idx = name;
   ast_Stats_addExpr(ast_ExprKind_Identifier, 24);
   return e;
}

static ast_Expr* ast_IdentifierExpr_instantiate(ast_IdentifierExpr* e, ast_Instantiator* inst)
{
   return ((ast_Expr*)(ast_IdentifierExpr_create(inst->c, e->parent.loc, e->name_idx)));
}

static uint32_t ast_IdentifierExpr_getSize(void)
{
   return 24;
}

static ast_Expr* ast_IdentifierExpr_asExpr(ast_IdentifierExpr* e)
{
   return &e->parent;
}

static void ast_IdentifierExpr_setDecl(ast_IdentifierExpr* e, ast_Decl* decl)
{
   e->decl = decl;
   e->parent.parent.identifierExprBits.has_decl = true;
}

static ast_Decl* ast_IdentifierExpr_getDecl(const ast_IdentifierExpr* e)
{
   if (!e->parent.parent.identifierExprBits.has_decl) return NULL;

   return e->decl;
}

static ast_Ref ast_IdentifierExpr_getRef(const ast_IdentifierExpr* e)
{
   ast_Ref ref = { e->parent.loc, ast_IdentifierExpr_getNameIdx(e), ast_IdentifierExpr_getDecl(e) };
   return ref;
}

static void ast_IdentifierExpr_setKind(ast_IdentifierExpr* e, ast_IdentifierKind kind)
{
   e->parent.parent.identifierExprBits.kind = kind;
}

static ast_IdentifierKind ast_IdentifierExpr_getKind(const ast_IdentifierExpr* e)
{
   return ((ast_IdentifierKind)(e->parent.parent.identifierExprBits.kind));
}

static const char* ast_IdentifierExpr_getName(const ast_IdentifierExpr* e)
{
   if (e->parent.parent.identifierExprBits.has_decl) return ast_Decl_getName(e->decl);

   return ast_idx2name(e->name_idx);
}

static void ast_IdentifierExpr_setCaseRange(ast_IdentifierExpr* e)
{
   e->parent.parent.identifierExprBits.is_case_range = 1;
}

static bool ast_IdentifierExpr_isCaseRange(const ast_IdentifierExpr* e)
{
   return e->parent.parent.identifierExprBits.is_case_range;
}

static uint32_t ast_IdentifierExpr_getNameIdx(const ast_IdentifierExpr* e)
{
   if (e->parent.parent.identifierExprBits.has_decl) return ast_Decl_getNameIdx(e->decl);

   return e->name_idx;
}

static void ast_IdentifierExpr_print(const ast_IdentifierExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   if (ast_IdentifierExpr_isCaseRange(e)) string_buffer_Buf_add(out, " range");
   string_buffer_Buf_space(out);
   ast_IdentifierKind kind = ast_IdentifierExpr_getKind(e);
   if ((kind == ast_IdentifierKind_Unresolved)) string_buffer_Buf_color(out, ast_col_Error);
   else string_buffer_Buf_color(out, ast_col_Attr);
   string_buffer_Buf_add(out, ast_identifierKind_names[kind]);
   string_buffer_Buf_space(out);
   if (e->parent.parent.identifierExprBits.has_decl) {
      string_buffer_Buf_color(out, ast_col_Value);
      string_buffer_Buf_add(out, ast_Decl_getName(e->decl));
   } else {
      string_buffer_Buf_color(out, ast_col_Value);
      string_buffer_Buf_add(out, ast_idx2name(e->name_idx));
   }
   string_buffer_Buf_newline(out);
}

static void ast_IdentifierExpr_printLiteral(const ast_IdentifierExpr* e, string_buffer_Buf* out)
{
   string_buffer_Buf_add(out, ast_IdentifierExpr_getName(e));
}

static ast_ImplicitCastExpr* ast_ImplicitCastExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_ImplicitCastKind kind, ast_Expr* inner)
{
   ast_ImplicitCastExpr* e = ast_context_Context_alloc(c, 24);
   ast_Expr_init(&e->parent, ast_ExprKind_ImplicitCast, loc, 0, 0, 0, ast_ValType_RValue);
   e->parent.parent.implicitCastBits.kind = kind;
   e->inner = inner;
   ast_Expr_copyConstantFlags(&e->parent, inner);
   switch (kind) {
   case ast_ImplicitCastKind_ArrayToPointerDecay:
      ast_Expr_copyValType(&e->parent, inner);
      e->parent.parent.exprBits.is_ctv = false;
      break;
   case ast_ImplicitCastKind_FunctionToPointerDecay:
      ast_Expr_copyValType(&e->parent, inner);
      break;
   case ast_ImplicitCastKind_LValueToRValue:
      e->parent.parent.exprBits.is_ctc = false;
      break;
   case ast_ImplicitCastKind_PointerToBoolean:
      ast_Expr_copyValType(&e->parent, inner);
      break;
   case ast_ImplicitCastKind_PointerToInteger:
      ast_Expr_copyValType(&e->parent, inner);
      break;
   case ast_ImplicitCastKind_IntegralCast:
      ast_Expr_copyValType(&e->parent, inner);
      break;
   case ast_ImplicitCastKind_BitCast:
      e->parent.parent.exprBits.is_ctc = false;
      break;
   }
   ast_Stats_addExpr(ast_ExprKind_ImplicitCast, 24);
   return e;
}

static ast_ImplicitCastKind ast_ImplicitCastExpr_getKind(const ast_ImplicitCastExpr* e)
{
   return ((ast_ImplicitCastKind)(e->parent.parent.implicitCastBits.kind));
}

static bool ast_ImplicitCastExpr_isArrayToPointerDecay(const ast_ImplicitCastExpr* e)
{
   return (ast_ImplicitCastExpr_getKind(e) == ast_ImplicitCastKind_ArrayToPointerDecay);
}

static ast_Expr* ast_ImplicitCastExpr_getInner(const ast_ImplicitCastExpr* e)
{
   return e->inner;
}

static void ast_ImplicitCastExpr_printLiteral(const ast_ImplicitCastExpr* e, string_buffer_Buf* out)
{
   ast_Expr_printLiteral(e->inner, out);
}

static void ast_ImplicitCastExpr_print(const ast_ImplicitCastExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_space(out);
   string_buffer_Buf_color(out, ast_col_Calc);
   string_buffer_Buf_add(out, ast_implicitCastKind_names[ast_ImplicitCastExpr_getKind(e)]);
   string_buffer_Buf_newline(out);
   ast_Expr_print(e->inner, out, (indent + 1));
}

static ast_InitListExpr* ast_InitListExpr_create(ast_context_Context* c, src_loc_SrcLoc left, src_loc_SrcLoc right, ast_Expr** values, uint32_t num_values)
{
   uint32_t size = (24 + (num_values * 8));
   ast_InitListExpr* e = ast_context_Context_alloc(c, size);
   ast_Expr_init(&e->parent, ast_ExprKind_InitList, left, 0, 0, 0, ast_ValType_RValue);
   e->parent.parent.initListExprBits.num_values = num_values;
   e->right = right;
   if (num_values) {
      memcpy(((void*)(e->values)), ((void*)(values)), (num_values * 8));
   }
   ast_Stats_addExpr(ast_ExprKind_InitList, size);
   return e;
}

static ast_Expr* ast_InitListExpr_instantiate(ast_InitListExpr* e, ast_Instantiator* inst)
{
   uint32_t num_values = ast_InitListExpr_getNumValues(e);
   uint32_t size = (24 + (num_values * 8));
   ast_InitListExpr* e2 = ast_context_Context_alloc(inst->c, size);
   e2->parent = e->parent;
   e2->right = e->right;
   for (uint32_t i = 0; (i < num_values); i++) {
      e2->values[i] = ast_Expr_instantiate(e->values[i], inst);
   }
   ast_Stats_addExpr(ast_ExprKind_InitList, size);
   return ((ast_Expr*)(e2));
}

static uint32_t ast_InitListExpr_getNumValues(const ast_InitListExpr* e)
{
   return e->parent.parent.initListExprBits.num_values;
}

static ast_Expr** ast_InitListExpr_getValues(ast_InitListExpr* e)
{
   return e->values;
}

static void ast_InitListExpr_print(const ast_InitListExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_newline(out);
   for (uint32_t i = 0; (i < ast_InitListExpr_getNumValues(e)); i++) {
      ast_Expr_print(e->values[i], out, (indent + 1));
   }
}

static ast_IntegerLiteral* ast_IntegerLiteral_create(ast_context_Context* c, src_loc_SrcLoc loc, uint8_t radix, uint64_t val)
{
   ast_IntegerLiteral* i = ast_context_Context_alloc(c, 24);
   ast_Expr_init(&i->parent, ast_ExprKind_IntegerLiteral, loc, 1, 1, 0, ast_ValType_RValue);
   i->parent.parent.integerLiteralBits.radix = radix;
   i->val = val;
   ast_Stats_addExpr(ast_ExprKind_IntegerLiteral, 24);
   ast_Expr_setType(&i->parent, ast_builtins[ast_BuiltinKind_Int32]);
   return i;
}

static ast_IntegerLiteral* ast_IntegerLiteral_createUnsignedConstant(ast_context_Context* c, src_loc_SrcLoc loc, uint64_t val, ast_QualType qt)
{
   ast_IntegerLiteral* i = ast_IntegerLiteral_create(c, loc, 10, val);
   ast_Expr_setCtv(&i->parent);
   ast_Expr_setCtc(&i->parent);
   ast_Expr_setType(&i->parent, qt);
   return i;
}

static ast_IntegerLiteral* ast_IntegerLiteral_createSignedConstant(ast_context_Context* c, src_loc_SrcLoc loc, int64_t val, ast_QualType qt)
{
   ast_IntegerLiteral* i = ast_IntegerLiteral_create(c, loc, 10, ((uint64_t)(val)));
   i->parent.parent.integerLiteralBits.is_signed = 1;
   ast_Expr_setCtv(&i->parent);
   ast_Expr_setCtc(&i->parent);
   ast_Expr_setType(&i->parent, qt);
   return i;
}

static uint64_t ast_IntegerLiteral_getValue(const ast_IntegerLiteral* e)
{
   return e->val;
}

static bool ast_IntegerLiteral_isDecimal(const ast_IntegerLiteral* e)
{
   return (e->parent.parent.integerLiteralBits.radix == 10);
}

static bool ast_IntegerLiteral_isSigned(const ast_IntegerLiteral* e)
{
   return e->parent.parent.integerLiteralBits.is_signed;
}

static void ast_printBinary(string_buffer_Buf* out, uint64_t value)
{
   char tmp[64];
   tmp[63] = 0;
   char* cp = &tmp[62];
   while (value) {
      *cp = ('0' + ((value & 0x1)));
      cp--;
      value /= 2;
   }
   *cp-- = 'b';
   *cp = '0';
   string_buffer_Buf_add(out, cp);
}

static void ast_printOctal(string_buffer_Buf* out, uint64_t value)
{
   char tmp[32];
   tmp[31] = 0;
   char* cp = &tmp[30];
   while (value) {
      *cp = ('0' + ((value & 0x7)));
      cp--;
      value /= 8;
   }
   *cp = '0';
   string_buffer_Buf_add(out, cp);
}

static void ast_IntegerLiteral_print(const ast_IntegerLiteral* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_color(out, ast_col_Value);
   string_buffer_Buf_space(out);
   ast_IntegerLiteral_printLiteral(e, out);
   string_buffer_Buf_newline(out);
}

static void ast_IntegerLiteral_printLiteral(const ast_IntegerLiteral* e, string_buffer_Buf* out)
{
   switch (e->parent.parent.integerLiteralBits.radix) {
   case 2:
      ast_printBinary(out, e->val);
      break;
   case 8:
      ast_printOctal(out, e->val);
      break;
   case 10:
      if (e->parent.parent.integerLiteralBits.is_signed) {
         string_buffer_Buf_print(out, "%ld", ((int64_t)(e->val)));
      } else {
         string_buffer_Buf_print(out, "%lu", e->val);
      }
      break;
   case 16:
      string_buffer_Buf_print(out, "0x%lx", e->val);
      break;
   }
}

static void ast_IntegerLiteral_printDecimal(const ast_IntegerLiteral* e, string_buffer_Buf* out)
{
   if (e->parent.parent.integerLiteralBits.is_signed) {
      string_buffer_Buf_print(out, "%ld", ((int64_t)(e->val)));
   } else {
      string_buffer_Buf_print(out, "%lu", e->val);
   }
}

static ast_MemberExpr* ast_MemberExpr_create(ast_context_Context* c, ast_Expr* base, const ast_Ref* refs, uint32_t refcount)
{
   uint32_t size = ((16 + (refcount * 8)) + (((refcount - 1)) * 4));
   if (base) size += 8;
   size = (((size + 7)) & ~0x7);
   ast_MemberExpr* e = ast_context_Context_alloc(c, size);
   ast_Expr_init(&e->parent, ast_ExprKind_Member, refs[0].loc, 0, 0, 0, ast_ValType_NValue);
   e->parent.parent.memberExprBits.num_refs = refcount;
   uint32_t offset = 0;
   if (base) {
      offset = 1;
      e->refs[0].expr = base;
      e->parent.parent.memberExprBits.has_expr = 1;
   }
   for (uint32_t i = 0; (i < refcount); i++) {
      e->refs[(i + offset)].name_idx = refs[i].name_idx;
   }
   src_loc_SrcLoc* locs = ((src_loc_SrcLoc*)(&e->refs[(refcount + offset)]));
   for (uint32_t i = 0; (i < (refcount - 1)); i++) {
      locs[i] = refs[(i + 1)].loc;
   }
   ast_Stats_addExpr(ast_ExprKind_Member, size);
   return e;
}

static ast_Expr* ast_MemberExpr_instantiate(ast_MemberExpr* e, ast_Instantiator* inst)
{
   uint32_t refcount = e->parent.parent.memberExprBits.num_refs;
   ast_Expr* base = ast_MemberExpr_getExprBase(e);
   uint32_t size = ((16 + (refcount * 8)) + (((refcount - 1)) * 4));
   if (base) size += 8;
   size = (((size + 7)) & ~0x7);
   ast_MemberExpr* e2 = ast_context_Context_alloc(inst->c, size);
   memcpy(e2, e, size);
   if (base) e2->refs[0].expr = ast_Expr_instantiate(base, inst);
   return ((ast_Expr*)(e2));
}

static bool ast_MemberExpr_hasExpr(const ast_MemberExpr* e)
{
   return e->parent.parent.memberExprBits.has_expr;
}

static ast_Expr* ast_MemberExpr_getExprBase(const ast_MemberExpr* e)
{
   if (ast_MemberExpr_hasExpr(e)) return e->refs[0].expr;

   return NULL;
}

static const char* ast_MemberExpr_getName(const ast_MemberExpr* e, uint32_t ref_idx)
{
   const ast_MemberRef* ref = &e->refs[(ref_idx + ast_MemberExpr_hasExpr(e))];
   if ((e->parent.parent.memberExprBits.num_decls > ref_idx)) {
      return ast_Decl_getName(ref->decl);
   }
   return ast_idx2name(ref->name_idx);
}

static uint32_t ast_MemberExpr_getNumRefs(const ast_MemberExpr* e)
{
   return e->parent.parent.memberExprBits.num_refs;
}

static uint32_t ast_MemberExpr_getNameIdx(const ast_MemberExpr* e, uint32_t ref_idx)
{
   const ast_MemberRef* ref = &e->refs[(ref_idx + ast_MemberExpr_hasExpr(e))];
   if ((e->parent.parent.memberExprBits.num_decls > ref_idx)) {
      ast_Decl* d = ref->decl;
      if (ast_Decl_isImport(d)) {
         const ast_ImportDecl* id = ((ast_ImportDecl*)(d));
         uint32_t alias_idx = ast_ImportDecl_getAliasNameIdx(id);
         if (alias_idx) return alias_idx;

      }
      return ast_Decl_getNameIdx(d);
   }
   return ref->name_idx;
}

static src_loc_SrcLoc ast_MemberExpr_getLoc(const ast_MemberExpr* e, uint32_t ref_idx)
{
   if ((ref_idx == 0)) return ast_Expr_getLoc(&e->parent);

   src_loc_SrcLoc* locs = ((src_loc_SrcLoc*)(&e->refs[(ast_MemberExpr_getNumRefs(e) + ast_MemberExpr_hasExpr(e))]));
   return locs[(ref_idx - 1)];
}

static ast_Ref ast_MemberExpr_getRef(const ast_MemberExpr* e, uint32_t ref_idx)
{
   ast_Ref ref;
   ref.loc = ast_MemberExpr_getLoc(e, ref_idx);
   ref.name_idx = ast_MemberExpr_getNameIdx(e, ref_idx);
   ref.decl = ast_MemberExpr_getDecl(e, ref_idx);
   return ref;
}

static ast_IdentifierKind ast_MemberExpr_getKind(const ast_MemberExpr* e)
{
   return ((ast_IdentifierKind)(e->parent.parent.memberExprBits.kind));
}

static void ast_MemberExpr_setKind(ast_MemberExpr* e, ast_IdentifierKind kind)
{
   e->parent.parent.memberExprBits.kind = kind;
}

static bool ast_MemberExpr_isVarKind(const ast_MemberExpr* e)
{
   return (ast_MemberExpr_getKind(e) == ast_IdentifierKind_Var);
}

static void ast_MemberExpr_setIsStructFunc(ast_MemberExpr* e)
{
   e->parent.parent.memberExprBits.is_struct_func = 1;
}

static bool ast_MemberExpr_isStructFunc(const ast_MemberExpr* e)
{
   return e->parent.parent.memberExprBits.is_struct_func;
}

static void ast_MemberExpr_setIsStaticStructFunc(ast_MemberExpr* e)
{
   e->parent.parent.memberExprBits.is_static_sf = 1;
}

static bool ast_MemberExpr_isStaticStructFunc(const ast_MemberExpr* e)
{
   return e->parent.parent.memberExprBits.is_static_sf;
}

static void ast_MemberExpr_setConstBase(ast_MemberExpr* e, bool b)
{
   e->parent.parent.memberExprBits.is_const_base = b;
}

static bool ast_MemberExpr_isConstBase(const ast_MemberExpr* e)
{
   return e->parent.parent.memberExprBits.is_const_base;
}

static ast_Decl* ast_MemberExpr_getPrevLastDecl(const ast_MemberExpr* e)
{
   uint32_t num = ast_MemberExpr_getNumRefs(e);
   if ((e->parent.parent.memberExprBits.num_decls < num)) return NULL;

   num += ast_MemberExpr_hasExpr(e);
   return e->refs[(num - 2)].decl;
}

static ast_Decl* ast_MemberExpr_getFullDecl(const ast_MemberExpr* e)
{
   uint32_t num = ast_MemberExpr_getNumRefs(e);
   if ((e->parent.parent.memberExprBits.num_decls < num)) return NULL;

   num += ast_MemberExpr_hasExpr(e);
   return e->refs[(num - 1)].decl;
}

static ast_Decl* ast_MemberExpr_getDecl(const ast_MemberExpr* e, uint32_t ref_idx)
{
   if ((e->parent.parent.memberExprBits.num_decls <= ref_idx)) return NULL;

   return e->refs[(ref_idx + ast_MemberExpr_hasExpr(e))].decl;
}

static void ast_MemberExpr_setDecl(ast_MemberExpr* e, ast_Decl* d, uint32_t ref_idx)
{
   e->parent.parent.memberExprBits.num_decls = (ref_idx + 1);
   e->refs[(ref_idx + ast_MemberExpr_hasExpr(e))].decl = d;
}

static src_loc_SrcLoc ast_MemberExpr_getStartLoc(const ast_MemberExpr* e)
{
   if (ast_MemberExpr_hasExpr(e)) return ast_Expr_getStartLoc(e->refs[0].expr);

   return ast_Expr_getLoc(&e->parent);
}

static src_loc_SrcLoc ast_MemberExpr_getEndLoc(const ast_MemberExpr* e)
{
   return ast_MemberExpr_getLoc(e, (ast_MemberExpr_getNumRefs(e) - 1));
}

static ast_QualType ast_MemberExpr_getBaseType(const ast_MemberExpr* m)
{
   uint32_t numRefs = ast_MemberExpr_getNumRefs(m);
   ast_QualType qt;
   if ((ast_MemberExpr_hasExpr(m) && (numRefs == 1))) {
      qt = ast_Expr_getType(m->refs[0].expr);
   } else {
      qt = ast_Decl_getType(m->refs[((ast_MemberExpr_hasExpr(m) + numRefs) - 2)].decl);
   }
   if (ast_MemberExpr_isConstBase(m)) ast_QualType_setConst(&qt);
   return qt;
}

static const char* ast_MemberExpr_getLastMemberName(const ast_MemberExpr* e)
{
   return ast_MemberExpr_getName(e, (ast_MemberExpr_getNumRefs(e) - 1));
}

static void ast_MemberExpr_print(const ast_MemberExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_space(out);
   ast_IdentifierKind kind = ast_MemberExpr_getKind(e);
   if ((kind == ast_IdentifierKind_Unresolved)) string_buffer_Buf_color(out, ast_col_Error);
   else string_buffer_Buf_color(out, ast_col_Attr);
   string_buffer_Buf_add(out, ast_identifierKind_names[kind]);
   string_buffer_Buf_color(out, ast_col_Attr);
   if (ast_MemberExpr_isStructFunc(e)) string_buffer_Buf_add(out, " SF");
   if (ast_MemberExpr_isStaticStructFunc(e)) string_buffer_Buf_add(out, " SSF");
   if (ast_MemberExpr_isConstBase(e)) string_buffer_Buf_add(out, " const-base");
   string_buffer_Buf_print(out, " refs=%u/%u ", e->parent.parent.memberExprBits.num_decls, ast_MemberExpr_getNumRefs(e));
   string_buffer_Buf_color(out, ast_col_Value);
   ast_MemberExpr_printLiteral(e, out);
   string_buffer_Buf_newline(out);
   if (ast_MemberExpr_hasExpr(e)) ast_Expr_print(e->refs[0].expr, out, (indent + 1));
}

static void ast_MemberExpr_printLiteral(const ast_MemberExpr* e, string_buffer_Buf* out)
{
   if (ast_MemberExpr_hasExpr(e)) {
      string_buffer_Buf_add(out, "<expr>.");
   }
   for (uint32_t i = 0; (i < ast_MemberExpr_getNumRefs(e)); i++) {
      if ((i != 0)) string_buffer_Buf_add(out, ".");
      string_buffer_Buf_add(out, ast_MemberExpr_getName(e, i));
   }
}

static void ast_MemberExpr_dump(const ast_MemberExpr* m)
{
   string_buffer_Buf* out = string_buffer_create((10 * 4096), ast_useColor(), 2);
   string_buffer_Buf_color(out, ast_col_Expr);
   string_buffer_Buf_print(out, "MemberExpr expr %u ref %u/%u\n", ast_MemberExpr_hasExpr(m), m->parent.parent.memberExprBits.num_decls, ast_MemberExpr_getNumRefs(m));
   if (ast_MemberExpr_hasExpr(m)) {
      string_buffer_Buf_indent(out, 1);
      string_buffer_Buf_color(out, ast_col_Value);
      string_buffer_Buf_print(out, "<expr>\n");
      ast_Expr* e = ast_MemberExpr_getExprBase(m);
      ast_Expr_print(e, out, 1);
   }
   for (uint32_t i = 0; (i < ast_MemberExpr_getNumRefs(m)); i++) {
      const ast_MemberRef* ref = &m->refs[(i + ast_MemberExpr_hasExpr(m))];
      string_buffer_Buf_indent(out, 1);
      if ((m->parent.parent.memberExprBits.num_decls > i)) {
         string_buffer_Buf_print(out, "[%u]\n", i);
         ast_Decl_print(ref->decl, out, 1);
      } else {
         string_buffer_Buf_print(out, "[%u] %s\n", i, ast_idx2name(ref->name_idx));
      }
   }
   string_buffer_Buf_color(out, ast_col_Normal);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static ast_NilExpr* ast_NilExpr_create(ast_context_Context* c, src_loc_SrcLoc loc)
{
   ast_NilExpr* e = ast_context_Context_alloc(c, 16);
   ast_Expr_init(&e->parent, ast_ExprKind_Nil, loc, 1, 1, 0, ast_ValType_RValue);
   ast_Expr_setType(&e->parent, ast_getVoidPtr());
   ast_Stats_addExpr(ast_ExprKind_Nil, 16);
   return e;
}

static void ast_NilExpr_print(const ast_NilExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_newline(out);
}

static void ast_NilExpr_printLiteral(const ast_NilExpr* _arg0, string_buffer_Buf* out)
{
   string_buffer_Buf_add(out, "nil");
}

static ast_ParenExpr* ast_ParenExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_Expr* inner)
{
   ast_ParenExpr* e = ast_context_Context_alloc(c, 24);
   ast_Expr_init(&e->parent, ast_ExprKind_Paren, loc, 0, 0, 0, ast_ValType_NValue);
   e->inner = inner;
   ast_Stats_addExpr(ast_ExprKind_Paren, 24);
   return e;
}

static ast_Expr* ast_ParenExpr_instantiate(ast_ParenExpr* e, ast_Instantiator* inst)
{
   return ((ast_Expr*)(ast_ParenExpr_create(inst->c, e->parent.loc, ast_Expr_instantiate(e->inner, inst))));
}

static ast_Expr* ast_ParenExpr_getInner(const ast_ParenExpr* e)
{
   return e->inner;
}

static ast_Expr** ast_ParenExpr_getInner2(ast_ParenExpr* e)
{
   return &e->inner;
}

static void ast_ParenExpr_print(const ast_ParenExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_newline(out);
   ast_Expr_print(e->inner, out, (indent + 1));
}

static void ast_ParenExpr_printLiteral(const ast_ParenExpr* e, string_buffer_Buf* out)
{
   string_buffer_Buf_lparen(out);
   ast_Expr_printLiteral(e->inner, out);
   string_buffer_Buf_rparen(out);
}

static ast_StringLiteral* ast_StringLiteral_create(ast_context_Context* c, src_loc_SrcLoc loc, uint32_t value, uint32_t len)
{
   ast_StringLiteral* e = ast_context_Context_alloc(c, 24);
   ast_Expr_init(&e->parent, ast_ExprKind_StringLiteral, loc, 0, 1, 0, ast_ValType_LValue);
   e->value = value;
   ast_Stats_addExpr(ast_ExprKind_StringLiteral, 24);
   ast_Expr_setType(&e->parent, ast_getStringType(len));
   return e;
}

static const char* ast_StringLiteral_getText(const ast_StringLiteral* e)
{
   return ast_idx2name(e->value);
}

static void ast_StringLiteral_printLiteral(const ast_StringLiteral* e, string_buffer_Buf* out)
{
   string_buffer_Buf_print(out, "\"%s\"", ast_idx2name(e->value));
}

static void ast_StringLiteral_print(const ast_StringLiteral* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_space(out);
   string_buffer_Buf_color(out, ast_col_Value);
   ast_StringLiteral_printLiteral(e, out);
   string_buffer_Buf_newline(out);
}

static ast_TypeExpr* ast_TypeExpr_create(ast_context_Context* c, src_loc_SrcLoc loc, const ast_TypeRefHolder* ref)
{
   uint32_t size = (24 + ast_TypeRefHolder_getExtraSize(ref));
   ast_TypeExpr* e = ast_context_Context_alloc(c, size);
   ast_Expr_init(&e->parent, ast_ExprKind_Type, loc, 0, 0, 0, ast_ValType_NValue);
   ast_TypeRefHolder_fill(ref, &e->typeRef);
   ast_Stats_addExpr(ast_ExprKind_Type, size);
   return e;
}

static ast_Expr* ast_TypeExpr_instantiate(ast_TypeExpr* e, ast_Instantiator* inst)
{
   bool matches = ast_TypeRef_matchesTemplate(&e->typeRef, inst->template_name);
   uint32_t extra = matches ? ast_TypeRef_getExtraSize(inst->ref) : ast_TypeRef_getExtraSize(&e->typeRef);
   uint32_t size = (24 + extra);
   ast_TypeExpr* e2 = ast_context_Context_alloc(inst->c, size);
   e2->parent = e->parent;
   ast_TypeRef_instantiate(&e2->typeRef, &e->typeRef, inst);
   ast_Stats_addExpr(ast_ExprKind_Type, size);
   return ((ast_Expr*)(e2));
}

static ast_TypeRef* ast_TypeExpr_getTypeRef(ast_TypeExpr* e)
{
   return &e->typeRef;
}

static void ast_TypeExpr_print(const ast_TypeExpr* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_space(out);
   ast_TypeRef_print(&e->typeRef, out, true);
   string_buffer_Buf_newline(out);
}

static ast_UnaryOperator* ast_UnaryOperator_create(ast_context_Context* c, src_loc_SrcLoc loc, ast_UnaryOpcode kind, ast_Expr* inner)
{
   ast_UnaryOperator* e = ast_context_Context_alloc(c, 24);
   ast_Expr_init(&e->parent, ast_ExprKind_UnaryOperator, loc, 0, 0, (kind <= ast_UnaryOpcode_PreDec), ast_ValType_RValue);
   e->parent.parent.unaryOperatorBits.kind = kind;
   e->inner = inner;
   ast_Stats_addExpr(ast_ExprKind_UnaryOperator, 24);
   return e;
}

static ast_Expr* ast_UnaryOperator_instantiate(ast_UnaryOperator* e, ast_Instantiator* inst)
{
   return ((ast_Expr*)(ast_UnaryOperator_create(inst->c, e->parent.loc, ast_UnaryOperator_getOpcode(e), ast_Expr_instantiate(e->inner, inst))));
}

static ast_UnaryOpcode ast_UnaryOperator_getOpcode(const ast_UnaryOperator* e)
{
   return ((ast_UnaryOpcode)(e->parent.parent.unaryOperatorBits.kind));
}

static ast_Expr* ast_UnaryOperator_getInner(const ast_UnaryOperator* e)
{
   return e->inner;
}

static ast_Expr** ast_UnaryOperator_getInner2(ast_UnaryOperator* e)
{
   return &e->inner;
}

static ast_Expr* ast_UnaryOperator_asExpr(ast_UnaryOperator* e)
{
   return &e->parent;
}

static bool ast_UnaryOperator_isBefore(const ast_UnaryOperator* e)
{
   switch (ast_UnaryOperator_getOpcode(e)) {
   case ast_UnaryOpcode_PostInc:
      return false;
   case ast_UnaryOpcode_PostDec:
      return false;
   case ast_UnaryOpcode_PreInc:
      return true;
   case ast_UnaryOpcode_PreDec:
      return true;
   case ast_UnaryOpcode_AddrOf:
      return true;
   case ast_UnaryOpcode_Deref:
      return true;
   case ast_UnaryOpcode_Minus:
      return true;
   case ast_UnaryOpcode_Not:
      return true;
   case ast_UnaryOpcode_LNot:
      return true;
   }
   return true;
}

static src_loc_SrcLoc ast_UnaryOperator_getStartLoc(const ast_UnaryOperator* e)
{
   if (ast_UnaryOperator_isBefore(e)) return ast_Expr_getLoc(&e->parent);

   return ast_Expr_getStartLoc(e->inner);
}

static src_loc_SrcLoc ast_UnaryOperator_getEndLoc(const ast_UnaryOperator* e)
{
   if (ast_UnaryOperator_isBefore(e)) return ast_Expr_getStartLoc(e->inner);

   return ast_Expr_getLoc(&e->parent);
}

static const char* ast_UnaryOperator_getOpcodeStr(const ast_UnaryOperator* e)
{
   return ast_unaryOpcode_names[ast_UnaryOperator_getOpcode(e)];
}

static void ast_UnaryOperator_print(const ast_UnaryOperator* e, string_buffer_Buf* out, uint32_t indent)
{
   ast_Expr_printKind(&e->parent, out, indent);
   ast_Expr_printTypeBits(&e->parent, out);
   string_buffer_Buf_space(out);
   string_buffer_Buf_color(out, ast_col_Value);
   string_buffer_Buf_add(out, ast_unaryOpcode_names[ast_UnaryOperator_getOpcode(e)]);
   string_buffer_Buf_newline(out);
   ast_Expr_print(e->inner, out, (indent + 1));
}

static void ast_UnaryOperator_printLiteral(const ast_UnaryOperator* e, string_buffer_Buf* out)
{
   const char* opcode = ast_unaryOpcode_names[ast_UnaryOperator_getOpcode(e)];
   if (ast_UnaryOperator_isBefore(e)) {
      string_buffer_Buf_add(out, opcode);
      ast_Expr_printLiteral(e->inner, out);
   } else {
      ast_Expr_printLiteral(e->inner, out);
      string_buffer_Buf_add(out, opcode);
   }
}

static void ast_Type_init(ast_Type* t, ast_TypeKind k)
{
   t->bits = 0;
   t->typeBits.kind = k;
   t->ptr_pool_idx = 0;
   t->canonicalType.ptr = 0;
}

static ast_TypeKind ast_Type_getKind(const ast_Type* t)
{
   return ((ast_TypeKind)(t->typeBits.kind));
}

static bool ast_Type_hasCanonicalType(const ast_Type* t)
{
   return (t->canonicalType.ptr != 0);
}

static ast_QualType ast_Type_getCanonicalType(const ast_Type* t)
{
   return t->canonicalType;
}

static void ast_Type_setCanonicalType(ast_Type* t, ast_QualType canon)
{
   t->canonicalType = canon;
}

static uint32_t ast_Type_getIndex(const ast_Type* t)
{
   return t->ptr_pool_idx;
}

static ast_ArrayType* ast_Type_asArray(ast_Type* t)
{
   if ((ast_Type_getKind(t) == ast_TypeKind_Array)) return ((ast_ArrayType*)(t));

   return NULL;
}

static bool ast_Type_isBuiltinType(const ast_Type* t)
{
   return ((ast_Type_getKind(t) == ast_TypeKind_Builtin));
}

static bool ast_Type_isArrayType(const ast_Type* t)
{
   return ((ast_Type_getKind(t) == ast_TypeKind_Array));
}

static bool ast_Type_isStructType(const ast_Type* t)
{
   return ((ast_Type_getKind(t) == ast_TypeKind_Struct));
}

static bool ast_Type_isPointerType(const ast_Type* t)
{
   return ((ast_Type_getKind(t) == ast_TypeKind_Pointer));
}

static bool ast_Type_isFunctionType(const ast_Type* t)
{
   return ((ast_Type_getKind(t) == ast_TypeKind_Function));
}

static bool ast_Type_isEnumType(const ast_Type* t)
{
   return ((ast_Type_getKind(t) == ast_TypeKind_Enum));
}

static bool ast_Type_isVoidType(const ast_Type* t)
{
   return (t == ast_QualType_getTypeOrNil(&ast_builtins[ast_BuiltinKind_Void]));
}

static void ast_Type_dump(const ast_Type* t)
{
   string_buffer_Buf* out = string_buffer_create(256, ast_useColor(), 2);
   ast_Type_print(t, out);
   string_buffer_Buf_color(out, color_Normal);
   printf("%s\n", string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static uint32_t ast_Type_getAlignment(const ast_Type* t)
{
   switch (ast_Type_getKind(t)) {
   case ast_TypeKind_Builtin: {
      const ast_BuiltinType* bi = ((ast_BuiltinType*)(t));
      return ast_BuiltinType_getAlignment(bi);
   }
   case ast_TypeKind_Pointer:
      break;
   case ast_TypeKind_Array: {
      const ast_ArrayType* at = ((ast_ArrayType*)(t));
      ast_QualType elem = ast_ArrayType_getElemType(at);
      return ast_QualType_getAlignment(&elem);
   }
   case ast_TypeKind_Struct: {
      const ast_StructType* s = ((ast_StructType*)(t));
      const ast_StructTypeDecl* std = ast_StructType_getDecl(s);
      return ast_StructTypeDecl_getAlignment(std);
   }
   case ast_TypeKind_Enum: {
      const ast_EnumType* e = ((ast_EnumType*)(t));
      const ast_EnumTypeDecl* etd = ast_EnumType_getDecl(e);
      ast_QualType it = ast_EnumTypeDecl_getImplType(etd);
      return ast_Type_getAlignment(ast_QualType_getTypeOrNil(&it));
   }
   case ast_TypeKind_Function:
      break;
   case ast_TypeKind_Alias: {
      ast_QualType canon = ast_Type_getCanonicalType(t);
      return ast_QualType_getAlignment(&canon);
   }
   case ast_TypeKind_Module:
      return 0;
   }
   return ast_getWordSize();
}

static uint32_t ast_Type_getSize(const ast_Type* t)
{
   switch (ast_Type_getKind(t)) {
   case ast_TypeKind_Builtin: {
      const ast_BuiltinType* bi = ((ast_BuiltinType*)(t));
      return ast_BuiltinType_getAlignment(bi);
   }
   case ast_TypeKind_Pointer:
      break;
   case ast_TypeKind_Array: {
      const ast_ArrayType* at = ((ast_ArrayType*)(t));
      ast_QualType elem = ast_ArrayType_getElemType(at);
      return (ast_ArrayType_getSize(at) * ast_QualType_getSize(&elem));
   }
   case ast_TypeKind_Struct: {
      const ast_StructType* s = ((ast_StructType*)(t));
      const ast_StructTypeDecl* std = ast_StructType_getDecl(s);
      return ast_StructTypeDecl_getSize(std);
   }
   case ast_TypeKind_Enum: {
      const ast_EnumType* e = ((ast_EnumType*)(t));
      const ast_EnumTypeDecl* etd = ast_EnumType_getDecl(e);
      ast_QualType it = ast_EnumTypeDecl_getImplType(etd);
      return ast_Type_getSize(ast_QualType_getTypeOrNil(&it));
   }
   case ast_TypeKind_Function:
      break;
   case ast_TypeKind_Alias: {
      ast_QualType canon = ast_Type_getCanonicalType(t);
      return ast_QualType_getAlignment(&canon);
   }
   case ast_TypeKind_Module:
      return 0;
   }
   return ast_getWordSize();
}

static void ast_Type_print(const ast_Type* t, string_buffer_Buf* out)
{
   string_buffer_Buf_color(out, ast_col_Type);
   switch (ast_Type_getKind(t)) {
   case ast_TypeKind_Builtin:
      ast_BuiltinType_print(((ast_BuiltinType*)(t)), out);
      break;
   case ast_TypeKind_Pointer:
      ast_PointerType_print(((ast_PointerType*)(t)), out);
      break;
   case ast_TypeKind_Array:
      ast_ArrayType_print(((ast_ArrayType*)(t)), out);
      break;
   case ast_TypeKind_Struct:
      ast_StructType_print(((ast_StructType*)(t)), out);
      break;
   case ast_TypeKind_Enum:
      ast_EnumType_print(((ast_EnumType*)(t)), out);
      break;
   case ast_TypeKind_Function:
      ast_FunctionType_print(((ast_FunctionType*)(t)), out);
      break;
   case ast_TypeKind_Alias:
      ast_AliasType_print(((ast_AliasType*)(t)), out);
      break;
   case ast_TypeKind_Module:
      ast_ModuleType_print(((ast_ModuleType*)(t)), out);
      break;
   }
}

static void ast_Type_fullPrint(const ast_Type* t, string_buffer_Buf* out, uint32_t indent)
{
   switch (ast_Type_getKind(t)) {
   case ast_TypeKind_Builtin:
      ast_BuiltinType_fullPrint(((ast_BuiltinType*)(t)), out, indent);
      break;
   case ast_TypeKind_Pointer:
      ast_PointerType_fullPrint(((ast_PointerType*)(t)), out, indent);
      break;
   case ast_TypeKind_Array:
      ast_ArrayType_fullPrint(((ast_ArrayType*)(t)), out, indent);
      break;
   case ast_TypeKind_Struct:
      ast_StructType_fullPrint(((ast_StructType*)(t)), out, indent);
      break;
   case ast_TypeKind_Enum:
      ast_EnumType_fullPrint(((ast_EnumType*)(t)), out, indent);
      break;
   case ast_TypeKind_Function:
      ast_FunctionType_fullPrint(((ast_FunctionType*)(t)), out, indent);
      break;
   case ast_TypeKind_Alias:
      ast_AliasType_fullPrint(((ast_AliasType*)(t)), out, indent);
      break;
   case ast_TypeKind_Module:
      ast_ModuleType_fullPrint(((ast_ModuleType*)(t)), out, indent);
      break;
   }
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_add(out, "canonical=");
   ast_Type* canon = ast_QualType_getTypeOrNil(&t->canonicalType);
   if (canon) {
      if ((canon == t)) {
         string_buffer_Buf_add(out, "this\n");
      } else {
         ast_Type_fullPrint(canon, out, (indent + 1));
      }
   } else {
      string_buffer_Buf_add(out, "NIL\n");
   }
}

static ast_AliasType* ast_AliasType_create(ast_context_Context* c, ast_AliasTypeDecl* decl)
{
   ast_AliasType* t = ast_context_Context_alloc(c, 24);
   ast_Type_init(&t->parent, ast_TypeKind_Alias);
   t->decl = decl;
   ast_Stats_addType(ast_TypeKind_Alias, 24);
   return t;
}

static ast_AliasTypeDecl* ast_AliasType_getDecl(const ast_AliasType* t)
{
   return t->decl;
}

static void ast_AliasType_print(const ast_AliasType* t, string_buffer_Buf* out)
{
   string_buffer_Buf_add(out, "(alias)");
   string_buffer_Buf_add(out, ast_Decl_getModuleName(&t->decl->parent));
   string_buffer_Buf_add1(out, '.');
   string_buffer_Buf_add(out, ast_Decl_getName(&t->decl->parent));
}

static void ast_AliasType_fullPrint(const ast_AliasType* t, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_print(out, "AliasType [%p] %s\n", t, ast_Decl_getName(ast_AliasTypeDecl_asDecl(t->decl)));
}

static ast_ArrayType* ast_ArrayType_create(ast_context_Context* c, ast_QualType elem, bool has_size, uint32_t size)
{
   ast_ArrayType* t = ast_context_Context_alloc(c, 32);
   ast_Type_init(&t->parent, ast_TypeKind_Array);
   t->parent.arrayTypeBits.is_incremental = false;
   t->parent.arrayTypeBits.has_size = has_size;
   t->elem = elem;
   t->size = size;
   ast_Stats_addType(ast_TypeKind_Array, 32);
   return t;
}

static ast_ArrayType* ast_ArrayType_createIncremental(ast_context_Context* c, ast_QualType elem)
{
   ast_ArrayType* t = ast_context_Context_alloc(c, 32);
   ast_Type_init(&t->parent, ast_TypeKind_Array);
   t->parent.arrayTypeBits.is_incremental = true;
   t->parent.arrayTypeBits.has_size = false;
   t->elem = elem;
   t->size = 0;
   ast_Stats_addType(ast_TypeKind_Array, 32);
   return t;
}

static ast_Type* ast_ArrayType_asType(ast_ArrayType* t)
{
   return &t->parent;
}

static ast_QualType ast_ArrayType_getElemType(const ast_ArrayType* t)
{
   return t->elem;
}

static bool ast_ArrayType_isIncremental(const ast_ArrayType* t)
{
   return t->parent.arrayTypeBits.is_incremental;
}

static uint32_t ast_ArrayType_hasSize(const ast_ArrayType* t)
{
   return t->parent.arrayTypeBits.has_size;
}

static uint32_t ast_ArrayType_getSize(const ast_ArrayType* t)
{
   return t->size;
}

static void ast_ArrayType_setSize(ast_ArrayType* t, uint32_t size)
{
   t->parent.arrayTypeBits.has_size = true;
   t->size = size;
}

static void ast_ArrayType_printPreName(const ast_ArrayType* t, string_buffer_Buf* out)
{
   ast_QualType_print(&t->elem, out);
}

static void ast_ArrayType_printPostName(const ast_ArrayType* t, string_buffer_Buf* out)
{
   string_buffer_Buf_add1(out, '[');
   if (t->parent.arrayTypeBits.is_incremental) {
      string_buffer_Buf_add(out, "+");
   } else {
      string_buffer_Buf_print(out, "%u", t->size);
   }
   string_buffer_Buf_add1(out, ']');
}

static void ast_ArrayType_print(const ast_ArrayType* t, string_buffer_Buf* out)
{
   ast_ArrayType* at = ast_QualType_getArrayTypeOrNil(&t->elem);
   if (at) ast_ArrayType_printPreName(at, out);
   else ast_QualType_printInner(&t->elem, out, false, true, true);
   ast_ArrayType_printPostName(t, out);
   if (at) ast_ArrayType_printPostName(at, out);
}

static void ast_ArrayType_fullPrint(const ast_ArrayType* t, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_print(out, "ArrayType [%p]", t);
   if (t->parent.arrayTypeBits.has_size) string_buffer_Buf_print(out, " size=%u", t->size);
   string_buffer_Buf_newline(out);
   ast_QualType_fullPrint(&t->elem, out, (indent + 1));
}

static bool ast_builtinKind2Signed(ast_BuiltinKind kind)
{
   return ast_BuiltinType_signed[kind];
}

static ast_BuiltinType* ast_BuiltinType_create(ast_context_Context* c, ast_BuiltinKind kind)
{
   ast_BuiltinType* b = ast_context_Context_alloc(c, 16);
   ast_Type_init(&b->parent, ast_TypeKind_Builtin);
   b->parent.builtinTypeBits.kind = kind;
   ast_Type_setCanonicalType(&b->parent, ast_QualType_init(&b->parent));
   ast_Stats_addType(ast_TypeKind_Builtin, 16);
   return b;
}

static ast_BuiltinKind ast_BuiltinType_getKind(const ast_BuiltinType* b)
{
   return ((ast_BuiltinKind)(b->parent.builtinTypeBits.kind));
}

static ast_BuiltinKind ast_BuiltinType_getBaseKind(const ast_BuiltinType* b)
{
   return ast_globals->builtinType_baseTypes[ast_BuiltinType_getKind(b)];
}

static bool ast_BuiltinType_isChar(const ast_BuiltinType* b)
{
   return (b->parent.builtinTypeBits.kind == ast_BuiltinKind_Char);
}

static bool ast_BuiltinType_isInt8(const ast_BuiltinType* b)
{
   return (b->parent.builtinTypeBits.kind == ast_BuiltinKind_Int8);
}

static bool ast_BuiltinType_isUInt8(const ast_BuiltinType* b)
{
   return (b->parent.builtinTypeBits.kind == ast_BuiltinKind_UInt8);
}

static bool ast_BuiltinType_isInt32(const ast_BuiltinType* b)
{
   return (b->parent.builtinTypeBits.kind == ast_BuiltinKind_Int32);
}

static bool ast_BuiltinType_isBool(const ast_BuiltinType* b)
{
   return (b->parent.builtinTypeBits.kind == ast_BuiltinKind_Bool);
}

static bool ast_BuiltinType_isVoid(const ast_BuiltinType* b)
{
   return (b->parent.builtinTypeBits.kind == ast_BuiltinKind_Void);
}

static bool ast_BuiltinType_isArithmeticType(const ast_BuiltinType* b)
{
   return (b->parent.builtinTypeBits.kind < ast_BuiltinKind_Bool);
}

static const char* ast_BuiltinType_kind2str(const ast_BuiltinType* b)
{
   return ast_builtinType_names[ast_BuiltinType_getKind(b)];
}

static bool ast_BuiltinType_isPromotableIntegerType(const ast_BuiltinType* b)
{
   return ast_BuiltinType_promotable[ast_BuiltinType_getKind(b)];
}

static bool ast_BuiltinType_isInteger(const ast_BuiltinType* b)
{
   return ast_BuiltinType_integer[ast_BuiltinType_getKind(b)];
}

static bool ast_BuiltinType_isIntegerOrBool(const ast_BuiltinType* b)
{
   return (ast_BuiltinType_isInteger(b) || (ast_BuiltinType_getKind(b) == ast_BuiltinKind_Bool));
}

static bool ast_BuiltinType_isFloatingPoint(const ast_BuiltinType* b)
{
   return (((ast_BuiltinType_getKind(b) == ast_BuiltinKind_Float32) || (ast_BuiltinType_getKind(b) == ast_BuiltinKind_Float64)));
}

static bool ast_BuiltinType_isSigned(const ast_BuiltinType* b)
{
   return ast_BuiltinType_signed[ast_BuiltinType_getKind(b)];
}

static bool ast_BuiltinType_isUnsigned(const ast_BuiltinType* b)
{
   return ast_BuiltinType_unsigned[ast_BuiltinType_getKind(b)];
}

static uint32_t ast_BuiltinType_getAlignment(const ast_BuiltinType* b)
{
   return ast_globals->builtinType_sizes[ast_BuiltinType_getKind(b)];
}

static uint32_t ast_BuiltinType_getWidth(const ast_BuiltinType* b)
{
   return ast_globals->builtinType_width[ast_BuiltinType_getKind(b)];
}

static void ast_BuiltinType_print(const ast_BuiltinType* b, string_buffer_Buf* out)
{
   string_buffer_Buf_add(out, ast_builtinType_names[ast_BuiltinType_getKind(b)]);
}

static void ast_BuiltinType_fullPrint(const ast_BuiltinType* t, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_print(out, "BuiltinType [%p] %s\n", t, ast_BuiltinType_kind2str(t));
}

static ast_EnumType* ast_EnumType_create(ast_context_Context* c, ast_EnumTypeDecl* decl)
{
   ast_EnumType* t = ast_context_Context_alloc(c, 24);
   ast_Type_init(&t->parent, ast_TypeKind_Enum);
   t->decl = decl;
   ast_Stats_addType(ast_TypeKind_Enum, 24);
   ast_Type_setCanonicalType(&t->parent, ast_QualType_init(((ast_Type*)(t))));
   return t;
}

static ast_EnumTypeDecl* ast_EnumType_getDecl(const ast_EnumType* t)
{
   return t->decl;
}

static ast_Type* ast_EnumType_asType(ast_EnumType* t)
{
   return &t->parent;
}

static ast_QualType ast_EnumType_getImplType(const ast_EnumType* t)
{
   return ast_EnumTypeDecl_getImplType(t->decl);
}

static const char* ast_EnumType_getName(const ast_EnumType* t)
{
   return ast_Decl_getName(&t->decl->parent);
}

static void ast_EnumType_print(const ast_EnumType* t, string_buffer_Buf* out)
{
   string_buffer_Buf_add(out, "(enum)");
   string_buffer_Buf_add(out, ast_Decl_getModuleName(&t->decl->parent));
   string_buffer_Buf_add1(out, '.');
   string_buffer_Buf_add(out, ast_Decl_getName(&t->decl->parent));
}

static void ast_EnumType_fullPrint(const ast_EnumType* t, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_print(out, "EnumType [%p] %s\n", t, ast_Decl_getName(ast_EnumTypeDecl_asDecl(t->decl)));
}

static ast_FunctionType* ast_FunctionType_create(ast_context_Context* c, ast_FunctionDecl* decl)
{
   ast_FunctionType* t = ast_context_Context_alloc(c, 24);
   ast_Type_init(&t->parent, ast_TypeKind_Function);
   t->decl = NULL;
   t->decl = decl;
   ast_Type_setCanonicalType(&t->parent, ast_QualType_init(&t->parent));
   ast_Stats_addType(ast_TypeKind_Function, 24);
   return t;
}

static ast_FunctionDecl* ast_FunctionType_getDecl(const ast_FunctionType* t)
{
   return t->decl;
}

static ast_Type* ast_FunctionType_asType(ast_FunctionType* t)
{
   return &t->parent;
}

static void ast_FunctionType_print(const ast_FunctionType* t, string_buffer_Buf* out)
{
   ast_FunctionDecl_printType(t->decl, out);
}

static void ast_FunctionType_fullPrint(const ast_FunctionType* t, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_print(out, "FunctionType [%p] %s\n", t, ast_Decl_getName(ast_FunctionDecl_asDecl(t->decl)));
}

static ast_ModuleType* ast_ModuleType_create(ast_context_Context* c, ast_Module* mod)
{
   ast_ModuleType* t = ast_context_Context_alloc(c, 24);
   ast_Type_init(&t->parent, ast_TypeKind_Module);
   t->mod = mod;
   ast_Type_setCanonicalType(&t->parent, ast_QualType_init(&t->parent));
   ast_Stats_addType(ast_TypeKind_Module, 24);
   return t;
}

static ast_Type* ast_ModuleType_asType(ast_ModuleType* t)
{
   return &t->parent;
}

static ast_Module* ast_ModuleType_getModule(const ast_ModuleType* t)
{
   return t->mod;
}

static void ast_ModuleType_print(const ast_ModuleType* t, string_buffer_Buf* out)
{
   string_buffer_Buf_print(out, "Module %s", ast_Module_getName(t->mod));
}

static void ast_ModuleType_fullPrint(const ast_ModuleType* t, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_print(out, "ModuleType %s\n", ast_Module_getName(t->mod));
}

static ast_PointerType* ast_PointerType_create(ast_context_Context* c, ast_QualType inner)
{
   ast_PointerType* t = ast_context_Context_alloc(c, 24);
   ast_Type_init(&t->parent, ast_TypeKind_Pointer);
   t->inner = inner;
   ast_Stats_addType(ast_TypeKind_Pointer, 24);
   return t;
}

static ast_Type* ast_PointerType_asType(ast_PointerType* t)
{
   return &t->parent;
}

static ast_QualType ast_PointerType_getInner(const ast_PointerType* t)
{
   return t->inner;
}

static void ast_PointerType_print(const ast_PointerType* t, string_buffer_Buf* out)
{
   ast_QualType_printInner(&t->inner, out, false, true, true);
   string_buffer_Buf_add1(out, '*');
}

static void ast_PointerType_fullPrint(const ast_PointerType* t, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_print(out, "PointerType [%p]\n", t);
   ast_QualType_fullPrint(&t->inner, out, (indent + 1));
}

static ast_QualType ast_QualType_init(ast_Type* t)
{
   ast_QualType qt = { ((size_t)(t)) };
   return qt;
}

static void ast_QualType_set(ast_QualType* qt, ast_Type* t)
{
   qt->ptr = ((size_t)(t));
}

static void ast_QualType_setConst(ast_QualType* qt)
{
   qt->ptr |= ast_QualType_Const;
}

static void ast_QualType_unsetConst(ast_QualType* qt)
{
   qt->ptr &= ~ast_QualType_Const;
}

static bool ast_QualType_isConst(ast_QualType* qt)
{
   return (((qt->ptr & ast_QualType_Const)) != 0);
}

static bool ast_QualType_isVolatile(ast_QualType* qt)
{
   return (((qt->ptr & ast_QualType_Volatile)) != 0);
}

static void ast_QualType_setVolatile(ast_QualType* qt)
{
   qt->ptr |= ast_QualType_Volatile;
}

static uint32_t ast_QualType_getQuals(const ast_QualType* qt)
{
   return (qt->ptr & ast_QualType_Mask);
}

static void ast_QualType_copyQuals(ast_QualType* qt, ast_QualType other)
{
   qt->ptr &= ~ast_QualType_Mask;
   qt->ptr |= ((other.ptr & ast_QualType_Mask));
}

static void ast_QualType_clearQuals(ast_QualType* qt)
{
   qt->ptr &= ~ast_QualType_Mask;
}

static bool ast_QualType_isConstant(const ast_QualType* qt)
{
   ast_QualType canon = ast_QualType_getCanonicalType(qt);
   const ast_Type* t = ast_QualType_getTypeOrNil(&canon);
   if (ast_QualType_isConst(&canon)) return true;

   if ((ast_Type_getKind(t) == ast_TypeKind_Array)) {
      const ast_ArrayType* at = ((ast_ArrayType*)(t));
      canon = ast_ArrayType_getElemType(at);
      return ast_QualType_isConstant(&canon);
   }
   return false;
}

static bool ast_QualType_isValid(const ast_QualType* qt)
{
   return (qt->ptr != 0);
}

static bool ast_QualType_isInvalid(const ast_QualType* qt)
{
   return (qt->ptr == 0);
}

static bool ast_QualType_isScalar(const ast_QualType* qt)
{
   ast_QualType canon = ast_QualType_getCanonicalType(qt);
   const ast_Type* t = ast_QualType_getTypeOrNil(&canon);
   if (ast_Type_isBuiltinType(t)) {
      ast_BuiltinType* bi = ((ast_BuiltinType*)(t));
      return !ast_BuiltinType_isVoid(bi);
   }
   if (ast_QualType_isPointer(&canon)) return true;

   if (ast_QualType_isFunction(&canon)) return true;

   if (ast_QualType_isEnum(&canon)) return true;

   return false;
}

static ast_Type* ast_QualType_getType(const ast_QualType* qt)
{
   size_t t = (qt->ptr & ~ast_QualType_Mask);
   c2_assert((t) != 0, "ast/qualtype.c2:109: ast.QualType.getType", "t");
   return ((ast_Type*)(t));
}

static ast_Type* ast_QualType_getTypeOrNil(const ast_QualType* qt)
{
   size_t temp = (qt->ptr & ~ast_QualType_Mask);
   return ((ast_Type*)(temp));
}

static bool ast_QualType_hasCanonicalType(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getType(qt);
   return (t->canonicalType.ptr != 0);
}

static ast_QualType ast_QualType_getCanonicalType(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getType(qt);
   ast_QualType canon = t->canonicalType;
   ast_QualType_copyQuals(&canon, *qt);
   return canon;
}

static void ast_QualType_setCanonicalType(ast_QualType* qt, ast_QualType canon)
{
   ast_Type* t = ast_QualType_getType(qt);
   ast_Type_setCanonicalType(t, canon);
}

static bool ast_QualType_isConstPtr(ast_QualType* qt)
{
   ast_PointerType* pt = ast_QualType_getPointerType(qt);
   return ast_QualType_isConst(&pt->inner);
}

static ast_TypeKind ast_QualType_getKind(ast_QualType* qt)
{
   ast_Type* t = ast_QualType_getType(qt);
   return ast_Type_getKind(t);
}

static uint32_t ast_QualType_getIndex(ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getType(qt);
   return ast_Type_getIndex(t);
}

static uint32_t ast_QualType_getAlignment(ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getType(qt);
   return ast_Type_getAlignment(t);
}

static uint32_t ast_QualType_getSize(ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getType(qt);
   return ast_Type_getSize(t);
}

static bool ast_QualType_isBool(const ast_QualType* qt)
{
   return (ast_QualType_getTypeOrNil(qt) == ast_QualType_getTypeOrNil(&ast_builtins[ast_BuiltinKind_Bool]));
}

static bool ast_QualType_isBuiltin(const ast_QualType* qt)
{
   return ast_Type_isBuiltinType(ast_QualType_getTypeOrNil(qt));
}

static bool ast_QualType_isArray(const ast_QualType* qt)
{
   return ast_Type_isArrayType(ast_QualType_getTypeOrNil(qt));
}

static bool ast_QualType_isStruct(const ast_QualType* qt)
{
   return ast_Type_isStructType(ast_QualType_getTypeOrNil(qt));
}

static bool ast_QualType_isInteger(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if (!ast_Type_isBuiltinType(t)) return false;

   const ast_BuiltinType* bi = ((ast_BuiltinType*)(t));
   return ast_BuiltinType_isInteger(bi);
}

static bool ast_QualType_isCharPointer(const ast_QualType* qt)
{
   if (!ast_QualType_isPointer(qt)) return false;

   const ast_PointerType* pt = ast_QualType_getPointerType(qt);
   ast_QualType inner = ast_PointerType_getInner(pt);
   return ast_QualType_isChar(&inner);
}

static bool ast_QualType_isPointer(const ast_QualType* qt)
{
   return ast_Type_isPointerType(ast_QualType_getTypeOrNil(qt));
}

static bool ast_QualType_isFunction(const ast_QualType* qt)
{
   return ast_Type_isFunctionType(ast_QualType_getTypeOrNil(qt));
}

static bool ast_QualType_isEnum(const ast_QualType* qt)
{
   return ast_Type_isEnumType(ast_QualType_getTypeOrNil(qt));
}

static bool ast_QualType_isVoid(const ast_QualType* qt)
{
   return ast_Type_isVoidType(ast_QualType_getTypeOrNil(qt));
}

static ast_BuiltinType* ast_QualType_getBuiltin(const ast_QualType* qt)
{
   return ((ast_BuiltinType*)(ast_QualType_getTypeOrNil(qt)));
}

static ast_BuiltinType* ast_QualType_getBuiltinTypeOrNil(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if ((t && (ast_Type_getKind(t) == ast_TypeKind_Builtin))) return ((ast_BuiltinType*)(t));

   return NULL;
}

static ast_StructType* ast_QualType_getStructType(const ast_QualType* qt)
{
   return ((ast_StructType*)(ast_QualType_getTypeOrNil(qt)));
}

static ast_PointerType* ast_QualType_getPointerType(const ast_QualType* qt)
{
   return ((ast_PointerType*)(ast_QualType_getTypeOrNil(qt)));
}

static ast_FunctionType* ast_QualType_getFunctionType(const ast_QualType* qt)
{
   return ((ast_FunctionType*)(ast_QualType_getTypeOrNil(qt)));
}

static ast_ArrayType* ast_QualType_getArrayType(const ast_QualType* qt)
{
   return ((ast_ArrayType*)(ast_QualType_getTypeOrNil(qt)));
}

static ast_EnumType* ast_QualType_getEnum(const ast_QualType* qt)
{
   return ((ast_EnumType*)(ast_QualType_getTypeOrNil(qt)));
}

static ast_FunctionType* ast_QualType_getFunctionTypeOrNil(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if ((t && (ast_Type_getKind(t) == ast_TypeKind_Function))) return ((ast_FunctionType*)(t));

   return NULL;
}

static ast_StructType* ast_QualType_getStructTypeOrNil(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if ((t && (ast_Type_getKind(t) == ast_TypeKind_Struct))) return ((ast_StructType*)(t));

   return NULL;
}

static ast_PointerType* ast_QualType_getPointerTypeOrNil(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if ((t && (ast_Type_getKind(t) == ast_TypeKind_Pointer))) return ((ast_PointerType*)(t));

   return NULL;
}

static ast_ArrayType* ast_QualType_getArrayTypeOrNil(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if ((t && (ast_Type_getKind(t) == ast_TypeKind_Array))) return ((ast_ArrayType*)(t));

   return NULL;
}

static ast_EnumType* ast_QualType_getEnumType(const ast_QualType* qt)
{
   return ((ast_EnumType*)(ast_QualType_getTypeOrNil(qt)));
}

static ast_EnumType* ast_QualType_getEnumTypeOrNil(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if ((ast_Type_getKind(t) == ast_TypeKind_Enum)) return ((ast_EnumType*)(t));

   return NULL;
}

static bool ast_QualType_isChar(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if ((ast_Type_getKind(t) != ast_TypeKind_Builtin)) return false;

   const ast_BuiltinType* bi = ((ast_BuiltinType*)(t));
   return ast_BuiltinType_isChar(bi);
}

static bool ast_QualType_isInt8(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if ((ast_Type_getKind(t) != ast_TypeKind_Builtin)) return false;

   const ast_BuiltinType* bi = ((ast_BuiltinType*)(t));
   return ast_BuiltinType_isInt8(bi);
}

static bool ast_QualType_isUInt8(const ast_QualType* qt)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if ((ast_Type_getKind(t) != ast_TypeKind_Builtin)) return false;

   const ast_BuiltinType* bi = ((ast_BuiltinType*)(t));
   return ast_BuiltinType_isUInt8(bi);
}

static bool ast_QualType_needsCtvInit(const ast_QualType* qt)
{
   ast_QualType canon = ast_QualType_getCanonicalType(qt);
   const ast_Type* t = ast_QualType_getTypeOrNil(&canon);
   if (!t) {
      printf("MISSING CANONICAL\n");
      ast_QualType_dump_full(qt);
      return false;
   }
   c2_assert((t) != NULL, "ast/qualtype.c2:312: ast.QualType.needsCtvInit", "t");
   switch (ast_Type_getKind(t)) {
   case ast_TypeKind_Builtin:
      return true;
   case ast_TypeKind_Pointer:
      return false;
   case ast_TypeKind_Array:
      return true;
   case ast_TypeKind_Struct:
      return true;
   case ast_TypeKind_Enum:
      return true;
   case ast_TypeKind_Function:
      return false;
   case ast_TypeKind_Alias:
      return false;
   case ast_TypeKind_Module:
      return false;
   }
   return false;
}

static const char* ast_QualType_diagName(const ast_QualType* qt)
{
   static char msgs[4][128];
   static uint32_t msg_id = 0;
   char* msg = msgs[msg_id];
   msg_id = (((msg_id + 1)) % 4);
   string_buffer_Buf* buf = string_buffer_create_static(128, false, msg);
   ast_QualType_printInner(qt, buf, true, true, true);
   string_buffer_Buf_free(buf);
   return msg;
}

static const char* ast_QualType_diagNameBare(const ast_QualType* qt)
{
   static char msgs[4][128];
   static uint32_t msg_id = 0;
   char* msg = msgs[msg_id];
   msg_id = (((msg_id + 1)) % 4);
   string_buffer_Buf* buf = string_buffer_create_static(128, false, msg);
   ast_QualType_printInner(qt, buf, true, false, true);
   string_buffer_Buf_free(buf);
   return msg;
}

static void ast_QualType_dump(const ast_QualType* qt)
{
   string_buffer_Buf* out = string_buffer_create(512, ast_useColor(), 2);
   ast_QualType_printInner(qt, out, false, true, false);
   string_buffer_Buf_color(out, color_Normal);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static void ast_QualType_dump_full(const ast_QualType* qt)
{
   string_buffer_Buf* out = string_buffer_create(512, ast_useColor(), 1);
   ast_QualType_fullPrint(qt, out, 0);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static void ast_QualType_printQuoted(const ast_QualType* qt, string_buffer_Buf* out)
{
   string_buffer_Buf_color(out, ast_col_Type);
   ast_QualType_printInner(qt, out, true, true, true);
   string_buffer_Buf_color(out, ast_col_Type);
}

static void ast_QualType_print(const ast_QualType* qt, string_buffer_Buf* out)
{
   ast_QualType_printInner(qt, out, true, true, true);
}

static void ast_QualType_printInner(const ast_QualType* qt, string_buffer_Buf* out, bool printCanon, bool printModifiers, bool print_error)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if (t) {
      uint32_t quals = ast_QualType_getQuals(qt);
      if (ast_QualType_isPointer(qt)) {
         ast_Type_print(t, out);
         if (printModifiers) {
            if ((quals & ast_QualType_Const)) string_buffer_Buf_add(out, " const");
            if ((quals & ast_QualType_Volatile)) string_buffer_Buf_add(out, " volatile");
         }
      } else {
         if (printModifiers) {
            if (quals) string_buffer_Buf_color(out, ast_col_Type);
            if ((quals & ast_QualType_Const)) string_buffer_Buf_add(out, "const ");
            if ((quals & ast_QualType_Volatile)) string_buffer_Buf_add(out, "volatile ");
         }
         ast_Type_print(t, out);
      }
      if (printCanon) {
         ast_QualType qt2 = ast_QualType_getCanonicalType(qt);
         const ast_Type* canon = ast_QualType_getTypeOrNil(&qt2);
         if ((canon && (canon != t))) {
            string_buffer_Buf_add(out, " => ");
            ast_QualType_printInner(&qt2, out, false, true, true);
         }
      }
   } else {
      if (print_error) {
         string_buffer_Buf_color(out, ast_col_Error);
         string_buffer_Buf_add(out, "??");
      } else {
         string_buffer_Buf_add(out, "QualType(nil)");
      }
   }
}

static void ast_QualType_fullPrint(const ast_QualType* qt, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_add(out, "QualType");
   uint32_t quals = ast_QualType_getQuals(qt);
   if ((quals & ast_QualType_Const)) string_buffer_Buf_add(out, " const");
   if ((quals & ast_QualType_Volatile)) string_buffer_Buf_add(out, " volatile");
   const ast_Type* t = ast_QualType_getTypeOrNil(qt);
   if (t) {
      string_buffer_Buf_newline(out);
      ast_Type_fullPrint(t, out, (indent + 1));
   } else string_buffer_Buf_add(out, " type=nil\n");
}

static ast_StructType* ast_StructType_create(ast_context_Context* c, ast_StructTypeDecl* decl)
{
   ast_StructType* t = ast_context_Context_alloc(c, 24);
   ast_Type_init(&t->parent, ast_TypeKind_Struct);
   t->decl = decl;
   ast_Stats_addType(ast_TypeKind_Struct, 24);
   return t;
}

static ast_StructTypeDecl* ast_StructType_getDecl(const ast_StructType* t)
{
   return t->decl;
}

static ast_Type* ast_StructType_asType(ast_StructType* t)
{
   return &t->parent;
}

static void ast_StructType_print(const ast_StructType* t, string_buffer_Buf* out)
{
   string_buffer_Buf_add(out, "(struct)");
   if (ast_StructTypeDecl_isGlobal(t->decl)) {
      string_buffer_Buf_add(out, ast_Decl_getModuleName(&t->decl->parent));
      string_buffer_Buf_add1(out, '.');
   }
   const char* name = ast_Decl_getName(&t->decl->parent);
   if (name) string_buffer_Buf_add(out, name);
   else string_buffer_Buf_add(out, "<anonymous>");
}

static void ast_StructType_fullPrint(const ast_StructType* t, string_buffer_Buf* out, uint32_t indent)
{
   string_buffer_Buf_indent(out, indent);
   string_buffer_Buf_print(out, "StructType [%p] %s\n", t, ast_Decl_getName(ast_StructTypeDecl_asDecl(t->decl)));
}

static const char* ast_Ref_getName(const ast_Ref* r)
{
   return ast_idx2name(r->name_idx);
}

static void ast_TypeRefHolder_init(ast_TypeRefHolder* h)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   r->flagBits = 0;
   r->dest = 0;
}

static uint32_t ast_TypeRefHolder_getExtraSize(const ast_TypeRefHolder* h)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   return ast_TypeRef_getExtraSize(r);
}

static void ast_TypeRefHolder_setQualifiers(ast_TypeRefHolder* h, uint32_t qualifiers)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   if ((qualifiers & ast_QualType_Volatile)) r->flags.is_volatile = 1;
   if ((qualifiers & ast_QualType_Const)) r->flags.is_const = 1;
}

static void ast_TypeRefHolder_setConst(ast_TypeRefHolder* h)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   r->flags.is_const = 1;
}

static void ast_TypeRefHolder_setVolatile(ast_TypeRefHolder* h)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   r->flags.is_volatile = 1;
}

static void ast_TypeRefHolder_addPointer(ast_TypeRefHolder* h)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   c2_assert(((r->flags.num_ptrs != 3)) != 0, "ast/type_ref.c2:97: ast.TypeRefHolder.addPointer", "r.flags.num_ptrs!=3");
   r->flags.num_ptrs++;
}

static bool ast_TypeRefHolder_isIncrArray(const ast_TypeRefHolder* h)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   return r->flags.incr_array;
}

static void ast_TypeRefHolder_setIncrArray(ast_TypeRefHolder* h)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   r->flags.incr_array = 1;
}

static uint32_t ast_TypeRefHolder_getNumArrays(const ast_TypeRefHolder* h)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   return ast_TypeRef_getNumArrays(r);
}

static void ast_TypeRefHolder_addArray(ast_TypeRefHolder* h, ast_Expr* array)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   c2_assert(((r->flags.num_arrays != 3)) != 0, "ast/type_ref.c2:118: ast.TypeRefHolder.addArray", "r.flags.num_arrays!=3");
   h->arrays[r->flags.num_arrays] = array;
   r->flags.num_arrays++;
}

static void ast_TypeRefHolder_setBuiltin(ast_TypeRefHolder* h, ast_BuiltinKind kind, src_loc_SrcLoc loc)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   r->flags.builtin_kind = kind;
   r->loc = loc;
}

static void ast_TypeRefHolder_setUser(ast_TypeRefHolder* h, src_loc_SrcLoc loc, uint32_t name_idx)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   r->flags.is_user = 1;
   h->user.loc = loc;
   h->user.name_idx = name_idx;
   h->user.decl = NULL;
}

static void ast_TypeRefHolder_setPrefix(ast_TypeRefHolder* h, src_loc_SrcLoc loc, uint32_t name_idx)
{
   ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   h->prefix = h->user;
   r->flags.has_prefix = 1;
   h->user.loc = loc;
   h->user.name_idx = name_idx;
   h->user.decl = NULL;
}

static void ast_TypeRefHolder_fill(const ast_TypeRefHolder* h, ast_TypeRef* dest)
{
   const ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   *dest = *r;
   if (ast_TypeRef_isUser(r)) {
      dest->refs[0] = r->refs[0];
      if (ast_TypeRef_hasPrefix(r)) {
         dest->refs[1] = r->refs[1];
      }
   }
   for (uint32_t i = 0; (i < r->flags.num_arrays); i++) {
      ast_Expr** a = ast_TypeRef_getArray2(dest, i);
      *a = h->arrays[i];
   }
}

static void ast_TypeRefHolder_dump(const ast_TypeRefHolder* h)
{
   const ast_TypeRef* r = ((ast_TypeRef*)(&h->ref));
   string_buffer_Buf* out = string_buffer_create(128, ast_useColor(), 2);
   ast_TypeRef_print(r, out, false);
   for (uint32_t i = 0; (i < ast_TypeRef_getNumArrays(r)); i++) {
      string_buffer_Buf_add(out, "[");
      ast_Expr* size = h->arrays[i];
      if (size) ast_Expr_printLiteral(h->arrays[i], out);
      string_buffer_Buf_add(out, "]");
   }
   string_buffer_Buf_color(out, ast_col_Normal);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static bool ast_TypeRef_matchesTemplate(const ast_TypeRef* r, uint32_t template_arg)
{
   if ((!r->flags.is_user || r->flags.has_prefix)) return false;

   return (r->refs[0].name_idx == template_arg);
}

static void ast_TypeRef_initRef0(ast_TypeRef* r, const ast_TypeRef* r2)
{
   r->flagBits = r2->flagBits;
   r->refs[0] = r2->refs[0];
}

static void ast_TypeRef_instantiate(ast_TypeRef* r, const ast_TypeRef* r1, ast_Instantiator* inst)
{
   const ast_TypeRef* r2 = inst->ref;
   if (ast_TypeRef_matchesTemplate(r1, inst->template_name)) {
      r->flagBits = r2->flagBits;
      if (r2->flags.is_user) {
         r->refs[0].name_idx = r2->refs[0].name_idx;
         r->refs[0].loc = r1->refs[0].loc;
         if (r2->flags.has_prefix) {
            r->refs[1].name_idx = r2->refs[1].name_idx;
         }
         r->refs[0].decl = r2->refs[0].decl;
      }
      r->flags.is_const |= r1->flags.is_const;
      r->flags.is_volatile |= r1->flags.is_volatile;
      r->flags.num_ptrs += r1->flags.num_ptrs;
      if ((inst->used_opaque && (r->flags.num_ptrs == 0))) {
         ast_Instantiator_on_opaque(inst, r->refs[0].loc, r->refs[0].decl);
      }
   } else {
      memcpy(r, r1, (8 + ast_TypeRef_getExtraSize(r1)));
   }
}

static void ast_TypeRef_setDest(ast_TypeRef* r, uint32_t dest)
{
   c2_assert((ast_TypeRef_isUser(r)) != 0, "ast/type_ref.c2:214: ast.TypeRef.setDest", "CALL TODO");
   r->dest = dest;
}

static uint32_t ast_TypeRef_getDest2(const ast_TypeRef* r)
{
   return r->dest;
}

static uint32_t ast_TypeRef_getExtraSize(const ast_TypeRef* r)
{
   uint32_t numrefs = (r->flags.is_user + r->flags.has_prefix);
   uint32_t extra = (numrefs * 16);
   extra += (r->flags.num_arrays * 8);
   return extra;
}

static uint32_t ast_TypeRef_getMaxSizeNoArray(void)
{
   return (8 + (2 * 16));
}

static void* ast_TypeRef_getPointerAfter(const ast_TypeRef* r)
{
   uint8_t* ptr = ((((uint8_t*)(r)) + 8) + ast_TypeRef_getExtraSize(r));
   return ptr;
}

static bool ast_TypeRef_isConst(const ast_TypeRef* r)
{
   return r->flags.is_const;
}

static bool ast_TypeRef_isVolatile(const ast_TypeRef* r)
{
   return r->flags.is_volatile;
}

static bool ast_TypeRef_isUser(const ast_TypeRef* r)
{
   return r->flags.is_user;
}

static bool ast_TypeRef_isVoid(const ast_TypeRef* r)
{
   return ((!r->flags.is_user && (r->flags.num_ptrs == 0)) && (ast_TypeRef_getBuiltinKind(r) == ast_BuiltinKind_Void));
}

static bool ast_TypeRef_isConstCharPtr(const ast_TypeRef* r)
{
   return ((((r->flags.is_const && !r->flags.is_user) && (r->flags.num_ptrs == 1)) && (r->flags.num_arrays == 0)) && (ast_TypeRef_getBuiltinKind(r) == ast_BuiltinKind_Char));
}

static bool ast_TypeRef_isU32(const ast_TypeRef* r)
{
   return (((!r->flags.is_user && (r->flags.num_ptrs == 0)) && (r->flags.num_arrays == 0)) && (ast_TypeRef_getBuiltinKind(r) == ast_BuiltinKind_UInt32));
}

static bool ast_TypeRef_hasPrefix(const ast_TypeRef* r)
{
   return r->flags.has_prefix;
}

static bool ast_TypeRef_isIncrArray(const ast_TypeRef* r)
{
   return r->flags.incr_array;
}

static bool ast_TypeRef_isPointerTo(const ast_TypeRef* r, uint32_t ptr_idx)
{
   if (((r->dest != ptr_idx) || (ptr_idx == 0))) return false;

   return ((((r->flags.num_ptrs == 1) && (r->flags.num_arrays == 0)) && r->flags.is_user));
}

static ast_BuiltinKind ast_TypeRef_getBuiltinKind(const ast_TypeRef* r)
{
   return ((ast_BuiltinKind)(r->flags.builtin_kind));
}

static src_loc_SrcLoc ast_TypeRef_getLoc(const ast_TypeRef* r)
{
   if (ast_TypeRef_isUser(r)) {
      if (ast_TypeRef_hasPrefix(r)) return r->refs[1].loc;
      else return r->refs[0].loc;

   }
   return r->loc;
}

static uint32_t ast_TypeRef_getNumPointers(const ast_TypeRef* r)
{
   return r->flags.num_ptrs;
}

static const ast_Ref* ast_TypeRef_getUser(const ast_TypeRef* r)
{
   if (r->flags.is_user) return &r->refs[0];

   return NULL;
}

static const ast_Decl* ast_TypeRef_getUserDecl(const ast_TypeRef* r)
{
   if (r->flags.is_user) return r->refs[0].decl;

   return NULL;
}

static const ast_Ref* ast_TypeRef_getPrefix(const ast_TypeRef* r)
{
   if (r->flags.has_prefix) return &r->refs[1];

   return NULL;
}

static void ast_TypeRef_setPrefix(ast_TypeRef* r, ast_Decl* d)
{
   r->refs[1].decl = d;
}

static void ast_TypeRef_setUser(ast_TypeRef* r, ast_Decl* d)
{
   r->refs[0].decl = d;
}

static uint32_t ast_TypeRef_getNumArrays(const ast_TypeRef* r)
{
   return r->flags.num_arrays;
}

static ast_Expr* ast_TypeRef_getArray(const ast_TypeRef* r, uint32_t idx)
{
   const uint32_t numrefs = (r->flags.is_user + r->flags.has_prefix);
   const uint8_t* ptr = (((uint8_t*)(r->refs)) + (numrefs * 16));
   ast_Expr** arrays = ((ast_Expr**)(ptr));
   return arrays[idx];
}

static ast_Expr** ast_TypeRef_getArray2(ast_TypeRef* r, uint32_t idx)
{
   const uint32_t numrefs = (r->flags.is_user + r->flags.has_prefix);
   const uint8_t* ptr = (((uint8_t*)(r->refs)) + (numrefs * 16));
   ast_Expr** arrays = ((ast_Expr**)(ptr));
   return &arrays[idx];
}

static void ast_TypeRef_printLiteral(const ast_TypeRef* r, string_buffer_Buf* out, bool print_prefix)
{
   if (ast_TypeRef_isConst(r)) string_buffer_Buf_add(out, "const ");
   if (ast_TypeRef_isVolatile(r)) string_buffer_Buf_add(out, "volatile ");
   if (r->flags.is_user) {
      ast_Decl* d = r->refs[0].decl;
      c2_assert((d) != NULL, "ast/type_ref.c2:339: ast.TypeRef.printLiteral", "d");
      if (print_prefix) {
         string_buffer_Buf_add(out, ast_Decl_getFullName(d));
      } else {
         string_buffer_Buf_add(out, ast_Decl_getName(d));
      }
   } else {
      string_buffer_Buf_add(out, ast_builtinType_names[r->flags.builtin_kind]);
   }
   for (uint32_t i = 0; (i < r->flags.num_ptrs); i++) string_buffer_Buf_add1(out, '*');
   if (r->flags.incr_array) string_buffer_Buf_add(out, "*");
   for (uint32_t i = 0; (i < r->flags.num_arrays); i++) {
      string_buffer_Buf_add(out, "*");
   }
}

static void ast_TypeRef_print(const ast_TypeRef* r, string_buffer_Buf* out, bool filled)
{
   string_buffer_Buf_color(out, ast_col_Error);
   if (ast_TypeRef_isConst(r)) string_buffer_Buf_add(out, "const ");
   if (ast_TypeRef_isVolatile(r)) string_buffer_Buf_add(out, "volatile ");
   if (r->flags.is_user) {
      if (r->flags.has_prefix) {
         string_buffer_Buf_add(out, ast_idx2name(r->refs[1].name_idx));
         string_buffer_Buf_add1(out, '.');
      }
      string_buffer_Buf_add(out, ast_idx2name(r->refs[0].name_idx));
   } else {
      string_buffer_Buf_add(out, ast_builtinType_names[r->flags.builtin_kind]);
   }
   for (uint32_t i = 0; (i < r->flags.num_ptrs); i++) string_buffer_Buf_add1(out, '*');
   if (r->flags.incr_array) {
      string_buffer_Buf_add(out, "[+]");
   }
   if (filled) {
      for (uint32_t i = 0; (i < r->flags.num_arrays); i++) {
         string_buffer_Buf_add1(out, '[');
         const ast_Expr* a = ast_TypeRef_getArray(r, i);
         if (a) ast_Expr_printLiteral(a, out);
         string_buffer_Buf_add1(out, ']');
      }
   }
}

static void ast_TypeRef_dump(const ast_TypeRef* r)
{
   string_buffer_Buf* out = string_buffer_create(128, ast_useColor(), 2);
   ast_TypeRef_print(r, out, true);
   string_buffer_Buf_color(out, ast_col_Normal);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static const char* ast_TypeRef_diagName(const ast_TypeRef* r)
{
   static char result[128];
   string_buffer_Buf* out = string_buffer_create_static(128, false, result);
   ast_TypeRef_print(r, out, true);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
   return result;
}

static void ast_ArrayValueList_init(ast_ArrayValueList* l, uint32_t initial_size)
{
   l->count = 0;
   l->capacity = initial_size;
   l->values = NULL;
   if (initial_size) {
      l->values = ((void*)(malloc((l->capacity * 8))));
   }
}

static void ast_ArrayValueList_free(ast_ArrayValueList* l)
{
   if (l->values) free(((void*)(l->values)));
}

static void ast_ArrayValueList_add(ast_ArrayValueList* l, ast_ArrayValue* v)
{
   if ((l->count >= l->capacity)) {
      l->capacity += 4;
      void* values2 = ((void*)(malloc((l->capacity * 8))));
      void* old = ((void*)(l->values));
      if (old) {
         memcpy(values2, old, (l->count * 8));
         free(old);
      }
      l->values = values2;
   }
   l->values[l->count] = v;
   l->count++;
}

static uint32_t ast_ArrayValueList_size(const ast_ArrayValueList* l)
{
   return l->count;
}

static ast_ArrayValue** ast_ArrayValueList_get(ast_ArrayValueList* l)
{
   return l->values;
}

static ast_AST* ast_AST_create(string_pool_Pool* auxPool, uint32_t name, ast_Module* mod, bool is_generated)
{
   ast_AST* a = calloc(1, 144);
   a->mod = mod;
   a->auxPool = auxPool;
   a->name = name;
   a->idx = ast_addAST(a);
   a->is_generated = is_generated;
   ast_ImportDeclList_init(&a->imports);
   ast_DeclList_init(&a->types, 0);
   ast_DeclList_init(&a->variables, 4);
   ast_FunctionDeclList_init(&a->functions);
   ast_StaticAssertList_init(&a->static_asserts, 0);
   ast_ArrayValueList_init(&a->array_values, 0);
   return a;
}

static void ast_AST_free(ast_AST* a)
{
   ast_ImportDeclList_free(&a->imports);
   ast_DeclList_free(&a->types);
   ast_DeclList_free(&a->variables);
   ast_FunctionDeclList_free(&a->functions);
   ast_StaticAssertList_free(&a->static_asserts);
   ast_ArrayValueList_free(&a->array_values);
   if (a->attrs) attr_table_Table_free(a->attrs);
   free(a);
}

static const char* ast_AST_getFilename(const ast_AST* a)
{
   return string_pool_Pool_idx2str(a->auxPool, a->name);
}

static uint32_t ast_AST_getIdx(const ast_AST* a)
{
   return a->idx;
}

static const char* ast_AST_getName(const ast_AST* a)
{
   return ast_Module_getName(a->mod);
}

static uint32_t ast_AST_getNameIdx(const ast_AST* a)
{
   return ast_Module_getNameIdx(a->mod);
}

static src_loc_SrcLoc ast_AST_getLoc(const ast_AST* a)
{
   ast_ImportDecl** imports = ast_ImportDeclList_getDecls(&a->imports);
   ast_Decl* d = ((ast_Decl*)(imports[0]));
   return ast_Decl_getLoc(d);
}

static void ast_AST_setPtr(ast_AST* a, void* ptr)
{
   a->ptr = ptr;
}

static void* ast_AST_getPtr(const ast_AST* a)
{
   return a->ptr;
}

static ast_Module* ast_AST_getMod(const ast_AST* a)
{
   return a->mod;
}

static void ast_AST_addImport(ast_AST* a, ast_ImportDecl* d)
{
   ast_ImportDeclList_add(&a->imports, d);
}

static ast_ImportDecl* ast_AST_findImport(const ast_AST* a, uint32_t name)
{
   ast_ImportDecl** imports = ast_ImportDeclList_getDecls(&a->imports);
   for (uint32_t i = 1; (i < ast_ImportDeclList_size(&a->imports)); i++) {
      ast_ImportDecl* d = imports[i];
      if ((ast_Decl_getNameIdx(ast_ImportDecl_asDecl(d)) == name)) return d;

   }
   return NULL;
}

static bool ast_AST_isGenerated(const ast_AST* a)
{
   return a->is_generated;
}

static void ast_AST_addFunc(ast_AST* a, ast_FunctionDecl* d)
{
   ast_FunctionDeclList_add(&a->functions, d);
}

static void ast_AST_addTypeDecl(ast_AST* a, ast_Decl* d)
{
   ast_DeclList_add(&a->types, d);
}

static void ast_AST_addVarDecl(ast_AST* a, ast_Decl* d)
{
   ast_DeclList_add(&a->variables, d);
}

static void ast_AST_addStaticAssert(ast_AST* a, ast_StaticAssert* s)
{
   ast_StaticAssertList_add(&a->static_asserts, s);
}

static void ast_AST_addArrayValue(ast_AST* a, ast_ArrayValue* v)
{
   ast_ArrayValueList_add(&a->array_values, v);
}

static void ast_AST_visitImports(const ast_AST* a, ast_ImportVisitor visitor, void* arg)
{
   ast_ImportDecl** imports = ast_ImportDeclList_getDecls(&a->imports);
   for (uint32_t i = 1; (i < ast_ImportDeclList_size(&a->imports)); i++) {
      visitor(arg, imports[i]);
   }
}

static const ast_ImportDeclList* ast_AST_getImports(const ast_AST* a)
{
   return &a->imports;
}

static void ast_AST_visitArrayValues(ast_AST* a, ast_ArrayValueVisitor visitor, void* arg)
{
   ast_ArrayValue** values = ast_ArrayValueList_get(&a->array_values);
   for (uint32_t i = 0; (i < ast_ArrayValueList_size(&a->array_values)); i++) {
      visitor(arg, values[i]);
   }
}

static void ast_AST_visitStructFunctions(const ast_AST* a, ast_FunctionVisitor visitor, void* arg)
{
   ast_FunctionDecl** functions = ast_FunctionDeclList_getDecls(&a->functions);
   for (uint32_t i = 0; (i < ast_FunctionDeclList_size(&a->functions)); i++) {
      ast_FunctionDecl* d = functions[i];
      if (ast_FunctionDecl_hasPrefix(d)) visitor(arg, d);
   }
}

static void ast_AST_visitFunctions(const ast_AST* a, ast_FunctionVisitor visitor, void* arg)
{
   ast_FunctionDecl** functions = ast_FunctionDeclList_getDecls(&a->functions);
   for (uint32_t i = 0; (i < ast_FunctionDeclList_size(&a->functions)); i++) {
      ast_FunctionDecl* d = functions[i];
      visitor(arg, d);
   }
}

static void ast_AST_visitTypeDecls(const ast_AST* a, ast_TypeDeclVisitor visitor, void* arg)
{
   ast_Decl** types = ast_DeclList_getDecls(&a->types);
   for (uint32_t i = 0; (i < ast_DeclList_size(&a->types)); i++) {
      visitor(arg, types[i]);
   }
}

static void ast_AST_visitVarDecls(const ast_AST* a, ast_VarDeclVisitor visitor, void* arg)
{
   ast_Decl** variables = ast_DeclList_getDecls(&a->variables);
   for (uint32_t i = 0; (i < ast_DeclList_size(&a->variables)); i++) {
      visitor(arg, ((ast_VarDecl*)(variables[i])));
   }
}

static void ast_AST_visitStaticAsserts(ast_AST* a, ast_StaticAssertVisitor visitor, void* arg)
{
   ast_StaticAssert** asserts = ast_StaticAssertList_get(&a->static_asserts);
   for (uint32_t i = 0; (i < ast_StaticAssertList_size(&a->static_asserts)); i++) {
      visitor(arg, asserts[i]);
   }
}

static void ast_AST_visitDecls(const ast_AST* a, ast_DeclVisitor visitor, void* arg)
{
   ast_ImportDecl** imports = ast_ImportDeclList_getDecls(&a->imports);
   for (uint32_t i = 0; (i < ast_ImportDeclList_size(&a->imports)); i++) {
      visitor(arg, ((ast_Decl*)(imports[i])));
   }
   ast_AST_visitDeclsWithoutImports(a, visitor, arg);
}

static void ast_AST_visitDeclsWithoutImports(const ast_AST* a, ast_DeclVisitor visitor, void* arg)
{
   ast_Decl** types = ast_DeclList_getDecls(&a->types);
   for (uint32_t i = 0; (i < ast_DeclList_size(&a->types)); i++) {
      visitor(arg, types[i]);
   }
   ast_Decl** variables = ast_DeclList_getDecls(&a->variables);
   for (uint32_t i = 0; (i < ast_DeclList_size(&a->variables)); i++) {
      visitor(arg, variables[i]);
   }
   ast_FunctionDecl** functions = ast_FunctionDeclList_getDecls(&a->functions);
   for (uint32_t i = 0; (i < ast_FunctionDeclList_size(&a->functions)); i++) {
      ast_FunctionDecl* d = functions[i];
      visitor(arg, ((ast_Decl*)(d)));
   }
}

static ast_Decl* ast_AST_findType(const ast_AST* a, uint32_t name_idx)
{
   ast_Decl** types = ast_DeclList_getDecls(&a->types);
   for (uint32_t i = 0; (i < ast_DeclList_size(&a->types)); i++) {
      ast_Decl* d = types[i];
      if ((ast_Decl_getNameIdx(d) == name_idx)) return d;

   }
   return NULL;
}

static void ast_AST_storeAttr(ast_AST* a, ast_Decl* d, attr_AttrKind kind, const attr_Value* value)
{
   if (!a->attrs) a->attrs = attr_table_create();
   attr_table_Table_add(a->attrs, d, kind, value);
}

static const attr_Value* ast_AST_getAttr(const ast_AST* a, const ast_Decl* d, attr_AttrKind kind)
{
   if (a->attrs) return attr_table_Table_find(a->attrs, d, kind);

   return NULL;
}

static void ast_AST_info(const ast_AST* a, string_buffer_Buf* out)
{
   string_buffer_Buf_print(out, "    %s\n", ast_AST_getFilename(a));
}

static void ast_AST_print(const ast_AST* a, string_buffer_Buf* out, bool show_funcs)
{
   string_buffer_Buf_print(out, "---- AST %s ----\n", ast_AST_getFilename(a));
   ast_ImportDecl** imports = ast_ImportDeclList_getDecls(&a->imports);
   for (uint32_t i = 1; (i < ast_ImportDeclList_size(&a->imports)); i++) {
      ast_ImportDecl_print(imports[i], out, 0);
   }
   if ((ast_ImportDeclList_size(&a->imports) > 1)) string_buffer_Buf_newline(out);
   ast_Decl** types = ast_DeclList_getDecls(&a->types);
   for (uint32_t i = 0; (i < ast_DeclList_size(&a->types)); i++) {
      ast_Decl_print(types[i], out, 0);
      string_buffer_Buf_newline(out);
   }
   ast_Decl** variables = ast_DeclList_getDecls(&a->variables);
   for (uint32_t i = 0; (i < ast_DeclList_size(&a->variables)); i++) {
      ast_Decl_print(variables[i], out, 0);
      string_buffer_Buf_newline(out);
   }
   if (show_funcs) {
      ast_FunctionDecl** functions = ast_FunctionDeclList_getDecls(&a->functions);
      for (uint32_t i = 0; (i < ast_FunctionDeclList_size(&a->functions)); i++) {
         ast_FunctionDecl_print(functions[i], out, 0);
         string_buffer_Buf_newline(out);
      }
   }
   for (uint32_t i = 0; (i < ast_StaticAssertList_size(&a->static_asserts)); i++) {
      ast_StaticAssert_print(ast_StaticAssertList_getAt(&a->static_asserts, i), out, 0);
      string_buffer_Buf_newline(out);
   }
}

static void ast_AST_setExported(ast_AST* a)
{
   ast_Decl** types = ast_DeclList_getDecls(&a->types);
   for (uint32_t i = 0; (i < ast_DeclList_size(&a->types)); i++) {
      ast_Decl_setExportedIfPublic(types[i]);
   }
   ast_Decl** variables = ast_DeclList_getDecls(&a->variables);
   for (uint32_t i = 0; (i < ast_DeclList_size(&a->variables)); i++) {
      ast_Decl_setExportedIfPublic(variables[i]);
   }
   ast_Decl** functions = ((ast_Decl**)(ast_FunctionDeclList_getDecls(&a->functions)));
   for (uint32_t i = 0; (i < ast_FunctionDeclList_size(&a->functions)); i++) {
      ast_Decl_setExportedIfPublic(functions[i]);
   }
}

static void ast_DeclList_init(ast_DeclList* l, uint32_t initial_size)
{
   memset(l, 0, 16);
   if (initial_size) {
      l->capacity = initial_size;
      l->decls = malloc((l->capacity * 8));
   }
}

static void ast_DeclList_free(ast_DeclList* l)
{
   if (l->decls) free(((void*)(l->decls)));
}

static void ast_DeclList_add(ast_DeclList* l, ast_Decl* d)
{
   if ((l->count >= l->capacity)) {
      l->capacity += 4;
      void* decls2 = malloc((l->capacity * 8));
      void* old = ((void*)(l->decls));
      if (old) {
         memcpy(decls2, old, (l->count * 8));
         free(old);
      }
      l->decls = decls2;
   }
   l->decls[l->count] = d;
   l->count++;
}

static void ast_DeclList_clear(ast_DeclList* l)
{
   l->count = 0;
}

static uint32_t ast_DeclList_size(const ast_DeclList* l)
{
   return l->count;
}

static ast_Decl* ast_DeclList_get(const ast_DeclList* l, uint32_t idx)
{
   return l->decls[idx];
}

static ast_Decl** ast_DeclList_getDecls(const ast_DeclList* l)
{
   return l->decls;
}

static void ast_DeclList_setSize(ast_DeclList* l, uint32_t size)
{
   l->count = size;
}

static uint32_t ast_DeclList_findIdx(const ast_DeclList* l, const ast_Decl* d)
{
   for (uint32_t i = 0; (i < l->count); i++) {
      if ((l->decls[i] == d)) return i;

   }
   return c2_max_u32;
}

static void ast_ExprList_init(ast_ExprList* l, uint32_t initial_size)
{
   memset(l, 0, 16);
   if (initial_size) {
      l->capacity = initial_size;
      l->exprs = malloc((l->capacity * 8));
   }
}

static void ast_ExprList_free(ast_ExprList* l)
{
   if (l->exprs) free(((void*)(l->exprs)));
}

static void ast_ExprList_add(ast_ExprList* l, ast_Expr* d)
{
   if ((l->count >= l->capacity)) {
      l->capacity += 4;
      void* exprs2 = malloc((l->capacity * 8));
      void* old = ((void*)(l->exprs));
      if (old) {
         memcpy(exprs2, old, (l->count * 8));
         free(old);
      }
      l->exprs = exprs2;
   }
   l->exprs[l->count] = d;
   l->count++;
}

static uint32_t ast_ExprList_size(const ast_ExprList* l)
{
   return l->count;
}

static ast_Expr** ast_ExprList_getExprs(const ast_ExprList* l)
{
   return l->exprs;
}

static void ast_FunctionDeclList_init(ast_FunctionDeclList* l)
{
   memset(l, 0, 16);
}

static void ast_FunctionDeclList_free(ast_FunctionDeclList* l)
{
   if (l->decls) free(((void*)(l->decls)));
}

static void ast_FunctionDeclList_clear(ast_FunctionDeclList* l)
{
   l->count = 0;
}

static void ast_FunctionDeclList_add(ast_FunctionDeclList* l, ast_FunctionDecl* d)
{
   if ((l->count >= l->capacity)) {
      l->capacity = ((l->capacity == 0)) ? 4 : (l->capacity * 2);
      void* decls2 = malloc((l->capacity * 8));
      void* old = ((void*)(l->decls));
      if (old) {
         memcpy(decls2, old, (l->count * 8));
         free(old);
      }
      l->decls = decls2;
   }
   l->decls[l->count] = d;
   l->count++;
}

static uint32_t ast_FunctionDeclList_size(const ast_FunctionDeclList* l)
{
   return l->count;
}

static ast_FunctionDecl** ast_FunctionDeclList_getDecls(const ast_FunctionDeclList* l)
{
   return l->decls;
}

static ast_FunctionDecl* ast_FunctionDeclList_find(const ast_FunctionDeclList* l, uint32_t name_idx)
{
   for (uint32_t i = 0; (i < l->count); i++) {
      ast_FunctionDecl* fd = l->decls[i];
      if ((ast_Decl_getNameIdx(ast_FunctionDecl_asDecl(fd)) == name_idx)) return fd;

   }
   return NULL;
}

static void ast_ImportDeclList_init(ast_ImportDeclList* l)
{
   memset(l, 0, 16);
}

static void ast_ImportDeclList_free(ast_ImportDeclList* l)
{
   if (l->decls) free(((void*)(l->decls)));
}

static void ast_ImportDeclList_add(ast_ImportDeclList* l, ast_ImportDecl* d)
{
   if ((l->count >= l->capacity)) {
      l->capacity += 4;
      void* decls2 = malloc((l->capacity * 8));
      void* old = ((void*)(l->decls));
      if (old) {
         memcpy(decls2, old, (l->count * 8));
         free(old);
      }
      l->decls = decls2;
   }
   l->decls[l->count] = d;
   l->count++;
}

static uint32_t ast_ImportDeclList_size(const ast_ImportDeclList* l)
{
   return l->count;
}

static ast_ImportDecl** ast_ImportDeclList_getDecls(const ast_ImportDeclList* l)
{
   return l->decls;
}

static ast_ImportDecl* ast_ImportDeclList_find(const ast_ImportDeclList* l, uint32_t name_idx)
{
   for (uint32_t i = 0; (i < l->count); i++) {
      ast_ImportDecl* d = l->decls[i];
      if ((ast_ImportDecl_getImportNameIdx(d) == name_idx)) return d;

   }
   return NULL;
}

static ast_ImportDecl* ast_ImportDeclList_findAny(const ast_ImportDeclList* l, uint32_t name_idx)
{
   for (uint32_t i = 0; (i < l->count); i++) {
      ast_ImportDecl* d = l->decls[i];
      if ((ast_Decl_getNameIdx(ast_ImportDecl_asDecl(d)) == name_idx)) return d;

   }
   return NULL;
}

static void ast_TemplateFunction_init(ast_TemplateFunction* f, const ast_FunctionDecl* fd)
{
   f->fd = fd;
   f->count = 0;
   f->capacity = 0;
   f->instances = NULL;
   ast_TemplateFunction_resize(f, 2);
}

static void ast_TemplateFunction_resize(ast_TemplateFunction* f, uint16_t capacity)
{
   f->capacity = capacity;
   ast_TemplateInstance* inst2 = malloc((capacity * 16));
   if (f->count) {
      memcpy(inst2, f->instances, (f->count * 16));
      free(f->instances);
   }
   f->instances = inst2;
}

static uint16_t ast_TemplateFunction_add(ast_TemplateFunction* f, ast_QualType qt, ast_FunctionDecl* instance)
{
   if ((f->count == f->capacity)) ast_TemplateFunction_resize(f, (f->capacity * 2));
   uint16_t idx = f->count;
   ast_TemplateInstance* ti = &f->instances[idx];
   f->count++;
   ti->qt = qt;
   ti->instance = instance;
   return idx;
}

static ast_FunctionDecl* ast_TemplateFunction_find(const ast_TemplateFunction* f, ast_QualType qt)
{
   for (uint32_t i = 0; (i < f->count); i++) {
      const ast_TemplateInstance* ti = &f->instances[i];
      if ((ti->qt.ptr == qt.ptr)) return ti->instance;

   }
   return NULL;
}

static ast_FunctionDecl* ast_TemplateFunction_get(const ast_TemplateFunction* f, uint32_t idx)
{
   return f->instances[idx].instance;
}

static void ast_InstanceTable_init(ast_InstanceTable* t)
{
   t->count = 0;
   t->capacity = 0;
   t->funcs = NULL;
}

static void ast_InstanceTable_free(ast_InstanceTable* t)
{
   for (uint32_t i = 0; (i < t->count); i++) {
      free(t->funcs[i].instances);
   }
   free(t->funcs);
}

static void ast_InstanceTable_resize(ast_InstanceTable* t, uint32_t capacity)
{
   t->capacity = capacity;
   ast_TemplateFunction* funcs2 = malloc((capacity * 24));
   if (t->count) {
      memcpy(funcs2, t->funcs, (t->count * 24));
      free(t->funcs);
   }
   t->funcs = funcs2;
}

static ast_TemplateFunction* ast_InstanceTable_findFunc(const ast_InstanceTable* t, const ast_FunctionDecl* fd)
{
   for (uint32_t i = 0; (i < t->count); i++) {
      ast_TemplateFunction* fi = &t->funcs[i];
      if ((fi->fd == fd)) return fi;

   }
   return NULL;
}

static ast_FunctionDecl* ast_InstanceTable_find(const ast_InstanceTable* t, const ast_FunctionDecl* fd, ast_QualType qt)
{
   const ast_TemplateFunction* fi = ast_InstanceTable_findFunc(t, fd);
   if (fi) return ast_TemplateFunction_find(fi, qt);

   return NULL;
}

static ast_FunctionDecl* ast_InstanceTable_get(const ast_InstanceTable* t, const ast_FunctionDecl* fd, uint32_t idx)
{
   const ast_TemplateFunction* fi = ast_InstanceTable_findFunc(t, fd);
   c2_assert((fi) != NULL, "ast/instance_table.c2:120: ast.InstanceTable.get", "fi");
   return ast_TemplateFunction_get(fi, idx);
}

static uint16_t ast_InstanceTable_add(ast_InstanceTable* t, const ast_FunctionDecl* fd, ast_QualType qt, ast_FunctionDecl* instance)
{
   ast_TemplateFunction* fi = ast_InstanceTable_findFunc(t, fd);
   if (!fi) {
      if ((t->count == t->capacity)) {
         if ((t->capacity == 0)) t->capacity = 2;
         ast_InstanceTable_resize(t, (t->capacity * 2));
      }
      fi = &t->funcs[t->count];
      t->count++;
      ast_TemplateFunction_init(fi, fd);
   }
   return ast_TemplateFunction_add(fi, qt, instance);
}

static void ast_InstanceTable_visit(const ast_InstanceTable* t, const ast_FunctionDecl* fd, ast_TemplateVisitor visitor, void* arg)
{
   ast_TemplateFunction* fi = ast_InstanceTable_findFunc(t, fd);
   if (!fi) return;

   for (uint32_t i = 0; (i < fi->count); i++) {
      visitor(arg, fi->instances[i].instance, (i + 1));
   }
}

static void ast_Instantiator_on_opaque(ast_Instantiator* inst, src_loc_SrcLoc loc, ast_Decl* decl)
{
   inst->on_error(inst->arg, loc, decl);
}

static ast_Module* ast_Module_create(ast_context_Context* c, uint32_t name_idx, bool is_external, bool is_direct)
{
   ast_Module* m = calloc(1, 88);
   m->mt = ast_ModuleType_create(c, m);
   m->name_idx = name_idx;
   m->is_external = is_external;
   m->is_internal = false;
   m->is_direct = is_direct;
   ast_Module_resizeFiles(m, 1);
   ast_SymbolTable_init(&m->symbols, 16);
   ast_InstanceTable_init(&m->instances);
   return m;
}

static void ast_Module_free(ast_Module* m)
{
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_free(m->files[i]);
   }
   free(((void*)(m->files)));
   ast_SymbolTable_free(&m->symbols);
   ast_InstanceTable_free(&m->instances);
   free(m);
}

static void ast_Module_setUsed(ast_Module* m)
{
   m->is_used = true;
}

static bool ast_Module_isUsed(const ast_Module* m)
{
   return m->is_used;
}

static void ast_Module_setInternal(ast_Module* m)
{
   m->is_internal = true;
}

static bool ast_Module_isInternal(const ast_Module* m)
{
   return m->is_internal;
}

static bool ast_Module_isExternal(const ast_Module* m)
{
   return m->is_external;
}

static void ast_Module_setExported(ast_Module* m)
{
   m->is_exported = true;
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_setExported(m->files[i]);
   }
}

static bool ast_Module_isExported(const ast_Module* m)
{
   return m->is_exported;
}

static bool ast_Module_isDirect(const ast_Module* m)
{
   return m->is_direct;
}

static const ast_SymbolTable* ast_Module_getSymbols(const ast_Module* m)
{
   return &m->symbols;
}

static ast_ModuleType* ast_Module_getType(const ast_Module* m)
{
   return m->mt;
}

static const char* ast_Module_getFirstFilename(const ast_Module* m)
{
   if (m->num_files) return ast_AST_getFilename(m->files[0]);

   return NULL;
}

static void ast_Module_visitASTs(const ast_Module* m, ast_ASTVisitor visitor, void* arg)
{
   for (uint32_t i = 0; (i < m->num_files); i++) {
      visitor(arg, m->files[i]);
   }
}

static void ast_Module_visitImports(const ast_Module* m, ast_ImportVisitor visitor, void* arg)
{
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_visitImports(m->files[i], visitor, arg);
   }
}

static void ast_Module_visitArrayValues(const ast_Module* m, ast_ArrayValueVisitor visitor, void* arg)
{
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_visitArrayValues(m->files[i], visitor, arg);
   }
}

static void ast_Module_visitStructFunctions(const ast_Module* m, ast_FunctionVisitor visitor, void* arg)
{
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_visitStructFunctions(m->files[i], visitor, arg);
   }
}

static void ast_Module_visitFunctions(const ast_Module* m, ast_FunctionVisitor visitor, void* arg)
{
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_visitFunctions(m->files[i], visitor, arg);
   }
}

static void ast_Module_visitTypeDecls(const ast_Module* m, ast_TypeDeclVisitor visitor, void* arg)
{
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_visitTypeDecls(m->files[i], visitor, arg);
   }
}

static void ast_Module_visitVarDecls(const ast_Module* m, ast_VarDeclVisitor visitor, void* arg)
{
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_visitVarDecls(m->files[i], visitor, arg);
   }
}

static void ast_Module_visitStaticAsserts(const ast_Module* m, ast_StaticAssertVisitor visitor, void* arg)
{
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_visitStaticAsserts(m->files[i], visitor, arg);
   }
}

static void ast_Module_visitDecls(const ast_Module* m, ast_DeclVisitor visitor, void* arg)
{
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_visitDecls(m->files[i], visitor, arg);
   }
}

static ast_Decl* ast_Module_findType(const ast_Module* m, uint32_t name_idx)
{
   ast_Decl* result = NULL;
   for (uint32_t i = 0; (i < m->num_files); i++) {
      result = ast_AST_findType(m->files[i], name_idx);
      if (result) break;

   }
   return result;
}

static const char* ast_Module_getName(const ast_Module* m)
{
   return ast_idx2name(m->name_idx);
}

static uint32_t ast_Module_getNameIdx(const ast_Module* m)
{
   return m->name_idx;
}

static void ast_Module_resizeFiles(ast_Module* m, uint32_t cap)
{
   m->max_files = cap;
   void* buf = malloc((m->max_files * 8));
   if (m->files) {
      void* old = ((void*)(m->files));
      memcpy(buf, old, (m->num_files * 8));
      free(old);
   }
   m->files = buf;
}

static ast_AST* ast_Module_add(ast_Module* m, string_pool_Pool* auxPool, uint32_t filename, bool is_generated)
{
   ast_AST* a = ast_AST_create(auxPool, filename, m, is_generated);
   if ((m->num_files == m->max_files)) ast_Module_resizeFiles(m, (m->max_files * 2));
   m->files[m->num_files] = a;
   m->num_files++;
   return a;
}

static void ast_Module_addSymbol(ast_Module* m, uint32_t name_idx, ast_Decl* d)
{
   ast_SymbolTable_add(&m->symbols, name_idx, d);
}

static ast_Decl* ast_Module_findSymbol(const ast_Module* m, uint32_t name_idx)
{
   return ast_SymbolTable_find(&m->symbols, name_idx);
}

static ast_FunctionDecl* ast_Module_findInstance(const ast_Module* m, ast_FunctionDecl* fd, ast_QualType qt)
{
   return ast_InstanceTable_find(&m->instances, fd, qt);
}

static uint16_t ast_Module_addInstance(ast_Module* m, ast_FunctionDecl* fd, ast_QualType qt, ast_FunctionDecl* instance)
{
   return ast_InstanceTable_add(&m->instances, fd, qt, instance);
}

static ast_FunctionDecl* ast_Module_getInstance(const ast_Module* m, ast_FunctionDecl* fd, uint32_t idx)
{
   return ast_InstanceTable_get(&m->instances, fd, idx);
}

static void ast_Module_visitInstances(const ast_Module* m, ast_FunctionDecl* fd, ast_TemplateVisitor visitor, void* arg)
{
   ast_InstanceTable_visit(&m->instances, fd, visitor, arg);
}

static void ast_Module_info(const ast_Module* m, string_buffer_Buf* out)
{
   string_buffer_Buf_print(out, "  module %s\n", ast_idx2name(m->name_idx));
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_info(m->files[i], out);
   }
}

static void ast_Module_print(const ast_Module* m, string_buffer_Buf* out, bool show_funcs)
{
   string_buffer_Buf_color(out, color_Normal);
   string_buffer_Buf_print(out, "------ module %s (used %u, exported %u) ------\n", ast_idx2name(m->name_idx), m->is_used, m->is_exported);
   for (uint32_t i = 0; (i < m->num_files); i++) {
      ast_AST_print(m->files[i], out, show_funcs);
   }
}

static void ast_PointerPool_init(ast_PointerPool* p, ast_context_Context* c)
{
   p->count = 1;
   p->capacity = 0;
   p->slots = NULL;
   p->context = c;
   ast_PointerPool_resize(p, 64);
}

static void ast_PointerPool_clear(ast_PointerPool* p)
{
   free(p->slots);
   p->count = 1;
   p->capacity = 0;
   p->slots = NULL;
}

static void ast_PointerPool_resize(ast_PointerPool* p, uint32_t cap)
{
   p->capacity = cap;
   ast_PointerPoolSlot* slots2 = malloc((p->capacity * 32));
   if ((p->count > 1)) {
      memcpy(slots2, p->slots, (p->count * 32));
      free(p->slots);
   }
   p->slots = slots2;
}

static ast_Type* ast_PointerPool_getPointer(ast_PointerPool* p, ast_QualType qt)
{
   ast_Type* t = ast_QualType_getTypeOrNil(&qt);
   c2_assert((t) != NULL, "ast/pointer_pool.c2:70: ast.PointerPool.getPointer", "t");
   const uint32_t ptr_pool_idx = t->ptr_pool_idx;
   ast_PointerPoolSlot* slot = &p->slots[ptr_pool_idx];
   if ((ptr_pool_idx == 0)) {
      uint32_t slot_idx = p->count;
      if ((slot_idx == p->capacity)) ast_PointerPool_resize(p, (p->capacity * 2));
      slot = &p->slots[slot_idx];
      memset(slot, 0, 32);
      t->ptr_pool_idx = slot_idx;
      p->count++;
   }
   uint32_t quals = ast_QualType_getQuals(&qt);
   ast_Type* ptr = slot->ptrs[quals];
   if (ptr) return ptr;

   ptr = ((ast_Type*)(ast_PointerType_create(p->context, qt)));
   slot->ptrs[quals] = ptr;
   return ptr;
}

static void ast_StaticAssertList_init(ast_StaticAssertList* l, uint32_t initial_size)
{
   l->count = 0;
   l->capacity = initial_size;
   l->asserts = NULL;
   if (initial_size) {
      l->asserts = ((void*)(malloc((l->capacity * 8))));
   }
}

static void ast_StaticAssertList_free(ast_StaticAssertList* l)
{
   if (l->asserts) free(((void*)(l->asserts)));
}

static void ast_StaticAssertList_add(ast_StaticAssertList* l, ast_StaticAssert* v)
{
   if ((l->count >= l->capacity)) {
      l->capacity += 4;
      void* asserts2 = ((void*)(malloc((l->capacity * 8))));
      void* old = ((void*)(l->asserts));
      if (old) {
         memcpy(asserts2, old, (l->count * 8));
         free(old);
      }
      l->asserts = asserts2;
   }
   l->asserts[l->count] = v;
   l->count++;
}

static uint32_t ast_StaticAssertList_size(const ast_StaticAssertList* l)
{
   return l->count;
}

static ast_StaticAssert* ast_StaticAssertList_getAt(const ast_StaticAssertList* l, uint32_t idx)
{
   return l->asserts[idx];
}

static ast_StaticAssert** ast_StaticAssertList_get(ast_StaticAssertList* l)
{
   return l->asserts;
}

static void ast_Stats_reset(ast_Stats* s)
{
   memset(s, 0, 472);
}

static void ast_Stats_addType(ast_TypeKind kind, uint32_t size)
{
   ast_globals->stats.types[kind].count++;
   ast_globals->stats.types[kind].size += size;
}

static void ast_Stats_addExpr(ast_ExprKind kind, uint32_t size)
{
   ast_globals->stats.exprs[kind].count++;
   ast_globals->stats.exprs[kind].size += size;
}

static void ast_Stats_addStmt(ast_StmtKind kind, uint32_t size)
{
   ast_globals->stats.stmts[kind].count++;
   ast_globals->stats.stmts[kind].size += size;
}

static void ast_Stats_addDecl(ast_DeclKind kind, uint32_t size)
{
   ast_globals->stats.decls[kind].count++;
   ast_globals->stats.decls[kind].size += size;
}

static void ast_Stats_addArrayValue(uint32_t size)
{
   ast_globals->stats.others[0].count++;
   ast_globals->stats.others[0].size += size;
}

static void ast_Stats_addStaticAssert(uint32_t size)
{
   ast_globals->stats.others[1].count++;
   ast_globals->stats.others[1].size += size;
}

static void ast_Stats_addSwitchCase(uint32_t size)
{
   ast_globals->stats.others[2].count++;
   ast_globals->stats.others[2].size += size;
}

static void ast_Stats_dump(const ast_Stats* s)
{
   printf("---------------------------------------\n");
   printf("--- Types ---\n");
   uint32_t typesTotal = 0;
   uint32_t typesCount = 0;
   for (uint32_t i = 0; (i <= 7); i++) {
      const ast_Stat* ss = &s->types[i];
      printf("  %20s  %6u  %7u\n", ast_typeKind_names[i], ss->count, ss->size);
      typesCount += ss->count;
      typesTotal += ss->size;
   }
   printf("  %20s  %6u  %7u\n", "total", typesCount, typesTotal);
   printf("--- Expressions ---\n");
   uint32_t exprTotal = 0;
   uint32_t exprCount = 0;
   for (uint32_t i = 0; (i <= 21); i++) {
      const ast_Stat* ss = &s->exprs[i];
      printf("  %20s  %6u  %7u\n", ast_exprKind_names[i], ss->count, ss->size);
      exprCount += ss->count;
      exprTotal += ss->size;
   }
   printf("  %20s  %6u  %7u\n", "total", exprCount, exprTotal);
   printf("--- Statements ---\n");
   uint32_t stmtTotal = 0;
   uint32_t stmtCount = 0;
   for (uint32_t i = 0; (i <= 15); i++) {
      const ast_Stat* ss = &s->stmts[i];
      printf("  %20s  %6u  %7u\n", ast_stmtKind_names[i], ss->count, ss->size);
      stmtCount += ss->count;
      stmtTotal += ss->size;
   }
   printf("  %20s  %6u  %7u\n", "total", stmtCount, stmtTotal);
   printf("--- Decls ---\n");
   uint32_t declTotal = 0;
   uint32_t declCount = 0;
   for (uint32_t i = 0; (i <= 7); i++) {
      const ast_Stat* ss = &s->decls[i];
      printf("  %20s  %6u  %7u\n", ast_declKind_names[i], ss->count, ss->size);
      declCount += ss->count;
      declTotal += ss->size;
   }
   printf("  %20s  %6u  %7u\n", "total", declCount, declTotal);
   printf("--- Other ---\n");
   uint32_t otherTotal = 0;
   uint32_t otherCount = 0;
   for (uint32_t i = 0; (i < 3); i++) {
      const ast_Stat* ss = &s->others[i];
      printf("  %20s  %6u  %7u\n", ast_other_names[i], ss->count, ss->size);
      otherCount += ss->count;
      otherTotal += ss->size;
   }
   printf("  %20s  %6u  %7u\n", "total", otherCount, otherTotal);
   printf("--- Total ---\n");
   uint32_t totalCount = ((((typesCount + exprCount) + stmtCount) + declCount) + otherCount);
   uint32_t totalSize = ((((typesTotal + exprTotal) + stmtTotal) + declTotal) + otherTotal);
   printf("  %20s  %6u  %7u\n", "objects", totalCount, totalSize);
   printf("---------------------------------------\n");
}

static void ast_StringTypePool_init(ast_StringTypePool* p, ast_context_Context* c)
{
   p->count = 0;
   p->capacity = 0;
   p->slots = NULL;
   p->context = c;
   ast_StringTypePool_resize(p, 8);
}

static void ast_StringTypePool_clear(ast_StringTypePool* p)
{
   free(p->slots);
   p->count = 0;
   p->capacity = 0;
   p->slots = NULL;
}

static void ast_StringTypePool_resize(ast_StringTypePool* p, uint32_t cap)
{
   p->capacity = cap;
   ast_StringTypeSlot* slots2 = malloc((p->capacity * 16));
   if (p->count) {
      memcpy(slots2, p->slots, (p->count * 16));
      free(p->slots);
   }
   p->slots = slots2;
}

static ast_QualType ast_StringTypePool_get(ast_StringTypePool* p, uint32_t len)
{
   for (uint32_t i = 0; (i < p->count); i++) {
      ast_StringTypeSlot* s = &p->slots[i];
      if ((s->len == len)) return ast_QualType_init(s->type_);

   }
   if ((p->count == p->capacity)) ast_StringTypePool_resize(p, (p->capacity * 2));
   ast_Type* t = ((ast_Type*)(ast_ArrayType_create(p->context, ast_builtins[ast_BuiltinKind_Char], true, len)));
   uint32_t idx = p->count;
   p->slots[idx].len = len;
   p->slots[idx].type_ = t;
   p->count++;
   ast_QualType qt = ast_QualType_init(t);
   ast_Type_setCanonicalType(t, qt);
   return qt;
}

static void ast_SymbolTable_init(ast_SymbolTable* t, uint32_t initial)
{
   t->num_symbols = 0;
   t->max_symbols = 0;
   ast_SymbolTable_resize(t, initial);
   ast_DeclList_init(&t->decls, initial);
}

static void ast_SymbolTable_free(ast_SymbolTable* t)
{
   free(t->symbols);
   ast_DeclList_free(&t->decls);
}

static uint32_t ast_SymbolTable_size(const ast_SymbolTable* t)
{
   return t->num_symbols;
}

static ast_Decl** ast_SymbolTable_getDecls(const ast_SymbolTable* l)
{
   return ast_DeclList_getDecls(&l->decls);
}

static void ast_SymbolTable_crop(ast_SymbolTable* t, uint32_t size)
{
   t->num_symbols = size;
   ast_DeclList_setSize(&t->decls, size);
}

static void ast_SymbolTable_resize(ast_SymbolTable* t, uint32_t capacity)
{
   uint32_t* symbols = malloc((capacity * 4));
   t->max_symbols = capacity;
   if (t->symbols) {
      memcpy(symbols, t->symbols, (t->num_symbols * 4));
      free(t->symbols);
   }
   t->symbols = symbols;
}

static void ast_SymbolTable_add(ast_SymbolTable* t, uint32_t name_idx, ast_Decl* d)
{
   if ((t->num_symbols == t->max_symbols)) ast_SymbolTable_resize(t, (t->max_symbols * 2));
   t->symbols[t->num_symbols] = name_idx;
   t->num_symbols++;
   ast_DeclList_add(&t->decls, d);
}

static ast_Decl* ast_SymbolTable_find(const ast_SymbolTable* t, uint32_t name_idx)
{
   for (uint32_t i = 0; (i < t->num_symbols); i++) {
      if ((t->symbols[i] == name_idx)) return ast_DeclList_get(&t->decls, i);

   }
   return NULL;
}

static void ast_SymbolTable_print(const ast_SymbolTable* t, string_buffer_Buf* out)
{
   for (uint32_t i = 0; (i < t->num_symbols); i++) {
      const ast_Decl* d = ast_DeclList_get(&t->decls, i);
      const char* col = ast_Decl_isUsed(d) ? color_Normal : color_Grey;
      string_buffer_Buf_color(out, col);
      const char* name = ast_idx2name(t->symbols[i]);
      string_buffer_Buf_print(out, "    %s", name);
      if (ast_Decl_isPublic(d)) {
         string_buffer_Buf_color(out, color_Yellow);
         string_buffer_Buf_add(out, " public");
      }
      string_buffer_Buf_newline(out);
      if (ast_Decl_isStructType(d)) {
         ast_StructTypeDecl* std = ((ast_StructTypeDecl*)(d));
         const ast_FunctionDecl** fds = ast_StructTypeDecl_getStructFunctions(std);
         for (uint32_t j = 0; (j < ast_StructTypeDecl_getNumStructFunctions(std)); j++) {
            ast_Decl* fd = ((ast_Decl*)(fds[j]));
            col = ast_Decl_isUsed(fd) ? color_Normal : color_Grey;
            string_buffer_Buf_color(out, col);
            string_buffer_Buf_indent(out, 6);
            string_buffer_Buf_print(out, "%s.%s", name, ast_Decl_getName(fd));
            if (ast_Decl_isPublic(fd)) {
               string_buffer_Buf_color(out, color_Yellow);
               string_buffer_Buf_add(out, " public");
            }
            string_buffer_Buf_newline(out);
         }
      }
   }
}

static void ast_SymbolTable_dump(const ast_SymbolTable* t)
{
   string_buffer_Buf* out = string_buffer_create(4096, ast_useColor(), 2);
   string_buffer_Buf_add(out, "Symbols:\n");
   for (uint32_t i = 0; (i < t->num_symbols); i++) {
      const ast_Decl* d = ast_DeclList_get(&t->decls, i);
      uint32_t name_idx = t->symbols[i];
      const char* name = ast_idx2name(name_idx);
      string_buffer_Buf_print(out, "  [%2u]  %6u  %s\n", i, name_idx, name);
   }
   string_buffer_Buf_stripNewline(out);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static ast_Globals* ast_getGlobals(void)
{
   return ast_globals;
}

static void ast_setGlobals(ast_Globals* g)
{
   ast_globals = g;
   attr_init(g->attr_name_indexes);
}

static void ast_init(ast_context_Context* c, string_pool_Pool* astPool, uint32_t wordsize, bool use_color)
{
   ast_globals = malloc(768);
   ast_PointerPool_init(&ast_globals->pointers, c);
   ast_StringTypePool_init(&ast_globals->string_types, c);
   ast_globals->wordsize = wordsize;
   ast_globals->use_color = use_color;
   ast_globals->names_start = string_pool_Pool_getStart(astPool);
   ast_globals->ast_count = 1;
   ast_globals->ast_capacity = 0;
   ast_globals->ast_list = NULL;
   ast_Stats_reset(&ast_globals->stats);
   ast_builtins = malloc((((15 + 1)) * 8));
   for (uint32_t i = 0; (i < 15); i++) {
      ast_QualType_set(&ast_builtins[i], ((ast_Type*)(ast_BuiltinType_create(c, ((ast_BuiltinKind)(i))))));
   }
   ast_Type* void_ptr = ast_getPointerType(ast_builtins[ast_BuiltinKind_Void]);
   ast_QualType_set(&ast_builtins[15], void_ptr);
   ast_Type_setCanonicalType(void_ptr, ast_builtins[15]);
   memcpy(ast_globals->builtinType_sizes, ast_BuiltinType_default_sizes, 60);
   ast_globals->builtinType_sizes[ast_BuiltinKind_ISize] = wordsize;
   ast_globals->builtinType_sizes[ast_BuiltinKind_USize] = wordsize;
   memcpy(ast_globals->builtinType_width, ast_BuiltinType_default_widths, 60);
   ast_globals->builtinType_width[ast_BuiltinKind_ISize] = ((wordsize * 8) - 1);
   ast_globals->builtinType_width[ast_BuiltinKind_USize] = (wordsize * 8);
   for (uint32_t i = 0; (i < 15); i++) {
      ast_globals->builtinType_baseTypes[i] = ((ast_BuiltinKind)(i));
   }
   if ((wordsize == 4)) {
      ast_globals->builtinType_baseTypes[ast_BuiltinKind_ISize] = ast_BuiltinKind_Int32;
      ast_globals->builtinType_baseTypes[ast_BuiltinKind_USize] = ast_BuiltinKind_UInt32;
   } else {
      ast_globals->builtinType_baseTypes[ast_BuiltinKind_ISize] = ast_BuiltinKind_Int64;
      ast_globals->builtinType_baseTypes[ast_BuiltinKind_USize] = ast_BuiltinKind_UInt64;
   }
   attr_register(astPool, ast_globals->attr_name_indexes);
   attr_init(ast_globals->attr_name_indexes);
}

static void ast_deinit(bool print_stats)
{
   if (print_stats) ast_Stats_dump(&ast_globals->stats);
   ast_globals->names_start = NULL;
   ast_globals->ast_count = 0;
   ast_globals->ast_capacity = 0;
   free(((void*)(ast_globals->ast_list)));
   ast_globals->ast_list = NULL;
   ast_PointerPool_clear(&ast_globals->pointers);
   ast_StringTypePool_clear(&ast_globals->string_types);
   free(ast_globals);
   free(ast_builtins);
}

static uint32_t ast_getWordSize(void)
{
   return ast_globals->wordsize;
}

static bool ast_useColor(void)
{
   return ast_globals->use_color;
}

static ast_QualType ast_getStringType(uint32_t len)
{
   return ast_StringTypePool_get(&ast_globals->string_types, len);
}

static const char* ast_idx2name(uint32_t idx)
{
   if (idx) return (ast_globals->names_start + idx);

   return NULL;
}

static ast_QualType ast_getVoidPtr(void)
{
   return ast_builtins[15];
}

static ast_Type* ast_getPointerType(ast_QualType inner)
{
   return ast_PointerPool_getPointer(&ast_globals->pointers, inner);
}

static uint32_t ast_addAST(ast_AST* ast_)
{
   if ((ast_globals->ast_count >= ast_globals->ast_capacity)) {
      if ((ast_globals->ast_capacity == 0)) ast_globals->ast_capacity = 16;
      else ast_globals->ast_capacity *= 2;
      void* buf = malloc((ast_globals->ast_capacity * 8));
      if (ast_globals->ast_list) {
         void* old = ((void*)(ast_globals->ast_list));
         memcpy(buf, old, (ast_globals->ast_count * 8));
         free(old);
      }
      ast_globals->ast_list = buf;
   }
   uint32_t idx = ast_globals->ast_count;
   ast_globals->ast_list[idx] = ast_;
   ast_globals->ast_count++;
   return idx;
}

static uint32_t ast_ast2idx(const ast_AST* ast_)
{
   if (ast_) return ast_->idx;

   return 0;
}

static ast_AST* ast_idx2ast(uint32_t idx)
{
   if ((idx == 0)) return NULL;

   return ast_globals->ast_list[idx];
}

static void ast_setTypePublicUsed(ast_QualType qt)
{
   const ast_Type* t = ast_QualType_getType(&qt);
   ast_Decl* d = NULL;
   switch (ast_Type_getKind(t)) {
   case ast_TypeKind_Builtin:
      return;
   case ast_TypeKind_Pointer: {
      ast_PointerType* pt = ((ast_PointerType*)(t));
      ast_setTypePublicUsed(pt->inner);
      return;
   }
   case ast_TypeKind_Array: {
      ast_ArrayType* at = ((ast_ArrayType*)(t));
      ast_setTypePublicUsed(at->elem);
      return;
   }
   case ast_TypeKind_Struct: {
      ast_StructType* st = ((ast_StructType*)(t));
      d = ((ast_Decl*)(st->decl));
      break;
   }
   case ast_TypeKind_Enum: {
      ast_EnumType* et = ((ast_EnumType*)(t));
      d = ((ast_Decl*)(et->decl));
      break;
   }
   case ast_TypeKind_Function:
      return;
   case ast_TypeKind_Alias: {
      ast_AliasType* at = ((ast_AliasType*)(t));
      d = ((ast_Decl*)(at->decl));
      break;
   }
   case ast_TypeKind_Module:
      return;
   }
   if (d) ast_Decl_setUsedPublic(d);
}

static const char* ast_getPrefixedName(const ast_Decl* d)
{
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_Function: {
      const ast_FunctionDecl* fd = ((ast_FunctionDecl*)(d));
      if (ast_FunctionDecl_hasPrefix(fd)) {
         static char fullname[64];
         sprintf(fullname, "%s.%s", ast_FunctionDecl_getPrefixName(fd), ast_Decl_getName(d));
         return fullname;
      }
      break;
   }
   case ast_DeclKind_Import:
      break;
   case ast_DeclKind_StructType:
      break;
   case ast_DeclKind_EnumType:
      break;
   case ast_DeclKind_EnumConstant:
      break;
   case ast_DeclKind_FunctionType:
      break;
   case ast_DeclKind_AliasType:
      break;
   case ast_DeclKind_Variable:
      break;
   }
   return ast_Decl_getName(d);
}

static bool ast_isGlobal(const ast_Decl* d)
{
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_Function:
      break;
   case ast_DeclKind_Import:
      return false;
   case ast_DeclKind_StructType: {
      const ast_StructTypeDecl* std = ((ast_StructTypeDecl*)(d));
      return ast_StructTypeDecl_isGlobal(std);
   }
   case ast_DeclKind_EnumType:
      break;
   case ast_DeclKind_EnumConstant:
      break;
   case ast_DeclKind_FunctionType:
      break;
   case ast_DeclKind_AliasType:
      break;
   case ast_DeclKind_Variable: {
      const ast_VarDecl* vd = ((ast_VarDecl*)(d));
      return ast_VarDecl_isGlobal(vd);
   }
   }
   return true;
}

static ast_QualType ast_getNativeType(void)
{
   ast_BuiltinKind kind = (ast_globals->wordsize == 8) ? ast_BuiltinKind_UInt64 : ast_BuiltinKind_UInt32;
   return ast_builtins[kind];
}


// --- module incr_array_list ---
typedef struct incr_array_list_Info_ incr_array_list_Info;
typedef struct incr_array_list_List_ incr_array_list_List;

struct incr_array_list_Info_ {
   uint32_t name;
   src_loc_SrcLoc loc;
   ast_ExprList values;
};

struct incr_array_list_List_ {
   incr_array_list_Info* entries;
   uint32_t count;
   uint32_t capacity;
};

static void incr_array_list_List_free(incr_array_list_List* v);
static void incr_array_list_List_resize(incr_array_list_List* v);
static ast_ExprList* incr_array_list_List_find(incr_array_list_List* v, uint32_t name);
static void incr_array_list_List_add(incr_array_list_List* v, uint32_t name, src_loc_SrcLoc loc, ast_Expr* value);
static void incr_array_list_List_free(incr_array_list_List* v)
{
   for (uint32_t i = 0; (i < v->count); i++) {
      ast_ExprList_free(&v->entries[i].values);
   }
   free(v->entries);
   v->count = 0;
   v->capacity = 0;
   v->entries = NULL;
}

static void incr_array_list_List_resize(incr_array_list_List* v)
{
   v->capacity = (v->capacity == 0) ? 4 : (v->capacity * 2);
   incr_array_list_Info* entries2 = malloc((v->capacity * 24));
   if (v->entries) {
      memcpy(entries2, v->entries, (v->count * 24));
      free(v->entries);
   }
   v->entries = entries2;
}

static ast_ExprList* incr_array_list_List_find(incr_array_list_List* v, uint32_t name)
{
   for (uint32_t i = 0; (i < v->count); i++) {
      incr_array_list_Info* info = &v->entries[i];
      if ((info->name == name)) return &info->values;

   }
   return NULL;
}

static void incr_array_list_List_add(incr_array_list_List* v, uint32_t name, src_loc_SrcLoc loc, ast_Expr* value)
{
   ast_ExprList* values = incr_array_list_List_find(v, name);
   if (!values) {
      if ((v->count == v->capacity)) incr_array_list_List_resize(v);
      incr_array_list_Info* info = &v->entries[v->count];
      info->name = name;
      info->loc = loc;
      ast_ExprList_init(&info->values, 4);
      v->count++;
      values = &info->values;
   }
   ast_ExprList_add(values, value);
}


// --- module struct_func_list ---
typedef struct struct_func_list_Info_ struct_func_list_Info;
typedef struct struct_func_list_List_ struct_func_list_List;

struct struct_func_list_Info_ {
   ast_Decl* decl;
   ast_FunctionDeclList functions;
};

struct struct_func_list_List_ {
   struct_func_list_Info* data;
   uint32_t count;
   uint32_t capacity;
};

static void struct_func_list_List_free(struct_func_list_List* v);
static void struct_func_list_List_resize(struct_func_list_List* v);
static void struct_func_list_List_addDecl(struct_func_list_List* v, ast_Decl* decl);
static ast_Decl* struct_func_list_List_getDecl(struct_func_list_List* v, uint32_t index);
static void struct_func_list_List_addFunc(struct_func_list_List* v, uint32_t index, ast_FunctionDecl* fd);
static ast_FunctionDecl* struct_func_list_List_findFunc(struct_func_list_List* v, uint32_t index, uint32_t name_idx);
static void struct_func_list_List_free(struct_func_list_List* v)
{
   for (uint32_t i = 0; (i < v->count); i++) {
      ast_FunctionDeclList_free(&v->data[i].functions);
   }
   free(v->data);
   v->count = 0;
   v->capacity = 0;
   v->data = NULL;
}

static void struct_func_list_List_resize(struct_func_list_List* v)
{
   v->capacity = (v->capacity == 0) ? 4 : (v->capacity * 2);
   struct_func_list_Info* data2 = malloc((v->capacity * 24));
   if (v->data) {
      memcpy(data2, v->data, (v->count * 24));
      free(v->data);
   }
   v->data = data2;
}

static void struct_func_list_List_addDecl(struct_func_list_List* v, ast_Decl* decl)
{
   if ((v->count == v->capacity)) struct_func_list_List_resize(v);
   struct_func_list_Info* info = &v->data[v->count];
   info->decl = decl;
   ast_FunctionDeclList_init(&info->functions);
   v->count++;
}

static ast_Decl* struct_func_list_List_getDecl(struct_func_list_List* v, uint32_t index)
{
   return v->data[index].decl;
}

static void struct_func_list_List_addFunc(struct_func_list_List* v, uint32_t index, ast_FunctionDecl* fd)
{
   c2_assert(((index < v->count)) != 0, "analyser/struct_func_list.c2:70: struct_func_list.List.addFunc", "index<v.count");
   struct_func_list_Info* info = &v->data[index];
   ast_FunctionDeclList_add(&info->functions, fd);
}

static ast_FunctionDecl* struct_func_list_List_findFunc(struct_func_list_List* v, uint32_t index, uint32_t name_idx)
{
   c2_assert(((index < v->count)) != 0, "analyser/struct_func_list.c2:76: struct_func_list.List.findFunc", "index<v.count");
   struct_func_list_Info* info = &v->data[index];
   return ast_FunctionDeclList_find(&info->functions, name_idx);
}


// --- module unused_checker ---
typedef struct unused_checker_Checker_ unused_checker_Checker;

struct unused_checker_Checker_ {
   diagnostics_Diags* diags;
   const warning_flags_Flags* warnings;
};

static void unused_checker_check(diagnostics_Diags* diags, const warning_flags_Flags* warnings, ast_Module* mod);
static void unused_checker_Checker_unused_module(void* arg, ast_AST* a);
static void unused_checker_Checker_check(void* arg, ast_Decl* d);
static void unused_checker_Checker_checkEnum(unused_checker_Checker* c, ast_EnumTypeDecl* d);
static void unused_checker_Checker_checkStructMembers(unused_checker_Checker* c, ast_Decl* d);
static void unused_checker_check(diagnostics_Diags* diags, const warning_flags_Flags* warnings, ast_Module* mod)
{
   unused_checker_Checker c = { .diags = diags, .warnings = warnings };
   if (ast_Module_isUsed(mod)) {
      ast_Module_visitDecls(mod, unused_checker_Checker_check, &c);
   } else {
      ast_Module_visitASTs(mod, unused_checker_Checker_unused_module, &c);
   }
}

static void unused_checker_Checker_unused_module(void* arg, ast_AST* a)
{
   unused_checker_Checker* c = arg;
   diagnostics_Diags_warn(c->diags, ast_AST_getLoc(a), "unused module '%s'", ast_AST_getName(a));
}

static void unused_checker_Checker_check(void* arg, ast_Decl* d)
{
   unused_checker_Checker* c = arg;
   bool used = ast_Decl_isUsed(d);
   if ((((used && ast_Decl_isPublic(d)) && !ast_Decl_isUsedPublic(d)) && !c->warnings->no_unused_public)) {
      diagnostics_Diags_warn(c->diags, ast_Decl_getLoc(d), "%s '%s' is not used public", ast_Decl_getKindName(d), ast_Decl_getFullName(d));
   }
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_Function:
      if (c->warnings->no_unused_function) return;

      if (ast_Decl_hasAttrUnused(d)) return;

      break;
   case ast_DeclKind_Import:
      if (c->warnings->no_unused_import) return;

      break;
   case ast_DeclKind_StructType:
      if (used) {
         unused_checker_Checker_checkStructMembers(c, d);
      }
      if (c->warnings->no_unused_type) return;

      break;
   case ast_DeclKind_EnumType:
      if ((used && !c->warnings->no_unused_enum_constant)) {
         unused_checker_Checker_checkEnum(c, ((ast_EnumTypeDecl*)(d)));
      }
      break;
   case ast_DeclKind_EnumConstant:
      break;
   case ast_DeclKind_FunctionType:
      if (c->warnings->no_unused_type) return;

      break;
   case ast_DeclKind_AliasType:
      if (c->warnings->no_unused_type) return;

      break;
   case ast_DeclKind_Variable:
      if (c->warnings->no_unused_variable) return;

      break;
   }
   if (((!used && !ast_Decl_hasAttrUnused(d)) && !ast_Decl_isExported(d))) {
      diagnostics_Diags_warn(c->diags, ast_Decl_getLoc(d), "unused %s '%s'", ast_Decl_getKindName(d), ast_Decl_getFullName(d));
      return;
   }
}

static void unused_checker_Checker_checkEnum(unused_checker_Checker* c, ast_EnumTypeDecl* d)
{
   uint32_t num_consts = ast_EnumTypeDecl_getNumConstants(d);
   ast_EnumConstantDecl** constants = ast_EnumTypeDecl_getConstants(d);
   for (uint32_t i = 0; (i < num_consts); i++) {
      ast_EnumConstantDecl* ecd = constants[i];
      ast_Decl* dd = ((ast_Decl*)(ecd));
      if (!ast_Decl_isUsed(dd)) {
         diagnostics_Diags_warn(c->diags, ast_Decl_getLoc(dd), "unused %s '%s'", ast_Decl_getKindName(dd), ast_Decl_getName(dd));
      }
   }
}

static void unused_checker_Checker_checkStructMembers(unused_checker_Checker* c, ast_Decl* d)
{
   ast_StructTypeDecl* std = ((ast_StructTypeDecl*)(d));
   uint32_t num_members = ast_StructTypeDecl_getNumMembers(std);
   ast_Decl** members = ast_StructTypeDecl_getMembers(std);
   for (uint32_t i = 0; (i < num_members); i++) {
      ast_Decl* member = members[i];
      if (ast_Decl_isStructType(member)) {
         unused_checker_Checker_checkStructMembers(c, member);
      } else {
         if ((!ast_Decl_isUsed(member) && !c->warnings->no_unused_variable)) {
            diagnostics_Diags_warn(c->diags, ast_Decl_getLoc(member), "unused %s member '%s'", ast_StructTypeDecl_isStruct(std) ? "struct" : "union", ast_Decl_getName(member));
         }
      }
   }
}


// --- module ctv_analyser ---
typedef struct ctv_analyser_Limit_ ctv_analyser_Limit;

struct ctv_analyser_Limit_ {
   int64_t min_val;
   uint64_t max_val;
   const char* min_str;
   const char* max_str;
};

static const ctv_analyser_Limit ctv_analyser_Limits[9] = {
   { 0, 1, "0", "1" },
   { -128, 127, "-128", "127" },
   { 0, 255, "0", "255" },
   { -32768, 32767, "-32768", "32767" },
   { 0, 65535, "0", "65535" },
   { -2147483648, 2147483647, "-2147483648", "2147483647" },
   { 0, 4294967295, "0", "4294967295" },
   { -9223372036854775807lu, 9223372036854775807lu, "-9223372036854775808", "9223372036854775807" },
   { 0, 18446744073709551615lu, "0", "18446744073709551615" }
};

static const ctv_analyser_Limit* ctv_analyser_getLimit(uint32_t width);
static ast_Value ctv_analyser_get_value(const ast_Expr* e);
static ast_Value ctv_analyser_truncate(ast_Value orig, bool is_signed, uint32_t width);
static ast_Value ctv_analyser_get_decl_value(const ast_Decl* d);
static ast_Value ctv_analyser_get_unaryop_value(const ast_UnaryOperator* e);
static ast_Value ctv_analyser_get_binaryop_value(const ast_BinaryOperator* e);
static bool ctv_analyser_check(diagnostics_Diags* diags, ast_QualType qt, const ast_Expr* e);
static bool ctv_analyser_checkRange(diagnostics_Diags* diags, ast_QualType qt, ast_Value* value, src_loc_SrcLoc loc, const ast_Expr* e);
static const ctv_analyser_Limit* ctv_analyser_getLimit(uint32_t width)
{
   switch (width) {
   case 1:
      return &ctv_analyser_Limits[0];
   case 7:
      return &ctv_analyser_Limits[1];
   case 8:
      return &ctv_analyser_Limits[2];
   case 15:
      return &ctv_analyser_Limits[3];
   case 16:
      return &ctv_analyser_Limits[4];
   case 31:
      return &ctv_analyser_Limits[5];
   case 32:
      return &ctv_analyser_Limits[6];
   case 63:
      return &ctv_analyser_Limits[7];
   case 64:
      return &ctv_analyser_Limits[8];
   }
   c2_assert((0) != 0, "analyser_utils/ctv_analyser.c2:63: ctv_analyser.getLimit", "0");
   return NULL;
}

static ast_Value ctv_analyser_get_value(const ast_Expr* e)
{
   ast_Value result = { };
   if (!ast_Expr_isCtv(e)) ast_Expr_dump(e);
   c2_assert((ast_Expr_isCtv(e)) != 0, "analyser_utils/ctv_analyser.c2:72: ctv_analyser.get_value", "CALL TODO");
   switch (ast_Expr_getKind(e)) {
   case ast_ExprKind_IntegerLiteral: {
      const ast_IntegerLiteral* i = ((ast_IntegerLiteral*)(e));
      result.uvalue = ast_IntegerLiteral_getValue(i);
      break;
   }
   case ast_ExprKind_FloatLiteral:
      break;
   case ast_ExprKind_BooleanLiteral: {
      const ast_BooleanLiteral* b = ((ast_BooleanLiteral*)(e));
      result.uvalue = ast_BooleanLiteral_getValue(b);
      break;
   }
   case ast_ExprKind_CharLiteral: {
      const ast_CharLiteral* c = ((ast_CharLiteral*)(e));
      result.uvalue = ast_CharLiteral_getValue(c);
      break;
   }
   case ast_ExprKind_StringLiteral:
      c2_assert((0) != 0, "analyser_utils/ctv_analyser.c2:91: ctv_analyser.get_value", "0");
      break;
   case ast_ExprKind_Nil:
      break;
   case ast_ExprKind_Identifier: {
      const ast_IdentifierExpr* i = ((ast_IdentifierExpr*)(e));
      return ctv_analyser_get_decl_value(ast_IdentifierExpr_getDecl(i));
   }
   case ast_ExprKind_Type:
      __attribute__((fallthrough));
   case ast_ExprKind_Call:
      __attribute__((fallthrough));
   case ast_ExprKind_InitList:
      __attribute__((fallthrough));
   case ast_ExprKind_FieldDesignatedInit:
      __attribute__((fallthrough));
   case ast_ExprKind_ArrayDesignatedInit:
      break;
   case ast_ExprKind_BinaryOperator:
      return ctv_analyser_get_binaryop_value(((ast_BinaryOperator*)(e)));
   case ast_ExprKind_UnaryOperator:
      return ctv_analyser_get_unaryop_value(((ast_UnaryOperator*)(e)));
   case ast_ExprKind_ConditionalOperator:
      break;
   case ast_ExprKind_Builtin: {
      const ast_BuiltinExpr* bi = ((ast_BuiltinExpr*)(e));
      result = ast_BuiltinExpr_getValue(bi);
      break;
   }
   case ast_ExprKind_ArraySubscript: {
      ast_ArraySubscriptExpr* a = ((ast_ArraySubscriptExpr*)(e));
      result = ctv_analyser_get_value(ast_ArraySubscriptExpr_getBase(a));
      c2_assert((!result.is_signed) != 0, "analyser_utils/ctv_analyser.c2:121: ctv_analyser.get_value", "!result.is_signed");
      ast_Expr* index = ast_ArraySubscriptExpr_getIndex(a);
      c2_assert((ast_Expr_isBitOffset(index)) != 0, "analyser_utils/ctv_analyser.c2:124: ctv_analyser.get_value", "CALL TODO");
      ast_BitOffsetExpr* bo = ((ast_BitOffsetExpr*)(index));
      ast_Value high = ctv_analyser_get_value(ast_BitOffsetExpr_getLHS(bo));
      ast_Value low = ctv_analyser_get_value(ast_BitOffsetExpr_getRHS(bo));
      ast_Value width = ast_Value_minus(&high, &low);
      width.uvalue++;
      result.uvalue >>= low.uvalue;
      ast_Value_mask(&result, ((uint32_t)(width.uvalue)));
      break;
   }
   case ast_ExprKind_Member: {
      const ast_MemberExpr* m = ((ast_MemberExpr*)(e));
      return ctv_analyser_get_decl_value(ast_MemberExpr_getFullDecl(m));
   }
   case ast_ExprKind_Paren: {
      const ast_ParenExpr* p = ((ast_ParenExpr*)(e));
      return ctv_analyser_get_value(ast_ParenExpr_getInner(p));
   }
   case ast_ExprKind_BitOffset:
      break;
   case ast_ExprKind_ExplicitCast: {
      const ast_ExplicitCastExpr* i = ((ast_ExplicitCastExpr*)(e));
      result = ctv_analyser_get_value(ast_ExplicitCastExpr_getInner(i));
      ast_QualType qt = ast_Expr_getType(e);
      qt = ast_QualType_getCanonicalType(&qt);
      c2_assert((ast_QualType_isBuiltin(&qt)) != 0, "analyser_utils/ctv_analyser.c2:148: ctv_analyser.get_value", "CALL TODO");
      ast_BuiltinType* bi = ast_QualType_getBuiltin(&qt);
      result = ctv_analyser_truncate(result, ast_BuiltinType_isSigned(bi), ast_BuiltinType_getWidth(bi));
      break;
   }
   case ast_ExprKind_ImplicitCast: {
      const ast_ImplicitCastExpr* i = ((ast_ImplicitCastExpr*)(e));
      return ctv_analyser_get_value(ast_ImplicitCastExpr_getInner(i));
   }
   }
   return result;
}

static ast_Value ctv_analyser_truncate(ast_Value orig, bool is_signed, uint32_t width)
{
   switch (width) {
   case 1:
      orig.is_signed = false;
      orig.svalue = ((orig.uvalue != 0)) ? 1 : 0;
      break;
   case 7:
      orig.uvalue = ((orig.uvalue & 0x7f));
      break;
   case 8:
      orig.is_signed = false;
      orig.uvalue = ((orig.uvalue & 0xff));
      break;
   case 15:
      orig.uvalue = ((orig.uvalue & 0x7fff));
      break;
   case 16:
      orig.is_signed = false;
      orig.uvalue = ((orig.uvalue & 0xffff));
      break;
   case 31:
      orig.uvalue = ((orig.uvalue & 0x7fffffff));
      break;
   case 32:
      orig.is_signed = false;
      orig.uvalue = ((orig.uvalue & 0xffffffff));
      break;
   case 63:
      orig.uvalue = ((orig.uvalue & 0xefffffffffffffff));
      break;
   case 64:
      orig.is_signed = false;
      break;
   }
   return orig;
}

static ast_Value ctv_analyser_get_decl_value(const ast_Decl* d)
{
   c2_assert((d) != NULL, "analyser_utils/ctv_analyser.c2:201: ctv_analyser.get_decl_value", "d");
   ast_Value result = { };
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_EnumConstant: {
      const ast_EnumConstantDecl* ecd = ((ast_EnumConstantDecl*)(d));
      result = ast_EnumConstantDecl_getValue(ecd);
      break;
   }
   case ast_DeclKind_Variable: {
      const ast_VarDecl* vd = ((ast_VarDecl*)(d));
      const ast_Expr* initval = ast_VarDecl_getInit(vd);
      c2_assert((initval) != NULL, "analyser_utils/ctv_analyser.c2:211: ctv_analyser.get_decl_value", "initval");
      return ctv_analyser_get_value(initval);
   }
   default:
      c2_assert((0) != 0, "analyser_utils/ctv_analyser.c2:214: ctv_analyser.get_decl_value", "0");
      break;
   }
   return result;
}

static ast_Value ctv_analyser_get_unaryop_value(const ast_UnaryOperator* e)
{
   ast_Value result = { };
   const ast_Expr* inner = ast_UnaryOperator_getInner(e);
   ast_Value res2 = ctv_analyser_get_value(inner);
   switch (ast_UnaryOperator_getOpcode(e)) {
   case ast_UnaryOpcode_PostInc:
      __attribute__((fallthrough));
   case ast_UnaryOpcode_PostDec:
      __attribute__((fallthrough));
   case ast_UnaryOpcode_PreInc:
      __attribute__((fallthrough));
   case ast_UnaryOpcode_PreDec:
      break;
   case ast_UnaryOpcode_AddrOf:
      __attribute__((fallthrough));
   case ast_UnaryOpcode_Deref:
      break;
   case ast_UnaryOpcode_Minus:
      result.is_signed = true;
      result.svalue = res2.is_signed ? -res2.svalue : ((int64_t)(-res2.uvalue));
      break;
   case ast_UnaryOpcode_Not:
      result.svalue = ((result.svalue == 0)) ? 1 : 0;
      break;
   case ast_UnaryOpcode_LNot:
      result.uvalue = res2.is_signed ? ((uint64_t)(~res2.svalue)) : ~res2.uvalue;
      break;
   }
   return result;
}

static ast_Value ctv_analyser_get_binaryop_value(const ast_BinaryOperator* e)
{
   ast_Value result = { };
   ast_Value left = ctv_analyser_get_value(ast_BinaryOperator_getLHS(e));
   ast_Value right = ctv_analyser_get_value(ast_BinaryOperator_getRHS(e));
   result.is_signed = left.is_signed;
   switch (ast_BinaryOperator_getOpcode(e)) {
   case ast_BinaryOpcode_Multiply:
      if ((left.is_signed || right.is_signed)) {
         int64_t lval = (left.is_signed) ? left.svalue : ((int64_t)(left.uvalue));
         int64_t rval = (right.is_signed) ? right.svalue : ((int64_t)(right.uvalue));
         result.svalue = (lval * rval);
         result.is_signed = true;
      } else {
         result.uvalue = (left.uvalue * right.uvalue);
      }
      break;
   case ast_BinaryOpcode_Divide:
      c2_assert(((right.svalue != 0)) != 0, "analyser_utils/ctv_analyser.c2:275: ctv_analyser.get_binaryop_value", "right.svalue!=0");
      if ((left.is_signed || right.is_signed)) {
         int64_t lval = (left.is_signed) ? left.svalue : ((int64_t)(left.uvalue));
         int64_t rval = (right.is_signed) ? right.svalue : ((int64_t)(right.uvalue));
         result.svalue = (lval / rval);
         result.is_signed = true;
      } else {
         result.uvalue = (left.uvalue / right.uvalue);
      }
      break;
   case ast_BinaryOpcode_Reminder:
      c2_assert(((right.svalue != 0)) != 0, "analyser_utils/ctv_analyser.c2:288: ctv_analyser.get_binaryop_value", "right.svalue!=0");
      if (left.is_signed) result.svalue = (left.svalue % right.svalue);
      else result.uvalue = (left.uvalue % right.uvalue);
      break;
   case ast_BinaryOpcode_Add:
      if ((left.is_signed || right.is_signed)) {
         int64_t lval = (left.is_signed) ? left.svalue : ((int64_t)(left.uvalue));
         int64_t rval = (right.is_signed) ? right.svalue : ((int64_t)(right.uvalue));
         result.svalue = (lval + rval);
         result.is_signed = true;
      } else {
         result.uvalue = (left.uvalue + right.uvalue);
      }
      break;
   case ast_BinaryOpcode_Subtract: {
      int64_t lval = (left.is_signed) ? left.svalue : ((int64_t)(left.uvalue));
      int64_t rval = (right.is_signed) ? right.svalue : ((int64_t)(right.uvalue));
      result.svalue = (lval - rval);
      result.is_signed = true;
      break;
   }
   case ast_BinaryOpcode_ShiftLeft:
      result.uvalue = (left.uvalue << right.uvalue);
      break;
   case ast_BinaryOpcode_ShiftRight:
      result.uvalue = (left.uvalue >> right.uvalue);
      break;
   case ast_BinaryOpcode_LessThan:
      result.is_signed = false;
      if (left.is_signed) result.svalue = (left.svalue < right.svalue);
      else result.uvalue = (left.uvalue < right.uvalue);
      break;
   case ast_BinaryOpcode_GreaterThan:
      result.is_signed = false;
      if (left.is_signed) result.svalue = (left.svalue > right.svalue);
      else result.uvalue = (left.uvalue > right.uvalue);
      break;
   case ast_BinaryOpcode_LessEqual:
      result.is_signed = false;
      if (left.is_signed) result.svalue = (left.svalue <= right.svalue);
      else result.uvalue = (left.uvalue <= right.uvalue);
      break;
   case ast_BinaryOpcode_GreaterEqual:
      result.is_signed = false;
      if (left.is_signed) result.svalue = (left.svalue >= right.svalue);
      else result.uvalue = (left.uvalue >= right.uvalue);
      break;
   case ast_BinaryOpcode_Equal:
      result.is_signed = false;
      if (left.is_signed) result.svalue = (left.svalue == right.svalue);
      else result.uvalue = (left.uvalue == right.uvalue);
      break;
   case ast_BinaryOpcode_NotEqual:
      result.is_signed = false;
      if (left.is_signed) result.svalue = (left.svalue != right.svalue);
      else result.uvalue = (left.uvalue != right.uvalue);
      break;
   case ast_BinaryOpcode_And:
      result.is_signed = false;
      if (left.is_signed) result.svalue = (left.svalue & right.svalue);
      else result.uvalue = (left.uvalue & right.uvalue);
      break;
   case ast_BinaryOpcode_Xor:
      result.is_signed = false;
      if (left.is_signed) result.svalue = (left.svalue ^ right.svalue);
      else result.uvalue = (left.uvalue ^ right.uvalue);
      break;
   case ast_BinaryOpcode_Or:
      result.is_signed = false;
      if (left.is_signed) result.svalue = (left.svalue | right.svalue);
      else result.uvalue = (left.uvalue | right.uvalue);
      break;
   case ast_BinaryOpcode_LAnd:
      result.is_signed = false;
      if (left.is_signed) result.svalue = (left.svalue && right.svalue);
      else result.uvalue = (left.uvalue && right.uvalue);
      break;
   case ast_BinaryOpcode_LOr:
      result.is_signed = false;
      if (left.is_signed) result.svalue = (left.svalue || right.svalue);
      else result.uvalue = (left.uvalue || right.uvalue);
      break;
   case ast_BinaryOpcode_Assign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_MulAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_DivAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_RemAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_AddAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_SubAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_ShlAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_ShrAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_AndAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_XorAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_OrAssign:
      c2_assert((0) != 0, "analyser_utils/ctv_analyser.c2:386: ctv_analyser.get_binaryop_value", "0");
      break;
   }
   return result;
}

static bool ctv_analyser_check(diagnostics_Diags* diags, ast_QualType qt, const ast_Expr* e)
{
   ast_QualType canon = ast_QualType_getCanonicalType(&qt);
   if (!ast_QualType_isBuiltin(&canon)) return true;

   c2_assert((ast_Expr_isCtv(e)) != 0, "analyser_utils/ctv_analyser.c2:396: ctv_analyser.check", "CALL TODO");
   ast_Value value = ctv_analyser_get_value((e));
   return ctv_analyser_checkRange(diags, qt, &value, 0, e);
}

static bool ctv_analyser_checkRange(diagnostics_Diags* diags, ast_QualType qt, ast_Value* value, src_loc_SrcLoc loc, const ast_Expr* e)
{
   ast_QualType canon = ast_QualType_getCanonicalType(&qt);
   if (!ast_QualType_isBuiltin(&canon)) return true;

   ast_BuiltinType* bi = ast_QualType_getBuiltin(&canon);
   uint32_t width = ast_BuiltinType_getWidth(bi);
   if ((width == 64)) return true;

   if (((ast_BuiltinType_getKind(bi) == ast_BuiltinKind_Float32) || (ast_BuiltinType_getKind(bi) == ast_BuiltinKind_Float64))) return true;

   const ctv_analyser_Limit* limit = ctv_analyser_getLimit(width);
   if (ast_BuiltinType_isSigned(bi)) value->is_signed = true;
   bool out_of_bounds = false;
   if (value->is_signed) {
      if (((value->svalue > 0) && (((uint64_t)(value->svalue)) > limit->max_val))) out_of_bounds = true;
      if (((value->svalue < 0) && (value->svalue < limit->min_val))) out_of_bounds = true;
   } else {
      if ((value->uvalue > limit->max_val)) out_of_bounds = true;
   }
   if (out_of_bounds) {
      if (e) {
         diagnostics_Diags_errorRange(diags, ast_Expr_getLoc(e), ast_Expr_getRange(e), "constant value %s out-of-bounds for type '%s', range [%s, %s]", ast_Value_str(value), ast_QualType_diagNameBare(&qt), limit->min_str, limit->max_str);
      } else {
         diagnostics_Diags_error(diags, loc, "constant value %s out-of-bounds for type '%s', range [%s, %s]", ast_Value_str(value), ast_QualType_diagNameBare(&qt), limit->min_str, limit->max_str);
      }
      return false;
   }
   return true;
}


// --- module printf_utils ---

typedef enum {
   printf_utils_Specifier_Other,
   printf_utils_Specifier_String,
   printf_utils_Specifier_Char,
   printf_utils_Specifier_Integer,
   printf_utils_Specifier_FloatingPoint,
   printf_utils_Specifier_Pointer,
   printf_utils_Specifier_Invalid,
   _printf_utils_Specifier_max = 255
} __attribute__((packed)) printf_utils_Specifier;

typedef bool (*printf_utils_FormatHandler)(void* arg, printf_utils_Specifier specifier, uint32_t offset, char letter);

static const char* printf_utils_get_format(ast_Expr* format, src_loc_SrcLoc* format_loc);
static printf_utils_Specifier printf_utils_getSpecifier(const char* specifier, uint32_t* len, char* c);
static bool printf_utils_parseFormat(const char* format, printf_utils_FormatHandler handler, void* arg);
static const char* printf_utils_get_format(ast_Expr* format, src_loc_SrcLoc* format_loc)
{
   const char* format_text = NULL;
   switch (ast_Expr_getKind(format)) {
   case ast_ExprKind_StringLiteral: {
      ast_StringLiteral* s = ((ast_StringLiteral*)(format));
      format_text = ast_StringLiteral_getText(s);
      *format_loc = (ast_Expr_getLoc(format) + 1);
      break;
   }
   case ast_ExprKind_Identifier: {
      ast_QualType qt = ast_Expr_getType(format);
      c2_assert((ast_QualType_isArray(&qt)) != 0, "analyser_utils/printf_utils.c2:32: printf_utils.get_format", "CALL TODO");
      ast_ArrayType* at = ast_QualType_getArrayType(&qt);
      qt = ast_ArrayType_getElemType(at);
      if (!ast_QualType_isConst(&qt)) return NULL;

      ast_IdentifierExpr* id = ((ast_IdentifierExpr*)(format));
      ast_Decl* decl = ast_IdentifierExpr_getDecl(id);
      c2_assert((ast_Decl_isVariable(decl)) != 0, "analyser_utils/printf_utils.c2:39: printf_utils.get_format", "CALL TODO");
      ast_VarDecl* vd = ((ast_VarDecl*)(decl));
      ast_Expr* initExpr = ast_VarDecl_getInit(vd);
      c2_assert((initExpr) != NULL, "analyser_utils/printf_utils.c2:42: printf_utils.get_format", "initExpr");
      return printf_utils_get_format(initExpr, format_loc);
   }
   case ast_ExprKind_Member: {
      ast_QualType qt = ast_Expr_getType(format);
      c2_assert((ast_QualType_isArray(&qt)) != 0, "analyser_utils/printf_utils.c2:46: printf_utils.get_format", "CALL TODO");
      ast_ArrayType* at = ast_QualType_getArrayType(&qt);
      qt = ast_ArrayType_getElemType(at);
      if (!ast_QualType_isConst(&qt)) return NULL;

      ast_MemberExpr* m = ((ast_MemberExpr*)(format));
      ast_Decl* decl = ast_MemberExpr_getFullDecl(m);
      c2_assert((ast_Decl_isVariable(decl)) != 0, "analyser_utils/printf_utils.c2:53: printf_utils.get_format", "CALL TODO");
      ast_VarDecl* vd = ((ast_VarDecl*)(decl));
      ast_Expr* initExpr = ast_VarDecl_getInit(vd);
      c2_assert((initExpr) != NULL, "analyser_utils/printf_utils.c2:56: printf_utils.get_format", "initExpr");
      return printf_utils_get_format(initExpr, format_loc);
   }
   default:
      c2_assert((0) != 0, "analyser_utils/printf_utils.c2:59: printf_utils.get_format", "0");
      break;
   }
   return format_text;
}

static printf_utils_Specifier printf_utils_getSpecifier(const char* specifier, uint32_t* len, char* c)
{
   const char* cp = specifier;
   while (1) {
      switch (*cp) {
      case '*':
         c2_assert((0) != 0, "analyser_utils/printf_utils.c2:82: printf_utils.getSpecifier", "0");
         break;
      case '%':
         cp++;
         *len = ((uint32_t)((cp - specifier)));
         if ((*len == 1)) return printf_utils_Specifier_Other;

         return printf_utils_Specifier_Invalid;
      case 'c':
         *c = *cp;
         cp++;
         *len = ((uint32_t)((cp - specifier)));
         return printf_utils_Specifier_Char;
      case 'f':
         *c = *cp;
         cp++;
         *len = ((uint32_t)((cp - specifier)));
         return printf_utils_Specifier_FloatingPoint;
      case 'p':
         *c = *cp;
         cp++;
         *len = ((uint32_t)((cp - specifier)));
         return printf_utils_Specifier_Pointer;
      case 's':
         *c = *cp;
         cp++;
         *len = ((uint32_t)((cp - specifier)));
         return printf_utils_Specifier_String;
      case 'd':
         __attribute__((fallthrough));
      case 'o':
         __attribute__((fallthrough));
      case 'x':
         __attribute__((fallthrough));
      case 'X':
         *c = *cp;
         cp++;
         *len = ((uint32_t)((cp - specifier)));
         return printf_utils_Specifier_Integer;
      default:
         if (isalpha(*cp)) {
            cp++;
            *len = ((uint32_t)((cp - specifier)));
            return printf_utils_Specifier_Invalid;
         }
         break;
      }
      cp++;
   }
   return printf_utils_Specifier_Other;
}

static bool printf_utils_parseFormat(const char* format, printf_utils_FormatHandler handler, void* arg)
{
   const char* cp = format;
   while (*cp) {
      if ((*cp == '%')) {
         uint32_t len = 0;
         cp++;
         char c = 0;
         printf_utils_Specifier s = printf_utils_getSpecifier(cp, &len, &c);
         cp += (len - 1);
         if ((s != printf_utils_Specifier_Other)) {
            if (!handler(arg, s, ((uint32_t)((cp - format))), c)) return false;

         }
      }
      cp++;
   }
   return true;
}


// --- module c2recipe ---
typedef struct c2recipe_Recipe_ c2recipe_Recipe;
typedef struct c2recipe_Token_ c2recipe_Token;
typedef struct c2recipe_Parser_ c2recipe_Parser;

struct c2recipe_Recipe_ {
   string_pool_Pool* pool;
   source_mgr_SourceMgr* sm;
   build_target_Target** targets;
   uint32_t num_targets;
   uint32_t max_targets;
   build_target_PluginList plugins;
};

static c2recipe_Recipe* c2recipe_create(source_mgr_SourceMgr* sm, string_pool_Pool* pool);
static void c2recipe_Recipe_free(c2recipe_Recipe* r);
static void c2recipe_Recipe_addPlugin(c2recipe_Recipe* r, uint32_t name, uint32_t options, src_loc_SrcLoc loc);
static build_target_Target* c2recipe_Recipe_addTarget(c2recipe_Recipe* r, uint32_t name, src_loc_SrcLoc loc, build_target_Kind kind);
static void c2recipe_Recipe_addDummyTarget(c2recipe_Recipe* r, const char* filename);
static bool c2recipe_Recipe_parse(c2recipe_Recipe* r, int32_t file_id);
static uint32_t c2recipe_Recipe_numTargets(const c2recipe_Recipe* r);
static build_target_Target* c2recipe_Recipe_getTarget(const c2recipe_Recipe* r, uint32_t idx);
static const build_target_PluginList* c2recipe_Recipe_getPlugins(const c2recipe_Recipe* r);
typedef enum {
   c2recipe_Kind_Plugin,
   c2recipe_Kind_PluginOptions,
   c2recipe_Kind_Text,
   c2recipe_Kind_Executable,
   c2recipe_Kind_Lib,
   c2recipe_Kind_Image,
   c2recipe_Kind_File,
   c2recipe_Kind_End,
   c2recipe_Kind_Warnings,
   c2recipe_Kind_Backend,
   c2recipe_Kind_DisableAsserts,
   c2recipe_Kind_NoLibc,
   c2recipe_Kind_Config,
   c2recipe_Kind_Export,
   c2recipe_Kind_Use,
   c2recipe_Kind_AsmFile,
   c2recipe_Kind_Eof,
   _c2recipe_Kind_max = 255
} __attribute__((packed)) c2recipe_Kind;

struct c2recipe_Token_ {
   c2recipe_Kind kind;
   src_loc_SrcLoc loc;
   bool more;
   uint32_t value;
};

struct c2recipe_Parser_ {
   c2recipe_Recipe* recipe;
   string_pool_Pool* pool;
   source_mgr_SourceMgr* sm;
   string_list_List global_configs;
   const char* input_start;
   const char* cur;
   src_loc_SrcLoc loc_start;
   __jmp_buf_tag jmpbuf;
   c2recipe_Token token;
   bool new_line;
   bool targets_started;
   build_target_Target* target;
};

static const char* c2recipe_kind_names[17] = {
   "plugin",
   "[plugin_options]",
   "text",
   "executable",
   "lib",
   "image",
   "file",
   "end",
   "$warnings",
   "$backend",
   "$disable-asserts",
   "$nolibc",
   "$config",
   "$export",
   "$use",
   "$asm",
   "eof"
};

static void c2recipe_Token_init(c2recipe_Token* t);
static bool c2recipe_Parser_parse(c2recipe_Recipe* recipe, string_pool_Pool* pool, source_mgr_SourceMgr* sm, int32_t file_id);
static bool c2recipe_Parser_addGlobalFeature(c2recipe_Parser* p, uint32_t feature);
__attribute__((__format__(printf, 2, 3))) 
static void c2recipe_Parser_error(c2recipe_Parser* p, const char* format, ...);
static void c2recipe_Parser_consumeToken(c2recipe_Parser* p);
static void c2recipe_Parser_expect(c2recipe_Parser* p, c2recipe_Kind kind, const char* msg);
static bool c2recipe_Parser_is(const c2recipe_Parser* p, c2recipe_Kind kind);
static void c2recipe_Parser_lex(c2recipe_Parser* p, c2recipe_Token* result);
static void c2recipe_Parser_lex_plugin_options(c2recipe_Parser* p, c2recipe_Token* result);
static void c2recipe_Parser_lex_option(c2recipe_Parser* p, c2recipe_Token* result);
static void c2recipe_Parser_skip_comments(c2recipe_Parser* p);
static void c2recipe_Parser_parseTop(c2recipe_Parser* p);
static void c2recipe_Parser_parsePlugin(c2recipe_Parser* p, bool is_global);
static void c2recipe_Parser_parseWarnings(c2recipe_Parser* p);
static void c2recipe_Parser_parseExecutable(c2recipe_Parser* p);
static void c2recipe_Parser_parseImage(c2recipe_Parser* p);
static void c2recipe_Parser_parseLibrary(c2recipe_Parser* p);
static void c2recipe_Parser_parseTarget(c2recipe_Parser* p);
static void c2recipe_Parser_parseBackend(c2recipe_Parser* p);
static void c2recipe_Parser_parseCGenOptions(c2recipe_Parser* p);
static bool c2recipe_equals(const char* str, const char* expect, uint32_t len);
static const char* c2recipe_get_prefix(const char* input, char* output, uint32_t maxlen);
static bool c2recipe_Recipe_getYamlInfo(c2recipe_Recipe* _arg0, const yaml_Parser* parser);
static bool c2recipe_Recipe_parseYaml(c2recipe_Recipe* r, int32_t file_id);
static c2recipe_Recipe* c2recipe_create(source_mgr_SourceMgr* sm, string_pool_Pool* pool)
{
   c2recipe_Recipe* r = calloc(1, 48);
   r->sm = sm;
   r->pool = pool;
   r->max_targets = 4;
   r->targets = calloc(r->max_targets, 8);
   return r;
}

static void c2recipe_Recipe_free(c2recipe_Recipe* r)
{
   for (uint32_t i = 0; (i < r->num_targets); i++) {
      build_target_Target_free(r->targets[i]);
   }
   free(((void*)(r->targets)));
   free(r);
}

static void c2recipe_Recipe_addPlugin(c2recipe_Recipe* r, uint32_t name, uint32_t options, src_loc_SrcLoc loc)
{
   build_target_PluginList_add(&r->plugins, name, options, loc);
}

static build_target_Target* c2recipe_Recipe_addTarget(c2recipe_Recipe* r, uint32_t name, src_loc_SrcLoc loc, build_target_Kind kind)
{
   if ((r->num_targets == r->max_targets)) {
      r->max_targets *= 2;
      build_target_Target** targets2 = malloc((r->max_targets * 8));
      memcpy(((void*)(targets2)), ((void*)(r->targets)), (r->num_targets * 8));
      free(((void*)(r->targets)));
      r->targets = targets2;
   }
   build_target_Target* t = build_target_create(name, loc, kind, r->pool);
   r->targets[r->num_targets] = t;
   r->num_targets++;
   return t;
}

static void c2recipe_Recipe_addDummyTarget(c2recipe_Recipe* r, const char* filename)
{
   uint32_t target_name = string_pool_Pool_addStr(r->pool, "dummy", true);
   uint32_t file_idx = string_pool_Pool_addStr(r->pool, filename, false);
   build_target_Target* t = c2recipe_Recipe_addTarget(r, target_name, 0, build_target_Kind_Executable);
   build_target_Target_addFile(t, file_idx, 0);
}

static bool c2recipe_Recipe_parse(c2recipe_Recipe* r, int32_t file_id)
{
   return c2recipe_Parser_parse(r, r->pool, r->sm, file_id);
}

static uint32_t c2recipe_Recipe_numTargets(const c2recipe_Recipe* r)
{
   return r->num_targets;
}

static build_target_Target* c2recipe_Recipe_getTarget(const c2recipe_Recipe* r, uint32_t idx)
{
   return r->targets[idx];
}

static const build_target_PluginList* c2recipe_Recipe_getPlugins(const c2recipe_Recipe* r)
{
   return &r->plugins;
}

static void c2recipe_Token_init(c2recipe_Token* t)
{
   memset(t, 0, 16);
   t->more = true;
}

static bool c2recipe_Parser_parse(c2recipe_Recipe* recipe, string_pool_Pool* pool, source_mgr_SourceMgr* sm, int32_t file_id)
{
   const char* data = source_mgr_SourceMgr_get_content(sm, file_id);
   c2recipe_Parser p = { };
   p.recipe = recipe;
   p.pool = pool;
   p.sm = sm;
   string_list_List_init(&p.global_configs, pool);
   p.input_start = data;
   p.cur = data;
   p.loc_start = source_mgr_SourceMgr_get_offset(sm, file_id);
   p.new_line = true;
   c2recipe_Token_init(&p.token);
   int32_t res = setjmp(&p.jmpbuf);
   if ((res == 0)) {
      c2recipe_Parser_consumeToken(&p);
      c2recipe_Parser_parseTop(&p);
   }
   string_list_List_free(&p.global_configs);
   return (res == 0);
}

static bool c2recipe_Parser_addGlobalFeature(c2recipe_Parser* p, uint32_t feature)
{
   if (string_list_List_contains_idx(&p->global_configs, feature)) return false;

   string_list_List_add(&p->global_configs, feature);
   return true;
}

__attribute__((__format__(printf, 2, 3))) 
static void c2recipe_Parser_error(c2recipe_Parser* p, const char* format, ...)
{
   char msg[128];
   va_list args;
   va_start(args, format);
   vsnprintf(msg, (128 - 1), format, args);
   va_end(args);
   if (color_useColor()) {
      fprintf(stderr, "%s: %serror:%s %s\n", source_mgr_SourceMgr_loc2str(p->sm, p->token.loc), color_Red, color_Normal, msg);
   } else {
      fprintf(stderr, "%s: error: %s\n", source_mgr_SourceMgr_loc2str(p->sm, p->token.loc), msg);
   }
   longjmp(&p->jmpbuf, 1);
}

static void c2recipe_Parser_consumeToken(c2recipe_Parser* p)
{
   c2recipe_Parser_lex(p, &p->token);
}

static void c2recipe_Parser_expect(c2recipe_Parser* p, c2recipe_Kind kind, const char* msg)
{
   if ((p->token.kind != kind)) c2recipe_Parser_error(p, "%s", msg);
}

static bool c2recipe_Parser_is(const c2recipe_Parser* p, c2recipe_Kind kind)
{
   return (p->token.kind == kind);
}

static void c2recipe_Parser_lex(c2recipe_Parser* p, c2recipe_Token* result)
{
   while (1) {
      switch (*p->cur) {
      case 0:
         p->cur--;
         result->loc = (p->loc_start + ((src_loc_SrcLoc)((p->cur - p->input_start))));
         result->kind = c2recipe_Kind_Eof;
         result->more = false;
         return;
      case ' ':
         __attribute__((fallthrough));
      case '\t':
         __attribute__((fallthrough));
      case '\r':
         p->cur++;
         break;
      case '\n':
         p->cur++;
         p->new_line = true;
         break;
      case '#':
         c2recipe_Parser_skip_comments(p);
         break;
      case '[':
         c2recipe_Parser_lex_plugin_options(p, result);
         p->new_line = false;
         return;
      case '$':
         c2recipe_Parser_lex_option(p, result);
         p->new_line = false;
         return;
      case '/': {
         const char* start = p->cur;
         while ((*p->cur && !isspace(*p->cur))) p->cur++;
         result->kind = p->new_line ? c2recipe_Kind_File : c2recipe_Kind_Text;
         if ((*p->cur == 0)) result->kind = c2recipe_Kind_Eof;
         p->new_line = false;
         uint32_t len = ((uint32_t)((p->cur - start)));
         result->value = string_pool_Pool_add(p->pool, start, len, true);
         return;
      }
      default:
         if (isalnum(*p->cur)) {
            result->loc = (p->loc_start + ((src_loc_SrcLoc)((p->cur - p->input_start))));
            if (c2recipe_equals(p->cur, "plugin ", 7)) {
               result->kind = c2recipe_Kind_Plugin;
               p->cur += 7;
               p->new_line = false;
               return;
            }
            if (c2recipe_equals(p->cur, "executable ", 11)) {
               result->kind = c2recipe_Kind_Executable;
               p->cur += 11;
               p->new_line = false;
               return;
            }
            if (c2recipe_equals(p->cur, "lib ", 4)) {
               result->kind = c2recipe_Kind_Lib;
               p->cur += 4;
               p->new_line = false;
               return;
            }
            if (c2recipe_equals(p->cur, "end", 3)) {
               result->kind = c2recipe_Kind_End;
               p->cur += 3;
               p->new_line = false;
               return;
            }
            if (c2recipe_equals(p->cur, "image ", 6)) {
               result->kind = c2recipe_Kind_Image;
               p->cur += 6;
               p->new_line = false;
               return;
            }
            if (c2recipe_equals(p->cur, "config ", 7)) {
               result->kind = c2recipe_Kind_Config;
               p->cur += 7;
               p->new_line = false;
               return;
            }
            const char* start = p->cur;
            while ((*p->cur && !isspace(*p->cur))) p->cur++;
            result->kind = p->new_line ? c2recipe_Kind_File : c2recipe_Kind_Text;
            p->new_line = false;
            uint32_t len = ((uint32_t)((p->cur - start)));
            result->value = string_pool_Pool_add(p->pool, start, len, true);
            return;
         }
         result->loc = (p->loc_start + ((src_loc_SrcLoc)((p->cur - p->input_start))));
         c2recipe_Parser_error(p, "unexpected input '%c'", *p->cur);
         return;
      }
   }
}

static void c2recipe_Parser_lex_plugin_options(c2recipe_Parser* p, c2recipe_Token* result)
{
   p->cur++;
   const char* start = p->cur;
   while (1) {
      if ((*p->cur == 0)) {
         return;
      }
      if ((*p->cur == ']')) {
         uint32_t len = ((uint32_t)((p->cur - start)));
         result->loc = (p->loc_start + ((src_loc_SrcLoc)((start - p->input_start))));
         result->kind = c2recipe_Kind_PluginOptions;
         result->value = string_pool_Pool_add(p->pool, start, len, true);
         p->cur++;
         return;
      }
      p->cur++;
   }
}

static void c2recipe_Parser_lex_option(c2recipe_Parser* p, c2recipe_Token* result)
{
   p->cur++;
   result->loc = (p->loc_start + ((src_loc_SrcLoc)((p->cur - p->input_start))));
   const char* end = p->cur;
   while ((*end && !isspace(*end))) end++;
   uint32_t len = ((uint32_t)((end - p->cur)));
   if ((len >= 20)) c2recipe_Parser_error(p, "unknown option");
   char option[24];
   memcpy(option, p->cur, len);
   option[len] = 0;
   do {
      const char* _tmp = option;
      if (c2_strequal(_tmp, "warnings")) {
         result->kind = c2recipe_Kind_Warnings;
      } else if (c2_strequal(_tmp, "backend")) {
         result->kind = c2recipe_Kind_Backend;
      } else if (c2_strequal(_tmp, "disable-asserts")) {
         result->kind = c2recipe_Kind_DisableAsserts;
      } else if (c2_strequal(_tmp, "nolibc")) {
         result->kind = c2recipe_Kind_NoLibc;
      } else if (c2_strequal(_tmp, "config")) {
         result->kind = c2recipe_Kind_Config;
      } else if (c2_strequal(_tmp, "export")) {
         result->kind = c2recipe_Kind_Export;
      } else if (c2_strequal(_tmp, "plugin")) {
         result->kind = c2recipe_Kind_Plugin;
      } else if (c2_strequal(_tmp, "use")) {
         result->kind = c2recipe_Kind_Use;
      } else if (c2_strequal(_tmp, "asm")) {
         result->kind = c2recipe_Kind_AsmFile;
      } else {
         c2recipe_Parser_error(p, "unknown option '%s'", option);
      }
   } while (0);
   p->cur += strlen(option);
}

static void c2recipe_Parser_skip_comments(c2recipe_Parser* p)
{
   while (*p->cur) {
      if ((*p->cur == '\n')) return;

      p->cur++;
   }
}

static void c2recipe_Parser_parseTop(c2recipe_Parser* p)
{
   while (1) {
      switch (p->token.kind) {
      case c2recipe_Kind_Plugin:
         c2recipe_Parser_parsePlugin(p, true);
         break;
      case c2recipe_Kind_PluginOptions:
         break;
      case c2recipe_Kind_Text:
         break;
      case c2recipe_Kind_Executable:
         c2recipe_Parser_parseExecutable(p);
         break;
      case c2recipe_Kind_Lib:
         c2recipe_Parser_parseLibrary(p);
         break;
      case c2recipe_Kind_Image:
         c2recipe_Parser_parseImage(p);
         break;
      case c2recipe_Kind_File:
         c2recipe_Parser_error(p, "syntax error");
         break;
      case c2recipe_Kind_End:
         break;
      case c2recipe_Kind_Warnings:
         __attribute__((fallthrough));
      case c2recipe_Kind_Backend:
         __attribute__((fallthrough));
      case c2recipe_Kind_DisableAsserts:
         __attribute__((fallthrough));
      case c2recipe_Kind_NoLibc:
         c2recipe_Parser_error(p, "must be inside target");
         break;
      case c2recipe_Kind_Config:
         if (p->targets_started) c2recipe_Parser_error(p, "global configs must come before targets");
         c2recipe_Parser_consumeToken(p);
         c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect config");
         if (!c2recipe_Parser_addGlobalFeature(p, p->token.value)) {
            c2recipe_Parser_error(p, "duplicate config '%s'", string_pool_Pool_idx2str(p->pool, p->token.value));
         }
         c2recipe_Parser_consumeToken(p);
         break;
      case c2recipe_Kind_Export:
         __attribute__((fallthrough));
      case c2recipe_Kind_Use:
         __attribute__((fallthrough));
      case c2recipe_Kind_AsmFile:
         c2recipe_Parser_error(p, "must be inside target");
         break;
      case c2recipe_Kind_Eof:
         return;
      }
   }
}

static void c2recipe_Parser_parsePlugin(c2recipe_Parser* p, bool is_global)
{
   c2recipe_Parser_consumeToken(p);
   c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect plugin name");
   uint32_t name = p->token.value;
   src_loc_SrcLoc loc = p->token.loc;
   c2recipe_Parser_consumeToken(p);
   uint32_t options = 0;
   if ((p->token.kind == c2recipe_Kind_PluginOptions)) {
      options = p->token.value;
      c2recipe_Parser_consumeToken(p);
   }
   if (is_global) {
      c2recipe_Recipe_addPlugin(p->recipe, name, options, loc);
   } else {
      build_target_Target_addPlugin(p->target, name, options, loc);
   }
}

static void c2recipe_Parser_parseWarnings(c2recipe_Parser* p)
{
   c2recipe_Parser_consumeToken(p);
   c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect options");
   warning_flags_Flags* warnings = build_target_Target_getWarnings2(p->target);
   while (c2recipe_Parser_is(p, c2recipe_Kind_Text)) {
      const char* option = string_pool_Pool_idx2str(p->pool, p->token.value);
      if ((strcmp(option, "no-unused") == 0)) {
         warnings->no_unused = true;
         warnings->no_unused_variable = true;
         warnings->no_unused_function = true;
         warnings->no_unused_parameter = true;
         warnings->no_unused_type = true;
         warnings->no_unused_module = true;
         warnings->no_unused_import = true;
         warnings->no_unused_public = true;
         warnings->no_unused_label = true;
         warnings->no_unused_enum_constant = true;
      } else if ((strcmp(option, "no-unused-variable") == 0)) {
         warnings->no_unused_variable = true;
      } else if ((strcmp(option, "no-unused-function") == 0)) {
         warnings->no_unused_function = true;
      } else if ((strcmp(option, "no-unused-parameter") == 0)) {
         warnings->no_unused_parameter = true;
      } else if ((strcmp(option, "no-unused-type") == 0)) {
         warnings->no_unused_type = true;
      } else if ((strcmp(option, "no-unused-module") == 0)) {
         warnings->no_unused_module = true;
      } else if ((strcmp(option, "no-unused-import") == 0)) {
         warnings->no_unused_import = true;
      } else if ((strcmp(option, "no-unused-public") == 0)) {
         warnings->no_unused_public = true;
      } else if ((strcmp(option, "no-unused-label") == 0)) {
         warnings->no_unused_label = true;
      } else if ((strcmp(option, "no-unused-enum-constant") == 0)) {
         warnings->no_unused_enum_constant = true;
      } else if ((strcmp(option, "promote-to-error") == 0)) {
         warnings->are_errors = true;
      } else {
         c2recipe_Parser_error(p, "unknown warning '%s'", option);
      }










      c2recipe_Parser_consumeToken(p);
   }
}

static void c2recipe_Parser_parseExecutable(c2recipe_Parser* p)
{
   c2recipe_Parser_consumeToken(p);
   c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect target name");
   p->target = c2recipe_Recipe_addTarget(p->recipe, p->token.value, p->token.loc, build_target_Kind_Executable);
   c2recipe_Parser_consumeToken(p);
   c2recipe_Parser_parseTarget(p);
}

static void c2recipe_Parser_parseImage(c2recipe_Parser* p)
{
   c2recipe_Parser_consumeToken(p);
   c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect image name");
   p->target = c2recipe_Recipe_addTarget(p->recipe, p->token.value, p->token.loc, build_target_Kind_Image);
   c2recipe_Parser_consumeToken(p);
   c2recipe_Parser_parseTarget(p);
}

static void c2recipe_Parser_parseLibrary(c2recipe_Parser* p)
{
   c2recipe_Parser_consumeToken(p);
   c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect target name");
   uint32_t name = p->token.value;
   src_loc_SrcLoc loc = p->token.loc;
   c2recipe_Parser_consumeToken(p);
   c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect lib type");
   uint32_t kind_name = p->token.value;
   build_target_Kind kind = build_target_Kind_StaticLibrary;
   do {
      const char* _tmp = string_pool_Pool_idx2str(p->pool, kind_name);
      if (c2_strequal(_tmp, "static")) {
         kind = build_target_Kind_StaticLibrary;
      } else if (c2_strequal(_tmp, "dynamic")) {
         kind = build_target_Kind_DynamicLibrary;
      } else {
         c2recipe_Parser_error(p, "invalid library type (allowed: dynamic|static)");
      }
   } while (0);
   c2recipe_Parser_consumeToken(p);
   p->target = c2recipe_Recipe_addTarget(p->recipe, name, loc, kind);
   c2recipe_Parser_parseTarget(p);
}

static void c2recipe_Parser_parseTarget(c2recipe_Parser* p)
{
   p->targets_started = true;
   bool files_started = false;
   for (uint32_t i = 0; (i < string_list_List_length(&p->global_configs)); i++) {
      build_target_Target_addFeature(p->target, string_list_List_get_idx(&p->global_configs, i));
   }
   while (1) {
      switch (p->token.kind) {
      case c2recipe_Kind_Plugin:
         c2recipe_Parser_parsePlugin(p, false);
         break;
      case c2recipe_Kind_PluginOptions:
         __attribute__((fallthrough));
      case c2recipe_Kind_Text:
         __attribute__((fallthrough));
      case c2recipe_Kind_Executable:
         __attribute__((fallthrough));
      case c2recipe_Kind_Lib:
         __attribute__((fallthrough));
      case c2recipe_Kind_Image:
         c2recipe_Parser_error(p, "syntax error");
         break;
      case c2recipe_Kind_File:
         files_started = true;
         if (!build_target_Target_addFile(p->target, p->token.value, p->token.loc)) {
            c2recipe_Parser_error(p, "duplicate file '%s'", string_pool_Pool_idx2str(p->pool, p->token.value));
         }
         c2recipe_Parser_consumeToken(p);
         break;
      case c2recipe_Kind_End:
         c2recipe_Parser_consumeToken(p);
         p->target = NULL;
         return;
      case c2recipe_Kind_Warnings:
         if (files_started) c2recipe_Parser_error(p, "$warnings must come before files");
         c2recipe_Parser_parseWarnings(p);
         break;
      case c2recipe_Kind_Backend:
         if (files_started) c2recipe_Parser_error(p, "$backend must come before files");
         c2recipe_Parser_parseBackend(p);
         break;
      case c2recipe_Kind_DisableAsserts:
         if (files_started) c2recipe_Parser_error(p, "$disable-asserts must come before files");
         c2recipe_Parser_consumeToken(p);
         build_target_Target_disableAsserts(p->target);
         break;
      case c2recipe_Kind_NoLibc:
         if (files_started) c2recipe_Parser_error(p, "$nolibc must come before files");
         c2recipe_Parser_consumeToken(p);
         build_target_Target_setNoLibC(p->target);
         break;
      case c2recipe_Kind_Config:
         if (files_started) c2recipe_Parser_error(p, "$config must come before files");
         c2recipe_Parser_consumeToken(p);
         c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect config");
         build_target_Target_addFeature(p->target, p->token.value);
         c2recipe_Parser_consumeToken(p);
         break;
      case c2recipe_Kind_Export:
         if (files_started) c2recipe_Parser_error(p, "$export must come before files");
         c2recipe_Parser_consumeToken(p);
         c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect export");
         while ((p->token.kind == c2recipe_Kind_Text)) {
            build_target_Target_addExport(p->target, p->token.value);
            c2recipe_Parser_consumeToken(p);
         }
         break;
      case c2recipe_Kind_Use: {
         if (files_started) c2recipe_Parser_error(p, "$use must come before files");
         c2recipe_Parser_consumeToken(p);
         c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect library name");
         uint32_t libname = p->token.value;
         if (build_target_Target_hasLib(p->target, libname)) {
            c2recipe_Parser_error(p, "duplicate use of %s", string_pool_Pool_idx2str(p->pool, libname));
         }
         c2recipe_Parser_consumeToken(p);
         c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect library type");
         bool is_static = false;
         const char* libtype = string_pool_Pool_idx2str(p->pool, p->token.value);
         do {
            const char* _tmp = libtype;
            if (c2_strequal(_tmp, "static")) {
               is_static = true;
            } else if (c2_strequal(_tmp, "dynamic")) {
            } else {
               c2recipe_Parser_error(p, "unknown library kind '%s'", libtype);
            }
         } while (0);
         build_target_Target_addLib(p->target, libname, is_static);
         c2recipe_Parser_consumeToken(p);
         while ((p->token.kind == c2recipe_Kind_Text)) c2recipe_Parser_consumeToken(p);
         break;
      }
      case c2recipe_Kind_AsmFile:
         if (files_started) c2recipe_Parser_error(p, "$asm must come before files");
         c2recipe_Parser_consumeToken(p);
         c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect filename");
         if (!build_target_Target_addAsmFile(p->target, p->token.value, p->token.loc)) {
            c2recipe_Parser_error(p, "duplicate asm file '%s'", string_pool_Pool_idx2str(p->pool, p->token.value));
         }
         c2recipe_Parser_consumeToken(p);
         break;
      case c2recipe_Kind_Eof:
         c2recipe_Parser_error(p, "un-terminated target");
         return;
      }
   }
}

static void c2recipe_Parser_parseBackend(c2recipe_Parser* p)
{
   c2recipe_Parser_consumeToken(p);
   c2recipe_Parser_expect(p, c2recipe_Kind_Text, "expect backend type");
   const char* backend_kind = string_pool_Pool_idx2str(p->pool, p->token.value);
   c2recipe_Parser_consumeToken(p);
   do {
      const char* _tmp = backend_kind;
      if (c2_strequal(_tmp, "c")) {
         if (build_target_Target_getBackendC(p->target)) c2recipe_Parser_error(p, "duplicate c backend");
         build_target_Target_setBackendC(p->target);
         c2recipe_Parser_parseCGenOptions(p);
      } else if (c2_strequal(_tmp, "llvm")) {
         if (build_target_Target_getBackendLLVM(p->target)) c2recipe_Parser_error(p, "duplicate llvm backend");
         build_target_Target_setBackendLLVM(p->target);
         while ((p->token.kind == c2recipe_Kind_Text)) c2recipe_Parser_consumeToken(p);
      } else if (c2_strequal(_tmp, "qbe")) {
         if (build_target_Target_getBackendQBE(p->target)) c2recipe_Parser_error(p, "duplicate qbe backend");
         build_target_Target_setBackendQBE(p->target);
         while ((p->token.kind == c2recipe_Kind_Text)) c2recipe_Parser_consumeToken(p);
      } else {
         c2recipe_Parser_error(p, "unknown backend type (supported: c,llvm,qbe)");
      }
   } while (0);
}

static void c2recipe_Parser_parseCGenOptions(c2recipe_Parser* p)
{
   while ((p->token.kind == c2recipe_Kind_Text)) {
      const char* option = string_pool_Pool_idx2str(p->pool, p->token.value);
      do {
         const char* _tmp = option;
         if (c2_strequal(_tmp, "no-build")) {
            build_target_Target_setCGenNoBuild(p->target);
         } else if (c2_strequal(_tmp, "fast")) {
            build_target_Target_setCGenFastBuild(p->target);
         } else if (c2_strequal(_tmp, "skip")) {
         } else {
            c2recipe_Parser_error(p, "invalid c backend option '%s'", option);
         }
      } while (0);
      c2recipe_Parser_consumeToken(p);
   }
}

static bool c2recipe_equals(const char* str, const char* expect, uint32_t len)
{
   for (uint32_t i = 0; (i < len); i++) {
      if ((str[i] != expect[i])) return false;

   }
   return true;
}

static const char* c2recipe_get_prefix(const char* input, char* output, uint32_t maxlen)
{
   maxlen--;
   while ((*input && maxlen)) {
      if ((*input == '.')) break;

      *output++ = *input++;
      maxlen--;
   }
   *output = 0;
   if ((maxlen == 0)) return NULL;

   return (input + 1);
}

static bool c2recipe_Recipe_getYamlInfo(c2recipe_Recipe* _arg0, const yaml_Parser* parser)
{
   const yaml_Node* root = yaml_Parser_getRoot(parser);
   if ((!root || !yaml_Node_isMap(root))) {
      fprintf(stderr, "empty recipe?\n");
      return false;
   }
   yaml_Iter iter = yaml_Parser_getNodeChildIter(parser, root);
   while (!yaml_Iter_done(&iter)) {
      const char* name = yaml_Iter_getName(&iter);
      char prefix[32];
      const char* after = c2recipe_get_prefix(name, prefix, 32);
      if (!after) {
         printf("invalid item %s\n", name);
         return false;
      }
      char* p = prefix;
      do {
         const char* _tmp = p;
         if (c2_strequal(_tmp, "plugin")) {
         } else if (c2_strequal(_tmp, "executable")) {
         } else {
            printf("unknown item %s\n", prefix);
            return false;
         }
      } while (0);
      yaml_Iter_next(&iter);
   }
   return true;
}

static bool c2recipe_Recipe_parseYaml(c2recipe_Recipe* r, int32_t file_id)
{
   const char* data = source_mgr_SourceMgr_get_content(r->sm, file_id);
   yaml_Parser* parser = yaml_Parser_create();
   bool ok = yaml_Parser_parse(parser, ((char*)(data)));
   if (ok) {
      if (!c2recipe_Recipe_getYamlInfo(r, parser)) return false;

   } else {
      fprintf(stderr, "Error: %s\n", yaml_Parser_getMessage(parser));
   }
   yaml_Parser_destroy(parser);
   return ok;
}


// --- module ast_visitor ---
typedef struct ast_visitor_Visitor_ ast_visitor_Visitor;

typedef void (*ast_visitor_OnRef)(void* arg, const ast_Ref* ref);

struct ast_visitor_Visitor_ {
   void* arg;
   ast_visitor_OnRef on_ref;
};

static ast_visitor_Visitor* ast_visitor_create(void* arg, ast_visitor_OnRef on_ref);
static void ast_visitor_Visitor_free(ast_visitor_Visitor* v);
static void ast_visitor_Visitor_handleAssert(ast_visitor_Visitor* v, ast_StaticAssert* a);
static void ast_visitor_Visitor_handle(ast_visitor_Visitor* v, ast_Decl* d);
static void ast_visitor_Visitor_handleFunction(ast_visitor_Visitor* v, ast_FunctionDecl* d);
static void ast_visitor_Visitor_handleVarDecl(ast_visitor_Visitor* v, ast_VarDecl* d);
static void ast_visitor_Visitor_handleTypeRef(ast_visitor_Visitor* v, const ast_TypeRef* r);
static void ast_visitor_Visitor_handleType(ast_visitor_Visitor* v, ast_QualType qt);
static void ast_visitor_Visitor_handleStmt(ast_visitor_Visitor* v, ast_Stmt* s);
static void ast_visitor_Visitor_handleCompoundStmt(ast_visitor_Visitor* v, ast_CompoundStmt* s);
static void ast_visitor_Visitor_handleExpr(ast_visitor_Visitor* v, ast_Expr* e);
static void ast_visitor_Visitor_handleCallExpr(ast_visitor_Visitor* v, ast_CallExpr* c);
static void ast_visitor_Visitor_handleMemberExpr(ast_visitor_Visitor* v, ast_MemberExpr* m);
static void ast_visitor_Visitor_handleBuiltinExpr(ast_visitor_Visitor* v, ast_BuiltinExpr* b);
static ast_visitor_Visitor* ast_visitor_create(void* arg, ast_visitor_OnRef on_ref)
{
   ast_visitor_Visitor* v = calloc(1, 16);
   v->arg = arg;
   v->on_ref = on_ref;
   return v;
}

static void ast_visitor_Visitor_free(ast_visitor_Visitor* v)
{
   free(v);
}

static void ast_visitor_Visitor_handleAssert(ast_visitor_Visitor* v, ast_StaticAssert* a)
{
   ast_visitor_Visitor_handleExpr(v, ast_StaticAssert_getLhs(a));
   ast_visitor_Visitor_handleExpr(v, ast_StaticAssert_getRhs(a));
}

static void ast_visitor_Visitor_handle(ast_visitor_Visitor* v, ast_Decl* d)
{
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_Function:
      ast_visitor_Visitor_handleFunction(v, ((ast_FunctionDecl*)(d)));
      break;
   case ast_DeclKind_Import:
      break;
   case ast_DeclKind_StructType: {
      ast_StructTypeDecl* s = ((ast_StructTypeDecl*)(d));
      uint32_t num_members = ast_StructTypeDecl_getNumMembers(s);
      ast_Decl** members = ast_StructTypeDecl_getMembers(s);
      for (uint32_t i = 0; (i < num_members); i++) {
         ast_visitor_Visitor_handle(v, members[i]);
      }
      break;
   }
   case ast_DeclKind_EnumType:
      break;
   case ast_DeclKind_EnumConstant:
      break;
   case ast_DeclKind_FunctionType: {
      ast_FunctionTypeDecl* ftd = ((ast_FunctionTypeDecl*)(d));
      ast_visitor_Visitor_handleFunction(v, ast_FunctionTypeDecl_getDecl(ftd));
      break;
   }
   case ast_DeclKind_AliasType: {
      ast_AliasTypeDecl* atd = ((ast_AliasTypeDecl*)(d));
      ast_visitor_Visitor_handleTypeRef(v, ast_AliasTypeDecl_getTypeRef(atd));
      break;
   }
   case ast_DeclKind_Variable:
      ast_visitor_Visitor_handleVarDecl(v, ((ast_VarDecl*)(d)));
      break;
   }
}

static void ast_visitor_Visitor_handleFunction(ast_visitor_Visitor* v, ast_FunctionDecl* d)
{
   if (ast_FunctionDecl_isTemplate(d)) return;

   ast_visitor_Visitor_handleTypeRef(v, ast_FunctionDecl_getReturnTypeRef(d));
   ast_Ref* prefix = ast_FunctionDecl_getPrefix(d);
   if (prefix) v->on_ref(v->arg, prefix);
   uint32_t num_params = ast_FunctionDecl_getNumParams(d);
   ast_VarDecl** args = ast_FunctionDecl_getParams(d);
   for (uint32_t i = 0; (i < num_params); i++) ast_visitor_Visitor_handleVarDecl(v, args[i]);
   ast_CompoundStmt* body = ast_FunctionDecl_getBody(d);
   if (body) {
      ast_visitor_Visitor_handleCompoundStmt(v, body);
   }
}

static void ast_visitor_Visitor_handleVarDecl(ast_visitor_Visitor* v, ast_VarDecl* d)
{
   ast_visitor_Visitor_handleTypeRef(v, ast_VarDecl_getTypeRef(d));
   ast_Expr* init_expr = ast_VarDecl_getInit(d);
   if (init_expr) ast_visitor_Visitor_handleExpr(v, init_expr);
}

static void ast_visitor_Visitor_handleTypeRef(ast_visitor_Visitor* v, const ast_TypeRef* r)
{
   if (ast_TypeRef_isUser(r)) {
      const ast_Ref* prefix = ast_TypeRef_getPrefix(r);
      if (prefix) v->on_ref(v->arg, prefix);
      const ast_Ref* user = ast_TypeRef_getUser(r);
      v->on_ref(v->arg, user);
   }
   uint32_t num_arrays = ast_TypeRef_getNumArrays(r);
   for (uint32_t i = 0; (i < num_arrays); i++) {
      ast_Expr* e = ast_TypeRef_getArray(r, i);
      if (e) ast_visitor_Visitor_handleExpr(v, e);
   }
}

static void ast_visitor_Visitor_handleType(ast_visitor_Visitor* v, ast_QualType qt)
{
}

static void ast_visitor_Visitor_handleStmt(ast_visitor_Visitor* v, ast_Stmt* s)
{
   switch (ast_Stmt_getKind(s)) {
   case ast_StmtKind_Return: {
      ast_ReturnStmt* r = ((ast_ReturnStmt*)(s));
      ast_Expr* e = ast_ReturnStmt_getValue(r);
      if (e) ast_visitor_Visitor_handleExpr(v, e);
      break;
   }
   case ast_StmtKind_Expr:
      ast_visitor_Visitor_handleExpr(v, ((ast_Expr*)(s)));
      break;
   case ast_StmtKind_If: {
      ast_IfStmt* i = ((ast_IfStmt*)(s));
      ast_visitor_Visitor_handleStmt(v, ast_IfStmt_getCond(i));
      ast_visitor_Visitor_handleStmt(v, ast_IfStmt_getThen(i));
      ast_Stmt* e = ast_IfStmt_getElse(i);
      if (e) ast_visitor_Visitor_handleStmt(v, e);
      break;
   }
   case ast_StmtKind_While: {
      ast_WhileStmt* w = ((ast_WhileStmt*)(s));
      ast_visitor_Visitor_handleStmt(v, ast_WhileStmt_getCond(w));
      ast_visitor_Visitor_handleStmt(v, ast_WhileStmt_getBody(w));
      break;
   }
   case ast_StmtKind_Do: {
      ast_DoStmt* d = ((ast_DoStmt*)(s));
      ast_visitor_Visitor_handleStmt(v, ast_DoStmt_getCond(d));
      ast_visitor_Visitor_handleStmt(v, ast_DoStmt_getBody(d));
      break;
   }
   case ast_StmtKind_For: {
      ast_ForStmt* f = ((ast_ForStmt*)(s));
      ast_Stmt* in = ast_ForStmt_getInit(f);
      if (in) ast_visitor_Visitor_handleStmt(v, in);
      ast_Expr* cond = ast_ForStmt_getCond(f);
      if (cond) ast_visitor_Visitor_handleExpr(v, cond);
      ast_Expr* incr = ast_ForStmt_getIncr(f);
      if (incr) ast_visitor_Visitor_handleExpr(v, incr);
      ast_Stmt* body = ast_ForStmt_getBody(f);
      if (body) ast_visitor_Visitor_handleStmt(v, body);
      break;
   }
   case ast_StmtKind_Switch: {
      ast_SwitchStmt* sw = ((ast_SwitchStmt*)(s));
      ast_visitor_Visitor_handleExpr(v, ast_SwitchStmt_getCond(sw));
      const uint32_t numcases = ast_SwitchStmt_getNumCases(sw);
      ast_SwitchCase** cases = ast_SwitchStmt_getCases(sw);
      for (uint32_t i = 0; (i < numcases); i++) {
         ast_SwitchCase* c = cases[i];
         if (ast_SwitchCase_numMulti(c)) {
            ast_IdentifierExpr** multi = ast_SwitchCase_getMultiCond(c);
            for (uint32_t j = 0; (j < ast_SwitchCase_numMulti(c)); j++) {
               ast_IdentifierExpr* id = multi[j];
               ast_Ref ref = ast_IdentifierExpr_getRef(id);
               v->on_ref(v->arg, &ref);
            }
         } else {
            if (ast_SwitchCase_getCond(c)) ast_visitor_Visitor_handleExpr(v, ast_SwitchCase_getCond(c));
         }
         const uint32_t numstmts = ast_SwitchCase_getNumStmts(c);
         ast_Stmt** stmts = ast_SwitchCase_getStmts(c);
         for (uint32_t j = 0; (j < numstmts); j++) ast_visitor_Visitor_handleStmt(v, stmts[j]);
      }
      break;
   }
   case ast_StmtKind_Break:
      break;
   case ast_StmtKind_Continue:
      break;
   case ast_StmtKind_Fallthrough:
      break;
   case ast_StmtKind_Label:
      break;
   case ast_StmtKind_Goto:
      break;
   case ast_StmtKind_Compound:
      ast_visitor_Visitor_handleCompoundStmt(v, ((ast_CompoundStmt*)(s)));
      break;
   case ast_StmtKind_Decl: {
      ast_DeclStmt* d = ((ast_DeclStmt*)(s));
      ast_visitor_Visitor_handleVarDecl(v, ast_DeclStmt_getDecl(d));
      break;
   }
   case ast_StmtKind_Asm: {
      ast_AsmStmt* a = ((ast_AsmStmt*)(s));
      uint32_t num_exprs = ast_AsmStmt_getNumExprs(a);
      ast_Expr** exprs = ast_AsmStmt_getExprs(a);
      for (uint32_t i = 0; (i < num_exprs); i++) {
         ast_visitor_Visitor_handleExpr(v, exprs[i]);
      }
      break;
   }
   case ast_StmtKind_Assert: {
      ast_AssertStmt* a = ((ast_AssertStmt*)(s));
      ast_visitor_Visitor_handleExpr(v, ast_AssertStmt_getInner(a));
      break;
   }
   }
}

static void ast_visitor_Visitor_handleCompoundStmt(ast_visitor_Visitor* v, ast_CompoundStmt* s)
{
   uint32_t count = ast_CompoundStmt_getCount(s);
   ast_Stmt** stmts = ast_CompoundStmt_getStmts(s);
   for (uint32_t i = 0; (i < count); i++) ast_visitor_Visitor_handleStmt(v, stmts[i]);
}

static void ast_visitor_Visitor_handleExpr(ast_visitor_Visitor* v, ast_Expr* e)
{
   c2_assert((e) != NULL, "generator/ast_visitor_expr.c2:21: ast_visitor.Visitor.handleExpr", "e");
   switch (ast_Expr_getKind(e)) {
   case ast_ExprKind_IntegerLiteral:
      return;
   case ast_ExprKind_FloatLiteral:
      return;
   case ast_ExprKind_BooleanLiteral:
      return;
   case ast_ExprKind_CharLiteral:
      return;
   case ast_ExprKind_StringLiteral:
      return;
   case ast_ExprKind_Nil:
      return;
   case ast_ExprKind_Identifier: {
      ast_IdentifierExpr* i = ((ast_IdentifierExpr*)(e));
      ast_Ref ref = ast_IdentifierExpr_getRef(i);
      v->on_ref(v->arg, &ref);
      break;
   }
   case ast_ExprKind_Type: {
      ast_TypeExpr* t = ((ast_TypeExpr*)(e));
      ast_visitor_Visitor_handleTypeRef(v, ast_TypeExpr_getTypeRef(t));
      break;
   }
   case ast_ExprKind_Call:
      ast_visitor_Visitor_handleCallExpr(v, ((ast_CallExpr*)(e)));
      break;
   case ast_ExprKind_InitList: {
      ast_InitListExpr* ili = ((ast_InitListExpr*)(e));
      uint32_t count = ast_InitListExpr_getNumValues(ili);
      ast_Expr** exprs = ast_InitListExpr_getValues(ili);
      for (uint32_t i = 0; (i < count); i++) ast_visitor_Visitor_handleExpr(v, exprs[i]);
      break;
   }
   case ast_ExprKind_FieldDesignatedInit: {
      ast_FieldDesignatedInitExpr* f = ((ast_FieldDesignatedInitExpr*)(e));
      ast_Ref ref = { .loc = ast_Expr_getLoc(e), .name_idx = ast_FieldDesignatedInitExpr_getField(f), .decl = ast_FieldDesignatedInitExpr_getDecl(f) };
      v->on_ref(v->arg, &ref);
      ast_visitor_Visitor_handleExpr(v, ast_FieldDesignatedInitExpr_getInit(f));
      break;
   }
   case ast_ExprKind_ArrayDesignatedInit: {
      ast_ArrayDesignatedInitExpr* a = ((ast_ArrayDesignatedInitExpr*)(e));
      ast_visitor_Visitor_handleExpr(v, ast_ArrayDesignatedInitExpr_getDesignator(a));
      ast_visitor_Visitor_handleExpr(v, ast_ArrayDesignatedInitExpr_getInit(a));
      break;
   }
   case ast_ExprKind_BinaryOperator: {
      ast_BinaryOperator* b = ((ast_BinaryOperator*)(e));
      ast_visitor_Visitor_handleExpr(v, ast_BinaryOperator_getLHS(b));
      ast_visitor_Visitor_handleExpr(v, ast_BinaryOperator_getRHS(b));
      break;
   }
   case ast_ExprKind_UnaryOperator: {
      ast_UnaryOperator* u = ((ast_UnaryOperator*)(e));
      ast_visitor_Visitor_handleExpr(v, ast_UnaryOperator_getInner(u));
      break;
   }
   case ast_ExprKind_ConditionalOperator: {
      ast_ConditionalOperator* c = ((ast_ConditionalOperator*)(e));
      ast_visitor_Visitor_handleExpr(v, ast_ConditionalOperator_getCond(c));
      ast_visitor_Visitor_handleExpr(v, ast_ConditionalOperator_getLHS(c));
      ast_visitor_Visitor_handleExpr(v, ast_ConditionalOperator_getRHS(c));
      break;
   }
   case ast_ExprKind_Builtin:
      ast_visitor_Visitor_handleBuiltinExpr(v, ((ast_BuiltinExpr*)(e)));
      break;
   case ast_ExprKind_ArraySubscript: {
      ast_ArraySubscriptExpr* a = ((ast_ArraySubscriptExpr*)(e));
      ast_visitor_Visitor_handleExpr(v, ast_ArraySubscriptExpr_getBase(a));
      ast_visitor_Visitor_handleExpr(v, ast_ArraySubscriptExpr_getIndex(a));
      break;
   }
   case ast_ExprKind_Member:
      ast_visitor_Visitor_handleMemberExpr(v, ((ast_MemberExpr*)(e)));
      break;
   case ast_ExprKind_Paren: {
      ast_ParenExpr* p = ((ast_ParenExpr*)(e));
      ast_visitor_Visitor_handleExpr(v, ast_ParenExpr_getInner(p));
      break;
   }
   case ast_ExprKind_BitOffset: {
      ast_BitOffsetExpr* bi = ((ast_BitOffsetExpr*)(e));
      ast_visitor_Visitor_handleExpr(v, ast_BitOffsetExpr_getLHS(bi));
      ast_visitor_Visitor_handleExpr(v, ast_BitOffsetExpr_getRHS(bi));
      break;
   }
   case ast_ExprKind_ExplicitCast: {
      ast_ExplicitCastExpr* ec = ((ast_ExplicitCastExpr*)(e));
      ast_visitor_Visitor_handleTypeRef(v, ast_ExplicitCastExpr_getTypeRef(ec));
      ast_visitor_Visitor_handleExpr(v, ast_ExplicitCastExpr_getInner(ec));
      break;
   }
   case ast_ExprKind_ImplicitCast: {
      ast_ImplicitCastExpr* ic = ((ast_ImplicitCastExpr*)(e));
      ast_visitor_Visitor_handleExpr(v, ast_ImplicitCastExpr_getInner(ic));
      break;
   }
   }
}

static void ast_visitor_Visitor_handleCallExpr(ast_visitor_Visitor* v, ast_CallExpr* c)
{
   ast_visitor_Visitor_handleExpr(v, ast_CallExpr_getFunc(c));
   uint32_t num_args = ast_CallExpr_getNumArgs(c);
   ast_Expr** args = ast_CallExpr_getArgs(c);
   for (uint32_t i = 0; (i < num_args); i++) ast_visitor_Visitor_handleExpr(v, args[i]);
}

static void ast_visitor_Visitor_handleMemberExpr(ast_visitor_Visitor* v, ast_MemberExpr* m)
{
   if (ast_MemberExpr_hasExpr(m)) ast_visitor_Visitor_handleExpr(v, ast_MemberExpr_getExprBase(m));
   for (uint32_t i = 0; (i < ast_MemberExpr_getNumRefs(m)); i++) {
      ast_Ref ref = ast_MemberExpr_getRef(m, i);
      v->on_ref(v->arg, &ref);
   }
}

static void ast_visitor_Visitor_handleBuiltinExpr(ast_visitor_Visitor* v, ast_BuiltinExpr* b)
{
   ast_visitor_Visitor_handleExpr(v, ast_BuiltinExpr_getInner(b));
   switch (ast_BuiltinExpr_getKind(b)) {
   case ast_BuiltinExprKind_Sizeof:
      break;
   case ast_BuiltinExprKind_Elemsof:
      break;
   case ast_BuiltinExprKind_EnumMin:
      break;
   case ast_BuiltinExprKind_EnumMax:
      break;
   case ast_BuiltinExprKind_OffsetOf:
      ast_visitor_Visitor_handleExpr(v, ast_BuiltinExpr_getOffsetOfMember(b));
      break;
   case ast_BuiltinExprKind_ToContainer:
      ast_visitor_Visitor_handleExpr(v, ast_BuiltinExpr_getToContainerMember(b));
      ast_visitor_Visitor_handleExpr(v, ast_BuiltinExpr_getToContainerPointer(b));
      break;
   }
}


// --- module dep_finder ---
typedef struct dep_finder_Finder_ dep_finder_Finder;

typedef void (*dep_finder_OnDecl)(void* arg, ast_Decl* d);

struct dep_finder_Finder_ {
   ast_Module* mod;
   void* arg;
   dep_finder_OnDecl on_decl;
};

static void dep_finder_Finder_init(dep_finder_Finder* f, ast_Module* mod, void* arg, dep_finder_OnDecl on_decl);
static void dep_finder_Finder_check(dep_finder_Finder* s, ast_Decl* d);
static void dep_finder_Finder_handleFunctionType(dep_finder_Finder* s, ast_FunctionTypeDecl* ftd);
static void dep_finder_Finder_handleFunction(dep_finder_Finder* s, ast_FunctionDecl* d);
static void dep_finder_Finder_handleStruct(dep_finder_Finder* s, ast_StructTypeDecl* d);
static void dep_finder_Finder_handleEnumType(dep_finder_Finder* s, ast_EnumTypeDecl* etd);
static void dep_finder_Finder_handleTypeRef(dep_finder_Finder* f, ast_TypeRef* r);
static void dep_finder_Finder_handleVarDecl(dep_finder_Finder* s, ast_VarDecl* d);
static void dep_finder_Finder_handleExpr(dep_finder_Finder* s, ast_Expr* e);
static void dep_finder_Finder_handleInitList(dep_finder_Finder* s, ast_InitListExpr* ile);
static void dep_finder_Finder_handleMemberExpr(dep_finder_Finder* s, ast_MemberExpr* m);
static void dep_finder_Finder_onDep(dep_finder_Finder* s, const ast_Decl* d, bool full);
static void dep_finder_Finder_init(dep_finder_Finder* f, ast_Module* mod, void* arg, dep_finder_OnDecl on_decl)
{
   f->mod = mod;
   f->arg = arg;
   f->on_decl = on_decl;
}

static void dep_finder_Finder_check(dep_finder_Finder* s, ast_Decl* d)
{
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_Function:
      dep_finder_Finder_handleFunction(s, ((ast_FunctionDecl*)(d)));
      break;
   case ast_DeclKind_Import:
      break;
   case ast_DeclKind_StructType:
      dep_finder_Finder_handleStruct(s, ((ast_StructTypeDecl*)(d)));
      break;
   case ast_DeclKind_EnumType:
      dep_finder_Finder_handleEnumType(s, ((ast_EnumTypeDecl*)(d)));
      break;
   case ast_DeclKind_EnumConstant:
      ast_Decl_dump(d);
      c2_assert((0) != 0, "generator/dep_finder.c2:49: dep_finder.Finder.check", "0");
      break;
   case ast_DeclKind_FunctionType: {
      ast_FunctionTypeDecl* ftd = ((ast_FunctionTypeDecl*)(d));
      dep_finder_Finder_handleFunction(s, ast_FunctionTypeDecl_getDecl(ftd));
      break;
   }
   case ast_DeclKind_AliasType: {
      ast_AliasTypeDecl* at = ((ast_AliasTypeDecl*)(d));
      dep_finder_Finder_handleTypeRef(s, ast_AliasTypeDecl_getTypeRef(at));
      break;
   }
   case ast_DeclKind_Variable:
      dep_finder_Finder_handleVarDecl(s, ((ast_VarDecl*)(d)));
      break;
   }
}

static void dep_finder_Finder_handleFunctionType(dep_finder_Finder* s, ast_FunctionTypeDecl* ftd)
{
   dep_finder_Finder_handleFunction(s, ast_FunctionTypeDecl_getDecl(ftd));
}

static void dep_finder_Finder_handleFunction(dep_finder_Finder* s, ast_FunctionDecl* d)
{
   if (ast_FunctionDecl_isTemplate(d)) return;

   dep_finder_Finder_handleTypeRef(s, ast_FunctionDecl_getReturnTypeRef(d));
   uint32_t num_params = ast_FunctionDecl_getNumParams(d);
   ast_VarDecl** args = ast_FunctionDecl_getParams(d);
   for (uint32_t i = 0; (i < num_params); i++) dep_finder_Finder_handleVarDecl(s, args[i]);
}

static void dep_finder_Finder_handleStruct(dep_finder_Finder* s, ast_StructTypeDecl* d)
{
   uint32_t num_members = ast_StructTypeDecl_getNumMembers(d);
   ast_Decl** members = ast_StructTypeDecl_getMembers(d);
   for (uint32_t i = 0; (i < num_members); i++) {
      ast_Decl* m = members[i];
      if (ast_Decl_isStructType(m)) {
         dep_finder_Finder_handleStruct(s, ((ast_StructTypeDecl*)(m)));
      } else {
         c2_assert((ast_Decl_isVariable(m)) != 0, "generator/dep_finder.c2:98: dep_finder.Finder.handleStruct", "CALL TODO");
         dep_finder_Finder_handleVarDecl(s, ((ast_VarDecl*)(m)));
      }
   }
}

static void dep_finder_Finder_handleEnumType(dep_finder_Finder* s, ast_EnumTypeDecl* etd)
{
   uint32_t num_constants = ast_EnumTypeDecl_getNumConstants(etd);
   ast_EnumConstantDecl** constants = ast_EnumTypeDecl_getConstants(etd);
   for (uint32_t i = 0; (i < num_constants); i++) {
      ast_EnumConstantDecl* c = constants[i];
      ast_Expr* initExpr = ast_EnumConstantDecl_getInit(c);
      if (initExpr) dep_finder_Finder_handleExpr(s, initExpr);
   }
}

static void dep_finder_Finder_handleTypeRef(dep_finder_Finder* f, ast_TypeRef* r)
{
   const ast_Decl* refDecl = ast_TypeRef_getUserDecl(r);
   if (refDecl) {
      if ((ast_TypeRef_getNumPointers(r) && ast_Decl_isStructType(refDecl))) {
      } else {
         dep_finder_Finder_onDep(f, refDecl, (ast_TypeRef_getNumPointers(r) == 0));
      }
   }
   uint32_t num_arrays = ast_TypeRef_getNumArrays(r);
   for (uint32_t i = 0; (i < num_arrays); i++) {
      ast_Expr* e = ast_TypeRef_getArray(r, i);
      if (e) dep_finder_Finder_handleExpr(f, e);
   }
}

static void dep_finder_Finder_handleVarDecl(dep_finder_Finder* s, ast_VarDecl* d)
{
   dep_finder_Finder_handleTypeRef(s, ast_VarDecl_getTypeRef(d));
   ast_Expr* init_expr = ast_VarDecl_getInit(d);
   if (init_expr) dep_finder_Finder_handleExpr(s, init_expr);
}

static void dep_finder_Finder_handleExpr(dep_finder_Finder* s, ast_Expr* e)
{
   switch (ast_Expr_getKind(e)) {
   case ast_ExprKind_IntegerLiteral:
      break;
   case ast_ExprKind_FloatLiteral:
      break;
   case ast_ExprKind_BooleanLiteral:
      break;
   case ast_ExprKind_CharLiteral:
      break;
   case ast_ExprKind_StringLiteral:
      break;
   case ast_ExprKind_Nil:
      break;
   case ast_ExprKind_Identifier: {
      ast_IdentifierExpr* i = ((ast_IdentifierExpr*)(e));
      ast_Decl* d = ast_IdentifierExpr_getDecl(i);
      dep_finder_Finder_onDep(s, d, true);
      break;
   }
   case ast_ExprKind_Type: {
      ast_TypeExpr* t = ((ast_TypeExpr*)(e));
      dep_finder_Finder_handleTypeRef(s, ast_TypeExpr_getTypeRef(t));
      break;
   }
   case ast_ExprKind_Call:
      break;
   case ast_ExprKind_InitList:
      dep_finder_Finder_handleInitList(s, ((ast_InitListExpr*)(e)));
      break;
   case ast_ExprKind_FieldDesignatedInit: {
      ast_FieldDesignatedInitExpr* i = ((ast_FieldDesignatedInitExpr*)(e));
      dep_finder_Finder_handleExpr(s, ast_FieldDesignatedInitExpr_getInit(i));
      break;
   }
   case ast_ExprKind_ArrayDesignatedInit: {
      ast_ArrayDesignatedInitExpr* a = ((ast_ArrayDesignatedInitExpr*)(e));
      dep_finder_Finder_handleExpr(s, ast_ArrayDesignatedInitExpr_getDesignator(a));
      dep_finder_Finder_handleExpr(s, ast_ArrayDesignatedInitExpr_getInit(a));
      break;
   }
   case ast_ExprKind_BinaryOperator: {
      ast_BinaryOperator* b = ((ast_BinaryOperator*)(e));
      dep_finder_Finder_handleExpr(s, ast_BinaryOperator_getLHS(b));
      dep_finder_Finder_handleExpr(s, ast_BinaryOperator_getRHS(b));
      break;
   }
   case ast_ExprKind_UnaryOperator: {
      ast_UnaryOperator* u = ((ast_UnaryOperator*)(e));
      dep_finder_Finder_handleExpr(s, ast_UnaryOperator_getInner(u));
      break;
   }
   case ast_ExprKind_ConditionalOperator:
      ast_Expr_dump(e);
      c2_assert((0) != 0, "generator/dep_finder.c2:190: dep_finder.Finder.handleExpr", "0");
      break;
   case ast_ExprKind_Builtin: {
      ast_BuiltinExpr* b = ((ast_BuiltinExpr*)(e));
      dep_finder_Finder_handleExpr(s, ast_BuiltinExpr_getInner(b));
      break;
   }
   case ast_ExprKind_ArraySubscript: {
      ast_ArraySubscriptExpr* a = ((ast_ArraySubscriptExpr*)(e));
      dep_finder_Finder_handleExpr(s, ast_ArraySubscriptExpr_getBase(a));
      dep_finder_Finder_handleExpr(s, ast_ArraySubscriptExpr_getIndex(a));
      break;
   }
   case ast_ExprKind_Member:
      dep_finder_Finder_handleMemberExpr(s, ((ast_MemberExpr*)(e)));
      break;
   case ast_ExprKind_Paren: {
      ast_ParenExpr* p = ((ast_ParenExpr*)(e));
      dep_finder_Finder_handleExpr(s, ast_ParenExpr_getInner(p));
      break;
   }
   case ast_ExprKind_BitOffset:
      break;
   case ast_ExprKind_ExplicitCast: {
      ast_ExplicitCastExpr* c = ((ast_ExplicitCastExpr*)(e));
      dep_finder_Finder_handleExpr(s, ast_ExplicitCastExpr_getInner(c));
      dep_finder_Finder_handleTypeRef(s, ast_ExplicitCastExpr_getTypeRef(c));
      break;
   }
   case ast_ExprKind_ImplicitCast: {
      ast_ImplicitCastExpr* c = ((ast_ImplicitCastExpr*)(e));
      dep_finder_Finder_handleExpr(s, ast_ImplicitCastExpr_getInner(c));
      break;
   }
   }
}

static void dep_finder_Finder_handleInitList(dep_finder_Finder* s, ast_InitListExpr* ile)
{
   uint32_t count = ast_InitListExpr_getNumValues(ile);
   ast_Expr** exprs = ast_InitListExpr_getValues(ile);
   for (uint32_t i = 0; (i < count); i++) {
      dep_finder_Finder_handleExpr(s, exprs[i]);
   }
}

static void dep_finder_Finder_handleMemberExpr(dep_finder_Finder* s, ast_MemberExpr* m)
{
   ast_Decl* d = ast_MemberExpr_getFullDecl(m);
   if (ast_Decl_isEnumConstant(d)) {
      d = ast_MemberExpr_getPrevLastDecl(m);
      c2_assert((ast_Decl_isEnum(d)) != 0, "generator/dep_finder.c2:235: dep_finder.Finder.handleMemberExpr", "CALL TODO");
   }
   dep_finder_Finder_onDep(s, d, true);
}

static void dep_finder_Finder_onDep(dep_finder_Finder* s, const ast_Decl* d, bool full)
{
   if ((ast_Decl_getModule(d) != s->mod)) return;

   if (ast_Decl_isGenerated(d)) return;

   s->on_decl(s->arg, ((ast_Decl*)(d)));
}


// --- module attr_handler ---
typedef struct attr_handler_Entry_ attr_handler_Entry;
typedef struct attr_handler_Handler_ attr_handler_Handler;

struct attr_handler_Entry_ {
   uint32_t name;
   ast_AttrHandlerFn func;
   void* arg;
};

struct attr_handler_Handler_ {
   diagnostics_Diags* diags;
   attr_handler_Entry* entries;
   uint32_t count;
   uint32_t capacity;
};

static attr_handler_Handler* attr_handler_create(diagnostics_Diags* diags);
static void attr_handler_Handler_free(attr_handler_Handler* h);
static bool attr_handler_Handler_register(attr_handler_Handler* h, uint32_t name, ast_AttrHandlerFn func, void* arg);
static bool attr_handler_Handler_handle(attr_handler_Handler* h, ast_Decl* d, const attr_Attr* a);
static attr_handler_Handler* attr_handler_create(diagnostics_Diags* diags)
{
   attr_handler_Handler* h = calloc(1, 24);
   h->diags = diags;
   return h;
}

static void attr_handler_Handler_free(attr_handler_Handler* h)
{
   if (h->entries) free(h->entries);
   free(h);
}

static bool attr_handler_Handler_register(attr_handler_Handler* h, uint32_t name, ast_AttrHandlerFn func, void* arg)
{
   if ((h->count == h->capacity)) {
      h->capacity += 2;
      attr_handler_Entry* entries2 = malloc((h->capacity * 24));
      if (h->count) {
         memcpy(entries2, h->entries, (h->count * 24));
         free(h->entries);
      }
      h->entries = entries2;
   }
   attr_handler_Entry* e = &h->entries[h->count];
   e->name = name;
   e->func = func;
   e->arg = arg;
   h->count++;
   return true;
}

static bool attr_handler_Handler_handle(attr_handler_Handler* h, ast_Decl* d, const attr_Attr* a)
{
   for (uint32_t i = 0; (i < h->count); i++) {
      attr_handler_Entry* e = &h->entries[i];
      if ((e->name == a->name)) return e->func(e->arg, d, a);

   }
   diagnostics_Diags_error(h->diags, a->loc, "unknown attribute '%s'", ast_idx2name(a->name));
   return false;
}


// --- module module_list ---
typedef struct module_list_List_ module_list_List;

typedef void (*module_list_Visitor)(void* arg, ast_Module* m);

struct module_list_List_ {
   ast_Module** mods;
   uint32_t num_mods;
   uint32_t max_mods;
   bool owns_modules;
};

static void module_list_List_init(module_list_List* l, bool owns_modules);
static void module_list_List_free(module_list_List* l);
static void module_list_List_visit(const module_list_List* l, module_list_Visitor visitor, void* arg);
static ast_Module* module_list_List_find(const module_list_List* l, uint32_t module_name);
static void module_list_List_add(module_list_List* list, ast_Module* m);
static uint32_t module_list_List_length(const module_list_List* l);
static ast_Module** module_list_List_get(const module_list_List* l);
static const ast_Module** module_list_List_getConst(const module_list_List* l);
static ast_Module* module_list_List_at(const module_list_List* l, uint32_t idx);
static bool module_list_List_contains(const module_list_List* l, const ast_Module* mod);
static void module_list_List_resize(module_list_List* l, uint32_t cap);
static void module_list_List_init(module_list_List* l, bool owns_modules)
{
   memset(l, 0, 24);
   l->owns_modules = owns_modules;
   module_list_List_resize(l, 4);
}

static void module_list_List_free(module_list_List* l)
{
   if (l->owns_modules) {
      for (uint32_t i = 0; (i < l->num_mods); i++) {
         ast_Module_free(l->mods[i]);
      }
   }
   free(((void*)(l->mods)));
}

static void module_list_List_visit(const module_list_List* l, module_list_Visitor visitor, void* arg)
{
   for (uint32_t i = 0; (i < l->num_mods); i++) {
      visitor(arg, l->mods[i]);
   }
}

static ast_Module* module_list_List_find(const module_list_List* l, uint32_t module_name)
{
   for (uint32_t i = 0; (i < l->num_mods); i++) {
      if ((ast_Module_getNameIdx(l->mods[i]) == module_name)) return l->mods[i];

   }
   return NULL;
}

static void module_list_List_add(module_list_List* list, ast_Module* m)
{
   if ((list->num_mods == list->max_mods)) module_list_List_resize(list, (list->max_mods * 2));
   list->mods[list->num_mods] = m;
   list->num_mods++;
}

static uint32_t module_list_List_length(const module_list_List* l)
{
   return l->num_mods;
}

static ast_Module** module_list_List_get(const module_list_List* l)
{
   return l->mods;
}

static const ast_Module** module_list_List_getConst(const module_list_List* l)
{
   return ((const ast_Module**)(l->mods));
}

static ast_Module* module_list_List_at(const module_list_List* l, uint32_t idx)
{
   return l->mods[idx];
}

static bool module_list_List_contains(const module_list_List* l, const ast_Module* mod)
{
   for (uint32_t i = 0; (i < l->num_mods); i++) {
      if ((l->mods[i] == mod)) return true;

   }
   return false;
}

static void module_list_List_resize(module_list_List* l, uint32_t cap)
{
   l->max_mods = cap;
   void* buf = malloc((l->max_mods * 8));
   if (l->mods) {
      void* old = ((void*)(l->mods));
      memcpy(buf, old, (l->num_mods * 8));
      free(old);
   }
   l->mods = buf;
}


// --- module case_list ---
typedef struct case_list_List_ case_list_List;

#define case_list_StackSize 4
struct case_list_List_ {
   uint32_t count;
   uint32_t capacity;
   ast_SwitchCase* stack[4];
   ast_SwitchCase** heap;
};

#define case_list_InitialHeapSize 8
static void case_list_List_init(case_list_List* l);
static void case_list_List_add(case_list_List* l, ast_SwitchCase* s);
static void case_list_List_free(case_list_List* l);
static uint32_t case_list_List_size(const case_list_List* l);
static ast_SwitchCase** case_list_List_getData(case_list_List* l);
static void case_list_List_init(case_list_List* l)
{
   memset(l, 0, 48);
}

static void case_list_List_add(case_list_List* l, ast_SwitchCase* s)
{
   if ((l->count < case_list_StackSize)) {
      l->stack[l->count] = s;
   } else if ((l->count > case_list_StackSize)) {
      if ((l->count == l->capacity)) {
         l->capacity *= 2;
         ast_SwitchCase** heap2 = malloc((l->capacity * 8));
         memcpy(((void*)(heap2)), ((void*)(l->heap)), (l->count * 8));
         free(((void*)(l->heap)));
         l->heap = heap2;
      }
      l->heap[l->count] = s;
   } else {
      l->capacity = case_list_InitialHeapSize;
      l->heap = malloc((l->capacity * 8));
      memcpy(((void*)(l->heap)), ((void*)(l->stack)), (case_list_StackSize * 8));
      l->heap[l->count] = s;
   }

   l->count++;
}

static void case_list_List_free(case_list_List* l)
{
   if (l->heap) free(((void*)(l->heap)));
}

static uint32_t case_list_List_size(const case_list_List* l)
{
   return l->count;
}

static ast_SwitchCase** case_list_List_getData(case_list_List* l)
{
   if (l->heap) return l->heap;

   return l->stack;
}


// --- module identifier_expr_list ---
typedef struct identifier_expr_list_List_ identifier_expr_list_List;

#define identifier_expr_list_MaxSize 32
struct identifier_expr_list_List_ {
   uint32_t count;
   ast_IdentifierExpr* data[32];
};

static void identifier_expr_list_List_init(identifier_expr_list_List* l);
static bool identifier_expr_list_List_add(identifier_expr_list_List* l, ast_IdentifierExpr* i);
static uint32_t identifier_expr_list_List_size(const identifier_expr_list_List* l);
static ast_IdentifierExpr** identifier_expr_list_List_getData(identifier_expr_list_List* l);
static void identifier_expr_list_List_init(identifier_expr_list_List* l)
{
   l->count = 0;
}

static bool identifier_expr_list_List_add(identifier_expr_list_List* l, ast_IdentifierExpr* i)
{
   if ((l->count == identifier_expr_list_MaxSize)) return false;

   l->data[l->count] = i;
   l->count++;
   return true;
}

static uint32_t identifier_expr_list_List_size(const identifier_expr_list_List* l)
{
   return l->count;
}

static ast_IdentifierExpr** identifier_expr_list_List_getData(identifier_expr_list_List* l)
{
   return l->data;
}


// --- module stmt_list ---
typedef struct stmt_list_List_ stmt_list_List;

#define stmt_list_StackSize 4
struct stmt_list_List_ {
   uint32_t count;
   uint32_t capacity;
   ast_Stmt* stack[4];
   ast_Stmt** heap;
};

#define stmt_list_InitialHeapSize 8
static void stmt_list_List_init(stmt_list_List* l);
static void stmt_list_List_add(stmt_list_List* l, ast_Stmt* s);
static void stmt_list_List_free(stmt_list_List* l);
static uint32_t stmt_list_List_size(const stmt_list_List* l);
static ast_Stmt** stmt_list_List_getData(stmt_list_List* l);
static void stmt_list_List_init(stmt_list_List* l)
{
   memset(l, 0, 48);
}

static void stmt_list_List_add(stmt_list_List* l, ast_Stmt* s)
{
   if ((l->count < stmt_list_StackSize)) {
      l->stack[l->count] = s;
   } else if ((l->count > stmt_list_StackSize)) {
      if ((l->count == l->capacity)) {
         l->capacity *= 2;
         ast_Stmt** heap2 = malloc((l->capacity * 8));
         memcpy(((void*)(heap2)), ((void*)(l->heap)), (l->count * 8));
         free(((void*)(l->heap)));
         l->heap = heap2;
      }
      l->heap[l->count] = s;
   } else {
      l->capacity = stmt_list_InitialHeapSize;
      l->heap = malloc((l->capacity * 8));
      memcpy(((void*)(l->heap)), ((void*)(l->stack)), (stmt_list_StackSize * 8));
      l->heap[l->count] = s;
   }

   l->count++;
}

static void stmt_list_List_free(stmt_list_List* l)
{
   if (l->heap) free(((void*)(l->heap)));
}

static uint32_t stmt_list_List_size(const stmt_list_List* l)
{
   return l->count;
}

static ast_Stmt** stmt_list_List_getData(stmt_list_List* l)
{
   if (l->heap) return l->heap;

   return l->stack;
}


// --- module scope ---
typedef struct scope_Level_ scope_Level;
typedef struct scope_Scope_ scope_Scope;

struct scope_Level_ {
   uint32_t flags;
   uint32_t first_index;
};

#define scope_MaxLevels 32
struct scope_Scope_ {
   const module_list_List* allmodules;
   diagnostics_Diags* diags;
   const ast_ImportDeclList* imports;
   const ast_Module* mod;
   const ast_SymbolTable* symbols;
   bool warn_on_unused;
   ast_SymbolTable local_scope;
   scope_Level levels[32];
   uint32_t lvl;
};

#define scope_Function 0x1
#define scope_Break 0x2
#define scope_Continue 0x4
#define scope_Decl 0x8
#define scope_Control 0x10
#define scope_Block 0x20
#define scope_Fallthrough 0x40
#define scope_Unreachable 0x80
static scope_Scope* scope_create(module_list_List* allmodules, diagnostics_Diags* diags, const ast_ImportDeclList* imports, ast_Module* mod, const ast_SymbolTable* symbols, bool warn_on_unused);
static void scope_Scope_free(scope_Scope* s);
static void scope_Scope_reset(scope_Scope* s);
static void scope_Scope_addImports(scope_Scope* s);
static void scope_Scope_enter(scope_Scope* s, uint32_t flags);
static void scope_Scope_exit(scope_Scope* s, bool has_error);
static void scope_Scope_setUnreachable(scope_Scope* s);
static void scope_Scope_setReachable(scope_Scope* s);
static bool scope_Scope_isUnreachable(const scope_Scope* s);
static bool scope_Scope_allowBreak(const scope_Scope* s);
static bool scope_Scope_allowContinue(const scope_Scope* s);
static bool scope_Scope_allowFallthrough(const scope_Scope* s);
static bool scope_Scope_add(scope_Scope* s, ast_Decl* d);
static ast_Decl* scope_Scope_find(scope_Scope* s, uint32_t name_idx, src_loc_SrcLoc loc, bool usedPublic);
static bool scope_Scope_checkGlobalSymbol(scope_Scope* s, uint32_t name_idx, src_loc_SrcLoc loc);
static ast_ImportDecl* scope_Scope_findModule(scope_Scope* s, uint32_t name_idx, src_loc_SrcLoc loc);
static ast_Decl* scope_Scope_findSymbolInModule(scope_Scope* s, ast_Module* mod, uint32_t name_idx, src_loc_SrcLoc loc);
static ast_Decl* scope_Scope_findType(scope_Scope* s, uint32_t name_idx, src_loc_SrcLoc loc, bool usedPublic);
static ast_Decl* scope_Scope_findGlobalSymbol(scope_Scope* s, uint32_t name_idx, src_loc_SrcLoc loc, bool* other_error, bool usedPublic);
static void scope_Scope_dump(const scope_Scope* s);
static bool scope_Scope_checkAccess(scope_Scope* s, ast_Decl* d, src_loc_SrcLoc loc);
static scope_Scope* scope_create(module_list_List* allmodules, diagnostics_Diags* diags, const ast_ImportDeclList* imports, ast_Module* mod, const ast_SymbolTable* symbols, bool warn_on_unused)
{
   scope_Scope* s = calloc(1, 344);
   s->allmodules = allmodules;
   s->diags = diags;
   s->imports = imports;
   s->mod = mod;
   s->symbols = symbols;
   s->warn_on_unused = warn_on_unused;
   ast_SymbolTable_init(&s->local_scope, 64);
   scope_Scope_addImports(s);
   return s;
}

static void scope_Scope_free(scope_Scope* s)
{
   ast_SymbolTable_free(&s->local_scope);
   free(s);
}

static void scope_Scope_reset(scope_Scope* s)
{
   s->lvl = 0;
   uint32_t first_index = ast_ImportDeclList_size(s->imports);
   ast_SymbolTable_crop(&s->local_scope, first_index);
}

static void scope_Scope_addImports(scope_Scope* s)
{
   uint32_t num_imports = ast_ImportDeclList_size(s->imports);
   ast_ImportDecl** imports = ast_ImportDeclList_getDecls(s->imports);
   for (uint32_t i = 0; (i < num_imports); i++) {
      ast_ImportDecl* id = imports[i];
      ast_Decl* d = ((ast_Decl*)(id));
      uint32_t name_idx = ast_ImportDecl_getImportNameIdx(id);
      ast_Decl* decl = ast_SymbolTable_find(&s->local_scope, name_idx);
      if (decl) {
         diagnostics_Diags_error(s->diags, ast_ImportDecl_getLoc(id), "duplicate import name '%s'", ast_idx2name(name_idx));
         c2_assert((ast_Decl_isImport(decl)) != 0, "analyser/scope.c2:114: scope.Scope.addImports", "CALL TODO");
         ast_ImportDecl* other = ((ast_ImportDecl*)(decl));
         diagnostics_Diags_note(s->diags, ast_ImportDecl_getLoc(other), "previous definition is here");
         continue;
      }
      decl = ast_SymbolTable_find(s->symbols, name_idx);
      if (decl) {
         diagnostics_Diags_error(s->diags, ast_ImportDecl_getLoc(id), "import redefinition of '%s'", ast_idx2name(name_idx));
         diagnostics_Diags_note(s->diags, ast_Decl_getLoc(decl), "previous definition is here");
         continue;
      }
      ast_SymbolTable_add(&s->local_scope, name_idx, d);
   }
}

static void scope_Scope_enter(scope_Scope* s, uint32_t flags)
{
   if ((s->lvl == scope_MaxLevels)) {
      diagnostics_Diags_error(s->diags, 0, "max scope depth reached");
      c2_assert((0) != 0, "analyser/scope.c2:136: scope.Scope.enter", "0");
      return;
   }
   scope_Level* top = &s->levels[s->lvl];
   if (s->lvl) {
      const scope_Level* parent = &s->levels[(s->lvl - 1)];
      flags |= ((parent->flags & ((scope_Break | scope_Continue))));
   }
   top->flags = flags;
   top->first_index = ast_SymbolTable_size(&s->local_scope);
   s->lvl++;
}

static void scope_Scope_exit(scope_Scope* s, bool has_error)
{
   c2_assert(((s->lvl != 0)) != 0, "analyser/scope.c2:152: scope.Scope.exit", "s.lvl!=0");
   s->lvl--;
   uint32_t first = s->levels[s->lvl].first_index;
   if ((s->warn_on_unused && !has_error)) {
      uint32_t last = ast_SymbolTable_size(&s->local_scope);
      ast_Decl** decls = ast_SymbolTable_getDecls(&s->local_scope);
      for (uint32_t i = first; (i < last); i++) {
         ast_Decl* d = decls[i];
         if (!ast_Decl_isUsed(d)) {
            ast_VarDecl* vd = ((ast_VarDecl*)(d));
            if (ast_VarDecl_isLocal(vd)) diagnostics_Diags_warn(s->diags, ast_Decl_getLoc(d), "unused variable '%s'", ast_Decl_getName(d));
         }
      }
   }
   ast_SymbolTable_crop(&s->local_scope, first);
}

static void scope_Scope_setUnreachable(scope_Scope* s)
{
   c2_assert((s->lvl) != 0, "analyser/scope.c2:175: scope.Scope.setUnreachable", "s.lvl");
   scope_Level* top = &s->levels[(s->lvl - 1)];
   top->flags |= scope_Unreachable;
}

static void scope_Scope_setReachable(scope_Scope* s)
{
   c2_assert((s->lvl) != 0, "analyser/scope.c2:181: scope.Scope.setReachable", "s.lvl");
   scope_Level* top = &s->levels[(s->lvl - 1)];
   top->flags &= ~scope_Unreachable;
}

static bool scope_Scope_isUnreachable(const scope_Scope* s)
{
   c2_assert((s->lvl) != 0, "analyser/scope.c2:187: scope.Scope.isUnreachable", "s.lvl");
   const scope_Level* top = &s->levels[(s->lvl - 1)];
   return ((top->flags & scope_Unreachable));
}

static bool scope_Scope_allowBreak(const scope_Scope* s)
{
   c2_assert((s->lvl) != 0, "analyser/scope.c2:193: scope.Scope.allowBreak", "s.lvl");
   const scope_Level* top = &s->levels[(s->lvl - 1)];
   return ((top->flags & scope_Break));
}

static bool scope_Scope_allowContinue(const scope_Scope* s)
{
   c2_assert((s->lvl) != 0, "analyser/scope.c2:199: scope.Scope.allowContinue", "s.lvl");
   const scope_Level* top = &s->levels[(s->lvl - 1)];
   return ((top->flags & scope_Continue));
}

static bool scope_Scope_allowFallthrough(const scope_Scope* s)
{
   c2_assert((s->lvl) != 0, "analyser/scope.c2:205: scope.Scope.allowFallthrough", "s.lvl");
   const scope_Level* top = &s->levels[(s->lvl - 1)];
   return ((top->flags & scope_Fallthrough));
}

static bool scope_Scope_add(scope_Scope* s, ast_Decl* d)
{
   c2_assert((s->lvl) != 0, "analyser/scope.c2:212: scope.Scope.add", "s.lvl");
   const uint32_t name_idx = ast_Decl_getNameIdx(d);
   ast_Decl* decl = ast_SymbolTable_find(&s->local_scope, name_idx);
   if (decl) {
      diagnostics_Diags_error(s->diags, ast_Decl_getLoc(d), "redefinition of '%s'", ast_Decl_getName(decl));
      diagnostics_Diags_note(s->diags, ast_Decl_getLoc(decl), "previous definition is here");
      return true;
   }
   bool other_error = false;
   decl = scope_Scope_findGlobalSymbol(s, name_idx, ast_Decl_getLoc(d), &other_error, false);
   if (decl) {
      diagnostics_Diags_error(s->diags, ast_Decl_getLoc(d), "redefinition of '%s'", ast_Decl_getName(decl));
      diagnostics_Diags_note(s->diags, ast_Decl_getLoc(decl), "previous definition is here");
      return true;
   }
   ast_SymbolTable_add(&s->local_scope, name_idx, d);
   return false;
}

static ast_Decl* scope_Scope_find(scope_Scope* s, uint32_t name_idx, src_loc_SrcLoc loc, bool usedPublic)
{
   ast_Decl* decl = ast_SymbolTable_find(&s->local_scope, name_idx);
   if (decl) {
      if (ast_Decl_isImport(decl)) {
         ast_Decl_setUsed(decl);
         if (usedPublic) ast_Decl_setUsedPublic(decl);
      }
      return decl;
   }
   bool other_error = false;
   decl = scope_Scope_findGlobalSymbol(s, name_idx, loc, &other_error, usedPublic);
   if ((!decl && !other_error)) {
      const char* name = ast_idx2name(name_idx);
      ast_ImportDecl* id = ast_ImportDeclList_findAny(s->imports, name_idx);
      if (id) {
         diagnostics_Diags_error(s->diags, loc, "module '%s' is imported with alias '%s'", ast_Decl_getName(ast_ImportDecl_asDecl(id)), ast_ImportDecl_getAliasName(id));
      } else {
         diagnostics_Diags_error(s->diags, loc, "use of undeclared identifier '%s'", name);
      }
      return NULL;
   }
   return decl;
}

static bool scope_Scope_checkGlobalSymbol(scope_Scope* s, uint32_t name_idx, src_loc_SrcLoc loc)
{
   uint32_t num_imports = ast_ImportDeclList_size(s->imports);
   ast_ImportDecl** imports = ast_ImportDeclList_getDecls(s->imports);
   ast_Decl* decl = NULL;
   for (uint32_t i = 0; (i < num_imports); i++) {
      ast_ImportDecl* id = imports[i];
      if ((name_idx == ast_ImportDecl_getImportNameIdx(id))) {
         decl = ((ast_Decl*)(id));
         break;
      }
      if (ast_ImportDecl_isLocal(id)) {
         ast_Module* dest = ast_ImportDecl_getDest(id);
         decl = ast_Module_findSymbol(dest, name_idx);
         if (decl) break;

      }
   }
   if (decl) {
      diagnostics_Diags_error(s->diags, loc, "redefinition of '%s'", ast_idx2name(name_idx));
      diagnostics_Diags_note(s->diags, ast_Decl_getLoc(decl), "previous definition is here");
      return false;
   }
   return true;
}

static ast_ImportDecl* scope_Scope_findModule(scope_Scope* s, uint32_t name_idx, src_loc_SrcLoc loc)
{
   c2_assert((s) != NULL, "analyser/scope.c2:300: scope.Scope.findModule", "s");
   ast_ImportDecl* d = ast_ImportDeclList_find(s->imports, name_idx);
   if (d) {
      ast_Decl_setUsed(ast_ImportDecl_asDecl(d));
      return d;
   }
   d = ast_ImportDeclList_findAny(s->imports, name_idx);
   if (d) {
      diagnostics_Diags_error(s->diags, loc, "module '%s' is imported with alias '%s'", ast_Decl_getName(ast_ImportDecl_asDecl(d)), ast_ImportDecl_getAliasName(d));
      return NULL;
   }
   ast_Module* mod = module_list_List_find(s->allmodules, name_idx);
   if (mod) {
      diagnostics_Diags_error(s->diags, loc, "module %s not imported", ast_idx2name(name_idx));
   } else {
      diagnostics_Diags_error(s->diags, loc, "unknown module: '%s'", ast_idx2name(name_idx));
   }
   return NULL;
}

static ast_Decl* scope_Scope_findSymbolInModule(scope_Scope* s, ast_Module* mod, uint32_t name_idx, src_loc_SrcLoc loc)
{
   c2_assert((s) != NULL, "analyser/scope.c2:324: scope.Scope.findSymbolInModule", "s");
   ast_Decl* d = ast_Module_findSymbol(mod, name_idx);
   if (!d) {
      diagnostics_Diags_error(s->diags, loc, "module '%s' has no symbol '%s'", ast_Module_getName(mod), ast_idx2name(name_idx));
      return NULL;
   }
   if ((mod != s->mod)) {
      if (!ast_Decl_isPublic(d)) {
         diagnostics_Diags_error(s->diags, loc, "symbol '%s' is not public", ast_Decl_getFullName(d));
         return NULL;
      }
      ast_Decl_setUsedPublic(d);
   }
   return d;
}

static ast_Decl* scope_Scope_findType(scope_Scope* s, uint32_t name_idx, src_loc_SrcLoc loc, bool usedPublic)
{
   c2_assert((s) != NULL, "analyser/scope.c2:346: scope.Scope.findType", "s");
   bool other_error = false;
   ast_Decl* decl = scope_Scope_findGlobalSymbol(s, name_idx, loc, &other_error, usedPublic);
   if ((!decl && !other_error)) {
      diagnostics_Diags_error(s->diags, loc, "unknown type '%s'", ast_idx2name(name_idx));
   }
   return decl;
}

static ast_Decl* scope_Scope_findGlobalSymbol(scope_Scope* s, uint32_t name_idx, src_loc_SrcLoc loc, bool* other_error, bool usedPublic)
{
   ast_Decl* decl = NULL;
   ast_ImportDecl* used_import = NULL;
   bool ambiguous = false;
   bool visible_match = false;
   uint32_t num_imports = ast_ImportDeclList_size(s->imports);
   ast_ImportDecl** imports = ast_ImportDeclList_getDecls(s->imports);
   for (uint32_t i = 0; (i < num_imports); i++) {
      ast_ImportDecl* id = imports[i];
      if ((name_idx == ast_ImportDecl_getImportNameIdx(id))) {
         decl = ((ast_Decl*)(id));
         used_import = id;
         visible_match = true;
         continue;
      }
      if (ast_ImportDecl_isLocal(id)) {
         ast_Module* dest = ast_ImportDecl_getDest(id);
         ast_Decl* d = ast_Module_findSymbol(dest, name_idx);
         if (!d) continue;

         bool visible = !((((s->mod != dest)) && !ast_Decl_isPublic(d)));
         if (decl) {
            if ((visible_match == visible)) {
               const char* name = ast_idx2name(name_idx);
               const char* mod2_name = ast_idx2name(ast_ImportDecl_getImportNameIdx(id));
               if (!ambiguous) {
                  diagnostics_Diags_error(s->diags, loc, "symbol '%s' is ambiguous", name);
                  const char* mod1_name = ast_idx2name(ast_ImportDecl_getImportNameIdx(used_import));
                  diagnostics_Diags_note(s->diags, ast_Decl_getLoc(decl), "did you mean '%s'?", ast_Decl_getFullName(decl));
                  diagnostics_Diags_note(s->diags, ast_Decl_getLoc(d), "did you mean '%s'?", ast_Decl_getFullName(d));
                  ambiguous = true;
                  *other_error = true;
               } else {
                  diagnostics_Diags_note(s->diags, ast_Decl_getLoc(d), "did you mean '%s'?", ast_Decl_getFullName(d));
               }
               continue;
            }
            if (!visible_match) {
               decl = d;
               used_import = id;
               visible_match = visible;
            }
         } else {
            decl = d;
            used_import = id;
            visible_match = visible;
         }
      }
   }
   if (ambiguous) return NULL;

   if (decl) {
      bool external = (((ast_ImportDecl_getDest(used_import) != s->mod)) && ((((ast_Decl*)(used_import)) != decl)));
      ast_Decl_setUsed(ast_ImportDecl_asDecl(used_import));
      if (!visible_match) {
         diagnostics_Diags_error(s->diags, loc, "symbol '%s' is not public", ast_Decl_getFullName(decl));
         *other_error = true;
         return NULL;
      }
      if ((external || usedPublic)) {
         ast_Decl_setUsedPublic(decl);
         if (usedPublic) ast_Decl_setUsedPublic(ast_ImportDecl_asDecl(used_import));
      }
   }
   return decl;
}

static void scope_Scope_dump(const scope_Scope* s)
{
   ast_SymbolTable_dump(&s->local_scope);
   printf("Scope (lvl %u) %u\n", s->lvl, ast_SymbolTable_size(&s->local_scope));
   for (uint32_t i = 0; (i < s->lvl); i++) {
      const scope_Level* l = &s->levels[i];
      printf("  [%u]  start %2u  flags 0x%02x\n", i, l->first_index, l->flags);
   }
}

static bool scope_Scope_checkAccess(scope_Scope* s, ast_Decl* d, src_loc_SrcLoc loc)
{
   bool external = ((s->mod != ast_Decl_getModule(d)));
   if ((!ast_Decl_isPublic(d) && external)) {
      diagnostics_Diags_error(s->diags, loc, "symbol '%s' is not public", ast_Decl_getFullName(d));
      return false;
   }
   return true;
}


// --- module size_analyser ---
typedef struct size_analyser_TypeSize_ size_analyser_TypeSize;

struct size_analyser_TypeSize_ {
   uint32_t size;
   uint32_t align;
   uint32_t bitfield_size;
   uint32_t bitfield_width;
};

static size_analyser_TypeSize size_analyser_sizeOfUnion(ast_StructTypeDecl* s);
static size_analyser_TypeSize size_analyser_sizeOfStruct(ast_StructTypeDecl* s);
static size_analyser_TypeSize size_analyser_sizeOfType(ast_QualType qt);
static size_analyser_TypeSize size_analyser_sizeOfUnion(ast_StructTypeDecl* s)
{
   size_analyser_TypeSize result = { 0, 1, 0, 0 };
   result.align = ast_StructTypeDecl_getAttrAlignment(s);
   uint32_t num_members = ast_StructTypeDecl_getNumMembers(s);
   ast_Decl** members = ast_StructTypeDecl_getMembers(s);
   for (uint32_t i = 0; (i < num_members); i++) {
      ast_Decl* d = members[i];
      size_analyser_TypeSize m_size = size_analyser_sizeOfType(ast_Decl_getType(d));
      if ((m_size.size > result.size)) result.size = m_size.size;
      if ((m_size.align > result.align)) result.align = m_size.align;
   }
   return result;
}

static size_analyser_TypeSize size_analyser_sizeOfStruct(ast_StructTypeDecl* s)
{
   if (ast_StructTypeDecl_isUnion(s)) return size_analyser_sizeOfUnion(s);

   size_analyser_TypeSize result = { 0, 1, 0, 0 };
   bool packed = ast_StructTypeDecl_isPacked(s);
   result.align = ast_StructTypeDecl_getAttrAlignment(s);
   uint32_t num_members = ast_StructTypeDecl_getNumMembers(s);
   ast_Decl** members = ast_StructTypeDecl_getMembers(s);
   if (packed) {
      for (uint32_t i = 0; (i < num_members); i++) {
         ast_Decl* d = members[i];
         size_analyser_TypeSize member = size_analyser_sizeOfType(ast_Decl_getType(d));
         ast_StructTypeDecl_setMemberOffset(s, i, result.size);
         result.size += member.size;
      }
   } else {
      for (uint32_t i = 0; (i < num_members); i++) {
         ast_Decl* d = members[i];
         size_analyser_TypeSize member = size_analyser_sizeOfType(ast_Decl_getType(d));
         ast_VarDecl* vd = ast_Decl_isVariable(d) ? ((ast_VarDecl*)(d)) : NULL;
         if (vd) {
            const ast_Expr* bitfield = ast_VarDecl_getBitfield(vd);
            if (bitfield) {
               ast_Value value = ctv_analyser_get_value(bitfield);
               member.bitfield_size = ((uint32_t)(value.uvalue));
               member.bitfield_width = (member.size * 8);
               member.size = 0;
               member.align = 0;
            }
         }
         if ((result.bitfield_width && (member.align != 0))) {
            uint32_t bytesize = (((result.bitfield_size + 7)) / 8);
            result.size += bytesize;
            if ((bytesize > result.align)) result.align = bytesize;
            result.bitfield_width = 0;
            result.bitfield_size = 0;
         }
         if ((member.align > 1)) {
            if ((member.align > result.align)) result.align = member.align;
            uint32_t rest = (result.size % member.align);
            if ((rest != 0)) {
               uint32_t pad = (member.align - rest);
               result.size += pad;
            }
         }
         ast_StructTypeDecl_setMemberOffset(s, i, result.size);
         if (member.bitfield_width) {
            uint32_t total_bitsize = (result.bitfield_size + member.bitfield_size);
            if ((total_bitsize > member.bitfield_width)) {
               uint32_t bytesize = (((result.bitfield_size + 7)) / 8);
               member.align = bytesize;
               if ((bytesize > 1)) {
                  uint32_t rest = (result.size % bytesize);
                  if ((rest != 0)) {
                     uint32_t pad = (member.align - rest);
                     result.size += pad;
                  }
               }
               result.size += bytesize;
               result.bitfield_size = member.bitfield_size;
               result.bitfield_width = member.bitfield_width;
            } else {
               result.bitfield_size = total_bitsize;
               result.bitfield_width = member.bitfield_width;
            }
         } else {
            result.size += member.size;
         }
      }
      if (result.bitfield_width) {
         uint32_t bytesize = (((result.bitfield_size + 7)) / 8);
         result.size += bytesize;
      }
      uint32_t rest = (result.size % result.align);
      if ((rest != 0)) {
         uint32_t pad = (result.align - rest);
         result.size += pad;
      }
   }
   return result;
}

static size_analyser_TypeSize size_analyser_sizeOfType(ast_QualType qt)
{
   size_analyser_TypeSize result = { 0, 1, 0, 0 };
   if (ast_QualType_isInvalid(&qt)) return result;

   uint32_t pointerSize = ast_getWordSize();
   qt = ast_QualType_getCanonicalType(&qt);
   ast_Type* t = ast_QualType_getType(&qt);
   switch (ast_Type_getKind(t)) {
   case ast_TypeKind_Builtin: {
      const ast_BuiltinType* bi = ((ast_BuiltinType*)(t));
      result.size = ast_BuiltinType_getAlignment(bi);
      result.align = result.size;
      break;
   }
   case ast_TypeKind_Pointer:
      result.size = pointerSize;
      result.align = result.size;
      break;
   case ast_TypeKind_Array: {
      ast_ArrayType* arrayType = ((ast_ArrayType*)(t));
      result = size_analyser_sizeOfType(ast_ArrayType_getElemType(arrayType));
      result.size *= ast_ArrayType_getSize(arrayType);
      break;
   }
   case ast_TypeKind_Struct: {
      ast_StructType* st = ((ast_StructType*)(t));
      ast_StructTypeDecl* d = ast_StructType_getDecl(st);
      c2_assert((ast_Decl_isChecked(ast_StructTypeDecl_asDecl(d))) != 0, "analyser/size_analyser.c2:192: size_analyser.sizeOfType", "CALL TODO");
      result.size = ast_StructTypeDecl_getSize(d);
      result.align = ast_StructTypeDecl_getAlignment(d);
      break;
   }
   case ast_TypeKind_Enum: {
      ast_EnumType* et = ((ast_EnumType*)(t));
      ast_EnumTypeDecl* etd = ast_EnumType_getDecl(et);
      return size_analyser_sizeOfType(ast_EnumTypeDecl_getImplType(etd));
   }
   case ast_TypeKind_Function:
      result.size = pointerSize;
      result.align = pointerSize;
      break;
   case ast_TypeKind_Alias:
      c2_assert((0) != 0, "analyser/size_analyser.c2:205: size_analyser.sizeOfType", "0");
      break;
   case ast_TypeKind_Module:
      c2_assert((0) != 0, "analyser/size_analyser.c2:208: size_analyser.sizeOfType", "0");
      break;
   }
   return result;
}


// --- module component ---
typedef struct component_Component_ component_Component;
typedef struct component_List_ component_List;

typedef enum {
   component_Kind_Internal,
   component_Kind_Image,
   component_Kind_Executable,
   component_Kind_StaticLibrary,
   component_Kind_DynamicLibrary,
   component_Kind_ExternalStatic,
   component_Kind_ExternalDynamic,
   _component_Kind_max = 255
} __attribute__((packed)) component_Kind;

struct component_Component_ {
   uint32_t name_idx;
   uint32_t dirname_idx;
   uint32_t linkname;
   component_Kind kind;
   bool is_direct;
   bool available_static;
   bool available_dynamic;
   ast_context_Context* context;
   string_pool_Pool* auxPool;
   module_list_List* allmodules;
   module_list_List mods;
   string_list_List deps;
};

struct component_List_ {
   component_Component** components;
   uint32_t count;
   uint32_t capacity;
};

static const char* component_kind_names[7] = {
   "internal",
   "image",
   "executable",
   "static library",
   "dynamic library",
   "external static library",
   "external dynamic library"
};

static component_Component* component_create(ast_context_Context* context, string_pool_Pool* auxPool, module_list_List* allmodules, uint32_t name_idx, component_Kind kind, bool is_direct);
static void component_Component_free(component_Component* c);
static const char* component_Component_getName(const component_Component* c);
static uint32_t component_Component_getNameIdx(const component_Component* c);
static void component_Component_setPath(component_Component* c, uint32_t dirname);
static const char* component_Component_getPath(const component_Component* c);
static void component_Component_setLinkName(component_Component* c, const char* name);
static const char* component_Component_getLinkName(const component_Component* c);
static bool component_Component_isExternal(const component_Component* c);
static bool component_Component_isInternal(const component_Component* c);
static bool component_Component_isDirect(const component_Component* c);
static void component_Component_setKind(component_Component* c, bool is_static, bool is_dynamic);
static bool component_Component_isStaticLib(const component_Component* c);
static bool component_Component_isAvailableStatic(const component_Component* c);
static bool component_Component_isAvailableDynamic(const component_Component* c);
static void component_Component_visitModules(const component_Component* c, module_list_Visitor visitor, void* arg);
static module_list_List* component_Component_getModules(component_Component* c);
static ast_Module* component_Component_getOrAddModule(component_Component* c, uint32_t name_idx);
static ast_Module* component_Component_createModule(component_Component* c, uint32_t name_idx);
static bool component_Component_hasModule(const component_Component* c, const ast_Module* mod);
static void component_Component_print(const component_Component* c, bool show_funcs);
static void component_Component_printModules(const component_Component* c);
static void component_Component_printSymbols(const component_Component* c, bool print_external);
static void component_Component_addDep(component_Component* c, const char* dep);
static string_list_List* component_Component_getDeps(component_Component* c);
static void component_List_init(component_List* l);
static void component_List_free(component_List* l);
static void component_List_resize(component_List* l, uint32_t capacity);
static void component_List_add(component_List* l, component_Component* c);
static uint32_t component_List_size(const component_List* l);
static component_Component* component_List_get(component_List* l, uint32_t idx);
static component_Component** component_List_get_all(component_List* l);
static bool component_Component_isLibrary(const component_Component* c);
static component_Kind component_Component_getKind(const component_Component* c);
static component_Component* component_create(ast_context_Context* context, string_pool_Pool* auxPool, module_list_List* allmodules, uint32_t name_idx, component_Kind kind, bool is_direct)
{
   component_Component* c = calloc(1, 88);
   c->name_idx = name_idx;
   c->kind = kind;
   c->is_direct = is_direct;
   c->context = context;
   c->auxPool = auxPool;
   c->allmodules = allmodules;
   module_list_List_init(&c->mods, true);
   string_list_List_init(&c->deps, auxPool);
   return c;
}

static void component_Component_free(component_Component* c)
{
   module_list_List_free(&c->mods);
   string_list_List_free(&c->deps);
   free(c);
}

static const char* component_Component_getName(const component_Component* c)
{
   return string_pool_Pool_idx2str(c->auxPool, c->name_idx);
}

static uint32_t component_Component_getNameIdx(const component_Component* c)
{
   return c->name_idx;
}

static void component_Component_setPath(component_Component* c, uint32_t dirname)
{
   c->dirname_idx = dirname;
}

static const char* component_Component_getPath(const component_Component* c)
{
   return string_pool_Pool_idx2str(c->auxPool, c->dirname_idx);
}

static void component_Component_setLinkName(component_Component* c, const char* name)
{
   c->linkname = string_pool_Pool_addStr(c->auxPool, name, false);
}

static const char* component_Component_getLinkName(const component_Component* c)
{
   if (c->linkname) return string_pool_Pool_idx2str(c->auxPool, c->linkname);

   return NULL;
}

static bool component_Component_isExternal(const component_Component* c)
{
   return (c->kind >= component_Kind_ExternalStatic);
}

static bool component_Component_isInternal(const component_Component* c)
{
   return (c->kind == component_Kind_Internal);
}

static bool component_Component_isDirect(const component_Component* c)
{
   return c->is_direct;
}

static void component_Component_setKind(component_Component* c, bool is_static, bool is_dynamic)
{
   c->available_static = is_static;
   c->available_dynamic = is_dynamic;
}

static bool component_Component_isStaticLib(const component_Component* c)
{
   return ((c->kind == component_Kind_StaticLibrary) || (c->kind == component_Kind_ExternalStatic));
}

static bool component_Component_isAvailableStatic(const component_Component* c)
{
   return c->available_static;
}

static bool component_Component_isAvailableDynamic(const component_Component* c)
{
   return c->available_dynamic;
}

static void component_Component_visitModules(const component_Component* c, module_list_Visitor visitor, void* arg)
{
   module_list_List_visit(&c->mods, visitor, arg);
}

static module_list_List* component_Component_getModules(component_Component* c)
{
   return &c->mods;
}

static ast_Module* component_Component_getOrAddModule(component_Component* c, uint32_t name_idx)
{
   ast_Module* m = module_list_List_find(&c->mods, name_idx);
   if (m) return m;

   return component_Component_createModule(c, name_idx);
}

static ast_Module* component_Component_createModule(component_Component* c, uint32_t name_idx)
{
   ast_Module* m = module_list_List_find(c->allmodules, name_idx);
   if (m) return NULL;

   m = ast_Module_create(c->context, name_idx, component_Component_isExternal(c), c->is_direct);
   module_list_List_add(&c->mods, m);
   module_list_List_add(c->allmodules, m);
   return m;
}

static bool component_Component_hasModule(const component_Component* c, const ast_Module* mod)
{
   return module_list_List_contains(&c->mods, mod);
}

static void component_Component_print(const component_Component* c, bool show_funcs)
{
   string_buffer_Buf* out = string_buffer_create((128 * 1024), color_useColor(), 1);
   const ast_Module** mods = module_list_List_getConst(&c->mods);
   for (uint32_t i = 0; (i < module_list_List_length(&c->mods)); i++) {
      ast_Module_print(mods[i], out, show_funcs);
   }
   string_buffer_Buf_color(out, color_Normal);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static void component_Component_printModules(const component_Component* c)
{
   string_buffer_Buf* out = string_buffer_create(4096, color_useColor(), 1);
   string_buffer_Buf_print(out, "--- %s [%s]", component_Component_getName(c), component_kind_names[c->kind]);
   if (!c->is_direct) string_buffer_Buf_add(out, " (indirect)");
   string_buffer_Buf_add(out, " ---\n");
   const ast_Module** mods = module_list_List_getConst(&c->mods);
   for (uint32_t i = 0; (i < module_list_List_length(&c->mods)); i++) {
      const ast_Module* m = mods[i];
      const char* col = ast_Module_isUsed(m) ? color_Normal : color_Grey;
      string_buffer_Buf_color(out, col);
      string_buffer_Buf_print(out, "   %s\n", ast_Module_getName(m));
   }
   string_buffer_Buf_color(out, color_Normal);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static void component_Component_printSymbols(const component_Component* c, bool print_external)
{
   if ((component_Component_isExternal(c) != print_external)) return;

   string_buffer_Buf* out = string_buffer_create(4096, color_useColor(), 1);
   string_buffer_Buf_print(out, "--- %s [%s] ---\n", component_Component_getName(c), component_kind_names[c->kind]);
   const ast_Module** mods = module_list_List_getConst(&c->mods);
   for (uint32_t i = 0; (i < module_list_List_length(&c->mods)); i++) {
      string_buffer_Buf_color(out, color_Cyan);
      string_buffer_Buf_print(out, "  %s\n", ast_Module_getName(mods[i]));
      const ast_SymbolTable* table = ast_Module_getSymbols(mods[i]);
      ast_SymbolTable_print(table, out);
   }
   string_buffer_Buf_color(out, color_Normal);
   puts(string_buffer_Buf_data(out));
   string_buffer_Buf_free(out);
}

static void component_Component_addDep(component_Component* c, const char* dep)
{
   uint32_t dep_idx = string_pool_Pool_addStr(c->auxPool, dep, true);
   string_list_List_add(&c->deps, dep_idx);
}

static string_list_List* component_Component_getDeps(component_Component* c)
{
   return &c->deps;
}

static void component_List_init(component_List* l)
{
   l->components = NULL;
   l->count = 0;
   component_List_resize(l, 4);
}

static void component_List_free(component_List* l)
{
   for (uint32_t i = 0; (i < l->count); i++) {
      component_Component_free(l->components[i]);
   }
   free(((void*)(l->components)));
}

static void component_List_resize(component_List* l, uint32_t capacity)
{
   l->capacity = capacity;
   component_Component** comps2 = malloc((l->capacity * 8));
   if (l->count) {
      memcpy(((void*)(comps2)), ((void*)(l->components)), (l->count * 8));
      free(((void*)(l->components)));
   }
   l->components = comps2;
}

static void component_List_add(component_List* l, component_Component* c)
{
   if ((l->count == l->capacity)) component_List_resize(l, (l->capacity * 2));
   l->components[l->count] = c;
   l->count++;
}

static uint32_t component_List_size(const component_List* l)
{
   return l->count;
}

static component_Component* component_List_get(component_List* l, uint32_t idx)
{
   return l->components[idx];
}

static component_Component** component_List_get_all(component_List* l)
{
   return l->components;
}

static bool component_Component_isLibrary(const component_Component* c)
{
   return ((c->kind == component_Kind_StaticLibrary) || (c->kind == component_Kind_DynamicLibrary));
}

static component_Kind component_Component_getKind(const component_Component* c)
{
   return c->kind;
}


// --- module component_sorter ---

static uint32_t component_sorter_find_idx(component_Component** comps, uint32_t count, uint32_t name);
static void component_sorter_sort(component_Component** orig, uint32_t count, diagnostics_Diags* diags);
static uint32_t component_sorter_find_idx(component_Component** comps, uint32_t count, uint32_t name)
{
   for (uint32_t i = 0; (i < count); i++) {
      if ((component_Component_getNameIdx(comps[i]) == name)) return i;

   }
   c2_assert((0) != 0, "common/component_sorter.c2:30: component_sorter.find_idx", "0");
   return 0;
}

static void component_sorter_sort(component_Component** orig, uint32_t count, diagnostics_Diags* diags)
{
   if ((count <= 1)) return;

   dsm_sorter_Sorter dsm;
   dsm_sorter_Sorter_init(&dsm, count);
   for (uint32_t i = 0; (i < count); i++) {
      component_Component* c = orig[i];
      const string_list_List* deps = component_Component_getDeps(c);
      for (uint32_t j = 0; (j < string_list_List_length(deps)); j++) {
         uint32_t dest_name = string_list_List_get_idx(deps, j);
         uint32_t dest_idx = component_sorter_find_idx(orig, count, dest_name);
         dsm_sorter_Sorter_add_dep(&dsm, i, dest_idx);
      }
   }
   const uint8_t* sorted = dsm_sorter_Sorter_sort(&dsm);
   if (!sorted) {
      diagnostics_Diags_error(diags, 0, "circular dependency between libraries");
      exit(-1);
   }
   component_Component** copy = malloc((count * 8));
   memcpy(((void*)(copy)), ((void*)(orig)), (count * 8));
   for (uint32_t i = 0; (i < count); i++) orig[i] = copy[sorted[i]];
   free(((void*)(copy)));
   dsm_sorter_Sorter_free(&dsm);
}


// --- module manifest_writer ---

static const char* manifest_writer_getKindStr(const component_Component* c);
static void manifest_writer_on_module(void* arg, ast_Module* m);
static void manifest_writer_write(const char* dir, component_Component* c, const char* filename);
static const char* manifest_writer_getKindStr(const component_Component* c)
{
   switch (component_Component_getKind(c)) {
   case component_Kind_Internal:
      __attribute__((fallthrough));
   case component_Kind_Image:
      c2_assert((0) != 0, "common/manifest_writer.c2:33: manifest_writer.getKindStr", "0");
      break;
   case component_Kind_Executable:
      c2_assert((0) != 0, "common/manifest_writer.c2:36: manifest_writer.getKindStr", "0");
      break;
   case component_Kind_StaticLibrary:
      return "static";
   case component_Kind_DynamicLibrary:
      return "dynamic";
   case component_Kind_ExternalStatic:
      __attribute__((fallthrough));
   case component_Kind_ExternalDynamic:
      c2_assert((0) != 0, "common/manifest_writer.c2:42: manifest_writer.getKindStr", "0");
      break;
   }
   return "";
}

static void manifest_writer_on_module(void* arg, ast_Module* m)
{
   if (!ast_Module_isExported(m)) return;

   string_buffer_Buf* out = arg;
   string_buffer_Buf_add(out, "[[module]]\n");
   string_buffer_Buf_print(out, "name = \"%s\"\n\n", ast_Module_getName(m));
}

static void manifest_writer_write(const char* dir, component_Component* c, const char* filename)
{
   string_buffer_Buf* out = string_buffer_create(4096, false, 2);
   string_buffer_Buf_add(out, "[library]\n");
   string_buffer_Buf_add(out, "language = \"C2\"\n");
   string_buffer_Buf_print(out, "type = [ \"%s\"\n", manifest_writer_getKindStr(c));
   string_buffer_Buf_print(out, "linkname = \"%s\"\n", component_Component_getName(c));
   string_buffer_Buf_newline(out);
   const string_list_List* deps = component_Component_getDeps(c);
   for (uint32_t i = 0; (i < string_list_List_length(deps)); i++) {
      string_buffer_Buf_add(out, "[[deps]]\n");
      string_buffer_Buf_print(out, "name = \"%s\"\n", string_list_List_get(deps, i));
      string_buffer_Buf_add(out, "type = \"\"\n");
      string_buffer_Buf_newline(out);
   }
   component_Component_visitModules(c, manifest_writer_on_module, out);
   char fullname[512];
   sprintf(fullname, "%s/%s", dir, filename);
   file_utils_Writer writer;
   if (!file_utils_Writer_write(&writer, fullname, ((uint8_t*)(string_buffer_Buf_data(out))), string_buffer_Buf_size(out))) {
      console_error("%s", file_utils_Writer_getError(&writer));
      exit(EXIT_FAILURE);
   }
}


// --- module ast_builder ---
typedef struct ast_builder_Builder_ ast_builder_Builder;

struct ast_builder_Builder_ {
   ast_context_Context* context;
   diagnostics_Diags* diags;
   string_pool_Pool* auxPool;
   component_Component* comp;
   ast_Module* mod;
   ast_AST* ast;
   uint32_t ast_idx;
   uint32_t c2_name;
   uint32_t main_name;
   bool is_interface;
   attr_Attr attrs[8];
   uint32_t num_attrs;
   attr_handler_Handler* attr_handler;
};

static ast_builder_Builder* ast_builder_create(ast_context_Context* context, diagnostics_Diags* diags, string_pool_Pool* auxPool, uint32_t c2_name, uint32_t main_name, attr_handler_Handler* attr_handler_);
static void ast_builder_Builder_free(ast_builder_Builder* b);
static void ast_builder_Builder_setComponent(ast_builder_Builder* b, component_Component* comp);
static void ast_builder_Builder_reserve(ast_builder_Builder* b, uint32_t size);
static void ast_builder_Builder_actOnModule(ast_builder_Builder* b, uint32_t mod_name, src_loc_SrcLoc mod_loc, uint32_t filename, bool is_generated);
static void ast_builder_Builder_actOnImport(ast_builder_Builder* b, uint32_t mod_name, src_loc_SrcLoc mod_loc, uint32_t alias_name, src_loc_SrcLoc alias_loc, bool islocal);
static ast_Decl* ast_builder_Builder_actOnAliasType(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* ref);
static ast_Decl* ast_builder_Builder_actOnFunctionTypeDecl(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* rtype, ast_VarDecl** params, uint32_t num_params, bool is_variadic);
static ast_StructTypeDecl* ast_builder_Builder_actOnStructType(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, bool is_struct, bool is_global, ast_Decl** members, uint32_t num_members);
static ast_VarDecl* ast_builder_Builder_actOnStructMember(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* ref, ast_Expr* bitfield);
static ast_Decl* ast_builder_Builder_actOnGlobalVarDecl(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, ast_TypeRefHolder* ref, src_loc_SrcLoc assignLoc, ast_Expr* initValue);
static ast_VarDecl* ast_builder_Builder_actOnFunctionParam(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* ref);
static ast_Stmt* ast_builder_Builder_actOnVarDeclStmt(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, const ast_TypeRefHolder* ref, src_loc_SrcLoc assignLoc, ast_Expr* initValue, bool has_local);
static bool ast_builder_Builder_hasAttr(const ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a);
static void ast_builder_Builder_storeAttr(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a);
static void ast_builder_Builder_actOnArrayValue(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, ast_Expr* initValue);
static bool ast_builder_Builder_checkAttr(ast_builder_Builder* b, const attr_Attr* a);
static void ast_builder_Builder_actOnFunctionAttr(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a);
static void ast_builder_Builder_actOnStructAttr(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a);
static bool ast_builder_Builder_actOnTypeAttr(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a);
static void ast_builder_Builder_actOnVarAttr(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a);
static void ast_builder_Builder_actOnParamAttr(ast_builder_Builder* b, ast_VarDecl* d, uint32_t name, src_loc_SrcLoc loc);
static void ast_builder_Builder_actOnAttr(ast_builder_Builder* b, attr_Attr* a);
static void ast_builder_Builder_clearAttributes(ast_builder_Builder* b);
static void ast_builder_Builder_applyAttribute(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a);
static void ast_builder_Builder_applyAttributes(ast_builder_Builder* b, ast_Decl* d);
static ast_QualType ast_builder_Builder_actOnBuiltinType(ast_builder_Builder* _arg0, ast_BuiltinKind kind);
static ast_QualType ast_builder_Builder_actOnPointerType(ast_builder_Builder* _arg0, ast_QualType inner);
static ast_QualType ast_builder_Builder_actOnArrayType(ast_builder_Builder* b, ast_QualType elem, bool has_size, uint32_t size);
static ast_QualType ast_builder_Builder_actOnIncrementalArrayType(ast_builder_Builder* b, ast_QualType elem);
static ast_FunctionDecl* ast_builder_Builder_actOnFunctionDecl(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* rtype, const ast_Ref* prefix, ast_VarDecl** params, uint32_t num_params, bool is_variadic);
static ast_FunctionDecl* ast_builder_Builder_actOnTemplateFunctionDecl(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* rtype, uint32_t template_name, src_loc_SrcLoc template_loc, ast_VarDecl** params, uint32_t num_params, bool is_variadic);
static void ast_builder_Builder_actOnFunctionBody(ast_builder_Builder* _arg0, ast_FunctionDecl* f, ast_CompoundStmt* body);
static ast_EnumConstantDecl* ast_builder_Builder_actOnEnumConstant(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, ast_Expr* init_expr);
static ast_Decl* ast_builder_Builder_actOnEnumType(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, bool is_incr, ast_QualType implType, ast_EnumConstantDecl** constants, uint32_t num_constants);
static ast_Stmt* ast_builder_Builder_actOnAsmStmt(ast_builder_Builder* b, src_loc_SrcLoc loc, bool is_basic, bool is_volatile, uint32_t num_outputs, uint32_t num_inputs, const uint32_t* names, ast_ExprList* constraints, ast_ExprList* exprs, ast_ExprList* clobbers, ast_Expr* asm_string);
static ast_CompoundStmt* ast_builder_Builder_actOnCompoundStmt(ast_builder_Builder* b, src_loc_SrcLoc endLoc, ast_Stmt** stmts, uint32_t count);
static ast_Stmt* ast_builder_Builder_actOnReturnStmt(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* ret);
static ast_Stmt* ast_builder_Builder_actOnIfStmt(ast_builder_Builder* b, ast_Stmt* cond, ast_Stmt* then, ast_Stmt* else_stmt);
static ast_Stmt* ast_builder_Builder_actOnDoStmt(ast_builder_Builder* b, ast_Stmt* cond, ast_Stmt* then);
static ast_Stmt* ast_builder_Builder_actOnWhileStmt(ast_builder_Builder* b, ast_Stmt* cond, ast_Stmt* then);
static ast_Stmt* ast_builder_Builder_actOnForStmt(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Stmt* init_, ast_Expr* cond, ast_Expr* incr, ast_Stmt* body);
static ast_Stmt* ast_builder_Builder_actOnSwitchStmt(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* cond, ast_SwitchCase** cases, uint32_t num_cases, bool is_sswitch);
static ast_SwitchCase* ast_builder_Builder_actOnCase(ast_builder_Builder* b, src_loc_SrcLoc loc, bool is_default, ast_Expr* cond, identifier_expr_list_List* conditions, ast_Stmt** stmts, uint32_t num_stmts);
static ast_Stmt* ast_builder_Builder_actOnAssertStmt(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* inner);
static ast_Stmt* ast_builder_Builder_actOnBreakStmt(ast_builder_Builder* b, src_loc_SrcLoc loc);
static ast_Stmt* ast_builder_Builder_actOnContinueStmt(ast_builder_Builder* b, src_loc_SrcLoc loc);
static ast_Stmt* ast_builder_Builder_actOnFallthroughStmt(ast_builder_Builder* b, src_loc_SrcLoc loc);
static ast_Stmt* ast_builder_Builder_actOnLabelStmt(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc);
static ast_Stmt* ast_builder_Builder_actOnGotoStmt(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc);
static ast_IdentifierExpr* ast_builder_Builder_actOnIdentifier(ast_builder_Builder* b, src_loc_SrcLoc loc, uint32_t name);
static ast_Expr* ast_builder_Builder_actOnIntegerLiteral(ast_builder_Builder* b, src_loc_SrcLoc loc, uint8_t radix, uint64_t value);
static ast_Expr* ast_builder_Builder_actOnFloatLiteral(ast_builder_Builder* b, src_loc_SrcLoc loc, double value);
static ast_Expr* ast_builder_Builder_actOnCharLiteral(ast_builder_Builder* b, src_loc_SrcLoc loc, uint8_t value, uint8_t radix);
static ast_Expr* ast_builder_Builder_actOnStringLiteral(ast_builder_Builder* b, src_loc_SrcLoc loc, uint32_t value, uint32_t len);
static ast_Expr* ast_builder_Builder_actOnNilExpr(ast_builder_Builder* b, src_loc_SrcLoc loc);
static ast_Expr* ast_builder_Builder_actOnParenExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* inner);
static ast_Expr* ast_builder_Builder_actOnUnaryOperator(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_UnaryOpcode opcode, ast_Expr* inner);
static ast_Expr* ast_builder_Builder_actOnPostFixUnaryOperator(ast_builder_Builder* b, src_loc_SrcLoc loc, bool is_increment, ast_Expr* inner);
static ast_Expr* ast_builder_Builder_actOnBinaryOperator(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_BinaryOpcode opcode, ast_Expr* lhs, ast_Expr* rhs);
static ast_Expr* ast_builder_Builder_actOnConditionalOperator(ast_builder_Builder* b, src_loc_SrcLoc questionLoc, src_loc_SrcLoc colonLoc, ast_Expr* cond, ast_Expr* lhs, ast_Expr* rhs);
static ast_Expr* ast_builder_Builder_actOnBooleanConstant(ast_builder_Builder* b, src_loc_SrcLoc loc, bool value);
static ast_Expr* ast_builder_Builder_actOnBuiltinExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* inner, ast_BuiltinExprKind kind);
static ast_Expr* ast_builder_Builder_actOnOffsetOfExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* structExpr, ast_Expr* member);
static ast_Expr* ast_builder_Builder_actOnToContainerExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* structExpr, ast_Expr* member, ast_Expr* pointer);
static ast_Expr* ast_builder_Builder_actOnTypeExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, const ast_TypeRefHolder* ref);
static ast_Expr* ast_builder_Builder_actOnBitOffsetExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* lhs, ast_Expr* rhs);
static ast_Expr* ast_builder_Builder_actOnArraySubscriptExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* base, ast_Expr* idx);
static ast_Expr* ast_builder_Builder_actOnCallExpr(ast_builder_Builder* b, src_loc_SrcLoc endLoc, ast_Expr* func, ast_Expr** args, uint32_t num_args);
static ast_Expr* ast_builder_Builder_actOnTemplateCallExpr(ast_builder_Builder* b, src_loc_SrcLoc endLoc, ast_Expr* func, ast_Expr** args, uint32_t num_args, const ast_TypeRefHolder* ref);
static ast_Expr* ast_builder_Builder_actOnExplicitCast(ast_builder_Builder* b, src_loc_SrcLoc loc, const ast_TypeRefHolder* ref, ast_Expr* inner);
static ast_Expr* ast_builder_Builder_actOnMemberExpr(ast_builder_Builder* b, ast_Expr* base, const ast_Ref* refs, uint32_t refcount);
static ast_Expr* ast_builder_Builder_actOnInitList(ast_builder_Builder* b, src_loc_SrcLoc left, src_loc_SrcLoc right, ast_Expr** values, uint32_t num_values);
static ast_Expr* ast_builder_Builder_actOnFieldDesignatedInit(ast_builder_Builder* b, uint32_t field, src_loc_SrcLoc loc, ast_Expr* initValue);
static ast_Expr* ast_builder_Builder_actOnArrayDesignatedInit(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* designator, ast_Expr* initValue);
static void ast_builder_Builder_actOnStaticAssert(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* lhs, ast_Expr* rhs);
static void ast_builder_Builder_insertImplicitCast(ast_builder_Builder* b, ast_ImplicitCastKind kind, ast_Expr** e_ptr, ast_QualType qt);
static void ast_builder_Builder_addSymbol(ast_builder_Builder* b, uint32_t name_idx, ast_Decl* d);
static ast_builder_Builder* ast_builder_create(ast_context_Context* context, diagnostics_Diags* diags, string_pool_Pool* auxPool, uint32_t c2_name, uint32_t main_name, attr_handler_Handler* attr_handler_)
{
   ast_builder_Builder* b = calloc(1, 240);
   b->context = context;
   b->diags = diags;
   b->auxPool = auxPool;
   b->c2_name = c2_name;
   b->main_name = main_name;
   b->attr_handler = attr_handler_;
   return b;
}

static void ast_builder_Builder_free(ast_builder_Builder* b)
{
   free(b);
}

static void ast_builder_Builder_setComponent(ast_builder_Builder* b, component_Component* comp)
{
   b->comp = comp;
   b->mod = NULL;
   b->is_interface = component_Component_isExternal(comp);
}

static void ast_builder_Builder_reserve(ast_builder_Builder* b, uint32_t size)
{
   ast_context_Context_reserve(b->context, size);
}

static void ast_builder_Builder_actOnModule(ast_builder_Builder* b, uint32_t mod_name, src_loc_SrcLoc mod_loc, uint32_t filename, bool is_generated)
{
   c2_assert((b->comp) != NULL, "parser/ast_builder.c2:83: ast_builder.Builder.actOnModule", "b.comp");
   if ((mod_name == b->c2_name)) {
      diagnostics_Diags_error(b->diags, mod_loc, "module name 'c2' is reserved");
      exit(-1);
   }
   if ((mod_name == b->main_name)) {
      diagnostics_Diags_error(b->diags, mod_loc, "module name 'main' is reserved");
      exit(-1);
   }
   b->mod = component_Component_getOrAddModule(b->comp, mod_name);
   if (!b->mod) {
      diagnostics_Diags_error(b->diags, mod_loc, "module '%s' is already defined in another component", ast_idx2name(mod_name));
      exit(-1);
   }
   b->ast = ast_Module_add(b->mod, b->auxPool, filename, is_generated);
   b->ast_idx = ast_AST_getIdx(b->ast);
   ast_ImportDecl* i = ast_ImportDecl_create(b->context, mod_name, mod_loc, 0, 0, b->ast_idx, true);
   ast_Decl* d = ((ast_Decl*)(i));
   ast_Decl_setUsed(d);
   ast_Decl_setChecked(d);
   ast_ImportDecl_setDest(i, b->mod);
   ast_Decl_setType(d, ast_QualType_init(((ast_Type*)(ast_Module_getType(b->mod)))));
   ast_AST_addImport(b->ast, i);
}

static void ast_builder_Builder_actOnImport(ast_builder_Builder* b, uint32_t mod_name, src_loc_SrcLoc mod_loc, uint32_t alias_name, src_loc_SrcLoc alias_loc, bool islocal)
{
   if ((ast_AST_getNameIdx(b->ast) == mod_name)) {
      diagnostics_Diags_error(b->diags, mod_loc, "cannot import own module '%s'", ast_idx2name(mod_name));
      return;
   }
   ast_ImportDecl* old = ast_AST_findImport(b->ast, mod_name);
   if (old) {
      diagnostics_Diags_error(b->diags, mod_loc, "duplicate import of module '%s'", ast_idx2name(mod_name));
      diagnostics_Diags_note(b->diags, ast_Decl_getLoc(ast_ImportDecl_asDecl(old)), "previous import is here");
      return;
   }
   if (alias_name) {
      if ((alias_name == mod_name)) {
         diagnostics_Diags_error(b->diags, alias_loc, "alias name is same as module name");
         return;
      }
   }
   ast_ImportDecl* d = ast_ImportDecl_create(b->context, mod_name, mod_loc, alias_name, alias_loc, b->ast_idx, islocal);
   ast_AST_addImport(b->ast, d);
}

static ast_Decl* ast_builder_Builder_actOnAliasType(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* ref)
{
   is_public |= b->is_interface;
   ast_AliasTypeDecl* d = ast_AliasTypeDecl_create(b->context, name, loc, is_public, b->ast_idx, ref);
   ast_AST_addTypeDecl(b->ast, ast_AliasTypeDecl_asDecl(d));
   ast_Decl* dd = ((ast_Decl*)(d));
   if (b->is_interface) ast_Decl_setExternal(dd);
   ast_builder_Builder_addSymbol(b, name, dd);
   return dd;
}

static ast_Decl* ast_builder_Builder_actOnFunctionTypeDecl(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* rtype, ast_VarDecl** params, uint32_t num_params, bool is_variadic)
{
   is_public |= b->is_interface;
   ast_FunctionDecl* fd = ast_FunctionDecl_create(b->context, name, loc, is_public, b->ast_idx, rtype, NULL, params, num_params, is_variadic, true);
   ast_FunctionTypeDecl* d = ast_FunctionTypeDecl_create(b->context, fd);
   ast_AST_addTypeDecl(b->ast, ast_FunctionTypeDecl_asDecl(d));
   ast_Decl* dd = ((ast_Decl*)(d));
   if (b->is_interface) {
      ast_Decl_setExternal(dd);
      ast_Decl_setExternal(ast_FunctionDecl_asDecl(fd));
   }
   ast_builder_Builder_addSymbol(b, name, dd);
   return dd;
}

static ast_StructTypeDecl* ast_builder_Builder_actOnStructType(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, bool is_struct, bool is_global, ast_Decl** members, uint32_t num_members)
{
   is_public |= b->is_interface;
   ast_StructTypeDecl* d = ast_StructTypeDecl_create(b->context, name, loc, is_public, b->ast_idx, is_struct, is_global, members, num_members);
   if (is_global) {
      ast_AST_addTypeDecl(b->ast, ast_StructTypeDecl_asDecl(d));
      ast_builder_Builder_addSymbol(b, name, ast_StructTypeDecl_asDecl(d));
   }
   if (b->is_interface) ast_Decl_setExternal(ast_StructTypeDecl_asDecl(d));
   return d;
}

static ast_VarDecl* ast_builder_Builder_actOnStructMember(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* ref, ast_Expr* bitfield)
{
   is_public |= b->is_interface;
   return ast_VarDecl_createStructMember(b->context, name, loc, is_public, ref, b->ast_idx, bitfield);
}

static ast_Decl* ast_builder_Builder_actOnGlobalVarDecl(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, ast_TypeRefHolder* ref, src_loc_SrcLoc assignLoc, ast_Expr* initValue)
{
   is_public |= b->is_interface;
   ast_VarDecl* vd = ast_VarDecl_create(b->context, ast_VarDeclKind_GlobalVar, name, loc, is_public, ref, b->ast_idx, assignLoc, initValue);
   ast_Decl* d = ast_VarDecl_asDecl(vd);
   ast_AST_addVarDecl(b->ast, d);
   ast_builder_Builder_addSymbol(b, name, d);
   if (b->is_interface) ast_Decl_setExternal(d);
   return d;
}

static ast_VarDecl* ast_builder_Builder_actOnFunctionParam(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* ref)
{
   is_public |= b->is_interface;
   return ast_VarDecl_create(b->context, ast_VarDeclKind_FunctionParam, name, loc, is_public, ref, b->ast_idx, 0, NULL);
}

static ast_Stmt* ast_builder_Builder_actOnVarDeclStmt(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, const ast_TypeRefHolder* ref, src_loc_SrcLoc assignLoc, ast_Expr* initValue, bool has_local)
{
   ast_VarDecl* d = ast_VarDecl_create(b->context, ast_VarDeclKind_LocalVar, name, loc, false, ref, b->ast_idx, assignLoc, initValue);
   ast_VarDecl_setLocal(d, has_local);
   return ((ast_Stmt*)(ast_DeclStmt_create(b->context, d)));
}

static bool ast_builder_Builder_hasAttr(const ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a)
{
   return false;
}

static void ast_builder_Builder_storeAttr(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a)
{
   ast_Decl_setHasAttr(d);
   if (ast_builder_Builder_hasAttr(b, d, a)) {
      diagnostics_Diags_error(b->diags, a->loc, "attribute '%s' is already applied", attr_kind2name(a->kind));
      return;
   }
   ast_AST_storeAttr(b->ast, d, a->kind, &a->value);
}

static void ast_builder_Builder_actOnArrayValue(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, ast_Expr* initValue)
{
   ast_ArrayValue* avd = ast_ArrayValue_create(b->context, name, loc, initValue);
   ast_AST_addArrayValue(b->ast, avd);
}

static bool ast_builder_Builder_checkAttr(ast_builder_Builder* b, const attr_Attr* a)
{
   switch (a->kind) {
   case attr_AttrKind_CName:
      if (!b->is_interface) {
         diagnostics_Diags_error(b->diags, a->loc, "attribute 'cname' can only be used in interface files");
         return false;
      }
      break;
   default:
      break;
   }
   return true;
}

static void ast_builder_Builder_actOnFunctionAttr(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a)
{
   ast_FunctionDecl* fd = ((ast_FunctionDecl*)(d));
   switch (a->kind) {
   case attr_AttrKind_Export:
      ast_Decl_setAttrExport(d);
      break;
   case attr_AttrKind_Unused:
      if (ast_Decl_hasAttrUnused(d)) {
         diagnostics_Diags_error(b->diags, a->loc, "attribute 'unused' is already applied");
      }
      ast_Decl_setAttrUnused(d);
      break;
   case attr_AttrKind_UnusedParams:
      ast_FunctionDecl_setAttrUnusedParams(fd);
      break;
   case attr_AttrKind_Section:
      ast_builder_Builder_storeAttr(b, d, a);
      break;
   case attr_AttrKind_NoReturn:
      ast_FunctionDecl_setAttrNoReturn(fd);
      break;
   case attr_AttrKind_Inline:
      ast_FunctionDecl_setAttrInline(fd);
      break;
   case attr_AttrKind_Weak:
      if (!ast_Decl_isPublic(d)) diagnostics_Diags_error(b->diags, a->loc, "weak declarations must be public");
      ast_FunctionDecl_setAttrWeak(fd);
      break;
   case attr_AttrKind_CName:
      ast_builder_Builder_storeAttr(b, d, a);
      break;
   case attr_AttrKind_Constructor:
      ast_FunctionDecl_setAttrConstructor(fd);
      break;
   case attr_AttrKind_Destructor:
      ast_FunctionDecl_setAttrDestructor(fd);
      break;
   case attr_AttrKind_Pure:
      ast_FunctionDecl_setAttrPure(fd);
      break;
   default:
      diagnostics_Diags_error(b->diags, a->loc, "attribute '%s' is not applicable to functions", attr_kind2name(a->kind));
      break;
   }
}

static void ast_builder_Builder_actOnStructAttr(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a)
{
   ast_StructTypeDecl* std = ((ast_StructTypeDecl*)(d));
   switch (a->kind) {
   case attr_AttrKind_Export:
      ast_Decl_setAttrExport(d);
      break;
   case attr_AttrKind_Packed:
      ast_StructTypeDecl_setPacked(std);
      break;
   case attr_AttrKind_Unused:
      if (ast_Decl_hasAttrUnused(d)) {
         diagnostics_Diags_error(b->diags, a->loc, "attribute 'unused' is already applied");
      }
      ast_Decl_setAttrUnused(d);
      break;
   case attr_AttrKind_Section:
      diagnostics_Diags_error(b->diags, a->loc, "attribute '%s' cannot be applied to type declarations", attr_kind2name(a->kind));
      break;
   case attr_AttrKind_Aligned:
      ast_StructTypeDecl_setAttrAlignment(std, a->value.number);
      break;
   case attr_AttrKind_Opaque:
      if (!ast_Decl_isPublic(d)) {
         diagnostics_Diags_error(b->diags, a->loc, "opaque declaration must be public");
      }
      ast_StructTypeDecl_setOpaque(std);
      break;
   case attr_AttrKind_CName:
      ast_builder_Builder_storeAttr(b, d, a);
      break;
   case attr_AttrKind_NoTypeDef:
      if (b->is_interface) {
         ast_StructTypeDecl_setAttrNoTypeDef(std);
      } else {
         diagnostics_Diags_error(b->diags, a->loc, "attribute '%s' can only be used in interfaces", attr_kind2name(a->kind));
      }
      break;
   default:
      diagnostics_Diags_error(b->diags, a->loc, "attribute '%s' is not applicable to structs", attr_kind2name(a->kind));
      break;
   }
}

static bool ast_builder_Builder_actOnTypeAttr(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a)
{
   switch (a->kind) {
   case attr_AttrKind_Export:
      ast_Decl_setAttrExport(d);
      break;
   case attr_AttrKind_Packed:
      diagnostics_Diags_error(b->diags, a->loc, "attribute '%s' can only be applied to struct/union types", attr_kind2name(a->kind));
      return false;
   case attr_AttrKind_Unused:
      if (ast_Decl_hasAttrUnused(d)) {
         diagnostics_Diags_error(b->diags, a->loc, "attribute 'unused' is already applied");
      }
      ast_Decl_setAttrUnused(d);
      break;
   case attr_AttrKind_Opaque:
      diagnostics_Diags_error(b->diags, a->loc, "attribute '%s' can only be applied to struct/union types", attr_kind2name(a->kind));
      return false;
   case attr_AttrKind_CName:
      ast_builder_Builder_storeAttr(b, d, a);
      break;
   default:
      diagnostics_Diags_error(b->diags, a->loc, "attribute '%s' is not applicable to Enum/Alias types", attr_kind2name(a->kind));
      return false;
   }
   return true;
}

static void ast_builder_Builder_actOnVarAttr(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a)
{
   ast_VarDecl* vd = ((ast_VarDecl*)(d));
   switch (a->kind) {
   case attr_AttrKind_Export:
      ast_Decl_setAttrExport(d);
      break;
   case attr_AttrKind_Unused:
      if (ast_Decl_hasAttrUnused(d)) {
         diagnostics_Diags_error(b->diags, a->loc, "attribute 'unused' is already applied");
      }
      ast_Decl_setAttrUnused(d);
      break;
   case attr_AttrKind_Section:
      ast_builder_Builder_storeAttr(b, d, a);
      break;
   case attr_AttrKind_Aligned:
      ast_builder_Builder_storeAttr(b, d, a);
      break;
   case attr_AttrKind_Weak:
      if (!ast_Decl_isPublic(d)) diagnostics_Diags_error(b->diags, a->loc, "weak declarations must be public");
      ast_VarDecl_setAttrWeak(vd);
      break;
   case attr_AttrKind_CName:
      ast_builder_Builder_storeAttr(b, d, a);
      break;
   default:
      diagnostics_Diags_error(b->diags, a->loc, "attribute '%s' is not applicable to variables", attr_kind2name(a->kind));
      break;
   }
}

static void ast_builder_Builder_actOnParamAttr(ast_builder_Builder* b, ast_VarDecl* d, uint32_t name, src_loc_SrcLoc loc)
{
   attr_AttrKind kind = attr_find(name);
   switch (kind) {
   case attr_AttrKind_Unknown:
      diagnostics_Diags_error(b->diags, loc, "unknown attribute '%s'", ast_idx2name(name));
      break;
   case attr_AttrKind_PrintfFormat:
      ast_VarDecl_setPrintfFormat(d);
      break;
   case attr_AttrKind_AutoFile:
      ast_VarDecl_setAttrAutoFile(d);
      break;
   case attr_AttrKind_AutoLine:
      ast_VarDecl_setAttrAutoLine(d);
      break;
   default:
      diagnostics_Diags_error(b->diags, loc, "attribute '%s' cannot be applied to function parameters", ast_idx2name(name));
      break;
   }
}

static void ast_builder_Builder_actOnAttr(ast_builder_Builder* b, attr_Attr* a)
{
   a->kind = attr_find(a->name);
   if (!ast_builder_Builder_checkAttr(b, a)) return;

   if ((a->kind != attr_AttrKind_Unknown)) {
      attr_AttrReq req = attr_check(a);
      switch (req) {
      case attr_AttrReq_NoArg:
         diagnostics_Diags_error(b->diags, a->loc, "attribute '%s' has no argument", attr_kind2name(a->kind));
         return;
      case attr_AttrReq_Arg:
         diagnostics_Diags_error(b->diags, a->loc, "attribute '%s' needs an argument", attr_kind2name(a->kind));
         return;
      case attr_AttrReq_Number:
         diagnostics_Diags_error(b->diags, a->value.loc, "attribute '%s' needs a number argument", attr_kind2name(a->kind));
         return;
      case attr_AttrReq_String:
         diagnostics_Diags_error(b->diags, a->value.loc, "attribute '%s' needs a string argument", attr_kind2name(a->kind));
         return;
      case attr_AttrReq_Power2:
         diagnostics_Diags_error(b->diags, a->value.loc, "requested alignment is not a power of 2");
         return;
      case attr_AttrReq_Ok:
         break;
      }
   }
   if ((b->num_attrs == 8)) {
      diagnostics_Diags_error(b->diags, a->loc, "too many attributes");
      return;
   }
   memcpy(&b->attrs[b->num_attrs], a, 20);
   b->num_attrs++;
}

static void ast_builder_Builder_clearAttributes(ast_builder_Builder* b)
{
   b->num_attrs = 0;
}

static void ast_builder_Builder_applyAttribute(ast_builder_Builder* b, ast_Decl* d, const attr_Attr* a)
{
   ast_DeclKind dk = ast_Decl_getKind(d);
   switch (dk) {
   case ast_DeclKind_Function:
      ast_builder_Builder_actOnFunctionAttr(b, d, a);
      break;
   case ast_DeclKind_StructType:
      ast_builder_Builder_actOnStructAttr(b, d, a);
      break;
   case ast_DeclKind_EnumType:
      ast_builder_Builder_actOnTypeAttr(b, d, a);
      break;
   case ast_DeclKind_FunctionType: {
      if (!ast_builder_Builder_actOnTypeAttr(b, d, a)) return;

      ast_FunctionTypeDecl* ftd = ((ast_FunctionTypeDecl*)(d));
      if ((a->kind == attr_AttrKind_PrintfFormat)) {
         ast_builder_Builder_applyAttribute(b, ((ast_Decl*)(ast_FunctionTypeDecl_getDecl(ftd))), a);
      } else {
         ast_builder_Builder_applyAttribute(b, ((ast_Decl*)(ast_FunctionTypeDecl_getDecl(ftd))), a);
      }
      break;
   }
   case ast_DeclKind_AliasType:
      ast_builder_Builder_actOnTypeAttr(b, d, a);
      break;
   case ast_DeclKind_Variable:
      ast_builder_Builder_actOnVarAttr(b, d, a);
      break;
   default:
      c2_assert((0) != 0, "parser/ast_builder.c2:526: ast_builder.Builder.applyAttribute", "0");
      return;
   }
}

static void ast_builder_Builder_applyAttributes(ast_builder_Builder* b, ast_Decl* d)
{
   c2_assert((d) != NULL, "parser/ast_builder.c2:532: ast_builder.Builder.applyAttributes", "d");
   for (uint32_t i = 0; (i < b->num_attrs); i++) {
      const attr_Attr* a = &b->attrs[i];
      if ((a->kind == attr_AttrKind_Unknown)) {
         attr_handler_Handler_handle(b->attr_handler, d, a);
      } else {
         ast_builder_Builder_applyAttribute(b, d, a);
      }
   }
   b->num_attrs = 0;
}

static ast_QualType ast_builder_Builder_actOnBuiltinType(ast_builder_Builder* _arg0, ast_BuiltinKind kind)
{
   return ast_builtins[kind];
}

static ast_QualType ast_builder_Builder_actOnPointerType(ast_builder_Builder* _arg0, ast_QualType inner)
{
   ast_QualType ptr = ast_QualType_init(ast_getPointerType(inner));
   ast_QualType canon = ast_QualType_getCanonicalType(&inner);
   if ((ast_QualType_getTypeOrNil(&inner) == ast_QualType_getTypeOrNil(&canon))) {
      canon = ptr;
   } else {
      canon = ast_QualType_init(ast_getPointerType(canon));
      ast_QualType_setCanonicalType(&canon, canon);
   }
   ast_QualType_setCanonicalType(&ptr, canon);
   return ptr;
}

static ast_QualType ast_builder_Builder_actOnArrayType(ast_builder_Builder* b, ast_QualType elem, bool has_size, uint32_t size)
{
   ast_ArrayType* t = ast_ArrayType_create(b->context, elem, has_size, size);
   ast_QualType a = ast_QualType_init(((ast_Type*)(t)));
   ast_QualType canon = ast_QualType_getCanonicalType(&elem);
   if ((ast_QualType_getTypeOrNil(&elem) == ast_QualType_getTypeOrNil(&canon))) {
      canon = a;
   } else {
      ast_ArrayType* t2 = ast_ArrayType_create(b->context, canon, has_size, size);
      canon = ast_QualType_init(((ast_Type*)(t2)));
   }
   ast_QualType_setCanonicalType(&a, canon);
   return a;
}

static ast_QualType ast_builder_Builder_actOnIncrementalArrayType(ast_builder_Builder* b, ast_QualType elem)
{
   ast_ArrayType* t = ast_ArrayType_createIncremental(b->context, elem);
   ast_QualType a = ast_QualType_init(((ast_Type*)(t)));
   ast_QualType canon = ast_QualType_getCanonicalType(&elem);
   if ((ast_QualType_getTypeOrNil(&elem) == ast_QualType_getTypeOrNil(&canon))) {
      canon = a;
   } else {
      ast_ArrayType* t2 = ast_ArrayType_createIncremental(b->context, canon);
      canon = ast_QualType_init(((ast_Type*)(t2)));
   }
   ast_QualType_setCanonicalType(&a, canon);
   return a;
}

static ast_FunctionDecl* ast_builder_Builder_actOnFunctionDecl(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* rtype, const ast_Ref* prefix, ast_VarDecl** params, uint32_t num_params, bool is_variadic)
{
   is_public |= b->is_interface;
   ast_FunctionDecl* f = ast_FunctionDecl_create(b->context, name, loc, is_public, b->ast_idx, rtype, prefix, params, num_params, is_variadic, false);
   ast_AST_addFunc(b->ast, f);
   if (!prefix) ast_builder_Builder_addSymbol(b, name, ast_FunctionDecl_asDecl(f));
   if (b->is_interface) ast_Decl_setExternal(ast_FunctionDecl_asDecl(f));
   return f;
}

static ast_FunctionDecl* ast_builder_Builder_actOnTemplateFunctionDecl(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, const ast_TypeRefHolder* rtype, uint32_t template_name, src_loc_SrcLoc template_loc, ast_VarDecl** params, uint32_t num_params, bool is_variadic)
{
   if (b->is_interface) diagnostics_Diags_error(b->diags, loc, "template functions are not allow in interfaces");
   is_public |= b->is_interface;
   ast_FunctionDecl* f = ast_FunctionDecl_createTemplate(b->context, name, loc, is_public, b->ast_idx, rtype, template_name, template_loc, params, num_params, is_variadic);
   ast_AST_addFunc(b->ast, f);
   ast_builder_Builder_addSymbol(b, name, ast_FunctionDecl_asDecl(f));
   if (b->is_interface) ast_Decl_setExternal(ast_FunctionDecl_asDecl(f));
   return f;
}

static void ast_builder_Builder_actOnFunctionBody(ast_builder_Builder* _arg0, ast_FunctionDecl* f, ast_CompoundStmt* body)
{
   ast_FunctionDecl_setBody(f, body);
}

static ast_EnumConstantDecl* ast_builder_Builder_actOnEnumConstant(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, ast_Expr* init_expr)
{
   return ast_EnumConstantDecl_create(b->context, name, loc, is_public, b->ast_idx, init_expr);
}

static ast_Decl* ast_builder_Builder_actOnEnumType(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc, bool is_public, bool is_incr, ast_QualType implType, ast_EnumConstantDecl** constants, uint32_t num_constants)
{
   is_public |= b->is_interface;
   ast_EnumTypeDecl* d = ast_EnumTypeDecl_create(b->context, name, loc, is_public, b->ast_idx, implType, is_incr, constants, num_constants);
   ast_AST_addTypeDecl(b->ast, ast_EnumTypeDecl_asDecl(d));
   ast_builder_Builder_addSymbol(b, name, ast_EnumTypeDecl_asDecl(d));
   return ((ast_Decl*)(d));
}

static ast_Stmt* ast_builder_Builder_actOnAsmStmt(ast_builder_Builder* b, src_loc_SrcLoc loc, bool is_basic, bool is_volatile, uint32_t num_outputs, uint32_t num_inputs, const uint32_t* names, ast_ExprList* constraints, ast_ExprList* exprs, ast_ExprList* clobbers, ast_Expr* asm_string)
{
   return ((ast_Stmt*)(ast_AsmStmt_create(b->context, loc, is_basic, is_volatile, num_outputs, num_inputs, names, constraints, exprs, clobbers, asm_string)));
}

static ast_CompoundStmt* ast_builder_Builder_actOnCompoundStmt(ast_builder_Builder* b, src_loc_SrcLoc endLoc, ast_Stmt** stmts, uint32_t count)
{
   return ast_CompoundStmt_create(b->context, endLoc, stmts, count);
}

static ast_Stmt* ast_builder_Builder_actOnReturnStmt(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* ret)
{
   return ((ast_Stmt*)(ast_ReturnStmt_create(b->context, loc, ret)));
}

static ast_Stmt* ast_builder_Builder_actOnIfStmt(ast_builder_Builder* b, ast_Stmt* cond, ast_Stmt* then, ast_Stmt* else_stmt)
{
   return ((ast_Stmt*)(ast_IfStmt_create(b->context, cond, then, else_stmt)));
}

static ast_Stmt* ast_builder_Builder_actOnDoStmt(ast_builder_Builder* b, ast_Stmt* cond, ast_Stmt* then)
{
   return ((ast_Stmt*)(ast_DoStmt_create(b->context, cond, then)));
}

static ast_Stmt* ast_builder_Builder_actOnWhileStmt(ast_builder_Builder* b, ast_Stmt* cond, ast_Stmt* then)
{
   return ((ast_Stmt*)(ast_WhileStmt_create(b->context, cond, then)));
}

static ast_Stmt* ast_builder_Builder_actOnForStmt(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Stmt* init_, ast_Expr* cond, ast_Expr* incr, ast_Stmt* body)
{
   return ((ast_Stmt*)(ast_ForStmt_create(b->context, loc, init_, cond, incr, body)));
}

static ast_Stmt* ast_builder_Builder_actOnSwitchStmt(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* cond, ast_SwitchCase** cases, uint32_t num_cases, bool is_sswitch)
{
   return ((ast_Stmt*)(ast_SwitchStmt_create(b->context, loc, cond, cases, num_cases, is_sswitch)));
}

static ast_SwitchCase* ast_builder_Builder_actOnCase(ast_builder_Builder* b, src_loc_SrcLoc loc, bool is_default, ast_Expr* cond, identifier_expr_list_List* conditions, ast_Stmt** stmts, uint32_t num_stmts)
{
   return ast_SwitchCase_create(b->context, loc, is_default, cond, identifier_expr_list_List_getData(conditions), identifier_expr_list_List_size(conditions), stmts, num_stmts);
}

static ast_Stmt* ast_builder_Builder_actOnAssertStmt(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* inner)
{
   return ((ast_Stmt*)(ast_AssertStmt_create(b->context, loc, inner)));
}

static ast_Stmt* ast_builder_Builder_actOnBreakStmt(ast_builder_Builder* b, src_loc_SrcLoc loc)
{
   return ((ast_Stmt*)(ast_BreakStmt_create(b->context, loc)));
}

static ast_Stmt* ast_builder_Builder_actOnContinueStmt(ast_builder_Builder* b, src_loc_SrcLoc loc)
{
   return ((ast_Stmt*)(ast_ContinueStmt_create(b->context, loc)));
}

static ast_Stmt* ast_builder_Builder_actOnFallthroughStmt(ast_builder_Builder* b, src_loc_SrcLoc loc)
{
   return ((ast_Stmt*)(ast_FallthroughStmt_create(b->context, loc)));
}

static ast_Stmt* ast_builder_Builder_actOnLabelStmt(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc)
{
   return ((ast_Stmt*)(ast_LabelStmt_create(b->context, name, loc)));
}

static ast_Stmt* ast_builder_Builder_actOnGotoStmt(ast_builder_Builder* b, uint32_t name, src_loc_SrcLoc loc)
{
   return ((ast_Stmt*)(ast_GotoStmt_create(b->context, name, loc)));
}

static ast_IdentifierExpr* ast_builder_Builder_actOnIdentifier(ast_builder_Builder* b, src_loc_SrcLoc loc, uint32_t name)
{
   return ast_IdentifierExpr_create(b->context, loc, name);
}

static ast_Expr* ast_builder_Builder_actOnIntegerLiteral(ast_builder_Builder* b, src_loc_SrcLoc loc, uint8_t radix, uint64_t value)
{
   return ((ast_Expr*)(ast_IntegerLiteral_create(b->context, loc, radix, value)));
}

static ast_Expr* ast_builder_Builder_actOnFloatLiteral(ast_builder_Builder* b, src_loc_SrcLoc loc, double value)
{
   return ((ast_Expr*)(ast_FloatLiteral_create(b->context, loc, value)));
}

static ast_Expr* ast_builder_Builder_actOnCharLiteral(ast_builder_Builder* b, src_loc_SrcLoc loc, uint8_t value, uint8_t radix)
{
   return ((ast_Expr*)(ast_CharLiteral_create(b->context, loc, value, radix)));
}

static ast_Expr* ast_builder_Builder_actOnStringLiteral(ast_builder_Builder* b, src_loc_SrcLoc loc, uint32_t value, uint32_t len)
{
   return ((ast_Expr*)(ast_StringLiteral_create(b->context, loc, value, len)));
}

static ast_Expr* ast_builder_Builder_actOnNilExpr(ast_builder_Builder* b, src_loc_SrcLoc loc)
{
   return ((ast_Expr*)(ast_NilExpr_create(b->context, loc)));
}

static ast_Expr* ast_builder_Builder_actOnParenExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* inner)
{
   return ((ast_Expr*)(ast_ParenExpr_create(b->context, loc, inner)));
}

static ast_Expr* ast_builder_Builder_actOnUnaryOperator(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_UnaryOpcode opcode, ast_Expr* inner)
{
   return ((ast_Expr*)(ast_UnaryOperator_create(b->context, loc, opcode, inner)));
}

static ast_Expr* ast_builder_Builder_actOnPostFixUnaryOperator(ast_builder_Builder* b, src_loc_SrcLoc loc, bool is_increment, ast_Expr* inner)
{
   ast_UnaryOpcode opcode = is_increment ? ast_UnaryOpcode_PostInc : ast_UnaryOpcode_PostDec;
   return ((ast_Expr*)(ast_UnaryOperator_create(b->context, loc, opcode, inner)));
}

static ast_Expr* ast_builder_Builder_actOnBinaryOperator(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_BinaryOpcode opcode, ast_Expr* lhs, ast_Expr* rhs)
{
   return ((ast_Expr*)(ast_BinaryOperator_create(b->context, loc, opcode, lhs, rhs)));
}

static ast_Expr* ast_builder_Builder_actOnConditionalOperator(ast_builder_Builder* b, src_loc_SrcLoc questionLoc, src_loc_SrcLoc colonLoc, ast_Expr* cond, ast_Expr* lhs, ast_Expr* rhs)
{
   return ((ast_Expr*)(ast_ConditionalOperator_create(b->context, questionLoc, colonLoc, cond, lhs, rhs)));
}

static ast_Expr* ast_builder_Builder_actOnBooleanConstant(ast_builder_Builder* b, src_loc_SrcLoc loc, bool value)
{
   return ((ast_Expr*)(ast_BooleanLiteral_create(b->context, loc, value)));
}

static ast_Expr* ast_builder_Builder_actOnBuiltinExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* inner, ast_BuiltinExprKind kind)
{
   return ((ast_Expr*)(ast_BuiltinExpr_create(b->context, loc, inner, kind)));
}

static ast_Expr* ast_builder_Builder_actOnOffsetOfExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* structExpr, ast_Expr* member)
{
   return ((ast_Expr*)(ast_BuiltinExpr_createOffsetOf(b->context, loc, structExpr, member)));
}

static ast_Expr* ast_builder_Builder_actOnToContainerExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* structExpr, ast_Expr* member, ast_Expr* pointer)
{
   return ((ast_Expr*)(ast_BuiltinExpr_createToContainer(b->context, loc, structExpr, member, pointer)));
}

static ast_Expr* ast_builder_Builder_actOnTypeExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, const ast_TypeRefHolder* ref)
{
   return ((ast_Expr*)(ast_TypeExpr_create(b->context, loc, ref)));
}

static ast_Expr* ast_builder_Builder_actOnBitOffsetExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* lhs, ast_Expr* rhs)
{
   return ((ast_Expr*)(ast_BitOffsetExpr_create(b->context, loc, lhs, rhs)));
}

static ast_Expr* ast_builder_Builder_actOnArraySubscriptExpr(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* base, ast_Expr* idx)
{
   return ((ast_Expr*)(ast_ArraySubscriptExpr_create(b->context, loc, base, idx)));
}

static ast_Expr* ast_builder_Builder_actOnCallExpr(ast_builder_Builder* b, src_loc_SrcLoc endLoc, ast_Expr* func, ast_Expr** args, uint32_t num_args)
{
   return ((ast_Expr*)(ast_CallExpr_create(b->context, endLoc, func, args, num_args)));
}

static ast_Expr* ast_builder_Builder_actOnTemplateCallExpr(ast_builder_Builder* b, src_loc_SrcLoc endLoc, ast_Expr* func, ast_Expr** args, uint32_t num_args, const ast_TypeRefHolder* ref)
{
   return ((ast_Expr*)(ast_CallExpr_createTemplate(b->context, endLoc, func, args, num_args, ref)));
}

static ast_Expr* ast_builder_Builder_actOnExplicitCast(ast_builder_Builder* b, src_loc_SrcLoc loc, const ast_TypeRefHolder* ref, ast_Expr* inner)
{
   return ((ast_Expr*)(ast_ExplicitCastExpr_create(b->context, loc, ref, inner)));
}

static ast_Expr* ast_builder_Builder_actOnMemberExpr(ast_builder_Builder* b, ast_Expr* base, const ast_Ref* refs, uint32_t refcount)
{
   return ((ast_Expr*)(ast_MemberExpr_create(b->context, base, refs, refcount)));
}

static ast_Expr* ast_builder_Builder_actOnInitList(ast_builder_Builder* b, src_loc_SrcLoc left, src_loc_SrcLoc right, ast_Expr** values, uint32_t num_values)
{
   return ((ast_Expr*)(ast_InitListExpr_create(b->context, left, right, values, num_values)));
}

static ast_Expr* ast_builder_Builder_actOnFieldDesignatedInit(ast_builder_Builder* b, uint32_t field, src_loc_SrcLoc loc, ast_Expr* initValue)
{
   return ((ast_Expr*)(ast_FieldDesignatedInitExpr_create(b->context, field, loc, initValue)));
}

static ast_Expr* ast_builder_Builder_actOnArrayDesignatedInit(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* designator, ast_Expr* initValue)
{
   return ((ast_Expr*)(ast_ArrayDesignatedInitExpr_create(b->context, loc, designator, initValue)));
}

static void ast_builder_Builder_actOnStaticAssert(ast_builder_Builder* b, src_loc_SrcLoc loc, ast_Expr* lhs, ast_Expr* rhs)
{
   ast_StaticAssert* d = ast_StaticAssert_create(b->context, b->ast_idx, loc, lhs, rhs);
   ast_AST_addStaticAssert(b->ast, d);
}

static void ast_builder_Builder_insertImplicitCast(ast_builder_Builder* b, ast_ImplicitCastKind kind, ast_Expr** e_ptr, ast_QualType qt)
{
   ast_Expr* inner = *e_ptr;
   ast_Expr* ic = ((ast_Expr*)(ast_ImplicitCastExpr_create(b->context, ast_Expr_getLoc(inner), kind, inner)));
   ast_Expr_setType(ic, qt);
   *e_ptr = ic;
}

static void ast_builder_Builder_addSymbol(ast_builder_Builder* b, uint32_t name_idx, ast_Decl* d)
{
   ast_Decl* old = ast_Module_findSymbol(b->mod, name_idx);
   if (old) {
      diagnostics_Diags_error(b->diags, ast_Decl_getLoc(d), "redefinition of '%s'", ast_idx2name(name_idx));
      diagnostics_Diags_note(b->diags, ast_Decl_getLoc(old), "previous definition is here");
   } else {
      ast_Module_addSymbol(b->mod, name_idx, d);
   }
}


// --- module c2_parser ---
typedef struct c2_parser_Parser_ c2_parser_Parser;

struct c2_parser_Parser_ {
   c2_tokenizer_Tokenizer tokenizer;
   token_Token tok;
   src_loc_SrcLoc prev_loc;
   int32_t file_id;
   source_mgr_SourceMgr* sm;
   diagnostics_Diags* diags;
   string_pool_Pool* pool;
   ast_builder_Builder* builder;
   const string_list_List* features;
   bool is_interface;
   __jmp_buf_tag jmpbuf;
};

static const ast_BuiltinKind c2_parser_Tok2builtin[19] = {
   ast_BuiltinKind_Bool,
   ast_BuiltinKind_Char,
   ast_BuiltinKind_Int8,
   ast_BuiltinKind_Int16,
   ast_BuiltinKind_Int32,
   ast_BuiltinKind_Int64,
   ast_BuiltinKind_UInt8,
   ast_BuiltinKind_UInt16,
   ast_BuiltinKind_UInt32,
   ast_BuiltinKind_UInt64,
   ast_BuiltinKind_UInt8,
   ast_BuiltinKind_UInt16,
   ast_BuiltinKind_UInt32,
   ast_BuiltinKind_UInt64,
   ast_BuiltinKind_ISize,
   ast_BuiltinKind_USize,
   ast_BuiltinKind_Float32,
   ast_BuiltinKind_Float64,
   ast_BuiltinKind_Void
};

static c2_parser_Parser* c2_parser_create(source_mgr_SourceMgr* sm, diagnostics_Diags* diags, string_pool_Pool* pool, ast_builder_Builder* builder, const string_list_List* features);
static void c2_parser_Parser_free(c2_parser_Parser* p);
static void c2_parser_Parser_parse(c2_parser_Parser* p, int32_t file_id, bool is_interface, bool is_generated);
static void c2_parser_Parser_consumeToken(c2_parser_Parser* p);
static void c2_parser_Parser_expectAndConsume(c2_parser_Parser* p, token_Kind kind);
static void c2_parser_Parser_expect(c2_parser_Parser* p, token_Kind kind);
static void c2_parser_Parser_expectIdentifier(c2_parser_Parser* p);
__attribute__((__format__(printf, 2, 3))) 
static void c2_parser_Parser_error(c2_parser_Parser* p, const char* format, ...);
static void c2_parser_Parser_parseModule(c2_parser_Parser* p, bool is_generated);
static void c2_parser_Parser_parseImports(c2_parser_Parser* p);
static void c2_parser_Parser_parseTopLevel(c2_parser_Parser* p);
static void c2_parser_Parser_parseOptionalAttributes(c2_parser_Parser* p);
static void c2_parser_Parser_parseParamOptionalAttributes(c2_parser_Parser* p, ast_VarDecl* d);
static void c2_parser_Parser_parseFuncDecl(c2_parser_Parser* p, bool is_public);
static bool c2_parser_Parser_parseFunctionParams(c2_parser_Parser* p, ast_DeclList* params, bool is_public);
static ast_VarDecl* c2_parser_Parser_parseParamDecl(c2_parser_Parser* p, bool is_public);
static void c2_parser_Parser_parseTypeSpecifier(c2_parser_Parser* p, ast_TypeRefHolder* ref, bool allow_qualifier, bool allow_array);
static void c2_parser_Parser_parseOptionalArray(c2_parser_Parser* p, ast_TypeRefHolder* ref, ast_QualType base, bool allow_array);
static void c2_parser_Parser_parseArrayEntry(c2_parser_Parser* p);
static void c2_parser_Parser_parseVarDecl(c2_parser_Parser* p, bool is_public);
static void c2_parser_Parser_parseStaticAssert(c2_parser_Parser* p);
static bool c2_parser_Parser_parseOptionalAccessSpecifier(c2_parser_Parser* p);
static uint32_t c2_parser_Parser_parseOptionalTypeQualifier(c2_parser_Parser* p);
static ast_BuiltinKind c2_parser_tokKindToBuiltinKind(token_Kind kind);
static void c2_parser_Parser_parseSingleTypeSpecifier(c2_parser_Parser* p, ast_TypeRefHolder* ref, bool allow_qualifier);
static void c2_parser_Parser_parseFullTypeIdentifier(c2_parser_Parser* p, ast_TypeRefHolder* ref);
static void c2_parser_Parser_dump_token(c2_parser_Parser* p, const token_Token* tok);
static bool c2_parser_checkName(const char* name, bool is_interface);
typedef enum {
   c2_parser_Prec_Unknown = 0,
   c2_parser_Prec_Comma = 1,
   c2_parser_Prec_Assignment = 2,
   c2_parser_Prec_Conditional = 3,
   c2_parser_Prec_LogicalAndOr = 4,
   c2_parser_Prec_Relational = 5,
   c2_parser_Prec_Additive = 6,
   c2_parser_Prec_Bitwise = 7,
   c2_parser_Prec_Shift = 8,
   c2_parser_Prec_Multiplicative = 9,
   _c2_parser_Prec_max = 255
} __attribute__((packed)) c2_parser_Prec;

static const c2_parser_Prec c2_parser_BinOpPrecLookup[128] = {
   [token_Kind_Comma] = c2_parser_Prec_Comma,
   [token_Kind_Equal] = c2_parser_Prec_Assignment,
   [token_Kind_StarEqual] = c2_parser_Prec_Assignment,
   [token_Kind_SlashEqual] = c2_parser_Prec_Assignment,
   [token_Kind_PercentEqual] = c2_parser_Prec_Assignment,
   [token_Kind_PlusEqual] = c2_parser_Prec_Assignment,
   [token_Kind_MinusEqual] = c2_parser_Prec_Assignment,
   [token_Kind_LessLessEqual] = c2_parser_Prec_Assignment,
   [token_Kind_GreaterGreaterEqual] = c2_parser_Prec_Assignment,
   [token_Kind_AmpEqual] = c2_parser_Prec_Assignment,
   [token_Kind_CaretEqual] = c2_parser_Prec_Assignment,
   [token_Kind_PipeEqual] = c2_parser_Prec_Assignment,
   [token_Kind_Question] = c2_parser_Prec_Conditional,
   [token_Kind_PipePipe] = c2_parser_Prec_LogicalAndOr,
   [token_Kind_AmpAmp] = c2_parser_Prec_LogicalAndOr,
   [token_Kind_ExclaimEqual] = c2_parser_Prec_Relational,
   [token_Kind_EqualEqual] = c2_parser_Prec_Relational,
   [token_Kind_LessEqual] = c2_parser_Prec_Relational,
   [token_Kind_Less] = c2_parser_Prec_Relational,
   [token_Kind_Greater] = c2_parser_Prec_Relational,
   [token_Kind_GreaterEqual] = c2_parser_Prec_Relational,
   [token_Kind_Plus] = c2_parser_Prec_Additive,
   [token_Kind_Minus] = c2_parser_Prec_Additive,
   [token_Kind_Pipe] = c2_parser_Prec_Bitwise,
   [token_Kind_Caret] = c2_parser_Prec_Bitwise,
   [token_Kind_Amp] = c2_parser_Prec_Bitwise,
   [token_Kind_LessLess] = c2_parser_Prec_Shift,
   [token_Kind_GreaterGreater] = c2_parser_Prec_Shift,
   [token_Kind_Percent] = c2_parser_Prec_Multiplicative,
   [token_Kind_Slash] = c2_parser_Prec_Multiplicative,
   [token_Kind_Star] = c2_parser_Prec_Multiplicative
};

static const ast_BinaryOpcode c2_parser_BinOpTokenLookup[128] = {
   [token_Kind_Star] = ast_BinaryOpcode_Multiply,
   [token_Kind_Slash] = ast_BinaryOpcode_Divide,
   [token_Kind_Percent] = ast_BinaryOpcode_Reminder,
   [token_Kind_Plus] = ast_BinaryOpcode_Add,
   [token_Kind_Minus] = ast_BinaryOpcode_Subtract,
   [token_Kind_LessLess] = ast_BinaryOpcode_ShiftLeft,
   [token_Kind_GreaterGreater] = ast_BinaryOpcode_ShiftRight,
   [token_Kind_Less] = ast_BinaryOpcode_LessThan,
   [token_Kind_Greater] = ast_BinaryOpcode_GreaterThan,
   [token_Kind_LessEqual] = ast_BinaryOpcode_LessEqual,
   [token_Kind_GreaterEqual] = ast_BinaryOpcode_GreaterEqual,
   [token_Kind_EqualEqual] = ast_BinaryOpcode_Equal,
   [token_Kind_ExclaimEqual] = ast_BinaryOpcode_NotEqual,
   [token_Kind_Amp] = ast_BinaryOpcode_And,
   [token_Kind_Caret] = ast_BinaryOpcode_Xor,
   [token_Kind_Pipe] = ast_BinaryOpcode_Or,
   [token_Kind_AmpAmp] = ast_BinaryOpcode_LAnd,
   [token_Kind_PipePipe] = ast_BinaryOpcode_LOr,
   [token_Kind_Equal] = ast_BinaryOpcode_Assign,
   [token_Kind_StarEqual] = ast_BinaryOpcode_MulAssign,
   [token_Kind_SlashEqual] = ast_BinaryOpcode_DivAssign,
   [token_Kind_PercentEqual] = ast_BinaryOpcode_RemAssign,
   [token_Kind_PlusEqual] = ast_BinaryOpcode_AddAssign,
   [token_Kind_MinusEqual] = ast_BinaryOpcode_SubAssign,
   [token_Kind_LessLessEqual] = ast_BinaryOpcode_ShlAssign,
   [token_Kind_GreaterGreaterEqual] = ast_BinaryOpcode_ShrAssign,
   [token_Kind_AmpEqual] = ast_BinaryOpcode_AndAssign,
   [token_Kind_CaretEqual] = ast_BinaryOpcode_XorAssign,
   [token_Kind_PipeEqual] = ast_BinaryOpcode_OrAssign
};

static const uint8_t c2_parser_CastExprTokenLookup[128] = {
   [token_Kind_Identifier] = 1,
   [token_Kind_IntegerLiteral] = 2,
   [token_Kind_FloatLiteral] = 3,
   [token_Kind_CharLiteral] = 4,
   [token_Kind_StringLiteral] = 5,
   [token_Kind_LParen] = 6,
   [token_Kind_Star] = 7,
   [token_Kind_Tilde] = 7,
   [token_Kind_Minus] = 7,
   [token_Kind_Exclaim] = 7,
   [token_Kind_Amp] = 8,
   [token_Kind_KW_cast] = 9,
   [token_Kind_Plus] = 10,
   [token_Kind_PlusPlus] = 11,
   [token_Kind_MinusMinus] = 11,
   [token_Kind_KW_elemsof] = 12,
   [token_Kind_KW_enum_min] = 13,
   [token_Kind_KW_enum_max] = 13,
   [token_Kind_KW_false] = 14,
   [token_Kind_KW_true] = 14,
   [token_Kind_KW_nil] = 15,
   [token_Kind_KW_offsetof] = 16,
   [token_Kind_KW_sizeof] = 17,
   [token_Kind_KW_to_container] = 18
};

static ast_Expr* c2_parser_Parser_parseExpr(c2_parser_Parser* p);
static ast_Expr* c2_parser_Parser_parseAssignmentExpression(c2_parser_Parser* p);
static ast_Expr* c2_parser_Parser_parseRHSOfBinaryExpression(c2_parser_Parser* p, ast_Expr* lhs, c2_parser_Prec minPrec);
static ast_UnaryOpcode c2_parser_convertTokenToUnaryOpcode(token_Kind kind);
static ast_Expr* c2_parser_Parser_parseCastExpr(c2_parser_Parser* p, bool _arg1, bool _arg2);
static ast_Expr* c2_parser_Parser_parsePostfixExprSuffix(c2_parser_Parser* p, ast_Expr* lhs, bool couldBeTemplateCall);
static ast_Expr* c2_parser_Parser_parseCallExpr(c2_parser_Parser* p, ast_Expr* func);
static ast_Expr* c2_parser_Parser_parseTemplateCallExpr(c2_parser_Parser* p, ast_Expr* func, const ast_TypeRefHolder* ref);
static ast_Expr* c2_parser_Parser_parseImpureMemberExpr(c2_parser_Parser* p, ast_Expr* base);
static ast_Expr* c2_parser_Parser_parsePureMemberExpr(c2_parser_Parser* p);
static ast_IdentifierExpr* c2_parser_Parser_parseIdentifier(c2_parser_Parser* p);
static ast_Expr* c2_parser_Parser_parseStringLiteral(c2_parser_Parser* p);
static ast_Expr* c2_parser_Parser_parseParenExpr(c2_parser_Parser* p);
static bool c2_parser_Parser_isTemplateFunctionCall(c2_parser_Parser* p);
static ast_Expr* c2_parser_Parser_parseSizeof(c2_parser_Parser* p);
static ast_Expr* c2_parser_Parser_parseElemsof(c2_parser_Parser* p);
static ast_Expr* c2_parser_Parser_parseInitValue(c2_parser_Parser* p, bool* need_semi, bool allow_designators);
static ast_Expr* c2_parser_Parser_parseInitList(c2_parser_Parser* p);
static ast_Expr* c2_parser_Parser_parseFieldDesignator(c2_parser_Parser* p, bool* need_semi);
static ast_Expr* c2_parser_Parser_parseArrayDesignator(c2_parser_Parser* p, bool* need_semi);
static ast_Expr* c2_parser_Parser_parseExplicitCastExpr(c2_parser_Parser* p);
static ast_Expr* c2_parser_Parser_parseEnumMinMax(c2_parser_Parser* p, bool is_min);
static ast_Expr* c2_parser_Parser_parseOffsetOfExpr(c2_parser_Parser* p);
static ast_Expr* c2_parser_Parser_parseToContainerExpr(c2_parser_Parser* p);
static ast_Expr* c2_parser_Parser_parseFullIdentifier(c2_parser_Parser* p);
static bool c2_parser_Parser_parseAsType(c2_parser_Parser* p, bool* has_brackets);
static ast_Stmt* c2_parser_Parser_parseStmt(c2_parser_Parser* p);
static bool c2_parser_Parser_isTypeSpec(c2_parser_Parser* p);
static uint32_t c2_parser_Parser_skipArray(c2_parser_Parser* p, uint32_t lookahead);
static ast_Stmt* c2_parser_Parser_parseDeclOrStmt(c2_parser_Parser* p);
static ast_CompoundStmt* c2_parser_Parser_parseCompoundStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseAsmStmt(c2_parser_Parser* p);
static void c2_parser_Parser_parseAsmOperandsOpt(c2_parser_Parser* p, string_list_List* names, ast_ExprList* constraints, ast_ExprList* exprs);
static ast_Stmt* c2_parser_Parser_parseAssertStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseBreakStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseContinueStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseFallthroughStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseCondition(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseIfStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseReturnStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseDoStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseForStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseWhileStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseDeclStmt(c2_parser_Parser* p, bool checkSemi, bool allowLocal);
static ast_Stmt* c2_parser_Parser_parseExprStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseLabelStmt(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseGotoStmt(c2_parser_Parser* p);
static bool c2_parser_Parser_isDeclaration(c2_parser_Parser* p);
static ast_Stmt* c2_parser_Parser_parseSwitchStmt(c2_parser_Parser* p, bool is_sswitch);
static ast_Expr* c2_parser_Parser_parseCaseCondition(c2_parser_Parser* p, identifier_expr_list_List* list);
static ast_SwitchCase* c2_parser_Parser_parseCase(c2_parser_Parser* p, bool is_default);
static void c2_parser_Parser_parseTypeDecl(c2_parser_Parser* p, bool is_public);
static void c2_parser_Parser_parseFunctionType(c2_parser_Parser* p, uint32_t name, src_loc_SrcLoc loc, bool is_public);
static void c2_parser_Parser_parseStructType(c2_parser_Parser* p, bool is_struct, uint32_t name, src_loc_SrcLoc loc, bool is_public);
static void c2_parser_Parser_parseStructBlock(c2_parser_Parser* p, ast_DeclList* members, bool is_public);
static void c2_parser_Parser_parseEnumType(c2_parser_Parser* p, uint32_t name, src_loc_SrcLoc loc, bool is_public);
static void c2_parser_Parser_parseAliasType(c2_parser_Parser* p, uint32_t name, src_loc_SrcLoc loc, bool is_public);
static c2_parser_Parser* c2_parser_create(source_mgr_SourceMgr* sm, diagnostics_Diags* diags, string_pool_Pool* pool, ast_builder_Builder* builder, const string_list_List* features)
{
   c2_parser_Parser* p = calloc(1, 872);
   p->sm = sm;
   p->diags = diags;
   p->pool = pool;
   p->builder = builder;
   p->features = features;
   return p;
}

static void c2_parser_Parser_free(c2_parser_Parser* p)
{
   free(p);
}

static void c2_parser_Parser_parse(c2_parser_Parser* p, int32_t file_id, bool is_interface, bool is_generated)
{
   p->file_id = file_id;
   p->is_interface = is_interface;
   string_buffer_Buf* buf = string_buffer_create(1024, 0, false);
   int32_t res = setjmp(&p->jmpbuf);
   if ((res == 0)) {
      c2_tokenizer_Tokenizer_init(&p->tokenizer, p->pool, buf, source_mgr_SourceMgr_get_content(p->sm, p->file_id), source_mgr_SourceMgr_get_offset(p->sm, p->file_id), p->features, false);
      token_Token_init(&p->tok);
      c2_parser_Parser_consumeToken(p);
      c2_parser_Parser_parseModule(p, is_generated);
      c2_parser_Parser_parseImports(p);
      while (p->tok.more) {
         c2_parser_Parser_parseTopLevel(p);
      }
   }
   string_buffer_Buf_free(buf);
}

static void c2_parser_Parser_consumeToken(c2_parser_Parser* p)
{
   p->prev_loc = p->tok.loc;
   c2_tokenizer_Tokenizer_lex(&p->tokenizer, &p->tok);
   if ((p->tok.kind == token_Kind_Error)) c2_parser_Parser_error(p, "%s", p->tok.error_msg);
}

static void c2_parser_Parser_expectAndConsume(c2_parser_Parser* p, token_Kind kind)
{
   if ((p->tok.kind == kind)) {
      c2_parser_Parser_consumeToken(p);
      return;
   }
   if (p->prev_loc) {
      const char* src = source_mgr_SourceMgr_get_token_source(p->sm, p->prev_loc);
      src_loc_SrcLoc loc = parser_utils_getTokenEnd(src, p->prev_loc);
      p->tok.loc = (loc + 1);
   }
   if ((p->tok.loc == 0)) {
      p->tok.loc = source_mgr_SourceMgr_get_offset(p->sm, p->file_id);
   }
   c2_parser_Parser_error(p, "expected '%s'", token_kind2str(kind));
}

static void c2_parser_Parser_expect(c2_parser_Parser* p, token_Kind kind)
{
   if ((p->tok.kind == kind)) return;

   if (p->prev_loc) {
      const char* src = source_mgr_SourceMgr_get_token_source(p->sm, p->prev_loc);
      src_loc_SrcLoc loc = parser_utils_getTokenEnd(src, p->prev_loc);
      p->tok.loc = (loc + 1);
   }
   if ((p->tok.loc == 0)) {
      p->tok.loc = source_mgr_SourceMgr_get_offset(p->sm, p->file_id);
   }
   c2_parser_Parser_error(p, "expected '%s'", token_kind2str(kind));
}

static void c2_parser_Parser_expectIdentifier(c2_parser_Parser* p)
{
   if ((p->tok.kind == token_Kind_Identifier)) return;

   c2_parser_Parser_error(p, "expected identifier");
}

__attribute__((__format__(printf, 2, 3))) 
static void c2_parser_Parser_error(c2_parser_Parser* p, const char* format, ...)
{
   char msg[256];
   va_list args;
   va_start(args, format);
   vsnprintf(msg, (256 - 1), format, args);
   va_end(args);
   diagnostics_Diags_error(p->diags, p->tok.loc, "%s", msg);
   longjmp(&p->jmpbuf, 1);
}

static void c2_parser_Parser_parseModule(c2_parser_Parser* p, bool is_generated)
{
   c2_parser_Parser_expectAndConsume(p, token_Kind_KW_module);
   c2_parser_Parser_expectIdentifier(p);
   const char* modname = string_pool_Pool_idx2str(p->pool, p->tok.text_idx);
   if (isupper(modname[0])) {
      c2_parser_Parser_error(p, "a module name must start with a lower case character");
   }
   ast_builder_Builder_actOnModule(p->builder, p->tok.text_idx, p->tok.loc, source_mgr_SourceMgr_getFileNameIdx(p->sm, p->file_id), is_generated);
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
}

static void c2_parser_Parser_parseImports(c2_parser_Parser* p)
{
   while ((p->tok.kind == token_Kind_KW_import)) {
      c2_parser_Parser_consumeToken(p);
      c2_parser_Parser_expectIdentifier(p);
      uint32_t mod_name = p->tok.text_idx;
      src_loc_SrcLoc mod_loc = p->tok.loc;
      uint32_t alias_name = 0;
      src_loc_SrcLoc alias_loc = 0;
      c2_parser_Parser_consumeToken(p);
      if ((p->tok.kind == token_Kind_KW_as)) {
         c2_parser_Parser_consumeToken(p);
         c2_parser_Parser_expectIdentifier(p);
         alias_name = p->tok.text_idx;
         alias_loc = p->tok.loc;
         if (!c2_parser_checkName(string_pool_Pool_idx2str(p->pool, alias_name), false)) {
            c2_parser_Parser_error(p, "a module name must start with a lower case character");
         }
         c2_parser_Parser_consumeToken(p);
      }
      bool islocal = false;
      if ((p->tok.kind == token_Kind_KW_local)) {
         c2_parser_Parser_consumeToken(p);
         islocal = true;
      }
      c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
      ast_builder_Builder_actOnImport(p->builder, mod_name, mod_loc, alias_name, alias_loc, islocal);
   }
}

static void c2_parser_Parser_parseTopLevel(c2_parser_Parser* p)
{
   ast_builder_Builder_clearAttributes(p->builder);
   bool is_public = c2_parser_Parser_parseOptionalAccessSpecifier(p);
   switch (p->tok.kind) {
   case token_Kind_KW_assert:
      c2_parser_Parser_error(p, "assert can only be used inside a function");
      break;
   case token_Kind_KW_fn:
      c2_parser_Parser_parseFuncDecl(p, is_public);
      break;
   case token_Kind_KW_import:
      c2_parser_Parser_error(p, "no imports allowed after declarations");
      break;
   case token_Kind_KW_static_assert:
      if (is_public) c2_parser_Parser_error(p, "static_assert cannot be public");
      c2_parser_Parser_parseStaticAssert(p);
      break;
   case token_Kind_KW_type:
      c2_parser_Parser_parseTypeDecl(p, is_public);
      break;
   case token_Kind_Eof:
      break;
   case token_Kind_Identifier: {
      token_Token next = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, 1);
      if ((next.kind == token_Kind_PlusEqual)) {
         if (is_public) c2_parser_Parser_error(p, "incremental array entries cannot be public");
         c2_parser_Parser_parseArrayEntry(p);
         break;
      }
      __attribute__((fallthrough));
   }
   default:
      c2_parser_Parser_parseVarDecl(p, is_public);
      break;
   }
}

static void c2_parser_Parser_parseOptionalAttributes(c2_parser_Parser* p)
{
   if ((p->tok.kind != token_Kind_At)) return;

   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   while (1) {
      c2_parser_Parser_expectIdentifier(p);
      attr_Attr a;
      a.name = p->tok.text_idx;
      a.loc = p->tok.loc;
      a.value_kind = attr_ValueKind_None;
      c2_parser_Parser_consumeToken(p);
      if ((p->tok.kind == token_Kind_Equal)) {
         c2_parser_Parser_consumeToken(p);
         a.value.loc = p->tok.loc;
         switch (p->tok.kind) {
         case token_Kind_StringLiteral: {
            a.value_kind = attr_ValueKind_String;
            a.value.text = p->tok.text_idx;
            const char* str = string_pool_Pool_idx2str(p->pool, a.value.text);
            if ((str[0] == 0)) {
               c2_parser_Parser_error(p, "attribute argument cannot be an empty string");
            }
            c2_parser_Parser_consumeToken(p);
            break;
         }
         case token_Kind_IntegerLiteral:
            a.value_kind = attr_ValueKind_Number;
            a.value.number = ((uint32_t)(p->tok.int_value));
            c2_parser_Parser_consumeToken(p);
            break;
         default:
            c2_parser_Parser_error(p, "expected attribute argument");
            return;
         }
      }
      ast_builder_Builder_actOnAttr(p->builder, &a);
      if ((p->tok.kind != token_Kind_Comma)) break;

      c2_parser_Parser_consumeToken(p);
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
}

static void c2_parser_Parser_parseParamOptionalAttributes(c2_parser_Parser* p, ast_VarDecl* d)
{
   if ((p->tok.kind != token_Kind_At)) return;

   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   while (1) {
      c2_parser_Parser_expectIdentifier(p);
      uint32_t attr_id = p->tok.text_idx;
      src_loc_SrcLoc loc = p->tok.loc;
      c2_parser_Parser_consumeToken(p);
      if ((p->tok.kind == token_Kind_Equal)) {
         c2_parser_Parser_error(p, "a parameter attribute cannot have a value");
      }
      ast_builder_Builder_actOnParamAttr(p->builder, d, attr_id, loc);
      if ((p->tok.kind != token_Kind_Comma)) break;

      c2_parser_Parser_consumeToken(p);
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
}

static void c2_parser_Parser_parseFuncDecl(c2_parser_Parser* p, bool is_public)
{
   c2_parser_Parser_consumeToken(p);
   ast_TypeRefHolder rtype;
   ast_TypeRefHolder_init(&rtype);
   c2_parser_Parser_parseTypeSpecifier(p, &rtype, true, true);
   c2_parser_Parser_expectIdentifier(p);
   uint32_t func_name = p->tok.text_idx;
   src_loc_SrcLoc func_loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   ast_Ref prefix_ref;
   ast_Ref* prefix = NULL;
   if ((p->tok.kind == token_Kind_Dot)) {
      c2_parser_Parser_consumeToken(p);
      c2_parser_Parser_expectIdentifier(p);
      prefix_ref.loc = func_loc;
      prefix_ref.name_idx = func_name;
      prefix_ref.decl = NULL;
      prefix = &prefix_ref;
      func_name = p->tok.text_idx;
      func_loc = p->tok.loc;
      c2_parser_Parser_consumeToken(p);
   }
   if (!c2_parser_checkName(string_pool_Pool_idx2str(p->pool, func_name), p->is_interface)) {
      p->tok.loc = func_loc;
      c2_parser_Parser_error(p, "a function name must start with a lower case character");
   }
   ast_DeclList params;
   ast_DeclList_init(&params, 4);
   bool is_variadic = c2_parser_Parser_parseFunctionParams(p, &params, is_public);
   ast_FunctionDecl* f;
   if ((p->tok.kind == token_Kind_KW_template)) {
      c2_parser_Parser_consumeToken(p);
      c2_parser_Parser_expectIdentifier(p);
      uint32_t template_name = p->tok.text_idx;
      src_loc_SrcLoc template_loc = p->tok.loc;
      c2_parser_Parser_consumeToken(p);
      f = ast_builder_Builder_actOnTemplateFunctionDecl(p->builder, func_name, func_loc, is_public, &rtype, template_name, template_loc, ((ast_VarDecl**)(ast_DeclList_getDecls(&params))), ast_DeclList_size(&params), is_variadic);
   } else {
      f = ast_builder_Builder_actOnFunctionDecl(p->builder, func_name, func_loc, is_public, &rtype, prefix, ((ast_VarDecl**)(ast_DeclList_getDecls(&params))), ast_DeclList_size(&params), is_variadic);
   }
   ast_DeclList_free(&params);
   c2_parser_Parser_parseOptionalAttributes(p);
   ast_builder_Builder_applyAttributes(p->builder, ((ast_Decl*)(f)));
   if (p->is_interface) {
      if ((p->tok.kind == token_Kind_LBrace)) {
         c2_parser_Parser_error(p, "interface functions cannot have a function body");
      }
      c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
      return;
   }
   ast_CompoundStmt* body = c2_parser_Parser_parseCompoundStmt(p);
   ast_builder_Builder_actOnFunctionBody(p->builder, f, body);
}

static bool c2_parser_Parser_parseFunctionParams(c2_parser_Parser* p, ast_DeclList* params, bool is_public)
{
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   if ((p->tok.kind == token_Kind_RParen)) {
      c2_parser_Parser_consumeToken(p);
      return false;
   }
   bool is_variadic = false;
   while (1) {
      ast_VarDecl* decl = c2_parser_Parser_parseParamDecl(p, is_public);
      ast_DeclList_add(params, ast_VarDecl_asDecl(decl));
      if ((p->tok.kind != token_Kind_Comma)) break;

      c2_parser_Parser_consumeToken(p);
      if ((p->tok.kind == token_Kind_Ellipsis)) {
         is_variadic = true;
         c2_parser_Parser_consumeToken(p);
         break;
      }
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   return is_variadic;
}

static ast_VarDecl* c2_parser_Parser_parseParamDecl(c2_parser_Parser* p, bool is_public)
{
   if ((p->tok.kind == token_Kind_KW_local)) c2_parser_Parser_error(p, "keyword 'local' is not allowed here");
   ast_TypeRefHolder ref;
   ast_TypeRefHolder_init(&ref);
   c2_parser_Parser_parseTypeSpecifier(p, &ref, true, true);
   uint32_t name = 0;
   src_loc_SrcLoc loc = p->tok.loc;
   if ((p->tok.kind == token_Kind_Identifier)) {
      name = p->tok.text_idx;
      if (!c2_parser_checkName(string_pool_Pool_idx2str(p->pool, name), p->is_interface)) {
         c2_parser_Parser_error(p, "a parameter name must start with a lower case character");
      }
      c2_parser_Parser_consumeToken(p);
   }
   if ((p->tok.kind == token_Kind_Equal)) {
      c2_parser_Parser_error(p, "default parameter values are not allowed");
   }
   ast_VarDecl* param = ast_builder_Builder_actOnFunctionParam(p->builder, name, loc, is_public, &ref);
   c2_parser_Parser_parseParamOptionalAttributes(p, param);
   return param;
}

static void c2_parser_Parser_parseTypeSpecifier(c2_parser_Parser* p, ast_TypeRefHolder* ref, bool allow_qualifier, bool allow_array)
{
   c2_parser_Parser_parseSingleTypeSpecifier(p, ref, allow_qualifier);
   c2_parser_Parser_parseOptionalArray(p, ref, ast_QualType_Invalid, allow_array);
}

static void c2_parser_Parser_parseOptionalArray(c2_parser_Parser* p, ast_TypeRefHolder* ref, ast_QualType base, bool allow_array)
{
   if ((p->tok.kind != token_Kind_LSquare)) return;

   if (!allow_array) c2_parser_Parser_error(p, "array types are not allowed here");
   if ((ast_TypeRefHolder_getNumArrays(ref) == 3)) c2_parser_Parser_error(p, "arrays cannot have more than 3 dimensions");
   if (ast_TypeRefHolder_isIncrArray(ref)) c2_parser_Parser_error(p, "incremental arrays cannot have more than 1 dimension");
   c2_parser_Parser_consumeToken(p);
   bool is_incremental = false;
   ast_Expr* size = NULL;
   if ((p->tok.kind == token_Kind_RSquare)) {
      ast_TypeRefHolder_addArray(ref, NULL);
      c2_parser_Parser_consumeToken(p);
   } else if ((p->tok.kind == token_Kind_Plus)) {
      if (ast_TypeRefHolder_getNumArrays(ref)) c2_parser_Parser_error(p, "incremental arrays cannot have more than 1 dimension");
      c2_parser_Parser_consumeToken(p);
      ast_TypeRefHolder_setIncrArray(ref);
      is_incremental = true;
      c2_parser_Parser_expectAndConsume(p, token_Kind_RSquare);
   } else {
      size = c2_parser_Parser_parseExpr(p);
      ast_TypeRefHolder_addArray(ref, size);
      c2_parser_Parser_expectAndConsume(p, token_Kind_RSquare);
   }

   c2_parser_Parser_parseOptionalArray(p, ref, base, true);
}

static void c2_parser_Parser_parseArrayEntry(c2_parser_Parser* p)
{
   uint32_t name = p->tok.text_idx;
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_consumeToken(p);
   bool need_semi = true;
   ast_Expr* initValue = c2_parser_Parser_parseInitValue(p, &need_semi, false);
   if (need_semi) c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   ast_builder_Builder_actOnArrayValue(p->builder, name, loc, initValue);
}

static void c2_parser_Parser_parseVarDecl(c2_parser_Parser* p, bool is_public)
{
   if ((p->tok.kind == token_Kind_KW_local)) c2_parser_Parser_error(p, "keyword 'local' cannot be used at file scope");
   ast_TypeRefHolder ref;
   ast_TypeRefHolder_init(&ref);
   c2_parser_Parser_parseTypeSpecifier(p, &ref, true, true);
   c2_parser_Parser_expectIdentifier(p);
   uint32_t name = p->tok.text_idx;
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   bool need_semi = true;
   ast_Expr* initValue = NULL;
   src_loc_SrcLoc assignLoc = 0;
   c2_parser_Parser_parseOptionalAttributes(p);
   if ((p->tok.kind == token_Kind_Equal)) {
      assignLoc = p->tok.loc;
      c2_parser_Parser_consumeToken(p);
      initValue = c2_parser_Parser_parseInitValue(p, &need_semi, false);
   }
   if (need_semi) c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   ast_Decl* d = ast_builder_Builder_actOnGlobalVarDecl(p->builder, name, loc, is_public, &ref, assignLoc, initValue);
   ast_builder_Builder_applyAttributes(p->builder, d);
}

static void c2_parser_Parser_parseStaticAssert(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   ast_Expr* lhs = c2_parser_Parser_parseExpr(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Comma);
   ast_Expr* rhs = c2_parser_Parser_parseExpr(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   ast_builder_Builder_actOnStaticAssert(p->builder, loc, lhs, rhs);
}

static bool c2_parser_Parser_parseOptionalAccessSpecifier(c2_parser_Parser* p)
{
   if ((p->tok.kind == token_Kind_KW_public)) {
      c2_parser_Parser_consumeToken(p);
      return true;
   }
   return false;
}

static uint32_t c2_parser_Parser_parseOptionalTypeQualifier(c2_parser_Parser* p)
{
   uint32_t qualifiers = 0;
   if ((p->tok.kind == token_Kind_KW_const)) {
      c2_parser_Parser_consumeToken(p);
      qualifiers |= ast_QualType_Const;
   }
   if ((p->tok.kind == token_Kind_KW_volatile)) {
      c2_parser_Parser_consumeToken(p);
      qualifiers |= ast_QualType_Volatile;
   }
   return qualifiers;
}

static ast_BuiltinKind c2_parser_tokKindToBuiltinKind(token_Kind kind)
{
   return c2_parser_Tok2builtin[(kind - token_Kind_KW_bool)];
}

static void c2_parser_Parser_parseSingleTypeSpecifier(c2_parser_Parser* p, ast_TypeRefHolder* ref, bool allow_qualifier)
{
   if (allow_qualifier) {
      uint32_t type_qualifier = c2_parser_Parser_parseOptionalTypeQualifier(p);
      ast_TypeRefHolder_setQualifiers(ref, type_qualifier);
   }
   token_Kind kind = p->tok.kind;
   if (((kind >= token_Kind_KW_bool) && (kind <= token_Kind_KW_void))) {
      ast_TypeRefHolder_setBuiltin(ref, c2_parser_tokKindToBuiltinKind(p->tok.kind), p->tok.loc);
      if (((kind >= token_Kind_KW_reg8) && (kind <= token_Kind_KW_reg64))) ast_TypeRefHolder_setVolatile(ref);
      c2_parser_Parser_consumeToken(p);
   } else if ((kind == token_Kind_Identifier)) {
      c2_parser_Parser_parseFullTypeIdentifier(p, ref);
   } else {
      c2_parser_Parser_error(p, "expected type specifier");
   }

   uint32_t depth = 0;
   while ((p->tok.kind == token_Kind_Star)) {
      depth++;
      if ((depth > 3)) c2_parser_Parser_error(p, "pointers have a maximum nesting of 3");
      ast_TypeRefHolder_addPointer(ref);
      c2_parser_Parser_consumeToken(p);
   }
}

static void c2_parser_Parser_parseFullTypeIdentifier(c2_parser_Parser* p, ast_TypeRefHolder* ref)
{
   ast_TypeRefHolder_setUser(ref, p->tok.loc, p->tok.text_idx);
   c2_parser_Parser_consumeToken(p);
   if ((p->tok.kind == token_Kind_Dot)) {
      c2_parser_Parser_consumeToken(p);
      c2_parser_Parser_expectIdentifier(p);
      ast_TypeRefHolder_setPrefix(ref, p->tok.loc, p->tok.text_idx);
      c2_parser_Parser_consumeToken(p);
   }
}

static void c2_parser_Parser_dump_token(c2_parser_Parser* p, const token_Token* tok)
{
   if (((tok->kind >= token_Kind_KW_bool) && (tok->kind <= token_Kind_KW_while))) {
      printf("%s%12s%s  %6u %s\n", color_Green, token_kind2str(tok->kind), color_Normal, tok->loc, source_mgr_SourceMgr_loc2str(p->sm, tok->loc));
      return;
   }
   printf("%12s  %6u %s  ", token_kind2str(tok->kind), tok->loc, source_mgr_SourceMgr_loc2str(p->sm, tok->loc));
   switch (tok->kind) {
   case token_Kind_Identifier:
      printf("  %s%s%s", color_Cyan, string_pool_Pool_idx2str(p->pool, tok->text_idx), color_Normal);
      break;
   case token_Kind_IntegerLiteral:
      switch (tok->radix) {
      case 16:
         printf("  (%u) %s0x%lx%s", tok->radix, color_Cyan, tok->int_value, color_Normal);
         break;
      default:
         printf("  (%u) %s%lu%s", tok->radix, color_Cyan, tok->int_value, color_Normal);
         break;
      }
      break;
   case token_Kind_FloatLiteral:
      printf("  (%u) %s%lf%s", tok->radix, color_Cyan, tok->float_value, color_Normal);
      break;
   case token_Kind_CharLiteral:
      switch (tok->radix) {
      case 8:
         printf("  %s'\\%o'%s", color_Cyan, tok->char_value, color_Normal);
         break;
      case 16:
         printf("  %s'\\x%x'%s", color_Cyan, tok->char_value, color_Normal);
         break;
      default:
         if (isprint(tok->char_value)) {
            printf("  %s'%c'%s", color_Cyan, tok->char_value, color_Normal);
         } else {
            printf("  %s(%u)%s", color_Cyan, tok->char_value, color_Normal);
         }
         break;
      }
      break;
   case token_Kind_StringLiteral:
      printf("  %s\"%s\"%s (len %u)", color_Cyan, string_pool_Pool_idx2str(p->pool, tok->text_idx), color_Normal, tok->text_len);
      break;
   case token_Kind_LineComment:
      printf("  '%s%s%s'", color_Cyan, string_pool_Pool_idx2str(p->pool, tok->text_idx), color_Normal);
      break;
   case token_Kind_BlockComment:
      printf("  '%s%s%s'", color_Cyan, string_pool_Pool_idx2str(p->pool, tok->text_idx), color_Normal);
      break;
   case token_Kind_Warning:
      printf("  %s%s%s", color_Yellow, tok->error_msg, color_Normal);
      break;
   case token_Kind_Error:
      printf("  %s%s%s", color_Red, tok->error_msg, color_Normal);
      break;
   default:
      break;
   }
   printf("\n");
}

static bool c2_parser_checkName(const char* name, bool is_interface)
{
   if (is_interface) return true;

   char c = name[0];
   if (islower(c)) return true;

   return false;
}

static ast_Expr* c2_parser_Parser_parseExpr(c2_parser_Parser* p)
{
   ast_Expr* lhs = c2_parser_Parser_parseAssignmentExpression(p);
   return c2_parser_Parser_parseRHSOfBinaryExpression(p, lhs, c2_parser_Prec_Comma);
}

static ast_Expr* c2_parser_Parser_parseAssignmentExpression(c2_parser_Parser* p)
{
   ast_Expr* lhs = c2_parser_Parser_parseCastExpr(p, false, false);
   return c2_parser_Parser_parseRHSOfBinaryExpression(p, lhs, c2_parser_Prec_Assignment);
}

static ast_Expr* c2_parser_Parser_parseRHSOfBinaryExpression(c2_parser_Parser* p, ast_Expr* lhs, c2_parser_Prec minPrec)
{
   c2_parser_Prec nextTokPrec = c2_parser_BinOpPrecLookup[p->tok.kind];
   src_loc_SrcLoc colonLoc = 0;
   while (1) {
      if ((nextTokPrec < minPrec)) return lhs;

      if ((p->tok.kind == token_Kind_Comma)) return lhs;

      token_Token opToken = p->tok;
      c2_parser_Parser_consumeToken(p);
      ast_Expr* ternaryMiddle = NULL;
      if ((nextTokPrec == c2_parser_Prec_Conditional)) {
         if ((p->tok.kind == token_Kind_Colon)) {
            c2_parser_Parser_error(p, "TODO conditional expr");
         } else {
            ternaryMiddle = c2_parser_Parser_parseExpr(p);
         }
         if ((p->tok.kind == token_Kind_Colon)) {
            colonLoc = p->tok.loc;
            c2_parser_Parser_consumeToken(p);
         }
      }
      ast_Expr* rhs = c2_parser_Parser_parseCastExpr(p, false, false);
      c2_parser_Prec thisPrec = nextTokPrec;
      nextTokPrec = c2_parser_BinOpPrecLookup[p->tok.kind];
      bool isRightAssoc = (((thisPrec == c2_parser_Prec_Conditional) || (thisPrec == c2_parser_Prec_Assignment)));
      if (((thisPrec < nextTokPrec) || (((thisPrec == nextTokPrec) && isRightAssoc)))) {
         rhs = c2_parser_Parser_parseRHSOfBinaryExpression(p, rhs, (thisPrec + !isRightAssoc));
         nextTokPrec = c2_parser_BinOpPrecLookup[p->tok.kind];
      }
      if (ternaryMiddle) {
         lhs = ast_builder_Builder_actOnConditionalOperator(p->builder, opToken.loc, colonLoc, lhs, ternaryMiddle, rhs);
      } else {
         ast_BinaryOpcode opcode = c2_parser_BinOpTokenLookup[opToken.kind];
         lhs = ast_builder_Builder_actOnBinaryOperator(p->builder, opToken.loc, opcode, lhs, rhs);
      }
   }
   return NULL;
}

static ast_UnaryOpcode c2_parser_convertTokenToUnaryOpcode(token_Kind kind)
{
   switch (kind) {
   case token_Kind_Exclaim:
      return ast_UnaryOpcode_LNot;
   case token_Kind_Star:
      return ast_UnaryOpcode_Deref;
   case token_Kind_Amp:
      return ast_UnaryOpcode_AddrOf;
   case token_Kind_PlusPlus:
      return ast_UnaryOpcode_PreInc;
   case token_Kind_Minus:
      return ast_UnaryOpcode_Minus;
   case token_Kind_MinusMinus:
      return ast_UnaryOpcode_PreDec;
   case token_Kind_Tilde:
      return ast_UnaryOpcode_Not;
   default:
      c2_assert((0) != 0, "parser/c2_parser_expr.c2:216: c2_parser.convertTokenToUnaryOpcode", "0");
      break;
   }
   return ast_UnaryOpcode_PreInc;
}

static ast_Expr* c2_parser_Parser_parseCastExpr(c2_parser_Parser* p, bool _arg1, bool _arg2)
{
   token_Kind savedKind = p->tok.kind;
   ast_Expr* res = NULL;
   bool couldBeTemplateCall = false;
   switch (c2_parser_CastExprTokenLookup[savedKind]) {
   case 0:
      c2_parser_Parser_error(p, "expected expression");
      break;
   case 1: {
      token_Token t2 = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, 1);
      if ((t2.kind == token_Kind_Dot)) {
         res = c2_parser_Parser_parsePureMemberExpr(p);
      } else {
         res = ast_IdentifierExpr_asExpr(c2_parser_Parser_parseIdentifier(p));
      }
      couldBeTemplateCall = true;
      break;
   }
   case 2:
      res = ast_builder_Builder_actOnIntegerLiteral(p->builder, p->tok.loc, p->tok.radix, p->tok.int_value);
      c2_parser_Parser_consumeToken(p);
      break;
   case 3:
      res = ast_builder_Builder_actOnFloatLiteral(p->builder, p->tok.loc, p->tok.float_value);
      c2_parser_Parser_consumeToken(p);
      break;
   case 4:
      res = ast_builder_Builder_actOnCharLiteral(p->builder, p->tok.loc, p->tok.char_value, p->tok.radix);
      c2_parser_Parser_consumeToken(p);
      break;
   case 5:
      res = c2_parser_Parser_parseStringLiteral(p);
      break;
   case 6:
      res = c2_parser_Parser_parseParenExpr(p);
      break;
   case 7: {
      src_loc_SrcLoc loc = p->tok.loc;
      c2_parser_Parser_consumeToken(p);
      res = c2_parser_Parser_parseCastExpr(p, false, false);
      ast_UnaryOpcode opcode = c2_parser_convertTokenToUnaryOpcode(savedKind);
      return ast_builder_Builder_actOnUnaryOperator(p->builder, loc, opcode, res);
   }
   case 8: {
      src_loc_SrcLoc loc = p->tok.loc;
      c2_parser_Parser_consumeToken(p);
      res = c2_parser_Parser_parseCastExpr(p, false, true);
      ast_UnaryOpcode opcode = c2_parser_convertTokenToUnaryOpcode(savedKind);
      return ast_builder_Builder_actOnUnaryOperator(p->builder, loc, opcode, res);
   }
   case 9:
      return c2_parser_Parser_parseExplicitCastExpr(p);
   case 10:
      c2_parser_Parser_consumeToken(p);
      return c2_parser_Parser_parseCastExpr(p, false, false);
   case 11: {
      src_loc_SrcLoc loc = p->tok.loc;
      c2_parser_Parser_consumeToken(p);
      res = c2_parser_Parser_parseCastExpr(p, false, false);
      ast_UnaryOpcode opcode = c2_parser_convertTokenToUnaryOpcode(savedKind);
      return ast_builder_Builder_actOnUnaryOperator(p->builder, loc, opcode, res);
   }
   case 12:
      res = c2_parser_Parser_parseElemsof(p);
      break;
   case 13:
      return c2_parser_Parser_parseEnumMinMax(p, (savedKind == token_Kind_KW_enum_min));
   case 14:
      res = ast_builder_Builder_actOnBooleanConstant(p->builder, p->tok.loc, (savedKind == token_Kind_KW_true));
      c2_parser_Parser_consumeToken(p);
      break;
   case 15:
      res = ast_builder_Builder_actOnNilExpr(p->builder, p->tok.loc);
      c2_parser_Parser_consumeToken(p);
      break;
   case 16:
      return c2_parser_Parser_parseOffsetOfExpr(p);
   case 17:
      return c2_parser_Parser_parseSizeof(p);
   case 18:
      return c2_parser_Parser_parseToContainerExpr(p);
   }
   return c2_parser_Parser_parsePostfixExprSuffix(p, res, couldBeTemplateCall);
}

static ast_Expr* c2_parser_Parser_parsePostfixExprSuffix(c2_parser_Parser* p, ast_Expr* lhs, bool couldBeTemplateCall)
{
   while (1) {
      switch (p->tok.kind) {
      case token_Kind_Identifier:
         return lhs;
      case token_Kind_LParen:
         lhs = c2_parser_Parser_parseCallExpr(p, lhs);
         break;
      case token_Kind_LSquare: {
         c2_parser_Parser_consumeToken(p);
         ast_Expr* idx = c2_parser_Parser_parseExpr(p);
         if ((p->tok.kind == token_Kind_Colon)) {
            src_loc_SrcLoc colLoc = p->tok.loc;
            c2_parser_Parser_consumeToken(p);
            ast_Expr* rhs = c2_parser_Parser_parseExpr(p);
            idx = ast_builder_Builder_actOnBitOffsetExpr(p->builder, colLoc, idx, rhs);
         }
         src_loc_SrcLoc rloc = p->tok.loc;
         c2_parser_Parser_expectAndConsume(p, token_Kind_RSquare);
         lhs = ast_builder_Builder_actOnArraySubscriptExpr(p->builder, rloc, lhs, idx);
         break;
      }
      case token_Kind_Dot:
         lhs = c2_parser_Parser_parseImpureMemberExpr(p, lhs);
         break;
      case token_Kind_PlusPlus:
         __attribute__((fallthrough));
      case token_Kind_MinusMinus:
         lhs = ast_builder_Builder_actOnPostFixUnaryOperator(p->builder, p->tok.loc, (p->tok.kind == token_Kind_PlusPlus), lhs);
         c2_parser_Parser_consumeToken(p);
         break;
      case token_Kind_Less:
         if ((couldBeTemplateCall && c2_parser_Parser_isTemplateFunctionCall(p))) {
            c2_parser_Parser_consumeToken(p);
            ast_TypeRefHolder ref;
            ast_TypeRefHolder_init(&ref);
            c2_parser_Parser_parseTypeSpecifier(p, &ref, false, false);
            c2_parser_Parser_expectAndConsume(p, token_Kind_Greater);
            lhs = c2_parser_Parser_parseTemplateCallExpr(p, lhs, &ref);
         }
         return lhs;
      default:
         return lhs;
      }
   }
   c2_assert((0) != 0, "parser/c2_parser_expr.c2:373: c2_parser.Parser.parsePostfixExprSuffix", "0");
   return NULL;
}

static ast_Expr* c2_parser_Parser_parseCallExpr(c2_parser_Parser* p, ast_Expr* func)
{
   c2_parser_Parser_consumeToken(p);
   ast_ExprList args;
   ast_ExprList_init(&args, 4);
   while ((p->tok.kind != token_Kind_RParen)) {
      ast_ExprList_add(&args, c2_parser_Parser_parseExpr(p));
      if ((p->tok.kind == token_Kind_RParen)) break;

      if ((p->tok.kind != token_Kind_Comma)) break;

      c2_parser_Parser_consumeToken(p);
   }
   src_loc_SrcLoc endLoc = p->tok.loc;
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   ast_Expr* res = ast_builder_Builder_actOnCallExpr(p->builder, endLoc, func, ast_ExprList_getExprs(&args), ast_ExprList_size(&args));
   ast_ExprList_free(&args);
   return res;
}

static ast_Expr* c2_parser_Parser_parseTemplateCallExpr(c2_parser_Parser* p, ast_Expr* func, const ast_TypeRefHolder* ref)
{
   c2_parser_Parser_consumeToken(p);
   ast_ExprList args;
   ast_ExprList_init(&args, 4);
   while ((p->tok.kind != token_Kind_RParen)) {
      ast_ExprList_add(&args, c2_parser_Parser_parseExpr(p));
      if ((p->tok.kind == token_Kind_RParen)) break;

      c2_parser_Parser_expectAndConsume(p, token_Kind_Comma);
   }
   src_loc_SrcLoc endLoc = p->tok.loc;
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   ast_Expr* res = ast_builder_Builder_actOnTemplateCallExpr(p->builder, endLoc, func, ast_ExprList_getExprs(&args), ast_ExprList_size(&args), ref);
   ast_ExprList_free(&args);
   return res;
}

static ast_Expr* c2_parser_Parser_parseImpureMemberExpr(c2_parser_Parser* p, ast_Expr* base)
{
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectIdentifier(p);
   ast_Ref ref[7];
   ref[0].loc = p->tok.loc;
   ref[0].name_idx = p->tok.text_idx;
   uint32_t refcount = 1;
   c2_parser_Parser_consumeToken(p);
   while ((p->tok.kind == token_Kind_Dot)) {
      c2_parser_Parser_consumeToken(p);
      c2_parser_Parser_expectIdentifier(p);
      if ((refcount == 7)) c2_parser_Parser_error(p, "max member depth is %u", ast_MemberExprMaxDepth);
      ref[refcount].loc = p->tok.loc;
      ref[refcount].name_idx = p->tok.text_idx;
      refcount++;
      c2_parser_Parser_consumeToken(p);
   }
   return ast_builder_Builder_actOnMemberExpr(p->builder, base, ref, refcount);
}

static ast_Expr* c2_parser_Parser_parsePureMemberExpr(c2_parser_Parser* p)
{
   ast_Ref ref[7];
   ref[0].loc = p->tok.loc;
   ref[0].name_idx = p->tok.text_idx;
   uint32_t refcount = 1;
   c2_parser_Parser_consumeToken(p);
   while ((p->tok.kind == token_Kind_Dot)) {
      c2_parser_Parser_consumeToken(p);
      c2_parser_Parser_expectIdentifier(p);
      if ((refcount == 7)) c2_parser_Parser_error(p, "max member depth is %u", ast_MemberExprMaxDepth);
      ref[refcount].loc = p->tok.loc;
      ref[refcount].name_idx = p->tok.text_idx;
      refcount++;
      c2_parser_Parser_consumeToken(p);
   }
   return ast_builder_Builder_actOnMemberExpr(p->builder, NULL, ref, refcount);
}

static ast_IdentifierExpr* c2_parser_Parser_parseIdentifier(c2_parser_Parser* p)
{
   ast_IdentifierExpr* e = ast_builder_Builder_actOnIdentifier(p->builder, p->tok.loc, p->tok.text_idx);
   c2_parser_Parser_consumeToken(p);
   return e;
}

static ast_Expr* c2_parser_Parser_parseStringLiteral(c2_parser_Parser* p)
{
   ast_Expr* e = ast_builder_Builder_actOnStringLiteral(p->builder, p->tok.loc, p->tok.text_idx, p->tok.text_len);
   c2_parser_Parser_consumeToken(p);
   return e;
}

static ast_Expr* c2_parser_Parser_parseParenExpr(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   ast_Expr* res = c2_parser_Parser_parseExpr(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   return ast_builder_Builder_actOnParenExpr(p->builder, loc, res);
}

static bool c2_parser_Parser_isTemplateFunctionCall(c2_parser_Parser* p)
{
   c2_assert(((p->tok.kind == token_Kind_Less)) != 0, "parser/c2_parser_expr.c2:495: c2_parser.Parser.isTemplateFunctionCall", "p.tok.kind==Kind.Less");
   uint32_t ahead = 1;
   token_Token t = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, ahead);
   if (((t.kind >= token_Kind_KW_bool) && (t.kind <= token_Kind_KW_void))) return true;

   while ((ahead < 8)) {
      t = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, ahead);
      switch (t.kind) {
      case token_Kind_Identifier:
         __attribute__((fallthrough));
      case token_Kind_Star:
         __attribute__((fallthrough));
      case token_Kind_Dot:
         break;
      case token_Kind_Greater:
         t = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, (ahead + 1));
         return (t.kind == token_Kind_LParen);
      case token_Kind_KW_const:
         return true;
      default:
         return false;
      }
      ahead++;
   }
   return false;
}

static ast_Expr* c2_parser_Parser_parseSizeof(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   ast_Expr* res = NULL;
   bool has_brackets = false;
   if (c2_parser_Parser_parseAsType(p, &has_brackets)) {
      if (has_brackets) {
         while ((p->tok.kind != token_Kind_LSquare)) c2_parser_Parser_consumeToken(p);
         c2_parser_Parser_error(p, "arrays or subscripts expressions are not allowed inside a sizeof expression");
      }
      ast_TypeRefHolder ref;
      ast_TypeRefHolder_init(&ref);
      c2_parser_Parser_parseTypeSpecifier(p, &ref, false, true);
      res = ast_builder_Builder_actOnTypeExpr(p->builder, p->tok.loc, &ref);
   } else {
      if ((p->tok.kind != token_Kind_Identifier)) {
         c2_parser_Parser_error(p, "expect a type or variable name");
      }
      res = c2_parser_Parser_parseFullIdentifier(p);
      while ((p->tok.kind == token_Kind_LSquare)) {
         c2_parser_Parser_consumeToken(p);
         ast_Expr* idx = c2_parser_Parser_parseExpr(p);
         src_loc_SrcLoc rloc = p->tok.loc;
         c2_parser_Parser_expectAndConsume(p, token_Kind_RSquare);
         res = ast_builder_Builder_actOnArraySubscriptExpr(p->builder, rloc, res, idx);
      }
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   return ast_builder_Builder_actOnBuiltinExpr(p->builder, loc, res, ast_BuiltinExprKind_Sizeof);
}

static ast_Expr* c2_parser_Parser_parseElemsof(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   c2_parser_Parser_expectIdentifier(p);
   ast_Expr* res = c2_parser_Parser_parseFullIdentifier(p);
   res = c2_parser_Parser_parsePostfixExprSuffix(p, res, false);
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   return ast_builder_Builder_actOnBuiltinExpr(p->builder, loc, res, ast_BuiltinExprKind_Elemsof);
}

static ast_Expr* c2_parser_Parser_parseInitValue(c2_parser_Parser* p, bool* need_semi, bool allow_designators)
{
   switch (p->tok.kind) {
   case token_Kind_LBrace:
      *need_semi = false;
      return c2_parser_Parser_parseInitList(p);
   case token_Kind_Dot:
      if (!allow_designators) c2_parser_Parser_error(p, "designator not allowed here");
      return c2_parser_Parser_parseFieldDesignator(p, need_semi);
   case token_Kind_LSquare:
      if (!allow_designators) c2_parser_Parser_error(p, "designator not allowed here");
      return c2_parser_Parser_parseArrayDesignator(p, need_semi);
   default:
      break;
   }
   *need_semi = true;
   return c2_parser_Parser_parseAssignmentExpression(p);
}

static ast_Expr* c2_parser_Parser_parseInitList(c2_parser_Parser* p)
{
   src_loc_SrcLoc left = p->tok.loc;
   c2_parser_Parser_expectAndConsume(p, token_Kind_LBrace);
   ast_ExprList values;
   ast_ExprList_init(&values, 8);
   while ((p->tok.kind != token_Kind_RBrace)) {
      bool unused;
      ast_Expr* e = c2_parser_Parser_parseInitValue(p, &unused, true);
      ast_ExprList_add(&values, e);
      if ((p->tok.kind == token_Kind_Comma)) {
         c2_parser_Parser_consumeToken(p);
      } else {
         break;
      }
   }
   src_loc_SrcLoc right = p->tok.loc;
   c2_parser_Parser_expectAndConsume(p, token_Kind_RBrace);
   ast_Expr* e = ast_builder_Builder_actOnInitList(p->builder, left, right, ast_ExprList_getExprs(&values), ast_ExprList_size(&values));
   ast_ExprList_free(&values);
   return e;
}

static ast_Expr* c2_parser_Parser_parseFieldDesignator(c2_parser_Parser* p, bool* need_semi)
{
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectIdentifier(p);
   uint32_t field = p->tok.text_idx;
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Equal);
   ast_Expr* value = c2_parser_Parser_parseInitValue(p, need_semi, false);
   return ast_builder_Builder_actOnFieldDesignatedInit(p->builder, field, loc, value);
}

static ast_Expr* c2_parser_Parser_parseArrayDesignator(c2_parser_Parser* p, bool* need_semi)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   ast_Expr* designator = c2_parser_Parser_parseAssignmentExpression(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_RSquare);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Equal);
   ast_Expr* initValue = c2_parser_Parser_parseInitValue(p, need_semi, false);
   return ast_builder_Builder_actOnArrayDesignatedInit(p->builder, loc, designator, initValue);
}

static ast_Expr* c2_parser_Parser_parseExplicitCastExpr(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Less);
   ast_TypeRefHolder ref;
   ast_TypeRefHolder_init(&ref);
   c2_parser_Parser_parseTypeSpecifier(p, &ref, true, false);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Greater);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   ast_Expr* expr = c2_parser_Parser_parseExpr(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   return ast_builder_Builder_actOnExplicitCast(p->builder, loc, &ref, expr);
}

static ast_Expr* c2_parser_Parser_parseEnumMinMax(c2_parser_Parser* p, bool is_min)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   c2_parser_Parser_expectIdentifier(p);
   ast_Expr* expr = c2_parser_Parser_parseExpr(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   return ast_builder_Builder_actOnBuiltinExpr(p->builder, loc, expr, is_min ? ast_BuiltinExprKind_EnumMin : ast_BuiltinExprKind_EnumMax);
}

static ast_Expr* c2_parser_Parser_parseOffsetOfExpr(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   ast_Expr* structExpr = c2_parser_Parser_parseFullIdentifier(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Comma);
   ast_Expr* member = c2_parser_Parser_parseFullIdentifier(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   return ast_builder_Builder_actOnOffsetOfExpr(p->builder, loc, structExpr, member);
}

static ast_Expr* c2_parser_Parser_parseToContainerExpr(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   ast_Expr* structExpr = c2_parser_Parser_parseFullIdentifier(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Comma);
   ast_Expr* member = c2_parser_Parser_parseFullIdentifier(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Comma);
   ast_Expr* pointer = c2_parser_Parser_parseExpr(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   return ast_builder_Builder_actOnToContainerExpr(p->builder, loc, structExpr, member, pointer);
}

static ast_Expr* c2_parser_Parser_parseFullIdentifier(c2_parser_Parser* p)
{
   c2_parser_Parser_expectIdentifier(p);
   token_Token t2 = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, 1);
   if ((t2.kind == token_Kind_Dot)) {
      return c2_parser_Parser_parsePureMemberExpr(p);
   }
   return ast_IdentifierExpr_asExpr(c2_parser_Parser_parseIdentifier(p));
}

static bool c2_parser_Parser_parseAsType(c2_parser_Parser* p, bool* has_brackets)
{
   const token_Kind kind = p->tok.kind;
   if (((kind >= token_Kind_KW_bool) && (kind <= token_Kind_KW_void))) return true;

   if ((p->tok.kind != token_Kind_Identifier)) return false;

   uint32_t lookahead = 1;
   while (1) {
      token_Token t2 = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, lookahead);
      switch (t2.kind) {
      case token_Kind_Identifier:
         break;
      case token_Kind_Star:
         return true;
      case token_Kind_Dot:
         break;
      case token_Kind_LSquare:
         *has_brackets = true;
         return true;
      case token_Kind_Less:
         return true;
      default:
         return false;
      }
      lookahead++;
   }
   return false;
}

static ast_Stmt* c2_parser_Parser_parseStmt(c2_parser_Parser* p)
{
   switch (p->tok.kind) {
   case token_Kind_Identifier:
      return c2_parser_Parser_parseDeclOrStmt(p);
   case token_Kind_LBrace:
      return ((ast_Stmt*)(c2_parser_Parser_parseCompoundStmt(p)));
   case token_Kind_RBrace:
      c2_parser_Parser_error(p, "expected stmt");
      break;
   case token_Kind_KW_asm:
      return c2_parser_Parser_parseAsmStmt(p);
   case token_Kind_KW_assert:
      return c2_parser_Parser_parseAssertStmt(p);
   case token_Kind_KW_break:
      return c2_parser_Parser_parseBreakStmt(p);
   case token_Kind_KW_continue:
      return c2_parser_Parser_parseContinueStmt(p);
   case token_Kind_KW_do:
      return c2_parser_Parser_parseDoStmt(p);
   case token_Kind_KW_fallthrough:
      return c2_parser_Parser_parseFallthroughStmt(p);
   case token_Kind_KW_for:
      return c2_parser_Parser_parseForStmt(p);
   case token_Kind_KW_goto:
      return c2_parser_Parser_parseGotoStmt(p);
   case token_Kind_KW_if:
      return c2_parser_Parser_parseIfStmt(p);
   case token_Kind_KW_return:
      return c2_parser_Parser_parseReturnStmt(p);
   case token_Kind_KW_switch:
      return c2_parser_Parser_parseSwitchStmt(p, false);
   case token_Kind_KW_sswitch:
      return c2_parser_Parser_parseSwitchStmt(p, true);
   case token_Kind_KW_bool:
      __attribute__((fallthrough));
   case token_Kind_KW_char:
      __attribute__((fallthrough));
   case token_Kind_KW_const:
      __attribute__((fallthrough));
   case token_Kind_KW_i8:
      __attribute__((fallthrough));
   case token_Kind_KW_i16:
      __attribute__((fallthrough));
   case token_Kind_KW_i32:
      __attribute__((fallthrough));
   case token_Kind_KW_i64:
      __attribute__((fallthrough));
   case token_Kind_KW_isize:
      __attribute__((fallthrough));
   case token_Kind_KW_f32:
      __attribute__((fallthrough));
   case token_Kind_KW_f64:
      __attribute__((fallthrough));
   case token_Kind_KW_local:
      __attribute__((fallthrough));
   case token_Kind_KW_reg8:
      __attribute__((fallthrough));
   case token_Kind_KW_reg16:
      __attribute__((fallthrough));
   case token_Kind_KW_reg32:
      __attribute__((fallthrough));
   case token_Kind_KW_reg64:
      __attribute__((fallthrough));
   case token_Kind_KW_u8:
      __attribute__((fallthrough));
   case token_Kind_KW_u16:
      __attribute__((fallthrough));
   case token_Kind_KW_u32:
      __attribute__((fallthrough));
   case token_Kind_KW_u64:
      __attribute__((fallthrough));
   case token_Kind_KW_usize:
      __attribute__((fallthrough));
   case token_Kind_KW_volatile:
      __attribute__((fallthrough));
   case token_Kind_KW_void:
      return c2_parser_Parser_parseDeclStmt(p, true, true);
   case token_Kind_KW_while:
      return c2_parser_Parser_parseWhileStmt(p);
   default:
      return c2_parser_Parser_parseExprStmt(p);
   }
   return NULL;
}

static bool c2_parser_Parser_isTypeSpec(c2_parser_Parser* p)
{
   c2_assert(((p->tok.kind == token_Kind_Identifier)) != 0, "parser/c2_parser_stmt.c2:107: c2_parser.Parser.isTypeSpec", "p.tok.kind==Kind.Identifier");
   token_Token t;
   uint32_t state = 0;
   uint32_t lookahead = 1;
   while (1) {
      token_Token t2 = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, lookahead);
      switch (t2.kind) {
      case token_Kind_Identifier:
         goto type_done;
      case token_Kind_LSquare:
         lookahead = c2_parser_Parser_skipArray(p, lookahead);
         state = 3;
         break;
      case token_Kind_Star:
         if ((state == 3)) return false;

         state = 2;
         lookahead++;
         break;
      case token_Kind_Dot:
         if ((state == 0)) {
            token_Token t3 = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, (lookahead + 1));
            if ((t3.kind != token_Kind_Identifier)) {
               return false;
            }
            state = 2;
            lookahead += 2;
         } else {
            return false;
         }
         break;
      default:
         goto type_done;
      }
   }
   type_done:
   t = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, lookahead);
   return (t.kind == token_Kind_Identifier);
}

static uint32_t c2_parser_Parser_skipArray(c2_parser_Parser* p, uint32_t lookahead)
{
   lookahead++;
   uint32_t depth = 1;
   while (depth) {
      token_Token next = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, lookahead);
      switch (next.kind) {
      case token_Kind_LSquare:
         depth++;
         break;
      case token_Kind_RSquare:
         depth--;
         break;
      case token_Kind_Eof:
         c2_parser_Parser_error(p, "unexpected end-of-file");
         break;
      default:
         break;
      }
      lookahead++;
   }
   return lookahead;
}

static ast_Stmt* c2_parser_Parser_parseDeclOrStmt(c2_parser_Parser* p)
{
   c2_assert(((p->tok.kind == token_Kind_Identifier)) != 0, "parser/c2_parser_stmt.c2:186: c2_parser.Parser.parseDeclOrStmt", "p.tok.kind==Kind.Identifier");
   bool isDecl = c2_parser_Parser_isTypeSpec(p);
   if (isDecl) return c2_parser_Parser_parseDeclStmt(p, true, true);

   token_Token next = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, 1);
   if ((next.kind == token_Kind_Colon)) return c2_parser_Parser_parseLabelStmt(p);

   return c2_parser_Parser_parseExprStmt(p);
}

static ast_CompoundStmt* c2_parser_Parser_parseCompoundStmt(c2_parser_Parser* p)
{
   c2_parser_Parser_expectAndConsume(p, token_Kind_LBrace);
   stmt_list_List stmts;
   stmt_list_List_init(&stmts);
   while ((p->tok.kind != token_Kind_RBrace)) {
      stmt_list_List_add(&stmts, c2_parser_Parser_parseStmt(p));
   }
   src_loc_SrcLoc endLoc = p->tok.loc;
   c2_parser_Parser_expectAndConsume(p, token_Kind_RBrace);
   ast_CompoundStmt* s = ast_builder_Builder_actOnCompoundStmt(p->builder, endLoc, stmt_list_List_getData(&stmts), stmt_list_List_size(&stmts));
   stmt_list_List_free(&stmts);
   return s;
}

static ast_Stmt* c2_parser_Parser_parseAsmStmt(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   uint32_t quals = c2_parser_Parser_parseOptionalTypeQualifier(p);
   bool is_volatile = ((quals == ast_QualType_Volatile));
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   c2_parser_Parser_expect(p, token_Kind_StringLiteral);
   ast_Expr* str = c2_parser_Parser_parseStringLiteral(p);
   ast_ExprList constraints;
   ast_ExprList_init(&constraints, 2);
   ast_ExprList exprs;
   ast_ExprList_init(&exprs, 2);
   ast_ExprList clobbers;
   ast_ExprList_init(&clobbers, 2);
   if ((p->tok.kind == token_Kind_RParen)) {
      c2_parser_Parser_consumeToken(p);
      c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
      ast_Stmt* s = ast_builder_Builder_actOnAsmStmt(p->builder, loc, true, is_volatile, 0, 0, NULL, &constraints, &exprs, &clobbers, str);
      ast_ExprList_free(&constraints);
      ast_ExprList_free(&exprs);
      ast_ExprList_free(&clobbers);
      return s;
   }
   string_list_List names;
   string_list_List_init(&names, p->pool);
   if ((p->tok.kind == token_Kind_Colon)) {
      c2_parser_Parser_consumeToken(p);
      c2_parser_Parser_parseAsmOperandsOpt(p, &names, &constraints, &exprs);
   }
   uint32_t num_outputs = string_list_List_length(&names);
   if ((p->tok.kind == token_Kind_Colon)) {
      c2_parser_Parser_consumeToken(p);
      c2_parser_Parser_parseAsmOperandsOpt(p, &names, &constraints, &exprs);
   }
   uint32_t num_inputs = (string_list_List_length(&names) - num_outputs);
   if ((p->tok.kind == token_Kind_Colon)) {
      c2_parser_Parser_consumeToken(p);
      if ((p->tok.kind != token_Kind_RParen)) {
         while (1) {
            c2_parser_Parser_expect(p, token_Kind_StringLiteral);
            ast_Expr* e = c2_parser_Parser_parseStringLiteral(p);
            ast_ExprList_add(&clobbers, e);
            if ((p->tok.kind != token_Kind_Comma)) break;

            c2_parser_Parser_consumeToken(p);
         }
      }
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   ast_Stmt* s = ast_builder_Builder_actOnAsmStmt(p->builder, loc, false, is_volatile, num_outputs, num_inputs, string_list_List_getData(&names), &constraints, &exprs, &clobbers, str);
   string_list_List_free(&names);
   ast_ExprList_free(&constraints);
   ast_ExprList_free(&exprs);
   ast_ExprList_free(&clobbers);
   return s;
}

static void c2_parser_Parser_parseAsmOperandsOpt(c2_parser_Parser* p, string_list_List* names, ast_ExprList* constraints, ast_ExprList* exprs)
{
   if (((p->tok.kind != token_Kind_StringLiteral) && (p->tok.kind != token_Kind_LSquare))) return;

   while (1) {
      if ((p->tok.kind == token_Kind_LSquare)) {
         c2_parser_Parser_consumeToken(p);
         c2_parser_Parser_expectIdentifier(p);
         uint32_t name = p->tok.text_idx;
         c2_parser_Parser_consumeToken(p);
         string_list_List_add(names, name);
         c2_parser_Parser_expectAndConsume(p, token_Kind_RSquare);
      } else {
         string_list_List_add(names, 0);
      }
      c2_parser_Parser_expect(p, token_Kind_StringLiteral);
      ast_Expr* e = c2_parser_Parser_parseStringLiteral(p);
      ast_ExprList_add(constraints, e);
      c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
      e = c2_parser_Parser_parseExpr(p);
      c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
      ast_ExprList_add(exprs, e);
      if ((p->tok.kind != token_Kind_Comma)) return;

      c2_parser_Parser_consumeToken(p);
   }
}

static ast_Stmt* c2_parser_Parser_parseAssertStmt(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   ast_Expr* inner = c2_parser_Parser_parseExpr(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   return ast_builder_Builder_actOnAssertStmt(p->builder, loc, inner);
}

static ast_Stmt* c2_parser_Parser_parseBreakStmt(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   return ast_builder_Builder_actOnBreakStmt(p->builder, loc);
}

static ast_Stmt* c2_parser_Parser_parseContinueStmt(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   return ast_builder_Builder_actOnContinueStmt(p->builder, loc);
}

static ast_Stmt* c2_parser_Parser_parseFallthroughStmt(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   return ast_builder_Builder_actOnFallthroughStmt(p->builder, loc);
}

static ast_Stmt* c2_parser_Parser_parseCondition(c2_parser_Parser* p)
{
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   ast_Stmt* s;
   if (c2_parser_Parser_isDeclaration(p)) {
      s = c2_parser_Parser_parseDeclStmt(p, false, false);
   } else {
      ast_Expr* cond = c2_parser_Parser_parseExpr(p);
      s = ast_Expr_asStmt(cond);
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   return s;
}

static ast_Stmt* c2_parser_Parser_parseIfStmt(c2_parser_Parser* p)
{
   c2_parser_Parser_consumeToken(p);
   ast_Stmt* cond = c2_parser_Parser_parseCondition(p);
   ast_Stmt* then = c2_parser_Parser_parseStmt(p);
   ast_Stmt* else_stmt = NULL;
   if ((p->tok.kind == token_Kind_KW_else)) {
      c2_parser_Parser_consumeToken(p);
      else_stmt = c2_parser_Parser_parseStmt(p);
   }
   return ast_builder_Builder_actOnIfStmt(p->builder, cond, then, else_stmt);
}

static ast_Stmt* c2_parser_Parser_parseReturnStmt(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   ast_Expr* ret = NULL;
   if ((p->tok.kind != token_Kind_Semicolon)) {
      ret = c2_parser_Parser_parseExpr(p);
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   return ast_builder_Builder_actOnReturnStmt(p->builder, loc, ret);
}

static ast_Stmt* c2_parser_Parser_parseDoStmt(c2_parser_Parser* p)
{
   c2_parser_Parser_consumeToken(p);
   ast_Stmt* then = c2_parser_Parser_parseStmt(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_KW_while);
   ast_Stmt* cond = c2_parser_Parser_parseCondition(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   return ast_builder_Builder_actOnDoStmt(p->builder, cond, then);
}

static ast_Stmt* c2_parser_Parser_parseForStmt(c2_parser_Parser* p)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   ast_Stmt* init_ = NULL;
   if ((p->tok.kind != token_Kind_Semicolon)) {
      if (c2_parser_Parser_isDeclaration(p)) {
         init_ = c2_parser_Parser_parseDeclStmt(p, false, false);
      } else {
         init_ = ast_Expr_asStmt(c2_parser_Parser_parseExpr(p));
      }
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   ast_Expr* cond = NULL;
   if ((p->tok.kind != token_Kind_Semicolon)) {
      cond = c2_parser_Parser_parseExpr(p);
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   ast_Expr* incr = NULL;
   if ((p->tok.kind != token_Kind_RParen)) {
      incr = c2_parser_Parser_parseExpr(p);
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   ast_Stmt* body = c2_parser_Parser_parseStmt(p);
   return ast_builder_Builder_actOnForStmt(p->builder, loc, init_, cond, incr, body);
}

static ast_Stmt* c2_parser_Parser_parseWhileStmt(c2_parser_Parser* p)
{
   c2_parser_Parser_consumeToken(p);
   ast_Stmt* cond = c2_parser_Parser_parseCondition(p);
   ast_Stmt* then = c2_parser_Parser_parseStmt(p);
   return ast_builder_Builder_actOnWhileStmt(p->builder, cond, then);
}

static ast_Stmt* c2_parser_Parser_parseDeclStmt(c2_parser_Parser* p, bool checkSemi, bool allowLocal)
{
   bool has_local = false;
   if ((p->tok.kind == token_Kind_KW_local)) {
      has_local = true;
      if (!allowLocal) c2_parser_Parser_error(p, "keyword 'local' is not allowed here");
      c2_parser_Parser_consumeToken(p);
   }
   ast_TypeRefHolder ref;
   ast_TypeRefHolder_init(&ref);
   c2_parser_Parser_parseTypeSpecifier(p, &ref, true, true);
   c2_parser_Parser_expectIdentifier(p);
   uint32_t name = p->tok.text_idx;
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   bool need_semi = true;
   ast_Expr* initValue = NULL;
   src_loc_SrcLoc assignLoc = 0;
   if ((p->tok.kind == token_Kind_Equal)) {
      assignLoc = p->tok.loc;
      c2_parser_Parser_consumeToken(p);
      initValue = c2_parser_Parser_parseInitValue(p, &need_semi, false);
   }
   ast_Stmt* s = ast_builder_Builder_actOnVarDeclStmt(p->builder, name, loc, &ref, assignLoc, initValue, has_local);
   if ((checkSemi && need_semi)) {
      c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   }
   return s;
}

static ast_Stmt* c2_parser_Parser_parseExprStmt(c2_parser_Parser* p)
{
   ast_Expr* e = c2_parser_Parser_parseExpr(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   return ast_Expr_asStmt(e);
}

static ast_Stmt* c2_parser_Parser_parseLabelStmt(c2_parser_Parser* p)
{
   uint32_t name = p->tok.text_idx;
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Colon);
   return ast_builder_Builder_actOnLabelStmt(p->builder, name, loc);
}

static ast_Stmt* c2_parser_Parser_parseGotoStmt(c2_parser_Parser* p)
{
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectIdentifier(p);
   uint32_t name = p->tok.text_idx;
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   return ast_builder_Builder_actOnGotoStmt(p->builder, name, loc);
}

static bool c2_parser_Parser_isDeclaration(c2_parser_Parser* p)
{
   const token_Kind kind = p->tok.kind;
   if ((kind == token_Kind_Identifier)) return c2_parser_Parser_isTypeSpec(p);

   if (((kind >= token_Kind_KW_bool) && (kind <= token_Kind_KW_void))) return true;

   if ((kind == token_Kind_KW_local)) return true;

   return false;
}

static ast_Stmt* c2_parser_Parser_parseSwitchStmt(c2_parser_Parser* p, bool is_sswitch)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LParen);
   if (c2_parser_Parser_isDeclaration(p)) {
      c2_parser_Parser_error(p, "declarations inside a %sswitch condition are not allowed", is_sswitch ? "s" : "");
   }
   ast_Expr* cond = c2_parser_Parser_parseExpr(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_RParen);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LBrace);
   case_list_List cases;
   case_list_List_init(&cases);
   while ((p->tok.kind != token_Kind_RBrace)) {
      ast_SwitchCase* c = NULL;
      switch (p->tok.kind) {
      case token_Kind_KW_case:
         c = c2_parser_Parser_parseCase(p, false);
         break;
      case token_Kind_KW_default:
         c = c2_parser_Parser_parseCase(p, true);
         break;
      default:
         c2_parser_Parser_error(p, "expected 'case' or 'default' keyword");
         break;
      }
      case_list_List_add(&cases, c);
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_RBrace);
   ast_Stmt* s = ast_builder_Builder_actOnSwitchStmt(p->builder, loc, cond, case_list_List_getData(&cases), case_list_List_size(&cases), is_sswitch);
   case_list_List_free(&cases);
   return s;
}

static ast_Expr* c2_parser_Parser_parseCaseCondition(c2_parser_Parser* p, identifier_expr_list_List* list)
{
   bool multi_case = false;
   if ((p->tok.kind == token_Kind_Identifier)) {
      token_Token t2 = c2_tokenizer_Tokenizer_lookahead(&p->tokenizer, 1);
      if (((t2.kind == token_Kind_Comma) || (t2.kind == token_Kind_Minus))) multi_case = true;
   }
   if (!multi_case) {
      ast_Expr* e;
      switch (p->tok.kind) {
      case token_Kind_Identifier:
         e = c2_parser_Parser_parseFullIdentifier(p);
         break;
      case token_Kind_IntegerLiteral:
         e = ast_builder_Builder_actOnIntegerLiteral(p->builder, p->tok.loc, p->tok.radix, p->tok.int_value);
         c2_parser_Parser_consumeToken(p);
         break;
      case token_Kind_CharLiteral:
         e = ast_builder_Builder_actOnCharLiteral(p->builder, p->tok.loc, p->tok.char_value, p->tok.radix);
         c2_parser_Parser_consumeToken(p);
         break;
      case token_Kind_StringLiteral:
         e = c2_parser_Parser_parseStringLiteral(p);
         break;
      case token_Kind_KW_nil:
         e = ast_builder_Builder_actOnNilExpr(p->builder, p->tok.loc);
         c2_parser_Parser_consumeToken(p);
         break;
      default:
         c2_parser_Parser_error(p, "expected case condition");
         return NULL;
      }
      if (((p->tok.kind == token_Kind_Comma) || (p->tok.kind == token_Kind_Minus))) {
         c2_parser_Parser_error(p, "multi-condition case statements are only allowed with unprefixed enum constants");
      }
      return e;
   }
   while (1) {
      c2_parser_Parser_expectIdentifier(p);
      ast_IdentifierExpr* id1 = c2_parser_Parser_parseIdentifier(p);
      if (!identifier_expr_list_List_add(list, id1)) c2_parser_Parser_error(p, "too many conditions");
      if ((p->tok.kind == token_Kind_Minus)) {
         c2_parser_Parser_consumeToken(p);
         c2_parser_Parser_expectIdentifier(p);
         ast_IdentifierExpr* id2 = c2_parser_Parser_parseIdentifier(p);
         ast_IdentifierExpr_setCaseRange(id1);
         if (!identifier_expr_list_List_add(list, id2)) c2_parser_Parser_error(p, "too many conditions");
      }
      if ((p->tok.kind == token_Kind_Colon)) break;

      c2_parser_Parser_expectAndConsume(p, token_Kind_Comma);
   }
   return NULL;
}

static ast_SwitchCase* c2_parser_Parser_parseCase(c2_parser_Parser* p, bool is_default)
{
   src_loc_SrcLoc loc = p->tok.loc;
   c2_parser_Parser_consumeToken(p);
   ast_Expr* cond = NULL;
   identifier_expr_list_List list;
   identifier_expr_list_List_init(&list);
   if (!is_default) {
      cond = c2_parser_Parser_parseCaseCondition(p, &list);
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_Colon);
   stmt_list_List stmts;
   stmt_list_List_init(&stmts);
   bool more = true;
   while (more) {
      switch (p->tok.kind) {
      case token_Kind_RBrace:
         __attribute__((fallthrough));
      case token_Kind_KW_case:
         __attribute__((fallthrough));
      case token_Kind_KW_default:
         more = false;
         break;
      default:
         stmt_list_List_add(&stmts, c2_parser_Parser_parseStmt(p));
         break;
      }
   }
   ast_SwitchCase* s = ast_builder_Builder_actOnCase(p->builder, loc, is_default, cond, &list, stmt_list_List_getData(&stmts), stmt_list_List_size(&stmts));
   stmt_list_List_free(&stmts);
   return s;
}

static void c2_parser_Parser_parseTypeDecl(c2_parser_Parser* p, bool is_public)
{
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_expectIdentifier(p);
   uint32_t type_name = p->tok.text_idx;
   src_loc_SrcLoc type_loc = p->tok.loc;
   const char* name = string_pool_Pool_idx2str(p->pool, type_name);
   if (!isupper(name[0])) c2_parser_Parser_error(p, "a type name must start with an upper case character");
   c2_parser_Parser_consumeToken(p);
   switch (p->tok.kind) {
   case token_Kind_KW_fn:
      c2_parser_Parser_parseFunctionType(p, type_name, type_loc, is_public);
      break;
   case token_Kind_KW_struct:
      c2_parser_Parser_parseStructType(p, true, type_name, type_loc, is_public);
      break;
   case token_Kind_KW_union:
      c2_parser_Parser_parseStructType(p, false, type_name, type_loc, is_public);
      break;
   case token_Kind_KW_enum:
      c2_parser_Parser_parseEnumType(p, type_name, type_loc, is_public);
      break;
   default:
      c2_parser_Parser_parseAliasType(p, type_name, type_loc, is_public);
      break;
   }
}

static void c2_parser_Parser_parseFunctionType(c2_parser_Parser* p, uint32_t name, src_loc_SrcLoc loc, bool is_public)
{
   c2_parser_Parser_consumeToken(p);
   ast_TypeRefHolder rtype;
   ast_TypeRefHolder_init(&rtype);
   c2_parser_Parser_parseSingleTypeSpecifier(p, &rtype, true);
   ast_DeclList params;
   ast_DeclList_init(&params, 4);
   bool is_variadic = c2_parser_Parser_parseFunctionParams(p, &params, is_public);
   ast_Decl* f = ast_builder_Builder_actOnFunctionTypeDecl(p->builder, name, loc, is_public, &rtype, ((ast_VarDecl**)(ast_DeclList_getDecls(&params))), ast_DeclList_size(&params), is_variadic);
   c2_parser_Parser_parseOptionalAttributes(p);
   ast_DeclList_free(&params);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   ast_builder_Builder_applyAttributes(p->builder, f);
}

static void c2_parser_Parser_parseStructType(c2_parser_Parser* p, bool is_struct, uint32_t name, src_loc_SrcLoc loc, bool is_public)
{
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_parseOptionalAttributes(p);
   ast_DeclList members;
   ast_DeclList_init(&members, 8);
   c2_parser_Parser_parseStructBlock(p, &members, is_public);
   ast_StructTypeDecl* d = ast_builder_Builder_actOnStructType(p->builder, name, loc, is_public, is_struct, true, ast_DeclList_getDecls(&members), ast_DeclList_size(&members));
   ast_DeclList_free(&members);
   ast_builder_Builder_applyAttributes(p->builder, ((ast_Decl*)(d)));
}

static void c2_parser_Parser_parseStructBlock(c2_parser_Parser* p, ast_DeclList* members, bool is_public)
{
   c2_parser_Parser_expectAndConsume(p, token_Kind_LBrace);
   while (1) {
      if ((p->tok.kind == token_Kind_RBrace)) break;

      if (((p->tok.kind == token_Kind_KW_union) || (p->tok.kind == token_Kind_KW_struct))) {
         bool is_struct = (p->tok.kind == token_Kind_KW_struct);
         c2_parser_Parser_consumeToken(p);
         uint32_t name = 0;
         src_loc_SrcLoc loc = 0;
         if ((p->tok.kind == token_Kind_Identifier)) {
            name = p->tok.text_idx;
            loc = p->tok.loc;
            if (!c2_parser_checkName(string_pool_Pool_idx2str(p->pool, name), p->is_interface)) {
               c2_parser_Parser_error(p, "a struct/union member name must start with a lower case character");
            }
            c2_parser_Parser_consumeToken(p);
         }
         ast_DeclList sub_members;
         ast_DeclList_init(&sub_members, 4);
         c2_parser_Parser_parseStructBlock(p, &sub_members, is_public);
         ast_StructTypeDecl* member = ast_builder_Builder_actOnStructType(p->builder, name, loc, is_public, is_struct, false, ast_DeclList_getDecls(&sub_members), ast_DeclList_size(&sub_members));
         ast_DeclList_free(&sub_members);
         ast_DeclList_add(members, ast_StructTypeDecl_asDecl(member));
      } else {
         ast_TypeRefHolder ref;
         ast_TypeRefHolder_init(&ref);
         c2_parser_Parser_parseTypeSpecifier(p, &ref, true, true);
         uint32_t name = 0;
         src_loc_SrcLoc loc;
         if ((p->tok.kind == token_Kind_Colon)) {
            loc = p->tok.loc;
         } else {
            c2_parser_Parser_expectIdentifier(p);
            name = p->tok.text_idx;
            loc = p->tok.loc;
            if (!c2_parser_checkName(string_pool_Pool_idx2str(p->pool, name), p->is_interface)) {
               c2_parser_Parser_error(p, "a struct/union member name must start with a lower case character");
            }
            c2_parser_Parser_consumeToken(p);
         }
         ast_Expr* bitfield = NULL;
         if ((p->tok.kind == token_Kind_Colon)) {
            c2_parser_Parser_consumeToken(p);
            bitfield = c2_parser_Parser_parseExpr(p);
         }
         ast_VarDecl* member = ast_builder_Builder_actOnStructMember(p->builder, name, loc, is_public, &ref, bitfield);
         ast_DeclList_add(members, ast_VarDecl_asDecl(member));
         c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
      }
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_RBrace);
}

static void c2_parser_Parser_parseEnumType(c2_parser_Parser* p, uint32_t name, src_loc_SrcLoc loc, bool is_public)
{
   c2_parser_Parser_consumeToken(p);
   switch (p->tok.kind) {
   case token_Kind_KW_char:
      __attribute__((fallthrough));
   case token_Kind_KW_f32:
      __attribute__((fallthrough));
   case token_Kind_KW_f64:
      c2_parser_Parser_error(p, "enum type must be an integer");
      break;
   case token_Kind_KW_i8:
      __attribute__((fallthrough));
   case token_Kind_KW_i16:
      __attribute__((fallthrough));
   case token_Kind_KW_i32:
      __attribute__((fallthrough));
   case token_Kind_KW_i64:
      __attribute__((fallthrough));
   case token_Kind_KW_isize:
      break;
   case token_Kind_KW_reg8:
      __attribute__((fallthrough));
   case token_Kind_KW_reg16:
      __attribute__((fallthrough));
   case token_Kind_KW_reg32:
      __attribute__((fallthrough));
   case token_Kind_KW_reg64:
      c2_parser_Parser_error(p, "enum type must be an integer");
      break;
   case token_Kind_KW_u8:
      __attribute__((fallthrough));
   case token_Kind_KW_u16:
      __attribute__((fallthrough));
   case token_Kind_KW_u32:
      __attribute__((fallthrough));
   case token_Kind_KW_u64:
      __attribute__((fallthrough));
   case token_Kind_KW_usize:
      break;
   case token_Kind_KW_void:
      c2_parser_Parser_error(p, "enum type must be an integer");
      break;
   default:
      c2_parser_Parser_error(p, "expected enum type specifier");
      break;
   }
   ast_QualType implType = ast_builder_Builder_actOnBuiltinType(p->builder, c2_parser_tokKindToBuiltinKind(p->tok.kind));
   c2_parser_Parser_consumeToken(p);
   c2_parser_Parser_parseOptionalAttributes(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_LBrace);
   bool is_incr = false;
   ast_DeclList constants;
   ast_DeclList_init(&constants, 16);
   if ((p->tok.kind == token_Kind_Plus)) {
      is_incr = true;
      c2_parser_Parser_consumeToken(p);
   } else {
      while ((p->tok.kind == token_Kind_Identifier)) {
         uint32_t const_name = p->tok.text_idx;
         const char* name_str = string_pool_Pool_idx2str(p->pool, const_name);
         if (islower(name_str[0])) {
            c2_parser_Parser_error(p, "an enum constant name must start with an upper case character");
         }
         src_loc_SrcLoc const_loc = p->tok.loc;
         c2_parser_Parser_consumeToken(p);
         ast_Expr* init_expr = NULL;
         if ((p->tok.kind == token_Kind_Equal)) {
            c2_parser_Parser_consumeToken(p);
            init_expr = c2_parser_Parser_parseExpr(p);
         }
         ast_EnumConstantDecl* constant = ast_builder_Builder_actOnEnumConstant(p->builder, const_name, const_loc, is_public, init_expr);
         ast_DeclList_add(&constants, ast_EnumConstantDecl_asDecl(constant));
         if ((p->tok.kind != token_Kind_Comma)) break;

         c2_parser_Parser_consumeToken(p);
      }
      if ((ast_DeclList_size(&constants) == 0)) {
         c2_parser_Parser_error(p, "enum without constants");
      }
   }
   c2_parser_Parser_expectAndConsume(p, token_Kind_RBrace);
   ast_Decl* d = ast_builder_Builder_actOnEnumType(p->builder, name, loc, is_public, is_incr, implType, ((ast_EnumConstantDecl**)(ast_DeclList_getDecls(&constants))), ast_DeclList_size(&constants));
   ast_DeclList_free(&constants);
   ast_builder_Builder_applyAttributes(p->builder, d);
}

static void c2_parser_Parser_parseAliasType(c2_parser_Parser* p, uint32_t name, src_loc_SrcLoc loc, bool is_public)
{
   ast_TypeRefHolder ref;
   ast_TypeRefHolder_init(&ref);
   c2_parser_Parser_parseTypeSpecifier(p, &ref, true, true);
   ast_Decl* d = ast_builder_Builder_actOnAliasType(p->builder, name, loc, is_public, &ref);
   c2_parser_Parser_parseOptionalAttributes(p);
   c2_parser_Parser_expectAndConsume(p, token_Kind_Semicolon);
   ast_builder_Builder_applyAttributes(p->builder, d);
}


// --- module plugin_info ---
typedef struct plugin_info_Info_ plugin_info_Info;
typedef struct plugin_info_Plugin_ plugin_info_Plugin;

typedef bool (*plugin_info_RegisterAttrFn)(void* arg, uint32_t name, ast_AttrHandlerFn func, void* arg2);

struct plugin_info_Info_ {
   source_mgr_SourceMgr* sm;
   diagnostics_Diags* diags;
   build_target_Target* target;
   component_List* components;
   string_pool_Pool* astPool;
   string_pool_Pool* auxPool;
   ast_context_Context* context;
   ast_builder_Builder* builder;
   c2_parser_Parser* parser;
   ast_Globals* ast_globals;
   ast_QualType* ast_builtins;
   plugin_info_RegisterAttrFn register_attr;
   void* register_arg;
   char target_name[32];
   char output_dir[256];
};

typedef void* (*plugin_info_LoadFn)(const char* options, bool console_timing, bool console_debug);

typedef void (*plugin_info_UnloadFn)(void* arg);

typedef void (*plugin_info_InitFn)(void* arg, plugin_info_Info* info);

typedef void (*plugin_info_PostParseFn)(void* arg);

typedef void (*plugin_info_PostAnalysisFn)(void* arg);

struct plugin_info_Plugin_ {
   plugin_info_LoadFn load;
   plugin_info_UnloadFn unload;
   plugin_info_InitFn init;
   plugin_info_PostParseFn post_parse;
   plugin_info_PostAnalysisFn post_analysis;
   const char* name;
};

static bool plugin_info_Info_registerAttr(plugin_info_Info* info, uint32_t name, ast_AttrHandlerFn func, void* arg);
static bool plugin_info_Info_registerAttr(plugin_info_Info* info, uint32_t name, ast_AttrHandlerFn func, void* arg)
{
   return info->register_attr(info->register_arg, name, func, arg);
}


// --- module conversion_checker ---
typedef struct conversion_checker_Checker_ conversion_checker_Checker;
typedef struct conversion_checker_ExprWidth_ conversion_checker_ExprWidth;

struct conversion_checker_Checker_ {
   diagnostics_Diags* diags;
   ast_builder_Builder* builder;
   src_loc_SrcLoc loc;
   ast_QualType lhs;
   ast_QualType rhs;
   ast_Expr** expr_ptr;
};

static const uint8_t conversion_checker_Conversions[8][8] = {
   {
   2,
   3,
   1,
   1,
   13,
   15,
   0,
   0
},
   {
   4,
   5,
   1,
   1,
   1,
   6,
   0,
   0
},
   {
   1,
   7,
   8,
   1,
   1,
   1,
   0,
   0
},
   {
   1,
   1,
   1,
   9,
   1,
   1,
   0,
   0
},
   {
   10,
   1,
   1,
   1,
   14,
   1,
   0,
   0
},
   {
   16,
   11,
   1,
   1,
   1,
   12,
   0,
   0
},
   {
   0,
   0,
   0,
   0,
   0,
   0,
   0,
   0
},
   {
   0,
   0,
   0,
   0,
   0,
   0,
   0,
   0
}
};

static const uint8_t conversion_checker_BuiltinConversions[15][15] = {
   {
   0,
   7,
   1,
   1,
   1,
   3,
   3,
   3,
   3,
   1,
   1,
   1,
   3,
   1,
   2
},
   {
   7,
   0,
   1,
   1,
   1,
   3,
   3,
   3,
   3,
   1,
   1,
   1,
   3,
   1,
   2
},
   {
   4,
   4,
   0,
   1,
   1,
   3,
   3,
   3,
   3,
   1,
   1,
   1,
   3,
   1,
   2
},
   {
   4,
   4,
   4,
   0,
   1,
   3,
   3,
   3,
   3,
   1,
   1,
   1,
   3,
   1,
   2
},
   {
   4,
   4,
   4,
   4,
   0,
   3,
   3,
   3,
   3,
   1,
   1,
   7,
   3,
   1,
   2
},
   {
   3,
   3,
   3,
   3,
   3,
   0,
   1,
   1,
   1,
   1,
   1,
   3,
   1,
   1,
   2
},
   {
   4,
   4,
   3,
   3,
   3,
   4,
   0,
   1,
   1,
   1,
   1,
   3,
   1,
   1,
   2
},
   {
   4,
   4,
   4,
   3,
   3,
   4,
   4,
   0,
   1,
   1,
   1,
   3,
   1,
   1,
   2
},
   {
   4,
   4,
   4,
   4,
   3,
   4,
   4,
   4,
   0,
   1,
   1,
   3,
   7,
   1,
   2
},
   {
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   0,
   1,
   5,
   5,
   2,
   2
},
   {
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   6,
   0,
   5,
   5,
   2,
   2
},
   {
   4,
   4,
   4,
   4,
   7,
   4,
   4,
   4,
   3,
   1,
   1,
   0,
   3,
   1,
   2
},
   {
   4,
   4,
   4,
   4,
   3,
   4,
   4,
   4,
   7,
   1,
   1,
   3,
   0,
   1,
   2
},
   {
   1,
   1,
   1,
   1,
   1,
   1,
   1,
   1,
   1,
   1,
   1,
   1,
   1,
   0,
   2
},
   {
   2,
   2,
   2,
   2,
   2,
   2,
   2,
   2,
   2,
   2,
   2,
   2,
   2,
   2,
   0
}
};

static const uint8_t conversion_checker_ConditionalOperatorResult[15][15] = {
   {
   0,
   1,
   2,
   3,
   4,
   5,
   6,
   7,
   8,
   9,
   10,
   11,
   12,
   0,
   14
},
   {
   1,
   1,
   2,
   3,
   4,
   5,
   6,
   7,
   8,
   9,
   10,
   11,
   12,
   1,
   14
},
   {
   2,
   2,
   2,
   3,
   4,
   2,
   6,
   7,
   8,
   9,
   10,
   11,
   12,
   2,
   14
},
   {
   3,
   3,
   3,
   3,
   4,
   3,
   3,
   7,
   8,
   9,
   10,
   11,
   12,
   3,
   14
},
   {
   4,
   4,
   4,
   4,
   4,
   4,
   4,
   4,
   8,
   9,
   10,
   11,
   12,
   4,
   14
},
   {
   5,
   5,
   2,
   3,
   4,
   5,
   6,
   7,
   8,
   9,
   10,
   11,
   12,
   5,
   14
},
   {
   6,
   6,
   6,
   3,
   4,
   6,
   6,
   7,
   8,
   9,
   10,
   11,
   12,
   6,
   14
},
   {
   7,
   7,
   7,
   7,
   7,
   7,
   7,
   7,
   8,
   9,
   10,
   11,
   12,
   7,
   14
},
   {
   8,
   8,
   8,
   8,
   8,
   8,
   8,
   8,
   8,
   9,
   10,
   11,
   12,
   8,
   14
},
   {
   9,
   9,
   9,
   9,
   9,
   9,
   9,
   9,
   9,
   9,
   10,
   9,
   9,
   9,
   14
},
   {
   10,
   10,
   10,
   10,
   10,
   10,
   10,
   10,
   10,
   10,
   10,
   10,
   10,
   10,
   14
},
   {
   11,
   11,
   11,
   11,
   11,
   11,
   11,
   11,
   8,
   9,
   10,
   11,
   12,
   11,
   14
},
   {
   12,
   12,
   12,
   12,
   12,
   12,
   12,
   12,
   8,
   9,
   10,
   12,
   12,
   12,
   14
},
   {
   0,
   1,
   2,
   3,
   4,
   5,
   6,
   7,
   8,
   9,
   10,
   11,
   12,
   13,
   14
},
   {
   14,
   14,
   14,
   14,
   14,
   14,
   14,
   14,
   14,
   14,
   14,
   14,
   14,
   14,
   14
}
};

static const uint8_t conversion_checker_UsualArithmeticConversions[15][15] = {
   {
   0,
   0,
   0,
   0,
   2,
   0,
   0,
   1,
   3,
   4,
   5,
   6,
   6,
   0,
   6
},
   {
   0,
   0,
   0,
   0,
   2,
   0,
   0,
   1,
   3,
   4,
   5,
   6,
   6,
   0,
   6
},
   {
   0,
   0,
   0,
   0,
   2,
   0,
   0,
   1,
   3,
   4,
   5,
   6,
   6,
   0,
   6
},
   {
   0,
   0,
   0,
   0,
   2,
   0,
   0,
   1,
   3,
   4,
   5,
   6,
   6,
   0,
   6
},
   {
   2,
   2,
   2,
   3,
   2,
   2,
   2,
   2,
   3,
   4,
   5,
   6,
   6,
   2,
   6
},
   {
   0,
   0,
   0,
   0,
   2,
   0,
   0,
   1,
   3,
   4,
   5,
   6,
   6,
   0,
   6
},
   {
   0,
   0,
   0,
   0,
   2,
   0,
   0,
   1,
   3,
   4,
   5,
   6,
   6,
   0,
   6
},
   {
   1,
   1,
   1,
   1,
   3,
   1,
   1,
   1,
   3,
   4,
   5,
   6,
   6,
   1,
   6
},
   {
   3,
   3,
   3,
   3,
   3,
   3,
   3,
   3,
   3,
   4,
   5,
   6,
   6,
   3,
   6
},
   {
   4,
   4,
   4,
   4,
   4,
   4,
   4,
   4,
   4,
   4,
   5,
   4,
   4,
   4,
   6
},
   {
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   5,
   6
},
   {
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   3,
   4,
   5,
   6,
   6,
   6,
   6
},
   {
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   4,
   5,
   6,
   6,
   6,
   6
},
   {
   0,
   0,
   0,
   0,
   2,
   0,
   0,
   1,
   3,
   4,
   5,
   6,
   6,
   0,
   6
},
   {
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   6,
   6
}
};

static void conversion_checker_Checker_init(conversion_checker_Checker* c, diagnostics_Diags* diags, ast_builder_Builder* builder);
static bool conversion_checker_Checker_check(conversion_checker_Checker* c, ast_QualType lhs, ast_QualType rhs, ast_Expr** e_ptr, src_loc_SrcLoc loc);
static bool conversion_checker_Checker_checkTypes(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon);
static bool conversion_checker_Checker_checkBuiltins(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon);
static bool conversion_checker_Checker_checkBuiltin2Pointer(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon);
static bool conversion_checker_Checker_checkPointer2Builtin(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon);
static bool conversion_checker_Checker_checkPointer2Func(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon);
static bool conversion_checker_Checker_checkIntConversion(conversion_checker_Checker* c, const ast_BuiltinType* bi);
static bool conversion_checker_Checker_try_to_fix_type(conversion_checker_Checker* c);
static bool conversion_checker_pointer_conversion_allowed(ast_QualType linner, ast_QualType rinner);
static bool conversion_checker_Checker_checkPointers(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon);
static bool conversion_checker_Checker_checkFunc2Pointer(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon);
static bool conversion_checker_Checker_checkEnum2Int(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon);
static bool conversion_checker_Checker_checkFunc2Func(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon);
static bool conversion_checker_checkFunc2Func(const ast_FunctionDecl* fdl, const ast_FunctionDecl* fdr);
static bool conversion_checker_Checker_checkFunc2Builtin(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon, bool* other_error);
static bool conversion_checker_Checker_checkCast(conversion_checker_Checker* c, ast_QualType lhs, ast_QualType rhs, src_loc_SrcLoc lhsLoc, src_loc_SrcLoc rhsLoc);
static bool conversion_checker_Checker_checkBuiltin2PointerCast(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon);
static bool conversion_checker_Checker_checkPointer2BuiltinCast(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon);
static ast_QualType conversion_checker_get_common_arithmetic_type(ast_QualType t1, ast_QualType t2);
static ast_QualType conversion_checker_usual_arithmetic_conversion(const ast_BuiltinType* b1, const ast_BuiltinType* b2);
struct conversion_checker_ExprWidth_ {
   uint8_t width;
   bool is_signed;
};

static conversion_checker_ExprWidth conversion_checker_ExprWidth_mergeSmaller(conversion_checker_ExprWidth* w1, conversion_checker_ExprWidth* w2);
static conversion_checker_ExprWidth conversion_checker_ExprWidth_mergeWider(conversion_checker_ExprWidth* w1, conversion_checker_ExprWidth* w2);
static uint8_t conversion_checker_val2width(uint64_t value);
static conversion_checker_ExprWidth conversion_checker_getExprWidth(const ast_Expr* e);
static conversion_checker_ExprWidth conversion_checker_getCondOpWidth(const ast_ConditionalOperator* c);
static conversion_checker_ExprWidth conversion_checker_getUnaryOpWidth(const ast_UnaryOperator* u);
static conversion_checker_ExprWidth conversion_checker_getBinOpWidth(const ast_BinaryOperator* b);
static conversion_checker_ExprWidth conversion_checker_getTypeWidth(ast_QualType qt);
static void conversion_checker_Checker_init(conversion_checker_Checker* c, diagnostics_Diags* diags, ast_builder_Builder* builder)
{
   c->diags = diags;
   c->builder = builder;
}

static bool conversion_checker_Checker_check(conversion_checker_Checker* c, ast_QualType lhs, ast_QualType rhs, ast_Expr** e_ptr, src_loc_SrcLoc loc)
{
   c2_assert((lhs.ptr) != 0, "analyser/conversion_checker.c2:123: conversion_checker.Checker.check", "lhs.ptr");
   c2_assert((rhs.ptr) != 0, "analyser/conversion_checker.c2:124: conversion_checker.Checker.check", "rhs.ptr");
   c2_assert((!ast_Expr_isNValue((*e_ptr))) != 0, "analyser/conversion_checker.c2:125: conversion_checker.Checker.check", "!CALL TODO");
   ast_QualType t1 = ast_QualType_getCanonicalType(&lhs);
   ast_QualType t2 = ast_QualType_getCanonicalType(&rhs);
   if (((ast_Expr_isCtv((*e_ptr)) && ast_QualType_isBuiltin(&t1)) && !ast_QualType_isPointer(&t2))) {
      if (ast_QualType_isBool(&t1)) return true;

      return ctv_analyser_check(c->diags, lhs, *e_ptr);
   }
   const ast_Type* lcanon = ast_QualType_getTypeOrNil(&t1);
   const ast_Type* rcanon = ast_QualType_getTypeOrNil(&t2);
   if ((lcanon == rcanon)) {
      uint32_t lquals = ast_QualType_getQuals(&lhs);
      uint32_t rquals = ast_QualType_getQuals(&rhs);
      if ((((ast_Type_getKind(lcanon) == ast_TypeKind_Pointer) && ast_QualType_isConst(&rhs)) && !ast_QualType_isConst(&lhs))) {
         diagnostics_Diags_error(c->diags, loc, "conversion discards const qualifier");
         return false;
      }
      return true;
   }
   c->lhs = lhs;
   c->rhs = rhs;
   c->expr_ptr = e_ptr;
   c->loc = loc;
   return conversion_checker_Checker_checkTypes(c, lcanon, rcanon);
}

static bool conversion_checker_Checker_checkTypes(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon)
{
   uint8_t res = conversion_checker_Conversions[ast_Type_getKind(rcanon)][ast_Type_getKind(lcanon)];
   switch (res) {
   case 0:
      diagnostics_Diags_error(c->diags, c->loc, "SHOULD NOT HAPPEN (%u - %u)\n", ast_Type_getKind(lcanon), ast_Type_getKind(rcanon));
      ast_QualType_dump_full(&c->lhs);
      ast_QualType_dump_full(&c->rhs);
      c2_assert((0) != 0, "analyser/conversion_checker.c2:172: conversion_checker.Checker.checkTypes", "0");
      return false;
   case 1:
      diagnostics_Diags_error(c->diags, c->loc, "invalid type conversion from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      return false;
   case 2:
      return conversion_checker_Checker_checkBuiltins(c, lcanon, rcanon);
   case 3:
      return conversion_checker_Checker_checkBuiltin2Pointer(c, lcanon, rcanon);
   case 4:
      return conversion_checker_Checker_checkPointer2Builtin(c, lcanon, rcanon);
   case 5:
      return conversion_checker_Checker_checkPointers(c, lcanon, rcanon);
   case 6:
      return conversion_checker_Checker_checkPointer2Func(c, lcanon, rcanon);
   case 7:
      diagnostics_Diags_note(c->diags, c->loc, "SHOULD NOT HAPPEN (Array -> Ptr)");
      ast_QualType_dump_full(&c->lhs);
      ast_QualType_dump_full(&c->rhs);
      c2_assert((0) != 0, "analyser/conversion_checker.c2:191: conversion_checker.Checker.checkTypes", "0");
      return false;
   case 8:
      diagnostics_Diags_error(c->diags, c->loc, "invalid type conversion from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      return false;
   case 9:
      diagnostics_Diags_error(c->diags, c->loc, "conversion between struct of different types ('%s' to '%s')", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      return false;
   case 10:
      return conversion_checker_Checker_checkEnum2Int(c, lcanon, rcanon);
   case 11:
      return conversion_checker_Checker_checkFunc2Pointer(c, lcanon, rcanon);
   case 12:
      return conversion_checker_Checker_checkFunc2Func(c, lcanon, rcanon);
   case 13:
      diagnostics_Diags_error(c->diags, c->loc, "invalid type conversion from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      return false;
   case 14:
      diagnostics_Diags_error(c->diags, c->loc, "invalid type conversion from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      return false;
   case 15:
      diagnostics_Diags_error(c->diags, c->loc, "invalid type conversion from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      return false;
   case 16: {
      bool other = false;
      bool ok = conversion_checker_Checker_checkFunc2Builtin(c, lcanon, rcanon, &other);
      if ((!ok && !other)) {
         diagnostics_Diags_error(c->diags, c->loc, "invalid type conversion from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      }
      return ok;
   }
   default:
      diagnostics_Diags_note(c->diags, c->loc, "TODO CONVERSION  %u)", res);
      return false;
   }
   return true;
}

static bool conversion_checker_Checker_checkBuiltins(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon)
{
   const ast_BuiltinType* lbuiltin = ((ast_BuiltinType*)(lcanon));
   const ast_BuiltinType* rbuiltin = ((ast_BuiltinType*)(rcanon));
   if (ast_Expr_isCtv((*c->expr_ptr))) {
      return true;
   }
   uint8_t res = conversion_checker_BuiltinConversions[ast_BuiltinType_getKind(rbuiltin)][ast_BuiltinType_getKind(lbuiltin)];
   switch (res) {
   case 0:
      printf("BUILTIN SHOULD NOT HAPPEN (%u - %u)\n", ast_Type_getKind(lcanon), ast_Type_getKind(rcanon));
      c2_assert((0) != 0, "analyser/conversion_checker.c2:244: conversion_checker.Checker.checkBuiltins", "0");
      return false;
   case 1:
      ast_builder_Builder_insertImplicitCast(c->builder, ast_ImplicitCastKind_IntegralCast, c->expr_ptr, c->lhs);
      break;
   case 2:
      diagnostics_Diags_error(c->diags, c->loc, "invalid type conversion from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      return false;
   case 3:
      if (conversion_checker_Checker_checkIntConversion(c, lbuiltin)) {
         ast_builder_Builder_insertImplicitCast(c->builder, ast_ImplicitCastKind_IntegralCast, c->expr_ptr, c->lhs);
      } else {
         diagnostics_Diags_error(c->diags, c->loc, "implicit conversion changes signedness: '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      }
      break;
   case 4:
      if (conversion_checker_Checker_checkIntConversion(c, lbuiltin)) {
         ast_builder_Builder_insertImplicitCast(c->builder, ast_ImplicitCastKind_IntegralCast, c->expr_ptr, c->lhs);
      } else {
         ast_Expr* e = *c->expr_ptr;
         diagnostics_Diags_errorRange(c->diags, c->loc, ast_Expr_getRange(e), "implicit conversion loses integer precision: '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      }
      break;
   case 5:
      diagnostics_Diags_error(c->diags, c->loc, "implicit conversion turns floating-point number into integer: '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      break;
   case 6:
      diagnostics_Diags_error(c->diags, c->loc, "implicit conversion loses floating-point precision: '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      break;
   case 7:
      break;
   default:
      c2_assert((0) != 0, "analyser/conversion_checker.c2:277: conversion_checker.Checker.checkBuiltins", "0");
      return false;
   }
   return true;
}

static bool conversion_checker_Checker_checkBuiltin2Pointer(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon)
{
   const ast_PointerType* ptr = ((ast_PointerType*)(lcanon));
   const ast_BuiltinType* bi = ((ast_BuiltinType*)(rcanon));
   ast_QualType inner = ast_PointerType_getInner(ptr);
   bool ok = ast_QualType_isVoid(&inner);
   ast_BuiltinKind kind = ast_BuiltinType_getKind(bi);
   ok &= (((kind == ast_BuiltinKind_USize) || (kind == ast_BuiltinKind_UInt64)));
   if (!ok) {
      diagnostics_Diags_error(c->diags, c->loc, "incompatible integer to pointer conversion: '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      return false;
   }
   return true;
}

static bool conversion_checker_Checker_checkPointer2Builtin(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon)
{
   const ast_BuiltinType* bi = ((ast_BuiltinType*)(lcanon));
   ast_BuiltinKind kind = ast_BuiltinType_getKind(bi);
   if ((kind == ast_BuiltinKind_Bool)) {
      ast_builder_Builder_insertImplicitCast(c->builder, ast_ImplicitCastKind_PointerToBoolean, c->expr_ptr, ast_builtins[ast_BuiltinKind_Bool]);
      return true;
   }
   const ast_PointerType* ptr = ((ast_PointerType*)(rcanon));
   ast_QualType inner = ast_PointerType_getInner(ptr);
   bool ok = ast_QualType_isVoid(&inner);
   ok &= (((kind == ast_BuiltinKind_USize) || (kind == ast_BuiltinKind_UInt64)));
   if (!ok) {
      if (conversion_checker_Checker_try_to_fix_type(c)) {
         diagnostics_Diags_error(c->diags, c->loc, "invalid type conversion from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      } else {
         diagnostics_Diags_error(c->diags, c->loc, "incompatible pointer to integer conversion: '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      }
      return false;
   }
   ast_builder_Builder_insertImplicitCast(c->builder, ast_ImplicitCastKind_PointerToInteger, c->expr_ptr, ast_builtins[ast_BuiltinKind_USize]);
   return true;
}

static bool conversion_checker_Checker_checkPointer2Func(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon)
{
   const ast_PointerType* ptr = ((ast_PointerType*)(rcanon));
   ast_QualType inner = ast_PointerType_getInner(ptr);
   if (!ast_QualType_isVoid(&inner)) {
      diagnostics_Diags_error(c->diags, c->loc, "incompatible pointer to function conversion '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      return false;
   }
   ast_builder_Builder_insertImplicitCast(c->builder, ast_ImplicitCastKind_BitCast, c->expr_ptr, c->lhs);
   return true;
}

static bool conversion_checker_Checker_checkIntConversion(conversion_checker_Checker* c, const ast_BuiltinType* bi)
{
   ast_Expr* e = *c->expr_ptr;
   const uint8_t wl = ((uint8_t)(ast_BuiltinType_getWidth(bi)));
   conversion_checker_ExprWidth w = conversion_checker_getExprWidth(e);
   if ((w.is_signed == ast_BuiltinType_isSigned(bi))) {
      return ((w.width <= wl));
   } else {
      if (w.is_signed) return false;

      return (w.width <= wl);
   }
   return false;
}

static bool conversion_checker_Checker_try_to_fix_type(conversion_checker_Checker* c)
{
   ast_Expr* e = *c->expr_ptr;
   if (ast_Expr_isImplicitCast(e)) {
      const ast_ImplicitCastExpr* ic = ((ast_ImplicitCastExpr*)(e));
      if (ast_ImplicitCastExpr_isArrayToPointerDecay(ic)) {
         e = ast_ImplicitCastExpr_getInner(ic);
         c->rhs = ast_Expr_getType(e);
         return true;
      }
   }
   return false;
}

static bool conversion_checker_pointer_conversion_allowed(ast_QualType linner, ast_QualType rinner)
{
   const ast_Type* in1 = ast_QualType_getTypeOrNil(&linner);
   const ast_Type* in2 = ast_QualType_getTypeOrNil(&rinner);
   if ((in1 == in2)) return true;

   if (ast_Type_isVoidType(in1)) return true;

   if (ast_Type_isVoidType(in2)) return true;

   if ((ast_QualType_isInt8(&linner) && ast_QualType_isChar(&rinner))) return true;

   if ((ast_QualType_isChar(&linner) && ast_QualType_isInt8(&rinner))) return true;

   return false;
}

static bool conversion_checker_Checker_checkPointers(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon)
{
   const ast_PointerType* ltype = ((ast_PointerType*)(lcanon));
   const ast_PointerType* rtype = ((ast_PointerType*)(rcanon));
   ast_QualType linner = ast_PointerType_getInner(ltype);
   ast_QualType rinner = ast_PointerType_getInner(rtype);
   if (!conversion_checker_pointer_conversion_allowed(linner, rinner)) {
      if (conversion_checker_Checker_try_to_fix_type(c)) {
         diagnostics_Diags_error(c->diags, c->loc, "invalid type conversion from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      } else {
         diagnostics_Diags_error(c->diags, c->loc, "invalid pointer conversion from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      }
      return false;
   }
   uint32_t rquals = ast_QualType_getQuals(&rinner);
   if ((rquals == 0)) return true;

   uint32_t lquals = ast_QualType_getQuals(&linner);
   if (((((~lquals) & rquals)) & 0x3)) {
      diagnostics_Diags_error(c->diags, c->loc, "pointer conversion discards const qualifier");
      return false;
   }
   return true;
}

static bool conversion_checker_Checker_checkFunc2Pointer(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon)
{
   ast_PointerType* pt = ((ast_PointerType*)(lcanon));
   ast_QualType inner = ast_PointerType_getInner(pt);
   if (ast_QualType_isVoid(&inner)) return true;

   diagnostics_Diags_note(c->diags, c->loc, "TODO F2P");
   ast_QualType_dump(&c->lhs);
   ast_QualType_dump(&c->rhs);
   return true;
}

static bool conversion_checker_Checker_checkEnum2Int(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon)
{
   const ast_BuiltinType* bi = ((ast_BuiltinType*)(lcanon));
   uint32_t width = ast_BuiltinType_getWidth(bi);
   if ((width == 64)) return true;

   if (ast_Expr_isCtv((*c->expr_ptr))) {
      return ctv_analyser_check(c->diags, c->lhs, *c->expr_ptr);
   } else {
      const ast_EnumType* et = ((ast_EnumType*)(rcanon));
      const ast_EnumTypeDecl* etd = ast_EnumType_getDecl(et);
      ast_QualType impl = ast_EnumTypeDecl_getImplType(etd);
      return conversion_checker_Checker_check(c, c->lhs, impl, c->expr_ptr, c->loc);
   }
   return true;
}

static bool conversion_checker_Checker_checkFunc2Func(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon)
{
   ast_FunctionType* ftl = ((ast_FunctionType*)(lcanon));
   ast_FunctionDecl* fdl = ast_FunctionType_getDecl(ftl);
   ast_FunctionType* ftr = ((ast_FunctionType*)(rcanon));
   ast_FunctionDecl* fdr = ast_FunctionType_getDecl(ftr);
   if (ast_FunctionDecl_getNumAutoArgs(fdr)) {
      diagnostics_Diags_error(c->diags, c->loc, "functions used as function pointers cannot have auto-arguments");
      return false;
   }
   if (!conversion_checker_checkFunc2Func(fdl, fdr)) {
      diagnostics_Diags_error(c->diags, c->loc, "invalid function conversion from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      return false;
   }
   return true;
}

static bool conversion_checker_checkFunc2Func(const ast_FunctionDecl* fdl, const ast_FunctionDecl* fdr)
{
   ast_QualType ql = ast_FunctionDecl_getRType(fdl);
   ast_QualType qr = ast_FunctionDecl_getRType(fdr);
   if ((ql.ptr != qr.ptr)) return false;

   uint32_t num1 = ast_FunctionDecl_getNumParams(fdl);
   uint32_t num2 = ast_FunctionDecl_getNumParams(fdr);
   if ((num1 != num2)) return false;

   ast_Decl** args1 = ((ast_Decl**)(ast_FunctionDecl_getParams(fdl)));
   ast_Decl** args2 = ((ast_Decl**)(ast_FunctionDecl_getParams(fdr)));
   for (uint32_t i = 0; (i < num1); i++) {
      ast_Decl* a1 = args1[i];
      ast_Decl* a2 = args2[i];
      ql = ast_Decl_getType(a1);
      qr = ast_Decl_getType(a2);
      if ((ql.ptr != qr.ptr)) return false;

   }
   if ((ast_FunctionDecl_isVariadic(fdl) != ast_FunctionDecl_isVariadic(fdr))) return false;

   return true;
}

static bool conversion_checker_Checker_checkFunc2Builtin(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon, bool* other_error)
{
   uint32_t wordsize = ast_getWordSize();
   const ast_BuiltinType* bi = ((ast_BuiltinType*)(lcanon));
   ast_BuiltinKind kind = ast_BuiltinType_getKind(bi);
   if ((kind == ast_BuiltinKind_USize)) return true;

   if ((kind == ast_BuiltinKind_Bool)) {
      ast_FunctionType* ft = ((ast_FunctionType*)(rcanon));
      ast_FunctionDecl* fd = ast_FunctionType_getDecl(ft);
      if ((ast_FunctionDecl_isType(fd) || ast_FunctionDecl_hasAttrWeak(fd))) return true;

      diagnostics_Diags_error(c->diags, c->loc, "comparison of function '%s' will always be true", ast_Decl_getFullName(ast_FunctionDecl_asDecl(fd)));
      *other_error = true;
      return false;
   }
   if (((wordsize == 4) && (kind == ast_BuiltinKind_UInt32))) return true;

   if (((wordsize == 8) && (kind == ast_BuiltinKind_UInt64))) return true;

   return false;
}

static bool conversion_checker_Checker_checkCast(conversion_checker_Checker* c, ast_QualType lhs, ast_QualType rhs, src_loc_SrcLoc lhsLoc, src_loc_SrcLoc rhsLoc)
{
   c->lhs = lhs;
   c->rhs = rhs;
   c->loc = lhsLoc;
   c->expr_ptr = NULL;
   if ((lhsLoc == 0)) c->loc = rhsLoc;
   ast_QualType t1 = ast_QualType_getCanonicalType(&lhs);
   ast_QualType t2 = ast_QualType_getCanonicalType(&rhs);
   const ast_Type* lcanon = ast_QualType_getTypeOrNil(&t1);
   const ast_Type* rcanon = ast_QualType_getTypeOrNil(&t2);
   uint8_t res = conversion_checker_Conversions[ast_Type_getKind(rcanon)][ast_Type_getKind(lcanon)];
   switch (res) {
   case 0:
      diagnostics_Diags_error(c->diags, lhsLoc, "SHOULD NOT HAPPEN (%u - %u)\n", ast_Type_getKind(lcanon), ast_Type_getKind(rcanon));
      ast_QualType_dump_full(&c->lhs);
      ast_QualType_dump_full(&c->rhs);
      c2_assert((0) != 0, "analyser/conversion_checker.c2:560: conversion_checker.Checker.checkCast", "0");
      return false;
   case 1:
      diagnostics_Diags_error(c->diags, c->loc, "invalid cast from '%s' to '%s'", ast_QualType_diagName(&c->rhs), ast_QualType_diagName(&c->lhs));
      return false;
   case 2:
      return true;
   case 3:
      return conversion_checker_Checker_checkBuiltin2PointerCast(c, lcanon, rcanon);
   case 4:
      return conversion_checker_Checker_checkPointer2BuiltinCast(c, lcanon, rcanon);
   case 5:
      return true;
   case 6:
      return true;
   case 7:
      diagnostics_Diags_note(c->diags, c->loc, "SHOULD NOT HAPPEN (Array -> Ptr)");
      c2_assert((0) != 0, "analyser/conversion_checker.c2:578: conversion_checker.Checker.checkCast", "0");
      return false;
   case 8:
      diagnostics_Diags_error(c->diags, lhsLoc, "SHOULD NOT HAPPEN (%u - %u)\n", ast_Type_getKind(lcanon), ast_Type_getKind(rcanon));
      c2_assert((0) != 0, "analyser/conversion_checker.c2:582: conversion_checker.Checker.checkCast", "0");
      return false;
   case 9:
      diagnostics_Diags_error(c->diags, lhsLoc, "SHOULD NOT HAPPEN (%u - %u)\n", ast_Type_getKind(lcanon), ast_Type_getKind(rcanon));
      c2_assert((0) != 0, "analyser/conversion_checker.c2:586: conversion_checker.Checker.checkCast", "0");
      return false;
   case 10:
      return true;
   case 11:
      return true;
   case 12:
      return true;
   case 13:
      return true;
   case 14:
      return true;
   case 15:
      return conversion_checker_Checker_checkBuiltin2PointerCast(c, lcanon, rcanon);
   case 16: {
      bool other = false;
      bool ok = conversion_checker_Checker_checkFunc2Builtin(c, lcanon, rcanon, &other);
      if ((!ok && !other)) {
         ast_QualType valid = ast_getNativeType();
         diagnostics_Diags_error(c->diags, c->loc, "pointers may only be cast to integer type '%s'", ast_QualType_diagName(&valid));
      }
      return ok;
   }
   default:
      diagnostics_Diags_note(c->diags, c->loc, "TODO CONVERSION  %u)", res);
      return false;
   }
   return true;
}

static bool conversion_checker_Checker_checkBuiltin2PointerCast(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon)
{
   uint32_t wordsize = ast_getWordSize();
   const ast_BuiltinType* bi = ((ast_BuiltinType*)(rcanon));
   ast_BuiltinKind kind = ast_BuiltinType_getKind(bi);
   if ((kind == ast_BuiltinKind_USize)) return true;

   if (((wordsize == 4) && (kind == ast_BuiltinKind_UInt32))) return true;

   if (((wordsize == 8) && (kind == ast_BuiltinKind_UInt64))) return true;

   ast_QualType valid = ast_getNativeType();
   diagnostics_Diags_error(c->diags, c->loc, "only integers of type '%s' may be cast to a pointer", ast_QualType_diagName(&valid));
   return false;
}

static bool conversion_checker_Checker_checkPointer2BuiltinCast(conversion_checker_Checker* c, const ast_Type* lcanon, const ast_Type* rcanon)
{
   uint32_t wordsize = ast_getWordSize();
   const ast_BuiltinType* bi = ((ast_BuiltinType*)(lcanon));
   ast_BuiltinKind kind = ast_BuiltinType_getKind(bi);
   if ((kind == ast_BuiltinKind_USize)) return true;

   if (((wordsize == 4) && (kind == ast_BuiltinKind_UInt32))) return true;

   if (((wordsize == 8) && (kind == ast_BuiltinKind_UInt64))) return true;

   ast_QualType valid = ast_getNativeType();
   diagnostics_Diags_error(c->diags, c->loc, "pointers may only be cast to integer type '%s'", ast_QualType_diagName(&valid));
   return false;
}

static ast_QualType conversion_checker_get_common_arithmetic_type(ast_QualType t1, ast_QualType t2)
{
   c2_assert((ast_QualType_isBuiltin(&t1)) != 0, "analyser/conversion_checker.c2:686: conversion_checker.get_common_arithmetic_type", "CALL TODO");
   c2_assert((ast_QualType_isBuiltin(&t2)) != 0, "analyser/conversion_checker.c2:687: conversion_checker.get_common_arithmetic_type", "CALL TODO");
   ast_BuiltinType* bi1 = ast_QualType_getBuiltin(&t1);
   ast_BuiltinType* bi2 = ast_QualType_getBuiltin(&t2);
   ast_BuiltinKind kind = ((ast_BuiltinKind)(conversion_checker_ConditionalOperatorResult[ast_BuiltinType_getKind(bi2)][ast_BuiltinType_getKind(bi1)]));
   return ast_builtins[kind];
}

static ast_QualType conversion_checker_usual_arithmetic_conversion(const ast_BuiltinType* b1, const ast_BuiltinType* b2)
{
   ast_BuiltinKind k1 = ast_BuiltinType_getBaseKind(b1);
   ast_BuiltinKind k2 = ast_BuiltinType_getBaseKind(b2);
   switch (conversion_checker_UsualArithmeticConversions[k2][k1]) {
   case 0:
      return ast_builtins[ast_BuiltinKind_Int32];
   case 1:
      return ast_builtins[ast_BuiltinKind_UInt32];
   case 2:
      return ast_builtins[ast_BuiltinKind_Int64];
   case 3:
      return ast_builtins[ast_BuiltinKind_UInt64];
   case 4:
      return ast_builtins[ast_BuiltinKind_Float32];
   case 5:
      return ast_builtins[ast_BuiltinKind_Float64];
   case 6:
      break;
   }
   return ast_QualType_Invalid;
}

static conversion_checker_ExprWidth conversion_checker_ExprWidth_mergeSmaller(conversion_checker_ExprWidth* w1, conversion_checker_ExprWidth* w2)
{
   conversion_checker_ExprWidth result;
   if ((w2->width < w1->width)) result.width = w2->width;
   else result.width = w1->width;
   result.is_signed = (w1->is_signed || w2->is_signed);
   return result;
}

static conversion_checker_ExprWidth conversion_checker_ExprWidth_mergeWider(conversion_checker_ExprWidth* w1, conversion_checker_ExprWidth* w2)
{
   conversion_checker_ExprWidth result;
   if ((w2->width > w1->width)) result.width = w2->width;
   else result.width = w1->width;
   result.is_signed = (w1->is_signed || w2->is_signed);
   return result;
}

static uint8_t conversion_checker_val2width(uint64_t value)
{
   if ((value <= 65535)) {
      if ((value <= 255)) {
         if ((value <= 128)) return 7;
         else return 8;

      }
      if ((value <= 32767)) return 15;
      else return 16;

   }
   if ((value <= 4294967295)) {
      if ((value <= 2147483647)) return 31;
      else return 32;

   }
   if ((value <= 9223372036854775807lu)) return 63;

   return 64;
}

static conversion_checker_ExprWidth conversion_checker_getExprWidth(const ast_Expr* e)
{
   conversion_checker_ExprWidth result = { };
   if (ast_Expr_isCtv(e)) {
      ast_Value v = ctv_analyser_get_value(e);
      if (ast_Value_isNegative(&v)) {
         result.is_signed = true;
         result.width = conversion_checker_val2width(((uint64_t)(-v.svalue)));
      } else {
         result.width = conversion_checker_val2width(v.uvalue);
         result.is_signed = false;
      }
      return result;
   }
   switch (ast_Expr_getKind(e)) {
   case ast_ExprKind_IntegerLiteral:
      break;
   case ast_ExprKind_FloatLiteral:
      break;
   case ast_ExprKind_BooleanLiteral:
      break;
   case ast_ExprKind_CharLiteral:
      break;
   case ast_ExprKind_StringLiteral:
      break;
   case ast_ExprKind_Nil:
      break;
   case ast_ExprKind_Identifier: {
      ast_QualType qt = ast_Expr_getType(e);
      qt = ast_QualType_getCanonicalType(&qt);
      if (ast_QualType_isBuiltin(&qt)) {
         ast_BuiltinType* bi = ast_QualType_getBuiltin(&qt);
         result.width = ((uint8_t)(ast_BuiltinType_getWidth(bi)));
         result.is_signed = ast_BuiltinType_isSigned(bi);
      } else {
         result.width = ((uint8_t)((ast_getWordSize() * 8)));
         result.is_signed = true;
      }
      return result;
   }
   case ast_ExprKind_Type:
      break;
   case ast_ExprKind_Call:
      return conversion_checker_getTypeWidth(ast_Expr_getType(e));
   case ast_ExprKind_InitList:
      break;
   case ast_ExprKind_FieldDesignatedInit:
      break;
   case ast_ExprKind_ArrayDesignatedInit:
      break;
   case ast_ExprKind_BinaryOperator:
      return conversion_checker_getBinOpWidth(((ast_BinaryOperator*)(e)));
   case ast_ExprKind_UnaryOperator:
      return conversion_checker_getUnaryOpWidth(((ast_UnaryOperator*)(e)));
   case ast_ExprKind_ConditionalOperator:
      return conversion_checker_getCondOpWidth(((ast_ConditionalOperator*)(e)));
   case ast_ExprKind_Builtin:
      break;
   case ast_ExprKind_ArraySubscript:
      __attribute__((fallthrough));
   case ast_ExprKind_Member: {
      ast_QualType qt = ast_Expr_getType(e);
      qt = ast_QualType_getCanonicalType(&qt);
      c2_assert((ast_QualType_isBuiltin(&qt)) != 0, "analyser/conversion_checker_expr.c2:123: conversion_checker.getExprWidth", "CALL TODO");
      ast_BuiltinType* bi = ast_QualType_getBuiltin(&qt);
      result.width = ((uint8_t)(ast_BuiltinType_getWidth(bi)));
      result.is_signed = ast_BuiltinType_isSigned(bi);
      return result;
   }
   case ast_ExprKind_Paren: {
      const ast_ParenExpr* p = ((ast_ParenExpr*)(e));
      return conversion_checker_getExprWidth(ast_ParenExpr_getInner(p));
   }
   case ast_ExprKind_BitOffset:
      break;
   case ast_ExprKind_ExplicitCast: {
      ast_QualType qt = ast_Expr_getType(e);
      qt = ast_QualType_getCanonicalType(&qt);
      c2_assert((ast_QualType_isBuiltin(&qt)) != 0, "analyser/conversion_checker_expr.c2:136: conversion_checker.getExprWidth", "CALL TODO");
      ast_BuiltinType* bi = ast_QualType_getBuiltin(&qt);
      result.width = ((uint8_t)(ast_BuiltinType_getWidth(bi)));
      result.is_signed = ast_BuiltinType_isSigned(bi);
      return result;
   }
   case ast_ExprKind_ImplicitCast: {
      const ast_ImplicitCastExpr* c = ((ast_ImplicitCastExpr*)(e));
      return conversion_checker_getExprWidth(ast_ImplicitCastExpr_getInner(c));
   }
   }
   ast_Expr_dump(e);
   c2_assert((0) != 0, "analyser/conversion_checker_expr.c2:147: conversion_checker.getExprWidth", "0");
   return result;
}

static conversion_checker_ExprWidth conversion_checker_getCondOpWidth(const ast_ConditionalOperator* c)
{
   conversion_checker_ExprWidth lhs = conversion_checker_getExprWidth(ast_ConditionalOperator_getLHS(c));
   conversion_checker_ExprWidth rhs = conversion_checker_getExprWidth(ast_ConditionalOperator_getRHS(c));
   return conversion_checker_ExprWidth_mergeWider(&lhs, &rhs);
}

static conversion_checker_ExprWidth conversion_checker_getUnaryOpWidth(const ast_UnaryOperator* u)
{
   conversion_checker_ExprWidth w;
   switch (ast_UnaryOperator_getOpcode(u)) {
   case ast_UnaryOpcode_PostInc:
      __attribute__((fallthrough));
   case ast_UnaryOpcode_PostDec:
      __attribute__((fallthrough));
   case ast_UnaryOpcode_PreInc:
      __attribute__((fallthrough));
   case ast_UnaryOpcode_PreDec:
      return conversion_checker_getExprWidth(ast_UnaryOperator_getInner(u));
   case ast_UnaryOpcode_AddrOf:
      w.is_signed = false;
      break;
   case ast_UnaryOpcode_Deref:
      break;
   case ast_UnaryOpcode_Minus:
      w = conversion_checker_getExprWidth(ast_UnaryOperator_getInner(u));
      w.is_signed = true;
      break;
   case ast_UnaryOpcode_Not:
      w = conversion_checker_getExprWidth(ast_UnaryOperator_getInner(u));
      w.is_signed = false;
      break;
   case ast_UnaryOpcode_LNot:
      w.width = 1;
      w.is_signed = false;
      break;
   }
   return w;
}

static conversion_checker_ExprWidth conversion_checker_getBinOpWidth(const ast_BinaryOperator* b)
{
   switch (ast_BinaryOperator_getOpcode(b)) {
   case ast_BinaryOpcode_Multiply:
      break;
   case ast_BinaryOpcode_Divide:
      break;
   case ast_BinaryOpcode_Reminder:
      break;
   case ast_BinaryOpcode_Add:
      break;
   case ast_BinaryOpcode_Subtract:
      break;
   case ast_BinaryOpcode_ShiftLeft:
      break;
   case ast_BinaryOpcode_ShiftRight:
      break;
   case ast_BinaryOpcode_LessThan:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_GreaterThan:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_LessEqual:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_GreaterEqual:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_Equal:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_NotEqual: {
      conversion_checker_ExprWidth result = { 1, 0 };
      return result;
   }
   case ast_BinaryOpcode_And: {
      conversion_checker_ExprWidth l = conversion_checker_getExprWidth(ast_BinaryOperator_getLHS(b));
      conversion_checker_ExprWidth r = conversion_checker_getExprWidth(ast_BinaryOperator_getRHS(b));
      return conversion_checker_ExprWidth_mergeSmaller(&l, &r);
   }
   case ast_BinaryOpcode_Xor:
      break;
   case ast_BinaryOpcode_Or:
      break;
   case ast_BinaryOpcode_LAnd:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_LOr: {
      conversion_checker_ExprWidth result = { 1, 0 };
      return result;
   }
   case ast_BinaryOpcode_Assign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_MulAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_DivAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_RemAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_AddAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_SubAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_ShlAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_ShrAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_AndAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_XorAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_OrAssign:
      return conversion_checker_getExprWidth(ast_BinaryOperator_getLHS(b));
   }
   conversion_checker_ExprWidth lhs = conversion_checker_getExprWidth(ast_BinaryOperator_getLHS(b));
   conversion_checker_ExprWidth rhs = conversion_checker_getExprWidth(ast_BinaryOperator_getRHS(b));
   return conversion_checker_ExprWidth_mergeWider(&lhs, &rhs);
}

static conversion_checker_ExprWidth conversion_checker_getTypeWidth(ast_QualType qt)
{
   c2_assert((ast_QualType_isBuiltin(&qt)) != 0, "analyser/conversion_checker_expr.c2:249: conversion_checker.getTypeWidth", "CALL TODO");
   const ast_BuiltinType* bi = ast_QualType_getBuiltin(&qt);
   conversion_checker_ExprWidth result;
   result.width = ((uint8_t)(ast_BuiltinType_getWidth(bi)));
   result.is_signed = ast_BuiltinType_isSigned(bi);
   return result;
}


// --- module module_analyser ---
typedef struct module_analyser_Analyser_ module_analyser_Analyser;
typedef struct module_analyser_Label_ module_analyser_Label;
typedef struct module_analyser_LabelVector_ module_analyser_LabelVector;
typedef struct module_analyser_StackLayer_ module_analyser_StackLayer;
typedef struct module_analyser_MainMarker_ module_analyser_MainMarker;
typedef struct module_analyser_FormatAnalyser_ module_analyser_FormatAnalyser;

struct module_analyser_LabelVector_ {
   module_analyser_Label* labels;
   uint32_t count;
   uint32_t capacity;
};

struct module_analyser_StackLayer_ {
   ast_Decl* decl;
   scope_Scope* scope;
   ast_FunctionDecl* function;
   bool usedPublic;
};

#define module_analyser_MaxDepth 8
struct module_analyser_Analyser_ {
   diagnostics_Diags* diags;
   conversion_checker_Checker checker;
   ast_context_Context* context;
   string_pool_Pool* astPool;
   ast_builder_Builder* builder;
   module_list_List* allmodules;
   const warning_flags_Flags* warnings;
   ast_Module* mod;
   bool usedPublic;
   uint32_t prefix_cache_name;
   uint32_t prefix_cache_idx;
   name_vector_NameVector prefixes;
   struct_func_list_List* struct_decls;
   incr_array_list_List* incr_values;
   module_analyser_LabelVector labels;
   module_analyser_StackLayer checkStack[8];
   uint32_t checkIndex;
   scope_Scope* scope;
   ast_FunctionDecl* curFunction;
   bool has_error;
};

struct module_analyser_Label_ {
   uint32_t name_idx;
   src_loc_SrcLoc loc;
   bool used;
   bool is_label;
};

struct module_analyser_MainMarker_ {
   uint32_t name_idx;
   ast_Decl* main;
};

#define module_analyser_LHS 0x1
#define module_analyser_RHS 0x2
static module_analyser_Analyser* module_analyser_create(diagnostics_Diags* diags, ast_context_Context* context, string_pool_Pool* astPool, ast_builder_Builder* builder, module_list_List* allmodules, const warning_flags_Flags* warnings);
static void module_analyser_Analyser_free(module_analyser_Analyser* ma);
static void module_analyser_Analyser_check(module_analyser_Analyser* ma, ast_Module* mod);
static void module_analyser_Analyser_collectStructFunctions(module_analyser_Analyser* ma);
static void module_analyser_Analyser_handleArrayValue(void* arg, ast_ArrayValue* avd);
static void module_analyser_Analyser_collectIncrementalArrays(module_analyser_Analyser* ma);
static void module_analyser_Analyser_handleIncrEntry(module_analyser_Analyser* ma, incr_array_list_Info* entry);
static void module_analyser_Analyser_handleImport(void* arg, ast_ImportDecl* id);
static void module_analyser_Analyser_setMod(module_analyser_Analyser* ma, ast_Module* mod);
static void module_analyser_LabelVector_init(module_analyser_LabelVector* v, uint32_t capacity);
static void module_analyser_LabelVector_free(module_analyser_LabelVector* v);
static void module_analyser_LabelVector_reset(module_analyser_LabelVector* v);
static void module_analyser_LabelVector_resize(module_analyser_LabelVector* v);
static module_analyser_Label* module_analyser_LabelVector_add(module_analyser_LabelVector* v, uint32_t name_idx, src_loc_SrcLoc loc, bool is_label);
static uint32_t module_analyser_LabelVector_getCount(const module_analyser_LabelVector* v);
static const module_analyser_Label* module_analyser_LabelVector_getLabels(const module_analyser_LabelVector* v);
static module_analyser_Label* module_analyser_LabelVector_find(module_analyser_LabelVector* v, uint32_t name_idx);
__attribute__((__format__(printf, 3, 4))) 
static void module_analyser_Analyser_note(module_analyser_Analyser* ma, src_loc_SrcLoc loc, const char* format, ...);
__attribute__((__format__(printf, 3, 4))) 
static void module_analyser_Analyser_warn(module_analyser_Analyser* ma, src_loc_SrcLoc loc, const char* format, ...);
__attribute__((__format__(printf, 3, 4))) 
static void module_analyser_Analyser_error(module_analyser_Analyser* ma, src_loc_SrcLoc loc, const char* format, ...);
__attribute__((__format__(printf, 4, 5))) 
static void module_analyser_Analyser_errorRange(module_analyser_Analyser* ma, src_loc_SrcLoc loc, src_loc_SrcRange range, const char* format, ...);
static void module_analyser_Analyser_createGlobalScope(void* arg, ast_AST* a);
static void module_analyser_Analyser_deleteScope(void* _arg0, ast_AST* a);
static void module_analyser_Analyser_handleStructFunc(void* arg, ast_FunctionDecl* fd);
static void module_analyser_Analyser_analyseFunctionProto(void* arg, ast_FunctionDecl* d);
static void module_analyser_Analyser_analyseFunctionBodies(void* arg, ast_FunctionDecl* d);
static bool module_analyser_Analyser_analyseGlobalDecl(module_analyser_Analyser* ma, ast_Decl* d);
static void module_analyser_Analyser_handleTypeDecl(void* arg, ast_Decl* d);
static void module_analyser_Analyser_handleStaticAssert(void* arg, ast_StaticAssert* d);
static void module_analyser_Analyser_handleVarDecl(void* arg, ast_VarDecl* v);
static void module_analyser_Analyser_checkName(module_analyser_Analyser* ma, ast_Decl* d, bool is_constant);
static void module_analyser_Analyser_analyseGlobalVarDecl(module_analyser_Analyser* ma, ast_VarDecl* v);
static void module_analyser_Analyser_checkVarDeclAttributes(module_analyser_Analyser* ma, ast_VarDecl* v);
static bool module_analyser_Analyser_pushCheck(module_analyser_Analyser* ma, ast_Decl* d, scope_Scope* s, ast_FunctionDecl* fd);
static void module_analyser_Analyser_popCheck(module_analyser_Analyser* ma);
static bool module_analyser_Analyser_globalScope(const module_analyser_Analyser* ma);
static void module_analyser_findMainFunction(void* arg, ast_FunctionDecl* fd);
static ast_Decl* module_analyser_Analyser_findMain(module_analyser_Analyser* ma, ast_Module* top, uint32_t name_idx);
static const uint32_t module_analyser_Binop_lhs[29] = {
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_RHS,
   module_analyser_LHS,
   (module_analyser_LHS | module_analyser_RHS),
   (module_analyser_LHS | module_analyser_RHS),
   (module_analyser_LHS | module_analyser_RHS),
   (module_analyser_LHS | module_analyser_RHS),
   (module_analyser_LHS | module_analyser_RHS),
   (module_analyser_LHS | module_analyser_RHS),
   (module_analyser_LHS | module_analyser_RHS),
   (module_analyser_LHS | module_analyser_RHS),
   (module_analyser_LHS | module_analyser_RHS),
   (module_analyser_LHS | module_analyser_RHS)
};

static const uint8_t module_analyser_BinOpConvAddSubAss[8][8] = {
   {
   2,
   1,
   0,
   1,
   3,
   0,
   0,
   0
},
   {
   4,
   1,
   0,
   1,
   4,
   1,
   0,
   0
},
   {
   0,
   0,
   0,
   0,
   0,
   0,
   0,
   0
},
   {
   1,
   1,
   0,
   1,
   1,
   1,
   0,
   0
},
   {
   5,
   1,
   0,
   1,
   6,
   1,
   0,
   0
},
   {
   0,
   0,
   0,
   1,
   0,
   0,
   0,
   0
}
};

static const uint8_t module_analyser_BinOpConvAdd[8][8] = {
   {
   2,
   3,
   0,
   1,
   5,
   0,
   0,
   0
},
   {
   4,
   1,
   0,
   1,
   4,
   1,
   0,
   0
},
   {
   0,
   0,
   0,
   0,
   0,
   0,
   0,
   0
},
   {
   1,
   1,
   0,
   1,
   1,
   1,
   0,
   0
},
   {
   6,
   3,
   0,
   1,
   7,
   1,
   0,
   0
},
   {
   0,
   0,
   0,
   1,
   0,
   0,
   0,
   0
}
};

static const uint8_t module_analyser_BinOpConvSub[8][8] = {
   {
   2,
   1,
   0,
   1,
   2,
   0,
   0,
   0
},
   {
   4,
   5,
   0,
   1,
   4,
   6,
   0,
   0
},
   {
   0,
   0,
   0,
   0,
   0,
   0,
   0,
   0
},
   {
   1,
   1,
   0,
   1,
   1,
   1,
   0,
   0
},
   {
   6,
   1,
   0,
   1,
   7,
   1,
   0,
   0
},
   {
   0,
   0,
   0,
   1,
   0,
   0,
   0,
   0
}
};

static const uint8_t module_analyser_BinOpConvComparision[8][8] = {
   {
   2,
   1,
   0,
   1,
   4,
   0,
   0,
   0
},
   {
   1,
   3,
   0,
   1,
   1,
   7,
   0,
   0
},
   {
   0,
   0,
   0,
   0,
   0,
   0,
   0,
   0
},
   {
   1,
   1,
   0,
   1,
   1,
   1,
   0,
   0
},
   {
   5,
   1,
   0,
   1,
   6,
   1,
   0,
   0
},
   {
   1,
   8,
   1,
   1,
   1,
   9,
   0,
   0
}
};

static bool module_analyser_validBinOpKind(ast_QualType t);
static ast_QualType module_analyser_Analyser_checkBinopIntArgs(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs);
static ast_QualType module_analyser_Analyser_checkBinopLogical(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs);
static ast_QualType module_analyser_Analyser_checkBinopAddSubAssign(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs);
static ast_QualType module_analyser_Analyser_checkBinopAddArgs(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs);
static ast_QualType module_analyser_Analyser_checkBinopSubArgs(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs);
static ast_QualType module_analyser_Analyser_checkBinopComparison(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs);
static ast_QualType module_analyser_Analyser_checkPointerFuncComparison(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs, ast_QualType lcanon, ast_QualType rcanon);
static ast_QualType module_analyser_Analyser_analyseBinaryOperator(module_analyser_Analyser* ma, ast_Expr** e_ptr);
static bool module_analyser_Analyser_checkShiftArgs(module_analyser_Analyser* ma, ast_Expr* lhs, ast_Expr* rhs);
static bool module_analyser_Analyser_checkZero(module_analyser_Analyser* ma, ast_Expr* e, const char* operation);
static ast_QualType module_analyser_Analyser_analyseBuiltin(module_analyser_Analyser* ma, ast_Expr** e_ptr);
static ast_QualType module_analyser_Analyser_analyseSizeof(module_analyser_Analyser* ma, ast_BuiltinExpr* e);
static ast_QualType module_analyser_Analyser_analyseElemsof(module_analyser_Analyser* ma, ast_BuiltinExpr* b);
static ast_QualType module_analyser_Analyser_analyseEnumMinMax(module_analyser_Analyser* ma, ast_BuiltinExpr* b);
static ast_QualType module_analyser_Analyser_analyseOffsetOf(module_analyser_Analyser* ma, ast_BuiltinExpr* b);
static ast_QualType module_analyser_Analyser_analyseToContainer(module_analyser_Analyser* ma, ast_BuiltinExpr* b);
static ast_Decl* module_analyser_Analyser_findMemberOffset(module_analyser_Analyser* ma, ast_BuiltinExpr* b, ast_StructTypeDecl* std, ast_Expr* member);
static ast_Decl* module_analyser_Analyser_findStructMemberOffset(module_analyser_Analyser* ma, ast_StructTypeDecl* s, uint32_t name_idx, src_loc_SrcLoc loc, uint32_t* base);
struct module_analyser_FormatAnalyser_ {
   module_analyser_Analyser* ma;
   const char* format;
   uint32_t num_args;
   uint32_t idx;
   src_loc_SrcLoc loc;
   bool change_format;
   ast_Expr** args;
};

static ast_QualType module_analyser_Analyser_analyseCallExpr(module_analyser_Analyser* ma, ast_Expr** e_ptr);
static bool module_analyser_on_format_specifier(void* context, printf_utils_Specifier specifier, uint32_t offset, char letter);
static void module_analyser_Analyser_checkPrintArgs(module_analyser_Analyser* ma, ast_Expr* format, uint32_t num_args, ast_Expr** args, bool* change_format);
static void module_analyser_create_template_name(char* name, const char* orig, uint16_t idx);
static void module_analyser_Analyser_opaque_callback(void* arg, src_loc_SrcLoc loc, ast_Decl* d);
static ast_FunctionDecl* module_analyser_Analyser_instantiateTemplateFunction(module_analyser_Analyser* ma, ast_CallExpr* call, ast_FunctionDecl* fd);
static ast_QualType module_analyser_Analyser_analysePureCallExpr(module_analyser_Analyser* ma, ast_Expr* e);
static const uint8_t module_analyser_CondOpTable[8][8] = {
   {
   2,
   1,
   0,
   1,
   3,
   0,
   0,
   0
},
   {
   1,
   4,
   0,
   1,
   1,
   1,
   0,
   0
},
   {
   0,
   0,
   0,
   0,
   0,
   0,
   0,
   0
},
   {
   1,
   1,
   0,
   5,
   1,
   1,
   0,
   0
},
   {
   6,
   1,
   0,
   1,
   7,
   1,
   0,
   0
},
   {
   0,
   0,
   0,
   1,
   0,
   8,
   0,
   0
}
};

static ast_QualType module_analyser_Analyser_analyseExpr(module_analyser_Analyser* ma, ast_Expr** e_ptr, bool need_rvalue, uint32_t side);
static ast_QualType module_analyser_Analyser_analyseExprInner(module_analyser_Analyser* ma, ast_Expr** e_ptr, uint32_t side);
static ast_Decl* module_analyser_Analyser_analyseIdentifier(module_analyser_Analyser* ma, ast_Expr** e_ptr, uint32_t side);
static ast_IdentifierKind module_analyser_Analyser_setExprFlags(module_analyser_Analyser* ma, ast_Expr** e_ptr, ast_Decl* d);
static ast_QualType module_analyser_Analyser_analyseConditionalOperator(module_analyser_Analyser* ma, ast_Expr** e_ptr);
static bool module_analyser_Analyser_checkAssignment(module_analyser_Analyser* ma, ast_Expr* assignee, ast_QualType tleft, const char* msg, src_loc_SrcLoc loc);
static ast_QualType module_analyser_usualUnaryConversions(ast_Expr* e);
static ast_QualType module_analyser_Analyser_analyseExplicitCast(module_analyser_Analyser* ma, ast_Expr** e_ptr);
static ast_QualType module_analyser_Analyser_analyseArraySubscriptExpr(module_analyser_Analyser* ma, ast_Expr** e_ptr, uint32_t side);
static ast_QualType module_analyser_Analyser_analyseBitOffsetExpr(module_analyser_Analyser* ma, ast_QualType ltype, ast_Expr* base, ast_Expr* e);
static bool module_analyser_Analyser_analyseBitOffsetIndex(module_analyser_Analyser* ma, ast_Expr** e_ptr, ast_QualType baseType, ast_Value* result);
static void module_analyser_Analyser_memberError(module_analyser_Analyser* ma, uint32_t name_idx, src_loc_SrcLoc loc, ast_StructTypeDecl* s);
static ast_Decl* module_analyser_Analyser_findStructMember(module_analyser_Analyser* ma, ast_StructTypeDecl* s, uint32_t name_idx, src_loc_SrcLoc loc, bool allow_funcs);
static ast_QualType module_analyser_getPointerFromArray(ast_builder_Builder* builder, ast_QualType q);
static void module_analyser_Analyser_analyseFunction(module_analyser_Analyser* ma, ast_FunctionDecl* fd);
static void module_analyser_Analyser_analyseFunctionBody(module_analyser_Analyser* ma, ast_FunctionDecl* fd, scope_Scope* s);
static void module_analyser_Analyser_checkPrintfFormat(module_analyser_Analyser* ma, ast_VarDecl* v, ast_QualType qt, uint32_t idx, ast_FunctionDecl* fd);
static bool module_analyser_Analyser_analyseInitExpr(module_analyser_Analyser* ma, ast_Expr** e_ptr, ast_QualType expectedType, src_loc_SrcLoc assignLoc);
static bool module_analyser_Analyser_analyseInitListExpr(module_analyser_Analyser* ma, ast_InitListExpr* ile, ast_QualType expectedType);
static bool module_analyser_Analyser_analyseArrayDesignatedInit(module_analyser_Analyser* ma, ast_Expr* e, ast_QualType expectedType);
static bool module_analyser_Analyser_analyseInitListArray(module_analyser_Analyser* ma, ast_InitListExpr* ile, ast_QualType expectedType);
static bool module_analyser_Analyser_checkArrayDesignators(module_analyser_Analyser* ma, ast_InitListExpr* ile, int32_t* size, init_checker_Checker* checker);
static bool module_analyser_Analyser_analyseInitListStruct(module_analyser_Analyser* ma, ast_InitListExpr* ile, ast_QualType expectedType);
static bool module_analyser_Analyser_checkFieldDesignators(module_analyser_Analyser* ma, ast_InitListExpr* ile, init_checker_Checker* checker);
static ast_QualType module_analyser_Analyser_analyseMemberExpr(module_analyser_Analyser* ma, ast_Expr** e_ptr, uint32_t side);
static ast_Decl* module_analyser_Analyser_analyseStructMemberAccess(module_analyser_Analyser* ma, ast_StructTypeDecl* std, uint32_t name_idx, src_loc_SrcLoc loc, ast_ValType valtype, uint32_t side, ast_CallKind* ck);
static ast_TypeKind module_analyser_Analyser_analyseBaseType(module_analyser_Analyser* ma, ast_QualType baseType);
static ast_ValType module_analyser_decl2valtype(const ast_Decl* d);
static bool module_analyser_hasReturn(const ast_Stmt* s);
static void module_analyser_Analyser_analyseStmt(module_analyser_Analyser* ma, ast_Stmt* s, bool checkEffect);
static void module_analyser_Analyser_analyseBreakStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseContinueStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseFallthroughStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseLabelStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseGotoStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseCompoundStmt(module_analyser_Analyser* ma, ast_CompoundStmt* c);
static ast_QualType module_analyser_Analyser_analyseCondition(module_analyser_Analyser* ma, ast_Stmt** s_ptr, bool check_assign);
static void module_analyser_Analyser_analyseIfStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseForStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseWhileStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseDoStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static ast_QualType module_analyser_Analyser_analyseDeclStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseAsmStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseAssertStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseReturnStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static void module_analyser_Analyser_analyseStructType(module_analyser_Analyser* ma, ast_StructTypeDecl* d);
static void module_analyser_Analyser_analyseStructMembers(module_analyser_Analyser* ma, ast_StructTypeDecl* d);
static void module_analyser_Analyser_analyseStructMember(module_analyser_Analyser* ma, ast_VarDecl* v);
static void module_analyser_Analyser_analyseStructNames(module_analyser_Analyser* ma, ast_StructTypeDecl* d, name_vector_NameVector* names, name_vector_NameVector* locs);
static void module_analyser_Analyser_analyseSwitchStmt(module_analyser_Analyser* ma, ast_Stmt* s);
static bool module_analyser_Analyser_analyseCase(module_analyser_Analyser* ma, ast_SwitchCase* c, init_checker_Checker* checker, ast_EnumTypeDecl* etd, bool is_sswitch);
static bool module_analyser_Analyser_checkLastStmt(module_analyser_Analyser* ma, uint32_t count, ast_Stmt* last, src_loc_SrcLoc loc, bool is_default, bool is_sswitch);
static bool module_analyser_Analyser_analyseCaseCondition(module_analyser_Analyser* ma, ast_SwitchCase* c, init_checker_Checker* checker, ast_EnumTypeDecl* etd, bool is_sswitch);
static bool module_analyser_Analyser_checkEnumConstantCase(module_analyser_Analyser* ma, ast_IdentifierExpr* id, init_checker_Checker* checker, ast_EnumTypeDecl* etd, uint32_t* enum_idx);
static bool module_analyser_Analyser_analyseMultiCaseCondition(module_analyser_Analyser* ma, ast_SwitchCase* c, init_checker_Checker* checker, ast_EnumTypeDecl* etd);
static ast_Stmt* module_analyser_get_last_stmt(ast_Stmt* s);
static bool module_analyser_isTerminatingStmt(const ast_Stmt* s);
static void module_analyser_Analyser_analyseFunctionType(module_analyser_Analyser* ma, ast_Decl* d);
static void module_analyser_Analyser_analyseAliasType(module_analyser_Analyser* ma, ast_AliasTypeDecl* a);
static void module_analyser_Analyser_analyseEnumType(module_analyser_Analyser* ma, ast_EnumTypeDecl* d);
static ast_QualType module_analyser_Analyser_analyseUserTypeRef(module_analyser_Analyser* ma, ast_TypeRef* ref);
static ast_QualType module_analyser_Analyser_analyseTypeRef(module_analyser_Analyser* ma, ast_TypeRef* ref);
static ast_QualType module_analyser_Analyser_analyseIncrTypeRef(module_analyser_Analyser* ma, ast_TypeRef* ref, uint32_t size);
static bool module_analyser_Analyser_checkOpaque(module_analyser_Analyser* ma, const ast_StructTypeDecl* std, src_loc_SrcLoc loc);
static ast_VarDecl* module_analyser_getVarDecl(const ast_Expr* e);
static ast_QualType module_analyser_Analyser_analyseUnaryOperator(module_analyser_Analyser* ma, ast_Expr** e_ptr, uint32_t side);
static bool module_analyser_Analyser_checkIncrDecr(module_analyser_Analyser* ma, ast_Expr* inner, ast_QualType t, bool is_incr, src_loc_SrcLoc loc);
static bool module_analyser_Analyser_getIdentifierKind(module_analyser_Analyser* ma, ast_Expr* e);
static ast_IdentifierKind module_analyser_getInnerExprAddressOf(ast_Expr* e);
static ast_QualType module_analyser_getMinusType(ast_QualType qt);
static module_analyser_Analyser* module_analyser_create(diagnostics_Diags* diags, ast_context_Context* context, string_pool_Pool* astPool, ast_builder_Builder* builder, module_list_List* allmodules, const warning_flags_Flags* warnings)
{
   module_analyser_Analyser* ma = calloc(1, 456);
   ma->diags = diags;
   conversion_checker_Checker_init(&ma->checker, diags, builder);
   ma->context = context;
   ma->astPool = astPool;
   ma->builder = builder;
   ma->allmodules = allmodules;
   ma->warnings = warnings;
   return ma;
}

static void module_analyser_Analyser_free(module_analyser_Analyser* ma)
{
   name_vector_NameVector_free(&ma->prefixes);
   module_analyser_LabelVector_free(&ma->labels);
   free(ma);
}

static void module_analyser_Analyser_check(module_analyser_Analyser* ma, ast_Module* mod)
{
   ma->mod = mod;
   ma->prefix_cache_name = 0;
   ma->prefix_cache_idx = 0;
   name_vector_NameVector_clear(&ma->prefixes);
   module_analyser_LabelVector_reset(&ma->labels);
   ma->checkIndex = 0;
   ma->scope = NULL;
   ma->curFunction = NULL;
   ma->has_error = false;
   ma->usedPublic = false;
   ast_Module_visitASTs(mod, module_analyser_Analyser_createGlobalScope, ma);
   if (diagnostics_Diags_hasErrors(ma->diags)) return;

   module_analyser_Analyser_collectStructFunctions(ma);
   if (ma->has_error) return;

   module_analyser_Analyser_collectIncrementalArrays(ma);
   if (ma->has_error) return;

   ast_Module_visitTypeDecls(mod, module_analyser_Analyser_handleTypeDecl, ma);
   ast_Module_visitVarDecls(mod, module_analyser_Analyser_handleVarDecl, ma);
   ma->usedPublic = false;
   ast_Module_visitStaticAsserts(mod, module_analyser_Analyser_handleStaticAssert, ma);
   ast_Module_visitFunctions(mod, module_analyser_Analyser_analyseFunctionProto, ma);
   if (ma->has_error) return;

   ast_Module_visitFunctions(mod, module_analyser_Analyser_analyseFunctionBodies, ma);
   if ((!ast_Module_isExternal(mod) && ast_Module_isExported(mod))) {
      ast_Module_visitImports(mod, module_analyser_Analyser_handleImport, ma);
   }
   ast_Module_visitASTs(mod, module_analyser_Analyser_deleteScope, ma);
}

static void module_analyser_Analyser_collectStructFunctions(module_analyser_Analyser* ma)
{
   struct_func_list_List struct_decls = { };
   ma->struct_decls = &struct_decls;
   ast_Module_visitStructFunctions(ma->mod, module_analyser_Analyser_handleStructFunc, ma);
   if (ma->has_error) return;

   for (uint32_t i = 0; (i < struct_decls.count); i++) {
      const struct_func_list_Info* info = &struct_decls.data[i];
      ast_StructTypeDecl* fd = ((ast_StructTypeDecl*)(info->decl));
      ast_StructTypeDecl_setStructFunctions(fd, ma->context, ast_FunctionDeclList_getDecls(&info->functions), ast_FunctionDeclList_size(&info->functions));
   }
   name_vector_NameVector_free(&ma->prefixes);
   struct_func_list_List_free(&struct_decls);
   ma->struct_decls = NULL;
}

static void module_analyser_Analyser_handleArrayValue(void* arg, ast_ArrayValue* avd)
{
   module_analyser_Analyser* ma = arg;
   incr_array_list_List_add(ma->incr_values, ast_ArrayValue_getNameIdx(avd), ast_ArrayValue_getLoc(avd), ast_ArrayValue_getValue(avd));
}

static void module_analyser_Analyser_collectIncrementalArrays(module_analyser_Analyser* ma)
{
   incr_array_list_List ialist = { };
   ma->incr_values = &ialist;
   ast_Module_visitArrayValues(ma->mod, module_analyser_Analyser_handleArrayValue, ma);
   for (uint32_t i = 0; (i < ialist.count); i++) {
      module_analyser_Analyser_handleIncrEntry(ma, &ialist.entries[i]);
   }
   incr_array_list_List_free(&ialist);
   ma->incr_values = NULL;
}

static void module_analyser_Analyser_handleIncrEntry(module_analyser_Analyser* ma, incr_array_list_Info* entry)
{
   uint32_t name = entry->name;
   ast_Decl* d = ast_Module_findSymbol(ma->mod, name);
   if (!d) {
      module_analyser_Analyser_error(ma, entry->loc, "module '%s' has no symbol '%s'", ast_Module_getName(ma->mod), ast_idx2name(name));
      return;
   }
   if (ast_Decl_isVariable(d)) {
      ast_VarDecl* vd = ((ast_VarDecl*)(d));
      ast_TypeRef* ref = ast_VarDecl_getTypeRef(vd);
      if (!ast_TypeRef_isIncrArray(ref)) {
         module_analyser_Analyser_error(ma, entry->loc, "'%s' is not an incremental array", ast_idx2name(name));
         module_analyser_Analyser_note(ma, ast_Decl_getLoc(d), "'%s' is defined here", ast_idx2name(name));
         return;
      }
      uint32_t num_values = ast_ExprList_size(&entry->values);
      ast_Expr** values = ast_ExprList_getExprs(&entry->values);
      ast_Expr* init_expr = ast_builder_Builder_actOnInitList(ma->builder, 0, 0, values, num_values);
      ast_VarDecl_setInit(vd, init_expr);
   } else if (ast_Decl_isEnum(d)) {
      ast_EnumTypeDecl* etd = ((ast_EnumTypeDecl*)(d));
      if (!ast_EnumTypeDecl_isIncremental(etd)) {
         module_analyser_Analyser_error(ma, entry->loc, "'%s' is not an incremental enum", ast_idx2name(name));
         module_analyser_Analyser_note(ma, ast_Decl_getLoc(d), "'%s' is defined here", ast_idx2name(name));
         return;
      }
      uint32_t num_values = ast_ExprList_size(&entry->values);
      ast_Expr** values = ast_ExprList_getExprs(&entry->values);
      bool error = false;
      for (uint32_t j = 0; (j < num_values); j++) {
         ast_Expr* e = values[j];
         if (!ast_Expr_isIdentifier(e)) {
            module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "expected identifier");
            return;
         }
      }
      ast_EnumTypeDecl_setIncrConstants(etd, ma->context, ((ast_IdentifierExpr**)(values)), num_values);
   } else {
      module_analyser_Analyser_error(ma, entry->loc, "'%s' is not an incremental array/enum", ast_idx2name(name));
      module_analyser_Analyser_note(ma, ast_Decl_getLoc(d), "'%s' is defined here", ast_idx2name(name));
   }

}

static void module_analyser_Analyser_handleImport(void* arg, ast_ImportDecl* id)
{
   module_analyser_Analyser* ma = arg;
   ast_Decl* d = ((ast_Decl*)(id));
   if (!ast_Decl_isUsedPublic(d)) return;

   ast_Module* dest = ast_ImportDecl_getDest(id);
   if ((ast_Module_isExternal(dest) || ast_Module_isInternal(dest))) return;

   if (!ast_Module_isExported(dest)) {
      module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "exported module '%s' publicly uses non-exported module '%s'", ast_Module_getName(ma->mod), ast_Module_getName(dest));
   }
}

static void module_analyser_Analyser_setMod(module_analyser_Analyser* ma, ast_Module* mod)
{
   ma->mod = mod;
}

static void module_analyser_LabelVector_init(module_analyser_LabelVector* v, uint32_t capacity)
{
   v->labels = NULL;
   v->count = 0;
   v->capacity = (capacity / 2);
}

static void module_analyser_LabelVector_free(module_analyser_LabelVector* v)
{
   if (v->labels) free(v->labels);
   v->count = 0;
   v->capacity = 0;
   v->labels = NULL;
}

static void module_analyser_LabelVector_reset(module_analyser_LabelVector* v)
{
   v->count = 0;
}

static void module_analyser_LabelVector_resize(module_analyser_LabelVector* v)
{
   v->capacity = (v->capacity == 0) ? 4 : (v->capacity * 2);
   void* data2 = malloc((v->capacity * 12));
   if (v->labels) {
      memcpy(data2, v->labels, (v->count * 12));
      free(v->labels);
   }
   v->labels = data2;
}

static module_analyser_Label* module_analyser_LabelVector_add(module_analyser_LabelVector* v, uint32_t name_idx, src_loc_SrcLoc loc, bool is_label)
{
   module_analyser_Label* l = module_analyser_LabelVector_find(v, name_idx);
   if (!l) {
      if ((v->count == v->capacity)) module_analyser_LabelVector_resize(v);
      uint32_t index = v->count;
      l = &v->labels[index];
      l->name_idx = name_idx;
      l->loc = loc;
      l->is_label = is_label;
      l->used = !is_label;
      v->count++;
   }
   return l;
}

static uint32_t module_analyser_LabelVector_getCount(const module_analyser_LabelVector* v)
{
   return v->count;
}

static const module_analyser_Label* module_analyser_LabelVector_getLabels(const module_analyser_LabelVector* v)
{
   return v->labels;
}

static module_analyser_Label* module_analyser_LabelVector_find(module_analyser_LabelVector* v, uint32_t name_idx)
{
   for (uint32_t i = 0; (i < v->count); i++) {
      if ((v->labels[i].name_idx == name_idx)) return &v->labels[i];

   }
   return NULL;
}

__attribute__((__format__(printf, 3, 4))) 
static void module_analyser_Analyser_note(module_analyser_Analyser* ma, src_loc_SrcLoc loc, const char* format, ...)
{
   va_list args;
   va_start(args, format);
   diagnostics_Diags_note2(ma->diags, loc, format, args);
   va_end(args);
}

__attribute__((__format__(printf, 3, 4))) 
static void module_analyser_Analyser_warn(module_analyser_Analyser* ma, src_loc_SrcLoc loc, const char* format, ...)
{
   va_list args;
   va_start(args, format);
   diagnostics_Diags_warn2(ma->diags, loc, format, args);
   va_end(args);
}

__attribute__((__format__(printf, 3, 4))) 
static void module_analyser_Analyser_error(module_analyser_Analyser* ma, src_loc_SrcLoc loc, const char* format, ...)
{
   va_list args;
   va_start(args, format);
   diagnostics_Diags_error2(ma->diags, loc, format, args);
   va_end(args);
   ma->has_error = true;
}

__attribute__((__format__(printf, 4, 5))) 
static void module_analyser_Analyser_errorRange(module_analyser_Analyser* ma, src_loc_SrcLoc loc, src_loc_SrcRange range, const char* format, ...)
{
   va_list args;
   va_start(args, format);
   diagnostics_Diags_errorRange2(ma->diags, loc, range, format, args);
   va_end(args);
   ma->has_error = true;
}

static void module_analyser_Analyser_createGlobalScope(void* arg, ast_AST* a)
{
   module_analyser_Analyser* ma = arg;
   scope_Scope* s = scope_create(ma->allmodules, ma->diags, ast_AST_getImports(a), ma->mod, ast_Module_getSymbols(ma->mod), !ma->warnings->no_unused_variable);
   ast_AST_setPtr(a, s);
}

static void module_analyser_Analyser_deleteScope(void* _arg0, ast_AST* a)
{
   scope_Scope* s = ast_AST_getPtr(a);
   ast_AST_setPtr(a, NULL);
   scope_Scope_free(s);
}

static void module_analyser_Analyser_handleStructFunc(void* arg, ast_FunctionDecl* fd)
{
   module_analyser_Analyser* ma = arg;
   ast_Ref* prefix = ast_FunctionDecl_getPrefix(fd);
   ast_Decl* d = ((ast_Decl*)(fd));
   c2_assert((prefix) != NULL, "analyser/module_analyser.c2:387: module_analyser.Analyser.handleStructFunc", "prefix");
   uint32_t prefix_name_idx = prefix->name_idx;
   c2_assert((ma->struct_decls) != NULL, "analyser/module_analyser.c2:390: module_analyser.Analyser.handleStructFunc", "ma.struct_decls");
   uint32_t index = 0;
   if ((prefix_name_idx == ma->prefix_cache_name)) {
      index = ma->prefix_cache_idx;
   } else {
      bool found = false;
      found = name_vector_NameVector_find(&ma->prefixes, prefix_name_idx, &index);
      if (!found) {
         ast_Decl* decl = ast_Module_findType(ma->mod, prefix_name_idx);
         if (!decl) {
            decl = ast_Module_findSymbol(ma->mod, prefix_name_idx);
            if (decl) {
               module_analyser_Analyser_error(ma, prefix->loc, "a struct-function type must be a struct/union");
            } else {
               module_analyser_Analyser_error(ma, prefix->loc, "module '%s' has no symbol '%s'", ast_Module_getName(ma->mod), ast_Ref_getName(prefix));
            }
            return;
         }
         if (!ast_Decl_isStructType(decl)) {
            module_analyser_Analyser_error(ma, prefix->loc, "a struct-function type must be a struct/union");
            return;
         }
         if ((ast_Decl_isPublic(d) && !ast_Decl_isPublic(decl))) {
            module_analyser_Analyser_error(ma, prefix->loc, "public struct-functions need a public struct/union");
            return;
         }
         index = name_vector_NameVector_add(&ma->prefixes, prefix_name_idx);
         struct_func_list_List_addDecl(ma->struct_decls, decl);
      }
      ma->prefix_cache_name = prefix_name_idx;
      ma->prefix_cache_idx = index;
   }
   ast_FunctionDecl* other = struct_func_list_List_findFunc(ma->struct_decls, index, ast_Decl_getNameIdx(d));
   if (other) {
      module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "redefinition of '%s'", ast_Decl_getFullName(d));
      module_analyser_Analyser_note(ma, ast_Decl_getLoc(ast_FunctionDecl_asDecl(other)), "previous definition is here");
      return;
   }
   prefix->decl = struct_func_list_List_getDecl(ma->struct_decls, index);
   ast_StructTypeDecl* std = ((ast_StructTypeDecl*)(prefix->decl));
   ast_Decl* match = ast_StructTypeDecl_findMember(std, ast_Decl_getNameIdx(d), NULL);
   if (match) {
      module_analyser_Analyser_error(ma, ast_Decl_getLoc(match), "member '%s' conflicts with struct-function '%s'", ast_Decl_getName(match), ast_Decl_getFullName(d));
      module_analyser_Analyser_note(ma, ast_Decl_getLoc(d), "previous declaration is here");
      return;
   }
   struct_func_list_List_addFunc(ma->struct_decls, index, fd);
}

static void module_analyser_Analyser_analyseFunctionProto(void* arg, ast_FunctionDecl* d)
{
   module_analyser_Analyser* ma = arg;
   module_analyser_Analyser_analyseGlobalDecl(ma, ((ast_Decl*)(d)));
}

static void module_analyser_Analyser_analyseFunctionBodies(void* arg, ast_FunctionDecl* d)
{
   module_analyser_Analyser* ma = arg;
   module_analyser_Analyser_analyseFunctionBody(ma, d, ast_AST_getPtr(ast_Decl_getAST(ast_FunctionDecl_asDecl(d))));
}

static bool module_analyser_Analyser_analyseGlobalDecl(module_analyser_Analyser* ma, ast_Decl* d)
{
   if (ast_Decl_isChecked(d)) return true;

   if (!module_analyser_Analyser_pushCheck(ma, d, ast_AST_getPtr(ast_Decl_getAST(d)), NULL)) return false;

   if ((ast_Decl_isExported(d) && !ast_Decl_isPublic(d))) {
      module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "attribute 'export' can only be applied to public declarations");
      return false;
   }
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_Function:
      module_analyser_Analyser_analyseFunction(ma, ((ast_FunctionDecl*)(d)));
      break;
   case ast_DeclKind_Import:
      break;
   case ast_DeclKind_StructType:
      module_analyser_Analyser_analyseStructType(ma, ((ast_StructTypeDecl*)(d)));
      break;
   case ast_DeclKind_EnumType:
      module_analyser_Analyser_analyseEnumType(ma, ((ast_EnumTypeDecl*)(d)));
      break;
   case ast_DeclKind_EnumConstant:
      c2_assert((0) != 0, "analyser/module_analyser.c2:483: module_analyser.Analyser.analyseGlobalDecl", "0");
      break;
   case ast_DeclKind_FunctionType:
      module_analyser_Analyser_analyseFunctionType(ma, d);
      break;
   case ast_DeclKind_AliasType:
      module_analyser_Analyser_analyseAliasType(ma, ((ast_AliasTypeDecl*)(d)));
      break;
   case ast_DeclKind_Variable:
      module_analyser_Analyser_analyseGlobalVarDecl(ma, ((ast_VarDecl*)(d)));
      break;
   }
   ast_Decl_setChecked(d);
   module_analyser_Analyser_popCheck(ma);
   return true;
}

static void module_analyser_Analyser_handleTypeDecl(void* arg, ast_Decl* d)
{
   module_analyser_Analyser* ma = arg;
   module_analyser_Analyser_analyseGlobalDecl(ma, d);
}

static void module_analyser_Analyser_handleStaticAssert(void* arg, ast_StaticAssert* d)
{
   module_analyser_Analyser* ma = arg;
   ma->scope = ast_AST_getPtr(ast_StaticAssert_getAST(d));
   ast_StaticAssert* sa = ((ast_StaticAssert*)(d));
   ast_Expr* lhs = ast_StaticAssert_getLhs(sa);
   ast_Expr* rhs = ast_StaticAssert_getRhs(sa);
   ast_QualType t1 = module_analyser_Analyser_analyseExpr(ma, &lhs, false, module_analyser_RHS);
   ast_QualType t2 = module_analyser_Analyser_analyseExpr(ma, &rhs, false, module_analyser_RHS);
   if ((ast_QualType_isInvalid(&t1) || ast_QualType_isInvalid(&t2))) return;

   bool error = false;
   if (!ast_Expr_isCtv(lhs)) {
      module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(lhs), ast_Expr_getRange(lhs), "static_assert element is not a compile-time value");
      error = true;
   }
   if (!ast_Expr_isCtv(rhs)) {
      module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(rhs), ast_Expr_getRange(rhs), "static_assert element is not a compile-time value");
      error = true;
   }
   if (error) return;

   ast_Value val1 = ctv_analyser_get_value(lhs);
   ast_Value val2 = ctv_analyser_get_value(rhs);
   if (!ast_Value_equals(&val1, &val2)) {
      module_analyser_Analyser_errorRange(ma, ast_Expr_getStartLoc(rhs), ast_Expr_getRange(rhs), "static_assert failed, expected %s, got %s", ast_Value_str(&val1), ast_Value_str(&val2));
   }
}

static void module_analyser_Analyser_handleVarDecl(void* arg, ast_VarDecl* v)
{
   module_analyser_Analyser* ma = arg;
   module_analyser_Analyser_analyseGlobalDecl(ma, ast_VarDecl_asDecl(v));
}

static void module_analyser_Analyser_checkName(module_analyser_Analyser* ma, ast_Decl* d, bool is_constant)
{
   const char* name = ast_Decl_getName(d);
   if (is_constant) {
      if (islower(name[0])) {
         module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "a global constant name must start with an upper case character");
      }
   } else {
      if (isupper(name[0])) {
         module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "a variable name must start with a lower case character");
      }
   }
}

static void module_analyser_Analyser_analyseGlobalVarDecl(module_analyser_Analyser* ma, ast_VarDecl* v)
{
   ast_Decl* d = ((ast_Decl*)(v));
   ast_TypeRef* ref = ast_VarDecl_getTypeRef(v);
   ast_Expr* init_expr = ast_VarDecl_getInit(v);
   ast_QualType res;
   if (ast_TypeRef_isIncrArray(ref)) {
      uint32_t size = 0;
      if (init_expr) {
         ast_InitListExpr* ile = ((ast_InitListExpr*)(init_expr));
         size = ast_InitListExpr_getNumValues(ile);
      }
      res = module_analyser_Analyser_analyseIncrTypeRef(ma, ref, size);
   } else {
      res = module_analyser_Analyser_analyseTypeRef(ma, ref);
   }
   if (ast_QualType_isInvalid(&res)) return;

   ast_QualType canon = ast_QualType_getCanonicalType(&res);
   if (ast_QualType_isVoid(&canon)) {
      module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(ref), "%s has invalid type 'void'", ast_QualType_isConst(&res) ? "constant" : "variable");
      return;
   }
   if ((ast_QualType_isArray(&canon) && !ast_TypeRef_isIncrArray(ref))) {
      const ast_ArrayType* at = ast_QualType_getArrayType(&canon);
      if (ast_ArrayType_hasSize(at)) {
         if ((ast_ArrayType_getSize(at) == 0)) {
            module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(ref), "only struct members may have array size zero");
            return;
         }
      } else {
         if (!init_expr) {
            module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "array-type variable '%s' needs an explicit size or an initializer", ast_Decl_getFullName(d));
            return;
         }
      }
   }
   ast_Decl_setType(d, res);
   module_analyser_Analyser_checkName(ma, d, ast_QualType_isConstant(&res));
   if (ast_Decl_isPublic(d)) ast_setTypePublicUsed(res);
   if (init_expr) {
      if (!ast_QualType_isConstant(&res)) {
         ma->checkStack[(ma->checkIndex - 1)].usedPublic = false;
         ma->usedPublic = false;
      }
      module_analyser_Analyser_analyseInitExpr(ma, ast_VarDecl_getInit2(v), res, ast_VarDecl_getAssignLoc(v));
   } else {
      if (ast_QualType_isConstant(&res)) {
         module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "constant variable '%s' must be initialized", ast_Decl_getFullName(d));
      }
   }
   module_analyser_Analyser_checkVarDeclAttributes(ma, v);
}

static void module_analyser_Analyser_checkVarDeclAttributes(module_analyser_Analyser* ma, ast_VarDecl* v)
{
   ast_Decl* d = ((ast_Decl*)(v));
   ast_QualType qt = ast_Decl_getType(d);
   if ((ast_QualType_isConst(&qt) && ast_QualType_isBuiltin(&qt))) {
      if (ast_VarDecl_hasAttrWeak(v)) {
         module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "attribute 'weak' cannot be applied to constants");
      }
      if (!ast_Decl_hasAttr(d)) return;

      const ast_AST* a = ast_Decl_getAST(d);
      const attr_Value* section = ast_AST_getAttr(a, d, attr_AttrKind_Section);
      if (section) {
         module_analyser_Analyser_error(ma, section->loc, "attribute 'section' cannot be applied to constants");
      }
      const attr_Value* aligned = ast_AST_getAttr(a, d, attr_AttrKind_Aligned);
      if (aligned) {
         module_analyser_Analyser_error(ma, aligned->loc, "attribute 'aligned' cannot be applied to constants");
      }
   }
}

static bool module_analyser_Analyser_pushCheck(module_analyser_Analyser* ma, ast_Decl* d, scope_Scope* s, ast_FunctionDecl* fd)
{
   for (uint32_t i = 0; (i < ma->checkIndex); i++) {
      if ((ma->checkStack[i].decl == d)) {
         for (uint32_t j = i; (j < ma->checkIndex); j++) {
            module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "circular declaration dependency '%s'", ast_Decl_getName(d));
         }
         return false;
      }
   }
   ma->scope = s;
   module_analyser_StackLayer* top = &ma->checkStack[ma->checkIndex];
   top->decl = d;
   top->scope = ma->scope;
   top->function = fd;
   top->usedPublic = ast_Decl_isPublic(d);
   if (fd) ma->curFunction = fd;
   ma->usedPublic = top->usedPublic;
   ma->checkIndex++;
   if (!ast_Decl_isChecked(d)) ast_Decl_setCheckState(d, ast_DeclCheckState_InProgress);
   c2_assert(((ma->checkIndex <= module_analyser_MaxDepth)) != 0, "analyser/module_analyser.c2:664: module_analyser.Analyser.pushCheck", "ma.checkIndex<=MaxDepth");
   return true;
}

static void module_analyser_Analyser_popCheck(module_analyser_Analyser* ma)
{
   c2_assert(((ma->checkIndex > 0)) != 0, "analyser/module_analyser.c2:669: module_analyser.Analyser.popCheck", "ma.checkIndex>0");
   ma->checkIndex--;
   if ((ma->checkIndex > 0)) {
      module_analyser_StackLayer* top = &ma->checkStack[(ma->checkIndex - 1)];
      ma->scope = top->scope;
      ma->curFunction = top->function;
      ma->usedPublic = top->usedPublic;
   } else {
      ma->scope = NULL;
      ma->curFunction = NULL;
   }
}

static bool module_analyser_Analyser_globalScope(const module_analyser_Analyser* ma)
{
   return (ma->curFunction == NULL);
}

static void module_analyser_findMainFunction(void* arg, ast_FunctionDecl* fd)
{
   module_analyser_MainMarker* m = (arg);
   ast_Decl* d = ((ast_Decl*)(fd));
   if ((ast_Decl_getNameIdx(d) == m->name_idx)) {
      if (m->main) {
         return;
      }
      m->main = d;
   }
}

static ast_Decl* module_analyser_Analyser_findMain(module_analyser_Analyser* ma, ast_Module* top, uint32_t name_idx)
{
   ast_Module_setUsed(top);
   module_analyser_MainMarker marker = { name_idx, NULL };
   ast_Module_visitFunctions(top, module_analyser_findMainFunction, &marker);
   if (marker.main) {
      ast_Decl_setUsed(marker.main);
      ast_Decl_setUsedPublic(marker.main);
      ast_Decl_setAttrExport(marker.main);
   }
   return marker.main;
}

static bool module_analyser_validBinOpKind(ast_QualType t)
{
   t = ast_QualType_getCanonicalType(&t);
   switch (ast_QualType_getKind(&t)) {
   case ast_TypeKind_Builtin:
      return true;
   case ast_TypeKind_Pointer:
      return true;
   case ast_TypeKind_Array:
      break;
   case ast_TypeKind_Struct:
      return true;
   case ast_TypeKind_Enum:
      return true;
   case ast_TypeKind_Function:
      return true;
   case ast_TypeKind_Alias:
      c2_assert((0) != 0, "analyser/module_analyser_binop.c2:39: module_analyser.validBinOpKind", "0");
      break;
   case ast_TypeKind_Module:
      break;
   }
   return false;
}

static ast_QualType module_analyser_Analyser_checkBinopIntArgs(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs)
{
   ast_QualType lcanon = ast_QualType_getCanonicalType(&lhs);
   ast_QualType rcanon = ast_QualType_getCanonicalType(&rhs);
   c2_assert((ast_QualType_isValid(&lcanon)) != 0, "analyser/module_analyser_binop.c2:129: module_analyser.Analyser.checkBinopIntArgs", "CALL TODO");
   c2_assert((ast_QualType_isValid(&rcanon)) != 0, "analyser/module_analyser_binop.c2:130: module_analyser.Analyser.checkBinopIntArgs", "CALL TODO");
   if (ast_QualType_isEnum(&lcanon)) {
      ast_EnumType* et = ast_QualType_getEnum(&lcanon);
      ast_EnumTypeDecl* etd = ast_EnumType_getDecl(et);
      lcanon = ast_EnumTypeDecl_getImplType(etd);
   }
   if (ast_QualType_isEnum(&rcanon)) {
      ast_EnumType* et = ast_QualType_getEnum(&rcanon);
      ast_EnumTypeDecl* etd = ast_EnumType_getDecl(et);
      rcanon = ast_EnumTypeDecl_getImplType(etd);
   }
   ast_BuiltinType* bl = ast_QualType_getBuiltinTypeOrNil(&lcanon);
   ast_BuiltinType* br = ast_QualType_getBuiltinTypeOrNil(&rcanon);
   if ((((!bl || !br) || ast_QualType_isVoid(&lcanon)) || ast_QualType_isVoid(&lcanon))) {
      ast_Expr* e = ((ast_Expr*)(b));
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "invalid operands to binary expression ('%s' and '%s')", ast_QualType_diagName(&lhs), ast_QualType_diagName(&rhs));
      return ast_QualType_Invalid;
   }
   ast_QualType optype = conversion_checker_usual_arithmetic_conversion(bl, br);
   ast_BuiltinType* bi = ast_QualType_getBuiltin(&optype);
   if ((bl != bi)) {
      ast_builder_Builder_insertImplicitCast(ma->builder, ast_ImplicitCastKind_IntegralCast, ast_BinaryOperator_getLHS2(b), optype);
   }
   if ((br != bi)) {
      ast_builder_Builder_insertImplicitCast(ma->builder, ast_ImplicitCastKind_IntegralCast, ast_BinaryOperator_getRHS2(b), optype);
   }
   return optype;
}

static ast_QualType module_analyser_Analyser_checkBinopLogical(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs)
{
   bool ok = conversion_checker_Checker_check(&ma->checker, ast_builtins[ast_BuiltinKind_Bool], lhs, ast_BinaryOperator_getLHS2(b), ast_Expr_getLoc(ast_BinaryOperator_getLHS(b)));
   ok &= conversion_checker_Checker_check(&ma->checker, ast_builtins[ast_BuiltinKind_Bool], rhs, ast_BinaryOperator_getRHS2(b), ast_Expr_getLoc(ast_BinaryOperator_getRHS(b)));
   if (ok) return ast_builtins[ast_BuiltinKind_Bool];

   ast_Expr* e = ((ast_Expr*)(b));
   module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "invalid operands to binary expression ('%s' and '%s')", ast_QualType_diagName(&lhs), ast_QualType_diagName(&rhs));
   return ast_QualType_Invalid;
}

static ast_QualType module_analyser_Analyser_checkBinopAddSubAssign(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs)
{
   ast_QualType lcanon = ast_QualType_getCanonicalType(&lhs);
   ast_QualType rcanon = ast_QualType_getCanonicalType(&rhs);
   c2_assert((ast_QualType_isValid(&lcanon)) != 0, "analyser/module_analyser_binop.c2:201: module_analyser.Analyser.checkBinopAddSubAssign", "CALL TODO");
   c2_assert((ast_QualType_isValid(&rcanon)) != 0, "analyser/module_analyser_binop.c2:202: module_analyser.Analyser.checkBinopAddSubAssign", "CALL TODO");
   uint8_t res = module_analyser_BinOpConvAddSubAss[ast_QualType_getKind(&lcanon)][ast_QualType_getKind(&rcanon)];
   switch (res) {
   case 0:
      break;
   case 1: {
      ast_Expr* e = ((ast_Expr*)(b));
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "invalid operands to binary expression ('%s' and '%s')", ast_QualType_diagName(&lhs), ast_QualType_diagName(&rhs));
      return ast_QualType_Invalid;
   }
   case 2:
      return lhs;
   case 3:
      return lhs;
   case 4:
      return lhs;
   case 5:
      return lhs;
   case 6:
      return lhs;
   }
   c2_assert((0) != 0, "analyser/module_analyser_binop.c2:227: module_analyser.Analyser.checkBinopAddSubAssign", "0");
   return ast_QualType_Invalid;
}

static ast_QualType module_analyser_Analyser_checkBinopAddArgs(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs)
{
   ast_QualType lcanon = ast_QualType_getCanonicalType(&lhs);
   ast_QualType rcanon = ast_QualType_getCanonicalType(&rhs);
   c2_assert((ast_QualType_isValid(&lcanon)) != 0, "analyser/module_analyser_binop.c2:261: module_analyser.Analyser.checkBinopAddArgs", "CALL TODO");
   c2_assert((ast_QualType_isValid(&rcanon)) != 0, "analyser/module_analyser_binop.c2:262: module_analyser.Analyser.checkBinopAddArgs", "CALL TODO");
   uint8_t res = module_analyser_BinOpConvAdd[ast_QualType_getKind(&lcanon)][ast_QualType_getKind(&rcanon)];
   switch (res) {
   case 0:
      break;
   case 1: {
      ast_Expr* e = ((ast_Expr*)(b));
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "invalid operands to binary expression ('%s' and '%s')", ast_QualType_diagName(&lhs), ast_QualType_diagName(&rhs));
      return ast_QualType_Invalid;
   }
   case 2: {
      ast_BuiltinType* bl = ast_QualType_getBuiltin(&lcanon);
      ast_BuiltinType* br = ast_QualType_getBuiltin(&rcanon);
      ast_QualType optype = conversion_checker_usual_arithmetic_conversion(bl, br);
      ast_BuiltinType* bi = ast_QualType_getBuiltin(&optype);
      if ((bl != bi)) {
         ast_builder_Builder_insertImplicitCast(ma->builder, ast_ImplicitCastKind_IntegralCast, ast_BinaryOperator_getLHS2(b), optype);
      }
      if ((br != bi)) {
         ast_builder_Builder_insertImplicitCast(ma->builder, ast_ImplicitCastKind_IntegralCast, ast_BinaryOperator_getRHS2(b), optype);
      }
      return optype;
   }
   case 3:
      return rhs;
   case 4:
      return lhs;
   case 5:
      return rhs;
   case 6:
      return lhs;
   case 7:
      return ast_builtins[ast_BuiltinKind_UInt32];
   }
   c2_assert((0) != 0, "analyser/module_analyser_binop.c2:301: module_analyser.Analyser.checkBinopAddArgs", "0");
   return ast_QualType_Invalid;
}

static ast_QualType module_analyser_Analyser_checkBinopSubArgs(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs)
{
   ast_QualType lcanon = ast_QualType_getCanonicalType(&lhs);
   ast_QualType rcanon = ast_QualType_getCanonicalType(&rhs);
   c2_assert((ast_QualType_isValid(&lcanon)) != 0, "analyser/module_analyser_binop.c2:332: module_analyser.Analyser.checkBinopSubArgs", "CALL TODO");
   c2_assert((ast_QualType_isValid(&rcanon)) != 0, "analyser/module_analyser_binop.c2:333: module_analyser.Analyser.checkBinopSubArgs", "CALL TODO");
   uint8_t res = module_analyser_BinOpConvSub[ast_QualType_getKind(&lcanon)][ast_QualType_getKind(&rcanon)];
   switch (res) {
   case 0:
      break;
   case 1: {
      ast_Expr* e = ((ast_Expr*)(b));
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "invalid operands to binary expression ('%s' and '%s')", ast_QualType_diagName(&lhs), ast_QualType_diagName(&rhs));
      return ast_QualType_Invalid;
   }
   case 2: {
      ast_BuiltinType* bl = ast_QualType_getBuiltin(&lcanon);
      ast_BuiltinType* br = ast_QualType_getBuiltin(&rcanon);
      return conversion_checker_usual_arithmetic_conversion(bl, br);
   }
   case 3: {
      ast_BuiltinType* bl = ast_QualType_getBuiltin(&lcanon);
      ast_BuiltinType* br = ast_QualType_getBuiltin(&rcanon);
      return conversion_checker_usual_arithmetic_conversion(bl, br);
   }
   case 4:
      return lhs;
   case 5: {
      ast_PointerType* pt1 = ast_QualType_getPointerType(&lcanon);
      ast_PointerType* pt2 = ast_QualType_getPointerType(&rcanon);
      ast_QualType t1 = ast_PointerType_getInner(pt1);
      ast_QualType t2 = ast_PointerType_getInner(pt2);
      if ((((ast_QualType_getTypeOrNil(&t1) != ast_QualType_getTypeOrNil(&t2)) || ast_QualType_isVoid(&t1)) || ast_QualType_isVoid(&t2))) {
         ast_Expr* e = ((ast_Expr*)(b));
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "invalid operands to binary expression ('%s' and '%s')", ast_QualType_diagName(&lhs), ast_QualType_diagName(&rhs));
         return ast_QualType_Invalid;
      }
      return ast_builtins[ast_BuiltinKind_ISize];
   }
   case 6:
      return lhs;
   case 7:
      return ast_builtins[ast_BuiltinKind_Int32];
   }
   c2_assert((0) != 0, "analyser/module_analyser_binop.c2:373: module_analyser.Analyser.checkBinopSubArgs", "0");
   return ast_QualType_Invalid;
}

static ast_QualType module_analyser_Analyser_checkBinopComparison(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs)
{
   ast_QualType lcanon = ast_QualType_getCanonicalType(&lhs);
   ast_QualType rcanon = ast_QualType_getCanonicalType(&rhs);
   ast_Expr* e = ((ast_Expr*)(b));
   c2_assert((ast_QualType_isValid(&lcanon)) != 0, "analyser/module_analyser_binop.c2:408: module_analyser.Analyser.checkBinopComparison", "CALL TODO");
   c2_assert((ast_QualType_isValid(&rcanon)) != 0, "analyser/module_analyser_binop.c2:409: module_analyser.Analyser.checkBinopComparison", "CALL TODO");
   uint8_t res = module_analyser_BinOpConvComparision[ast_QualType_getKind(&lcanon)][ast_QualType_getKind(&rcanon)];
   switch (res) {
   case 0:
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "invalid operands to binary expression ('%s' and '%s')", ast_QualType_diagName(&lhs), ast_QualType_diagName(&rhs));
      return ast_QualType_Invalid;
   case 1:
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "invalid operands to binary expression ('%s' and '%s')", ast_QualType_diagName(&lhs), ast_QualType_diagName(&rhs));
      return ast_QualType_Invalid;
   case 2:
      return ast_builtins[ast_BuiltinKind_Bool];
   case 3:
      return ast_builtins[ast_BuiltinKind_Bool];
   case 4:
      return ast_builtins[ast_BuiltinKind_Bool];
   case 5:
      return ast_builtins[ast_BuiltinKind_Bool];
   case 6:
      if ((ast_QualType_getTypeOrNil(&lcanon) != ast_QualType_getTypeOrNil(&rcanon))) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "comparing enums of different types ('%s' and '%s')", ast_QualType_diagName(&lhs), ast_QualType_diagName(&rhs));
         return ast_QualType_Invalid;
      }
      return ast_builtins[ast_BuiltinKind_Bool];
   case 7:
      return module_analyser_Analyser_checkPointerFuncComparison(ma, b, lhs, rhs, lcanon, rcanon);
   case 8:
      return module_analyser_Analyser_checkPointerFuncComparison(ma, b, lhs, rhs, rcanon, lcanon);
   case 9:
      break;
   }
   module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "TODO BINOP %u", res);
   c2_assert((0) != 0, "analyser/module_analyser_binop.c2:447: module_analyser.Analyser.checkBinopComparison", "0");
   return ast_QualType_Invalid;
}

static ast_QualType module_analyser_Analyser_checkPointerFuncComparison(module_analyser_Analyser* ma, ast_BinaryOperator* b, ast_QualType lhs, ast_QualType rhs, ast_QualType lcanon, ast_QualType rcanon)
{
   c2_assert((ast_QualType_isPointer(&lcanon)) != 0, "analyser/module_analyser_binop.c2:455: module_analyser.Analyser.checkPointerFuncComparison", "CALL TODO");
   ast_PointerType* pt = ast_QualType_getPointerType(&lcanon);
   ast_QualType inner = ast_PointerType_getInner(pt);
   ast_Expr* e = ((ast_Expr*)(b));
   if (!ast_QualType_isVoid(&inner)) {
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "invalid operands to binary expression ('%s' and '%s')", ast_QualType_diagName(&lhs), ast_QualType_diagName(&rhs));
      return ast_QualType_Invalid;
   }
   c2_assert((ast_QualType_isFunction(&rcanon)) != 0, "analyser/module_analyser_binop.c2:465: module_analyser.Analyser.checkPointerFuncComparison", "CALL TODO");
   ast_FunctionType* ft = ast_QualType_getFunctionType(&rcanon);
   ast_FunctionDecl* fd = ast_FunctionType_getDecl(ft);
   if ((ast_FunctionDecl_isType(fd) || ast_FunctionDecl_hasAttrWeak(fd))) return ast_builtins[ast_BuiltinKind_Bool];

   module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "comparison of function '%s' will always be true", ast_Decl_getFullName(ast_FunctionDecl_asDecl(fd)));
   return ast_QualType_Invalid;
}

static ast_QualType module_analyser_Analyser_analyseBinaryOperator(module_analyser_Analyser* ma, ast_Expr** e_ptr)
{
   ast_Expr* e = *e_ptr;
   ast_BinaryOperator* b = ((ast_BinaryOperator*)(e));
   bool need_lhs_rvalue = true;
   if ((ast_BinaryOperator_getOpcode(b) >= ast_BinaryOpcode_Assign)) need_lhs_rvalue = false;
   ast_QualType ltype = module_analyser_Analyser_analyseExpr(ma, ast_BinaryOperator_getLHS2(b), need_lhs_rvalue, module_analyser_Binop_lhs[ast_BinaryOperator_getOpcode(b)]);
   if (ast_QualType_isInvalid(&ltype)) return ast_QualType_Invalid;

   ast_QualType rtype = module_analyser_Analyser_analyseExpr(ma, ast_BinaryOperator_getRHS2(b), true, module_analyser_RHS);
   if (ast_QualType_isInvalid(&rtype)) return ast_QualType_Invalid;

   ast_Expr* lhs = ast_BinaryOperator_getLHS(b);
   ast_Expr* rhs = ast_BinaryOperator_getRHS(b);
   if (!need_lhs_rvalue) {
      if (!module_analyser_Analyser_checkAssignment(ma, lhs, ltype, "left operand of assignment", ast_Expr_getLoc(e))) {
         return ast_QualType_Invalid;
      }
   }
   if ((!module_analyser_validBinOpKind(ltype) || ast_QualType_isVoid(&ltype))) {
      ast_QualType tl = ast_Expr_getType(lhs);
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(lhs), "invalid operand to binary expression '%s'", ast_QualType_diagName(&tl));
      return ast_QualType_Invalid;
   }
   if ((!module_analyser_validBinOpKind(rtype) || ast_QualType_isVoid(&rtype))) {
      ast_QualType tr = ast_Expr_getType(rhs);
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(rhs), "invalid operand to binary expression '%s'", ast_QualType_diagName(&tr));
      return ast_QualType_Invalid;
   }
   ast_QualType result = ast_QualType_Invalid;
   switch (ast_BinaryOperator_getOpcode(b)) {
   case ast_BinaryOpcode_Multiply:
      result = module_analyser_Analyser_checkBinopIntArgs(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_Divide:
      if (!module_analyser_Analyser_checkZero(ma, rhs, "division")) return ast_QualType_Invalid;

      result = module_analyser_Analyser_checkBinopIntArgs(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_Reminder:
      if (!module_analyser_Analyser_checkZero(ma, rhs, "remainder")) return ast_QualType_Invalid;

      result = module_analyser_Analyser_checkBinopIntArgs(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_Add:
      result = module_analyser_Analyser_checkBinopAddArgs(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_Subtract:
      result = module_analyser_Analyser_checkBinopSubArgs(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_ShiftLeft:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_ShiftRight:
      result = module_analyser_Analyser_checkBinopIntArgs(ma, b, ltype, rtype);
      if (!module_analyser_Analyser_checkShiftArgs(ma, ast_BinaryOperator_getLHS(b), ast_BinaryOperator_getRHS(b))) return ast_QualType_Invalid;

      break;
   case ast_BinaryOpcode_LessThan:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_GreaterThan:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_LessEqual:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_GreaterEqual:
      result = module_analyser_Analyser_checkBinopComparison(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_Equal:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_NotEqual:
      result = module_analyser_Analyser_checkBinopComparison(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_And:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_Xor:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_Or:
      result = module_analyser_Analyser_checkBinopIntArgs(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_LAnd:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_LOr:
      result = module_analyser_Analyser_checkBinopLogical(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_Assign: {
      bool ok = conversion_checker_Checker_check(&ma->checker, ltype, rtype, ast_BinaryOperator_getRHS2(b), ast_Expr_getLoc(e));
      if (ok) result = ltype;
      break;
   }
   case ast_BinaryOpcode_MulAssign:
      result = module_analyser_Analyser_checkBinopIntArgs(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_DivAssign:
      if (!module_analyser_Analyser_checkZero(ma, rhs, "division")) return ast_QualType_Invalid;

      result = module_analyser_Analyser_checkBinopIntArgs(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_RemAssign:
      if (!module_analyser_Analyser_checkZero(ma, rhs, "remainder")) return ast_QualType_Invalid;

      result = module_analyser_Analyser_checkBinopIntArgs(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_AddAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_SubAssign:
      result = module_analyser_Analyser_checkBinopAddSubAssign(ma, b, ltype, rtype);
      break;
   case ast_BinaryOpcode_ShlAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_ShrAssign:
      if (!module_analyser_Analyser_checkShiftArgs(ma, lhs, rhs)) return ast_QualType_Invalid;

      __attribute__((fallthrough));
   case ast_BinaryOpcode_AndAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_XorAssign:
      __attribute__((fallthrough));
   case ast_BinaryOpcode_OrAssign:
      result = module_analyser_Analyser_checkBinopIntArgs(ma, b, ltype, rtype);
      break;
   }
   ast_Expr_combineConstantFlags(e, lhs, rhs);
   return result;
}

static bool module_analyser_Analyser_checkShiftArgs(module_analyser_Analyser* ma, ast_Expr* lhs, ast_Expr* rhs)
{
   ast_QualType qt = ast_Expr_getType(lhs);
   ast_QualType canon = ast_QualType_getCanonicalType(&qt);
   if (!ast_QualType_isBuiltin(&canon)) {
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(lhs), "cannot shift .. TODO");
      return false;
   }
   ast_BuiltinType* bi = ast_QualType_getBuiltinTypeOrNil(&canon);
   uint32_t width = ast_BuiltinType_getWidth(bi);
   bool is_signed = ast_BuiltinType_isSigned(bi);
   width += ast_BuiltinType_isSigned(bi);
   if (ast_Expr_isCtv(lhs)) {
      ast_Value val = ctv_analyser_get_value(lhs);
      if (ast_Value_isNegative(&val)) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(lhs), "shifting a negative signed value is undefined");
         return false;
      }
   }
   if (ast_Expr_isCtv(rhs)) {
      ast_Value val = ctv_analyser_get_value(rhs);
      if (ast_Value_isNegative(&val)) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(rhs), "shift count is negative");
         return false;
      }
      if ((val.uvalue >= width)) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(rhs), "shift count >= width of type");
         return false;
      }
   }
   return true;
}

static bool module_analyser_Analyser_checkZero(module_analyser_Analyser* ma, ast_Expr* e, const char* operation)
{
   if (!ast_Expr_isCtv(e)) return true;

   ast_Value val = ctv_analyser_get_value(e);
   if ((val.uvalue == 0)) {
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "%s by zero is undefined", operation);
      return false;
   }
   return true;
}

static ast_QualType module_analyser_Analyser_analyseBuiltin(module_analyser_Analyser* ma, ast_Expr** e_ptr)
{
   ast_Expr* e = *e_ptr;
   ast_BuiltinExpr* b = ((ast_BuiltinExpr*)(e));
   switch (ast_BuiltinExpr_getKind(b)) {
   case ast_BuiltinExprKind_Sizeof:
      return module_analyser_Analyser_analyseSizeof(ma, b);
   case ast_BuiltinExprKind_Elemsof:
      return module_analyser_Analyser_analyseElemsof(ma, b);
   case ast_BuiltinExprKind_EnumMin:
      __attribute__((fallthrough));
   case ast_BuiltinExprKind_EnumMax:
      return module_analyser_Analyser_analyseEnumMinMax(ma, b);
   case ast_BuiltinExprKind_OffsetOf:
      return module_analyser_Analyser_analyseOffsetOf(ma, b);
   case ast_BuiltinExprKind_ToContainer:
      return module_analyser_Analyser_analyseToContainer(ma, b);
   }
   return ast_QualType_Invalid;
}

static ast_QualType module_analyser_Analyser_analyseSizeof(module_analyser_Analyser* ma, ast_BuiltinExpr* e)
{
   ast_Expr* inner = ast_BuiltinExpr_getInner(e);
   c2_assert((inner) != NULL, "analyser/module_analyser_builtin.c2:45: module_analyser.Analyser.analyseSizeof", "inner");
   bool savedPublic = ma->usedPublic;
   ma->usedPublic = false;
   ast_QualType qt;
   if (ast_Expr_isType(inner)) {
      ast_TypeExpr* te = ((ast_TypeExpr*)(inner));
      ast_TypeRef* ref = ast_TypeExpr_getTypeRef(te);
      qt = module_analyser_Analyser_analyseTypeRef(ma, ref);
      ast_Expr_setType(inner, qt);
   } else {
      qt = module_analyser_Analyser_analyseExpr(ma, &inner, false, 0);
      if (ast_QualType_isInvalid(&qt)) return ast_QualType_Invalid;

      if (ast_QualType_isStruct(&qt)) {
         ast_StructType* st = ast_QualType_getStructType(&qt);
         const ast_StructTypeDecl* std = ast_StructType_getDecl(st);
         if (ast_StructTypeDecl_isOpaque(std)) {
            const ast_Decl* d = ((ast_Decl*)(std));
            bool is_external = (ma->mod != ast_Decl_getModule(d));
            if (is_external) {
               module_analyser_Analyser_error(ma, ast_Expr_getLoc(inner), "opaque type '%s' used by value", ast_QualType_diagName(&qt));
            }
         }
      }
   }
   ma->usedPublic = savedPublic;
   if (ast_QualType_isInvalid(&qt)) return ast_QualType_Invalid;

   size_analyser_TypeSize info = size_analyser_sizeOfType(qt);
   ast_BuiltinExpr_setUValue(e, info.size);
   return ast_builtins[ast_BuiltinKind_UInt32];
}

static ast_QualType module_analyser_Analyser_analyseElemsof(module_analyser_Analyser* ma, ast_BuiltinExpr* b)
{
   ast_Expr* inner = ast_BuiltinExpr_getInner(b);
   bool savedPublic = ma->usedPublic;
   ma->usedPublic = false;
   ast_QualType qt = module_analyser_Analyser_analyseExpr(ma, &inner, false, module_analyser_RHS);
   ma->usedPublic = savedPublic;
   if (ast_QualType_isInvalid(&qt)) return qt;

   const ast_ArrayType* at = ast_QualType_getArrayTypeOrNil(&qt);
   if (at) {
      ast_BuiltinExpr_setUValue(b, ast_ArrayType_getSize(at));
      return ast_builtins[ast_BuiltinKind_UInt32];
   }
   const ast_EnumType* et = ast_QualType_getEnumTypeOrNil(&qt);
   if (et) {
      const ast_EnumTypeDecl* etd = ast_EnumType_getDecl(et);
      ast_BuiltinExpr_setUValue(b, ast_EnumTypeDecl_getNumConstants(etd));
      return ast_builtins[ast_BuiltinKind_UInt32];
   }
   module_analyser_Analyser_error(ma, ast_Expr_getLoc(inner), "elemsof can only be used on arrays/enums");
   return ast_QualType_Invalid;
}

static ast_QualType module_analyser_Analyser_analyseEnumMinMax(module_analyser_Analyser* ma, ast_BuiltinExpr* b)
{
   ast_Expr* inner = ast_BuiltinExpr_getInner(b);
   ast_QualType qt = module_analyser_Analyser_analyseExpr(ma, &inner, false, module_analyser_RHS);
   if (ast_QualType_isInvalid(&qt)) return ast_QualType_Invalid;

   ast_EnumType* et = ast_QualType_getEnumTypeOrNil(&qt);
   if (!et) {
      const char* kind = ((ast_BuiltinExpr_getKind(b) == ast_BuiltinExprKind_EnumMin)) ? "enum_min" : "enum_max";
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(inner), "%s can only be used on enum types", kind);
      return ast_QualType_Invalid;
   }
   ast_EnumTypeDecl* etd = ast_EnumType_getDecl(et);
   uint32_t num = ast_EnumTypeDecl_getNumConstants(etd);
   ast_EnumConstantDecl** constants = ast_EnumTypeDecl_getConstants(etd);
   uint32_t index = 0;
   if ((ast_BuiltinExpr_getKind(b) == ast_BuiltinExprKind_EnumMax)) index = (num - 1);
   ast_BuiltinExpr_setValue(b, ast_EnumConstantDecl_getValue(constants[index]));
   return ast_EnumTypeDecl_getImplType(etd);
}

static ast_QualType module_analyser_Analyser_analyseOffsetOf(module_analyser_Analyser* ma, ast_BuiltinExpr* b)
{
   ast_Expr* e = ((ast_Expr*)(b));
   ast_Expr* inner = ast_BuiltinExpr_getInner(b);
   ast_QualType qt = module_analyser_Analyser_analyseExpr(ma, &inner, false, module_analyser_RHS);
   if (ast_QualType_isInvalid(&qt)) return ast_QualType_Invalid;

   ast_Expr_setType(e, ast_builtins[ast_BuiltinKind_UInt32]);
   ast_StructType* st = ast_QualType_getStructTypeOrNil(&qt);
   if (!st) {
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(inner), "offsetof can only be used on struct types");
      return ast_QualType_Invalid;
   }
   ast_StructTypeDecl* std = ast_StructType_getDecl(st);
   if (!module_analyser_Analyser_checkOpaque(ma, std, ast_Expr_getLoc(inner))) return ast_QualType_Invalid;

   ast_Expr* member = ast_BuiltinExpr_getOffsetOfMember(b);
   ast_Decl* d = module_analyser_Analyser_findMemberOffset(ma, b, std, member);
   if (!d) return ast_QualType_Invalid;

   return ast_Expr_getType(e);
}

static ast_QualType module_analyser_Analyser_analyseToContainer(module_analyser_Analyser* ma, ast_BuiltinExpr* b)
{
   ast_Expr* inner = ast_BuiltinExpr_getInner(b);
   ast_QualType qt = module_analyser_Analyser_analyseExpr(ma, &inner, false, module_analyser_RHS);
   if (ast_QualType_isInvalid(&qt)) return ast_QualType_Invalid;

   ast_StructType* st = ast_QualType_getStructTypeOrNil(&qt);
   if (!st) {
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(inner), "to_container can only be used on struct types");
      return ast_QualType_Invalid;
   }
   ast_StructTypeDecl* std = ast_StructType_getDecl(st);
   if (!module_analyser_Analyser_checkOpaque(ma, std, ast_Expr_getLoc(inner))) return ast_QualType_Invalid;

   ast_Expr* member = ast_BuiltinExpr_getToContainerMember(b);
   ast_Decl* d = module_analyser_Analyser_findMemberOffset(ma, b, std, member);
   if (!d) return ast_QualType_Invalid;

   ast_QualType qmem = ast_Decl_getType(d);
   qmem = ast_builder_Builder_actOnPointerType(ma->builder, qmem);
   ast_QualType qptr = module_analyser_Analyser_analyseExpr(ma, ast_BuiltinExpr_getToContainerPointer2(b), false, module_analyser_RHS);
   if (ast_QualType_isInvalid(&qptr)) return ast_QualType_Invalid;

   ast_Expr* eptr = ast_BuiltinExpr_getToContainerPointer(b);
   if (!conversion_checker_Checker_check(&ma->checker, qptr, qmem, ast_BuiltinExpr_getToContainerPointer2(b), ast_Expr_getLoc(eptr))) {
      return ast_QualType_Invalid;
   }
   if (ast_QualType_isConstPtr(&qptr)) ast_QualType_setConst(&qt);
   return ast_builder_Builder_actOnPointerType(ma->builder, qt);
}

static ast_Decl* module_analyser_Analyser_findMemberOffset(module_analyser_Analyser* ma, ast_BuiltinExpr* b, ast_StructTypeDecl* std, ast_Expr* member)
{
   uint32_t base_offset = 0;
   ast_Decl* d = NULL;
   if (ast_Expr_isIdentifier(member)) {
      ast_IdentifierExpr* i = ((ast_IdentifierExpr*)(member));
      uint32_t name_idx = ast_IdentifierExpr_getNameIdx(i);
      d = module_analyser_Analyser_findStructMemberOffset(ma, std, name_idx, ast_Expr_getLoc(member), &base_offset);
      if (!d) return NULL;

      ast_IdentifierExpr_setDecl(i, d);
      ast_Decl_setUsed(d);
      ast_Expr_setLValue(member);
      ast_IdentifierExpr_setKind(i, ast_IdentifierKind_StructMember);
   } else {
      c2_assert((ast_Expr_isMember(member)) != 0, "analyser/module_analyser_builtin.c2:207: module_analyser.Analyser.findMemberOffset", "CALL TODO");
      ast_MemberExpr* m = ((ast_MemberExpr*)(member));
      for (uint32_t i = 0; (i < ast_MemberExpr_getNumRefs(m)); i++) {
         uint32_t name_idx = ast_MemberExpr_getNameIdx(m, i);
         src_loc_SrcLoc loc = ast_MemberExpr_getLoc(m, i);
         d = module_analyser_Analyser_findStructMemberOffset(ma, std, name_idx, loc, &base_offset);
         if (!d) return NULL;

         if (ast_Decl_isStructType(d)) std = ((ast_StructTypeDecl*)(d));
         ast_Decl_setUsed(d);
         ast_MemberExpr_setDecl(m, d, i);
      }
      ast_MemberExpr_setKind(m, ast_IdentifierKind_StructMember);
   }
   ast_Expr_setType(member, ast_Decl_getType(d));
   ast_BuiltinExpr_setUValue(b, base_offset);
   return d;
}

static ast_Decl* module_analyser_Analyser_findStructMemberOffset(module_analyser_Analyser* ma, ast_StructTypeDecl* s, uint32_t name_idx, src_loc_SrcLoc loc, uint32_t* base)
{
   ast_Decl* d = ast_StructTypeDecl_findMember(s, name_idx, base);
   if (!d) module_analyser_Analyser_memberError(ma, name_idx, loc, s);
   return d;
}

static ast_QualType module_analyser_Analyser_analyseCallExpr(module_analyser_Analyser* ma, ast_Expr** e_ptr)
{
   ast_Expr* e = *e_ptr;
   ast_CallExpr* call = ((ast_CallExpr*)(e));
   ast_Expr** func = ast_CallExpr_getFunc2(call);
   ast_Expr* origFn = ast_CallExpr_getFunc(call);
   ast_QualType qt = module_analyser_Analyser_analyseExpr(ma, func, true, module_analyser_RHS);
   if (ast_QualType_isInvalid(&qt)) return ast_QualType_Invalid;

   if (ast_Expr_isNValue(origFn)) {
      module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(origFn), ast_Expr_getRange(origFn), "called object is not a function of function pointer");
      return ast_QualType_Invalid;
   }
   ast_FunctionType* ft = ast_QualType_getFunctionTypeOrNil(&qt);
   if (!ft) {
      ast_Expr* fn2 = ast_CallExpr_getFunc(call);
      module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(fn2), ast_Expr_getRange(fn2), "called object type %s is not a function or function pointer", ast_QualType_diagName(&qt));
      return ast_QualType_Invalid;
   }
   ast_FunctionDecl* fd = ast_FunctionType_getDecl(ft);
   ast_Decl_setUsed(ast_FunctionDecl_asDecl(fd));
   if (ast_FunctionDecl_isTemplate(fd)) {
      if (!ast_CallExpr_getTemplateArg(call)) {
         module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(e), ast_Expr_getRange(e), "function %s requires a template argument", ast_Decl_getFullName(ast_FunctionDecl_asDecl(fd)));
         return ast_QualType_Invalid;
      }
      fd = module_analyser_Analyser_instantiateTemplateFunction(ma, call, fd);
      if (!fd) return ast_QualType_Invalid;

   } else {
      if (ast_CallExpr_getTemplateArg(call)) {
         module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(e), ast_Expr_getRange(e), "function %s is not a template function", ast_Decl_getFullName(ast_FunctionDecl_asDecl(fd)));
         return ast_QualType_Invalid;
      }
   }
   uint32_t func_num_args = ast_FunctionDecl_getNumParams(fd);
   uint32_t call_num_args = ast_CallExpr_getNumArgs(call);
   bool isStructFuncCall = false;
   src_loc_SrcLoc loc = ast_Expr_getLoc(e);
   ast_QualType baseType;
   if (ast_Expr_isMember(origFn)) {
      const ast_MemberExpr* m = ((ast_MemberExpr*)(origFn));
      if (ast_MemberExpr_isStaticStructFunc(m)) {
         ast_CallExpr_setCallsStaticStructFunc(call);
      }
      if (ast_MemberExpr_isStructFunc(m)) {
         isStructFuncCall = true;
         ast_CallExpr_setCallsStructFunc(call);
         baseType = ast_MemberExpr_getBaseType(m);
         loc = ast_MemberExpr_getEndLoc(m);
      }
   }
   uint32_t num_auto_args = ast_FunctionDecl_getNumAutoArgs(fd);
   if (num_auto_args) ast_CallExpr_setHasAutoArgs(call);
   ast_VarDecl** func_args = ast_FunctionDecl_getParams(fd);
   ast_Expr** call_args = ast_CallExpr_getArgs(call);
   uint32_t func_arg_index = 0;
   uint32_t call_arg_index = 0;
   if (isStructFuncCall) {
      c2_assert(((ast_FunctionDecl_getNumParams(fd) >= 1)) != 0, "analyser/module_analyser_call.c2:99: module_analyser.Analyser.analyseCallExpr", "CALL TODO>=1");
      ast_VarDecl* arg0 = func_args[0];
      ast_QualType expectedType = ast_Decl_getType(ast_VarDecl_asDecl(arg0));
      if (!ast_QualType_isPointer(&baseType)) {
         baseType = ast_builder_Builder_actOnPointerType(ma->builder, baseType);
      }
      bool ok = conversion_checker_Checker_check(&ma->checker, expectedType, baseType, e_ptr, loc);
      if (!ok) return ast_QualType_Invalid;

      func_arg_index++;
   }
   bool has_printf_format = false;
   uint32_t printf_call_idx;
   while (1) {
      if ((func_arg_index >= func_num_args)) break;

      if ((call_arg_index >= call_num_args)) break;

      ast_VarDecl* vd = func_args[func_arg_index];
      if (ast_VarDecl_hasAutoAttr(vd)) {
         func_arg_index++;
         continue;
      }
      ast_QualType callType = module_analyser_Analyser_analyseExpr(ma, &call_args[call_arg_index], true, module_analyser_RHS);
      if (ast_QualType_isInvalid(&callType)) return ast_QualType_Invalid;

      ast_Expr* call_arg = call_args[call_arg_index];
      bool ok = conversion_checker_Checker_check(&ma->checker, ast_Decl_getType(ast_VarDecl_asDecl(vd)), callType, &call_args[call_arg_index], ast_Expr_getLoc(call_arg));
      if (!ok) return ast_QualType_Invalid;

      if (ast_VarDecl_hasPrintfFormat(vd)) {
         has_printf_format = true;
         printf_call_idx = call_arg_index;
      }
      func_arg_index++;
      call_arg_index++;
   }
   uint32_t expected_args = ((func_num_args - num_auto_args) - isStructFuncCall);
   if ((call_num_args < expected_args)) {
      module_analyser_Analyser_error(ma, ast_CallExpr_getEndLoc(call), "too few arguments to %sfunction call, expected %u, have %u", ast_FunctionDecl_getDiagKind(fd), expected_args, call_num_args);
      module_analyser_Analyser_note(ma, ast_Decl_getLoc(ast_FunctionDecl_asDecl(fd)), "'%s' declared here", ast_Decl_getFullName(ast_FunctionDecl_asDecl(fd)));
      return ast_QualType_Invalid;
   }
   if (((call_arg_index != call_num_args) || ast_FunctionDecl_isVariadic(fd))) {
      if (!ast_FunctionDecl_isVariadic(fd)) {
         ast_Expr* call_arg = call_args[call_arg_index];
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(call_arg), "too many arguments to %sfunction call, expected %u, have %u", ast_FunctionDecl_getDiagKind(fd), expected_args, call_num_args);
         module_analyser_Analyser_note(ma, ast_Decl_getLoc(ast_FunctionDecl_asDecl(fd)), "'%s' declared here", ast_Decl_getFullName(ast_FunctionDecl_asDecl(fd)));
         return ast_QualType_Invalid;
      }
      while ((call_arg_index != call_num_args)) {
         ast_QualType callType = module_analyser_Analyser_analyseExpr(ma, &call_args[call_arg_index], true, module_analyser_RHS);
         if (ast_QualType_isInvalid(&callType)) return ast_QualType_Invalid;

         if (ast_QualType_isVoid(&callType)) {
            ast_Expr* call_arg = call_args[call_arg_index];
            module_analyser_Analyser_error(ma, ast_Expr_getLoc(call_arg), "passing 'void' as variadic argument is invalid");
            return ast_QualType_Invalid;
         }
         call_arg_index++;
      }
      if (has_printf_format) {
         uint32_t num_args = ((call_num_args - printf_call_idx) - 1);
         bool change_format = false;
         module_analyser_Analyser_checkPrintArgs(ma, call_args[printf_call_idx], num_args, &call_args[(printf_call_idx + 1)], &change_format);
         ast_CallExpr_setPrintfFormat(call, printf_call_idx, change_format);
      }
   }
   return ast_FunctionDecl_getRType(fd);
}

static bool module_analyser_on_format_specifier(void* context, printf_utils_Specifier specifier, uint32_t offset, char letter)
{
   module_analyser_FormatAnalyser* fa = context;
   module_analyser_Analyser* ma = fa->ma;
   ast_Expr** args = fa->args;
   uint32_t idx = fa->idx;
   if ((idx >= fa->num_args)) {
      module_analyser_Analyser_error(ma, (fa->loc + offset), "too many format specifiers or not enough arguments");
      return false;
   }
   ast_Expr* arg = args[idx];
   ast_QualType qt = ast_Expr_getType(arg);
   qt = ast_QualType_getCanonicalType(&qt);
   switch (specifier) {
   case printf_utils_Specifier_Other:
      break;
   case printf_utils_Specifier_String:
      if (!ast_QualType_isCharPointer(&qt)) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(arg), "format '%%s' expects a string argument");
      }
      break;
   case printf_utils_Specifier_Char:
      if (((!ast_QualType_isChar(&qt) && !ast_QualType_isInt8(&qt)) && !ast_QualType_isUInt8(&qt))) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(arg), "format '%%c' expects a character argument");
      }
      break;
   case printf_utils_Specifier_Integer: {
      fa->change_format = true;
      if (ast_QualType_isEnum(&qt)) {
         ast_EnumType* et = ast_QualType_getEnumType(&qt);
         qt = ast_EnumType_getImplType(et);
      }
      ast_BuiltinType* bi = ast_QualType_getBuiltinTypeOrNil(&qt);
      if ((!bi || !ast_BuiltinType_isIntegerOrBool(bi))) {
         char c = fa->format[offset];
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(arg), "format '%%%c' expects an integer argument", c);
      }
      break;
   }
   case printf_utils_Specifier_FloatingPoint: {
      fa->change_format = true;
      ast_BuiltinType* bi = ast_QualType_getBuiltinTypeOrNil(&qt);
      if ((!bi || !ast_BuiltinType_isFloatingPoint(bi))) {
         char c = fa->format[offset];
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(arg), "format '%%%c' expects a floating-point argument", c);
      }
      break;
   }
   case printf_utils_Specifier_Pointer:
      if ((!ast_QualType_isPointer(&qt) && !ast_QualType_isFunction(&qt))) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(arg), "format '%%p' expects a pointer argument");
      }
      break;
   case printf_utils_Specifier_Invalid: {
      char c = fa->format[offset];
      switch (c) {
      case 'i':
         __attribute__((fallthrough));
      case 'l':
         __attribute__((fallthrough));
      case 'u':
         module_analyser_Analyser_error(ma, (fa->loc + offset), "invalid format specifier '%%%c', did you mean '%%d'?", c);
         break;
      default:
         module_analyser_Analyser_error(ma, (fa->loc + offset), "invalid format specifier '%%%c'", c);
         break;
      }
      return false;
   }
   }
   fa->idx++;
   return true;
}

static void module_analyser_Analyser_checkPrintArgs(module_analyser_Analyser* ma, ast_Expr* format, uint32_t num_args, ast_Expr** args, bool* change_format)
{
   if (!ast_Expr_isImplicitCast(format)) return;

   ast_ImplicitCastExpr* ic = ((ast_ImplicitCastExpr*)(format));
   if (!ast_ImplicitCastExpr_isArrayToPointerDecay(ic)) return;

   format = ast_ImplicitCastExpr_getInner(ic);
   src_loc_SrcLoc format_loc;
   const char* format_text = printf_utils_get_format(format, &format_loc);
   if (!format_text) return;

   module_analyser_FormatAnalyser fa = {
   ma,
   format_text,
   num_args,
   0,
   format_loc,
   false,
   args
};
   if (!printf_utils_parseFormat(format_text, module_analyser_on_format_specifier, &fa)) return;

   *change_format = fa.change_format;
   if ((fa.idx < num_args)) {
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(args[fa.idx]), "too many arguments for format");
   }
}

static void module_analyser_create_template_name(char* name, const char* orig, uint16_t idx)
{
   sprintf(name, "%s_%u_", orig, idx);
}

static void module_analyser_Analyser_opaque_callback(void* arg, src_loc_SrcLoc loc, ast_Decl* d)
{
   module_analyser_Analyser* ma = arg;
   ast_QualType qt = ast_Decl_getType(d);
   module_analyser_Analyser_error(ma, loc, " using opaque type '%s'", ast_QualType_diagName(&qt));
}

static ast_FunctionDecl* module_analyser_Analyser_instantiateTemplateFunction(module_analyser_Analyser* ma, ast_CallExpr* call, ast_FunctionDecl* fd)
{
   ast_TypeRef* template_arg = ast_CallExpr_getTemplateArg(call);
   ast_QualType templateType = module_analyser_Analyser_analyseTypeRef(ma, template_arg);
   if (ast_QualType_isInvalid(&templateType)) return NULL;

   ast_FunctionDecl* instance = ast_Module_findInstance(ma->mod, fd, templateType);
   if (!instance) {
      bool used_opaque = false;
      ast_StructType* st = ast_QualType_getStructTypeOrNil(&templateType);
      if (st) {
         ast_StructTypeDecl* std = ast_StructType_getDecl(st);
         ast_Decl* d = ((ast_Decl*)(std));
         used_opaque = ((ast_StructTypeDecl_isOpaque(std) && (ast_Decl_getModule(d) != ma->mod)));
      }
      ast_Instantiator inst = { .c = ma->context, .ref = template_arg, .template_name = ast_FunctionDecl_getTemplateNameIdx(fd), .used_opaque = used_opaque, .arg = ma, .on_error = module_analyser_Analyser_opaque_callback };
      instance = ast_FunctionDecl_instantiate(fd, &inst);
      ast_Decl* d = ((ast_Decl*)(instance));
      module_analyser_Analyser_analyseFunction(ma, instance);
      if (ma->has_error) return NULL;

      ast_Decl_setChecked(d);
      ast_Module* template_mod = ast_Decl_getModule(ast_FunctionDecl_asDecl(fd));
      module_analyser_Analyser* analyser = module_analyser_create(ma->diags, ma->context, ma->astPool, ma->builder, ma->allmodules, ma->warnings);
      module_analyser_Analyser_setMod(analyser, template_mod);
      scope_Scope* tmpScope = scope_create(ma->allmodules, ma->diags, ast_AST_getImports(ast_Decl_getAST(d)), template_mod, ast_Module_getSymbols(template_mod), !ma->warnings->no_unused_variable);
      module_analyser_Analyser_analyseFunctionBody(analyser, instance, tmpScope);
      scope_Scope_free(tmpScope);
      module_analyser_Analyser_free(analyser);
      if (ma->has_error) return NULL;

      uint16_t instance_idx = ast_Module_addInstance(ma->mod, fd, templateType, instance);
      ast_FunctionDecl_setTemplateInstanceIdx(instance, instance_idx);
      char name[64];
      module_analyser_create_template_name(name, ast_Decl_getName(d), instance_idx);
      ast_FunctionDecl_setInstanceName(instance, string_pool_Pool_addStr(ma->astPool, name, true));
   }
   ast_CallExpr_setTemplateIdx(call, ast_FunctionDecl_getTemplateInstanceIdx(instance));
   return instance;
}

static ast_QualType module_analyser_Analyser_analysePureCallExpr(module_analyser_Analyser* ma, ast_Expr* e)
{
   ast_CallExpr* call = ((ast_CallExpr*)(e));
   ast_Expr** func = ast_CallExpr_getFunc2(call);
   ast_Expr* origFn = ast_CallExpr_getFunc(call);
   ast_QualType qt = module_analyser_Analyser_analyseExpr(ma, func, true, module_analyser_RHS);
   if (ast_QualType_isInvalid(&qt)) return ast_QualType_Invalid;

   if (ast_Expr_isNValue(origFn)) {
      module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(origFn), ast_Expr_getRange(origFn), "called object is not a function of function pointer");
      return ast_QualType_Invalid;
   }
   ast_FunctionType* ft = ast_QualType_getFunctionTypeOrNil(&qt);
   if (!ft) {
      ast_Expr* fn2 = ast_CallExpr_getFunc(call);
      module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(fn2), ast_Expr_getRange(fn2), "called object type %s is not a function or function pointer", ast_QualType_diagName(&qt));
      return ast_QualType_Invalid;
   }
   ast_FunctionDecl* fd = ast_FunctionType_getDecl(ft);
   ast_Decl_setUsed(ast_FunctionDecl_asDecl(fd));
   if (!ast_FunctionDecl_hasAttrPure(fd)) {
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "only pure functions can be called in global initializers");
      module_analyser_Analyser_note(ma, ast_Decl_getLoc(ast_FunctionDecl_asDecl(fd)), "'%s' declared here", ast_Decl_getFullName(ast_FunctionDecl_asDecl(fd)));
      return ast_QualType_Invalid;
   }
   uint32_t func_num_args = ast_FunctionDecl_getNumParams(fd);
   uint32_t call_num_args = ast_CallExpr_getNumArgs(call);
   ast_VarDecl** func_args = ast_FunctionDecl_getParams(fd);
   ast_Expr** call_args = ast_CallExpr_getArgs(call);
   if ((func_num_args != call_num_args)) {
      if ((call_num_args > func_num_args)) {
         ast_Expr* call_arg = call_args[func_num_args];
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(call_arg), "too many arguments to %sfunction call, expected %u, have %u", ast_FunctionDecl_getDiagKind(fd), func_num_args, call_num_args);
         module_analyser_Analyser_note(ma, ast_Decl_getLoc(ast_FunctionDecl_asDecl(fd)), "'%s' declared here", ast_Decl_getFullName(ast_FunctionDecl_asDecl(fd)));
      } else {
         module_analyser_Analyser_error(ma, ast_CallExpr_getEndLoc(call), "too few arguments to %sfunction call, expected %u, have %u", ast_FunctionDecl_getDiagKind(fd), func_num_args, call_num_args);
         module_analyser_Analyser_note(ma, ast_Decl_getLoc(ast_FunctionDecl_asDecl(fd)), "'%s' declared here", ast_Decl_getFullName(ast_FunctionDecl_asDecl(fd)));
      }
      return ast_QualType_Invalid;
   }
   return ast_QualType_Invalid;
}

static ast_QualType module_analyser_Analyser_analyseExpr(module_analyser_Analyser* ma, ast_Expr** e_ptr, bool need_rvalue, uint32_t side)
{
   c2_assert((e_ptr) != NULL, "analyser/module_analyser_expr.c2:25: module_analyser.Analyser.analyseExpr", "e_ptr");
   ast_QualType result = module_analyser_Analyser_analyseExprInner(ma, e_ptr, side);
   if (ast_QualType_isInvalid(&result)) return result;

   ast_Expr* e = *e_ptr;
   ast_Expr_setType(e, result);
   if (need_rvalue) {
      if (ast_Expr_isLValue(e)) {
         ast_QualType canon = ast_QualType_getCanonicalType(&result);
         c2_assert((ast_QualType_isValid(&canon)) != 0, "analyser/module_analyser_expr.c2:35: module_analyser.Analyser.analyseExpr", "CALL TODO");
         if (ast_QualType_isArray(&canon)) {
            result = module_analyser_getPointerFromArray(ma->builder, canon);
            ast_builder_Builder_insertImplicitCast(ma->builder, ast_ImplicitCastKind_ArrayToPointerDecay, e_ptr, result);
         } else {
            ast_QualType_unsetConst(&result);
            ast_builder_Builder_insertImplicitCast(ma->builder, ast_ImplicitCastKind_LValueToRValue, e_ptr, result);
         }
      } else if (ast_Expr_isNValue(e)) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "lvalue/rvalue required");
         return ast_QualType_Invalid;
      }

   }
   return result;
}

static ast_QualType module_analyser_Analyser_analyseExprInner(module_analyser_Analyser* ma, ast_Expr** e_ptr, uint32_t side)
{
   ast_Expr* e = *e_ptr;
   switch (ast_Expr_getKind(e)) {
   case ast_ExprKind_IntegerLiteral:
      return ast_Expr_getType(e);
   case ast_ExprKind_FloatLiteral:
      return ast_Expr_getType(e);
   case ast_ExprKind_BooleanLiteral:
      return ast_builtins[ast_BuiltinKind_Bool];
   case ast_ExprKind_CharLiteral:
      return ast_builtins[ast_BuiltinKind_Int8];
   case ast_ExprKind_StringLiteral:
      return ast_Expr_getType(e);
   case ast_ExprKind_Nil:
      return ast_getVoidPtr();
   case ast_ExprKind_Identifier: {
      ast_Decl* d = module_analyser_Analyser_analyseIdentifier(ma, e_ptr, side);
      if (!d) break;

      return ast_Decl_getType(d);
   }
   case ast_ExprKind_Type:
      break;
   case ast_ExprKind_Call:
      return module_analyser_Analyser_analyseCallExpr(ma, e_ptr);
   case ast_ExprKind_InitList:
      ast_Expr_dump((*e_ptr));
      c2_assert((0) != 0, "analyser/module_analyser_expr.c2:80: module_analyser.Analyser.analyseExprInner", "0");
      break;
   case ast_ExprKind_FieldDesignatedInit:
      ast_Expr_dump((*e_ptr));
      c2_assert((0) != 0, "analyser/module_analyser_expr.c2:84: module_analyser.Analyser.analyseExprInner", "0");
      break;
   case ast_ExprKind_ArrayDesignatedInit:
      ast_Expr_dump((*e_ptr));
      c2_assert((0) != 0, "analyser/module_analyser_expr.c2:88: module_analyser.Analyser.analyseExprInner", "0");
      break;
   case ast_ExprKind_BinaryOperator:
      return module_analyser_Analyser_analyseBinaryOperator(ma, e_ptr);
   case ast_ExprKind_UnaryOperator:
      return module_analyser_Analyser_analyseUnaryOperator(ma, e_ptr, side);
   case ast_ExprKind_ConditionalOperator:
      return module_analyser_Analyser_analyseConditionalOperator(ma, e_ptr);
   case ast_ExprKind_Builtin:
      return module_analyser_Analyser_analyseBuiltin(ma, e_ptr);
   case ast_ExprKind_ArraySubscript:
      return module_analyser_Analyser_analyseArraySubscriptExpr(ma, e_ptr, side);
   case ast_ExprKind_Member:
      return module_analyser_Analyser_analyseMemberExpr(ma, e_ptr, side);
   case ast_ExprKind_Paren: {
      ast_ParenExpr* p = ((ast_ParenExpr*)(e));
      ast_QualType qt = module_analyser_Analyser_analyseExpr(ma, ast_ParenExpr_getInner2(p), false, side);
      ast_Expr* inner = ast_ParenExpr_getInner(p);
      ast_Expr_copyConstantFlags(e, inner);
      ast_Expr_copyValType(e, inner);
      return qt;
   }
   case ast_ExprKind_BitOffset:
      break;
   case ast_ExprKind_ExplicitCast:
      return module_analyser_Analyser_analyseExplicitCast(ma, e_ptr);
   case ast_ExprKind_ImplicitCast:
      break;
   }
   return ast_QualType_Invalid;
}

static ast_Decl* module_analyser_Analyser_analyseIdentifier(module_analyser_Analyser* ma, ast_Expr** e_ptr, uint32_t side)
{
   ast_Expr* e = *e_ptr;
   ast_IdentifierExpr* i = ((ast_IdentifierExpr*)(e));
   ast_Decl* d = scope_Scope_find(ma->scope, ast_IdentifierExpr_getNameIdx(i), ast_Expr_getLoc(e), ma->usedPublic);
   if (!d) {
      ma->has_error = true;
      return NULL;
   }
   if (!ast_Decl_isChecked(d)) {
      if (!module_analyser_Analyser_analyseGlobalDecl(ma, d)) return NULL;

   }
   ast_QualType qt = ast_Decl_getType(d);
   c2_assert((ast_QualType_isValid(&qt)) != 0, "analyser/module_analyser_expr.c2:133: module_analyser.Analyser.analyseIdentifier", "CALL TODO");
   ast_Expr_setType(e, qt);
   ast_IdentifierExpr_setDecl(i, d);
   if (((side & module_analyser_RHS) || (side == 0))) ast_Decl_setUsed(d);
   else {
      if (ast_Decl_isVarDecl(d)) {
         ast_VarDecl* vd = ((ast_VarDecl*)(d));
         if (ast_VarDecl_isParameter(vd)) ast_Decl_setUsed(d);
      }
   }
   ast_IdentifierKind kind = module_analyser_Analyser_setExprFlags(ma, e_ptr, d);
   ast_IdentifierExpr_setKind(i, kind);
   if ((ma->usedPublic && !ast_Decl_isPublic(d))) {
      const char* kind_str = ast_QualType_isConst(&qt) ? "constant" : "variable";
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "public declaration using non-public %s '%s'", kind_str, ast_Decl_getFullName(d));
      return NULL;
   }
   return d;
}

static ast_IdentifierKind module_analyser_Analyser_setExprFlags(module_analyser_Analyser* ma, ast_Expr** e_ptr, ast_Decl* d)
{
   ast_Expr* e = *e_ptr;
   ast_IdentifierKind kind = ast_IdentifierKind_Unresolved;
   switch (ast_Decl_getKind(d)) {
   case ast_DeclKind_Function:
      ast_Expr_setCtc(e);
      ast_Expr_setRValue(e);
      ast_builder_Builder_insertImplicitCast(ma->builder, ast_ImplicitCastKind_FunctionToPointerDecay, e_ptr, ast_Decl_getType(d));
      kind = ast_IdentifierKind_Function;
      break;
   case ast_DeclKind_Import:
      ast_Expr_setCtc(e);
      kind = ast_IdentifierKind_Module;
      break;
   case ast_DeclKind_StructType:
      kind = ast_IdentifierKind_Type;
      break;
   case ast_DeclKind_EnumType:
      ast_Expr_setCtc(e);
      kind = ast_IdentifierKind_Type;
      break;
   case ast_DeclKind_EnumConstant:
      ast_Expr_setCtc(e);
      ast_Expr_setCtv(e);
      ast_Expr_setRValue(e);
      kind = ast_IdentifierKind_EnumConstant;
      break;
   case ast_DeclKind_FunctionType:
      ast_Expr_setCtc(e);
      kind = ast_IdentifierKind_Type;
      break;
   case ast_DeclKind_AliasType:
      kind = ast_IdentifierKind_Type;
      break;
   case ast_DeclKind_Variable: {
      ast_VarDecl* vd = ((ast_VarDecl*)(d));
      ast_QualType t = ast_Decl_getType(ast_VarDecl_asDecl(vd));
      if (ast_VarDecl_isGlobal(vd)) ast_Expr_setCtc(e);
      ast_Expr_setLValue(e);
      const ast_Expr* init_ = ast_VarDecl_getInit(vd);
      if (((init_ && ast_QualType_isConst(&t)) && ast_Expr_isCtv(init_))) ast_Expr_setCtv(e);
      switch (ast_VarDecl_getKind(vd)) {
      case ast_VarDeclKind_GlobalVar:
         __attribute__((fallthrough));
      case ast_VarDeclKind_LocalVar:
         __attribute__((fallthrough));
      case ast_VarDeclKind_FunctionParam:
         kind = ast_IdentifierKind_Var;
         break;
      case ast_VarDeclKind_StructMember:
         kind = ast_IdentifierKind_StructMember;
         break;
      }
      break;
   }
   }
   return kind;
}

static ast_QualType module_analyser_Analyser_analyseConditionalOperator(module_analyser_Analyser* ma, ast_Expr** e_ptr)
{
   ast_Expr* e = *e_ptr;
   ast_ConditionalOperator* cond = ((ast_ConditionalOperator*)(e));
   ast_QualType qt = module_analyser_Analyser_analyseExpr(ma, ast_ConditionalOperator_getCond2(cond), true, module_analyser_RHS);
   if (ast_QualType_isInvalid(&qt)) return ast_QualType_Invalid;

   conversion_checker_Checker_check(&ma->checker, ast_builtins[ast_BuiltinKind_Bool], qt, ast_ConditionalOperator_getCond2(cond), ast_Expr_getLoc(ast_ConditionalOperator_getCond(cond)));
   ast_QualType lhs = module_analyser_Analyser_analyseExpr(ma, ast_ConditionalOperator_getLHS2(cond), true, module_analyser_RHS);
   ast_QualType rhs = module_analyser_Analyser_analyseExpr(ma, ast_ConditionalOperator_getRHS2(cond), true, module_analyser_RHS);
   if ((ast_QualType_isInvalid(&lhs) || ast_QualType_isInvalid(&rhs))) return ast_QualType_Invalid;

   ast_QualType lcanon = ast_QualType_getCanonicalType(&lhs);
   ast_QualType rcanon = ast_QualType_getCanonicalType(&rhs);
   c2_assert((ast_QualType_isValid(&lcanon)) != 0, "analyser/module_analyser_expr.c2:252: module_analyser.Analyser.analyseConditionalOperator", "CALL TODO");
   c2_assert((ast_QualType_isValid(&rcanon)) != 0, "analyser/module_analyser_expr.c2:253: module_analyser.Analyser.analyseConditionalOperator", "CALL TODO");
   uint8_t res = module_analyser_CondOpTable[ast_QualType_getKind(&lcanon)][ast_QualType_getKind(&rcanon)];
   switch (res) {
   case 0:
      break;
   case 1:
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "invalid operands to ternary operator (%s and %s)", ast_QualType_diagName(&lhs), ast_QualType_diagName(&rhs));
      return ast_QualType_Invalid;
   case 2:
      return conversion_checker_get_common_arithmetic_type(lcanon, rcanon);
   case 3:
      return lhs;
   case 4:
      return lhs;
   case 5: {
      bool ok = conversion_checker_Checker_check(&ma->checker, lhs, rhs, e_ptr, ast_Expr_getLoc(e));
      if (!ok) return ast_QualType_Invalid;

      return lhs;
   }
   case 6:
      return rhs;
   case 7: {
      bool ok = conversion_checker_Checker_check(&ma->checker, lhs, rhs, e_ptr, ast_Expr_getLoc(e));
      if (!ok) return ast_QualType_Invalid;

      return lhs;
   }
   case 8:
      return lhs;
   }
   ast_Expr_dump(e);
   c2_assert((0) != 0, "analyser/module_analyser_expr.c2:286: module_analyser.Analyser.analyseConditionalOperator", "0");
   return ast_QualType_Invalid;
}

static bool module_analyser_Analyser_checkAssignment(module_analyser_Analyser* ma, ast_Expr* assignee, ast_QualType tleft, const char* msg, src_loc_SrcLoc loc)
{
   if (ast_QualType_isConst(&tleft)) {
      if (ast_Expr_isIdentifier(assignee)) {
         ast_IdentifierExpr* i = ((ast_IdentifierExpr*)(assignee));
         module_analyser_Analyser_error(ma, loc, "cannot assign to read-only variable '%s'", ast_Decl_getFullName(ast_IdentifierExpr_getDecl(i)));
         return false;
      }
      if (ast_Expr_isMember(assignee)) {
         ast_MemberExpr* m = ((ast_MemberExpr*)(assignee));
         switch (ast_MemberExpr_getKind(m)) {
         case ast_IdentifierKind_Unresolved:
            c2_assert((0) != 0, "analyser/module_analyser_expr.c2:302: module_analyser.Analyser.checkAssignment", "0");
            break;
         case ast_IdentifierKind_Module:
            c2_assert((0) != 0, "analyser/module_analyser_expr.c2:305: module_analyser.Analyser.checkAssignment", "0");
            break;
         case ast_IdentifierKind_Function:
            break;
         case ast_IdentifierKind_Type:
            break;
         case ast_IdentifierKind_Var:
            module_analyser_Analyser_error(ma, loc, "cannot assign to read-only variable '%s'", ast_MemberExpr_getLastMemberName(m));
            return false;
         case ast_IdentifierKind_EnumConstant:
            break;
         case ast_IdentifierKind_StructMember:
            module_analyser_Analyser_error(ma, loc, "assignment of member '%s' in read-only object", ast_MemberExpr_getLastMemberName(m));
            return false;
         case ast_IdentifierKind_Label:
            break;
         }
      }
      module_analyser_Analyser_error(ma, loc, "cannot assign to variable with const-qualified type '%s'", ast_QualType_diagName(&tleft));
      return false;
   }
   if (!ast_Expr_isLValue(assignee)) {
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(assignee), "lvalue required as %s", msg);
      return false;
   }
   if (ast_QualType_isArray(&tleft)) {
      module_analyser_Analyser_error(ma, loc, "array type '%s' is not assignable", ast_QualType_diagName(&tleft));
      return false;
   }
   return true;
}

static ast_QualType module_analyser_usualUnaryConversions(ast_Expr* e)
{
   ast_QualType qt = ast_Expr_getType(e);
   ast_QualType canon = ast_QualType_getCanonicalType(&qt);
   if (ast_QualType_isBuiltin(&canon)) {
      ast_BuiltinType* bi = ast_QualType_getBuiltin(&canon);
      if (ast_BuiltinType_isPromotableIntegerType(bi)) return ast_builtins[ast_BuiltinKind_Int32];

   } else if (ast_QualType_isPointer(&canon)) {
      return ast_builtins[ast_BuiltinKind_UInt64];
   }

   return qt;
}

static ast_QualType module_analyser_Analyser_analyseExplicitCast(module_analyser_Analyser* ma, ast_Expr** e_ptr)
{
   ast_Expr* e = *e_ptr;
   ast_ExplicitCastExpr* c = ((ast_ExplicitCastExpr*)(e));
   ast_TypeRef* ref = ast_ExplicitCastExpr_getTypeRef(c);
   ast_QualType destType = module_analyser_Analyser_analyseTypeRef(ma, ref);
   ast_QualType srcType = module_analyser_Analyser_analyseExpr(ma, ast_ExplicitCastExpr_getInner2(c), true, module_analyser_RHS);
   if ((ast_QualType_isInvalid(&srcType) || ast_QualType_isInvalid(&destType))) return ast_QualType_Invalid;

   ast_Expr* inner = ast_ExplicitCastExpr_getInner(c);
   ast_Expr_copyConstantFlags(e, inner);
   ast_Expr_copyValType(e, inner);
   if (!ast_QualType_isScalar(&destType)) {
      module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(ref), "used type '%s' where arithmetic or pointer type is required", ast_QualType_diagName(&destType));
      return ast_QualType_Invalid;
   }
   if (!conversion_checker_Checker_checkCast(&ma->checker, destType, srcType, ast_TypeRef_getLoc(ref), ast_Expr_getLoc(inner))) return ast_QualType_Invalid;

   return destType;
}

static ast_QualType module_analyser_Analyser_analyseArraySubscriptExpr(module_analyser_Analyser* ma, ast_Expr** e_ptr, uint32_t side)
{
   ast_Expr* e = *e_ptr;
   ast_ArraySubscriptExpr* sub = ((ast_ArraySubscriptExpr*)(e));
   ast_QualType q = module_analyser_Analyser_analyseExpr(ma, ast_ArraySubscriptExpr_getBase2(sub), true, side);
   if (ast_QualType_isInvalid(&q)) return q;

   ast_Expr* index = ast_ArraySubscriptExpr_getIndex(sub);
   if (ast_Expr_isBitOffset(index)) {
      if ((side & module_analyser_LHS)) {
         module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(e), ast_Expr_getRange(e), "bitoffset cannot be used as left hand side expression");
         return ast_QualType_Invalid;
      }
      ast_Expr* base = ast_ArraySubscriptExpr_getBase(sub);
      q = module_analyser_Analyser_analyseBitOffsetExpr(ma, q, base, index);
      ast_Expr_combineConstantFlags(e, base, index);
      return q;
   }
   q = ast_QualType_getCanonicalType(&q);
   if (!ast_QualType_isPointer(&q)) {
      module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(e), ast_Expr_getRange(e), "subscripted value is not an array or pointer");
      return ast_QualType_Invalid;
   }
   ast_QualType qidx = module_analyser_Analyser_analyseExpr(ma, ast_ArraySubscriptExpr_getIndex2(sub), true, module_analyser_RHS);
   if (ast_QualType_isInvalid(&qidx)) return qidx;

   ast_PointerType* pt = ast_QualType_getPointerType(&q);
   return ast_PointerType_getInner(pt);
}

static ast_QualType module_analyser_Analyser_analyseBitOffsetExpr(module_analyser_Analyser* ma, ast_QualType ltype, ast_Expr* base, ast_Expr* e)
{
   ast_BitOffsetExpr* bo = ((ast_BitOffsetExpr*)(e));
   ast_QualType canon = ast_QualType_getCanonicalType(&ltype);
   ast_BuiltinType* bi = ast_QualType_getBuiltin(&canon);
   if ((!ast_QualType_isBuiltin(&canon) || !ast_BuiltinType_isUnsigned(bi))) {
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(base), "bitoffsets are only allowed on unsigned integer type");
      return ast_QualType_Invalid;
   }
   ast_Value lval;
   ast_Value rval;
   bool lvalid = module_analyser_Analyser_analyseBitOffsetIndex(ma, ast_BitOffsetExpr_getLHS2(bo), canon, &lval);
   bool rvalid = module_analyser_Analyser_analyseBitOffsetIndex(ma, ast_BitOffsetExpr_getRHS2(bo), canon, &rval);
   if ((lvalid && rvalid)) {
      if (!ast_Value_less_than(&rval, &lval)) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "left bitoffset index is smaller than right index");
         return ast_QualType_Invalid;
      }
      ast_Value width = ast_Value_minus(&lval, &rval);
      width.uvalue++;
      if ((width.uvalue <= 8)) {
         ltype = ast_builtins[ast_BuiltinKind_UInt8];
      } else if ((width.uvalue <= 16)) {
         ltype = ast_builtins[ast_BuiltinKind_UInt16];
      } else if ((width.uvalue <= 32)) {
         ltype = ast_builtins[ast_BuiltinKind_UInt32];
      } else {
         ltype = ast_builtins[ast_BuiltinKind_UInt64];
      }


      ast_BitOffsetExpr_setWidth(bo, ((uint8_t)(width.uvalue)));
      ast_Expr_setType(e, ltype);
   }
   ast_Expr_combineConstantFlags(e, ast_BitOffsetExpr_getLHS(bo), ast_BitOffsetExpr_getRHS(bo));
   return ltype;
}

static bool module_analyser_Analyser_analyseBitOffsetIndex(module_analyser_Analyser* ma, ast_Expr** e_ptr, ast_QualType baseType, ast_Value* result)
{
   ast_BuiltinType* base_bi = ast_QualType_getBuiltin(&baseType);
   ast_QualType qt = module_analyser_Analyser_analyseExpr(ma, e_ptr, true, module_analyser_RHS);
   if (ast_QualType_isInvalid(&qt)) return false;

   ast_Expr* e = *e_ptr;
   ast_QualType canon = ast_QualType_getCanonicalType(&qt);
   ast_BuiltinType* bi = ast_QualType_getBuiltin(&canon);
   if ((!ast_QualType_isBuiltin(&canon) || !ast_BuiltinType_isInteger(bi))) {
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "index of bitoffset has non-integer type '%s'", ast_QualType_diagName(&qt));
      return false;
   }
   if (!ast_Expr_isCtv(e)) return false;

   ast_Value val = ctv_analyser_get_value(e);
   if (ast_Value_isNegative(&val)) {
      module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(e), ast_Expr_getRange(e), "bitoffset index value '%s' is negative", ast_Value_str(&val));
      return false;
   }
   if (ast_Value_ugt(&val, (ast_BuiltinType_getWidth(base_bi) - 1))) {
      module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(e), ast_Expr_getRange(e), "bitoffset index value '%s' too large for type '%s'", ast_Value_str(&val), ast_QualType_diagName(&baseType));
      return false;
   }
   *result = val;
   return true;
}

static void module_analyser_Analyser_memberError(module_analyser_Analyser* ma, uint32_t name_idx, src_loc_SrcLoc loc, ast_StructTypeDecl* s)
{
   module_analyser_Analyser_error(ma, loc, "no member named '%s' in %s '%s'", ast_idx2name(name_idx), ast_StructTypeDecl_isStruct(s) ? "struct" : "union", ast_Decl_getFullName(ast_StructTypeDecl_asDecl(s)));
}

static ast_Decl* module_analyser_Analyser_findStructMember(module_analyser_Analyser* ma, ast_StructTypeDecl* s, uint32_t name_idx, src_loc_SrcLoc loc, bool allow_funcs)
{
   ast_Decl* d = ast_StructTypeDecl_findAny(s, name_idx);
   if ((!d || ((!allow_funcs && ast_Decl_isFunction(d))))) {
      module_analyser_Analyser_memberError(ma, name_idx, loc, s);
      return NULL;
   }
   return d;
}

static ast_QualType module_analyser_getPointerFromArray(ast_builder_Builder* builder, ast_QualType q)
{
   const ast_ArrayType* a = ((ast_ArrayType*)(ast_QualType_getTypeOrNil(&q)));
   ast_QualType elem = ast_ArrayType_getElemType(a);
   if (ast_QualType_isConst(&q)) ast_QualType_setConst(&elem);
   ast_QualType res = ast_builder_Builder_actOnPointerType(builder, elem);
   return res;
}

static void module_analyser_Analyser_analyseFunction(module_analyser_Analyser* ma, ast_FunctionDecl* fd)
{
   if (ast_FunctionDecl_isTemplate(fd)) {
      scope_Scope_checkGlobalSymbol(ma->scope, ast_FunctionDecl_getTemplateNameIdx(fd), ast_FunctionDecl_getTemplateLoc(fd));
      ast_TypeRef* rtype = ast_FunctionDecl_getReturnTypeRef(fd);
      if (ast_TypeRef_getNumArrays(rtype)) {
         module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(rtype), "functions are not allowed to return array types");
      }
      uint32_t num_params = ast_FunctionDecl_getNumParams(fd);
      ast_VarDecl** params = ast_FunctionDecl_getParams(fd);
      for (uint32_t i = 0; (i < num_params); i++) {
         ast_VarDecl* v = params[i];
         ast_TypeRef* ref = ast_VarDecl_getTypeRef(v);
         if (ast_TypeRef_getNumArrays(ref)) {
            module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(ref), "array types are not allowed here");
         }
      }
      return;
   }
   ast_TypeRef* rtype = ast_FunctionDecl_getReturnTypeRef(fd);
   ast_QualType qt = module_analyser_Analyser_analyseTypeRef(ma, rtype);
   if (ast_QualType_isInvalid(&qt)) return;

   ast_QualType canon = ast_QualType_getCanonicalType(&qt);
   if (ast_QualType_isArray(&canon)) {
      module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(rtype), "functions are not allowed to return array types");
   }
   if ((ast_QualType_isConst(&canon) && !ast_QualType_isPointer(&canon))) {
      module_analyser_Analyser_warn(ma, ast_TypeRef_getLoc(rtype), "'const' type qualifier on return type has no effect");
   }
   bool is_public = ast_Decl_isPublic(ast_FunctionDecl_asDecl(fd));
   if (is_public) ast_setTypePublicUsed(qt);
   ast_FunctionDecl_setRType(fd, qt);
   uint32_t num_params = ast_FunctionDecl_getNumParams(fd);
   ast_VarDecl** params = ast_FunctionDecl_getParams(fd);
   uint32_t auto_arg_count = 0;
   uint32_t first_auto_arg = 0;
   for (uint32_t i = 0; (i < num_params); i++) {
      ast_VarDecl* v = params[i];
      ast_TypeRef* ref = ast_VarDecl_getTypeRef(v);
      ast_QualType res = module_analyser_Analyser_analyseTypeRef(ma, ref);
      if (ast_QualType_isInvalid(&res)) continue;

      canon = ast_QualType_getCanonicalType(&res);
      if (ast_QualType_isArray(&canon)) {
         module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(ref), "array types are not allowed here");
         continue;
      }
      if (ast_QualType_isVoid(&canon)) {
         module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(ref), "parameter has invalid type 'void'");
         continue;
      }
      if (is_public) ast_setTypePublicUsed(res);
      if (ast_VarDecl_hasAutoAttr(v)) {
         if ((auto_arg_count == 0)) first_auto_arg = i;
         auto_arg_count++;
         if ((ast_VarDecl_hasAttrAutoFile(v) && !ast_TypeRef_isConstCharPtr(ref))) {
            module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(ref), "attribute 'auto_file' requires a parameter of type 'const char*'");
         }
         if ((ast_VarDecl_hasAttrAutoLine(v) && !ast_TypeRef_isU32(ref))) {
            module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(ref), "attribute 'auto_line' requires a parameter of type 'u32'");
         }
      }
      if (ast_VarDecl_hasPrintfFormat(v)) {
         module_analyser_Analyser_checkPrintfFormat(ma, v, res, i, fd);
         ast_FunctionDecl_setAttrPrintf(fd, ((uint8_t)((i + 1))));
      }
      ast_Decl* d = ((ast_Decl*)(v));
      ast_Decl_setType(d, res);
      ast_Decl_setChecked(d);
   }
   bool is_sf = false;
   if ((num_params && ast_FunctionDecl_hasPrefix(fd))) {
      const ast_Ref* prefix = ast_FunctionDecl_getPrefix(fd);
      c2_assert((prefix->decl) != NULL, "analyser/module_analyser_function.c2:120: module_analyser.Analyser.analyseFunction", "prefix.decl");
      ast_QualType prefixType = ast_Decl_getType(prefix->decl);
      ast_TypeRef* ref = ast_VarDecl_getTypeRef(params[0]);
      if (ast_TypeRef_isPointerTo(ref, ast_QualType_getIndex(&prefixType))) {
         ast_FunctionDecl_setCallKind(fd, ast_CallKind_StructFunc);
         is_sf = true;
      }
   }
   if (auto_arg_count) {
      ast_FunctionDecl_setNumAutoArgs(fd, auto_arg_count);
      bool seen_normal_arg = false;
      uint32_t start = is_sf ? 1 : 0;
      for (uint32_t i = start; (i < num_params); i++) {
         ast_VarDecl* v = params[i];
         if (ast_VarDecl_hasAutoAttr(v)) {
            if (seen_normal_arg) {
               module_analyser_Analyser_error(ma, ast_Decl_getLoc(ast_VarDecl_asDecl(v)), "auto-arguments must come before normal arguments");
            }
         } else {
            seen_normal_arg = true;
         }
      }
   }
   if ((ast_FunctionDecl_hasAttrConstructor(fd) || ast_FunctionDecl_hasAttrDestructor(fd))) {
      if (!ast_TypeRef_isVoid(rtype)) {
         module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(rtype), "functions marked with '%s' cannot return a value", ast_FunctionDecl_hasAttrConstructor(fd) ? "constructor" : "destructor");
      }
      if ((num_params || ast_FunctionDecl_isVariadic(fd))) {
         module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(rtype), "functions marked with '%s' cannot have arguments", ast_FunctionDecl_hasAttrConstructor(fd) ? "constructor" : "destructor");
      }
   }
   if (ast_FunctionDecl_hasAttrPure(fd)) {
      if ((num_params == 0)) {
         module_analyser_Analyser_error(ma, ast_Decl_getLoc(ast_FunctionDecl_asDecl(fd)), "pure functions must have arguments");
         return;
      }
      if (ast_FunctionDecl_isVariadic(fd)) {
         module_analyser_Analyser_error(ma, ast_Decl_getLoc(ast_FunctionDecl_asDecl(fd)), "pure functions cannot be variadic");
         return;
      }
      if (!ast_FunctionDecl_hasReturn(fd)) {
         module_analyser_Analyser_error(ma, ast_TypeRef_getLoc(rtype), "pure functions must return a value");
         return;
      }
      if (auto_arg_count) {
         ast_VarDecl* v = params[first_auto_arg];
         module_analyser_Analyser_error(ma, ast_Decl_getLoc(ast_VarDecl_asDecl(v)), "pure functions cannot have auto-arguments");
         return;
      }
   }
}

static void module_analyser_Analyser_analyseFunctionBody(module_analyser_Analyser* ma, ast_FunctionDecl* fd, scope_Scope* s)
{
   if (ast_FunctionDecl_isTemplate(fd)) return;

   ast_CompoundStmt* body = ast_FunctionDecl_getBody(fd);
   if (!body) return;

   ast_Decl* d = ((ast_Decl*)(fd));
   module_analyser_Analyser_pushCheck(ma, d, s, fd);
   ma->checkStack[0].usedPublic = false;
   ma->usedPublic = false;
   scope_Scope_reset(ma->scope);
   scope_Scope_enter(ma->scope, (scope_Function | scope_Decl));
   uint32_t num_params = ast_FunctionDecl_getNumParams(fd);
   ast_VarDecl** params = ast_FunctionDecl_getParams(fd);
   for (uint32_t i = 0; (i < num_params); i++) {
      ast_Decl* p = ((ast_Decl*)(params[i]));
      if (ast_Decl_getNameIdx(p)) {
         bool error = scope_Scope_add(ma->scope, p);
         if (error) return;

      }
   }
   ma->has_error = false;
   module_analyser_LabelVector_reset(&ma->labels);
   module_analyser_Analyser_analyseCompoundStmt(ma, body);
   ast_QualType rtype = ast_FunctionDecl_getRType(fd);
   if (!ast_QualType_isVoid(&rtype)) {
      if (!module_analyser_hasReturn(((ast_Stmt*)(body)))) {
         module_analyser_Analyser_error(ma, ast_CompoundStmt_getEndLoc(body), "control reaches end of non-void function");
      }
   }
   if ((!ma->warnings->no_unused_parameter && !ast_FunctionDecl_hasAttrUnusedParams(fd))) {
      for (uint32_t i = 0; (i < num_params); i++) {
         ast_Decl* p = ((ast_Decl*)(params[i]));
         if ((!ast_Decl_isUsed(p) && ast_Decl_getNameIdx(p))) {
            module_analyser_Analyser_warn(ma, ast_Decl_getLoc(p), "unused parameter '%s'", ast_Decl_getName(p));
         }
      }
   }
   scope_Scope_exit(ma->scope, ma->has_error);
   uint32_t num_labels = module_analyser_LabelVector_getCount(&ma->labels);
   const module_analyser_Label* labels = module_analyser_LabelVector_getLabels(&ma->labels);
   for (uint32_t i = 0; (i < num_labels); i++) {
      const module_analyser_Label* l = &labels[i];
      if (l->is_label) {
         if ((!l->used && !ma->warnings->no_unused_label)) {
            module_analyser_Analyser_warn(ma, l->loc, "unused label '%s'", ast_idx2name(l->name_idx));
         }
      } else {
         module_analyser_Analyser_error(ma, l->loc, "use of undeclared label '%s'", ast_idx2name(l->name_idx));
      }
   }
   module_analyser_Analyser_popCheck(ma);
}

static void module_analyser_Analyser_checkPrintfFormat(module_analyser_Analyser* ma, ast_VarDecl* v, ast_QualType qt, uint32_t idx, ast_FunctionDecl* fd)
{
   ast_Decl* d = ((ast_Decl*)(v));
   if (!ast_QualType_isCharPointer(&qt)) {
      module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "printf_format parameter must have type 'const char*'");
      return;
   }
   if (ast_VarDecl_hasAutoAttr(v)) {
      module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "printf_format parameter cannot be an auto-argument");
      return;
   }
   if (!ast_FunctionDecl_isVariadic(fd)) {
      module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "printf_format functions must have a variable number of arguments");
      return;
   }
   uint32_t num_params = ast_FunctionDecl_getNumParams(fd);
   if ((idx != (num_params - 1))) {
      module_analyser_Analyser_error(ma, ast_Decl_getLoc(d), "printf_format parameter must be the last parameter)");
      return;
   }
}

static bool module_analyser_Analyser_analyseInitExpr(module_analyser_Analyser* ma, ast_Expr** e_ptr, ast_QualType expectedType, src_loc_SrcLoc assignLoc)
{
   ast_Expr* e = *e_ptr;
   if (ast_Expr_isInitList(e)) {
      return module_analyser_Analyser_analyseInitListExpr(ma, ((ast_InitListExpr*)(e)), expectedType);
   }
   if (ast_Expr_isStringLiteral(e)) {
      ast_ArrayType* at = ast_QualType_getArrayTypeOrNil(&expectedType);
      if (at) {
         ast_QualType elem = ast_ArrayType_getElemType(at);
         if (((ast_QualType_getTypeOrNil(&elem) != ast_QualType_getTypeOrNil(&ast_builtins[ast_BuiltinKind_Char])) && (ast_QualType_getTypeOrNil(&elem) != ast_QualType_getTypeOrNil(&ast_builtins[ast_BuiltinKind_Int8])))) {
            module_analyser_Analyser_errorRange(ma, assignLoc, ast_Expr_getRange(e), "cannot initialize array of '%s' with a string literal", ast_QualType_diagName(&elem));
            return false;
         }
         ast_QualType st = ast_Expr_getType(e);
         ast_ArrayType* at2 = ast_QualType_getArrayType(&st);
         uint32_t rhs_len = ast_ArrayType_getSize(at2);
         if (ast_ArrayType_hasSize(at)) {
            uint32_t lhs_len = ast_ArrayType_getSize(at);
            if ((rhs_len > lhs_len)) {
               module_analyser_Analyser_errorRange(ma, assignLoc, ast_Expr_getRange(e), "initializer-string for char array is too long");
               return false;
            }
         } else {
            ast_ArrayType_setSize(at, rhs_len);
         }
         ast_Expr_setRValue(e);
      } else {
         ast_QualType qt = module_analyser_Analyser_analyseExpr(ma, e_ptr, true, module_analyser_RHS);
         e = *e_ptr;
         if (!conversion_checker_Checker_check(&ma->checker, expectedType, ast_Expr_getType(e), e_ptr, assignLoc)) return false;

      }
      return true;
   }
   if ((ast_QualType_isArray(&expectedType) && !ast_Expr_isInitList(e))) {
      module_analyser_Analyser_error(ma, assignLoc, "array initializer must be an initializer list");
      return false;
   }
   if ((ast_Expr_isCall(e) && module_analyser_Analyser_globalScope(ma))) {
      ast_CallExpr* c = ((ast_CallExpr*)(e));
      ast_QualType qt = module_analyser_Analyser_analysePureCallExpr(ma, e);
      return true;
   }
   ast_QualType res = module_analyser_Analyser_analyseExpr(ma, e_ptr, true, module_analyser_RHS);
   if (ast_QualType_isInvalid(&res)) return false;

   e = *e_ptr;
   if (((ast_Expr_isCtv(e) && ast_QualType_isBuiltin(&expectedType)) && !ast_QualType_isPointer(&res))) {
      if (ast_QualType_isBool(&expectedType)) return true;

      return ctv_analyser_check(ma->diags, expectedType, e);
   }
   if (!conversion_checker_Checker_check(&ma->checker, expectedType, res, e_ptr, assignLoc)) return false;

   if ((!ma->curFunction && !ast_Expr_isCtv(e))) {
      if (!ast_Expr_isCtc(e)) {
         module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(e), ast_Expr_getRange(e), "initializer element is not a compile-time constant");
         return false;
      }
      if (ast_QualType_needsCtvInit(&expectedType)) {
         module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(e), ast_Expr_getRange(e), "initializer element is not a compile-time value");
         return false;
      }
   }
   return true;
}

static bool module_analyser_Analyser_analyseInitListExpr(module_analyser_Analyser* ma, ast_InitListExpr* ile, ast_QualType expectedType)
{
   const ast_Type* t = ast_QualType_getTypeOrNil(&expectedType);
   if (ast_Type_isArrayType(t)) {
      return module_analyser_Analyser_analyseInitListArray(ma, ile, expectedType);
   }
   if (ast_Type_isStructType(t)) {
      return module_analyser_Analyser_analyseInitListStruct(ma, ile, expectedType);
   }
   ast_Expr* e = ((ast_Expr*)(ile));
   module_analyser_Analyser_error(ma, ast_Expr_getLoc(e), "cannot initialize variable of type '%s' with initializer list", ast_QualType_diagName(&expectedType));
   return false;
}

static bool module_analyser_Analyser_analyseArrayDesignatedInit(module_analyser_Analyser* ma, ast_Expr* e, ast_QualType expectedType)
{
   ast_ArrayDesignatedInitExpr* ad = ((ast_ArrayDesignatedInitExpr*)(e));
   ast_QualType qt = module_analyser_Analyser_analyseExpr(ma, ast_ArrayDesignatedInitExpr_getDesignator2(ad), false, module_analyser_RHS);
   if (ast_QualType_isInvalid(&qt)) return false;

   ast_Expr* de = ast_ArrayDesignatedInitExpr_getDesignator(ad);
   if (!ast_Expr_isCtv(de)) {
      module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(de), ast_Expr_getRange(de), "array index is not a compile-time value");
      return false;
   }
   ast_Expr* val = ast_ArrayDesignatedInitExpr_getInit(ad);
   if (ast_Expr_isInitList(val)) {
      return module_analyser_Analyser_analyseInitListExpr(ma, ((ast_InitListExpr*)(val)), expectedType);
   }
   bool ok = module_analyser_Analyser_analyseInitExpr(ma, ast_ArrayDesignatedInitExpr_getInit2(ad), expectedType, ast_Expr_getLoc(val));
   if (!ok) return false;

   if (!ma->curFunction) {
      if (!ast_Expr_isCtc(val)) {
         module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(val), ast_Expr_getRange(val), "initializer element is not a compile-time constant");
         return false;
      }
      if ((!ast_Expr_isCtv(val) && ast_QualType_needsCtvInit(&expectedType))) {
         module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(val), ast_Expr_getRange(val), "initializer element is not a compile-time value");
         return false;
      }
   }
   val = ast_ArrayDesignatedInitExpr_getInit(ad);
   ast_Expr_copyConstantFlags(e, val);
   ast_Expr_setType(e, expectedType);
   return true;
}

static bool module_analyser_Analyser_analyseInitListArray(module_analyser_Analyser* ma, ast_InitListExpr* ile, ast_QualType expectedType)
{
   ast_Expr* e = ((ast_Expr*)(ile));
   uint32_t numValues = ast_InitListExpr_getNumValues(ile);
   ast_Expr** values = ast_InitListExpr_getValues(ile);
   ast_QualType_clearQuals(&expectedType);
   ast_ArrayType* at = ast_QualType_getArrayType(&expectedType);
   ast_QualType et = ast_ArrayType_getElemType(at);
   ast_QualType_clearQuals(&et);
   bool ok = true;
   bool ctc = true;
   bool have_designators = false;
   uint32_t current_index = 0;
   for (uint32_t i = 0; (i < numValues); i++) {
      ast_Expr* value = values[i];
      if (ast_Expr_isFieldDesignatedInit(value)) {
         module_analyser_Analyser_errorRange(ma, ast_Expr_getLoc(value), ast_Expr_getRange(value), "field designator cannot initialize an array");
         ok = false;
         continue;
      }
      if (ast_Expr_isArrayDesignatedInit(value)) {
         ok &= module_analyser_Analyser_analyseArrayDesignatedInit(ma, value, et);
         have_designators = true;
         continue;
      }
      ok &= module_analyser_Analyser_analyseInitExpr(ma, &values[i], et, ast_Expr_getLoc(values[i]));
      ctc &= ast_Expr_isCtc(values[i]);
   }
   current_index = numValues;
   if (ctc) ast_Expr_setCtc(e);
   if (!ok) return false;

   if (have_designators) {
      int32_t array_size = -1;
      if (ast_ArrayType_hasSize(at)) array_size = ((int32_t)(ast_ArrayType_getSize(at)));
      init_checker_Checker checker = init_checker_Checker_create(numValues);
      ok = module_analyser_Analyser_checkArrayDesignators(ma, ile, &array_size, &checker);
      init_checker_Checker_free(&checker);
      if (!ast_ArrayType_hasSize(at)) ast_ArrayType_setSize(at, ((uint32_t)(array_size)));
   } else {
      if (ast_ArrayType_hasSize(at)) {
         uint32_t arraySize = ast_ArrayType_getSize(at);
         if ((current_index > arraySize)) {
            module_analyser_Analyser_error(ma, ast_Expr_getLoc(values[arraySize]), "excess elements in array initializer");
            return false;
         }
      } else {
         ast_ArrayType_setSize(at, numValues);
      }
   }
   ast_Expr_setType(e, expectedType);
   return ok;
}

static bool module_analyser_Analyser_checkArrayDesignators(module_analyser_Analyser* ma, ast_InitListExpr* ile, int32_t* size, init_checker_Checker* checker)
{
   uint32_t numValues = ast_InitListExpr_getNumValues(ile);
   ast_Expr** values = ast_InitListExpr_getValues(ile);
   int32_t max_index = 0;
   int32_t current_index = -1;
   for (uint32_t i = 0; (i < numValues); i++) {
      src_loc_SrcLoc loc;
      ast_Expr* value = values[i];
      if (ast_Expr_isArrayDesignatedInit(value)) {
         ast_ArrayDesignatedInitExpr* ad = ((ast_ArrayDesignatedInitExpr*)(value));
         ast_Expr* desig = ast_ArrayDesignatedInitExpr_getDesignator(ad);
         loc = ast_Expr_getLoc(desig);
         ast_Value idx = ctv_analyser_get_value(desig);
         if (ast_Value_isNegative(&idx)) {
            module_analyser_Analyser_error(ma, loc, "array designator value '%s' is negative", ast_Value_str(&idx));
            return false;
         }
         current_index = ((int32_t)(idx.uvalue));
         if (((*size != -1) && (current_index >= *size))) {
            module_analyser_Analyser_error(ma, loc, "array designator index (%d) exceeds array bounds (%d)", current_index, *size);
            return false;
         }
      } else {
         loc = ast_Expr_getLoc(value);
         current_index++;
      }
      if (((*size != -1) && (current_index >= *size))) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(value), "excess elements in array initializer");
         return false;
      }
      src_loc_SrcLoc duplicate = init_checker_Checker_find(checker, ((uint32_t)(current_index)));
      if (duplicate) {
         module_analyser_Analyser_error(ma, loc, "duplicate initialization of array index");
         module_analyser_Analyser_note(ma, duplicate, "previous initialization is here");
      } else {
         init_checker_Checker_add(checker, ((uint32_t)(current_index)), loc);
      }
      if ((current_index > max_index)) max_index = current_index;
   }
   if ((*size == -1)) *size = (max_index + 1);
   return true;
}

static bool module_analyser_Analyser_analyseInitListStruct(module_analyser_Analyser* ma, ast_InitListExpr* ile, ast_QualType expectedType)
{
   ast_Expr* e = ((ast_Expr*)(ile));
   uint32_t numValues = ast_InitListExpr_getNumValues(ile);
   ast_Expr** values = ast_InitListExpr_getValues(ile);
   ast_StructType* st = ast_QualType_getStructType(&expectedType);
   ast_StructTypeDecl* std = ast_StructType_getDecl(st);
   if ((numValues == 0)) {
      ast_Expr_setType(e, expectedType);
      return true;
   }
   const bool haveDesignators = (ast_Expr_isFieldDesignatedInit(values[0]));
   if ((!haveDesignators && ast_StructTypeDecl_isUnion(std))) {
      module_analyser_Analyser_error(ma, ast_Expr_getLoc(values[0]), "union member initializer needs field designator");
      return false;
   }
   const uint32_t num_members = ast_StructTypeDecl_getNumMembers(std);
   ast_Decl** members = ast_StructTypeDecl_getMembers(std);
   for (uint32_t i = 0; (i < numValues); i++) {
      ast_Expr* value = values[i];
      if ((i >= num_members)) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(value), "excess initializer elements in struct");
         return false;
      }
      if (ast_Expr_isArrayDesignatedInit(value)) {
         module_analyser_Analyser_error(ma, ast_Expr_getLoc(value), "array designator cannot initialize non-array type '%s'", ast_QualType_diagName(&expectedType));
         return false;
      }
      bool is_designator = ast_Expr_isFieldDesignatedInit(value);
      if ((haveDesignators != is_designator)) {
      }
      if (is_