% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Evaluate.R
\name{evaluateROC}
\alias{evaluateROC}
\title{Receiver operator characteristics of simulation results}
\usage{
evaluateROC(simRes,
alpha.type=c("adjusted","raw"),
MTC=c('BY', 'BH', 'Storey', 'IHW',
'holm', 'hochberg', 'hommel', 'bonferroni'),
alpha.nominal = 0.1,
target.by=c("lfc", "effectsize"), delta=0,
raw = FALSE, verbose = TRUE)
}
\arguments{
\item{simRes}{The result from \code{\link{simulateDE}}.}

\item{alpha.type}{A string to represent the way to call DE genes.
Available options are \code{"adjusted"} i.e. applying multiple testing correction and
\code{"raw"} i.e. using p-values. Default is \code{"adjusted"}.}

\item{MTC}{Multiple testing correction method to use. Available options are
1) see \link[stats]{p.adjust.methods},
2) Storey's qvalue see \link[qvalue]{qvalue} and
3) Independent Hypothesis Weighting considering mean expression as covariate (see \link[IHW]{ihw}).
Default is \code{BY}, i.e. Benjamini-Yekutieli FDR correction method.}

\item{alpha.nominal}{The nomial value of significance. Default is 0.1.}

\item{target.by}{A string to specify the method to define "biologically important" DE genes.
Available options are (1) \code{"lfc"}: interesting genes are defined by absolute log2 fold changes.
(2) \code{"effectsize"}: interesting genes are defined by
absolute log2 fold changes divided by the square root of 1/(mean+dispersion).}

\item{delta}{A threshold used for defining "biologically important" genes.
Genes with absolute log2 fold changes (when target.by is "lfc")
or effect sizes (when target.by is "effectsize") greater than this value
are deemed DE in error rates calculations.
If the default \code{delta=0}, then no threshold is applied.}

\item{raw}{Logical vector. The default \code{FALSE} removes
intermediate calculations from the output since they can be quite large.}

\item{verbose}{Logical vector to indicate whether to show progress report of evaluation.
Default is \code{TRUE}.}
}
\value{
A list with the following entries:
\item{Performances}{The output of \code{\link[iCOBRA]{calculate_performance}} of aspect "fdrtprcurve" calculating the proportions of TN, FN, TP and FP and related rates which uses \code{\link[ROCR]{prediction}} and \code{\link[ROCR]{performance}} internally.}
\item{TPRvsFDR}{The output of \code{\link[iCOBRA]{calculate_performance}} of aspect "fdrtpr" calculating the proportions of TN, FN, TP and FP and associated TPR and observed FDR for each nominal level from 0.01 to 1 in steps of 0.01.}
\item{Scores}{The mean +/- standard error of performance measures and area under the curve. These are calculated once for conservative (extension "conv") and once for liberal nominal (extension "lib") FDR levels. Measures include: accuracy (ACC), F-measure of precision and recall (F1score), Matthew's correlation coefficient (MCC), partial Area under the Curve of TPR vs FDR (TPRvsFDR_pAUC), area under the ROC-curve (TPRvsFPR_AUC) and area under the PR-curve (TPRvsPPV_AUC).}
\item{Raw}{If \code{raw} is \code{TRUE}, then the intermediate calculations of the above three list entries is also included in the output.}
\item{Settings}{Reiterating chosen evaluation parameters.}
}
\description{
This function takes the simulation output from \code{\link{simulateDE}}
and computes receiver operator characteristics (ROC) and area under the curve values (AUC) with the help of functions implemented in \code{\link[iCOBRA]{calculate_performance}}.
}
\examples{
\dontrun{
# estimate gene parameters
data("SmartSeq2_Gene_Read_Counts")
estparam_gene <- estimateParam(countData = SmartSeq2_Gene_Read_Counts,
                               readData = NULL,
                              batchData = NULL,
                               spikeData = NULL, spikeInfo = NULL,
                               Lengths = NULL, MeanFragLengths = NULL,
                               RNAseq = 'singlecell', Protocol = 'Read',
                               Distribution = 'ZINB', Normalisation = "scran",
                               GeneFilter = 0.1, SampleFilter = 3,
                               sigma = 1.96, NCores = NULL, verbose = TRUE)
# define log2 fold change
p.lfc <- function(x) sample(c(-1,1), size=x,replace=T)*rgamma(x, shape = 1, rate = 2)
# set up simulations
setupres <- Setup(ngenes = 10000, nsims = 10,
                  p.DE = 0.1, pLFC = p.lfc,
                  n1 = c(20,50,100), n2 = c(30,60,120),
                  Thinning = c(1,0.9,0.8), LibSize = 'given',
                  estParamRes = estparam_gene,
                  estSpikeRes = NULL,
                  DropGenes = TRUE,
                  sim.seed = 83596, verbose = TRUE)
# run simulation
simres <- simulateDE(SetupRes = setupres,
                     Prefilter = "FreqFilter", Imputation = NULL,
                     Normalisation = 'scran', Label = 'none',
                     DEmethod = "limma-trend", DEFilter = FALSE,
                     NCores = NULL, verbose = TRUE)
# evaluation
evalrocres <- evaluateROC(simRes = simres,
                          alpha.type = "adjusted",
                          MTC = 'BH', alpha.nominal = 0.05,
                          raw = FALSE)
# plot evaluation
plotEvalROC(evalRes = evalrocres)
}
}
\seealso{
\code{\link{estimateParam}} for parameter estimation needed for simulations,
\code{\link{Setup}} for setting up simulation parameters and
\code{\link{simulateDE}} for simulating differential expression.
}
\author{
Beate Vieth
}
