package models

import (
	"errors"
	"html"
	"log"
	"net/mail"
	"strings"
	"time"

	"github.com/badoux/checkmail"
	"golang.org/x/crypto/bcrypt"
	"gorm.io/gorm"
)

type User struct {
	Id        int        `gorm:"primary_key;auto_increment" json:"id"`
	FirstName string     `gorm:"size:255;not null;" json:"firstname"`
	LastName  string     `gorm:"size:255;not null;" json:"lastname"`
	Email     string     `gorm:"size:100;not null;unique" json:"email"`
	Password  string     `gorm:"size:100;not null;" json:"password"`
	CreatedAt time.Time  `gorm:"default:CURRENT_TIMESTAMP" json:"created_at"`
	UpdatedAt time.Time  `gorm:"default:CURRENT_TIMESTAMP" json:"updated_at"`
	DeletedAt *time.Time `gorm:"default:null" json:"deleted_at"`
	GroupId   *int       `gorm:"default:null" json:"group_id"`
	Group     *Group     `gorm:"foreignkey:GroupId"`
}

func Hash(password string) ([]byte, error) {
	return bcrypt.GenerateFromPassword([]byte(password), bcrypt.DefaultCost)
}

func (u *User) Setup() {
	u.Id = 0
	u.FirstName = html.EscapeString(strings.TrimSpace(u.FirstName))
	u.LastName = html.EscapeString(strings.TrimSpace(u.LastName))
	u.Email = html.EscapeString(strings.TrimSpace(u.Email))
	u.CreatedAt = time.Now()
	u.UpdatedAt = time.Now()
	u.DeletedAt = nil
}

func (u *User) Validate(action string) error {
	switch strings.ToLower(action) {

	case "update":
		if u.FirstName == "" {
			return errors.New("Required User First Name")
		}
		if u.LastName == "" {
			return errors.New("Required User Last Name")
		}
		if u.Password == "" {
			return errors.New("Required User Password")
		}
		if u.Email == "" {
			return errors.New("Required User Email")
		}
		if _, err := mail.ParseAddress(u.Email); err != nil {
			return errors.New("Invalid Email Address")
		}

		return nil

	case "login":
		if u.Password == "" {
			return errors.New("Required Password")
		}
		if u.Email == "" {
			return errors.New("Required Email")
		}
		if err := checkmail.ValidateFormat(u.Email); err != nil {
			return errors.New("Invalid Email")
		}
		return nil

	default:
		if u.FirstName == "" {
			return errors.New("Required User First Name")
		}
		if u.LastName == "" {
			return errors.New("Required User Last Name")
		}
		if u.Password == "" {
			return errors.New("Required User Password")
		}
		if u.Email == "" {
			return errors.New("Required User Email")
		}
		if _, err := mail.ParseAddress(u.Email); err != nil {
			return errors.New("Invalid Email Address")
		}
		return nil
	}
}

func VerifyPassword(hashedPassword, password string) error {
	return bcrypt.CompareHashAndPassword([]byte(hashedPassword), []byte(password))
}

func (u *User) SecurePassword() error {
	hashedPassword, err := Hash(u.Password)
	if err != nil {
		return err
	}
	u.Password = string(hashedPassword)
	return nil
}

func (u *User) FindAllUsers(db *gorm.DB) (*[]User, error) {
	var err error

	users := []User{}
	err = db.Debug().Model(&User{}).Limit(100).Find(&users).Error
	if err != nil {
		return &[]User{}, err
	}
	return &users, err
}

func (u *User) SaveUser(db *gorm.DB) (*User, error) {
	// Secure the password
	err := u.SecurePassword()
	if err != nil {
		log.Fatal(err)
	}

	err = db.Debug().Create(&u).Error
	if err != nil {
		return &User{}, err
	}
	return u, nil
}

func (u *User) UpdateAUser(db *gorm.DB, uid uint32) (*User, error) {
	// Secure the password
	err := u.SecurePassword()
	if err != nil {
		log.Fatal(err)
	}
	db = db.Debug().Model(&User{}).Where("id = ?", uid).Take(&User{}).UpdateColumns(
		map[string]interface{}{
			"password":   u.Password,
			"first_name": u.FirstName,
			"last_name":  u.LastName,
			"group_id":   u.GroupId,
			"email":      u.Email,
			"updated_at": time.Now(),
		},
	)
	if db.Error != nil {
		return &User{}, db.Error
	}
	// This is the display the updated user
	err = db.Debug().Model(&User{}).Where("id = ?", uid).Take(&u).Error
	if err != nil {
		return &User{}, err
	}
	return u, nil
}

func (u *User) FindUserByID(db *gorm.DB, uid uint32) (*User, error) {
	var err error
	err = db.Debug().Model(User{}).Where("id = ?", uid).Take(&u).Error
	if err != nil {
		return &User{}, err
	}
	if errors.Is(db.Error, gorm.ErrRecordNotFound) {
		return &User{}, errors.New("User Not Found")
	}
	return u, err
}

func (u *User) DeleteUserByID(db *gorm.DB, uid uint32) (int64, error) {

	db = db.Debug().Model(&User{}).Where("id = ?", uid).Take(&User{}).Delete(&User{})

	if db.Error != nil {
		return 0, db.Error
	}
	return db.RowsAffected, nil
}
