import { Stack } from './stack';

export interface GameConfig {
  /**
   * Selector for the root element.
   * This element should be of type `HTMLCanvasElement`.
   */
  selector: string;
  /**
   * Height of the rendering canvas in pixels.
   */
  height?: number;
  /**
   * Width of the rendering canvas in pixels.
   */
  width?: number;

  /**
   * Full screen canvas
   */
  fullscreen?: boolean;
}

export class Game {
  /**
   * Rendering
   */
  public readonly canvas: HTMLCanvasElement;
  public readonly ctx: CanvasRenderingContext2D;

  /**
   * Frame data
   */
  private time: number = 0;

  constructor({ selector, height, width }: GameConfig) {
    this.canvas = document.querySelector(selector);
    this.ctx = this.canvas.getContext('2d');

    if (!height || !width) {
      throw new Error(
        'height and width must be provided if fullscreen is false.'
      );
    }
    this.canvas.height = height;
    this.canvas.width = width;
  }

  public start() {
    this.tick(this.time);
  }

  public tick(time: number) {
    const delta = time - this.time;
    this.time = time;

    Stack.instance.resolve();
    Stack.instance.top?.update?.({ time, delta });

    for (let i = 0; i < Stack.instance.size; i++) {
      const state = Stack.instance.states[i];

      if (typeof state.render === 'function') {
        state.render(this.ctx);
      }
    }

    requestAnimationFrame((t) => this.tick(t));
  }
}
