<p align="center"><img src="images/icon.png" width="96"></p>

<h1 align="center">
  🕹️ State Driven Game Development Starter
</h1>

<p align="center">
  A template repository to kick start modern game development on the web.
</p>

<p align="center">
  <img alt="HTML5" src="https://img.shields.io/badge/html5-%23E34F26.svg?style=for-the-badge&logo=html5&logoColor=white" />
  <img alt="TypeScript" src="https://img.shields.io/badge/typescript-%23007ACC.svg?style=for-the-badge&logo=typescript&logoColor=white" />
  <img alt="NodeJS" src="https://img.shields.io/badge/node.js-6DA55F?style=for-the-badge&logo=node.js&logoColor=white" />
  <img alt="Express.js" src="https://img.shields.io/badge/express.js-%23404d59.svg?style=for-the-badge&logo=express&logoColor=%2361DAFB" />
  <img alt="Webpack" src="https://img.shields.io/badge/webpack-%238DD6F9.svg?style=for-the-badge&logo=webpack&logoColor=black" />
  <img alt="Progressive Web App" src="https://img.shields.io/badge/pwa-%235A0FC8.svg?style=for-the-badge&logo=pwa&logoColor=white" />
</p>

---

- [Quick Start](#quick-start)
- [Overview](#overview)
- [Project Structure](#project-structure)
- [Game States](#game-states)
- [Sprites](#sprites)
- [Example Projects](#examples)

---

<h2 id="quick-start">🚀 Quick Start</h2>

### 🛑 Prerequisites

- `node` >= 14
- `npm` >= 6

### 🖥️ Local Development

Install dependencies.

```bash
npm i
```

Start the development server.

```bash
npm run start
```

See your game by opening your browser at `localhost:5000`.

![Example screen](images/screenshot.PNG)

Changes made to the source files will trigger a recompilation. Refresh the browser to see the changes.

### 🛠️ Building

Building the project.

```bash
npm run build
```

The output will be located at `/dist`.

---

<h2 id="project-structure">📚 Project Structure</h2>

```text
dist                          [Build output]
src                           [Source root]
  /assets                     [Web assets]
  /icons                      [Web icons]
  /lib                        [Game libraries]
    /core                     [Core libraries - @core]
    /models                   [Game models - @models]
    /sprites                  [Sprite definitions - @sprites]
    /state                    [State definitions - @state]
    /utils                    [Utility libraries - @utils]
    /game.ts                  [Game entry point - @game]
  /index.html                 [Web index page]
  /main.ts                    [App entry point]
  /manifest.webmanifest       [PWA manifest]
  /service-worker.js          [PWA service worker]
dev-server.js                 [Development express server]
tsconfig.json                 [TypeScript configuration]
tslint.json                   [tslint configuration]
webpack.config.js             [Webpack configuration]
```

<h2 id="game-states">🧩 Game States</h2>

Game states provide frame by frame functionality for your game.

### Creating States

```ts
import { OnEnter, OnExit, OnRender, OnUpdate } from '@core';

export class MyGameState implements OnEnter, OnExit, OnRender, OnUpdate {
  enter(): void {
    console.log('Pushed to the stack!');
  }

  exit(): void {
    console.log('Popped from the stack!');
  }

  render(ctx: CanvasRenderingContext2D): void {
    console.log('Rendering frame!');
  }

  update({ time, delta }): void {
    console.log('Updating frame!');
  }
}
```

### Using States

To advance your game to the next state, simply `push` an instance of your state to the game stack.

```ts
import { push } from '@core';

// Push a new state to the stack
push(new MyGameState());
```

States can also be popped from the stack as long as it has states to `pop`.

```ts
import { pop } from '@core';

// Remove top-most state
pop();
```

<h2 id="sprites">👾 Sprites</h2>

Sprites are objects that render an image on a `HTMLCanvasContext2D`.

A sprite can be created in the following way.

```ts
import { Sprite } from '@core';

// Create and load the image to be rendered.
const image = new Image();
image.src = 'path/to/image.png';

// Create a sprite using the image and parameters.
const sprite = new Sprite(image, {
  // Source width
  w: 16,
  // Source height
  h: 16,
  // Origin x position
  ox: 0,
  // Origin y position
  oy: 0,
});
```

The sprite can be rendered to a canvas context.

```ts
sprite.render(ctx, {
  // Rendering x position
  x: 100,
  // Rendering y position
  y: 50,
});
```

Optionally, the sprite can be scaled in both directions.

```ts
sprite.render(ctx, {
  x: 100,
  y: 50,
  // Scale the sprite by 3
  scaleX: 3,
  scaleY: 3,
});
```

### Sprite Sheets

A `SpriteSheet` is used to define multiple `Sprite` objects from a single image.

A `SpriteSheet` is defined as follows.

```ts
const sprites = new SpriteSheet('/path/to/image.png');
```

Define sprites directly on the sprite sheet to build multiple sprites from the same image.

```ts
sprites
  .define('idle_1', { w: 16, h: 16, ox: 0, oy: 0 })
  .define('idle_2', { w: 16, h: 16, ox: 16, oy: 16 });
```

Sprites can then be retrieved from the sprite sheet.

```ts
/**
 * Note that, the `.get` method can return undefined.
 * Accommodate for this by optionally calling render
 * when doing so.
 *
 * `sprite?.render(ctx, config)`
 * */
const sprite = sprites.get('idle_1');
```

<h2 id="examples">💡 Example Projects</h2>

- tic-tac-toe - [[GitHub]](https://github.com/brookesb91/tic-tac-toe)
