/* eslint global-require: off, no-console: off, promise/always-return: off */

/**
 * This module executes inside of electron's main process. You can start
 * electron renderer process from here and communicate with the other processes
 * through IPC.
 *
 * When running `npm run build` or `npm run build:main`, this file is compiled to
 * `./src/main.js` using webpack. This gives us some performance wins.
 */
import { app, BrowserWindow } from 'electron';

import { ConfigCache } from 'caches';

import { createMainWindow } from './windows/main';
import MenuBuilder from './menu';
import { setupSever } from './server';
import { setupTray } from './tray';
import { resolveHtmlPath } from './util';

import 'lib/log';
import './ipcEvents';

if (process.env.NODE_ENV === 'production') {
  const sourceMapSupport = require('source-map-support');
  sourceMapSupport.install();
}

const isDebug =
  process.env.NODE_ENV === 'development' || process.env.DEBUG_PROD === 'true';

if (isDebug) {
  require('electron-debug')();
}

const installExtensions = async () => {
  const installer = require('electron-devtools-installer');
  const forceDownload = !!process.env.UPGRADE_EXTENSIONS;
  const extensions = ['REACT_DEVELOPER_TOOLS'];

  return installer
    .default(
      extensions.map((name) => installer[name]),
      forceDownload
    )
    .catch(console.error);
};

let mainWindow: BrowserWindow | null = null;

const createWindow = async () => {
  if (isDebug) {
    await installExtensions();
  }
  mainWindow = await createMainWindow();

  const route = await ConfigCache.getRoute();

  mainWindow.loadURL(`${resolveHtmlPath(`index.html`)}#${route}`);

  mainWindow.on('ready-to-show', () => {
    if (!mainWindow) {
      throw new Error('"mainWindow" is not defined');
    }
    if (process.env.START_MINIMIZED) {
      mainWindow.minimize();
    } else {
      mainWindow.show();
    }
  });

  mainWindow.on('closed', () => {
    mainWindow = null;
  });

  const menuBuilder = new MenuBuilder(mainWindow);
  menuBuilder.buildMenu();
};

app.on('activate', () => {
  // On macOS it's common to re-create a window in the app when the
  // dock icon is clicked and there are no other windows open.
  if (mainWindow === null) createWindow();
});

app.on('window-all-closed', () => {
  // Respect the OSX convention of having the application in memory even
  // after all windows have been closed
  // if (process.platform !== 'darwin') {
  //   app.quit();
  // }
});

app
  .whenReady()
  .then(() => {
    createWindow();
    setupSever();
    setupTray();
  })
  .catch(console.error);

app.commandLine.appendSwitch('autoplay-policy', 'no-user-gesture-required');
