﻿
namespace TheBoxSoftware.Reflection.Tests.Unit.Signatures
{
    using NUnit.Framework;
    using Reflection.Core.COFF;
    using Reflection.Signatures;

    [TestFixture]
    public class SignitureBuilderTests
    {
        [Test]
        public void OffsetOutOfBounds_Read_ReturnsNull()
        {
            SignatureBuilder builder = CreateBuilder(new byte[5]);

            Signature lower = builder.Read(-1);
            Signature larger = builder.Read(5);

            Assert.AreSame(null, lower);
            Assert.AreSame(null, larger);
        }

        [Test]
        public void OffsetIsOutOfBounds_ReadLength_ThrowsException()
        {
            SignatureBuilder builder = CreateBuilder(new byte[5]);

            Assert.Throws<System.IndexOutOfRangeException>(delegate ()
            {
                builder.GetLength(7);
            });
        }

        [TestCase(new byte[] { 0x03 }, 0x00000003)]
        [TestCase(new byte[] { 0x7F }, 0x0000007F)]
        [TestCase(new byte[] { 0x80, 0x80 }, 0x00000080)]
        [TestCase(new byte[] { 0xae, 0x57 }, 0x00002e57)]
        [TestCase(new byte[] { 0xbf, 0xff }, 0x00003fff)]
        [TestCase(new byte[] { 0xc0, 0x00, 0x40, 0x00 }, 0x00004000)]
        [TestCase(new byte[] { 0xdf, 0xff, 0xff, 0xff }, 0x1fffffff)]
        public void OffsetIsCorrect_ReadLength_GetsLength(byte[] signiture, int expected)
        {
            BlobStream stream = new BlobStream(signiture, 0, signiture.Length);
            SignatureBuilder builder = new SignatureBuilder(stream);
            int offset = 0;

            uint result = builder.GetLength(offset);

            Assert.AreEqual(expected, result);
        }

        [Test]
        public void OffsetIsOutOfBounds_GetSignitureBytes_ThrowsException()
        {
            SignatureBuilder builder = CreateEmptyBuilder();
            Assert.Throws<System.IndexOutOfRangeException>(delegate ()
            {
                builder.GetSignitureBytes(-1);
            });
        }

        [TestCase(1, 3)]
        [TestCase(685, 97)]
        public void SignitureIsValid_GetSignitureBytes_IsCorrect(int offset, int expectedLength)
        {
            SignatureBuilder builder = CreateTestBuilder();

            byte[] result = builder.GetSignitureBytes(offset);

            Assert.AreEqual(expectedLength, result.Length);
        }

        [Test]
        public void SignitureIsMethod_Read_ReturnsMethodSigniture()
        {
            SignatureBuilder builder = CreateTestBuilder();

            Signature result = builder.Read(1);

            Assert.AreEqual(Signatures.MethodDef, result.Type);
        }

        [Test]
        public void SignitureIsProperty_Read_DoesntReturnMethodSigniture()
        {
            byte[] contents = new byte[] { 0x03, 0x28, 0x00, 0x08 };
            SignatureBuilder builder = CreateBuilder(contents);

            Signature result = builder.Read(0);

            Assert.AreNotEqual(Signatures.MethodDef, result.Type);
        }

        private SignatureBuilder CreateTestBuilder() => CreateBuilder(_blobStream);

        private SignatureBuilder CreateEmptyBuilder() => CreateBuilder(new byte[0]);

        private SignatureBuilder CreateBuilder(byte[] contents)
        {
            BlobStream underlyingStream = new BlobStream(contents, 0, contents.Length);
            return new SignatureBuilder(underlyingStream);
        }

        // full blob stream pulled from documentationtest.dll
        private byte[] _blobStream = new byte[] {
            /*   0 */ 0x00, // first entry is always the empty blob
            /*   1 */ 0x03, 0x20, 0x00, 0x01,
            /*   5 */ 0x04, 0x20, 0x01, 0x01, 0x08,
            /*  10 */ 0x05, 0x20, 0x01, 0x01, 0x11, 0x15,
            /*  16 */ 0x04, 0x20, 0x01, 0x01, 0x0E,
            /*  21 */ 0x04, 0x20, 0x01, 0x01, 0x02,
            /*  26 */ 0x05, 0x20, 0x01, 0x01, 0x11, 0x45,
            /*  32 */ 0x05, 0x20, 0x01, 0x01, 0x11, 0x71,
            /*  38 */ 0x06, 0x15, 0x12, 0x80, 0x9C, 0x01, 0x0E,
            /*  45 */ 0x06, 0x15, 0x12, 0x80, 0x9C, 0x01, 0x08,
            /*  52 */ 0x0A, 0x15, 0x12, 0x80, 0x9C, 0x01, 0x15, 0x12, 0x61, 0x01, 0x08,
            /*  63 */ 0x0F, 0x15, 0x12, 0x80, 0x9C, 0x01, 0x15, 0x12, 0x61, 0x01, 0x15, 0x12, 0x80, 0x9C, 0x01, 0x0E,
            /*  79 */ 0x06, 0x20, 0x01, 0x01, 0x11, 0x80, 0x8D,
            /*  86 */ 0x02, 0x1D, 0x08,
            /*  89 */ 0x06, 0x15, 0x12, 0x61, 0x01, 0x1D, 0x0E,
            /*  96 */ 0x03, 0x20, 0x00, 0x08,
            /* 100 */ 0x05, 0x15, 0x12, 0x61, 0x01, 0x08,
            /* 106 */ 0x06, 0x15, 0x12, 0x61, 0x01, 0x1E, 0x00,
            0x0B, 0x07, 0x03, 0x12, 0x81, 0x8C, 0x12, 0x81, 0x8C, 0x12, 0x81, 0x8C, 0x0B, 0x00, 0x02,
            0x12, 0x80, 0xB5, 0x12, 0x80, 0xB5, 0x12, 0x80, 0xB5, 0x0C, 0x10, 0x01, 0x03, 0x1E, 0x00, 0x10,
            0x1E, 0x00, 0x1E, 0x00, 0x1E, 0x00, 0x05, 0x0A, 0x01, 0x12, 0x81, 0x8C, 0x02, 0x1E, 0x00, 0x03,
            0x20, 0x00, 0x0E, 0x06, 0x15, 0x12, 0x64, 0x01, 0x13, 0x00, 0x02, 0x06, 0x0E, 0x06, 0x06, 0x15,
            0x12, 0x64, 0x01, 0x0E, 0x06, 0x15, 0x12, 0x7D, 0x02, 0x08, 0x02, 0x05, 0x20, 0x02, 0x01, 0x1C,
            0x18, 0x18, 0x10, 0x01, 0x02, 0x15, 0x12, 0x80, 0xA1, 0x01, 0x1E, 0x00, 0x15, 0x12, 0x80, 0xA1,
            0x01, 0x1E, 0x00, 0x15, 0x12, 0x7D, 0x02, 0x1E, 0x00, 0x02, 0x03, 0x0A, 0x01, 0x08, 0x13, 0x10,
            0x01, 0x02, 0x1E, 0x00, 0x15, 0x12, 0x80, 0xA1, 0x01, 0x1E, 0x00, 0x15, 0x12, 0x7D, 0x02, 0x1E,
            0x00, 0x02, 0x02, 0x13, 0x00, 0x05, 0x07, 0x01, 0x11, 0x80, 0xD0, 0x05, 0x07, 0x01, 0x11, 0x80,
            0xD4, 0x0B, 0x07, 0x03, 0x12, 0x80, 0x9D, 0x12, 0x80, 0x9D, 0x12, 0x80, 0x9D, 0x05, 0x0A, 0x01,
            0x12, 0x80, 0x9D, 0x05, 0x15, 0x12, 0x61, 0x01, 0x0E, 0x05, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x19,
            0x10, 0x02, 0x02, 0x15, 0x12, 0x80, 0xA1, 0x01, 0x1E, 0x01, 0x15, 0x12, 0x80, 0xA1, 0x01, 0x1E,
            0x00, 0x15, 0x12, 0x7D, 0x02, 0x1E, 0x00, 0x1E, 0x01, 0x04, 0x0A, 0x02, 0x08, 0x02, 0x07, 0x15,
            0x12, 0x81, 0x74, 0x01, 0x13, 0x00, 0x03, 0x06, 0x13, 0x00, 0x08, 0xB7, 0x7A, 0x5C, 0x56, 0x19,
            0x34, 0xE0, 0x89, 0x08, 0x1F, 0x85, 0xEB, 0x51, 0xB8, 0x1E, 0x09, 0x40, 0x10, 0x43, 0x00, 0x6F,
            0x00, 0x6E, 0x00, 0x73, 0x00, 0x74, 0x00, 0x61, 0x00, 0x6E, 0x00, 0x74, 0x00, 0x04, 0x03, 0x00,
            0x00, 0x00, 0x04, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x80, 0x9E,
            0x2E, 0x01, 0x80, 0x84, 0x53, 0x79, 0x73, 0x74, 0x65, 0x6D, 0x2E, 0x53, 0x65, 0x63, 0x75, 0x72,
            0x69, 0x74, 0x79, 0x2E, 0x50, 0x65, 0x72, 0x6D, 0x69, 0x73, 0x73, 0x69, 0x6F, 0x6E, 0x73, 0x2E,
            0x53, 0x65, 0x63, 0x75, 0x72, 0x69, 0x74, 0x79, 0x50, 0x65, 0x72, 0x6D, 0x69, 0x73, 0x73, 0x69,
            0x6F, 0x6E, 0x41, 0x74, 0x74, 0x72, 0x69, 0x62, 0x75, 0x74, 0x65, 0x2C, 0x20, 0x6D, 0x73, 0x63,
            0x6F, 0x72, 0x6C, 0x69, 0x62, 0x2C, 0x20, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x34,
            0x2E, 0x30, 0x2E, 0x30, 0x2E, 0x30, 0x2C, 0x20, 0x43, 0x75, 0x6C, 0x74, 0x75, 0x72, 0x65, 0x3D,
            0x6E, 0x65, 0x75, 0x74, 0x72, 0x61, 0x6C, 0x2C, 0x20, 0x50, 0x75, 0x62, 0x6C, 0x69, 0x63, 0x4B,
            0x65, 0x79, 0x54, 0x6F, 0x6B, 0x65, 0x6E, 0x3D, 0x62, 0x37, 0x37, 0x61, 0x35, 0x63, 0x35, 0x36,
            0x31, 0x39, 0x33, 0x34, 0x65, 0x30, 0x38, 0x39, 0x15, 0x01, 0x54, 0x02, 0x10, 0x53, 0x6B, 0x69,
            0x70, 0x56, 0x65, 0x72, 0x69, 0x66, 0x69, 0x63, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x01, 0x02, 0x2A,
            0x50, 0x01, 0x02, 0x01, 0x07, 0x01, 0x1F, 0x02, 0x06, 0x08, 0x02, 0x06, 0x0D, 0x04, 0x06, 0x12,
            0x81, 0x8C, 0x03, 0x06, 0x1D, 0x0E, 0x02, 0x06, 0x02, 0x04, 0x06, 0x12, 0x80, 0x9D, 0x04, 0x06,
            0x12, 0x81, 0x90, 0x07, 0x06, 0x15, 0x12, 0x7D, 0x02, 0x08, 0x02, 0x04, 0x06, 0x11, 0x81, 0x94,
            0x02, 0x06, 0x05, 0x04, 0x06, 0x11, 0x81, 0x98, 0x02, 0x06, 0x04, 0x02, 0x06, 0x06, 0x02, 0x06,
            0x07, 0x02, 0x06, 0x09, 0x02, 0x06, 0x0A, 0x04, 0x06, 0x12, 0x82, 0x14, 0x05, 0x20, 0x00, 0x1D,
            0x1D, 0x08, 0x04, 0x20, 0x00, 0x12, 0x10, 0x04, 0x20, 0x00, 0x1D, 0x05, 0x08, 0x20, 0x00, 0x15,
            0x12, 0x80, 0x9C, 0x01, 0x0E, 0x03, 0x00, 0x00, 0x01, 0x05, 0x20, 0x01, 0x01, 0x10, 0x08, 0x04,
            0x30, 0x01, 0x00, 0x01, 0x08, 0x20, 0x03, 0x08, 0x0E, 0x10, 0x08, 0x0F, 0x01, 0x09, 0x20, 0x03,
            0x08, 0x0E, 0x10, 0x08, 0x0F, 0x11, 0x60, 0x09, 0x20, 0x01, 0x01, 0x15, 0x12, 0x61, 0x01, 0x1D,
            0x0E, 0x07, 0x20, 0x03, 0x01, 0x10, 0x08, 0x08, 0x08, 0x0C, 0x20, 0x02, 0x08, 0x1D, 0x06, 0x14,
            0x08, 0x02, 0x00, 0x02, 0x00, 0x00, 0x04, 0x20, 0x00, 0x12, 0x5C, 0x03, 0x20, 0x00, 0x0A, 0x04,
            0x20, 0x00, 0x1D, 0x08, 0x07, 0x20, 0x00, 0x15, 0x12, 0x61, 0x01, 0x08, 0x09, 0x30, 0x01, 0x00,
            0x15, 0x12, 0x61, 0x01, 0x1E, 0x00, 0x09, 0x20, 0x04, 0x01, 0x08, 0x10, 0x08, 0x0E, 0x12, 0x5C,
            0x08, 0x00, 0x02, 0x12, 0x5C, 0x12, 0x5C, 0x12, 0x5C, 0x06, 0x20, 0x01, 0x01, 0x12, 0x81, 0x8C,
            0x04, 0x20, 0x01, 0x08, 0x0E, 0x05, 0x20, 0x02, 0x08, 0x0E, 0x08, 0x05, 0x00, 0x01, 0x08, 0x12,
            0x5C, 0x06, 0x30, 0x01, 0x01, 0x01, 0x1E, 0x00, 0x08, 0x20, 0x01, 0x1D, 0x1D, 0x0E, 0x1D, 0x1D,
            0x0E, 0x04, 0x20, 0x00, 0x1D, 0x0E, 0x05, 0x20, 0x01, 0x01, 0x1D, 0x0E, 0x07, 0x20, 0x00, 0x15,
            0x12, 0x64, 0x01, 0x0E, 0x08, 0x20, 0x01, 0x01, 0x15, 0x12, 0x64, 0x01, 0x0E, 0x04, 0x20, 0x01,
            0x0E, 0x08, 0x04, 0x20, 0x01, 0x0E, 0x0E, 0x03, 0x20, 0x00, 0x02, 0x06, 0x20, 0x01, 0x08, 0x10,
            0x1D, 0x0E, 0x05, 0x20, 0x02, 0x02, 0x0E, 0x0E, 0x08, 0x20, 0x03, 0x12, 0x69, 0x0E, 0x12, 0x6D,
            0x1C, 0x05, 0x20, 0x01, 0x08, 0x12, 0x69, 0x05, 0x20, 0x01, 0x01, 0x13, 0x00, 0x09, 0x20, 0x03,
            0x12, 0x69, 0x13, 0x00, 0x12, 0x6D, 0x1C, 0x05, 0x20, 0x01, 0x01, 0x12, 0x69, 0x06, 0x10, 0x01,
            0x01, 0x08, 0x1E, 0x00, 0x05, 0x20, 0x00, 0x11, 0x80, 0x89, 0x06, 0x20, 0x01, 0x01, 0x1D, 0x13,
            0x00, 0x08, 0x30, 0x01, 0x02, 0x0E, 0x13, 0x00, 0x1E, 0x00, 0x07, 0x20, 0x02, 0x0E, 0x13, 0x00,
            0x13, 0x01, 0x0A, 0x30, 0x01, 0x03, 0x0E, 0x13, 0x00, 0x13, 0x01, 0x1E, 0x00, 0x09, 0x00, 0x02,
            0x02, 0x11, 0x80, 0xCC, 0x11, 0x80, 0xCC, 0x06, 0x00, 0x01, 0x02, 0x11, 0x80, 0xCC, 0x08, 0x00,
            0x01, 0x11, 0x80, 0xCC, 0x11, 0x80, 0xCC, 0x0B, 0x00, 0x02, 0x11, 0x80, 0xCC, 0x11, 0x80, 0xCC,
            0x11, 0x80, 0xCC, 0x09, 0x00, 0x02, 0x11, 0x80, 0xCC, 0x11, 0x80, 0xCC, 0x08, 0x06, 0x00, 0x01,
            0x08, 0x11, 0x80, 0xCC, 0x08, 0x00, 0x01, 0x11, 0x80, 0xD0, 0x11, 0x80, 0xCC, 0x06, 0x00, 0x01,
            0x09, 0x11, 0x80, 0xCC, 0x08, 0x00, 0x01, 0x11, 0x80, 0xD4, 0x11, 0x80, 0xCC, 0x03, 0x20, 0x00,
            0x18, 0x04, 0x20, 0x01, 0x01, 0x18, 0x08, 0x20, 0x03, 0x01, 0x18, 0x1D, 0x08, 0x1D, 0x08, 0x06,
            0x20, 0x01, 0x01, 0x12, 0x81, 0xDC, 0x06, 0x20, 0x01, 0x01, 0x12, 0x80, 0x9D, 0x10, 0x00, 0x02,
            0x15, 0x12, 0x80, 0xA1, 0x01, 0x0E, 0x15, 0x12, 0x80, 0xA1, 0x01, 0x0E, 0x1D, 0x0E, 0x09, 0x20,
            0x05, 0x01, 0x08, 0x18, 0x18, 0x08, 0x10, 0x18, 0x08, 0x20, 0x01, 0x01, 0x15, 0x12, 0x61, 0x01,
            0x0E, 0x09, 0x20, 0x01, 0x01, 0x15, 0x12, 0x61, 0x01, 0x11, 0x60, 0x09, 0x20, 0x01, 0x01, 0x15,
            0x12, 0x64, 0x01, 0x11, 0x60, 0x81, 0x05, 0x20, 0x32, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12,
            0x61, 0x01, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01,
            0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C,
            0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15,
            0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12,
            0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61,
            0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01,
            0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C,
            0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15,
            0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12,
            0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61,
            0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01,
            0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C,
            0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15,
            0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12,
            0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61,
            0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x15, 0x12, 0x61, 0x01, 0x1C, 0x05, 0x20, 0x02, 0x08,
            0x08, 0x08, 0x06, 0x20, 0x03, 0x01, 0x08, 0x08, 0x08, 0x04, 0x20, 0x01, 0x08, 0x08, 0x05, 0x20,
            0x02, 0x01, 0x08, 0x08, 0x04, 0x20, 0x01, 0x08, 0x1C, 0x04, 0x20, 0x01, 0x01, 0x1C, 0x04, 0x20,
            0x00, 0x13, 0x00, 0x08, 0x20, 0x03, 0x12, 0x69, 0x08, 0x12, 0x6D, 0x1C, 0x04, 0x20, 0x01, 0x02,
            0x08, 0x07, 0x20, 0x02, 0x12, 0x69, 0x12, 0x6D, 0x1C, 0x05, 0x20, 0x02, 0x0E, 0x08, 0x08, 0x06,
            0x20, 0x03, 0x01, 0x08, 0x08, 0x0E, 0x07, 0x20, 0x04, 0x01, 0x08, 0x08, 0x08, 0x08, 0x09, 0x20,
            0x03, 0x01, 0x13, 0x02, 0x13, 0x01, 0x13, 0x00, 0x03, 0x28, 0x00, 0x08, 0x04, 0x28, 0x01, 0x08,
            0x0E, 0x05, 0x28, 0x02, 0x08, 0x0E, 0x08, 0x04, 0x28, 0x00, 0x1D, 0x0E, 0x03, 0x28, 0x00, 0x0E,
            0x07, 0x28, 0x00, 0x15, 0x12, 0x64, 0x01, 0x0E, 0x05, 0x28, 0x00, 0x11, 0x80, 0x89, 0x05, 0x28,
            0x02, 0x08, 0x08, 0x08, 0x04, 0x28, 0x01, 0x08, 0x08, 0x03, 0x28, 0x00, 0x02, 0x04, 0x28, 0x00,
            0x13, 0x00, 0x05, 0x28, 0x02, 0x0E, 0x08, 0x08, 0x08, 0x01, 0x00, 0x08, 0x00, 0x00, 0x00, 0x00,
            0x00, 0x1E, 0x01, 0x00, 0x01, 0x00, 0x54, 0x02, 0x16, 0x57, 0x72, 0x61, 0x70, 0x4E, 0x6F, 0x6E,
            0x45, 0x78, 0x63, 0x65, 0x70, 0x74, 0x69, 0x6F, 0x6E, 0x54, 0x68, 0x72, 0x6F, 0x77, 0x73, 0x01,
            0x08, 0x01, 0x00, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x16, 0x01, 0x00, 0x11, 0x44, 0x6F, 0x63,
            0x75, 0x6D, 0x65, 0x6E, 0x74, 0x61, 0x74, 0x69, 0x6F, 0x6E, 0x54, 0x65, 0x73, 0x74, 0x00, 0x00,
            0x05, 0x01, 0x00, 0x00, 0x00, 0x00, 0x17, 0x01, 0x00, 0x12, 0x43, 0x6F, 0x70, 0x79, 0x72, 0x69,
            0x67, 0x68, 0x74, 0x20, 0xC2, 0xA9, 0x20, 0x20, 0x32, 0x30, 0x30, 0x39, 0x00, 0x00, 0x29, 0x01,
            0x00, 0x24, 0x35, 0x37, 0x38, 0x33, 0x66, 0x61, 0x37, 0x62, 0x2D, 0x33, 0x37, 0x37, 0x35, 0x2D,
            0x34, 0x64, 0x38, 0x65, 0x2D, 0x61, 0x37, 0x32, 0x66, 0x2D, 0x32, 0x35, 0x61, 0x33, 0x62, 0x37,
            0x32, 0x36, 0x62, 0x38, 0x38, 0x33, 0x00, 0x00, 0x0C, 0x01, 0x00, 0x07, 0x31, 0x2E, 0x30, 0x2E,
            0x30, 0x2E, 0x30, 0x00, 0x00, 0x4D, 0x01, 0x00, 0x1C, 0x2E, 0x4E, 0x45, 0x54, 0x46, 0x72, 0x61,
            0x6D, 0x65, 0x77, 0x6F, 0x72, 0x6B, 0x2C, 0x56, 0x65, 0x72, 0x73, 0x69, 0x6F, 0x6E, 0x3D, 0x76,
            0x34, 0x2E, 0x36, 0x2E, 0x31, 0x01, 0x00, 0x54, 0x0E, 0x14, 0x46, 0x72, 0x61, 0x6D, 0x65, 0x77,
            0x6F, 0x72, 0x6B, 0x44, 0x69, 0x73, 0x70, 0x6C, 0x61, 0x79, 0x4E, 0x61, 0x6D, 0x65, 0x14, 0x2E,
            0x4E, 0x45, 0x54, 0x20, 0x46, 0x72, 0x61, 0x6D, 0x65, 0x77, 0x6F, 0x72, 0x6B, 0x20, 0x34, 0x2E,
            0x36, 0x2E, 0x31, 0x09, 0x01, 0x00, 0x04, 0x49, 0x74, 0x65, 0x6D, 0x00, 0x00, 0x08, 0x01, 0x00,
            0xFF, 0x7F, 0x00, 0x00, 0x00, 0x00, 0x08, 0x01, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x29,
            0x01, 0x00, 0x24, 0x37, 0x42, 0x46, 0x38, 0x30, 0x39, 0x38, 0x30, 0x2D, 0x42, 0x46, 0x33, 0x32,
            0x2D, 0x31, 0x30, 0x31, 0x41, 0x2D, 0x38, 0x42, 0x42, 0x42, 0x2D, 0x30, 0x30, 0x41, 0x41, 0x30,
            0x30, 0x33, 0x30, 0x30, 0x43, 0x41, 0x42, 0x00, 0x00, 0x00, 0x00, 0x00
        };
    }
}
