import React, { useRef, useState } from "react"
import EllipsisTextControl from "@renderer/components/EllipsisTextControl"
import { Trans, useTranslation } from "react-i18next"
import { formatTime } from "@renderer/utils/formatTime"
import { Progress, Button, Modal, Checkbox, message } from "antd"
import { openFile, openFolder, separator } from "@renderer/utils/fileHelper"

// 文本
export const tableText = {
  title: <Trans i18nKey={"translation:commonText.text" as string & never[]} />,
  dataIndex: "text",
  key: "text",
  width: 400,
  render: (text) => <EllipsisTextControl content={text} width={360} type="single" />,
}

// 文件
export const tableFile = {
  title: <Trans i18nKey={"translation:commonText.file" as string & never[]} />,
  dataIndex: "outputFileName",
  key: "outputFileName",
  width: 400,
  render: (text) => <EllipsisTextControl content={text} width={360} type="single" />,
}

// 源文件
export const tableOriginFile = {
  title: <Trans i18nKey={"translation:commonText.originFile" as string & never[]} />,
  dataIndex: "inputFilePath",
  key: "inputFilePath",
  width: 240,
  render: (inputFilePath) => <EllipsisTextControl content={inputFilePath} width={200} type="single" />,
}

//
export const tableFps = {
  title: <Trans i18nKey={"translation:commonText.fps" as string & never[]} />,
  dataIndex: "fps",
  key: "fps",
  width: 60,
}

// 进度
export const tableProgress = {
  title: <Trans i18nKey={"translation:commonText.progress" as string & never[]} />,
  dataIndex: "progress",
  key: "progress",
  render: (progress) => <Progress percent={progress} />,
}

// 创建时间
export const tableCreateTime = {
  title: <Trans i18nKey={"translation:commonText.createTime" as string & never[]} />,
  dataIndex: "createTime",
  key: "createTime",
  width: 160,
  render: (createTime: number) => formatTime(createTime),
}

interface ITableBtnProps {
  record: any
  suffix?: string
  hasFile?: boolean
  callback?: (data?: any) => void
}

// 保存文件按钮
export const SaveFileBtn = (props: ITableBtnProps) => {
  const { record, suffix, callback } = props
  const { t } = useTranslation()
  const handleSaveFile = async () => {
    const res = await window.ipcInvoke("WIN_SAVE_AND_RENAME", { fileName: record.taskId + suffix })
    callback && callback(res)
  }

  return (
    <Button onClick={handleSaveFile} type="link" className="common-table-link-btn">
      {t("translation:commonText.save")}
    </Button>
  )
}

// 打开文件按钮
export const OpenFileBtn = (props: ITableBtnProps) => {
  const { record } = props
  const { t } = useTranslation()
  return (
    <Button
      onClick={() => openFolder(`${record.outputFloaderPath}${separator}${record.outputFileName}`)}
      type="link"
      className="common-table-link-btn"
    >
      {t("translation:commonText.openFile")}
    </Button>
  )
}

// 打开文件夹按钮
export const OpenFolderBtn = (props: ITableBtnProps) => {
  const { record } = props
  const { t } = useTranslation()
  return (
    <Button onClick={() => openFolder(record.outputFloaderPath)} type="link" className="common-table-link-btn">
      {t("translation:commonText.openFolder")}
    </Button>
  )
}

// 删除按钮
export const DeleteRecordBtn = (props: ITableBtnProps) => {
  const { record, hasFile, callback } = props
  const isCheckRef = useRef(true)

  const { t } = useTranslation()

  // 选择是否删除文件
  const onCheckBoxChnage = (e) => {
    isCheckRef.current = e.target.checked
  }

  // 删除记录
  const deleteData = async ({ record, isDeleteFile }) => {
    const recordDeleteRes = await window.ipcInvoke("QUEUE_STORE_DELETE", {
      key: `${record.code}List`,
      idKey: "taskId",
      id: record.taskId,
    })
    if (isDeleteFile && hasFile) {
      const path = `${record.outputFloaderPath}${separator}${record.outputFileName}`
      const res = await window.ipcInvoke("WIN_DELETE_FILE", {
        path,
      })
      res ? message.success(t("translation:commonText.success")) : message.error(t("translation:commonText.error"))
    } else {
      recordDeleteRes
        ? message.success(t("translation:commonText.success"))
        : message.error(t("translation:commonText.error"))
    }
    callback && callback()
  }

  // 删除按钮点击事件, 弹出确认框
  const onDelete = (record) => {
    Modal.confirm({
      title: t("translation:siteMain.components.deleteModal.content"),
      content: hasFile ? (
        <Checkbox defaultChecked onChange={onCheckBoxChnage}>
          {t("translation:siteMain.components.deleteModal.deleteFileText")}
        </Checkbox>
      ) : null,
      onOk: () => {
        deleteData({ record, isDeleteFile: isCheckRef.current })
      },
    })
  }
  return (
    <Button onClick={() => onDelete(record)} type="link" className="common-table-link-btn">
      {t("translation:commonText.delete")}
    </Button>
  )
}
