import { BrowserWindow, ipcMain, screen } from "electron"
import { ChildProcessWithoutNullStreams, spawn } from "child_process"
import { getFfmpegPath, getFfmpegPathForMac } from "@main/plugin/modules/ffmpeg"
import { checkFolderExists } from "@main/utils/fileHelper"
import { queueStoreAdd, queueStoreUpdate } from "@main/utils/storeHelper"

let ffmpegProcess: ChildProcessWithoutNullStreams | null = null

ipcMain.handle("SCREEN_RECORD_GET_CURRENT_INFO", async (e, data) => {
  const cur = BrowserWindow.fromWebContents(e.sender)
  if (!cur) return null
  const bounds = cur.getBounds()
  const currentDisplay = screen.getDisplayNearestPoint({
    x: bounds.x,
    y: bounds.y,
  })
  const display = screen.getDisplayMatching({ x: bounds.x, y: bounds.y, width: bounds.width, height: bounds.height });

  // 获取屏幕的索引
  const screenIndex = screen.getAllDisplays().findIndex(d => d.id === display.id);
  const scaleFactor = currentDisplay.scaleFactor
  const windowX = currentDisplay.nativeOrigin.x - currentDisplay.bounds.x + bounds.x
  const windowY = currentDisplay.nativeOrigin.y - currentDisplay.bounds.y + bounds.y
  bounds.x = windowX
  bounds.y = windowY
  bounds["scaleFactor"] = scaleFactor
  bounds["screenIndex"] = screenIndex
  return bounds
})

// 开始录屏
ipcMain.handle("SCREEN_RECORD_START", async (e, params) => {
  const win = BrowserWindow.fromWebContents(e.sender)
  win?.setResizable(false)
  win?.setMovable(false)
  const ffmpegPath = process.platform === "darwin" ? getFfmpegPathForMac() : getFfmpegPath()
  checkFolderExists(params.outputFloaderPath)
  queueStoreAdd({
    params: { ...params },
    key: `${params.code}List`,
  })

  const allWindows = BrowserWindow.getAllWindows()
  allWindows.forEach((window) => {
    if (window["customId"] === "main") {
      window.webContents.send("SCREEN_RECORD_DATA_CHANGE")
    }
  })

  ffmpegProcess = spawn(ffmpegPath, params.command)
  ffmpegProcess.stdout.on("data", (data) => {
    console.log(`stdout: ${data}`)
  })
  ffmpegProcess.stderr.on("data", (data) => {
    console.error(`stderr: ${data}`)
  })
  ffmpegProcess.on("close", (code) => {
    console.log(`child process exited with code ${code}`)
  })
})

// 结束录屏
ipcMain.handle("SCREEN_RECORD_STOP", async (e, data) => {
  const win = BrowserWindow.fromWebContents(e.sender)
  win?.setResizable(true)
  win?.setMovable(true)
  ffmpegProcess?.on("exit", (code, signal) => {
    console.log(`FFmpeg进程退出，退出码：${code}，信号：${signal}`)
  })
  ffmpegProcess?.stdin.write("q\n")
})
