/**
 * @file
 * @brief Static sets of BACnet Property members for each object type
 * @author Steve Karg <skarg@users.sourceforge.net>
 * @date 2012
 * @copyright SPDX-License-Identifier: GPL-2.0-or-later WITH GCC-exception-2.0
 */
#include <stdint.h>
/* BACnet Stack defines - first */
#include "bacnet/bacdef.h"
/* BACnet Stack API */
#include "bacnet/bacdcode.h"
#include "bacnet/rpm.h"
#include "bacnet/rp.h"
#include "bacnet/proplist.h"
#include "bacnet/property.h"

#ifndef BACNET_PROPERTY_LISTS
#define BACNET_PROPERTY_LISTS 0
#endif

#if BACNET_PROPERTY_LISTS
/** @file property.c  List of Required and Optional object properties */
/* note: the PROP_PROPERTY_LIST is NOT included in these lists, on purpose */

static const int Default_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME, PROP_OBJECT_TYPE, -1
};

static const int Default_Properties_Optional[] = {
    /* unordered list of properties */
    -1
};

static const int Access_Credential_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_GLOBAL_IDENTIFIER,
    PROP_STATUS_FLAGS,
    PROP_RELIABILITY,
    PROP_CREDENTIAL_STATUS,
    PROP_REASON_FOR_DISABLE,
    PROP_AUTHENTICATION_FACTORS,
    PROP_ACTIVATION_TIME,
    PROP_EXPIRATION_TIME,
    PROP_CREDENTIAL_DISABLE,
    PROP_ASSIGNED_ACCESS_RIGHTS,
    -1
};

static const int Access_Credential_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_DAYS_REMAINING,
    PROP_USES_REMAINING,
    PROP_ABSENTEE_LIMIT,
    PROP_BELONGS_TO,
    PROP_LAST_ACCESS_POINT,
    PROP_LAST_ACCESS_EVENT,
    PROP_LAST_USE_TIME,
    PROP_TRACE_FLAG,
    PROP_THREAT_AUTHORITY,
    PROP_EXTENDED_TIME_ENABLE,
    PROP_AUTHORIZATION_EXEMPTIONS,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Access_Door_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_DOOR_PULSE_TIME,
    PROP_DOOR_EXTENDED_PULSE_TIME,
    PROP_DOOR_OPEN_TOO_LONG_TIME,
    PROP_CURRENT_COMMAND_PRIORITY,
    -1
};

static const int Access_Door_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_DOOR_STATUS,
    PROP_LOCK_STATUS,
    PROP_SECURED_STATUS,
    PROP_DOOR_MEMBERS,
    PROP_DOOR_UNLOCK_DELAY_TIME,
    PROP_DOOR_ALARM_STATE,
    PROP_MASKED_ALARM_VALUES,
    PROP_MAINTENANCE_REQUIRED,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_ALARM_VALUES,
    PROP_FAULT_VALUES,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Access_Point_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_AUTHENTICATION_STATUS,
    PROP_ACTIVE_AUTHENTICATION_POLICY,
    PROP_NUMBER_OF_AUTHENTICATION_POLICIES,
    PROP_AUTHORIZATION_MODE,
    PROP_ACCESS_EVENT,
    PROP_ACCESS_EVENT_TAG,
    PROP_ACCESS_EVENT_TIME,
    PROP_ACCESS_EVENT_CREDENTIAL,
    PROP_ACCESS_DOORS,
    PROP_PRIORITY_FOR_WRITING,
    -1
};

static const int Access_Point_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_AUTHENTICATION_POLICY_LIST,
    PROP_AUTHENTICATION_POLICY_NAMES,
    PROP_VERIFICATION_TIME,
    PROP_LOCKOUT,
    PROP_LOCKOUT_RELINQUISH_TIME,
    PROP_FAILED_ATTEMPTS,
    PROP_FAILED_ATTEMPT_EVENTS,
    PROP_MAX_FAILED_ATTEMPTS,
    PROP_FAILED_ATTEMPTS_TIME,
    PROP_THREAT_LEVEL,
    PROP_OCCUPANCY_UPPER_LIMIT_ENFORCED,
    PROP_OCCUPANCY_LOWER_LIMIT_ENFORCED,
    PROP_OCCUPANCY_COUNT_ADJUST,
    PROP_ACCOMPANIMENT_TIME,
    PROP_ACCESS_EVENT_AUTHENTICATION_FACTOR,
    PROP_MUSTER_POINT,
    PROP_ZONE_TO,
    PROP_ZONE_FROM,
    PROP_NOTIFICATION_CLASS,
    PROP_TRANSACTION_NOTIFICATION_CLASS,
    PROP_ACCESS_ALARM_EVENTS,
    PROP_ACCESS_TRANSACTION_EVENTS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Access_Rights_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_GLOBAL_IDENTIFIER,
    PROP_STATUS_FLAGS,
    PROP_RELIABILITY,
    PROP_ENABLE,
    PROP_NEGATIVE_ACCESS_RULES,
    PROP_POSITIVE_ACCESS_RULES,
    -1
};

static const int Access_Rights_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_ACCOMPANIMENT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Access_User_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_GLOBAL_IDENTIFIER, PROP_STATUS_FLAGS, PROP_RELIABILITY,
    PROP_USER_TYPE,         PROP_CREDENTIALS,  -1
};

static const int Access_User_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_USER_NAME,
    PROP_USER_EXTERNAL_IDENTIFIER,
    PROP_USER_INFORMATION_REFERENCE,
    PROP_MEMBERS,
    PROP_MEMBER_OF,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Access_Zone_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,     PROP_OBJECT_TYPE,
    PROP_GLOBAL_IDENTIFIER, PROP_OCCUPANCY_STATE, PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,       PROP_RELIABILITY,     PROP_OUT_OF_SERVICE,
    PROP_ENTRY_POINTS,      PROP_EXIT_POINTS,     -1
};

static const int Access_Zone_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_OCCUPANCY_COUNT,
    PROP_OCCUPANCY_COUNT_ENABLE,
    PROP_ADJUST_VALUE,
    PROP_OCCUPANCY_UPPER_LIMIT,
    PROP_OCCUPANCY_LOWER_LIMIT,
    PROP_CREDENTIALS_IN_ZONE,
    PROP_LAST_CREDENTIAL_ADDED,
    PROP_LAST_CREDENTIAL_ADDED_TIME,
    PROP_LAST_CREDENTIAL_REMOVED,
    PROP_LAST_CREDENTIAL_REMOVED_TIME,
    PROP_PASSBACK_MODE,
    PROP_PASSBACK_TIMEOUT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_ALARM_VALUES,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Accumulator_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,
    PROP_SCALE,
    PROP_UNITS,
    PROP_MAX_PRES_VALUE,
    -1
};

static const int Accumulator_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_DEVICE_TYPE,
    PROP_RELIABILITY,
    PROP_PRESCALE,
    PROP_VALUE_CHANGE_TIME,
    PROP_VALUE_BEFORE_CHANGE,
    PROP_VALUE_SET,
    PROP_LOGGING_RECORD,
    PROP_LOGGING_OBJECT,
    PROP_PULSE_RATE,
    PROP_HIGH_LIMIT,
    PROP_LOW_LIMIT,
    PROP_LIMIT_MONITORING_INTERVAL,
    PROP_NOTIFICATION_CLASS,
    PROP_TIME_DELAY,
    PROP_LIMIT_ENABLE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_FAULT_HIGH_LIMIT,
    PROP_FAULT_LOW_LIMIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Alert_Enrollment_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,  PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,        PROP_PRESENT_VALUE,
    PROP_EVENT_STATE,        PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS, PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,  PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,  -1
};

static const int Alert_Enrollment_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Analog_Input_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,    PROP_UNITS,        -1
};

static const int Analog_Input_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_DEVICE_TYPE,
    PROP_RELIABILITY,
    PROP_UPDATE_INTERVAL,
    PROP_MIN_PRES_VALUE,
    PROP_MAX_PRES_VALUE,
    PROP_RESOLUTION,
    PROP_COV_INCREMENT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_HIGH_LIMIT,
    PROP_LOW_LIMIT,
    PROP_DEADBAND,
    PROP_LIMIT_ENABLE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_INTERFACE_VALUE,
    PROP_FAULT_HIGH_LIMIT,
    PROP_FAULT_LOW_LIMIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Analog_Output_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,
    PROP_UNITS,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
#if (BACNET_PROTOCOL_REVISION >= 17)
    PROP_CURRENT_COMMAND_PRIORITY,
#endif
    -1
};

static const int Analog_Output_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_DEVICE_TYPE,
    PROP_RELIABILITY,
    PROP_MIN_PRES_VALUE,
    PROP_MAX_PRES_VALUE,
    PROP_RESOLUTION,
    PROP_COV_INCREMENT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_HIGH_LIMIT,
    PROP_LOW_LIMIT,
    PROP_DEADBAND,
    PROP_LIMIT_ENABLE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_INTERFACE_VALUE,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Analog_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,    PROP_UNITS,        -1
};

static const int Analog_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_RELIABILITY,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_COV_INCREMENT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_HIGH_LIMIT,
    PROP_LOW_LIMIT,
    PROP_DEADBAND,
    PROP_LIMIT_ENABLE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_MIN_PRES_VALUE,
    PROP_MAX_PRES_VALUE,
    PROP_RESOLUTION,
    PROP_FAULT_HIGH_LIMIT,
    PROP_FAULT_LOW_LIMIT,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Averaging_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,    PROP_OBJECT_TYPE,
    PROP_MINIMUM_VALUE,     PROP_AVERAGE_VALUE,  PROP_MAXIMUM_VALUE,
    PROP_ATTEMPTED_SAMPLES, PROP_VALID_SAMPLES,  PROP_OBJECT_PROPERTY_REFERENCE,
    PROP_WINDOW_INTERVAL,   PROP_WINDOW_SAMPLES, -1
};

static const int Averaging_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_MINIMUM_VALUE_TIMESTAMP,
    PROP_VARIANCE_VALUE,
    PROP_MAXIMUM_VALUE_TIMESTAMP,
    PROP_DESCRIPTION,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Binary_Input_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,    PROP_POLARITY,     -1
};

static const int Binary_Input_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_DEVICE_TYPE,
    PROP_RELIABILITY,
    PROP_INACTIVE_TEXT,
    PROP_ACTIVE_TEXT,
    PROP_CHANGE_OF_STATE_TIME,
    PROP_CHANGE_OF_STATE_COUNT,
    PROP_TIME_OF_STATE_COUNT_RESET,
    PROP_ELAPSED_ACTIVE_TIME,
    PROP_TIME_OF_ACTIVE_TIME_RESET,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_ALARM_VALUE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_INTERFACE_VALUE,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Binary_Lighting_Output_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_OUT_OF_SERVICE,
    PROP_BLINK_WARN_ENABLE,
    PROP_EGRESS_TIME,
    PROP_EGRESS_ACTIVE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_CURRENT_COMMAND_PRIORITY,
    -1
};

static const int Binary_Lighting_Output_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_RELIABILITY,
    PROP_FEEDBACK_VALUE,
    PROP_POWER,
    PROP_POLARITY,
    PROP_ELAPSED_ACTIVE_TIME,
    PROP_TIME_OF_ACTIVE_TIME_RESET,
    PROP_STRIKE_COUNT,
    PROP_TIME_OF_STRIKE_COUNT_RESET,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Binary_Output_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,
    PROP_POLARITY,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
#if (BACNET_PROTOCOL_REVISION >= 17)
    PROP_CURRENT_COMMAND_PRIORITY,
#endif
    -1
};

static const int Binary_Output_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_DEVICE_TYPE,
    PROP_RELIABILITY,
    PROP_INACTIVE_TEXT,
    PROP_ACTIVE_TEXT,
    PROP_CHANGE_OF_STATE_TIME,
    PROP_CHANGE_OF_STATE_COUNT,
    PROP_TIME_OF_STATE_COUNT_RESET,
    PROP_ELAPSED_ACTIVE_TIME,
    PROP_TIME_OF_ACTIVE_TIME_RESET,
    PROP_MINIMUM_OFF_TIME,
    PROP_MINIMUM_ON_TIME,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_FEEDBACK_VALUE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_INTERFACE_VALUE,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Binary_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,       PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,      PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,    -1
};

static const int Binary_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_RELIABILITY,
    PROP_INACTIVE_TEXT,
    PROP_ACTIVE_TEXT,
    PROP_CHANGE_OF_STATE_TIME,
    PROP_CHANGE_OF_STATE_COUNT,
    PROP_TIME_OF_STATE_COUNT_RESET,
    PROP_ELAPSED_ACTIVE_TIME,
    PROP_TIME_OF_ACTIVE_TIME_RESET,
    PROP_MINIMUM_OFF_TIME,
    PROP_MINIMUM_ON_TIME,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_ALARM_VALUE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int BitString_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, -1
};

static const int BitString_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_ALARM_VALUES,
    PROP_BIT_MASK,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Calendar_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME, PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_DATE_LIST,   -1
};

static const int Calendar_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Channel_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_LAST_PRIORITY,
    PROP_WRITE_STATUS,
    PROP_STATUS_FLAGS,
    PROP_OUT_OF_SERVICE,
    PROP_LIST_OF_OBJECT_PROPERTY_REFERENCES,
    PROP_CHANNEL_NUMBER,
    PROP_CONTROL_GROUPS,
    -1
};

static const int Channel_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_RELIABILITY,
    PROP_EXECUTION_DELAY,
    PROP_ALLOW_GROUP_DELAY_INHIBIT,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_EVENT_STATE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_VALUE_SOURCE,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Command_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_IN_PROCESS,
    PROP_ALL_WRITES_SUCCESSFUL,
    PROP_ACTION,
    -1
};

static const int Command_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_ACTION_TEXT,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_VALUE_SOURCE,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int CharacterString_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, -1
};

static const int CharacterString_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_ALARM_VALUES,
    PROP_FAULT_VALUES,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Color_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,       PROP_PRESENT_VALUE,
    PROP_TRACKING_VALUE,    PROP_COLOR_COMMAND,
    PROP_IN_PROGRESS,       PROP_DEFAULT_COLOR,
    PROP_DEFAULT_FADE_TIME, -1
};

static const int Color_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_RELIABILITY,  PROP_DESCRIPTION,
    PROP_TRANSITION,   PROP_VALUE_SOURCE,
    PROP_AUDIT_LEVEL,  PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,         PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME, -1
};

static const int Color_Temperature_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_TRACKING_VALUE,
    PROP_COLOR_COMMAND,
    PROP_IN_PROGRESS,
    PROP_DEFAULT_COLOR_TEMPERATURE,
    PROP_DEFAULT_FADE_TIME,
    PROP_DEFAULT_RAMP_RATE,
    PROP_DEFAULT_STEP_INCREMENT,
    -1
};

static const int Color_Temperature_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_MIN_PRES_VALUE,
    PROP_MAX_PRES_VALUE,
    PROP_TRANSITION,
    PROP_VALUE_SOURCE,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Credential_Data_Input_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,       PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,      PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,    PROP_SUPPORTED_FORMATS,
    PROP_UPDATE_TIME,       -1
};

static const int Credential_Data_Input_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_SUPPORTED_FORMAT_CLASSES,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_EVENT_STATE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Date_Pattern_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, -1
};

static const int Date_Pattern_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Date_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, -1
};

static const int Date_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int DateTime_Pattern_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, -1
};

static const int DateTime_Pattern_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_IS_UTC,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int DateTime_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, -1
};

static const int DateTime_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_IS_UTC,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Device_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_SYSTEM_STATUS,
    PROP_VENDOR_NAME,
    PROP_VENDOR_IDENTIFIER,
    PROP_MODEL_NAME,
    PROP_FIRMWARE_REVISION,
    PROP_APPLICATION_SOFTWARE_VERSION,
    PROP_PROTOCOL_VERSION,
    PROP_PROTOCOL_REVISION,
    PROP_PROTOCOL_SERVICES_SUPPORTED,
    PROP_PROTOCOL_OBJECT_TYPES_SUPPORTED,
    PROP_OBJECT_LIST,
    PROP_MAX_APDU_LENGTH_ACCEPTED,
    PROP_SEGMENTATION_SUPPORTED,
    PROP_APDU_TIMEOUT,
    PROP_NUMBER_OF_APDU_RETRIES,
    PROP_DEVICE_ADDRESS_BINDING,
    PROP_DATABASE_REVISION,
    -1
};

static const int Device_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_LOCATION,
    PROP_DESCRIPTION,
    PROP_STRUCTURED_OBJECT_LIST,
    PROP_MAX_SEGMENTS_ACCEPTED,
    PROP_VT_CLASSES_SUPPORTED,
    PROP_ACTIVE_VT_SESSIONS,
    PROP_LOCAL_TIME,
    PROP_LOCAL_DATE,
    PROP_UTC_OFFSET,
    PROP_DAYLIGHT_SAVINGS_STATUS,
    PROP_APDU_SEGMENT_TIMEOUT,
    PROP_TIME_SYNCHRONIZATION_RECIPIENTS,
    PROP_MAX_MASTER,
    PROP_MAX_INFO_FRAMES,
    PROP_CONFIGURATION_FILES,
    PROP_LAST_RESTORE_TIME,
    PROP_BACKUP_FAILURE_TIMEOUT,
    PROP_BACKUP_PREPARATION_TIME,
    PROP_RESTORE_PREPARATION_TIME,
    PROP_RESTORE_COMPLETION_TIME,
    PROP_BACKUP_AND_RESTORE_STATE,
    PROP_ACTIVE_COV_SUBSCRIPTIONS,
    PROP_LAST_RESTART_REASON,
    PROP_TIME_OF_DEVICE_RESTART,
    PROP_RESTART_NOTIFICATION_RECIPIENTS,
    PROP_UTC_TIME_SYNCHRONIZATION_RECIPIENTS,
    PROP_TIME_SYNCHRONIZATION_INTERVAL,
    PROP_ALIGN_INTERVALS,
    PROP_INTERVAL_OFFSET,
    PROP_SERIAL_NUMBER,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_ACTIVE_COV_MULTIPLE_SUBSCRIPTIONS,
    PROP_AUDIT_NOTIFICATION_RECIPIENT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_DEVICE_UUID,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Elevator_Group_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_MACHINE_ROOM_ID,
    PROP_GROUP_ID,
    PROP_GROUP_MEMBERS,
    -1
};

static const int Elevator_Group_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,   PROP_GROUP_MODE,
    PROP_LANDING_CALLS, PROP_LANDING_CALL_CONTROL,
    PROP_AUDIT_LEVEL,   PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,          PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,  -1
};

static const int Escalator_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_STATUS_FLAGS,
    PROP_ELEVATOR_GROUP,
    PROP_GROUP_ID,
    PROP_INSTALLATION_ID,
    PROP_OPERATION_DIRECTION,
    PROP_OUT_OF_SERVICE,
    PROP_PASSENGER_ALARM,
    -1
};

static const int Escalator_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_POWER_MODE,
    PROP_ESCALATOR_MODE,
    PROP_ENERGY_METER,
    PROP_ENERGY_METER_REF,
    PROP_RELIABILITY,
    PROP_FAULT_SIGNALS,
    PROP_TIME_DELAY,
    PROP_TIME_DELAY_NORMAL,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_EVENT_STATE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Event_Enrollment_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_EVENT_TYPE,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_PARAMETERS,
    PROP_OBJECT_PROPERTY_REFERENCE,
    PROP_EVENT_STATE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_STATUS_FLAGS,
    PROP_RELIABILITY,
    -1
};

static const int Event_Enrollment_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_FAULT_TYPE,
    PROP_FAULT_PARAMETERS,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Event_Log_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,        PROP_OBJECT_TYPE,
    PROP_STATUS_FLAGS,      PROP_EVENT_STATE,        PROP_ENABLE,
    PROP_STOP_WHEN_FULL,    PROP_BUFFER_SIZE,        PROP_LOG_BUFFER,
    PROP_RECORD_COUNT,      PROP_TOTAL_RECORD_COUNT, -1
};

static const int Event_Log_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_RELIABILITY,
    PROP_START_TIME,
    PROP_STOP_TIME,
    PROP_NOTIFICATION_THRESHOLD,
    PROP_RECORDS_SINCE_NOTIFICATION,
    PROP_LAST_NOTIFY_RECORD,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int File_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_FILE_TYPE,
    PROP_FILE_SIZE,
    PROP_MODIFICATION_DATE,
    PROP_ARCHIVE,
    PROP_READ_ONLY,
    PROP_FILE_ACCESS_METHOD,
    -1
};

static const int File_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,  PROP_RECORD_COUNT,
    PROP_AUDIT_LEVEL,  PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,         PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME, -1
};

static const int Global_Group_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,       PROP_GROUP_MEMBERS,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,       PROP_MEMBER_STATUS_FLAGS,
    PROP_OUT_OF_SERVICE,    -1
};

static const int Global_Group_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_GROUP_MEMBER_NAMES,
    PROP_RELIABILITY,
    PROP_UPDATE_INTERVAL,
    PROP_REQUESTED_UPDATE_INTERVAL,
    PROP_COV_RESUBSCRIPTION_INTERVAL,
    PROP_CLIENT_COV_INCREMENT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_COVU_PERIOD,
    PROP_COVU_RECIPIENTS,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Group_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,     PROP_OBJECT_NAME,   PROP_OBJECT_TYPE,
    PROP_LIST_OF_GROUP_MEMBERS, PROP_PRESENT_VALUE, -1
};

static const int Group_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Integer_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_UNITS,
    -1
};

static const int Integer_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_COV_INCREMENT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_HIGH_LIMIT,
    PROP_LOW_LIMIT,
    PROP_DEADBAND,
    PROP_LIMIT_ENABLE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_MIN_PRES_VALUE,
    PROP_MAX_PRES_VALUE,
    PROP_RESOLUTION,
    PROP_FAULT_HIGH_LIMIT,
    PROP_FAULT_LOW_LIMIT,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Large_Analog_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_UNITS,
    -1
};

static const int Large_Analog_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_COV_INCREMENT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_HIGH_LIMIT,
    PROP_LOW_LIMIT,
    PROP_DEADBAND,
    PROP_LIMIT_ENABLE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_MIN_PRES_VALUE,
    PROP_MAX_PRES_VALUE,
    PROP_RESOLUTION,
    PROP_FAULT_HIGH_LIMIT,
    PROP_FAULT_LOW_LIMIT,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Lift_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,          PROP_STATUS_FLAGS,
    PROP_ELEVATOR_GROUP,       PROP_GROUP_ID,
    PROP_INSTALLATION_ID,      PROP_CAR_POSITION,
    PROP_CAR_MOVING_DIRECTION, PROP_CAR_DOOR_STATUS,
    PROP_PASSENGER_ALARM,      PROP_OUT_OF_SERVICE,
    PROP_FAULT_SIGNALS,        -1
};

static const int Lift_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_FLOOR_TEXT,
    PROP_CAR_DOOR_TEXT,
    PROP_ASSIGNED_LANDING_CALLS,
    PROP_MAKING_CAR_CALL,
    PROP_REGISTERED_CAR_CALL,
    PROP_CAR_ASSIGNED_DIRECTION,
    PROP_CAR_DOOR_COMMAND,
    PROP_CAR_DOOR_ZONE,
    PROP_CAR_MODE,
    PROP_CAR_LOAD,
    PROP_CAR_LOAD_UNITS,
    PROP_NEXT_STOPPING_FLOOR,
    PROP_TIME_DELAY,
    PROP_TIME_DELAY_NORMAL,
    PROP_ENERGY_METER,
    PROP_ENERGY_METER_REF,
    PROP_RELIABILITY,
    PROP_CAR_DRIVE_STATUS,
    PROP_LANDING_DOOR_STATUS,
    PROP_HIGHER_DECK,
    PROP_LOWER_DECK,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_EVENT_STATE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Lighting_Output_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_TRACKING_VALUE,
    PROP_LIGHTING_COMMAND,
    PROP_IN_PROGRESS,
    PROP_STATUS_FLAGS,
    PROP_OUT_OF_SERVICE,
    PROP_BLINK_WARN_ENABLE,
    PROP_EGRESS_TIME,
    PROP_EGRESS_ACTIVE,
    PROP_DEFAULT_FADE_TIME,
    PROP_DEFAULT_RAMP_RATE,
    PROP_DEFAULT_STEP_INCREMENT,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_LIGHTING_COMMAND_DEFAULT_PRIORITY,
    PROP_CURRENT_COMMAND_PRIORITY,
    -1
};

static const int Lighting_Output_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_RELIABILITY,
    PROP_TRANSITION,
    PROP_FEEDBACK_VALUE,
    PROP_POWER,
    PROP_INSTANTANEOUS_POWER,
    PROP_MIN_ACTUAL_VALUE,
    PROP_MAX_ACTUAL_VALUE,
    PROP_COV_INCREMENT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Load_Control_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_REQUESTED_SHED_LEVEL,
    PROP_START_TIME,
    PROP_SHED_DURATION,
    PROP_DUTY_WINDOW,
    PROP_ENABLE,
    PROP_EXPECTED_SHED_LEVEL,
    PROP_ACTUAL_SHED_LEVEL,
    PROP_SHED_LEVELS,
    PROP_SHED_LEVEL_DESCRIPTIONS,
    -1
};

static const int Load_Control_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_STATE_DESCRIPTION,
    PROP_RELIABILITY,
    PROP_FULL_DUTY_BASELINE,
    PROP_NOTIFICATION_CLASS,
    PROP_TIME_DELAY,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_VALUE_SOURCE,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Life_Safety_Point_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,  PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,        PROP_PRESENT_VALUE,
    PROP_TRACKING_VALUE,     PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,        PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,     PROP_MODE,
    PROP_ACCEPTED_MODES,     PROP_SILENCED,
    PROP_OPERATION_EXPECTED, -1
};

static const int Life_Safety_Point_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_DEVICE_TYPE,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_LIFE_SAFETY_ALARM_VALUES,
    PROP_ALARM_VALUES,
    PROP_FAULT_VALUES,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_MAINTENANCE_REQUIRED,
    PROP_SETTING,
    PROP_DIRECT_READING,
    PROP_UNITS,
    PROP_MEMBER_OF,
    PROP_VALUE_SOURCE,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Life_Safety_Zone_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_TRACKING_VALUE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_MODE,
    PROP_ACCEPTED_MODES,
    PROP_SILENCED,
    PROP_OPERATION_EXPECTED,
    PROP_ZONE_MEMBERS,
    -1
};

static const int Life_Safety_Zone_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_DEVICE_TYPE,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_LIFE_SAFETY_ALARM_VALUES,
    PROP_ALARM_VALUES,
    PROP_FAULT_VALUES,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_MAINTENANCE_REQUIRED,
    PROP_MEMBER_OF,
    PROP_VALUE_SOURCE,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Loop_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,
    PROP_OUTPUT_UNITS,
    PROP_MANIPULATED_VARIABLE_REFERENCE,
    PROP_CONTROLLED_VARIABLE_REFERENCE,
    PROP_CONTROLLED_VARIABLE_VALUE,
    PROP_CONTROLLED_VARIABLE_UNITS,
    PROP_SETPOINT_REFERENCE,
    PROP_SETPOINT,
    PROP_ACTION,
    PROP_PRIORITY_FOR_WRITING,
    -1
};

static const int Loop_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_RELIABILITY,
    PROP_PROPORTIONAL_CONSTANT,
    PROP_PROPORTIONAL_CONSTANT_UNITS,
    PROP_INTEGRAL_CONSTANT,
    PROP_INTEGRAL_CONSTANT_UNITS,
    PROP_DERIVATIVE_CONSTANT,
    PROP_DERIVATIVE_CONSTANT_UNITS,
    PROP_BIAS,
    PROP_MAXIMUM_OUTPUT,
    PROP_MINIMUM_OUTPUT,
    PROP_COV_INCREMENT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_ERROR_LIMIT,
    PROP_DEADBAND,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_LOW_DIFF_LIMIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Multistate_Input_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,      PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS,     PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,    PROP_NUMBER_OF_STATES, -1
};

static const int Multistate_Input_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_DEVICE_TYPE,
    PROP_RELIABILITY,
    PROP_STATE_TEXT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_ALARM_VALUES,
    PROP_FAULT_VALUES,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_INTERFACE_VALUE,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Multistate_Output_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,
    PROP_NUMBER_OF_STATES,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
#if (BACNET_PROTOCOL_REVISION >= 17)
    PROP_CURRENT_COMMAND_PRIORITY,
#endif
    -1
};

static const int Multistate_Output_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_DEVICE_TYPE,
    PROP_RELIABILITY,
    PROP_STATE_TEXT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_FEEDBACK_VALUE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_INTERFACE_VALUE,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Multistate_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,      PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS,     PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,    PROP_NUMBER_OF_STATES, -1
};

static const int Multistate_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_RELIABILITY,
    PROP_STATE_TEXT,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_ALARM_VALUES,
    PROP_FAULT_VALUES,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Network_Port_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,       PROP_STATUS_FLAGS,
    PROP_RELIABILITY,       PROP_OUT_OF_SERVICE,
    PROP_NETWORK_TYPE,      PROP_PROTOCOL_LEVEL,
    PROP_NETWORK_NUMBER,    PROP_NETWORK_NUMBER_QUALITY,
    PROP_CHANGES_PENDING,   PROP_APDU_LENGTH,
    PROP_LINK_SPEED,        -1
};

static const int Network_Port_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_REFERENCE_PORT,
    PROP_COMMAND,
    PROP_MAC_ADDRESS,
    PROP_LINK_SPEEDS,
    PROP_LINK_SPEED_AUTONEGOTIATE,
    PROP_NETWORK_INTERFACE_NAME,
    PROP_BACNET_IP_MODE,
    PROP_IP_ADDRESS,
    PROP_BACNET_IP_UDP_PORT,
    PROP_IP_SUBNET_MASK,
    PROP_IP_DEFAULT_GATEWAY,
    PROP_BACNET_IP_MULTICAST_ADDRESS,
    PROP_IP_DNS_SERVER,
    PROP_IP_DHCP_ENABLE,
    PROP_IP_DHCP_LEASE_TIME,
    PROP_IP_DHCP_LEASE_TIME_REMAINING,
    PROP_IP_DHCP_SERVER,
    PROP_BACNET_IP_NAT_TRAVERSAL,
    PROP_BACNET_IP_GLOBAL_ADDRESS,
    PROP_BBMD_BROADCAST_DISTRIBUTION_TABLE,
    PROP_BBMD_ACCEPT_FD_REGISTRATIONS,
    PROP_BBMD_FOREIGN_DEVICE_TABLE,
    PROP_FD_BBMD_ADDRESS,
    PROP_FD_SUBSCRIPTION_LIFETIME,
    PROP_BACNET_IPV6_MODE,
    PROP_IPV6_ADDRESS,
    PROP_IPV6_PREFIX_LENGTH,
    PROP_BACNET_IPV6_UDP_PORT,
    PROP_IPV6_DEFAULT_GATEWAY,
    PROP_BACNET_IPV6_MULTICAST_ADDRESS,
    PROP_IPV6_DNS_SERVER,
    PROP_IPV6_AUTO_ADDRESSING_ENABLE,
    PROP_IPV6_DHCP_LEASE_TIME,
    PROP_IPV6_DHCP_LEASE_TIME_REMAINING,
    PROP_IPV6_DHCP_SERVER,
    PROP_IPV6_ZONE_INDEX,
    PROP_MAX_MASTER,
    PROP_MAX_INFO_FRAMES,
    PROP_SLAVE_PROXY_ENABLE,
    PROP_MANUAL_SLAVE_ADDRESS_BINDING,
    PROP_AUTO_SLAVE_DISCOVERY,
    PROP_SLAVE_ADDRESS_BINDING,
    PROP_VIRTUAL_MAC_ADDRESS_TABLE,
    PROP_ROUTING_TABLE,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_STATE,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Network_Security_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_BASE_DEVICE_SECURITY_POLICY,
    PROP_NETWORK_ACCESS_SECURITY_POLICIES,
    PROP_SECURITY_TIME_WINDOW,
    PROP_PACKET_REORDER_TIME,
    PROP_DISTRIBUTION_KEY_REVISION,
    PROP_KEY_SETS,
    PROP_LAST_KEY_SERVER,
    PROP_SECURITY_PDU_TIMEOUT,
    PROP_UPDATE_KEY_SET_TIMEOUT,
    PROP_SUPPORTED_SECURITY_ALGORITHM,
    PROP_DO_NOT_HIDE,
    -1
};

static const int Network_Security_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Notification_Class_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,       PROP_NOTIFICATION_CLASS,
    PROP_PRIORITY,          PROP_ACK_REQUIRED,
    PROP_RECIPIENT_LIST,    -1
};

static const int Notification_Class_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Notification_Forwarder_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_STATUS_FLAGS,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_RECIPIENT_LIST,
    PROP_SUBSCRIBED_RECIPIENTS,
    PROP_PROCESS_IDENTIFIER_FILTER,
    PROP_LOCAL_FORWARDING_ONLY,
    -1
};

static const int Notification_Forwarder_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_PORT_FILTER,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int OctetString_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, -1
};

static const int OctetString_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Positive_Integer_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_UNITS,
    -1
};

static const int Positive_Integer_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_COV_INCREMENT,
    PROP_TIME_DELAY,
    PROP_NOTIFICATION_CLASS,
    PROP_HIGH_LIMIT,
    PROP_LOW_LIMIT,
    PROP_DEADBAND,
    PROP_LIMIT_ENABLE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_MIN_PRES_VALUE,
    PROP_MAX_PRES_VALUE,
    PROP_RESOLUTION,
    PROP_FAULT_HIGH_LIMIT,
    PROP_FAULT_LOW_LIMIT,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Program_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,       PROP_PROGRAM_STATE,
    PROP_PROGRAM_CHANGE,    PROP_STATUS_FLAGS,
    PROP_OUT_OF_SERVICE,    -1
};

static const int Program_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_REASON_FOR_HALT,
    PROP_DESCRIPTION_OF_HALT,
    PROP_PROGRAM_LOCATION,
    PROP_DESCRIPTION,
    PROP_INSTANCE_OF,
    PROP_RELIABILITY,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_EVENT_STATE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Pulse_Converter_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_OUT_OF_SERVICE,
    PROP_UNITS,
    PROP_SCALE_FACTOR,
    PROP_ADJUST_VALUE,
    PROP_COUNT,
    PROP_UPDATE_TIME,
    PROP_COUNT_CHANGE_TIME,
    PROP_COUNT_BEFORE_CHANGE,
    -1
};

static const int Pulse_Converter_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_INPUT_REFERENCE,
    PROP_RELIABILITY,
    PROP_COV_INCREMENT,
    PROP_COV_PERIOD,
    PROP_NOTIFICATION_CLASS,
    PROP_TIME_DELAY,
    PROP_HIGH_LIMIT,
    PROP_LOW_LIMIT,
    PROP_DEADBAND,
    PROP_LIMIT_ENABLE,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_TIME_DELAY_NORMAL,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Schedule_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_EFFECTIVE_PERIOD,
    PROP_SCHEDULE_DEFAULT,
    PROP_LIST_OF_OBJECT_PROPERTY_REFERENCES,
    PROP_PRIORITY_FOR_WRITING,
    PROP_STATUS_FLAGS,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    -1
};

static const int Schedule_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_WEEKLY_SCHEDULE,
    PROP_EXCEPTION_SCHEDULE,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_EVENT_STATE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Staging_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,
    PROP_PRESENT_STAGE,
    PROP_STAGES,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_UNITS,
    PROP_TARGET_REFERENCES,
    PROP_PRIORITY_FOR_WRITING,
    PROP_MIN_PRES_VALUE,
    PROP_MAX_PRES_VALUE,
    -1
};

static const int Staging_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_STAGE_NAMES,
    PROP_DESCRIPTION,
    PROP_DEFAULT_PRESENT_VALUE,
    PROP_COV_INCREMENT,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_VALUE_SOURCE,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Structured_View_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,      PROP_OBJECT_TYPE,
    PROP_NODE_TYPE,         PROP_SUBORDINATE_LIST, -1
};

static const int Structured_View_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_NODE_SUBTYPE,
    PROP_SUBORDINATE_ANNOTATIONS,
    PROP_SUBORDINATE_TAGS,
    PROP_SUBORDINATE_NODE_TYPES,
    PROP_SUBORDINATE_RELATIONSHIPS,
    PROP_DEFAULT_SUBORDINATE_RELATIONSHIP,
    PROP_REPRESENTS,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Time_Pattern_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, -1
};

static const int Time_Pattern_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Time_Value_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,  PROP_OBJECT_TYPE,
    PROP_PRESENT_VALUE,     PROP_STATUS_FLAGS, -1
};

static const int Time_Value_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_PRIORITY_ARRAY,
    PROP_RELINQUISH_DEFAULT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_CURRENT_COMMAND_PRIORITY,
    PROP_VALUE_SOURCE,
    PROP_VALUE_SOURCE_ARRAY,
    PROP_LAST_COMMAND_TIME,
    PROP_COMMAND_TIME_ARRAY,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_AUDIT_PRIORITY_FILTER,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Timer_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER, PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,       PROP_PRESENT_VALUE,
    PROP_STATUS_FLAGS,      PROP_TIMER_STATE,
    PROP_TIMER_RUNNING,     -1
};

static const int Timer_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_EVENT_STATE,
    PROP_RELIABILITY,
    PROP_OUT_OF_SERVICE,
    PROP_UPDATE_TIME,
    PROP_LAST_STATE_CHANGE,
    PROP_EXPIRATION_TIME,
    PROP_INITIAL_TIMEOUT,
    PROP_DEFAULT_TIMEOUT,
    PROP_MIN_PRES_VALUE,
    PROP_MAX_PRES_VALUE,
    PROP_RESOLUTION,
    PROP_STATE_CHANGE_VALUES,
    PROP_LIST_OF_OBJECT_PROPERTY_REFERENCES,
    PROP_PRIORITY_FOR_WRITING,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_NOTIFICATION_CLASS,
    PROP_TIME_DELAY,
    PROP_TIME_DELAY_NORMAL,
    PROP_ALARM_VALUES,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Trend_Log_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_ENABLE,
    PROP_STOP_WHEN_FULL,
    PROP_BUFFER_SIZE,
    PROP_LOG_BUFFER,
    PROP_RECORD_COUNT,
    PROP_TOTAL_RECORD_COUNT,
    PROP_LOGGING_TYPE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    -1
};

static const int Trend_Log_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_START_TIME,
    PROP_STOP_TIME,
    PROP_LOG_DEVICE_OBJECT_PROPERTY,
    PROP_LOG_INTERVAL,
    PROP_COV_RESUBSCRIPTION_INTERVAL,
    PROP_CLIENT_COV_INCREMENT,
    PROP_ALIGN_INTERVALS,
    PROP_INTERVAL_OFFSET,
    PROP_TRIGGER,
    PROP_RELIABILITY,
    PROP_NOTIFICATION_THRESHOLD,
    PROP_RECORDS_SINCE_NOTIFICATION,
    PROP_LAST_NOTIFY_RECORD,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

static const int Trend_Log_Multiple_Properties_Required[] = {
    /* unordered list of properties */
    PROP_OBJECT_IDENTIFIER,
    PROP_OBJECT_NAME,
    PROP_OBJECT_TYPE,
    PROP_STATUS_FLAGS,
    PROP_EVENT_STATE,
    PROP_ENABLE,
    PROP_LOG_DEVICE_OBJECT_PROPERTY,
    PROP_LOGGING_TYPE,
    PROP_LOG_INTERVAL,
    PROP_STOP_WHEN_FULL,
    PROP_BUFFER_SIZE,
    PROP_LOG_BUFFER,
    PROP_RECORD_COUNT,
    PROP_TOTAL_RECORD_COUNT,
    -1
};

static const int Trend_Log_Multiple_Properties_Optional[] = {
    /* unordered list of properties */
    PROP_DESCRIPTION,
    PROP_RELIABILITY,
    PROP_START_TIME,
    PROP_STOP_TIME,
    PROP_ALIGN_INTERVALS,
    PROP_INTERVAL_OFFSET,
    PROP_TRIGGER,
    PROP_NOTIFICATION_THRESHOLD,
    PROP_RECORDS_SINCE_NOTIFICATION,
    PROP_LAST_NOTIFY_RECORD,
    PROP_NOTIFICATION_CLASS,
    PROP_EVENT_ENABLE,
    PROP_ACKED_TRANSITIONS,
    PROP_NOTIFY_TYPE,
    PROP_EVENT_TIME_STAMPS,
    PROP_EVENT_MESSAGE_TEXTS,
    PROP_EVENT_MESSAGE_TEXTS_CONFIG,
    PROP_EVENT_DETECTION_ENABLE,
    PROP_EVENT_ALGORITHM_INHIBIT_REF,
    PROP_EVENT_ALGORITHM_INHIBIT,
    PROP_RELIABILITY_EVALUATION_INHIBIT,
    PROP_AUDIT_LEVEL,
    PROP_AUDITABLE_OPERATIONS,
    PROP_TAGS,
    PROP_PROFILE_LOCATION,
    PROP_PROFILE_NAME,
    -1
};

/**
 * Function that returns the list of all Optional properties
 * of known standard objects.
 *
 * @param object_type - enumerated BACNET_OBJECT_TYPE
 * @return returns a pointer to a '-1' terminated array of
 * type 'int' that contain BACnet object properties for the given object
 * type.
 */
const int *property_list_optional(BACNET_OBJECT_TYPE object_type)
{
    const int *pList = NULL;

    switch (object_type) {
        case OBJECT_ACCESS_CREDENTIAL:
            pList = Access_Credential_Properties_Optional;
            break;
        case OBJECT_ACCESS_DOOR:
            pList = Access_Door_Properties_Optional;
            break;
        case OBJECT_ACCESS_POINT:
            pList = Access_Point_Properties_Optional;
            break;
        case OBJECT_ACCESS_RIGHTS:
            pList = Access_Rights_Properties_Optional;
            break;
        case OBJECT_ACCESS_USER:
            pList = Access_User_Properties_Optional;
            break;
        case OBJECT_ACCESS_ZONE:
            pList = Access_Zone_Properties_Optional;
            break;
        case OBJECT_ACCUMULATOR:
            pList = Accumulator_Properties_Optional;
            break;
        case OBJECT_ALERT_ENROLLMENT:
            pList = Alert_Enrollment_Properties_Optional;
            break;
        case OBJECT_ANALOG_INPUT:
            pList = Analog_Input_Properties_Optional;
            break;
        case OBJECT_ANALOG_OUTPUT:
            pList = Analog_Output_Properties_Optional;
            break;
        case OBJECT_ANALOG_VALUE:
            pList = Analog_Value_Properties_Optional;
            break;
        case OBJECT_AVERAGING:
            pList = Averaging_Properties_Optional;
            break;
        case OBJECT_BINARY_INPUT:
            pList = Binary_Input_Properties_Optional;
            break;
        case OBJECT_BINARY_LIGHTING_OUTPUT:
            pList = Binary_Lighting_Output_Properties_Optional;
            break;
        case OBJECT_BINARY_OUTPUT:
            pList = Binary_Output_Properties_Optional;
            break;
        case OBJECT_BINARY_VALUE:
            pList = Binary_Value_Properties_Optional;
            break;
        case OBJECT_BITSTRING_VALUE:
            pList = BitString_Value_Properties_Optional;
            break;
        case OBJECT_CALENDAR:
            pList = Calendar_Properties_Optional;
            break;
        case OBJECT_CHANNEL:
            pList = Channel_Properties_Optional;
            break;
        case OBJECT_COMMAND:
            pList = Command_Properties_Optional;
            break;
        case OBJECT_CHARACTERSTRING_VALUE:
            pList = CharacterString_Value_Properties_Optional;
            break;
        case OBJECT_COLOR:
            pList = Color_Properties_Optional;
            break;
        case OBJECT_COLOR_TEMPERATURE:
            pList = Color_Temperature_Properties_Optional;
            break;
        case OBJECT_CREDENTIAL_DATA_INPUT:
            pList = Credential_Data_Input_Properties_Optional;
            break;
        case OBJECT_DATE_PATTERN_VALUE:
            pList = Date_Pattern_Value_Properties_Optional;
            break;
        case OBJECT_DATE_VALUE:
            pList = Date_Value_Properties_Optional;
            break;
        case OBJECT_DATETIME_PATTERN_VALUE:
            pList = DateTime_Pattern_Value_Properties_Optional;
            break;
        case OBJECT_DATETIME_VALUE:
            pList = DateTime_Value_Properties_Optional;
            break;
        case OBJECT_DEVICE:
            pList = Device_Properties_Optional;
            break;
        case OBJECT_ELEVATOR_GROUP:
            pList = Elevator_Group_Properties_Optional;
            break;
        case OBJECT_ESCALATOR:
            pList = Escalator_Properties_Optional;
            break;
        case OBJECT_EVENT_ENROLLMENT:
            pList = Event_Enrollment_Properties_Optional;
            break;
        case OBJECT_EVENT_LOG:
            pList = Event_Log_Properties_Optional;
            break;
        case OBJECT_FILE:
            pList = File_Properties_Optional;
            break;
        case OBJECT_GLOBAL_GROUP:
            pList = Global_Group_Properties_Optional;
            break;
        case OBJECT_GROUP:
            pList = Group_Properties_Optional;
            break;
        case OBJECT_INTEGER_VALUE:
            pList = Integer_Value_Properties_Optional;
            break;
        case OBJECT_LARGE_ANALOG_VALUE:
            pList = Large_Analog_Value_Properties_Optional;
            break;
        case OBJECT_LIFE_SAFETY_POINT:
            pList = Life_Safety_Point_Properties_Optional;
            break;
        case OBJECT_LIFE_SAFETY_ZONE:
            pList = Life_Safety_Zone_Properties_Optional;
            break;
        case OBJECT_LIFT:
            pList = Lift_Properties_Optional;
            break;
        case OBJECT_LIGHTING_OUTPUT:
            pList = Lighting_Output_Properties_Optional;
            break;
        case OBJECT_LOAD_CONTROL:
            pList = Load_Control_Properties_Optional;
            break;
        case OBJECT_LOOP:
            pList = Loop_Properties_Optional;
            break;
        case OBJECT_MULTI_STATE_INPUT:
            pList = Multistate_Input_Properties_Optional;
            break;
        case OBJECT_MULTI_STATE_OUTPUT:
            pList = Multistate_Output_Properties_Optional;
            break;
        case OBJECT_MULTI_STATE_VALUE:
            pList = Multistate_Value_Properties_Optional;
            break;
        case OBJECT_NETWORK_PORT:
            pList = Network_Port_Properties_Optional;
            break;
        case OBJECT_NETWORK_SECURITY:
            pList = Network_Security_Properties_Optional;
            break;
        case OBJECT_NOTIFICATION_CLASS:
            pList = Notification_Class_Properties_Optional;
            break;
        case OBJECT_NOTIFICATION_FORWARDER:
            pList = Notification_Forwarder_Properties_Optional;
            break;
        case OBJECT_OCTETSTRING_VALUE:
            pList = OctetString_Value_Properties_Optional;
            break;
        case OBJECT_POSITIVE_INTEGER_VALUE:
            pList = Positive_Integer_Value_Properties_Optional;
            break;
        case OBJECT_PROGRAM:
            pList = Program_Properties_Optional;
            break;
        case OBJECT_PULSE_CONVERTER:
            pList = Pulse_Converter_Properties_Optional;
            break;
        case OBJECT_SCHEDULE:
            pList = Schedule_Properties_Optional;
            break;
        case OBJECT_STAGING:
            pList = Staging_Properties_Optional;
            break;
        case OBJECT_STRUCTURED_VIEW:
            pList = Structured_View_Properties_Optional;
            break;
        case OBJECT_TIME_PATTERN_VALUE:
            pList = Time_Pattern_Value_Properties_Optional;
            break;
        case OBJECT_TIME_VALUE:
            pList = Time_Value_Properties_Optional;
            break;
        case OBJECT_TIMER:
            pList = Timer_Properties_Optional;
            break;
        case OBJECT_TRENDLOG:
            pList = Trend_Log_Properties_Optional;
            break;
        case OBJECT_TREND_LOG_MULTIPLE:
            pList = Trend_Log_Multiple_Properties_Optional;
            break;
        default:
            pList = Default_Properties_Optional;
            break;
    }

    return pList;
}

/**
 * Function that returns the list of Required properties
 * of known standard objects.
 *
 * @param object_type - enumerated BACNET_OBJECT_TYPE
 * @return returns a pointer to a '-1' terminated array of
 * type 'int' that contain BACnet object properties for the given object
 * type.
 */
const int *property_list_required(BACNET_OBJECT_TYPE object_type)
{
    const int *pList = NULL;

    switch (object_type) {
        case OBJECT_ACCESS_CREDENTIAL:
            pList = Access_Credential_Properties_Required;
            break;
        case OBJECT_ACCESS_DOOR:
            pList = Access_Door_Properties_Required;
            break;
        case OBJECT_ACCESS_POINT:
            pList = Access_Point_Properties_Required;
            break;
        case OBJECT_ACCESS_RIGHTS:
            pList = Access_Rights_Properties_Required;
            break;
        case OBJECT_ACCESS_USER:
            pList = Access_User_Properties_Required;
            break;
        case OBJECT_ACCESS_ZONE:
            pList = Access_Zone_Properties_Required;
            break;
        case OBJECT_ACCUMULATOR:
            pList = Accumulator_Properties_Required;
            break;
        case OBJECT_ALERT_ENROLLMENT:
            pList = Alert_Enrollment_Properties_Required;
            break;
        case OBJECT_ANALOG_INPUT:
            pList = Analog_Input_Properties_Required;
            break;
        case OBJECT_ANALOG_OUTPUT:
            pList = Analog_Output_Properties_Required;
            break;
        case OBJECT_ANALOG_VALUE:
            pList = Analog_Value_Properties_Required;
            break;
        case OBJECT_AVERAGING:
            pList = Averaging_Properties_Required;
            break;
        case OBJECT_BINARY_INPUT:
            pList = Binary_Input_Properties_Required;
            break;
        case OBJECT_BINARY_LIGHTING_OUTPUT:
            pList = Binary_Lighting_Output_Properties_Required;
            break;
        case OBJECT_BINARY_OUTPUT:
            pList = Binary_Output_Properties_Required;
            break;
        case OBJECT_BINARY_VALUE:
            pList = Binary_Value_Properties_Required;
            break;
        case OBJECT_BITSTRING_VALUE:
            pList = BitString_Value_Properties_Required;
            break;
        case OBJECT_CALENDAR:
            pList = Calendar_Properties_Required;
            break;
        case OBJECT_CHANNEL:
            pList = Channel_Properties_Required;
            break;
        case OBJECT_COMMAND:
            pList = Command_Properties_Required;
            break;
        case OBJECT_CHARACTERSTRING_VALUE:
            pList = CharacterString_Value_Properties_Required;
            break;
        case OBJECT_COLOR:
            pList = Color_Properties_Required;
            break;
        case OBJECT_COLOR_TEMPERATURE:
            pList = Color_Temperature_Properties_Required;
            break;
        case OBJECT_CREDENTIAL_DATA_INPUT:
            pList = Credential_Data_Input_Properties_Required;
            break;
        case OBJECT_DATE_PATTERN_VALUE:
            pList = Date_Pattern_Value_Properties_Required;
            break;
        case OBJECT_DATE_VALUE:
            pList = Date_Value_Properties_Required;
            break;
        case OBJECT_DATETIME_PATTERN_VALUE:
            pList = DateTime_Pattern_Value_Properties_Required;
            break;
        case OBJECT_DATETIME_VALUE:
            pList = DateTime_Value_Properties_Required;
            break;
        case OBJECT_DEVICE:
            pList = Device_Properties_Required;
            break;
        case OBJECT_ELEVATOR_GROUP:
            pList = Elevator_Group_Properties_Required;
            break;
        case OBJECT_ESCALATOR:
            pList = Escalator_Properties_Required;
            break;
        case OBJECT_EVENT_ENROLLMENT:
            pList = Event_Enrollment_Properties_Required;
            break;
        case OBJECT_EVENT_LOG:
            pList = Event_Log_Properties_Required;
            break;
        case OBJECT_FILE:
            pList = File_Properties_Required;
            break;
        case OBJECT_GLOBAL_GROUP:
            pList = Global_Group_Properties_Required;
            break;
        case OBJECT_GROUP:
            pList = Group_Properties_Required;
            break;
        case OBJECT_INTEGER_VALUE:
            pList = Integer_Value_Properties_Required;
            break;
        case OBJECT_LARGE_ANALOG_VALUE:
            pList = Large_Analog_Value_Properties_Required;
            break;
        case OBJECT_LIFE_SAFETY_POINT:
            pList = Life_Safety_Point_Properties_Required;
            break;
        case OBJECT_LIFE_SAFETY_ZONE:
            pList = Life_Safety_Zone_Properties_Required;
            break;
        case OBJECT_LIFT:
            pList = Lift_Properties_Required;
            break;
        case OBJECT_LIGHTING_OUTPUT:
            pList = Lighting_Output_Properties_Required;
            break;
        case OBJECT_LOAD_CONTROL:
            pList = Load_Control_Properties_Required;
            break;
        case OBJECT_LOOP:
            pList = Loop_Properties_Required;
            break;
        case OBJECT_MULTI_STATE_INPUT:
            pList = Multistate_Input_Properties_Required;
            break;
        case OBJECT_MULTI_STATE_OUTPUT:
            pList = Multistate_Output_Properties_Required;
            break;
        case OBJECT_MULTI_STATE_VALUE:
            pList = Multistate_Value_Properties_Required;
            break;
        case OBJECT_NETWORK_PORT:
            pList = Network_Port_Properties_Required;
            break;
        case OBJECT_NETWORK_SECURITY:
            pList = Network_Security_Properties_Required;
            break;
        case OBJECT_NOTIFICATION_CLASS:
            pList = Notification_Class_Properties_Required;
            break;
        case OBJECT_NOTIFICATION_FORWARDER:
            pList = Notification_Forwarder_Properties_Required;
            break;
        case OBJECT_OCTETSTRING_VALUE:
            pList = OctetString_Value_Properties_Required;
            break;
        case OBJECT_POSITIVE_INTEGER_VALUE:
            pList = Positive_Integer_Value_Properties_Required;
            break;
        case OBJECT_PROGRAM:
            pList = Program_Properties_Required;
            break;
        case OBJECT_PULSE_CONVERTER:
            pList = Pulse_Converter_Properties_Required;
            break;
        case OBJECT_SCHEDULE:
            pList = Schedule_Properties_Required;
            break;
        case OBJECT_STAGING:
            pList = Staging_Properties_Required;
            break;
        case OBJECT_STRUCTURED_VIEW:
            pList = Structured_View_Properties_Required;
            break;
        case OBJECT_TIME_PATTERN_VALUE:
            pList = Time_Pattern_Value_Properties_Required;
            break;
        case OBJECT_TIME_VALUE:
            pList = Time_Value_Properties_Required;
            break;
        case OBJECT_TIMER:
            pList = Timer_Properties_Required;
            break;
        case OBJECT_TRENDLOG:
            pList = Trend_Log_Properties_Required;
            break;
        case OBJECT_TREND_LOG_MULTIPLE:
            pList = Trend_Log_Multiple_Properties_Required;
            break;
        default:
            pList = Default_Properties_Required;
            break;
    }

    return pList;
}

/**
 * Function that returns the list of Required or Optional properties
 * of known standard objects.
 *
 * @param object_type - enumerated BACNET_OBJECT_TYPE
 * @param pPropertyList - returns a pointer to two '-1' terminated arrays of
 * type 'int' that contain BACnet object properties for the given object
 * type.
 */
void property_list_special(
    BACNET_OBJECT_TYPE object_type,
    struct special_property_list_t *pPropertyList)
{
    if (pPropertyList == NULL) {
        return;
    }
    pPropertyList->Required.pList = property_list_required(object_type);
    pPropertyList->Optional.pList = property_list_optional(object_type);
    pPropertyList->Proprietary.pList = NULL;
    /* Fetch the counts if available otherwise zero them */
    pPropertyList->Required.count =
        property_list_count(pPropertyList->Required.pList);
    pPropertyList->Optional.count =
        property_list_count(pPropertyList->Optional.pList);
    pPropertyList->Proprietary.count = 0;

    return;
}

BACNET_PROPERTY_ID property_list_special_property(
    BACNET_OBJECT_TYPE object_type,
    BACNET_PROPERTY_ID special_property,
    unsigned index)
{
    BACNET_PROPERTY_ID property = UINT32_MAX; /* return value */
    unsigned required, optional, proprietary;
    struct special_property_list_t PropertyList = { 0 };

    property_list_special(object_type, &PropertyList);
    required = PropertyList.Required.count;
    optional = PropertyList.Optional.count;
    proprietary = PropertyList.Proprietary.count;
    if (special_property == PROP_ALL) {
        if (index < required) {
            if (PropertyList.Required.pList) {
                property = PropertyList.Required.pList[index];
            }
        } else if (index < (required + optional)) {
            if (PropertyList.Optional.pList) {
                index -= required;
                property = PropertyList.Optional.pList[index];
            }
        } else if (index < (required + optional + proprietary)) {
            if (PropertyList.Proprietary.pList) {
                index -= (required + optional);
                property = PropertyList.Proprietary.pList[index];
            }
        }
    } else if (special_property == PROP_REQUIRED) {
        if (index < required) {
            if (PropertyList.Required.pList) {
                property = PropertyList.Required.pList[index];
            }
        }
    } else if (special_property == PROP_OPTIONAL) {
        if (index < optional) {
            if (PropertyList.Optional.pList) {
                property = PropertyList.Optional.pList[index];
            }
        }
    }

    return property;
}

unsigned property_list_special_count(
    BACNET_OBJECT_TYPE object_type, BACNET_PROPERTY_ID special_property)
{
    unsigned count = 0; /* return value */
    struct special_property_list_t PropertyList = { 0 };

    property_list_special(object_type, &PropertyList);
    if (special_property == PROP_ALL) {
        count = PropertyList.Required.count + PropertyList.Optional.count +
            PropertyList.Proprietary.count;
    } else if (special_property == PROP_REQUIRED) {
        count = PropertyList.Required.count;
    } else if (special_property == PROP_OPTIONAL) {
        count = PropertyList.Optional.count;
    }

    return count;
}
#endif

/**
 * @brief Determine if this property is required to be writable
 * @note Property Conformance Code of W means writable
 * @param object_type - object-type to be checked
 * @param object_property - object-property to be checked
 * @return true if the property is a required writable property
 */
bool property_list_writable_member(
    BACNET_OBJECT_TYPE object_type, BACNET_PROPERTY_ID object_property)
{
    bool writable = false;

    switch (object_type) {
        case OBJECT_ANALOG_OUTPUT:
        case OBJECT_BINARY_OUTPUT:
        case OBJECT_MULTI_STATE_OUTPUT:
        case OBJECT_BINARY_LIGHTING_OUTPUT:
        case OBJECT_ACCESS_DOOR:
        case OBJECT_COMMAND:
        case OBJECT_STAGING:
            if (object_property == PROP_PRESENT_VALUE) {
                writable = true;
            }
            break;
        case OBJECT_AVERAGING:
            if ((object_property == PROP_ATTEMPTED_SAMPLES) ||
                (object_property == PROP_WINDOW_INTERVAL) ||
                (object_property == PROP_WINDOW_SAMPLES)) {
                writable = true;
            }
            break;
        case OBJECT_FILE:
            if (object_property == PROP_ARCHIVE) {
                writable = true;
            }
            break;
        case OBJECT_LIFE_SAFETY_POINT:
        case OBJECT_LIFE_SAFETY_ZONE:
            if (object_property == PROP_MODE) {
                writable = true;
            }
            break;
        case OBJECT_PROGRAM:
            if (object_property == PROP_PROGRAM_CHANGE) {
                writable = true;
            }
            break;
        case OBJECT_PULSE_CONVERTER:
            if (object_property == PROP_ADJUST_VALUE) {
                writable = true;
            }
            break;
        case OBJECT_EVENT_LOG:
        case OBJECT_TRENDLOG:
        case OBJECT_TREND_LOG_MULTIPLE:
            if ((object_property == PROP_ENABLE) ||
                (object_property == PROP_RECORD_COUNT)) {
                writable = true;
            }
            break;
        case OBJECT_LOAD_CONTROL:
            if ((object_property == PROP_REQUESTED_SHED_LEVEL) ||
                (object_property == PROP_START_TIME) ||
                (object_property == PROP_SHED_DURATION) ||
                (object_property == PROP_DUTY_WINDOW) ||
                (object_property == PROP_SHED_LEVELS)) {
                writable = true;
            }
            break;
        case OBJECT_ACCESS_CREDENTIAL:
        case OBJECT_ACCESS_RIGHTS:
        case OBJECT_ACCESS_USER:
        case OBJECT_ACCESS_ZONE:
            if (object_property == PROP_GLOBAL_IDENTIFIER) {
                writable = true;
            }
            break;
        case OBJECT_NETWORK_SECURITY:
            if ((object_property == PROP_BASE_DEVICE_SECURITY_POLICY) ||
                (object_property == PROP_NETWORK_ACCESS_SECURITY_POLICIES) ||
                (object_property == PROP_SECURITY_TIME_WINDOW) ||
                (object_property == PROP_PACKET_REORDER_TIME) ||
                (object_property == PROP_LAST_KEY_SERVER) ||
                (object_property == PROP_SECURITY_PDU_TIMEOUT) ||
                (object_property == PROP_LAST_KEY_SERVER) ||
                (object_property == PROP_SECURITY_PDU_TIMEOUT) ||
                (object_property == PROP_DO_NOT_HIDE)) {
                writable = true;
            }
            break;
        case OBJECT_NOTIFICATION_FORWARDER:
            if (object_property == PROP_SUBSCRIBED_RECIPIENTS) {
                writable = true;
            }
            break;
        case OBJECT_CHANNEL:
            if ((object_property == PROP_PRESENT_VALUE) ||
                (object_property == PROP_LIST_OF_OBJECT_PROPERTY_REFERENCES) ||
                (object_property == PROP_CHANNEL_NUMBER) ||
                (object_property == PROP_CONTROL_GROUPS)) {
                writable = true;
            }
            break;
        case OBJECT_LIGHTING_OUTPUT:
            if ((object_property == PROP_PRESENT_VALUE) ||
                (object_property == PROP_LIGHTING_COMMAND)) {
                writable = true;
            }
            break;
        case OBJECT_AUDIT_LOG:
            if (object_property == PROP_ENABLE) {
                writable = true;
            }
            break;
        case OBJECT_COLOR:
        case OBJECT_COLOR_TEMPERATURE:
            if ((object_property == PROP_PRESENT_VALUE) ||
                (object_property == PROP_COLOR_COMMAND)) {
                writable = true;
            }
            break;
        default:
            break;
    }

    return writable;
}
