/**
 * @file
 * @brief BACnet defines and enumerations
 * @author Steve Karg <skarg@users.sourceforge.net>
 * @date 2004
 * @copyright SPDX-License-Identifier: MIT
 */
#ifndef BACNET_ENUMERATIONS_H
#define BACNET_ENUMERATIONS_H

typedef enum {
    PROP_ACKED_TRANSITIONS = 0,
    PROP_ACK_REQUIRED = 1,
    PROP_ACTION = 2,
    PROP_ACTION_TEXT = 3,
    PROP_ACTIVE_TEXT = 4,
    PROP_ACTIVE_VT_SESSIONS = 5,
    PROP_ALARM_VALUE = 6,
    PROP_ALARM_VALUES = 7,
    PROP_ALL = 8,
    PROP_ALL_WRITES_SUCCESSFUL = 9,
    PROP_APDU_SEGMENT_TIMEOUT = 10,
    PROP_APDU_TIMEOUT = 11,
    PROP_APPLICATION_SOFTWARE_VERSION = 12,
    PROP_ARCHIVE = 13,
    PROP_BIAS = 14,
    PROP_CHANGE_OF_STATE_COUNT = 15,
    PROP_CHANGE_OF_STATE_TIME = 16,
    PROP_NOTIFICATION_CLASS = 17,
    PROP_BLANK_1 = 18,
    PROP_CONTROLLED_VARIABLE_REFERENCE = 19,
    PROP_CONTROLLED_VARIABLE_UNITS = 20,
    PROP_CONTROLLED_VARIABLE_VALUE = 21,
    PROP_COV_INCREMENT = 22,
    PROP_DATE_LIST = 23,
    PROP_DAYLIGHT_SAVINGS_STATUS = 24,
    PROP_DEADBAND = 25,
    PROP_DERIVATIVE_CONSTANT = 26,
    PROP_DERIVATIVE_CONSTANT_UNITS = 27,
    PROP_DESCRIPTION = 28,
    PROP_DESCRIPTION_OF_HALT = 29,
    PROP_DEVICE_ADDRESS_BINDING = 30,
    PROP_DEVICE_TYPE = 31,
    PROP_EFFECTIVE_PERIOD = 32,
    PROP_ELAPSED_ACTIVE_TIME = 33,
    PROP_ERROR_LIMIT = 34,
    PROP_EVENT_ENABLE = 35,
    PROP_EVENT_STATE = 36,
    PROP_EVENT_TYPE = 37,
    PROP_EXCEPTION_SCHEDULE = 38,
    PROP_FAULT_VALUES = 39,
    PROP_FEEDBACK_VALUE = 40,
    PROP_FILE_ACCESS_METHOD = 41,
    PROP_FILE_SIZE = 42,
    PROP_FILE_TYPE = 43,
    PROP_FIRMWARE_REVISION = 44,
    PROP_HIGH_LIMIT = 45,
    PROP_INACTIVE_TEXT = 46,
    PROP_IN_PROCESS = 47,
    PROP_INSTANCE_OF = 48,
    PROP_INTEGRAL_CONSTANT = 49,
    PROP_INTEGRAL_CONSTANT_UNITS = 50,
    PROP_ISSUE_CONFIRMED_NOTIFICATIONS = 51,
    PROP_LIMIT_ENABLE = 52,
    PROP_LIST_OF_GROUP_MEMBERS = 53,
    PROP_LIST_OF_OBJECT_PROPERTY_REFERENCES = 54,
    PROP_LIST_OF_SESSION_KEYS = 55,
    PROP_LOCAL_DATE = 56,
    PROP_LOCAL_TIME = 57,
    PROP_LOCATION = 58,
    PROP_LOW_LIMIT = 59,
    PROP_MANIPULATED_VARIABLE_REFERENCE = 60,
    PROP_MAXIMUM_OUTPUT = 61,
    PROP_MAX_APDU_LENGTH_ACCEPTED = 62,
    PROP_MAX_INFO_FRAMES = 63,
    PROP_MAX_MASTER = 64,
    PROP_MAX_PRES_VALUE = 65,
    PROP_MINIMUM_OFF_TIME = 66,
    PROP_MINIMUM_ON_TIME = 67,
    PROP_MINIMUM_OUTPUT = 68,
    PROP_MIN_PRES_VALUE = 69,
    PROP_MODEL_NAME = 70,
    PROP_MODIFICATION_DATE = 71,
    PROP_NOTIFY_TYPE = 72,
    PROP_NUMBER_OF_APDU_RETRIES = 73,
    PROP_NUMBER_OF_STATES = 74,
    PROP_OBJECT_IDENTIFIER = 75,
    PROP_OBJECT_LIST = 76,
    PROP_OBJECT_NAME = 77,
    PROP_OBJECT_PROPERTY_REFERENCE = 78,
    PROP_OBJECT_TYPE = 79,
    PROP_OPTIONAL = 80,
    PROP_OUT_OF_SERVICE = 81,
    PROP_OUTPUT_UNITS = 82,
    PROP_EVENT_PARAMETERS = 83,
    PROP_POLARITY = 84,
    PROP_PRESENT_VALUE = 85,
    PROP_PRIORITY = 86,
    PROP_PRIORITY_ARRAY = 87,
    PROP_PRIORITY_FOR_WRITING = 88,
    PROP_PROCESS_IDENTIFIER = 89,
    PROP_PROGRAM_CHANGE = 90,
    PROP_PROGRAM_LOCATION = 91,
    PROP_PROGRAM_STATE = 92,
    PROP_PROPORTIONAL_CONSTANT = 93,
    PROP_PROPORTIONAL_CONSTANT_UNITS = 94,
    PROP_PROTOCOL_CONFORMANCE_CLASS = 95, /* deleted in version 1 revision 2 */
    PROP_PROTOCOL_OBJECT_TYPES_SUPPORTED = 96,
    PROP_PROTOCOL_SERVICES_SUPPORTED = 97,
    PROP_PROTOCOL_VERSION = 98,
    PROP_READ_ONLY = 99,
    PROP_REASON_FOR_HALT = 100,
    PROP_RECIPIENT = 101,
    PROP_RECIPIENT_LIST = 102,
    PROP_RELIABILITY = 103,
    PROP_RELINQUISH_DEFAULT = 104,
    PROP_REQUIRED = 105,
    PROP_RESOLUTION = 106,
    PROP_SEGMENTATION_SUPPORTED = 107,
    PROP_SETPOINT = 108,
    PROP_SETPOINT_REFERENCE = 109,
    PROP_STATE_TEXT = 110,
    PROP_STATUS_FLAGS = 111,
    PROP_SYSTEM_STATUS = 112,
    PROP_TIME_DELAY = 113,
    PROP_TIME_OF_ACTIVE_TIME_RESET = 114,
    PROP_TIME_OF_STATE_COUNT_RESET = 115,
    PROP_TIME_SYNCHRONIZATION_RECIPIENTS = 116,
    PROP_UNITS = 117,
    PROP_UPDATE_INTERVAL = 118,
    PROP_UTC_OFFSET = 119,
    PROP_VENDOR_IDENTIFIER = 120,
    PROP_VENDOR_NAME = 121,
    PROP_VT_CLASSES_SUPPORTED = 122,
    PROP_WEEKLY_SCHEDULE = 123,
    PROP_ATTEMPTED_SAMPLES = 124,
    PROP_AVERAGE_VALUE = 125,
    PROP_BUFFER_SIZE = 126,
    PROP_CLIENT_COV_INCREMENT = 127,
    PROP_COV_RESUBSCRIPTION_INTERVAL = 128,
    PROP_CURRENT_NOTIFY_TIME = 129,
    PROP_EVENT_TIME_STAMPS = 130,
    PROP_LOG_BUFFER = 131,
    PROP_LOG_DEVICE_OBJECT_PROPERTY = 132,
    /* The enable property is renamed from log-enable in
       Addendum b to ANSI/ASHRAE 135-2004(135b-2) */
    PROP_ENABLE = 133,
    PROP_LOG_INTERVAL = 134,
    PROP_MAXIMUM_VALUE = 135,
    PROP_MINIMUM_VALUE = 136,
    PROP_NOTIFICATION_THRESHOLD = 137,
    PROP_PREVIOUS_NOTIFY_TIME = 138,
    PROP_PROTOCOL_REVISION = 139,
    PROP_RECORDS_SINCE_NOTIFICATION = 140,
    PROP_RECORD_COUNT = 141,
    PROP_START_TIME = 142,
    PROP_STOP_TIME = 143,
    PROP_STOP_WHEN_FULL = 144,
    PROP_TOTAL_RECORD_COUNT = 145,
    PROP_VALID_SAMPLES = 146,
    PROP_WINDOW_INTERVAL = 147,
    PROP_WINDOW_SAMPLES = 148,
    PROP_MAXIMUM_VALUE_TIMESTAMP = 149,
    PROP_MINIMUM_VALUE_TIMESTAMP = 150,
    PROP_VARIANCE_VALUE = 151,
    PROP_ACTIVE_COV_SUBSCRIPTIONS = 152,
    PROP_BACKUP_FAILURE_TIMEOUT = 153,
    PROP_CONFIGURATION_FILES = 154,
    PROP_DATABASE_REVISION = 155,
    PROP_DIRECT_READING = 156,
    PROP_LAST_RESTORE_TIME = 157,
    PROP_MAINTENANCE_REQUIRED = 158,
    PROP_MEMBER_OF = 159,
    PROP_MODE = 160,
    PROP_OPERATION_EXPECTED = 161,
    PROP_SETTING = 162,
    PROP_SILENCED = 163,
    PROP_TRACKING_VALUE = 164,
    PROP_ZONE_MEMBERS = 165,
    PROP_LIFE_SAFETY_ALARM_VALUES = 166,
    PROP_MAX_SEGMENTS_ACCEPTED = 167,
    PROP_PROFILE_NAME = 168,
    PROP_AUTO_SLAVE_DISCOVERY = 169,
    PROP_MANUAL_SLAVE_ADDRESS_BINDING = 170,
    PROP_SLAVE_ADDRESS_BINDING = 171,
    PROP_SLAVE_PROXY_ENABLE = 172,
    PROP_LAST_NOTIFY_RECORD = 173,
    PROP_SCHEDULE_DEFAULT = 174,
    PROP_ACCEPTED_MODES = 175,
    PROP_ADJUST_VALUE = 176,
    PROP_COUNT = 177,
    PROP_COUNT_BEFORE_CHANGE = 178,
    PROP_COUNT_CHANGE_TIME = 179,
    PROP_COV_PERIOD = 180,
    PROP_INPUT_REFERENCE = 181,
    PROP_LIMIT_MONITORING_INTERVAL = 182,
    PROP_LOGGING_OBJECT = 183,
    PROP_LOGGING_RECORD = 184,
    PROP_PRESCALE = 185,
    PROP_PULSE_RATE = 186,
    PROP_SCALE = 187,
    PROP_SCALE_FACTOR = 188,
    PROP_UPDATE_TIME = 189,
    PROP_VALUE_BEFORE_CHANGE = 190,
    PROP_VALUE_SET = 191,
    PROP_VALUE_CHANGE_TIME = 192,
    /* enumerations 193-206 are new */
    PROP_ALIGN_INTERVALS = 193,
    /* enumeration 194 is unassigned */
    PROP_INTERVAL_OFFSET = 195,
    PROP_LAST_RESTART_REASON = 196,
    PROP_LOGGING_TYPE = 197,
    /* enumeration 198-201 is unassigned */
    PROP_RESTART_NOTIFICATION_RECIPIENTS = 202,
    PROP_TIME_OF_DEVICE_RESTART = 203,
    PROP_TIME_SYNCHRONIZATION_INTERVAL = 204,
    PROP_TRIGGER = 205,
    PROP_UTC_TIME_SYNCHRONIZATION_RECIPIENTS = 206,
    /* enumerations 207-211 are used in Addendum d to ANSI/ASHRAE 135-2004 */
    PROP_NODE_SUBTYPE = 207,
    PROP_NODE_TYPE = 208,
    PROP_STRUCTURED_OBJECT_LIST = 209,
    PROP_SUBORDINATE_ANNOTATIONS = 210,
    PROP_SUBORDINATE_LIST = 211,
    /* enumerations 212-225 are used in Addendum e to ANSI/ASHRAE 135-2004 */
    PROP_ACTUAL_SHED_LEVEL = 212,
    PROP_DUTY_WINDOW = 213,
    PROP_EXPECTED_SHED_LEVEL = 214,
    PROP_FULL_DUTY_BASELINE = 215,
    /* enumerations 216-217 are unassigned */
    /* enumerations 212-225 are used in Addendum e to ANSI/ASHRAE 135-2004 */
    PROP_REQUESTED_SHED_LEVEL = 218,
    PROP_SHED_DURATION = 219,
    PROP_SHED_LEVEL_DESCRIPTIONS = 220,
    PROP_SHED_LEVELS = 221,
    PROP_STATE_DESCRIPTION = 222,
    /* enumerations 223-225 are unassigned  */
    /* enumerations 226-235 are used in Addendum f to ANSI/ASHRAE 135-2004 */
    PROP_DOOR_ALARM_STATE = 226,
    PROP_DOOR_EXTENDED_PULSE_TIME = 227,
    PROP_DOOR_MEMBERS = 228,
    PROP_DOOR_OPEN_TOO_LONG_TIME = 229,
    PROP_DOOR_PULSE_TIME = 230,
    PROP_DOOR_STATUS = 231,
    PROP_DOOR_UNLOCK_DELAY_TIME = 232,
    PROP_LOCK_STATUS = 233,
    PROP_MASKED_ALARM_VALUES = 234,
    PROP_SECURED_STATUS = 235,
    /* enumerations 236-243 are unassigned  */
    /* enumerations 244-311 are used in Addendum j to ANSI/ASHRAE 135-2004 */
    PROP_ABSENTEE_LIMIT = 244,
    PROP_ACCESS_ALARM_EVENTS = 245,
    PROP_ACCESS_DOORS = 246,
    PROP_ACCESS_EVENT = 247,
    PROP_ACCESS_EVENT_AUTHENTICATION_FACTOR = 248,
    PROP_ACCESS_EVENT_CREDENTIAL = 249,
    PROP_ACCESS_EVENT_TIME = 250,
    PROP_ACCESS_TRANSACTION_EVENTS = 251,
    PROP_ACCOMPANIMENT = 252,
    PROP_ACCOMPANIMENT_TIME = 253,
    PROP_ACTIVATION_TIME = 254,
    PROP_ACTIVE_AUTHENTICATION_POLICY = 255,
    PROP_ASSIGNED_ACCESS_RIGHTS = 256,
    PROP_AUTHENTICATION_FACTORS = 257,
    PROP_AUTHENTICATION_POLICY_LIST = 258,
    PROP_AUTHENTICATION_POLICY_NAMES = 259,
    PROP_AUTHENTICATION_STATUS = 260,
    PROP_AUTHORIZATION_MODE = 261,
    PROP_BELONGS_TO = 262,
    PROP_CREDENTIAL_DISABLE = 263,
    PROP_CREDENTIAL_STATUS = 264,
    PROP_CREDENTIALS = 265,
    PROP_CREDENTIALS_IN_ZONE = 266,
    PROP_DAYS_REMAINING = 267,
    PROP_ENTRY_POINTS = 268,
    PROP_EXIT_POINTS = 269,
    PROP_EXPIRATION_TIME = 270,
    PROP_EXTENDED_TIME_ENABLE = 271,
    PROP_FAILED_ATTEMPT_EVENTS = 272,
    PROP_FAILED_ATTEMPTS = 273,
    PROP_FAILED_ATTEMPTS_TIME = 274,
    PROP_LAST_ACCESS_EVENT = 275,
    PROP_LAST_ACCESS_POINT = 276,
    PROP_LAST_CREDENTIAL_ADDED = 277,
    PROP_LAST_CREDENTIAL_ADDED_TIME = 278,
    PROP_LAST_CREDENTIAL_REMOVED = 279,
    PROP_LAST_CREDENTIAL_REMOVED_TIME = 280,
    PROP_LAST_USE_TIME = 281,
    PROP_LOCKOUT = 282,
    PROP_LOCKOUT_RELINQUISH_TIME = 283,
    PROP_MASTER_EXEMPTION = 284,
    PROP_MAX_FAILED_ATTEMPTS = 285,
    PROP_MEMBERS = 286,
    PROP_MUSTER_POINT = 287,
    PROP_NEGATIVE_ACCESS_RULES = 288,
    PROP_NUMBER_OF_AUTHENTICATION_POLICIES = 289,
    PROP_OCCUPANCY_COUNT = 290,
    PROP_OCCUPANCY_COUNT_ADJUST = 291,
    PROP_OCCUPANCY_COUNT_ENABLE = 292,
    PROP_OCCUPANCY_EXEMPTION = 293,
    PROP_OCCUPANCY_LOWER_LIMIT = 294,
    PROP_OCCUPANCY_LOWER_LIMIT_ENFORCED = 295,
    PROP_OCCUPANCY_STATE = 296,
    PROP_OCCUPANCY_UPPER_LIMIT = 297,
    PROP_OCCUPANCY_UPPER_LIMIT_ENFORCED = 298,
    PROP_PASSBACK_EXEMPTION = 299,
    PROP_PASSBACK_MODE = 300,
    PROP_PASSBACK_TIMEOUT = 301,
    PROP_POSITIVE_ACCESS_RULES = 302,
    PROP_REASON_FOR_DISABLE = 303,
    PROP_SUPPORTED_FORMATS = 304,
    PROP_SUPPORTED_FORMAT_CLASSES = 305,
    PROP_THREAT_AUTHORITY = 306,
    PROP_THREAT_LEVEL = 307,
    PROP_TRACE_FLAG = 308,
    PROP_TRANSACTION_NOTIFICATION_CLASS = 309,
    PROP_USER_EXTERNAL_IDENTIFIER = 310,
    PROP_USER_INFORMATION_REFERENCE = 311,
    /* enumerations 312-316 are unassigned */
    PROP_USER_NAME = 317,
    PROP_USER_TYPE = 318,
    PROP_USES_REMAINING = 319,
    PROP_ZONE_FROM = 320,
    PROP_ZONE_TO = 321,
    PROP_ACCESS_EVENT_TAG = 322,
    PROP_GLOBAL_IDENTIFIER = 323,
    /* enumerations 324-325 are unassigned */
    PROP_VERIFICATION_TIME = 326,
    PROP_BASE_DEVICE_SECURITY_POLICY = 327,
    PROP_DISTRIBUTION_KEY_REVISION = 328,
    PROP_DO_NOT_HIDE = 329,
    PROP_KEY_SETS = 330,
    PROP_LAST_KEY_SERVER = 331,
    PROP_NETWORK_ACCESS_SECURITY_POLICIES = 332,
    PROP_PACKET_REORDER_TIME = 333,
    PROP_SECURITY_PDU_TIMEOUT = 334,
    PROP_SECURITY_TIME_WINDOW = 335,
    PROP_SUPPORTED_SECURITY_ALGORITHM = 336,
    PROP_UPDATE_KEY_SET_TIMEOUT = 337,
    PROP_BACKUP_AND_RESTORE_STATE = 338,
    PROP_BACKUP_PREPARATION_TIME = 339,
    PROP_RESTORE_COMPLETION_TIME = 340,
    PROP_RESTORE_PREPARATION_TIME = 341,
    /* enumerations 342-344 are defined in Addendum 2008-w */
    PROP_BIT_MASK = 342,
    PROP_BIT_TEXT = 343,
    PROP_IS_UTC = 344,
    PROP_GROUP_MEMBERS = 345,
    PROP_GROUP_MEMBER_NAMES = 346,
    PROP_MEMBER_STATUS_FLAGS = 347,
    PROP_REQUESTED_UPDATE_INTERVAL = 348,
    PROP_COVU_PERIOD = 349,
    PROP_COVU_RECIPIENTS = 350,
    PROP_EVENT_MESSAGE_TEXTS = 351,
    /* enumerations 352-363 are defined in Addendum 2010-af */
    PROP_EVENT_MESSAGE_TEXTS_CONFIG = 352,
    PROP_EVENT_DETECTION_ENABLE = 353,
    PROP_EVENT_ALGORITHM_INHIBIT = 354,
    PROP_EVENT_ALGORITHM_INHIBIT_REF = 355,
    PROP_TIME_DELAY_NORMAL = 356,
    PROP_RELIABILITY_EVALUATION_INHIBIT = 357,
    PROP_FAULT_PARAMETERS = 358,
    PROP_FAULT_TYPE = 359,
    PROP_LOCAL_FORWARDING_ONLY = 360,
    PROP_PROCESS_IDENTIFIER_FILTER = 361,
    PROP_SUBSCRIBED_RECIPIENTS = 362,
    PROP_PORT_FILTER = 363,
    /* enumeration 364 is defined in Addendum 2010-ae */
    PROP_AUTHORIZATION_EXEMPTIONS = 364,
    /* enumerations 365-370 are defined in Addendum 2010-aa */
    PROP_ALLOW_GROUP_DELAY_INHIBIT = 365,
    PROP_CHANNEL_NUMBER = 366,
    PROP_CONTROL_GROUPS = 367,
    PROP_EXECUTION_DELAY = 368,
    PROP_LAST_PRIORITY = 369,
    PROP_WRITE_STATUS = 370,
    /* enumeration 371 is defined in Addendum 2010-ao */
    PROP_PROPERTY_LIST = 371,
    /* enumeration 372 is defined in Addendum 2010-ak */
    PROP_SERIAL_NUMBER = 372,
    /* enumerations 373-386 are defined in Addendum 2010-i */
    PROP_BLINK_WARN_ENABLE = 373,
    PROP_DEFAULT_FADE_TIME = 374,
    PROP_DEFAULT_RAMP_RATE = 375,
    PROP_DEFAULT_STEP_INCREMENT = 376,
    PROP_EGRESS_TIME = 377,
    PROP_IN_PROGRESS = 378,
    PROP_INSTANTANEOUS_POWER = 379,
    PROP_LIGHTING_COMMAND = 380,
    PROP_LIGHTING_COMMAND_DEFAULT_PRIORITY = 381,
    PROP_MAX_ACTUAL_VALUE = 382,
    PROP_MIN_ACTUAL_VALUE = 383,
    PROP_POWER = 384,
    PROP_TRANSITION = 385,
    PROP_EGRESS_ACTIVE = 386,
    /* enumerations 387-398 */
    PROP_INTERFACE_VALUE = 387,
    PROP_FAULT_HIGH_LIMIT = 388,
    PROP_FAULT_LOW_LIMIT = 389,
    PROP_LOW_DIFF_LIMIT = 390,
    PROP_STRIKE_COUNT = 391,
    PROP_TIME_OF_STRIKE_COUNT_RESET = 392,
    PROP_DEFAULT_TIMEOUT = 393,
    PROP_INITIAL_TIMEOUT = 394,
    PROP_LAST_STATE_CHANGE = 395,
    PROP_STATE_CHANGE_VALUES = 396,
    PROP_TIMER_RUNNING = 397,
    PROP_TIMER_STATE = 398,
    /* enumerations 399-427 are defined in Addendum 2012-ai */
    PROP_APDU_LENGTH = 399,
    PROP_IP_ADDRESS = 400,
    PROP_IP_DEFAULT_GATEWAY = 401,
    PROP_IP_DHCP_ENABLE = 402,
    PROP_IP_DHCP_LEASE_TIME = 403,
    PROP_IP_DHCP_LEASE_TIME_REMAINING = 404,
    PROP_IP_DHCP_SERVER = 405,
    PROP_IP_DNS_SERVER = 406,
    PROP_BACNET_IP_GLOBAL_ADDRESS = 407,
    PROP_BACNET_IP_MODE = 408,
    PROP_BACNET_IP_MULTICAST_ADDRESS = 409,
    PROP_BACNET_IP_NAT_TRAVERSAL = 410,
    PROP_IP_SUBNET_MASK = 411,
    PROP_BACNET_IP_UDP_PORT = 412,
    PROP_BBMD_ACCEPT_FD_REGISTRATIONS = 413,
    PROP_BBMD_BROADCAST_DISTRIBUTION_TABLE = 414,
    PROP_BBMD_FOREIGN_DEVICE_TABLE = 415,
    PROP_CHANGES_PENDING = 416,
    PROP_COMMAND = 417,
    PROP_FD_BBMD_ADDRESS = 418,
    PROP_FD_SUBSCRIPTION_LIFETIME = 419,
    PROP_LINK_SPEED = 420,
    PROP_LINK_SPEEDS = 421,
    PROP_LINK_SPEED_AUTONEGOTIATE = 422,
    PROP_MAC_ADDRESS = 423,
    PROP_NETWORK_INTERFACE_NAME = 424,
    PROP_NETWORK_NUMBER = 425,
    PROP_NETWORK_NUMBER_QUALITY = 426,
    PROP_NETWORK_TYPE = 427,
    PROP_ROUTING_TABLE = 428,
    PROP_VIRTUAL_MAC_ADDRESS_TABLE = 429,
    /* enumerations 430-491 are defined in Addendum-135-2012as */
    PROP_COMMAND_TIME_ARRAY = 430,
    PROP_CURRENT_COMMAND_PRIORITY = 431,
    PROP_LAST_COMMAND_TIME = 432,
    PROP_VALUE_SOURCE = 433,
    PROP_VALUE_SOURCE_ARRAY = 434,
    PROP_BACNET_IPV6_MODE = 435,
    PROP_IPV6_ADDRESS = 436,
    PROP_IPV6_PREFIX_LENGTH = 437,
    PROP_BACNET_IPV6_UDP_PORT = 438,
    PROP_IPV6_DEFAULT_GATEWAY = 439,
    PROP_BACNET_IPV6_MULTICAST_ADDRESS = 440,
    PROP_IPV6_DNS_SERVER = 441,
    PROP_IPV6_AUTO_ADDRESSING_ENABLE = 442,
    PROP_IPV6_DHCP_LEASE_TIME = 443,
    PROP_IPV6_DHCP_LEASE_TIME_REMAINING = 444,
    PROP_IPV6_DHCP_SERVER = 445,
    PROP_IPV6_ZONE_INDEX = 446,
    PROP_ASSIGNED_LANDING_CALLS = 447,
    PROP_CAR_ASSIGNED_DIRECTION = 448,
    PROP_CAR_DOOR_COMMAND = 449,
    PROP_CAR_DOOR_STATUS = 450,
    PROP_CAR_DOOR_TEXT = 451,
    PROP_CAR_DOOR_ZONE = 452,
    PROP_CAR_DRIVE_STATUS = 453,
    PROP_CAR_LOAD = 454,
    PROP_CAR_LOAD_UNITS = 455,
    PROP_CAR_MODE = 456,
    PROP_CAR_MOVING_DIRECTION = 457,
    PROP_CAR_POSITION = 458,
    PROP_ELEVATOR_GROUP = 459,
    PROP_ENERGY_METER = 460,
    PROP_ENERGY_METER_REF = 461,
    PROP_ESCALATOR_MODE = 462,
    PROP_FAULT_SIGNALS = 463,
    PROP_FLOOR_TEXT = 464,
    PROP_GROUP_ID = 465,
    /* value 466 is unassigned */
    PROP_GROUP_MODE = 467,
    PROP_HIGHER_DECK = 468,
    PROP_INSTALLATION_ID = 469,
    PROP_LANDING_CALLS = 470,
    PROP_LANDING_CALL_CONTROL = 471,
    PROP_LANDING_DOOR_STATUS = 472,
    PROP_LOWER_DECK = 473,
    PROP_MACHINE_ROOM_ID = 474,
    PROP_MAKING_CAR_CALL = 475,
    PROP_NEXT_STOPPING_FLOOR = 476,
    PROP_OPERATION_DIRECTION = 477,
    PROP_PASSENGER_ALARM = 478,
    PROP_POWER_MODE = 479,
    PROP_REGISTERED_CAR_CALL = 480,
    PROP_ACTIVE_COV_MULTIPLE_SUBSCRIPTIONS = 481,
    PROP_PROTOCOL_LEVEL = 482,
    PROP_REFERENCE_PORT = 483,
    PROP_DEPLOYED_PROFILE_LOCATION = 484,
    PROP_PROFILE_LOCATION = 485,
    PROP_TAGS = 486,
    PROP_SUBORDINATE_NODE_TYPES = 487,
    PROP_SUBORDINATE_TAGS = 488,
    PROP_SUBORDINATE_RELATIONSHIPS = 489,
    PROP_DEFAULT_SUBORDINATE_RELATIONSHIP = 490,
    PROP_REPRESENTS = 491,
    PROP_DEFAULT_PRESENT_VALUE = 492,
    PROP_PRESENT_STAGE = 493,
    PROP_STAGES = 494,
    PROP_STAGE_NAMES = 495,
    PROP_TARGET_REFERENCES = 496,
    PROP_AUDIT_SOURCE_LEVEL = 497,
    PROP_AUDIT_LEVEL = 498,
    PROP_AUDIT_NOTIFICATION_RECIPIENT = 499,
    PROP_AUDIT_PRIORITY_FILTER = 500,
    PROP_AUDITABLE_OPERATIONS = 501,
    PROP_DELETE_ON_FORWARD = 502,
    PROP_MAXIMUM_SEND_DELAY = 503,
    PROP_MONITORED_OBJECTS = 504,
    PROP_SEND_NOW = 505,
    PROP_FLOOR_NUMBER = 506,
    PROP_DEVICE_UUID = 507,
    /* enumerations 508-511 are defined in Addendum 2020cc */
    PROP_ADDITIONAL_REFERENCE_PORTS = 508,
    PROP_CERTIFICATE_SIGNING_REQUEST_FILE = 509,
    PROP_COMMAND_VALIDATION_RESULT = 510,
    PROP_ISSUER_CERTIFICATE_FILES = 511,
    /* The special property identifiers all, optional, and required  */
    /* are reserved for use in the ReadPropertyConditional and */
    /* ReadPropertyMultiple services or services not defined in this standard.
     */
    /* Enumerated values 0-511 are reserved for definition by ASHRAE.  */
    /* Enumerated values 512-4194303 may be used by others subject to the  */
    /* procedures and constraints described in Clause 23.  */
    PROP_RESERVED_RANGE_MAX = 511,
    PROP_PROPRIETARY_RANGE_MIN = 512,
    PROP_PROPRIETARY_RANGE_MAX = 4194303,
    PROP_RESERVED_RANGE_MIN2 = 4194304,
    /* enumerations 4194304-4194327 are defined in Addendum 2020cc */
    PROP_MAX_BVLC_LENGTH_ACCEPTED = 4194304,
    PROP_MAX_NPDU_LENGTH_ACCEPTED = 4194305,
    PROP_OPERATIONAL_CERTIFICATE_FILE = 4194306,
    PROP_CURRENT_HEALTH = 4194307,
    PROP_SC_CONNECT_WAIT_TIMEOUT = 4194308,
    PROP_SC_DIRECT_CONNECT_ACCEPT_ENABLE = 4194309,
    PROP_SC_DIRECT_CONNECT_ACCEPT_URIS = 4194310,
    PROP_SC_DIRECT_CONNECT_BINDING = 4194311,
    PROP_SC_DIRECT_CONNECT_CONNECTION_STATUS = 4194312,
    PROP_SC_DIRECT_CONNECT_INITIATE_ENABLE = 4194313,
    PROP_SC_DISCONNECT_WAIT_TIMEOUT = 4194314,
    PROP_SC_FAILED_CONNECTION_REQUESTS = 4194315,
    PROP_SC_FAILOVER_HUB_CONNECTION_STATUS = 4194316,
    PROP_SC_FAILOVER_HUB_URI = 4194317,
    PROP_SC_HUB_CONNECTOR_STATE = 4194318,
    PROP_SC_HUB_FUNCTION_ACCEPT_URIS = 4194319,
    PROP_SC_HUB_FUNCTION_BINDING = 4194320,
    PROP_SC_HUB_FUNCTION_CONNECTION_STATUS = 4194321,
    PROP_SC_HUB_FUNCTION_ENABLE = 4194322,
    PROP_SC_HEARTBEAT_TIMEOUT = 4194323,
    PROP_SC_PRIMARY_HUB_CONNECTION_STATUS = 4194324,
    PROP_SC_PRIMARY_HUB_URI = 4194325,
    PROP_SC_MAXIMUM_RECONNECT_TIME = 4194326,
    PROP_SC_MINIMUM_RECONNECT_TIME = 4194327,
    /* enumerations 4194328-4194332 are defined in Addendum 2020ca */
    PROP_COLOR_OVERRIDE = 4194328,
    PROP_COLOR_REFERENCE = 4194329,
    PROP_DEFAULT_COLOR = 4194330,
    PROP_DEFAULT_COLOR_TEMPERATURE = 4194331,
    PROP_OVERRIDE_COLOR_REFERENCE = 4194332,
    PROP_COLOR_COMMAND = 4194334,
    PROP_HIGH_END_TRIM = 4194335,
    PROP_LOW_END_TRIM = 4194336,
    PROP_TRIM_FADE_TIME = 4194337,
    /* Enumerated values 4194303-16777215 are reserved
       for definition by ASHRAE.  */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    MAX_BACNET_PROPERTY_ID = 16777215
} BACNET_PROPERTY_ID;

typedef enum {
    EVENT_LOW_LIMIT_ENABLE = 1,
    EVENT_HIGH_LIMIT_ENABLE = 2
} BACNET_LIMIT_ENABLE;

typedef enum { ACTION_DIRECT = 0, ACTION_REVERSE = 1 } BACNET_ACTION;

typedef enum {
    MIN_BINARY_PV = 0, /* for validating incoming values */
    BINARY_INACTIVE = 0,
    BINARY_ACTIVE = 1,
    MAX_BINARY_PV = 1, /* for validating incoming values */
    BINARY_NULL = 255 /* our homemade way of storing this info */
} BACNET_BINARY_PV;

typedef enum {
    ACTION_BINARY_PV,
    ACTION_UNSIGNED,
    ACTION_FLOAT
} BACNET_ACTION_VALUE_TYPE;

typedef enum {
    EVENT_STATE_NORMAL = 0,
    EVENT_STATE_FAULT = 1,
    EVENT_STATE_OFFNORMAL = 2,
    EVENT_STATE_HIGH_LIMIT = 3,
    EVENT_STATE_LOW_LIMIT = 4,
    EVENT_STATE_MAX = 5
} BACNET_EVENT_STATE;

typedef enum {
    EVENT_ENABLE_TO_OFFNORMAL = 1,
    EVENT_ENABLE_TO_FAULT = 2,
    EVENT_ENABLE_TO_NORMAL = 4
} BACNET_EVENT_ENABLE;

typedef enum {
    STATUS_OPERATIONAL = 0,
    STATUS_OPERATIONAL_READ_ONLY = 1,
    STATUS_DOWNLOAD_REQUIRED = 2,
    STATUS_DOWNLOAD_IN_PROGRESS = 3,
    STATUS_NON_OPERATIONAL = 4,
    STATUS_BACKUP_IN_PROGRESS = 5,
    MAX_DEVICE_STATUS = 6
} BACNET_DEVICE_STATUS;

typedef enum {
    /* Enumerated values 0-255 are reserved for definition by ASHRAE. */
    /* Acceleration */
    UNITS_METERS_PER_SECOND_PER_SECOND = 166,
    /* Area */
    UNITS_SQUARE_METERS = 0,
    UNITS_SQUARE_CENTIMETERS = 116,
    UNITS_SQUARE_FEET = 1,
    UNITS_SQUARE_INCHES = 115,
    /* Currency */
    UNITS_CURRENCY1 = 105,
    UNITS_CURRENCY2 = 106,
    UNITS_CURRENCY3 = 107,
    UNITS_CURRENCY4 = 108,
    UNITS_CURRENCY5 = 109,
    UNITS_CURRENCY6 = 110,
    UNITS_CURRENCY7 = 111,
    UNITS_CURRENCY8 = 112,
    UNITS_CURRENCY9 = 113,
    UNITS_CURRENCY10 = 114,
    /* Efficiency */
    UNITS_BTU_PER_HOUR_PER_WATT = 47898,
    UNITS_BTU_PER_WATT_HOUR_SEASONAL = 47899,
    UNITS_COEFFICIENT_OF_PERFORMANCE = 47900,
    UNITS_COEFFICIENT_OF_PERFORMANCE_SEASONAL = 47901,
    UNITS_KILOWATT_PER_TON_REFRIGERATION = 47902,
    UNITS_LUMENS_PER_WATT = 47903,
    /* Electrical */
    UNITS_MILLIAMPERES = 2,
    UNITS_AMPERES = 3,
    UNITS_AMPERES_PER_METER = 167,
    UNITS_AMPERES_PER_SQUARE_METER = 168,
    UNITS_AMPERE_SQUARE_METERS = 169,
    UNITS_DECIBELS = 199,
    UNITS_DECIBELS_MILLIVOLT = 200,
    UNITS_DECIBELS_VOLT = 201,
    UNITS_FARADS = 170,
    UNITS_HENRYS = 171,
    UNITS_OHMS = 4,
    UNITS_OHM_METERS = 172,
    UNITS_MILLIOHMS = 145,
    UNITS_KILOHMS = 122,
    UNITS_MEGOHMS = 123,
    UNITS_MICROSIEMENS = 190,
    UNITS_MILLISIEMENS = 202,
    UNITS_SIEMENS = 173, /* 1 mho equals 1 siemens */
    UNITS_SIEMENS_PER_METER = 174,
    UNITS_MICROSIEMENS_PER_CENTIMETER = 47909,
    UNITS_MILLISIEMENS_PER_CENTIMETER = 47910,
    UNITS_MILLISIEMENS_PER_METER = 47911,
    UNITS_TESLAS = 175,
    UNITS_VOLTS = 5,
    UNITS_MILLIVOLTS = 124,
    UNITS_KILOVOLTS = 6,
    UNITS_MEGAVOLTS = 7,
    UNITS_VOLT_AMPERES = 8,
    UNITS_KILOVOLT_AMPERES = 9,
    UNITS_MEGAVOLT_AMPERES = 10,
    UNITS_VOLT_AMPERES_REACTIVE = 11,
    UNITS_KILOVOLT_AMPERES_REACTIVE = 12,
    UNITS_MEGAVOLT_AMPERES_REACTIVE = 13,
    UNITS_VOLTS_PER_DEGREE_KELVIN = 176,
    UNITS_VOLTS_PER_METER = 177,
    UNITS_DEGREES_PHASE = 14,
    UNITS_POWER_FACTOR = 15,
    UNITS_WEBERS = 178,
    /* Energy */
    UNITS_JOULES = 16,
    UNITS_KILOJOULES = 17,
    UNITS_KILOJOULES_PER_KILOGRAM = 125,
    UNITS_MEGAJOULES = 126,
    UNITS_GIGAJOULES = 47925,
    UNITS_TERAJOULES = 47926,
    UNITS_WATT_HOURS = 18,
    UNITS_KILOWATT_HOURS = 19,
    UNITS_MEGAWATT_HOURS = 146,
    UNITS_GIGAWATT_HOURS = 47927,
    UNITS_WATT_REACTIVE_HOURS = 203,
    UNITS_KILOWATT_REACTIVE_HOURS = 204,
    UNITS_MEGAWATT_REACTIVE_HOURS = 205,
    UNITS_GIGAWATT_REACTIVE_HOURS = 47928,
    UNITS_BTUS = 20,
    UNITS_KILO_BTUS = 147,
    UNITS_MEGA_BTUS = 148,
    UNITS_THERMS = 21,
    UNITS_TON_HOURS = 22,
    UNITS_ACTIVE_ENERGY_PULSE_VALUE = 47918,
    UNITS_REACTIVE_ENERGY_PULSE_VALUE = 47919,
    UNITS_APPARENT_ENERGY_PULSE_VALUE = 47920,
    UNITS_VOLT_SQUARED_HOUR_PULSE_VALUE = 47921,
    UNITS_AMPERE_SQUARED_HOUR_PULSE_VALUE = 47922,
    /* Enthalpy */
    UNITS_JOULES_PER_KILOGRAM_DRY_AIR = 23,
    UNITS_KILOJOULES_PER_KILOGRAM_DRY_AIR = 149,
    UNITS_MEGAJOULES_PER_KILOGRAM_DRY_AIR = 150,
    UNITS_BTUS_PER_POUND_DRY_AIR = 24,
    UNITS_BTUS_PER_POUND = 117,
    /* Entropy */
    UNITS_JOULES_PER_DEGREE_KELVIN = 127,
    UNITS_KILOJOULES_PER_DEGREE_KELVIN = 151,
    UNITS_MEGAJOULES_PER_DEGREE_KELVIN = 152,
    UNITS_JOULES_PER_KILOGRAM_DEGREE_KELVIN = 128,
    /* Force */
    UNITS_NEWTON = 153,
    /* Frequency */
    UNITS_CYCLES_PER_HOUR = 25,
    UNITS_CYCLES_PER_MINUTE = 26,
    UNITS_HERTZ = 27,
    UNITS_KILOHERTZ = 129,
    UNITS_MEGAHERTZ = 130,
    UNITS_PER_DAY = 47823,
    UNITS_PER_HOUR = 131,
    UNITS_PER_MILLISECOND = 47824,
    /* Humidity */
    UNITS_GRAMS_OF_WATER_PER_KILOGRAM_DRY_AIR = 28,
    UNITS_PERCENT_RELATIVE_HUMIDITY = 29,
    /* Length */
    UNITS_MICROMETERS = 194,
    UNITS_MILLIMETERS = 30,
    UNITS_CENTIMETERS = 118,
    UNITS_KILOMETERS = 193,
    UNITS_METERS = 31,
    UNITS_INCHES = 32,
    UNITS_FEET = 33,
    UNITS_YARDS = 47825,
    UNITS_MILES = 47826,
    UNITS_NAUTICAL_MILES = 47827,
    /* Light */
    UNITS_CANDELAS = 179,
    UNITS_CANDELAS_PER_SQUARE_METER = 180,
    UNITS_WATTS_PER_SQUARE_FOOT = 34,
    UNITS_WATTS_PER_SQUARE_METER = 35,
    UNITS_LUMENS = 36,
    UNITS_LUXES = 37,
    UNITS_FOOT_CANDLES = 38,
    /* Mass */
    UNITS_NANOGRAMS = 47828,
    UNITS_MICROGRAMS = 47829,
    UNITS_MILLIGRAMS = 196,
    UNITS_GRAMS = 195,
    UNITS_KILOGRAMS = 39,
    UNITS_POUNDS_MASS = 40,
    UNITS_TONS = 41,
    UNITS_METRIC_TONNES = 47830,
    UNITS_SHORT_TONS = 47831,
    UNITS_LONG_TONS = 47832,
    /* Mass Flow */
    UNITS_GRAMS_PER_SECOND = 154,
    UNITS_GRAMS_PER_MINUTE = 155,
    UNITS_GRAMS_PER_HOUR = 47833,
    UNITS_GRAMS_PER_DAY = 47834,
    UNITS_KILOGRAMS_PER_SECOND = 42,
    UNITS_KILOGRAMS_PER_MINUTE = 43,
    UNITS_KILOGRAMS_PER_HOUR = 44,
    UNITS_KILOGRAMS_PER_DAY = 47835,
    UNITS_POUNDS_MASS_PER_SECOND = 119,
    UNITS_POUNDS_MASS_PER_MINUTE = 45,
    UNITS_POUNDS_MASS_PER_HOUR = 46,
    UNITS_TONS_PER_HOUR = 156,
    UNITS_SHORT_TONS_PER_SECOND = 47836,
    UNITS_SHORT_TONS_PER_MINUTE = 47837,
    UNITS_SHORT_TONS_PER_HOUR = 47838,
    UNITS_SHORT_TONS_PER_DAY = 47839,
    UNITS_METRIC_TONNES_PER_SECOND = 47840,
    UNITS_METRIC_TONNES_PER_MINUTE = 47841,
    UNITS_METRIC_TONNES_PER_HOUR = 47842,
    UNITS_METRIC_TONNES_PER_DAY = 47843,
    UNITS_LONG_TONS_PER_SECOND = 47844,
    UNITS_LONG_TONS_PER_MINUTE = 47845,
    UNITS_LONG_TONS_PER_HOUR = 47846,
    UNITS_LONG_TONS_PER_DAY = 47847,
    /* Power */
    UNITS_MILLIWATTS = 132,
    UNITS_WATTS = 47,
    UNITS_KILOWATTS = 48,
    UNITS_MEGAWATTS = 49,
    UNITS_GIGAWATTS = 47924,
    UNITS_BTUS_PER_SECOND = 47848,
    UNITS_BTUS_PER_MINUTE = 47849,
    UNITS_BTUS_PER_HOUR = 50,
    UNITS_BTUS_PER_DAY = 47850,
    UNITS_KILO_BTUS_PER_SECOND = 47851,
    UNITS_KILO_BTUS_PER_MINUTE = 47852,
    UNITS_KILO_BTUS_PER_HOUR = 157,
    UNITS_KILO_BTUS_PER_DAY = 47853,
    UNITS_MEGA_BTUS_PER_SECOND = 47854,
    UNITS_MEGA_BTUS_PER_MINUTE = 47855,
    UNITS_MEGA_BTUS_PER_HOUR = 47856,
    UNITS_MEGA_BTUS_PER_DAY = 47857,
    UNITS_JOULES_PER_SECOND = 47858,
    UNITS_JOULES_PER_MINUTE = 47859,
    UNITS_JOULES_PER_DAY = 47860,
    UNITS_KILOJOULES_PER_SECOND = 47861,
    UNITS_KILOJOULES_PER_MINUTE = 47862,
    UNITS_KILOJOULES_PER_HOUR = 47863,
    UNITS_KILOJOULES_PER_DAY = 47864,
    UNITS_MEGAJOULES_PER_SECOND = 47865,
    UNITS_MEGAJOULES_PER_MINUTE = 47866,
    UNITS_MEGAJOULES_PER_HOUR = 47867,
    UNITS_MEGAJOULES_PER_DAY = 47868,
    UNITS_HORSEPOWER = 51,
    UNITS_TONS_REFRIGERATION = 52,
    /* Pressure */
    UNITS_PASCALS = 53,
    UNITS_HECTOPASCALS = 133,
    UNITS_KILOPASCALS = 54,
    UNITS_MILLIBARS = 134,
    UNITS_BARS = 55,
    UNITS_POUNDS_FORCE_PER_SQUARE_INCH = 56,
    UNITS_POUNDS_FORCE_PER_SQUARE_INCH_ABSOLUTE = 47907,
    UNITS_POUNDS_FORCE_PER_SQUARE_INCH_GAUGE = 47908,
    UNITS_MILLIMETERS_OF_WATER = 206,
    UNITS_CENTIMETERS_OF_WATER = 57,
    UNITS_INCHES_OF_WATER = 58,
    UNITS_MILLIMETERS_OF_MERCURY = 59,
    UNITS_CENTIMETERS_OF_MERCURY = 60,
    UNITS_INCHES_OF_MERCURY = 61,
    /* Temperature */
    UNITS_DEGREES_CELSIUS = 62,
    UNITS_DEGREES_CELSIUS_PER_DAY = 47869,
    UNITS_KELVIN = 63,
    UNITS_KELVIN_PER_DAY = 47870,
    UNITS_KELVIN_PER_HOUR = 181,
    UNITS_KELVIN_PER_MINUTE = 182,
    UNITS_DEGREES_FAHRENHEIT = 64,
    UNITS_DEGREES_FAHRENHEIT_PER_DAY = 47871,
    UNITS_DEGREE_DAYS_CELSIUS = 65,
    UNITS_DEGREE_DAYS_FAHRENHEIT = 66,
    UNITS_DELTA_DEGREES_CELSIUS = 47872,
    UNITS_DELTA_DEGREES_FAHRENHEIT = 120,
    UNITS_DELTA_KELVIN = 121,
    /* Time */
    UNITS_YEARS = 67,
    UNITS_MONTHS = 68,
    UNITS_WEEKS = 69,
    UNITS_DAYS = 70,
    UNITS_HOURS = 71,
    UNITS_MINUTES = 72,
    UNITS_SECONDS = 73,
    UNITS_HUNDREDTHS_SECONDS = 158,
    UNITS_MILLISECONDS = 159,
    /* Torque */
    UNITS_NEWTON_METERS = 160,
    UNITS_POUND_FORCE_FEET = 47904,
    UNITS_POUND_FORCE_INCHES = 47905,
    UNITS_OUNCE_FORCE_INCHES = 47906,
    /* Velocity */
    UNITS_MILLIMETERS_PER_SECOND = 161,
    UNITS_MILLIMETERS_PER_MINUTE = 162,
    UNITS_METERS_PER_SECOND = 74,
    UNITS_METERS_PER_MINUTE = 163,
    UNITS_METERS_PER_HOUR = 164,
    UNITS_KILOMETERS_PER_HOUR = 75,
    UNITS_FEET_PER_SECOND = 76,
    UNITS_FEET_PER_MINUTE = 77,
    UNITS_MILES_PER_HOUR = 78,
    /* Volume */
    UNITS_CUBIC_FEET = 79,
    UNITS_CUBIC_METERS = 80,
    UNITS_IMPERIAL_GALLONS = 81,
    UNITS_MILLILITERS = 197,
    UNITS_LITERS = 82,
    UNITS_US_GALLONS = 83,
    UNITS_MILLIONS_OF_US_GALLONS = 47912,
    UNITS_MILLIONS_OF_IMPERIAL_GALLONS = 47913,
    UNITS_VOLUME1 = 47937,
    UNITS_VOLUME2 = 47938,
    UNITS_VOLUME3 = 47939,
    UNITS_VOLUME4 = 47940,
    UNITS_VOLUME5 = 47941,
    UNITS_VOLUME6 = 47942,
    UNITS_VOLUME7 = 47943,
    UNITS_VOLUME8 = 47944,
    UNITS_VOLUME9 = 47945,
    UNITS_VOLUME10 = 47946,
    /* Volumetric Flow */
    UNITS_CUBIC_FEET_PER_SECOND = 142,
    UNITS_CUBIC_FEET_PER_MINUTE = 84,
    UNITS_CUBIC_FEET_PER_HOUR = 191,
    UNITS_MILLION_CUBIC_FEET_PER_MINUTE = 47873,
    UNITS_MILLION_CUBIC_FEET_PER_DAY = 47874,
    UNITS_CUBIC_METERS_PER_SECOND = 85,
    UNITS_CUBIC_METERS_PER_MINUTE = 165,
    UNITS_CUBIC_METERS_PER_HOUR = 135,
    UNITS_IMPERIAL_GALLONS_PER_SECOND = 47875,
    UNITS_IMPERIAL_GALLONS_PER_MINUTE = 86,
    UNITS_IMPERIAL_GALLONS_PER_HOUR = 47876,
    UNITS_IMPERIAL_GALLONS_PER_DAY = 47877,
    UNITS_MILLILITERS_PER_SECOND = 198,
    UNITS_MILLILITERS_PER_MINUTE = 47914,
    UNITS_LITERS_PER_SECOND = 87,
    UNITS_LITERS_PER_MINUTE = 88,
    UNITS_LITERS_PER_HOUR = 136,
    UNITS_LITERS_PER_DAY = 47878,
    UNITS_US_GALLONS_PER_SECOND = 47879,
    UNITS_US_GALLONS_PER_MINUTE = 89,
    UNITS_US_GALLONS_PER_HOUR = 192,
    UNITS_US_GALLONS_PER_DAY = 47880,
    UNITS_CUBIC_METER_PULSE_VALUE = 47923,
    UNITS_VOLUMETRIC_FLOW1 = 47947,
    UNITS_VOLUMETRIC_FLOW2 = 47948,
    UNITS_VOLUMETRIC_FLOW3 = 47949,
    UNITS_VOLUMETRIC_FLOW4 = 47950,
    UNITS_VOLUMETRIC_FLOW5 = 47951,
    UNITS_VOLUMETRIC_FLOW6 = 47952,
    UNITS_VOLUMETRIC_FLOW7 = 47953,
    UNITS_VOLUMETRIC_FLOW8 = 47954,
    UNITS_VOLUMETRIC_FLOW9 = 47955,
    UNITS_VOLUMETRIC_FLOW10 = 47956,
    /* Other */
    UNITS_DEGREES_ANGULAR = 90,
    UNITS_DEGREES_CELSIUS_PER_HOUR = 91,
    UNITS_DEGREES_CELSIUS_PER_MINUTE = 92,
    UNITS_DEGREES_FAHRENHEIT_PER_HOUR = 93,
    UNITS_DEGREES_FAHRENHEIT_PER_MINUTE = 94,
    UNITS_JOULE_SECONDS = 183,
    UNITS_KILOGRAMS_PER_CUBIC_METER = 186,
    UNITS_KW_HOURS_PER_SQUARE_METER = 137,
    UNITS_KW_HOURS_PER_SQUARE_FOOT = 138,
    UNITS_MEGAJOULES_PER_SQUARE_METER = 139,
    UNITS_MEGAJOULES_PER_SQUARE_FOOT = 140,
    UNITS_NO_UNITS = 95,
    UNITS_NEWTON_SECONDS = 187,
    UNITS_NEWTONS_PER_METER = 188,
    UNITS_PARTS_PER_MILLION = 96,
    UNITS_PARTS_PER_BILLION = 97,
    UNITS_PERCENT = 98,
    UNITS_PERCENT_OBSCURATION_PER_FOOT = 143,
    UNITS_PERCENT_OBSCURATION_PER_METER = 144,
    UNITS_PERCENT_PER_SECOND = 99,
    UNITS_PERCENT_PER_MINUTE = 47881,
    UNITS_PERCENT_PER_HOUR = 47882,
    UNITS_PERCENT_PER_DAY = 47883,
    UNITS_PER_MINUTE = 100,
    UNITS_PER_SECOND = 101,
    UNITS_PSI_PER_DEGREE_FAHRENHEIT = 102,
    UNITS_RADIANS = 103,
    UNITS_RADIANS_PER_SECOND = 184,
    UNITS_REVOLUTIONS_PER_MINUTE = 104,
    UNITS_SQUARE_METERS_PER_NEWTON = 185,
    UNITS_WATTS_PER_METER_PER_DEGREE_KELVIN = 189,
    UNITS_WATTS_PER_SQUARE_METER_DEGREE_KELVIN = 141,
    UNITS_PER_MILLE = 207,
    UNITS_PER_MILLION = 47884,
    UNITS_PER_BILLION = 47885,
    UNITS_GRAMS_PER_GRAM = 208,
    UNITS_MICROGRAMS_PER_GRAM = 47886,
    UNITS_NANOGRAMS_PER_GRAM = 47887,
    UNITS_KILOGRAMS_PER_KILOGRAM = 209,
    UNITS_GRAMS_PER_KILOGRAM = 210,
    UNITS_MILLIGRAMS_PER_GRAM = 211,
    UNITS_MILLIGRAMS_PER_KILOGRAM = 212,
    UNITS_MICROGRAMS_PER_KILOGRAM = 47888,
    UNITS_NANOGRAMS_PER_KILOGRAM = 47889,
    UNITS_GRAMS_PER_MILLILITER = 213,
    UNITS_MILLIGRAMS_PER_MILLILITER = 47890,
    UNITS_MICROGRAMS_PER_MILLILITER = 47891,
    UNITS_NANOGRAMS_PER_MILLILITER = 47892,
    UNITS_KILOGRAMS_PER_LITER = 47893,
    UNITS_GRAMS_PER_LITER = 214,
    UNITS_MILLIGRAMS_PER_LITER = 215,
    UNITS_MICROGRAMS_PER_LITER = 216,
    UNITS_NANOGRAMS_PER_LITER = 47894,
    UNITS_GRAMS_PER_CUBIC_METER = 217,
    UNITS_MILLIGRAMS_PER_CUBIC_METER = 218,
    UNITS_MICROGRAMS_PER_CUBIC_METER = 219,
    UNITS_NANOGRAMS_PER_CUBIC_METER = 220,
    UNITS_GRAMS_PER_CUBIC_CENTIMETER = 221,
    UNITS_MILLIGRAMS_PER_CUBIC_CENTIMETER = 47895,
    UNITS_MICROGRAMS_PER_CUBIC_CENTIMETER = 47896,
    UNITS_NANOGRAMS_PER_CUBIC_CENTIMETER = 47897,
    UNITS_BECQUERELS = 222,
    UNITS_KILOBECQUERELS = 223,
    UNITS_MEGABECQUERELS = 224,
    UNITS_GRAY = 225,
    UNITS_MILLIGRAY = 226,
    UNITS_MICROGRAY = 227,
    UNITS_SIEVERTS = 228,
    UNITS_MILLISIEVERTS = 229,
    UNITS_MICROSIEVERTS = 230,
    UNITS_MICROSIEVERTS_PER_HOUR = 231,
    UNITS_DECIBELS_A = 232,
    UNITS_NEPHELOMETRIC_TURBIDITY_UNIT = 233,
    UNITS_PH = 234,
    UNITS_GRAMS_PER_SQUARE_METER = 235,
    UNITS_MINUTES_PER_DEGREE_KELVIN = 236,
    UNITS_DEGREES_LOVIBOND = 47816,
    UNITS_ALCOHOL_BY_VOLUME = 47817,
    UNITS_INTERNATIONAL_BITTERING_UNITS = 47818,
    UNITS_EUROPEAN_BITTERNESS_UNITS = 47819,
    UNITS_DEGREES_PLATO = 47820,
    UNITS_SPECIFIC_GRAVITY = 47821,
    UNITS_EUROPEAN_BREWING_CONVENTION = 47822,
    UNITS_MILS_PER_YEAR = 47915,
    UNITS_MILLIMETERS_PER_YEAR = 47916,
    UNITS_PULSES_PER_MINUTE = 47917,
    UNITS_BITS_PER_SECOND = 47929,
    UNITS_KILOBITS_PER_SECOND = 47930,
    UNITS_MEGABITS_PER_SECOND = 47931,
    UNITS_GIGABITS_PER_SECOND = 47932,
    UNITS_BYTES_PER_SECOND = 47933,
    UNITS_KILOBYTES_PER_SECOND = 47934,
    UNITS_MEGABYTES_PER_SECOND = 47935,
    UNITS_GIGABYTES_PER_SECOND = 47936,
    UNITS_SITE_UNIT1 = 47958,
    UNITS_SITE_UNIT2 = 47959,
    UNITS_SITE_UNIT3 = 47960,
    UNITS_SITE_UNIT4 = 47961,
    UNITS_SITE_UNIT5 = 47962,
    UNITS_SITE_UNIT6 = 47963,
    UNITS_SITE_UNIT7 = 47964,
    UNITS_SITE_UNIT8 = 47965,
    UNITS_SITE_UNIT9 = 47966,
    UNITS_SITE_UNIT10 = 47967,

    UNITS_OHM_METER_SQUARED_PER_METER = 237,
    UNITS_AMPERE_SECONDS = 238,
    UNITS_VOLT_AMPERE_HOURS = 239,
    UNITS_KILOVOLT_AMPERE_HOURS = 240,
    UNITS_MEGAVOLT_AMPERE_HOURS = 241,
    UNITS_VOLT_AMPERE_REACTIVE_HOURS = 242,
    UNITS_KILOVOLT_AMPERE_REACTIVE_HOURS = 243,
    UNITS_MEGAVOLT_AMPERE_REACTIVE_HOURS = 244,
    UNITS_VOLT_SQUARE_HOURS = 245,
    UNITS_AMPERE_SQUARE_HOURS = 246,
    UNITS_JOULE_PER_HOURS = 247,
    UNITS_CUBIC_FEET_PER_DAY = 248,
    UNITS_CUBIC_METERS_PER_DAY = 249,
    UNITS_WATT_HOURS_PER_CUBIC_METER = 250,
    UNITS_JOULES_PER_CUBIC_METER = 251,
    UNITS_MOLE_PERCENT = 252,
    UNITS_PASCAL_SECONDS = 253,
    UNITS_MILLION_STANDARD_CUBIC_FEET_PER_MINUTE = 254,
    UNITS_RESERVED_RANGE_MAX = 255,
    /* Enumerated values 256-47807 may be used by others
       subject to the procedures and constraints described in Clause 23. */
    UNITS_PROPRIETARY_RANGE_MIN = 256,
    UNITS_PROPRIETARY_RANGE_MAX = 47807,
    /* Enumerated values 47808-49999 are reserved for definition by ASHRAE. */
    UNITS_RESERVED_RANGE_MIN2 = 47808,
    UNITS_STANDARD_CUBIC_FEET_PER_DAY = 47808,
    UNITS_MILLION_STANDARD_CUBIC_FEET_PER_DAY = 47809,
    UNITS_THOUSAND_CUBIC_FEET_PER_DAY = 47810,
    UNITS_THOUSAND_STANDARD_CUBIC_FEET_PER_DAY = 47811,
    UNITS_POUNDS_MASS_PER_DAY = 47812,
    /* 47813 - NOT USED */
    UNITS_MILLIREMS = 47814,
    UNITS_MILLIREMS_PER_HOUR = 47815,
    UNITS_RESERVED_RANGE_MAX2 = 49999,
    UNITS_PROPRIETARY_RANGE_MIN2 = 50000,
    /* Enumerated values 50000-65535 may be used by others
       subject to the procedures and constraints described in Clause 23. */
    /* do the proprietary range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    UNITS_PROPRIETARY_RANGE_MAX2 = 65535
} BACNET_ENGINEERING_UNITS;

typedef enum {
    POLARITY_NORMAL = 0,
    POLARITY_REVERSE = 1,
    MAX_POLARITY = 2
} BACNET_POLARITY;

typedef enum {
    PROGRAM_REQUEST_READY = 0,
    PROGRAM_REQUEST_LOAD = 1,
    PROGRAM_REQUEST_RUN = 2,
    PROGRAM_REQUEST_HALT = 3,
    PROGRAM_REQUEST_RESTART = 4,
    PROGRAM_REQUEST_UNLOAD = 5
} BACNET_PROGRAM_REQUEST;

typedef enum {
    PROGRAM_STATE_IDLE = 0,
    PROGRAM_STATE_LOADING = 1,
    PROGRAM_STATE_RUNNING = 2,
    PROGRAM_STATE_WAITING = 3,
    PROGRAM_STATE_HALTED = 4,
    PROGRAM_STATE_UNLOADING = 5
} BACNET_PROGRAM_STATE;

typedef enum {
    PROGRAM_ERROR_NORMAL = 0,
    PROGRAM_ERROR_LOAD_FAILED = 1,
    PROGRAM_ERROR_INTERNAL = 2,
    PROGRAM_ERROR_PROGRAM = 3,
    PROGRAM_ERROR_OTHER = 4,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.  */
    /* Enumerated values 64-65535 may be used by others subject to  */
    /* the procedures and constraints described in Clause 23. */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    PROGRAM_ERROR_PROPRIETARY_MIN = 64,
    PROGRAM_ERROR_PROPRIETARY_MAX = 65535
} BACNET_PROGRAM_ERROR;

typedef enum {
    RESTART_REASON_UNKNOWN = 0,
    RESTART_REASON_COLDSTART = 1,
    RESTART_REASON_WARMSTART = 2,
    RESTART_REASON_DETECTED_POWER_LOST = 3,
    RESTART_REASON_DETECTED_POWER_OFF = 4,
    RESTART_REASON_HARDWARE_WATCHDOG = 5,
    RESTART_REASON_SOFTWARE_WATCHDOG = 6,
    RESTART_REASON_SUSPENDED = 7,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.
       Enumerated values 64-255 may be used by others subject to the
       procedures and constraints described in Clause 23. */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    RESTART_REASON_PROPRIETARY_MIN = 64,
    RESTART_REASON_PROPRIETARY_MAX = 255
} BACNET_RESTART_REASON;

typedef enum {
    PROP_STATE_BOOLEAN_VALUE = 0,
    PROP_STATE_BINARY_VALUE = 1,
    PROP_STATE_EVENT_TYPE = 2,
    PROP_STATE_POLARITY = 3,
    PROP_STATE_PROGRAM_CHANGE = 4,
    PROP_STATE_PROGRAM_STATE = 5,
    PROP_STATE_REASON_FOR_HALT = 6,
    PROP_STATE_RELIABILITY = 7,
    PROP_STATE_EVENT_STATE = 8,
    PROP_STATE_SYSTEM_STATUS = 9,
    PROP_STATE_UNITS = 10,
    PROP_STATE_UNSIGNED_VALUE = 11,
    PROP_STATE_LIFE_SAFETY_MODE = 12,
    PROP_STATE_LIFE_SAFETY_STATE = 13,
    PROP_STATE_RESTART_REASON = 14,
    PROP_STATE_DOOR_ALARM_STATE = 15,
    PROP_STATE_ACTION = 16,
    PROP_STATE_DOOR_SECURED_STATUS = 17,
    PROP_STATE_DOOR_STATUS = 18,
    PROP_STATE_DOOR_VALUE = 19,
    PROP_STATE_FILE_ACCESS_METHOD = 20,
    PROP_STATE_LOCK_STATUS = 21,
    PROP_STATE_LIFE_SAFETY_OPERATION = 22,
    PROP_STATE_MAINTENANCE = 23,
    PROP_STATE_NODE_TYPE = 24,
    PROP_STATE_NOTIFY_TYPE = 25,
    PROP_STATE_SECURITY_LEVEL = 26,
    PROP_STATE_SHED_STATE = 27,
    PROP_STATE_SILENCED_STATE = 28,
    PROP_STATE_ACCESS_EVENT = 30,
    PROP_STATE_ZONE_OCCUPANCY_STATE = 31,
    PROP_STATE_ACCESS_CRED_DISABLE_REASON = 32,
    PROP_STATE_ACCESS_CRED_DISABLE = 33,
    PROP_STATE_AUTHENTICATION_STATUS = 34,
    PROP_STATE_BACKUP_STATE = 36,
    PROP_STATE_WRITE_STATUS = 37,
    PROP_STATE_LIGHTING_IN_PROGRESS = 38,
    PROP_STATE_LIGHTING_OPERATION = 39,
    PROP_STATE_LIGHTING_TRANSITION = 40,
    PROP_STATE_INTEGER_VALUE = 41,
    PROP_STATE_BINARY_LIGHTING_VALUE = 42,
    PROP_STATE_TIMER_STATE = 43,
    PROP_STATE_TIMER_TRANSITION = 44,
    PROP_STATE_BACNET_IP_MODE = 45,
    PROP_STATE_NETWORK_PORT_COMMAND = 46,
    PROP_STATE_NETWORK_TYPE = 47,
    PROP_STATE_NETWORK_NUMBER_QUALITY = 48,
    PROP_STATE_ESCALATOR_OPERATION_DIRECTION = 49,
    PROP_STATE_ESCALATOR_FAULT = 50,
    PROP_STATE_ESCALATOR_MODE = 51,
    PROP_STATE_LIFT_CAR_DIRECTION = 52,
    PROP_STATE_LIFT_CAR_DOOR_COMMAND = 53,
    PROP_STATE_LIFT_CAR_DRIVE_STATUS = 54,
    PROP_STATE_LIFT_CAR_MODE = 55,
    PROP_STATE_LIFT_GROUP_MODE = 56,
    PROP_STATE_LIFT_FAULT = 57,
    PROP_STATE_PROTOCOL_LEVEL = 58,
    PROP_STATE_AUDIT_LEVEL = 59,
    PROP_STATE_AUDIT_OPERATION = 60,
    PROP_STATE_EXTENDED_VALUE = 63,
    /* Tag values 0-63 are reserved for definition by ASHRAE.*/
    PROP_STATE_PROPRIETARY_MIN = 64,
    PROP_STATE_PROPRIETARY_MAX = 254,
    /* Tag values of 64-254 may be used by others to
       accommodate vendor specific properties that have discrete
       or enumerated values, subject to the constraints
       described in Clause 23.*/
    /* Tag values greater than 254 are not encoded as ASN context tags.
       In these cases, the tag value is multiplied
       by 100000 and is added to the enumeration value
       and the sum is encoded using context tag 63,
       the extended-value choice.*/
    PROP_STATE_EXAMPLE_ONE = 256,
    PROP_STATE_EXAMPLE_TWO = 257
} BACNET_PROPERTY_STATES;

typedef enum BACnetReliability {
    RELIABILITY_NO_FAULT_DETECTED = 0,
    RELIABILITY_NO_SENSOR = 1,
    RELIABILITY_OVER_RANGE = 2,
    RELIABILITY_UNDER_RANGE = 3,
    RELIABILITY_OPEN_LOOP = 4,
    RELIABILITY_SHORTED_LOOP = 5,
    RELIABILITY_NO_OUTPUT = 6,
    RELIABILITY_UNRELIABLE_OTHER = 7,
    RELIABILITY_PROCESS_ERROR = 8,
    RELIABILITY_MULTI_STATE_FAULT = 9,
    RELIABILITY_CONFIGURATION_ERROR = 10,
    RELIABILITY_RESERVED_11 = 11,
    RELIABILITY_COMMUNICATION_FAILURE = 12,
    RELIABILITY_MEMBER_FAULT = 13,
    RELIABILITY_MONITORED_OBJECT_FAULT = 14,
    RELIABILITY_TRIPPED = 15,
    RELIABILITY_LAMP_FAILURE = 16,
    RELIABILITY_ACTIVATION_FAILURE = 17,
    RELIABILITY_RENEW_DHCP_FAILURE = 18,
    RELIABILITY_RENEW_FD_REGISTRATION_FAILURE = 19,
    RELIABILITY_RESTART_AUTO_NEGOTIATION_FAILURE = 20,
    RELIABILITY_RESTART_FAILURE = 21,
    RELIABILITY_PROPRIETARY_COMMAND_FAILURE = 22,
    RELIABILITY_FAULTS_LISTED = 23,
    RELIABILITY_REFERENCED_OBJECT_FAULT = 24,
    RELIABILITY_MULTI_STATE_OUT_OF_RANGE = 25,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.  */
    /* Enumerated values 64-65535 may be used by others subject to  */
    /* the procedures and constraints described in Clause 23. */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    RELIABILITY_PROPRIETARY_MIN = 64,
    RELIABILITY_PROPRIETARY_MAX = 65535
} BACNET_RELIABILITY;

typedef enum {
    EVENT_CHANGE_OF_BITSTRING = 0,
    EVENT_CHANGE_OF_STATE = 1,
    EVENT_CHANGE_OF_VALUE = 2,
    EVENT_COMMAND_FAILURE = 3,
    EVENT_FLOATING_LIMIT = 4,
    EVENT_OUT_OF_RANGE = 5,
    /*  complex-event-type        (6), -- see comment below */
    /*  event-buffer-ready   (7), -- context tag 7 is deprecated */
    EVENT_CHANGE_OF_LIFE_SAFETY = 8,
    EVENT_EXTENDED = 9,
    EVENT_BUFFER_READY = 10,
    EVENT_UNSIGNED_RANGE = 11,
    EVENT_ACCESS_EVENT = 13,
    EVENT_DOUBLE_OUT_OF_RANGE = 14,
    EVENT_SIGNED_OUT_OF_RANGE = 15,
    EVENT_UNSIGNED_OUT_OF_RANGE = 16,
    EVENT_CHANGE_OF_CHARACTERSTRING = 17,
    EVENT_CHANGE_OF_STATUS_FLAGS = 18,
    EVENT_CHANGE_OF_RELIABILITY = 19,
    EVENT_NONE = 20,
    EVENT_CHANGE_OF_DISCRETE_VALUE = 21,
    EVENT_CHANGE_OF_TIMER = 22,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.  */
    /* Enumerated values 64-65535 may be used by others subject to  */
    /* the procedures and constraints described in Clause 23.  */
    /* It is expected that these enumerated values will correspond to  */
    /* the use of the complex-event-type CHOICE [6] of the  */
    /* BACnetNotificationParameters production. */
    /* The last enumeration used in this version is 22. */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    EVENT_PROPRIETARY_MIN = 64,
    EVENT_PROPRIETARY_MAX = 65535
} BACNET_EVENT_TYPE;

typedef enum {
    FILE_RECORD_ACCESS = 0,
    FILE_STREAM_ACCESS = 1,
    FILE_RECORD_AND_STREAM_ACCESS = 2
} BACNET_FILE_ACCESS_METHOD;

typedef enum {
    MIN_LIFE_SAFETY_MODE = 0,
    LIFE_SAFETY_MODE_OFF = 0,
    LIFE_SAFETY_MODE_ON = 1,
    LIFE_SAFETY_MODE_TEST = 2,
    LIFE_SAFETY_MODE_MANNED = 3,
    LIFE_SAFETY_MODE_UNMANNED = 4,
    LIFE_SAFETY_MODE_ARMED = 5,
    LIFE_SAFETY_MODE_DISARMED = 6,
    LIFE_SAFETY_MODE_PREARMED = 7,
    LIFE_SAFETY_MODE_SLOW = 8,
    LIFE_SAFETY_MODE_FAST = 9,
    LIFE_SAFETY_MODE_DISCONNECTED = 10,
    LIFE_SAFETY_MODE_ENABLED = 11,
    LIFE_SAFETY_MODE_DISABLED = 12,
    LIFE_SAFETY_MODE_AUTOMATIC_RELEASE_DISABLED = 13,
    LIFE_SAFETY_MODE_DEFAULT = 14,
    MAX_LIFE_SAFETY_MODE = 15,
    /* Enumerated values 0-255 are reserved for definition by ASHRAE.  */
    /* Enumerated values 256-65535 may be used by others subject to  */
    /* procedures and constraints described in Clause 23. */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    LIFE_SAFETY_MODE_PROPRIETARY_MIN = 256,
    LIFE_SAFETY_MODE_PROPRIETARY_MAX = 65535
} BACNET_LIFE_SAFETY_MODE;

typedef enum {
    LIFE_SAFETY_OP_NONE = 0,
    LIFE_SAFETY_OP_SILENCE = 1,
    LIFE_SAFETY_OP_SILENCE_AUDIBLE = 2,
    LIFE_SAFETY_OP_SILENCE_VISUAL = 3,
    LIFE_SAFETY_OP_RESET = 4,
    LIFE_SAFETY_OP_RESET_ALARM = 5,
    LIFE_SAFETY_OP_RESET_FAULT = 6,
    LIFE_SAFETY_OP_UNSILENCE = 7,
    LIFE_SAFETY_OP_UNSILENCE_AUDIBLE = 8,
    LIFE_SAFETY_OP_UNSILENCE_VISUAL = 9,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.  */
    /* Enumerated values 64-65535 may be used by others subject to  */
    /* procedures and constraints described in Clause 23. */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    LIFE_SAFETY_OP_PROPRIETARY_MIN = 64,
    LIFE_SAFETY_OP_PROPRIETARY_MAX = 65535
} BACNET_LIFE_SAFETY_OPERATION;

typedef enum {
    MIN_LIFE_SAFETY_STATE = 0,
    LIFE_SAFETY_STATE_QUIET = 0,
    LIFE_SAFETY_STATE_PRE_ALARM = 1,
    LIFE_SAFETY_STATE_ALARM = 2,
    LIFE_SAFETY_STATE_FAULT = 3,
    LIFE_SAFETY_STATE_FAULT_PRE_ALARM = 4,
    LIFE_SAFETY_STATE_FAULT_ALARM = 5,
    LIFE_SAFETY_STATE_NOT_READY = 6,
    LIFE_SAFETY_STATE_ACTIVE = 7,
    LIFE_SAFETY_STATE_TAMPER = 8,
    LIFE_SAFETY_STATE_TEST_ALARM = 9,
    LIFE_SAFETY_STATE_TEST_ACTIVE = 10,
    LIFE_SAFETY_STATE_TEST_FAULT = 11,
    LIFE_SAFETY_STATE_TEST_FAULT_ALARM = 12,
    LIFE_SAFETY_STATE_HOLDUP = 13,
    LIFE_SAFETY_STATE_DURESS = 14,
    LIFE_SAFETY_STATE_TAMPER_ALARM = 15,
    LIFE_SAFETY_STATE_ABNORMAL = 16,
    LIFE_SAFETY_STATE_EMERGENCY_POWER = 17,
    LIFE_SAFETY_STATE_DELAYED = 18,
    LIFE_SAFETY_STATE_BLOCKED = 19,
    LIFE_SAFETY_STATE_LOCAL_ALARM = 20,
    LIFE_SAFETY_STATE_GENERAL_ALARM = 21,
    LIFE_SAFETY_STATE_SUPERVISORY = 22,
    LIFE_SAFETY_STATE_TEST_SUPERVISORY = 23,
    MAX_LIFE_SAFETY_STATE = 24,
    /* Enumerated values 0-255 are reserved for definition by ASHRAE.  */
    /* Enumerated values 256-65535 may be used by others subject to  */
    /* procedures and constraints described in Clause 23. */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    LIFE_SAFETY_STATE_PROPRIETARY_MIN = 256,
    LIFE_SAFETY_STATE_PROPRIETARY_MAX = 65535
} BACNET_LIFE_SAFETY_STATE;

typedef enum {
    SILENCED_STATE_UNSILENCED = 0,
    SILENCED_STATE_AUDIBLE_SILENCED = 1,
    SILENCED_STATE_VISIBLE_SILENCED = 2,
    SILENCED_STATE_ALL_SILENCED = 3,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE. */
    /* Enumerated values 64-65535 may be used by others subject to */
    /* procedures and constraints described in Clause 23. */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    SILENCED_STATE_PROPRIETARY_MIN = 64,
    SILENCED_STATE_PROPRIETARY_MAX = 65535
} BACNET_SILENCED_STATE;

typedef enum {
    MAINTENANCE_NONE = 0,
    MAINTENANCE_PERIODIC_TEST = 1,
    MAINTENANCE_NEED_SERVICE_OPERATIONAL = 2,
    MAINTENANCE_NEED_SERVICE_INOPERATIVE = 3,
    /* Enumerated values 0-255 are reserved for definition by ASHRAE.  */
    /* Enumerated values 256-65535 may be used by others subject to  */
    /* procedures and constraints described in Clause 23. */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    MAINTENANCE_PROPRIETARY_MIN = 256,
    MAINTENANCE_PROPRIETARY_MAX = 65535
} BACNET_MAINTENANCE;

typedef enum {
    NOTIFY_ALARM = 0,
    NOTIFY_EVENT = 1,
    NOTIFY_ACK_NOTIFICATION = 2,
    NOTIFY_MAX = 3
} BACNET_NOTIFY_TYPE;

typedef enum BACnetObjectType {
    OBJECT_ANALOG_INPUT = 0,
    OBJECT_ANALOG_OUTPUT = 1,
    OBJECT_ANALOG_VALUE = 2,
    OBJECT_BINARY_INPUT = 3,
    OBJECT_BINARY_OUTPUT = 4,
    OBJECT_BINARY_VALUE = 5,
    OBJECT_CALENDAR = 6,
    OBJECT_COMMAND = 7,
    OBJECT_DEVICE = 8,
    OBJECT_EVENT_ENROLLMENT = 9,
    OBJECT_FILE = 10,
    OBJECT_GROUP = 11,
    OBJECT_LOOP = 12,
    OBJECT_MULTI_STATE_INPUT = 13,
    OBJECT_MULTI_STATE_OUTPUT = 14,
    OBJECT_NOTIFICATION_CLASS = 15,
    OBJECT_PROGRAM = 16,
    OBJECT_SCHEDULE = 17,
    OBJECT_AVERAGING = 18,
    OBJECT_MULTI_STATE_VALUE = 19,
    OBJECT_TRENDLOG = 20,
    OBJECT_LIFE_SAFETY_POINT = 21,
    OBJECT_LIFE_SAFETY_ZONE = 22,
    OBJECT_ACCUMULATOR = 23,
    OBJECT_PULSE_CONVERTER = 24,
    OBJECT_EVENT_LOG = 25,
    OBJECT_GLOBAL_GROUP = 26,
    OBJECT_TREND_LOG_MULTIPLE = 27,
    OBJECT_LOAD_CONTROL = 28,
    OBJECT_STRUCTURED_VIEW = 29,
    OBJECT_ACCESS_DOOR = 30,
    OBJECT_TIMER = 31,
    /* Note: 31 was lighting output, but BACnet editor changed it... */
    OBJECT_ACCESS_CREDENTIAL = 32, /* Addendum 2008-j */
    OBJECT_ACCESS_POINT = 33,
    OBJECT_ACCESS_RIGHTS = 34,
    OBJECT_ACCESS_USER = 35,
    OBJECT_ACCESS_ZONE = 36,
    OBJECT_CREDENTIAL_DATA_INPUT = 37, /* authentication-factor-input */
    OBJECT_NETWORK_SECURITY = 38, /* Addendum 2008-g */
    OBJECT_BITSTRING_VALUE = 39, /* Addendum 2008-w */
    OBJECT_CHARACTERSTRING_VALUE = 40, /* Addendum 2008-w */
    OBJECT_DATE_PATTERN_VALUE = 41, /* Addendum 2008-w */
    OBJECT_DATE_VALUE = 42, /* Addendum 2008-w */
    OBJECT_DATETIME_PATTERN_VALUE = 43, /* Addendum 2008-w */
    OBJECT_DATETIME_VALUE = 44, /* Addendum 2008-w */
    OBJECT_INTEGER_VALUE = 45, /* Addendum 2008-w */
    OBJECT_LARGE_ANALOG_VALUE = 46, /* Addendum 2008-w */
    OBJECT_OCTETSTRING_VALUE = 47, /* Addendum 2008-w */
    OBJECT_POSITIVE_INTEGER_VALUE = 48, /* Addendum 2008-w */
    OBJECT_TIME_PATTERN_VALUE = 49, /* Addendum 2008-w */
    OBJECT_TIME_VALUE = 50, /* Addendum 2008-w */
    OBJECT_NOTIFICATION_FORWARDER = 51, /* Addendum 2010-af */
    OBJECT_ALERT_ENROLLMENT = 52, /* Addendum 2010-af */
    OBJECT_CHANNEL = 53, /* Addendum 2010-aa */
    OBJECT_LIGHTING_OUTPUT = 54, /* Addendum 2010-i */
    OBJECT_BINARY_LIGHTING_OUTPUT = 55, /* Addendum 135-2012az */
    OBJECT_NETWORK_PORT = 56, /* Addendum 135-2012az */
    OBJECT_ELEVATOR_GROUP = 57, /* Addendum 135-2012aq */
    OBJECT_ESCALATOR = 58, /* Addendum 135-2012aq */
    OBJECT_LIFT = 59, /* Addendum 135-2012aq */
    OBJECT_STAGING = 60, /* Addendum 135-2016bd */
    OBJECT_AUDIT_LOG = 61, /* Addendum 135-2016bi */
    OBJECT_AUDIT_REPORTER = 62, /* Addendum 135-2016bi */
    OBJECT_COLOR = 63, /* Addendum 135-2020ca */
    OBJECT_COLOR_TEMPERATURE = 64, /* Addendum 135-2020ca */
    BACNET_OBJECT_TYPE_LAST = OBJECT_COLOR_TEMPERATURE,
    BACNET_OBJECT_TYPE_RESERVED_MAX = 127,
    /* Enumerated values 0-127 are reserved for definition by ASHRAE. */
    /* Enumerated values 128-1023 may be used by others subject to  */
    /* the procedures and constraints described in Clause 23. */
    OBJECT_PROPRIETARY_MIN = 128,
    OBJECT_PROPRIETARY_MAX = 1023,
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    MAX_BACNET_OBJECT_TYPE = 1024,
    /* special usage for this library 16-bit max */
    OBJECT_NONE = 65535
} BACNET_OBJECT_TYPE;

typedef enum {
    SEGMENTATION_BOTH = 0,
    SEGMENTATION_TRANSMIT = 1,
    SEGMENTATION_RECEIVE = 2,
    SEGMENTATION_NONE = 3,
    MAX_BACNET_SEGMENTATION = 4
} BACNET_SEGMENTATION;

typedef enum {
    VT_CLASS_DEFAULT = 0,
    VT_CLASS_ANSI_X34 = 1, /* real name is ANSI X3.64 */
    VT_CLASS_DEC_VT52 = 2,
    VT_CLASS_DEC_VT100 = 3,
    VT_CLASS_DEC_VT220 = 4,
    VT_CLASS_HP_700_94 = 5, /* real name is HP 700/94 */
    VT_CLASS_IBM_3130 = 6,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.  */
    /* Enumerated values 64-65535 may be used by others subject to  */
    /* the procedures and constraints described in Clause 23. */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    VT_CLASS_PROPRIETARY_MIN = 64,
    VT_CLASS_PROPRIETARY_MAX = 65535
} BACNET_VT_CLASS;

typedef enum {
    CHARACTER_ANSI_X34 = 0, /* deprecated */
    CHARACTER_UTF8 = 0,
    CHARACTER_MS_DBCS = 1,
    CHARACTER_JISC_6226 = 2,
    CHARACTER_UCS4 = 3,
    CHARACTER_UCS2 = 4,
    CHARACTER_ISO8859 = 5,
    MAX_CHARACTER_STRING_ENCODING = 6
} BACNET_CHARACTER_STRING_ENCODING;

typedef enum {
    BACNET_APPLICATION_TAG_NULL = 0,
    BACNET_APPLICATION_TAG_BOOLEAN = 1,
    BACNET_APPLICATION_TAG_UNSIGNED_INT = 2,
    BACNET_APPLICATION_TAG_SIGNED_INT = 3,
    BACNET_APPLICATION_TAG_REAL = 4,
    BACNET_APPLICATION_TAG_DOUBLE = 5,
    BACNET_APPLICATION_TAG_OCTET_STRING = 6,
    BACNET_APPLICATION_TAG_CHARACTER_STRING = 7,
    BACNET_APPLICATION_TAG_BIT_STRING = 8,
    BACNET_APPLICATION_TAG_ENUMERATED = 9,
    BACNET_APPLICATION_TAG_DATE = 10,
    BACNET_APPLICATION_TAG_TIME = 11,
    BACNET_APPLICATION_TAG_OBJECT_ID = 12,
    BACNET_APPLICATION_TAG_RESERVE1 = 13,
    BACNET_APPLICATION_TAG_RESERVE2 = 14,
    BACNET_APPLICATION_TAG_RESERVE3 = 15,
    MAX_BACNET_APPLICATION_TAG = 16,

    /* Complex tagged constructed data types - not specifically enumerated */

    /* Means : "nothing", an empty list, not even a null character */
    BACNET_APPLICATION_TAG_EMPTYLIST,
    /* BACnetWeeknday */
    BACNET_APPLICATION_TAG_WEEKNDAY,
    /* BACnetDateRange */
    BACNET_APPLICATION_TAG_DATERANGE,
    /* BACnetDateTime */
    BACNET_APPLICATION_TAG_DATETIME,
    /* BACnetTimeStamp */
    BACNET_APPLICATION_TAG_TIMESTAMP,
    /* Error Class, Error Code */
    BACNET_APPLICATION_TAG_ERROR,
    /* BACnetDeviceObjectPropertyReference */
    BACNET_APPLICATION_TAG_DEVICE_OBJECT_PROPERTY_REFERENCE,
    /* BACnetDeviceObjectReference */
    BACNET_APPLICATION_TAG_DEVICE_OBJECT_REFERENCE,
    /* BACnetObjectPropertyReference */
    BACNET_APPLICATION_TAG_OBJECT_PROPERTY_REFERENCE,
    /* BACnetDestination (Recipient_List) */
    BACNET_APPLICATION_TAG_DESTINATION,
    /* BACnetRecipient */
    BACNET_APPLICATION_TAG_RECIPIENT,
    /* BACnetCOVSubscription */
    BACNET_APPLICATION_TAG_COV_SUBSCRIPTION,
    /* BACnetCalendarEntry */
    BACNET_APPLICATION_TAG_CALENDAR_ENTRY,
    /* BACnetWeeklySchedule */
    BACNET_APPLICATION_TAG_WEEKLY_SCHEDULE,
    /* BACnetSpecialEvent */
    BACNET_APPLICATION_TAG_SPECIAL_EVENT,
    /* BACnetReadAccessSpecification */
    BACNET_APPLICATION_TAG_READ_ACCESS_SPECIFICATION,
    /* BACnetLightingCommand */
    BACNET_APPLICATION_TAG_LIGHTING_COMMAND,
    /* BACnetHostNPort */
    BACNET_APPLICATION_TAG_HOST_N_PORT,
    /* BACnetxyColor */
    BACNET_APPLICATION_TAG_XY_COLOR,
    /* BACnetColorCommand */
    BACNET_APPLICATION_TAG_COLOR_COMMAND,
    /* BACNET_SC_FAILED_CONNECTION_REQUEST */
    BACNET_APPLICATION_TAG_SC_FAILED_CONNECTION_REQUEST,
    /* BACNET_SC_HUB_FUNCTION_CONNECTION_STATUS */
    BACNET_APPLICATION_TAG_SC_HUB_FUNCTION_CONNECTION_STATUS,
    /* BACNET_SC_DIRECT_CONNECTION_STATUS */
    BACNET_APPLICATION_TAG_SC_DIRECT_CONNECTION_STATUS,
    /* BACNET_SC_HUB_CONNECTION_STATUS */
    BACNET_APPLICATION_TAG_SC_HUB_CONNECTION_STATUS,
    /* BACnetBDTEntry */
    BACNET_APPLICATION_TAG_BDT_ENTRY,
    /* BACnetFDTEntry */
    BACNET_APPLICATION_TAG_FDT_ENTRY,
    /* BACnetActionCommand */
    BACNET_APPLICATION_TAG_ACTION_COMMAND,
    /* BACnetScale */
    BACNET_APPLICATION_TAG_SCALE,
    /* BACnetShedLevel */
    BACNET_APPLICATION_TAG_SHED_LEVEL,
    /* BACnetAccessRule */
    BACNET_APPLICATION_TAG_ACCESS_RULE,
    /* BACnetChannelValue */
    BACNET_APPLICATION_TAG_CHANNEL_VALUE
} BACNET_APPLICATION_TAG;

/* note: these are not the real values, */
/* but are shifted left for easy encoding */
typedef enum {
    PDU_TYPE_CONFIRMED_SERVICE_REQUEST = 0,
    PDU_TYPE_UNCONFIRMED_SERVICE_REQUEST = 0x10,
    PDU_TYPE_SIMPLE_ACK = 0x20,
    PDU_TYPE_COMPLEX_ACK = 0x30,
    PDU_TYPE_SEGMENT_ACK = 0x40,
    PDU_TYPE_ERROR = 0x50,
    PDU_TYPE_REJECT = 0x60,
    PDU_TYPE_ABORT = 0x70
} BACNET_PDU_TYPE;

/* BACnetConfirmedServiceChoice ::= ENUMERATED */
typedef enum BACnet_Confirmed_Service_Choice {
    /* Alarm and Event Services */
    SERVICE_CONFIRMED_ACKNOWLEDGE_ALARM = 0,
    SERVICE_CONFIRMED_AUDIT_NOTIFICATION = 32,
    SERVICE_CONFIRMED_COV_NOTIFICATION = 1,
    SERVICE_CONFIRMED_COV_NOTIFICATION_MULTIPLE = 31,
    SERVICE_CONFIRMED_EVENT_NOTIFICATION = 2,
    SERVICE_CONFIRMED_GET_ALARM_SUMMARY = 3,
    SERVICE_CONFIRMED_GET_ENROLLMENT_SUMMARY = 4,
    SERVICE_CONFIRMED_GET_EVENT_INFORMATION = 29,
    SERVICE_CONFIRMED_LIFE_SAFETY_OPERATION = 27,
    SERVICE_CONFIRMED_SUBSCRIBE_COV = 5,
    SERVICE_CONFIRMED_SUBSCRIBE_COV_PROPERTY = 28,
    SERVICE_CONFIRMED_SUBSCRIBE_COV_PROPERTY_MULTIPLE = 30,
    /* File Access Services */
    SERVICE_CONFIRMED_ATOMIC_READ_FILE = 6,
    SERVICE_CONFIRMED_ATOMIC_WRITE_FILE = 7,
    /* Object Access Services */
    SERVICE_CONFIRMED_ADD_LIST_ELEMENT = 8,
    SERVICE_CONFIRMED_REMOVE_LIST_ELEMENT = 9,
    SERVICE_CONFIRMED_CREATE_OBJECT = 10,
    SERVICE_CONFIRMED_DELETE_OBJECT = 11,
    SERVICE_CONFIRMED_READ_PROPERTY = 12,
    SERVICE_CONFIRMED_READ_PROP_CONDITIONAL = 13,
    SERVICE_CONFIRMED_READ_PROP_MULTIPLE = 14,
    SERVICE_CONFIRMED_READ_RANGE = 26,
    SERVICE_CONFIRMED_WRITE_PROPERTY = 15,
    SERVICE_CONFIRMED_WRITE_PROP_MULTIPLE = 16,
    SERVICE_CONFIRMED_AUDIT_LOG_QUERY = 33,
    /* Remote Device Management Services */
    SERVICE_CONFIRMED_DEVICE_COMMUNICATION_CONTROL = 17,
    SERVICE_CONFIRMED_PRIVATE_TRANSFER = 18,
    SERVICE_CONFIRMED_TEXT_MESSAGE = 19,
    SERVICE_CONFIRMED_REINITIALIZE_DEVICE = 20,
    /* Virtual Terminal Services */
    SERVICE_CONFIRMED_VT_OPEN = 21,
    SERVICE_CONFIRMED_VT_CLOSE = 22,
    SERVICE_CONFIRMED_VT_DATA = 23,
    /* Security Services */
    SERVICE_CONFIRMED_AUTHENTICATE = 24,
    SERVICE_CONFIRMED_REQUEST_KEY = 25,
    /* Services added after 1995 */
    /* readRange [26] see Object Access Services */
    /* lifeSafetyOperation [27] see Alarm and Event Services */
    /* subscribeCOVProperty [28] see Alarm and Event Services */
    /* getEventInformation [29] see Alarm and Event Services */
    /* Services added after 2012 */
    /* subscribe-cov-property-multiple [30] see Alarm and Event Services */
    /* confirmed-cov-notification-multiple [31] see Alarm and Event Services */
    /* Services added after 2016 */
    /* confirmed-audit-notification [32] see Alarm and Event Services */
    /* audit-log-query [33] see Object Access Services */
    MAX_BACNET_CONFIRMED_SERVICE = 34
} BACNET_CONFIRMED_SERVICE;

/* BACnetUnconfirmedServiceChoice ::= ENUMERATED */
typedef enum BACnet_Unconfirmed_Service_Choice {
    SERVICE_UNCONFIRMED_I_AM = 0,
    SERVICE_UNCONFIRMED_I_HAVE = 1,
    SERVICE_UNCONFIRMED_COV_NOTIFICATION = 2,
    SERVICE_UNCONFIRMED_EVENT_NOTIFICATION = 3,
    SERVICE_UNCONFIRMED_PRIVATE_TRANSFER = 4,
    SERVICE_UNCONFIRMED_TEXT_MESSAGE = 5,
    SERVICE_UNCONFIRMED_TIME_SYNCHRONIZATION = 6,
    SERVICE_UNCONFIRMED_WHO_HAS = 7,
    SERVICE_UNCONFIRMED_WHO_IS = 8,
    SERVICE_UNCONFIRMED_UTC_TIME_SYNCHRONIZATION = 9,
    /* addendum 2010-aa */
    SERVICE_UNCONFIRMED_WRITE_GROUP = 10,
    /* addendum 2012-aq */
    SERVICE_UNCONFIRMED_COV_NOTIFICATION_MULTIPLE = 11,
    /* addendum 2016-bi */
    SERVICE_UNCONFIRMED_AUDIT_NOTIFICATION = 12,
    /* addendum 2016-bz */
    SERVICE_UNCONFIRMED_WHO_AM_I = 13,
    SERVICE_UNCONFIRMED_YOU_ARE = 14,
    /* Other services to be added as they are defined. */
    /* All choice values in this production are reserved */
    /* for definition by ASHRAE. */
    /* Proprietary extensions are made by using the */
    /* UnconfirmedPrivateTransfer service. See Clause 23. */
    MAX_BACNET_UNCONFIRMED_SERVICE = 15
} BACNET_UNCONFIRMED_SERVICE;

/* BACnetServicesSupported ::= BIT STRING */
typedef enum BACnet_Services_Supported {
    /* Alarm and Event Services */
    SERVICE_SUPPORTED_ACKNOWLEDGE_ALARM = 0,
    SERVICE_SUPPORTED_CONFIRMED_COV_NOTIFICATION = 1,
    SERVICE_SUPPORTED_CONFIRMED_COV_NOTIFICATION_MULTIPLE = 42,
    SERVICE_SUPPORTED_CONFIRMED_EVENT_NOTIFICATION = 2,
    SERVICE_SUPPORTED_GET_ALARM_SUMMARY = 3,
    SERVICE_SUPPORTED_GET_ENROLLMENT_SUMMARY = 4,
    SERVICE_SUPPORTED_GET_EVENT_INFORMATION = 39,
    SERVICE_SUPPORTED_SUBSCRIBE_COV = 5,
    SERVICE_SUPPORTED_SUBSCRIBE_COV_PROPERTY = 38,
    SERVICE_SUPPORTED_SUBSCRIBE_COV_PROPERTY_MULTIPLE = 41,
    SERVICE_SUPPORTED_LIFE_SAFETY_OPERATION = 37,
    SERVICE_SUPPORTED_CONFIRMED_AUDIT_NOTIFICATION = 44,
    SERVICE_SUPPORTED_AUDIT_LOG_QUERY = 45,
    /* File Access Services */
    SERVICE_SUPPORTED_ATOMIC_READ_FILE = 6,
    SERVICE_SUPPORTED_ATOMIC_WRITE_FILE = 7,
    /* Object Access Services */
    SERVICE_SUPPORTED_ADD_LIST_ELEMENT = 8,
    SERVICE_SUPPORTED_REMOVE_LIST_ELEMENT = 9,
    SERVICE_SUPPORTED_CREATE_OBJECT = 10,
    SERVICE_SUPPORTED_DELETE_OBJECT = 11,
    SERVICE_SUPPORTED_READ_PROPERTY = 12,
    SERVICE_SUPPORTED_READ_PROP_CONDITIONAL = 13,
    SERVICE_SUPPORTED_READ_PROP_MULTIPLE = 14,
    SERVICE_SUPPORTED_READ_RANGE = 35,
    SERVICE_SUPPORTED_WRITE_PROPERTY = 15,
    SERVICE_SUPPORTED_WRITE_PROP_MULTIPLE = 16,
    SERVICE_SUPPORTED_WRITE_GROUP = 40,
    /* Remote Device Management Services */
    SERVICE_SUPPORTED_DEVICE_COMMUNICATION_CONTROL = 17,
    SERVICE_SUPPORTED_PRIVATE_TRANSFER = 18,
    SERVICE_SUPPORTED_TEXT_MESSAGE = 19,
    SERVICE_SUPPORTED_REINITIALIZE_DEVICE = 20,
    SERVICE_SUPPORTED_WHO_AM_I = 47, /* Addendum 135-2016bz */
    SERVICE_SUPPORTED_YOU_ARE = 48, /* Addendum 135-2016bz */
    /* Virtual Terminal Services */
    SERVICE_SUPPORTED_VT_OPEN = 21,
    SERVICE_SUPPORTED_VT_CLOSE = 22,
    SERVICE_SUPPORTED_VT_DATA = 23,
    /* Security Services */
    SERVICE_SUPPORTED_AUTHENTICATE = 24,
    SERVICE_SUPPORTED_REQUEST_KEY = 25,
    /* Unconfirmed Services */
    SERVICE_SUPPORTED_I_AM = 26,
    SERVICE_SUPPORTED_I_HAVE = 27,
    SERVICE_SUPPORTED_UNCONFIRMED_COV_NOTIFICATION = 28,
    SERVICE_SUPPORTED_UNCONFIRMED_COV_NOTIFICATION_MULTIPLE = 43,
    SERVICE_SUPPORTED_UNCONFIRMED_EVENT_NOTIFICATION = 29,
    SERVICE_SUPPORTED_UNCONFIRMED_PRIVATE_TRANSFER = 30,
    SERVICE_SUPPORTED_UNCONFIRMED_TEXT_MESSAGE = 31,
    SERVICE_SUPPORTED_TIME_SYNCHRONIZATION = 32,
    SERVICE_SUPPORTED_UTC_TIME_SYNCHRONIZATION = 36,
    SERVICE_SUPPORTED_WHO_HAS = 33,
    SERVICE_SUPPORTED_WHO_IS = 34,
    SERVICE_SUPPORTED_UNCONFIRMED_AUDIT_NOTIFICATION = 46
    /* Other services to be added as they are defined. */
    /* All values in this production are reserved */
    /* for definition by ASHRAE. */
} BACNET_SERVICES_SUPPORTED;

/* Bit String Enumerations */
typedef enum {
    STATUS_FLAG_IN_ALARM = 0,
    STATUS_FLAG_FAULT = 1,
    STATUS_FLAG_OVERRIDDEN = 2,
    STATUS_FLAG_OUT_OF_SERVICE = 3
} BACNET_STATUS_FLAGS;

typedef enum {
    LOG_STATUS_LOG_DISABLED = 0,
    LOG_STATUS_BUFFER_PURGED = 1,
    LOG_STATUS_LOG_INTERRUPTED = 2
} BACNET_LOG_STATUS;

typedef enum {
    LOGGING_TYPE_POLLED = 0,
    LOGGING_TYPE_COV = 1,
    LOGGING_TYPE_TRIGGERED = 2
} BACNET_LOGGING_TYPE;

typedef enum {
    ACKNOWLEDGMENT_FILTER_ALL = 0,
    ACKNOWLEDGMENT_FILTER_ACKED = 1,
    ACKNOWLEDGMENT_FILTER_NOT_ACKED = 2
} BACNET_ACKNOWLEDGMENT_FILTER;

typedef enum {
    EVENT_STATE_FILTER_OFFNORMAL = 0,
    EVENT_STATE_FILTER_FAULT = 1,
    EVENT_STATE_FILTER_NORMAL = 2,
    EVENT_STATE_FILTER_ALL = 3,
    EVENT_STATE_FILTER_ACTIVE = 4
} BACNET_EVENT_STATE_FILTER;

typedef enum {
    SELECTION_LOGIC_AND = 0,
    SELECTION_LOGIC_OR = 1,
    SELECTION_LOGIC_ALL = 2
} BACNET_SELECTION_LOGIC;

typedef enum {
    RELATION_SPECIFIER_EQUAL = 0,
    RELATION_SPECIFIER_NOT_EQUAL = 1,
    RELATION_SPECIFIER_LESS_THAN = 2,
    RELATION_SPECIFIER_GREATER_THAN = 3,
    RELATION_SPECIFIER_LESS_THAN_OR_EQUAL = 4,
    RELATION_SPECIFIER_GREATER_THAN_OR_EQUAL = 5
} BACNET_RELATION_SPECIFIER;

typedef enum {
    COMMUNICATION_ENABLE = 0,
    COMMUNICATION_DISABLE = 1,
    COMMUNICATION_DISABLE_INITIATION = 2,
    MAX_BACNET_COMMUNICATION_ENABLE_DISABLE = 3
} BACNET_COMMUNICATION_ENABLE_DISABLE;

typedef enum {
    MESSAGE_PRIORITY_NORMAL = 0,
    MESSAGE_PRIORITY_URGENT = 1,
    MESSAGE_PRIORITY_CRITICAL_EQUIPMENT = 2,
    MESSAGE_PRIORITY_LIFE_SAFETY = 3
} BACNET_MESSAGE_PRIORITY;

/*Network Layer Message Type */
/*If Bit 7 of the control octet described in 6.2.2 is 1, */
/* a message type octet shall be present as shown in Figure 6-1. */
/* The following message types are indicated: */
typedef enum {
    NETWORK_MESSAGE_WHO_IS_ROUTER_TO_NETWORK = 0,
    NETWORK_MESSAGE_I_AM_ROUTER_TO_NETWORK = 1,
    NETWORK_MESSAGE_I_COULD_BE_ROUTER_TO_NETWORK = 2,
    NETWORK_MESSAGE_REJECT_MESSAGE_TO_NETWORK = 3,
    NETWORK_MESSAGE_ROUTER_BUSY_TO_NETWORK = 4,
    NETWORK_MESSAGE_ROUTER_AVAILABLE_TO_NETWORK = 5,
    NETWORK_MESSAGE_INIT_RT_TABLE = 6,
    NETWORK_MESSAGE_INIT_RT_TABLE_ACK = 7,
    NETWORK_MESSAGE_ESTABLISH_CONNECTION_TO_NETWORK = 8,
    NETWORK_MESSAGE_DISCONNECT_CONNECTION_TO_NETWORK = 9,
    NETWORK_MESSAGE_CHALLENGE_REQUEST = 10,
    NETWORK_MESSAGE_SECURITY_PAYLOAD = 11,
    NETWORK_MESSAGE_SECURITY_RESPONSE = 12,
    NETWORK_MESSAGE_REQUEST_KEY_UPDATE = 13,
    NETWORK_MESSAGE_UPDATE_KEY_SET = 14,
    NETWORK_MESSAGE_UPDATE_DISTRIBUTION_KEY = 15,
    NETWORK_MESSAGE_REQUEST_MASTER_KEY = 16,
    NETWORK_MESSAGE_SET_MASTER_KEY = 17,
    NETWORK_MESSAGE_WHAT_IS_NETWORK_NUMBER = 18,
    NETWORK_MESSAGE_NETWORK_NUMBER_IS = 19,
    /* X'14' to X'7F': Reserved for use by ASHRAE, */
    NETWORK_MESSAGE_ASHRAE_RESERVED_MIN = 20,
    NETWORK_MESSAGE_ASHRAE_RESERVED_MAX = 127,
    /* X'80' to X'FF': Available for vendor proprietary messages */
    NETWORK_MESSAGE_INVALID = 0x100
} BACNET_NETWORK_MESSAGE_TYPE;

typedef enum {
    ABORT_REASON_OTHER = 0,
    ABORT_REASON_BUFFER_OVERFLOW = 1,
    ABORT_REASON_INVALID_APDU_IN_THIS_STATE = 2,
    ABORT_REASON_PREEMPTED_BY_HIGHER_PRIORITY_TASK = 3,
    ABORT_REASON_SEGMENTATION_NOT_SUPPORTED = 4,
    ABORT_REASON_SECURITY_ERROR = 5,
    ABORT_REASON_INSUFFICIENT_SECURITY = 6,
    ABORT_REASON_WINDOW_SIZE_OUT_OF_RANGE = 7,
    ABORT_REASON_APPLICATION_EXCEEDED_REPLY_TIME = 8,
    ABORT_REASON_OUT_OF_RESOURCES = 9,
    ABORT_REASON_TSM_TIMEOUT = 10,
    ABORT_REASON_APDU_TOO_LONG = 11,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE. */
    /* Enumerated values 64-255 may be used by others subject to */
    /* the procedures and constraints described in Clause 23. */
    MAX_BACNET_ABORT_REASON = 12,
    ABORT_REASON_PROPRIETARY_FIRST = 64,
    ABORT_REASON_PROPRIETARY_LAST = 255
} BACNET_ABORT_REASON;

typedef enum {
    REJECT_REASON_OTHER = 0,
    REJECT_REASON_BUFFER_OVERFLOW = 1,
    REJECT_REASON_INCONSISTENT_PARAMETERS = 2,
    REJECT_REASON_INVALID_PARAMETER_DATA_TYPE = 3,
    REJECT_REASON_INVALID_TAG = 4,
    REJECT_REASON_MISSING_REQUIRED_PARAMETER = 5,
    REJECT_REASON_PARAMETER_OUT_OF_RANGE = 6,
    REJECT_REASON_TOO_MANY_ARGUMENTS = 7,
    REJECT_REASON_UNDEFINED_ENUMERATION = 8,
    REJECT_REASON_UNRECOGNIZED_SERVICE = 9,
    REJECT_REASON_INVALID_DATA_ENCODING = 10,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE. */
    /* Enumerated values 64-255 may be used by others subject to */
    /* the procedures and constraints described in Clause 23. */
    MAX_BACNET_REJECT_REASON = 11,
    REJECT_REASON_PROPRIETARY_FIRST = 64,
    REJECT_REASON_PROPRIETARY_LAST = 255
} BACNET_REJECT_REASON;

typedef enum {
    ERROR_CLASS_DEVICE = 0,
    ERROR_CLASS_OBJECT = 1,
    ERROR_CLASS_PROPERTY = 2,
    ERROR_CLASS_RESOURCES = 3,
    ERROR_CLASS_SECURITY = 4,
    ERROR_CLASS_SERVICES = 5,
    ERROR_CLASS_VT = 6,
    ERROR_CLASS_COMMUNICATION = 7,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE. */
    /* Enumerated values 64-65535 may be used by others subject to */
    /* the procedures and constraints described in Clause 23. */
    MAX_BACNET_ERROR_CLASS = 8,
    /* do the MAX here instead of outside of enum so that
       compilers will allocate adequate sized datatype for enum */
    ERROR_CLASS_PROPRIETARY_FIRST = 64,
    ERROR_CLASS_PROPRIETARY_LAST = 65535
} BACNET_ERROR_CLASS;

/* These are sorted in the order given in
   Clause 18. ERROR, REJECT AND ABORT CODES
   The Class and Code pairings are required
   to be used in accordance with Clause 18. */
typedef enum {
    /* valid for all classes */
    ERROR_CODE_OTHER = 0,

    /* Error Class - Device */
    ERROR_CODE_DEVICE_BUSY = 3,
    ERROR_CODE_CONFIGURATION_IN_PROGRESS = 2,
    ERROR_CODE_OPERATIONAL_PROBLEM = 25,

    /* Error Class - Object */
    ERROR_CODE_DYNAMIC_CREATION_NOT_SUPPORTED = 4,
    ERROR_CODE_NO_OBJECTS_OF_SPECIFIED_TYPE = 17,
    ERROR_CODE_OBJECT_DELETION_NOT_PERMITTED = 23,
    ERROR_CODE_OBJECT_IDENTIFIER_ALREADY_EXISTS = 24,
    ERROR_CODE_READ_ACCESS_DENIED = 27,
    ERROR_CODE_UNKNOWN_OBJECT = 31,
    ERROR_CODE_UNSUPPORTED_OBJECT_TYPE = 36,

    /* Error Class - Property */
    ERROR_CODE_CHARACTER_SET_NOT_SUPPORTED = 41,
    ERROR_CODE_DATATYPE_NOT_SUPPORTED = 47,
    ERROR_CODE_INCONSISTENT_SELECTION_CRITERION = 8,
    ERROR_CODE_INVALID_ARRAY_INDEX = 42,
    ERROR_CODE_INVALID_DATA_TYPE = 9,
    ERROR_CODE_NOT_COV_PROPERTY = 44,
    ERROR_CODE_OPTIONAL_FUNCTIONALITY_NOT_SUPPORTED = 45,
    ERROR_CODE_PROPERTY_IS_NOT_AN_ARRAY = 50,
    /* ERROR_CODE_READ_ACCESS_DENIED = 27, */
    ERROR_CODE_UNKNOWN_PROPERTY = 32,
    ERROR_CODE_VALUE_OUT_OF_RANGE = 37,
    ERROR_CODE_WRITE_ACCESS_DENIED = 40,

    /* Error Class - Resources */
    ERROR_CODE_NO_SPACE_FOR_OBJECT = 18,
    ERROR_CODE_NO_SPACE_TO_ADD_LIST_ELEMENT = 19,
    ERROR_CODE_NO_SPACE_TO_WRITE_PROPERTY = 20,

    /* Error Class - Security */
    ERROR_CODE_AUTHENTICATION_FAILED = 1,
    /* ERROR_CODE_CHARACTER_SET_NOT_SUPPORTED = 41, */
    ERROR_CODE_INCOMPATIBLE_SECURITY_LEVELS = 6,
    ERROR_CODE_INVALID_OPERATOR_NAME = 12,
    ERROR_CODE_KEY_GENERATION_ERROR = 15,
    ERROR_CODE_PASSWORD_FAILURE = 26,
    ERROR_CODE_SECURITY_NOT_SUPPORTED = 28,
    ERROR_CODE_TIMEOUT = 30,

    /* Error Class - Services */
    /* ERROR_CODE_CHARACTER_SET_NOT_SUPPORTED = 41, */
    ERROR_CODE_COV_SUBSCRIPTION_FAILED = 43,
    ERROR_CODE_DUPLICATE_NAME = 48,
    ERROR_CODE_DUPLICATE_OBJECT_ID = 49,
    ERROR_CODE_FILE_ACCESS_DENIED = 5,
    ERROR_CODE_INCONSISTENT_PARAMETERS = 7,
    ERROR_CODE_INVALID_CONFIGURATION_DATA = 46,
    ERROR_CODE_INVALID_FILE_ACCESS_METHOD = 10,
    ERROR_CODE_INVALID_FILE_START_POSITION = 11,
    ERROR_CODE_INVALID_PARAMETER_DATA_TYPE = 13,
    ERROR_CODE_INVALID_TIME_STAMP = 14,
    ERROR_CODE_MISSING_REQUIRED_PARAMETER = 16,
    /* ERROR_CODE_OPTIONAL_FUNCTIONALITY_NOT_SUPPORTED = 45, */
    ERROR_CODE_PROPERTY_IS_NOT_A_LIST = 22,
    ERROR_CODE_SERVICE_REQUEST_DENIED = 29,

    /* Error Class - VT */
    ERROR_CODE_UNKNOWN_VT_CLASS = 34,
    ERROR_CODE_UNKNOWN_VT_SESSION = 35,
    ERROR_CODE_NO_VT_SESSIONS_AVAILABLE = 21,
    ERROR_CODE_VT_SESSION_ALREADY_CLOSED = 38,
    ERROR_CODE_VT_SESSION_TERMINATION_FAILURE = 39,

    /* unused */
    ERROR_CODE_RESERVED1 = 33,
    /* new error codes from new addenda */
    ERROR_CODE_ABORT_BUFFER_OVERFLOW = 51,
    ERROR_CODE_ABORT_INVALID_APDU_IN_THIS_STATE = 52,
    ERROR_CODE_ABORT_PREEMPTED_BY_HIGHER_PRIORITY_TASK = 53,
    ERROR_CODE_ABORT_SEGMENTATION_NOT_SUPPORTED = 54,
    ERROR_CODE_ABORT_PROPRIETARY = 55,
    ERROR_CODE_ABORT_OTHER = 56,
    ERROR_CODE_INVALID_TAG = 57,
    ERROR_CODE_NETWORK_DOWN = 58,
    ERROR_CODE_REJECT_BUFFER_OVERFLOW = 59,
    ERROR_CODE_REJECT_INCONSISTENT_PARAMETERS = 60,
    ERROR_CODE_REJECT_INVALID_PARAMETER_DATA_TYPE = 61,
    ERROR_CODE_REJECT_INVALID_TAG = 62,
    ERROR_CODE_REJECT_MISSING_REQUIRED_PARAMETER = 63,
    ERROR_CODE_REJECT_PARAMETER_OUT_OF_RANGE = 64,
    ERROR_CODE_REJECT_TOO_MANY_ARGUMENTS = 65,
    ERROR_CODE_REJECT_UNDEFINED_ENUMERATION = 66,
    ERROR_CODE_REJECT_UNRECOGNIZED_SERVICE = 67,
    ERROR_CODE_REJECT_PROPRIETARY = 68,
    ERROR_CODE_REJECT_OTHER = 69,
    ERROR_CODE_UNKNOWN_DEVICE = 70,
    ERROR_CODE_UNKNOWN_ROUTE = 71,
    ERROR_CODE_VALUE_NOT_INITIALIZED = 72,
    ERROR_CODE_INVALID_EVENT_STATE = 73,
    ERROR_CODE_NO_ALARM_CONFIGURED = 74,
    ERROR_CODE_LOG_BUFFER_FULL = 75,
    ERROR_CODE_LOGGED_VALUE_PURGED = 76,
    ERROR_CODE_NO_PROPERTY_SPECIFIED = 77,
    ERROR_CODE_NOT_CONFIGURED_FOR_TRIGGERED_LOGGING = 78,
    ERROR_CODE_UNKNOWN_SUBSCRIPTION = 79,
    ERROR_CODE_PARAMETER_OUT_OF_RANGE = 80,
    ERROR_CODE_LIST_ELEMENT_NOT_FOUND = 81,
    ERROR_CODE_BUSY = 82,
    ERROR_CODE_COMMUNICATION_DISABLED = 83,
    ERROR_CODE_SUCCESS = 84,
    ERROR_CODE_ACCESS_DENIED = 85,
    ERROR_CODE_BAD_DESTINATION_ADDRESS = 86,
    ERROR_CODE_BAD_DESTINATION_DEVICE_ID = 87,
    ERROR_CODE_BAD_SIGNATURE = 88,
    ERROR_CODE_BAD_SOURCE_ADDRESS = 89,
    ERROR_CODE_BAD_TIMESTAMP = 90,
    ERROR_CODE_CANNOT_USE_KEY = 91,
    ERROR_CODE_CANNOT_VERIFY_MESSAGE_ID = 92,
    ERROR_CODE_CORRECT_KEY_REVISION = 93,
    ERROR_CODE_DESTINATION_DEVICE_ID_REQUIRED = 94,
    ERROR_CODE_DUPLICATE_MESSAGE = 95,
    ERROR_CODE_ENCRYPTION_NOT_CONFIGURED = 96,
    ERROR_CODE_ENCRYPTION_REQUIRED = 97,
    ERROR_CODE_INCORRECT_KEY = 98,
    ERROR_CODE_INVALID_KEY_DATA = 99,
    ERROR_CODE_KEY_UPDATE_IN_PROGRESS = 100,
    ERROR_CODE_MALFORMED_MESSAGE = 101,
    ERROR_CODE_NOT_KEY_SERVER = 102,
    ERROR_CODE_SECURITY_NOT_CONFIGURED = 103,
    ERROR_CODE_SOURCE_SECURITY_REQUIRED = 104,
    ERROR_CODE_TOO_MANY_KEYS = 105,
    ERROR_CODE_UNKNOWN_AUTHENTICATION_TYPE = 106,
    ERROR_CODE_UNKNOWN_KEY = 107,
    ERROR_CODE_UNKNOWN_KEY_REVISION = 108,
    ERROR_CODE_UNKNOWN_SOURCE_MESSAGE = 109,
    ERROR_CODE_NOT_ROUTER_TO_DNET = 110,
    ERROR_CODE_ROUTER_BUSY = 111,
    ERROR_CODE_UNKNOWN_NETWORK_MESSAGE = 112,
    ERROR_CODE_MESSAGE_TOO_LONG = 113,
    ERROR_CODE_SECURITY_ERROR = 114,
    ERROR_CODE_ADDRESSING_ERROR = 115,
    ERROR_CODE_WRITE_BDT_FAILED = 116,
    ERROR_CODE_READ_BDT_FAILED = 117,
    ERROR_CODE_REGISTER_FOREIGN_DEVICE_FAILED = 118,
    ERROR_CODE_READ_FDT_FAILED = 119,
    ERROR_CODE_DELETE_FDT_ENTRY_FAILED = 120,
    ERROR_CODE_DISTRIBUTE_BROADCAST_FAILED = 121,
    ERROR_CODE_UNKNOWN_FILE_SIZE = 122,
    ERROR_CODE_ABORT_APDU_TOO_LONG = 123,
    ERROR_CODE_ABORT_APPLICATION_EXCEEDED_REPLY_TIME = 124,
    ERROR_CODE_ABORT_OUT_OF_RESOURCES = 125,
    ERROR_CODE_ABORT_TSM_TIMEOUT = 126,
    ERROR_CODE_ABORT_WINDOW_SIZE_OUT_OF_RANGE = 127,
    ERROR_CODE_FILE_FULL = 128,
    ERROR_CODE_INCONSISTENT_CONFIGURATION = 129,
    ERROR_CODE_INCONSISTENT_OBJECT_TYPE = 130,
    ERROR_CODE_INTERNAL_ERROR = 131,
    ERROR_CODE_NOT_CONFIGURED = 132,
    ERROR_CODE_OUT_OF_MEMORY = 133,
    ERROR_CODE_VALUE_TOO_LONG = 134,
    ERROR_CODE_ABORT_INSUFFICIENT_SECURITY = 135,
    ERROR_CODE_ABORT_SECURITY_ERROR = 136,
    ERROR_CODE_DUPLICATE_ENTRY = 137,
    ERROR_CODE_INVALID_VALUE_IN_THIS_STATE = 138,
    ERROR_CODE_INVALID_OPERATION_IN_THIS_STATE = 139,
    ERROR_CODE_LIST_ITEM_NOT_NUMBERED = 140,
    ERROR_CODE_LIST_ITEM_NOT_TIMESTAMPED = 141,
    ERROR_CODE_INVALID_DATA_ENCODING = 142,
    ERROR_CODE_BVLC_FUNCTION_UNKNOWN = 143,
    ERROR_CODE_BVLC_PROPRIETARY_FUNCTION_UNKNOWN = 144,
    ERROR_CODE_HEADER_ENCODING_ERROR = 145,
    ERROR_CODE_HEADER_NOT_UNDERSTOOD = 146,
    ERROR_CODE_MESSAGE_INCOMPLETE = 147,
    ERROR_CODE_NOT_A_BACNET_SC_HUB = 148,
    ERROR_CODE_PAYLOAD_EXPECTED = 149,
    ERROR_CODE_UNEXPECTED_DATA = 150,
    ERROR_CODE_NODE_DUPLICATE_VMAC = 151,
    ERROR_CODE_HTTP_UNEXPECTED_RESPONSE_CODE = 152,
    ERROR_CODE_HTTP_NO_UPGRADE = 153,
    ERROR_CODE_HTTP_RESOURCE_NOT_LOCAL = 154,
    ERROR_CODE_HTTP_PROXY_AUTHENTICATION_FAILED = 155,
    ERROR_CODE_HTTP_RESPONSE_TIMEOUT = 156,
    ERROR_CODE_HTTP_RESPONSE_SYNTAX_ERROR = 157,
    ERROR_CODE_HTTP_RESPONSE_VALUE_ERROR = 158,
    ERROR_CODE_HTTP_RESPONSE_MISSING_HEADER = 159,
    ERROR_CODE_HTTP_WEBSOCKET_HEADER_ERROR = 160,
    ERROR_CODE_HTTP_UPGRADE_REQUIRED = 161,
    ERROR_CODE_HTTP_UPGRADE_ERROR = 162,
    ERROR_CODE_HTTP_TEMPORARY_UNAVAILABLE = 163,
    ERROR_CODE_HTTP_NOT_A_SERVER = 164,
    ERROR_CODE_HTTP_ERROR = 165,
    ERROR_CODE_WEBSOCKET_SCHEME_NOT_SUPPORTED = 166,
    ERROR_CODE_WEBSOCKET_UNKNOWN_CONTROL_MESSAGE = 167,
    ERROR_CODE_WEBSOCKET_CLOSE_ERROR = 168,
    ERROR_CODE_WEBSOCKET_CLOSED_BY_PEER = 169,
    ERROR_CODE_WEBSOCKET_ENDPOINT_LEAVES = 170,
    ERROR_CODE_WEBSOCKET_PROTOCOL_ERROR = 171,
    ERROR_CODE_WEBSOCKET_DATA_NOT_ACCEPTED = 172,
    ERROR_CODE_WEBSOCKET_CLOSED_ABNORMALLY = 173,
    ERROR_CODE_WEBSOCKET_DATA_INCONSISTENT = 174,
    ERROR_CODE_WEBSOCKET_DATA_AGAINST_POLICY = 175,
    ERROR_CODE_WEBSOCKET_FRAME_TOO_LONG = 176,
    ERROR_CODE_WEBSOCKET_EXTENSION_MISSING = 177,
    ERROR_CODE_WEBSOCKET_REQUEST_UNAVAILABLE = 178,
    ERROR_CODE_WEBSOCKET_ERROR = 179,
    ERROR_CODE_TLS_CLIENT_CERTIFICATE_ERROR = 180,
    ERROR_CODE_TLS_SERVER_CERTIFICATE_ERROR = 181,
    ERROR_CODE_TLS_CLIENT_AUTHENTICATION_FAILED = 182,
    ERROR_CODE_TLS_SERVER_AUTHENTICATION_FAILED = 183,
    ERROR_CODE_TLS_CLIENT_CERTIFICATE_EXPIRED = 184,
    ERROR_CODE_TLS_SERVER_CERTIFICATE_EXPIRED = 185,
    ERROR_CODE_TLS_CLIENT_CERTIFICATE_REVOKED = 186,
    ERROR_CODE_TLS_SERVER_CERTIFICATE_REVOKED = 187,
    ERROR_CODE_TLS_ERROR = 188,
    ERROR_CODE_DNS_UNAVAILABLE = 189,
    ERROR_CODE_DNS_NAME_RESOLUTION_FAILED = 190,
    ERROR_CODE_DNS_RESOLVER_FAILURE = 191,
    ERROR_CODE_DNS_ERROR = 192,
    ERROR_CODE_TCP_CONNECT_TIMEOUT = 193,
    ERROR_CODE_TCP_CONNECTION_REFUSED = 194,
    ERROR_CODE_TCP_CLOSED_BY_LOCAL = 195,
    ERROR_CODE_TCP_CLOSED_OTHER = 196,
    ERROR_CODE_TCP_ERROR = 197,
    ERROR_CODE_IP_ADDRESS_NOT_REACHABLE = 198,
    ERROR_CODE_IP_ERROR = 199,
    ERROR_CODE_CERTIFICATE_EXPIRED = 200,
    ERROR_CODE_CERTIFICATE_INVALID = 201,
    ERROR_CODE_CERTIFICATE_MALFORMED = 202,
    ERROR_CODE_CERTIFICATE_REVOKED = 203,
    ERROR_CODE_UNKNOWN_SECURITY_KEY = 204,
    ERROR_CODE_REFERENCED_PORT_IN_ERROR = 205,
    /* Enumerated values 0-255 are reserved for definition by ASHRAE. */
    ERROR_CODE_RESERVED_MAX = 255,
    /* Enumerated values 256-65535 may be used by others subject to */
    /* the procedures and constraints described in Clause 23. */
    /* do the max range inside of enum so that
       compilers will allocate adequate sized datatype for enum
       which is used to store decoding */
    ERROR_CODE_PROPRIETARY_FIRST = 256,
    /* some error codes for internal stack usage */
    ERROR_CODE_ = 65535,
    ERROR_CODE_DISCARD = 65534,
    ERROR_CODE_DEFAULT = 65535,
    ERROR_CODE_PROPRIETARY_LAST = 65535
} BACNET_ERROR_CODE;

typedef enum {
    BACNET_REINIT_COLDSTART = 0,
    BACNET_REINIT_WARMSTART = 1,
    BACNET_REINIT_STARTBACKUP = 2,
    BACNET_REINIT_ENDBACKUP = 3,
    BACNET_REINIT_STARTRESTORE = 4,
    BACNET_REINIT_ENDRESTORE = 5,
    BACNET_REINIT_ABORTRESTORE = 6,
    BACNET_REINIT_ACTIVATE_CHANGES = 7,
    BACNET_REINIT_IDLE = 255,
    BACNET_REINIT_MAX = 255
} BACNET_REINITIALIZED_STATE;

typedef enum BACnetNodeType {
    BACNET_NODE_UNKNOWN = 0,
    BACNET_NODE_SYSTEM = 1,
    BACNET_NODE_NETWORK = 2,
    BACNET_NODE_DEVICE = 3,
    BACNET_NODE_ORGANIZATIONAL = 4,
    BACNET_NODE_AREA = 5,
    BACNET_NODE_EQUIPMENT = 6,
    BACNET_NODE_POINT = 7,
    BACNET_NODE_COLLECTION = 8,
    BACNET_NODE_PROPERTY = 9,
    BACNET_NODE_FUNCTIONAL = 10,
    BACNET_NODE_OTHER = 11,
    BACNET_NODE_SUBSYSTEM = 12,
    BACNET_NODE_BUILDING = 13,
    BACNET_NODE_FLOOR = 14,
    BACNET_NODE_SECTION = 15,
    BACNET_NODE_MODULE = 16,
    BACNET_NODE_TREE = 17,
    BACNET_NODE_MEMBER = 18,
    BACNET_NODE_PROTOCOL = 19,
    BACNET_NODE_ROOM = 20,
    BACNET_NODE_ZONE = 21
} BACNET_NODE_TYPE;

typedef enum BACnetRelationship {
    BACNET_RELATIONSHIP_UNKNOWN = 0,
    BACNET_RELATIONSHIP_DEFAULT = 1,
    BACNET_RELATIONSHIP_CONTAINS = 2,
    BACNET_RELATIONSHIP_CONTAINED_BY = 3,
    BACNET_RELATIONSHIP_USES = 4,
    BACNET_RELATIONSHIP_USED_BY = 5,
    BACNET_RELATIONSHIP_COMMANDS = 6,
    BACNET_RELATIONSHIP_COMMANDED_BY = 7,
    BACNET_RELATIONSHIP_ADJUSTS = 8,
    BACNET_RELATIONSHIP_ADJUSTED_BY = 9,
    BACNET_RELATIONSHIP_INGRESS = 10,
    BACNET_RELATIONSHIP_EGRESS = 11,
    BACNET_RELATIONSHIP_SUPPLIES_AIR = 12,
    BACNET_RELATIONSHIP_RECEIVES_AIR = 13,
    BACNET_RELATIONSHIP_SUPPLIES_HOT_AIR = 14,
    BACNET_RELATIONSHIP_RECEIVES_HOT_AIR = 15,
    BACNET_RELATIONSHIP_SUPPLIES_COOL_AIR = 16,
    BACNET_RELATIONSHIP_RECEIVES_COOL_AIR = 17,
    BACNET_RELATIONSHIP_SUPPLIES_POWER = 18,
    BACNET_RELATIONSHIP_RECEIVES_POWER = 19,
    BACNET_RELATIONSHIP_SUPPLIES_GAS = 20,
    BACNET_RELATIONSHIP_RECEIVES_GAS = 21,
    BACNET_RELATIONSHIP_SUPPLIES_WATER = 22,
    BACNET_RELATIONSHIP_RECEIVES_WATER = 23,
    BACNET_RELATIONSHIP_SUPPLIES_HOT_WATER = 24,
    BACNET_RELATIONSHIP_RECEIVES_HOT_WATER = 25,
    BACNET_RELATIONSHIP_SUPPLIES_COOL_WATER = 26,
    BACNET_RELATIONSHIP_RECEIVES_COOL_WATER = 27,
    BACNET_RELATIONSHIP_SUPPLIES_STEAM = 28,
    BACNET_RELATIONSHIP_RECEIVES_STEAM = 29,
    BACNET_RELATIONSHIP_RESERVED_MAX = 1023,
    BACNET_RELATIONSHIP_PROPRIETARY_MIN = 1024,
    BACNET_RELATIONSHIP_PROPRIETARY_MAX = 65535
    /* Enumerated values 0-1023 are reserved for definition by ASHRAE.
       Enumerated values 1024-65535 may be used by others subject
       to the procedures and constraints described in Clause 23. */
} BACNET_RELATIONSHIP;

typedef enum BACnetShedState {
    BACNET_SHED_INACTIVE = 0,
    BACNET_SHED_REQUEST_PENDING = 1,
    BACNET_SHED_COMPLIANT = 2,
    BACNET_SHED_NON_COMPLIANT = 3
} BACNET_SHED_STATE;

typedef enum BACnetShedLevelType {
    BACNET_SHED_TYPE_PERCENT, /* Unsigned */
    BACNET_SHED_TYPE_LEVEL, /* Unsigned */
    BACNET_SHED_TYPE_AMOUNT /* REAL */
} BACNET_SHED_LEVEL_TYPE;

typedef enum BACnetLightingOperation {
    BACNET_LIGHTS_NONE = 0,
    BACNET_LIGHTS_FADE_TO = 1,
    BACNET_LIGHTS_RAMP_TO = 2,
    BACNET_LIGHTS_STEP_UP = 3,
    BACNET_LIGHTS_STEP_DOWN = 4,
    BACNET_LIGHTS_STEP_ON = 5,
    BACNET_LIGHTS_STEP_OFF = 6,
    BACNET_LIGHTS_WARN = 7,
    BACNET_LIGHTS_WARN_OFF = 8,
    BACNET_LIGHTS_WARN_RELINQUISH = 9,
    BACNET_LIGHTS_STOP = 10,
    MAX_BACNET_LIGHTING_OPERATION = 11,
    /* Enumerated values 0-255 are reserved for definition by ASHRAE.
       Enumerated values 256-65535 may be used by others subject to
       the procedures and constraints described in Clause 23 */
    BACNET_LIGHTS_PROPRIETARY_FIRST = 256,
    BACNET_LIGHTS_PROPRIETARY_LAST = 65535
} BACNET_LIGHTING_OPERATION;

typedef enum BACnetLightingInProgress {
    BACNET_LIGHTING_IDLE = 0,
    BACNET_LIGHTING_FADE_ACTIVE = 1,
    BACNET_LIGHTING_RAMP_ACTIVE = 2,
    BACNET_LIGHTING_NOT_CONTROLLED = 3,
    BACNET_LIGHTING_OTHER = 4,
    BACNET_LIGHTING_TRIM_ACTIVE = 5,
    MAX_BACNET_LIGHTING_IN_PROGRESS = 6
} BACNET_LIGHTING_IN_PROGRESS;

typedef enum BACnetLightingTransition {
    BACNET_LIGHTING_TRANSITION_NONE = 0,
    BACNET_LIGHTING_TRANSITION_FADE = 1,
    BACNET_LIGHTING_TRANSITION_RAMP = 2,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.
       Enumerated values 64-255 may be used by others subject to
       the procedures and constraints described in Clause 23. */
    BACNET_LIGHTING_TRANSITION_PROPRIETARY_FIRST = 64,
    BACNET_LIGHTING_TRANSITION_PROPRIETARY_LAST = 255
} BACNET_LIGHTING_TRANSITION;

typedef enum BACnetBinaryLightingPV {
    BINARY_LIGHTING_PV_OFF = 0,
    BINARY_LIGHTING_PV_ON = 1,
    BINARY_LIGHTING_PV_WARN = 2,
    BINARY_LIGHTING_PV_WARN_OFF = 3,
    BINARY_LIGHTING_PV_WARN_RELINQUISH = 4,
    BINARY_LIGHTING_PV_STOP = 5,
    BINARY_LIGHTING_PV_MAX = 6,
    /* -- Enumerated values 0-63 are reserved for definition by ASHRAE.
    -- Enumerated values 64-255 may be used by others
    -- subject to the procedures and constraints described in Clause 23. */
    BINARY_LIGHTING_PV_PROPRIETARY_MIN = 64,
    BINARY_LIGHTING_PV_PROPRIETARY_MAX = 255
} BACNET_BINARY_LIGHTING_PV;

typedef enum BACnetColorOperation {
    BACNET_COLOR_OPERATION_NONE = 0,
    BACNET_COLOR_OPERATION_FADE_TO_COLOR = 1,
    BACNET_COLOR_OPERATION_FADE_TO_CCT = 2,
    BACNET_COLOR_OPERATION_RAMP_TO_CCT = 3,
    BACNET_COLOR_OPERATION_STEP_UP_CCT = 4,
    BACNET_COLOR_OPERATION_STEP_DOWN_CCT = 5,
    BACNET_COLOR_OPERATION_STOP = 6,
    BACNET_COLOR_OPERATION_MAX = 7
} BACNET_COLOR_OPERATION;

typedef enum BACnetColorOperationInProgress {
    BACNET_COLOR_OPERATION_IN_PROGRESS_IDLE = 0,
    BACNET_COLOR_OPERATION_IN_PROGRESS_FADE_ACTIVE = 1,
    BACNET_COLOR_OPERATION_IN_PROGRESS_RAMP_ACTIVE = 2,
    BACNET_COLOR_OPERATION_IN_PROGRESS_NOT_CONTROLLED = 3,
    BACNET_COLOR_OPERATION_IN_PROGRESS_OTHER = 4,
    BACNET_COLOR_OPERATION_IN_PROGRESS_MAX = 5
} BACNET_COLOR_OPERATION_IN_PROGRESS;

typedef enum BACnetColorTransition {
    BACNET_COLOR_TRANSITION_NONE = 0,
    BACNET_COLOR_TRANSITION_FADE = 1,
    BACNET_COLOR_TRANSITION_RAMP = 2,
    BACNET_COLOR_TRANSITION_MAX = 3
} BACNET_COLOR_TRANSITION;

/* NOTE: BACNET_DAYS_OF_WEEK is different than BACNET_WEEKDAY */
/* 0=Monday-6=Sunday */
typedef enum BACnetDaysOfWeek {
    BACNET_DAYS_OF_WEEK_MONDAY = 0,
    BACNET_DAYS_OF_WEEK_TUESDAY = 1,
    BACNET_DAYS_OF_WEEK_WEDNESDAY = 2,
    BACNET_DAYS_OF_WEEK_THURSDAY = 3,
    BACNET_DAYS_OF_WEEK_FRIDAY = 4,
    BACNET_DAYS_OF_WEEK_SATURDAY = 5,
    BACNET_DAYS_OF_WEEK_SUNDAY = 6,
    MAX_BACNET_DAYS_OF_WEEK = 7
} BACNET_DAYS_OF_WEEK;

typedef enum BACnetEventTransitionBits {
    TRANSITION_TO_OFFNORMAL = 0,
    TRANSITION_TO_FAULT = 1,
    TRANSITION_TO_NORMAL = 2,
    MAX_BACNET_EVENT_TRANSITION = 3
} BACNET_EVENT_TRANSITION_BITS;

/* Not in standard, but useful for store Ack_Required */
typedef enum BACnetEventTransitionMask {
    TRANSITION_TO_OFFNORMAL_MASKED = 1,
    TRANSITION_TO_FAULT_MASKED = 2,
    TRANSITION_TO_NORMAL_MASKED = 4
} BACNET_EVENT_TRANSITION_MASK;

/* Access Enumerations */
/* Door Alarm State */
typedef enum {
    DOOR_ALARM_STATE_NORMAL = 0,
    DOOR_ALARM_STATE_ALARM = 1,
    DOOR_ALARM_STATE_DOOR_OPEN_TOO_LONG = 2,
    DOOR_ALARM_STATE_FORCED_OPEN = 3,
    DOOR_ALARM_STATE_TAMPER = 4,
    DOOR_ALARM_STATE_DOOR_FAULT = 5,
    DOOR_ALARM_STATE_LOCK_DOWN = 6,
    DOOR_ALARM_STATE_FREE_ACCESS = 7,
    DOOR_ALARM_STATE_EGRESS_OPEN = 8
} BACNET_DOOR_ALARM_STATE;

/* Door Secured Status */
typedef enum {
    DOOR_SECURED_STATUS_SECURED = 0,
    DOOR_SECURED_STATUS_UNSECURED = 1,
    DOOR_SECURED_STATUS_UNKNOWN = 2
} BACNET_DOOR_SECURED_STATUS;

/* Door Status */
typedef enum {
    DOOR_STATUS_CLOSED = 0,
    DOOR_STATUS_OPENED = 1,
    DOOR_STATUS_UNKNOWN = 2,
    DOOR_STATUS_DOOR_FAULT = 3,
    DOOR_STATUS_UNUSED = 4,
    DOOR_STATUS_NONE = 5,
    DOOR_STATUS_CLOSING = 6,
    DOOR_STATUS_OPENING = 7,
    DOOR_STATUS_SAFETY_LOCKED = 8,
    DOOR_STATUS_LIMITED_OPENED = 9
} BACNET_DOOR_STATUS;

/* Door Value */
typedef enum {
    DOOR_VALUE_LOCK = 0,
    DOOR_VALUE_UNLOCK = 1,
    DOOR_VALUE_PULSE_UNLOCK = 2,
    DOOR_VALUE_EXTENDED_PULSE_UNLOCK = 3
} BACNET_DOOR_VALUE;

/* Lock Status */
typedef enum {
    LOCK_STATUS_LOCKED = 0,
    LOCK_STATUS_UNLOCKED = 1,
    LOCK_STATUS_LOCK_FAULT = 2,
    LOCK_STATUS_UNUSED = 3,
    LOCK_STATUS_UNKNOWN = 4
} BACNET_LOCK_STATUS;

/* Access Event */
typedef enum {
    ACCESS_EVENT_NONE = 0,
    ACCESS_EVENT_GRANTED = 1,
    ACCESS_EVENT_MUSTER = 2,
    ACCESS_EVENT_PASSBACK_DETECTED = 3,
    ACCESS_EVENT_DURESS = 4,
    ACCESS_EVENT_TRACE = 5,
    ACCESS_EVENT_LOCKOUT_MAX_ATTEMPTS = 6,
    ACCESS_EVENT_LOCKOUT_OTHER = 7,
    ACCESS_EVENT_LOCKOUT_RELINQUISHED = 8,
    ACCESS_EVENT_LOCKED_BY_HIGHER_AUTHORITY = 9,
    ACCESS_EVENT_OUT_OF_SERVICE = 10,
    ACCESS_EVENT_OUT_OF_SERVICE_RELINQUISHED = 11,
    ACCESS_EVENT_ACCOMPANIMENT_BY = 12,
    ACCESS_EVENT_AUTHENTICATION_FACTOR_READ = 13,
    ACCESS_EVENT_AUTHORIZATION_DELAYED = 14,
    ACCESS_EVENT_VERIFICATION_REQUIRED = 15,
    /* values over 128 indicate that access is denied */
    ACCESS_EVENT_DENIED_DENY_ALL = 128,
    ACCESS_EVENT_DENIED_UNKNOWN_CREDENTIAL = 129,
    ACCESS_EVENT_DENIED_AUTHENTICATION_UNAVAILABLE = 130,
    ACCESS_EVENT_DENIED_AUTHENTICATION_FACTOR_TIMEOUT = 131,
    ACCESS_EVENT_DENIED_INCORRECT_AUTHENTICATION_FACTOR = 132,
    ACCESS_EVENT_DENIED_ZONE_NO_ACCESS_RIGHTS = 133,
    ACCESS_EVENT_DENIED_POINT_NO_ACCESS_RIGHTS = 134,
    ACCESS_EVENT_DENIED_NO_ACCESS_RIGHTS = 135,
    ACCESS_EVENT_DENIED_OUT_OF_TIME_RANGE = 136,
    ACCESS_EVENT_DENIED_THREAT_LEVEL = 137,
    ACCESS_EVENT_DENIED_PASSBACK = 138,
    ACCESS_EVENT_DENIED_UNEXPECTED_LOCATION_USAGE = 139,
    ACCESS_EVENT_DENIED_MAX_ATTEMPTS = 140,
    ACCESS_EVENT_DENIED_LOWER_OCCUPANCY_LIMIT = 141,
    ACCESS_EVENT_DENIED_UPPER_OCCUPANCY_LIMIT = 142,
    ACCESS_EVENT_DENIED_AUTHENTICATION_FACTOR_LOST = 143,
    ACCESS_EVENT_DENIED_AUTHENTICATION_FACTOR_STOLEN = 144,
    ACCESS_EVENT_DENIED_AUTHENTICATION_FACTOR_DAMAGED = 145,
    ACCESS_EVENT_DENIED_AUTHENTICATION_FACTOR_DESTROYED = 146,
    ACCESS_EVENT_DENIED_AUTHENTICATION_FACTOR_DISABLED = 147,
    ACCESS_EVENT_DENIED_AUTHENTICATION_FACTOR_ERROR = 148,
    ACCESS_EVENT_DENIED_CREDENTIAL_UNASSIGNED = 149,
    ACCESS_EVENT_DENIED_CREDENTIAL_NOT_PROVISIONED = 150,
    ACCESS_EVENT_DENIED_CREDENTIAL_NOT_YET_ACTIVE = 151,
    ACCESS_EVENT_DENIED_CREDENTIAL_EXPIRED = 152,
    ACCESS_EVENT_DENIED_CREDENTIAL_MANUAL_DISABLE = 153,
    ACCESS_EVENT_DENIED_CREDENTIAL_LOCKOUT = 154,
    ACCESS_EVENT_DENIED_CREDENTIAL_MAX_DAYS = 155,
    ACCESS_EVENT_DENIED_CREDENTIAL_MAX_USES = 156,
    ACCESS_EVENT_DENIED_CREDENTIAL_INACTIVITY = 157,
    ACCESS_EVENT_DENIED_CREDENTIAL_DISABLED = 158,
    ACCESS_EVENT_DENIED_NO_ACCOMPANIMENT = 159,
    ACCESS_EVENT_DENIED_INCORRECT_ACCOMPANIMENT = 160,
    ACCESS_EVENT_DENIED_LOCKOUT = 161,
    ACCESS_EVENT_DENIED_VERIFICATION_FAILED = 162,
    ACCESS_EVENT_DENIED_VERIFICATION_TIMEOUT = 163,
    ACCESS_EVENT_DENIED_OTHER = 164
} BACNET_ACCESS_EVENT;

/* Authentication Status */
typedef enum {
    AUTHENTICATION_STATUS_NOT_READY = 0,
    AUTHENTICATION_STATUS_READY = 1,
    AUTHENTICATION_STATUS_DISABLED = 2,
    AUTHENTICATION_STATUS_WAITING_FOR_AUTHENTICATION_FACTOR = 3,
    AUTHENTICATION_STATUS_WAITING_FOR_ACCOMPANIMENT = 4,
    AUTHENTICATION_STATUS_WAITING_FOR_VERIFICATION = 5,
    AUTHENTICATION_STATUS_IN_PROGRESS = 6
} BACNET_AUTHENTICATION_STATUS;

/* Authorization Mode */
typedef enum {
    AUTHORIZATION_MODE_AUTHORIZE = 0,
    AUTHORIZATION_MODE_GRANT_ACTIVE = 1,
    AUTHORIZATION_MODE_DENY_ALL = 2,
    AUTHORIZATION_MODE_VERIFICATION_REQUIRED = 3,
    AUTHORIZATION_MODE_AUTHORIZATION_DELAYED = 4,
    AUTHORIZATION_MODE_NONE = 5
} BACNET_AUTHORIZATION_MODE;

/* Access Passback Mode */
typedef enum {
    ACCESS_PASSBACK_MODE_PASSBACK_OFF = 0,
    ACCESS_PASSBACK_MODE_HARD_PASSBACK = 1,
    ACCESS_PASSBACK_MODE_SOFT_PASSBACK = 2
} BACNET_ACCESS_PASSBACK_MODE;

/* Access Zone Occupancy State */
typedef enum BACnetAccessZoneOccupancyState {
    ACCESS_ZONE_OCCUPANCY_STATE_NORMAL = 0,
    ACCESS_ZONE_OCCUPANCY_STATE_BELOW_LOWER_LIMIT = 1,
    ACCESS_ZONE_OCCUPANCY_STATE_AT_LOWER_LIMIT = 2,
    ACCESS_ZONE_OCCUPANCY_STATE_AT_UPPER_LIMIT = 3,
    ACCESS_ZONE_OCCUPANCY_STATE_ABOVE_UPPER_LIMIT = 4,
    ACCESS_ZONE_OCCUPANCY_STATE_DISABLED = 5,
    ACCESS_ZONE_OCCUPANCY_STATE_NOT_SUPPORTED = 6,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.
    Enumerated values 64-65535 may be used by others subject
    to the procedures and constraints described in Clause 23. */
    ACCESS_ZONE_OCCUPANCY_STATE_PROPRIETARY_MIN = 64,
    ACCESS_ZONE_OCCUPANCY_STATE_PROPRIETARY_MAX = 65535
} BACNET_ACCESS_ZONE_OCCUPANCY_STATE;

/* Access User Type */
typedef enum {
    ACCESS_USER_TYPE_ASSET = 0,
    ACCESS_USER_TYPE_GROUP = 1,
    ACCESS_USER_TYPE_PERSON = 2
} BACNET_ACCESS_USER_TYPE;

/* Access Authentication Factor Disable */
typedef enum {
    ACCESS_AUTHENTICATION_FACTOR_DISABLE_NONE = 0,
    ACCESS_AUTHENTICATION_FACTOR_DISABLE_DISABLED = 1,
    ACCESS_AUTHENTICATION_FACTOR_DISABLE_DISABLED_LOST = 2,
    ACCESS_AUTHENTICATION_FACTOR_DISABLE_DISABLED_STOLEN = 3,
    ACCESS_AUTHENTICATION_FACTOR_DISABLE_DISABLED_DAMAGED = 4,
    ACCESS_AUTHENTICATION_FACTOR_DISABLE_DISABLED_DESTROYED = 5,
    ACCESS_AUTHENTICATION_FACTOR_DISABLE_MAX = 6
} BACNET_ACCESS_AUTHENTICATION_FACTOR_DISABLE;

/* Authorization Exemption */
typedef enum {
    AUTHORIZATION_EXEMPTION_PASSBACK = 0,
    AUTHORIZATION_EXEMPTION_OCCUPANCY_CHECK = 1,
    AUTHORIZATION_EXEMPTION_ACCESS_RIGHTS = 2,
    AUTHORIZATION_EXEMPTION_LOCKOUT = 3,
    AUTHORIZATION_EXEMPTION_DENY = 4,
    AUTHORIZATION_EXEMPTION_VERIFICATION = 5,
    AUTHORIZATION_EXEMPTION_AUTHORIZATION_DELAY = 6
} BACNET_AUTHORIZATION_EXEMPTION;

/* The Network Reject Reasons  for NETWORK_MESSAGE_REJECT_MESSAGE_TO_NETWORK */
typedef enum {
    NETWORK_REJECT_UNKNOWN_ERROR = 0,
    NETWORK_REJECT_NO_ROUTE = 1,
    NETWORK_REJECT_ROUTER_BUSY = 2,
    NETWORK_REJECT_UNKNOWN_MESSAGE_TYPE = 3,
    NETWORK_REJECT_MESSAGE_TOO_LONG = 4,
    NETWORK_REJECT_BACNET_SECURITY = 5,
    NETWORK_REJECT_BAD_ADDRESS = 6,
    /* Reasons this value or above we don't know about */
    NETWORK_REJECT_REASON_INVALID
} BACNET_NETWORK_REJECT_REASONS;

typedef enum BACnetWriteStatus {
    BACNET_WRITE_STATUS_IDLE = 0,
    BACNET_WRITE_STATUS_IN_PROGRESS = 1,
    BACNET_WRITE_STATUS_SUCCESSFUL = 2,
    BACNET_WRITE_STATUS_FAILED = 3
} BACNET_WRITE_STATUS;

/* BACnetNetworkType ::= ENUMERATED */
typedef enum {
    PORT_TYPE_ETHERNET = 0,
    PORT_TYPE_ARCNET = 1,
    PORT_TYPE_MSTP = 2,
    PORT_TYPE_PTP = 3,
    PORT_TYPE_LONTALK = 4,
    PORT_TYPE_BIP = 5,
    PORT_TYPE_ZIGBEE = 6,
    PORT_TYPE_VIRTUAL = 7,
    /* note: (8), removed in version 1 revision 18 */
    PORT_TYPE_NON_BACNET = 8,
    PORT_TYPE_BIP6 = 9,
    PORT_TYPE_SERIAL = 10,
    /*  For BACnet/SC network port implementations with
        a Protocol_Revision 24 and higher, BACnet/SC network ports shall be
        represented by a Network Port object at the BACNET_APPLICATION
        protocol level with network type of SECURE_CONNECT. */
    PORT_TYPE_BSC = 11,
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.
       Enumerated values 64-255 may be used by others subject to the
       procedures and constraints described in Clause 23. */
    PORT_TYPE_RESERVED_FIRST = 12,
    PORT_TYPE_RESERVED_LAST = 63,
    PORT_TYPE_PROPRIETARY_FIRST = 64,
    /*  For BACnet/SC network port implementations with
        a protocol revision Protocol_Revision 17 and higher through 23,
        BACnet/SC network ports shall be represented by a Network Port
        object at the BACNET_APPLICATION protocol level with
        a proprietary network type value. */
    PORT_TYPE_BSC_INTERIM = 64,
    PORT_TYPE_PROPRIETARY_LAST = 255,
    PORT_TYPE_MAX = 255
} BACNET_PORT_TYPE;

/* BACnetNetworkNumberQuality ::= ENUMERATED */
typedef enum {
    PORT_QUALITY_UNKNOWN = 0,
    PORT_QUALITY_LEARNED = 1,
    PORT_QUALITY_LEARNED_CONFIGURED = 2,
    PORT_QUALITY_CONFIGURED = 3
} BACNET_PORT_QUALITY;

/* BACnetNetworkPortCommand :: = ENUMERATED */
typedef enum {
    PORT_COMMAND_IDLE = 0,
    PORT_COMMAND_DISCARD_CHANGES = 1,
    PORT_COMMAND_RENEW_FD_REGISTRATION = 2,
    PORT_COMMAND_RESTART_SLAVE_DISCOVERY = 3,
    PORT_COMMAND_RENEW_DHCP = 4,
    PORT_COMMAND_RESTART_AUTONEGOTIATION = 5,
    PORT_COMMAND_DISCONNECT = 6,
    PORT_COMMAND_RESTART_PORT = 7,
    PORT_COMMAND_GENERATE_CSR_FILE = 8,
    PORT_COMMAND_VALIDATE_CHANGES = 9,
    PORT_COMMAND_RESERVED_FIRST = 10,
    PORT_COMMAND_RESERVED_LAST = 127,
    /* Enumerated values 0-127 are reserved for definition by ASHRAE.
       Enumerated values 128-255 may be used by others subject to the
       procedures and constraints described in Clause 23. */
    PORT_COMMAND_PROPRIETARY_FIRST = 128,
    PORT_COMMAND_PROPRIETARY_LAST = 255,
    PORT_COMMAND_MAX = 255
} BACNET_PORT_COMMAND;

typedef enum {
    BACNET_SECURITY_LEVEL_INCAPABLE = 0,
    BACNET_SECURITY_LEVEL_PLAIN = 1,
    BACNET_SECURITY_LEVEL_SIGNED = 2,
    BACNET_SECURITY_LEVEL_ENCRYPTED = 3,
    BACNET_SECURITY_LEVEL_SIGNED_END_TO_END = 4,
    BACNET_SECURITY_LEVEL_ENCRYPTED_END_TO_END = 5
} BACNET_SECURITY_LEVEL;

typedef enum {
    BACNET_SECURITY_POLICY_PLAIN_NOT_TRUSTED = 0,
    BACNET_SECURITY_POLICY_PLAIN_TRUSTED = 1,
    BACNET_SECURITY_POLICY_SIGNED_TRUSTED = 2,
    BACNET_SECURITY_POLICY_ENCRYPTED_TRUSTED = 3
} BACNET_SECURITY_POLICY;

typedef enum {
    KIA_AES_MD5 = 0,
    KIA_AES_SHA256 = 1,
    /* 2-255 reserved */
    KIA_MAX_KEY_IDENTIFIER_ALGORITHM = 255
} BACNET_KEY_IDENTIFIER_ALGORITHM;

typedef enum {
    KIKN_NOT_USED = 0,
    KIKN_DEVICE_MASTER = 1,
    KIKN_DISTRIBUTION = 2,
    KIKN_INSTALLATION = 3,
    KIKN_GENERAL_NETWORK_ACCESS = 4,
    KIKN_USER_AUTHENTICATED = 5,
    KIKN_MIN_APPLICATION_SPECIFIC = 6,
    KIKN_MAX_APPLICATION_SPECIFIC = 127,
    /* 128-255 reserved */
    KIKN_MAX_KEY_IDENTIFIER_KEY_NUMBER = 255
} BACNET_KEY_IDENTIFIER_KEY_NUMBER;

typedef enum {
    SEC_RESP_SUCCESS = 0,
    SEC_RESP_ACCESS_DENIED = 1,
    SEC_RESP_BAD_DESTINATION_ADDRESS = 2,
    SEC_RESP_BAD_DESTINATION_DEVICE_ID = 3,
    SEC_RESP_BAD_SIGNATURE = 4,
    SEC_RESP_BAD_SOURCE_ADDRESS = 5,
    SEC_RESP_BAD_TIMESTAMP = 6,
    SEC_RESP_CANNOT_USE_KEY = 7,
    SEC_RESP_CANNOT_VERIFY_MESSAGE_ID = 8,
    SEC_RESP_CORRECT_KEY_REVISION = 9,
    SEC_RESP_DESTINATION_DEVICE_ID_REQUIRED = 10,
    SEC_RESP_DUPLICATE_MESSAGE = 11,
    SEC_RESP_ENCRYPTION_NOT_CONFIGURED = 12,
    SEC_RESP_ENCRYPTION_REQUIRED = 13,
    SEC_RESP_INCORRECT_KEY = 14,
    SEC_RESP_INVALID_KEY_DATA = 15,
    SEC_RESP_KEY_UPDATE_IN_PROGRESS = 16,
    SEC_RESP_MALFORMED_MESSAGE = 17,
    SEC_RESP_NOT_KEY_SERVER = 18,
    SEC_RESP_SECURITY_NOT_CONFIGURED = 19,
    SEC_RESP_SOURCE_SECURITY_REQUIRED = 20,
    SEC_RESP_TOO_MANY_KEYS = 21,
    SEC_RESP_UNKNOWN_AUTHENTICATION_TYPE = 22,
    SEC_RESP_UNKNOWN_KEY = 23,
    SEC_RESP_UNKNOWN_KEY_REVISION = 24,
    SEC_RESP_UNKNOWN_SOURCE_MESSAGE = 25
} BACNET_SECURITY_RESPONSE_CODE;

typedef enum {
    ACCESS_CREDENTIAL_DISABLE_NONE = 0,
    ACCESS_CREDENTIAL_DISABLE = 1,
    ACCESS_CREDENTIAL_DISABLE_MANUAL = 2,
    ACCESS_CREDENTIAL_DISABLE_LOCKOUT = 3,
    ACCESS_CREDENTIAL_DISABLE_MAX = 4
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.
       Enumerated values 64-65535 may be used by others subject to
       the procedures and constraints described in Clause 23. */
} BACNET_ACCESS_CREDENTIAL_DISABLE;

typedef enum {
    CREDENTIAL_DISABLED = 0,
    CREDENTIAL_DISABLED_NEEDS_PROVISIONING = 1,
    CREDENTIAL_DISABLED_UNASSIGNED = 2,
    CREDENTIAL_DISABLED_NOT_YET_ACTIVE = 3,
    CREDENTIAL_DISABLED_EXPIRED = 4,
    CREDENTIAL_DISABLED_LOCKOUT = 5,
    CREDENTIAL_DISABLED_MAX_DAYS = 6,
    CREDENTIAL_DISABLED_MAX_USES = 7,
    CREDENTIAL_DISABLED_INACTIVITY = 8,
    CREDENTIAL_DISABLED_MANUAL = 9,
    CREDENTIAL_DISABLED_MAX = 10
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.
       Enumerated values 64-65535 may be used by others subject to
       the procedures and constraints described in Clause 23. */
} BACNET_ACCESS_CREDENTIAL_DISABLE_REASON;

typedef enum {
    AUTHENTICATION_NONE = 0,
    AUTHENTICATION_DISABLED = 1,
    AUTHENTICATION_DISABLED_LOST = 2,
    AUTHENTICATION_DISABLED_STOLEN = 3,
    AUTHENTICATION_DISABLED_DAMAGED = 4,
    AUTHENTICATION_DISABLED_DESTROYED = 5,
    AUTHENTICATION_DISABLED_MAX = 6
    /* Enumerated values 0-63 are reserved for definition by ASHRAE.
       Enumerated values 64-65535 may be used by others subject to
       the procedures and constraints described in Clause 23. */
} BACNET_AUTHENTICATION_DISABLE_REASON;

typedef enum {
    AUTHENTICATION_FACTOR_UNDEFINED = 0,
    AUTHENTICATION_FACTOR_ERROR = 1,
    AUTHENTICATION_FACTOR_CUSTOM = 2,
    AUTHENTICATION_FACTOR_SIMPLE_NUMBER16 = 3,
    AUTHENTICATION_FACTOR_SIMPLE_NUMBER32 = 4,
    AUTHENTICATION_FACTOR_SIMPLE_NUMBER56 = 5,
    AUTHENTICATION_FACTOR_SIMPLE_ALPHA_NUMERIC = 6,
    AUTHENTICATION_FACTOR_ABA_TRACK2 = 7,
    AUTHENTICATION_FACTOR_WIEGAND26 = 8,
    AUTHENTICATION_FACTOR_WIEGAND37 = 9,
    AUTHENTICATION_FACTOR_WIEGAND37_FACILITY = 10,
    AUTHENTICATION_FACTOR_FACILITY16_CARD32 = 11,
    AUTHENTICATION_FACTOR_FACILITY32_CARD32 = 12,
    AUTHENTICATION_FACTOR_FASC_N = 13,
    AUTHENTICATION_FACTOR_FASC_N_BCD = 14,
    AUTHENTICATION_FACTOR_FASC_N_LARGE = 15,
    AUTHENTICATION_FACTOR_FASC_N_LARGE_BCD = 16,
    AUTHENTICATION_FACTOR_GSA75 = 17,
    AUTHENTICATION_FACTOR_CHUID = 18,
    AUTHENTICATION_FACTOR_CHUID_FULL = 19,
    AUTHENTICATION_FACTOR_GUID = 20,
    AUTHENTICATION_FACTOR_CBEFF_A = 21,
    AUTHENTICATION_FACTOR_CBEFF_B = 22,
    AUTHENTICATION_FACTOR_CBEFF_C = 23,
    AUTHENTICATION_FACTOR_USER_PASSWORD = 24,
    AUTHENTICATION_FACTOR_MAX = 25
} BACNET_AUTHENTICATION_FACTOR_TYPE;

typedef enum BACnetProtocolLevel_T {
    BACNET_PROTOCOL_LEVEL_PHYSICAL = 0,
    BACNET_PROTOCOL_LEVEL_PROTOCOL = 1,
    BACNET_PROTOCOL_LEVEL_BACNET_APPLICATION = 2,
    BACNET_PROTOCOL_LEVEL_NON_BACNET_APPLICATION = 3
} BACNET_PROTOCOL_LEVEL;

typedef enum BACnetIPMode_T {
    BACNET_IP_MODE_NORMAL = 0,
    BACNET_IP_MODE_FOREIGN = 1,
    BACNET_IP_MODE_BBMD = 2
} BACNET_IP_MODE;

typedef enum BACnetBackupState {
    BACKUP_STATE_IDLE = 0,
    BACKUP_STATE_PREPARING_FOR_BACKUP = 1,
    BACKUP_STATE_PREPARING_FOR_RESTORE = 2,
    BACKUP_STATE_PERFORMING_A_BACKUP = 3,
    BACKUP_STATE_PERFORMING_A_RESTORE = 4,
    BACKUP_STATE_BACKUP_FAILURE = 5,
    BACKUP_STATE_RESTORE_FAILURE = 6
} BACNET_BACKUP_STATE;

typedef enum BACnetTimerState {
    TIMER_STATE_IDLE = 0,
    TIMER_STATE_RUNNING = 1,
    TIMER_STATE_EXPIRED = 2
} BACNET_TIMER_STATE;

typedef enum BACnetTimerTransition {
    TIMER_TRANSITION_NONE = 0,
    TIMER_TRANSITION_IDLE_TO_RUNNING = 1,
    TIMER_TRANSITION_RUNNING_TO_IDLE = 2,
    TIMER_TRANSITION_RUNNING_TO_RUNNING = 3,
    TIMER_TRANSITION_RUNNING_TO_EXPIRED = 4,
    TIMER_TRANSITION_FORCED_TO_EXPIRED = 5,
    TIMER_TRANSITION_EXPIRED_TO_IDLE = 6,
    TIMER_TRANSITION_EXPIRED_TO_RUNNING = 7
} BACNET_TIMER_TRANSITION;

typedef enum BACnetEscalatorFault {
    ESCALATOR_FAULT_CONTROLLER_FAULT = 0,
    ESCALATOR_FAULT_DRIVE_AND_MOTOR_FAULT = 1,
    ESCALATOR_FAULT_MECHANICAL_COMPONENT_FAULT = 2,
    ESCALATOR_FAULT_OVERSPEED_FAULT = 3,
    ESCALATOR_FAULT_POWER_SUPPLY_FAULT = 4,
    ESCALATOR_FAULT_SAFETY_DEVICE_FAULT = 5,
    ESCALATOR_FAULT_CONTROLLER_SUPPLY_FAULT = 6,
    ESCALATOR_FAULT_DRIVE_TEMPERATURE_EXCEEDED = 7,
    ESCALATOR_FAULT_COMB_PLATE_FAULT = 8,
    /* Enumerated values 0-1023 are reserved for definition by ASHRAE.
    Enumerated values 1024-65535 may be used by others subject to
    the procedures and constraints described in Clause 23. */
    ESCALATOR_FAULT_PROPRIETARY_MIN = 1024,
    ESCALATOR_FAULT_PROPRIETARY_MAX = 65535
} BACNET_ESCALATOR_FAULT;

typedef enum BACnetEscalatorMode {
    ESCALATOR_MODE_UNKNOWN = 0,
    ESCALATOR_MODE_STOP = 1,
    ESCALATOR_MODE_UP = 2,
    ESCALATOR_MODE_DOWN = 3,
    ESCALATOR_MODE_INSPECTION = 4,
    ESCALATOR_MODE_OUT_OF_SERVICE = 5,
    /* Enumerated values 0-1023 are reserved for definition by ASHRAE.
    Enumerated values 1024-65535 may be used by others subject to
    the procedures and constraints described in Clause 23. */
    ESCALATOR_MODE_PROPRIETARY_MIN = 1024,
    ESCALATOR_MODE_PROPRIETARY_MAX = 65535
} BACNET_ESCALATOR_MODE;

typedef enum BACnetEscalatorOperationDirection {
    ESCALATOR_OPERATION_DIRECTION_UNKNOWN = 0,
    ESCALATOR_OPERATION_DIRECTION_STOPPED = 1,
    ESCALATOR_OPERATION_DIRECTION_UP_RATED_SPEED = 2,
    ESCALATOR_OPERATION_DIRECTION_UP_REDUCED_SPEED = 3,
    ESCALATOR_OPERATION_DIRECTION_DOWN_RATED_SPEED = 4,
    ESCALATOR_OPERATION_DIRECTION_DOWN_REDUCED_SPEED = 5,
    /* Enumerated values 0-1023 are reserved for definition by ASHRAE.
    Enumerated values 1024-65535 may be used by others subject to
    the procedures and constraints described in Clause 23. */
    ESCALATOR_OPERATION_DIRECTION_PROPRIETARY_MIN = 1024,
    ESCALATOR_OPERATION_DIRECTION_PROPRIETARY_MAX = 65535
} BACNET_ESCALATOR_OPERATION_DIRECTION;

typedef enum BACnetLiftCarDirection {
    LIFT_CAR_DIRECTION_UNKNOWN = 0,
    LIFT_CAR_DIRECTION_NONE = 1,
    LIFT_CAR_DIRECTION_STOPPED = 2,
    LIFT_CAR_DIRECTION_UP = 3,
    LIFT_CAR_DIRECTION_DOWN = 4,
    LIFT_CAR_DIRECTION_UP_AND_DOWN = 5,
    /* Enumerated values 0-1023 are reserved for definition by ASHRAE.
    Enumerated values 1024-65535 may be used by others subject to
    the procedures and constraints described in Clause 23. */
    LIFT_CAR_DIRECTION_PROPRIETARY_MIN = 1024,
    LIFT_CAR_DIRECTION_PROPRIETARY_MAX = 65535
} BACNET_LIFT_CAR_DIRECTION;

typedef enum BACnetLiftCarDoorCommand {
    LIFT_CAR_DOOR_COMMAND_NONE = 0,
    LIFT_CAR_DOOR_COMMAND_OPEN = 1,
    LIFT_CAR_DOOR_COMMAND_CLOSE = 2
} BACNET_LIFT_CAR_DOOR_COMMAND;

typedef enum BACnetLiftCarDriveStatus {
    LIFT_CAR_DRIVE_STATUS_UNKNOWN = 0,
    LIFT_CAR_DRIVE_STATUS_STATIONARY = 1,
    LIFT_CAR_DRIVE_STATUS_BRAKING = 2,
    LIFT_CAR_DRIVE_STATUS_ACCELERATE = 3,
    LIFT_CAR_DRIVE_STATUS_DECELERATE = 4,
    LIFT_CAR_DRIVE_STATUS_RATED_SPEED = 5,
    LIFT_CAR_DRIVE_STATUS_SINGLE_FLOOR_JUMP = 6,
    LIFT_CAR_DRIVE_STATUS_TWO_FLOOR_JUMP = 7,
    LIFT_CAR_DRIVE_STATUS_THREE_FLOOR_JUMP = 8,
    LIFT_CAR_DRIVE_STATUS_MULTI_FLOOR_JUMP = 9,
    /* Enumerated values 0-1023 are reserved for definition by ASHRAE.
    Enumerated values 1024-65535 may be used by others subject to
    the procedures and constraints described in Clause 23. */
    LIFT_CAR_DRIVE_STATUS_PROPRIETARY_MIN = 1024,
    LIFT_CAR_DRIVE_STATUS_PROPRIETARY_MAX = 65535
} BACNET_LIFT_CAR_DRIVE_STATUS;

typedef enum BACnetLiftCarMode {
    LIFT_CAR_MODE_UNKNOWN = 0,
    LIFT_CAR_MODE_NORMAL = 1, /* in service */
    LIFT_CAR_MODE_VIP = 2,
    LIFT_CAR_MODE_HOMING = 3,
    LIFT_CAR_MODE_PARKING = 4,
    LIFT_CAR_MODE_ATTENDANT_CONTROL = 5,
    LIFT_CAR_MODE_FIREFIGHTER_CONTROL = 6,
    LIFT_CAR_MODE_EMERGENCY_POWER = 7,
    LIFT_CAR_MODE_INSPECTION = 8,
    LIFT_CAR_MODE_CABINET_RECALL = 9,
    LIFT_CAR_MODE_EARTHQUAKE_OPERATION = 10,
    LIFT_CAR_MODE_FIRE_OPERATION = 11,
    LIFT_CAR_MODE_OUT_OF_SERVICE = 12,
    LIFT_CAR_MODE_OCCUPANT_EVACUATION = 13,
    /* Enumerated values 0-1023 are reserved for definition by ASHRAE.
    Enumerated values 1024-65535 may be used by others subject to
    the procedures and constraints described in Clause 23. */
    LIFT_CAR_MODE_PROPRIETARY_MIN = 1024,
    LIFT_CAR_MODE_PROPRIETARY_MAX = 65535
} BACNET_LIFT_CAR_MODE;

typedef enum BACnetLiftFault {
    LIFT_FAULT_CONTROLLER_FAULT = 0,
    LIFT_FAULT_DRIVE_AND_MOTOR_FAULT = 1,
    LIFT_FAULT_GOVERNOR_AND_SAFETY_GEAR_FAULT = 2,
    LIFT_FAULT_LIFT_SHAFT_DEVICE_FAULT = 3,
    LIFT_FAULT_POWER_SUPPLY_FAULT = 4,
    LIFT_FAULT_SAFETY_INTERLOCK_FAULT = 5,
    LIFT_FAULT_DOOR_CLOSING_FAULT = 6,
    LIFT_FAULT_DOOR_OPENING_FAULT = 7,
    LIFT_FAULT_CAR_STOPPED_OUTSIDE_LANDING_ZONE = 8,
    LIFT_FAULT_CALL_BUTTON_STUCK = 9,
    LIFT_FAULT_START_FAILURE = 10,
    LIFT_FAULT_CONTROLLER_SUPPLY_FAULT = 11,
    LIFT_FAULT_SELF_TEST_FAILURE = 12,
    LIFT_FAULT_RUNTIME_LIMIT_EXCEEDED = 13,
    LIFT_FAULT_POSITION_LOST = 14,
    LIFT_FAULT_DRIVE_TEMPERATURE_EXCEEDED = 15,
    LIFT_FAULT_LOAD_MEASUREMENT_FAULT = 16,
    /* Enumerated values 0-1023 are reserved for definition by ASHRAE.
    Enumerated values 1024-65535 may be used by others subject to
    the procedures and constraints described in Clause 23. */
    LIFT_FAULT_PROPRIETARY_MIN = 1024,
    LIFT_FAULT_PROPRIETARY_MAX = 65535
} BACNET_LIFT_FAULT;

typedef enum BACnetLiftGroupMode {
    LIFT_GROUP_MODE_UNKNOWN = 0,
    LIFT_GROUP_MODE_NORMAL = 1,
    LIFT_GROUP_MODE_DOWN_PEAK = 2,
    LIFT_GROUP_MODE_TWO_WAY = 3,
    LIFT_GROUP_MODE_FOUR_WAY = 4,
    LIFT_GROUP_MODE_EMERGENCY_POWER = 5,
    LIFT_GROUP_MODE_UP_PEAK = 6
} BACNET_LIFT_GROUP_MODE;

typedef enum BACnetAuditLevel {
    AUDIT_LEVEL_NONE = 0,
    AUDIT_LEVEL_AUDIT_ALL = 1,
    AUDIT_LEVEL_AUDIT_CONFIG = 2,
    AUDIT_LEVEL_DEFAULT = 3,
    /* Enumerated values 0-127 are reserved for definition by ASHRAE.
    Enumerated values 128-255 may be used by others subject to
    the procedures and constraints described in Clause 23. */
    AUDIT_LEVEL_PROPRIETARY_MIN = 128,
    AUDIT_LEVEL_PROPRIETARY_MAX = 255
} BACNET_AUDIT_LEVEL;

typedef enum BACnetAuditOperation {
    AUDIT_OPERATION_READ = 0,
    AUDIT_OPERATION_WRITE = 1,
    AUDIT_OPERATION_CREATE = 2,
    AUDIT_OPERATION_DELETE = 3,
    AUDIT_OPERATION_LIFE_SAFETY = 4,
    AUDIT_OPERATION_ACKNOWLEDGE_ALARM = 5,
    AUDIT_OPERATION_DEVICE_DISABLE_COMM = 6,
    AUDIT_OPERATION_DEVICE_ENABLE_COMM = 7,
    AUDIT_OPERATION_DEVICE_RESET = 8,
    AUDIT_OPERATION_DEVICE_BACKUP = 9,
    AUDIT_OPERATION_DEVICE_RESTORE = 10,
    AUDIT_OPERATION_SUBSCRIPTION = 11,
    AUDIT_OPERATION_NOTIFICATION = 12,
    AUDIT_OPERATION_AUDITING_FAILURE = 13,
    AUDIT_OPERATION_NETWORK_CHANGES = 14,
    AUDIT_OPERATION_GENERAL = 15,
    /* Enumerated values 0-31 are reserved for definition by ASHRAE.
    Enumerated values 32-63 may be used by others subject to
    the procedures and constraints described in Clause 23.

    The enumerated values match the bit positions in
    BACnetAuditOperationFlags. */
    AUDIT_OPERATION_PROPRIETARY_MIN = 32,
    AUDIT_OPERATION_PROPRIETARY_MAX = 63
} BACNET_AUDIT_OPERATION;

typedef enum BACnetSCHubConnectorState {
    /* FIXME: prefix with typedef name as much as possible */
    BACNET_SC_HUB_CONNECTOR_STATE_NO_HUB_CONNECTION = 0,
    BACNET_SC_HUB_CONNECTOR_STATE_CONNECTED_TO_PRIMARY = 1,
    BACNET_SC_HUB_CONNECTOR_STATE_CONNECTED_TO_FAILOVER = 2,
    BACNET_SC_HUB_CONNECTOR_STATE_MAX = 3
} BACNET_SC_HUB_CONNECTOR_STATE;

typedef enum BACnetSCConnectionState {
    BACNET_SC_CONNECTION_STATE_NOT_CONNECTED = 0,
    BACNET_SC_CONNECTION_STATE_CONNECTED = 1,
    BACNET_SC_CONNECTION_STATE_DISCONNECTED_WITH_ERRORS = 2,
    BACNET_SC_CONNECTION_STATE_FAILED_TO_CONNECT = 3,
    BACNET_SC_CONNECTION_STATE_MAX = 4
} BACNET_SC_CONNECTION_STATE;

#endif /* end of BACENUM_H */
