require "rails_helper"

RSpec.feature "password", type: :feature do
  context "edit" do
    let(:url) { "/admin/resources/users/#{user.slug}/edit" }

    subject do
      visit url
      find_field_value_element("password")
    end

    describe "with password set" do
      let(:old_password) { "foobar" }
      let(:new_password) { "new_foobar" }
      let!(:user) { create :user, password: old_password }

      it { is_expected.to have_field type: "password", id: "user_password", placeholder: "User Password", text: nil }

      it "does not change the password" do
        visit url
        expect(page).to have_field type: "password", id: "user_password", placeholder: "User Password", text: nil

        fill_in "user_first_name", with: "Johnny"

        save

        expect(current_path).to eql "/admin/resources/users/#{user.slug}"
        expect(page).to have_text "Johnny"

        expect(user.valid_password?(old_password)).to be(true)
      end

      it "changes the password" do
        visit url
        expect(page).to have_field type: "password", id: "user_password", placeholder: "User Password", text: nil

        fill_in "user_password", with: new_password
        fill_in "user_password_confirmation", with: new_password

        save

        expect(current_path).to eql "/admin/resources/users/#{user.slug}"

        expect(User.last.valid_password?(old_password)).to be(false)
        expect(User.last.valid_password?(new_password)).to be(true)
      end
    end
  end

  context "create" do
    let(:url) { "/admin/resources/users/new" }

    describe "new user with password" do
      it "checks placeholder" do
        visit url

        expect(page).to have_field type: "password", id: "user_password", placeholder: "User Password", text: nil
        expect(page).to have_field type: "password", id: "user_password_confirmation", placeholder: "Password confirmation", text: nil
      end

      it "checks password revealable button" do
        visit url

        expect(page).to have_selector "[data-password-visibility-target]"
      end

      it "saves the resource with password" do
        visit url

        expect(page).to have_field type: "password", id: "user_password", placeholder: "User Password", text: nil
        expect(page).to have_field type: "password", id: "user_password_confirmation", placeholder: "Password confirmation", text: nil

        fill_in "user_first_name", with: "John"
        fill_in "user_last_name", with: "Doe"
        fill_in "user_email", with: "johndoe@test.com"
        fill_in "user_password", with: "passwordtest"
        fill_in "user_password_confirmation", with: "passwordtest"

        save

        expect(current_path).to eql "/admin/resources/users/#{User.last.slug}"
        expect(page).to have_text "John"
        expect(page).to have_text "Doe"
        expect(User.last.valid_password?("passwordtest")).to be(true)
      end
    end
  end
end
