---
layout: post
title: "Probability distribution sampling in C# using StatDist and Troschuetz.Random"
excerpt: "Learn how to sample various common probability distributions in C#"
thumbtext: StatDist + TRandom
image: assets/img-min/social/J9mO6DVgPJY.webp
categories: [csharp]
tags: [csharp]
author: apkd
series: false
featured: false
hidden: true
license: cc-by
contributors: []
---

# Introduction
{:.no_toc}

Let's talk about sampling [probability density functions](https://en.wikipedia.org/wiki/Probability_density_function).

For example, imagine that you're generating a combat encounter.[^0] You need to spawn several enemies, but not a fixed count - let's assume a range of `[1 .. 10]`{{site.code.cs}} opponents.

[^0]: Sorry, gamedev examples only, but the general technique could come in useful for all kinds of C# projects.

At this point we could [roll a d10](https://rollthedice.online/en/dice/d10) and call it a day, but that wouldn't feel very creative. What if we'd prefer to *usually* choose a small number of opponents, and only occasionally challenge the player with a larger fight?

Basically, we want to sample a **probability distribution** that looks like this:

![statdist.com screenshot](assets/img/posts/statdist-03.png){:style="max-height:400px;"}  
*[Binomial distribution](https://statdist.com/distributions/binomial) with `n = 30, p = 0.11` - usually yields a number around 3, but sometimes even up to 10.*
{:.text-center.small}

Different scenario. You're spawning some loot and need to decide whether to include a rare item. Again, if your game is simple a dice roll might be sufficient. But let's assume that you wish the player to have a `Luck` stat that affects the drop chance nonlinearly. Consider these constraints:
* At low levels, you want the player to immediately feel the benefits of increasing their `Luck`
* Obviously, the drop probability cannot exceed 100%
* But at the same time it would be nice if stacking huge amounts of `Luck` still benefited the player *somewhat*, even if by a small amount

What we need is a random event that depends on a difficulty level with [diminishing returns](https://en.wikipedia.org/wiki/Diminishing_returns). Basically, we're looking for a distribution with a **cumulative distribution** that looks something like this:

![statdist.com screenshot](assets/img/posts/statdist-04.png){:style="max-height:440px;"}  
*Cumulative distribution function for the [Erlang distribution](https://statdist.com/distributions/erlang) with `k = 1, λ = 0.01`.  
In our example, 𝑥 represents our character's `Luck`.  
P(𝒳 < 𝑥) - the green area - represents the probability that `Luck` is sufficient to spawn the item.*
{:.text-center.small}

These are just two examples, but what you probably want is some different probability distribution, with a set of parameters that is specific to your game or program.

If you're a visual thinker, you might have a rough idea of what *shape* your imaginary reward value distribution is, but no idea how to represent it in code. How do we solve this? Here's one workflow I like...

* chapter 1
* chapter 2
* chapter 3
{:toc}

# Finding your dream distribution w/ [StatDist](https://statdist.com/)

[StatDist](https://statdist.com/) is a cool little website that lets you plot dozens of common statistical distributions.

![statdist.com screenshot](assets/img/posts/statdist-01.png)
*Distributions shown in red are not supported by Troschuetz.Random*
{:.text-center.small}

Simply choose a distribution, input the parameters and you'll get an overview of what the distribution is like. Thankfully, the scary equations are hidden by default, but there's a *Details* button for
the brave.

The graphs and the 𝑥 parameter (which lets us preview the value of the function at 𝑥) can be used to build intuition for how the distribution behaves by simply playing with it.

![statdist.com screenshot](assets/img/posts/statdist-02.png)

# Sampling distributions w/ [Troschuetz.Random](https://nuget.org/packages/Troschuetz.Random)

[Troschuetz.Random](https://nuget.org/packages/Troschuetz.Random) is a library which implements various random number generators and distributions. In most cases, you can simply copy the
parameters chosen using StatDist into the function call. Super convenient.

```csharp
var random = new Troschuetz.Random.TRandom();

// get a random value based on some chosen distribution
double sample = random.Binomial(alpha: 30, beta: 0.1);

// clamp the value to our indended value range (recommended)
sample = Math.Clamp(sample, min: 0, max: 10);
```

This yields us a random 𝑥 based on the distribution we previously nailed down. No longer are we bound to the uniformly-distributed random numbers generated by `System.Random` - now we can sample any random distribution we like!

# Generating random events w/ [Troschuetz.Random](https://nuget.org/packages/Troschuetz.Random)

Sometimes we want to decide whether a random event has occured based on a parameter that represents difficulty.[^1]

If you ever used the `System.Random` class like this: `random.Next() < 0.9 // 90% probability`{{site.code.cs}}, then this is basically the same thing, except with `TRandom` we can use a more interesting, non-uniform distribution of random values. This allows us to have a non-linear relationship between the difficulty and the probability of the random event.[^2]

[^1]: Note that when you input an 𝑥, StatDist will plot P(𝒳 < 𝑥), which is more similar to our `luck` stat. If you want to think in terms of a `difficulty` check instead, you'd need to use P(𝒳 > 𝑥). In practice, to switch between them you can simply invert the test result.

[^2]: You can explore this relationship by looking at the **cumulative distribution function** graph on the right side of StatDist's UI.

Basically, you just do this:

```csharp
var random = new Troschuetz.Random.TRandom();

// the "luck" value that we want our random value to undershoot
// (if you prefer to think in terms of a "difficulty" value instead, simply invert the final result)
double luck = 100;

// get a random value based on some chosen distribution
double sample = random.Erlang(alpha: 1, lambda: 0.01);

// test whether the random value is less than the player's "luck"
// (or higher than the "difficulty" value)
bool randomEventOccurred = sample < luck;
```

# Distributions supported by both [Troschuetz.Random](https://nuget.org/packages/Troschuetz.Random) and [StatDist](https://statdist.com/)

Here's the functions you can use. Also check out [this article by the package author](https://www.codeproject.com/articles/15102/net-random-number-generators-and-distributions) and the [API docs](https://pomma89.gitlab.io/troschuetz-random/class_troschuetz_1_1_random_1_1_t_random.html).

<style>
.distrib {
    background-color: #222; 
    margin-top: 0 !important; 
    margin-bottom: 8px !important; 
}
</style>

## `double Beta(double alpha, double beta);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/beta.png){:.pixel-perfect.distrib}  
0 < α < ∞  
0 < β < ∞  
0 ≤ X ≤ 1  
[View on StatDist](https://statdist.com/distributions/beta)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Beta_distribution)

## `int Binomial(double alpha /* trials */, int beta /* probability */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/binomial.png){:.pixel-perfect.distrib}  
0 ≤ α ≤ 1  
β ∈ { 0, 1, ... }  
X ∈ { 0, 1, ..., β }  
[View on StatDist](https://statdist.com/distributions/binomial)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Binomial_distribution)

## `double Cauchy(double alpha /* location */, double gamma /* scale */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/cauchy.png){:.pixel-perfect.distrib}  
-∞ < α < ∞  
0 < γ < ∞  
-∞ < X < ∞  
[View on StatDist](https://statdist.com/distributions/cauchy)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Cauchy_distribution)

## `double ChiSquare(int alpha /* degrees of freedom */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/chi-squared.png){:.pixel-perfect.distrib}  
α ∈ { 1, 2, ... }  
0 ≤ X < ∞  
[View on StatDist](https://statdist.com/distributions/chi-squared)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Chi-squared_distribution)

## `double ContinuousUniform(double alpha /* min */, double beta /* max */);`{{site.code.cs}}
{:.h5.mt-5 #ContinuousUniform}
> 
![distribution graph](assets/img/posts/distributions/uniform.png){:.pixel-perfect.distrib}  
α ≤ β < ∞  
-∞ < α ≤ β  
α ≤ X < β  
[View on StatDist](https://statdist.com/distributions/uniform)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Continuous_uniform_distribution)

## `int DiscreteUniform(int alpha /* min */, int beta /* min */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/uniform.png){:.pixel-perfect.distrib}  
α ∈ { ..., β-1, β}  
β ∈ { α, α+1, ... }  
X ∈ { α, α+1, ..., β-1, β }  
Not on StatDist, but see the [continuous version](#ContinuousUniform) above  
[View on Wikipedia](https://en.wikipedia.org/wiki/Discrete_uniform_distribution)

## `double Erlang(int alpha /* shape */, double lambda /* rate */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/erlang.png){:.pixel-perfect.distrib}  
0 < α < ∞  
λ ∈ { 1, 2, ... }  
0 ≤ X < ∞  
[View on StatDist](https://statdist.com/distributions/erlang)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Erlang_distribution)

## `double Exponential(double lambda /* rate */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/exponential.png){:.pixel-perfect.distrib}  
0 < λ < ∞  
0 ≤ X < ∞  
[View on StatDist](https://statdist.com/distributions/exponential)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Exponential_distribution)

## `double FisherSnedecor(int alpha, int beta);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/f.png){:.pixel-perfect.distrib}  
α ∈ {1, 2, ... }  
β ∈ {1, 2, ... }  
0 ≤ X < ∞  
[View on StatDist](https://statdist.com/distributions/f)  
[View on Wikipedia](https://en.wikipedia.org/wiki/F-distribution)

## `double Gamma(double alpha /* shape */, double beta /* scale */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/gamma.png){:.pixel-perfect.distrib}  
0 < α < ∞  
0 < β < ∞  
0 ≤ X < ∞  
[View on StatDist](https://statdist.com/distributions/gamma)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Gamma_distribution)

## `double Laplace(double alpha /* location */, double mu /* scale */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/laplace.png){:.pixel-perfect.distrib}  
0 < α < ∞  
-∞ < μ < ∞  
-∞ < X < ∞  
[View on StatDist](https://statdist.com/distributions/laplace)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Laplace_distribution)

## `double Logistic(double mu /* mean */, double sigma /* scale */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/logistic.png){:.pixel-perfect.distrib}  
-∞ < μ < ∞  
0 < σ < ∞  
-∞ < X < ∞  
[View on StatDist](https://statdist.com/distributions/logistic)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Logistic_distribution)

## `double Lognormal(double mu /* mean */, double sigma /* standard deviation */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/log-normal.png){:.pixel-perfect.distrib}  
-∞ < μ < ∞  
0 ≤ σ < ∞  
0 ≤ X < ∞  
[View on StatDist](https://statdist.com/distributions/log-normal)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Log-normal_distribution)

## `double Normal(double mu /* mean */, double sigma /* standard deviation */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/normal.png){:.pixel-perfect.distrib}  
-∞ < μ < ∞  
0 < σ < ∞  
-∞ < X < ∞  
[View on StatDist](https://statdist.com/distributions/normal)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Normal_distribution)

## `double Pareto(double alpha /* scale */, double beta /* shape */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/pareto.png){:.pixel-perfect.distrib}  
0 < α < ∞  
0 < β < ∞  
α ≤ X < ∞  
[View on StatDist](https://statdist.com/distributions/pareto)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Pareto_distribution)

## `int Poisson(double lambda /* rate */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/poisson.png){:.pixel-perfect.distrib}  
0 < λ < ∞  
X ∈ { 0, 1, ... }  
[View on StatDist](https://statdist.com/distributions/poisson)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Poisson_distribution)

## `double Rayleigh(double sigma /* scale */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/rayleigh.png){:.pixel-perfect.distrib}  
0 < σ < ∞  
0 ≤ X < ∞  
[View on StatDist](https://statdist.com/distributions/rayleigh)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Rayleigh_distribution)

## `double StudentsT(int nu /* degrees of freedom */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/students-t.png){:.pixel-perfect.distrib}  
ν ∈ { 1, 2, ... }  
-∞ < X < ∞  
[View on StatDist](https://statdist.com/distributions/student-t)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Student%27s_t-distribution)

## `double Triangular(double alpha /* min */, double beta /* mode */, double gamma /* max */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/triangular.png){:.pixel-perfect.distrib}  
-∞ < α < β  
α < β < ∞  
α ≤ γ ≤ β  
α ≤ X ≤ β  
[View on StatDist](https://statdist.com/distributions/triangular)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Triangular_distribution)

## `double Weibull(double alpha /* shape */, double lambda /* scale */);`{{site.code.cs}}
{:.h5.mt-5}
> 
![distribution graph](assets/img/posts/distributions/weibull.png){:.pixel-perfect.distrib}  
0 < α < ∞  
0 < λ < ∞  
0 ≤ X < ∞  
[View on StatDist](https://statdist.com/distributions/weibull)  
[View on Wikipedia](https://en.wikipedia.org/wiki/Weibull_distribution)
