<?php

/*
 * Copyright 2018 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

namespace Apigee\Edge\Exception;

use Http\Message\Formatter;
use Http\Message\Formatter\FullHttpMessageFormatter;
use Psr\Http\Message\RequestInterface;
use Throwable;

/**
 * Exception for when a request failed, providing access to the failed request.
 *
 * This could be due to an invalid request, or one of the extending exceptions
 * for network errors or HTTP error responses.
 */
class ApiRequestException extends ApiException
{
    /** @var RequestInterface */
    protected $request;

    /** @var Formatter */
    protected $formatter;

    /**
     * ApiException constructor.
     *
     * @param RequestInterface $request
     * @param string $message
     * @param int $code
     * @param Throwable|null $previous
     * @param Formatter|null $formatter
     */
    public function __construct(
        RequestInterface $request,
        string $message = '',
        int $code = 0,
        ?Throwable $previous = null,
        ?Formatter $formatter = null,
    ) {
        $this->request = $request;
        $this->formatter = $formatter ?: new FullHttpMessageFormatter();
        parent::__construct($message, $code, $previous);
    }

    /**
     * {@inheritdoc}
     */
    public function __toString(): string
    {
        $output = [
            get_called_class() . PHP_EOL,
            'Request:' . PHP_EOL . $this->formatter->formatRequest($this->request) . PHP_EOL,
            'Stack trace: ' . PHP_EOL . $this->getTraceAsString(),
        ];

        return implode(PHP_EOL, $output);
    }

    /**
     * @return RequestInterface
     */
    public function getRequest(): RequestInterface
    {
        return $this->request;
    }

    /**
     * @return Formatter
     */
    public function getFormatter(): Formatter
    {
        return $this->formatter;
    }
}
