---
title: Some change on getContainer
date: 2022-12-08
author: zombieJ
zhihu_url: https://zhuanlan.zhihu.com/p/606878571
yuque_url: https://www.yuque.com/ant-design/ant-design/eegn0tn5fy94uwk8
---

We often encounter the need for pop-up elements when developing, such as the Select drop-down box, or the Modal component. When it is directly rendered under the current node, it may be clipped by the `overflow: hidden` of the parent node:

<img alt="Overflow" height="200" src="https://mdn.alipayobjects.com/huamei_7uahnr/afts/img/A*Noh-TYJ0BdcAAAAAAAAAAAAADrJ8AQ/original" />

Therefore we render it under `body` by default in Ant Design, but this will bring new problems. Since they are not under the same container, when the user scrolls the screen, they will find that the popup layer does not follow the scrolling:

<img alt="Scroll" height="370" src="https://mdn.alipayobjects.com/huamei_7uahnr/afts/img/A*d44KQqkTX90AAAAAAAAAAAAADrJ8AQ/original" />

To solve this problem, we provide the `getContainer` property, which allows users to customize the rendered container. The `getContainer` method will be called when the component is mounted, returning a container node, and the component will be rendered under this node through `createPortal`.

```tsx
// Fake Code. Just for Demo
const PopupWrapper = () => {
  const eleRef = React.useRef<HTMLDivElement>(null);

  React.useEffect(() => {
    // It's much complex with timing in real world. You can view the source for more detail:
    // https://github.com/react-component/portal/blob/master/src/Portal.tsx
    const container: HTMLElement = getContainer(eleRef.current);

    // ...
  }, []);

  return (
    <div ref={eleRef}>
      {...}
    </div>
  );
}
```

```tsx
// Fake Code. Just for Demo
const defaultGetContainer = () => {
  const div = document.createElement('div');
  document.body.appendChild(div);
  return div;
};

const SomeComponent = ({ getContainer = defaultGetContainer }) => (
  <PopupWrapper getContainer={getContainer} />
);
```

For the time being, we don’t pay attention to `getContainer`’s need to dynamically switch the mount node (in fact, it has not been able to switch for a long time in the past), only from the perspective of React 18, it has encountered some problems.

## React 18 Concurrent Mode

In React 18, effects may fire multiple times. In order to prevent inadvertently breaking the developer's behavior, it has also been adjusted accordingly under [StrictMode](https://reactjs.org/docs/strict-mode.html):

> - React mounts the component.
>   - Layout effects are created.
>   - Effect effects are created.
> - React simulates effects being destroyed on a mounted component.
>   - Layout effects are destroyed.
>   - Effects are destroyed.
> - React simulates effects being re-created on a mounted component.
>   - Layout effects are created
>   - Effect setup code runs

The simple understanding is that under StrictMode, even if your deps contains empty objects, the effect will still be triggered multiple times. When switching to React 18 StrictMode, we will find that there will be a pair of mount nodes in the HTML, and the previous one is empty:

```html
<body>
  <div id="root">...</div>

  <!-- Empty -->
  <div className="sample-holder"></div>

  <!-- Real in use -->
  <div className="sample-holder">
    <div className="ant-component-wrapper">...</div>
  </div>
</body>
```

Therefore, we adjusted the call implementation, and the default `getContainer` is also managed through state to ensure that the nodes generated by the previous effect will be cleaned up in StrictMode:

```tsx
// Fake Code. Just for Demo
const SomeComponent = ({ getContainer }) => {
  const [myContainer, setMyContainer] = React.useState<HTMLElement | null>(null);

  React.useEffect(() => {
    if (getContainer) {
      setMyContainer(getContainer());
      return;
    }

    const div = document.createElement('div');
    document.body.appendChild(div);
    setMyContainer(div);

    return () => {
      document.body.removeChild(div);
    };
  }, [getContainer]);

  return <PopupWrapper getContainer={() => myContainer} />;
};
```

After putting `getContainer` into effect management, we can manage nodes in a way that is more in line with the React life cycle, and we can also clean up when `getContainer` changes. So as to support the scenario of dynamically changing `getContainer` (although I personally doubt the universality of this usage scenario).

## Finally

Due to the fix that `getContainer` does not support dynamic changes, it also introduces a potential breaking change at the same time. If the developer customizes `getContainer` to create a new DOM node every time, it will cause an infinite loop because of the continuous execution of the effect, resulting in the continuous creation of nodes. If you use this method and encounter problems, you need to pay attention to check.
