//
//  ConfigureLocalModelSourceViewModel.swift
//  LlamaChat
//
//  Created by Alex Rozanski on 01/04/2023.
//

import Foundation
import Combine
import SwiftUI

struct ConfiguredSource {
  let name: String
  let avatarImageName: String?
  let settings: SourceSettings
}

class ConfigureLocalModelSourceViewModel: ObservableObject, ConfigureSourceViewModel {
  typealias NextHandler = (ConfiguredSource) -> Void

  private lazy var nameGenerator = SourceNameGenerator()

  // MARK: - Info

  @Published var name: String
  @Published var avatarImageName: String?

  var modelType: String {
    return chatSourceType.readableName
  }

  var modelSourcingDescription: LocalizedStringKey {
    switch chatSourceType {
    case .llama:
      return "The LLaMA model checkpoints and tokenizer are required to add this chat source. Learn more and request access to these on the [LLaMA GitHub repo](https://github.com/facebookresearch/llama)."
    case .alpaca:
      return "The Alpaca model checkpoints and tokenizer are required to add this chat source. Learn more on the [Alpaca GitHub repo](https://github.com/tatsu-lab/stanford_alpaca)."
    case .gpt4All:
      return "The GPT4All .ggml model file is required to add this chat source. Learn more on the [llama.cpp GitHub repo](https://github.com/ggerganov/llama.cpp/blob/a0caa34/README.md#using-gpt4all)."
    }
  }

  // MARK: - Model Settings

  var settingsViewModels = [ConfigureLocalModelSourceType: ConfigureLocalModelSettingsViewModel]()

  @Published private(set) var modelSourceType: ConfigureLocalModelSourceType? = nil {
    didSet {
      guard let modelSourceType else {
        settingsViewModel = nil
        return
      }

      if let existingModel = settingsViewModels[modelSourceType] {
        settingsViewModel = existingModel
      } else {
        switch modelSourceType {
        case .pyTorch:
          let viewModel = ConfigureLocalPyTorchModelSettingsViewModel(chatSourceType: chatSourceType)
          viewModel.determineConversionStateIfNeeded()
          settingsViewModels[.pyTorch] = viewModel
          settingsViewModel = viewModel
        case .ggml:
          let viewModel = ConfigureLocalGgmlModelSettingsViewModel(
            chatSourceType: chatSourceType,
            exampleModelPath: exampleGgmlModelPath
          )
          settingsViewModels[.ggml] = viewModel
          settingsViewModel = viewModel
        }
      }
    }
  }

  @Published private(set) var settingsViewModel: ConfigureLocalModelSettingsViewModel?

  // MARK: - Validation

  let primaryActionsViewModel: ConfigureSourcePrimaryActionsViewModel

  let chatSourceType: ChatSourceType
  let exampleGgmlModelPath: String
  private let nextHandler: NextHandler

  private var subscriptions = Set<AnyCancellable>()

  init(
    defaultName: String? = nil,
    chatSourceType: ChatSourceType,
    exampleGgmlModelPath: String,
    nextHandler: @escaping NextHandler
  ) {
    self.name = defaultName ?? ""
    self.chatSourceType = chatSourceType
    self.exampleGgmlModelPath = exampleGgmlModelPath
    self.nextHandler = nextHandler
    primaryActionsViewModel = ConfigureSourcePrimaryActionsViewModel()
    primaryActionsViewModel.delegate = self

    let configuredSource = $settingsViewModel
      .compactMap { $0 }
      .map { $0.sourceSettings }
      .switchToLatest()

    $name
      .map { !$0.trimmingCharacters(in: .whitespacesAndNewlines).isEmpty }
      .combineLatest(configuredSource)
      .sink { [weak self] nameValid, configuredSource in
        self?.primaryActionsViewModel.canContinue = nameValid && configuredSource != nil
      }.store(in: &subscriptions)

    $modelSourceType
      .sink { [weak self] newSourceType in
        self?.primaryActionsViewModel.showContinueButton = newSourceType != nil

        if let newSourceType {
          switch newSourceType {
          case .pyTorch:
            self?.primaryActionsViewModel.nextButtonTitle = "Continue"
          case .ggml:
            self?.primaryActionsViewModel.nextButtonTitle = "Add"
          }
        }
      }.store(in: &subscriptions)
  }

  func generateName() {
    if let generatedName = nameGenerator.generateName(for: chatSourceType) {
      name = generatedName
    }
  }

  func select(modelSourceType: ConfigureLocalModelSourceType?) {
    self.modelSourceType = modelSourceType
  }
}

extension ConfigureLocalModelSourceType {
  var label: String {
    switch self {
    case .pyTorch: return "PyTorch Checkpoint (.pth)"
    case .ggml: return "GGML (.ggml)"
    }
  }
}

extension ConfigureLocalModelSourceViewModel: ConfigureSourcePrimaryActionsViewModelDelegate {
  func next() {
    guard let sourceSettings = settingsViewModel?.sourceSettings.value else { return }
    nextHandler(
      ConfiguredSource(name: name, avatarImageName: avatarImageName, settings: sourceSettings)
    )
  }
}
