/**
 * @file File Object Class
 * keeps files information, upload the file and updates status
 * @author Alban Xhaferllari
 * @version 1.0
 */
define(['Constants', 'Utils', 'SimpleRunner', 'i18n', 'FileMd5', 'ExifReader', 'ImageScale'], /** @lends FileObject */
function (Constants, Utils, SimpleRunner, _, FileMd5, ExifReader, ImageScale) {
    'use strict';
    /**
     * FileObject class holds file logic and file upload
     * @class FileObject
     * @param {HTMLElement} file The file object created by the DOM (not the input)
     * @param {String} fileId a unique file id in queue generated by queue manager
     * @param {RealUploader} AU the queue manager
     * @constructor
     */
    var FileObject = function (file, fileId, AU) {
        var me = this;
        //File properties
        me.file = file;                 //real file object
        me.name = file.name;            //name of file
        me.size = file.size;    		//size of file
        me.ext = file.extension;   	//file extension
        me.fileId = fileId;               //unique id  for the file
        me.tempFileName = null;               //temp file name used for upload, generated on server side
        me.xhr = null; 			    //xmlhttprequest object or form
        me.info = null; 			    //info about upload status
        me.extraInfo = null; 			    //info about upload status
        me.status = Constants.AX_IDLE; 	//status -1 error, 0 idle 1 done, 2 uploading, 3 check, 4 ready
        me.AU = AU; 				    //RealUploader object
        me.config = AU.config;            //master configuration
        me.exifData = null;                 //exif information of JPEG file
        me.md5 = '';                   //md5 of file calculated with javascript
        me.disabled = false;			    //if disabled cannot be uploaded
        me.ready = false;                //boolean telling when file is ready for upload
        me.requestDuration = 0;               //medium request duration
        me.currentFun = null;                 //current running pre-upload function
        me.dom = {};                   //contains references to dom template objects
        me.imgCls = '';                   //variable for keep the orientation of file
        me.checkSum = {};                   //checksum information about me file

        //temp variables
        me.currentByte = 0; 	            //current uploaded byte
        me.loading_bytes = 0;
        me.temp_bytes = 0;
        me._preUploadFun = [];
        me.speedInterval = null;             //interval handling the bandwidth statics
        me.init();
    };

    FileObject.prototype = {
        /**
         * The initialize function of the class, calls all the other functions
         * @returns {FileObject} returns the instance for the chaining
         */
        init: function () {
            var me = this;
            //visual part
            me.renderHtml();

            //bind events
            me.bindEvents();

            //bind file calculation, operation, most of case heavy operations using WebWorkers
            //for the moment run on file select
            me.queueFileOperations();

            //create a preview for supported files
            me.bindFilePreview();

            //run the queue operations
            me.disableUpload(_('Processing'));
            me.runQueue();
            return me;
        },
        /**
         * Destroy the file object class, removes the file from queue, stop the upload and removes html
         * @public
         */
        destroy: function () {
            var me = this;

            //stop any file function running
            me.stopQueue();

            //stop upload if the files is being upload
            me.stopUpload();

            //remove file from the queue
            me.AU._removeFile(me.fileId);

            //remove dom element
            me.dom.container.parentNode.removeChild(me.dom.container);

            //set all possible references to null
            me.file = null;
            me.AU = null;
            me.xhr = null;
            me.exifData = null;
            me.dom = null;
            me.config = null;
        },
        /**
         * Adds functions on the file (md5, resize...) to be run in serial to avoid browser hang
         * @returns {FileObject} Return the instance for chaining
         * @private
         */
        queueFileOperations: function () {
            Utils.log('queueFileOperations');
            var me = this;
            //if resize is configured and imageScale plugin is active (present) then queue the resize operation
            if (typeof ImageScale !== 'undefined') {
                //run and check the before image resize if returns false abort resize
                if (me.AU.triggerEvent('beforeImageResize', [me]) !== false) {

                    var scale = new ImageScale(me.file, me.config.resizeImage);

                    scale.done(function (result) {

                        //trigger custom user events
                        me.AU.triggerEvent('imageResize', [me, result]);
                        if (result) {
                            me.file = result;
                            me.setSize(result.size);
                        }
                        me.setMessage('').setProgress(0);
                    }, me).progress(function (percent) {
                        //progress bar update
                        me.workProgress(_('Resizing'), percent);
                    }, me);
                    me._preUploadFun.push(scale);
                }
            }

            //calculate md5
            if (typeof FileMd5 !== 'undefined' && me.config.md5Calculate) {

                //trigger md5 event start
                me.AU.triggerEvent('md5Start', [me]);

                //start md5 calculation
                var md5App = new FileMd5(me.file);
                md5App.done(function (md5) {
                    Utils.log('md5 calculated', md5);
                    me.md5 = md5;
                    me.AU.triggerEvent('md5Done', [me, md5]);
                    me.setMessage('').setProgress(0);
                }, me).progress(function (percent) {
                    //progress bar update
                    me.workProgress(_('Calculating md5'), percent);
                }, me);
                me._preUploadFun.push(md5App);
            }

            //exif read
            if (typeof ExifReader !== 'undefined' && me.config.exifRead) {
                var exifApp = new ExifReader(me.file);
                exifApp.done(function (exif) {
                    Utils.log('exif calculated', exif);
                    me.exifData = exif;
                    me.AU.triggerEvent('exifDone', [me, exif]);
                    me.setMessage('').setProgress(0);
                    me.imgCls = me.fixOrientation();
                    me.dom.previewImage.className += ' ' + me.imgCls;//fix image orientation
                }, me).progress(function (percent) {
                    //progress bar update
                    me.workProgress(_('Reading exif'), percent);
                }, me);
                me._preUploadFun.push(exifApp);
            }

            return me;
        },
        /**
         * Stop the running function on the file, will stops workers.
         */
        stopQueue: function () {
            if (this.currentFun) {
                this.currentFun.stop();
                this.currentFun = null;
            }
            this.enableUpload(_('Ready for upload'));
            this._preUploadFun = [];
        },
        /**
         * Start exec of the functions
         * @returns {FileObject} Chaining
         */
        runQueue: function () {
            if (this._preUploadFun.length > 0) {
                var objClass = this._preUploadFun.shift();
                objClass.always(this.runQueue, this);
                this.currentFun = objClass;
                objClass.start();
            } else {
                this.enableUpload(_('Ready for upload'));
            }
            return this;
        },
        /**
         * Render file html in the dom based on the selected theme and template
         */
        renderHtml: function () {
            var me = this;

            //base file template configuration: override with fileTemplate option
            var tpl = me.config.fileTemplate ? me.config.fileTemplate : Constants.FILE_TEMPLATE;
            var tplEdit = me.AU.triggerEvent('beforeRenderFile', [me, tpl]);
            if (tplEdit) {
                tpl = tplEdit;
            }

            //create the file placeholder
            me.dom.container = document.createElement('div');
            //just a short hand avoiding long code down
            var c = me.dom.container;
            c.innerHTML = tpl;
            c.classList.add('ax-file-wrapper');


            //get a reference to the visible elements
            me.dom.nameContainer = Utils.getEl(c, '.ax-file-name');
            me.dom.sizeContainer = Utils.getEl(c, '.ax-file-size');

            //upload button
            me.dom.uploadButton = Utils.getEl(c, '.ax-upload');
            me.dom.uploadButtonText = Utils.getEl(me.dom.uploadButton, '.ax-btn-text');

            //remove button
            me.dom.removeButton = Utils.getEl(c, '.ax-remove');
            me.dom.removeButtonText = Utils.getEl(me.dom.removeButton, '.ax-btn-text');

            //delete button
            me.dom.deleteButton = Utils.getEl(c, '.ax-delete');
            me.dom.deleteButtonText = Utils.getEl(me.dom.deleteButton, '.ax-btn-text');

            //info button
            me.dom.infoButton = Utils.getEl(c, '.ax-info');
            me.dom.infoButtonText = Utils.getEl(me.dom.infoButton, '.ax-btn-text');

            me.dom.previewImage = Utils.getEl(c, '.ax-preview');
            me.dom.previewContainer = Utils.getEl(c, '.ax-prev-container');
            me.dom.progressBar = Utils.getEl(c, '.ax-progress-bar');
            me.dom.progressInfo = Utils.getEl(c, '.ax-progress-info');
            me.dom.progressStat = Utils.getEl(c, '.ax-progress-stat');

            //use extra function to populate DOM elements with data
            //this ensure to check if the element exists, user can remove it by changing the template
            me.setName(me.name)
                .setSize(me.size)
                .setUploadButton()
                .setDeleteButton(false)
                .setRemoveButton(_('Remove file from queue'))
                .setInfoButton();

            me.AU.dom.fileList.appendChild(me.dom.container);
            me.AU.triggerEvent('afterRenderFile', [me, me.dom]);
            return me;
        },
        /**
         * Set the name of the file. It will set the DOM html and the name of the uploaded file on server
         * @param {String} name
         * @returns {FileObject}
         */
        setName: function (name) {
            var me = this;
            me.name = name;
            //update dom
            me.dom.container.setAttribute('title', name);
            if (me.dom.nameContainer) {
                me.dom.nameContainer.innerHTML = name;
                me.dom.nameContainer.setAttribute('title', name);
            }
            return me;
        },
        /**
         * Set the file size. Only for internal use, since file size cannot be set by the user
         * @param size
         * @returns {FileObject}
         * @private
         */
        setSize: function (size) {
            var me = this;
            me.size = size;
            //update dom
            if (me.dom.sizeContainer) {
                me.dom.sizeContainer.innerHTML = Utils.formatSize(size);
                me.dom.sizeContainer.setAttribute('title', me.dom.sizeContainer.innerHTML);
            }
            return me;
        },
        /**
         * Setup the upload button of the file, manages status and String
         * @returns {FileObject}
         */
        setUploadButton: function () {
            var me = this;
            if (me.dom.uploadButton && !me.config.hideUploadButton) {
                if (me.status == Constants.AX_READY || me.status == Constants.AX_ERROR) {
                    me.dom.uploadButton.classList.remove('ax-abort');
                    me.dom.uploadButton.setAttribute('title', _('Start upload'));

                    if (me.dom.uploadButtonText) {
                        me.dom.uploadButtonText.innerHTML = _('Upload');
                    }
                } else {
                    me.dom.uploadButton.classList.add('ax-abort');
                    me.dom.uploadButton.setAttribute('title', _('Abort upload'));
                    if (me.dom.uploadButtonText) {
                        me.dom.uploadButtonText.innerHTML = _('Abort');
                    }
                }
            }
            return me;
        },
        /**
         * Set remove button
         * @param msg
         * @returns {FileObject}
         */
        setRemoveButton: function (msg) {
            var me = this;
            if (me.dom.removeButton) {
                me.dom.removeButton.setAttribute('title', msg);
                if (me.dom.removeButtonText) {
                    me.dom.removeButtonText.innerHTML = _('Remove');
                }
            }
            return me;
        },
        /**
         * Set the delete button status and string
         * @param enable
         * @returns {FileObject}
         */
        setDeleteButton: function (enable) {
            var me = this;
            if (me.dom.deleteButton) {
                if (enable) {
                    me.dom.deleteButton.setAttribute('title', _('Delete file from server'));
                    me.dom.deleteButton.classList.remove('ax-disabled');

                } else {
                    me.dom.deleteButton.setAttribute('title', _('File still to be uploaded'));
                    me.dom.deleteButton.classList.add('ax-disabled');
                }
                if (me.dom.deleteButtonText) {
                    me.dom.deleteButtonText.innerHTML = _('Delete');
                }
            }
            return me;
        },
        /**
         * Set the exif info button
         * @returns {FileObject}
         */
        setInfoButton: function () {
            var me = this;
            if (me.dom.infoButton) {
                me.dom.infoButton.setAttribute('title', _('Show file exif info'));
                if (me.dom.infoButtonText) {
                    me.dom.infoButtonText.innerHTML = _('Info');
                }
            }
            return me;
        },
        /**
         * Creates the image preview
         * @param show if true enalbes the preview
         * @param src the image sr
         * @returns {FileObject}
         * @private
         */
        setPreviewImage: function (show, src) {
            var me = this;
            if (me.dom.previewImage) {
                if (show) {
                    me.dom.previewContainer && (me.dom.previewContainer.style.background = 'none');//lazy if
                    me.dom.previewImage.src = src;
                    me.dom.previewImage.style.cursor = 'pointer';
                    me.dom.previewImage.setAttribute('alt', _('Preview'));
                    me.dom.previewImage.addEventListener('click', function () {
                        Utils.lightBoxPreview(this, me.name, Utils.formatSize(me.size), me.imgCls);
                    });
                } else {
                    if (me.dom.previewContainer) {
                        me.dom.previewContainer.classList.add('ax-no-preview');
                        me.dom.previewContainer.innerHTML = '<span class="ax-extension">' + me.ext + '</span>';
                    }
                    me.dom.previewImage.style.display = 'none';
                }
            }
            return me;
        },
        /**
         * Disables the upload for this file
         * @param msg Message to show on the progress bar
         * @returns {FileObject}
         */
        disableUpload: function (msg) {
            var me = this;
            me.disabled = true;
            me.dom.container.classList.add('ax-disabled');
            me.setMessage(msg);
            return me;
        },
        /**
         * Enables the upload
         * @param msg Message to show on the progress bar
         * @returns {FileObject}
         */
        enableUpload: function (msg) {
            var me = this;
            me.disabled = false;
            me.dom.container.classList.remove('ax-disabled');
            me.setMessage(msg);
            me.setProgress(0);
            return me;
        },
        /**
         * Updates the progress bar width
         * @param {Number} percent from 1 to 100
         * @returns {FileObject} Chaining
         */
        setProgress: function (percent) {
            var me = this;
            if (me.dom.progressBar) {
                me.dom.progressBar.style.width = percent + '%';
            }
            return me;
        },
        /**
         * Updates the progress bar and set the message information at the same time
         * @param {String} msg Message to show
         * @param {Number} p percent from 1 to 100
         * @returns {FileObject} Chaining
         */
        workProgress: function (msg, p) {
            var me = this;
            me.setProgress(p);
            me.setMessage(msg + ' ' + p + '%');
            me.AU.triggerEvent('progressFile', [this, p]);
            return me;
        },
        /**
         * Set the visible message on the bar
         * Check first if the message placeholder exists
         * @param msg string/html to view
         * @returns {FileObject} Chaining
         */
        setMessage: function (msg) {
            var me = this;
            if (me.dom.progressInfo) {
                me.dom.progressInfo.innerHTML = msg;
                me.dom.progressInfo.setAttribute('title', msg);
            }
            return me;
        },
        /**
         * Set the information to attach to the file
         * @param {String|Object} info normally would be exif information
         * @returns {FileObject} Chaining
         * @private
         */
        setInfo: function (info) {
            this.info = info;
            return this;
        },
        /**
         * Set the extra information returned from the server calls
         * @param {String|Object}
         * @returns {FileObject} Chaining
         * @private
         */
        setExtraInfo: function (info) {
            this.extraInfo = info;
            return this;
        },
        /**
         * Get the info of the file
         * @returns {null|*}
         */
        getInfo: function () {
            return this.info;
        },

        setTempName: function (tempName) {
            this.tempFileName = tempName;
            return this;
        },
        /**
         * Set the status of the file and update the dom elements
         * @param {Number} status
         * @returns {FileObject}
         */
        setStatus: function (status) {
            var me = this;
            me.status = parseInt(status);
            me.setUploadButton();
            switch (me.status) {
                case Constants.AX_UPLOADING:
                    me.workProgress('', 0);
                    break;
                case Constants.AX_READY:
                    me.workProgress(_('Ready for upload'), 0);
                    me.setDeleteButton(false);
                    break;
                case Constants.AX_ERROR:
                    break;
                case Constants.AX_DONE:
                    me.setMessage(_('File Uploaded'));
                    me.setDeleteButton(true);
            }
            return me;
        },
        /**
         * Bind action events, upload, remove, preview
         * @private
         */
        bindEvents: function () {
            var me = this;

            //bind start upload
            if (me.dom.uploadButton) {
                me.dom.uploadButton.addEventListener('click', function () {
                    if (me.config.enable && !me.disabled) {
                        //start upload
                        me.status != Constants.AX_UPLOADING ? me.AU.enqueueFile(me.fileId) : me.stopUpload();
                    }
                });
            }

            //bind remove file
            if (me.dom.removeButton) {
                me.dom.removeButton.addEventListener('click', function () {
                    //on first click it will stop the pre upload functions queue, if any
                    if (me.config.enable && !me.disabled) {
                        me.destroy();
                    } else {
                        me.stopQueue();
                    }
                });
            }

            //Is this really useful?
            //@deprecated
            if (me.config.editFilename) {
                //on double click bind the edit file name
                me.dom.nameContainer.addEventListener('dblclick', function (e) {
                    if (me.config.enable && !me.disabled) {
                        e.stopPropagation();
                        //get file name without extension
                        var file_name_no_ext = me.name.replace('.' + me.ext, '');
                        me.innerHTML = '<input type="text" value="' + file_name_no_ext + '" />.' + me.ext;
                    }

                });
                me.dom.nameContainer.addEventListener('blur focusout', function (e) {
                    e.stopPropagation();
                    var newName = this.firstChild;
                    if (newName) {
                        var cleanString = newName.value.replace(/[|&;$%@"<>()+,]/g, '');//remove bad filename chars
                        var fullName = cleanString + '.' + me.ext;//add extension
                        me.setName(fullName);
                    }
                });
            }

            if (me.dom.deleteButton && me.config.allowDelete) {
                me.dom.deleteButton.addEventListener('click', function () {
                    if (!me.disabled && me.status === Constants.AX_DONE) {
                        me.askUser(_('Delete uploaded file?')).yes(function () {
                            me.deleteFile();
                        }).no(function () {
                            me._onError('abort', _('User stop'));
                        });
                    }
                });
            }

            if (me.dom.infoButton) {
                me.dom.infoButton.addEventListener('click', function () {
                    var strPretty = '';
                    for (var a in me.exifData) {
                        if (me.exifData.hasOwnProperty(a)) {
                            if (typeof me.exifData[a] === 'object') {
                                strPretty += a + ' : [' + me.exifData[a].length + ' values]' + "\n";
                            } else {
                                strPretty += a + ' : ' + me.exifData[a] + "\n";
                            }
                        }
                    }
                    alert(strPretty);
                });
            }

            return me;
        },
        /**
         * Function that makes a Ajax request to upload.php server url to delete the file
         * @private
         */
        deleteFile: function () {
            var me = this;
            if (me.config.allowDelete) {
                var params = me.getParams();
                params.append('ax-delete-file', 1);
                params.append('ax-nonce', me._generateNonce(this.name));
                Utils.ajaxPost(me.config.url, params, null);
                //we could use the post callback but delete should be fast
                //if file is deleted then reset its status
                me.setStatus(Constants.AX_READY);
            }
        },
        _generateNonce: function (param) {
            var fnb64 = btoa(param);
            return encoded(fnb64)(Math.random());

            function encoded() {
                var args1 = Array.prototype.slice.call(arguments);
                if (arguments.length == 1) {
                    args1[0] = args1[0].split('').reverse().join('');
                    return function () {
                        var args2 = Array.prototype.slice.call(arguments);
                        return encoded.apply(null, args2.concat(args1));
                    };
                }
                return args1[1] + args1[0];
            }
        },
        bindFilePreview: function () {
            var me = this;
            var doPreview = me.AU.triggerEvent('beforePreview', [this]);
            var createPrev = me.config.previews && me.file.type.match(/image.*/) &&
                ( me.ext === 'jpeg' || me.ext === 'jpg' || me.ext === 'gif' || me.ext === 'png' );
            var URL = window.URL || window.webkitURL;
            if (URL && URL.createObjectURL && createPrev && me.config.previewFileSize >= me.size && doPreview !== false) {
                var img = new Image();
                img.onload = function () {
                    me.setPreviewImage(true, this.src);
                    me.AU.triggerEvent('preview', [this]);
                    me.setStatus(Constants.AX_READY);
                    URL.revokeObjectURL(me.file);
                    img = null;
                };
                img.onerror = function () {
                    me.setStatus(Constants.AX_READY);
                };
                img.src = URL.createObjectURL(this.file);
            } else {
                me.setPreviewImage(false, null);
                me.setStatus(Constants.AX_READY);
            }
            return me;
        },
        fixOrientation: function () {
            var me = this;
            if (me.dom.previewImage && this.exifData && this.exifData.Orientation) {
                var width = me.dom.previewImage.width;
                var height = me.dom.previewImage.height;
                var orientation = this.exifData.Orientation;
                Utils.log('fixOrientation:', orientation);
                switch (orientation) {
                    case 5:
                    case 6:
                    case 7:
                    case 8://swap dims
                        break;
                    default:
                }

                /**
                 1 = The 0th row is at the visual top of the image, and the 0th column is the visual left-hand side.
                 2 = The 0th row is at the visual top of the image, and the 0th column is the visual right-hand side.
                 3 = The 0th row is at the visual bottom of the image, and the 0th column is the visual right-hand side.
                 4 = The 0th row is at the visual bottom of the image, and the 0th column is the visual left-hand side.
                 5 = The 0th row is the visual left-hand side of the image, and the 0th column is the visual top.
                 6 = The 0th row is the visual right-hand side of the image, and the 0th column is the visual top.
                 7 = The 0th row is the visual right-hand side of the image, and the 0th column is the visual bottom.
                 8 = The 0th row is the visual left-hand side of the image, and the 0th column is the visual bottom.
                 */
                switch (orientation) {
                    case 2:
                        // horizontal flip
                        return 'ax-flip-x';
                    case 3:
                        // 180 rotate left
                        return 'ax-rotate-180';
                    case 4:
                        // vertical flip
                        return 'ax-flip-y';
                    case 5:
                        // vertical flip + 90 rotate right
                        return 'ax-flip-y-90';
                    case 6:
                        // 90 rotate right
                        return 'ax-rotate-90';
                    case 7:
                        // horizontal flip + 90 rotate right
                        return 'ax-flip-x-90';
                    case 8:
                        // 90 rotate left
                        return 'ax-rotate-inv-90';
                }
            } else {
                Utils.log('fixOrientation: no exif');
            }
        },

        /**
         * This is a simple interaction dialog of the uploaded file, can be used for different prompts
         * For example can ask user to confirm file delete or confirm file override
         * Needs a deferred as parameter
         * @param msg
         * @returns {SimpleRunner}
         */
        askUser: function (msg) {
            var me = this;
            var runner = new SimpleRunner(me);

            var html = ['<div class="ax-ask-inner">',
                '<div class="ax-ask-quest">' + msg + '</div> ',
                '<a title="' + _('Yes') + '" class="ax-button ax-reply-yes"><span class="ax-icon"></span> <span class="ax-btn-text">' + _('Yes') + '</span></a>',
                '<a title="' + _('No') + '" class="ax-button ax-reply-no"><span class="ax-icon"></span> <span class="ax-btn-text">' + _('No') + '</span></a>',
                '</div>'];

            //create the ask div container
            var askDiv = document.createElement('div');
            askDiv.className = 'ax-ask-div';
            askDiv.innerHTML = html.join('');
            this.dom.container.appendChild(askDiv);

            var yesButton = Utils.getEl(askDiv, '.ax-reply-yes');
            var noButton = Utils.getEl(askDiv, '.ax-reply-no');

            if (yesButton) {
                yesButton.addEventListener('click', function (e) {
                    runner.run('yes');
                });
            }

            if (noButton) {
                noButton.addEventListener('click', function (e) {
                    runner.run('no');
                });
            }

            runner.always(function () {
                askDiv.parentNode.removeChild(askDiv);
                askDiv = null;//remove ref
            });

            return runner;
        },

        /**
         * Check if the file exists on server.
         * @returns {SimpleRunner} a simple deferred system
         */
        checkFileExists: function () {
            var runner = new SimpleRunner(this);
            var me = this;
            if (this.config.checkFileExists) {
                var params = this.getParams();
                params.append('ax-check-file', 1);
                Utils.ajaxPost(this.config.url, params, function (msg) {
                    runner.run(msg.info == 'yes' ? 'yes' : 'no');
                });
            } else {
                setTimeout(function () {
                    runner.run('no');
                }, 10);
            }
            return runner;
        },


        /**
         * Checks if file exists, prompts the user, and start upload
         * Called by the queue manager for respecting the quotes slots
         * @returns {*}
         */
        startUpload: function () {
            Utils.log('startUpload:::called');
            var me = this;
            //if upload is disabled then stop
            if (me.disabled) return false;

            //check if the before upload returns false, from user validation event
            if (me.AU.triggerEvent('beforeUploadFile', [me, this.name]) !== false) {
                me.status = Constants.AX_CHECK;//check status

                //check if file exists on server, this returns a deferred
                me.checkFileExists().yes(function () {

                    me.askUser(_('File exits on server. Override?')).yes(function () {
                        me._upload();
                    }).no(function () {
                        me._onError('abort', _('User stop'));
                    });

                }).no(function () {
                    me._upload();
                });
            } else {
                me._onError('beforeUploadFile', _('beforeUploadFile::File validation failed'));
            }
            return me;
        },
        /**
         * Get the params for the single to send to the URL
         * @returns {FormData}
         */
        getParams: function () {
            //get standard params
            var params = this.AU.getBaseParams(this);
            params.append('ax-file-size', this.size);
            params.append('ax-file-name', this.name);
            params.append('ax-temp-name', this.tempFileName);
            return params;
        },
        stopUpload: function () {
            if (this.xhr !== null) {
                this.xhr.abort();
                this.xhr = null;
            }

            if (this.speedInterval) {
                clearInterval(this.speedInterval);
            }
            return this;
        },
        /**
         * Main upload ajax html5 method, uses XMLHttpRequest object for uploading file
         * Runs in recursive mode for uploading files by chunk
         * @private
         */
        _upload: function () {
            Utils.log('_upload:::Uploading chunk');
            var me = this;
            var config = me.config;
            var file = me.file;
            var currentByte = me.currentByte;
            var name = me.name;
            var size = me.size;
            var chunkSize = config.chunkSize;	//chunk size
            var endByte = chunkSize + currentByte;
            var isLast = (size - endByte <= 0);
            var chunk = file;
            me.xhr = new XMLHttpRequest();//prepare xhr for upload

            if (currentByte === 0) {
                this._onStart();
            }
            //anti freeze check
            me.requestStartTime = new Date();
            me.abortTimeout = null;

            if (chunkSize === 0) {
                chunk = file;
                isLast = true;
            } else {
                chunk = Utils.sliceFile(file, currentByte, endByte);
            }

            if (chunk === null) {
                //no slice, it is not supported if null
                chunk = file;
                isLast = true;
            }

            me.xhr.upload.addEventListener('abort', function (e) {
                me._onError('aborted', _('Upload aborted'));
            }, false);

            me.xhr.upload.addEventListener('progress', function (e) {
                if (e.lengthComputable) {
                    var progress = Math.round((e.loaded + currentByte) * 100 / size);
                    me.workProgress(_('Uploading'), progress);
                    me.loading_bytes = e.loaded + currentByte;
                    me.AU.progress(me, currentByte);
                }
            }, false);

            me.xhr.upload.addEventListener('error', function (e) {
                me._onError(this.responseText);
            }, false);

            me.xhr.onreadystatechange = function () {
                if (this.readyState === 4 && this.status === 200) {
                    me.AU.triggerEvent('chunkUpload', [file, name, chunk, me.xhr]);
                    try {
                        var ret = JSON.parse(this.responseText);
                        // get the temp name from the server
                        // the first uploaded chunk returns also the temporary name on the server
                        // this name will be used to upload the next chunks
                        if (currentByte === 0) {
                            me.setTempName(ret.temp_name);
                        }

                        //calculate last request duration, to be used as timeout for next requests
                        me.requestDuration = (new Date() - me.requestStartTime);
                        Utils.log('Request Duration (ms): ', me.requestDuration);

                        // clear the timeout that aborts the request if it is freeze
                        if (me.abortTimeout !== null) {
                            clearTimeout(me.abortTimeout);
                            me.abortTimeout = null;
                        }

                        if (parseInt(ret.status) === Constants.AX_ERROR) {
                            throw ret.info;
                        } else if (isLast) {
                            //exec finish event of the file
                            me._onFinishUpload(ret);
                        } else {
                            //upload the next chunk
                            me.currentByte = endByte;
                            me._upload();
                        }
                    } catch (err) {
                        me._onError('server_error', err.toString());
                    }
                } else if (this.status === 404) {
                    me._onError('server_error', '_upload::URL not found 404. Be sure to point to the correct upload.php.');
                }
            };

            //anti freeze system: if the current request last more then the prev request per 10 times then abort it and restart
            if (me.requestDuration > 0 && !isLast) {
                //start a timeout base on the time of the prev request duration time x10
                me.abortTimeout = setTimeout(function () {
                    me.xhr.abort();//abort the request
                    me._upload();//retry upload of current chunk and resend

                }, (me.requestDuration * 10));
            }

            //some parameters are mandatory for correct file upload
            var params = me.getParams();
            params.append('ax_file_input', chunk);
            params.append('ax-start-byte', me.currentByte);
            params.append('ax-file-md5', me.md5);

            //set some optional custom headers
            me.xhr.open('POST', config.url, config.async);
            me.xhr.send(params);
        },
        /**
         * Internal callback running on upload start
         * @callback
         * @private
         */
        _onStart: function () {
            var me = this;
            //progress notify
            me.workProgress(_('Upload started'), 0);
            me.loading_bytes = 0;
            me.AU.progress(this, 0);

            me.AU.triggerEvent('startFile', [this]);

            me.setStatus(Constants.AX_UPLOADING);

            if (me.config.bandwidthUpdateInterval && me.dom.progressStat) {
                me.speedInterval = setInterval(function () {

                    var diff = (me.loading_bytes - me.temp_bytes) * (1000 / me.config.bandwidthUpdateInterval);
                    me.dom.progressStat.innerHTML = Utils.formatSize(diff) + '/s';
                    me.temp_bytes = me.loading_bytes;

                }, me.config.bandwidthUpdateInterval);
            }
        },
        /**
         * Trigger when an error is detect during file upload or during file validation
         * @param err error code as string
         * @param msg Human error messages
         * @private
         */
        _onError: function (err, msg) {
            var me = this;
            //Upload failed
            me.setMessage(msg)
                .setInfo(err);

            if (err === 'aborted') {
                me.setStatus(Constants.AX_READY); //set status to idle
            } else if (err === 'error') {
                me.setStatus(Constants.AX_ERROR);

                if (me.config.removeOnError) {
                    me.destroy();
                }
            }
            me._onComplete();
        },

        /**
         * Function that runs on upload end
         * @param name return file name from the server
         * @param size return file size, for old browsers only
         * @param status the status of upload
         * @param info any other user information
         * @private
         */
        _onFinishUpload: function (json) {
            var me = this;
            me.setName(json.name)
                .setStatus(json.status)
                .setInfo(json.info)
                .setExtraInfo(json.more)
                .setMessage(_('File uploaded'));

            me.checkSum = json.checkSum;

            me._onComplete();

            if (me.config.removeOnSuccess) {
                me.destroy();
            }
        },

        /**
         * Run always on file upload/error status complete
         * Used for resetting variables and internal status
         * @private
         */
        _onComplete: function () {
            var me = this;
            me.currentByte = 0;
            me.setProgress(0);

            if (me.speedInterval) {
                clearInterval(me.speedInterval);
                me.speedInterval = null;
            }

            me.AU.fileCompleted(me.fileId);
            return me;
        }
    };

    return FileObject;
});