﻿// Copyright (c) Microsoft Open Technologies, Inc. All rights reserved. See License.txt in the project root for license information.

using System.Net.Http;
using System.Web.Http.Routing;
using Microsoft.TestCommon;

namespace System.Web.Http.OData.Routing
{
    public class ODataRouteTest
    {
        [Fact]
        public void GenerateLinkDirectly_ReturnsNull_IfHelperRequestHasNoConfiguration()
        {
            HttpRequestMessage request = new HttpRequestMessage(HttpMethod.Get, "http://localhost/vpath/prefix/Customers");
            ODataRoute odataRoute = new ODataRoute("prefix", pathConstraint: null);

            Assert.Null(odataRoute.GenerateLinkDirectly(request, "odataPath"));
        }

        [Fact]
        public void GenerateLinkDirectly_ReturnsNull_IfRouteTemplateHasParameterInPrefix()
        {
            HttpRequestMessage request = new HttpRequestMessage(HttpMethod.Get, "http://localhost/vpath/prefix/Customers");
            HttpConfiguration config = new HttpConfiguration(new HttpRouteCollection("http://localhost/vpath"));
            request.SetConfiguration(config);
            ODataRoute odataRoute = new ODataRoute("{prefix}", pathConstraint: null);

            Assert.Null(odataRoute.GenerateLinkDirectly(request, "odataPath"));
        }

        [Theory]
        [InlineData("")]
        [InlineData("odataPath")]
        [InlineData("Customers('$&+,/:;=?@ <>#%{}|\\^~[]` ')")]
        public void GetVirtualPath_MatchesHttpRoute(string odataPath)
        {
            HttpRequestMessage request = new HttpRequestMessage(HttpMethod.Get, "http://localhost/vpath/prefix/Customers");
            HttpConfiguration config = new HttpConfiguration(new HttpRouteCollection("http://localhost/vpath"));
            request.SetConfiguration(config);

            IHttpRoute httpRoute = config.Routes.CreateRoute("prefix/{*odataPath}", defaults: null, constraints: null);
            ODataRoute odataRoute = new ODataRoute("prefix", pathConstraint: null);

            // Test that the link generated by ODataRoute matches the one generated by HttpRoute
            Assert.Equal(
                httpRoute.GetVirtualPath(request, new HttpRouteValueDictionary { { "odataPath", odataPath }, { "httproute", true } }).VirtualPath,
                odataRoute.GetVirtualPath(request, new HttpRouteValueDictionary { { "odataPath", odataPath }, { "httproute", true } }).VirtualPath);
        }
    }
}
