import React from 'react'
import { space, SpaceProps } from 'styled-system'
import styled, { css } from 'styled-components'

export type TabProps = SpaceProps & {
  index: number
  children: any
  value?: number
  active?: boolean
  variant?: 'default' | 'contained' | 'card'
  disabled?: boolean
}

interface StyledProps {
  disabled?: boolean
  active?: boolean
  variant?: 'default' | 'contained' | 'card'
}

const tabVariations: { [index: string]: any } = {
  default: css<StyledProps>`
    padding: 8px 21px;
    ${({ active, disabled, theme }) => {
      if (active)
        return css`
          font-weight: ${theme.fontWeights.medium};
          color: ${theme.colors.primary};
          &::before {
            background: ${theme.colors.primary};
            animation: show 0.25s;
            @keyframes show {
              from {
                transform: scale(0);
              }
              to {
                transform: scale(1);
              }
            }
          }
        `
      if (disabled)
        return css`
          cursor: not-allowed !important;
          font-weight: ${theme.fontWeights.regular};
          color: ${theme.colors.grey};
          &::before {
            background: transparent;
          }
        `

      return css`
        font-weight: ${theme.fontWeights.regular};
        color: ${theme.colors.darkGrey};
        &::before {
          background: transparent;
        }
      `
    }}
  `,
  contained: css<StyledProps>`
    padding: 10px 17px;
    ${({ active, theme, disabled }) => {
      if (active) {
        return css`
          font-weight: ${theme.fontWeights.semiBold};
          color: ${theme.colors.black};
          background: ${theme.colors.white};
          border-radius: 5px;
          border: 1px solid ${theme.colors.mediumGrey};
        `
      }
      if (disabled) {
        return css`
          cursor: not-allowed !important;
          font-weight: ${theme.fontWeights.regular};
          color: ${theme.colors.grey};
          &::before {
            background: transparent;
          }
        `
      }

      return css`
        font-weight: ${theme.fontWeights.regular};
        color: ${theme.colors.darkGrey};
      `
    }}
  `,

  card: css<StyledProps>`
    padding: 10px 17px;
    ${({ active, theme, disabled }) => {
      if (active) {
        return css`
          color: ${theme.colors.almostBlack};
          background: ${theme.colors.white};
          border-radius: 5px 5px 0 0;
          border: 1px solid ${theme.colors.mediumGrey};
          border-bottom: 0px;
          color: ${theme.colors.primary};
          font-weight: ${theme.fontWeights.medium};

          &::before {
            background: ${theme.colors.white};
          }
        `
      }

      if (disabled) {
        return css`
          cursor: not-allowed !important;
          font-weight: ${theme.fontWeights.regular};
          color: ${theme.colors.grey};
          &::before {
            background: transparent;
          }
        `
      }

      return css`
        font-weight: ${theme.fontWeights.regular};
        color: ${theme.colors.darkGrey};
      `
    }}
  `
}

const TabStyled = styled.li<StyledProps>`
  ${space}

  position: relative;
  display: flex;
  flex-direction: row;
  align-items: center;
  justify-content: center;
  min-width: max-content;

  &:hover {
    cursor: pointer;
  }

  &::before {
    position: absolute;
    content: '';
    bottom: -1px;
    width: 100%;
    height: 3px;
    border-radius: 3px 3px 0 0;
  }

  ${({ variant }) => tabVariations[variant || 'default']}
`
export const Tab = React.forwardRef<HTMLLIElement, TabProps>(
  ({ children, value = 0, index, disabled, ...props }, ref) => {
    function handleClick(event) {
      event.currentTarget.parentNode.setAttribute('data-id', index)
    }

    return (
      <TabStyled
        data-testid='tab'
        onClick={e => {
          if (disabled) {
            return
          }

          handleClick(e)
        }}
        disabled={disabled}
        active={index === value}
        ref={ref}
        {...props}
      >
        {children}
      </TabStyled>
    )
  }
)

Tab.defaultProps = {
  variant: 'default'
}
