/*
 * Decompiled with CFR 0.152.
 */
package io.flutter.plugins.camera;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.graphics.SurfaceTexture;
import android.hardware.camera2.CameraAccessException;
import android.hardware.camera2.CameraCaptureSession;
import android.hardware.camera2.CameraCharacteristics;
import android.hardware.camera2.CameraDevice;
import android.hardware.camera2.CameraManager;
import android.hardware.camera2.CaptureFailure;
import android.hardware.camera2.CaptureRequest;
import android.hardware.camera2.params.StreamConfigurationMap;
import android.media.CamcorderProfile;
import android.media.Image;
import android.media.ImageReader;
import android.media.MediaRecorder;
import android.os.Build;
import android.util.Size;
import android.view.OrientationEventListener;
import android.view.Surface;
import androidx.annotation.NonNull;
import io.flutter.plugin.common.EventChannel;
import io.flutter.plugin.common.MethodChannel;
import io.flutter.plugins.camera.CameraUtils;
import io.flutter.view.FlutterView;
import io.flutter.view.TextureRegistry;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

public class Camera {
    private final TextureRegistry.SurfaceTextureEntry flutterTexture;
    private final CameraManager cameraManager;
    private final OrientationEventListener orientationEventListener;
    private final boolean isFrontFacing;
    private final int sensorOrientation;
    private final String cameraName;
    private final Size captureSize;
    private final Size previewSize;
    private final boolean enableAudio;
    private CameraDevice cameraDevice;
    private CameraCaptureSession cameraCaptureSession;
    private ImageReader pictureImageReader;
    private ImageReader imageStreamReader;
    private EventChannel.EventSink eventSink;
    private CaptureRequest.Builder captureRequestBuilder;
    private MediaRecorder mediaRecorder;
    private boolean recordingVideo;
    private CamcorderProfile recordingProfile;
    private int currentOrientation = -1;

    public Camera(Activity activity, FlutterView flutterView, String cameraName, String resolutionPreset, boolean enableAudio) throws CameraAccessException {
        if (activity == null) {
            throw new IllegalStateException("No activity available!");
        }
        this.cameraName = cameraName;
        this.enableAudio = enableAudio;
        this.flutterTexture = flutterView.createSurfaceTexture();
        this.cameraManager = (CameraManager)activity.getSystemService("camera");
        this.orientationEventListener = new OrientationEventListener(activity.getApplicationContext()){

            public void onOrientationChanged(int i) {
                if (i == -1) {
                    return;
                }
                Camera.this.currentOrientation = (int)Math.round((double)i / 90.0) * 90;
            }
        };
        this.orientationEventListener.enable();
        CameraCharacteristics characteristics = this.cameraManager.getCameraCharacteristics(cameraName);
        StreamConfigurationMap streamConfigurationMap = (StreamConfigurationMap)characteristics.get(CameraCharacteristics.SCALER_STREAM_CONFIGURATION_MAP);
        this.sensorOrientation = (Integer)characteristics.get(CameraCharacteristics.SENSOR_ORIENTATION);
        this.isFrontFacing = (Integer)characteristics.get(CameraCharacteristics.LENS_FACING) == 0;
        ResolutionPreset preset = ResolutionPreset.valueOf(resolutionPreset);
        this.recordingProfile = CameraUtils.getBestAvailableCamcorderProfileForResolutionPreset(cameraName, preset);
        this.captureSize = new Size(this.recordingProfile.videoFrameWidth, this.recordingProfile.videoFrameHeight);
        this.previewSize = CameraUtils.computeBestPreviewSize(cameraName, preset);
    }

    public void setupCameraEventChannel(EventChannel cameraEventChannel) {
        cameraEventChannel.setStreamHandler(new EventChannel.StreamHandler(){

            public void onListen(Object arguments, EventChannel.EventSink sink) {
                Camera.this.eventSink = sink;
            }

            public void onCancel(Object arguments) {
                Camera.this.eventSink = null;
            }
        });
    }

    private void prepareMediaRecorder(String outputFilePath) throws IOException {
        if (this.mediaRecorder != null) {
            this.mediaRecorder.release();
        }
        this.mediaRecorder = new MediaRecorder();
        if (this.enableAudio) {
            this.mediaRecorder.setAudioSource(1);
        }
        this.mediaRecorder.setVideoSource(2);
        this.mediaRecorder.setOutputFormat(this.recordingProfile.fileFormat);
        if (this.enableAudio) {
            this.mediaRecorder.setAudioEncoder(this.recordingProfile.audioCodec);
        }
        this.mediaRecorder.setVideoEncoder(this.recordingProfile.videoCodec);
        this.mediaRecorder.setVideoEncodingBitRate(this.recordingProfile.videoBitRate);
        if (this.enableAudio) {
            this.mediaRecorder.setAudioSamplingRate(this.recordingProfile.audioSampleRate);
        }
        this.mediaRecorder.setVideoFrameRate(this.recordingProfile.videoFrameRate);
        this.mediaRecorder.setVideoSize(this.recordingProfile.videoFrameWidth, this.recordingProfile.videoFrameHeight);
        this.mediaRecorder.setOutputFile(outputFilePath);
        this.mediaRecorder.setOrientationHint(this.getMediaOrientation());
        this.mediaRecorder.prepare();
    }

    @SuppressLint(value={"MissingPermission"})
    public void open(final @NonNull MethodChannel.Result result) throws CameraAccessException {
        this.pictureImageReader = ImageReader.newInstance((int)this.captureSize.getWidth(), (int)this.captureSize.getHeight(), (int)256, (int)2);
        this.imageStreamReader = ImageReader.newInstance((int)this.previewSize.getWidth(), (int)this.previewSize.getHeight(), (int)35, (int)2);
        this.cameraManager.openCamera(this.cameraName, new CameraDevice.StateCallback(){

            public void onOpened(@NonNull CameraDevice device) {
                Camera.this.cameraDevice = device;
                try {
                    Camera.this.startPreview();
                }
                catch (CameraAccessException e) {
                    result.error("CameraAccess", e.getMessage(), null);
                    Camera.this.close();
                    return;
                }
                HashMap<String, Number> reply = new HashMap<String, Number>();
                reply.put("textureId", Camera.this.flutterTexture.id());
                reply.put("previewWidth", Camera.this.previewSize.getWidth());
                reply.put("previewHeight", Camera.this.previewSize.getHeight());
                result.success(reply);
            }

            public void onClosed(@NonNull CameraDevice camera) {
                Camera.this.sendEvent(EventType.CAMERA_CLOSING);
                super.onClosed(camera);
            }

            public void onDisconnected(@NonNull CameraDevice cameraDevice) {
                Camera.this.close();
                Camera.this.sendEvent(EventType.ERROR, "The camera was disconnected.");
            }

            public void onError(@NonNull CameraDevice cameraDevice, int errorCode) {
                String errorDescription;
                Camera.this.close();
                switch (errorCode) {
                    case 1: {
                        errorDescription = "The camera device is in use already.";
                        break;
                    }
                    case 2: {
                        errorDescription = "Max cameras in use";
                        break;
                    }
                    case 3: {
                        errorDescription = "The camera device could not be opened due to a device policy.";
                        break;
                    }
                    case 4: {
                        errorDescription = "The camera device has encountered a fatal error";
                        break;
                    }
                    case 5: {
                        errorDescription = "The camera service has encountered a fatal error.";
                        break;
                    }
                    default: {
                        errorDescription = "Unknown camera error";
                    }
                }
                Camera.this.sendEvent(EventType.ERROR, errorDescription);
            }
        }, null);
    }

    private void writeToFile(ByteBuffer buffer, File file) throws IOException {
        try (FileOutputStream outputStream = new FileOutputStream(file);){
            while (0 < buffer.remaining()) {
                outputStream.getChannel().write(buffer);
            }
        }
    }

    TextureRegistry.SurfaceTextureEntry getFlutterTexture() {
        return this.flutterTexture;
    }

    public void takePicture(String filePath, final @NonNull MethodChannel.Result result) {
        File file = new File(filePath);
        if (file.exists()) {
            result.error("fileExists", "File at path '" + filePath + "' already exists. Cannot overwrite.", null);
            return;
        }
        this.pictureImageReader.setOnImageAvailableListener(reader -> {
            try (Image image = reader.acquireLatestImage();){
                ByteBuffer buffer = image.getPlanes()[0].getBuffer();
                this.writeToFile(buffer, file);
                result.success(null);
            }
            catch (IOException e) {
                result.error("IOError", "Failed saving image", null);
            }
        }, null);
        try {
            CaptureRequest.Builder captureBuilder = this.cameraDevice.createCaptureRequest(2);
            captureBuilder.addTarget(this.pictureImageReader.getSurface());
            captureBuilder.set(CaptureRequest.JPEG_ORIENTATION, (Object)this.getMediaOrientation());
            this.cameraCaptureSession.capture(captureBuilder.build(), new CameraCaptureSession.CaptureCallback(){

                public void onCaptureFailed(@NonNull CameraCaptureSession session, @NonNull CaptureRequest request, @NonNull CaptureFailure failure) {
                    String reason;
                    switch (failure.getReason()) {
                        case 0: {
                            reason = "An error happened in the framework";
                            break;
                        }
                        case 1: {
                            reason = "The capture has failed due to an abortCaptures() call";
                            break;
                        }
                        default: {
                            reason = "Unknown reason";
                        }
                    }
                    result.error("captureFailure", reason, null);
                }
            }, null);
        }
        catch (CameraAccessException e) {
            result.error("cameraAccess", e.getMessage(), null);
        }
    }

    private void createCaptureSession(int templateType, Surface ... surfaces) throws CameraAccessException {
        this.createCaptureSession(templateType, (Runnable)null, surfaces);
    }

    private void createCaptureSession(int templateType, final Runnable onSuccessCallback, Surface ... surfaces) throws CameraAccessException {
        this.closeCaptureSession();
        this.captureRequestBuilder = this.cameraDevice.createCaptureRequest(templateType);
        SurfaceTexture surfaceTexture = this.flutterTexture.surfaceTexture();
        surfaceTexture.setDefaultBufferSize(this.previewSize.getWidth(), this.previewSize.getHeight());
        Surface flutterSurface = new Surface(surfaceTexture);
        this.captureRequestBuilder.addTarget(flutterSurface);
        List<Surface> remainingSurfaces = Arrays.asList(surfaces);
        if (templateType != 1) {
            for (Surface surface : remainingSurfaces) {
                this.captureRequestBuilder.addTarget(surface);
            }
        }
        CameraCaptureSession.StateCallback callback = new CameraCaptureSession.StateCallback(){

            public void onConfigured(@NonNull CameraCaptureSession session) {
                try {
                    if (Camera.this.cameraDevice == null) {
                        Camera.this.sendEvent(EventType.ERROR, "The camera was closed during configuration.");
                        return;
                    }
                    Camera.this.cameraCaptureSession = session;
                    Camera.this.captureRequestBuilder.set(CaptureRequest.CONTROL_MODE, (Object)1);
                    Camera.this.cameraCaptureSession.setRepeatingRequest(Camera.this.captureRequestBuilder.build(), null, null);
                    if (onSuccessCallback != null) {
                        onSuccessCallback.run();
                    }
                }
                catch (CameraAccessException | IllegalArgumentException | IllegalStateException e) {
                    Camera.this.sendEvent(EventType.ERROR, e.getMessage());
                }
            }

            public void onConfigureFailed(@NonNull CameraCaptureSession cameraCaptureSession) {
                Camera.this.sendEvent(EventType.ERROR, "Failed to configure camera session.");
            }
        };
        ArrayList<Surface> surfaceList = new ArrayList<Surface>();
        surfaceList.add(flutterSurface);
        surfaceList.addAll(remainingSurfaces);
        this.cameraDevice.createCaptureSession(surfaceList, callback, null);
    }

    public void startVideoRecording(String filePath, MethodChannel.Result result) {
        if (new File(filePath).exists()) {
            result.error("fileExists", "File at path '" + filePath + "' already exists.", null);
            return;
        }
        try {
            this.prepareMediaRecorder(filePath);
            this.recordingVideo = true;
            this.createCaptureSession(3, () -> this.mediaRecorder.start(), this.mediaRecorder.getSurface());
            result.success(null);
        }
        catch (CameraAccessException | IOException e) {
            result.error("videoRecordingFailed", e.getMessage(), null);
        }
    }

    public void stopVideoRecording(@NonNull MethodChannel.Result result) {
        if (!this.recordingVideo) {
            result.success(null);
            return;
        }
        try {
            this.recordingVideo = false;
            this.mediaRecorder.stop();
            this.mediaRecorder.reset();
            this.startPreview();
            result.success(null);
        }
        catch (CameraAccessException | IllegalStateException e) {
            result.error("videoRecordingFailed", e.getMessage(), null);
        }
    }

    public void pauseVideoRecording(@NonNull MethodChannel.Result result) {
        if (!this.recordingVideo) {
            result.success(null);
            return;
        }
        try {
            if (Build.VERSION.SDK_INT < 24) {
                result.error("videoRecordingFailed", "pauseVideoRecording requires Android API +24.", null);
                return;
            }
            this.mediaRecorder.pause();
        }
        catch (IllegalStateException e) {
            result.error("videoRecordingFailed", e.getMessage(), null);
            return;
        }
        result.success(null);
    }

    public void resumeVideoRecording(@NonNull MethodChannel.Result result) {
        if (!this.recordingVideo) {
            result.success(null);
            return;
        }
        try {
            if (Build.VERSION.SDK_INT < 24) {
                result.error("videoRecordingFailed", "resumeVideoRecording requires Android API +24.", null);
                return;
            }
            this.mediaRecorder.resume();
        }
        catch (IllegalStateException e) {
            result.error("videoRecordingFailed", e.getMessage(), null);
            return;
        }
        result.success(null);
    }

    public void startPreview() throws CameraAccessException {
        this.createCaptureSession(1, this.pictureImageReader.getSurface());
    }

    public void startPreviewWithImageStream(EventChannel imageStreamChannel) throws CameraAccessException {
        this.createCaptureSession(2, this.imageStreamReader.getSurface());
        imageStreamChannel.setStreamHandler(new EventChannel.StreamHandler(){

            public void onListen(Object o, EventChannel.EventSink imageStreamSink) {
                Camera.this.setImageStreamImageAvailableListener(imageStreamSink);
            }

            public void onCancel(Object o) {
                Camera.this.imageStreamReader.setOnImageAvailableListener(null, null);
            }
        });
    }

    private void setImageStreamImageAvailableListener(EventChannel.EventSink imageStreamSink) {
        this.imageStreamReader.setOnImageAvailableListener(reader -> {
            Image img = reader.acquireLatestImage();
            if (img == null) {
                return;
            }
            ArrayList planes = new ArrayList();
            for (Image.Plane plane : img.getPlanes()) {
                ByteBuffer buffer = plane.getBuffer();
                byte[] bytes = new byte[buffer.remaining()];
                buffer.get(bytes, 0, bytes.length);
                HashMap<String, Object> planeBuffer = new HashMap<String, Object>();
                planeBuffer.put("bytesPerRow", plane.getRowStride());
                planeBuffer.put("bytesPerPixel", plane.getPixelStride());
                planeBuffer.put("bytes", bytes);
                planes.add(planeBuffer);
            }
            HashMap<String, Serializable> imageBuffer = new HashMap<String, Serializable>();
            imageBuffer.put("width", Integer.valueOf(img.getWidth()));
            imageBuffer.put("height", Integer.valueOf(img.getHeight()));
            imageBuffer.put("format", Integer.valueOf(img.getFormat()));
            imageBuffer.put("planes", planes);
            imageStreamSink.success(imageBuffer);
            img.close();
        }, null);
    }

    private void sendEvent(EventType eventType) {
        this.sendEvent(eventType, null);
    }

    private void sendEvent(EventType eventType, String description) {
        if (this.eventSink != null) {
            HashMap<String, String> event = new HashMap<String, String>();
            event.put("eventType", eventType.toString().toLowerCase());
            if (eventType != EventType.ERROR) {
                event.put("errorDescription", description);
            }
            this.eventSink.success(event);
        }
    }

    private void closeCaptureSession() {
        if (this.cameraCaptureSession != null) {
            this.cameraCaptureSession.close();
            this.cameraCaptureSession = null;
        }
    }

    public void close() {
        this.closeCaptureSession();
        if (this.cameraDevice != null) {
            this.cameraDevice.close();
            this.cameraDevice = null;
        }
        if (this.pictureImageReader != null) {
            this.pictureImageReader.close();
            this.pictureImageReader = null;
        }
        if (this.imageStreamReader != null) {
            this.imageStreamReader.close();
            this.imageStreamReader = null;
        }
        if (this.mediaRecorder != null) {
            this.mediaRecorder.reset();
            this.mediaRecorder.release();
            this.mediaRecorder = null;
        }
    }

    public void dispose() {
        this.close();
        this.flutterTexture.release();
        this.orientationEventListener.disable();
    }

    private int getMediaOrientation() {
        int sensorOrientationOffset = this.currentOrientation == -1 ? 0 : (this.isFrontFacing ? -this.currentOrientation : this.currentOrientation);
        return (sensorOrientationOffset + this.sensorOrientation + 360) % 360;
    }

    private static enum EventType {
        ERROR,
        CAMERA_CLOSING;

    }

    public static enum ResolutionPreset {
        low,
        medium,
        high,
        veryHigh,
        ultraHigh,
        max;

    }
}

