{ *********************************************************************************** }
{ *                              CryptoLib Library                                  * }
{ *                Copyright (c) 2018 - 20XX Ugochukwu Mmaduekwe                    * }
{ *                 Github Repository <https://github.com/Xor-el>                   * }

{ *  Distributed under the MIT software license, see the accompanying file LICENSE  * }
{ *          or visit http://www.opensource.org/licenses/mit-license.php.           * }

{ *                              Acknowledgements:                                  * }
{ *                                                                                 * }
{ *      Thanks to Sphere 10 Software (http://www.sphere10.com/) for sponsoring     * }
{ *                           development of this library                           * }

{ * ******************************************************************************* * }

(* &&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&&& *)

unit XSalsa20Tests;

interface

{$IFDEF FPC}
{$MODE DELPHI}
{$ENDIF FPC}

uses
  SysUtils,
{$IFDEF FPC}
  fpcunit,
  testregistry,
{$ELSE}
  TestFramework,
{$ENDIF FPC}
  ClpXSalsa20Engine,
  ClpIXSalsa20Engine,
  ClpKeyParameter,
  ClpIKeyParameter,
  ClpParametersWithIV,
  ClpIParametersWithIV,
  ClpCryptoLibTypes,
  CryptoLibTestBase;

type

  TTestXSalsa20 = class(TCryptoLibAlgorithmTestCase)

  private
  var
    FTestVectors: TCryptoLibGenericArray<TCryptoLibStringArray>;

    procedure DoXSalsa20Test(number: Int32;
      const ATestVectorParams: TCryptoLibStringArray);

  protected
    procedure SetUp; override;
    procedure TearDown; override;
  published

    procedure TestXSalsa20Test();

  end;

implementation

{ TTestXSalsa20 }

procedure TTestXSalsa20.DoXSalsa20Test(number: Int32;
  const ATestVectorParams: TCryptoLibStringArray);
var
  engine: IXSalsa20Engine;
var
  LPlainText, LCipherText, LOutput, LKey, LIV: TBytes;
begin
  LPlainText := DecodeHex(ATestVectorParams[2]);
  System.SetLength(LOutput, System.Length(LPlainText));
  LKey := DecodeHex(ATestVectorParams[0]);
  LIV := DecodeHex(ATestVectorParams[1]);
  LCipherText := DecodeHex(ATestVectorParams[3]);

  engine := TXSalsa20Engine.Create();
  engine.Init(false, TParametersWithIV.Create(TKeyParameter.Create(LKey)
    as IKeyParameter, LIV) as IParametersWithIV);

  engine.ProcessBytes(LPlainText, 0, System.Length(LPlainText), LOutput, 0);

  if not(AreEqual(LCipherText, LOutput)) then
  begin
    Fail(Format('Mismatch on %d, Expected %s but found %s',
      [number, EncodeHex(LCipherText), EncodeHex(LOutput)]));
  end;

end;

procedure TTestXSalsa20.SetUp;
begin
  inherited;
  // Test cases generated by naclcrypto-20090308, as used by cryptopp
  FTestVectors := TCryptoLibGenericArray<TCryptoLibStringArray>.Create
    (TCryptoLibStringArray.Create
    ('A6A7251C1E72916D11C2CB214D3C252539121D8E234E652D651FA4C8CFF88030',
    '9E645A74E9E0A60D8243ACD9177AB51A1BEB8D5A2F5D700C',
    '093C5E5585579625337BD3AB619D615760D8C5B224A85B1D0EFE0EB8A7EE163ABB0376529FCC09BAB506C6'
    + '18E13CE777D82C3AE9D1A6F972D4160287CBFE60BF2130FC0A6FF6049D0A5C8A82F429231F008082E845D7E189D37F9ED2B464E6B9'
    + '19E6523A8C1210BD52A02A4C3FE406D3085F5068D1909EEECA6369ABC981A42E87FE665583F0AB85AE71F6F84F528E6B397AF86F6917D9754B7320DBDC2FEA81496F2732F532AC78C4E9C6CFB18F8E9BDF74622EB126141416776971A84F94D156BEAF67AECBF2AD412E76E66E'
    + '8FAD7633F5B6D7F3D64B5C6C69CE29003C6024465AE3B89BE78E915D88B4B5621D',
    'B2AF688E7D8FC4B508C05CC39DD583D6714322C64D7F3E63147AEDE2D9534934B04FF6F337B031815CD094BDBC6D7A92077DCE709412286822EF0737EE47F6B7FF'
    + 'A22F9D53F11DD2B0A3BB9FC01D9A88F9D53C26E9365C2C3C063BC4840BFC812E4B80463E69D179530B25C158F543191CFF993106511AA036043BBC75866AB7E34AFC57E2CCE4934A5FAAE6EABE4F2217'
    + '70183DD060467827C27A354159A081275A291F69D946D6FE28ED0B9CE08206CF484925A51B9498DBDE178DDD3AE91A8581B91682D860F840782F6EEA49DBB9BD721'
    + '501D2C67122DEA3B7283848C5F13E0C0DE876BD227A856E4DE593A3'),

    TCryptoLibStringArray.Create
    ('D5C7F6797B7E7E9C1D7FD2610B2ABF2BC5A7885FB3FF78092FB3ABE8986D35E2',
    '744E17312B27969D826444640E9C4A378AE334F185369C95',
    '7758298C628EB3A4B6963C5445EF66971222BE5D1A4AD839715D1188071739B77CC6E05D5410F963A64167629757',
    '27B8CFE81416A76301FD1EEC6A4D99675069B2DA2776C360DB1BDFEA7C0AA613913E10F7A60FEC04D11E65F2D64E'),

    TCryptoLibStringArray.Create
    ('6799D76E5FFB5B4920BC2768BAFD3F8C16554E65EFCF9A16F4683A7A06927C11',
    '61AB951921E54FF06D9B77F313A4E49DF7A057D5FD627989', '472766', '8FD7DF'),

    TCryptoLibStringArray.Create
    ('AD1A5C47688874E6663A0F3FA16FA7EFB7ECADC175C468E5432914BDB480FFC6',
    'E489EED440F1AAE1FAC8FB7A9825635454F8F8F1F52E2FCC',
    'AA6C1E53580F03A9ABB73BFDADEDFECADA4C6B0EBE020EF10DB745E54BA861CAF65F0E40DFC520203BB54D29E0A8F78F16B3F1AA525D6BFA33C54726E59988CFBEC78056',
    '02FE84CE81E178E7AABDD3BA925A766C3C24756EEFAE33942AF75E8B464556B5997E616F3F2DFC7FCE91848AFD79912D9FB55201B5813A5A074D2C0D4292C1FD441807C5')
    );
end;

procedure TTestXSalsa20.TearDown;
begin
  inherited;

end;

procedure TTestXSalsa20.TestXSalsa20Test;
var
  LIdx: Int32;
begin
  for LIdx := System.Low(FTestVectors) to System.High(FTestVectors) do
  begin
    DoXSalsa20Test(LIdx, FTestVectors[LIdx]);
  end;
end;

initialization

// Register any test cases with the test runner

{$IFDEF FPC}
  RegisterTest(TTestXSalsa20);
{$ELSE}
  RegisterTest(TTestXSalsa20.Suite);
{$ENDIF FPC}

end.
