"""Service impl"""

from typing import List
from loguru import logger

from langchain_core.documents import Document
from langchain_text_splitters import RecursiveCharacterTextSplitter

from component.data_loader.firecrawl import FireCrawlLoader
from component.model_runtime.model_engine.base.llm_model import LargeLanguageModel
from component.model_runtime.model_engine.factory import (
    get_embeddings_model,
    get_llm_model,
)
from component.model_runtime.schema.chat import ChatRequest
from component.prompt.prompt import (
    PREDEFINED_RESPONSE,
    REPHRASE_TEMPLATE,
    RESPONSE_TEMPLATE,
)
from component.vector_store.pg_vector import search_documents, add_documents
from server.data_object.history import ChatHistoryDO
from server.mapper.chat_history import (
    add_chat_history,
    search_chat_history,
    clear_all_chat_history,
)
from server.service.service import Service


class ServiceImpl(Service):
    async def embed_documents(self, texts: List[str]) -> List[List[float]]:
        """
        Embed a list of documents.

        :param texts: A list of document texts.
        :return: A list of embeddings for each document.
        """
        logger.info(f"Word count {str(texts)}")
        textEmbeddingModel = await get_embeddings_model()
        return textEmbeddingModel.embed_documents(texts)

    async def rephrase(self, request: ChatRequest, llm: LargeLanguageModel) -> str:
        """
        Rephrase a given question based on the chat history using a language model_runtime.

        Args:
            request (ChatRequest): Request object containing the chat history and the question to rephrase.
            llm: LargeLanguageModel to chat

        Returns:
            str: The rephrased question generated by the language model_runtime.
        """
        chat_history_list = []
        chat_history_str = ""
        chat_history = request.chat_history
        limit = request.limit
        if chat_history is None or len(chat_history) == 0:
            if limit > 0:
                chat_history: List[ChatHistoryDO] = await search_chat_history(limit)
                question_list = []
                for chat_record in chat_history:
                    question_list.append(chat_record.query_history)
                chat_history_list = question_list
        else:
            chat_history_list = chat_history[:limit]
        if len(chat_history_list) > 0:
            chat_history_str = "\n".join(chat_history_list)
        else:
            return chat_history_str
        rephrase_template = REPHRASE_TEMPLATE.format(chat_history_str, request.question)
        return llm.generate(question=request.question, content=rephrase_template)

    async def chat(self, request: ChatRequest):
        """
        Large model responds to user's dialogue
        :param request: A chat request for a conversational agent
        :return: Large model responds
        """
        question = request.question
        logger.info(f"Question url {question}")
        chatHistoryDO = ChatHistoryDO(query_history=question)
        await add_chat_history(chatHistoryDO)
        docs: List[Document] = await search_documents(question)
        if len(docs) == 0 and request.strict_model:
            return PREDEFINED_RESPONSE
        llm = await get_llm_model()
        rephrase_question = await self.rephrase(request=request, llm=llm)
        if len(rephrase_question) == 0:
            rephrase_question = question
        if len(docs) == 0:
            return llm.chat(question=rephrase_question, stream=request.stream)
        page_content_list = []
        for doc in docs:
            page, meta = doc
            page_content_list.append(page.page_content)
        relate_doc_str = "\n".join(page_content_list)
        response_template = RESPONSE_TEMPLATE.format(relate_doc_str)
        return llm.chat(
            question=rephrase_question, content=response_template, stream=request.stream
        )

    async def crawl_document(self, url: str):
        """
        Crawling web page content from a URL and coexisting it in a vector database

        Args:
            url (str): A url starts with http(https)

        Returns:
            dict: A success response containing the added id in vector store.
        """
        logger.info(f"Crawl url {url}")
        if (not url.startswith("http")) and (not url.startswith("https")):
            raise
        loader = FireCrawlLoader(
            api_key="no need key for local deployment", url=url, mode="crawl"
        )
        crawled_docs = loader.load()
        text_splitter = RecursiveCharacterTextSplitter(
            chunk_size=2000, chunk_overlap=200
        )
        split_doc = text_splitter.split_documents(crawled_docs)
        return await add_documents(split_doc)

    async def clear_history(self) -> None:
        """
        Clear all chat history
        """
        await clear_all_chat_history()


async def get_service():
    return ServiceImpl()
