using System;
using AsmResolver.PE.DotNet.Metadata;
using Xunit;

namespace AsmResolver.PE.Tests.DotNet.Metadata
{
    public class GuidStreamTest
    {
        private readonly GuidStream _guidStream;

        public GuidStreamTest()
        {
            _guidStream = new SerializedGuidStream(GuidStream.DefaultName, new byte[]
            {
                0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
                0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F,
                0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F
            });
        }

        [Fact]
        public void IndexZeroGivesZeroGuid()
        {
            Assert.Equal(Guid.Empty, _guidStream.GetGuidByIndex(0));
        }

        [Fact]
        public void IndexOneGivesFirstGuidInArray()
        {
            Assert.Equal(new Guid(new byte[]
            {
                0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
            }), _guidStream.GetGuidByIndex(1));
        }

        [Fact]
        public void IndexTwoGivesSecondGuidInArray()
        {
            Assert.Equal(new Guid(new byte[]
            {
                0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
            }), _guidStream.GetGuidByIndex(2));
        }

        [Fact]
        public void IndexThreeGivesThirdGuidInArray()
        {
            Assert.Equal(new Guid(new byte[]
            {
                0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F,
            }), _guidStream.GetGuidByIndex(3));
        }

        private void AssertDoesNotHaveGuid(Guid needle)
        {
            Assert.False(_guidStream.TryFindGuidIndex(needle, out _));
        }

        private void AssertHasGuid(Guid needle)
        {
            Assert.True(_guidStream.TryFindGuidIndex(needle, out uint actualIndex));
            Assert.Equal(needle, _guidStream.GetGuidByIndex(actualIndex));
        }

        [Theory]
        [InlineData(new byte[]
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F})]
        [InlineData(new byte[]
            {0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F})]
        [InlineData(new byte[]
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F})]
        public void FindExistingGuid(byte[] guidData) => AssertHasGuid(new Guid(guidData));

        [Theory]
        [InlineData(new byte[]
            {0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17})]
        [InlineData(new byte[]
            {0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00})]
        public void FindNonExistingGuid(byte[] guidData) => AssertDoesNotHaveGuid(new Guid(guidData));
    }
}
