using AsmResolver.IO;

namespace AsmResolver.PE.File
{
    /// <summary>
    /// Represents a data directory header, consisting of the starting address, and the size.
    /// </summary>
    /// <remarks>
    /// This structure does not contain the actual contents of the data directory.
    /// </remarks>
    public readonly struct DataDirectory : IWritable
    {
        /// <summary>
        /// Indicates the size of a single data directory header.
        /// </summary>
        public const uint DataDirectorySize = 2 * sizeof(uint);

        /// <summary>
        /// Reads a single data directory at the current position of the provided input stream.
        /// </summary>
        /// <param name="reader">The input stream to read from.</param>
        /// <returns>The data directory that was read.</returns>
        public static DataDirectory FromReader(ref BinaryStreamReader reader)
        {
            return new(reader.ReadUInt32(), reader.ReadUInt32());
        }

        /// <summary>
        /// Creates a new data directory header.
        /// </summary>
        /// <param name="virtualAddress">The starting virtual address (RVA) of the directory.</param>
        /// <param name="size">The size in bytes of the directory.</param>
        public DataDirectory(uint virtualAddress, uint size)
        {
            VirtualAddress = virtualAddress;
            Size = size;
        }

        /// <summary>
        /// Gets the relative virtual address (RVA) of the directory.
        /// </summary>
        public uint VirtualAddress
        {
            get;
        }

        /// <summary>
        /// Gets the size of the directory.
        /// </summary>
        public uint Size
        {
            get;
        }

        /// <summary>
        /// Gets a value indicating the data directory is present in the portable executable file.
        /// </summary>
        public bool IsPresentInPE => VirtualAddress != 0 || Size != 0;

        /// <summary>
        /// Creates a data directory descriptor for the provided segment.
        /// </summary>
        /// <param name="segment">The segment to create a data directory descriptor for.</param>
        /// <returns>The data directory.</returns>
        public static DataDirectory CreateForSegment(ISegment? segment)
        {
            return segment is not null
                ? new DataDirectory(segment.Rva, segment.GetPhysicalSize())
                : default;
        }

        /// <inheritdoc />
        public uint GetPhysicalSize() => DataDirectorySize;

        /// <inheritdoc />
        public void Write(BinaryStreamWriter writer)
        {
            writer.WriteUInt32(VirtualAddress);
            writer.WriteUInt32(Size);
        }

        /// <summary>
        /// Deconstructs the data directory into a 2-tuple.
        /// </summary>
        /// <param name="virtualAddress">The virtual address.</param>
        /// <param name="size">The size.</param>
        public void Deconstruct(out uint virtualAddress, out uint size)
        {
            virtualAddress = VirtualAddress;
            size = Size;
        }

        /// <inheritdoc />
        public override string ToString() => $"RVA: 0x{VirtualAddress:X8}, Size: 0x{Size:X8}";
    }
}
