using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using AsmResolver.Shims;

namespace AsmResolver.DotNet.Signatures.Parsing
{
    /// <summary>
    /// Provides a mechanism for parsing a fully assembly qualified name of a type.
    /// </summary>
    public struct TypeNameParser
    {
        // src/coreclr/src/vm/typeparse.cpp
        // https://docs.microsoft.com/en-us/dotnet/framework/reflection-and-codedom/specifying-fully-qualified-type-names

        private static readonly SignatureComparer Comparer = new();
        private readonly ModuleDefinition _module;
        private TypeNameLexer _lexer;

        private TypeNameParser(ModuleDefinition module, TypeNameLexer lexer)
        {
            _module = module ?? throw new ArgumentNullException(nameof(module));
            _lexer = lexer;
        }

        /// <summary>
        /// Parses a single fully assembly qualified name.
        /// </summary>
        /// <param name="module">The module containing the assembly qualified name.</param>
        /// <param name="canonicalName">The fully qualified assembly name of the type.</param>
        /// <returns>The parsed type.</returns>
        public static TypeSignature Parse(ModuleDefinition module, string canonicalName)
        {
            var lexer = new TypeNameLexer(new StringReader(canonicalName));
            var parser = new TypeNameParser(module, lexer);
            return parser.ParseTypeSpec();
        }

        private TypeSignature ParseTypeSpec()
        {
            bool lastHasConsumedTypeName = _lexer.HasConsumedTypeName;

            // Parse type signature.
            _lexer.HasConsumedTypeName = false;
            var typeSpec = ParseSimpleTypeSpec();
            _lexer.HasConsumedTypeName = true;

            // See if the type full name contains an assembly ref.
            var scope = TryExpect(TypeNameTerminal.Comma).HasValue
                ? (IResolutionScope) ParseAssemblyNameSpec()
                : null;

            _lexer.HasConsumedTypeName = lastHasConsumedTypeName;

            if (typeSpec.GetUnderlyingTypeDefOrRef() is TypeReference reference)
                SetScope(reference, scope);

            // Ensure corlib type sigs are used.
            if (Comparer.Equals(typeSpec.Scope, _module.CorLibTypeFactory.CorLibScope))
            {
                var corlibType = _module.CorLibTypeFactory.FromType(typeSpec);
                if (corlibType != null)
                    return corlibType;
            }

            return typeSpec;
        }

        private void SetScope(TypeReference reference, IResolutionScope? newScope)
        {
            // Find the top-most type.
            while (reference.Scope is TypeReference declaringType)
                reference = declaringType;

            // If the scope is null, it means it was omitted from the fully qualified type name.
            // In this case, the CLR first looks into the current assembly, and then into corlib.
            if (newScope is not null)
            {
                // Update scope.
                reference.Scope = newScope;
            }
            else
            {
                // First look into the current module.
                reference.Scope = _module;
                var definition = reference.Resolve();
                if (definition is null)
                {
                    // If that fails, try corlib.
                    reference.Scope = _module.CorLibTypeFactory.CorLibScope;
                    definition = reference.Resolve();

                    // If both lookups fail, revert to the normal module as scope as a fallback.
                    if (definition is null)
                        reference.Scope = _module;
                }
            }
        }

        private TypeSignature ParseSimpleTypeSpec()
        {
            // Parse type name.
            var typeName = ParseTypeName();

            // Check for annotations.
            while (true)
            {
                switch (_lexer.Peek().Terminal)
                {
                    case TypeNameTerminal.Ampersand:
                        typeName = ParseByReferenceTypeSpec(typeName);
                        break;

                    case TypeNameTerminal.Star:
                        typeName = ParsePointerTypeSpec(typeName);
                        break;

                    case TypeNameTerminal.OpenBracket:
                        typeName = ParseArrayOrGenericTypeSpec(typeName);
                        break;

                    default:
                        return typeName;
                }
            }
        }

        private TypeSignature ParseByReferenceTypeSpec(TypeSignature typeName)
        {
            Expect(TypeNameTerminal.Ampersand);
            return new ByReferenceTypeSignature(typeName);
        }

        private TypeSignature ParsePointerTypeSpec(TypeSignature typeName)
        {
            Expect(TypeNameTerminal.Star);
            return new PointerTypeSignature(typeName);
        }

        private TypeSignature ParseArrayOrGenericTypeSpec(TypeSignature typeName)
        {
            Expect(TypeNameTerminal.OpenBracket);

            switch (_lexer.Peek().Terminal)
            {
                case TypeNameTerminal.OpenBracket:
                case TypeNameTerminal.Identifier:
                    return ParseGenericTypeSpec(typeName);

                default:
                    return ParseArrayTypeSpec(typeName);
            }
        }

        private TypeSignature ParseArrayTypeSpec(TypeSignature typeName)
        {
            var dimensions = new List<ArrayDimension>
            {
                ParseArrayDimension()
            };

            bool stop = false;
            while (!stop)
            {
                var nextToken = Expect(TypeNameTerminal.CloseBracket, TypeNameTerminal.Comma);
                switch (nextToken.Terminal)
                {
                    case TypeNameTerminal.CloseBracket:
                        stop = true;
                        break;
                    case TypeNameTerminal.Comma:
                        dimensions.Add(ParseArrayDimension());
                        break;
                }
            }

            if (dimensions.Count == 1 && dimensions[0].Size == null && dimensions[0].LowerBound == null)
                return new SzArrayTypeSignature(typeName);

            var result = new ArrayTypeSignature(typeName);
            foreach (var dimension in dimensions)
                result.Dimensions.Add(dimension);
            return result;
        }

        private ArrayDimension ParseArrayDimension()
        {
            switch (_lexer.Peek().Terminal)
            {
                case TypeNameTerminal.CloseBracket:
                case TypeNameTerminal.Comma:
                    return new ArrayDimension();

                case TypeNameTerminal.Number:
                    int? size = null;
                    int? lowerBound = null;

                    int firstNumber = int.Parse(_lexer.Next().Text);
                    var dots = TryExpect(TypeNameTerminal.Ellipsis, TypeNameTerminal.DoubleDot);
                    if (dots.HasValue)
                    {
                        var secondNumberToken = TryExpect(TypeNameTerminal.Number);
                        if (secondNumberToken.HasValue)
                        {
                            int secondNumber = int.Parse(secondNumberToken.Value.Text);
                            size = secondNumber - firstNumber;
                            lowerBound = firstNumber;
                        }
                        else
                        {
                            lowerBound = firstNumber;
                        }
                    }

                    return new ArrayDimension(size, lowerBound);

                default:
                    // Fail intentionally:
                    Expect(TypeNameTerminal.CloseBracket, TypeNameTerminal.Comma, TypeNameTerminal.Number);
                    return new ArrayDimension();
            }
        }

        private TypeSignature ParseGenericTypeSpec(TypeSignature typeName)
        {
            var result = new GenericInstanceTypeSignature(typeName.ToTypeDefOrRef(), typeName.IsValueType);

            result.TypeArguments.Add(ParseGenericTypeArgument(result));

            bool stop = false;
            while (!stop)
            {
                var nextToken = Expect(TypeNameTerminal.CloseBracket, TypeNameTerminal.Comma);
                switch (nextToken.Terminal)
                {
                    case TypeNameTerminal.CloseBracket:
                        stop = true;
                        break;
                    case TypeNameTerminal.Comma:
                        result.TypeArguments.Add(ParseGenericTypeArgument(result));
                        break;
                }
            }

            return result;
        }

        private TypeSignature ParseGenericTypeArgument(GenericInstanceTypeSignature genericInstance)
        {
            var extraBracketToken = TryExpect(TypeNameTerminal.OpenBracket);
            var result = !extraBracketToken.HasValue
                ? ParseSimpleTypeSpec()
                : ParseTypeSpec();
            if (extraBracketToken.HasValue)
                Expect(TypeNameTerminal.CloseBracket);
            return result;
        }

        private TypeSignature ParseTypeName()
        {
            // Note: This is a slight deviation from grammar (but is equivalent), to make the parsing easier.
            //       We read all components
            (string? ns, var names) = ParseNamespaceTypeName();

            TypeReference? result = null;
            for (int i = 0; i < names.Count; i++)
            {
                result = result is null
                    ? new TypeReference(_module, _module, ns, names[i])
                    : new TypeReference(_module, result, null, names[i]);
            }

            if (result is null)
                throw new FormatException();

            return result.ToTypeSignature();
        }

        private (string? Namespace, IList<string> TypeNames) ParseNamespaceTypeName()
        {
            var names = ParseDottedExpression(TypeNameTerminal.Identifier);

            // The namespace is every name concatenated except for the last one.
            string? ns;
            if (names.Count > 1)
            {
                ns = StringShim.Join(".", names.Take(names.Count - 1));
                names.RemoveRange(0, names.Count - 1);
            }
            else
            {
                ns = null;
            }

            // Check if we have any nested identifiers.
            while (TryExpect(TypeNameTerminal.Plus).HasValue)
            {
                var nextIdentifier = Expect(TypeNameTerminal.Identifier);
                names.Add(nextIdentifier.Text);
            }

            return (ns, names);
        }

        private List<string> ParseDottedExpression(TypeNameTerminal terminal)
        {
            var result = new List<string>();

            while (true)
            {
                var nextIdentifier = TryExpect(terminal);
                if (!nextIdentifier.HasValue)
                    break;

                result.Add(nextIdentifier.Value.Text);

                if (!TryExpect(TypeNameTerminal.Dot).HasValue)
                    break;
            }

            if (result.Count == 0)
                throw new FormatException($"Expected {terminal}.");

            return result;
        }

        private AssemblyReference ParseAssemblyNameSpec()
        {
            string assemblyName = StringShim.Join(".", ParseDottedExpression(TypeNameTerminal.Identifier));
            var newReference = new AssemblyReference(assemblyName, new Version());

            while (TryExpect(TypeNameTerminal.Comma).HasValue)
            {
                string propertyName = Expect(TypeNameTerminal.Identifier).Text;
                Expect(TypeNameTerminal.Equals);
                if (propertyName.Equals("version", StringComparison.OrdinalIgnoreCase))
                {
                    newReference.Version = ParseVersion();
                }
                else if (propertyName.Equals("publickey", StringComparison.OrdinalIgnoreCase))
                {
                    newReference.PublicKeyOrToken = ParseHexBlob();
                    newReference.HasPublicKey = true;
                }
                else if (propertyName.Equals("publickeytoken", StringComparison.OrdinalIgnoreCase))
                {
                    newReference.PublicKeyOrToken = ParseHexBlob();
                    newReference.HasPublicKey = false;
                }
                else if (propertyName.Equals("culture", StringComparison.OrdinalIgnoreCase))
                {
                    string culture = ParseCulture();
                    newReference.Culture = !culture.Equals("neutral", StringComparison.OrdinalIgnoreCase)
                        ? culture
                        : null;
                }
                else
                {
                    throw new FormatException($"Unsupported {propertyName} assembly property.");
                }
            }

            // Reuse imported assembly reference instance if possible.
            for (int i = 0; i < _module.AssemblyReferences.Count; i++)
            {
                var existingReference = _module.AssemblyReferences[i];
                if (Comparer.Equals((AssemblyDescriptor) existingReference, newReference))
                    return existingReference;
            }

            return newReference;
        }

        private Version ParseVersion()
        {
            string versionString = StringShim.Join(".", ParseDottedExpression(TypeNameTerminal.Number));
            return VersionShim.Parse(versionString);
        }

        private byte[]? ParseHexBlob()
        {
            string hexString = Expect(TypeNameTerminal.Identifier, TypeNameTerminal.Number).Text;
            if (hexString == "null")
                return null;
            if (hexString.Length % 2 != 0)
                throw new FormatException("Provided hex string does not have an even length.");

            byte[] result = new byte[hexString.Length / 2];
            for (int i = 0; i < hexString.Length; i += 2)
                result[i / 2] = ParseHexByte(hexString, i);
            return result;
        }

        private static byte ParseHexByte(string hexString, int index)
        {
            return (byte) ((ParseHexNibble(hexString[index]) << 4) | ParseHexNibble(hexString[index + 1]));
        }

        private static byte ParseHexNibble(char nibble) => nibble switch
        {
            >= '0' and <= '9' => (byte) (nibble - '0'),
            >= 'A' and <= 'F' => (byte) (nibble - 'A' + 10),
            >= 'a' and <= 'f' => (byte) (nibble - 'a' + 10),
            _ => throw new FormatException()
        };

        private string ParseCulture()
        {
            return Expect(TypeNameTerminal.Identifier).Text;
        }

        private TypeNameToken Expect(TypeNameTerminal terminal)
        {
            return TryExpect(terminal)
                   ?? throw new FormatException($"Expected {terminal}.");
        }

        private TypeNameToken? TryExpect(TypeNameTerminal terminal)
        {
            var token = _lexer.Peek();

            if (terminal != token.Terminal)
                return null;

            _lexer.Next();
            return token;
        }

        private TypeNameToken Expect(params TypeNameTerminal[] terminals)
        {
            return TryExpect(terminals)
                ?? throw new FormatException(
                    $"Expected one of {StringShim.Join(", ", terminals.Select(x => x.ToString()))}.");
        }

        private TypeNameToken? TryExpect(params TypeNameTerminal[] terminals)
        {
            var token = _lexer.Peek();

            if (!terminals.Contains(token.Terminal))
                return null;

            _lexer.Next();
            return token;
        }
    }
}
