using System.Collections.Generic;
using System.Runtime.CompilerServices;

namespace AsmResolver.DotNet
{
    /// <summary>
    /// Provides extensions for various metadata members.
    /// </summary>
    public static class HasCustomAttributeExtensions
    {
        /// <summary>
        /// Determines whether a metadata member is assigned an attributes that match a particular namespace and name.
        /// </summary>
        /// <param name="self">The metadata member.</param>
        /// <param name="ns">The namespace of the attribute type.</param>
        /// <param name="name">The name of the attribute type.</param>
        /// <returns>The matching attributes.</returns>
        public static bool HasCustomAttribute(this IHasCustomAttribute self, string? ns, string? name)
        {
            // Note: we are not using the FindCustomAttributes to avoid allocations.

            for (int i = 0; i < self.CustomAttributes.Count; i++)
            {
                var attribute = self.CustomAttributes[i];
                var declaringType = attribute.Constructor?.DeclaringType;
                if (declaringType is null)
                    continue;

                if (declaringType.IsTypeOf(ns, name))
                    return true;
            }

            return false;
        }

        /// <summary>
        /// Finds all custom attributes that were assigned to a metadata member that match a particular namespace and name.
        /// </summary>
        /// <param name="self">The metadata member.</param>
        /// <param name="ns">The namespace of the attribute type.</param>
        /// <param name="name">The name of the attribute type.</param>
        /// <returns>The matching attributes.</returns>
        public static IEnumerable<CustomAttribute> FindCustomAttributes(this IHasCustomAttribute self, string? ns, string? name)
        {
            for (int i = 0; i < self.CustomAttributes.Count; i++)
            {
                var attribute = self.CustomAttributes[i];
                var declaringType = attribute.Constructor?.DeclaringType;
                if (declaringType is null)
                    continue;

                if (declaringType.IsTypeOf(ns, name))
                    yield return attribute;
            }
        }

        /// <summary>
        /// Indicates whether the specified member is compiler generated.
        /// </summary>
        /// <param name="self">The referenced member to check</param>
        /// <returns><c>true</c> if the member was generated by the compiler, otherwise <c>false</c></returns>
        public static bool IsCompilerGenerated(this IHasCustomAttribute self) =>
            self.HasCustomAttribute("System.Runtime.CompilerServices", nameof(CompilerGeneratedAttribute));
    }
}
