/**
 * Generate Data-URI stylesheets with grunticon.
 */

	// Generic libs
const Grunticon = require('grunticon-lib');
const globby = require('globby');
const chalk = require('chalk');
const fs = require('fs-extra');
const commandLineArgs = require('command-line-args');
const getUsage = require('command-line-usage');

// Configs & Helpers
const veamsConfig = require('../../../veams-cli.json');

/**
 * Command Line Usage
 */
const sections = [
	{
		header: 'Grunticon Generation ',
		content: `grunticon takes a folder of SVG/PNG files (typically, icons that you've drawn in an application like Adobe Illustrator), and outputs them to CSS in 3 formats: svg data urls, png data urls, and a third fallback CSS file with references to regular png images, which are also automatically generated and placed in a folder.`
	},
	{
		header: 'Options ',
		optionList: [
			{
				name: 'config',
				typeLabel: '[underline]{filepath}',
				description: 'The config file with your options object in there.'
			},
			{
				name: 'input',
				typeLabel: '[underline]{filepath}',
				description: 'The input directory you want to process.'
			},
			{
				name: 'dest',
				typeLabel: '[underline]{filepath}',
				description: 'The destination for your generated style file.'
			},
			{
				name: 'removeAdditional',
				typeLabel: '[underline]{filepath}',
				description: 'Remove HTML and JS files which are generated by lib.'
			},
			{
				name: 'help',
				description: 'Print this usage guide.'
			}
		]
	}
];
const usage = getUsage(sections);

/**
 * Command line options
 */
const optionDefinitions = [
	{
		name: 'config',
		alias: 'c',
		type: String
	},
	{
		name: 'input',
		alias: 'i',
		type: String
	},
	{
		name: 'dest',
		alias: 'd',
		type: String
	},
	{
		name: 'removeAdditional',
		alias: 'r',
		type: Boolean
	},
	{
		name: 'help',
		alias: 'h',
		type: Boolean
	}
];

// Parsing options
const cmdOptions = commandLineArgs(optionDefinitions);
cmdOptions.config = cmdOptions.config || `${veamsConfig.paths.config}/tasks/icons/grunticon.config.js`;

/**
 * Print usage
 */
if (cmdOptions.help) {
	return console.log(usage);
}

/**
 * Get config file
 */
let defaultConfig = require(`${process.cwd()}/${cmdOptions.config}`);
const config = {
	...defaultConfig,
	input: cmdOptions.input || `${defaultConfig.input}`,
	dest: cmdOptions.dest || `${defaultConfig.dest}`,
	options: {
		...defaultConfig.pluginOptions
	}
};

/**
 * Generation Task
 */

globby(`${config.input}`)
	.then(files => generateGrunticons(files))
	.then(() => {
		if (cmdOptions.removeAdditional) {
			const removedFiles = [];

			console.log(chalk.green('Grunticon :: Generation of all files successful!\n') +
				chalk.yellow('Grunticon :: Starting with clean up ...'));

			removedFiles.push(fs.remove(`${config.dest}/icons-data-fallback.scss`));
			removedFiles.push(fs.remove(`${config.dest}/grunticon.loader.js`));
			removedFiles.push(fs.remove(`${config.dest}/preview.html`));

			return Promise.all(removedFiles);
		} else {
			return false;
		}

	})
	.then(data => {
		if (data) {
			console.log(chalk.green(`Grunticon :: Clean up & task execution successful!`))
		} else {
			console.log(chalk.green(`Grunticon :: Task successfully executed!`))
		}
	})
	.catch(err => console.log(chalk.red('Grunticon :: Error\n', err)));

/**
 * Grunticon promisified
 */
function generateGrunticons(files) {
	const grunticon = new Grunticon(files, `${config.dest}`, config.pluginOptions);

	return new Promise((resolve, reject) => {
		grunticon.process(() => resolve());
	});
}
