import { ReactNode, PropsWithoutRef } from 'react'
import {
  Form as FinalForm,
  FormProps as FinalFormProps,
} from 'react-final-form'
import { z } from 'zod'
import { validateZodSchema } from 'blitz'
export { FORM_ERROR } from 'final-form'

export interface FormProps<S extends z.ZodType<any, any>>
  extends Omit<PropsWithoutRef<JSX.IntrinsicElements['form']>, 'onSubmit'> {
  /** All your form fields */
  children?: ReactNode
  /** Text to display in the submit button */
  submitText?: string
  schema?: S
  onSubmit: FinalFormProps<z.infer<S>>['onSubmit']
  initialValues?: FinalFormProps<z.infer<S>>['initialValues']
}

export function Form<S extends z.ZodType<any, any>>({
  children,
  submitText,
  schema,
  initialValues,
  onSubmit,
  ...props
}: FormProps<S>) {
  return (
    <FinalForm
      initialValues={initialValues}
      validate={validateZodSchema(schema)}
      onSubmit={onSubmit}
      render={({ handleSubmit, submitting, submitError }) => (
        <form onSubmit={handleSubmit} className="form" {...props}>
          {/* Form fields supplied as children are rendered here */}
          {children}

          {submitError && (
            <div role="alert" style={{ color: 'red' }}>
              {submitError}
            </div>
          )}

          {submitText && (
            <button
              className="mt-4 rounded-md border border-ladderly-light-purple-1 bg-ladderly-violet-700 p-2 px-4 text-white"
              disabled={submitting}
              type="submit"
            >
              {submitText}
            </button>
          )}
        </form>
      )}
    />
  )
}

export default Form
