# Хеш-функция выполняет преобразование массива входных данных произвольной длины (ключа, сообщения)
# в (выходную) битовую строку установленной длины (хеш, хеш-код, хеш-сумму).

# При построении хеш-функции хорошим подходом является подбор функции таким образом,
# чтобы она никак не коррелировала с закономерностями, которым могут подчиняться существующие данные.

# Например, мы можем потребовать, чтобы "близкие" в некотором смысле ключи давали далекие хеш-значения
# (например, хеш функция для подряд идущих целых чисел давала далекие хеш-значения).

# В некоторых приложениях хеш-функций требуется противоположное свойство - непрерывность
# (близкие ключи должны порождать близкие хеш-значения).

# Обычно от хеш-функций ожидается, что значения хеш-функции находиятся в диапазоне от 0 до m - 1.
# Причём, часто удобно, есчли m = 2 ^ n.
# Таким образом значение хеш-функции может, например, без преобразований хранится в машинном слове.


# Хеш-функции применяются в следующих задачах:
# 1) Построение ассоциативных массивов;

# 2) Поиске дубликатов в сериях наборов данных;

# 3) Построение уникальных идентификаторов для наборов данных;

# 4) Вычислении контрольных сумм от данных (сигнала) для последующего обнаружения в них ошибок
# (возникших случайно или внесённых намеренно), возникающих при хранении и/или передаче данных;

# 5) Сохранении паролей в системах защиты в виде хеш-кода (для восстановления пароля
# по хеш-коду требуется функция, являющаяся обратной по отношению к использованной хеш-функции);

# 6) Выработке электронной подписи (на практике часто подписывается не само сообщение,
# а его «хеш-образ»); и многих других.


# "Хорошая" хеш-функция должна удовлетворять двум свойствам:
# 1) Быстрое вычисление;
# 2) Минимальное количество коллизий.

#           Метод деления в хеш-функциях
# Построение хеш-функции методом деления состоит в отображении ключа k в одну из ячеек путем
# получения остатка от деления k на m, т.е. хеш-функция имеет вид h(k) = k mod m.

print()
print('Пример выполнения хеш-функции методом деления:')


def str_h(s, m=701):
    return sum(ord(symb) for symb in s) % m


s1 = 'Хеш-функция выполняет преобразование массива входных данных произвольной длины (ключа, сообщения) в (выходную) битовую строку установленной длины (хеш, хеш-код, хеш-сумму).'
s1_cod = [(s, str_h(s)) for s in s1.split()]
s2_cod = [str((s, str_h(s))) + '\n' for s in s1.split()]
print(*s2_cod)
print()
print('Сортировка хешированных ключей:')
print('[', end='')
count = 0
for i in sorted(s[1] for s in s1_cod):
    if count != len(sorted(s[1] for s in s1_cod)) - 1:
        print(i, end=',\n')
    else:
        print(i, end=']')
    count += 1
print()

#           Метод MAD
# Хеш-функция multiply-add-and-divide (часто именуемая как MAD) преобразует целое число k по следующему алгоритму.
# У хеш-функци имеются следующие параметры: p - большое простое число, a: {1, 2, ..., p - 1} и b: {0, 1, ..., p - 1},
# m - количество значений в диапозоне значений хеш-функции.

# h{a,b}(k) = ((ak + b) mod p) mod m

# Этот класс хеш-функций удобен тем, что размер m выходного диапазона произволен и не обязательно
# представляет собой простое число.
# Поскольку число а можно выбрать р - 1 способом, и р способами – число b, всего в данном семействе будет содержаться
# р(р - 1) хеш-функций.

# Данную хеш-функцию (семейство хеш-функций) можно использовать для универсального хеширования.
# Универсальным хешированием называется хеширование, при котором используется не одна конкретная хеш-функция,
# а происходит выбор хеш-функции из заданного семейства по случайному алгоритму.

# Семейство универсальных хеш-функций называется универсальным, если для любых двух допустимых ключей вероятность
# коллизии является наименьшей из возможных:
# {Pr}{h: {H}} [h(x) = h(y)] = (1 / m), x != y.
print('    ----------Реализация метода MAD----------')
print()

import random

mad_p = 4294967311  # количество целых чисел в 32 битовом слове, https://www.numberempire.com/primenumbers.php


class MadHash:
    def __init__(self, m):
        self.m = m
        self.p = mad_p
        assert self.p > self.m
        self.a = random.randint(1, self.p)
        self.b = random.randint(0, self.p)

    def h(self, k):
        return ((self.a * k + self.b) % self.p) % self.m


mh = MadHash(2 ** 8)
print(mh.a, mh.b, sep=', ')
print()
hr1, count = [mh.h(v) for v in range(128)], 0
print('[', end='')
for i in hr1:
    if count != len(hr1) - 1:
        print(i, end=',\n')
    else:
        print(i, end=']')
    count += 1
print()
print()
for i, h in enumerate(hr1):
    print(i, h, sep=' -> ')
print()
print('Нахождение коллизий в функции и таблице:')

hr1s = sorted(hr1)
x = list(zip(hr1s, [x0 == x1 for x0, x1 in zip(hr1s[:-1], hr1s[1:])]))

print('[', end='')
ct = 0
for i in x:
    if ct != len(x) - 1:
        print(i, end=',\n')
    else:
        print(i, end=']')
    ct += 1
print()
print()

#                   Полиномиальная хеш-функция
# Приведенная ранее функция построения хеш-кода, основанная на суммировании (или опреации xor) хеш-кодов,
# плохо подходит для работы с символьными строками и другими объектами различной длины, которые могут быть
# представлены в виде кортежа (x_0 , x_1 , ..., x_{n−1}), в котором позиция элемента x_i имеет значение,
# т.к. такой хеш-код создает коллизии для строк (последовательностей) с одинаковым составом элементов.

# Примеры коллизий для хеш-функций строк основанных на суммировании
# s2 = ["stop", "tops", "pots", "spot"]
# [(s, str_h(s)) for s in s2]
# Результат:
# [('stop', 454), ('tops', 454), ('pots', 454), ('spot', 454)]

# Такого рода коллизии не будут возникать в хеш-функции, которая учитыват положение элементов в массиве
# входных данных. Примером такой хеш-функции является функция, использующая константу
# a (a \neq 0, a \neq 1$) при построении хеш-функции вида:

# x_0 * a^{n-1} + x_1 * a^{n-2} + ... + x_{n−2} * a + x_{n−1} * 1
# Т.е. это полином, использующий элементы массива входных данных (x_0 , x_1 , ..., x_{n−1}) в качестве
# коэффициентов. Такая функция назвается полиниомиальным хеш-кодом.

# Для использования ее в качестве хеш-функции к ней необходимо только добавить функцию компресии
# в соответствующий диапозон значений.
# Используя схему Горнера полиномиальный хеш-код можно эффективно вычислить по формуле:

# x_{n-1} + a(x_{n-2} + a(x_{n-3} + ... + a(x_2 + a(x_1 + a * x_0))...))



#                       Функция hash в Python
# Стандартным способом для получения хеш-кода в Python является встроенная функция hash(x).
# Она возвращает целочисленное значение для объекта x.

# Однако, в Python только неизменяемые типы данных могут возвращать значение хеш-кода.

# Это ограчиние гарантирует, что хеш-код для объека не изменится во время его жизни.

# Это свойство очень важно для корректной работы при использовании хеш-кодов объектов в хеш-таблицах,
# например в dict().
print()
print('         ----------Функция hash в Python----------')
print()
print("hash('Hello world!'):", hash('Hello world!'))
print()
print('hash(42):', hash(42))
print()
print('hash(3.141):', hash(3.141))
print()
print('hash((1, 2)):', hash((1, 2)))
print()
print('hash(None):', hash(None))
print()
print('hash(frozenset([1, 2])):', hash(frozenset([1, 2])))
print()
# ошибка:
# hash([1, 2])
# ---------------------------------------------------------------------------
# TypeError                                 Traceback (most recent call last)
# <ipython-input-80-210e02e9c6e1> in <module>
#       1 # ошибка:
# ----> 2 hash([1, 2])
#
# TypeError: unhashable type: 'list'


# Важным правилом реализации функции hash для классов явлется необходимость сохранять
# консистентность между равенством (x == y) и равенством хеш-функций (hash(x) == hash(y)).

# Для любых двух объектов из равенства x == y должно следовать hash(x) == hash(y)
# (из-за возможности коллизий у хеш-функций следствие в обратную сторону в общем случае не
# выполняется).

# Это необходимо для того, чтобы в случае использования объекта в качестве ключа в хеш таблицы
# для равных объектов (x == y) результат поиска в таблице (который ведется с ипользованием
# hash(x), hash(y)) был идентичен.

print('42 == 42.0:', 42 == 42.0)
print()
print('hash(42), hash(42.0), hash(42.0000001):', hash(42), hash(42.0), hash(42.0000001))
print()
print()

# реализация hash для пользовательского типа данных:

class Color:
    def __init__(self, r, g, b):
        assert type(r) is int
        assert 0 <= r <= 255
        self.__red = r
        assert type(g) is int
        assert 0 <= g <= 255
        self.__green = g
        assert type(b) is int
        assert 0 <= b <= 255
        self.__blue = b

    @property
    def red(self):
        return self.__red

    @property
    def green(self):
        return self.__green

    @property
    def blue(self):
        return self.__blue

    def __hash__(self):
        return hash((self.__red, self.__green, self.__blue))

    def __eq__(self, other):
        return self.__red == other.red and \
               self.__green == other.green and self.__blue == other.blue

c1 = Color(2, 2, 115)
print('c1.red, c1.green, c1.blue:', f'({c1.red}, {c1.green}, {c1.blue})')
print()
print('hash(c1):', hash(c1))
print()
c2 = Color(2, 2, 115)
print('c1 == c2:', c1 == c2)
print()
print('hash(c1) == hash(c2):', hash(c1) == hash(c2))
print()
dc = dict()
dc[c1] = 110
print('dc[c1]:', dc[c1])

# Пример выполнения хеш-функции методом деления:
# ('Хеш-функция', 391)
#  ('выполняет', 671)
#  ('преобразование', 403)
#  ('массива', 550)
#  ('входных', 596)
#  ('данных', 201)
#  ('произвольной', 397)
#  ('длины', 516)
#  ('(ключа,', 611)
#  ('сообщения)', 3)
#  ('в', 373)
#  ('(выходную)', 375)
#  ('битовую', 586)
#  ('строку', 217)
#  ('установленной', 64)
#  ('длины', 516)
#  ('(хеш,', 546)
#  ('хеш-код,', 290)
#  ('хеш-сумму).', 425)
#
#
# Сортировка хешированных ключей:
# [3,
# 64,
# 201,
# 217,
# 290,
# 373,
# 375,
# 391,
# 397,
# 403,
# 425,
# 516,
# 516,
# 546,
# 550,
# 586,
# 596,
# 611,
# 671]
#     ----------Реализация метода MAD----------
#
# 786656826, 3938180618
#
# [10,
# 53,
# 111,
# 169,
# 227,
# 29,
# 72,
# 130,
# 188,
# 246,
# 48,
# 106,
# 149,
# 207,
# 9,
# 67,
# 125,
# 168,
# 226,
# 28,
# 86,
# 144,
# 202,
# 245,
# 47,
# 105,
# 163,
# 221,
# 8,
# 66,
# 124,
# 182,
# 240,
# 42,
# 85,
# 143,
# 201,
# 3,
# 61,
# 104,
# 162,
# 220,
# 22,
# 80,
# 138,
# 181,
# 239,
# 41,
# 99,
# 157,
# 200,
# 2,
# 60,
# 118,
# 176,
# 234,
# 21,
# 79,
# 137,
# 195,
# 253,
# 40,
# 98,
# 156,
# 214,
# 16,
# 59,
# 117,
# 175,
# 233,
# 35,
# 93,
# 136,
# 194,
# 252,
# 54,
# 112,
# 155,
# 213,
# 15,
# 73,
# 131,
# 189,
# 232,
# 34,
# 92,
# 150,
# 208,
# 251,
# 53,
# 111,
# 169,
# 227,
# 29,
# 72,
# 130,
# 188,
# 246,
# 48,
# 91,
# 149,
# 207,
# 9,
# 67,
# 125,
# 168,
# 226,
# 28,
# 86,
# 144,
# 187,
# 245,
# 47,
# 105,
# 163,
# 221,
# 8,
# 66,
# 124,
# 182,
# 240,
# 27,
# 85,
# 143,
# 201,
# 3,
# 61,
# 104]
#
# 0 -> 10
# 1 -> 53
# 2 -> 111
# 3 -> 169
# 4 -> 227
# 5 -> 29
# 6 -> 72
# 7 -> 130
# 8 -> 188
# 9 -> 246
# 10 -> 48
# 11 -> 106
# 12 -> 149
# 13 -> 207
# 14 -> 9
# 15 -> 67
# 16 -> 125
# 17 -> 168
# 18 -> 226
# 19 -> 28
# 20 -> 86
# 21 -> 144
# 22 -> 202
# 23 -> 245
# 24 -> 47
# 25 -> 105
# 26 -> 163
# 27 -> 221
# 28 -> 8
# 29 -> 66
# 30 -> 124
# 31 -> 182
# 32 -> 240
# 33 -> 42
# 34 -> 85
# 35 -> 143
# 36 -> 201
# 37 -> 3
# 38 -> 61
# 39 -> 104
# 40 -> 162
# 41 -> 220
# 42 -> 22
# 43 -> 80
# 44 -> 138
# 45 -> 181
# 46 -> 239
# 47 -> 41
# 48 -> 99
# 49 -> 157
# 50 -> 200
# 51 -> 2
# 52 -> 60
# 53 -> 118
# 54 -> 176
# 55 -> 234
# 56 -> 21
# 57 -> 79
# 58 -> 137
# 59 -> 195
# 60 -> 253
# 61 -> 40
# 62 -> 98
# 63 -> 156
# 64 -> 214
# 65 -> 16
# 66 -> 59
# 67 -> 117
# 68 -> 175
# 69 -> 233
# 70 -> 35
# 71 -> 93
# 72 -> 136
# 73 -> 194
# 74 -> 252
# 75 -> 54
# 76 -> 112
# 77 -> 155
# 78 -> 213
# 79 -> 15
# 80 -> 73
# 81 -> 131
# 82 -> 189
# 83 -> 232
# 84 -> 34
# 85 -> 92
# 86 -> 150
# 87 -> 208
# 88 -> 251
# 89 -> 53
# 90 -> 111
# 91 -> 169
# 92 -> 227
# 93 -> 29
# 94 -> 72
# 95 -> 130
# 96 -> 188
# 97 -> 246
# 98 -> 48
# 99 -> 91
# 100 -> 149
# 101 -> 207
# 102 -> 9
# 103 -> 67
# 104 -> 125
# 105 -> 168
# 106 -> 226
# 107 -> 28
# 108 -> 86
# 109 -> 144
# 110 -> 187
# 111 -> 245
# 112 -> 47
# 113 -> 105
# 114 -> 163
# 115 -> 221
# 116 -> 8
# 117 -> 66
# 118 -> 124
# 119 -> 182
# 120 -> 240
# 121 -> 27
# 122 -> 85
# 123 -> 143
# 124 -> 201
# 125 -> 3
# 126 -> 61
# 127 -> 104
#
# Нахождение коллизий в функции и таблице:
# [(2, False),
# (3, True),
# (3, False),
# (8, True),
# (8, False),
# (9, True),
# (9, False),
# (10, False),
# (15, False),
# (16, False),
# (21, False),
# (22, False),
# (27, False),
# (28, True),
# (28, False),
# (29, True),
# (29, False),
# (34, False),
# (35, False),
# (40, False),
# (41, False),
# (42, False),
# (47, True),
# (47, False),
# (48, True),
# (48, False),
# (53, True),
# (53, False),
# (54, False),
# (59, False),
# (60, False),
# (61, True),
# (61, False),
# (66, True),
# (66, False),
# (67, True),
# (67, False),
# (72, True),
# (72, False),
# (73, False),
# (79, False),
# (80, False),
# (85, True),
# (85, False),
# (86, True),
# (86, False),
# (91, False),
# (92, False),
# (93, False),
# (98, False),
# (99, False),
# (104, True),
# (104, False),
# (105, True),
# (105, False),
# (106, False),
# (111, True),
# (111, False),
# (112, False),
# (117, False),
# (118, False),
# (124, True),
# (124, False),
# (125, True),
# (125, False),
# (130, True),
# (130, False),
# (131, False),
# (136, False),
# (137, False),
# (138, False),
# (143, True),
# (143, False),
# (144, True),
# (144, False),
# (149, True),
# (149, False),
# (150, False),
# (155, False),
# (156, False),
# (157, False),
# (162, False),
# (163, True),
# (163, False),
# (168, True),
# (168, False),
# (169, True),
# (169, False),
# (175, False),
# (176, False),
# (181, False),
# (182, True),
# (182, False),
# (187, False),
# (188, True),
# (188, False),
# (189, False),
# (194, False),
# (195, False),
# (200, False),
# (201, True),
# (201, False),
# (202, False),
# (207, True),
# (207, False),
# (208, False),
# (213, False),
# (214, False),
# (220, False),
# (221, True),
# (221, False),
# (226, True),
# (226, False),
# (227, True),
# (227, False),
# (232, False),
# (233, False),
# (234, False),
# (239, False),
# (240, True),
# (240, False),
# (245, True),
# (245, False),
# (246, True),
# (246, False),
# (251, False),
# (252, False)]
#
#
#          ----------Функция hash в Python----------
#
# hash('Hello world!'): 6866628724218699186
#
# hash(42): 42
#
# hash(3.141): 325123864299130883
#
# hash((1, 2)): -3550055125485641917
#
# hash(None): 8795981494408
#
# hash(frozenset([1, 2])): -1826646154956904602
#
# 42 == 42.0: True
#
# hash(42), hash(42.0), hash(42.0000001): 42 42 230584303658
#
#
# c1.red, c1.green, c1.blue: (2, 2, 115)
#
# hash(c1): 4644449387623648775
#
# c1 == c2: True
#
# hash(c1) == hash(c2): True
#
# dc[c1]: 110
#
# Process finished with exit code 0