# https://dsp.stackexchange.com/questions/52982/perform-non-power-of-two-fft-using-arm-cmsis-library
import numpy as np

# 1280 FFT based on five FFTs of 256 each
n = 1280

# Create a piece of noise
x = np.expand_dims(np.array(
    [-4.1488e-01, -9.2448e-01, 3.5027e-01, -2.4207e+00, 1.9487e-01, -7.8371e-01, -1.3600e-01,
     2.8558e-01, 1.3649e+00, 3.9745e-01, 1.8342e-01, -1.2420e-01, -1.0443e+00, 1.3939e-01,
     -4.9939e-01, -3.4967e-01, 8.4696e-01, -1.8719e-01, -4.8928e-01, -2.0881e+00, -1.3318e+00,
     -2.1042e+00, -4.8688e-01, -1.7142e-01, 1.3219e+00, 1.0797e+00, -2.0530e+00, 6.1470e-01,
     1.2031e+00, -1.6347e-01, 1.1620e-02, 1.6592e+00, -1.0170e+00, 6.9519e-01, -1.1510e+00,
     -2.0430e-01, -1.0328e+00, -1.8557e+00, -4.2818e-01, 5.7056e-01, 5.3481e-02, -8.4769e-01,
     6.3757e-01, -3.2138e-01, -1.7087e+00, -1.5569e+00, 1.5555e+00, 7.0832e-01, 2.9571e+00,
     -8.9484e-01, 1.5150e+00, -2.6829e-01, 1.6363e+00, -2.3867e-01, -5.0999e-01, 1.2587e+00,
     -3.4107e-01, -3.1700e-01, 6.0932e-01, -2.2974e+00, -6.2836e-01, 3.3953e-01, -1.2361e+00,
     9.9197e-01, 3.9961e-01, 7.1825e-01, 1.7606e+00, -1.7261e+00, -1.1246e+00, -4.6820e-01,
     9.5441e-02, -1.2826e+00, -8.5144e-01, -1.6001e+00, 6.8586e-01, -8.8040e-01, -1.3032e+00,
     1.4903e+00, 7.5253e-01, 3.1416e-01, -1.1736e+00, 1.2384e-01, 1.4356e+00, -4.2002e-01,
     4.6197e-01, 3.7264e-01, 1.3238e+00, -5.0668e-01, 1.6777e+00, 1.0156e+00, 1.5524e+00,
     -1.4940e+00, 4.7429e-02, -4.2382e-01, -1.1864e+00, 1.6299e-01, -9.6608e-01, 8.7988e-01,
     -9.5379e-01, -6.8413e-01, -3.9483e-02, -2.4163e-01, 1.3079e+00, -1.4672e-01, 6.8055e-01,
     -2.2861e+00, 7.7490e-01, -1.4670e+00, -2.2253e-01, 1.4340e+00, 1.0121e+00, -1.0459e+00,
     -3.2284e-01, 2.2410e+00, 1.1848e+00, 1.3242e-01, -2.6312e+00, -8.9672e-01, -3.7693e-01,
     -6.7873e-02, 1.1887e-01, -9.1289e-01, 2.5304e-01, -1.9358e-01, -8.9879e-02, 8.7993e-02,
     -2.2463e+00, -9.7670e-01, -1.0129e+00, 2.9357e-01, -1.0843e-01, 5.7513e-01, -2.1635e-01,
     1.4231e-01, 3.9493e-01, 1.6919e+00, 9.6167e-01, -7.7692e-01, -7.3883e-01, -6.0918e-01,
     -1.5647e-01, -1.0422e-01, 2.8992e-01, 6.6684e-01, 1.7778e-01, 6.9553e-01, 3.3273e-01,
     -1.2264e+00, -1.7459e+00, 8.1993e-01, -1.0940e-01, 1.1302e+00, 1.1487e+00, 1.7854e+00,
     1.5899e+00, 1.5074e+00, -5.3930e-02, 2.0262e+00, 1.1897e+00, 1.4754e+00, 2.8854e-01,
     8.2244e-01, 3.7323e-01, -2.4043e-03, 3.6744e-02, 5.9796e-01, -5.4140e-01, -2.1389e-01,
     -9.3853e-01, -3.2756e-01, 9.0069e-02, 1.9206e+00, -4.9497e-01, -6.9619e-01, 2.7113e-01,
     -6.7159e-01, -3.8296e-01, -6.2717e-01, 9.9195e-01, -1.0369e+00, -4.5746e-03, -1.0043e+00,
     -1.6438e-01, 5.9607e-02, -6.3473e-01, 1.3317e-01, -3.2152e-01, -1.4103e+00, -2.7484e-01,
     1.2029e+00, -1.2080e+00, 4.2627e-01, 1.0301e+00, -8.3935e-01, 2.7005e-01, -5.1658e-01,
     -1.2983e-01, -1.0014e-01, 9.0529e-01, -1.4560e-01, 3.7585e-01, -8.7873e-01, -2.4088e+00,
     9.0976e-02, -8.8576e-01, -7.0332e-01, -2.5367e+00, 7.6970e-01, 5.7537e-01, 2.8019e-01,
     4.8469e-01, -5.6636e-01, 2.8767e-01, 1.0489e+00, 1.3031e+00, 4.6826e-01, -1.9177e+00,
     9.9355e-01, 9.5695e-01, -3.6231e-01, -8.1648e-01, -3.6484e-01, 7.9047e-01, -5.0989e-01,
     8.4394e-01, 7.5021e-02, -1.5262e+00, 3.5884e-01, 2.2213e-01, -5.2161e-01, -1.9997e+00,
     9.7885e-04, 1.4570e+00, 4.6854e-01, 9.0670e-01, -2.3209e-01, 1.2066e+00, 1.8578e-01,
     6.6934e-01, 1.3838e-01, 6.2549e-01, -3.6912e-01, 7.5273e-02, 1.3751e-01, 9.4630e-01,
     4.3422e-01, -1.6949e+00, -1.6021e+00, 1.3015e-01, 5.3158e-01, -6.8727e-01, 6.7309e-01,
     1.4493e+00, 7.4439e-01, 1.8175e+00, -4.1744e-01, 5.6308e-01, 7.8390e-02, -8.7782e-01,
     -1.7291e+00, -6.2982e-03, -1.6799e-01, -1.0595e+00, -5.1189e-01, 1.7525e+00, -5.2519e-01,
     -1.1611e+00, 7.8727e-01, -5.3809e-01, -6.6755e-01, 1.4389e+00, 5.7077e-01, 9.6383e-01,
     1.8389e+00, -4.2953e-01, 4.6895e-02, -1.6928e+00, 2.0049e-01, 2.5526e+00, 1.1468e+00,
     7.7482e-01, -1.8454e+00, -2.0514e+00, -3.8614e-01, -9.6613e-01, -6.9353e-01, 1.0960e+00,
     -1.1870e+00, 1.1646e+00, 2.2491e+00, 6.3257e-01, 1.7369e+00, 4.3160e-01, 1.7561e+00,
     4.0193e-01, 9.2237e-01, -2.8238e-01, 1.0852e+00, -1.1271e-01, -1.8490e+00, -7.2364e-01,
     -5.6986e-01, -6.8518e-01, -6.2122e-01, 2.5198e-01, -2.8666e-01, -5.5868e-01, -2.4283e-01,
     6.6530e-02, 1.9580e+00, 6.2159e-01, 1.6752e+00, -2.0359e+00, 5.0398e-01, -2.0165e-01,
     -4.0322e-01, -1.2912e+00, -4.5723e-02, 1.7038e+00, -1.0314e+00, -1.3402e+00, 5.6550e-01,
     1.0266e-01, 1.0877e+00, -2.2055e-01, -3.3177e-01, 3.0165e-01, 4.6620e-02, -6.6769e-01,
     6.2928e-01, -7.2397e-01, -2.5726e-01, -2.7200e-01, 2.0872e-01, -8.7203e-01, -1.3144e+00,
     8.9952e-01, -3.8221e-01, 6.2872e-02, 1.0304e+00, -3.5753e-01, -2.6747e-01, 1.3758e+00,
     -4.8414e-01, -9.1888e-01, 1.1780e+00, -6.9825e-01, 1.0606e+00, -2.1483e+00, -6.0914e-01,
     -7.2031e-02, -4.6350e-01, -1.3708e-01, 6.0922e-01, 1.2510e+00, 1.0087e+00, 1.7439e-01,
     5.2244e-01, 1.4969e+00, -7.1393e-01, 1.1180e+00, -1.1983e-01, -1.3842e+00, 4.7895e-01,
     -6.4305e-01, -1.5597e+00, -3.9605e-01, 6.1447e-02, 1.8299e+00, -7.0158e-01, -1.8043e+00,
     -3.3648e-01, 7.9996e-01, 6.4264e-01, -2.5850e-01, 1.4992e+00, 1.0895e-01, -1.0542e+00,
     1.5164e-01, -1.6523e+00, 2.2730e-01, 1.8635e+00, -1.1931e+00, -7.5171e-01, -9.1451e-01,
     -1.0726e+00, 4.8927e-01, 1.5187e-01, 8.5337e-02, -2.0334e+00, -2.2892e-01, -8.3675e-01,
     -4.3078e-01, 2.2367e+00, -8.5644e-01, 9.0270e-02, 6.2115e-01, 8.4920e-01, -1.3196e+00,
     1.0723e+00, -3.5813e-01, -5.5137e-01, -7.5038e-01, -4.5461e-01, 6.2216e-01, -5.9859e-01,
     -6.5886e-02, 1.4825e-01, -2.2378e-01, 9.5793e-01, 4.0491e-01, -1.8373e+00, 1.7815e+00,
     -7.9259e-01, 4.8308e-02, 1.4843e+00, 1.0098e+00, 6.2844e-01, 1.1212e+00, -4.9345e-01,
     2.1573e-01, 2.8339e-01, -1.7444e+00, -7.2886e-01, 1.4045e+00, -4.0127e-01, -1.8413e-01,
     2.5687e+00, -7.7999e-01, -1.1350e-02, 4.2623e-01, -5.0716e-01, 7.7376e-01, -1.5475e+00,
     9.4497e-01, 1.1898e+00, -1.0726e+00, -2.7838e-01, 1.2455e+00, 1.0615e+00, -5.6313e-01,
     1.0755e+00, 1.9948e+00, 2.1403e+00, -1.5469e+00, 1.1753e+00, 8.8331e-01, 6.4893e-01,
     -2.2493e+00, -1.3032e-01, -2.7018e+00, -8.2875e-01, -3.5987e-01, 3.0482e-01, -7.0763e-01,
     -1.0907e-01, -6.3233e-01, -3.1732e-01, 9.4530e-01, -1.8611e-01, -7.7528e-01, 1.3630e+00,
     -2.3203e-01, -9.2287e-01, 4.6118e-01, -1.0747e+00, 1.5899e+00, -3.4540e-02, 7.7994e-01,
     8.0845e-01, 2.0075e-01, 1.6965e+00, -1.4165e+00, 1.8064e+00, -9.9992e-01, 1.6257e-01,
     -1.3907e-01, 2.9551e-01, -2.3676e+00, 1.2543e+00, 2.4931e+00, 9.0623e-01, 6.4736e-01,
     -1.1231e+00, 6.7251e-01, -1.2621e+00, -2.7173e-01, 8.4136e-01, 1.1671e+00, -1.2321e+00,
     -7.2945e-01, 1.0178e-01, 1.5719e-01, 2.7354e-01, -2.8636e-01, 7.1915e-01, -2.0683e-01,
     -5.9685e-01, 8.1272e-01, 1.5457e+00, 3.4248e-01, -2.5198e-01, 5.1473e-01, -5.0769e-01,
     -6.8485e-01, 3.3319e-01, -3.9144e-01, 9.6591e-02, 1.1278e+00, -1.4809e+00, 4.3509e-01,
     7.7548e-01, -1.0593e+00, -3.9162e-01, 4.3329e-01, 1.0829e-01, -1.9463e-01, 4.9631e-01,
     8.8766e-01, 1.2863e-01, -6.6877e-01, 1.4338e+00, -1.0119e+00, -1.0608e+00, 2.6860e-02,
     1.9727e-01, 5.8275e-01, 8.3553e-01, -2.9511e-01, 1.8918e+00, 2.0398e+00, -5.3011e-01,
     4.5546e-01, 7.9428e-01, -8.9882e-02, -2.0151e-01, -1.7641e-01, 7.0682e-01, 1.7320e+00,
     1.4191e+00, -4.8626e-01, 6.1453e-01, -1.0752e+00, 1.1338e-02, -1.4466e+00, 7.2053e-01,
     -1.4958e+00, -2.3536e-01, 2.6940e-01, -8.2904e-01, -6.0504e-01, -1.1634e+00, -2.1776e+00,
     6.8866e-01, -1.4810e+00, 1.8634e+00, -9.7114e-01, 4.6251e-01, -5.9909e-01, 9.5367e-01,
     7.5680e-01, -1.5910e-01, 1.8921e+00, -1.3641e+00, 1.2488e+00, -1.3582e-01, 2.2218e-01,
     -1.1610e+00, -1.6205e+00, 1.9881e-01, -6.5954e-01, 1.8082e-01, 1.0377e-01, -2.1137e+00,
     -1.4346e-01, -4.9851e-01, -3.1803e-01, -9.3909e-01, 6.2689e-01, -2.0858e-02, 6.4558e-01,
     -2.6198e-01, -2.5766e-01, -4.9697e-01, -1.6314e-02, -1.5443e+00, -3.7084e-01, -6.0892e-01,
     2.9410e+00, 2.1076e-01, 1.7835e+00, 1.2227e+00, 1.4020e-01, 4.4618e-01, -2.1935e-01,
     -6.0509e-01, -3.0275e-01, -7.5706e-01, 1.0663e+00, -1.9490e+00, 7.6509e-01, -1.4397e-01,
     -2.5672e+00, 1.3260e+00, 6.4298e-01, 4.6866e-01, 7.6328e-01, 9.7330e-02, -1.5036e+00,
     -1.4019e+00, 9.0652e-01, 1.4668e+00, -1.7287e+00, 1.3305e-01, -1.4205e+00, 4.5626e-01,
     1.1478e+00, -6.0446e-01, 1.1878e+00, -1.1610e+00, 1.7156e+00, 1.3885e+00, -7.3295e-01,
     -5.2411e-01, -1.3990e+00, 1.7097e-01, 1.1922e+00, -3.6641e-03, -9.7733e-01, 1.1643e+00,
     -1.3249e-01, 1.2953e+00, 7.1774e-01, 3.3401e-01, -9.0487e-01, -8.3334e-01, -1.8897e-01,
     -7.0907e-01, -2.8963e+00, 4.4920e-02, 1.0431e-01, -6.8174e-01, 5.7715e-01, -7.6168e-02,
     9.1019e-01, 6.8638e-01, 1.3136e-02, 4.3426e-01, -5.4715e-01, -2.7566e+00, -7.2318e-01,
     1.6177e+00, 8.1904e-01, -9.3565e-01, 1.6113e-01, 1.0525e+00, -8.5731e-01, -4.6700e-01,
     7.9942e-01, 7.6692e-01, 2.7156e-01, 4.5682e-01, -2.4310e-01, 1.1496e-01, -6.0460e-01,
     2.6421e-01, -3.7128e-01, 4.6117e-01, -1.5826e+00, -1.2678e+00, 7.1415e-01, 4.1213e-01,
     -1.2572e-01, 3.6206e-01, 1.0925e+00, 5.1066e-02, 4.0794e-01, -5.5582e-01, -6.7038e-01,
     1.7340e-01, 2.3633e+00, -4.2998e-01, 4.0816e-01, 5.0638e-01, 1.7899e+00, -3.8226e-01,
     2.0800e+00, 1.6086e-01, -8.4434e-01, 1.0596e+00, 2.2525e+00, 3.6725e-01, -8.0882e-01,
     -2.5660e-01, 1.6244e-01, -1.5824e-01, 9.7691e-01, 8.2820e-01, 2.0129e+00, -2.0329e+00,
     7.2982e-01, -4.0008e-01, -7.2421e-01, 1.9667e-01, -1.9192e+00, -5.1371e-01, 1.6182e+00,
     1.4591e+00, 2.4087e-01, 3.0823e-01, 3.4511e-01, -1.4152e+00, -1.5426e-01, -8.6402e-01,
     1.9109e+00, 6.8347e-01, -5.8475e-01, 4.0220e-01, 3.2204e-01, -7.4958e-01, 9.1149e-01,
     4.3767e-01, -7.6190e-01, -1.1828e+00, -1.4967e+00, 5.0689e-01, 1.0821e+00, -3.8596e-01,
     5.9077e-01, 2.6202e-01, -1.2097e+00, -1.7156e+00, -8.1916e-01, -3.6850e-01, 4.1804e-01,
     -1.6833e+00, -1.1350e+00, -7.4182e-01, -3.0695e-01, 2.0065e-01, -8.1373e-01, -8.5884e-01,
     6.9345e-01, -2.2506e-01, 1.1069e-01, -1.0960e+00, 2.7062e-01, -1.7595e-01, 2.1322e+00,
     -9.9583e-01, -1.1861e+00, -2.0261e+00, 2.3163e+00, -1.0466e+00, -5.9040e-01, 2.1633e+00,
     7.6991e-02, -3.9589e-01, -1.7468e-01, -7.6032e-01, -1.2996e+00, 1.0743e+00, 6.4865e-01,
     5.8944e-03, -8.9003e-01, 6.5900e-01, 1.0958e+00, -1.3044e+00, -6.2841e-01, -7.9062e-01,
     1.2435e+00, -9.2536e-01, 1.9286e+00, -2.1453e+00, 8.9145e-01, -5.2080e-01, -1.3932e-01,
     -2.4506e-01, 1.9380e+00, -5.6642e-01, -6.4525e-01, -1.3943e+00, -1.1109e+00, 3.9823e-01,
     -2.4828e+00, -1.0659e+00, 2.4474e-02, -4.6595e-01, -2.2136e-01, 1.3674e+00, -1.8494e-01,
     -3.0039e-01, 7.1307e-01, -9.8606e-01, 5.4703e-01, 9.0783e-01, -3.4092e-01, 2.4723e-01,
     1.3889e+00, 5.0881e-01, 5.1822e-01, 1.6415e+00, 1.1699e+00, 3.4461e-01, 1.0253e+00, 1.6036e+00,
     -1.3741e-01, 8.4919e-01, 8.9790e-01, -1.9921e-01, 9.7984e-01, -4.8201e-01, -5.4573e-01,
     -2.2753e+00, -8.3550e-01, 7.8487e-02, 1.1744e+00, 5.9869e-01, -7.3392e-02, -1.4236e+00,
     1.7176e+00, -1.5527e+00, -1.3585e+00, -1.1185e+00, -2.4091e-01, -7.3293e-01, -1.0728e+00,
     2.1748e-01, 1.7487e+00, -7.3377e-01, 2.0428e+00, 1.7342e+00, -5.6145e-01, 5.6491e-01,
     -1.8101e+00, -3.7734e-01, -2.9498e+00, -8.9287e-01, 1.9070e+00, -2.5263e-01, -1.7800e+00,
     -3.8260e-01, 1.3162e+00, 1.2760e+00, 1.4475e+00, -2.0453e+00, 9.7177e-01, 1.1695e+00,
     1.4783e+00, 1.1681e+00, -1.3491e-01, -4.0207e-01, -8.3138e-01, -1.1760e+00, -1.1923e-01,
     3.6632e-01, 5.2624e-01, -1.2686e+00, 3.7546e-01, 1.0659e+00, -5.0498e-01, -4.9066e-01,
     -1.5263e+00, -1.5813e+00, 1.0673e-01, 3.8973e-01, 5.1687e-01, 1.3344e+00, -1.0606e+00,
     -9.2464e-01, -1.7718e+00, -3.0530e-01, 5.7619e-01, 1.7497e+00, 6.8695e-01, 5.0385e-01,
     -1.2883e+00, 8.1203e-02, -7.7810e-01, 1.3922e-01, 1.2512e+00, 8.2796e-01, -9.1979e-01,
     -3.0747e-01, -1.7976e+00, -4.5424e-01, 1.5745e-01, 7.9170e-01, 7.1659e-01, 9.1052e-01,
     1.4410e+00, 8.6432e-01, 7.6028e-02, 3.7553e-01, 1.2112e+00, -2.0902e-02, 8.7232e-01,
     -9.7506e-03, 7.0521e-01, -6.7970e-01, 1.2619e+00, 1.8471e+00, 8.1033e-01, -1.7458e+00,
     7.1404e-02, 7.3276e-02, 1.6234e+00, -9.2911e-01, -1.1025e+00, -1.9376e-01, -1.1583e+00,
     -2.2860e+00, 6.9238e-01, 1.3554e+00, 1.2764e+00, -4.5544e-02, 3.8845e-02, 1.3993e+00,
     -1.4547e+00, 2.1694e-01, -1.8131e-01, -6.6751e-02, -5.2279e-01, -7.2893e-01, 8.0611e-01,
     -9.0736e-01, -5.8881e-02, -1.8291e+00, 4.9943e-01, 1.0184e+00, -8.5020e-01, 7.4167e-01,
     -1.5637e+00, -5.7415e-01, -3.2104e-01, 3.1837e-01, -4.7992e-01, -1.2488e-01, 1.1063e+00,
     -1.5025e+00, 1.9074e+00, 3.6550e-01, -6.3435e-01, -1.2599e+00, 6.5697e-02, 5.3798e-01,
     9.5039e-01, 6.1239e-01, -5.7909e-01, -3.2882e-01, 1.2214e+00, -3.4580e-01, 5.8557e-01,
     1.1800e+00, 1.0803e-01, 1.0405e+00, 7.7834e-01, 8.6186e-01, 6.3910e-01, 1.2711e+00,
     -8.0222e-01, 5.2783e-02, -2.5477e-01, 7.3108e-02, 1.8420e+00, -1.1845e+00, -1.2322e+00,
     1.1468e+00, -1.7349e+00, -1.6351e-01, 4.4958e-01, 6.5334e-01, 1.4267e-01, 2.0860e+00,
     -5.2512e-02, -5.4237e-01, 1.6328e+00, 2.0744e-01, 9.1028e-01, -4.3513e-01, -1.0373e+00,
     -8.0590e-01, -1.1957e+00, -9.0254e-01, 1.6954e+00, -7.7844e-01, 1.1398e+00, -1.1129e+00,
     4.2439e-01, 2.6492e-01, 7.2903e-01, 1.5550e-01, -9.2760e-01, 7.8894e-01, -9.1396e-01,
     -1.7810e+00, -6.4848e-01, -2.4744e+00, -9.7357e-01, -3.0472e-01, 8.5593e-01, -4.3757e-01,
     4.7301e-01, -4.9829e-02, -9.6333e-01, 9.4439e-01, -6.6561e-01, 6.6517e-01, 4.6433e-01,
     4.2831e-01, 7.0934e-01, -6.1992e-01, 1.9511e+00, 7.9865e-01, -4.1635e-01, -5.1635e-02,
     5.7854e-01, 9.7156e-01, 7.0995e-01, -1.9433e-01, -6.7057e-01, -2.0511e+00, -2.3526e-01,
     1.0844e+00, 1.0730e-01, -2.4880e-01, 9.1865e-01, -1.0797e+00, 1.8629e-01, 8.7834e-01,
     2.5032e-01, -1.0772e+00, 4.0091e-01, -1.8913e+00, -2.8724e-01, 1.5534e-01, -1.7529e+00,
     -1.1153e-01, 7.4164e-01, -6.1898e-02, -1.0627e+00, -1.1334e+00, 3.3294e-02, 3.9319e-01,
     3.4313e-01, 8.8878e-01, 2.4400e-03, -3.2165e-02, 1.0344e+00, 1.3676e-01, -1.0802e-01,
     -9.7432e-03, 1.5905e+00, 2.4696e-01, -6.6787e-01, 4.7492e-02, 8.9841e-01, 5.0459e-01,
     9.0784e-03, -1.9140e+00, 9.9598e-01, 8.3746e-02, -1.0714e+00, -2.3219e+00, -6.6653e-02,
     -7.3808e-01, -6.7713e-01, -3.6206e-02, -3.7875e-01, 8.7921e-01, -2.2187e-01, 6.7972e-01,
     1.0345e+00, 8.3253e-01, 1.4252e-02, 5.6941e-02, 7.5515e-01, -7.3773e-02, 3.2071e-01,
     5.2695e-01, 9.9983e-01, 5.8305e-02, 1.3729e-01, -3.9623e-01, -1.0020e+00, 3.3285e-02,
     -4.9971e-01, 6.4403e-01, -4.7039e-02, 9.9160e-01, 2.0073e-01, 7.0882e-01, -3.5299e-01,
     -1.8020e-01, -1.8129e+00, -9.5295e-01, -1.1631e+00, -1.1918e-01, 6.5101e-01, -8.4805e-01,
     2.4961e-01, -7.8247e-01, -3.3530e-01, -1.7831e+00, -1.5012e+00, 1.4051e-01, 1.9088e+00,
     -1.2237e+00, -4.8878e-02, 7.8567e-01, -1.0965e-01, -7.1596e-01, -1.1034e-01, 2.7963e-01,
     2.2372e-01, 1.8480e-01, -6.2744e-01, 4.9713e-01, 1.0479e+00, -6.0197e-02, -6.1440e-01,
     3.5720e-01, -4.8342e-01, -4.8204e-01, 5.3126e-01, 2.1878e-01, -1.4723e+00, 8.1245e-01,
     5.1301e-01, -1.9520e-02, 1.9900e-01, 1.6386e-01, 1.1365e-01, 6.4509e-03, 1.2729e+00,
     4.9291e-01, 6.1764e-01, -2.6792e-01, -5.7030e-01, -2.4164e+00, -7.3968e-01, 1.1633e+00,
     7.8950e-01, 8.6144e-01, 1.8745e+00, -1.0049e+00, -1.3111e+00, 6.1854e-01, 6.4753e-01,
     -1.1100e+00, 1.5352e-01, -3.2226e-01, 7.4135e-01, -1.0010e+00, -1.7474e-01, -1.5241e+00,
     -5.6116e-01, 4.0870e-01, 7.5695e-01, 1.0604e+00, -1.0358e+00, -1.6777e+00, 6.4488e-01,
     5.8582e-01, 1.4466e+00, -6.7446e-01, 9.1643e-01, 1.4548e+00, -8.8701e-01, 2.1853e+00,
     -5.3285e-01, 1.8380e-01, 2.0897e+00, 1.2017e+00, -1.0458e+00, -2.4686e-01, 6.1483e-01,
     -1.0641e+00, -1.0404e+00, 2.3223e-01, -1.2346e+00, 1.3533e+00, -8.9224e-01, 1.8574e-01,
     -2.7008e-02, 1.9834e+00, -1.5936e-01, -1.5425e+00, 7.3876e-01, -7.0803e-01, -2.6807e-01,
     -1.2448e+00, -9.5569e-01, 7.2522e-01, 1.7769e-01, 1.2771e-01, -5.2491e-01, 7.4665e-01,
     9.9348e-02, 1.1900e+00, -1.0808e+00, -8.1579e-02, -1.7205e+00, 5.2716e-01, 2.3826e-01,
     -2.0841e+00, 2.7033e-01, 4.4091e-01, -1.6447e+00, -1.2119e+00, 3.2267e-01, -4.6370e-03,
     -7.4356e-01, -8.5964e-01, 2.6228e-01, 6.9574e-01, -9.6830e-01, 2.1486e+00, 2.9226e-01,
     6.4278e-01, -6.4754e-01, 2.2273e-01, 3.1553e-01, -9.1593e-01, 1.9176e+00, 3.9915e-01,
     8.0985e-01, -3.1640e-01, 7.6797e-01, 8.0035e-01, -2.1371e+00, 2.0538e+00, 1.8872e+00,
     2.8187e-01, -8.7993e-01, 4.6678e-01, -7.5733e-01, 9.0605e-01, -5.5566e-01, -1.5131e+00,
     -1.2057e+00, -1.0686e+00, -4.9018e-01, 7.7064e-01, 1.3722e-01, 2.6215e-01, -1.0863e+00,
     -1.3151e+00, -1.2168e-01, -7.3414e-01, 4.7929e-01, 5.3309e-01, -5.1574e-01, -1.4742e+00,
     -1.3417e+00, -6.8831e-02, -2.2740e+00, 8.9694e-01, 2.4879e-01, 1.6895e+00, -5.3846e-01,
     -6.2556e-01, 3.8016e-02, -5.4459e-01, 1.1485e+00, 3.6656e-01, -8.0543e-01, -1.7693e+00,
     8.1608e-01, 1.4643e+00, 1.1667e+00, 1.0407e+00, -9.4399e-01, 9.2806e-02, -3.2360e-02,
     2.2728e+00, 2.5641e+00, 4.1048e-01, -7.7594e-01, 7.9364e-01]), axis=-1)

# Calculate FFT using NumPy's fft() function.
# We'll use this as a reference to prove it works
fx = np.fft.fft(x)

# Break down into five signals of 256 points each, interleaved
p = x[::5]
q = x[1::5]
r = x[2::5]
s = x[3::5]
t = x[4::5]

# FFT each of those. This is a 256 power-of-two standard FFT
fp = np.fft.fft(p)
fq = np.fft.fft(q)
fr = np.fft.fft(r)
fs = np.fft.fft(s)
ft = np.fft.fft(t)

# Do five times periodic extension (just repeat it 5 times)
fp5 = np.tile(fp, (5, 1))
fq5 = np.tile(fq, (5, 1))
fr5 = np.tile(fr, (5, 1))
fs5 = np.tile(fs, (5, 1))
ft5 = np.tile(ft, (5, 1))

# Calculate the 1280 twiddle factors
k5 = np.arange(n)
W5 = np.exp(-2j * np.pi * k5 / n)

# Assemble the result
fy5 = fp5 + W5[:, np.newaxis] * fq5 + W5[:, np.newaxis] ** 2 * fr5 + W5[:,
                                                                     np.newaxis] ** 3 * fs5 + W5[:,
                                                                                              np.newaxis] ** 4 * ft5

# Calculate the error
ferror = fy5 - fx
error = 10 * np.log10(np.sum(np.abs(ferror) ** 2) / np.sum(np.abs(fx) ** 2))
print(f'Error = {error:.2f} dB')
