#include "GlobalVariableUsageMatchCallback.h"

#include "clang-utils/AlignmentFetcher.h"
#include "clang-utils/ClangUtils.h"
#include "clang-utils/Matchers.h"

#include "loguru.h"

#include <utility>

void GlobalVariableUsageMatchCallback::run(const MatchFinder::MatchResult &Result) {
    ExecUtils::throwIfCancelled();
    checkUsage(Result);
}

static std::unordered_set<std::string> BLACK_LIST = {"stdin", "stdout", "stderr"};

void GlobalVariableUsageMatchCallback::checkUsage(const MatchFinder::MatchResult &Result) {
    if (const auto *pVarDecl =
            Result.Nodes.getNodeAs<clang::VarDecl>(Matchers::GLOBAL_VARIABLE_USAGE)) {
        clang::QualType varType = pVarDecl->getType();
        std::string name = pVarDecl->getNameAsString();
        if (!pVarDecl->isKnownToBeDefined() && !pVarDecl->isExternC()) {
            LOG_S(DEBUG) << "Variable \"" << name << "\" was skipped - it has no definition.";
            return;
        }
        if (const auto *pFunctionDecl = Result.Nodes.getNodeAs<clang::FunctionDecl>(
                Matchers::FUNCTION_USED_GLOBAL_VARIABLE)) {
            if (varType.isConstant(pVarDecl->getASTContext())) {
                LOG_S(MAX) << "Variable \"" << name << "\" was skipped - it is being constant.";
                return;
            }
            if (CollectionUtils::contains(BLACK_LIST, name)) {
                LOG_S(MAX) << "Variable \"" << name << "\" was skipped - it is being blacklisted.";
                return;
            }
            if (ClangUtils::isIncomplete(pVarDecl->getType())) {
                LOG_S(MAX) << "Variable \"" << name
                           << "\" was skipped - it's type has no definition in current "
                              "translation unit.";
                return;
            }
            handleUsage(pFunctionDecl, pVarDecl);
        }
    }
}

void GlobalVariableUsageMatchCallback::handleUsage(const clang::FunctionDecl *functionDecl,
                                                   const clang::VarDecl *varDecl) {
    clang::SourceManager &sourceManager = functionDecl->getASTContext().getSourceManager();
    fs::path sourceFilePath = ClangUtils::getSourceFilePath(sourceManager);
    auto const &[iterator, inserted] =
            usages.emplace(varDecl->getNameAsString(), functionDecl->getNameAsString());
    auto const &usage = *iterator;

    LOG_S(MAX) << "Found usage of global variable \'" << usage.variableName << "\' in function \'"
               << usage.functionName << "\'";

    if (!inserted) {
        LOG_S(MAX) << "Skip it, as it has been already occurred";
        return;
    }

    auto &tests = (*parent->projectTests).at(sourceFilePath);
    auto &method = tests.methods[usage.functionName];
    const clang::QualType realParamType = varDecl->getType().getCanonicalType();
    const std::string usedParamTypeString = varDecl->getType().getAsString();
    types::Type paramType = types::Type(realParamType, usedParamTypeString, sourceManager);
    method.globalParams.emplace_back(paramType, usage.variableName, AlignmentFetcher::fetch(varDecl));
    if (!paramType.isPointerToFunction() && varDecl->isExternC() && !varDecl->hasDefinition()) {
        tests.externVariables.insert({paramType, usage.variableName});
    }
}

GlobalVariableUsageMatchCallback::Usage::Usage(std::string variableName, std::string functionName)
        : variableName(std::move(variableName)), functionName(std::move(functionName)) {
}

bool GlobalVariableUsageMatchCallback::Usage::operator==(
        const GlobalVariableUsageMatchCallback::Usage &rhs) const {
    return variableName == rhs.variableName && functionName == rhs.functionName;
}

bool GlobalVariableUsageMatchCallback::Usage::operator!=(
        const GlobalVariableUsageMatchCallback::Usage &rhs) const {
    return !(rhs == *this);
}

std::size_t GlobalVariableUsageMatchCallback::UsageHash::operator()(
        const GlobalVariableUsageMatchCallback::Usage &usage) const {
    size_t seed = 0;
    HashUtils::hashCombine(seed, usage.variableName, usage.functionName);
    return seed;
}
