using Xunit;
using System;
using System.Collections.Generic;
using FluentAssertions;
using System.IO;
using System.Linq;
using System.Security.Cryptography;
using System.Threading.Tasks;
using static TrueLayer.Signing.Tests.TestData;

namespace TrueLayer.Signing.Tests
{
    public class UsageTest
    {
        public static IEnumerable<object[]> TestCases = new[]
        {
            new TestCase(
                "Shared Test Key",
                Kid,
                PrivateKey,
                PublicKey),
            new TestCase(
                "Length Error Reproduction",
                BugReproduction.LengthError.Kid,
                BugReproduction.LengthError.PrivateKey,
                BugReproduction.LengthError.PublicKey),
        }.Select(x => new object[] { x });

        [Theory]
        [MemberData(nameof(TestCases))]
        public void SignAndVerify(TestCase testCase)
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000}";
            var idempotency_key = "idemp-2076717c-9005-4811-a321-9e0787fa0382";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";

            var tlSignature = Signer.SignWithPem(testCase.Kid, testCase.PrivateKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Sign();

            Verifier.VerifyWithPem(testCase.PublicKey)
                .Method("post") // case-insensitive: no troubles
                .Path(path)
                .Header("X-Whatever-2", "t2345d")
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Verify(tlSignature); // should not throw
        }

        [Theory]
        [MemberData(nameof(TestCases))]
        public void SignAndVerify_NoHeaders(TestCase testCase)
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000}";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";

            var tlSignature = Signer.SignWithPem(testCase.Kid, testCase.PrivateKey)
                .Method("POST")
                .Path(path)
                .Body(body)
                .Sign();

            Verifier.VerifyWithPem(testCase.PublicKey)
                .Method("POST")
                .Path(path)
                .Body(body)
                .Verify(tlSignature); // should not throw
        }

        // Signing a path with a single trailing slash & trying to verify
        // without that slash should still work. 
        // Verify a path that matches except it has an additional trailing slash
        // should still work too.
        // See #80.
        [Theory]
        [InlineData("/tl-webhook/", "/tl-webhook")]
        [InlineData("/tl-webhook", "/tl-webhook/")]
        public void SignAndVerify_SignedTrailingSlash(string signedPath, string verifyPath)
        {
            var body = "{\"foo\":\"bar\"}";

            var tlSignature = Signer.SignWithPem(Kid, PrivateKey)
                .Method("POST")
                .Path(signedPath)
                .Body(body)
                .Sign();

            Verifier.VerifyWithPem(PublicKey)
                .Method("POST")
                .Path(verifyPath)
                .Body(body)
                .Verify(tlSignature);
        }

        // Verify the a static signature used in all lang tests to ensure
        // cross-lang consistency and prevent regression.
        [Fact]
        public void VerifyStaticSignature()
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000,\"name\":\"Foo???\"}";
            var idempotency_key = "idemp-2076717c-9005-4811-a321-9e0787fa0382";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";
            var tlSignature = File.ReadAllText(TestResourcePath("tl-signature.txt")).Trim();

            Verifier.VerifyWithPem(PublicKey)
                .Method("POST")
                .Path(path)
                .Header("X-Whatever-2", "t2345d")
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Verify(tlSignature); // should not throw
        }

        [Fact]
        public void SignAndVerify_MethodMismatch()
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000}";
            var idempotency_key = "idemp-2076717c-9005-4811-a321-9e0787fa0382";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";

            var tlSignature = Signer.SignWithPem(Kid, PrivateKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Sign();

            Action verify = () => Verifier.VerifyWithPem(PublicKey)
                .Method("DELETE") // different
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Verify(tlSignature);

            verify.Should().Throw<SignatureException>();
        }

        [Fact]
        public void SignAndVerify_PathMismatch()
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000}";
            var idempotency_key = "idemp-2076717c-9005-4811-a321-9e0787fa0382";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";

            var tlSignature = Signer.SignWithPem(Kid, PrivateKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Sign();

            Action verify = () => Verifier.VerifyWithPem(PublicKey)
                .Method("POST")
                .Path("/merchant_accounts/67b5b1cf-1d0c-45d4-a2ea-61bdc044327c/sweeping") // different
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Verify(tlSignature);

            verify.Should().Throw<SignatureException>();
        }

        [Fact]
        public void SignAndVerify_HeaderMismatch()
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000}";
            var idempotency_key = "idemp-2076717c-9005-4811-a321-9e0787fa0382";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";

            var tlSignature = Signer.SignWithPem(Kid, PrivateKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Sign();

            Action verify = () => Verifier.VerifyWithPem(PublicKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", "something-else") // different
                .Body(body)
                .Verify(tlSignature);

            verify.Should().Throw<SignatureException>();
        }

        [Fact]
        public void SignAndVerify_BodyMismatch()
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000}";
            var idempotency_key = "idemp-2076717c-9005-4811-a321-9e0787fa0382";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";

            var tlSignature = Signer.SignWithPem(Kid, PrivateKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Sign();

            Action verify = () => Verifier.VerifyWithPem(PublicKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body("{\"currency\":\"GBP\",\"max_amount_in_minor\":5000001}") // different
                .Verify(tlSignature);

            verify.Should().Throw<SignatureException>();
        }

        [Fact]
        public void SignAndVerify_MissingSignatureHeader()
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000}";
            var idempotency_key = "idemp-2076717c-9005-4811-a321-9e0787fa0382";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";

            var tlSignature = Signer.SignWithPem(Kid, PrivateKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Sign();

            Action verify = () => Verifier.VerifyWithPem(PublicKey)
                .Method("POST")
                .Path(path)
                // missing Idempotency-Key
                .Body(body)
                .Verify(tlSignature);

            verify.Should().Throw<SignatureException>();
        }

        [Fact]
        public void SignAndVerify_RequiredHeaderMissingFromSignature()
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000}";
            var idempotency_key = "idemp-2076717c-9005-4811-a321-9e0787fa0382";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";

            var tlSignature = Signer.SignWithPem(Kid, PrivateKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Sign();

            Action verify = () => Verifier.VerifyWithPem(PublicKey)
                .Method("POST")
                .Path(path)
                .RequireHeader("X-Required") // missing from signature
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Verify(tlSignature);

            verify.Should().Throw<SignatureException>();
        }

        [Fact]
        public void SignAndVerify_RequiredHeaderCaseInsensitivity()
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000}";
            var idempotency_key = "idemp-2076717c-9005-4811-a321-9e0787fa0382";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";

            var tlSignature = Signer.SignWithPem(Kid, PrivateKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Sign();

            Verifier.VerifyWithPem(PublicKey)
                .Method("POST")
                .Path(path)
                .RequireHeader("IdEmPoTeNcY-KeY")
                .Header("iDeMpOtEnCy-kEy", idempotency_key)
                .Body(body)
                .Verify(tlSignature); // should not throw
        }

        [Fact]
        public void SignAndVerify_FlexibleHeaderCaseOrderVerify()
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000}";
            var idempotency_key = "idemp-2076717c-9005-4811-a321-9e0787fa0382";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";

            var tlSignature = Signer.SignWithPem(Kid, PrivateKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Header("X-Custom", "123")
                .Body(body)
                .Sign();

            Verifier.VerifyWithPem(PublicKey)
                .Method("POST")
                .Path(path)
                .Header("X-CUSTOM", "123") // different order & case, it's ok!
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Verify(tlSignature);
        }

        [Fact]
        public void Verifier_ExtractKid()
        {
            var tlSignature = Signer.SignWithPem(Kid, PrivateKey)
                .Method("delete")
                .Path("/foo")
                .Header("X-Custom", "123")
                .Sign();

            Verifier.ExtractKid(tlSignature).Should().Be(Kid);
        }

        [Fact]
        public void VerifierExtractKid_FromInvalidSignature_ShouldThrowSignatureException()
        {
            Action action = () => { Verifier.ExtractKid("an-invalid..signature"); };
            action
                .Should()
                .Throw<SignatureException>()
// exception message changed between version 2.2 and 3.1 due to a migration from Newtonsoft.Json (Json.NET) to System.Text.Json
#if (NETCOREAPP3_1 || NETCOREAPP3_1_OR_GREATER)
                .WithMessage("Failed to parse JWS: 'j' is an invalid start of a value. Path: $ | LineNumber: 0 | BytePositionInLine: 0.");
#else
                .WithMessage("Failed to parse JWS: Unexpected character encountered while parsing value: j. Path '', line 0, position 0.");
#endif
        }

        [Fact]
        public void Verifier_ExtractJku()
        {
            var tlSignature = File.ReadAllText(TestResourcePath("webhook-signature.txt")).Trim();
            Verifier.ExtractJku(tlSignature).Should().Be("https://webhooks.truelayer.com/.well-known/jwks");
        }

        [Fact]
        public void VerifierExtractJku_FromInvalidSignature_ShouldThrowSignatureException()
        {
            Action action = () => { Verifier.ExtractJku("an-invalid..signature"); };
            action
                .Should()
                .Throw<SignatureException>()
// exception message changed between version 2.2 and 3.1 due to a migration from Newtonsoft.Json (Json.NET) to System.Text.Json
#if (NETCOREAPP3_1 || NETCOREAPP3_1_OR_GREATER)
                .WithMessage("Failed to parse JWS: 'j' is an invalid start of a value. Path: $ | LineNumber: 0 | BytePositionInLine: 0.");
#else
                .WithMessage("Failed to parse JWS: Unexpected character encountered while parsing value: j. Path '', line 0, position 0.");
#endif
        }

        [Fact]
        public void Verifier_Jwks()
        {
            var tlSignature = File.ReadAllText(TestResourcePath("webhook-signature.txt")).Trim();
            var jwks = File.ReadAllText(TestResourcePath("jwks.json"));

            Verifier.VerifyWithJwks(jwks)
                .Method("POST")
                .Path("/tl-webhook")
                .Header("x-tl-webhook-timestamp", "2021-11-29T11:42:55Z")
                .Header("content-type", "application/json")
                .Body("{\"event_type\":\"example\",\"event_id\":\"18b2842b-a57b-4887-a0a6-d3c7c36f1020\"}")
                .Verify(tlSignature); // should not throw

            Action verify = () => Verifier.VerifyWithJwks(jwks)
                .Method("POST")
                .Path("/tl-webhook")
                .Header("x-tl-webhook-timestamp", "2021-12-02T14:18:00Z") // different
                .Header("content-type", "application/json")
                .Body("{\"event_type\":\"example\",\"event_id\":\"18b2842b-a57b-4887-a0a6-d3c7c36f1020\"}")
                .Verify(tlSignature);

            verify.Should().Throw<SignatureException>();
        }

        [Fact]
        public void VerifierVerify_InvalidSignature_ShouldThrowSignatureException()
        {
            Action action = () => Verifier.VerifyWithPem(PublicKey)
                .Method("POST")
                .Path("/bar")
                .Body("{}")
                .Verify("an-invalid..signature");

            action
                .Should()
                .Throw<SignatureException>()
// exception message changed between version 2.2 and 3.1 due to a migration from Newtonsoft.Json (Json.NET) to System.Text.Json
#if (NETCOREAPP3_1 || NETCOREAPP3_1_OR_GREATER)
                .WithMessage("Failed to parse JWS: 'j' is an invalid start of a value. Path: $ | LineNumber: 0 | BytePositionInLine: 0.");
#else
                .WithMessage("Failed to parse JWS: Unexpected character encountered while parsing value: j. Path '', line 0, position 0.");
#endif
        }

        [Theory]
        [MemberData(nameof(TestCases))]
        public async Task SignAndVerify_AsyncFunction(TestCase testCase)
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":5000000}";
            var idempotency_key = "idemp-2076717c-9005-4811-a321-9e0787fa0382";
            var path = "/merchant_accounts/a61acaef-ee05-4077-92f3-25543a11bd8d/sweeping";

            Func<string, Task<string>> signingFunction = payload =>
            {
                var privateKey = Util.ParsePem(testCase.PrivateKey);
                var payloadBytes = System.Text.Encoding.UTF8.GetBytes(payload);
                var signatureBytes = privateKey.SignData(payloadBytes, HashAlgorithmName.SHA512);
                return Task.FromResult(Convert.ToBase64String(signatureBytes));
            };
            
            var tlSignature = await Signer.SignWithFunction(testCase.Kid, signingFunction)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .SignAsync();

            Verifier.VerifyWithPem(testCase.PublicKey)
                .Method("post") // case-insensitive: no troubles
                .Path(path)
                .Header("X-Whatever-2", "t2345d")
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Verify(tlSignature); // should not throw
        }

        [Theory]
        [MemberData(nameof(TestCases))]
        public void SignAndVerify_WithJku(TestCase testCase)
        {
            var body = "{\"currency\":\"GBP\",\"max_amount_in_minor\":1}";
            var idempotency_key = $"idemp-{Guid.NewGuid()}";
            var path = "/payments";
            var jku = $"https://{Guid.NewGuid()}.com/.well-known/jwks";

            var tlSignature = Signer.SignWithPem(testCase.Kid, testCase.PrivateKey)
                .Method("POST")
                .Path(path)
                .Header("Idempotency-Key", idempotency_key)
                .Body(body)
                .Jku(jku)
                .Sign();

            Verifier.VerifyWithPem(testCase.PublicKey)
                .Method("post")
                .Path(path)
                .Header("X-Extra-1", "qwerty")
                .Header("IDEMPOTENCY-KEY", idempotency_key)
                .Body(body)
                .Verify(tlSignature); // should not throw

            var signatureJku = Verifier.ExtractJku(tlSignature);
            signatureJku.Should().Be(jku);
        }
        
        public sealed class TestCase
        {
            public TestCase(string name, string kid, string privateKey, string publicKey)
            {
                Name = name;
                Kid = kid;
                PrivateKey = privateKey;
                PublicKey = publicKey;
            }

            private string Name { get; }
            public string Kid { get; }
            public string PrivateKey { get; }
            public string PublicKey { get; }

            public override string ToString() => Name;
        }
    }
}
