import { useState } from 'react';
import Table from '@mui/material/Table';

import TableBody from '@mui/material/TableBody';
import TableRow from '@mui/material/TableRow';
import TableCell from '@mui/material/TableCell';
import TableContainer from '@mui/material/TableContainer';
import TableHead from '@mui/material/TableHead';
import TablePagination from '@mui/material/TablePagination';
import TableSortLabel from '@mui/material/TableSortLabel';

function descendingComparator(a, b, orderBy) {
	if (b[orderBy] < a[orderBy]) {
		return -1;
	}
	if (b[orderBy] > a[orderBy]) {
		return 1;
	}
	return 0;
}

function getComparator(order, orderBy) {
	return order === 'desc'
		? (a, b) => descendingComparator(a, b, orderBy)
		: (a, b) => -descendingComparator(a, b, orderBy);
}

function EnhancedTableHead(props) {
	const { order, orderBy, onRequestSort, headCells } = props;
	const createSortHandler = (property) => (event) => {
		onRequestSort(event, property);
	};

	return (
		<TableHead sx={{ bgcolor: 'background.default' }}>
			<TableRow>
				{headCells.map((headCell, i) => (
					<TableCell
						key={headCell.id || i}
						align={headCell.numeric ? 'right' : 'left'}
						padding={headCell.disablePadding ? 'none' : 'normal'}
						sortDirection={orderBy === headCell.id ? order : false}
					>
						<TableSortLabel
							active={orderBy === headCell.id}
							direction={orderBy === headCell.id ? order : 'asc'}
							onClick={createSortHandler(headCell.id)}
						>
							{headCell.label}
							{orderBy === headCell.id ? (
								<span
									style={{
										fontSize: '10px',
									}}
								>
									{order === 'desc' ? '•Decendente' : '•Ascendente'}
								</span>
							) : null}
						</TableSortLabel>
					</TableCell>
				))}
			</TableRow>
		</TableHead>
	);
}

/**
 * @param {object} props
 * @param {Array} props.rows
 * @param {Function} props.render
 * @param {boolean} props.dense
 * @param {Object} props.emptyRowsHeight
 * @param {Boolean} props.stickyHeader
 * @param {object} props.tableContainerProps
 */
function EnhancedTable(props) {
	const {
		rows,
		headCells,
		render,
		dense = false,
		emptyRowsHeight = { default: 76, dense: 43 },
		stickyHeader,
		tableContainerProps,
	} = props;
	const [order, setOrder] = useState('desc');
	const [orderBy, setOrderBy] = useState('');
	const [page, setPage] = useState(0);
	const [rowsPerPage, setRowsPerPage] = useState(5);

	const handleRequestSort = (event, property) => {
		const isAsc = orderBy === property && order === 'asc';
		setOrder(isAsc ? 'desc' : 'asc');
		setOrderBy(property);
	};

	const handleChangePage = (event, newPage) => {
		setPage(newPage);
	};

	const handleChangeRowsPerPage = (event) => {
		setRowsPerPage(parseInt(event.target.value, 10));
		setPage(0);
	};

	// Avoid a layout jump when reaching the last page with empty rows.
	const emptyRows = page > 0 ? Math.max(0, (1 + page) * rowsPerPage - rows.length) : 0;

	return (
		<>
			<TableContainer {...tableContainerProps}>
				<Table
					sx={{ width: '100%' }}
					stickyHeader={stickyHeader}
					aria-labelledby="tableTitle"
					size={dense ? 'small' : 'medium'}
				>
					<EnhancedTableHead
						order={order}
						orderBy={orderBy}
						onRequestSort={handleRequestSort}
						rowCount={rows.length}
						headCells={headCells}
					/>
					<TableBody>
						{rows
							.slice()
							.sort(getComparator(order, orderBy))
							.slice(page * rowsPerPage, page * rowsPerPage + rowsPerPage)
							.map((row, i) => render(row, i))}
						{emptyRows > 0 && (
							<TableRow
								style={{
									height: (dense ? emptyRowsHeight.dense : emptyRowsHeight.default) * emptyRows,
								}}
							>
								<TableCell colSpan="100%" />
							</TableRow>
						)}
					</TableBody>
				</Table>
			</TableContainer>
			<TablePagination
				rowsPerPageOptions={[5, 10, 25]}
				component="div"
				count={rows.length}
				rowsPerPage={rowsPerPage}
				page={page}
				onPageChange={handleChangePage}
				onRowsPerPageChange={handleChangeRowsPerPage}
				labelRowsPerPage="Resultados por Pagina"
			/>
		</>
	);
}

export default EnhancedTable;
