//////////////////////////////////////////////////////////////////////
/// \class TreeViewItems
/// \author Timo "TimoSoft" Kunze
/// \brief <em>Manages a collection of \c TreeViewItem objects</em>
///
/// This class provides easy access (including filtering) to collections of \c TreeViewItem
/// objects. While a \c TreeViewItemContainer object is used to group any items and acts more like
/// a clipboard, a \c TreeViewItems object is used to group items that have certain properties in
/// common.
///
/// \if UNICODE
///   \sa ExTVwLibU::ITreeViewItems, TreeViewItem, TreeViewItemContainer, ExplorerTreeView
/// \else
///   \sa ExTVwLibA::ITreeViewItems, TreeViewItem, TreeViewItemContainer, ExplorerTreeView
/// \endif
//////////////////////////////////////////////////////////////////////


#pragma once

#include "res/resource.h"
#ifdef UNICODE
	#include "ExTVwU.h"
#else
	#include "ExTVwA.h"
#endif
#include "_ITreeViewItemsEvents_CP.h"
#include "helpers.h"
#include "ExplorerTreeView.h"
#include "TreeViewItem.h"


class ATL_NO_VTABLE TreeViewItems : 
    public CComObjectRootEx<CComSingleThreadModel>,
    public CComCoClass<TreeViewItems, &CLSID_TreeViewItems>,
    public ISupportErrorInfo,
    public IConnectionPointContainerImpl<TreeViewItems>,
    public Proxy_ITreeViewItemsEvents<TreeViewItems>,
    public IEnumVARIANT,
    #ifdef UNICODE
    	public IDispatchImpl<ITreeViewItems, &IID_ITreeViewItems, &LIBID_ExTVwLibU, /*wMajor =*/ VERSION_MAJOR, /*wMinor =*/ VERSION_MINOR>
    #else
    	public IDispatchImpl<ITreeViewItems, &IID_ITreeViewItems, &LIBID_ExTVwLibA, /*wMajor =*/ VERSION_MAJOR, /*wMinor =*/ VERSION_MINOR>
    #endif
{
	friend class ExplorerTreeView;
	friend class ClassFactory;

public:
	#ifndef DOXYGEN_SHOULD_SKIP_THIS
		DECLARE_REGISTRY_RESOURCEID(IDR_TREEVIEWITEMS)

		BEGIN_COM_MAP(TreeViewItems)
			COM_INTERFACE_ENTRY(ITreeViewItems)
			COM_INTERFACE_ENTRY(IDispatch)
			COM_INTERFACE_ENTRY(ISupportErrorInfo)
			COM_INTERFACE_ENTRY(IConnectionPointContainer)
			COM_INTERFACE_ENTRY(IEnumVARIANT)
		END_COM_MAP()

		BEGIN_CONNECTION_POINT_MAP(TreeViewItems)
			CONNECTION_POINT_ENTRY(__uuidof(_ITreeViewItemsEvents))
		END_CONNECTION_POINT_MAP()

		DECLARE_PROTECT_FINAL_CONSTRUCT()
	#endif

	//////////////////////////////////////////////////////////////////////
	/// \name Implementation of ISupportErrorInfo
	///
	//@{
	/// \brief <em>Retrieves whether an interface supports the \c IErrorInfo interface</em>
	///
	/// \param[in] interfaceToCheck The IID of the interface to check.
	///
	/// \return \c S_OK if the interface identified by \c interfaceToCheck supports \c IErrorInfo;
	///         otherwise \c S_FALSE.
	///
	/// \sa <a href="https://msdn.microsoft.com/en-us/library/ms221233.aspx">IErrorInfo</a>
	virtual HRESULT STDMETHODCALLTYPE InterfaceSupportsErrorInfo(REFIID interfaceToCheck);
	//@}
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	/// \name Implementation of IEnumVARIANT
	///
	//@{
	/// \brief <em>Clones the \c VARIANT iterator used to iterate the items</em>
	///
	/// Clones the \c VARIANT iterator including its current state. This iterator is used to iterate
	/// the \c TreeViewItem objects managed by this collection object.
	///
	/// \param[out] ppEnumerator Receives the clone's \c IEnumVARIANT implementation.
	///
	/// \return An \c HRESULT error code.
	///
	/// \sa Next, Reset, Skip,
	///     <a href="https://msdn.microsoft.com/en-us/library/ms221053.aspx">IEnumVARIANT</a>,
	///     <a href="https://msdn.microsoft.com/en-us/library/ms690336.aspx">IEnumXXXX::Clone</a>
	virtual HRESULT STDMETHODCALLTYPE Clone(IEnumVARIANT** ppEnumerator);
	/// \brief <em>Retrieves the next x items</em>
	///
	/// Retrieves the next \c numberOfMaxItems items from the iterator.
	///
	/// \param[in] numberOfMaxItems The maximum number of items the array identified by \c pItems can
	///            contain.
	/// \param[in,out] pItems An array of \c VARIANT values. On return, each \c VARIANT will contain
	///                the pointer to an item's \c ITreeViewItem implementation.
	/// \param[out] pNumberOfItemsReturned The number of items that actually were copied to the array
	///             identified by \c pItems.
	///
	/// \return An \c HRESULT error code.
	///
	/// \sa Clone, Reset, Skip, TreeViewItem,
	///     <a href="https://msdn.microsoft.com/en-us/library/ms695273.aspx">IEnumXXXX::Next</a>
	virtual HRESULT STDMETHODCALLTYPE Next(ULONG numberOfMaxItems, VARIANT* pItems, ULONG* pNumberOfItemsReturned);
	/// \brief <em>Resets the \c VARIANT iterator</em>
	///
	/// Resets the \c VARIANT iterator so that the next call of \c Next or \c Skip starts at the first
	/// item in the collection.
	///
	/// \return An \c HRESULT error code.
	///
	/// \sa Clone, Next, Skip,
	///     <a href="https://msdn.microsoft.com/en-us/library/ms693414.aspx">IEnumXXXX::Reset</a>
	virtual HRESULT STDMETHODCALLTYPE Reset(void);
	/// \brief <em>Skips the next x items</em>
	///
	/// Instructs the \c VARIANT iterator to skip the next \c numberOfItemsToSkip items.
	///
	/// \param[in] numberOfItemsToSkip The number of items to skip.
	///
	/// \return An \c HRESULT error code.
	///
	/// \sa Clone, Next, Reset,
	///     <a href="https://msdn.microsoft.com/en-us/library/ms690392.aspx">IEnumXXXX::Skip</a>
	virtual HRESULT STDMETHODCALLTYPE Skip(ULONG numberOfItemsToSkip);
	//@}
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	/// \name Implementation of ITreeViewItems
	///
	//@{
	/// \brief <em>Retrieves the current setting of the \c CaseSensitiveFilters property</em>
	///
	/// Retrieves whether string comparisons, that are done when applying the filters on an item, are case
	/// sensitive. If this property is set to \c VARIANT_TRUE, string comparisons are case sensitive;
	/// otherwise not.
	///
	/// \param[out] pValue The property's current setting.
	///
	/// \return An \c HRESULT error code.
	///
	/// \sa put_CaseSensitiveFilters, get_Filter, get_ComparisonFunction
	virtual HRESULT STDMETHODCALLTYPE get_CaseSensitiveFilters(VARIANT_BOOL* pValue);
	/// \brief <em>Sets the \c CaseSensitiveFilters property</em>
	///
	/// Sets whether string comparisons, that are done when applying the filters on an item, are case
	/// sensitive. If this property is set to \c VARIANT_TRUE, string comparisons are case sensitive;
	/// otherwise not.
	///
	/// \param[in] newValue The setting to apply.
	///
	/// \return An \c HRESULT error code.
	///
	/// \sa get_CaseSensitiveFilters, put_Filter, put_ComparisonFunction
	virtual HRESULT STDMETHODCALLTYPE put_CaseSensitiveFilters(VARIANT_BOOL newValue);
	/// \brief <em>Retrieves the current setting of the \c ComparisonFunction property</em>
	///
	/// Retrieves an item filter's comparison function. This property takes the address of a function
	/// having the following signature:\n
	/// \code
	///   BOOL IsEqual(T itemProperty, T pattern);
	/// \endcode
	/// where T stands for the filtered property's type (\c VARIANT_BOOL, \c LONG, \c BSTR or
	/// \c ITreeViewItem). This function must compare its arguments and return a non-zero value if the
	/// arguments are equal and zero otherwise.\n
	/// If this property is set to 0, the control compares the values itself using the "==" operator
	/// (\c lstrcmp and \c lstrcmpi for string filters).
	///
	/// \param[in] filteredProperty A value specifying the property that the filter refers to. Any of the
	///            values defined by the \c FilteredPropertyConstants enumeration is valid.
	/// \param[out] pValue The property's current setting.
	///
	/// \return An \c HRESULT error code.
	///
	/// \if UNICODE
	///   \sa put_ComparisonFunction, get_Filter, get_CaseSensitiveFilters,
	///       ExTVwLibU::FilteredPropertyConstants,
	///       <a href="https://msdn.microsoft.com/en-us/library/ms647488.aspx">lstrcmp</a>,
	///       <a href="https://msdn.microsoft.com/en-us/library/ms647489.aspx">lstrcmpi</a>
	/// \else
	///   \sa put_ComparisonFunction, get_Filter, get_CaseSensitiveFilters,
	///       ExTVwLibA::FilteredPropertyConstants,
	///       <a href="https://msdn.microsoft.com/en-us/library/ms647488.aspx">lstrcmp</a>,
	///       <a href="https://msdn.microsoft.com/en-us/library/ms647489.aspx">lstrcmpi</a>
	/// \endif
	virtual HRESULT STDMETHODCALLTYPE get_ComparisonFunction(FilteredPropertyConstants filteredProperty, LONG* pValue);
	/// \brief <em>Sets the \c ComparisonFunction property</em>
	///
	/// Sets an item filter's comparison function. This property takes the address of a function
	/// having the following signature:\n
	/// \code
	///   BOOL IsEqual(T itemProperty, T pattern);
	/// \endcode
	/// where T stands for the filtered property's type (\c VARIANT_BOOL, \c LONG, \c BSTR or
	/// \c ITreeViewItem). This function must compare its arguments and return a non-zero value if the
	/// arguments are equal and zero otherwise.\n
	/// If this property is set to 0, the control compares the values itself using the "==" operator
	/// (\c lstrcmp and \c lstrcmpi for string filters).
	///
	/// \param[in] filteredProperty A value specifying the property that the filter refers to. Any of the
	///            values defined by the \c FilteredPropertyConstants enumeration is valid.
	/// \param[in] newValue The setting to apply.
	///
	/// \return An \c HRESULT error code.
	///
	/// \if UNICODE
	///   \sa get_ComparisonFunction, put_Filter, put_CaseSensitiveFilters,
	///       ExTVwLibU::FilteredPropertyConstants,
	///       <a href="https://msdn.microsoft.com/en-us/library/ms647488.aspx">lstrcmp</a>,
	///       <a href="https://msdn.microsoft.com/en-us/library/ms647489.aspx">lstrcmpi</a>
	/// \else
	///   \sa get_ComparisonFunction, put_Filter, put_CaseSensitiveFilters,
	///       ExTVwLibA::FilteredPropertyConstants,
	///       <a href="https://msdn.microsoft.com/en-us/library/ms647488.aspx">lstrcmp</a>,
	///       <a href="https://msdn.microsoft.com/en-us/library/ms647489.aspx">lstrcmpi</a>
	/// \endif
	virtual HRESULT STDMETHODCALLTYPE put_ComparisonFunction(FilteredPropertyConstants filteredProperty, LONG newValue);
	/// \brief <em>Retrieves the current setting of the \c Filter property</em>
	///
	/// Retrieves an item filter.\n
	/// An \c ITreeViewItems collection can be filtered by any of \c ITreeViewItem's properties, that
	/// the \c FilteredPropertyConstants enumeration defines a constant for. Combinations of multiple
	/// filters are possible, too. A filter is a \c VARIANT containing an array whose elements are of
	/// type \c VARIANT. Each element of this array contains a valid value for the property, that the
	/// filter refers to.\n
	/// When applying the filter, the elements of the array are connected using the logical Or operator.\n\n
	/// Setting this property to \c Empty or any other value, that doesn't match the described structure,
	/// deactivates the filter.
	///
	/// \param[in] filteredProperty A value specifying the property that the filter refers to. Any of the
	///            values defined by the \c FilteredPropertyConstants enumeration is valid.
	/// \param[out] pValue The property's current setting.
	///
	/// \return An \c HRESULT error code.
	///
	/// \if UNICODE
	///   \sa put_Filter, get_FilterType, get_ComparisonFunction, ExTVwLibU::FilteredPropertyConstants
	/// \else
	///   \sa put_Filter, get_FilterType, get_ComparisonFunction, ExTVwLibA::FilteredPropertyConstants
	/// \endif
	virtual HRESULT STDMETHODCALLTYPE get_Filter(FilteredPropertyConstants filteredProperty, VARIANT* pValue);
	/// \brief <em>Sets the \c Filter property</em>
	///
	/// Sets an item filter.\n
	/// An \c ITreeViewItems collection can be filtered by any of \c ITreeViewItem's properties, that
	/// the \c FilteredPropertyConstants enumeration defines a constant for. Combinations of multiple
	/// filters are possible, too. A filter is a \c VARIANT containing an array whose elements are of
	/// type \c VARIANT. Each element of this array contains a valid value for the property, that the
	/// filter refers to.\n
	/// When applying the filter, the elements of the array are connected using the logical Or operator.\n\n
	/// Setting this property to \c Empty or any other value, that doesn't match the described structure,
	/// deactivates the filter.
	///
	/// \param[in] filteredProperty A value specifying the property that the filter refers to. Any of the
	///            values defined by the \c FilteredPropertyConstants enumeration is valid.
	/// \param[in] newValue The setting to apply.
	///
	/// \return An \c HRESULT error code.
	///
	/// \if UNICODE
	///   \sa get_Filter, put_FilterType, put_ComparisonFunction, IsPartOfCollection,
	///       ExTVwLibU::FilteredPropertyConstants
	/// \else
	///   \sa get_Filter, put_FilterType, put_ComparisonFunction, IsPartOfCollection,
	///       ExTVwLibA::FilteredPropertyConstants
	/// \endif
	virtual HRESULT STDMETHODCALLTYPE put_Filter(FilteredPropertyConstants filteredProperty, VARIANT newValue);
	/// \brief <em>Retrieves the current setting of the \c FilterType property</em>
	///
	/// Retrieves an item filter's type.
	///
	/// \param[in] filteredProperty A value specifying the property that the filter refers to. Any of the
	///            values defined by the \c FilteredPropertyConstants enumeration is valid.
	/// \param[out] pValue The property's current setting.
	///
	/// \return An \c HRESULT error code.
	///
	/// \if UNICODE
	///   \sa put_FilterType, get_Filter, ExTVwLibU::FilteredPropertyConstants,
	///       ExTVwLibU::FilterTypeConstants
	/// \else
	///   \sa put_FilterType, get_Filter, ExTVwLibA::FilteredPropertyConstants,
	///       ExTVwLibA::FilterTypeConstants
	/// \endif
	virtual HRESULT STDMETHODCALLTYPE get_FilterType(FilteredPropertyConstants filteredProperty, FilterTypeConstants* pValue);
	/// \brief <em>Sets the \c FilterType property</em>
	///
	/// Sets an item filter's type.
	///
	/// \param[in] filteredProperty A value specifying the property that the filter refers to. Any of the
	///            values defined by the \c FilteredPropertyConstants enumeration is valid.
	/// \param[in] newValue The setting to apply.
	///
	/// \return An \c HRESULT error code.
	///
	/// \if UNICODE
	///   \sa get_FilterType, put_Filter, IsPartOfCollection, ExTVwLibU::FilteredPropertyConstants,
	///       ExTVwLibU::FilterTypeConstants
	/// \else
	///   \sa get_FilterType, put_Filter, IsPartOfCollection, ExTVwLibA::FilteredPropertyConstants,
	///       ExTVwLibA::FilterTypeConstants
	/// \endif
	virtual HRESULT STDMETHODCALLTYPE put_FilterType(FilteredPropertyConstants filteredProperty, FilterTypeConstants newValue);
	/// \brief <em>Retrieves a \c TreeViewItem object from the collection</em>
	///
	/// Retrieves a \c TreeViewItem object from the collection that wraps the treeview item identified
	/// by \c itemIdentifier.
	///
	/// \param[in] itemIdentifier A value that identifies the treeview item to be retrieved.
	/// \param[in] itemIdentifierType A value specifying the meaning of \c itemIdentifier. Any of the
	///            values defined by the \c ItemIdentifierTypeConstants enumeration is valid.
	/// \param[out] ppItem Receives the item's \c ITreeViewItem implementation.
	///
	/// \return An \c HRESULT error code.
	///
	/// \remarks This property is read-only.
	///
	/// \if UNICODE
	///   \sa TreeViewItem, Add, Remove, Contains, ExTVwLibU::ItemIdentifierTypeConstants
	/// \else
	///   \sa TreeViewItem, Add, Remove, Contains, ExTVwLibA::ItemIdentifierTypeConstants
	/// \endif
	virtual HRESULT STDMETHODCALLTYPE get_Item(LONG itemIdentifier, ItemIdentifierTypeConstants itemIdentifierType = iitHandle, ITreeViewItem** ppItem = NULL);
	/// \brief <em>Retrieves a \c VARIANT enumerator</em>
	///
	/// Retrieves a \c VARIANT enumerator that may be used to iterate the \c TreeViewItem objects
	/// managed by this collection object. This iterator is used by Visual Basic's \c For...Each
	/// construct.
	///
	/// \param[out] ppEnumerator Receives the iterator's \c IEnumVARIANT implementation.
	///
	/// \return An \c HRESULT error code.
	///
	/// \remarks This property is read-only and hidden.
	///
	/// \sa <a href="https://msdn.microsoft.com/en-us/library/ms221053.aspx">IEnumVARIANT</a>
	virtual HRESULT STDMETHODCALLTYPE get__NewEnum(IUnknown** ppEnumerator);
	/// \brief <em>Retrieves all items' parent item</em>
	///
	/// Retrieves a \c TreeViewItem object wrapping the treeview item that is the immediate parent item
	/// of all items in the collection.
	///
	/// \param[out] ppParentItem Receives the parent item's \c ITreeViewItem implementation.
	///
	/// \return An \c HRESULT error code.
	///
	/// \remarks The collection must be restricted to a single branch and level (i. e. it's a collection
	///          retrieved through the \c ExplorerTreeView::get_RootItems or \c TreeViewItem::get_SubItems
	///          property). Otherwise retrieving the parent item will fail.
	///
	/// \sa putref_ParentItem, get_SingleLevel, ExplorerTreeView::get_RootItems, TreeViewItem::get_SubItems
	virtual HRESULT STDMETHODCALLTYPE get_ParentItem(ITreeViewItem** ppParentItem);
	/// \brief <em>Sets all items' parent item</em>
	///
	/// Moves all \c TreeViewItem objects in the collection so that they become sub-items of the item
	/// identified by \c pNewParentItem.
	///
	/// \param[in] pNewParentItem The new parent item.
	///
	/// \return An \c HRESULT error code.
	///
	/// \remarks The collection must be restricted to a single branch and level (i. e. it's a collection
	///          retrieved through the \c ExplorerTreeView::get_RootItems or \c TreeViewItem::get_SubItems
	///          property). Otherwise changing the parent item will fail.
	///
	/// \sa get_ParentItem, get_SingleLevel, ExplorerTreeView::get_RootItems, TreeViewItem::get_SubItems
	virtual HRESULT STDMETHODCALLTYPE putref_ParentItem(ITreeViewItem* pNewParentItem);
	/// \brief <em>Retrieves the current setting of the \c SingleLevel property</em>
	///
	/// Retrieves whether the collection recursively dives into sub-branches while enumerating the items.
	/// If this property is set to \c VARIANT_FALSE, the collection dives into sub-branches; otherwise not.
	///
	/// \param[out] pValue The property's current setting.
	///
	/// \return An \c HRESULT error code.
	///
	/// \remarks This property is read-only.
	///
	/// \sa get_ParentItem, get_Filter, ExplorerTreeView::get_RootItems, TreeViewItem::get_SubItems
	virtual HRESULT STDMETHODCALLTYPE get_SingleLevel(VARIANT_BOOL* pValue);

	/// \brief <em>Adds an item to the treeview</em>
	///
	/// Adds an item with the specified properties at the specified position in the control and returns a
	/// \c TreeViewItem object wrapping the inserted item.
	///
	/// \param[in] itemText The new item's caption text. The maximum number of characters in this text
	///            is \c MAX_ITEMTEXTLENGTH. If set to \c NULL, the control will fire the
	///            \c ItemGetDisplayInfo event each time this property's value is required.
	/// \param[in] pParentItem The new item's immediate parent item. If set to \c NULL, the item will be
	///            a top-level item.
	/// \param[in] insertAfter A \c VARIANT value specifying the new item's preceding item. This may
	///            be either a \c TreeViewItem object or a handle returned by \c TreeViewItem::get_Handle
	///            or a value defined by the \c InsertAfterConstants enumeration. If set to \c Empty,
	///            the item will be inserted after the last item that has the same immediate parent item.
	/// \param[in] hasExpando A value specifying whether to draw a "+" or "-" next to the item
	///            indicating the item has sub-items. Any of the values defined by the \c HasExpandoConstants
	///            enumeration is valid. If set to \c heCallback, the control will fire the
	///            \c ItemGetDisplayInfo event each time this property's value is required.
	/// \param[in] iconIndex The zero-based index of the item's icon in the control's \c ilItems imagelist.
	///            If set to -1, the control will fire the \c ItemGetDisplayInfo event each time this
	///            property's value is required. A value of -2 means 'not specified' and is valid if there's
	///            no imagelist associated with the control.
	/// \param[in] selectedIconIndex The zero-based index of the item's selected icon in the control's
	///            \c ilItems imagelist. This icon will be used instead of the normal icon identified by
	///            \c iconIndex if the item is the caret item. If set to -1, the control will fire the
	///            \c ItemGetDisplayInfo event each time this property's value is required. If set to -2, the
	///            normal icon specified by \c iconIndex will be used.
	/// \param[in] expandedIconIndex The zero-based index of the item's expanded icon in the control's
	///            \c ilItems imagelist. This icon will be used instead of the normal icon identified by
	///            \c iconIndex if the item is expanded. If set to -1, the control will fire the
	///            \c ItemGetDisplayInfo event each time this property's value is required. If set to -2, the
	///            normal icon specified by \c iconIndex will be used.
	/// \param[in] itemData A \c LONG value that will be associated with the item.
	/// \param[in] isVirtual If set to \c VARIANT_TRUE, the item will be treated as not existent when drawing
	///            the treeview and therefore won't be drawn. Instead its sub-items will be drawn at this
	///            item's position. If set to \c VARIANT_FALSE, the item and its sub-items will be drawn
	///            normally.
	/// \param[in] heightIncrement The item's height in multiples of the control's basic item height. E. g.
	///            a value of 2 means that the item will be twice as high as an item with \c HeighIncrement
	///            set to 1.
	/// \param[out] ppAddedItem Receives the added item's \c ITreeViewItem implementation.
	///
	/// \return An \c HRESULT error code.
	///
	/// \remarks The \c expandedIconIndex and \c isVirtual parameters will be ignored if comctl32.dll is used
	///          in a version older than 6.10.
	///
	/// \if UNICODE
	///   \sa Count, Remove, RemoveAll, TreeViewItem::get_Text, TreeViewItem::get_ParentItem,
	///       TreeViewItem::get_PreviousSiblingItem, TreeViewItem::get_HasExpando,
	///       TreeViewItem::get_IconIndex, TreeViewItem::get_SelectedIconIndex,
	///       TreeViewItem::get_ExpandedIconIndex, TreeViewItem::get_ItemData,
	///       TreeViewItem::get_Virtual, TreeViewItem::get_HeightIncrement,
	///       ExplorerTreeView::get_hImageList, ExplorerTreeView::get_ItemHeight,
	///       ExplorerTreeView::Raise_ItemGetDisplayInfo, ExTVwLibU::InsertAfterConstants,
	///       ExTVwLibU::HasExpandoConstants, ExTVwLibU::ImageListConstants, MAX_ITEMTEXTLENGTH
	/// \else
	///   \sa Count, Remove, RemoveAll, TreeViewItem::get_Text, TreeViewItem::get_ParentItem,
	///       TreeViewItem::get_PreviousSiblingItem, TreeViewItem::get_HasExpando,
	///       TreeViewItem::get_IconIndex, TreeViewItem::get_SelectedIconIndex,
	///       TreeViewItem::get_ExpandedIconIndex, TreeViewItem::get_ItemData,
	///       TreeViewItem::get_Virtual, TreeViewItem::get_HeightIncrement,
	///       ExplorerTreeView::get_hImageList, ExplorerTreeView::get_ItemHeight,
	///       ExplorerTreeView::Raise_ItemGetDisplayInfo, ExTVwLibA::InsertAfterConstants,
	///       ExTVwLibA::HasExpandoConstants, ExTVwLibA::ImageListConstants, MAX_ITEMTEXTLENGTH
	/// \endif
	virtual HRESULT STDMETHODCALLTYPE Add(BSTR itemText, ITreeViewItem* pParentItem = NULL, VARIANT insertAfter = _variant_t(DISP_E_PARAMNOTFOUND, VT_ERROR), HasExpandoConstants hasExpando = heNo, LONG iconIndex = -2, LONG selectedIconIndex = -2, LONG expandedIconIndex = -2, LONG itemData = 0, VARIANT_BOOL isVirtual = VARIANT_FALSE, LONG heightIncrement = 1, ITreeViewItem** ppAddedItem = NULL);
	/// \brief <em>Retrieves whether the specified item is part of the item collection</em>
	///
	/// \param[in] itemIdentifier A value that identifies the item to be checked.
	/// \param[in] itemIdentifierType A value specifying the meaning of \c itemIdentifier. Any of the
	///            values defined by the \c ItemIdentifierTypeConstants enumeration is valid.
	/// \param[out] pValue \c VARIANT_TRUE, if the item is part of the collection; otherwise
	///             \c VARIANT_FALSE.
	///
	/// \return An \c HRESULT error code.
	///
	/// \if UNICODE
	///   \sa get_Filter, Add, Remove, ExTVwLibU::ItemIdentifierTypeConstants
	/// \else
	///   \sa get_Filter, Add, Remove, ExTVwLibA::ItemIdentifierTypeConstants
	/// \endif
	virtual HRESULT STDMETHODCALLTYPE Contains(LONG itemIdentifier, ItemIdentifierTypeConstants itemIdentifierType = iitHandle, VARIANT_BOOL* pValue = NULL);
	/// \brief <em>Counts the items in the collection</em>
	///
	/// Retrieves the number of \c TreeViewItem objects in the collection.
	///
	/// \param[out] pValue The number of elements in the collection.
	///
	/// \return An \c HRESULT error code.
	///
	/// \sa Add, Remove, RemoveAll
	virtual HRESULT STDMETHODCALLTYPE Count(LONG* pValue);
	/// \brief <em>Searches for an item by its path</em>
	///
	/// Takes a string of the form ...\\a\\b\\c, where each part represents an item's text, and searches
	/// for a corresponding item. The search starts within the sub-items of the items defined by the
	/// \c ParentItem property.\n
	/// The separator is customizable and defaults to "\".
	///
	/// \param[in] path The item path to search for.
	/// \param[in] returnDeepestMatch Specifies the behavior for the case that no exactly matching item is
	///            found. If set to \c VARIANT_FALSE, no item is returned in such cases. If set to
	///            \c VARIANT_TRUE, the last item that matches a part of the path, is returned.
	/// \param[in] separator The \c String specifying the separator to use. If not specified, a
	///            backslash (\\) is used.
	///
	/// \return An \c HRESULT error code.
	/// \return The tree view item that matches the specified path.
	///
	/// \remarks Case sensitivity is specified by the \c CaseSensitiveFilters property. The functioned that
	///          is used for text comparison can be customized using the \c ComparisonFunction property
	///          (for the \c fpText filter).
	///
	/// \sa TreeViewItem::GetPath, get_CaseSensitiveFilters, get_ComparisonFunction
	virtual HRESULT STDMETHODCALLTYPE FindByPath(BSTR path, VARIANT_BOOL returnDeepestMatch = VARIANT_FALSE, BSTR* pSeparator = NULL, ITreeViewItem** ppItem = NULL);
	/// \brief <em>Removes the specified item in the collection from the treeview</em>
	///
	/// \param[in] itemIdentifier A value that identifies the treeview item to be removed.
	/// \param[in] itemIdentifierType A value specifying the meaning of \c itemIdentifier. Any of the
	///            values defined by the \c ItemIdentifierTypeConstants enumeration is valid.
	///
	/// \return An \c HRESULT error code.
	///
	/// \if UNICODE
	///   \sa Add, Count, RemoveAll, Contains, ExTVwLibU::ItemIdentifierTypeConstants
	/// \else
	///   \sa Add, Count, RemoveAll, Contains, ExTVwLibA::ItemIdentifierTypeConstants
	/// \endif
	virtual HRESULT STDMETHODCALLTYPE Remove(LONG itemIdentifier, ItemIdentifierTypeConstants itemIdentifierType = iitHandle);
	/// \brief <em>Removes all items in the collection from the treeview</em>
	///
	/// \return An \c HRESULT error code.
	///
	/// \sa Add, Count, Remove
	virtual HRESULT STDMETHODCALLTYPE RemoveAll(void);
	//@}
	//////////////////////////////////////////////////////////////////////

protected:
	//////////////////////////////////////////////////////////////////////
	/// \name Filter validation
	///
	//@{
	/// \brief <em>Validates a filter for a \c VARIANT_BOOL type property</em>
	///
	/// Retrieves whether a \c VARIANT value can be used as a filter for a property of type \c VARIANT_BOOL.
	///
	/// \param[in] filter The \c VARIANT to check.
	///
	/// \return \c TRUE, if the filter is valid; otherwise \c FALSE.
	///
	/// \sa IsValidIntegerFilter, IsValidStringFilter, IsValidTreeViewItemFilter, put_Filter
	BOOL IsValidBooleanFilter(VARIANT& filter);
	/// \brief <em>Validates a filter for a \c LONG (or compatible) type property</em>
	///
	/// Retrieves whether a \c VARIANT value can be used as a filter for a property of type
	/// \c LONG or compatible.
	///
	/// \param[in] filter The \c VARIANT to check.
	/// \param[in] minValue The minimum value that the corresponding property would accept.
	/// \param[in] maxValue The maximum value that the corresponding property would accept.
	///
	/// \return \c TRUE, if the filter is valid; otherwise \c FALSE.
	///
	/// \sa IsValidBooleanFilter, IsValidStringFilter, IsValidTreeViewItemFilter, put_Filter
	BOOL IsValidIntegerFilter(VARIANT& filter, int minValue, int maxValue);
	/// \brief <em>Validates a filter for a \c LONG (or compatible) type property</em>
	///
	/// \overload
	BOOL IsValidIntegerFilter(VARIANT& filter, int minValue);
	/// \brief <em>Validates a filter for a \c LONG (or compatible) type property</em>
	///
	/// \overload
	BOOL IsValidIntegerFilter(VARIANT& filter);
	/// \brief <em>Validates a filter for a \c BSTR type property</em>
	///
	/// Retrieves whether a \c VARIANT value can be used as a filter for a property of type \c BSTR.
	///
	/// \param[in] filter The \c VARIANT to check.
	///
	/// \return \c TRUE, if the filter is valid; otherwise \c FALSE.
	///
	/// \sa IsValidBooleanFilter, IsValidIntegerFilter, IsValidTreeViewItemFilter, put_Filter
	BOOL IsValidStringFilter(VARIANT& filter);
	/// \brief <em>Validates a filter for a \c ITreeViewItem type property</em>
	///
	/// Retrieves whether a \c VARIANT value can be used as a filter for a property of type \c ITreeViewItem.
	///
	/// \param[in] filter The \c VARIANT to check.
	///
	/// \return \c TRUE, if the filter is valid; otherwise \c FALSE.
	///
	/// \sa IsValidBooleanFilter, IsValidIntegerFilter, IsValidStringFilter, put_Filter
	BOOL IsValidTreeViewItemFilter(VARIANT& filter);
	//@}
	//////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////
	/// \name Filter appliance
	///
	//@{
	/// \brief <em>Retrieves the control's first item that might be in the collection</em>
	///
	/// \param[in] hWndTvw The treeview window the method will work on.
	///
	/// \return The item being the collection's first candidate. \c NULL if no item was found.
	///
	/// \sa GetNextItemToProcess, Count, RemoveAll, Next
	HTREEITEM GetFirstItemToProcess(HWND hWndTvw);
	/// \brief <em>Retrieves the control's next item that might be in the collection</em>
	///
	/// \param[in] hPreviousItem The item at which to start the search for a new collection candidate.
	/// \param[in] hWndTvw The treeview window the method will work on.
	///
	/// \return The next item being a candidate for the collection. \c NULL if no item was found.
	///
	/// \sa GetFirstItemToProcess, Count, RemoveAll, Next
	HTREEITEM GetNextItemToProcess(HTREEITEM hPreviousItem, HWND hWndTvw);
	/// \brief <em>Retrieves whether the specified \c SAFEARRAY contains the specified boolean value</em>
	///
	/// \param[in] pSafeArray The \c SAFEARRAY to search.
	/// \param[in] value The value to search for.
	/// \param[in] pComparisonFunction The address of the comparison function to use.
	///
	/// \return \c TRUE, if the array contains the value; otherwise \c FALSE.
	///
	/// \sa IsPartOfCollection, IsIntegerInSafeArray, IsStringInSafeArray, get_ComparisonFunction
	BOOL IsBooleanInSafeArray(LPSAFEARRAY pSafeArray, VARIANT_BOOL value, LPVOID pComparisonFunction);
	/// \brief <em>Retrieves whether the specified \c SAFEARRAY contains the specified integer value</em>
	///
	/// \param[in] pSafeArray The \c SAFEARRAY to search.
	/// \param[in] value The value to search for.
	/// \param[in] pComparisonFunction The address of the comparison function to use.
	///
	/// \return \c TRUE, if the array contains the value; otherwise \c FALSE.
	///
	/// \sa IsPartOfCollection, IsBooleanInSafeArray, IsStringInSafeArray, get_ComparisonFunction
	BOOL IsIntegerInSafeArray(LPSAFEARRAY pSafeArray, int value, LPVOID pComparisonFunction);
	/// \brief <em>Retrieves whether the specified \c SAFEARRAY contains the specified \c BSTR value</em>
	///
	/// \param[in] pSafeArray The \c SAFEARRAY to search.
	/// \param[in] value The value to search for.
	/// \param[in] pComparisonFunction The address of the comparison function to use.
	///
	/// \return \c TRUE, if the array contains the value; otherwise \c FALSE.
	///
	/// \sa IsPartOfCollection, IsBooleanInSafeArray, IsIntegerInSafeArray, get_ComparisonFunction
	BOOL IsStringInSafeArray(LPSAFEARRAY pSafeArray, BSTR value, LPVOID pComparisonFunction);
	/// \brief <em>Retrieves whether an item is part of the collection (applying the filters)</em>
	///
	/// \param[in] hItem The item to check.
	/// \param[in] skipFPSelected If \c TRUE, the \c fpSelected filter is skipped.
	/// \param[in] hWndTvw The treeview window the method will work on.
	/// \param[in] forceMultiLevel If \c TRUE, the \c singleNodeAndLevel flag will be ignored.
	///
	/// \return \c TRUE, if the item is part of the collection; otherwise \c FALSE.
	///
	/// \sa Contains, Count, Remove, RemoveAll, Next, properties.singleNodeAndLevel
	BOOL IsPartOfCollection(HTREEITEM hItem, BOOL skipFPSelected = FALSE, HWND hWndTvw = NULL, BOOL forceMultiLevel = FALSE);
	//@}
	//////////////////////////////////////////////////////////////////////

	#ifdef INCLUDESHELLBROWSERINTERFACE
		/// \brief <em>Adds an item to the treeview</em>
		///
		/// Adds an item with the specified properties at the specified position in the control and returns the
		/// the inserted item's handle.
		///
		/// \param[in] pItemText The new item's caption text. The maximum number of characters in this text
		///            is \c MAX_ITEMTEXTLENGTH.
		/// \param[in] hParentItem The new item's immediate parent item. If set to \c NULL, the item will be
		///            a top-level item.
		/// \param[in] hInsertAfter The new item's preceding item.
		/// \param[in] hasExpando A value specifying whether to draw a "+" or "-" next to the item
		///            indicating the item has sub-items. Any of the values defined by the
		///            \c HasExpandoConstants enumeration is valid. If set to \c heCallback, the control will
		///            fire the \c ItemGetDisplayInfo event each time this property's value is required.
		/// \param[in] iconIndex The zero-based index of the item's icon in the control's \c ilItems imagelist.
		///            If set to -1, the control will fire the \c ItemGetDisplayInfo event each time this
		///            property's value is required. A value of -2 means 'not specified' and is valid if
		///            there's no imagelist associated with the control.
		/// \param[in] selectedIconIndex The zero-based index of the item's selected icon in the control's
		///            \c ilItems imagelist. This icon will be used instead of the normal icon identified by
		///            \c iconIndex if the item is the caret item. If set to -1, the control will fire the
		///            \c ItemGetDisplayInfo event each time this property's value is required. If set to -2,
		///            the normal icon specified by \c iconIndex will be used.
		/// \param[in] expandedIconIndex The zero-based index of the item's expanded icon in the control's
		///            \c ilItems imagelist. This icon will be used instead of the normal icon identified by
		///            \c iconIndex if the item is expanded. If set to -1, the control will fire the
		///            \c ItemGetDisplayInfo event each time this property's value is required. If set to -2,
		///            the normal icon specified by \c iconIndex will be used.
		/// \param[in] overlayIndex The one-based index of the item's overlay icon in the control's \c ilItems
		///            imagelist. If set to 0, the new item won't have an overlay icon.
		/// \param[in] itemData A \c LPARAM value that will be associated with the item.
		/// \param[in] isGhosted If set to \c TRUE, the item will be drawn ghosted; otherwise not.
		/// \param[in] isVirtual If set to \c TRUE, the item will be treated as not existent when drawing the
		///            treeview and therefore won't be drawn. Instead its sub-items will be drawn at this
		///            item's position. If set to \c FALSE, the item and its sub-items will be drawn normally.
		/// \param[in] heightIncrement The item's height in multiples of the control's basic item height. E. g.
		///            a value of 2 means that the item will be twice as high as an item with \c HeighIncrement
		///            set to 1.
		/// \param[in] setShellItemFlag If \c TRUE, the control will flag the item as a shell item; otherwise
		///            not.
		///
		/// \return The inserted item's handle.
		///
		/// \remarks The \c expandedIconIndex and \c isVirtual parameters will be ignored if comctl32.dll is
		///          used in a version older than 6.10.
		///
		/// \if UNICODE
		///   \sa TreeViewItem::get_Text, TreeViewItem::get_ParentItem, TreeViewItem::get_PreviousSiblingItem,
		///       TreeViewItem::get_HasExpando, TreeViewItem::get_IconIndex,
		///       TreeViewItem::get_SelectedIconIndex, TreeViewItem::get_ExpandedIconIndex,
		///       TreeViewItem::get_OverlayIndex, TreeViewItem::get_ItemData, TreeViewItem::get_Ghosted,
		///       TreeViewItem::get_Virtual, TreeViewItem::get_HeightIncrement,
		///       ExplorerTreeView::get_hImageList, ExplorerTreeView::get_ItemHeight,
		///       ExplorerTreeView::Raise_ItemGetDisplayInfo, ExplorerTreeView::Raise_InsertedItem,
		///       ExTVwLibU::HasExpandoConstants, ExTVwLibU::ImageListConstants, MAX_ITEMTEXTLENGTH
		/// \else
		///   \sa TreeViewItem::get_Text, TreeViewItem::get_ParentItem, TreeViewItem::get_PreviousSiblingItem,
		///       TreeViewItem::get_HasExpando, TreeViewItem::get_IconIndex,
		///       TreeViewItem::get_SelectedIconIndex, TreeViewItem::get_ExpandedIconIndex,
		///       TreeViewItem::get_OverlayIndex, TreeViewItem::get_ItemData, TreeViewItem::get_Ghosted,
		///       TreeViewItem::get_Virtual, TreeViewItem::get_HeightIncrement,
		///       ExplorerTreeView::get_hImageList, ExplorerTreeView::get_ItemHeight,
		///       ExplorerTreeView::Raise_ItemGetDisplayInfo, ExplorerTreeView::Raise_InsertedItem,
		///       ExTVwLibA::HasExpandoConstants, ExTVwLibA::ImageListConstants, MAX_ITEMTEXTLENGTH
		/// \endif
		HTREEITEM Add(LPTSTR pItemText, HTREEITEM hParentItem, HTREEITEM hInsertAfter, int hasExpando, int iconIndex, int selectedIconIndex, int expandedIconIndex, int overlayIndex, LPARAM itemData, BOOL isGhosted, BOOL isVirtual, int heightIncrement, BOOL setShellItemFlag);
	#else
		/// \brief <em>Adds an item to the treeview</em>
		///
		/// Adds an item with the specified properties at the specified position in the control and returns the
		/// the inserted item's handle.
		///
		/// \param[in] pItemText The new item's caption text. The maximum number of characters in this text
		///            is \c MAX_ITEMTEXTLENGTH.
		/// \param[in] hParentItem The new item's immediate parent item. If set to \c NULL, the item will be
		///            a top-level item.
		/// \param[in] hInsertAfter The new item's preceding item.
		/// \param[in] hasExpando A value specifying whether to draw a "+" or "-" next to the item
		///            indicating the item has sub-items. Any of the values defined by the
		///            \c HasExpandoConstants enumeration is valid. If set to \c heCallback, the control will
		///            fire the \c ItemGetDisplayInfo event each time this property's value is required.
		/// \param[in] iconIndex The zero-based index of the item's icon in the control's \c ilItems imagelist.
		///            If set to -1, the control will fire the \c ItemGetDisplayInfo event each time this
		///            property's value is required. A value of -2 means 'not specified' and is valid if
		///            there's no imagelist associated with the control.
		/// \param[in] selectedIconIndex The zero-based index of the item's selected icon in the control's
		///            \c ilItems imagelist. This icon will be used instead of the normal icon identified by
		///            \c iconIndex if the item is the caret item. If set to -1, the control will fire the
		///            \c ItemGetDisplayInfo event each time this property's value is required. If set to -2,
		///            the normal icon specified by \c iconIndex will be used.
		/// \param[in] expandedIconIndex The zero-based index of the item's expanded icon in the control's
		///            \c ilItems imagelist. This icon will be used instead of the normal icon identified by
		///            \c iconIndex if the item is expanded. If set to -1, the control will fire the
		///            \c ItemGetDisplayInfo event each time this property's value is required. If set to -2,
		///            the normal icon specified by \c iconIndex will be used.
		/// \param[in] overlayIndex The one-based index of the item's overlay icon in the control's \c ilItems
		///            imagelist. If set to 0, the new item won't have an overlay icon.
		/// \param[in] itemData A \c LPARAM value that will be associated with the item.
		/// \param[in] isGhosted If set to \c TRUE, the item will be drawn ghosted; otherwise not.
		/// \param[in] isVirtual If set to \c TRUE, the item will be treated as not existent when drawing the
		///            treeview and therefore won't be drawn. Instead its sub-items will be drawn at this
		///            item's position. If set to \c FALSE, the item and its sub-items will be drawn normally.
		/// \param[in] heightIncrement The item's height in multiples of the control's basic item height. E. g.
		///            a value of 2 means that the item will be twice as high as an item with \c HeighIncrement
		///            set to 1.
		///
		/// \return The inserted item's handle.
		///
		/// \remarks The \c expandedIconIndex and \c isVirtual parameters will be ignored if comctl32.dll is
		///          used in a version older than 6.10.
		///
		/// \if UNICODE
		///   \sa TreeViewItem::get_Text, TreeViewItem::get_ParentItem, TreeViewItem::get_PreviousSiblingItem,
		///       TreeViewItem::get_HasExpando, TreeViewItem::get_IconIndex,
		///       TreeViewItem::get_SelectedIconIndex, TreeViewItem::get_ExpandedIconIndex,
		///       TreeViewItem::get_OverlayIndex, TreeViewItem::get_ItemData, TreeViewItem::get_Ghosted,
		///       TreeViewItem::get_Virtual, TreeViewItem::get_HeightIncrement,
		///       ExplorerTreeView::get_hImageList, ExplorerTreeView::get_ItemHeight,
		///       ExplorerTreeView::Raise_ItemGetDisplayInfo, ExplorerTreeView::Raise_InsertedItem,
		///       ExTVwLibU::HasExpandoConstants, ExTVwLibU::ImageListConstants, MAX_ITEMTEXTLENGTH
		/// \else
		///   \sa TreeViewItem::get_Text, TreeViewItem::get_ParentItem, TreeViewItem::get_PreviousSiblingItem,
		///       TreeViewItem::get_HasExpando, TreeViewItem::get_IconIndex,
		///       TreeViewItem::get_SelectedIconIndex, TreeViewItem::get_ExpandedIconIndex,
		///       TreeViewItem::get_OverlayIndex, TreeViewItem::get_ItemData, TreeViewItem::get_Ghosted,
		///       TreeViewItem::get_Virtual, TreeViewItem::get_HeightIncrement,
		///       ExplorerTreeView::get_hImageList, ExplorerTreeView::get_ItemHeight,
		///       ExplorerTreeView::Raise_ItemGetDisplayInfo, ExplorerTreeView::Raise_InsertedItem,
		///       ExTVwLibA::HasExpandoConstants, ExTVwLibA::ImageListConstants, MAX_ITEMTEXTLENGTH
		/// \endif
		HTREEITEM Add(LPTSTR pItemText, HTREEITEM hParentItem, HTREEITEM hInsertAfter, int hasExpando, int iconIndex, int selectedIconIndex, int expandedIconIndex, int overlayIndex, LPARAM itemData, BOOL isGhosted, BOOL isVirtual, int heightIncrement);
	#endif
	/// \brief <em>Retrieves whether a given item is selected</em>
	///
	/// \param[in] hItem The item to check.
	/// \param[in] hWndTvw The treeview window the method will work on.
	///
	/// \return \c TRUE if the item is selected; otherwise \c FALSE.
	///
	/// \sa TreeViewItem::get_Selected
	BOOL IsSelected(HTREEITEM hItem, HWND hWndTvw = NULL);
	#ifdef USE_STL
		/// \brief <em>Moves the specified items to a new parent item</em>
		///
		/// \param[in] itemsToMove A vector containing the handles to all items to move.
		/// \param[in] hNewParentItem The item that the items to move will become immediate sub-items of.
		///            May be \c TVI_ROOT or \c NULL to make the items top-level items.
		/// \param[in] hWndTvw The treeview window the method will work on.
		///
		/// \return An \c HRESULT error code.
		HRESULT MoveItems(std::vector<HTREEITEM>& itemsToMove, HTREEITEM hNewParentItem, HWND hWndTvw = NULL);
	#else
		/// \brief <em>Moves the specified items to a new parent item</em>
		///
		/// \param[in] itemsToMove A vector containing the handles to all items to move.
		/// \param[in] hNewParentItem The item that the items to move will become immediate sub-items of.
		///            May be \c TVI_ROOT or \c NULL to make the items top-level items.
		/// \param[in] hWndTvw The treeview window the method will work on.
		///
		/// \return An \c HRESULT error code.
		HRESULT MoveItems(CAtlArray<HTREEITEM>& itemsToMove, HTREEITEM hNewParentItem, HWND hWndTvw = NULL);
	#endif
	/// \brief <em>Shortens a filter as much as possible</em>
	///
	/// Optimizes a filter by detecting redundancies, tautologies and so on.
	///
	/// \param[in] filteredProperty The filter to optimize. Any of the values defined by the
	///            \c FilteredPropertyConstants enumeration is valid.
	///
	/// \sa put_Filter, put_FilterType
	void OptimizeFilter(FilteredPropertyConstants filteredProperty);
	#ifdef USE_STL
		/// \brief <em>Removes the specified items</em>
		///
		/// \param[in] itemsToRemove A vector containing all items to remove.
		/// \param[in] hWndTvw The treeview window the method will work on.
		///
		/// \return An \c HRESULT error code.
		HRESULT RemoveItems(std::vector<HTREEITEM>& itemsToRemove, HWND hWndTvw);
	#else
		/// \brief <em>Removes the specified items</em>
		///
		/// \param[in] itemsToRemove A vector containing all items to remove.
		/// \param[in] hWndTvw The treeview window the method will work on.
		///
		/// \return An \c HRESULT error code.
		HRESULT RemoveItems(CAtlArray<HTREEITEM>& itemsToRemove, HWND hWndTvw);
	#endif

	/// \brief <em>Sets the owner of this collection</em>
	///
	/// \param[in] pOwner The owner to set.
	///
	/// \sa Properties::pOwnerExTvw
	void SetOwner(__in_opt ExplorerTreeView* pOwner);

	/// \brief <em>Holds the object's properties' settings</em>
	struct Properties
	{
		#define NUMBEROFFILTERS 18
		/// \brief <em>Holds the \c CaseSensitiveFilters property's setting</em>
		///
		/// \sa get_CaseSensitiveFilters, put_CaseSensitiveFilters
		UINT caseSensitiveFilters : 1;
		/// \brief <em>Holds the \c ComparisonFunction property's setting</em>
		///
		/// \sa get_ComparisonFunction, put_ComparisonFunction
		LPVOID comparisonFunction[NUMBEROFFILTERS];
		/// \brief <em>Holds the \c Filter property's setting</em>
		///
		/// \sa get_Filter, put_Filter
		VARIANT filter[NUMBEROFFILTERS];
		/// \brief <em>Holds the \c FilterType property's setting</em>
		///
		/// \sa get_FilterType, put_FilterType
		FilterTypeConstants filterType[NUMBEROFFILTERS];

		/// \brief <em>The \c ExplorerTreeView object that owns this collection</em>
		///
		/// \sa SetOwner
		ExplorerTreeView* pOwnerExTvw;
		/// \brief <em>Holds the last enumerated item</em>
		HTREEITEM hLastEnumeratedItem;
		/// \brief <em>If \c TRUE, the collection contains immediate sub-items of \c hSimpleParentItem only</em>
		///
		/// \sa hSimpleParentItem, get_SingleLevel, ExplorerTreeView::get_RootItems, TreeViewItem::get_SubItems
		UINT singleNodeAndLevel : 1;
		/// \brief <em>If \c TRUE, we must filter the items</em>
		///
		/// \sa put_Filter, put_FilterType
		UINT usingFilters : 1;
		/// \brief <em>Holds the immediate parent item of the collection's items</em>
		///
		/// Holds the immediate parent item of all items in the collection. If set to -1, the \c fpParentItem
		/// filter specified by the \c Filter property is used.
		///
		/// \sa singleNodeAndLevel, get_ParentItem, get_SingleLevel, get_Filter
		HTREEITEM hSimpleParentItem;

		Properties()
		{
			pOwnerExTvw = NULL;
			hLastEnumeratedItem = NULL;
			hSimpleParentItem = (HTREEITEM) -1;
			caseSensitiveFilters = FALSE;
			singleNodeAndLevel = FALSE;

			for(int i = 0; i < NUMBEROFFILTERS; ++i) {
				VariantInit(&filter[i]);
				filterType[i] = ftDeactivated;
				comparisonFunction[i] = NULL;
			}
			usingFilters = FALSE;
		}

		~Properties();

		/// \brief <em>Copies this struct's content to another \c Properties struct</em>
		void CopyTo(Properties* pTarget);

		/// \brief <em>Retrieves the owning treeview's window handle</em>
		///
		/// \return The window handle of the treeview that contains the items in this collection.
		///
		/// \sa pOwnerExTvw
		HWND GetExTvwHWnd(void);
	} properties;
};     // TreeViewItems

OBJECT_ENTRY_AUTO(__uuidof(TreeViewItems), TreeViewItems)