// Copyright © 2019 The Things Network Foundation, The Things Industries B.V.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package ttnmage

import (
	"bufio"
	"fmt"
	"os"
	"strings"

	"github.com/blang/semver"
	"github.com/magefile/mage/mg"
)

// Version namespace.
type Version mg.Namespace

var goVersionFile = `// Code generated by Makefile. DO NOT EDIT.

package version

// TTN Version
var TTN = "%s-dev"
`

const (
	goVersionFilePath = "pkg/version/ttn.go"
)

var packageJSONFilePaths = []string{"package.json", "sdk/js/package.json"}

// Files writes given version to files that contain version info
func (Version) Files() error {
	if mg.Verbose() {
		fmt.Println("Enter new version")
	}
	reader := bufio.NewReader(os.Stdin)
	version, err := reader.ReadString('\n')
	if err != nil {
		return err
	}
	version = strings.TrimPrefix(strings.TrimSpace(version), "v")
	_, err = semver.Parse(version)
	if err != nil {
		return err
	}
	err = os.WriteFile(goVersionFilePath, []byte(fmt.Sprintf(goVersionFile, version)), 0o644)
	if err != nil {
		return err
	}
	for _, packageJSONFile := range packageJSONFilePaths {
		if err = runYarn("run", "json",
			"-f", packageJSONFile,
			"-I",
			"-e", fmt.Sprintf(`this.version="%s"`, version),
		); err != nil {
			return err
		}
	}
	return nil
}
