// Copyright © 2020 The Things Network Foundation, The Things Industries B.V.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package webmiddleware

import (
	"net/http"
	"net/textproto"
	"strings"

	"google.golang.org/grpc/metadata"
)

// Metadata returns a middleware that sets gRPC metadata from the request headers.
func Metadata(keys ...string) MiddlewareFunc {
	return func(next http.Handler) http.Handler {
		return http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
			ctx := r.Context()
			header := textproto.MIMEHeader(r.Header)
			md := make(metadata.MD)
			for _, key := range keys {
				if value := header.Values(key); value != nil {
					md[strings.ToLower(key)] = value
				}
			}
			if len(md) > 0 {
				if existingMD, ok := metadata.FromIncomingContext(ctx); ok {
					md = metadata.Join(existingMD, md)
				}
				ctx = metadata.NewIncomingContext(ctx, md)
			}
			next.ServeHTTP(w, r.WithContext(ctx))
		})
	}
}
