# Copyright (c) 2022-2023 THALES. All Rights Reserved

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

# http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


# Import Cocotb
import cocotb
from cocotb.triggers import Timer
from cocotb.clock import Clock
from cocotb.triggers import RisingEdge
from cocotbext.axi import (AxiStreamBus, AxiStreamSource, AxiStreamSink, AxiStreamFrame)

# Others
import random
from random import randbytes
from random import Random
import logging

# UDP library
from lib import UdpFrame

# Global Parameters
NB_FRAMES = 20
PAYLOAD_MIN_SIZE = 1
PAYLOAD_MAX_SIZE = 50
SEED = 587691
DEBUG = 1

# Variable declarations
SRC_MAC_ADDR = 0x11_22_33_44_55_66
DEST_MAC_ADDR = 0xaa_bb_cc_dd_ee_ff
ETH_IP_ADDR_1 = 0xC0_A8_01_0A  # 192.168.1.10
ETH_IP_ADDR_2 = 0xC0_A8_01_0F  # 192.168.1.15
ETH_IP_ADDR_3 = 0xC0_A8_01_14  # 192.168.1.20
ETH_IP_ADDR_4 = 0xC0_A8_01_19  # 192.168.1.25
ETH_IP_ADDR_LIST = [ETH_IP_ADDR_1, ETH_IP_ADDR_2, ETH_IP_ADDR_3, ETH_IP_ADDR_4]
ETH_PORT_ADDR_1 = 0x12_34
ETH_PORT_ADDR_2 = 0x56_78
ETH_PORT_ADDR_3 = 0x9A_BC
ETH_PORT_ADDR_4 = 0xDE_F0
ETH_PORT_ADDR_LIST = [ETH_PORT_ADDR_1, ETH_PORT_ADDR_2, ETH_PORT_ADDR_3, ETH_PORT_ADDR_4]
UDP_HEADER_SIZE = 8
FRAME_SIZE_MAX = UDP_HEADER_SIZE + PAYLOAD_MAX_SIZE


def genRandomTransfer(random_gen):
    """Generation of UDP frame with pseudo-random way"""
    while True:
        size = random_gen.randint(PAYLOAD_MIN_SIZE, PAYLOAD_MAX_SIZE)  # Generate random size
        ip = ETH_IP_ADDR_LIST[random_gen.randint(0, 3)]
        port_dest = ETH_PORT_ADDR_LIST[random_gen.randint(0, 3)]
        port_src = ETH_PORT_ADDR_LIST[random_gen.randint(0, 3)]
        tdata = random_gen.randbytes(size)  # Generate tdata with random bytes
        tkeep = [1] * len(tdata)  # Generate tkeep
        tuser = int.from_bytes(port_dest.to_bytes(2, 'big') + port_src.to_bytes(2, 'big') + size.to_bytes(2, 'big') + ip.to_bytes(4, 'big'), 'big')
        frame = AxiStreamFrame(tdata=tdata, tkeep=tkeep, tid=None, tdest=None, tuser=tuser)
        yield frame


# coroutine to handle Reset
async def handlerReset(dut):
    """Reset management"""
    dut.rst.value = 0
    await Timer(30, units='ns')
    dut.rst.value = 1


# coroutine to handle Initdone
async def handlerInitdone(dut):
    """Init done management"""
    dut.init_done.value = 0
    await Timer(400, units='ns')
    dut.init_done.value = 1


# coroutine to handle Slave interface
async def handlerSlave(dut):
    """Sending data frames generated by genRandomTransfer to AXI-Stream bus"""

    # Init source
    logging.getLogger("cocotb.uoe_udp_module_tx.s").setLevel("WARNING")
    slave = AxiStreamSource(AxiStreamBus.from_prefix(dut, "s"), dut.clk, dut.rst, reset_active_level=False)

    # Init random generator
    s_random = Random()
    s_random.seed(SEED)
    s_trans = genRandomTransfer(s_random)

    await RisingEdge(dut.rst)
    await RisingEdge(dut.clk)

    # Data send
    for _ in range(NB_FRAMES):
        frame = next(s_trans)
        await slave.send(frame)

    cocotb.log.info("End of handlerSlave")


# coroutine to handle Master interface
async def handlerMaster(dut):
    """Read data from AXI-Stream bus"""

    # Error variable
    global simulation_err

    # Init source
    logging.getLogger("cocotb.uoe_udp_module_tx.m").setLevel("WARNING")
    master = AxiStreamSink(AxiStreamBus.from_prefix(dut, "m"), dut.clk, dut.rst, reset_active_level=False)

    # Init random generator
    m_random_ctrl = Random()
    m_random_ctrl.seed(SEED)

    # Data reception
    for _ in range(NB_FRAMES):
        data = await master.recv()
        data = UdpFrame.from_bytes(data.tdata)

        # Value for test
        m_size = m_random_ctrl.randint(PAYLOAD_MIN_SIZE, PAYLOAD_MAX_SIZE)  # Generate random size
        m_ip = ETH_IP_ADDR_LIST[m_random_ctrl.randint(0, 3)]
        m_port_dest = ETH_PORT_ADDR_LIST[m_random_ctrl.randint(0, 3)]
        m_port_src = ETH_PORT_ADDR_LIST[m_random_ctrl.randint(0, 3)]
        m_payload = m_random_ctrl.randbytes(m_size)

        data_ctrl = UdpFrame(src_port=m_port_src,
                             dst_port=m_port_dest,
                             payload=m_payload)

        # Validity test
        if data_ctrl == data:
            if DEBUG == 1:
                cocotb.log.info(f"UPD_TX [{_}] is OK")
        else:
            cocotb.log.error(f"UDP_TX [{_}] faillure / size {len(data.payload)}:{len(data_ctrl.payload)}(test)")
            cocotb.log.error(f"Port_dest : {hex(data.dst_port)} / Port_dest_ctrl : {hex(data_ctrl.dst_port)}")
            cocotb.log.error(f"Port_src : {hex(data.src_port)} / Port_src_ctrl : {hex(data_ctrl.src_port)}")
            cocotb.log.error(f"Data : {data.payload.hex()} / Data_ctrl : {data_ctrl.payload.hex()}")
            simulation_err += 1

    cocotb.log.info("End of handlerMaster")


@cocotb.test()
async def handlermain(dut):
    """Main function for starting coroutines"""

    description = "\n\n**********************************************************************************************************************************************************\n"
    description += "*                                                                    Description                                                                         *\n"
    description += "**********************************************************************************************************************************************************\n"
    description += "*  The role of the UDP module layer is to manage the UDP protocol.                                                                                       *\n"
    description += "*  On reception, the UDP header is removed.                                                                                                              *\n"
    description += "*  The aim is to send random bytes with an Ethernet header and check whether the header has been removed correctly.                                      *\n"
    description += "**********************************************************************************************************************************************************\n"

    cocotb.log.info(f"{description}")
    cocotb.log.info("Start coroutines")

    # Error variable
    global simulation_err
    simulation_err = 0

    # Init clock
    clk100M = Clock(dut.clk, 10, units='ns')
    # start clock
    cocotb.start_soon(clk100M.start())

    # start coroutine of reset management
    cocotb.start_soon(handlerReset(dut))

    # start coroutines
    h_init_done = cocotb.start_soon(handlerInitdone(dut))
    h_slave = cocotb.start_soon(handlerSlave(dut))
    h_master = cocotb.start_soon(handlerMaster(dut))

    # Wait Reset
    await RisingEdge(dut.rst)
    await RisingEdge(dut.clk)

    # wait that coroutines are finished
    await h_slave
    await h_master

    await Timer(100, units='ns')

    if simulation_err >= 1:
        print_rsl = "\n\n\n******************************************************************************************\n"
        print_rsl += "**                                   There are " + str(simulation_err) + " errors !                              **\n"
        print_rsl += "******************************************************************************************"
        cocotb.log.error(f"{print_rsl}")
    else:
        print_rsl = "\n\n\n******************************************************************************************\n"
        print_rsl += "**                                        Simulation OK !                               **\n"
        print_rsl += "******************************************************************************************"
        cocotb.log.info(f"{print_rsl}")
