# Copyright (c) 2022-2023 THALES. All Rights Reserved

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

# http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


# Import Cocotb
import cocotb
from cocotb.triggers import Timer
from cocotb.clock import Clock
from cocotb.triggers import RisingEdge
from cocotbext.axi import (AxiStreamBus, AxiStreamSource, AxiStreamSink, AxiStreamFrame)

# Others
import random
from random import randbytes
from random import Random
import logging

# Ethernet library
from lib import EthFrame

# Global Parameters
NB_FRAMES = 20
MIN_SIZE = 5
MAX_SIZE = 20
SEED = 587691
DEBUG = 1

# Variable declarations
SRC_MAC_ADDR = 0x11_22_33_44_55_66
DEST_MAC_ADDR = 0xaa_bb_cc_dd_ee_ff
ETHERTYPE_1 = 0x1234
ETHERTYPE_2 = 0xABCD
ETHERTYPE_3 = 0xA55A
ETHERTYPE_4 = 0xBEAF
ETHERTYPE_LIST = [ETHERTYPE_1, ETHERTYPE_2, ETHERTYPE_3, ETHERTYPE_4]


def genRandomTransfer(random_gen):
    """Generation of RAW frame with pseudo-random way"""
    while True:
        size = random_gen.randint(MIN_SIZE, MAX_SIZE)  # Generate random size
        tdata = random_gen.randbytes(size)  # Generate tdata with random bytes
        tkeep = [1] * len(tdata)  # Generate tkeep
        tid = ETHERTYPE_LIST[random_gen.randint(0, 3)]  # Generate tid with random ethertype
        frame = AxiStreamFrame(tdata=tdata, tkeep=tkeep, tid=tid, tdest=None, tuser=None)
        yield frame


# coroutine to handle Reset
async def handlerReset(dut):
    """Reset management"""
    dut.rst.value = 0
    await Timer(30, units='ns')
    dut.rst.value = 1


# coroutine to handle Slave interface
async def handlerSlave(dut):
    """Sending data frames generated by genRandomTransfer to AXI-Stream bus"""

    # Init source
    logging.getLogger("cocotb.uoe_raw_ethernet_tx.s").setLevel("WARNING")
    slave = AxiStreamSource(AxiStreamBus.from_prefix(dut, "s"), dut.clk, dut.rst, reset_active_level=False)

    # Init random generator
    s_random = Random()
    s_random.seed(SEED)
    s_trans = genRandomTransfer(s_random)

    await RisingEdge(dut.rst)
    await RisingEdge(dut.clk)

    # Data send
    for _ in range(NB_FRAMES):
        frame = next(s_trans)
        await slave.send(frame)

    cocotb.log.info("End of handlerSlave")


# coroutine to handle Master interface
async def handlerMaster(dut):
    """Read data from AXI-Stream bus"""

    # Error variable
    global simulation_err

    # Init source
    logging.getLogger("cocotb.uoe_raw_ethernet_tx.m").setLevel("WARNING")
    master = AxiStreamSink(AxiStreamBus.from_prefix(dut, "m"), dut.clk, dut.rst, reset_active_level=False)

    # Init random generator
    m_random_ctrl = Random()
    m_random_ctrl.seed(SEED)

    # Data reception
    for _ in range(NB_FRAMES):
        data = await master.recv()
        data = EthFrame.from_bytes(data.tdata)

        # Value for test
        m_size = m_random_ctrl.randint(MIN_SIZE, MAX_SIZE)
        m_payload = m_random_ctrl.randbytes(m_size)
        m_ethertype = ETHERTYPE_LIST[m_random_ctrl.randint(0, 3)]

        data_ctrl = EthFrame(dst_mac_addr=DEST_MAC_ADDR.to_bytes(6, 'big'),
                             src_mac_addr=SRC_MAC_ADDR.to_bytes(6, 'big'),
                             ethertype=m_ethertype,
                             payload=m_payload)

        # Validity test
        if data_ctrl == data:
            if DEBUG == 1:
                cocotb.log.info(f"RAW_TX [{_}] is OK")
        else:
            cocotb.log.error(f"RAW_TX [{_}] faillure / size {len(data.payload)}:{len(data_ctrl.payload)}(test)")
            cocotb.log.error(f"Dst_mac_addr : {data.dst_mac_addr.hex()} / Dst_mac_addr_ctrl : {data_ctrl.dst_mac_addr.hex()}")
            cocotb.log.error(f"Src_mac_addr : {data_ctrl.src_mac_addr.hex()} / Src_mac_addr_ctrl : {data_ctrl.src_mac_addr.hex()}")
            cocotb.log.error(f"Ethertype : {hex(data_ctrl.ethertype)} / Ethertype_ctrl : {hex(data_ctrl.ethertype)}")
            cocotb.log.error(f"Data : {data.payload.hex()} / Data_ctrl : {data_ctrl.payload.hex()}")
            simulation_err += 1

    cocotb.log.info("End of handlerMaster")


@cocotb.test()
async def handlermain(dut):
    """Main function for starting coroutines"""

    description = "\n\n**********************************************************************************************************************************************************\n"
    description += "*                                                                    Description                                                                         *\n"
    description += "**********************************************************************************************************************************************************\n"
    description += "*  In Emmission, the RAW Ethernet sub-module manages the insertion of the Ethernet header (MAC).                                                         *\n"
    description += "*  The aim is to send one of the random bytes to check whether the Ethernet header has been inserted.                                                    *\n"
    description += "**********************************************************************************************************************************************************\n"

    cocotb.log.info(f"{description}")
    cocotb.log.info("Start coroutines")

    # Error variable
    global simulation_err
    simulation_err = 0

    # Init clock
    clk100M = Clock(dut.clk, 10, units='ns')
    # start clock
    cocotb.start_soon(clk100M.start())

    # start coroutine of reset management
    cocotb.start_soon(handlerReset(dut))

    # start coroutines
    h_slave = cocotb.start_soon(handlerSlave(dut))
    h_master = cocotb.start_soon(handlerMaster(dut))

    # Init signal
    dut.init_done = 0

    # Wait Reset
    await RisingEdge(dut.rst)
    await RisingEdge(dut.clk)

    # update signals
    dut.init_done = 1
    dut.dest_mac_addr = DEST_MAC_ADDR
    dut.local_mac_addr = SRC_MAC_ADDR

    # wait that coroutines are finished
    await h_slave
    await h_master

    await Timer(100, units='ns')

    if simulation_err >= 1:
        print_rsl = "\n\n\n********************************************************************************************\n"
        print_rsl += "**                                    There are " + str(simulation_err) + " errors !                               **\n"
        print_rsl += "********************************************************************************************"
        cocotb.log.error(f"{print_rsl}")
    else:
        print_rsl = "\n\n\n********************************************************************************************\n"
        print_rsl += "**                                         Simulation OK !                                **\n"
        print_rsl += "********************************************************************************************"
        cocotb.log.info(f"{print_rsl}")
