# Copyright (c) 2022-2023 THALES. All Rights Reserved

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

# http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


# Import Cocotb
import cocotb
from cocotb.triggers import Timer
from cocotb.clock import Clock
from cocotb.triggers import RisingEdge
from cocotbext.axi import (AxiStreamBus, AxiStreamSource, AxiStreamSink, AxiStreamMonitor, AxiStreamFrame)

# Others
import random
from random import randbytes
from random import Random
import logging

# Ethernet library
from lib import EthFrame
from lib import Ipv4Frame
from lib import UdpFrame

# Global Parameters
NB_FRAMES = 20
PAYLOAD_SIZE_MIN = 5
PAYLOAD_SIZE_MAX = 50
SEED = 158647
DEBUG = 1

# Variable declarations
LOCAL_MAC_ADDR = 0x01_23_45_67_89_AB
ETH_IP_ADDR_1 = 0xC0_A8_01_0A  # 192.168.1.10
ETH_IP_ADDR_2 = 0xC0_A8_01_0F  # 192.168.1.15
ETH_IP_ADDR_3 = 0xC0_A8_01_14  # 192.168.1.20
ETH_IP_ADDR_4 = 0xC0_A8_01_19  # 192.168.1.25
ETH_IP_ADDR_LIST = [ETH_IP_ADDR_1, ETH_IP_ADDR_2, ETH_IP_ADDR_3, ETH_IP_ADDR_4]
ETHERTYPE_IPV4 = 0x0800


def genRandomTransfer(payload_random_gen, user_random_gen):
    """Function which generate AXIS frames"""
    while True:
        size = payload_random_gen.randint(PAYLOAD_SIZE_MIN, PAYLOAD_SIZE_MAX)  # Generate random size for payload
        ip_dest = ETH_IP_ADDR_LIST[user_random_gen.randint(0, 3)]
        ipv4_part = Ipv4Frame(frame_id=0,
                              sub_protocol=0x11,
                              ip_src=0xC0_A8_01_0A.to_bytes(4, 'big'),
                              ip_dest=ip_dest.to_bytes(4, 'big'),
                              payload=payload_random_gen.randbytes(size))
        tdata = Ipv4Frame.__bytes__(ipv4_part)  # Generate tdata
        tkeep = [1] * len(tdata)  # Generate tkeep
        tuser = ip_dest  # Generate tuser
        tid = ETHERTYPE_IPV4
        frame = AxiStreamFrame(tdata=tdata, tkeep=tkeep, tid=tid, tdest=None, tuser=tuser)
        yield frame


def genRandomTransfer_mac_addr(random_gen):
    """Functon which generate destination mac address"""
    while True:
        tdata = random_gen.randbytes(6)  # Generate tdata with mac addr
        tdata_size = len(tdata)  # Generate tkeep
        tuser = random_gen.randint(0, 1)  # Generate tuser
        frame = AxiStreamFrame(tdata=tdata, tkeep=None, tid=None, tdest=None, tuser=tuser)
        yield frame


# coroutine to handle Reset
async def handlerReset(dut):
    """Reset management"""
    dut.rst.value = 0
    await Timer(30, units='ns')
    dut.rst.value = 1


# coroutine to handle Slave interface
async def handlerSlave(dut):
    """Sending data frames generated by genRandomTransfer on AXI-Stream bus"""

    # Init source and random generator
    logging.getLogger("cocotb.uoe_mac_shaping_tx.s").setLevel("WARNING")
    slave = AxiStreamSource(AxiStreamBus.from_prefix(dut, "s"), dut.clk, dut.rst, reset_active_level=False)

    s_payload_random = Random()
    s_payload_random.seed(SEED)
    s_user_random = Random()
    s_user_random.seed(SEED)
    s_trans = genRandomTransfer(s_payload_random, s_user_random)

    # Init signals
    dut.s_tdata = 0
    dut.s_tvalid = 0
    dut.s_tkeep = 0
    dut.s_tid = 0
    dut.s_tuser = 0

    await RisingEdge(dut.rst)
    await RisingEdge(dut.clk)

    # Data send
    for _ in range(NB_FRAMES):
        frame = next(s_trans)
        await slave.send(frame)

    await Timer(100, units='ns')
    cocotb.log.info("End of handlerSlave")


# coroutine to handle Slave interface
async def handler_mac_addr(dut):
    """Send IP address and recept MAC address"""

    # Init source
    logging.getLogger("cocotb.uoe_mac_shaping_tx.s_mac_addr").setLevel("WARNING")
    slave = AxiStreamSource(AxiStreamBus.from_prefix(dut, "s_mac_addr"), dut.clk, dut.rst, reset_active_level=False)

    logging.getLogger("cocotb.uoe_mac_shaping_tx.m_ip_addr").setLevel("WARNING")
    master = AxiStreamSink(AxiStreamBus.from_prefix(dut, "m_ip_addr"), dut.clk, dut.rst, reset_active_level=False)

    # Init random generator
    s_random = Random()
    s_random.seed(SEED)
    s_trans = genRandomTransfer_mac_addr(s_random)

    # Init signals
    dut.s_mac_addr_tdata = 0
    dut.s_mac_addr_tvalid = 0
    dut.s_mac_addr_tuser = 0
    dut.m_ip_addr_tready = 0

    await RisingEdge(dut.rst)
    await RisingEdge(dut.clk)

    # Data send and read
    for _ in range(NB_FRAMES):
        data = await master.recv()
        frame = next(s_trans)
        await slave.send(frame)

    cocotb.log.info("End of handler_mac_addr")


# coroutine to handle Master interface
async def handlerMaster(dut):
    """Read data from AXI-Stream bus"""

    global simulation_err

    # Init source
    logging.getLogger("cocotb.uoe_mac_shaping_tx.m").setLevel("WARNING")
    master = AxiStreamSink(AxiStreamBus.from_prefix(dut, "m"), dut.clk, dut.rst, reset_active_level=False)

    # Init random generator
    m_payload_random = Random()
    m_payload_random.seed(SEED)

    m_user_random_gen = Random()
    m_user_random_gen.seed(SEED)

    mac_random = Random()
    mac_random.seed(SEED)

    # Init signal
    dut.m_tready = 0

    await RisingEdge(dut.rst)
    await RisingEdge(dut.clk)

    # Data reception
    for _ in range(NB_FRAMES):

        # Value for test
        mac_addr_ctrl = int.from_bytes(mac_random.randbytes(6), 'little')
        mac_status = mac_random.randint(0, 1)

        m_size = m_payload_random.randint(PAYLOAD_SIZE_MIN, PAYLOAD_SIZE_MAX)
        m_payload = m_payload_random.randbytes(m_size)

        ipv4_part = Ipv4Frame(frame_id=0,
                              sub_protocol=0x11,
                              ip_src=0xC0_A8_01_0A.to_bytes(4, 'big'),
                              ip_dest=ETH_IP_ADDR_LIST[m_user_random_gen.randint(0, 3)].to_bytes(4, 'big'),
                              payload=m_payload)

        data_ctrl = EthFrame(dst_mac_addr=mac_addr_ctrl.to_bytes(6, 'big'),
                             src_mac_addr=LOCAL_MAC_ADDR.to_bytes(6, 'big'),
                             ethertype=ETHERTYPE_IPV4,
                             payload=ipv4_part)

        # Convert UDP part to compare results
        data_ctrl = EthFrame.__bytes__(data_ctrl)
        data_ctrl = EthFrame.from_bytes(data_ctrl)

        # Validity test
        if mac_status == 0:
            data = await master.recv()
            data_rslt = EthFrame.from_bytes(data.tdata)

            if data_rslt == data_ctrl:
                if DEBUG == 1:
                    cocotb.log.info(f"M_MAC_SHAPING [{_}] is OK")
            else:
                cocotb.log.error(f"M_MAC_SHAPING [{_}] faillure / size {len(data_rslt.payload.payload)}:{len(data_ctrl.payload.payload)}(test)")
                cocotb.log.error(f"Dst_mac_addr : {data_rslt.dst_mac_addr.hex()} / Dst_mac_addr_ctrl : {data_ctrl.dst_mac_addr.hex()}")
                cocotb.log.error(f"Src_mac_addr : {data_rslt.src_mac_addr.hex()} / Src_mac_addr_ctrl : {data_ctrl.src_mac_addr.hex()}")
                cocotb.log.error(f"Ethertype : {hex(data_rslt.ethertype)} / Ethertype_ctrl : {hex(data_ctrl.ethertype)}")
                cocotb.log.error(f"ip_dest : {data_rslt.payload.ip_dest.hex()} / ip_dest_ctrl : {data_ctrl.payload.ip_dest.hex()}")
                cocotb.log.error(f"ip_src : {data_rslt.payload.ip_src.hex()} / ip_src_ctrl : {data_ctrl.payload.ip_src.hex()}")
                cocotb.log.error(f"sub_protocol : {hex(data_rslt.payload.sub_protocol)} / sub_protocol_ctrl : {hex(data_ctrl.payload.sub_protocol)}")
                cocotb.log.error(f"frame_id : {hex(data_rslt.payload.frame_id)} / frame_id_ctrl : {hex(data_ctrl.payload.frame_id)}")
                cocotb.log.error(f"ttl : {hex(data_rslt.payload.ttl)} / ttl_ctrl : {hex(data_ctrl.payload.ttl)}")
                cocotb.log.error(f"frag_flags : {hex(data_rslt.payload.frag_flags)} / frag_flags_ctrl : {hex(data_ctrl.payload.frag_flags)}")
                cocotb.log.error(f"frag_offset : {hex(data_rslt.payload.frag_offset)} / frag_offset_ctrl : {hex(data_ctrl.payload.frag_offset)}")
                cocotb.log.error(f"src_port : {hex(data_rslt.payload.payload.src_port)} / src_port_ctrl : {hex(data_ctrl.payload.payload.src_port)}")
                cocotb.log.error(f"dst_port : {hex(data_rslt.payload.payload.dst_port)} / dst_port_ctrl : {hex(data_ctrl.payload.payload.dst_port)}")
                cocotb.log.error(f"Data : {data_rslt.payload.payload.hex()} / Data_ctrl : {data_ctrl.payload.payload.hex()}")
                simulation_err += 1
        else:
            cocotb.log.info(f"MAC_STATUS = 1 [{_}]")

    await Timer(100, units='ns')
    cocotb.log.info("End of handlerMaster")


@cocotb.test()
async def handlermain(dut):
    """Main function for starting coroutines"""

    description = "\n\n**********************************************************************************************************************************************************\n"
    description += "*                                                                    Description                                                                         *\n"
    description += "**********************************************************************************************************************************************************\n"
    description += "*  In Emmission, the MAC Shaping sub-module manages the insertion of the Ethernet header (MAC). The header's Destination MAC Address field is defined    *\n"
    description += "*  using an IP Address <=> MAC Address association table.                                                                                                *\n"
    description += "*  The aim is to send one of the random bytes to check whether the Ethernet header has been correctly inserted.                                          *\n"
    description += "**********************************************************************************************************************************************************\n"

    cocotb.log.info(f"{description}")
    cocotb.log.info("Start coroutines")

    # Error variable
    global simulation_err
    simulation_err = 0

    # Init clock
    clk100M = Clock(dut.clk, 10, units='ns')
    # start clock
    cocotb.start_soon(clk100M.start())
    # start coroutine of reset management
    cocotb.start_soon(handlerReset(dut))

    # Start RX process
    h_slave = cocotb.start_soon(handlerSlave(dut))
    h_master = cocotb.start_soon(handlerMaster(dut))
    h_mac_addr = cocotb.start_soon(handler_mac_addr(dut))

    # Init signal
    dut.local_mac_addr = LOCAL_MAC_ADDR

    # Wait reset and clock
    await RisingEdge(dut.rst)
    await RisingEdge(dut.clk)

    # Await RX process
    await h_slave
    await h_master
    await h_mac_addr

    await Timer(100, units='ns')

    if simulation_err >= 1:
        print_rsl = "\n\n\n*****************************************************************************************\n"
        print_rsl += "**                                  There are " + str(simulation_err) + " errors !                              **\n"
        print_rsl += "*****************************************************************************************"
        cocotb.log.error(f"{print_rsl}")
    else:
        print_rsl = "\n\n\n*****************************************************************************************\n"
        print_rsl += "**                                       Simulation OK !                               **\n"
        print_rsl += "*****************************************************************************************"
        cocotb.log.info(f"{print_rsl}")
