# Copyright (c) 2022-2023 THALES. All Rights Reserved

# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at

# http://www.apache.org/licenses/LICENSE-2.0

# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


# Import Cocotb
import cocotb
from cocotb.triggers import Timer
from cocotb.clock import Clock
from cocotb.triggers import RisingEdge
from cocotbext.axi import (AxiStreamBus, AxiStreamSource, AxiStreamSink, AxiStreamMonitor, AxiStreamFrame)

# Others
import random
from random import randbytes
from random import Random
import logging

# Ethernet library
from lib import EthFrame

# Global Parameters
NB_FRAMES = 20
PAYLOAD_SIZE_MIN = 5
PAYLOAD_SIZE_MAX = 20
SEED = 158647
DEBUG = 1

# Variable declaration
DEST_MAC_ADDR = 0xAA_BB_CC_DD_EE_FF
SRC_MAC_ADDR = 0x11_22_33_44_55_66
ETHERTYPE_1 = 0x1234
ETHERTYPE_2 = 0xABCD
ETHERTYPE_3 = 0xA55A
ETHERTYPE_4 = 0xBEAF
ETHERTYPE_LIST_TX = [ETHERTYPE_1, ETHERTYPE_2, ETHERTYPE_3, ETHERTYPE_4]


def genRandomEthernetFrame(random_gen):
    """Generation of Ethernet frame with pseudo-random way"""
    while True:
        size = random_gen.randint(PAYLOAD_SIZE_MIN, PAYLOAD_SIZE_MAX)  # Generate random size for payload
        # Generate tdata with dest_mac/scr_mac/ethertype
        tdata = EthFrame(dst_mac_addr=DEST_MAC_ADDR.to_bytes(6, 'big'),
                         src_mac_addr=SRC_MAC_ADDR.to_bytes(6, 'big'),
                         ethertype=ETHERTYPE_LIST_TX[random_gen.randint(0, 1)],
                         payload=random_gen.randbytes(size))
        tdata = EthFrame.__bytes__(tdata)
        tkeep = [1] * len(tdata)  # Generate tkeep
        frame = AxiStreamFrame(tdata=tdata, tkeep=tkeep, tid=None, tdest=None, tuser=None)
        yield frame


# coroutine to handle Reset
async def handlerReset(dut):
    """Reset management"""
    dut.rst.value = 0
    await Timer(30, units='ns')
    dut.rst.value = 1


# coroutine to handle Slave interface
async def handlerSlave(dut):
    """Sending data frames generated by genRandomTransfer on AXI-Stream bus"""

    # Init source
    logging.getLogger("cocotb.uoe_mac_shaping_rx.s").setLevel("WARNING")
    slave = AxiStreamSource(AxiStreamBus.from_prefix(dut, "s"), dut.clk, dut.rst, reset_active_level=False)

    # Init random generator
    s_random = Random()
    s_random.seed(SEED)
    s_trans = genRandomEthernetFrame(s_random)

    # Init signals
    dut.s_tdata = 0
    dut.s_tvalid = 0
    dut.s_tkeep = 0

    await RisingEdge(dut.rst)
    await RisingEdge(dut.clk)

    # Data send
    for _ in range(NB_FRAMES):
        frame = next(s_trans)
        await slave.send(frame)

    cocotb.log.info("End of handlerSlave")


# coroutine to handle Master interface
async def handlerMaster(dut):
    """Read data from AXI-Stream bus for raw_ethernet_rx"""

    # Error variable
    global simulation_err

    # Init source
    logging.getLogger("cocotb.uoe_mac_shaping_rx.m").setLevel("WARNING")
    master = AxiStreamSink(AxiStreamBus.from_prefix(dut, "m"), dut.clk, dut.rst, reset_active_level=False)

    # Init random generator
    m_random = Random()
    m_random.seed(SEED)

    # Init signal
    dut.m_tready = 0

    await RisingEdge(dut.rst)
    await RisingEdge(dut.clk)

    # Data reception
    for _ in range(NB_FRAMES):
        data = await master.recv()

        # Value for test
        m_size = m_random.randint(PAYLOAD_SIZE_MIN, PAYLOAD_SIZE_MAX)
        m_ethertype = ETHERTYPE_LIST_TX[m_random.randint(0, 1)]
        m_payload = m_random.randbytes(m_size)

        # Validity test
        if data.tid == m_ethertype and data.tdata == m_payload:
            if DEBUG == 1:
                cocotb.log.info(f"M_MAC_SHAPING [{_}] is OK")
        else:
            cocotb.log.error(f"M_MAC_SHAPING [{_}] faillure / size {len(data.tdata)}:{len(m_payload)}(test)")
            cocotb.log.error(f"Ethertype : {hex(data.tid)} / Ethertype_ctrl : {hex(m_ethertype)}")
            cocotb.log.error(f"Data : {data.tdata.hex()} / Data_ctrl : {m_payload.hex()}")
            simulation_err += 1

    cocotb.log.info("End of handlerMaster")


@cocotb.test()
async def handlermain(dut):
    """Main function for starting coroutines"""

    description = "\n\n**********************************************************************************************************************************************************\n"
    description += "*                                                                    Description                                                                         *\n"
    description += "**********************************************************************************************************************************************************\n"
    description += "*  On the receiving side, the MAC Shaping sub-module manages the removal of the Ethernet header (MAC).                                                   *\n"
    description += "*  The aim is to send random bytes with an Ethernet header and check whether the header has been removed correctly.                                      *\n"
    description += "**********************************************************************************************************************************************************\n"

    cocotb.log.info(f"{description}")
    cocotb.log.info("Start coroutines")

    # Error variable
    global simulation_err
    simulation_err = 0

    # Init clock
    clk100M = Clock(dut.clk, 10, units='ns')
    # start clock
    cocotb.start_soon(clk100M.start())
    # start coroutine of reset management
    cocotb.start_soon(handlerReset(dut))

    # Start RX process
    h_slave = cocotb.start_soon(handlerSlave(dut))
    h_master = cocotb.start_soon(handlerMaster(dut))

    # Wait Reset
    await RisingEdge(dut.rst)
    await RisingEdge(dut.clk)

    # Await RX process
    await h_slave
    await h_master

    await Timer(100, units='ns')

    if simulation_err >= 1:
        print_rsl = "\n\n\n*****************************************************************************************\n"
        print_rsl += "**                                  There are " + str(simulation_err) + " errors !                              **\n"
        print_rsl += "*****************************************************************************************"
        cocotb.log.error(f"{print_rsl}")
    else:
        print_rsl = "\n\n\n*****************************************************************************************\n"
        print_rsl += "**                                       Simulation OK !                               **\n"
        print_rsl += "*****************************************************************************************"
        cocotb.log.info(f"{print_rsl}")
