-- Copyright (c) 2022-2024 THALES. All Rights Reserved
--
-- Licensed under the SolderPad Hardware License v 2.1 (the "License");
-- you may not use this file except in compliance with the License, or,
-- at your option. You may obtain a copy of the License at
--
-- https://solderpad.org/licenses/SHL-2.1/
--
-- Unless required by applicable law or agreed to in writing, any
-- work distributed under the License is distributed on an "AS IS"
-- BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
-- either express or implied. See the License for the specific
-- language governing permissions and limitations under the
-- License.
--
-- File subject to timestamp TSP22X5365 Thales, in the name of Thales SIX GTS France, made on 10/06/2022.
--

------------------------------------------------
--
--             AXIS_FRAME_CHK
--
------------------------------------------------
-- AXI4-Stream frame generator and checker
------------------------------
-- This module is used to compare input data from AXI4-Stream interface 
-- with data generated by module axis_pkt_gen.
-- 
-- It uses axis_pkt_gen to generate datas to compare with.
-- It uses axis_pkt_chk to compare the data generated and the ones received
-- on AXI4-Stream slave interface.
-- The module can be configured by the user.
-- It returns error signals.
--
------------------------------

library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;

library common;
use common.axis_utils_pkg.axis_enable;

use common.datatest_tools_pkg.all;

entity axis_frame_chk is
  generic(
    G_ASYNC_RST      : boolean   := false;
    G_ACTIVE_RST     : std_logic := '1';
    G_TDATA_WIDTH    : positive  := 64;                                                   -- Data bus size
    G_TUSER_WIDTH    : positive  := 8;                                                    -- User bus size used to transmit frame size 
    G_LSB_TKEEP      : boolean   := true;                                                 -- To choose if the TKEEP must be in LSB or MSB
    G_FRAME_SIZE_MIN : positive  := 1;                                                    -- Minimum size for data frame : must be between 1 and (2^G_TUSER_WIDTH) - 1
    G_FRAME_SIZE_MAX : positive  := 255;                                                  -- Maximum size for data frame : must be between 1 and (2^G_TUSER_WIDTH) - 1
    G_DATA_TYPE      : integer   := C_GEN_PRBS                                            -- PRBS : 0 / RAMP : 1
  );
  port(
    CLK               : in  std_logic;
    RST               : in  std_logic;
    -- Input ports
    S_TDATA           : in  std_logic_vector(G_TDATA_WIDTH - 1 downto 0);
    S_TVALID          : in  std_logic;
    S_TLAST           : in  std_logic;
    S_TUSER           : in  std_logic_vector(G_TUSER_WIDTH - 1 downto 0);
    S_TKEEP           : in  std_logic_vector(((G_TDATA_WIDTH + 7) / 8) - 1 downto 0);
    S_TREADY          : out std_logic;
    --Configuration ports
    ENABLE            : in  std_logic;
    NB_FRAME          : in  std_logic_vector(15 downto 0);                                -- Number of trame to generate : if 0, frame are generated endlessly
    FRAME_TYPE        : in  std_logic;                                                    -- '0' (static) : frames generated will always have the same size / '1' (dynamic) : frames will have different sizes
    FRAME_STATIC_SIZE : in  std_logic_vector(G_TUSER_WIDTH - 1 downto 0);                 -- Number of bytes in each frame in case the frame type is static
    DONE              : out std_logic;                                                    -- When asserted, indicate the end of data generation
    -- Error ports
    DATA_ERROR        : out std_logic;                                                    -- Indicate a difference in data between the two interfaces
    LAST_ERROR        : out std_logic;                                                    -- Indicate a difference on tlast between the two interfaces
    KEEP_ERROR        : out std_logic;                                                    -- Indicate a difference on tkeep between the two interfaces
    USER_ERROR        : out std_logic                                                     -- Indicate a difference on tuser between the two interfaces
  );
end axis_frame_chk;

architecture rtl of axis_frame_chk is

  --------------------------------------------------------------------
  -- Constants declaration
  --------------------------------------------------------------------
  constant C_TKEEP_WIDTH : integer := (G_TDATA_WIDTH + 7) / 8;

  --------------------------------------------------------------------
  -- Signals declaration
  --------------------------------------------------------------------
  -- AXIS from axis_pkt_gen
  signal pkt_gen_tready : std_logic;
  signal pkt_gen_tvalid : std_logic;
  signal pkt_gen_tdata  : std_logic_vector(G_TDATA_WIDTH - 1 downto 0);
  signal pkt_gen_tlast  : std_logic;
  signal pkt_gen_tkeep  : std_logic_vector(C_TKEEP_WIDTH - 1 downto 0);
  signal pkt_gen_tuser  : std_logic_vector(G_TUSER_WIDTH - 1 downto 0);

  -- AXIS from axis_enable
  signal axis_en_tready : std_logic;
  signal axis_en_tvalid : std_logic;
  signal axis_en_tdata  : std_logic_vector(G_TDATA_WIDTH - 1 downto 0);
  signal axis_en_tlast  : std_logic;
  signal axis_en_tkeep  : std_logic_vector(C_TKEEP_WIDTH - 1 downto 0);
  signal axis_en_tuser  : std_logic_vector(G_TUSER_WIDTH - 1 downto 0);

begin

  --===================================
  -- DATA GENERATION
  --===================================
  inst_axis_pkt_gen : axis_pkt_gen
    generic map(
      G_ASYNC_RST      => G_ASYNC_RST,
      G_ACTIVE_RST     => G_ACTIVE_RST,
      G_TDATA_WIDTH    => G_TDATA_WIDTH,
      G_TUSER_WIDTH    => G_TUSER_WIDTH,
      G_LSB_TKEEP      => G_LSB_TKEEP,
      G_FRAME_SIZE_MIN => G_FRAME_SIZE_MIN,
      G_FRAME_SIZE_MAX => G_FRAME_SIZE_MAX,
      G_DATA_TYPE      => G_DATA_TYPE
    )
    port map(
      CLK               => CLK,
      RST               => RST,
      M_TREADY          => pkt_gen_tready,
      M_TVALID          => pkt_gen_tvalid,
      M_TDATA           => pkt_gen_tdata,
      M_TLAST           => pkt_gen_tlast,
      M_TKEEP           => pkt_gen_tkeep,
      M_TUSER           => pkt_gen_tuser,
      ENABLE            => ENABLE,
      NB_FRAME          => NB_FRAME,
      FRAME_TYPE        => FRAME_TYPE,
      FRAME_STATIC_SIZE => FRAME_STATIC_SIZE,
      DONE              => DONE
    );

  inst_axis_enable : axis_enable
    generic map(
      G_ACTIVE_RST     => G_ACTIVE_RST,
      G_ASYNC_RST      => G_ASYNC_RST,
      G_TDATA_WIDTH    => G_TDATA_WIDTH,
      G_TUSER_WIDTH    => G_TUSER_WIDTH,
      G_TID_WIDTH      => 1,
      G_TDEST_WIDTH    => 1,
      G_REG_FORWARD    => false,
      G_REG_BACKWARD   => false,
      G_FULL_BANDWIDTH => true,
      G_PACKET_MODE    => true
    )
    port map(
      CLK         => CLK,
      RST         => RST,
      S_EN_TDATA  => ENABLE,
      S_EN_TVALID => '1',
      S_EN_TREADY => open,
      S_TDATA     => pkt_gen_tdata,
      S_TVALID    => pkt_gen_tvalid,
      S_TLAST     => pkt_gen_tlast,
      S_TUSER     => pkt_gen_tuser,
      S_TSTRB     => (others => '-'),
      S_TKEEP     => pkt_gen_tkeep,
      S_TID       => (others => '-'),
      S_TDEST     => (others => '-'),
      S_TREADY    => pkt_gen_tready,
      M_TDATA     => axis_en_tdata,
      M_TVALID    => axis_en_tvalid,
      M_TLAST     => axis_en_tlast,
      M_TUSER     => axis_en_tuser,
      M_TSTRB     => open,
      M_TKEEP     => axis_en_tkeep,
      M_TID       => open,
      M_TDEST     => open,
      M_TREADY    => axis_en_tready
    );

  --===================================
  -- DATA CHECKING
  --===================================
  inst_axis_pkt_chk : axis_pkt_chk
    generic map(
      G_ASYNC_RST   => G_ASYNC_RST,
      G_ACTIVE_RST  => G_ACTIVE_RST,
      G_TDATA_WIDTH => G_TDATA_WIDTH,
      G_TUSER_WIDTH => G_TUSER_WIDTH,
      G_TDEST_WIDTH => 1,
      G_TID_WIDTH   => 1
    )
    port map(
      CLK       => CLK,
      RST       => RST,
      S0_TDATA  => axis_en_tdata,
      S0_TVALID => axis_en_tvalid,
      S0_TLAST  => axis_en_tlast,
      S0_TUSER  => axis_en_tuser,
      S0_TSTRB  => (others => '-'),
      S0_TKEEP  => axis_en_tkeep,
      S0_TID    => (others => '-'),
      S0_TDEST  => (others => '-'),
      S0_TREADY => axis_en_tready,
      S1_TDATA  => S_TDATA,
      S1_TVALID => S_TVALID,
      S1_TLAST  => S_TLAST,
      S1_TUSER  => S_TUSER,
      S1_TSTRB  => (others => '-'),
      S1_TKEEP  => S_TKEEP,
      S1_TID    => (others => '-'),
      S1_TDEST  => (others => '-'),
      S1_TREADY => S_TREADY,
      ERR_DATA  => DATA_ERROR,
      ERR_LAST  => LAST_ERROR,
      ERR_KEEP  => KEEP_ERROR,
      ERR_STRB  => open,
      ERR_USER  => USER_ERROR,
      ERR_DEST  => open,
      ERR_ID    => open
    );

end rtl;
